!=========================================================================
! Copyright (C) VMware, Inc. 1986-2011.  All Rights Reserved.
!
! $Id: identitybag.gs,v 1.15.2.1 2008-05-07 22:26:09 bretlb Exp $
!
! Superclass Hierarchy:
!   IdentityBag, UnorderedCollection, Collection, Object.
!
!=========================================================================

removeallmethods IdentityBag
removeallclassmethods IdentityBag

category: 'For Documentation Installation only'
classmethod: IdentityBag
installDocumentation

| doc txt |
doc := GsClassDocumentation newForClass: self.

txt := (GsDocText new) details:
'An IdentityBag is an UnorderedCollection in which any distinct object can occur
 any number of times.  Adding the same (identical) object to an IdentityBag
 multiple times simply causes it to occur multiple times in the IdentityBag.

 Since an IdentityBag is an identity-based collection, different (non-identical)
 but equivalent (equal) objects are treated as distinct from each other.  In
 Bags, they are not distinct.  Adding multiple equivalent objects to an
 IdentityBag yields an IdentityBag with multiple objects as elements, each
 occurring once.' .

doc documentClassWith: txt.

txt := (GsDocText new) details:
'Methods in this category are obsolete and are provided only for compatibility
 with earlier releases of GemStone.  They will be removed in a future release.'.
doc documentClassCategory: #'Backward Compatibility' with: txt.

self description: doc.
%

category: 'Instance Creation'
classmethod: IdentityBag
new: anInteger

"(R) Returns an instance of the receiver with sufficient space to hold 
 anInteger number of elements.

 For IdentityBags, this method is the same as #new."

^ self new.
%

category: 'Modifying Classes'
classmethod: IdentityBag
isModifiable

"Returns true if the receiver and its Array of named instance variables are
 modifiable."

^ ((self isInvariant) | (instVarNames isInvariant) ) not
%

category: 'Backward Compatibility'
classmethod: IdentityBag
elementKind

"Obsolete in GemStone 3.2."

^ self varyingConstraint
%

category: 'Searching'
method: IdentityBag
_indexOf: anElement

"Returns the numerical index into the receiver of the given element (or nil if
 the element cannot be found)."

"Uses _at: to do the search and matches on equality."

1 to: self size do: [ :each |
  (self _at: each) = anElement ifTrue: [ ^each ]
  ].
^nil
%

category: 'Set Arithmetic'
method: IdentityBag
+ aBagOrSet

"Union.  Returns a kind of IdentityBag containing exactly the elements that are
 present in either the receiver or the argument aBagOrSet.

 The class of the result is the lowest class in the hierarchy of which both the
 receiver and argument are some kind.

 If the result is a kind of IdentitySet, then each element that occurs in either
 the receiver or aBagOrSet occurs exactly once in the result.  If the result is
 an IdentityBag that is not an IdentitySet, and if an element occurs m times in
 the receiver and n times in the argument aBagOrSet, then the result contains
 m + n occurrences of that element."

^ (aBagOrSet _isRcIdentityBag _or: [ aBagOrSet _isIdentityBag not ])
  ifTrue: [ self _primUnion: aBagOrSet _asIdentityBag ]
  ifFalse: [ self _primUnion: aBagOrSet ]
%

category: 'Set Arithmetic'
method: IdentityBag
_primUnion: aBagOrSet

""

<primitive: 221 >
aBagOrSet _validateClass: IdentityBag.
self _primitiveFailed: #_primUnion: .
self _uncontinuableError
%

category: 'Set Arithmetic'
method: IdentityBag
* aBagOrSet

"Intersection.  Returns a kind of IdentityBag containing only the elements that
 are present in both the receiver and the argument aBagOrSet.

 The class of the result is the lowest class in the hierarchy of which both the
 receiver and argument are some kind.

 If the result is a kind of Set, then each element that occurs in both the
 receiver and aBagOrSet occurs exactly once in the result.  If the result is a
 IdentityBag that is not an IdentitySet, and if an element occurs m times in the
 receiver and n times in the argument aBagOrSet, then the result contains the
 lesser of m or n occurrences of that element."

^ (aBagOrSet _isRcIdentityBag _or: [ aBagOrSet _isIdentityBag not ])
  ifTrue: [ self _primIntersect: aBagOrSet _asIdentityBag ]
  ifFalse: [ self _primIntersect: aBagOrSet ]
%

category: 'Set Arithmetic'
method: IdentityBag
_primIntersect: aBagOrSet

""

<primitive: 222>
aBagOrSet _validateClass: IdentityBag.
self _primitiveFailed: #_primIntersect: .
self _uncontinuableError
%

category: 'Set Arithmetic'
method: IdentityBag
- aBagOrSet

"Difference.  Returns a kind of IdentityBag containing exactly those elements of
 the receiver that have a greater number of occurrences in the receiver than in
 the argument aBagOrSet.  If an element occurs m times in the receiver and n
 times in aBagOrSet (where m >= n), then the result will contain m - n
 occurrences of that element.

 The class of the result is the class of the receiver."

^ (aBagOrSet _isRcIdentityBag _or: [ aBagOrSet _isIdentityBag not ])
  ifTrue: [ self _primDifference: aBagOrSet _asIdentityBag ]
  ifFalse: [ self _primDifference: aBagOrSet ]
%

category: 'Set Arithmetic'
method: IdentityBag
_primDifference: aBagOrSet

""

<primitive: 223>
aBagOrSet _validateClass: IdentityBag.
self _primitiveFailed: #_primDifference: .
self _uncontinuableError
%

category: 'Enumerating'
method: IdentityBag
do: aBlock

"Evaluates aBlock with each of the receiver's elements as the argument.
 The argument aBlock must be a one-argument block."

| theSize |
theSize := self size.
1 to: theSize do: [:i | aBlock value: (self _at: i) ].
^ self
%

category: 'Removing'
method: IdentityBag
remove: anObject

"Removes anObject from the receiver and returns anObject.  If anObject is
 present several times in the receiver, only one occurrence is removed.
 Generates an error if anObject is not in the receiver."

"Note: In GemStone 4.1, this method returned the receiver if an object
 equivalent to oldObject was found in the receiver."

<primitive: 207>  "primitive fails if anObject not found"
^ self _errorNotFound: anObject
%

category: 'Removing'
method: IdentityBag
removeIdentical: anObject

"Removes anObject from the receiver and returns anObject.  If anObject is
 present several times in the receiver, only one occurrence is removed.
 Generates an error if anObject is not in the receiver."

^ self remove: anObject
%

category: 'Removing'
method: IdentityBag
_remove: anObject

"Removes anObject from the receiver and returns anObject.  If anObject is
 present several times in the receiver, only one occurrence is removed.
 Generates an error if anObject is not in the receiver."

"Needed by certain subclasses where the super remove: message picks up an
 undesired reimplementation."

<primitive: 207>  "primitive fails if anObject not found"
^ self _errorNotFound: anObject
%

category: 'Removing'
method: IdentityBag
removeIdentical: anObject ifAbsent: exceptionBlock

"Removes from the receiver an object that is identical to anObject and
 returns anObject.  If several elements of the receiver are identical to
 oldObject, only one instance is removed.  If oldObject is not present in
 the receiver, evaluates anExceptionBlock and returns the result of the
 evaluation."

^ self remove: anObject ifAbsent: exceptionBlock.
%

category: 'Removing'
method: IdentityBag
remove: anObject ifAbsent: exceptionBlock

"Removes from the receiver an object that is identical to anObject and
 returns anObject.  If several elements of the receiver are identical to
 oldObject, only one instance is removed.  If oldObject is not present in
 the receiver, evaluates anExceptionBlock and returns the result of the
 evaluation."

"Note: In GemStone 4.1, this method returned the receiver."

<primitive: 209>  "primitive fails if anObject not found"
^ exceptionBlock value
%

category: 'Accessing'
method: IdentityBag
at: anIndex

"Returns the element of the receiver that is currently located at position
 anIndex.

 The elements of an IdentityBag are inherently unordered, and can change 
 position (index) when the IdentityBag is altered.  Thus, after an IdentityBag
 is altered, a given element may reside at a different index than before, and a
 given index may house a different element.  You should not infer an ordering
 for an IdentityBag's elements when you access them by index.

 This method is useful primarily as a code optimizer for iterating over all the
 elements of an IdentityBag (using a loop that runs the index from 1 to 
 the size of the IdentityBag).  

 The IdentityBag must not change during the iteration.  But the iteration may
 run faster than it would if you use other alternatives such as the do:
 method."

<primitive: 64>
(anIndex _isInteger)
   ifTrue: [ ^ self _error: #objErrBadOffsetIncomplete args: #[anIndex] ]
ifFalse: [ ^ self _errorNonIntegerIndex: anIndex ].
self _primitiveFailed: #at: .
self _uncontinuableError
%

category: 'Accessing'
method: IdentityBag
_at: anInteger

"Private.  Functionally equivalent to 'at:'."

"Used to implement the 'do:' method, which cannot be implemented as a
 primitive itself, since blocks cannot be sent as arguments to primitives."

<primitive: 64>
(anInteger _isInteger)
ifTrue: [ ^ self _error: #objErrBadOffsetIncomplete args: #[anInteger] ]
ifFalse: [ ^ self _errorNonIntegerIndex: anInteger ].
self _primitiveFailed: #_at: .
self _uncontinuableError
%

category: 'Updating'
method: IdentityBag
at: anIndex put: anObject

"Disallowed.  Generates an error, since the elements of an IdentityBag are not
 externally accessible through numeric indices."

^ self _errorNotIndexable
%

category: 'Updating'
method: IdentityBag
_basicAdd: anObject

"Same as IdentityBag | add:.  Has no effect if anObject is nil."

<primitive: 208>

self _primitiveFailed: #_basicAdd: .
self _uncontinuableError
%

category: 'Updating'
method: IdentityBag
_primitiveAt: anIndex put: anObject

"Disallowed.  Generates an error, since the elements of an IdentityBag are not
 externally accessible through numeric indices."

^ self _errorNotIndexable
%

category: 'Removing'
method: IdentityBag
_removeAll: aCollection errIfAbsent: aBool

"For each element in aCollection, removes the one element that is identical
 from the receiver.  Returns aCollection if successful."

"Note: In GemStone 4.1, this method returned the receiver."

<primitive: 226>

aBool _validateClass: Boolean .

(aCollection isKindOf: Array)
  ifFalse: [ 
    self _removeAll: (aCollection asArray) errIfAbsent: aBool.
    ^ aCollection
    ].

(aCollection isKindOf: CharacterCollection) ifTrue:[
  aBool
    ifTrue:[
      1 to: aCollection size do:[:j| self remove: (aCollection _at: j) ].
    ]
    ifFalse:[
      1 to: aCollection size do:
        [:j| self remove: (aCollection _at: j) ifAbsent:[] ].
    ].
  ^ aCollection
  ].

aCollection _validateClass: Collection .
self _primitiveFailed: #_removeAll:errIfAbsent: .
self _uncontinuableError
%

category: 'Removing'
method: IdentityBag
removeAll: aCollection

"Removes one occurrence of each element of aCollection from the
 receiver and returns the receiver.  Generates an error if any
 element of aCollection is not present in the receiver."

"Note: In GemStone 4.1, this method returned the receiver."

^ aCollection _isRcIdentityBag
  ifTrue: [ self _removeAll: aCollection _asIdentityBag errIfAbsent: true ]
  ifFalse: [ self _removeAll: aCollection errIfAbsent: true ]
%

category: 'Removing'
method: IdentityBag
removeAllPresent: aCollection

"Removes from the receiver one occurrence of each element of aCollection that is
 also an element of the receiver.  Differs from removeAll: in that, if some
 elements of aCollection are not present in the receiver, no error is generated.
 Returns aCollection."

"Note: In GemStone 4.1, this method returned the receiver."

^ aCollection _isRcIdentityBag
  ifTrue: [ self _removeAll: aCollection _asIdentityBag errIfAbsent: false ]
  ifFalse: [ self _removeAll: aCollection errIfAbsent: false ]
%

category: 'Comparing'
method: IdentityBag
= aBag

"Returns true if all of the following conditions are true:

 1.  The receiver and aBag are of the same class.
 2.  The two collections are of the same size.
 
 3.  Their public named instance variables are identical.
 4.  The elements of the receiver and aBag are identical.
 5.  Each element occurs the same number of times in the receiver and in aBag.

 Returns false otherwise."

<primitive: 600>
self _primitiveFailed: #= .
^false
%

category: 'Comparing'
method: IdentityBag
hash

"Returns an Integer hash code for the receiver."

| mySize |
(mySize := self size) = 0
  ifTrue: [ ^12794 ]
  ifFalse: [
    ^(((self _at: 1) identityHash bitShift: -1)
      bitXor: (self _at: mySize) identityHash)
        bitXor: mySize
  ]
%

category: 'Searching'
method: IdentityBag
collect: aBlock

"(R) Evaluates aBlock with each of the receiver's elements as the argument.
 Collects the resulting values into a collection of class specified by 
 sending #speciesForCollect message to the receiver and returns the new 
 collection.  The argument aBlock must be a one-argument block."

| result mySize |

result:= self speciesForCollect new.
mySize := self size.
1 to: mySize do: [ :i | result add: (aBlock value:(self _at:i)) ].
^ result
%

category: 'Searching'
method: IdentityBag
includesIdentical: anObject

"Returns true if the argument anObject is an element of the receiver.
 Returns false otherwise.  (Compare with includesValue:, which is based
 on equality.)"

<primitive: 65>
self _primitiveFailed: #includes: .
self _uncontinuableError
%

category: 'Searching'
method: IdentityBag
includes: anObject

"Returns true if the argument anObject is an element of the receiver.
 Returns false otherwise.  (Compare with includesValue:, which is based
 on equality.)"

<primitive: 65>
self _primitiveFailed: #includes: .
self _uncontinuableError
%

category: 'Searching'
method: IdentityBag
includesValue: anObject

"Returns true if the receiver contains an object of the same value as the
 argument, anObject.  Returns false otherwise.  (Compare with includes:,
 which is based on identity.)"

1 to: self size do: [ :each |
  (anObject = (self _at: each)) ifTrue: [ ^true ]
].
^ false
%

category: 'Searching'
method: IdentityBag
speciesForCollect

"Returns a class, an instance of which should be used as the result of
 collect: or other projections applied to the receiver."

^ IdentityBag
%

category: 'Searching'
method: IdentityBag
occurrencesOf: anObject

"Returns the number of the receiver's elements that are identical (==) to
 anObject."

<primitive: 67>
self _primitiveFailed: #occurrencesOf: .
self _uncontinuableError
%

category: 'Searching'
method: IdentityBag
_offsetOf: anObject

"Returns the first SmallInteger offset such that
 (self _at: (self _offSetOf: anObject)) == anObject.

 If the argument is not in the receiver, returns 0.  The result of this method
 is only valid while no insertions or deletions are made to the IdentityBag."

"Implemented with sequential scan of the receiver."

<primitive: 62>
self _primitiveFailed: #_offsetOf: .
self _uncontinuableError
%

category: 'Adding'
method: IdentityBag
add: newObject

"Adds newObject to the receiver.  Has no effect if newObject is nil."

<primitive: 208>
self _primitiveFailed: #add: .
self _uncontinuableError
%

! add:withOccurrences: made a primitive to fix 31939
category: 'Adding'
method: IdentityBag
add: anObject withOccurrences: aSmallInteger

"Includes anObject as an element of the receiver aSmallInteger number of times.
 
 Has no effect if anObject is nil."

<primitive: 549>
aSmallInteger _validateClass: SmallInteger .
self _primitiveFailed: #add:withOccurrences: .
self _uncontinuableError
%

category: 'Adding'
method: IdentityBag
addAll: aCollection

"Adds all of the elements of aCollection to the receiver."

^ aCollection _isRcIdentityBag
  ifTrue: [ super addAll: aCollection ]
  ifFalse: [ self _addAll: aCollection ]
%

! fixed 32782
category: 'Adding'
method: IdentityBag
_addAll: aCollection

"Adds all of the elements of aCollection to the receiver.  Occurrences 
 of nil in aCollection are not added to the receiver. 

 The primitive code handles aCollection if it is a kind of 
 Array or OrderedCollection, without faulting in elements of aCollection.
 Otherwise the do: implementation in this method is executed."

<primitive: 225>
(self == aCollection) ifTrue: [ ^ self addAll: (aCollection copy) ].
aCollection _validateClass: Collection.
aCollection do: [:each | self add: each].
^ aCollection
%

category: 'Copying'
method: IdentityBag
copyFrom: index1 count: aCount into: aColl startingAt: destIndex

"Copies elements of the receiver from (receiver at: index1)
  to (receiver at: (index2 + aCount - 1)), inclusive 
 into aColl starting at destIndex, overwriting the previous contents.

 If (index2 + aCount - 1) is beyond the end of the receiver, then 
 the copy will stop with the element  (receiver at: (receiver size)) .

 aColl must be a kind of Array or OrderedCollection .  

 returns the number of elements copied.
"
| numToCopy endIdx limit |
numToCopy := aCount .
endIdx := index1 + numToCopy - 1 .
limit := self size .
endIdx > limit ifTrue:[
  endIdx := limit .
  numToCopy := endIdx - index1 + 1 .
].
self copyFrom: index1 to: endIdx into: aColl startingAt: destIndex .
^ numToCopy
%

category: 'Copying'
method: IdentityBag
copyFrom: index1 to: index2 into: aColl startingAt: destIndex

"Copies the elements of the receiver from (receiver at: index1)
  to (receiver at: index2), inclusive,
 into aColl starting at destIndex, overwriting the previous contents.  
 aColl must be a kind of Array or OrderedCollection .  
 Returns the receiver. "

<primitive: 213>

(index1 > index2)
  ifTrue:[ index1 _error: #rtErrBadCopyFromTo args: #[index2]].

(index1 < 1) ifTrue:[ self _errorIndexOutOfRange: index1].

(index2 > self size) ifTrue:[ self _errorIndexOutOfRange: index2].

((destIndex < 1) _or: [(destIndex > (aColl size + 1))])
  ifTrue:[ self _errorIndexOutOfRange: destIndex].
aColl _validateClasses: #[ Array, OrderedCollection ].
self _primitiveFailed: #copyFrom:to:into:startingAt:  .
self _uncontinuableError
%


category: 'Reduced Conflict Support'
method: IdentityBag
_rcDifference: aBag

"Returns the set arithmetic difference between the receiver and aBag.  Also
 places the root object and interior nodes that are accessed in the
 reduced-conflict read set (RcReadSet) for both the receiver and aBag."

<primitive: 224>
aBag _validateClass: IdentityBag.
self _primitiveFailed: #_rcDifference: .
self _uncontinuableError
%

category: 'Reduced Conflict Support'
method: IdentityBag
_rcIncludes: anObject

"Returns true if anObject is contained identically in the receiver; returns
 false otherwise.  Also places the root object and any interior nodes that are
 accessed in the reduced-conflict read set(RcReadSet) ."

<primitive: 66>
self _primitiveFailed: #_rcIncludes: .
self _uncontinuableError
%

category: 'Reduced Conflict Support'
method: IdentityBag
_rcIncludesValue: anObject

"Returns true if anObject is contained in the receiver using equality
 comparison; returns false otherwise.  Also places the the root object of the
 receiver and any interior nodes that are accessed in the reduced-conflict read
 set(RcReadSet) ."

1 to: self size do: [ :each |
  (anObject = (self _rcAt: each)) ifTrue: [ ^true ]
].
^ false
%

category: 'Reduced Conflict Support'
method: IdentityBag
_rcOccurrencesOf: anObject

"Returns the number of occurrences of anObject in the receiver.  Places the
 root of the receiver and any interior nodes that are accessed in the
 reduced-conflict read set(RcReadSet) ."

<primitive: 68>
self _primitiveFailed: #_rcOccurrencesOf: .
self _uncontinuableError
%

!  category: 'Reduced Conflict Support', NOTES:
!
!  IdentityBag | _rcAt: is inherited from Object
!  The primitive in Object handles NSCs
!
! _rcRemoveAll:errIfAbsent: deleted
!
! _rcAdd: deleted

!category: 'Instance Migration'
!method: IdentityBag
!migrateFrom: anotherObject instVarMap: otherivi
!moved to UnorderedCollection

category: 'Removing'
method: IdentityBag
removeIfPresent: anObject

"Removes anObject from the receiver and returns anObject.  If anObject is
 present several times in the receiver, only one occurrence is removed.
 Returns nil if anObject is missing from the receiver."

"Note: In GemStone 4.1, this method returned the receiver."

<primitive: 207>  "primitive fails if anObject not found"
^nil
%

! category: 'Storing and Loading' methods moved to UnorderedCollection

category: 'Reduced Conflict Support'
method: IdentityBag
_selectiveAbort

"Performs an abort operation on the receiver. That is, if the object is
 committed, it removes any changes made by the current transaction and allows
 access to the committed state of the object.  This is not allowed if the object
 participates in an index."

_indexedPaths ~~ nil
  ifTrue: [ self _error: #rtErrSelectiveAbort ].
^ super _selectiveAbort
%

category: 'Repository Conversion'
method: IdentityBag
conversionRebuild

"Rebuilds an NSC to fix up internal sort order of the elements.
 Returns the receiver."

<primitive: 901> "enter protected mode"
| aClone numInstVars myClass myConstraint |

myClass := self _class.
myConstraint := myClass varyingConstraint.

"If constrained to hold only Pom oops, no need to rebuild."
((myConstraint == Object) _or: [myConstraint isSubclassOf: Magnitude]) ifFalse: [ 
  System _disableProtectedMode.
  ^ self 
  ].

aClone := myClass new.
aClone assignToSegment: self segment.

"First copy the named instance variables."
numInstVars:= myClass allInstVarNames size.
(self _class firstPublicInstVar) to: numInstVars do: [ :k | 
  aClone instVarAt: k put: (self instVarAt: k)
  ].

"Now copy the elements."
1 to: self size do:[:j | | anElement |
  anElement := self _at: j .
  aClone add: anElement .
  ].
aClone _primitiveBecome: self.
System _disableProtectedMode.
^aClone
%

category: 'Testing'
method: IdentityBag
_isIdentityBag

"Return whether the receiver is an identity-based collection."

^ true
%

category: 'Set Arithmetic'
method: IdentityBag
intersectAll: anArrayOfSetsOrBags into: aBagOrSet

"April 5, 2000 Norm Green - Created for OOCL

Starts by sorting by ascending size anArrayOfSetsOrBags.  The intersect the first
element with the receiver.  Take that result and intersect it with the second element.
Continue to all intersections are completed and place the result set into aBagOrSet.
Returns aBagOrSet.  If the array is empty, aBagOrSet will be a copy of the receiver.

This method will fail if any of the following is true:
 -the array size is greater than 2031.
 -any element of the array is not an IdentitySet or IdentityBag.
 -any element of the array is empty.
 -aBagOrSet is not an IdentityBag or IdentitySet"

<primitive: 532>
anArrayOfSetsOrBags _validateClass: Array.
aBagOrSet _validateClass: IdentityBag.
self _primitiveFailed: #intersectAll:into: .
self _uncontinuableError
%

category: 'Set Arithmetic'
method: IdentityBag
intersect: aBagOrSet into: anotherBagOrSet

"April 5, 2000 Norm Green - Created for OOCL
Same as the * method, except the result of the intersection
is put into anotherBagOrSet"

<primitive: 533>
aBagOrSet _validateClass: IdentityBag.
anotherBagOrSet _validateClass: IdentityBag.
self _primitiveFailed: #intersect:into: .
self _uncontinuableError
%

category: 'Set Arithmetic'
method: IdentityBag
sizeOfIntersectionWith: aBagOrSet

"April 5, 2000 Norm Green - Created for OOCL
Perform the intersection and return only the size of the result set"
<primitive: 531>
aBagOrSet _validateClass: IdentityBag.
self _primitiveFailed: #sizeOfIntersectionWith: .
self _uncontinuableError
%

category: 'Set Arithmetic'
method: IdentityBag
sizeOfIntersectionWithAll: anArrayOfBagsOrSets

"April 7, 2000 Norm Green - Created for OOCL
Starts by sorting by ascending size anArrayOfSetsOrBags.  The intersect the first
element with the receiver.  Take that result and intersect it with the second element.
Continue to all intersections are completed. Return the size of the resultant set.
If the array is empty, return the size of the receiver.

This method will fail if any of the following is true:
 -the array size is greater than 2031.
 -any element of the array is not an IdentitySet or IdentityBag.
 -any element of the array is empty.

All intermediate objects are disposed (no garbage is created)"

<primitive: 534>
anArrayOfBagsOrSets _validateClass: Array.
self _primitiveFailed: #sizeOfIntersectionWithAll: .
self _uncontinuableError
%


category: 'Private'
classmethod: IdentityBag
_clearEnumCache 

"Remove all entries from the enumeration cache"

<primitive: 547>
self _primitiveFailed: #_clearEnumCache .
self _uncontinuableError
%

category: 'Private'
method: IdentityBag
_removeFromEnumCache 

"Explicitly remove the receiver from the enumeration cache ,
 to facilitate garbage collection of temporary IdentityBags.

 Enumeration of the last element of an IdentityBag will
 automatically remove the receiver from the enumeration cache, so this
 method would be of use only if enumeration was terminated before
 reaching the end of an IdentityBag."

<primitive: 548>
self _primitiveFailed: #_removeFromEnumCache .
self _uncontinuableError
%

category: 'Repository Conversion'
method: IdentityBag
fixReferencesInFirstPass

"Return whether this object should have references to Float, SmallFloat,
 LargePositiveInteger and LargeNegativeInteger fixed inthe first pass.
 false means fix them in the 2nd pass.  IdentityBag objects can be part
 of an RcIdentityBag, so we do those later"

^self class ~~ IdentityBag
%
