!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! The Gs64 v3.0 AbstractException hierarchy is described in comment method below.
!
! Addtional code in AbstractException2.gs
!
! for methods of subclasses,  see files
!  Exception.gs, MessageNotUnderstood.gs , ZeroDivide.gs,  Warning.gs 

!=========================================================================

set class AbstractException

removeallmethods 
removeallclassmethods 

category: 'Documentation'
classmethod:
comment
^ '
 AbstractException describes the behavior of class objects that are used to
 create, signal, and select exceptions that exist within a specialization
 hierarchy.

 Beginning with Gemstone64 v3.0,  ANSI and legacy Gemstone
 exception behavior are unified.  The class ExceptionA no longer exists.

 The instance protocol describes the behavior of instances of class AbstractException.
 Typically, actual exceptions used by an application will be either
 direct or indirect subclasses of this class. 

 New trappable exception classes should be created as subclasses of Error, 
 not of AbstractException.  

 Note that instances created within the VM are initialized by VM code, and
 do not execute the initialize methods in the image.  Initializer methods
 in the image are only executed for those instances created by sending 
 some sort of signal: or error: message within Smalltalk code.

 The instVar   gsTrappable   has these values
    false - not trappable by any form of on:do: in Smalltalk code
    true  - trappable
    1       trappable except by onSynchronous:do:  ; exception was
            generated asynchrously (usually a ControlInterrupt)

 The instVar gsDetails holds the ANSI message text, if any.
 The instVar messageText holds the full GemStone error text .

 The exception class hierarchy follows, inst var names in parentheses.
 Unless noted, the default for all classes is  
   gsResumable==true, gsTrappable==true . 
 If an exception was signaled from within C code of a primtive, 
 the instance will have gsResumable==false.
 
  AbstractException (gsResumable gsTrappable gsNumber
                  currGsHandler gsStack gsReason gsDetails tag messageText args)
    Exception     --Highest class that customer Smalltalk code
                                should ever handle, ANSI Global--
      ControlInterrupt
        Break     -- from GciSoftBreak, ctl-C, etc
        Breakpoint (context stepPoint)
        ClientForwarderSend ([args=arguments] receiver clientObj selector)
        Halt    [gsTrappable==false]
      Error
        CompileError 
        EndOfStream
        ExternalError         -- Error from OS
          IOError
            SocketError
              SecureSocketError
          SystemCallError  (errno)
	  CryptoError -- Error from OpenSSL crypto library
        ImproperOperation [ args = object ]  -- Disallowed under the circumstances
          ArgumentError   -- Arg never appropriate for this message
          ArgumentTypeError ( expectedClass actualArg)  -- Class of arg
                                        never acceptable here
          CannotReturn        -- Non-local return attempt
          LookupError  (key) -- A lookup failed (or succeeded) improperly
          OffsetError (maximum actual) -- Array or String at:, at:put: out of bounds
          OutOfRange (minimum maximum actual) -- Numeric range error, or wrong num args
            FloatingPointError 
        IndexingErrorPreventingCommit
        InternalError         -- something that should never happen
          GciTransportError   -- Protocol or other error specific to GCI RPC link
        LockError [ args = object ]
        MigrationError
        NameError (selector)  -- used in Ruby
          MessageNotUnderstood ([args=arguments] envId receiver )  --ANSI Global
        NumericError [ args = object ]
          ZeroDivide  (dividend)       --ANSI Global
        RepositoryError
        SecurityError
        SignalBufferFull
	ThreadError 
        TransactionError
        UncontinuableError
        UserDefinedError 
      Notification     --ANSI Global
        Admonition
          AlmostOutOfStack [gsTrappable, gsResumable both false if in red zone]
          AlmostOutOfMemory
          RepositoryViewLost
        Deprecated
        FloatingPointException
        InterSessionSignal (sendingSession signal [args = message] )
        ObjectsCommittedNotification
        TerminateProcess
        TransactionBacklog (inTransaction)
        Warning      --ANSI Global 
          CompileWarning
      TestFailure
    RubyBreakException  -- used in implementation of Ruby 
    RubyThrowException ([args=name] value signalNum) -- used in implementation of Ruby

  See comments for primitive 33 , AbstractException >> defaultAction for
  how instVars of an exception are enumerated in to a GciErrSType.args.

Trappability of errors signaled from with VM

  OBJ_ERR_DOES_NOT_EXIST, OBJ_ERR_CORRUPT_OBJ not trappable in a slow-build vm
  RT_ERR_OBJ_IS_NP  not trappable if TrapAddToClosure enabled

  following signaled from C as not trappable
    RT_ERR_STACK_LIMIT_RED 
    GCI_ERR_EXEC_CLIENT_USERACT  
    RT_ERR_HARD_BREAK
    RT_ERR_STEP
    RT_ERR_CODE_BREAKPOINT
    RT_ERR_STACK_BREAKPOINT
    RT_ERR_SOFT_BREAK
    RT_ERR_PAUSE

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
'
%

classmethod:
errnoTables
 "Returns the class variable ErrnoTables .
  The class variable ErrnoTables is a zero-based invariant Array , 
   to be indexed by the one-based result of  AbstractException(C)>>cpuOsKind .

   Each element of ErrnoTables is an Array of Strings, to be accessed
     by a one-based errno value, translating the errno value to a name.

   On some cpu/OS there are multiple names for some errno values.
   The tables contain only the preferred name, as follows
     Solaris:  EWOULDBLOCK==EAGAIN , EAGAIN is the preferred name .
     Linux: EWOULDBLOCK == EAGAIN, EAGAIN is the preferred name ,
            EDEADLOCK == EDEADLK,  EDEADLK is the preferred name .
     Apple Unix: EWOULDBLOCK==EAGAIN , EAGAIN is the preferred name ,
                 EOPNOTSUPP == ENOTSUP, ENOTSUP is the preferred name .
  "
  ^ ErrnoTables
%

classmethod
errnoToName: aSmallInteger
 "Return the name for specified errno value, or nil if argument
  is out of range. 

  See AbstractException>>errnoTables for discussion of preferred names
" 

 | arr |
 arr := ErrnoTables at: self cpuOsKind .
 (aSmallInteger >= 1 and:[aSmallInteger <= arr size]) ifTrue:[ 
   ^ arr at: aSmallInteger .
 ].
 ^ nil
%

! --------------------------------------------------------------------------
category: 'Default handlers'
classmethod:
addDefaultHandler: handlerBlock

"Install the one-argument block handlerBlock as a static GsExceptionHandler
 to field exceptions whose classes are subclasses of the receiver.
 The handlerBlock may send  #resume: , #pass, or #outer to the 
 instance of Exception .

 If handlerBlock returns normally, or sends #return:
 to the instance of Exception, an Error will be signalled
 because there is no send of on:do:  which installed handlerBlock .

 Returns a new instance of GsExceptionHandler .

 When searching for a handlerBlock to handle a signaled exception, the VM
 uses Behavior>>_subclassOf: semantics . classHistories of the 
 class of the signaled exception and of self are ignored.  "

^ self _installStaticException: handlerBlock class: self
       category: nil number: nil subtype: nil
%

classmethod:
defaultHandlers
  "Returns all currently installed ANSI default handlers which
   handle the receiver. "
  | handler res hcls |
  res := { } .
  handler := self _staticExceptions .
  [ handler ~~ nil ] whileTrue:[
    (hcls := handler exceptionClass) ifNotNil:[
      (self _subclassOf: hcls) ifTrue:[ res add: handler ].
    ].
    handler := handler next 
  ].
  ^ res
% 
classmethod:
removeAllDefaultHandlers

"Removes all ANSI default handlers and static legacy handlers.
 A faster implementation of
  AbstractException _staticExceptions do:[:h | h remove ].
"
<primitive: 380>
^ self _primitiveFailed: #removeAllDefaultHandlers
%

category: 'Legacy Handlers'
classmethod:
removeStaticException: aGsExceptionHandler

"Remove a static exception handler (either an ANSI default handler,
 or a static legacy handler.)
 Returns the removed handler, or nil if not found by identity."

<primitive: 379>    
aGsExceptionHandler _validateClass: GsExceptionHandler .
^ self _primitiveFailed: #removeStaticException args: { aGsExceptionHandler }
%

category: 'Private'
classmethod:
_staticExceptions

"Returns the head of the static exception list. The list includes
 static legacy handlers, and ANSI default handlers.  Elements of the
 list are instances of GsExceptionHandler."

<primitive: 377>
^ self _primitiveFailed: #_staticExceptions
%

classmethod:
_installStaticException: handlerBlock
  class: exceptionClass
  category: aCategory number: aNumber subtype: atype

"Install the specified exception block at the head of the static handlers 
 list, not associated with a particular stack frame ,
 to field exceptions of the specified class .
 or errors of the specified category, number, and subtype.
 If exceptionClass is not nil   ignores  aCategory,  aNumber, atype  ;
 else assumes aCategory to be GemstoneError .

 The handler is represented as an instance of GsExceptionHandler,
 and will be a part of the session state.  
 The new  GsExceptionHandler is returned ."
<primitive: 378>
| nBlockArgs |
handlerBlock _validateClass: ExecBlock .
exceptionClass ifNotNil:[
  nBlockArgs := 1 .
  exceptionClass _validateClass: AbstractException .  
] ifNil:[
  nBlockArgs := 4 .
  " aCategory ifNotNil:[ aCategory _validateClass: SymbolDictionary ] . "
  aNumber ifNotNil:[ aNumber _validateClass: SmallInteger ] .
].
handlerBlock argumentCount ~~ nBlockArgs ifTrue: [
  ^ self _error: #rtErrExceptBlockNumArgs args: { nBlockArgs . handlerBlock argumentCount }.
].
^ self _primitiveFailed: #_installStaticException:class:category:number:subtype: 
       args: { handlerBlock . exceptionClass . aCategory . aNumber . atype }
%

!----------------------------------------------------------------
category: 'Instance creation'

classmethod:
signal

 "An exception of the type associated with the receiver is signaled."

  ^ self new _signalWith: nil
%

classmethod:
signalNotTrappable

  ^ self new signalNotTrappable
%

classmethod:
signal: signalText

 "An exception of the type associated with the receiver is signaled."
  ^ self new signal: signalText.
%
classmethod:
new
  "Return a newly created object initialized to a standard initial state"

  ^ self _basicNew initialize
%

category: 'ExceptionSet support'

classmethod:
, anExceptionSetOrClass

 "Return an ExceptionSet containing the receiver and argument."

  ^ ExceptionSet with: self with: anExceptionSetOrClass
%

classmethod:
handles: anException

^ anException isKindOf: self
%

category: 'Debugging Support'
classmethod:
cpuOsKind

"Returns an integer specifying the CPU and OS on which
 this VM is running. The result is one of
   1 - reserved for use within VM 
   2 - sparc Solaris
   3 - x86_64 Linux
   4 - PowerPc AIX
   5 - x86_64 Apple Unix
   6 - x86_64 Solaris
   7 - Itanium HPUX
"
<primitive: 345>
self _primitiveFailed: #cpuOsKind
%

classmethod:
cpuOsKindString

^ #( 'unknown'
     'sparc-solaris'
     'x86_64-linux'
     'powerpc-AIX'
     'x86_64-darwin'
     'x86_64-solaris'
     'ia64-HPUX'       ) at: self cpuOsKind
%

!----------------------------------------------------------------
category: 'Instance initialization'
method:
initialize

 "subclasses may override but must send   super initialize  
  at start of the reimplementation, or else initialize
  the 3 instVars   gsNumber, gsResumable, gsTrappable . "

  gsNumber := ERR_AbstractException.
  gsResumable := true .
  gsTrappable := true .
%

method
_number: aSmallInt
  gsNumber := aSmallInt
%
method:
_resumable: aBoolean
  gsResumable := aBoolean
%

!----------------------------------------------------------------
category: 'Compatibility'
method:
gsNumber
  "Will be deprecated"
  ^ gsNumber
%

method:
gsArguments
  "Will be deprecated"
  "Returns an Array"

  ^ self _legacyHandlerArgs
%

method:
gsCategory
  "Will be deprecated"
  ^ GemStoneError 
%
method:
category
  "Will be deprecated"
  ^ GemStoneError
%

!----------------------------------------------------------------
category: 'Private'
method:
number
  ^ gsNumber
%

method:
_gsStack: anArray
  gsStack := anArray
%

category: 'Accessing'
method:
isResumable
  ^ gsResumable .
%

method:
isTrappable
  ^ gsTrappable ~~ false
%

method:
details: aStringOrArray
  "If arg is a String it will be appended to the fully formed error 
   message.

   If arg is an Array and the receiver does not reimplement _description:,
   the arg will be used as a legacy style error message template."
   
  gsDetails := aStringOrArray  "store ANSI messageText"
%

method:
details
  ^ gsDetails ifNil:[ messageText ]
%

method:
messageText: aString
  gsDetails := aString   "store ANSI messageText"
%

method:
messageText
  "return ANSI messageText"
  ^ gsDetails ifNil:[ messageText ]
  
%

method: 
_description: subclassDetails

 "Actual implementation in AbstractException2.gs "

  ^ 'error messages not available early in slow filein'
%

method:
_legacyDetails: templateString
  | str theArgs elem |
  theArgs := self _legacyHandlerArgs .
  str := String new .
  1 to: templateString size do:[:j |
    elem := templateString at: j .
    elem _isOneByteString ifTrue:[
      str add: elem   
    ] ifFalse:[
      elem _isSmallInteger ifTrue:[
	elem < 0 ifTrue:[ str add: (theArgs atOrNil: 0 - elem) asOop asString ]
	      ifFalse:[ str add: (theArgs atOrNil: elem) describe1K ]
      ]
    ]
  ].
  ^ str
%

method:
asString
 "Reimplement asString in subclasses to generate class-specific
  detail string as argument to _description: "

  ^ self _description: nil
%

method:
genericDescription
  ^  self asString
%

method:
description
  "Returns a String."
 | res |
 gsDetails ifNotNil:[ :dt |
   res := [ 
     | str | 
     (str := String withAll: self class name) 
         add: ': ' ; add: dt asString  .
     str
   ] onException: Error do:[:ex | 
     ex return: nil "ignore" 
   ]
 ].
 ^ res ifNil:[ self asString ]
%

method:
describe
  "used by topaz and VM's error message building"

  ^ self asString
%

method:
reason: aString
  "aString must be a Symbol or String 
   Only the first 63 bytes of aString are used. "
  | str |
  aString _isSymbol ifTrue:[
    gsReason := aString asString .
  ] ifFalse:[
    (str := aString) _isOneByteString ifFalse:[ 
       str := aString asString .
       str _isOneByteString ifFalse:[  
         str := aString class name  .
         str _isOneByteString ifFalse:[ ^ self "ignore arg" ].
       ].
    ].
    (str size > 64) ifTrue:[
       gsReason := aString copyFrom: 1 to: 63 
    ] ifFalse:[
       gsReason := aString
    ].
  ].
%

method:
reason
  ^ gsReason
%

method:
tag
  ^ tag ifNil:[ gsDetails ]
%
method:
tag: anObject
  tag := anObject
%
method:
arg: anObject
  gsArgs ifNotNil:[:a | a _isArray ifTrue:[ a add: anObject ]]
       ifNil:[ gsArgs := { anObject } ]
%
method:
args: anArray
  gsArgs :=  anArray ifNotNil:[:a| a _isArray ifTrue:[ a ] 
                                           ifFalse:[ { a } ]] 
%

method:
_gsStack

  " If (System gemConfigurationAt:#GemExceptionSignalCapturesStack) == true ,
   and gsStack==nil when primitive 2022 (AbstractException>>_signalWith:) is invoked,
   then primitive 2022 will fill in gsStack with an Array .
   The Array contains a Boolean (inNativeCode), 
   followed by triples of  aGsNMethod , an ipOffset, a receiver   
   per the result of GsProcess>>_frameContentsAt: ,
   terminated by the end of the array, or a nil."
    
  ^ gsStack
%

! fixed 44259
method:
stackReport

"Returns a formatted String derived from gsStack instVar, or nil .

 If (System gemConfigurationAt:#GemExceptionSignalCapturesStack) == true ,
 and gsStack==nil when primitive 2022 (AbstractException>>_signalWith:) is invoked,
 then primitive 2022 will fill in gsStack with an Array .
"
| arr nativeStk |
arr := gsStack .
(arr _isArray and:[ (nativeStk := arr atOrNil:1 ) class == Boolean]) ifTrue:[
  | report lf | 
  report := String new .  lf := Character lf .
  2 to: arr size by: 3 do:[:j | | meth ip stepPoint rcvr |
    (ip := arr atOrNil: j + 1 ) ifNotNil:[
      meth :=  arr at: j .
      rcvr := arr at: j + 2 .
      ip < 0 ifTrue:[ ip := 0 ].
      report add: (meth _descrForStackPadTo: 0 rcvr: rcvr)  .
      nativeStk ifTrue:[ ip := meth _nativeIpOffsetToPortable: ip asReturn: false].
      stepPoint := meth _stepPointForIp: ip level: 2 useNext: nativeStk .  
      report add:' @' ; add: stepPoint asString ;
          add: ' line ';
          add: (meth _lineNumberForStep: stepPoint) asString ;
          add:'  [GsNMethod '; add: meth asOop asString ; add:']';  add: lf .
    ].
  ].
  ^ report
].
^ nil
%

!----------------------------------------------------------------
category: 'Signaling'
method:
signal: signalText

  "The message text of the receiver is set to signalText,
   and the receiver is signaled.  See also  #signal ."

  gsDetails := signalText .  "store ANSI messageText"
  ^ self _signalWith: nil . 
%

method:
signalNotTrappable

  "Receiver is signaled and it will not be trappable by an
   exception handler.  An Error will be return to the GCI. "

  gsTrappable := false .
  ^ self _signalWith: nil .
%

method:
signal

  "The current stack is searched for an 'exception handler' which matches
  the receiver.  The search proceeds from the top of stack downwards,
  and then checks the static legacy handlers.

  ANSI handlers are installed by using
    ExecBlock>>on:do:    installs an ANSI handler 
    ExecBlock>>onException:do:    installs an ANSI handler 
  ANSI static handlers may be installed by
    AbstractException(C)>>defaultAction:
  Legacy handlers may be installed by 
    AbstractException(C)>>category:number:do:
  Legacy static handlers may be installed by
    AbstractException(C)>>installStaticException:category:number: 

  A matching ANSI handler is defined to be one which would return 'true' 
  if the message #handles: was sent to its 'exception selector' 
  with the 'signaled exception' as the argument.

  If a matching handler is found, its handlerBlock is executed.

    For a legacy handler or ANSI default handler, if the handlerBlock 
    returns normally, execution resumes from the point of the signal , 
    with the value returned by the handlerBlock.

    For a handlerBlock installed by on:do: , if the handlerBlock returns normally,
    any ensure: blocks are executed, and execution returns from the send of on:do:
    with the value returned by the handlerBlock .  However if the return from
    the send of on:do: would cross the frame of a C primitive or user action,
    an uncontinuable error is generated.
    Within the handlerBlock, the methods  
      outer, pass , resignalAs:, resume, return ,  
    can be send to this instance of AbstractException  to alter flow . 

  If a matching handler is not found, 'default action' for the receiver 
  is performed, as if #defaultAction were sent to the receiver.
  If the receiver is 'resumable' the value returned from the
  #defaultAction method is returned as the value of the #signal message, 
  otherwise an error is returned to the GCI client. 
 
  If (System gemConfigurationAt:#GemExceptionSignalCapturesStack) == true ,
  and gsStack==nil when primitive 2022 (AbstractException>>_signalWith:) is invoked,
  then primitive 2022 will fill in gsStack with an Array.  
  See AbstractException>>stackReport. "

  ^ self _signalWith: nil
%

category: 'Private'
method
_signalWith: inCextensionArg
  <primitive: 2022>

   "The current stack is searched for an 'exception handler' per
    AbstractException>>signal .
    If a handler found, new frame pushed 
      to execute  _executeHandler:   or   _executeGsHandler:  .
      and the primitive does not return.
    If exception handling succeeds and execution is to resume, 
    either the resume: or the return: primitive will do the 'goto'
    and we don't actually return from this frame .

    inCextensionArg must be nil at entry, and may be modified
    by primitive.

    If a Smalltalk handler not found, 
    primitive fails so we can defer to a C extension or send defaultAction here."
 
  | res |
  inCextensionArg ifNotNil:[
    "primitive found a C extension which wants to handle receiver."
    self _executeEnsuresBelow: 1 .  "execute ensures from TOS to C extension"
    self _handleInCextension.  "trims stack and returns to C extension"
  ].
  res := self defaultAction .  
    "if you change code from beginning of method to here , may need to
     alter C code at  IntSwiReplaceTos_IPOFFSET"

  1 timesRepeat:[ self class ]. "loop to detect/handle termination interrupt"
  self isResumable ifTrue:[
    ^ res .
  ].
  self _signalGciError .
  self _uncontinuableError . "should never reach here"
%

method:
_signalFromPrimitive: inCextensionArg

  "Send from within the VM only.
   An exception is being signalled from within a C primitive.
   The signalled exception is not resumable .
   otherwise same behavior as AbstractException>>_signalWith: . "

  <primitive: 2022>
  | res num |
  inCextensionArg ifNotNil:[
    "primitive found a C extension which wants to handle receiver."
    self _executeEnsuresBelow: 1 .  "execute ensures from TOS to C extension"
    self _handleInCextension.  "trims stack and returns to C extension"
  ].
  res := self defaultAction .
    "if you change code from beginning of method to here , may need to
     alter C code at  IntSwiReplaceTos_IPOFFSET"

  ((num := gsNumber) > 6000 and:[   "allow breakpoints to be resumable from Smalltalk."
     num == 6002 or:[ num == 6005 or:[ num == 6006 ]]]) ifTrue:[
       self isResumable ifTrue:[
         ^ res        
     ]
  ].
  1 timesRepeat:[ self class ]. "loop to detect/handle termination interrupt"
  self _signalGciError .
  2 timesRepeat:[ self class ]. "loop to detect/handle termination interrupt"
  self _uncontinuableError . "should never reach here"
%

method:
_handleInCextension

"Private.
 Caller responsible for executing ensure blocks.
 Unwinds Smalltalk stack to the most recent C extension and
 returns to the C code with receiver as the signaled exception."

<primitive: 556>  "primitive fails unless this is Maglev VM"
"should never reach here"
self _primitiveFailed: #_handleInCextension.
self _uncontinuableError . 
%

!----------------------------------------------------------------
category: 'Debugging Support'
classmethod:
installDebugBlock: aBlock

"aBlock must be either nil or a one-argument ExecBlock taking an instance
 of AbstractException as the argument.   
 aBlock == nil clears any previously installed block from the VM. 
 Each GciLogin includes an automatic   AbstractException installDebugBlock: nil
 for the newly created session.

 aBlock ~~ nil installs the specified block into VM transient state,
 for use by AbstractException>>_debugException:  .
 aBlock will be sent   value: anException   prior to invocation of any
 handler block installed with an on:do:  (or prior to _executeGsHandler:).
 If debugger desires to handle the exception normally, aBlock must return
 normally (i.e. do not return from home, or signal any exception within aBlock).

 aBlock will not be invoked for non-trappable exceptions such as Object>>pause.

 aBlock is only executed if a matching handler was found on the stack.
 If there is no matching handler (i.e. no matching on:do:) then the 
 exception is signalled directly to the GCI client.
"

<primitive: 798>
aBlock _validateClass: ExecBlock .
aBlock numArgs ~~ 1 ifTrue:[ ArgumentError signal:'must be a 1-arg block' ].
self _primitiveFailed: #installDebugBlock args: { aBlock }
%

!----------------------------------------------------------------
category: 'Private'
method:
_handlerActive
  "an exception currently being handled by an on:do: will have
     FP information appended."

  ^ self size ~~ 0
%


method:
_debugException: fakeArg

"If a debugger block has been installed with AbstractException(C)>>installDebugBlock:
 execute that block.  Primitive succeeds and returns receiver if no
 debugger block is installed.
 Returns receiver"

<primitive: 799>
"if we get here, a debugger block was found and the argument fakeArg has 
 been changed to be that block."

 fakeArg value: self .
 ^ self
%

! fixed 46512
method:
_executeHandler: aBlock

  "Execute aBlock , which was the second arg of an #on:do: ,
   to handle the receiver.  Sent from VM C code only .
   aBlock must be a block taking zero or one arguments." 
  <primitive: 2025> "mark frame with executeHandler_Mark_NIL, always fails"
  | res |
  self _debugException: nil .
  res := aBlock argumentCount == 0 
           ifTrue:[ aBlock value  ] 
           ifFalse:[ aBlock value: self ].

  "If we get here, then handler block did not send a #resume,
   so continue program execution by returning from the #on:do: send
   which installed the handlerBlock we just executed."

  self return: res .  
%
! fixed 47456
method:
_executeOuterHandler: aBlock
  "Execute aBlock , which was the second arg of an #on:do: ,
   to handle the receiver.   Sent from implementation of #outer only ."

  <primitive: 2025> "mark frame with executeHandler_Mark_NIL, always fails"
  self _debugException: nil .

  ^ aBlock argumentCount == 0 
           ifTrue:[ aBlock value  ] 
           ifFalse:[ aBlock value: self ].
%

method:
_executeGsHandler: aGsExceptionHandler
  "Execute aGsExceptionHandler's block .  Sent from VM C code only"
  | res err |
  self _debugException: nil .
  currGsHandler := aGsExceptionHandler .
  aGsExceptionHandler _ansiBlock ifNotNil:[ :blk | 
    "static ansi handler, normal return from handler disallowed"
    res := blk value: self .
    (err := UncontinuableError new) 
       details:'an ANSI defaultHandler requires explicit resume:'.  
    (self isKindOf: UncontinuableError) 
       ifTrue:[ err signalNotTrappable  "avoid infinite recursion"]
      ifFalse:[ err signal ]. 
  ] ifNil:[  "legacy handler , if handler block returns normally
              resume program execution from the point of the signal."
    res := aGsExceptionHandler block value: self value: GemStoneError
		value: gsNumber value: self _legacyHandlerArgs .
  ].
  self isResumable ifFalse:[ | handNum |
    (err := Error new) 
       details: 'cannot resume from a not-resumable Exception' .
     handNum := aGsExceptionHandler number .
     (handNum == nil or:[ handNum == err number]) 
        ifTrue:[ err signalNotTrappable  "avoid infinite recursion" ]
       ifFalse:[ err signal ].
  ].
  currGsHandler := nil .   "clear to allow use of resume:"
  self _resume: res  
%
method:
_executeOuterGsHandler: aGsExceptionHandler
  "Execute aGsExceptionHandler's block .  Sent from image only"
  self _debugException: nil .
  currGsHandler := aGsExceptionHandler .
  aGsExceptionHandler _ansiBlock ifNotNil:[ :blk | 
    "static ansi handler, normal return returns to sender of outer "
    ^ blk value: self .
  ] ifNil:[ | res |  "have a legacy handler" 
    res := aGsExceptionHandler block value: self value: GemStoneError
                value: gsNumber value: self _legacyHandlerArgs .
    "if legacy handler returns normally, do a resume:"
    self isResumable ifFalse:[ | handNum err |
      (err := Error new)
         details: 'cannot resume from a not-resumable Exception' .
       handNum := aGsExceptionHandler number .
       (handNum == nil or:[ handNum == err number])
          ifTrue:[ err signalNotTrappable  "avoid infinite recursion" ]
         ifFalse:[ err signal ].
    ].
    currGsHandler := nil . "clear to allow use of resume:"
    self _resume: res
  ].
%

method:
_legacyHandlerArgs
  "Must be reimplemented in subclasses of AbstractException
   which have named instVar beyond 'args' "

  ^ gsArgs ifNil:[ #() ]
%

method:
_enableEvents

"perform the enable interrupts part of _gsReturnNothingEnableEvents"

<primitive: 141>
self _uncontinuableError
%

method:
_signalTerminateProcess
  "sent from within the VM "

  GsProcess _current _serviceTerminationInterrupt 
%

method:
_signalAsync

  "To be sent only by the virtual machine.
   The receiver is an asynchronous Exception, usually a ControlInterrupt.
   Examples are  such as soft-break, (ErrorSymbols at: #rtErrSoftBreak, error 6003) 
   or signalAbort (ErrorSymbols at: #rtErrSignalAbort , error 6009) .

   Handling of an asynchronous Exception happens between bytecodes and
   if successful does not alter the stack.  This method is always
   executed for an asynchronous Exception, regardless of whether a
   handler exists. "

  | ok |
  gsTrappable == true ifTrue:[ 
    gsTrappable := 1  "prevent onSynchronous:do: from handling this".
  ].
  ok := false .
  [
    self _signalWith: nil . "if handler not found  synchronous error thrown to GCI"
    ok := true .  "if handled ok we get to here"
  ] ensure:[
    ok ifFalse:[
      self _enableEvents . "reenable interrupts if resume not successful"
    ]
  ].
  self _gsReturnNothingEnableEvents "special selector optimized by compiler"
%

method:
_signalTimeout

  "To be sent only by the virtual machine.
   The receiver is an asynchronous Exception, error 6015 "

  | ok |
  ok := false .
  [
    gsNumber == ERR_TIMEOUT_INTERRUPT ifTrue:[
      ProcessorScheduler scheduler _yieldForTimeout .
      ok := true .  "if handled ok we get to here"
    ] ifFalse:[
      self _signalWith: nil .
    ].
  ] ensure:[
    ok ifFalse:[
      self _enableEvents . "reenable interrupts if resume not successful"
    ]
  ].
  self _gsReturnNothingEnableEvents "special selector optimized by compiler"
%

method:
_signalGcFinalize

  "To be sent only by the virtual machine.
   The receiver is an asynchronous Exception for error 6017 "
[ | chunk chunkSize anEphemeron  |
  chunk := System __sessionStateAt: 2 . "head of finalization queue"
  [ chunk ~~ nil ] whileTrue:[ 
    chunkSize := chunk size .
    1 to: chunkSize - 1 do:[:n | 
      anEphemeron := chunk at: n .
      [ anEphemeron mourn 
      ] onException: Error do:[:ex |
        GsFile gciLogServer: ex asString , 'during mourn'
      ].
    ].
    chunk := chunk at: chunkSize . "last element is next chunk"
    System __sessionStateAt: 2 put: chunk. "deref chunk just processed"
  ]
] ensure:[
  self _enableEvents . "reenable interrupts if resume not successful"
].
self _gsReturnNothingEnableEvents "special selector optimized by compiler"
%

method:
_executeEnsuresBelow: kind
 | ensBlks |
 ensBlks := self _getEnsuresBelow: kind .
 1 to: ensBlks size by: 2 do:[:j|
   self _removeEnsureAtFP: (ensBlks at: j) .
   (ensBlks at: j + 1) value .
 ].
%

method:
_removeEnsureAtFP: anOffset

 "Dereference the ensure block in specified frame,
  so it won't be executed more than once.
  Frame changed  from  ensure:[] to  ensure: nil ."

 <primitive: 696>
 self _uncontinuableError
%

method:
_getEnsuresBelow: kind

 "kind determines specified frame ,
     kind 0 -->  frame where receiver originally signaled

     kind 1 -->  #on:do: frame for currently active handler, 
         and also clears mark_nil from _executeHandler frames
         while searching for the ensure blocks. Stops the
         search at a C FFI or UA  to Smalltalk boundary.

  Returns an Array containing pairs of 
    FP offset,  block that was arg to an ensure:  
  for all #ensure:  frames  between top of stack and and specified frame

  Does not remove the ensure block from their frames.
  Returns nil if there are none."

 <primitive: 695>
 self _uncontinuableError
%

method:
_retryUsing: alternativeBlock

 "When sent within an ANSI handler's handlerBlock ,
  resume execution in the frame of the #on:do: send which
  installed the currently active handlerBlock.

  caller has executed ensure: blocks between top of stack and that #on:do: .
  this primitive will trim stack back to that #on:do: .

  If alternativeBlock  is non-nil , it must be a zero arg
  ExecBlock and it is substituted for the original receiver of the #on:do: .
  Then #value is sent to the non-nil alternativeBlock, else to the
  original receiver of #on:do: .
  
  Generates an error if sent within a legacy handler's handlerBlock."

  <primitive: 2019>  "does not return"
  self _uncontinuableError
%

method:
_resignalAs: replacementException

  "Caller has executed ensure: blocks between top of stack and 
   point where receiver was originally signalled.
   This primitive will trim stack back to that signal frame.

   Then replacementException is substituted for the receiver
   and execution of the signal method is restarted."

 <primitive: 2023>
 replacementException _validateClass: AbstractException .
  self _uncontinuableError
%

method:
_return: returnValue
 "When sent within an ANSI handler's handlerBlock,
  resume execution by returning returnValue from the #on:do: send 
  which installed the currently active handlerBlock .   

  Caller has executed ensure: blocks between top of stack and that #on:do: .
  This primitive will trim stack back to that #on:do: .
  Generates an error if sent within a legacy handler's handlerBlock.

  Will return into a C extension exception handling C code, or
  to from the currently active handlerBlock, which ever is closer
  to the top of the stack. 
"
<primitive: 2020> "does not return"
self _uncontinuableError
%

method: AbstractException
_passToDefaultHandler

  "Caller has executed ensure: blocks between top of stack and
   point where receiver was originally signalled.
   This primitive will trim stack back to that signal frame and
   restart execution of AbstractException>>signal such that
   the search for a handler will always fail, thus executing
   the defaultAction."

<primitive: 2009>
self _uncontinuableError . "should never reach here"
%


method:
_pass: aHandler with: isNonStaticAnsi
  "Search for an enclosing handler below the active one that will
   handle the receiver.
   Caller must always pass  nil for aHandler and isNonStaticAnsi .
   resume:  from within the outer handler will return from this frame."
 <primitive: 697>   
 "primitive always fails.
  If handler found, it was substituted in frame for aHandler ,
  and isNonStaticAnsi set to true if an ansi handler "

  aHandler ifNotNil:[   "handler found"               
    isNonStaticAnsi ifNotNil: [
      self return:( self _executeOuterHandler: aHandler )
    ] ifNil:[
      self return:( self _executeOuterGsHandler: aHandler )
    ].
    self _uncontinuableError. "should not reach here"
  ] ifNil:[ "no enclosing handler found"
    self _executeEnsuresBelow: 0 . "ensures down to original signal"
    self _passToDefaultHandler .
    self _uncontinuableError.  "should not be here"
  ].
%

method:
_outer: aHandler with: isNonStaticAnsi
  "Search for an enclosing handler below the active one that will
   handle the receiver.
   Caller must always pass  nil for aHandler and isAnsi.
   resume:  from within the outer handler will return from this frame."

 <primitive: 697>   
 "primitive always fails.
  if handler found, it was substituted in frame for aHandler ,
  and isNonStaticAnsi set to true if an ansi handler "

  aHandler ifNotNil:[   "handler found"               
    isNonStaticAnsi ifNotNil: [
      self return:( self _executeOuterHandler: aHandler )
    ] ifNil:[
      self return:( self _executeOuterGsHandler: aHandler )
    ].
  ] ifNil:[ "no enclosing handler found"
    ^ self defaultAction .
  ].
  self _uncontinuableError. "should not reach here"
%


!-----------------------------------------------------------------
category: 'Handling'
method:
outer
  "When sent within an ANSI handler's handlerBlock ,
   search for an enclosing handler below the active one that will
   handle the receiver.  

   If an enclosing handler found,  send #value: to
   the enclosing handler's handlerBlock , with receiver as the argument.
   If the receiver is 'resumable' and the enclosing handler's handlerBlock
   sends #resume: to the receiver,  the resumption value 
   will be returned from this message.
   If the receiver is not 'resumable' or if the enclosing handlerBlock 
   does not send #resume: , then this message will not return.

   If no such handler found,  returns the result of
   sending #defaultAction to the receiver. 

   For exceptions not 'resumable', #outer is equivalent to #pass ."

   ^ self _outer: nil with: nil .
%

method:
pass
  "When sent within an ANSI handler's handlerBlock ,
   search for an enclosing handler below the active one that will
   handle the receiver.

   If an enclosing handler found, push a new stack frame and send #value: to
   the enclosing handler's handlerBlock , with receiver as the argument.

   If no such handler found, send #defaultAction to the receiver.
   Control does not return to the currently active handler."

   "Following resume: only happens if the _pass:with: invoked a default handler
    which returned normally. "
   self resume:( self _pass: nil with: nil ) .
   self _uncontinuableError. "should not reach here"
%

method:
isNested
 "When sent within an ANSI handler's handlerBlock ,
  returns true if there is an enclosing handler below the  on:do:
  of the active handler that could handle the receiver. 

  Returns false if sent within a legacy handler block.
  "

 <primitive: 700>
 self _uncontinuableError.
%

method:
resignalAs: replacementException

  "When sent within an ANSI handler's handlerBlock,
   the stack is trimmed back to the point where the receiver was
   orginally signaled, executing any ensure: blocks installed by
   handlerBlock execution .

   Then replacementException is substituted for the originally
   signaled exception and #signal  is sent to it to restart
   exception handling. 

   Generates an error if the stack trim would cross the frame 
   of a C primitive or user action."

  gsResumable ifFalse:[ replacementException _resumable: false ].
  self _executeEnsuresBelow: 0 .
  self _resignalAs: replacementException  . "does not return"
  self _uncontinuableError.  "should not be here"
%

method:
retryUsing: alternativeBlock

  "When sent within an ANSI handler's handlerBlock ,
   resume execution in the frame of the #on:do: send which 
   installed the currently active handlerBlock.

   Any ensure: blocks between top of stack and that #on:do: will 
   be executed and stack trimmed back. If alternativeBlock  is non-nil
   it is substituted for the original receiver of the #on:do: .  
   Then #value is sent to the non-nil alternativeBlock, else to the
   original receiver of #on:do: .

   Generates an error if sent within a legacy handler's handlerBlock,
   or if stack trim would cross the frame of a C primitive or user action."

  self _executeEnsuresBelow: 1 .
  self _retryUsing: alternativeBlock
%


method:
retry
  "When sent within an ANSI handler's handlerBlock ,
   resume execution by sending  #value  to the receiver of the
   #on:do: send which installed the currently active handlerBlock ,
   and using the stack frame of that #on:do: .

   Any ensure: blocks between top of stack and that #on:do: will 
   be executed and stack trimmed back before resuming execution.
 
  Generates an error if sent within a legacy handler's handlerBlock,
  or if stack trim would cross the frame of a C primitive or user action."

^self retryUsing: nil
%

!  public API enfoces:  resume: only usable within ANSI handler blocks.
!    internal implementation uses it for legacy blocks also.
!
method:
_resume: resumptionValue

"To be sent from within an ANSI handler's handlerBlock only.

 If the current handler block was invoked by #outer from a previous
 handler block,  #resume will return to that previous handler block,
 with resumptionValue being returned from the send of #outer .
 Otherwise resume execution from the send that signaled the receiver
 with the specified value.

 calling methods in image responsible for checking isResumable."

 <primitive: 2021>  
 self _uncontinuableError .  "should not be here"
%

method:
resume: resumptionValue

"To be sent from within an ANSI handler's handlerBlock only.

 If the current handler block was invoked by #outer from a previous
 handler block,  #resume will return to that previous handler block,
 with resumptionValue being returned from the send of #outer .
 Otherwise resume execution from the send that signaled the receiver
 with the specified value.
 
 An Error is signaled if this is send directly or indirectly from
 within a defaultAction method to the receiver of defaultAction."

 self isResumable ifFalse:[
   "cannot resume from a not-resumable Exception."
   ^ self error:'cannot resume from a not-resumable Exception' .
 ].
 ^ self _resume: resumptionValue
%

method:
resume

"See  resume:  for documentation ."

 self isResumable ifFalse:[ 
   "cannot resume from a not-resumable Exception."
   ^ self error:'cannot resume from a not-resumable Exception'
 ].
 ^ self _resume: nil
%

method:
return: returnValue

"When sent within an ANSI handler's handlerBlock,
 resume execution by returning returnValue from the #on:do: send
 which installed the currently active handlerBlock .

 Any ensure: blocks between top of stack and that #on:do:
 will be executed and the stack trimmed back.

 If return would cross a C extension , the _executeEnsuresBelow:
 will execute ensure blocks below the C frame, and
 return: will trim frames below the C frame.

 Generates an error if sent within a legacy handler's handlerBlock.
 or if the return would cross the frame of a C primitive or user action.
"
 self _executeEnsuresBelow: 1 .
 self _return: returnValue .
%


method:
return

"When sent within an ANSI handler's handlerBlock,
 resume execution by returning nil from the #on:do: send
 which installed the currently active handlerBlock .

 Any ensure: blocks between top of stack and that #on:do:
 will be executed and the stack trimmed back.

 Generates an error if sent within a legacy handler's handlerBlock,
 or if the return would cross the frame of a C primitive or user action.
"
 self _executeEnsuresBelow: 1 .
 self _return: nil .
%

method:
defaultAction

"Return an error to the controlling GCI client. Stack is saved
 and available as an argument to the GCI error struct.

 Instance variable of the receiver are enumerated into the GciErrSType.args 
 as follows.
 If there are named instVars after AbstractException.args
 the values of those instVars are stored into GciErrSType.args .
 AbstractException.args is then added if space is available.
  
 If AbstractException.args is the last named instVar and it is an Array
 of size <= GCI_MAX_ERR_ARGS , it is enumerated into GciErrSType.args.
 otherwise the oop of the Array is stored into GciErrSType.args[0] .

 See also ClientForwarderSend >> defaultAction "

<primitive: 33>  "this frame removed from stack before saving stack."
self _uncontinuableError
%

method:
_signalGciError

"Return an error to the controlling GCI client. Stack is saved
 and available as an argument to the GCI error struct."

<primitive: 33>  "this frame removed from stack before saving stack."
self _uncontinuableError
%
! --------------------------------------------------------------------------
category: 'Private Legacy'
method:
_legacyResignal: newException

"Search for a handler for newException below this handler.
 Current handler should be a legacy handler .

 If the newException is handled successfully,
 a normal return of the handlerBlock found by this resignal will
 result in a return from this method."

<primitive: 2024>  "succeeds or throws error to GCI"
self _uncontinuableError.
%

classmethod: 
_new: anInteger args: anArray 

"return a new exception for the specified error number."

<primitive: 140>  
  "If args ok and handler found, this stack frame reused to invoke handler.
   If exception handling succeeds and execution is to resume, 
   we return from this method with the resumption value."
anInteger _validateClass: SmallInteger.
anInteger < 1 ifTrue: [ 
  ^ self _error: #rtErrBadErr args: { anInteger . anArray  }
].
anArray _validateClass: Array.
anArray size > 10 ifTrue: [ 
 ^ self _error: #rtErrTooManyErrArgs args: { anArray size }
].
self _primitiveFailed: #_new:args: args: { anArray }.
self _uncontinuableError
%

! ------------------------------------------------
category: 'Legacy Handlers'

classmethod:
category: aCategory number: aNumber do: handlerBlock

"Install a legacy exception handler in the frame of the sender of 
   category:number:do:  .  
The handler is represented as an instance of GsExceptionHandler .

aCategory is ignored; it is assumed to be nil or 
the SymbolDictionary GemStoneError .
All exceptions except those with instVar gsTrappable==false  are trapped,
subject to the value of aNumber .

handlerBlock must be a 4-argument instance of ExecBloc.

aNumber is an error number of an exception to trap 
or nil (to trap all exceptions) .

The subtype of the handler is always nil."

<primitive: 376>
" aCategory ~~ nil ifTrue:[ aCategory _validateClass: SymbolDictionary ] . "
aNumber ~~ nil ifTrue:[ aNumber _validateClass: SmallInteger ] .
handlerBlock _validateClass: ExecBlock .
handlerBlock argumentCount == 4 ifFalse:[
  handlerBlock _error: #rtErrBadBlockArgCount args: { 4 . handlerBlock argumentCount } .
  ] .
^ self _primitiveFailed: #category:number:do: 
       args: { aCategory . aNumber . handlerBlock }
%

classmethod:
block: aBlock category: aCategory number: num subtype: atype

"This form of creating a Legacy handler is no longer supported.
 You must use category:number:do:  or  installStaticException:category:number: ."

self shouldNotImplement: #block:category:number:subtype: .
self _uncontinuableError
%

classmethod:
removeActivationException: aGsExceptionHandler

"Search the current GemStone Smalltalk call stack for a method or block context
 that specified handler installed, and remove it.  The stack is searched by
 starting with the top method or block context and moving down.  Generates an
 error if aGsExceptionHandler is not installed anywhere in the current GemStone
 Smalltalk call stack or if aGsExceptionHandler is not an instance of
 GsExceptionHandler .

 Does not search the static legacy handlers list."

^ GsExceptionHandler removeActivationException: aGsExceptionHandler

%

method:
remove

"To be sent only within a legacy handler's handlerBlock. 

 Remove the receiver's  GsExceptionHandler from
 the GemStone Smalltalk call stack . 

 Does not search the static legacy handlers list.
"
  currGsHandler ifNotNil:[ :h | h remove ] 
                ifNil:[ self _uncontinuableError ]
%

method:
resignal: anErrorDict number: anInteger args: anArray

  "When sent within a legacy handler's handlerBlock,
   create a new instance of Exception as specified by the arguments,
   and search for the next handler below this handler.

   If the newly created Exception is handled successfully,
   a normal return of that handler's handlerBlock will
   result in a return from this method. 

   anErrorDict is ignored, and may be nil ."

  | ex |
  currGsHandler ifNil:[
     self _uncontinuableError. "illegal in ANSI handler"
  ].
  ex := AbstractException _new: anInteger args: anArray .
  gsResumable ifFalse:[ ex _resumable: false ].
  ^ self _legacyResignal: ex 
%
!----------------------------------------------------------------
category: 'Legacy Accessing'
method:
next

  "No longer supported.
   Within a legacyHandler block, to get the next legacyHandler after 
   the one currently active  use
       anException legacyHandler next
   where anException is the first arg to the legacyHandler block. "
 
  self shouldNotImplement: #next .
  self _uncontinuableError
%

method
subtype

   "No longer supported."

   ^ nil
%

! --------------------------------------------------------------------------
category: 'Static legacy handlers'
classmethod:
installStaticException: handlerBlock category: category number: num

"Install the 4-argument handlerBlock as a static GsExceptionHandler
 to field errors of the specified category and number. 
 Returns a new instance of GsExceptionHandler "

^ self _installStaticException: handlerBlock class: nil
       category: category number: num subtype: nil
%

classmethod:
installStaticException: handlerBlock category: aCategory number: aNumber subtype: atype

"Install the 4-argument handlerBlock at the head of the static handlers 
 list, not associated with a particular stack frame ,
 to field errors of the specified category, number, and subtype.

 The handler is represented as an instance of GsExceptionHandler,
 and will be a part of the session state.  
 The new  GsExceptionHandler is returned ."

^ self _installStaticException: handlerBlock class: nil 
      category: aCategory number: aNumber subtype: atype
%


! --------------------------------------------------------------------------
category: 'SUnit'

classmethod: 
sunitSignalWith: aString
        ^self signal: aString
%
method: 
sunitExitWith: aValue
        ^self return: aValue
%
method: 
sunitSignalWith: aString
        ^self signal: aString
%

category: 'Copying'
method:
shallowCopy
  "clears any exception handling information in the copy. Do this
   in shallowCopy  in case a subclass fails to send postCopy.  "
  | res |
  ( res := super shallowCopy ) size: 0 .
  ^ res
%
