!========================================================================
! Copyright (C) GemTalk Systems 2016-2020.  All Rights Reserved.

! $Id: Btree-PathTerm-Core.gs 38384 2016-01-08 18:22:36Z lalmarod $
!
! Btree-PathTerm-Core.gs  -  source code for the gs path term classes
!
!========================================================================

! class created in btreeplusclasses.gs

! Class Implementation for BtreePlusTraversalMapDictionary

! Remove existing behavior from BtreePlusTraversalMapDictionary
removeallmethods BtreePlusTraversalMapDictionary
removeallclassmethods BtreePlusTraversalMapDictionary

! ------------------- Instance methods for BtreePlusTraversalMapDictionary

category: 'Accessing'
method: BtreePlusTraversalMapDictionary
at: aKey

aKey ifNil: [ ^ super at: self].
^ super at: aKey
%
category: 'Accessing'
method: BtreePlusTraversalMapDictionary
at: aKey ifAbsent: aBlock

aKey ifNil: [ ^ super at: self ifAbsent: aBlock ].
^ super at: aKey ifAbsent: aBlock
%
category: 'Accessing'
method: BtreePlusTraversalMapDictionary
at: aKey ifAbsentPut: aBlock

aKey ifNil: [ ^ super at: self ifAbsentPut: aBlock ].
^ super at: aKey ifAbsentPut: aBlock
%
category: 'Accessing'
method: BtreePlusTraversalMapDictionary
at: aKey otherwise: defaultValue

aKey ifNil: [ ^ super at: self otherwise: defaultValue ].
^ super at: aKey otherwise: defaultValue
%
category: 'Updating'
method: BtreePlusTraversalMapDictionary
at: aKey put: aValue

aKey ifNil: [ ^ super at: self put: aValue ].
^ super at: aKey put: aValue
%

! Class Implementation for GsPathTerm

! Remove existing behavior from GsPathTerm
removeallmethods GsPathTerm
removeallclassmethods GsPathTerm

! ------------------- Class methods for GsPathTerm

category: 'Instance Creation'
classmethod: GsPathTerm
new

"Create an initialized instance of the receiver."

^ self basicNew initialize
%

! ------------------- Instance methods for GsPathTerm

category: 'Updating'
method: GsPathTerm
_addChildTerm: aPathTerm
  children addLast: aPathTerm
%

category: 'Testing'
method: GsPathTerm
_checkBtreeComparisonWith: anObject

"Returns whether the receiver's B-tree can hold an entry with anObject as the
 key."

^ updateBtree _canCompareWith: anObject
%

category: 'Accessing'
method: GsPathTerm
_classOf: anObject
  "Reflection classOf:"

  <primitive: 166>
  self _primitiveFailed: #'_classOf:' args: {anObject}.
  self _uncontinuableError
%

category: 'Updating'
method: GsPathTerm
_determineChildren
  "Determine the children of the receiver by getting the next path term for each
 index object that utilizes the receiver path term."

  | indexObj |
  children size: 0.
  1 to: self size do: [ :i | 
    indexObj := self at: i.
    (indexObj size > offset
      and: [ (children includesIdentical: (indexObj at: offset + 1)) not ])
      ifTrue: [ self _addChildTerm: (indexObj at: offset + 1) ] ]
%

category: 'Testing'
method: GsPathTerm
_doNotPerformPretraversalFor: anObject
  "Returns whether to do a preparatory traversal (using MappingInfo objects)
 when anObject is being added to an indexed Bag or IdentityBag."

  ^ true
%

category: 'Traversing'
method: GsPathTerm
_getNextObjectForTraversal: anObject
  " get the next object along the path "

  | ivOffset nextObj |
  ivOffset := anObject _findIVOffsetForPathTerm: self.
  ivOffset == nil
    ifTrue: [ 
      anObject _errorPathTermNotInDependencyList: self.
      nextObj := nil ]
    ifFalse: [ nextObj := anObject instVarAt: ivOffset ].
  ^ nextObj
%

category: 'Private'
method: GsPathTerm
_invalidIvOffset: anObject
  "If missing path slots are tolerated, return. Otherwise signal an error"

  self termsRequired
    ifFalse: [ ^ self ].
  anObject _errorInvalidOffset: name.
  self _uncontinuableError
%

category: 'Testing'
method: GsPathTerm
_isObsoletePathTerm

"Returns whether the receiver is obsolete."

^ self size == 0
%

category: 'Accessing'
method: GsPathTerm
_ivOffsetFor: anObject

"Returns the instance variable offset of anObject for the instance variable
 corresponding to the receiver path term."

^ (self _classOf: anObject) _ivOffsetOf: name
%

category: 'Accessing'
method: GsPathTerm
_nextObjectFor: anObject

"Returns the object at the instance variable that corresponds to the receiver
 path term."


^ self _nextObjectFor: anObject atInstVar: (self _ivOffsetFor: anObject)
%

category: 'Accessing'
method: GsPathTerm
_nextObjectFor: anObject atInstVar: ivOffset

"Returns the object at the instance variable that corresponds to the receiver
 path term."

<primitive: 167>
self _primitiveFailed: #_nextObjectFor:atInstVar: 
     args: { anObject . ivOffset } .
self _uncontinuableError
%

category: 'Accessing'
method: GsPathTerm
_thisAndAllChildTermsInto: array

"Returns an Array containing the receiver and all child path terms."


array add: self.

1 to: children size do: [ :i |
  (children at: i) _thisAndAllChildTermsInto: array
].

^ array
%

category: 'Testing'
method: GsPathTerm
_totalNumSetValuedTerms
  ""

  | total |
  self indicatesNsc
    ifTrue: [ total := 1 ]
    ifFalse: [ total := 0 ].
  1 to: children size do: [ :i | total := total + (children at: i) _totalNumSetValuedTerms ].
  ^ total
%

category: 'Adding'
method: GsPathTerm
addMappingsForObject: anObject at: ivOffset root: rootObject logging: aBoolean
  ^self addMappingsForObject: anObject root: rootObject logging: aBoolean
%

category: 'Adding'
method: GsPathTerm
addMappingsForObject: anObject logging: aBoolean
  self addMappingsForObject: anObject root: anObject logging: aBoolean
%

category: 'Adding'
method: GsPathTerm
addMappingsForObject: anObject root: rootObject logging: aBoolean
  "Add dependency list entries for anObject."

  | ivOffset nextObj |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (ivOffset := self _ivOffsetFor: anObject) == nil
    ifTrue: [ ^ self _errorInvalidOffset: anObject ].
  anObject
    getDepListAndAddPathTerm: self
    withIVOffset: ivOffset
    logging: aBoolean.	" add an entry to the objects dependency list "
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.	" get the next object along the path "
  updateBtree ~~ nil
    ifTrue: [ " insert an entry into the B-tree "
      (self _checkBtreeComparisonWith: nextObj)
        ifFalse: [ 
          ImproperOperation new
            _number:
                (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
            args: { nextObj class. self name. updateBtree lastElementClassDescription };
            signal ].
      updateBtree btreeAt: nextObj put: anObject root: rootObject.
      needsDepList
        ifTrue: [ " add an entry to the very last object's dependency list "
          nextObj getDepListAndAddLastElementPathTerm: self logging: aBoolean ] ].
  nil == nextObj
    ifTrue: [ ^ self recordNilOnPathForRoot: rootObject].
  1 to: children size do: [ :i | " make recursive call to add mappings "
    (children at: i)
      addMappingsForObject: nextObj
      root: rootObject
      logging: aBoolean ]
%

category: 'Adding'
method: GsPathTerm
addMappingsForObject: rootObject traverseUpTo: aPathTerm for: anIndexObject logging: aBoolean
  | object |
  object := anIndexObject traverse: rootObject upTo: aPathTerm.
  self addMappingsForObject: object root: rootObject logging: aBoolean
%

category: 'Audit'
method: GsPathTerm
auditDepListFor: obj index: indexObj using: auditor optionalSentinel: optionalSentinel
  "Private."

  | depList j nextObj ivOffset |
  (nil == obj or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  depList := DependencyList for: obj.
  depList == nil
    ifTrue: [ 
      (obj isInvariant not and: [ self isOptionalTerm not ])
        ifTrue: [ auditor pathTermObjectHasNoDependencyList: self object: obj indexObj: indexObj depList: depList ].
      nextObj := self _nextObjectFor: obj ]
    ifFalse: [ 
      (DepListTable _hasDependencyBitFor: obj)
        ifFalse: [ auditor pathTermObjectHasNoDependencyBitSet: self object: obj indexObj: indexObj depList: depList ].
      j := depList _findOffsetForPathTerm: self.
      j == nil
        ifTrue: [ 
          | ivOff |
          "no entry for self in depList"
          ivOff := self _ivOffsetFor: obj.
          (self isOptionalTerm and: [ ivOff isNil ])
            ifTrue: [ 
              "no dependencyList if there's no slot"
              nextObj := optionalSentinel ]
            ifFalse: [ auditor dependencyListHasNoEntryForPathTerm: self object: obj indexObj: indexObj depList: depList ] ]
        ifFalse: [ 
          "found entry for self in depList"
          ivOffset := depList at: j + 1.	"may be negative for last element reference count"
          ivOffset < 0
            ifTrue: [ 
              "reference count for last element, but we don't expect reference 
               count here ... see needsDepList clause below where depList 
               existence is sufficient"
              auditor dependencyListUnexpectedLastElementLocation: self object: obj indexObj: indexObj depList: depList ivOffset: ivOffset.
              ^ obj ]
            ifFalse: [ | expectedIvOffset |
              ivOffset ~~ (expectedIvOffset := self _ivOffsetFor: obj)
                ifTrue: [ 
                  auditor dependencyListHasIncorrectIvOffset: self object: obj indexObj: indexObj depList: depList ivOffset: ivOffset expectedIvOffset: expectedIvOffset.
                  ivOffset := expectedIvOffset "allow audit to continue, despite fatal nature of this situation" ].
              nextObj := obj instVarAt: ivOffset ] ] ].
  ((needsDepList and: [ nextObj ~~ nil ])
    and: [ DependencyList needsDepList: nextObj ])
    ifTrue: [ 
      depList := DependencyList for: nextObj.
      (depList == nil and: [ self isOptionalTerm not ])
        ifTrue: [ auditor pathTermObjectHasNoDependencyList: self object: obj indexObj: indexObj depList: depList ] ].
  ^ nextObj
%

category: 'Audit'
method: GsPathTerm
auditDepListForLastElementObject: obj occurrences: num index: indexObj using: auditor
  | depList j |
  (self size == 0 or: [ obj isInvariant ])
    ifTrue: [ ^ self ].
  depList := DependencyList for: obj.
  depList == nil
    ifTrue: [ auditor
        pathTermObjectHasNoDependencyList: self
        object: obj
        indexObj: indexObj
        depList: depList ]
    ifFalse: [ (DepListTable _hasDependencyBitFor: obj)
        ifFalse: [ auditor
            pathTermObjectHasNoDependencyBitSet: self
            object: obj
            indexObj: indexObj
            depList: depList ].
      j := depList _findOffsetForPathTerm: self.
      j == nil
        ifTrue: [ "hack to avoid false positives for index paths ending in *"
          self name = #'*'
            ifFalse: [ auditor
                dependencyListHasNoEntryForPathTerm: self
                object: obj
                indexObj: indexObj
                depList: depList ] ]
        ifFalse: [ | referenceCount nonReferenceCountTerm |
          referenceCount := depList at: j + 1.
          nonReferenceCountTerm := false.
          referenceCount > 0
            ifTrue: [ "upgraded indexes from pre-3.2 may have positive rerence count, but only for Byte indexable ojbects"
              obj class isBytes
                ifFalse: [ "ivOffset and not a reference count"
                  nonReferenceCountTerm := true ] ]
            ifFalse: [ referenceCount := referenceCount negated ].
          nonReferenceCountTerm
            ifTrue: [ "not a reference count and will be audited by PathTerm>>auditDepListFor:index:on:optionalSentinel: "
              "noop"
               ]
            ifFalse: [ referenceCount == num
                ifFalse: [ auditor
                    dependencyListHasIncorrectRefCount: self
                    object: obj
                    indexObj: indexObj
                    depList: depList
                    referenceCount: referenceCount
                    numOccurrences: num ] ] ] ]
%

category: 'Audit'
method: GsPathTerm
auditDirectNscCountsFor: obj using: auditor count: btreeCounts
  "Private. "

  | index |
  self size == 0
    ifTrue: [ ^ self ].
  (needsDepList and: [ nil ~~ obj ])
    ifTrue: [ | depList |
      depList := DependencyList for: obj.
      (depList == nil and: [ obj isInvariant not ])
        ifTrue: [ auditor pathTermObjectHasNoDependencyList: self object: obj indexObj: nil depList: depList ]
        ifFalse: [ obj isInvariant not
            ifTrue: [ (DepListTable _hasDependencyBitFor: obj)
                ifFalse: [ auditor pathTermObjectHasNoDependencyBitSet: self object: obj indexObj: nil depList: depList ] ] ] ].
  index := 1.
  1 to: self size do: [ :i | | count indexObj |
    indexObj := self at: i.
    indexObj size == offset
      ifTrue: [ count := btreeCounts at: index.
        count == 0
          ifTrue: [ auditor pathTermIncorrectNumberOfBtreeEntries: self index: i offset: offset ].
        btreeCounts at: index put: count - 1.
        index := index + 1 ] ]
%

category: 'Audit'
method: GsPathTerm
auditInfo: array using: auditor
  auditor auditInfoFor: self info: array
%

category: 'Audit'
method: GsPathTerm
auditNsc: nsc on: aString level: level
  "Verifies that the equality index objects are consistent.
 Returns a string that describes any inconsistencies found."

  | indexObj btreeCounts count |
  "aString add: 'entering obj, num is ', num printString; lf."
  self size == 0
    ifTrue: [ ^ aString ].	" for each index that utilizes the path term "
  level == 1 ifTrue: [ self auditUpdateCache: nsc on: aString ].
  btreeCounts := {}.
  1 to: self size do: [ :i | indexObj := self at: i.
    indexObj size == offset
      ifTrue: [ " verify B-tree has correct entries "
        count := indexObj btreeRoot
          auditNsc: nsc
          for: self
          offset: i
          on: aString.
        btreeCounts add: count ] ].
  1 to: children size do: [ :i | | resultArray |
    resultArray := (children at: i) auditNsc: nsc on: aString level: level + 1.
    btreeCounts add: resultArray ].
  ^ btreeCounts
%

category: 'Audit'
method: GsPathTerm
auditNscCountsFor: obj on: aString count: btreeCounts
  "Private. "

  | auditor |
  auditor := BtreePlusNodeAuditor new auditResultString: aString; yourself.
  true
    ifTrue:[ self auditNscCountsFor: obj using: auditor count: btreeCounts ]
    ifFalse: [ 
      "use this audit to provide more detailed information about btree element count audit failures"
      "not fully tested, but it is know to provide better information audit errors with the 
       BtreePlusNodeAuditor>>pathTermIncorrectNumberOfBtreeEntries:index:offset: message."

      "bypass call to _auditBtreeCounts:on: in UnorderedCollection>>_fastAuditEqualityIndexes as well"
      self auditNscForRootObj: obj rootObj: obj using: auditor ].
  ^ auditor auditResultString
%

category: 'Audit'
method: GsPathTerm
auditNscCountsFor: obj using: auditor count: btreeCounts
  "Private. "

  | nextObj index optionalSentinel |
  self size == 0
    ifTrue: [ ^ self ].
  self indicatesIndexOnNscElements
    ifTrue: [ ^ self auditDirectNscCountsFor: obj using: auditor count: btreeCounts ].
  nil == obj
    ifTrue: [ ^ self ].
  optionalSentinel := Object new.
  nextObj := self
    auditDepListFor: obj
    index: nil
    using: auditor
    optionalSentinel: optionalSentinel.
  nextObj == optionalSentinel
    ifTrue: [ ^ self ].
  index := 1.
  1 to: self size do: [ :i | | count indexObj |
    indexObj := self at: i.
    indexObj size == offset
      ifTrue: [ count := btreeCounts at: index.
        (count == 0 and: [ self isOptionalTerm not ])
          ifTrue: [ "an index with optional path terms may have 0 entries"
             auditor pathTermIncorrectNumberOfBtreeEntries: self index: i offset: offset ].
        (count == 0 and: [ self isOptionalTerm ])
          ifFalse: [ btreeCounts at: index put: count - 1 ].
        index := index + 1 ] ].
  1 to: children size do: [ :i | (children at: i)
      auditNscCountsFor: nextObj
      using: auditor
      count: (btreeCounts at: index).
    index := index + 1 ]
%

category: 'Accessing'
method: GsPathTerm
children

"Returns the value of the instance variable 'children'."

^children
%

category: 'Updating'
method: GsPathTerm
children: newValue

"Modify the value of the instance variable 'children'."

children := newValue
%

category: 'Updating Indexes'
method: GsPathTerm
cleanupDependencyListFor: anObject
  ""

  | ivOffset nextObj depList |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (depList := DependencyList for: anObject) == nil
    ifTrue: [ ^ self ].
  ivOffset := depList removeCompletelyPathTerm: self for: anObject.	" remove the receiver from the dependency list "
  (self indicatesIndexOnRootNsc
    or: [ 
      "last element dependency list"
      ivOffset < 0 ])
    ifTrue: [ ^ self ].
  ivOffset == 0
    ifTrue: [ nextObj := self _nextObjectFor: anObject ]
    ifFalse: [ 
      [ nextObj := self _nextObjectFor: anObject atInstVar: ivOffset ]
        onException: Error
        do: [ :ex | 
          "Handle any error thrown during execution of _nextObjectFor:
     Cannot resolve the path for this guy --- can't have been added to index,
     in fact this object could have caused the failure in the first place"
          ^ self ] ].
  nil == nextObj
    ifTrue: [ ^ self ].
  needsDepList
    ifTrue: [ 
      (depList := DependencyList for: nextObj) ~~ nil
        ifTrue: [ depList removeCompletelyPathTerm: self for: nextObj ].
      ^ self ].
  1 to: children size do: [ :i | (children at: i) cleanupDependencyListFor: nextObj ]
%

category: 'Testing'
method: GsPathTerm
coversIvOffset: anIvOffset for: anObject
  "Answer true if the receiver covers the given iv offset (see bug 46705)" 

  ^false
%

category: 'Collection Based Modification'
method: GsPathTerm
findReachableRootsFor: anObject 
  | rootObjects indexObj reachableRoots |
  rootObjects := self findRootObjectsFor: anObject.
  indexObj := self at: self size.
  "select rootObjects for which anNsc is reachable"
  reachableRoots := IdentityBag new.
  rootObjects asIdentitySet do: [:root | 
    (indexObj traverseAllWithParents: root upTo: self startingAt: 1)
      do: [ :ar | | intermediateObjectReachableFromRoot |
         intermediateObjectReachableFromRoot := ar at: 2.
        intermediateObjectReachableFromRoot == anObject
          ifTrue: [ reachableRoots add: root withOccurrences: (rootObjects occurrencesOf: root) ] ] ].
  ^ reachableRoots
%

category: 'Collection Based Modification'
method: GsPathTerm
findRootObjectsFor: anObject
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject.
  ivOffset ifNil: [ self _invalidIvOffset: anObject ].
  ^ self findRootObjectsFor: anObject ivOffset: ivOffset
%

category: 'Collection Based Modification'
method: GsPathTerm
findRootObjectsFor: anObject ivOffset: ivOffset
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject at is a key or parent of a key."

  | rootObjects nextObj roots |
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
  rootObjects := updateBtree ~~ nil
    ifTrue: [ 
      "btree exists for receiver, collect the root objects, where anObject is the key and nextObj
       is the value."
      self findRootObjectsForKey: nextObj value: anObject ]
    ifFalse: [ IdentityBag new ].
  (nil == nextObj or: [ self children isEmpty ])
    ifTrue: [ ^ rootObjects ].
  1 to: children size do: [:i | 
    roots := (children at: i) findRootObjectsFor: nextObj.
    rootObjects := rootObjects _union: roots  ].
  ^ rootObjects
%

category: 'Modification'
method: GsPathTerm
findRootObjectsForKey: aKey value: aValue
  ^ updateBtree findRootObjectsForKey: aKey value: aValue
%

category: 'Accessing'
method: GsPathTerm
getIndexDictionary

"Returns the index dictionary associated with the receiver."

"no idexDictionary associated with the receiver"

^ nil
%

category: 'Accessing'
method: GsPathTerm
getParentTerm

"Returns the parent path term of the receiver.  This is determined by looking
 at the previous path term of the first index object that utilizes this path
 term.  If there is no parent, returns nil."

offset > 1
    ifTrue: [ ^ (self at: 1) at: offset - 1 ]
    ifFalse: [ ^ nil ]
%

category: 'Testing'
method: GsPathTerm
hasIndexDictionary
  "Answer if receiver's index may have an indexDictionary."

  ^ false
%

category: 'Testing'
method: GsPathTerm
hasMatchingTerm: aSymbol
  "Answer true if the receiver's term matches <aSymbol>"

  ^ self name == aSymbol
%

category: 'Testing'
method: GsPathTerm
indicatesIndexOnNscElements

"Returns true if the path term indicates that the index is on elements of the
 NSC itself.  This is the case if the path is '' (two single quotes).  This
 method is overridden by SetValuedPathTerm to return true."

^ name ==  #'' 
%

category: 'Testing'
method: GsPathTerm
indicatesIndexOnRootNsc
  "Returns true if the path term indicates that the index is on elements of the
 NSC itself.  This is the case if the path is '' (two single quotes)."

  ^ name == #''
%

category: 'Testing'
method: GsPathTerm
indicatesMultiValue
  "Returns false, the receiver does not indicate that the index is on an
 instance variable that may have multiple values (i.e., collection based path term)"

  ^ false
%

category: 'Testing'
method: GsPathTerm
indicatesNsc
  "Returns false, the receiver does not indicate that the index is on an
 instance variable that is expected to be a collection (i.e., set valued pathterm."

  ^ false
%

category: 'Initialization'
method: GsPathTerm
initialize

"Initialize the receiver with default values."

securityPolicies := GsObjectSecurityPolicySet new.
children := { } .
needsDepList := false.
nilOnPath := RcLowMaintenanceIdentityBag new
%

category: 'Testing'
method: GsPathTerm
isEnumeratedTerm
  "Returns true if the receiver indicates an enumerated path term."

  ^ false
%

category: 'Testing'
method: GsPathTerm
isRangeEqualityIndexLastPathTerm

"Returns true if the receiver is the last path term of a RangeEqualityIndex."

1 to: self size do: [ :i |
    ( (self at: i) isRangeEqualityIndex and: [ (self at: i) size == offset ] )
        ifTrue: [ ^ true ]
].
^ false
%

category: 'Testing'
method: GsPathTerm
isSelectorTerm
  ^ false
%

category: 'Testing'
method: GsPathTerm
isSetValuedTerm
  "Returns true if the receiver indicates a set-valued instance variable."

  ^ false
%

category: 'Accessing'
method: GsPathTerm
name

"Returns the value of the instance variable 'name'."

^name
%

category: 'Updating'
method: GsPathTerm
name: newValue

"Modify the value of the instance variable 'name'."

name := newValue
%

category: 'Accessing'
method: GsPathTerm
needsDepList

"Returns the value of the instance variable 'needsDepList'."

^needsDepList
%

category: 'Updating'
method: GsPathTerm
needsDepList: newValue

"Modify the value of the instance variable 'needsDepList'."

needsDepList := newValue
%

category: 'Accessing'
method: GsPathTerm
objectSecurityPolicy: anObjectSecurityPolicy

"Assign the receiver and its children to the given security policy."

super objectSecurityPolicy: anObjectSecurityPolicy.
children objectSecurityPolicy: anObjectSecurityPolicy.
1 to: children size do: [ :i |
    (children at: i) objectSecurityPolicy: anObjectSecurityPolicy
].
securityPolicies objectSecurityPolicy: anObjectSecurityPolicy.
%

category: 'Accessing'
method: GsPathTerm
offset

"Returns the value of the instance variable 'offset'."

^offset
%

category: 'Updating'
method: GsPathTerm
offset: newValue

"Modify the value of the instance variable 'offset'."

offset := newValue
%

category: 'Removing'
method: GsPathTerm
removeDirectMappingFor: anObject logging: doLogging
  "There is a range index directly on the elements of the NSC.  Update the
 B-tree."

  updateBtree ~~ nil
    ifTrue: [ " remove it from the B-tree "
      updateBtree btreeRemoveKey: anObject value: anObject root: anObject ]
%

category: 'Removing'
method: GsPathTerm
removeMappingsFor: anObject at: ivOffset root: rootObject lastOne: lastOne logging: doLogging

  ^self removeMappingsFor: anObject root: rootObject lastOne: lastOne logging: doLogging
%

category: 'Removing'
method: GsPathTerm
removeMappingsFor: anObject lastOne: lastOne logging: doLogging
  ^self removeMappingsFor: anObject root: anObject lastOne: lastOne logging: doLogging
%

category: 'Removing'
method: GsPathTerm
removeMappingsFor: anObject root: rootObject lastOne: aBoolean logging: doLogging
  "Remove entries in the btree and dependency lists for anObject."

  | nextObj depList more isLast |
  self size == 0
    ifTrue: [ ^ self ].
  nextObj := self _nextObjectFor: anObject.
  updateBtree ~~ nil
    ifTrue: [ 
      " remove an entry from the B-tree "
      (updateBtree btreeRemoveKey: nextObj value: anObject root: rootObject)
        ifFalse: [ self _errorObjectNotInBtree: nextObj value: anObject root: rootObject ].
      DependencyList removePathTerm: self for: nextObj logging: doLogging	" remove dependency list entry for next object " ].
  nextObj ifNil: [ self removeNilOnPathForRoot: rootObject ].
  more := false.
  (nil ~~ nextObj and: [ children isEmpty not ])
    ifTrue: [ 
      aBoolean
        ifTrue: [
          | last |
          last := (children at: 1) _isLastOccurrenceInIndexFor: nextObj.
          isLast := aBoolean and: [ last ] ]
        ifFalse: [ isLast := false ].
          " make recursive call to remove mappings "
      1 to: children size do: [ :i | 
        (children at: i) removeMappingsFor: nextObj root: rootObject lastOne: isLast logging: doLogging ].
      isLast
        ifFalse: [
          "check again in case there are no more references left in btree"
          isLast := ((children at: 1) findReachableRootsFor: nextObj) isEmpty ].
      isLast
        ifTrue: [ 
          " remove dependency list entries for next object "
          (depList := DependencyList for: nextObj) ~~ nil
            ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ] ] ]
%

category: 'Accessing'
method: GsPathTerm
requirePathTerms: aBool
  "compat with PathTerm implementation"

  ^ self termsRequired: aBool
%

category: 'Accessing'
method: GsPathTerm
termsRequired
  ^ termsRequired ifNil: [ termsRequired := true ]
%

category: 'Accessing'
method: GsPathTerm
termsRequired: aBool
  termsRequired := aBool
%

category: 'Modification'
method: GsPathTerm
update: anObject at: ivOffset to: newValue
  "The instance variable for anObject is being changed to newValue.
 Update the btrees and dependency lists for the old value and the
 newValue.  Returns true if the index objects were modified correctly;
 otherwise returns an Array containing error information."

  | oldValue reachableRoots |
  self size == 0
    ifTrue: [ ^ true ].
  oldValue := anObject instVarAt: ivOffset.
  reachableRoots := self findReachableRootObjectMapsFor: anObject ivOffset: ivOffset oldValue: oldValue.
  updateBtree ifNotNil: [
    self
      updateBtreeDirectFor: anObject
      at: ivOffset
      oldValue: oldValue
      to: newValue
      roots: reachableRoots ].
  self
    updateBtreeFor: anObject
    at: ivOffset
    oldValue: oldValue
    to: newValue
    roots: reachableRoots.
  ^ true
%

category: 'Updating'
method: GsPathTerm
updateBtree
  "Answer the value of the instance variable 'updateBtree'."

  ^ updateBtree
%

category: 'Updating'
method: GsPathTerm
updateBtree: aGsAbstractIndex
  "Modify the value of the instance variable 'updateBtree'."

  updateBtree := aGsAbstractIndex
%

category: 'Modification'
method: GsPathTerm
updateBtreeDirectFor: anObject at: ivOffset oldValue: oldValue to: newValue roots: reachableRoots
  "The instance variable for anObject is being changed from olValue to newValue.
 Update the btrees associated with the receiver for the old value and the newValue."

  "BtreePlusEqualityTests>>#testEnumeratedIndexUpdateLeaf fails wrong result set"
  "BtreePlusEqualityTests>>#testMultiEnumeratedIndexUpdateB passes (modulo deplist count)"

  (self _checkBtreeComparisonWith: newValue)
    ifFalse: [ 
      ImproperOperation new
        args: {newValue class. self name. updateBtree lastElementClassDescription};
        _number:
            (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
        signal].
  reachableRoots do: [ :root |
    (updateBtree
      btreeRemoveKey: oldValue
      value: anObject
      root: root)
        ifFalse: [ self _errorObjectNotInBtree: oldValue value: anObject root: root ] ].
  (needsDepList and: [ nil ~~ oldValue ])
    ifTrue: [ 
      reachableRoots size
        timesRepeat: [ DependencyList removePathTerm: self for: oldValue logging: false ] ].
  reachableRoots do: [ :root | 
    " insert an entry into the B-tree "
    updateBtree btreeAt: newValue put: anObject root: root.
    needsDepList
      ifTrue: [ 
        " add an entry in the dependency list "
        newValue getDepListAndAddLastElementPathTerm: self logging: false ] ].
%

category: 'Modification'
method: GsPathTerm
updateBtreeFor: anObject at: ivOffset oldValue: oldValue to: newValue roots: reachableRoots
  "The instance variable for anObject is being changed from olValue to newValue.
 There are no btrees associated directly with the receiver. Traverse the receiver's 
 children and update the btrees and dependency lists for the old value and the newValue."

  | doLogging |
  doLogging := UnorderedCollection _isRcIndexLoggingEnabled.
  reachableRoots do: [ :root |
    self updateEntriesFor: oldValue
      to: newValue
      root: root
      lastOne: true
      logging: doLogging ].
  (oldValue ~~ nil and: [ children size > 0 ])
    ifTrue: [
      | childRoots |
      childRoots := (children at: 1) findReachableRootObjectMapsFor: oldValue.
      (childRoots - reachableRoots) isEmpty
        ifTrue: [ 
          | depList |
          " remove dependency list entries for next object "
          (depList := DependencyList for: oldValue) ~~ nil
            ifTrue: [ depList removePathTerms: children for: oldValue logging: doLogging ] ] ] 
%

category: 'Modification'
method: GsPathTerm
updateEntriesFor: oldValue to: newValue root: rootObject lastOne: aBoolean logging: doLogging
  "Remove entries in the index dictionary and dependency lists for anObject."

  self size == 0
    ifTrue: [ ^ self ].
  oldValue == nil
    ifTrue: [ self removeNilOnPathForRoot: rootObject ]
    ifFalse: [ 
      1 to: children size do: [ :i | 
        (children at: i)
          removeMappingsFor: oldValue
          root: rootObject
          lastOne: aBoolean
          logging: doLogging ] ].	
  newValue == nil
    ifTrue: [ self recordNilOnPathForRoot: rootObject ]
    ifFalse: [
      1 to: children size do: [ :i | 
        (children at: i) 
          addMappingsForObject: newValue root: rootObject logging: doLogging ] ] 
%

! Class Implementation for GsCollectionBasedPathTerm

! Remove existing behavior from GsCollectionBasedPathTerm
removeallmethods GsCollectionBasedPathTerm
removeallclassmethods GsCollectionBasedPathTerm

! ------------------- Instance methods for GsCollectionBasedPathTerm

category: 'Accessing'
method: GsCollectionBasedPathTerm
_nextObjectFor: anObject
  "Cannot use this shortcut method"

  self shouldNotImplement: #'_nextObjectFor:'
%

category: 'Testing'
method: GsCollectionBasedPathTerm
_totalNumSetValuedTerms
  "for use of AddMappingsForObject:root:logging: in UnorderedCollection>>_updateIndexesForAdditionOf:logging:"

  ^ 3
%

category: 'Testing'
method: GsCollectionBasedPathTerm
indicatesMultiValue
  "Returns true, the receiver indicates that the index is on an
 instance variable that may have multiple values (i.e., collection based path term)"

  ^ true
%

category: 'Accessing'
method: GsCollectionBasedPathTerm
nextObj: anObj do: aBlock
  self subclassResponsibility: #'nextObj:do:'
%

! Class Implementation for GsEnumeratedPathTerm

! Remove existing behavior from GsEnumeratedPathTerm
removeallmethods GsEnumeratedPathTerm
removeallclassmethods GsEnumeratedPathTerm

! ------------------- Instance methods for GsEnumeratedPathTerm

category: 'Updating'
method: GsEnumeratedPathTerm
_canonicalizeTerm: aSymbol
  "Canonicalize the pathTerm name, so that terms are in alphabetical order"

  | terms sz canon |
  terms := (aSymbol subStringsDelimitedBy: $|) sortAscending.
  canon := String new.
  sz := terms size.
  1 to: sz do: [ :index | 
    | term |
    term := terms at: index.
    canon add: term.
    index < sz
      ifTrue: [ canon add: $| ] ].
  ^ canon asSymbol
%

category: 'Accessing'
method: GsEnumeratedPathTerm
_ivOffsetFor: anObject pathName: pathName
  "Returns the instance variable offset of anObject for the pathName which is one of the enumarated instance variables."

  ^ (self _classOf: anObject) _ivOffsetOf: pathName asSymbol
%

category: 'Accessing'
method: GsEnumeratedPathTerm
_ivOffsetsFor: anObject
  ^ self pathTermElements
    collect: [ :pathTermElement | self _ivOffsetFor: anObject pathName: pathTermElement ]
%

category: 'Accessing'
method: GsEnumeratedPathTerm
_nextObjectFor: anObject pathName: aPathName
  "Returns the object at the instance variable that corresponds to the receiver
 path term."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject pathName: aPathName.
  ivOffset ifNil: [ ^ nil ].
  ^ self _nextObjectFor: anObject atInstVar: ivOffset
%

category: 'Updating Indexes'
method: GsEnumeratedPathTerm
addMappingsForObject: anObject root: rootObject logging: aBoolean
  "Add dependency list entries for anObject."

  | ivOffset sz |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathTermElement | (self _ivOffsetFor: anObject pathName: pathTermElement)
        ifNil: [ self _invalidIvOffset: anObject ]
        ifNotNil: [ :off | ivOffset := off ] ].
  anObject
    getDepListAndAddPathTerm: self
    withIVOffset: ivOffset
    logging: aBoolean.
	" add an entry to the objects dependency list ... note that we ar only using the last ivOffset and that is okay as we compensate in the places that the ivOffset is used..."
  sz := children size.
  self nextObj: anObject do: [ :nextObj | 
      updateBtree ~~ nil
        ifTrue: [
          " insert an entry into the btree "
          (self _checkBtreeComparisonWith: nextObj)
            ifFalse: [ 
              ImproperOperation new
                _number:
                    (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
                args: { nextObj class. self name. updateBtree lastElementClassDescription };
                signal ].
          updateBtree btreeAt: nextObj put: anObject root: rootObject.
          needsDepList
            ifTrue: [ 
              " add an entry in the dependency list "
              nextObj getDepListAndAddLastElementPathTerm: self logging: false ] ].
      nil == nextObj
        ifTrue: [ self recordNilOnPathForRoot: rootObject ]
        ifFalse: [ 
          1 to: sz do: [ :i | " make recursive call to add mappings "
            (children at: i)
              addMappingsForObject: nextObj
              root: rootObject
              logging: aBoolean ] ] ]
%

category: 'Audit'
method: GsEnumeratedPathTerm
auditDepListFor: obj index: indexObj using: auditor optionalSentinel: optionalSentinel
  "Private."

  | depList j ivOffset |
  (nil == obj or: [ self size == 0 ])
    ifTrue: [ ^ nil ].
  depList := DependencyList for: obj.
  depList == nil
    ifTrue: [ 
      obj isInvariant
        ifFalse: [ auditor pathTermObjectHasNoDependencyList: self object: obj indexObj: indexObj depList: depList ] ]
    ifFalse: [ 
      (DepListTable _hasDependencyBitFor: obj)
        ifFalse: [ auditor pathTermObjectHasNoDependencyBitSet: self object: obj indexObj: indexObj depList: depList ].
      j := depList _findOffsetForPathTerm: self.
      j == nil
        ifTrue: [ auditor dependencyListHasNoEntryForPathTerm: self object: obj indexObj: indexObj depList: depList ]
        ifFalse: [ 
          "found entry for self in depList"
          ivOffset := depList at: j + 1.	"may be negative for last element reference count"
          ivOffset < 0
            ifTrue: [ 
              "reference count for last element, but we don't expect reference 
               count here ... see needsDepList clause below where depList 
               existence is sufficient"
              auditor dependencyListUnexpectedLastElementLocation: self object: obj indexObj: indexObj depList: depList ivOffset: ivOffset.
              ^ self ] ] ].
  needsDepList
    ifTrue: [ 
      self
        nextObj: obj
        do: [ :nextObj | 
          nextObj ifNotNil: [
            (DependencyList needsDepList: nextObj)
              ifTrue: [ 
                depList := DependencyList for: nextObj.
                depList == nil
                  ifTrue: [ 
                    nextObj isInvariant
                      ifFalse: [ auditor pathTermObjectHasNoDependencyList: self object: nextObj indexObj: indexObj depList: depList ] ] ] ] ] ].
  ^ self
%

category: 'Audit'
method: GsEnumeratedPathTerm
auditNscCountsFor: obj using: auditor count: btreeCounts
  "Private. "

  | sz |
  sz := self size.
  (nil == obj _or: [ sz == 0 ])
    ifTrue: [ ^ self ].
  self
    auditDepListFor: obj
    index: nil
    using: auditor
    optionalSentinel: Object new.
  self nextObj: obj do: [ :nextObj | | index |
      index := 1.
      children isEmpty
        ifTrue: [ self auditDirectNscCountsFor: nextObj using: auditor count: btreeCounts ]
        ifFalse: [ 1 to: sz do: [ :i | | count indexObj |
            indexObj := self at: i.
            indexObj size == offset
              ifTrue: [ count := btreeCounts at: index.
                count == 0
                  ifTrue: [ auditor pathTermIncorrectNumberOfBtreeEntries: self index: i offset: offset ].
                btreeCounts at: index put: count - 1.
                index := index + 1 ] ].
          nextObj ~~ nil
            ifTrue: [ 1 to: children size do: [ :i | (children at: i)
                  auditNscCountsFor: nextObj
                  using: auditor
                  count: (btreeCounts at: index).
                index := index + 1 ] ] ] ]
%

category: 'Testing'
method: GsEnumeratedPathTerm
coversIvOffset: anIvOffset for: anObject
  "Answer true if the receiver covers the given iv offset (see bug 46705)" 

  self pathTermElements
    do: [ :pathTermElement | 
      (self _ivOffsetFor: anObject pathName: pathTermElement)
        ifNil: [ self _invalidIvOffset: anObject ]
        ifNotNil: [ :off | anIvOffset = off ifTrue: [ ^true ] ] ].
  ^ false
%

category: 'Collection Based Modification'
method: GsEnumeratedPathTerm
findRootObjectsFor: anObject
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | rootObjects |
  rootObjects := IdentityBag new.
  self pathTermElements
    do: [ :pathName | 
      "Make a full pass for each of the enumerated instance variables"
      | roots ivOffset |
      ivOffset := self _ivOffsetFor: anObject pathName: pathName.
      ivOffset ifNil: [ self _invalidIvOffset: anObject ].
      roots := self findRootObjectsFor: anObject ivOffset: ivOffset.
      rootObjects := rootObjects _union: roots ].
  ^ rootObjects
%

category: 'Testing'
method: GsEnumeratedPathTerm
hasMatchingTerm: aSymbol
  "Answer true if the receiver's term matches <aSymbol>"

  ^ self name == (self _canonicalizeTerm: aSymbol)
%

category: 'Testing'
method: GsEnumeratedPathTerm
indicatesIndexOnNscElements
  "Returns false. The ides is not on elements of an nsc."

  ^ false
%

category: 'Testing'
method: GsEnumeratedPathTerm
isEnumeratedTerm
  "Returns true if the receiver indicates an enumerated path term."

  ^ true
%

category: 'Accessing'
method: GsEnumeratedPathTerm
nextObj: anObj do: aBlock
  | found |
  (nil == anObj or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  found := false.
  self pathTermElements
    do: [ :pathName | 
      | ivOffset |
      (ivOffset := self _ivOffsetFor: anObj pathName: pathName) ~~ nil
        ifTrue: [ 
          | nextObj |
          found := true.
          nextObj := self _nextObjectFor: anObj atInstVar: ivOffset.
          aBlock cull: nextObj cull: ivOffset ] ].
  found
    ifFalse: [ ^ self _invalidIvOffset: anObj ]
%

category: 'Accessing'
method: GsEnumeratedPathTerm
pathTermElements
  "return the individual pathTerm elements"

  ^ self name subStringsDelimitedBy: $|
%

category: 'Removing'
method: GsEnumeratedPathTerm
removeMappingsFor: anObject root: rootObject lastOne: aBoolean logging: doLogging
  "Remove entries in the btree and dependency lists for anObject."

  | nextObj depList isLast |
  self size == 0
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathName | 
      nextObj := self _nextObjectFor: anObject pathName: pathName.
      updateBtree ~~ nil
        ifTrue: [ 
          " remove an entry from the B-tree "
          (updateBtree btreeRemoveKey: nextObj value: anObject root: rootObject)
            ifFalse: [ self _errorObjectNotInBtree: nextObj value: anObject root: rootObject ].
          DependencyList removePathTerm: self for: nextObj logging: doLogging	" remove dependency list entry for next object " ].
      nextObj ifNil: [ self removeNilOnPathForRoot: rootObject ].
      (nil ~~ nextObj and: [ children isEmpty not ])
        ifTrue: [ 
          aBoolean
            ifTrue: [
              | last |
              last := (children at: 1) _isLastOccurrenceInIndexFor: nextObj.
              isLast := aBoolean and: [ last ] ]
            ifFalse: [ isLast := false ].
          " make recursive call to remove mappings "
          1 to: children size do: [ :i | (children at: i) removeMappingsFor: nextObj root: rootObject lastOne: isLast logging: doLogging ].
          isLast
            ifFalse: [
              "check again in case there are no more references left in btree"
              isLast := ((children at: 1) findReachableRootsFor: nextObj) isEmpty ].
          isLast
            ifTrue: [ 
              " remove dependency list entries for next object "
              (depList := DependencyList for: nextObj) ~~ nil
                ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ] ] ] ]
%
! Class Implementation for GsSetValuedPathTerm

! Remove existing behavior from GsSetValuedPathTerm
removeallmethods GsSetValuedPathTerm
removeallclassmethods GsSetValuedPathTerm

! ------------------- Instance methods for GsSetValuedPathTerm

category: 'Error Handling'
method: GsSetValuedPathTerm
_errorPathObjectNotAnNsc: anObject
  "An object traversed along an index path through a set-valued instance
 variable was not an NSC."

  ImproperOperation new
    reason: #'rtErrPathTermObjectNotAnNsc';
    object: anObject;
    details: 'Expected an UnorderedCollection';
    signal
%

category: 'Updating Indexes'
method: GsSetValuedPathTerm
addMappingsForObject: anNsc root: rootObject logging: aBoolean
  "Adds index dictionary entries for the objects in anNsc.  Also update the
 NSC's index list."

  | iList sz |
  (nil == anNsc _or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  anNsc class isNsc
    ifFalse: [ ^ self _errorPathObjectNotAnNsc: anNsc ].
  sz := self size.
  iList := anNsc _getIndexList.
  anNsc _putInWriteSet.	" for each index that utilizes this path term "
  1 to: sz do: [ :i | iList addIndex: (self at: i) withOffset: offset + 1 nsc: anNsc ].
  sz := children size.
  anNsc
    do: [ :setElement | " insert dictionary entry mapping set element -> NSC "
      updateBtree ~~ nil
        ifTrue: [
          (self _checkBtreeComparisonWith: setElement)
            ifFalse: [ 
              ImproperOperation new
                _number:
                    (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
                args: { setElement class. self name. updateBtree lastElementClassDescription };
                signal ].
          updateBtree btreeAt: setElement put: anNsc root: rootObject.
          needsDepList
            ifTrue: [ " add an entry in the dependency list "
              setElement getDepListAndAddLastElementPathTerm: self logging: aBoolean ] ].
      1 to: sz do: [ :j | (children at: j) addMappingsForObject: setElement root: rootObject logging: aBoolean ] ]
%

category: 'Audit'
method: GsSetValuedPathTerm
auditDepListFor: obj index: indexObj using: auditor optionalSentinel: optionalSentinel
  "the set does not have a dependency list"

  self shouldNotImplement: #'auditDepListFor:index:using:optionalSentinel:'
%

category: 'Audit'
method: GsSetValuedPathTerm
auditNscCountsFor: anNsc using: auditor count: btreeCounts
  "Private. "

  | sz |
  sz := self size.
  (nil == anNsc _or: [ sz == 0 ])
    ifTrue: [ ^ self ].
  anNsc do: [ :setElement | | index |
      index := 1.
      children isEmpty
        ifTrue: [ self auditDirectNscCountsFor: setElement using: auditor count: btreeCounts ]
        ifFalse: [ 1 to: sz do: [ :i | | count indexObj |
            indexObj := self at: i.
            indexObj size == offset
              ifTrue: [ count := btreeCounts at: index.
                count == 0
                  ifTrue: [ auditor pathTermIncorrectNumberOfBtreeEntries: self index: i offset: offset ].
                btreeCounts at: index put: count - 1.
                index := index + 1 ] ].
          1 to: children size do: [ :i | (children at: i)
              auditNscCountsFor: setElement
              using: auditor
              count: (btreeCounts at: index).
            index := index + 1 ] ] ]
%

category: 'Collection Based Modification'
method: GsSetValuedPathTerm
findRootObjectsFor: anNsc
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | rootObjects roots |
  rootObjects := IdentityBag new.
  anNsc
    do: [ :nextObj | 
      "Make a full pass for each of the enumerated instance variables"
      updateBtree ~~ nil
        ifTrue: [ 
          "btree exists for receiver, collect the root objects, where nextObj is the key and anNsc
           is the value."
          roots := self findRootObjectsForKey: nextObj value: anNsc.
          rootObjects := rootObjects _union: roots ].
      (self children isEmpty)
        ifTrue: [ ^ rootObjects ].
      1 to: children size do: [:i | 
        roots := (children at: i) findRootObjectsFor: nextObj.
        rootObjects := rootObjects _union: roots ] ].
  ^ rootObjects
%

category: 'Testing'
method: GsSetValuedPathTerm
indicatesIndexOnNscElements
  "Returns true if the path term indicates that the index is on elements of the
 NSC itself.  This is the case if the path term ends in '*' (that is, this
 path term has no children)."

  ^ children isEmpty
%

category: 'Testing'
method: GsSetValuedPathTerm
indicatesNsc
  "Returns true, the receiver indicates that the index is on an
 instance variable that is expected to be a collection (i.e., set valued pathterm."

  ^ true
%

category: 'Testing'
method: GsSetValuedPathTerm
isSetValuedTerm
  "Returns true if the receiver indicates a set-valued instance variable."

  ^ true
%

category: 'Accessing'
method: GsSetValuedPathTerm
nextObj: anNsc do: aBlock
  anNsc do: aBlock
%

! Class Implementation for GsOptionalPathTerm

! Remove existing behavior from GsOptionalPathTerm
removeallmethods GsOptionalPathTerm
removeallclassmethods GsOptionalPathTerm

! ------------------- Instance methods for GsOptionalPathTerm

category: 'Private'
method: GsOptionalPathTerm
_findAllValuesForIdenticalRootObject: rootObject
	" if index directly on NSC elements or anObject is nil "

	| ivOffset |
	ivOffset := self _ivOffsetFor: rootObject.
	ivOffset
		ifNil: [ 
			"no values in index for rootObject"
			^ IdentityBag new ].
	^ super _findAllValuesForIdenticalRootObject: rootObject
%

category: 'Traversing'
method: GsOptionalPathTerm
_getNextObjectForTraversal: anObject
  " get the next object along the path "

  | ivOffset nextObj |
  ivOffset := anObject _findIVOffsetForPathTerm: self.
  ivOffset == nil
    ifTrue: [ nextObj := nil ]
    ifFalse: [ nextObj := anObject instVarAt: ivOffset ].
  ^ nextObj
%

category: 'Accessing'
method: GsOptionalPathTerm
_nextObjectFor: anObject atInstVar: anIvOffset
	"Returns the object at the instance variable that corresponds to the receiver
 path term."

	| ivOffset |
	ivOffset := self _ivOffsetFor: anObject.
	ivOffset ifNil: [ ^ nil ].
	^ super _nextObjectFor: anObject atInstVar: anIvOffset
%

category: 'Adding'
method: GsOptionalPathTerm
addMappingsForObject: anObject root: rootObject logging: aBoolean
	"Add dependency list entries for anObject."

	| ivOffset |
	(nil == anObject or: [ self size == 0 ])
		ifTrue: [ ^ self ].
	(ivOffset := self _ivOffsetFor: anObject) == nil
		ifTrue: [ ^ self recordNilOnPathForRoot: rootObject ].
	^ super addMappingsForObject: anObject root: rootObject logging: aBoolean
%

category: 'Removing'
method: GsOptionalPathTerm
removeMappingsFor: anObject root: rootObject lastOne: aBoolean logging: doLogging
  "Remove entries in the btree and dependency lists for anObject."

	| ivOffset |
	(nil == anObject or: [ self size == 0 ])
		ifTrue: [ ^ self ].
	(ivOffset := self _ivOffsetFor: anObject) == nil
		ifTrue: [ ^ self removeNilOnPathForRoot: rootObject ].
	^ super removeMappingsFor: anObject root: rootObject lastOne: aBoolean logging: doLogging
%

category: 'Testing'
method: GsOptionalPathTerm
termsRequired
  "Answer true if receiver requires that instance variables of indexed objects are present "

  ^ false
%

! Class Implementation for GsSelectorPathTerm

! Remove existing behavior from GsSelectorPathTerm
removeallmethods GsSelectorPathTerm
removeallclassmethods GsSelectorPathTerm

! ------------------- Instance methods for GsSelectorPathTerm

category: 'Updating'
method: GsSelectorPathTerm
_addChildTerm: aPathTerm
  self
    error:
      'Selector path term may not have child terms: ' , aPathTerm printString
%

category: 'Accessing'
method: GsSelectorPathTerm
_nextObjectFor: anObject
  "Returns the object at the instance variable that corresponds to the receiver
 path term."

  ^ anObject perform: self termSelector
%

category: 'Accessing'
method: GsSelectorPathTerm
_nextObjectFor: anObject atInstVar: ivOffset
  self shouldNotImplement: #'_nextObjectFor:atInstVar:'
%

category: 'Updating Indexes'
method: GsSelectorPathTerm
addMappingsForObject: anObject root: rootObject logging: aBoolean
  "No dependency list logic for receiver."

  | nextObj |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  nextObj := self _nextObjectFor: anObject.	" get the next object along the path "
  updateBtree ~~ nil
    ifTrue: [ " insert an entry into the B-tree "
      (self _checkBtreeComparisonWith: nextObj)
        ifFalse: [ 
          ImproperOperation new
            _number:
                (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
            args: { nextObj class. self name. updateBtree lastElementClassDescription };
            signal ].
      updateBtree btreeAt: nextObj put: anObject root: rootObject ].
  nil == nextObj
    ifTrue: [ ^ self recordNilOnPathForRoot: rootObject ].
  1 to: children size do: [ :i | " make recursive call to add mappings "
    (children at: i)
      addMappingsForObject: nextObj
      root: rootObject
      logging: aBoolean ]
%

category: 'Audit'
method: GsSelectorPathTerm
auditDepListFor: obj index: indexObj using: auditor optionalSentinel: optionalSentinel
  "No dependency lists"

  ^ obj
%

category: 'Testing'
method: GsSelectorPathTerm
isSelectorTerm
  ^ true
%

category: 'Accessing'
method: GsSelectorPathTerm
needsDepList
  "receiver does not do automatic maintenance updates"

  ^ false
%

category: 'Updating'
method: GsSelectorPathTerm
needsDepList: ignored
  "receiver does not do automatic maintenance updates"

%

category: 'Accessing'
method: GsSelectorPathTerm
termSelector
  | term |
  term := self name.
  ^ term copyFrom: 2 to: term size
%

category: 'Updating Indexes'
method: GsSelectorPathTerm
update: anObject at: ivOffset to: newValue
  "should never be sent to the receiver as automatic index maintenance is not supported"

  self shouldNotImplement: #'update:at:to:'
%

! Class Extensions

! Class initializers 

doit
true.
%

category: 'Indexing Support'
method: GsCollectionBasedPathTerm
_updateCollectionBasedIndexFor: anIndex on: anNsc offset: anOffset addingIndex: addingIndex
  "noop by default"

  
%
category: 'Indexing Support'
method: GsSetValuedPathTerm
_updateCollectionBasedIndexFor: anIndex on: anNsc offset: anOffset addingIndex: addingIndex
  "if anNsc already has an index, add/remove anIndex as appropriate"

  anNsc _indexedPaths ~~ nil
    ifTrue: [ 
      | nscOffset |
      nscOffset := anOffset + 1.
      addingIndex
        ifTrue: [ 
          anNsc _indexedPaths
            addIndex: anIndex
            withOffset: (anIndex at: anOffset) offset + 1
            nsc: anNsc ]
        ifFalse: [ anNsc _indexedPaths removeIndex: anIndex withOffset: nscOffset for: anNsc ] ]
%
category: 'Private'
method: GsPathTerm
_findAllValuesForIdenticalRootObject: rootObject
	" if index directly on NSC elements or anObject is nil "

	| key tmpList |
	(self indicatesIndexOnNscElements or: [ nil == rootObject ])
		ifTrue: [ key := rootObject ]
		ifFalse: [ key := self _nextObjectFor: rootObject ].
	tmpList := IdentityBag new.
	self updateBtree btreeRoot _findAllValuesForIdenticalKey: key into: tmpList.
	^ (tmpList occurrencesOf: rootObject) <= 1
%
category: 'Modification'
method: GsPathTerm
findReachableRootsFor: anObject ivOffset: ivOffset oldValue: oldValue
  | rootObjects indexObj reachableRoots reachableNilOnPathRoots |
  rootObjects := self findRootObjectsFor: anObject ivOffset: ivOffset oldValue: oldValue.
  reachableNilOnPathRoots := self findReachableNilOnPathRoots.
  rootObjects addAll: reachableNilOnPathRoots.
  indexObj := self at: self size.
  "select rootObjects for which anObject is reachable"
  reachableRoots := IdentityBag new.
  rootObjects asIdentitySet do: [:root | 
    (self traverseAllReachableFor: root indexObj: indexObj)
      do: [ :intermediateObjectReachableFromRoot |
        intermediateObjectReachableFromRoot == anObject
          ifTrue: [ reachableRoots add: root withOccurrences: (rootObjects occurrencesOf: root) ] ] ].
  ^ reachableRoots
%
category: 'Collection Based Modification'
method: GsPathTerm
findRootObjectsFor: anObject ivOffset: ivOffset oldValue: oldValue
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject at is a key or parent of a key."

  | rootObjects nextObj roots |
  rootObjects := IdentityBag new.
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
  nextObj == oldValue ifFalse: [ self error: 'unexpected error: oldValue and nextObj should be identical by definition.' ].
  updateBtree ~~ nil
    ifTrue: [ 
      "btree exists for receiver, collect the root objects, where anObject is the key and nextObj
       is the value."
      rootObjects := self findRootObjectsForKey: nextObj value: anObject ].
  (nil == nextObj or: [ self children isEmpty ])
    ifTrue: [ ^ rootObjects ].
  1 to: children size do: [:i | 
    roots := (children at: i) findRootObjectsFor: nextObj oldValue: oldValue.
    rootObjects := rootObjects _union: roots  ].
  ^ rootObjects
%
category: 'Collection Based Modification'
method: GsPathTerm
findRootObjectsFor: anObject oldValue: oldValue
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject.
  ivOffset ifNil: [ self _invalidIvOffset: anObject ].
  ^ self 
      findRootObjectsFor: anObject 
      ivOffset: ivOffset  
      oldValue: (self _nextObjectFor: oldValue atInstVar: ivOffset).
%
category: 'Private'
method: GsPathTerm
recordNilOnPathForRoot: rootObject

  "record fact that rootObject has a nil on the path for this path term"

  updateBtree ifNotNil: [ ^ self ].
  nilOnPath add: rootObject
%
category: 'Modification'
method: GsEnumeratedPathTerm
findReachableRootsFor: anObject ivOffset: ivOffset oldValue: oldValue

  | rootObjects indexObj reachableRoots factor |
  rootObjects := super findReachableRootsFor: anObject ivOffset: ivOffset oldValue: oldValue.
  indexObj := self at: self size.
  (rootObjects includes: anObject)
    ifTrue: [ factor := rootObjects occurrencesOf: anObject ]
    ifFalse: [ 
      | enumeratedValues |
      enumeratedValues := IdentityBag new.
      self nextObj: anObject do: [:nextObj |
        enumeratedValues add: nextObj].
      factor := (enumeratedValues occurrencesOf: oldValue) ].
  factor == 1
    ifTrue: [ ^ rootObjects ].
  reachableRoots := IdentityBag new.
  rootObjects asIdentitySet do: [:rootObject |
    | num theFactor |
    theFactor := factor + (indexObj cumulativeFactorFor: rootObject upTo: self startingAt: 1).
    num := rootObjects occurrencesOf: rootObject.
    reachableRoots add: rootObject withOccurrences: (num / theFactor) ceiling ].
  ^ reachableRoots
%
category: 'Collection Based Modification'
method: GsEnumeratedPathTerm
findRootObjectsFor: anObject oldValue: oldValue
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | rootObjects factor enumeratedValues reachableRoots |
  rootObjects := IdentityBag new.
  enumeratedValues := IdentityBag new.
  self nextObj: anObject do: [:nextObj |
    enumeratedValues add: nextObj].
  factor := 0.
  self pathTermElements
    do: [ :pathName | 
      "Make a full pass for each of the enumerated instance variables"
      | roots ivOffset nextOldValue |
      ivOffset := self _ivOffsetFor: anObject pathName: pathName.
      ivOffset ifNil: [ self _invalidIvOffset: anObject ].
      nextOldValue := self _nextObjectFor: oldValue atInstVar: ivOffset.
      factor := factor + (enumeratedValues occurrencesOf: nextOldValue).
      roots := self 
        findRootObjectsFor: anObject 
        ivOffset: ivOffset  
        oldValue: nextOldValue.
      rootObjects := rootObjects _union: roots ].
  factor == 1 ifTrue: [ ^ rootObjects ].
  reachableRoots := IdentityBag new.
  rootObjects asIdentitySet do: [:rootObject |
    | num |
    num := rootObjects occurrencesOf: rootObject.
    reachableRoots add: rootObject withOccurrences: (num / factor) ceiling ].
  ^ reachableRoots
%
category: 'Accessing'
method: GsPathTerm
nilOnPath
  "return nil or identity bag that contains root objects with nil along the path"

  ^ nilOnPath
%
category: 'Modification'
method: GsPathTerm
findReachableNilOnPathRoots

  | reachableRoots |
  reachableRoots := IdentityBag new.
  reachableRoots addAll: nilOnPath.
  1 to: children size do: [:i |
    reachableRoots addAll: (children at: i) nilOnPath ].
  ^ reachableRoots
%
category: 'Testing'
method: GsPathTerm
isOptionalTerm
  ^ false
%
category: 'Private'
method: GsPathTerm
removeNilOnPathForRoot: rootObject

  "rootObject with nil on path has been removed"

  updateBtree ifNotNil: [ ^ self ].
  nilOnPath remove: rootObject
%
category: 'Testing'
method: GsOptionalPathTerm
isOptionalTerm
  ^ true
%
category: 'Audit'
method: GsPathTerm
auditUpdateCache: nsc on: aString
  "Verifies that the update caches (nilOnPath) are consistent.
 Returns a string that describes any inconsistencies found."

  | auditor prevObj |
  auditor := BtreePlusNodeAuditor new auditResultString: aString; yourself.
  prevObj := #_incompletePathTraversal.
  nsc do: [ :obj |
    obj ~~ prevObj
      ifTrue: [
        self auditUpdateCacheFor: obj
          root: obj
          nilOnPathRoots: nil
          occurrences: (nsc occurrencesOf: obj)
          using: auditor ].
    prevObj := obj ].
%
category: 'Audit'
method: GsPathTerm
auditUpdateCacheFor: anObject root: root nilOnPathRoots: nilOnPathRoots occurrences: num using: auditor
  "verify proper accounting for objects with nil on path"

  | ivOffset nextObj |
  self indicatesIndexOnRootNsc
    ifTrue: [ ^ self ].
  (ivOffset := self _ivOffsetFor: anObject) == nil
    ifTrue: [ 
      self isOptionalTerm ifTrue: [ ^ self ].
      auditor pathTermInvalidIvOffset: self ivOffset: ivOffset object: anObject ].
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
  (nil == nextObj and: [ updateBtree isNil ])
    ifTrue: [
      | nilOnPathNum |
      nilOnPathNum := nilOnPath occurrencesOf: root.
      nilOnPathRoots 
        ifNil: [ 
          nilOnPathNum = num
           ifFalse: [ auditor pathTermIncorrectNilOnPathCount: self root: root got: nilOnPathNum  expected: num ] ]
        ifNotNil: [ 
          | expected count |
          expected := nilOnPathRoots occurrencesOf: root.
          count := 0.
          expected
            timesRepeat: [ 
              nilOnPathRoots 
                remove: root 
                ifAbsent: [ 
                  auditor pathTermIncorrectNilOnPathCount: self root: root got: count expected: expected.
                  ^ self ].
              count := count + 1 ] ].
      ^ self ].
  1 to: children size do: [ :i |
    (children at: i)
      auditUpdateCacheFor: nextObj
      root: root
      nilOnPathRoots: nilOnPathRoots 
      occurrences: num 
      using: auditor ]
%
category: 'Audit'
method: GsEnumeratedPathTerm
auditUpdateCacheFor: anObject root: root nilOnPathRoots: nilOnPathRoots occurrences: num using: auditor
  "verify proper accounting for objects with nil on path"

  self nextObj: anObject do: [ :nextObj | 
     (nil == nextObj and: [ updateBtree isNil ]) 
      ifTrue: [
        | nilOnPathNum |
        nilOnPathNum := nilOnPath asIdentitySet occurrencesOf: root.
        nilOnPathNum = num
          ifFalse: [ auditor pathTermIncorrectNilOnPathCount: self root: root got: nilOnPathNum  expected: num ] ]
      ifFalse: [ 
        1 to: children size do: [ :i |
          (children at: i)
            auditUpdateCacheFor: nextObj
            root: root
            nilOnPathRoots: nilOnPathRoots 
            occurrences: num 
            using: auditor ] ] ]
%
category: 'Audit'
method: GsSetValuedPathTerm
auditUpdateCacheFor: anNsc root: root nilOnPathRoots: nilOnPathRoots occurrences: num using: auditor
  "verify proper accounting for objects with nil on path"

  | prevObj reachableNilOnPathRoots excessNilOnPathCount |
  prevObj := #_incompletePathTraversal.
  reachableNilOnPathRoots := self findAllReachableNilOnPathRoots.
  nilOnPathRoots ifNotNil: [ reachableNilOnPathRoots addAll: nilOnPathRoots ].
  anNsc do: [ :nextObj |
     updateBtree 
       ifNil: [
        nextObj ~~ prevObj
          ifTrue: [ | newNum |
            newNum := num * (anNsc occurrencesOf: nextObj).
            1 to: children size do: [ :i |
              (children at: i)
                auditUpdateCacheFor: nextObj
                root: root
                nilOnPathRoots: reachableNilOnPathRoots 
                occurrences: newNum 
                using: auditor ] ] ].
    prevObj := nextObj ].
  (((excessNilOnPathCount := reachableNilOnPathRoots occurrencesOf: root) > 0) and: [ nilOnPathRoots isNil ])
    ifTrue: [ 
      "uppermost set valued term is where the accounts must balance"
      auditor setValuedPathTermExtraNilOnPathCount: self root: root excessCount: excessNilOnPathCount ].
%
category: 'Modification'
method: GsPathTerm
traverseAllReachableFor: root indexObj: indexObj

  | res |
  res := IdentitySet new.
  (indexObj traverseAllWithParents: root upTo: self startingAt: 1) do: [:each | res add: (each at: 2) ].
  ^ res
%
category: 'Modification'
method: GsEnumeratedPathTerm
traverseAllReachableFor: root indexObj: indexObj

  ^ (indexObj traverseAllWithParents: root upTo: self startingAt: 1) collect: [:each | each at: 2]
%
category: 'Audit'
method: GsSelectorPathTerm
auditUpdateCacheFor: anObject root: root nilOnPathRoots: nilOnPathRoots occurrences: num using: auditor
  "verify proper accounting for objects with nil on path"

  | nextObj |
  nextObj := self _nextObjectFor: anObject.
  (nil == nextObj and: [ updateBtree isNil ])
    ifTrue: [
      | nilOnPathNum |
      nilOnPathNum := nilOnPath occurrencesOf: root.
      nilOnPathNum = num
        ifFalse: [ auditor pathTermIncorrectNilOnPathCount: self root: root got: nilOnPathNum  expected: num ].
      ^ self ].
  1 to: children size do: [ :i |
    (children at: i)
      auditUpdateCacheFor: nextObj
      root: root
      nilOnPathRoots: nilOnPathRoots 
      occurrences: num 
      using: auditor ]
%
category: 'Accessing'
method: GsPathTerm
_thisAndAllChildTerms

"Returns an Array containing the receiver and all child path terms."


^ self _thisAndAllChildTermsInto: { } 
%
category: 'Index Maintenance'
method: GsPathTerm
removeIndex: indexObj

"The given index is being removed from this path term."

1 to: self size do: [ :i |
    indexObj == (self at: i)
        ifTrue: [ ^ self removeAtIndex: i ]
]

%
category: 'Modification'
method: GsCollectionBasedPathTerm
findRootObjectsForKey: aKey value: aValue
 ^true ifTrue: [super findRootObjectsForKey: aKey value: aValue]
ifFalse: [  updateBtree findRootObjectsComparingForKey: aKey value: aValue]
%
category: 'Become Support'
method: GsPathTerm
_getPathTermIndexReferencesInto: refsToRcvr for: anObject
  "Place information about references to the receiver due to the receiver's 
 participation in an index into the given Array.  The Array consists of pairs:

 1) An object that references the receiver.  If the object is directly
    contained in an indexed NSC, then this object is the NSC.  If the object
    in the dependency list is a tracking object (for object modification
    tracking), then this object is the tracking object.
 2) The offset of the instance variable in that object that references the
    receiver.  
 3) If the object is the last path term, this number is < 0 and is a reference 
    count for the number of times .  
 3) If the object in the dependency list is a tracking
    object, this number is 0.

 This is used only in support of the 'become:' method."
 
  | indexObj rootObjects parents prevPathTerm |
  self size = 0
    ifTrue: [ ^ self ].
  indexObj := self at: 1.
  self offset == 1
    ifTrue: [ 
      "anObject is the first path term in an index (which means anObject
       is directly contained in an indexed NSC) "
      anObject
        _addToReferences: indexObj nscRoot
        offset: 0
        occurrences: (indexObj nscRoot occurrencesOf: anObject)
        into: refsToRcvr.
      ^ self ].
  rootObjects := self findRootObjectsFor: anObject.
  parents := IdentityBag new.
  rootObjects asIdentitySet do: [:root | 
    (indexObj traverseAllWithParents: root upTo: self startingAt: 1)
      do: [ :ar | | intermediateObjectReachableFromRoot parentObject |
        parentObject := ar at: 1.
        intermediateObjectReachableFromRoot := ar at: 2.
        intermediateObjectReachableFromRoot == anObject
          ifTrue: [ parents add: parentObject withOccurrences: (rootObjects occurrencesOf: root) ] ] ].
  " get the previous path term in the index "
  prevPathTerm := indexObj at: self offset - 1.
  parents do: [:parentObj |
    prevPathTerm indicatesMultiValue
      ifTrue: [ 
        anObject
          _addToReferences: parentObj
          offset: 0
          occurrences: (parentObj occurrencesOf: anObject)
          into: refsToRcvr ]
      ifFalse: [
        anObject
          _addToReferences: parentObj
          offset: (prevPathTerm _ivOffsetFor: parentObj)
          occurrences: 1
          into: refsToRcvr ] ]
%
category: 'Updating Indexes'
method: GsSelectorPathTerm
undoMappingsFor: anObject lastOne: aBoolean logging: doLogging
  "No dependency lists for anObject."

%
category: 'Updating Indexes'
method: GsPathTerm
_clear


"Assigns nil to important instance variables and sets the receiver's size
 to zero."

children := nil.
updateBtree := nil.
self size: 0.
%
category: 'Private'
method: GsPathTerm
_makeObsolete

"Makes the receiver and all its children obsolete."



1 to: children size do: [ :i | (children at: i) _makeObsolete ].
self _clear.
%
category: 'Updating Indexes'
method: GsPathTerm
undoMappingsFor: anObject lastOne: aBoolean logging: doLogging

"Remove entries in the index dictionary and dependency lists for anObject."


| nextObj depList more isLast |
self size == 0
  ifTrue: [ ^ self ].

[
    nextObj := self _nextObjectFor: anObject.
] onException: Error do:[ :ex |  
    "Handle any error thrown during execution of _nextObjectFor:
     Cannot resolve the path for this guy --- can't have been added to index,
     in fact this object could have caused the failure in the first place"
    ^ self
].

updateBtree ~~ nil
  ifTrue: [
    " remove dependency list entry for next object "
    DependencyList removePathTerm: self for: nextObj logging: doLogging.
  ].

more := false.
( nil ~~ nextObj and: [ children isEmpty not ] )
  ifTrue: [ " make recursive call to remove mappings "
    isLast := aBoolean and: [ more not ].
    1 to: children size do: [ :i |
      (children at: i) undoMappingsFor: nextObj
        lastOne: isLast
        logging: doLogging
    ].
    isLast
      ifTrue: [
        " remove dependency list entries for next object "
        (depList := DependencyList for: nextObj) ~~ nil
          ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ]
      ]
  ]
%
category: 'Updating Indexes'
method: GsCollectionBasedPathTerm
cleanupDependencyListFor: anNsc
  ""

  self subclassResponsibility: #'cleanupDependencyListFor:'
%
category: 'Updating Indexes'
method: GsSetValuedPathTerm
cleanupDependencyListFor: anNsc
  ""

  | sz iList |
  (nil == anNsc _or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  anNsc class isNsc
    ifFalse: [ ^ self _errorPathObjectNotAnNsc: anNsc ].
  (iList := anNsc _indexedPaths) ~~ nil
    ifTrue: [ iList removeIndexesInPathTerm: self for: anNsc ].
  sz := children size.
  anNsc
    do: [ :setElement | 
      | depList |
      needsDepList
        ifTrue: [ 
          depList := DependencyList for: setElement.
          depList ~~ nil
            ifTrue: [ depList removeCompletelyPathTerm: self for: setElement ] ].
      1 to: sz do: [ :j | (children at: j) cleanupDependencyListFor: setElement ] ]
%
category: 'Updating Indexes'
method: GsSelectorPathTerm
cleanupDependencyListFor: anObject
  ""

  
%
category: 'Updating Indexes'
method: GsEnumeratedPathTerm
cleanupDependencyListFor: anObject
  ""

  | deplistOffset nextObj depList |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (depList := DependencyList for: anObject) == nil
    ifTrue: [ ^ self ].
  deplistOffset := depList removeCompletelyPathTerm: self for: anObject.	" remove the receiver from the dependency list "
  (self indicatesIndexOnRootNsc
    or: [ 
      "last element dependency list"
      deplistOffset < 0 ])
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathName | 
      (self _ivOffsetFor: anObject pathName: pathName)
        ifNotNil: [ :ivOffset | 
          nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
          nextObj
            ifNotNil: [ 
              needsDepList
                ifTrue: [ 
                  (depList := DependencyList for: nextObj) ~~ nil
                    ifTrue: [ depList removeCompletelyPathTerm: self for: nextObj ] ].
              1 to: children size do: [ :i | (children at: i) cleanupDependencyListFor: nextObj ] ] ] ]
%
category: 'Updating Indexes'
method: GsPathTerm
undoDirectMappingFor: anObject logging: doLogging
  "There is a range index directly on the elements of the NSC."
  " no indexDictionary in the receiver ... noop"
%
category: 'Index Maintenance'
method: GsPathTerm
_doRemoveIndex: indexObj nsc: nsc
      "Remove the index associated with the receiver from the nsc."
 
      | bag incomplete indexManager allTerms |
      indexManager := IndexManager current.
      bag := nsc _asIdentityBag.
      incomplete := #'_incompletePathTraversal'.
      indexObj _findSetValuedPathTermOffset ~~ 0
        ifTrue: [ 
          | updateIndexList objNsc |
          "See if there is a set-valued instance variable before the unshared path
     term."
          updateIndexList := false.
          (self == indexObj lastPathTerm
            _and: [ self size > 1 _and: [ (self at: 1) size ~= (self at: 2) size ] ])
            ifTrue: [ 
              " see if index is subsumed by an existing index "
              " if last path term is #*, will need to update the
              last objects' index list "
              self indicatesMultiValue
                ifTrue: [ 
                  updateIndexList := true.
                  objNsc := NscBuilder
                    for: IdentityBag new
                    max: indexObj sizeForNscBuilder ]
                ifFalse: [ objNsc := nil ] ]
            ifFalse: [ objNsc := NscBuilder for: IdentityBag new max: indexObj sizeForNscBuilder ].
          1 to: bag size do: [ :i | 
            " for each element in the NSC ... "
            " traverse up to the point of the unshared path term "
            indexObj
              traverse: (bag _at: i)
              upTo: self
              startingAt: 1
              addingIndex: false
              do: [:endObj :rootObj |
                objNsc ~~ nil
                  ifTrue: [
                    updateIndexList
                      ifTrue: [ self removeIndexListEntriesFor: objNsc completeBag index: indexObj ]
                      ifFalse: [ 
                        endObj ~~ incomplete
                          ifTrue: [ 
                            " only remove mappings once for a given object "
                            self
                              removeMappingsFor: endObj
                              root: rootObj
                              lastOne: true
                              logging: false.
                            DependencyList removePathTerm: self for: endObj ] ] ] ] ] ]
        ifFalse: [ | obj |
          self indicatesIndexOnRootNsc
            ifTrue: [ 
              " see if the index is on elements of the NSC "
              1 to: bag size do: [ :j | 
                obj := bag _at: j.
                self removeDirectMappingFor: obj logging: false.
                DependencyList removePathTerm: self for: obj.
                indexManager commitIndexMaintenance: indexObj at: j ] ]
            ifFalse: [ 
              [ | endObj | 
              1 to: bag size do: [ :j | 
                obj := bag _at: j.
                endObj := indexObj traverse: obj upTo: self.
                (nil ~~ endObj and: [ incomplete ~~ endObj ])
                  ifTrue: [ 
                    self removeMappingsSkipBtreeFor: endObj root: obj lastOne: true logging: false.
                    DependencyList removePathTerm: self for: endObj.
                    indexManager commitIndexMaintenance: indexObj at: j ] ] ]
                onException: Error
                do: [ :ex | 
                  ex number == (ErrorSymbols at: #'rtErrObjectPathTermNotInDependencyList')
                    ifTrue: [ 
                      "continue execution"
                      ex resume ]
                    ifFalse: [ ex outer ] ] ].
          allTerms := self _thisAndAllChildTerms ].
      ^ allTerms
%
category: 'Updating Indexes'
method: GsPathTerm
removeMappingsSkipBtreeFor: anObject root: rootObj lastOne: aBoolean logging: doLogging

"Remove entries in the dependency lists for anObject.
 This version for index removal, it does not update the btree since the btree
 is going away (#40858)."

| nextObj depList isLast |
self size == 0
  ifTrue: [ ^ self ].

nextObj := self _nextObjectFor: anObject.

updateBtree ~~ nil
  ifTrue: [
    " skip update of btree, index is going away"

    " remove dependency list entry for next object "
    DependencyList removePathTerm: self for: nextObj logging: doLogging.
  ].

( nil ~~ nextObj and: [ children isEmpty not ] )
  ifTrue: [ 
    aBoolean 
      ifTrue: [
        | last |
        last := (children at: 1) _isLastOccurrenceInIndexFor: nextObj.
        isLast := aBoolean and: [last ] ]
      ifFalse: [ isLast := false ].
    " make recursive call to remove mappings "
    1 to: children size do: [ :i |
      (children at: i) removeMappingsSkipBtreeFor: nextObj
        root: rootObj 
        lastOne: isLast
        logging: doLogging
    ].
        " remove dependency list entries for next object "
        (depList := DependencyList for: nextObj) ~~ nil
          ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ]
  ]
%
category: 'Index Maintenance'
method: GsPathTerm
_doAddMappingsFor: anNSC on: indexObj indexManager: indexManager hasSet: hasSet
  "Adds the mappings in the index dictionary, dependency lists, and so on for all
 elements in the receiver."
 
  | bag |
  bag := anNSC _asIdentityBag.
  hasSet
    ifTrue: [ 
      | updateIndexList objNsc incomplete |
      " if there is a set-valued instance variable, collect objects in a
      Bag on which to update the index dictionary "
      updateIndexList := false.
      (self == indexObj lastPathTerm
        _and: [ self size > 1 _and: [ (self at: 1) size ~= (self at: 2) size ] ])
        ifTrue: [ 
          " see if index is subsumed by an existing index "
          " if last path term is #*, will need to update the
          last objects' index list "
          self indicatesMultiValue
            ifTrue: [ 
              updateIndexList := true.
              objNsc := NscBuilder
                for: IdentityBag new
                max: indexObj sizeForNscBuilder ]
            ifFalse: [ objNsc := nil ] ]
        ifFalse: [ objNsc := NscBuilder for: IdentityBag new max: indexObj sizeForNscBuilder ].
      incomplete := #'_incompletePathTraversal'.
       1 to: bag size do: [ :i | 
        " for each element in the NSC ... "
        " traverse up to the point of the unshared path term "
        indexObj
          traverse: (bag _at: i)
          upTo: self
          startingAt: 1
          addingIndex: true
          do: [:endObj :rootObj |
            objNsc ~~ nil
              ifTrue: [
                updateIndexList
                  ifTrue: [ self addIndexListEntriesFor: objNsc completeBag index: indexObj ]
                  ifFalse: [ 
                    endObj ~~ incomplete
                      ifTrue: [ 
                        " only remove mappings once for a given object "
                        self
                          addMappingsForObject: endObj
                          root: rootObj
                          logging: false ] ].
                i \\ 100 == 0
                  ifTrue: [ indexManager commitIndexMaintenance: indexObj at: i ] ] ] ] ]
    ifFalse: [ 
      | object |
      "no set-valued terms"
      indexManager autoCommit
        ifTrue: [ anNSC _lockForIndexCreation ].
      indexObj firstPathTerm == self
        ifTrue: [ 
          " see if need to traverse to the first unshared path term "
          1 to: bag size do: [ :i | 
            object := bag _at: i.
            self addMappingsForObject: object logging: false.
            i \\ 100 == 0
              ifTrue: [ indexManager commitIndexMaintenance: indexObj at: i ] ] ]
        ifFalse: [ 
          1 to: bag size do: [ :i | 
            " traverse up to the point of the unshared path term "
            object := bag _at: i.
            self addMappingsForObject: object traverseUpTo: self for: indexObj logging: false.
            i \\ 100 == 0
              ifTrue: [ indexManager commitIndexMaintenance: indexObj at: i ] ] ] ].
  ^ anNSC
%
category: 'Testing'
method: GsSetValuedPathTerm
_isLastOccurrenceInIndexFor: anNsc

  ^ (self findReachableRootsFor: anNsc) size <= anNsc size
%
category: 'Testing'
method: GsPathTerm
_isLastOccurrenceInIndexFor: anObject

  ^ (self findReachableRootsFor: anObject) size <= 1
%
category: 'Testing'
method: GsEnumeratedPathTerm
_isLastOccurrenceInIndexFor: anObject

  ^ (self findReachableRootsFor: anObject) size <= 1
%
category: 'Collection Based Modification'
method: GsSelectorPathTerm
findRootObjectsFor: anObject
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | rootObjects nextObj |
  rootObjects := IdentityBag new.
  nextObj := self _nextObjectFor: anObject.
  updateBtree ~~ nil
    ifTrue: [ 
      "btree exists for receiver, collect the root objects, where anObject is the key and nextObj
       is the value."
      rootObjects := self findRootObjectsForKey: nextObj value: anObject ].
  ^ rootObjects
%
category: 'Collection Based Modification'
method: GsSelectorPathTerm
findRootObjectsFor: anObject oldValue: oldValue
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  ^self findRootObjectsFor: anObject
%

category: 'Collection Based Modification'
method: GsSelectorPathTerm
findRootObjectMaps: rootObjectMap for: anObject
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | rootObjects nextObj |
  rootObjects := IdentityBag new.
  nextObj := self _nextObjectFor: anObject.
  updateBtree ~~ nil
    ifTrue: [ 
      "btree exists for receiver, collect the root objects, where anObject is the key and nextObj
       is the value."
      rootObjects := self findRootObjectMaps: rootObjectMap forKey: nextObj value: anObject ].
  ^ rootObjects
%
category: 'Collection Based Modification'
method: GsSelectorPathTerm
findRootObjectMaps: rootObjectMap for: anObject oldValue: oldValue
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  ^self findRootObjectMaps: rootObjectMap for: anObject
%
category: 'Modification'
method: GsPathTerm
findReachableRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue sanity: sanity
  | traversalRootObjects |
  self findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue.
  traversalRootObjects := rootObjectMap resolveTraversalRootObjectsFor: anObject.
  ^ traversalRootObjects
%
category: 'Collection Based Modification'
method: GsPathTerm
findReachableRootObjectMapsFor: anObject 
  | rootObjectMap indexObj reachableRoots rootObjects |
  rootObjectMap := BtreePlusRootObjectMap new
    pivotObject: anObject;
    pivotPathTerm: self;
    yourself.
  self findRootObjectMaps: rootObjectMap for: anObject.
  rootObjects := rootObjectMap resolveTraversalRootObjectsFor: anObject.
  indexObj := self at: self size.
  reachableRoots := IdentityBag new.
  rootObjects asIdentitySet do: [:root | 
    (indexObj traverseAllWithParents: root upTo: self startingAt: 1)
      do: [ :ar | | intermediateObjectReachableFromRoot |
         intermediateObjectReachableFromRoot := ar at: 2.
        intermediateObjectReachableFromRoot == anObject
          ifTrue: [ reachableRoots add: root withOccurrences: (rootObjects occurrencesOf: root) ] ] ].
  ^ reachableRoots
%
category: 'Modification'
method: GsPathTerm
findReachableRootObjectMapsFor: anObject ivOffset: ivOffset oldValue: oldValue
  | rootObjectMap |
  rootObjectMap := BtreePlusRootObjectMap new
    pivotObject: anObject;
    pivotPathTerm: self;
    yourself.
  ^ self 
    findReachableRootObjectMaps: rootObjectMap 
    for: anObject 
    ivOffset: ivOffset 
    oldValue: oldValue 
    sanity: false
%
category: 'Collection Based Modification'
method: GsPathTerm
findRootObjectMaps: rootObjectMap for: anObject
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject.
  ivOffset ifNil: [ self _invalidIvOffset: anObject ].
  self findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset
%
category: 'Collection Based Modification'
method: GsPathTerm
findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject at is a key or parent of a key."

  | nextObj |
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
  nextObj
    ifNil: [ 
      | indexObj |
      indexObj := self at: self size.
      nilOnPath do: [:nilOnPathRoot |
        indexObj 
          traverse: nilOnPathRoot
          parent: nil
          thru: self
          startingAt: 1
          do: [ :term :parent :child | 
            (rootObjectMap traversalMap at: child ifAbsentPut: [ IdentityBag new ]) add: parent  ] ] ].
  updateBtree ~~ nil
    ifTrue: [ 
      "btree exists for receiver, collect the root objects, where anObject is the key and nextObj
       is the value."
      self findRootObjectMaps: rootObjectMap forKey: nextObj value: anObject ].
  (nil == nextObj or: [ self children isEmpty ])
    ifTrue: [ ^ self ].
  1 to: children size do: [:i | 
    (children at: i) findRootObjectMaps: rootObjectMap for: nextObj ]
%
category: 'Collection Based Modification'
method: GsPathTerm
findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject at is a key or parent of a key."

  | nextObj |
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
  oldValue
    ifNil: [ 
      (rootObjectMap traversalMap includesKey: anObject)
        ifFalse: [ 
          | indexObj pivotObject  nscRoot |
          indexObj := self at: self size.
          nscRoot := indexObj nscRoot.
          pivotObject := rootObjectMap pivotObject.
          nilOnPath asIdentitySet do: [:nilOnPathRoot |
            "nilOnPath collection may have more entries than exist in the nscRoot - set valued 
               path terms, but we are only interested in the root object counts for nscRoot."
            indexObj 
              traverse: nilOnPathRoot
              parent: nil
              thru: self
              startingAt: 1
              do: [ :term :parent :child |
                (nscRoot occurrencesOf: nilOnPathRoot) timesRepeat: [ (rootObjectMap traversalMap at: child ifAbsentPut: [ IdentityBag new ]) add: parent ] ].
            pivotObject == nilOnPathRoot
              ifTrue: [ (nscRoot occurrencesOf: nilOnPathRoot) timesRepeat: [ (rootObjectMap traversalMap at: nilOnPathRoot ifAbsentPut: [ IdentityBag new ]) add: nilOnPathRoot ] ] ] ] ].
  nextObj == oldValue ifFalse: [ self error: 'unexpected error: oldValue and nextObj should be identical by definition.' ].
  updateBtree ~~ nil
    ifTrue: [ 
      "btree exists for receiver, collect the root objects, where anObject is the key and nextObj
       is the value."
      self findRootObjectMaps: rootObjectMap forKey: nextObj value: anObject ].
  (nil == nextObj or: [ self children isEmpty ])
    ifTrue: [ ^ self ].
  1 to: children size do: [:i | 
    (children at: i) findRootObjectMaps: rootObjectMap for: nextObj oldValue: oldValue ]
%
category: 'Collection Based Modification'
method: GsPathTerm
findRootObjectMaps: rootObjectMap for: anObject oldValue: oldValue
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject.
  ivOffset ifNil: [ self _invalidIvOffset: anObject ].
  self 
      findRootObjectMaps: rootObjectMap
      for: anObject 
      ivOffset: ivOffset  
      oldValue: (self _nextObjectFor: oldValue atInstVar: ivOffset).
%
category: 'Modification'
method: GsPathTerm
findRootObjectMaps: rootObjectMap forKey: aKey value: aValue
  | indexRootObjectMap  |
  indexRootObjectMap := rootObjectMap copy.
  updateBtree
    findRootObjectMaps: indexRootObjectMap
    pathTerm: self
    key: aKey
    value: aValue.
  rootObjectMap mergeFromCopy: indexRootObjectMap.
%
category: 'Modification'
method: GsEnumeratedPathTerm
findReachableRootObjectMapsFor: anObject ivOffset: ivOffset oldValue: oldValue

  | indexObj factor theFactor rootObjectMap traversalRootObjects traversalReachableRoots |
  rootObjectMap := BtreePlusRootObjectMap new
    pivotObject: anObject;
    pivotPathTerm: self;
    yourself.
  self findReachableRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue sanity: false.
  traversalRootObjects := rootObjectMap resolveTraversalRootObjectsFor: oldValue.
  indexObj := self at: self size.
factor := rootObjectMap cumulativeFactor.
  (traversalRootObjects includes: anObject)
    ifTrue: [ theFactor := (traversalRootObjects occurrencesOf: anObject) ]
    ifFalse: [ 
      | enumeratedValues |
      enumeratedValues := IdentityBag new.
      self nextObj: anObject do: [:nextObj |
        enumeratedValues add: nextObj].
      theFactor := (enumeratedValues occurrencesOf: oldValue) ].
  theFactor ~~ 1
    ifTrue: [ factor := factor + theFactor ].
factor == 0 ifTrue: [ ^ traversalRootObjects ].
  traversalReachableRoots := IdentityBag new.
  traversalRootObjects asIdentitySet do: [:rootObject |
    | num |
    theFactor := factor + (indexObj cumulativeFactorFor: rootObject upTo: self startingAt: 1).
    num := traversalRootObjects occurrencesOf: rootObject.
    traversalReachableRoots add: rootObject withOccurrences: (num / theFactor) ceiling ].
  ^ traversalReachableRoots
%
category: 'Collection Based Modification'
method: GsEnumeratedPathTerm
findRootObjectMaps: rootObjectMap for: anObject
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  (rootObjectMap traversalMap includesKey: anObject)
    ifTrue: [
      "only way for the key to be present, is that to have come through this pathTerm/object before.
       And that can happen if one or more of the enumerated terms has identical values"
      ^ self  ].
  self pathTermElements
    do: [ :pathName | 
      "Make a full pass for each of the enumerated instance variables"
      | ivOffset |
      ivOffset := self _ivOffsetFor: anObject pathName: pathName.
      ivOffset ifNil: [ self _invalidIvOffset: anObject ].
      self findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset]
%
category: 'Collection Based Modification'
method: GsEnumeratedPathTerm
findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset

  (rootObjectMap traversalMap includesKey: anObject)
    ifTrue: [
      "only way for the key to be present, is that to have come through this pathTerm/object before.
       And that can happen if one or more of the enumerated terms has identical values"
      ^ self  ].
  super findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset
%
category: 'Collection Based Modification'
method: GsEnumeratedPathTerm
findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue

  (rootObjectMap traversalMap includesKey: anObject)
    ifTrue: [
      "only way for the key to be present, is that to have come through this pathTerm/object before.
       And that can happen if one or more of the enumerated terms has identical values"
      ^ self  ].
  super findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue
%
category: 'Collection Based Modification'
method: GsEnumeratedPathTerm
findRootObjectMaps: rootObjectMap for: anObject oldValue: oldValue
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."
| enumeratedValues factor |
  enumeratedValues := IdentityBag new.
  self nextObj: anObject do: [:nextObj |
    enumeratedValues add: nextObj].
factor := 0.
  self pathTermElements
    do: [ :pathName | 
      "Make a full pass for each of the enumerated instance variables"
      | ivOffset nextOldValue |
      ivOffset := self _ivOffsetFor: anObject pathName: pathName.
      ivOffset ifNil: [ self _invalidIvOffset: anObject ].
      nextOldValue := self _nextObjectFor: oldValue atInstVar: ivOffset.
      factor := factor + (enumeratedValues occurrencesOf: nextOldValue)].
factor > 1 ifTrue: [
rootObjectMap cumulativeFactor: (factor + (rootObjectMap cumulativeFactor))].
  (rootObjectMap traversalMap includesKey: anObject)
    ifTrue: [
      "only way for the key to be present, is that to have come through this pathTerm/object before.
       And that can happen if one or more of the enumerated terms has identical values"
      ^ self  ].
  self pathTermElements
    do: [ :pathName | 
      "Make a full pass for each of the enumerated instance variables"
      | ivOffset nextOldValue |
      ivOffset := self _ivOffsetFor: anObject pathName: pathName.
      ivOffset ifNil: [ self _invalidIvOffset: anObject ].
      nextOldValue := self _nextObjectFor: oldValue atInstVar: ivOffset.
      self 
         findRootObjectMaps: rootObjectMap 
         for: anObject 
         ivOffset: ivOffset  
         oldValue: nextOldValue ].
%
category: 'Collection Based Modification'
method: GsSetValuedPathTerm
findRootObjectMaps: rootObjectMap for:  anNsc
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  anNsc
    do: [ :nextObj | 
      "Make a full pass for each of the enumerated instance variables"
      updateBtree ~~ nil
        ifTrue: [ 
          "btree exists for receiver, collect the root objects, where nextObj is the key and anNsc
           is the value."
          self findRootObjectMaps: rootObjectMap forKey: nextObj value: anNsc ].
      (self children isEmpty)
        ifTrue: [ ^ self ].
      1 to: children size do: [:i | 
        (children at: i) findRootObjectMaps: rootObjectMap for:  nextObj] ].
%
category: 'Enumerating'
method: BtreePlusTraversalMapDictionary
keysAndValuesDo: aBlock

"Evaluates aBlock with each of the receiver's key/value pairs as the
 arguments.  The argument aBlock must be a two-argument block.  The
 first argument is the key and the second argument is the value of
 each key/value pair."

| aKey collisionBkt |

1 to: tableSize * 2 by: 2 do: [ :tableIndex |
  (aKey := self _at: tableIndex) ifNil: [
    (collisionBkt := self _at: (tableIndex + 1)) ifNotNil: [
      1 to: collisionBkt _basicSize by: 2 do: [ :j |
        (aKey := collisionBkt _at: j) ifNotNil: [
          aKey == self ifTrue: [ aKey := nil ].
          aBlock value: aKey value: (collisionBkt _at: j + 1).
	].
      ].
    ].
  ] ifNotNil: [
    aKey == self ifTrue: [ aKey := nil ].
    aBlock value: aKey value: (self _at: tableIndex + 1).
  ].
].
%
category: 'Removing'
method: BtreePlusTraversalMapDictionary
removeKey: aKey ifAbsent: aBlock
  aKey ifNil: [ ^ super removeKey: self ifAbsent: aBlock ].
  ^ super removeKey: aKey ifAbsent: aBlock
%
category: 'Collection Based Modification'
method: GsSetValuedPathTerm
findRootObjectMaps: rootObjectMap for: anNsc oldValue: oldValue
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  anNsc
    do: [ :nextObj | 
      "Make a full pass for each of the enumerated instance variables"
      updateBtree ~~ nil
        ifTrue: [ 
          "btree exists for receiver, collect the root objects, where nextObj is the key and anNsc
           is the value."
          self findRootObjectMaps: rootObjectMap forKey: nextObj value: anNsc ].
      (self children isEmpty)
        ifTrue: [ ^ self ].
      1 to: children size do: [:i | 
        (children at: i) findRootObjectMaps: rootObjectMap for:  nextObj] ].
%
category: 'Error Handling'
method: GsPathTerm
_errorObjectNotInBtree: key value: value root: root
  "An entry for the key/value/root was not present in the B-tree."

  ImproperOperation new
    args:
        {key.
          value.
          root};
    _number: (ErrorSymbols at: #'rtErrRangeEqualityIndexObjectNotInBtree');
    signal:
        'An entry for the key/value/root tuple (' , key asOop printString , '/'
            , value asOop printString , '/' , root asOop printString
            , ') was not present in the index [', updateBtree asOop printString, ']'
%
category: 'Removing'
method: GsSelectorPathTerm
removeMappingsFor: anObject root: rootObject lastOne: aBoolean logging: doLogging
  "Remove entries in the btree for anObject."

  | nextObj more isLast |
  self size == 0
    ifTrue: [ ^ self ].
  nextObj := self _nextObjectFor: anObject.
  updateBtree ~~ nil
    ifTrue: [ 
      " remove an entry from the B-tree "
      (updateBtree btreeRemoveKey: nextObj value: anObject root: rootObject)
        ifFalse: [ self _errorObjectNotInBtree: nextObj value: anObject root: rootObject ] ].
  more := false.
  nextObj ifNil: [ self removeNilOnPathForRoot: rootObject ].
  (nil ~~ nextObj and: [ children isEmpty not ])
    ifTrue: [ 
      aBoolean
        ifTrue: [
          | last |
          last := (children at: 1) _isLastOccurrenceInIndexFor: nextObj.
          isLast := aBoolean and: [ last ] ]
        ifFalse: [ isLast := false ].
          " make recursive call to remove mappings "
      1 to: children size do: [ :i | 
        (children at: i) removeMappingsFor: nextObj root: rootObject lastOne: isLast logging: doLogging ] ]
%
category: 'Private'
method: GsPathTerm
_errorInvalidOffset: anObject
  "If missing path slots are tolerated, return an empty MappingInfo. Otherwise 
   return an error array"

  (LookupError new
    _number:(ErrorSymbols at: #'rtErrObjectInvalidOffset');
    args: {anObject. name printString}) signal
%
category: 'Removing'
method: GsSetValuedPathTerm
removeMappingsFor: anNsc root: rootObject lastOne: aBoolean logging: doLogging
  "Remove entries in the btree and dependency lists for anNsc."

  | depList isLast |
  self size == 0
    ifTrue: [ ^ self ].
  anNsc
    do: [ :setElement | 
      updateBtree ~~ nil
        ifTrue: [ 
          " remove an entry from the B-tree "
          (updateBtree btreeRemoveKey: setElement value: anNsc root: rootObject)
            ifFalse: [ self _errorObjectNotInBtree: setElement value: anNsc root: rootObject ].
          needsDepList
            ifTrue: [ 
              " remove dependency list entry for next object " 
              DependencyList removePathTerm: self for: setElement ] ].
      (nil ~~ setElement and: [ children isEmpty not ])
        ifTrue: [ 
          aBoolean
            ifTrue: [
              | last |
              last := (children at: 1) _isLastOccurrenceInIndexFor: setElement.
              isLast := aBoolean and: [ last ] ]
            ifFalse: [ isLast := false ].
          " make recursive call to remove mappings "
          1 to: children size do: [ :i | 
            (children at: i) removeMappingsFor: setElement root: rootObject lastOne: isLast logging: doLogging ].
          isLast
            ifFalse: [
              "check again in case there are no more references left in btree"
              isLast := ((children at: 1) findReachableRootsFor: setElement) isEmpty ].
          isLast
            ifTrue: [ 
              " remove dependency list entries for next object "
              (depList := DependencyList for: setElement) ~~ nil
                ifTrue: [ depList removePathTerms: children for: setElement logging: doLogging ] ] ] ]
%
category: 'Adding'
method: GsSelectorPathTerm
addDirectMappingFor: anObject root: rootObject nsc: anNsc logging: aBoolean
  " Update the B-tree and the dependency list for anObject."

  updateBtree ~~ nil
    ifTrue: [ updateBtree btreeAt: anObject put: anObject root: rootObject ].
%
category: 'Adding'
method: GsPathTerm
addDirectMappingFor: anObject root: rootObject nsc: anNsc logging: aBoolean
  " Update the B-tree and the dependency list for anObject."

  updateBtree ~~ nil
    ifTrue: [ updateBtree btreeAt: anObject put: anObject root: rootObject ].
  needsDepList
    ifTrue: [ " add an entry in the dependency list "
      anObject getDepListAndAddLastElementPathTerm: self logging: aBoolean ]
%
category: 'Removing'
method: GsPathTerm
removeDirectMappingFor: anObject root: root nsc: anNsc logging: doLogging
  "There is a range index directly on the elements of the NSC.  Update the
 B-tree."

  updateBtree ~~ nil
    ifTrue: [ " remove it from the B-tree "
      (updateBtree btreeRemoveKey: anObject value: anObject root: root)
         ifFalse: [ self _errorObjectNotInBtree: anObject value: anObject root: root ] ]
%
category: 'Adding'
method: GsSetValuedPathTerm
addDirectMappingFor: anObject root: rootObject nsc: anNsc logging: aBoolean
  " Update the B-tree and the dependency list for anObject."

  updateBtree ~~ nil
    ifTrue: [ updateBtree btreeAt: anObject put: anNsc root: rootObject ].
  needsDepList
    ifTrue: [ " add an entry in the dependency list "
      anObject getDepListAndAddLastElementPathTerm: self logging: aBoolean ]
%
category: 'Modification'
method: GsSetValuedPathTerm
findRootObjectsForKey: aKey value: aValue
  ^ updateBtree findRootObjectsComparingForKey: aKey value: aValue
%
category: 'Removing'
method: GsSetValuedPathTerm
removeDirectMappingFor: anObject root: root nsc: anNsc logging: doLogging
  "There is a range index directly on the elements of the NSC.  Update the
 B-tree."

  updateBtree ~~ nil
    ifTrue: [ " remove it from the B-tree "
      (updateBtree btreeRemoveKey: anObject value: anNsc root: root)
         ifFalse: [ self _errorObjectNotInBtree: anObject value: anNsc root: root ] ]
%
category: 'Reduced Conflict Support'
method: GsPathTerm
_selectiveAbort

 "Do nothing"
%
category: 'Formatting'
method: GsPathTerm
asString
  ^ super asString , ' on ' , self name asString printString
%
category: 'Clustering'
method: GsPathTerm
clusterDepthFirst

"Cluster the receiver."

name cluster.
securityPolicies cluster.
children cluster.
1 to: children size do: [ :i | (children at: i) cluster ]
%
category: 'Accessing'
method: GsPathTerm
getRootNsc

"Returns the root NSC for the index in which the receiver is a term."

^ (self at: 1) nscRoot
%
category: 'Audit'
method: GsPathTerm
auditNscForRootObj: obj rootObj: rootObj using: auditor
  "Private. "

  | nextObj optionalSentinel |
  self size == 0
    ifTrue: [ ^ self ].
  self indicatesIndexOnNscElements
    ifTrue: [ "^ self auditDirectNscRootObj: obj using: auditor" self error: 'not yet implemented' ].
  nil == obj
    ifTrue: [ ^ self ].
  optionalSentinel := Object new.
  nextObj := self
    auditDepListFor: obj
    index: nil
    using: auditor
    optionalSentinel: optionalSentinel.
  (nextObj == optionalSentinel or: [ nextObj == nil ])
    ifTrue: [ ^ self ].
  1 to: self size do: [ :i | | indexObj |
    indexObj := self at: i.
    indexObj size == offset
      ifTrue: [ | col found expected | 
        col := IdentityBag new.
        indexObj btreeRoot btreeRootNode _findAndCountKey: nextObj value: obj root: rootObj into: col.
        (found := col size) == (expected := indexObj nscRoot occurrencesOf: rootObj)
          ifFalse: [
            auditor 
              btree: indexObj btreeRoot 
              incorrectCountForKeyValueRootPairs: self 
              key: nextObj 
              value: obj  
              root: rootObj 
              found: found 
              expected: expected ] ] ].
  1 to: children size do: [ :i | (children at: i)
      auditNscForRootObj: nextObj
      rootObj: rootObj
      using: auditor ]
%
category: 'accessing'
method: BtreePlusRootObjectMap
cumulativeFactor

   ^cumulativeFactor
%
category: 'accessing'
method: BtreePlusRootObjectMap
cumulativeFactor: anObject

   cumulativeFactor := anObject
%
category: 'initialization'
method: BtreePlusRootObjectMap
initialize
  roots := BtreePlusTraversalMapDictionary new.
  cumulativeFactor := 0.
  traversalMap := BtreePlusTraversalMapDictionary new.
%
category: 'instance creation'
classmethod: BtreePlusRootObjectMap
new
  ^ self basicNew
    initialize;
    yourself
%
category: 'accessing'
method: BtreePlusRootObjectMap
pivotObject

   ^pivotObject
%
category: 'accessing'
method: BtreePlusRootObjectMap
pivotObject: anObject

   pivotObject := anObject
%
category: 'accessing'
method: BtreePlusRootObjectMap
pivotPathTerm

   ^pivotPathTerm
%
category: 'accessing'
method: BtreePlusRootObjectMap
pivotPathTerm: anObject

   pivotPathTerm := anObject
%
category: 'accessing'
method: BtreePlusRootObjectMap
roots

   ^roots
%
category: 'accessing'
method: BtreePlusRootObjectMap
roots: anObject

   roots := anObject
%
category: 'copying'
method: BtreePlusRootObjectMap
mergeFromCopy: aRootObjectMap
  | newBag | 
  aRootObjectMap roots keysAndValuesDo: [ :rootObj :rootIndexBag |
    (self roots at: rootObj otherwise: nil)
      ifNotNil: [ :oldBag | 
        newBag := rootIndexBag _union: oldBag ]
      ifNil: [ 
        newBag := rootIndexBag ].
    self roots at: rootObj put: newBag ].
  aRootObjectMap traversalMap keysAndValuesDo: [ :obj :indexBag |
    (traversalMap at: obj otherwise: nil)
      ifNotNil: [ :oldBag | 
        newBag := indexBag _union: oldBag ]
      ifNil: [ 
        newBag := indexBag ].
    self traversalMap at: obj put: newBag ]
%
category: 'copying'
method: BtreePlusRootObjectMap
postCopy
  roots := BtreePlusTraversalMapDictionary new.
  traversalMap := BtreePlusTraversalMapDictionary new
%
category: 'resolving'
method: BtreePlusRootObjectMap
resolveTraversalRootObjectsFor: anObject
  ^ self 
    resolveTraversalRootObjectsFor: anObject
    level: self pivotPathTerm offset 
    into: IdentityBag new 
    visited: IdentitySet new.
%
category: 'resolving'
method: BtreePlusRootObjectMap
resolveTraversalRootObjectsFor: anObject level: level into: defaultRoots visited: visited

  | objs bag currentLevel |
  level = 0
    ifTrue: [ ^ defaultRoots ].
  bag := self traversalMap at: anObject ifAbsent: [ ^ self roots at: anObject ifAbsent: [defaultRoots]].
  objs := IdentityBag new.
  currentLevel := level.
  bag
    do: [ :obj | 
      (visited includes: obj)
        ifFalse: [ 
          obj class isNsc
            ifTrue: [ currentLevel := level + 1 ].
          visited add: obj.
          objs := objs
            _union:
              (self
                resolveTraversalRootObjectsFor: obj
                level: currentLevel - 1
                into: bag
                visited: visited) ] ].
  ^ objs
%
category: 'accessing'
method: BtreePlusRootObjectMap
traversalMap

   ^traversalMap
%
category: 'accessing'
method: BtreePlusRootObjectMap
traversalMap: anObject

   traversalMap := anObject
%
category: 'Modification'
method: GsPathTerm
findAllReachableNilOnPathRoots

  | reachableRoots |
  reachableRoots := IdentityBag new.
  reachableRoots addAll: nilOnPath.
  1 to: children size do: [:i |
    reachableRoots addAll: (children at: i) findAllReachableNilOnPathRoots ].
  ^ reachableRoots
%
category: 'Audit'
method: GsSetValuedPathTerm
auditNscForRootObj: anNsc rootObj: rootObj using: auditor
  "Private. "

  self size == 0
    ifTrue: [ ^ self ].
  self indicatesIndexOnNscElements
    ifTrue: [ "^ self auditDirectNscRootObj: obj using: auditor" self error: 'not yet implemented' ].
  nil == anNsc
    ifTrue: [ ^ self ].
self nextObj: anNsc do: [:nextObj | 
  1 to: self size do: [ :i | | indexObj |
    indexObj := self at: i.
    indexObj size == offset
      ifTrue: [ | col found expected | 
        col := IdentityBag new.
        indexObj btreeRoot btreeRootNode _findAndCountKey: nextObj value: anNsc root: rootObj into: col.
        (found := col size) == (expected := indexObj nscRoot occurrencesOf: rootObj)
          ifFalse: [
            auditor 
              btree: indexObj btreeRoot 
              incorrectCountForKeyValueRootPairs: self 
              key: nextObj 
              value: anNsc  
              root: rootObj 
              found: found 
              expected: expected ] ] ].
  1 to: children size do: [ :i | (children at: i)
      auditNscForRootObj: nextObj
      rootObj: rootObj
      using: auditor ]
]
%
