! ========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
! ========================================================================

set compile_env: 0
! ------------------- Class definition for CDeclaration
expectvalue /Class
doit
Object subclass: 'CDeclaration'
  instVarNames: #( header name storage
                    linkageSpec type count pointer
                    fields parameters enumTag isConstant
                    includesCode isVaryingArgCount bitCount source
                    file line)
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals
  options: #()

%
expectvalue /Class
doit
CDeclaration comment: 
'This class is internal to the FFI implementation and 
represents a declaration from a C header file.'
%
expectvalue /Class
doit
CDeclaration category: 'FFI'
%
! ------------------- Remove existing behavior from CDeclaration
expectvalue /Metaclass3       
doit
CDeclaration removeAllMethods.
CDeclaration class removeAllMethods.
%
set compile_env: 0
! ------------------- Class methods for CDeclaration
category: 'other'
classmethod: CDeclaration
header: aHeader
	"Return a CDeclaration (might not be self)"

	^(self readTypeFrom: aHeader) readDeclaration
%
category: 'other'
classmethod: CDeclaration
readTypeFrom: aHeader
	"Return a CDeclaration (might not be self)"

	^self basicNew readTypeFrom: aHeader
%
! ------------------- Instance methods for CDeclaration
category: 'CByteArray get'
method: CDeclaration
accessorForOffset: anInteger

	1 < count size ifTrue: [ self error: 'Multi-dimensional arrays not supported']. 
	0 == pointer ifTrue: [
		type == #'class' ifTrue: [^self structAccessorForOffset: anInteger].
		(type isKindOf: CDeclaration) ifTrue: [^self structAccessorForOffset: anInteger].
		count == nil ifTrue: [^self simpleAccessorForOffset: anInteger].
		type == #'uint8' ifTrue: [^self stringAccessorForOffset: anInteger].
		^self arrayAccessorForOffset: anInteger.
	].
	1 < pointer ifTrue: [self error: 'Multi-level pointers not supported'].
	count == nil ifTrue: [^self singlePointerAccessorForOffset: anInteger].
	type == #'uint8' ifTrue: [^self stringPointerAccessorForOffset: anInteger].
	^self arrayPointersAccessorForOffset: anInteger.
%
category: 'CCallout'
method: CDeclaration
addSourceTo: aStream
  | fn |
  file ifNil:[ 
    fn := 'test'
  ] ifNotNil:[ 
    fn := file .
    fn size > 0 ifTrue:[ | gsIncl exp |
      gsIncl := GsFile _expandEnvVariable:'GEMSTONE' isClient: false .
      gsIncl := GsFile _expandFilename: gsIncl isClient: false . "follow symlinks"
      (gsIncl at: 1 equals: (exp := '/export')) ifTrue:[
         gsIncl := gsIncl copyFrom: exp size + 1  to: gsIncl size .
      ].
      gsIncl ifNotNil:[  | ofs |
        ofs := 1 .
        gsIncl last == $/ ifFalse:[ gsIncl add: $/ ].
        gsIncl add: 'include' .
        (fn at: 1 equals: exp) ifTrue:[ ofs := ofs + exp size ].
        (fn at: ofs  equals: gsIncl) ifTrue:[ 
          fn := '$GEMSTONE/include' , (fn copyFrom: gsIncl size + ofs  to: fn size)
        ].
        (System gemVersionReport at: 'gsBuildArchitecture') = 'Darwin (macOS)' ifTrue:[
          (fn at: 1 equals: '/Volumes') ifTrue:[ "Darwin"
            fn := fn copyFrom: '/Volumes' size + 1 to: fn size .
            (fn at: 1 equals: gsIncl) ifTrue:[ 
              fn := '$GEMSTONE/include' , (fn copyFrom: gsIncl size + 1 to: fn size)
            ].
          ].
        ]
      ].
    ].
  ].
	aStream 
		lf; tab; nextPut: $";
		nextPutAll: fn ;
		nextPutAll: ' line ';
		nextPutAll: line printString;
		lf;
		yourself.
	source trimSeparators do: [:char | aStream nextPut: char. char == $" ifTrue: [aStream nextPut: char]].
	aStream nextPut: $"; lf.
%
category: 'CCallout'
method: CDeclaration
argumentType

	(self isSimple and: [self knownArgumentTypes includesIdentical: type]) ifTrue: [^type].
	(type == #'uint8' and: [isConstant]) ifTrue: [
		(count size == 1 and: [pointer == 0]) ifTrue: [^#'const char*'].		"const char foo[]"
		(count == nil and: [pointer == 1]) ifTrue: [^#'const char*'].			"const char * foo"
	].
	type == #'enum' ifTrue: [^#'int32'].
	^#'ptr'.
%
category: 'CByteArray get'
method: CDeclaration
arrayAccessorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name; lf; lf; tab;
		nextPutAll: '| array offset |'; lf; tab;
		nextPutAll: 'array := Array new: ';
		nextPutAll: count first printString;
		nextPut: $.; lf; tab;
		nextPutAll: 'offset := ';
		nextPutAll: anInteger printString;
		nextPut: $.; lf; tab;
		nextPutAll: '1 to: array size do: [:i | '; lf; tab; tab;
		nextPutAll: 'array at: i put: (self ';
		nextPutAll: self getSelector;
		nextPutAll: ' offset).'; lf; tab; tab;
		nextPutAll: 'offset := offset + ';
		nextPutAll: self baseByteSize printString;
		nextPut: $.; lf; tab;
		nextPutAll: '].'; lf; tab;
		nextPutAll: '^array.'; lf;
		contents.
%
category: 'CByteArray get'
method: CDeclaration
arrayPointersAccessorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name; lf; lf; tab;
		nextPutAll: '| array offset |'; lf; tab;
		nextPutAll: 'array := Array new: ';
		nextPutAll: count first printString;
		nextPut: $.; lf; tab;
		nextPutAll: 'offset := ';
		nextPutAll: anInteger printString;
		nextPut: $.; lf; tab;
		nextPutAll: '1 to: array size do: [:i | '; lf; tab; tab;
		nextPutAll: 'array at: i put: (self pointerAt: offset resultClass: ';
		nextPutAll: self cByteArraySpecies name asString;
		nextPutAll: ').'; lf; tab; tab;
		nextPutAll: 'offset := offset + 8.'; lf; tab;
		nextPutAll: '].'; lf; tab;
		nextPutAll: '^array.'; lf;
		contents.
%
category: 'CByteArray put'
method: CDeclaration
arrayPointersUpdatorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name;
		nextPutAll: ': anArrayOfCByteArrayInstances'; lf; lf; tab;
		nextPutAll: '| offset |'; lf; tab;
		nextPutAll: 'offset := ';
		nextPutAll: anInteger printString;
		nextPut: $.; lf; tab;
		nextPutAll: '1 to: anArrayOfCByteArrayInstances size do: [:i | '; lf; tab; tab;
		nextPutAll: 'self'; lf; tab; tab; tab;
		nextPutAll: 'pointerAt: offset'; lf; tab; tab; tab; 
		nextPutAll: 'put: (anArrayOfCByteArrayInstances at: i).'; lf; tab; tab;
		nextPutAll: 'offset := offset + 8.'; lf; tab;
		nextPutAll: '].'; lf;
		contents.
%
category: 'CByteArray put'
method: CDeclaration
arrayUpdatorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name;
		nextPutAll: ': aSequenceableCollection'; lf; lf; tab;
		nextPutAll: '| offset |'; lf; tab;
		nextPutAll: 'offset := ';
		nextPutAll: anInteger printString;
		nextPut: $.; lf; tab;
		nextPutAll: '1 to: aSequenceableCollection size do: [:i | '; lf; tab; tab;
		nextPutAll: 'self '; lf; tab; tab; tab;
		nextPutAll: self getSelector; space;
		nextPutAll: 'offset'; lf; tab; tab; tab;
		nextPutAll: 'put: (aSequenceableCollection at: i).'; lf; tab; tab;
		nextPutAll: 'offset := offset + ';
		nextPutAll: self baseByteSize printString;
		nextPut: $.; lf; tab;
		nextPutAll: '].'; lf;
		contents.
%
category: 'CCallout'
method: CDeclaration
asClassVarName

	^'Function' , name.
%
category: 'other'
method: CDeclaration
atEnd

	^header atEnd.
%
category: 'CByteArray get'
method: CDeclaration
baseAccessorForOffset: anInteger

	^'self ' , self getSelector , ' ' , anInteger printString.
%
category: 'size'
method: CDeclaration
baseByteSize

	0 < pointer ifTrue: [^8].
	bitCount ifNotNil: [^bitCount].
	(type == #'int8' or: [type == #'uint8']) ifTrue: [^1].
	(type == #'int16' or: [type == #'uint16']) ifTrue: [^2].
	(type == #'int32' or: [type == #'uint32']) ifTrue: [^4].
	(type == #'int64' or: [type == #'uint64']) ifTrue: [^8].
	(type == #'void' and: [0 < pointer]) ifTrue: [^8].
	type == #'class' ifTrue: [^self baseStructByteSize].
	type == #'struct' ifTrue: [^self baseStructByteSize].
	type == #'union' ifTrue: [^self baseUnionByteSize].
	type == #'enum' ifTrue: [^4].
	type == #'function' ifTrue: [^8].
	type == #'double' ifTrue: [^8].
	(type isKindOf: CDeclaration) ifTrue: [^type baseByteSize].
	self halt.
%
category: 'size'
method: CDeclaration
baseStructByteSize

	| bits |
	bits := 0.
	fields do: [:each |  
		each bitCount ifNil: [
			bits := bits + (each byteSize * 8).
		] ifNotNil: [
			each count ifNotNil: [self error: 'Array of bits!?'].
			bits := bits + each bitCount.
		].
	].
	^(bits / 8) ceiling.
%
category: 'size'
method: CDeclaration
baseUnionByteSize

	^fields 
		inject: 0
		into: [:max :each | max max: each baseByteSize].
%
category: 'other'
method: CDeclaration
beExternStorage

	storage := #'extern'.
%
category: 'accessors'
method: CDeclaration
bitCount

	^bitCount.
%
category: 'size'
method: CDeclaration
byteSize

	| x |
	x := self baseByteSize.
	count ifNil: [^x].
	count do: [:y | x := x * y].
	^x.
%
category: 'size'
method: CDeclaration
byteSizeRounded
  "Return a size aligned on an 8 byte boundary, like a C compiler would do"
  | x pad rem | 
  x := self byteSize .
  pad := 0 .
  (rem := x \\ 8) ~~ 0 ifTrue:[
    pad := 8 - rem
  ].
  ^ x + pad
%
category: 'querying'
method: CDeclaration
canStorageBeMadeExternal

	^(#(#auto #inline #static) includesIdentical: storage)
		or: [storage = #extern and: [linkageSpec == nil ]]
%
category: 'CCallout'
method: CDeclaration
cCalloutForLibrary: aCLibrary

	^self cCalloutSpecies
		library: aCLibrary 
		name: name 
		result: self resultType 
		args: (parameters collect: [:each | each argumentType])
		varArgsAfter: (isVaryingArgCount ifTrue: [parameters size] ifFalse: [-1]).
%
category: 'Class Membership'
method: CDeclaration
cCalloutSpecies
"Answer the class to be used for CCallout objects. Subclasses may
 overload this method as needed."
^ CCallout
%
category: 'Class Membership'
method: CDeclaration
cDeclarationSpecies
"Answer the class to be used for CDeclaration objects. Subclasses may
 overload this method as needed."
^ CDeclaration
%
category: 'accessors'
method: CDeclaration
count

	^count.
%
category: 'Class Membership'
method: CDeclaration
cPreprocessorSpecies
"Answer the class to be used for CPreprocessor objects. Subclasses may
 overload this method as needed."
^ CPreprocessor
%
category: 'CCallout'
method: CDeclaration
createFunctionInitializerCode

	| stream space |
	stream := AppendStream on: String new.
	stream
		nextPutAll: self initializerFunctionName;
		nextPutAll: ' cLibrary'; lf; lf; tab;
		nextPutAll: self asClassVarName;
		nextPutAll: ' := ';
		nextPutAll: self cCalloutSpecies name asString; lf; tab; tab;
		nextPutAll: 'library: cLibrary'; lf; tab; tab;
		nextPutAll: 'name: ''';
		nextPutAll: name; 
		nextPut: $'; lf; tab; tab;
		nextPutAll: 'result: #''';
		nextPutAll: self resultType;
		nextPutAll: ''''; lf; tab; tab;
		nextPutAll: 'args: #(';
		yourself.
	space := ''.
	parameters do: [:each | 
		stream 
			nextPutAll: space;
			nextPutAll: each argumentType printString;
			yourself.
		space := ' '.
	].
	^stream
		nextPutAll: ')'; lf; tab; tab;
		nextPutAll: 'varArgsAfter: ';
		nextPutAll: self varArgsAfter printString;
		nextPut: $.; lf;
		contents.
%
category: 'querying'
method: CDeclaration
derivedType

	storage == #'typedef' ifFalse: [^self].
	type == #'class' ifTrue: [^self].
	self isFunction ifTrue: [^self].
	^type
%
category: 'accessors'
method: CDeclaration
enumList

	^type == #'enum'
		ifTrue: [fields]
		ifFalse: [nil].
%
category: 'accessors'
method: CDeclaration
enumTag

	^enumTag.
%
category: 'Errors'
method: CDeclaration
error: aString

  Error signal: aString , ' near line ' , line asString , ' in file ' , file asString
%
category: 'other'
method: CDeclaration
expectCloseParenthesisCount: anInteger

	anInteger timesRepeat: [
		| token |
		(token := self next) isCloseParenthesisToken ifFalse: [token error: 'Expected '')'' '].
	].
%
category: 'accessors'
method: CDeclaration
fields

	^type == #'struct'
		ifTrue: [fields]
		ifFalse: [nil].
%
category: 'accessors'
method: CDeclaration
file

	^file.
%
category: 'CByteArray get'
method: CDeclaration
getSelector

	(#(
		#'int8' #'int16' #'int32' #'int64'
		#'uint8' #'uint16' #'uint32' #'uint64'
	) includesIdentical: type) ifTrue: [^type , 'At:'].
	type == #'enum' ifTrue: [^'int32At:'].
	type == #'function' ifTrue: [^'uint64At:'].
	self halt.
%
category: 'accessors'
method: CDeclaration
header

	^header.
%
category: 'accessors'
method: CDeclaration
includesCode

	^includesCode.
%
category: 'type'
method: CDeclaration
initializeAsStructOrUnion

	| token |
	storage := #'auto'.
	(token := self peek) isOpenCurlyBracketToken ifFalse: [^self].
	self next.
	fields ifNotNil: [self error: 'Fields already defined!'].
	fields := { }.
	[
		(token := self peek) isCloseCurlyBracketToken not.
	] whileTrue: [
		| field |
		field := self class header: header.
		field storage == #'auto' ifFalse: [ token error: 'storage is not #auto'].
		[
			fields add: field.
			(token := self next) isCommaToken.
		] whileTrue: [
			field := field copy.
			field readDeclaration.
		].
		token isSemicolonToken ifFalse: [ token error: 'expected '';'' '].
	].
	self next.
%
category: 'CCallout'
method: CDeclaration
initializerFunctionName
	^'initializeFunction_', name, '_inLibrary:'
%
category: 'type'
method: CDeclaration
interpretTypeSpecifier

	type class == Array ifFalse: [^self].
	(type size == 1 and:[
   #(#'class' #'enum' #'float' #'float128' #'void') includesIdentical: type first]) 
     ifTrue: [type := type first. ^self].

	"https://en.cppreference.com/w/c/language/arithmetic_types#Complex_floating_types
	https://gcc.gnu.org/onlinedocs/gcc-4.8.0/gcc/Complex.html"
	(type includesIdentical: #'complex') ifTrue: [
		| string |
		type removeIdentical: #'complex'.
		self interpretTypeSpecifier.
		(type isKindOf: Symbol) ifFalse: [self error: 'Expected to be simplified to a single type'].
		string := String withAll: type.
		string at: 1 put: string first asUppercase.
		type := ('complex' , string) asSymbol.
		^self
	].

	(type includesIdentical: #'char') ifTrue: [
		type removeIdentical: #'char'.
		type = #(#'signed') ifTrue: [type := #'int8'. ^self].
		(type isEmpty or: [type = #(#'unsigned')]) ifTrue: [type := #'uint8'. ^self].
		self halt.
	].
	(type includesIdentical: #'double') ifTrue: [
		type removeIdentical: #'double'.
		type isEmpty ifTrue: [type := #'double'. ^self].
		type = #(#'long') ifTrue: [type := #'longDouble'. ^self].
		self halt.
	].
	(type isEmpty or: [#(#'int' #'unsigned' #'long' #'signed' #'short') 
                     anySatisfy: [:each | type includesIdentical: each]]) ifTrue: [
		| isUnsigned |
		(type includesIdentical: #'int') ifTrue: [type removeIdentical: #'int'].
		(isUnsigned := type includesIdentical: #'unsigned') ifTrue: [type removeIdentical: #'unsigned'].
		(type includesIdentical: #'signed') ifTrue: [type removeIdentical: #'signed'].
		type = #(#'long') ifTrue: [
			type := isUnsigned ifTrue: [#'uint64'] ifFalse: [#'int64'].
			^self.
		].
		type = #(#'long' #'long') ifTrue: [
			type := isUnsigned ifTrue: [#'uint64'] ifFalse: [#'int64'].
			^self.
		].
		type = #(#'short') ifTrue: [
			type := isUnsigned ifTrue: [#'uint16'] ifFalse: [#'int16'].
			^self.
		].
		type isEmpty ifTrue: [
			type := isUnsigned ifTrue: [#'uint32'] ifFalse: [#'int32'].
			^self.
		].
		self halt.
	].
	(type size == 1 and:[
     #(#'uint64' #'int64' #'uint32' #'int32' #'uint16' #'int16' #'uint8' #'int8') 
         includesIdentical: type first]) ifTrue: [
		type := type first.
		^self.
	].
	(type size == 1 and: [type first isKindOf: CDeclaration]) ifTrue: [type := type first. ^self].
	self halt.
%
category: 'CCallout'
method: CDeclaration
invokeFunctionCode

	| args stream delimiter |
	args := parameters collect: [:each | each name].
	1 to: args size do: [:i | 
		(args at: i) ifNil: [args at: i put: 'arg' , i printString].
	].
	stream := (AppendStream on: String new)
		nextPutAll: name;
		yourself.
	args do: [:each | 
		stream 
			nextPutAll: '_: ';
			nextPutAll: each;
			space;
			yourself.
	].
	self addSourceTo: stream.
	stream
		lf; tab; 
		nextPutAll: '"Interpreted as #';
		nextPutAll: self resultType;
		nextPutAll: ' from #(';
		yourself.
	parameters do: [:each | stream space. each argumentType printOn: stream].
	stream 
		nextPutAll: ' )';
		nextPut: $"; lf; lf; 
		tab; nextPut: $^;
		nextPutAll: self asClassVarName;
		nextPutAll: ' callWith: { ';
		yourself.
	delimiter := ''.
	args do: [:each | 
		stream 
			nextPutAll: delimiter;
			nextPutAll: each;
			yourself.
		delimiter := '. '.
	].
	isVaryingArgCount ifTrue: [stream nextPutAll: '"varArgs array should be appended to this array"'].
	^stream 
		nextPutAll: ' }'; lf;
		contents.
%
category: 'accessors'
method: CDeclaration
isConstant

	^isConstant.
%
category: 'querying'
method: CDeclaration
isEmptyExternalLinkage
	"Answer whether the receiver is the introduction of:
		extern ''C'' { ... }
	 Note that we also end up with this after reading the entire { ... } block."

	^self storage = #extern
		and: [linkageSpec ~~ nil 
		and: [self name == nil 
		and: [self type == nil ]]]
%
category: 'querying'
method: CDeclaration
isEmptyVoid

	^name == nil and: [
		storage == #'auto' and: [
		type == #'void' and: [
		count == nil and: [
		pointer == 0 and: [
		fields == nil and: [
		isConstant == false and: [
		self isFunction == false and: [
		bitCount == nil]]]]]]]].
%
category: 'querying'
method: CDeclaration
isFunction

	^parameters ~~ nil .
%
category: 'CCallout'
method: CDeclaration
isSimple

	^count == nil  and: [pointer = 0].
%
category: 'accessors'
method: CDeclaration
isVaryingArgCount

	^isVaryingArgCount.
%
category: 'CCallout'
method: CDeclaration
knownArgumentTypes

	^#( #int64 #uint64 #int32 #uint32 #int16 #uint16 #int8 #uint8 #bool #double #float #ptr #'char*' "#'&ptr'" #'const char*' ).
%
category: 'CCallout'
method: CDeclaration
knownResultTypes

	^#( #int64 #uint64 #int32 #uint32 #int16 #uint16 #int8 #uint8 #bool #double #float #ptr #'char*' #void ).
%
category: 'accessors'
method: CDeclaration
line

	^line.
%
category: 'accessors'
method: CDeclaration
linkageSpec

	^linkageSpec.
%
category: 'declarator'
method: CDeclaration
mayHaveCode

	storage == #'static' 	ifTrue: [^true].
	storage == #'inline' 	ifTrue: [^true].
	type == #'class'		ifTrue: [^true].
	((type isKindOf: CDeclaration) and: [type type == #'class']) ifTrue: [^true].
	^false
%
category: 'accessors'
method: CDeclaration
members

	^type == #'union'
		ifTrue: [fields]
		ifFalse: [nil].
%
category: 'accessors'
method: CDeclaration
name

	^name.
%
category: 'other'
method: CDeclaration
next
  | res |
	res := self peek ; _next .
  ^ res
%
category: 'accessors'
method: CDeclaration
parameters

	^parameters.
%
category: 'other'
method: CDeclaration
peek

	| next wasIdentifierInSystemNamespace |
	wasIdentifierInSystemNamespace := false.
	[
		| wasEmpty |
		next := header peek.
		next ifNil:[ ^ nil ].
		wasEmpty := next isEmptyToken.
		wasEmpty ifFalse: [
			| value |
			(wasIdentifierInSystemNamespace and: [next isOpenParenthesisToken]) ifTrue: [
				self skipFunction.
				^self peek.
			] ifFalse: [
				wasIdentifierInSystemNamespace := next isIdentifierToken and: [2 < (value := next value) size and: 
					[(value at: 1) == $_ and: [(value at: 2) == $~]]].
			].
		].
		wasEmpty or: [wasIdentifierInSystemNamespace].
	] whileTrue: [
		self _next.
	].
	^ next
%
category: 'accessors'
method: CDeclaration
pointer

	^pointer.
%
category: 'other'
method: CDeclaration
printFunctionParametersOn: aStream

	| comma |
	(name == nil  or: [name last ~~ $(]) ifTrue: [
		aStream nextPut: $(.
	].
	(parameters == nil  or: [parameters isEmpty]) ifTrue: [aStream nextPutAll: 'void)'. ^self].
	comma := ''.
	parameters do: [:each | 
		aStream nextPutAll: comma.
		each printOn: aStream.
		comma := ', '.
	].
	aStream nextPut: $).
%
category: 'other'
method: CDeclaration
printOn: aStream

	storage ifNotNil: [
		(storage == #'auto' or: [storage == #'typedef' and: [type == #'class']]) ifFalse: [aStream nextPutAll: storage; space].
		linkageSpec ifNotNil:[ aStream nextPut: $"; nextPutAll: linkageSpec; nextPut: $"; space].
	].
	isConstant ifTrue: [aStream nextPutAll: 'const '].
	((type isKindOf: CDeclaration) and: [type isFunction]) ifTrue: [
		type printOn: aStream from: self.
		^self.
	].
	type ~= #() ifTrue: [
		(type isKindOf: Symbol)
			ifTrue: [aStream nextPutAll: type]
			ifFalse: [type printOn: aStream]. 
		aStream space.
	].
	pointer timesRepeat: [aStream nextPut: $*].
	enumTag ifNotNil: [aStream nextPutAll: enumTag; space].
	(storage == #'typedef' and: [self isFunction and: [0 < pointer]]) ifTrue: [aStream nextPut: $(].
	name ifNotNil: [aStream nextPutAll: name].
	count ifNotNil: [
		count do: [:each | 
			aStream nextPut: $[.
			0 < each ifTrue: [each printOn: aStream].
			aStream nextPut: $].
		].
	].
	(storage == #'typedef' and: [self isFunction and: [0 < pointer]]) ifTrue: [aStream nextPut: $)].
	self isFunction ifTrue: [self printFunctionParametersOn: aStream].
%
category: 'other'
method: CDeclaration
printOn: aStream from: aCDeclaration

	| open close |
	(type isKindOf: Symbol) ifTrue: [
		aStream nextPutAll: type.
	] ifFalse: [
		aStream nextPutAll: '<???>'.
	].
	aStream space.
	0 == aCDeclaration pointer ifTrue: [
		open := ''.
		close := ''.
	] ifFalse: [
		open := '('. 
		close := ')'.
	].
	pointer timesRepeat: [aStream nextPut: $*].
	aStream nextPutAll: open.
	aCDeclaration pointer timesRepeat: [aStream nextPut: $*].
	(name ~~ nil  and: [storage ~~ #'typedef']) ifTrue: [aStream nextPutAll: name].
	aStream nextPutAll: close.
	self printFunctionParametersOn: aStream.
%
category: 'accessors'
method: CDeclaration
properties

	^type == #'class'
		ifTrue: [fields]
		ifFalse: [nil].
%
category: 'attributes'
method: CDeclaration
readAttributes
"
	http://tigcc.ticalc.org/doc/gnuexts.html
"
	| parenthesisCount token |
	(token := self next) isOpenParenthesisToken ifFalse: [token error: 'Expected ''('' '].
	(token := self next) isOpenParenthesisToken ifFalse: [token error: 'Expected ''('' '].
	parenthesisCount := 2.
	[
		0 < parenthesisCount.
	] whileTrue: [
		token := self next.
		token isOpenParenthesisToken ifTrue: [
			parenthesisCount := parenthesisCount + 1.
		] ifFalse: [
			token isCloseParenthesisToken ifTrue: [
				parenthesisCount := parenthesisCount - 1.
			] ifFalse: [
				(token isIdentifierToken and: [token value = '__mode__']) ifTrue: [^self readModeAttributes].
			].
		].
	].
%
category: 'other'
method: CDeclaration
readBitFields
	"We parse bit fields, but we don't handle them properly!"

	bitCount := self cPreprocessorSpecies evaluateHeader: header.
	(bitCount isKindOf: SmallInteger) ifFalse: [ self error: 'Expected a SmallInteger in readBitFields'].
%
category: 'type'
method: CDeclaration
readComplexType
"
>>--+--struct--+--+-----identifier---------------------------+--><
      |              |   |                        .----------------.     |
      '--union----'    |                       V                   |     |
                          '--+----------+--{----member--;--+--}-'
                             '-identifier-'
"
	self peek isIdentifierToken ifTrue: [
		| dictionary |
		name := self next value.
    "  self trapName ."
		dictionary := type == #'struct'
			ifTrue: [header structs]
			ifFalse: [header unions].
		type := dictionary
			at: name
			ifAbsentPut: [self copy].
	] ifFalse: [
		type := self copy.
	].
	type fields ifNil: [type initializeAsStructOrUnion].
	name := nil.
%
category: 'declarator'
method: CDeclaration
readConstantExpression

	| token |
	[
		self atEnd not and: [self peek isOpenSquareBracketToken].
	] whileTrue: [
		| value tok |
		self next.
		count ifNil: [count := { } ].
		value := (tok := self peek) isCloseSquareBracketToken 
			ifTrue: [0]
			ifFalse: [self cPreprocessorSpecies evaluateHeader: header].
		value _isSmallInteger  ifFalse: [ tok error: 'Expected a SmallInteger'].
		count add: value.  
		(token := self next) isCloseSquareBracketToken ifFalse:[ token error:'expected '']'' '].
	].
%
category: 'declarator'
method: CDeclaration
readDeclaration
	"Return a CDeclaration (might not be self)
	   .-,------------------------------.
	  V                                      |
	----declarator--+-------------+-+--;--><
						  '--initializer--'

	We handle only one declarator; the caller (in CHeader>>#'readOne') handles the comma.
"
	| declaration token1 token2 |
	token1 := self peek.	"for debugging"
	(declaration := self readDeclarator) ifNil: [
		^self.
	].
	declaration includesCode ifTrue: [^declaration].
	token2 := self peek.	"for debugging"
	^declaration 
		readInitializer;
		readPostlude;
		yourself
%
category: 'declarator'
method: CDeclaration
readDeclarator
	"Return a CDeclaration (might not be self)
	>>--+-------------------------+--directDeclarator--><
	      |  .---------------------.  |
	      |  V                        |  |
	      '----pointerOperator--+-'
"
	| declaration token1 token2 |
	token1 := self peek.	"for debugging"
	(declaration := self readPointerOperator) ifNotNil: [
		token2 := self peek.	"for debugging"
		^declaration readDeclarator.
	].
	^self readDirectDeclarator.
%
category: 'declarator'
method: CDeclaration
readDirectDeclarator
	"Return a CDeclaration (might not be self)
	>>--+-----------------------------------------------------------------------------------------------------------------+-><
	      +--declaratorName---------------------------------------------------------------------------------------------+
	      +--directDeclarator--(--parameterDeclarationList--)--+----------------+---+----------------------------+--+
	      |                                                                       '--cvQualifiers--'     '--exceptionSpecification--'    |
	      +--directDeclarator--[--+-----------------------+--]-----------------------------------------------------------+
	      |                                '-constantExpression-'                                                                           |
	      '--(--declarator--)-----------------------------------------------------------------------------------------------'
"
	| token declarator |
	(token := self peek) ifNil: [^self].
	token isIdentifierToken ifTrue: [
		name := token value.
    "  self trapName ."
		(token := self next; peek) ifNil: [^self].
	] ifFalse: [ 
		(((type isKindOf: CDeclaration) and: [type type == #'class']) not and: [token isOpenParenthesisToken]) ifTrue: [ 
			self next.	"skip the open parenthesis token"
			declarator := self cDeclarationSpecies header: header.
			(token := self next) isUpArrowToken ifTrue: [	"Block argument, not pointer to function"
				storage := #'block'. 
				(token := self next) isIdentifierToken ifTrue: [token := self next].
			].
			token isCloseParenthesisToken ifFalse: [self error: 'expected '')'''].
			(token := self peek) ifNil: [self error: 'next token is empty'].
		].
	].
	token isOpenParenthesisToken ifTrue: [
		self readParameterDeclarationList.
		declarator ifNil: [^self].
		declarator 
			setStorage: storage
			type: self.
		storage := #'auto'.
		^declarator
	].
	token isOpenSquareBracketToken ifTrue: [
		declarator ifNotNil: [ token error:'unexpected declarator after ''['' ' ].
		self readConstantExpression.
		^self
	].
	declarator ifNil: [^self].
	"this is a specialized check for a function argument of the form 'struct dirent *(*[])'."
	(declarator storage == #'auto' and: [
		declarator type == #'int32' and: [
		declarator name == nil  and: [
		declarator count = #(0)]]]) ifTrue: [
			pointer := pointer + declarator pointer + 1.
		].
	^self
%
category: 'type'
method: CDeclaration
readEnumeration
	"
	'enum' [ identifier ] '{' enumerator-list '}' [ variable-list ]
	or
	'enum' identifier variable-list 
	or
	'enum' identifier function-definition // #48859
	"

	| token integralConstant |
	token := self peek.
	token isIdentifierToken ifTrue: [
		enumTag := self next value.
		token := self peek.
	].
	token isOpenCurlyBracketToken ifTrue: [
		fields := Dictionary new.	"this allows us to tell if the enum included an enumerator-list"
		self next.
		integralConstant := -1.
		[
			(token := self next) isCloseCurlyBracketToken.
		] whileFalse: [
			| identifier |
			token isIdentifierToken ifFalse: [token error: 'expected an identifier' ].
			identifier := token value.
			integralConstant := self peek isEqualsToken ifTrue: [ | tok |
				tok := self next.
				integralConstant := self cPreprocessorSpecies evaluateHeader: header.
				(integralConstant _isSmallInteger) ifFalse: [ tok error: 'Expected a SmallInteger'].
				integralConstant.
			] ifFalse: [
				integralConstant + 1.
			].
			(header enums includesKey: identifier) ifTrue: [ token error: 'Duplicate enum definition!'].
			header enums at: identifier put: integralConstant.
			fields at: identifier put: integralConstant.
			self peek isCommaToken ifTrue: [self next].
		].
	].
%
category: 'declarator'
method: CDeclaration
readInitializer
	"Since we don't have a real expression parser, we will go till we reach
	a comma, semicolon, or close parenthesis not in a nesting"

	| token parenthesisCount curlyBracketCount |
	(token := self peek) ifNil: [^self].
	token value = '__attribute__' ifTrue: [
		self next.
		self readAttributes.
		^self readInitializer
	].
	token isEqualsToken ifFalse: [^self].
	self next.
	parenthesisCount := 0.
	curlyBracketCount := 0.
	[true] whileTrue: [ | pkk |
		(pkk := self peek) ifNil: [ token error:'next token is empty' ].
    token := pkk .
		(parenthesisCount == 0 and: [curlyBracketCount == 0]) ifTrue: [
			token isCommaToken ifTrue: [^self].
			token isSemicolonToken ifTrue: [^self].
			token isCloseParenthesisToken ifTrue: [^self].
		].
		token isOpenParenthesisToken ifTrue: [parenthesisCount := parenthesisCount + 1] ifFalse: [
		token isCloseParenthesisToken ifTrue: [parenthesisCount := parenthesisCount - 1] ifFalse: [
		token isOpenCurlyBracketToken ifTrue: [curlyBracketCount := curlyBracketCount + 1] ifFalse: [
		token isCloseCurlyBracketToken ifTrue: [curlyBracketCount := curlyBracketCount - 1]]]].
		self next.
	].
%
category: 'attributes'
method: CDeclaration
readModeAttributes
"
	http://tigcc.ticalc.org/doc/gnuexts.html#SEC91_mode
	http://www.linuxtopia.org/online_books/programming_tool_guides/linux_using_gnu_compiler_collection/vector-extensions.html
		QI = 8-bit integer, HI = 16-bit integer, SI = 32-bit integer, DI = 64-bit integer, SF = 32-bit float, DF = 64-bit float, TC = 128-bit complex
		byte = 8-bit integer, word = 64-bit integer, pointer = 64-bit integer

	typedef int int8_t __attribute__ ( ( __mode__ ( __QI__ ) ) ) ;
"
	| token mode |
	(token := self peek) isOpenParenthesisToken ifFalse: [ token error: 'expected ''('' '].
	self next.
	(token := self next) key == #'identifier' ifFalse: [token error: 'Expected an identifier'].
	mode := token value.
	type := 
		mode = '__QI__' ifTrue: [type first == $u ifTrue: [#'uint8'] ifFalse: [#'int8']] ifFalse: [
		mode = '__HI__' ifTrue: [type first == $u ifTrue: [#'uint16'] ifFalse: [#'int16']] ifFalse: [
		mode = '__SI__' ifTrue: [type first == $u ifTrue: [#'uint32'] ifFalse: [#'int32']] ifFalse: [
		mode = '__DI__' ifTrue: [type first == $u ifTrue: [#'uint64'] ifFalse: [#'int64']] ifFalse: [
		mode = '__byte__' ifTrue: [type first == $u ifTrue: [#'uint8'] ifFalse: [#'int8']] ifFalse: [
		mode = '__word__' ifTrue: [type first == $u ifTrue: [#'uint64'] ifFalse: [#'int64']] ifFalse: [
		mode = '__pointer__' ifTrue: [type first == $u ifTrue: [#'uint64'] ifFalse: [#'int64']] 
      ifFalse: [
		mode = '__TC__' ifTrue: [#'complexDouble'] ifFalse: [
		self halt]]]]]]]].
	self expectCloseParenthesisCount: 3.
%
category: 'declarator'
method: CDeclaration
readParameterDeclarationList
  | token |
  self next.  "open parenthesis token"
  parameters := { } .
  "self trapName ."
  [
    (token := self peek ) isCloseParenthesisToken 
  ] whileFalse: [
    self peek isDotToken ifTrue: [
      self next.
      2 timesRepeat: [(token := self next) isDotToken ifFalse: [ token error:'expected ''.'' ']].
      isVaryingArgCount := true.
      (token := self peek) isCloseParenthesisToken ifFalse: [ token error:'expected '')'' '].
    ] ifFalse: [
      parameters add: (self cDeclarationSpecies header: header).
      (token := self peek) isCommaToken ifTrue: [self next].
    ].
  ].
  self next.  "close parenthesis token"
  (parameters size = 1 and: [parameters first isEmptyVoid]) ifTrue: [
    parameters := #().
  ].
  self mayHaveCode ifFalse:[ ^self].
  [
    (token := self peek) ifNil: [^self].
    token isSemicolonToken ifTrue: [^self].
    token isOpenCurlyBracketToken
  ] whileFalse: [
    self next.
  ].
  self next.
  self skipToEndOfCodeBlock.
%
category: 'declarator'
method: CDeclaration
readPointerOperator
	"Return a CDeclaration (might not be self)
	>>--*--+----------------+--><
	          '-typeQualifiers-' 
"
	| token |
	(token := self peek) ifNil: [^nil].
	token isStarToken ifFalse: [^nil].
	pointer := pointer + 1.
	(token := self next; peek) ifNil: [^self].
	token isIdentifierToken ifFalse: [^self].
	(#('const' 'volatile') includes: token value) ifFalse: [^self].
	self next.
	"we don't keep track of const pointers, only the data"
%
category: 'declarator'
method: CDeclaration
readPostlude

	| token1 token2 |
	(token1 := self peek) ifNil: [^self].
	token1 isIdentifierToken ifTrue: [
		(2 < token1 value size and: [(token1 value at: 1) == $_ and: [(token1 value at: 2) == $_]]) ifTrue: [
			(token2 := self next; peek) ifNil: [self next. ^self].	"Ignore trailing tokens beginning with double-underscore"
			token2 isOpenParenthesisToken ifTrue: [
				token1 value = '__attribute__' ifTrue: [
					self readAttributes.
				] ifFalse: [
					self skipFunction.
				].
			].
			^self readPostlude.
		].
		self error: 'unexpected token: ', token1 printString.
	].
	token1 isColonToken ifTrue: [
		self next. 
		self readBitFields. 
		^self readPostlude.
	].
	token1 isOpenCurlyBracketToken ifTrue: [
		((type isKindOf: CDeclaration) and: [type type == #'class']) ifTrue: [
			self skipToEndOfCodeBlock.
			^self.
		].
		self isFunction ifTrue: [		"Functions might have code (bugs #43426 & 43782)"
			self next.
			self skipToEndOfCodeBlock.
			^self
		].
	].
	token1 isEqualsToken ifTrue: [	"Check for an initialization"
		[
			(token2 := self next; peek) ~~ nil  and: [token2 isSemicolonToken].
		] whileFalse.
		^self
	].
%
category: 'type'
method: CDeclaration
readStorageClassSpecifier
	"Return a CDeclaration (might not be self)
	( 'auto' | 'extern' | 'register' | 'static' | 'typedef' | 'inline' )
"
	| token value |
	(token := self peek) ifNil: [^nil].
	token isTildeToken ifTrue: [token := self next; peek].	"indicates a class destructor function, which we can ignore"
	[token isExtensionIdentifierToken] whileTrue: [token := self next; peek].	"safe to ignore"
	token  isIdentifierToken ifFalse: [^nil].
	token value = 'inline' ifTrue: [
		self next.
		storage := #'inline'.
		^self.
	].
	(#('auto' 'extern' 'register' 'static' 'typedef') includes: token value) ifTrue: [
		(storage ~~ nil  and: [storage ~= token value asSymbol])
			ifTrue: [ token error: 'We support only one storage specifier'].
		storage := self next value asSymbol.
		(token := self peek) ifNil: [^nil].
		value := token value.
		(storage = #'extern' and: [token type = #'string']) ifTrue: [
			linkageSpec := value.
			(token := self next; peek) isOpenCurlyBracketToken
				ifTrue: [type := nil].
		].
		^self.
	].
	^nil.
%
category: 'type'
method: CDeclaration
readType
	"Return a CDeclaration (might not be self)
"
	| declaration |
	declaration := self readTypeA.
	storage ifNil: [storage := #'auto'].
	self interpretTypeSpecifier.
	^self.
%
category: 'type'
method: CDeclaration
readTypeA
	"Return a CDeclaration (might not be self)
	        .-----------------------------.  
	        V                                 |
	>>---+----------------------------+--><
           +--storageClassSpecifier--+			( 'auto' | 'extern' | 'register' | 'static' | 'typedef' | 'inline' )
           +--typeSpecifier-----------+			( 'void' | 'char' | 'short' | 'int' | 'long' | 'float' | 'double' | ... )
           '--typeQualifier-------------'				( 'const' | 'volatile' )
"
	| declaration token |
	(token := self peek) isAttributeIdentifierToken ifTrue: [self next; readAttributes. ^self readTypeA].
	(declaration := self readStorageClassSpecifier) ifNotNil: [^declaration readTypeA].
	(declaration := self readTypeSpecifier) ifNotNil: [^declaration readTypeA].
	(declaration := self readTypeQualifier) ifNotNil: [^declaration readTypeA].
	^self.
%
category: 'type'
method: CDeclaration
readTypeFrom: aHeader
	"Return a CDeclaration (might not be self)"

	header := aHeader.
	[self peek isEmptyToken] whileTrue: [self next].
	type := { } .
	pointer := 0.
	isConstant := false.
	includesCode := false.
	isVaryingArgCount := false.
	^self readType.
%
category: 'type'
method: CDeclaration
readTypeQualifier
 	"Return a CDeclaration (might not be self)
	( 'const' | 'volatile' )
"
	| token value |
	(token := self peek) 	ifNil: [^nil].
	token isIdentifierToken 	ifFalse: [^nil].
	value := token value.
	value = 'const' 	ifTrue: [self next. isConstant := true. ^self].
	value = 'volatile' 	ifTrue: [self next. ^self].
	^nil.
%
category: 'type'
method: CDeclaration
readTypeSpecifier
 	"Return a CDeclaration (might not be self)
		( 'void' | 'char' | 'short' | 'int' | 'long' | 'float' | 'double' | 'signed' | 'unsigned' 
       | '_Complex' | struct_or_union_spec | enum_spec | typedef_name )
	"
	| token value declaration |
	(token := self peek) ifNil: [^nil].
	token isIdentifierToken ifFalse: [^nil].
	value := token value.
	( #('void' 'char' 'short' 'int' 'long' 'float' 'double' 'signed' 'unsigned') 
        includes: value) ifTrue: [
		type add: value asSymbol.
		self next.
		^ self.
	].
	value = '__float128'	ifTrue: [type := #'float128'.		self next. ^self].
	value = '_Complex'	ifTrue: [type add: #'complex'.	self next. ^self].		"this means that there are two of something else (e.g., double)"
	value = 'ptr'			ifTrue: [type := #'ptr'.				self next. ^self].
	value = 'enum'			ifTrue: [type := #'enum'.			self next. self readEnumeration. 	^self].
	value = 'struct'		ifTrue: [type := #'struct'.			self next. self readComplexType. ^self].
	value = 'union' 		ifTrue: [type := #'union'.			self next. self readComplexType. ^self].
	value = 'class' 		ifTrue: [type := #'class'.			self next. storage := #'typedef'.	^self].
	type = #() ifTrue: [
		declaration :=
			header types at: value ifAbsent: [
			header structs at: value ifAbsent: [
			header unions at: value ifAbsent: [
			header enums at: value ifAbsent: [nil]]]].
		declaration ifNotNil:[ 
      type := declaration derivedType. 
      type == declaration ifFalse:[ pointer := declaration pointer ].  "fix 48748"
      self next. 
      ^self
    ].
	].
	"These are a bit troubling. Sometimes the are expected to be built-in, and other times they are typedefs!"
	(value = '__builtin_va_list' 	and: [type = #()])	ifTrue: [type := #'ptr'. 		self next. ^self].
	(value = '__gnuc_va_list' 	and: [type = #()])	ifTrue: [type := #'ptr'. 		self next. ^self].
	^nil.
%
category: 'CCallout'
method: CDeclaration
resultType

	self isFunction ifFalse: [ self error: 'Not a function!'].
	(self isSimple and: [self knownResultTypes includesIdentical: type]) ifTrue: [^type].
	type == #'enum' ifTrue: [^#'int32'].
	type == #'uint8' ifTrue: [
		(count size == 1 and: [pointer == 0]) ifTrue: [^#'char*'].		"const char foo[]"
		(count == nil and: [pointer == 1]) ifTrue: [^#'char*'].			"const char * foo"
	].
	^#'ptr'.
%
category: 'other'
method: CDeclaration
setClassProperties: anArray

	fields := anArray.
%
category: 'Updating'
method: CDeclaration
setStorage: aSymbol type: aCDeclaration

	storage := aSymbol.
	type := aCDeclaration.
%
category: 'CByteArray get'
method: CDeclaration
simpleAccessorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name; lf; lf; tab;
		nextPutAll: '^self ';
		nextPutAll: self getSelector; space;
		nextPutAll: anInteger printString;
		nextPut: $.; lf;
		contents.
%
category: 'CByteArray put'
method: CDeclaration
simpleUpdatorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name;
		nextPutAll: ': anObject'; lf; lf; tab;
		nextPutAll: 'self '; lf; tab; tab;
		nextPutAll: self getSelector; space;
		nextPutAll: anInteger printString; lf; tab; tab;
		nextPutAll: 'put: anObject.'; lf;
		contents.
%
category: 'CByteArray get'
method: CDeclaration
singlePointerAccessorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name; lf; lf; tab;
		nextPutAll: '^self'; lf; tab; tab;
		nextPutAll: 'pointerAt: ';
		nextPutAll: anInteger printString; lf; tab; tab;
		nextPutAll: 'resultClass: ';
		nextPutAll: self cByteArraySpecies name asString;
		nextPut: $. ; lf;
		contents.
%
category: 'CByteArray put'
method: CDeclaration
singlePointerUpdatorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name; 
		nextPutAll: ': aCByteArray'; lf; lf; tab;
		nextPutAll: 'self'; lf; tab; tab;
		nextPutAll: 'pointerAt: ';
		nextPutAll: anInteger printString; lf; tab; tab;
		nextPutAll: 'put: aCByteArray.'; lf;
		contents.
%
category: 'other'
method: CDeclaration
skipFunction

	| parenthesisCount |
	parenthesisCount := 1.
	self next.
	[
		0 < parenthesisCount.
	] whileTrue: [
		| token |
		token := self next.
		token isCloseParenthesisToken ifTrue: [parenthesisCount := parenthesisCount - 1] ifFalse: [
		(token isIdentifierToken and: [token value last == $(]) ifTrue: [parenthesisCount := parenthesisCount + 1] ifFalse: [
		token isOpenParenthesisToken ifTrue: [parenthesisCount := parenthesisCount + 1]]].
	].
%
category: 'declarator'
method: CDeclaration
skipToEndOfCodeBlock

	| depth |
	includesCode := true.
	depth := 1.
	[
		0 < depth.
	] whileTrue: [
		| token |
		token := self next.
		token isOpenCurlyBracketToken ifTrue: [depth := depth + 1].
		token isCloseCurlyBracketToken ifTrue: [depth := depth - 1].
	].
%
category: 'accessors'
method: CDeclaration
source
	^ source
%
category: 'Updating'
method: CDeclaration
source: aString

	source ifNotNil: [^self].
	source := aString.
	(type isKindOf: CDeclaration) ifTrue: [type source: aString].
	fields ifNotNil: [
		type ~~ #'enum' ifTrue: [
			fields do: [:each | 
				each source ifNil: [
					each source: aString.
				].
			].
		].
	].
	parameters ifNotNil: [
		parameters do: [:each | 
			each source ifNil: [
				each source: aString.
			].
		].
	].
%
category: 'CByteArray get'
method: CDeclaration
sourceStringsForAccessors

	| list offset |
	list := OrderedCollection new.
	offset := 0.
	fields do: [:each | 
		list add: (each accessorForOffset: offset).
		offset := offset + each byteSize.
	].
	^list.
%
category: 'CByteArray put'
method: CDeclaration
sourceStringsForUpdators

	| list offset |
	list := OrderedCollection new.
	offset := 0.
	fields do: [:each | 
		list add: (each updatorForOffset: offset).
		offset := offset + each byteSize.
	].
	^list.
%
category: 'accessors'
method: CDeclaration
storage

	^storage.
%
category: 'Updating'
method: CDeclaration
storage: aSymbol linkageSpec: optionalString

	storage := aSymbol.
	linkageSpec := optionalString.
%
category: 'other'
method: CDeclaration
stream

	^header stream.
%
category: 'CByteArray get'
method: CDeclaration
stringAccessorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name; lf; lf; tab;
		nextPutAll: '^self _stringFromBytes: (self '; lf; tab; tab;
		nextPutAll: 'byteArrayFrom: ';
		nextPutAll: anInteger printString; lf; tab; tab;
		nextPutAll: 'to: ';
		nextPutAll: (anInteger + count first - 1) printString;
		nextPutAll: ').'; lf;
		contents.
%
category: 'CByteArray get'
method: CDeclaration
stringPointerAccessorForOffset: anInteger

	1 == count ifTrue: [
		^(AppendStream on: String new)
			nextPutAll: name; lf; lf; tab;
			nextPutAll: '^self stringFromCharStarAt: ';
			nextPutAll: anInteger printString; 
			nextPut: $.; lf; tab;
			nextPutAll: '"^self'; lf; tab; tab;
			nextPutAll: 'pointerAt: ';
			nextPutAll: anInteger printString; lf; tab; tab;
			nextPutAll: 'resultClass: ';
			nextPutAll: self cByteArraySpecies name asString;
			nextPutAll: '."'; lf;
			contents.
	] ifFalse: [
		^(AppendStream on: String new)
			nextPutAll: name; lf; lf; tab;
			nextPutAll: '| array offset |'; lf; tab;
			nextPutAll: 'array := Array new: ';
			nextPutAll: count first printString;
			nextPut: $.; lf; tab;
			nextPutAll: 'offset := ';
			nextPutAll: anInteger printString;
			nextPut: $.; lf; tab;
			nextPutAll: '1 to: array size do: [:i | '; lf; tab; tab;
			nextPutAll: 'array at: i put: (self stringFromCharStarAt: offset).'; lf; tab; tab;
			nextPutAll: '"array at: i put: (self pointerAt: offset resultClass: ';
			nextPutAll: self cByteArraySpecies name asString;
			nextPutAll: ')."'; lf; tab; tab;
			nextPutAll: 'offset := offset + 8.'; lf; tab;
			nextPutAll: '].'; lf; tab;
			nextPutAll: '^array.'; lf;
			contents.
	].
%
category: 'CByteArray put'
method: CDeclaration
stringPointerUpdatorForOffset: anInteger

	1 == count ifTrue: [
		^(AppendStream on: String new)
			nextPutAll: name;
			nextPutAll: ': aCByteArray'; lf; lf; tab;
			nextPutAll: '"self'; lf; tab; tab;
			nextPutAll: 'pointerAt: ';
			nextPutAll: anInteger printString; lf; tab; tab;
			nextPutAll: 'put: aCByteArray."'; lf;
			contents.
	] ifFalse: [
		^self arrayPointersUpdatorForOffset: anInteger.
	].
%
category: 'CByteArray put'
method: CDeclaration
stringUpdatorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name;
		nextPutAll: ': aByteArray'; lf; lf; tab;
		nextPutAll: 'self'; lf; tab; tab; 
		nextPutAll: 'replaceFrom: ';
		nextPutAll: anInteger printString; lf; tab; tab;
		nextPutAll: 'to: ';
		nextPutAll: (anInteger - 1) printString;
		nextPutAll: ' + (';
		nextPutAll: count first printString;
		nextPutAll: ' min: aByteArray size)'; lf; tab; tab; 
		nextPutAll: 'with: aByteArray'; lf; tab; tab;
		nextPutAll: 'startingAt: 1.'; lf;
		contents.
%
category: 'CByteArray get'
method: CDeclaration
structAccessorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name; lf; lf; tab;
		nextPutAll: '^self'; lf; tab; tab;
		nextPutAll: 'newFrom: ';
		nextPutAll: anInteger printString; lf; tab; tab;
		nextPutAll: 'numBytes: ';
		nextPutAll: self baseByteSize printString;
		nextPut: $.; lf;
		contents.
%
category: 'CByteArray put'
method: CDeclaration
structUpdatorForOffset: anInteger

	^(AppendStream on: String new)
		nextPutAll: name;
		nextPutAll: ': aByteArray'; lf; lf; tab;
		nextPutAll: 'self'; lf; tab; tab; 
		nextPutAll: 'replaceFrom: ';
		nextPutAll: anInteger printString; lf; tab; tab;
		nextPutAll: 'to: ';
		nextPutAll: (anInteger - 1) printString;
		nextPutAll: ' + (';
		nextPutAll: self baseByteSize printString;
		nextPutAll: ' min: aByteArray size)'; lf; tab; tab; 
		nextPutAll: 'with: aByteArray'; lf; tab; tab;
		nextPutAll: 'startingAt: 1.'; lf;
		contents.
%
category: 'Debugging'
method: CDeclaration
trapFile
  "uncomment this and senders for debugging
    file ifNotNil:[:f |
      (f includesString:'t.hf') ifTrue:[ nil pause ].
    ].
  "
%
category: 'Debugging'
method: CDeclaration
trapLine
  "uncomment this and senders for debugging 
  file ifNotNil:[:f |
    (f includesString:'t.hf') ifTrue:[ 
       line = 26 ifTrue:[ nil pause ].
    ]. 
  ]
  "
%
category: 'Debugging'
method: CDeclaration
trapName
  "uncomment this and senders for debugging
   name = 'GciTsFetchX' ifTrue:[ nil pause ].
  "
%
category: 'accessors'
method: CDeclaration
type
	"For many types, the value is a Symbol (e.g., #'uint64' or #'struct').
	For other types, the value is a CDeclaration for a function or a class"
  "name = 'IPtr' ifTrue:[ nil pause ]."
	^type.
%
category: 'CByteArray put'
method: CDeclaration
updatorForOffset: anInteger

	0 == pointer ifTrue: [
		type == #'class' ifTrue: [^self structUpdatorForOffset: anInteger].
		(type isKindOf: CDeclaration) ifTrue: [^self structUpdatorForOffset: anInteger].
		count == nil ifTrue: [^self simpleUpdatorForOffset: anInteger].
		1 < count size ifTrue: [self error: 'Multi-dimensional arrays not supported!'].
		type == #'uint8' ifTrue: [^self stringUpdatorForOffset: anInteger].
		^self arrayUpdatorForOffset: anInteger.
	].
	1 < pointer ifTrue: [self error: 'Multi-level pointers not supported!'].
	count == nil ifTrue: [^self singlePointerUpdatorForOffset: anInteger].
	type == #'uint8' ifTrue: [^self stringPointerUpdatorForOffset: anInteger].
	^self arrayPointersUpdatorForOffset: anInteger.
%
category: 'CCallout'
method: CDeclaration
varArgsAfter

	^ isVaryingArgCount ifTrue: [parameters size] ifFalse: [-1].
%
category: 'Reporting'
method: CDeclaration
_addToReport: rpt
  | brace |
  (name ~~ nil ) ifTrue:[ rpt add: (brace := '{') ].
  name ifNotNil:[:n | rpt add: n ; add: ' ' ; add: self byteSize asString ; add: ' '].
  fields ifNotNil:[:f | f do:[ :aField | aField _addToReport: rpt ]].
  (pointer = 0 and:[ type isKindOf: CDeclaration]) ifTrue:[ type _addToReport: rpt ].
  brace ifNotNil:[ rpt add:'}' . ]
%
category: 'accessors'
method: CDeclaration
_fields

	^fields.
%
category: 'other'
method: CDeclaration
_next
	| next |
	(next := header next) ifNil: [^nil].
	file ifNil: [
		file := next file.
		line := next line. 
    "self trapLine ."
	].
	^next.
%
