! ==============================================================
set class CZstream
removeallmethods
removeallclassmethods

category: 'Documentation'
classmethod:
comment

^ '
A CZstream encapsulates the state of a zstream from zlib.so ,
and associates that zstream with an IO object.

 InstVars
   buffer - a String < 16K bytes
   bufSize - a SmallInteger
   comprBuffer - a String or ByteArray < 16K bytes
   comprSize - a SmallInteger , only used for output streams
   forWrite  - a Boolean
   ioObj       - an instance of IO (a GsSocket or GsFile)
   errorClass  - a subclass of Exception used to signal errors
   header - gzip header of a file, when reading
   comprOffset a SmallInteger , only used for output streams
   
   hidden instVar cData is an instance of struct z_stream_s  per zlib.h ,
   which will be automatically freed when this object is GC''ed 

Constraints:
	buffer: String
	bufSize: SmallInteger
	comprBuffer: ByteArray
	comprSize: SmallInteger
	forWrite: Boolean
	ioObj: IO
	errorClass: AbstractException
	header: Array
	comprOffset: SmallInteger
'
%
classmethod
_initializeConstants
  self _addInvariantClassVar: #ComprBufSize value: 4096 ;
     _addInvariantClassVar: #HalfComprBufSize value: 2048 . 
%

run
CZstream _initializeConstants .
true
%

category: 'Private'
classmethod:
_basicNew

  "creates an instance registered with VM for finalization of cData"
  <primitive: 674>
  ^self _primitiveFailed: #_basicNew
%

category: 'Instance creation'
classmethod:
openRead: anIO errorClass: anExceptionClass

| inst |
(inst := self _basicNew )
   open: false io: anIO errorClass: anExceptionClass 
	comprLevel: -1 "Z_DEFAULT_COMPRESSION".
^ inst
%

classmethod:
openRead: anIO
^ self openRead: anIO errorClass: IOError
%

classmethod:
openWrite: anIO errorClass: anExceptionClass comprLevel: anInt
| inst |
(inst := self _basicNew )
   open: true io: anIO errorClass: anExceptionClass comprLevel: anInt .
^ inst
%

classmethod:
openWrite: anIO
^ self openWrite: anIO errorClass: IOError comprLevel: -1"Z_DEFAULT_COMPRESSION"
%

classmethod:
openWrite: anIO errorClass: anExceptionClass comprLevel: anInt
| inst |
(inst := self _basicNew )
   open: true io: anIO errorClass: anExceptionClass comprLevel: anInt .
^ inst
%

category: 'Initialization'
method:
open: writeBoolean io: anIO errorClass: anExceptionClass comprLevel: anInt
  self _setNoStubbing . "ensure buffers stay in memory if rcvr committed"
  forWrite := writeBoolean .
  ioObj := anIO .
  errorClass := anExceptionClass .
  writeBoolean ifTrue:[ buffer := String new: 1024 ]
             ifFalse:[ buffer := nil ].
  bufSize := 0 .
  "logically comprBuffer is a ByteArray, but for use from Ruby,
     we need to use a String "
  writeBoolean ifTrue:[ comprBuffer := nil "created on demand" ]
              ifFalse:[ comprBuffer := String new: 1024 "grown as needed" ].
  comprSize := 0 .
  comprOffset := 1 .
  header := nil .
  self _streamOp: 0 with: anInt . "initialize C z_stream"
%

category: 'Stream processing'
classmethod:
adler32: byteObject initialCrc: startCrc

"compute the adler32 checksum of the byteObject, per zlib.h .
 startCrc, must be aSmallInteger, >=0 and <= 16rFFFFFFFF .
 Normally startCrc should be 1 . "

<primitive: 886>
self _primitiveFailed: #adler32:initialCrc: 
     args: { byteObject . startCrc }
%

classmethod:
crc32: byteObject initialCrc: startCrc

"compute the crc32 checksum of the byteObject, per zlib.h .
 startCrc, must be aSmallInteger, >=0 and <= 16rFFFFFFFF .
 Normally startCrc should be 0 . "

<primitive: 888>
self _primitiveFailed: #crc32:initialCrc:
     args: { byteObject . startCrc }
%

classmethod:
crcTable
"Returns an Array containing the contents of crc_table[0] per crc32.h"
<primitive: 887>

self _primitiveFailed: #crcTable
%

method:
_streamOp: opcode with: anArg

"opcode  0  initialize the C z_stream , anArg is compression level for output
         1  close the zstream , sets ioObj to nil
         2  flush zstream to comprBuf, returns true of no more output
         3  get position, from the C z_stream  return total_in for an output stream
               or total_out for an output stream"

<primitive: 647>
opcode == 1 ifTrue:[ errorClass signal:'stream already closed'].
opcode == 2 ifTrue:[ errorClass signal:'stream already closed'].
self _primitiveFailed: #_streamOp: args: { opcode . anArg }
%

method:
header
  "Returns an Array, { originalName. comment. mtime } 
   from header of a gzip file, or signals an error.  mtime is a time_t as an Integer"
  forWrite ifTrue:[ errorClass signal:'stream not opened for reading'].
  header ifNil:[ 
    self _readAndDecompress .
    header == false ifTrue:[ errorClass signal:'stream has no gzip header'].
  ].
  ^ header
%

method:
_decompress

"decompress the bytes  (comprBuffer copyFrom: comprOffset to: comprSize) 
            into       buffer at:1  . 
 grows buffer as needed up to max of 16K and 
 returns new value of bufSize instVar,  
 which reflects number of bytes available in buffer.
 Also updates comprOffset instVar to reflect bytes consumed. "

<primitive: 705>
self _primitiveFailed: #_decompress:
%

method:
_readAndDecompress
  "Returns the number of available bytes in uncompressed  buffer,
   or zero if at EOF.  Ruby has additional code in git "
  | bs |
  bs := bufSize .
  [ bs == 0 ] whileTrue:[ 
    comprOffset > comprSize ifTrue:[ | cbuf count |
      count := ioObj read: (cbuf := comprBuffer) size into: comprBuffer  .
      count == 0 ifTrue:[ ^ 0 "EOF" ].
      count ifNil:[ errorClass signal:'error on input stream' ].
      comprOffset := 1 .
      comprSize := count .
    ].
    bs := self _decompress
  ].
  ^ bs
%

method:
_finishReading

"Closes a stream that was open for input and deallocates zlib C memory"
self _streamOp: 1 with: nil 
%

method:
atEnd
"Returns true if an input stream is at eof, or has been closed by
  _finishReading"

 ioObj ifNotNil:[ | bs |
   bufSize ~~ 0 ifTrue:[ ^ false ].
   bs := self _readAndDecompress .
   ^ bs == 0 
 ]. 
 ^ true
%

method:
position
 "Returns the current position of the receiver, or nil receiver has been closed"

 ioObj ifNil:[ ^ nil ].
 forWrite ifTrue:[
   ^ (self _streamOp: 3 with:nil )"z_stream.total_out" + bufSize  
 ] ifFalse:[
   ^ (self _streamOp: 3 with:nil )"z_stream.total_in" - bufSize
 ]
%  


method:
_compress: aBuffer from: offset to: endOffset
  
"Usages  
  _compress: aStringOrByteArray from: aSmallInt to: aSmallInt
     buffer is a String or ByteArray, containing uncompressed data 
     per offset, endOffset.  primitive writes compressedData into
     self.comprBuffer, and updates self.comprSize  
  
     result is offset to use for next call, or endOffset + 1
     if buffer is was completely consumed.

  _compress: anGzipHeaderArray from: nil to: nil 
     result is nil , must be first call after init for write.
"

<primitive: 648>
%

method:
_compressAndWriteFrom: inputObj count: numBytes 
  "Ruby has additional code in git "
  | ofs | 
  ofs := 1 .
  [ true ] whileTrue:[ | csiz |
    (csiz := comprSize) > HalfComprBufSize ifTrue:[ | status |
      status := ioObj write: csiz from: comprBuffer .
      status ifNil:[ errorClass signal: 'error on output object' ].
      comprSize := 0
    ].
    ofs > numBytes ifTrue:[
      ^ self "DONE"
    ].
    comprBuffer ifNil:[ comprBuffer := String new: ComprBufSize ].
    "compress primitive updates comprSize to reflect bytes added
      to comprBuffer ."
    ofs := self _compress: inputObj from: ofs to: numBytes .
  ]
%

method:
close
 forWrite ifTrue:[ 
   self flush: nil .
   self _streamOp: 1 with: nil . "close underlying stream"
 ] ifFalse:[ 
   self _finishReading
 ]
%

method:
flush: flagsInt
  "flagsInt may be nil (implies Z_FINISH)
   or should be Z_FLUSH_SYNC .
   Ruby has additional code in git "
  | bs csiz done |
  (bs := bufSize) ~~ 0 ifTrue:[
    comprBuffer ifNil:[ comprBuffer := String new: ( bs max: 256) ].
    self _compressAndWriteFrom: buffer count: bs 
    "at this point buffer has been consumed "
  ].
  done := false .
  [ done ] whileFalse:[
    comprBuffer ifNil:[ comprBuffer := String new: 256 ].
    done := self _streamOp: 2 with: flagsInt .  "deflate(zstream, flags)"
    (csiz := comprSize) ~~ 0 ifTrue:[ | status |
      status := ioObj write: csiz from: comprBuffer .
      status ifNil:[ errorClass signal: 'error on output object' ].
      comprSize := 0
    ]
  ].
%

method:
flush

  ^ self flush: nil "equivalent to Z_FINISH"
%

method:
writeHeader: aGzipHeaderArray
  "must be the first write after opening a new stream, to
   be used if a gzip header is desired on the compressed output

   aGzipHeaderArray is of form
     { originalNameString. commentString . mtimeSmallInt} "

  ^ self _compress: aGzipHeaderArray from: nil to: nil
%

method:
write: aString
 "aString may also be aByteArray"
  ^ self write: aString count: aString size
%

method:
write: aString  count: numBytes
 "aString may also be aByteArray"

 | bs |
 numBytes < HalfComprBufSize ifTrue:[
   (numBytes + (bs:= bufSize))  >  HalfComprBufSize ifTrue:[ 
     self _compressAndWriteFrom: buffer count: bs .
     bs := 0 .
     bufSize := 0 .
   ].
  "aString copyFrom: 1 to: numBytes into: buffer startingAt: bs + 1 . "
   buffer replaceFrom: bs + 1 to: bs + numBytes with: aString startingAt: 1 .
   bufSize := bs + numBytes . 
 ] ifFalse:[
   (bs := bufSize) ~~ 0 ifTrue:[ 
     self _compressAndWriteFrom: buffer count: bs . 
     bufSize := 0 .
   ].
   self _compressAndWriteFrom: aString count: numBytes  
 ]
%

method: 
read: maxBytes
  "read and decompress up to maxBytes from receiver,  
   returning a String which may be smaller than maxBytes.
   Result will be of size 0 if EOF hit."
  | res bs |
  maxBytes == 0 ifTrue:[ errorClass signal:'maxBytes must be > 0'].
  (bs := bufSize) == 0 ifTrue:[
    bs := self _readAndDecompress .
    bs == 0 ifTrue:[ ^ String new ].
  ].
  maxBytes >= bs ifTrue:[
    res := buffer .
    res size: bs .
    buffer := nil .
    bufSize := 0 .
  ] ifFalse:[
    res := buffer copyFrom:1 to: maxBytes .
   "buffer copyFrom: maxBytes + 1 to: bs into: buffer startingAt: 1 ."
    buffer replaceFrom: 1 to:  bs - maxBytes with: buffer startingAt: maxBytes + 1 .
    bufSize := bs - maxBytes . 
  ].
  ^ res
%

method:
readAll
| res str |
res := String new .
[ true ] whileTrue:[ 
  str := self read: 16384 .
  str size == 0 ifTrue:[ ^ res ].
  res addAll: str .
]
%


