!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Definition of DateAndTimeANSI (from TimeZone2007.gs)
!
!=========================================================================

set class DateAndTimeANSI

removeallmethods 
removeallclassmethods 

! ------------------- Class methods for DateAndTimeANSI
category: 'For Documentation Installation only'
classmethod: DateAndTimeANSI
installDocumentation

self comment:
'This protocol describes the behavior that is common to date 
time objects. Date time objects represent individual points 
in Coordinated Universal Time (UTC) as represented in an
implementation defined local time, with a default resolution 
of seconds.

The exact properties of local times are unspecified. Local 
times may differ in their offset from UTC. A given local time 
may have different offsets from UTC at different points in time.

All dates and times in the UTC local time are in the Gregorian 
calendar. Date times prior to the adoption of the Gregorian 
calendar are given in the retrospective astronomical Gregorian 
calendar. The year 1 B.C. is astronomical Gregorian year 0. 
The year 2 B.C. is astronomical Gregorian year-1. The year 1 A.D. 
is astronomical Gregorian year 1. The offset of the UTC local 
time is zero.'.
%

category: 'DateAndTime factory'
classmethod: DateAndTimeANSI
clockPrecision
	"Answer a <Duration> such that after that period of time passes, 
	#now is guaranteed to give a different result.
	Ideally implementations should answer the least such duration."

	^Duration seconds: 1.
%
category: 'Migration'
classmethod: DateAndTimeANSI
migrateNew

	^self basicNew.
%
category: 'DateAndTime factory'
classmethod: DateAndTimeANSI
now
	"Synopsis
	Answer a <DateAndTime> representing the current date and time.
	Definition: <DateAndTime factory>
	Answer a <DateAndTime> representing the current date and time in the local time specified by the
	implementation.
	Return Values
	<DateAndTime> unspecified"

  | res |
  (res := super new) initializeAsNow.
  "res beRounded."
  ^ res.
%
category: 'DateAndTime factory'
classmethod: DateAndTimeANSI
year: year day: dayOfYear hour: hour minute: minute second: second
	"Synopsis
	Answer a <DateAndTime> which is the second second of the minute minute of the hour hour of
	the day dayOfYear of the year year of the astronomical Gregorian calendar in local time.
	Definition: <DateAndTime factory>
	Answer the least <DateAndTime> which is the second second of the minute minute of the hour
	hour of the day dayOfYear of the year year of the astronomical Gregorian calendar in the local
	time specified by the implementation. The second must be a <number> greater than or equal to 0
	and strictly less than 60. The minute must be an <integer> between 0 and 59 inclusive. The hour
	must be an <integer> between 0 and 23 inclusive. The day must be an <integer> between 1 and
	366 inclusive. An implementation may not impose any limits on the year other than those imposed
	on <integer> constants.
	It is possible that the time specified does not exist in the local time specified by the implementation.
	If there is a time change such that the local time is set forward and the time specified is in the
	interregnum, then that time does not exist in the local time. For example if at 02:00 in California on
	April 26, 1997 there is a time change that sets local time forward one hour, then the local time
	02:30 in California does not exist. Conversely if there is a time change that sets the local time back
	there are times which are ambiguous. For example if instead of setting the local time forward from
	02:00 to 03:00 it is set back to 01:00 the local time 01:30 in California is ambiguious. The result
	is the least <DateAndTime> that conforms to the given parameters.
	It is worth noting that the year 1 B.C. is year 0 in the astronomical Gregorian calendar. Similarly the
	year 2 B.C. is year -1 in the astronomical Gregorian calendar and so on. The year 1 A.D. is year 1
	in the astronomical Gregorian calendar.
	Parameters
	year <integer> unspecified
	dayOfYear <integer> unspecified
	hour <integer> unspecified
	minute <integer> unspecified
	second <number> unspecified
	Return Values
	<DateAndTime> new
	Errors
	month is not between 1 and 12 inclusive.
	dayOfYear greater than the number of days in the year year of the astronomical Gregorian
	calendar.
	hour is not between 0 and 23 inclusive.
	minute is not between 0 and 59 inclusive.
	second is not greater than or equal to 0 and strictly less than 60.
	the time specified does not exist."

	^self 
		year: year 
		day: dayOfYear 
		hour: hour 
		minute: minute 
		second: second 
		offset: nil.
%
category: 'DateAndTime factory'
classmethod: DateAndTimeANSI
year: year day: dayOfYear hour: hour minute: minute second: second offset: offset
	"Synopsis
	Answer a <DateAndTime> which is the second second of the minute minute of the hour hour of
	the day dayOfYear of the year year of the astronomical Gregorian calendar offset from UTC by
	offset.
	Definition: <DateAndTime factory>
	Answer the least <DateAndTime> which is the second second of the minute minute of the hour
	hour of the day dayOfYear of the year year of the astronomical Gregorian calendar in the local
	time of the local time specified by the implementation. The second must be a <number> greater
	than or equal to 0 and strictly less than 60. The minute must be an <integer> between 0 and 59
	inclusive. The hour must be an <integer> between 0 and 23 inclusive. The day must be an
	<integer> between 1 and 366 inclusive. An implementation may not impose any limits on the year
	other than those imposed on <integer> constants.
	It is possible that the time specified does not exist in the local time defined by the implementation. If
	there is a time change such that the local time is set forward and the time specified is in the
	interregnum, then that time does not exist in the local time. For example if at 02:00 in California on
	April 26, 1997 there is a time change that sets local time forward one hour, then the local time
	02:30 in California does not exist. Conversely if there is a time change that sets the local time back
	there are times which are ambiguous. For example if instead of setting the local time forward from
	02:00 to 03:00 it is set back to 01:00 the local time 01:30 in California is ambiguious. The result
	is the least <DateAndTime> that conforms to the given parameters.
	Parameters
	year <integer> unspecified
	dayOfYear <integer> unspecified
	hour <integer> unspecified
	minute <integer> unspecified
	second <number> unspecified
	offset <Duration> unspecified
	Return Values
	<DateAndTime> new
	Errors
	month is not between 1 and 12 inclusive.
	dayOfYear greater than the number of days in the year year of the astronomical Gregorian
	calendar.
	hour is not between 0 and 23 inclusive.
	minute is not between 0 and 59 inclusive.
	second is not greater than or equal to 0 and strictly less than the number of seconds in the minute
	specified."

	| isLeapYear daysInYear x days seconds |
	isLeapYear := year \\ 4 == 0 and: [year \\ 100 ~~ 0 or: [year \\ 400 == 0]].
	daysInYear := isLeapYear ifTrue: [366] ifFalse: [365].
	(1 <= dayOfYear and: [dayOfYear <= daysInYear]) ifFalse: [ self error: 'Invalid date'].
	(0 <= second and: [second < 60]) ifFalse: [self error: 'Invalid time'].
	(0 <= minute and: [minute < 60]) ifFalse: [self error: 'Invalid time'].
	(0 <= hour and: [hour < 24]) ifFalse: [self error: 'Invalid time'].
	x := year - 1901.
	days := x * 365
		+ (x // 4)
		- (x // 100)
		+ (x + 300 "1901 - 1601" // 400)
		+ dayOfYear
		- 36526.		"set base to 2001-01-01"
	seconds := days * 24 + hour * 60 + minute * 60 + second.
	^self 
		secondsLocal: seconds
		offset: offset.
%
category: 'DateAndTime factory'
classmethod: DateAndTimeANSI
year: year month: month day: dayOfMonth hour: hour minute: minute second: second
"Synopsis
Answer a <DateAndTime> which is the second second of the minute minute of the hour hour of
the day dayOfMonth of the month month of the year year of the astronomical Gregorian calendar
in local time.
Definition: <DateAndTime factory>
Answer the least <DateAndTime> which is the second second of the minute minute of the hour
hour of the day dayOfMonth of the month month of the year year of the astronomical Gregorian
calendar in the local time specified by the implementation. The second must be a <number>
greater than or equal to 0 and strictly less than 60. The minute must be an <integer> between 0
and 59 inclusive. The hour must be an <integer> between 0 and 23 inclusive. The day must be an
<integer> between 1 and 31 inclusive. The month must be an <integer> between 1 and 12
inclusive. An implementation may not impose any limits on the year other than those imposed on
<integer> constants.
It is possible that the time specified does not exist in the local time defined by the implementation. If
there is a time change such that the local time is set forward and the time specified is in the
interregnum, then that time does not exist in the local time. For example if at 02:00 in California on
April 26, 1997 there is a time change that sets local time forward one hour, then the local time
02:30 in California does not exist. Conversely if there is a time change that sets the locale time
back there are times which are ambiguous. For example if instead of setting the local time forward
from 02:00 to 03:00 it is set back to 01:00 the local time 01:30 in California is ambiguious. The
result is the least <DateAndTime> that conforms to the given parameters.
Parameters
year <integer> unspecified
month <integer> unspecified
dayOfMonth <integer> unspecified
hour <integer> unspecified
minute <integer> unspecified
second <number> unspecified
Return Values
<DateAndTime> new
Errors
month is not between 1 and 12 inclusive.
dayOfMonth greater than the number of days in the month month of year year of the
astronomical Gregorian calendar.
hour is not between 0 and 23 inclusive.
minute is not between 0 and 59 inclusive.
second is not greater than or equal to 0 and strictly less than 60.
the time specified does not exist."

	^self
		year: year 
		month: month 
		day: dayOfMonth 
		hour: hour 
		minute: minute 
		second: second 
		offset: nil.
%
category: 'DateAndTime factory'
classmethod: DateAndTimeANSI
year: year month: month day: dayOfMonth hour: hour minute: minute second: second offset: offset
	"Synopsis
	Answer a <DateAndTime> which is the second second of the minute minute of the hour hour of
	the day dayOfMonth of the month month of the year year of the astronomical Gregorian calendar
	offset from UTC by offset.
	Definition: <DateAndTime factory>
	Answer the least <DateAndTime> which is the second second of the minute minute of the hour
	hour of the day dayOfMonth of the month month of the year year of the astronomical Gregorian
	calendar offset from UTC by offset. The second must be a <number> greater than or equal to 0
	and strictly less than 60. The minute must be an <integer> between 0 and 59 inclusive. The hour
	must be an <integer> between 0 and 23 inclusive. The day must be an <integer> between 1 and
	31 inclusive. The month must be an <integer> between 1 and 12 inclusive. An implementation may
	not impose any limits on the year other than those imposed on <integer> constants.
	It is possible that the time specified does not exist in the local time defined by the implementation. If
	there is a time change such that the local time is set forward and the time specified is in the
	interregnum, then that time does not exist in the local time. For example if at 02:00 in California on
	April 26, 1997 there is a time change that sets local time forward one hour, then the local time
	02:30 in California does not exist. Conversely if there is a time change that sets the local time back
	there are times which are ambiguous. For example if instead of setting the local time forward from
	02:00 to 03:00 it is set back to 01:00 the local time 01:30 in California is ambiguious. The result
	is the least <DateAndTime> that conforms to the given parameters.
	Parameters
	year <integer> unspecified
	month <integer> unspecified
	dayOfMonth <integer> unspecified
	hour <integer> unspecified
	minute <integer> unspecified
	second <number> unspecified
	offset <Duration> unspecified
	Return Values
	<DateAndTime> new
	Errors
	month is not between 1 and 12 inclusive.
	dayOfMonth greater than the number of days in the month month of year year of the
	astronomical Gregorian calendar.
	hour is not between 0 and 23 inclusive.
	minute is not between 0 and 59 inclusive.
	second is not greater than or equal to 0 and strictly less than 60."

	| isLeapYear dayOfYear daysPerMonth daysInMonth |
	(1 <= month and: [month <= 12]) ifFalse: [self error: 'Invalid date'].
	isLeapYear := year \\ 4 == 0 and: [year \\ 100 ~~ 0 or: [year \\ 400 == 0]].
	daysPerMonth := isLeapYear
		ifTrue: [#(31 29 31 30 31 30 31 31 30 31 30 31)]
		ifFalse: [#(31 28 31 30 31 30 31 31 30 31 30 31)].
	daysInMonth := daysPerMonth at: month.
	(1 <= dayOfMonth and: [dayOfMonth <= daysInMonth]) ifFalse: [self error: 'Invalid date'].
	dayOfYear := (#(0 31 59 90 120 151 181 212 243 273 304 334) at: month) + dayOfMonth.
	(month > 2 and: [isLeapYear]) ifTrue: [dayOfYear := dayOfYear + 1].
	^self
		year: year 
		day: dayOfYear 
		hour: hour 
		minute: minute 
		second: second 
		offset: offset.
%
! fix 42920
category: 'other'
classmethod: DateAndTimeANSI
fromString: aString
	"YYYY-MM-DDTHH:MM:SS+HH:MM"

  | sz | sz := aString size .
	^self
		year:   (aString copyFrom:  1 to:  4) asNumber
		month:  (aString copyFrom:  6 to:  7) asNumber
		day:    (aString copyFrom:  9 to: 10) asNumber
		hour:   (aString copyFrom: 12 to: 13) asNumber
		minute: (aString copyFrom: 15 to: 16) asNumber
		second: (aString copyFrom: 18 to: sz - 6) asNumber 
		offset: (Duration
			days:    0 
			hours:   (aString copyFrom: sz - 4 to: sz - 3 ) asNumber * 
								  ((aString at: sz - 5) == $+ ifTrue: [1] ifFalse: [-1])
			minutes: (aString copyFrom: sz - 1 to: sz) asNumber
			seconds: 0)
%
category: 'other'
classmethod: DateAndTimeANSI
posixSeconds: aNumber offset: aDuration
	"Unix time, or POSIX time, is a system for describing instances in time, 
	defined as the number of seconds that have elapsed since 00:00:00 Coordinated 
	Universal Time (UTC), 1 January 1970, not counting leap seconds. It is used 
	widely in Unix-like and many other operating systems and file formats. 
	Due to its handling of leap seconds, it is neither a linear representation 
	of time nor a true representation of UTC. 
	--http://en.wikipedia.org/wiki/Unix_time"
	
	| res |
	(res := super new) _posixSeconds: aNumber offset: aDuration.
	 ^ res.
%
category: 'other'
classmethod: DateAndTimeANSI
secondsLocal: aNumber offset: aDuration
  | res |
	(res := super new)
		_secondsLocal: aNumber offset: aDuration.
  ^ res
%
category: 'other'
classmethod: DateAndTimeANSI
secondsSince2001

	self subclassResponsibility: #secondsSince2001
%
category: 'other'
classmethod: DateAndTimeANSI
secondsUTC: aNumber offset: aDuration
  | res |
	(res := super new)
		_secondsUTC: aNumber offset: aDuration.
	^ res
%
category: 'instance creation'
classmethod: DateAndTimeANSI
new

"Disallowed"

self shouldNotImplement: #new
%

! ------------------- Instance methods for DateAndTimeANSI
category: 'DateAndTime'
method: DateAndTimeANSI
+ operand
	"Answer a <DateAndTime> that represents the UTC time that is operand after the receiver and
	whose local time is the same as the receiver's. If operand is less than <Duration factory> #zero,
	the result is the <DateAndTime> that is that is the absolute value of operand before the receiver."

	^self class 
		secondsUTC: self asSeconds + operand asSeconds
		offset: self offset.
%
category: 'DateAndTime'
method: DateAndTimeANSI
- operand
	"If operand is a <DateAndTime>, answer a <Duration> whose value is the period of time between
	the operand and the receiver. If operand is a <DateAndTime> prior to the receiver then the result
	is a <Duration> less than <Duration factory> #zero.
	If operand is a <Duration>, answer a new <DateAndTime> which represents the UTC time that is
	operand before the receiver and whose local time is the same as the receiver's. If operand is a
	duration less than <Duration factory> #zero then the result is a <DateAndTime> that is the
	absolute value of operand after the receiver."

	^(operand isKindOf: Duration) ifTrue: [
		self class 
			secondsUTC: self asSeconds - operand asSeconds
			offset: self offset.
	] ifFalse: [
		Duration seconds: self asSeconds - operand asSeconds.
	].
%
category: 'DateAndTime'
method: DateAndTimeANSI
< aDateAndTime
	"Synopsis
	Answer true if the receiver is less than operand. Answer false otherwise.
	Definition: <magnitude>
	Answer true if the receiver is less than operand with respect to the ordering defined for them.
	Answer false otherwise.
	It is erroneous if the receiver and operand are not comparable.
	The semantics of the natural ordering must be defined by refinement, which may also restrict the
	type of operand.
	Refinement: <DateAndTime>
	Answer true if the UTC time represented by operand follows the UTC time represented by the
	receiver. Answer false otherwise.
	If the offsets of the receiver and operand are the same then their order is determined by their
	lexical order in the sequence #year, #month, #day, #hour24, #minute, #second. If their
	offsets differ then result is the same as if receiver asUTC < operand asUTC were evaluated."

	^self asSeconds < aDateAndTime asSeconds.
%
category: 'DateAndTime'
method: DateAndTimeANSI
= aMagnitude
	"Synopsis
	Object equivalence test.
	Definition: <Object>
	This message tests whether the receiver and the comparand are equivalent objects at the time the
	message is processed. Return true if the receiver is equivalent to comparand. Otherwise return
	false.
	The meaning of 'equivalent' cannot be precisely defined but the intent is that two objects are
	considered equivalent if they can be used interchangeably. Conforming protocols may choose to
	more precisely define the meaning of 'equivalent.'
	The value of
	receiver = comparand
	is true if and only if the value of
	comparand = receiver
	would also be true. If the value of
	receiver = comparand
	is true then the receiver and comparand must have equivalent hash values. Thus if
	receiver = comparand
	is true then
	receiver hash = comparand hash
	must also be true.
	The equivalence of objects need not be temporally invariant. Two independent invocations of #=
	with the same receiver and operand objects may not always yield the same results. Note that a
	collection that uses #= to discriminate objects may only reliably store objects whose hash values
	do not change while the objects are contained in the collection.
	Refinement: <DateAndTime>
	Answer true if the comparand conforms to <DateAndTime> and if it represents the
	same UTC time as the receiver. Answer false otherwise. The local times of the receiver and
	operand are ignored."

	^(aMagnitude isKindOf: DateAndTimeANSI) _and: [self asSeconds = aMagnitude asSeconds].
%
category: 'public'
method: DateAndTimeANSI
asDays

	| x |
	x := seconds.	"seconds from base"
	x := x // 60.	"minutes from base"
	x := x // 60.	"hours from base"
	x := x // 24.	"days from base"
	^x.
%
category: 'private'
method: DateAndTimeANSI
asFloatParts
	"#(year dayOfYear monthIndex dayOfMonth hour minute second)"

	^self partsFrom: seconds + offset.	"seconds from base"
%
category: 'private'
method: DateAndTimeANSI
asIntegerParts
	"#(year dayOfYear monthIndex dayOfMonth hour minute second)"

	^ self partsFrom: (seconds + offset) roundedNoFpe "seconds from base"
%
category: 'DateAndTime'
method: DateAndTimeANSI
asLocal
	"Synopsis
	Answer a <DateAndTime> that represents the same UTC time as the receiver but in the local time
	specified by the implementation.
	Definition: <DateAndTime>
	Answer a <DateAndTime> that represents the same UTC time as the receiver but in the local time
	specified by the implementation.
	Return Values
	<DateAndTime> unspecified"

	^self class 
		secondsUTC: seconds
		offset: nil.
%
category: 'private'
method: DateAndTimeANSI
asParts
	"#(year dayOfYear monthIndex dayOfMonth hour minute second)"

	^self asFloatParts.
%
category: 'public'
method: DateAndTimeANSI
asPosixSeconds
	"Unix time, or POSIX time, is a system for describing instances in time, 
	defined as the number of seconds that have elapsed since 00:00:00 Coordinated 
	Universal Time (UTC), 1 January 1970, not counting leap seconds. It is used 
	widely in Unix-like and many other operating systems and file formats. 
	Due to its handling of leap seconds, it is neither a linear representation 
	of time nor a true representation of UTC. 
	--http://en.wikipedia.org/wiki/Unix_time"

	^seconds + 978307200.
%
category: 'public'
method: DateAndTimeANSI
asSeconds
	"The number of seconds that have elapsed since 00:00:00 Coordinated 
	Universal Time (UTC), 1 January 2001, not counting leap seconds."

	^seconds.
%
category: 'DateAndTime'
method: DateAndTimeANSI
asUTC
	"Synopsis
	Answer a <DateAndTime> that represents the same absolute time as the receiver but in the local
	time UTC.
	Definition: <DateAndTime>
	Answer a <DateAndTime> that represents the same absolute time as the receiver but in the local
	time UTC. The exact meaning of UTC local time is specified by the implementation. The UTC local
	time must use the Gregorian calendar. <DateAndTimes> representing UTC times prior to the
	adoption of the Gregorian calendar must use the retrospective astronomical Gregorian calendar. It
	is an invariant that
	<DateAndTime> asUTC offset = Duration zero.
	Return Values
	<DateAndTime> unspecified"

	^self class
		secondsLocal: seconds
		offset: Duration zero.

%
category: 'public'
method: DateAndTimeANSI
beRounded 

	seconds := seconds roundedNoFpe
%
category: 'initialize-release'
method: DateAndTimeANSI
currentTimeZone

	^TimeZoneInfo default.
%
category: 'DateAndTime'
method: DateAndTimeANSI
dayOfMonth
	"Synopsis
	Answer the number of the day in the month in the local time of the receiver which includes the
	receiver.
	Definition: <DateAndTime>
	Answer an <integer> between 1 and 31 inclusive representing the number of the day in the month,
	in the local time of the receiver, which includes the receiver.
	Return Values
	<integer>unspecified"

	^self asParts at: 4.
%
category: 'DateAndTime'
method: DateAndTimeANSI
dayOfWeek
	"Synopsis
	Answer the number of the day in the week, in the local time of the receiver, which includes the
	receiver.
	Definition: <DateAndTime>
	Answer an <integer> between 1 and 7 inclusive representing the number of the day in the week, in
	the local time of the receiver, which includes the receiver. Sunday is 1, Monday is 2, and so on.
	Return Values
	<integer>unspecified"

	^ ((seconds + offset) // 60) // 60 // 24 + 1 \\ 7 + 1  "1 January 2001 was a Monday"
%
category: 'DateAndTime'
method: DateAndTimeANSI
dayOfWeekAbbreviation
	"Synopsis
	Answer the abbreviation of the name, in the local time of the receiver, of the day of the week which
	includes the receiver.
	Definition: <DateAndTime>
	Answer an <readableString> which is the abbreviation of the name, in the local time of the
	receiver, of the day of the week which includes the receiver.
	Return Values
	<readableString> unspecified"

	^self dayOfWeekName copyFrom: 1 to: 3.
%
category: 'DateAndTime'
method: DateAndTimeANSI
dayOfWeekName
	"Synopsis
	Answer the name, in the local time of the receiver, of the day of the week which includes the
	receiver.
	Definition: <DateAndTime>
	Answer an <readableString> which is the name, in the local time of the receiver, of the day of the
	week which includes the receiver.
	Return Values
	<readableString> unspecified"

	^#('Sunday' 'Monday' 'Tuesday' 'Wednesday' 'Thursday' 'Friday' 'Saturday') at: self dayOfWeek.
%
category: 'DateAndTime'
method: DateAndTimeANSI
dayOfYear
	"Synopsis
	Answer the number of the day in the year, in the local time of the receiver, which includes the
	receiver.
	Definition: <DateAndTime>
	Answer an <integer> between 1 and 366 inclusive representing the number of the day in the year,
	in the local time of the receiver, which includes the receiver.
	Return Values
	<integer>unspecified"

	^self asParts at: 2.
%
category: 'comparing'
method: DateAndTimeANSI
hash

	^seconds hash.
%
category: 'DateAndTime'
method: DateAndTimeANSI
hour
	"Synopsis
	Answer the number of the hour in the day, in the local time of the receiver, which includes the
	receiver.
	Definition: <DateAndTime>
	Answer an <integer> between 0 and 23 inclusive representing the number of the hour in the day, in
	the local time of the receiver, which includes the receiver. It is implementation defined whether a
	given local time uses the 12-hour clock or the 24-hour clock, except that the UTC local time must
	use the 24-hour clock.
	Return Values
	<integer>unspecified"

	^ ((seconds + offset) // 60) // 60 \\ 24.
%
category: 'DateAndTime'
method: DateAndTimeANSI
hour12

	"Synopsis
	Answer the hour in the day in the 12-hour clock of the local time of the receiver.
	Definition: <DateAndTime>
	Answer an <integer> between 1 and 12 inclusive representing the hour in the day in the 12-hour
	clock of the local time of the receiver.
	Return Values
	<integer>unspecified"

	^self hour - 1 \\ 12 + 1.
%
category: 'DateAndTime'
method: DateAndTimeANSI
hour24
	"Synopsis
	Answer the hour in the day in the 24-hour clock of the local time of the receiver.
	Definition: <DateAndTime>
	Answer an <integer> between 0 and 23 inclusive representing the hour in the day in the 24-hour
	clock of the local time of the receiver.
	Return Values
	<integer>unspecified"

	^self hour.
%
category: 'initialize-release'
method: DateAndTimeANSI
initializeAsNow

	seconds := self class secondsSince2001.
	offset := (self currentTimeZone transitionAtUTC: self) offsetFromUTC.
%
category: 'DateAndTime'
method: DateAndTimeANSI
isLeapYear
	"Synopsis
	Test for leap year.
	Definition: <DateAndTime>
	Answer true if the year, which includes the receiver, in the local time of the receiver is a leap year,
	false otherwise.
	Two <DateAndTime> objects that are equal can give different results for #isLeapYear. Equality
	depends on their UTC time whereas #isLeapYear depends on their local time.
	Return Values
	<boolean> unspecified"

	| year |
	year := self year.
	^year \\ 4 == 0 _and: [year \\ 100 ~~ 0 _or: [year \\ 400 == 0]].
%
category: 'DateAndTime'
method: DateAndTimeANSI
meridianAbbreviation
	"Synopsis
	Answer the abbreviation, in the local time of the receiver, of the name of the half of the day, which
	includes the receiver.
	Definition: <DateAndTime>
	Answer a <readableString> that is the abbreviation, in the local time of the receiver, of the name of
	the half of the day, which includes the receiver.
	Return Values
	<readableString> unspecified"

	^self hour < 12
		ifTrue: ['AM']
		ifFalse: ['PM'].
%
category: 'DateAndTime'
method: DateAndTimeANSI
minute
	"Synopsis
	Answer the minute of the hour in the local time of the receiver.
	Definition: <DateAndTime>
	Answer an <integer> between 0 and 59 inclusive representing the minute of hour in the local time
	of the receiver.
	Return Values
	<integer>unspecified"

	^ ((seconds + offset) // 60) \\ 60.
%
category: 'DateAndTime'
method: DateAndTimeANSI
month
	"Synopsis
	Answer the number of the month in the year, in the local time of the receiver, which includes the
	receiver.
	Definition: <DateAndTime>
	Answer an <integer> between 1 and 12 inclusive representing the number of the month in the year,
	in the local time of the receiver, which includes the receiver.
	Return Values
	<integer>unspecified"

	^self asParts at: 3.
%
category: 'DateAndTime'
method: DateAndTimeANSI
monthAbbreviation
	"Synopsis
	Answer the abbreviation of the name of the month, in the local time of the receiver, which includes
	the receiver.
	Definition: <DateAndTime>
	Answer a <readableString> that is the abbreviation of the name of the month, in the local time of
	the receiver, which includes the receiver.
	Return Values
	<readableString> unspecified"

	^self monthName copyFrom: 1 to: 3.
%
category: 'DateAndTime'
method: DateAndTimeANSI
monthName
	"Synopsis
	Answer the name of the month, in the local time of the receiver, which includes the receiver.
	Definition: <DateAndTime>
	Answer a <readableString> that is the name of the month, in the local time of the receiver, which
	includes the receiver.
	Return Values
	<readableString> unspecified"

	^#(
		'January' 'February' 'March'
		'April' 'May' 'June'
		'July' 'August' 'September'
		'October' 'November' 'December'
	) at: self month.
%
category: 'DateAndTime'
method: DateAndTimeANSI
offset
	"Synopsis
	Answer the difference between the local time of the receiver and UTC at the time of the receiver.
	Definition: <DateAndTime>
	Answer a <Duration> representing the difference between the local time of the receiver and UTC at
	the time of the receiver.
	Return Values
	<Duration> unspecified"

	^Duration seconds: offset.
%
category: 'DateAndTime'
method: DateAndTimeANSI
offset: aDuration
	"Synopsis
	Answer a <DateAndTime> equivalent to the receiver but with its local time being offset from UTC
	by offset.
	Definition: <DateAndTime>
	Answer a <DateAndTime> equivalent to the receiver but with its local time being offset from UTC
	by offset. The impact of this on any other local time property is unspecified.
	Implementations may define a limit to the range of offset, but it must be at least
	-12:00:00 to 12:00:00 inclusive.
	It is an invariant that if x is a <Duration> in range then
	(<DateAndTime> offset: x) offset = x
	Parameters
	offset <Duration> unspecified
	Return Values
	<DateAndTime> unspecified
	Errors
	offset out of range"

	offset := aDuration asSeconds.
%
category: 'private'
method: DateAndTimeANSI
partsFrom: secondsFromBase
	"#(year dayOfYear monthIndex dayOfMonth hour minute second)"
	| x y array year isLeapYear numDaysArr |
	array := Array new: 7.
      Float noInexactResultDo:[ 
	x := secondsFromBase .	"secondsFromBase is possibly a Float"
	array at: 7 put: x \\ 60 .
	x := x // 60 .	"minutes from base, an Integer"
      ].
	array at: 6 put: x \\ 60 .
	x := x // 60.	"hours from base"
	array at: 5 put: x \\ 24.
	x := x // 24.	"days from base"
	year := 2001.
	y := x // 146097.
	year := y * 400 + year.
	x := x - (y * 146097).		"days since beginning of 400-year cycle"

	y := x // 36524 min: 3.
	year := y * 100 + year.
	x := x - (y * 36524).		"days since beginning of 100-year cycle"

	y := x // 1461 min: 96.
	year := y * 4 + year.
	x := x - (y * 1461).			"days since beginning of 4-year cycle"

	y := x // 365 min: 3.
	year := y + year.
	x := x - (y * 365) + 1.		"days since beginning of year"
	array at: 1 put: year;
	      at: 2 put: x .
	x <= 31 ifTrue: [
		 array at: 3 put: 1;
			at: 4 put: x .
	         ^ array
	].
	x <= 59 ifTrue: [
		 array at: 3 put: 2;
			at: 4 put: x - 31 .
		^ array .
	].
	isLeapYear := year \\ 4 == 0 _and: [year \\ 100 ~~ 0 _or: [year \\ 400 == 0]].
	isLeapYear ifTrue: [
		x == 60 ifTrue: [
			array at: 3 put: 2;
				at: 4 put: 29 .
			^ array .
		].
		x := x - 1.
	].
	array at: 3 put: 3.
	x := x - 59.
	numDaysArr := #(31 30 31 30 31 31 30 31 30 31)  .
	1 to: numDaysArr size do:[:j | | each |
		each := numDaysArr at: j .
		x <= each ifTrue: [
			array at: 4 put: x .
			^ array .
		].
		array at: 3 put: (array at: 3) + 1.
		x := x - each.
	].
	self error: 'invalid date'.
%
category: 'public'
method: DateAndTimeANSI
posixSeconds: anInteger
	"Unix time, or POSIX time, is a system for describing instances in time, 
	defined as the number of seconds that have elapsed since 00:00:00 Coordinated 
	Universal Time (UTC), 1 January 1970, not counting leap seconds. It is used 
	widely in Unix-like and many other operating systems and file formats. 
	Due to its handling of leap seconds, it is neither a linear representation 
	of time nor a true representation of UTC. 
	--http://en.wikipedia.org/wiki/Unix_time"

	seconds := anInteger - 978307200.
%

category: 'private'
method: DateAndTimeANSI
printTimeFrom: anArray on: aStream 

	| x |
	(x := anArray at: 5) < 10 ifTrue: [aStream nextPut: $0].
	x printOn: aStream.
	aStream nextPut: $:.
	(x := anArray at: 6) < 10 ifTrue: [aStream nextPut: $0].
	x printOn: aStream.
	aStream nextPut: $:.
	x := anArray at: 7.
	x \\ 1 = 0 ifTrue: [
		x < 10 ifTrue: [aStream nextPut: $0].
		x roundedNoFpe printOn: aStream.
	] ifFalse: [	"See bugs 36718, 40624, 40643, and 40883"
		(x _isFloat or:[ x _isScaledDecimal]) ifFalse:[ x := x asFloat "handle fraction"].
		x asInteger < 10 ifTrue:[ aStream nextPut: $0 ].
		aStream nextPutAll: x asStringLocaleC  
	].
%

category: 'private'
method: DateAndTimeANSI
printDateFrom: anArray on: aStream 
	| x |
	x := anArray at: 1.
	x < 0 ifTrue: [aStream nextPut: $-].
	(x := x abs) < 1000 ifTrue: [aStream nextPut: $0].
	x < 100 ifTrue: [aStream nextPut: $0].
	x < 10 ifTrue: [aStream nextPut: $0].
	x printOn: aStream.
	aStream nextPut: $-.
	(x := anArray at: 3) < 10 ifTrue: [aStream nextPut: $0].
	x printOn: aStream.
	aStream nextPut: $-.
	(x := anArray at: 4) < 10 ifTrue: [aStream nextPut: $0].
	x printOn: aStream.
%

category: 'private'
method: DateAndTimeANSI
print: anArray on: aStream 

	self printDateFrom: anArray on: aStream.
	aStream nextPut: $T.
	self printTimeFrom: anArray on: aStream.
	self offset printOnHoursMinutes: aStream.
%

category: 'private'
method: DateAndTimeANSI
printAnsiOn: aStream 

	self 
		print: self asFloatParts
		on: aStream.
%
category: 'public'
method: DateAndTimeANSI
printOn: aStream 

	self printAnsiOn: aStream.
%
category: 'private'
method: DateAndTimeANSI
printRoundedOn: aStream 

	self 
		print: self asIntegerParts
		on: aStream.
%
category: 'DateAndTime'
method: DateAndTimeANSI
printString
	"Synopsis
	Return a string that describes the receiver.
	Definition: <Object>
	A string consisting of a sequence of characters that describe the receiver are returned as the
	result.
	The exact sequence of characters that describe an object are implementation defined.
	Refinement: <DateAndTime>
	The returned string will represent the UTC time of the receiver offset from UTC by the offset of the
	receiver. All dates are in the astronomical Gregorian calendar. The result will be formatted as
	-YYYY-MM-DDThh:mm:ss.s+ZZ:zz:z where
	- is the <Character> $- if the year is less than 0 otherwise it is the <Character> that is
	returned from the message #space sent to the standard global Character
	[but see VAST which makes a case that the space should be excluded],
	YYYY is the year left zero filled to four places,
	- is the <Character> $-,
	MM is the month of the year left zero filled to two places,
	- is the <Character> $-,
	DD is the day of the month left zero filled to two places,
	T is the <Character> $T,
	hh is the hour in the 24-hour clock left zero filled to two places,
	: is the <Character> $:,
	mm is the minute left zero filled to two places,
	: is the <Character> $:,
	ss is the second left zero filled to two places,
	. is the <Character> $. and is present only if the fraction of a second is non-zero,
	s is the fraction of a second and is present only if non-zero,
	+ is the <Character> $+ if the offset is greater than or equal to <Duration factory> #zero
	and the <Character> $- if it is less,
	ZZ is the hours of the offset left zero filled to two places, and
	: is the <Character> $:,
	zz is the minutes of the offset left zero filled to two places,
	: is the <Character> $: and is present only if the seconds of the offset is non-zero,
	z is the seconds of the offset including any fractional part and is present only if nonzero.
	This format is based on ISO 8601 sections 5.3.3 and 5.4.1.
	Example: 8:33:14.321 PM EST January 5, 1200 B.C.
	-1199-01-05T20:33:14.321-05:00
	Example: 12 midnight UTC January 1, 2001 A.D.
	 2001-01-01T00:00:00+00:00
	Return Values
	<readableString> unspecified"

	^super printString.
"
	^self printStringWithRoundedSeconds.
"
%

! add asString so topaz will display an instance in a useful way.
method: DateAndTimeANSI
asString
^ self printStringWithRoundedSeconds
%

category: 'private'
method: DateAndTimeANSI
printStringWithRoundedSeconds 

	| stream |
	stream := AppendStream on: String new .
	self printRoundedOn: stream.
	^stream contents.
%
category: 'public'
method: DateAndTimeANSI
rounded 

	^self copy beRounded.
%
category: 'DateAndTime'
method: DateAndTimeANSI
second
	"Synopsis
	Answer the second of the minute of the local time of the receiver.
	Definition: <DateAndTime>
	Answer a <number> greater than or equal to 0 and strictly less than 60 representing the second of
	the minute of the local time of the receiver.
	Return Values
	<number> unspecified"

	^seconds + offset \\ 60.
%
category: 'DateAndTime'
method: DateAndTimeANSI
timeZoneAbbreviation
	"Synopsis
	Answer the abbreviation of the name, in the local time of the receiver, of the time zone of the
	receiver.
	Definition: <DateAndTime>
	Answer a <readableString> that is the abbreviation of the name, in the local time of the receiver, of
	the time zone of the receiver.
	Return Values
	<readableString> unspecified"

	^self timeZoneName
%
category: 'DateAndTime'
method: DateAndTimeANSI
timeZoneName
	"Synopsis
	Answer the name in the local time of the receiver of the time zone of the receiver.
	Definition: <DateAndTime>
	Answer a <readableString> that is the name in the local time of the receiver of the time zone of the
	receiver.
	Return Values
	<readableString> unspecified"

	"There are currently (2013d version) 1759 distinct timezone names in the IANA 
	database. There *might* be a few offsets that can be easily disambiguated, but 
	most offsets will be common to dozens of different timezones. (Bug #43388)"
	
	| stream |
	stream := AppendStream on: String new.
	self offset printOnHoursMinutes: stream.
	^stream contents.
%
category: 'DateAndTime'
method: DateAndTimeANSI
year
	"Synopsis
	Answer the number of the year in the local time of the receiver which includes the receiver.
	Definition: <DateAndTime>
	Answer an <integer> the number of the year which includes the receiver.
	Return Values
	<integer>unspecified"

	^self asParts at: 1.
%
category: 'private'
method: DateAndTimeANSI
_secondsLocal: aNumber offset: anObject

	seconds := aNumber.
	anObject ifNil:[ 
		offset := TimeZoneInfo default offsetAtLocal: self.
	] ifNotNil: [
		offset  := anObject asSeconds.
	].
	seconds := seconds - offset.
%
category: 'private'
method: DateAndTimeANSI
_secondsUTC: aNumber offset: anObject

	seconds := aNumber.
	anObject ifNil: [
		offset := TimeZoneInfo default offsetAtUTC: self.
	] ifNotNil: [
		offset  := anObject asSeconds.
	].
%

category: 'private'
method: DateAndTimeANSI
_posixSeconds: aNumber offset: aDuration
  self posixSeconds: aNumber .
  aDuration ifNil: [
    offset := TimeZoneInfo default offsetAtUTC: self.
  ] ifNotNil: [
    offset  := aDuration asSeconds.
  ].
%
