! ========================================================================
! GsExternalSession.gs
!
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
! ========================================================================


! Provide placeholders for references to classes that cannot be generated yet.
! (The classes will be generated during the "bomlastconv.gs" phase.
doit
	Globals
		at: #GciLibrary ifAbsentPut: [nil];
		at: #GciErrSType ifAbsentPut: [nil] ;
    at: #GsLegacyExternalSession ifAbsentPut:[ nil ] .
	^true
%

expectvalue %String
doit
(Object subclass: 'GsExternalSession'
	instVarNames: #( parameters gciSessionId stoneSessionId stoneSessionSerial gemProcessId gciErrSType nbResult logger)
	classVars: #()
	classInstVars: #()
	poolDictionaries: #()
	inDictionary: Globals
	options: #()) definition
%

! ------------------- Class comment for GsExternalSession
doit
GsExternalSession comment: 
'GsExternalSession provides a means of running Smalltalk expressions in a different Gem,
possibly on another server or against a different Stone. The expression to execute can
be provided as a String (via #executeString: and #forkString:) or as a Block (via #executeBlock:
and #forkBlock:).

The #executeBlock: protocol allows the use of all the usual Smalltalk tools for working
with Smalltalk code, but does not support arguments to the Block and also requires that
the code compiles in the local image.

The #executeString: protocol supports the use of any String, but being a String, it is
opaque to the operation of most Smalltalk tools, such as the Refactoring Browser, for example.

The "execute" methods are synchronous; the "fork" methods are asynchronous and require checking
for results sometime later. When forking execution, use the #isResultAvailable, #waitForResult (and
variations), and #lastResult methods to get the results of the remote execution.

The Gems launched by the external sessions are not managed by the system, so ensure sessions are 
logged out to prevent lingering sessions consuming system resources unnecessarily.


Examples:

UserGlobals at: #Sample put: GsExternalSession new. "for testing; instances should not be persisted"
Sample 
	stoneNRS: GsNetworkResourceString defaultStoneNRSFromCurrent;
	gemNRS: GsNetworkResourceString defaultGemNRSFromCurrent;
	username: ''DataCurator'';
	password: ''swordfish'';
	yourself.
Sample login.
Sample executeString: ''System stoneName''.
Sample executeBlock: [2 + 5].
Sample executeBlock: [nil].
Sample executeString: ''true''.
Sample logout.
'.
true
%

doit
GsExternalSession category: 'External Sessions'.
true
%
set class GsExternalSession

! Remove existing behavior from GsExternalSession
removeallmethods 
removeallclassmethods 

! ------------------- Class methods for GsExternalSession
category: 'GciLibrary'
classmethod:
gciErrSTypeClass

	^GciErrSType.
%
category: 'GciLibrary'
classmethod:
gciLibrary

	^SessionTemps current
		at: #'gciLibrary'
		ifAbsentPut: [
			GciLibrary new
				GciInit;
				yourself.
		].
%
category: 'Instance Creation'
classmethod:
gemNRS: gemNRS stoneNRS: stoneNRS username: aUsername password: aPassword
  | res |
	(res := self new)
		gemNRS: gemNRS;
		stoneNRS: stoneNRS;
		username: aUsername;
		password: aPassword.
  res class gciLibrary .
  ^ res .
%
category: 'Instance Creation'
classmethod:
gemNRS: gemNRS stoneNRS: stoneNRS username: gsUsername password: gsPassword hostUsername: hostUsername hostPassword: hostPassword
  | res |
	(res := self new)
		gemNRS: gemNRS;
		stoneNRS: stoneNRS;
		username: gsUsername;
		password: gsPassword;
		hostUsername: hostUsername;
		hostPassword: hostPassword .
  res class gciLibrary .
  ^ res
%
category: 'Private' 
classmethod: GsExternalSession
sessionClass
	^(System gemVersionAt: 'osName') = 'AIX' 
		ifTrue: [GsLegacyExternalSession]
		ifFalse: [ self "change to GsLegacyExternalSession to not use FFI"]
%

category: 'Instance Creation'
classmethod: GsExternalSession
new
	^(self sessionClass basicNew)
		initialize;
		yourself
%

category: 'Instance Creation'
classmethod:
newDefault
  "This creates an external session that is set to the user, host,
   and stone of the current gem with the password 'swordfish'. This
   is enough to use in an environment where NetLDI uses a captive
   account, but is not sufficient for a more complex environment or
   one where the password has been changed. In that case you use 
   other Instance Creation methods." 

	 | res | 
	 (res := self new)
		  initializeDefaultResources .
   res class gciLibrary .
   ^ res
%

! ------------------- Instance methods for GsExternalSession
category: 'Public'
method:
abort
	"Abort the current transaction in the external Gem."

  | lib | 
  lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	lib GciNbAbort.
	self waitForResult .
	nbResult ~~ 0 ifTrue: [self error: 'Unexpected result!'].
%
category: 'Private'
method:
clearStackFor: anError

	| contextOop lib |
	(contextOop := anError context) == 20"nil asOop" ifTrue: [^self].
  lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	lib GciClearStack_: contextOop.
	self _signalIfError: lib .
%
category: 'Public'
method:
commit
	"Commit the current transaction in the external Gem."

	| result lib |
  lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	result := lib GciCommit.
  result == 1 ifFalse:[
	  self _signalIfError: lib .
  ].
	^ result == 1.
%
category: 'Public'
method:
continue: contextOop
	"Continue execution in the external Gem following an exception."

	^self 
		continue: contextOop 
		replacingTopOfStackWithOop: 1	"src/gcioop.ht:#define OOP_ILLEGAL         ((OopType)0x01)" 
%
category: 'Private'
method:
continue: contextOop replacingTopOfStackWithOop: valueOop
	"Continue execution in the external Gem following an exception,
	 replacing the top of the stack with the specified object.
	 It is an error to specify an oop not visible to the remote Gem."
  | lib |
	lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	lib GciNbContinueWith_: contextOop 
		_: valueOop
		_: 0
		_: nil.
	^self 
		waitForResult;
		_lastResult: lib .
%
category: 'Public'
method:
continue: contextOop with: anObject
	"Continue execution in the external Gem following an exception,
	 replacing the top of the stack with the specified object.
	 It is an error if anObject is not visible to the remote Gem."

	^self 
		continue: contextOop 
		replacingTopOfStackWithOop: anObject asOop
%
category: 'Private'
method:
_stringForBlock: aBlock
	| string |
	string := aBlock method _sourceStringForBlock .
  (string at: 1) == $[  ifFalse:[ Error signal:'malformed source'].
  "replace [ ]  with spaces"
  string at: 1 put: $  ; at: string size put: $  .
  ^ string
%

category: 'Public'
method:
executeBlock: aBlock
	"Execute the code in the Block argument in the external Gem
	 and answer the result. The best values to return are specials.
	 Strings and other byte objects are copied to the local Gem.
	 All other responses are returned as a Array containing a single OOP."

	^self executeString: (self _stringForBlock: aBlock)
%
category: 'Public'
method:
executeBlock: aBlock with: aValue
	"Execute the code in the Block argument in the external Gem, passing
	 in the specified value, and answer the result. The value passed to the
	 Block must be one whose printString allows the correct object state to
	 be recreated (such as numbers and strings, for example).
	 The best values to return are specials.
	 Strings and other byte objects are copied to the local Gem.
	 All other responses are returned as a Array containing a single OOP."

	^self executeBlock: aBlock withArguments: {aValue}.
%
category: 'Public'
method:
executeBlock: aBlock with: aValue with: anotherValue
	"Execute the code in the Block argument in the external Gem, passing
	 in the specified values, and answer the result. The values passed to the
	 Block must be ones whose printString allows the correct object state to
	 be recreated (such as numbers and strings, for example).
	 The best values to return are specials.
	 Strings and other byte objects are copied to the local Gem.
	 All other responses are returned as a Array containing a single OOP."

	^self executeBlock: aBlock withArguments: {aValue. anotherValue}.
%
category: 'Private'
method:
_stringForBlock: aBlock withArguments: someValues
  | stream string |
	aBlock numArgs == someValues size ifFalse: [self error: 'Wrong number of arguments'].
	stream := AppendStream on: String new .
	stream nextPutAll: aBlock method _sourceStringForBlock .
	stream nextPutAll: ' valueWithArguments: {'.
	1 to: someValues size do: [:index | | each |
    each := someValues at: index .
		index > 1 ifTrue: [stream nextPutAll: '. '].
		each printOn: stream.
	].
	stream nextPut: $}  .
	string := stream contents.
  (string at: 1) == $[  ifFalse:[ Error signal:'malformed source'].
  ^ string
%

category: 'Public'
method:
executeBlock: aBlock withArguments: someValues
	"Execute the code in the Block argument in the external Gem, passing
	 in the specified values, and answer the result. The values passed to the
	 Block must be ones whose printString allows the correct object state to
	 be recreated (such as numbers and strings, for example).
	 The best values to return are specials.
	 Strings and other byte objects are copied to the local Gem.
	 All other responses are returned as a Array containing a single OOP."

	^self executeString: (self _stringForBlock: aBlock withArguments: someValues)
%
category: 'Public'
method:
executeString: aString
	"Execute the string expression in the external Gem and answer the result.
	 The best values to return are specials.
	 Strings and other byte objects are copied to the local Gem.
	 All other responses are returned as a Array containing a single OOP."

  | lib |
  aString _isOneByteString ifFalse:[ ArgumentError signal:'arg is not a String' ].
  lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	lib GciNbExecuteStr_: aString _: 20"nil asOop".
  self waitForResult .
  ^ self _lastResult: lib 
%
category: 'Public'
method:
forceLogout

	stoneSessionId ifNil: [^self].
	[self hardBreak] onException: Error do: [:ex | ].
	self _logout.
%
category: 'Public'
method:
forkBlock: aBlock
	"Execute the code in the Block argument in the external Gem
	 and do not wait for a result. At some later point, you would
	 check for a result. Otherwise you cannot issue another call, as
	 the current call would remain in progress. Refer to #executeString:
	 for an example of the complete send, wait, response sequence."

	self forkString: (self _stringForBlock: aBlock)
%
category: 'Public'
method:
forkBlock: aBlock with: aValue
	"Execute the code in the Block argument in the external Gem, passing
	 in the specified value, and do not wait for a result. The value passed 
	 to the Block must be one whose printString allows the correct object 
	 state to be recreated (such as numbers and strings, for example).
	 At some later point, you would check for a result. Otherwise you cannot
	 issue another call, as the current call would remain in progress.  
	 Refer to #executeString: for an example of the complete send, wait, response sequence."

	self forkBlock: aBlock withArguments: {aValue}.
%
category: 'Public'
method:
forkBlock: aBlock with: aValue with: anotherValue
	"Execute the code in the Block argument in the external Gem, passing
	 in the specified values, and do not wait for a result. The values passed 
	 to the Block must be ones whose printString allows the correct object 
	 state to be recreated (such as numbers and strings, for example).
	 At some later point, you would check for a result. Otherwise you cannot
	 issue another call, as the current call would remain in progress.  
	 Refer to #executeString: for an example of the complete send, wait, response sequence."

	self forkBlock: aBlock withArguments: {aValue. anotherValue}.
%
category: 'Public'
method:
forkBlock: aBlock withArguments: someValues
	"Execute the code in the Block argument in the external Gem, passing
	 in the specified values, and do not wait for a result. The values passed 
	 to the Block must be ones whose printString allows the correct object 
	 state to be recreated (such as numbers and strings, for example).
	 At some later point, you would check for a result. Otherwise you cannot
	 issue another call, as the current call would remain in progress.  
	 Refer to #executeString: for an example of the complete send, wait, response sequence."

	self forkString: (self _stringForBlock: aBlock withArguments: someValues)
%
category: 'Public'
method:
forkString: aString
	"Execute the string expression in the external Gem and do not wait for 
	 a result.  At some later point, you would check for a result. Otherwise you cannot
	 issue another call, as the current call would remain in progress.  
	 Refer to #executeString: for an example of the complete send, wait, response sequence."
  | lib |
  aString _isOneByteString ifFalse:[ ArgumentError signal:'arg is not a String' ].
  lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	lib GciNbExecuteStr_: aString _: 20"nil asOop".
%
! fix 47868
category: 'Parameters'
method:
gemNRS: anNRS
	"Set the GemService parameters for the logon to the value
	 of anNRS, which may be a String or a GsNetworkResourceString instance."

	parameters gemService: anNRS asString .
  (anNRS isKindOf: GsNetworkResourceString) ifTrue:[	
    self dynamicInstVarAt: #gemHost put: anNRS node .
  ].
%
category: 'Accessors'
method:
gemProcessId

  ^gemProcessId.
%
category: 'Public'
method:
hardBreak
	"Interrupt the external Gem and abort the current transaction."
  | lib |
  lib := self _gciLibrary .
	self _setSessionId: lib .
	lib GciHardBreak.
	self _signalIfError: lib .
%
category: 'Parameters'
method:
hostPassword: aString

	parameters hostPassword: aString copy.
%
category: 'Parameters'
method:
hostUsername: aString

	parameters hostUsername: aString copy.
%
category: 'Private'
method:
initialize

	gciErrSType := self class gciErrSTypeClass new.
	parameters := GemStoneParameters new.
	self loggingToServer.
%
category: 'Private'
method:
initializeDefaultResources

	self
		gemNRS: GsNetworkResourceString defaultGemNRSFromCurrent;
		stoneNRS: GsNetworkResourceString defaultStoneNRSFromCurrent;
		username: System myUserProfile userId;
		password: 'swordfish';
		yourself.
%
category: 'Public'
method:
isCallInProgress
	"Answer whether there is currently a call in progress to
	 the external Gem.
	 
	 The following calls are OK during a nonblocking call:
		GciCallInProgress
		GciErr
		GciGetSessionId
		GciHardBreak
		GciNbEnd
		GciSetSessionId
		GciShutdown
		GciSoftBreak"

	self isLoggedIn ifFalse:[ ^ false "not logged in"].
  ^ self _isCallInProgress: self _gciLibrary .
%
category: 'Private'
method:
_isCallInProgress: lib
  | result |
	self _setSessionId: lib .
	result := lib GciCallInProgress. "no Gci error possible"
	^ result == 1
%
category: 'Private'
method:
_isOnMyStone
  | val |
  (val := self dynamicInstVarAt: #_isOnMyStone) ifNil:[
    (GsSession currentSession isSolo) ifTrue:[ 
      val := false .
      self dynamicInstVarAt: #_isOnMyStone put: val . 
    ].
  ].
  ^ val 
%
category: 'Private'
method:
_isOnMyHost
  (self dynamicInstVarAt: #_isOnMyHost) ifNotNil:[ :x | ^ x ].
  ^ false
%
category: 'Public'
method:
isRemoteServerBigEndian

	^self _gciLibrary GciServerIsBigEndian ~~ 0
%
category: 'Public'
method:
isResultAvailable

	"Check whether the current call in progress has finished
	 and save the result if it has. Most operations cannot be
	 started while another is in progress. You must call this
	 method and receive a true result before starting another
	 remote operation."
	 
  ^ self _isResultAvailable: (CByteArray gcMalloc: 8)
%

category: 'Private'
method: 
_isResultAvailable: cByteArray
  "cByteArray is allocated by the caller to avoid a gcMalloc: in each
   invocation of this method."
	| result lib |
	nbResult := nil.
  lib := self _gciLibrary .
	(self _isCallInProgress: lib) ifFalse: [self error: 'no call in progress'].
	result := lib GciNbEnd_: cByteArray.
	result < 2 ifTrue: [ ^ false].
	self _signalIfError: lib .
	nbResult := (cByteArray pointerAt: 0 resultClass: CByteArray) uint64At: 0.
	^ true.
%

category: 'Public'
method:
lastResult
  ^ nbResult ifNotNil:[:r | 
     "r is an Integer, the value of a remote OopType"
     self resolveResult: r
  ]
%
category: 'Private'
method:
_lastResult: lib 
  ^ nbResult ifNotNil:[:r | 
     "r is an Integer, the value of a remote OopType"
     self _resolveResult: r lib: lib
  ]
%

category: 'Logging'
method:
log: aString

	logger value: aString
%
category: 'Logging'
method:
logger: aOneArgBlock
	"Use the specified one-argument Block for logging messages.
	 The argument to the Block is the message to log."

	logger := aOneArgBlock
%
category: 'Logging'
method:
loggingToClient

	self logger: [:message | GsFile gciLogClient: message]
%
category: 'Logging'
method:
loggingToServer

	self logger: [:message | GsFile gciLogServer: message]
%
! fix 47978
category: 'Logging'
method:
quiet
  "By default login and logout are logged using GsFile >> gciLogServer: .
   This disables logging of login and logout."
   self dynamicInstVarAt: #quiet put: 2 .
%
category: 'Logging'
method:
quietLogout
  "By default login and logout are logged using GsFile >> gciLogServer:.
   This disables logging of logout."
   self dynamicInstVarAt: #quiet put: 1 .
%
category: 'Public'
method:
loginSolo
  "login as a Solo session using  GCI_LOGIN_SOLO flag. 
   Requires an appropriate GEM_SOLO_EXTENT value in the config file
   used by the gem process for the new session.
   See GsSession(C)>>isSolo for details of a Solo session."

	stoneSessionId ifNotNil: [
		ImproperOperation signal: 'Stone session ' , stoneSessionId printString , 
			' already associated with this GsExternalSession!'.
	].
  parameters loginFlags: (parameters loginFlags bitOr: parameters soloLoginFlag). 
  ^ self login
%

category: 'Public'
method:
login
	| result lib |
	stoneSessionId ifNotNil: [
		ImproperOperation signal: 'Stone session ' , stoneSessionId printString , 
			' already associated with this GsExternalSession!'.
	].
  lib := self _gciLibrary .
	result := lib 
    GciSetNetEx_: parameters gemStoneName
		_: parameters hostUsername
		_: parameters hostPassword
		_: parameters gemService
		_: parameters passwordIsEncryptedAsIntegerBoolean.	"1 or 0: GCI_LOGIN_PW_ENCRYPTED"
        result == 0 ifTrue:[ self error:'GciSetNetEx_ failed'].
	result := lib
		GciLoginEx_: parameters username
		_: parameters password
		_: parameters loginFlags
		_: 0. "haltOnErrNum"
	0 == result ifTrue: [
	  self _signalIfError: lib arg: 'Using ', parameters printString .
	  self error: 'Login failed for unknown reason!'.
	].
	gciSessionId := lib GciGetSessionId.
  self _postLogin: lib  .
	((self dynamicInstVarAt: #quiet) ifNil:[ 0 ]) < 2 ifTrue:[
	  self log: 'GsExternalSession login: ' , self _describe.
  ].
%

category: 'Private'
method:
_postLogin: lib 
	| onMyStn onMyHost gemHostIdStr oopSystem fd sock |
  oopSystem := 76033 "System asOop".
	stoneSessionId := Object _objectForOop: (lib GciPerform_: oopSystem _: 'session' _: nil _: 0).
	self _signalIfError: lib .
  gemProcessId := Object _objectForOop: (lib GciPerform_: oopSystem _: 'gemProcessId' _: nil _: 0).
	self _signalIfError: lib .
  
  gemHostIdStr := self executeString:'System hostId asString' .
  onMyHost :=   gemHostIdStr = System hostId asString .

  GsSession isSolo ifTrue:[
    onMyStn := false .
  ] ifFalse:[ | stoneStartupStr |
    stoneStartupStr := self executeString:'System stoneStartupId asString'.
    onMyStn :=  stoneStartupStr = System stoneStartupId asString .
	  stoneSessionSerial := onMyStn ifTrue:[ GsSession serialOfSession: stoneSessionId ]
                              ifFalse:[ self executeString: 'GsSession currentSession serialNumber'].
  ].
  self dynamicInstVarAt: #_isOnMyStone put: onMyStn .
  self dynamicInstVarAt: #_isOnMyHost put:  onMyHost .
  fd := lib GciNbGetNotifyHandle .
  sock := GsSocket fromFileHandle: fd .
  self dynamicInstVarAt: #_socket put: sock
%

category: 'Public'
method:
logout
	stoneSessionId ifNil: [^self].
	self isCallInProgress ifTrue: [
		[
			self waitForResult.
		] onException: Error do: [:ex | 
			ex return.
		].
	].
	self _logout.
%

category: 'Private'
method:
nbLogout
  "Private. 
   Should be followed by a send of _waitForLogout ."

	stoneSessionId ifNil: [^self].
	self isCallInProgress ifTrue: [
		[
			self waitForResult.
		] onException: Error do: [:ex | 
			ex return.
		].
	].
	self _nbLogout .
%

category: 'Parameters'
method:
password: aString

	parameters password: aString copy.
%
category: 'Public'
Set compile_env: 0
method:
printOn: aStream

	aStream
		nextPutAll: 'a';
		nextPutAll: self class name;
		nextPutAll: '(';
		nextPutAll: stoneSessionId printString;
		nextPutAll: '/';
		nextPutAll: stoneSessionSerial printString;
		nextPutAll: ')';
		yourself.
%
category: 'Public'
method:
send: selector to: anOop
	"Answer the result of having the specified remote object
	 sent the message with the specified selector.
	 The best values to return are specials.
	 Strings and other byte objects are copied to the local Gem.
	 All other responses are returned as a Array containing a single OOP."

	^self send: selector to: anOop withArguments: nil
%
category: 'Public'
method:
send: selector to: anOop withArguments: someValues
	"Answer the result of having the specified remote object
	 sent the message with the specified selector and arguments.
	 Argument values are passed by OOP; beware of inconsistent views
	 between the local Gem and the external Gem.
	 The best values to return are specials.
	 Strings and other byte objects are copied to the local Gem.
	 All other responses are returned as a Array containing a single OOP."

	| nArgs args lib |
	lib := self _gciLibrary .
	self _errorIfCallInProgress: lib .
	(someValues == nil or: [ (nArgs := someValues size) == 0])
		ifFalse: [ | ofs |
        ofs := 0 .
			  args := CByteArray gcMalloc: 8 * nArgs .
			  1 to: nArgs do: [:index | | each |
          each := someValues at: index .
					args uint64At: ofs put: each asOop.
          ofs := ofs + 8 .
        ]
     ].
	lib
		GciNbPerform_: anOop
		_: selector
		_: args
		_: someValues size.
	^self
		waitForResult;
		_lastResult: lib 
%
category: 'Accessors'
method:
sessionId
	"0 => not logged in"

	^gciSessionId.
%

category: 'Accessors'
method:
isLoggedIn
 ^ gciSessionId ~~ nil and:[ gciSessionId > 0].
%

category: 'Public'
method:
softBreak
	"Interrupt the external Gem, but permit it to be restarted."
  | lib |
  lib := self _gciLibrary .
	self _setSessionId: lib .
	lib GciSoftBreak.
	self _signalIfError: lib .
%
category: 'Parameters'
method:
stoneNRS: anNRS
	"Set the Stone parameters for the logon to the value
	 of anNRS, which may be a String or a GsNetworkResourceString instance."

	parameters gemStoneName: anNRS asString
%
category: 'Accessors'
method:
stoneSessionId

	^stoneSessionId.
%
category: 'Accessors'
method:
stoneSessionSerial

	^stoneSessionSerial.
%
category: 'Logging'
method:
suppressLogging

	self logger: [:message | ]
%
category: 'Parameters'
method:
username: aString

	parameters username: aString.
%
category: 'Parameters'
method:
username
  ^ parameters username
%
! fix 47021
category: 'Public'
method:
resolveResult: anOop
	"Answer the object, or Array containing the OOP, of the result 
	received when the last #isResultAvailable answered true. 
	Specials can be fully resolved in the current session, and are the 
	preferred return type. 
	Results that are byte objects will be copied into a String or ByteArray 
	and that will be returned, but the OOP of the byte object will remain 
	in the remote Gem's export set. 
	For objects of all other types, return a result as an Array containing 
	the object's OOP in the remote gem, which is also recorded in the remote 
	Gem's export set."
	 
  ^ self _resolveResult: anOop lib: self _gciLibrary .
%
category: 'Private'
method:
_resolveResult: anOop lib: lib
	| type |
	self _setSessionId: lib .
	type := lib GciFetchObjImpl_: anOop.
	type == 3 ifTrue: [^Object _objectForOop: anOop]. "result is a special"
	type == 1 ifTrue: [^self _getBytes: anOop lib: lib ].	  "result is a byte object"
  type < 0 ifTrue:[
	  self _signalIfError: lib .
  ].
	^ { anOop } .
%
! edited for 46919
category: 'Public'
method:
resolveResult: anOop toLevel: anInteger
	"Similar to resolveResult:, but this recognizes more classes.
	If the class is not recognized, then return a CByteArray 
	with the OOP"

	| lib object oop cByteArray |
	lib := self _gciLibrary.
	object := self _resolveResult: anOop lib: lib .
	(object _isArray) ifFalse: [^object].	"a special or a byte object"
	oop := lib GciFetchClass_: anOop.
	(oop == 66817"Array asOop" and: [0 < anInteger]) ifTrue: [
		| size array |
		size := lib GciFetchSize__: anOop.
		array := Array new.
		1 to: size do: [:i | 
			oop := lib GciFetchOop_: anOop _: i. 
			array add: (self resolveResult: oop toLevel: anInteger - 1).
		].
		^array.
	].
	"Not a recognized object"
	cByteArray := CByteArray gcMalloc: 8.
	cByteArray uint64At: 0 put: anOop.
	^cByteArray.
%
category: 'Public'
method:
waitForResult
	"Wait as long as it takes for the external Gem to complete
	 the current operation.
   Does not allow other GsProcess to run. "

	self waitForResultForSeconds: 1000000000000 
%
category: 'Public'
method:
waitForResultForSeconds: aNumber
	"Wait as long as the specified seconds for the external Gem
	 to complete the current operation.
   Does not allow other GsProcess to run. "

	self 
		waitForResultForSeconds: aNumber
		otherwise: [self error: 
			'Wait time of ' , aNumber printString , 
			' exceeded for session ' , stoneSessionId printString , '/' , stoneSessionSerial printString ,
			' (PID ' , gemProcessId printString , ')']
%
category: 'Public'
method:
waitForResultForSeconds: aNumber otherwise: aBlock
	"Wait as long as the specified seconds for the external Gem
	 to complete the current operation. If the operation does not
	 complete within that time, answer the result of evaluating aBlock.
   Does not allow other GsProcess to run. "

	| cByteArray lib res msLeft |
  lib := self _gciLibrary .
  cByteArray := CByteArray gcMalloc: 8 .
  msLeft := aNumber asInteger * 1000 .
  [ msLeft > 0 ] whileTrue:[ | tMs | 
    tMs := msLeft min: 2000000000 .
    self _setSessionId: lib .
    res := lib GciNbEndPoll_: cByteArray _: tMs .
    res >= 2 ifTrue:[ 
      self _signalIfError: lib .
      nbResult := cByteArray uint64At: 0 .
      ^ self 
    ].
    msLeft := msLeft - tMs .
  ].
  ^ aBlock value 
%
category: 'Private'
method:
_describe
  | str |
	(str := 'stone session ID ' copy )
     add: stoneSessionId asString ;
     add: ' gem processId '; add: gemProcessId asString .
  self _gemHost ifNotNil:[ :host |
     str add: ' on host ' , host asString.
  ].
  str add: ' stone serialNumber ' ; add: stoneSessionSerial asString.
  ^ str
%

method:
_gemHost
  ^ self dynamicInstVarAt: #gemHost
%
category: 'Private'
method:
_errorIfCallInProgress: lib

	(self _isCallInProgress: lib) ifTrue: [self error: 'call in progress'].
%
category: 'Private'
method:
_gciLibrary
  "This entry in SessionTemps initialized by instance creation paths."
	^ SessionTemps current at: #'gciLibrary' 
%
category: 'Private'
method:
_getBytes: anOop lib: lib
	| size bytes numRet classOop |
	classOop := lib GciFetchClass_: anOop.
  classOop == 20"nil asOop" ifTrue:[
	  self _signalIfError: lib .
  ].
	size := lib GciFetchSize__: anOop.
  size == 0 ifTrue:[ 
    self _signalIfError: lib 
  ].
	bytes := CByteArray gcMalloc: size.
	numRet := lib GciFetchBytes__: anOop _: 1 _: bytes _: bytes size.
  numRet == 0 ifTrue:[
	  self _signalIfError: lib .
  ].
	numRet ~~ size ifTrue: [self error: 'Unexpected size!'].
	classOop == 74753"String asOop" ifTrue: [
		^bytes 
			stringFrom: 0
			to: size - 1.
	].
	classOop == 110849"Symbol asOop" ifTrue: [
		^ Symbol withAll: (bytes 
			stringFrom: 0
			to: size - 1).
	].
	^bytes 
		byteArrayFrom: 0
		to: size - 1.
%
category: 'Private'
method:
_logout
	| descr |
	descr := self _describe.
	self isLoggedIn ifTrue:[
		self _nbLogout: descr .
		self _waitForLogout.
		gciSessionId := 0.
	].
	stoneSessionId := nil.
	stoneSessionSerial := nil.
	((self dynamicInstVarAt: #quiet) ifNil:[ 0 ]) < 1 ifTrue:[
		self log: 'GsExternalSession logout: ' , descr. 
	].
%

category: 'Private'
method:
_nbLogout
  | descr |
  descr := self _describe.
	((self dynamicInstVarAt: #quiet) ifNil:[ 0 ]) < 1 ifTrue:[
	  self log: 'GsExternalSession nbLogout: ' , descr.
  ].
  ^ self _nbLogout: descr
%

category: 'Private'
method:
_nbLogout: descr
  self isLoggedIn ifTrue:[ | lib |
    lib := self _gciLibrary .
		self _setSessionId: lib .
		lib GciLogout__: 0 .
		[ [ self _signalIfError: lib 
			] onException: GciError do:[:ex | 
				 ex originalNumber == 4100 
					 ifTrue:[ "ignore invalid session error from GciError"]
					ifFalse:[ ex pass ]. 
			].
		] onException: Error do: [:ex | | msg |
			(msg := '---(During GsExternalSession logout:') , descr lf .
			msg := msg , '    ', ex description , ')---'.
			self log: msg lf .
		].  
  ].  
%

category: 'Private'
method:
_setSessionId: lib

  self isLoggedIn ifFalse:[
     ^ Error signal:'invalid sessionId (session not logged in)' 
  ].
  lib GciSetSessionId_: gciSessionId .
  self _signalIfError: lib .
%
category: 'Private'
method:
_signalIfError: lib 
	(lib GciErr_: gciErrSType) == 1 ifFalse: [^self].
	(gciErrSType number  between: 4000 and: 4999) ifTrue: [gciSessionId := 0].
	GciError 
		signal: gciErrSType 
		in: self
%
method:
_signalIfError: lib arg: detailString
	(lib GciErr_: gciErrSType) == 1 ifFalse: [^self].
	(gciErrSType number  between: 4000 and: 4999) ifTrue: [gciSessionId := 0].
	GciError 
		signal: gciErrSType 
		in: self details: detailString
%

! fixed 46165
category: 'Private'
method:
_waitForLogout
  | onMyStn onMyHost |
	stoneSessionId ifNil:[ ^ self].
  onMyStn := self _isOnMyStone .
  onMyHost := self _isOnMyHost .
  (onMyStn or:[ onMyHost]) ifFalse:[  ^ self "no way to wait reliably"].
	1 to: 2000 do:[ :j |
		onMyStn ifTrue: [
			(GsSession serialOfSession: stoneSessionId) = stoneSessionSerial ifFalse:[ 
         ^self
      ].
		].
    onMyHost ifTrue:[ 
      (System _hostProcessExists: gemProcessId) ifFalse:[
        ^ self 
      ].
    ].
		(Delay forMilliseconds: 10) wait. 
	].
  self error: 'session with stone session ID of ' , stoneSessionId printString , 
              ' gemProcessId = ' , gemProcessId printString, 
              ' still present 20 seconds after logout'.
%

category: 'Private'
classmethod:
_stackReport: contextOop
 "contextOop is from   aGciErrSType context"
 | aGsProcess |
 aGsProcess := (Object _objectForOop: contextOop) ifNil:[ ^ ' < NO PROCESS FOUND > '].
 ^ aGsProcess stackReportToLevel: 300 withArgsAndTemps: true andMethods: false
%

category: 'Private'
method: 
_getStackForOop: gcierrContextOop
  | str start cByteArray |
  nbResult := nil .
  str := 'GsExternalSession _stackReport: ' , gcierrContextOop asString .
  self forkString: str .
  start := System timeGmt .
  cByteArray := CByteArray gcMalloc: 8 .
  [ | result lib |
    lib := self _gciLibrary .
    (self _isCallInProgress: lib) ifFalse:[ ^ 'NO STACK, ERROR no call in progress'].
    result := lib GciNbEnd_: cByteArray.
    result >= 2 ifTrue:[
      (self _gciLibrary GciErr_: gciErrSType) == 1 ifTrue:[ 
        (gciErrSType number  between: 4000 and: 4999) ifTrue: [gciSessionId := 0].
        ^ GciError new _error: gciErrSType in: self .
      ].
      nbResult := (cByteArray pointerAt: 0 resultClass: CByteArray) uint64At: 0.
      ^ self _lastResult: lib .
    ].
    Delay waitForMilliseconds: 20 .
    (System timeGmt - start) > 20 ifTrue:[ ^ 'NO STACK, getStack timedout'].
  ] repeat
%
category: 'Public'
method:
waitForReadReady
  "Use the ProcessorScheduler to wait for this session's socket to 
   be ready to read, allowing other GsProcess to run while we are waiting."
  (self dynamicInstVarAt: #_socket) _waitForReadReady
%
category: 'Private'
method:
nbResult
  | result lib cByteArray |
  nbResult := nil.
  lib := self _gciLibrary .
  (self _isCallInProgress: lib) ifFalse: [self error: 'no call in progress'].
  cByteArray := CByteArray gcMalloc: 8 .
  result := lib GciNbEnd_: cByteArray .
  result < 2 ifTrue: [ self error:'result not ready'].
  self _signalIfError: lib .
  nbResult := (cByteArray pointerAt: 0 resultClass: CByteArray) uint64At: 0.
  ^ nbResult ifNotNil:[:r | 
    "r is an Integer, the value of a remote OopType"
    self resolveResult: r
  ]
% 

