!=========================================================================
! Copyright (C) GemTalk Systems 2008-2020.  All Rights Reserved.
!
! $Id$
!
! File: Module.gs
!   
!=========================================================================

! Module created by bom.c

! should not need SELF_CAN_BE_SPECIAL, self should always be non-nil
!  in instances of Module

set class Module
removeallmethods 
removeallclassmethods 

category: 'Documentation'
classmethod: 
comment
^ '
 Module represented the Ruby  class Module .
 In a Smalltalk repository, there are no instances of Module and
 creating instances of Module is not supported.
 
   InstVars            Values
      name             a Symbol
      classHistory     nil or a ClassHistory 
      transientMethDicts   always nil on disk, in memory, nil or an Array .
        transient methodDict for envId is      transientMethDicts at:(envId*4 + 1)
        transient rubyNameSpace for envId is   transientMethDicts at:(envId*4 + 2)
        transient superclass for envId is      transientMethDicts at:(envId*4 + 3)
        spare element                          transientMethDicts at:(envId*4 + 4)

     Note that the persistent superclass link for envId==0 is always in
     the superClass instVar defined in behavior. (see class comment in Behavior)
     Image and primitives code for following the superClass link for
     envId == 0 ignores (transientMethodDicts at: (envId*4 + 3)).

Constraints:
	superClass: Behavior
	format: SmallInteger
	instVarsInfo: SmallInteger
	instVarNames: Array
	constraints: Object
	classVars: SymbolDictionary
	methDicts: Object
	poolDictionaries: Array
	categorys: Object
	primaryCopy: Object
	name: Symbol
	classHistory: ClassHistory
	transientMethDicts: Object
 '
%

! instance creation not supported in Smalltalk image , see newModule in .mcz

category: 'Smalltalk Printing'
classmethod:
sprintf: formatString with: argsArray

"printf per Ruby Pickaxe book page 529..532  ."
<primitive: 767>
formatString _validateClass: String .
argsArray _validateClass: Array  .
self _primitiveFailed: #sprintf:with: args: { formatString . argsArray }
%

! --------------------
category: 'Browser Methods'
method:
_versionedName
"used by topaz"

^ [ self versionedName ] onException: Error do:[:ex| ^ self describe]
%

category: 'Accessing'
method:
name
  ^ name
%
method:
versionedName
| ofs str |
classHistory last == self ifTrue:[ ^ name ].
(ofs := classHistory indexOfIdentical: self) ~~ 0 ifTrue:[
  (str := String new) addAll: name ; add: $[ ; add: ofs asString; add:$] .
  ^ str
].
Error signal: self asOop asString , ', oop not found in classHistory(oop ' , 
    classHistory asOop asString , $) 
% 

method:
_name
  ^ name
%

method:
name: aName
  self isInvariant ifTrue:[ 
     ArgumentTypeError signal:'illegal attempt to change name of a Module'.
  ].
  name := aName asSymbol
%

method:
classHistory
  "Modules have a classHistory instVar, but it is not used"
  ^ nil
%

method:
transientMethodDictForEnv: envId
"result will be nil if no methods exist for specified environmentId."

^ transientMethDicts atOrNil: (envId*4 + 1) .
%

!final version of transientMethodDictForEnv:put: is in Module2.gs 

method:
transientMethodDictForEnv: envId put: aValue
  "aValue should be a GsMethodDictionary, or nil ,
   caller responsible for _refreshClassCache "

<protected>
| ofs |
ofs := envId*4 + 1 .
transientMethDicts size < ofs ifTrue:[ transientMethDicts size: ofs ].
transientMethDicts at: ofs put: aValue
%

category: 'Accessing the Class Hierarchy'
method:
transientSuperclassForEnv: envId

 "Result is normally nil for all environments in a Smalltalk VM."

 ^ transientMethDicts ifNotNil:[:tmd | tmd atOrNil: (envId*4 + 3 )]
                      ifNil:[ nil ]
%

category: 'Updating the Class Hierarchy'
method:
transientSuperclassForEnv: envId put: aClass

"Intended for use only in a Ruby VM.  
 Smalltalk VM method lookup does not use the transient superclass."

| ofs |
ofs := envId*4 + 3 .
transientMethDicts size < ofs ifTrue:[ transientMethDicts size: ofs ].
transientMethDicts at: ofs put: aClass .
%

category: 'Accessing the Class Hierarchy'
method:
superclassForEnv: envId

"for receiver and specified environment > 0, return first non-nil of:
    persistent super class
    env 0 super class (smalltalk superclass) 
 If envId == 0, equivalent to  self superClass .
"

<primitive: 676>
envId _validateClass: SmallInteger .
(envId < 0 or:[ envId > 255]) ifTrue:[ "COMPILE_ENV_MAX"
  self _errorIndexOutOfRange: envId 
].
self _primitiveFailed: #superclassForEnv: args: { envId }
%


! --------------------
category: 'Queries'
method: 
isMeta

"Returns whether the receiver is a kind of metaclass."

^ false
%

method: 
thisClass

"Returns the receiver.  
 For a Class the receiver's metaclass returns this Class as well.
 This method is useful to get the base version of a Class if one is
 holding either the Class or its metaclass."

^ self
%

! --------------------
! fixed 47123
category: 'Authorization'
method: 
objectSecurityPolicy: anObjectSecurityPolicy
	"Assigns the receiver and its non-shared components to the given security policy.
 The security policies of class variable values are not changed.  The current user 
 must have write access to both the old and new security policies for this method 
 to succeed."
	"enter protected mode"
	<primitive: 2001>
	| prot |
	prot := System _protectedMode.
	[| anArray aCls categBlk parts |
		parts := self lockableParts asIdentitySet.
		parts remove: self; remove: self class.
		self _objectSecurityPolicy: anObjectSecurityPolicy.
		self class _objectSecurityPolicy: anObjectSecurityPolicy.
		parts accompaniedBy: anObjectSecurityPolicy do: [:policy :each |
			  "categorys, methDicts instVars for self and self class
         handled here.  For Class, lockableParts includes extraDict. "
			each objectSecurityPolicy: policy
    ].
	  anArray := {}.
	  aCls := self.
	  categBlk := [:arr |
			arr ifNotNil:[ 
        anArray add: arr.
				arr _isArray ifTrue:[
          anArray addAll: arr.
					arr do: [:dict | dict ifNotNil:[ anArray addAll: dict values]]]
				ifFalse: [anArray addAll: arr values]]].
		2 timesRepeat:[ 
 			aCls persistentMethodDictsDo:[:aDict | anArray add: aDict; addAll: aDict values].
			aCls persistentNameSpacesDo: [:aRns | anArray add: aRns].
			categBlk value: aCls __categorys.  "categs array may contain nils"
			aCls _poolDictionaries do: [:d | anArray add: d].
			aCls := self class
		].
		anArray add: classHistory.
		anArray asIdentitySet accompaniedBy: anObjectSecurityPolicy do: [:policy :anObj |
			(anObj isSpecial or: [anObj _isSymbol or: [anObj == #()]])
				ifFalse: [anObj objectSecurityPolicy: policy]]
	] ensure: [prot _leaveProtectedMode]
%

category: 'Locking'
method: 
lockableParts

"Returns an Array of the receiver's contents that are locked by browsers
 and folders."

| mcls parts |
mcls := self class .
parts := { self . 
  instVarNames . 
  classVars .  "if not nil will be shared with the metaClass"
  categorys . 
  methDicts . 

  mcls . 
  mcls.instVarNames . 
  mcls.categorys . 
  mcls.methDicts . 
  mcls.poolDictionaries }.
parts := parts reject:[:ea | ea == nil ].
^parts
%

category: 'Modifying Classes'
method:
_setClassVars: aDict old: previousDict

  classVars ~~ aDict ifTrue:[ 
    previousDict ~~ classVars ifTrue:[ self error:'invalid store to classVars'].
    classVars := aDict .
  ].
%
!---------------------------------
category: 'Browser Methods'

method: 
changeNameTo: aString
  "Sets the receiver's 'name' instance variable."

  self _validatePrivilege ifTrue:[
    aString size == 0 ifTrue: [ ^ self error: 'class names must have size > 0'].
    name := Symbol withAll: aString.
  ]
%

method: 
hierarchy

"Returns a String that enumerates the receiver's superclasses (up to Object)
 and the instance variables defined by the receiver and each of its
 superclasses."

^ self hierarchy: 0
%

method:
hierarchy: envId

"Returns a String that enumerates the receiver's superclasses (up to Object)
 and the instance variables defined by the receiver and each of its
 superclasses."

^ self _superClassHierarchy: envId includeOops: false includeVirtual: false
%

method: 
_superClassHierarchy: envId includeOops: withOopsBool includeVirtual: vBool

"Returns a String showing the superClass hierarchy of the receiver.
 Used by implementation of topaz. "

| lf depth clsList rptList result |
lf := Character lf.
depth := 0.
(clsList := self _allSuperList: vBool env: envId ) addLast: self .
rptList := { }  .
depth := 0 .
clsList do:[ :aCls | | aLine ivNames |
  aLine := String new .
  1 to: depth do: [:i | aLine addAll: '  '].
  aLine add: (aCls name  ifNil:[ '(nil name)' ]) .
  ivNames := aCls _instVarNamesWithSeparator:  ''   .
  ivNames size ~~ 0 ifTrue: [
     aLine add: $(; add: ivNames ; add: $) .
  ].
  depth := depth + 1 .
  rptList add: aLine .
].
withOopsBool ifTrue:[ | width padStr |
  width := 0 .
  rptList do:[:aLine | | sz |
    ((sz := aLine size) > width and:[ sz < 40 ]) ifTrue:[ width := sz ]
  ].
  width := width + 4 .
  padStr := String new .
  width timesRepeat:[ padStr add: $  . ].
  1 to: rptList size do:[:idx | | aCls priCls aLine lsz |
    aLine := rptList at: idx .
    lsz := aLine size .
    lsz < width ifTrue:[
      aLine size: width .
      aLine replaceFrom: lsz + 1 to: width with: padStr startingAt: lsz + 1 .

    ] ifFalse:[  aLine add: $  ].
    aLine add: (aCls := clsList at: idx) asOop asString .
    vBool ifTrue:[ | ns |
      "this path only used for Ruby"
      aLine add: '  format=16r'; add: (aCls format asHexString) .
      priCls := aCls primaryCopy .
      priCls ~~ aCls ifTrue:[
        aLine add: ' primaryCopy '; add: priCls asOop asString .
      ].
      "persistentNameSpace:, transientNameSpace: implemented in Maglev*.mcz"
      ns := aCls persistentNameSpace: envId .
      aLine add: ' pns=' ; add: ( ns ifNil:[ 'nil'] ifNotNil:[ ns asOop asString ]).
      ns := aCls transientNameSpace: envId .
      aLine add: ' tns=' ; add: ( ns ifNil:[ 'nil'] ifNotNil:[ ns asOop asString ]).
    ].
  ].
].
result := String new .
rptList do:[ :aLine | result add: aLine ; add: lf .].
^ result
%
method: 
_instVarNamesWithSeparator: sep

"Returns a string showing my instance variables, with the given
 separator string inserted after every three names."

| result i theIvs numIvs each |

result := String new: 0.
i := 0.
theIvs := self instVarNames.
numIvs := theIvs size.
1 to: numIvs do:[:j|
  each := theIvs at: j .
  (i := i + 1) > 3 ifTrue:
    [
    result addAll: sep.
    i := 0.
    ].
  result addLast: $  .
  (each includesValue: $') 
    ifTrue:[ result addAll: each _asSource ]
    ifFalse:[ result addAll: each ].
  ].
^result
%

!--------------
category: 'Browsing'
method:
definition

  InternalError new 
    details:'Unexpected instance of Module in a Smalltalk repository' ; 
    signalNotTrappable  .

  ^ 'Module newModule name: ', name printString , '
       "does not include code to install in a dictionary" '
%

! deleted squeak browser support methods  
!    ogDefinition , allSubclassesDo:, allSubclasses, subclasses
!  which were used at one time for Maglev

!-----------------
category: 'Clustering'
method: 
clusterBehavior

 super clusterBehavior ifTrue:[ ^ true ].
 classHistory ifNotNil:[ classHistory cluster ].
 ^ false
%

category: 'Instance Migration'
method:
allInstancesInMemory

^ ((SystemRepository listInstancesInMemory: {self}) at: 1)
%

! migrateInstancesTo* moved to Module2.gs 

method:
instVarMappingTo: anotherClass

"See also class InstVarMappingArray for new functionality specific to
 dynamic instVars.
 Returns an instance-variable mapping from the receiver's named instance
 variables to those in the given class.  If an entry is 0, the other
 class does not have the corresponding instance variable.

 For migrating instances of oldClass to instances of newClass, 
 create a mapping via    
     newClass instVarMappingTo: oldClass  
 "

| otherivn result numIvs |
otherivn := anotherClass allInstVarNames .
numIvs := self instSize .
result := Array new: numIvs .
1 to: numIvs do: [:i |
  result at: i put: (otherivn indexOfIdentical: (instVarNames at: i))
].
^result
%

