!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   AbstractCollisionBucket, Array, SequenceableCollection, Collection, Object.
!
!=========================================================================

removeallmethods AbstractCollisionBucket
removeallclassmethods AbstractCollisionBucket

category: 'For Documentation Installation only'
classmethod: AbstractCollisionBucket
installDocumentation

self comment: 
'An AbstractCollisionBucket is an Array that is used in a KeyValueDictionary to
 store a collection of key/value pairs for which the keys hash to the same
 value.

--- instVar numElements
A SmallInteger that gives the number of key/value pairs in the bucket.
'
%

category 'Accessing'
method: AbstractCollisionBucket
keyValueDictionary

"Returns nil.  Only IdentityCollisionBuckets have the keyValueDictionary
 instance variable."

^nil
%

category 'Updating'
method: AbstractCollisionBucket
keyValueDictionary: aDict

"No-op for AbstractCollisionBuckets.  Used only for IdentityCollisionBuckets."
%

category: 'Instance Creation'
classmethod: AbstractCollisionBucket
new: aSize

"Returns an AbstractCollisionBucket with the specified size."

|result|
result := super new: (aSize + aSize).
result initialize.
^result
%

category: 'Instance Creation'
classmethod: AbstractCollisionBucket
new

"Returns an AbstractCollisionBucket with a default capacity of four key/value
 pairs."

^ self new: 4
%

category: 'Error Handling'
method: AbstractCollisionBucket
_errorKeyNotFound: aKey

"No key/value pair with given key, 'aKey', was found."

^ self _error: #rtErrKeyNotFound args: { aKey }.
%

category: 'Deprecated'
method: AbstractCollisionBucket
doKeys: aBlock

self deprecated: 'doKeys: Obsolete in GemStone/64.  Use the keysDo: method instead.'.
^ self keysDo: aBlock.
%

category: 'Enumerating'
method: AbstractCollisionBucket
do: aBlock

"For each key/value pair in the receiver, evaluates the one-argument block
 aBlock with the value as the argument.  Returns the receiver."

	self valuesDo: aBlock
%

category: 'Enumerating'
method: AbstractCollisionBucket
keysDo: aBlock

"For each key/value pair in the receiver, evaluates the one-argument block
 aBlock with the key as the argument.  Returns the receiver."

| aKey |
1 to: self tableSize do: [ :index |
  (aKey := self keyAt: index) ~~ nil ifTrue: [ aBlock value: aKey ]
  ].
%

category: 'Enumerating'
method: AbstractCollisionBucket
valuesDo: aBlock

"For each key/value pair in the receiver, evaluates the one-argument block
 aBlock with the value as the argument.  Returns the receiver."

1 to: self tableSize do: [ :index |
  (self keyAt: index) == nil ifFalse: [ aBlock value: (self valueAt: index) ]
  ].
%

category: 'Enumerating'
method: AbstractCollisionBucket
keysAndValuesDo: aBlock

"For each key/value pair in the receiver, evaluates the two-argument block
 aBlock with the key and value as the arguments.  Returns the receiver."

| aKey |
1 to: self tableSize do: [ :index |
  (aKey := self keyAt: index) ~~ nil 
    ifTrue: [ 
      aBlock value: aKey value: (self valueAt: index)
      ]
  ].
%

category: 'Deprecated'
method: AbstractCollisionBucket
doValues: aBlock

self deprecated: 'doValues: Obsolete in GemStone/64.  Use the valuesDo: method instead.'.
^ self valuesDo: aBlock.
%

category: 'Removing'
method: AbstractCollisionBucket
removeKey: aKey ifAbsent: aBlock

"Removes the key/value pair having the key aKey.  If aKey is not found,
 returns the result of evaluating the zero-argument block aBlock."

  | keyIndex aValue |

  keyIndex := self searchForKey: aKey.
  keyIndex ifNotNil:[ 
      numElements := numElements - 1.
      aValue := self valueAt: keyIndex.
      self at: keyIndex putKey: nil.
      self at: keyIndex putValue: nil.
      ^aValue 
  ].
  aBlock ifNil:[ ^ self _errorKeyNotFound: aKey ] .
  ^ aBlock value
%

! added for 36675
category: 'Removing'
method: AbstractCollisionBucket
removeKey: aKey otherwise: notFoundValue

"Removes the key/value pair having the key aKey.  If aKey is not found,
 returns the notFoundValue. "

  | keyIndex aValue |

  keyIndex := self searchForKey: aKey.
  keyIndex == nil
    ifFalse: [
      numElements := numElements - 1.
      aValue := self valueAt: keyIndex.
      self at: keyIndex putKey: nil.
      self at: keyIndex putValue: nil.
      ^aValue ]
    ifTrue: [ ^ notFoundValue ]
%


category: 'Removing'
method: AbstractCollisionBucket
_removeKey: aKey 

"Removes the key/value pair having the key aKey.  If aKey is not found,
 generates an error."

  | keyIndex |

  keyIndex := self searchForKey: aKey.
  keyIndex == nil
    ifFalse: [
      numElements := numElements - 1.
      self at: keyIndex putKey: nil.
      self at: keyIndex putValue: nil.
    ]
    ifTrue: [ self _errorKeyNotFound: aKey ]
%

category: 'Accessing'
method: AbstractCollisionBucket
_reportKeyNotFound: aKey with: aBlock

"Private."

aBlock == nil ifTrue:[^ self _errorKeyNotFound: aKey ] .
^aBlock value
%

category: 'Accessing'
method: AbstractCollisionBucket
at: aKey

"Returns the value that corresponds to aKey."

^self at: aKey
    ifAbsent: nil "optimization, errors reported via _reportKeyNotFound"
%

category: 'Accessing'
method: AbstractCollisionBucket
keyAt: index

"Returns the key at the specified index."

^super at: (index + index - 1)
%

category: 'Accessing'
method: AbstractCollisionBucket
valueAt: index

"Returns the value at the specified index."

^super at: (index + index)
%

category: 'Accessing'
method: AbstractCollisionBucket
at: aKey ifAbsent: aBlock

"Returns the value that corresponds aKey.  If no such key/value pair exists,
 returns the result of evaluating the zero-argument block aBlock."

  | index |
  index := self searchForKey: aKey.
  index ifNotNil:[ ^ self valueAt: index ] .
  aBlock ifNil:[ ^ self _errorKeyNotFound: aKey ] .
  ^ aBlock value
%

category: 'Accessing'
method: AbstractCollisionBucket
tableSize

"Returns the number of key/value pairs in the capacity of the receiver."

^self _basicSize // 2
%

category: 'Accessing'
method: AbstractCollisionBucket
numElements

"Returns value of the numElements instance variable.  (The name numElements
 is provided for compatibility with earlier releases.  The instance method
 size is preferred for new code.)"

^numElements
%

category: 'Accessing'
method: AbstractCollisionBucket
size

"Returns value of the numElements instance variable."

^numElements
%

category: 'Accessing'
method: AbstractCollisionBucket
at: aKey otherwise: aValue

"Returns the value that corresponds to aKey by searching the elements in the
 bucket.  If no such key/value pair exists, returns the given alternate value."

  | index |
  index := self searchForKey: aKey.
  index ifNotNil:[ ^ self valueAt: index ].
   ^aValue .
%

category: 'Accessing'
method: AbstractCollisionBucket
keyAt: aKey otherwise: aValue

"Returns the key that corresponds to aKey by searching the elements in the
 bucket.  If no such key/value pair exists, returns the given alternate value."

  | index |
  index := self searchForKey: aKey.
  index ifNotNil:[ ^ self keyAt: index ].
  ^ aValue 
%

category: 'Updating'
method: AbstractCollisionBucket
at: anIndex putKey: aKey

"Stores the key aKey into the key part of the key/value pair referenced by
 anIndex.  Note that this method overwrites the key value at the given index.
 Returns aKey."

super at: (anIndex + anIndex) - 1 put: aKey
%

category: 'Updating'
method: AbstractCollisionBucket
at: anIndex putValue: aValue

"Stores the value aValue into the value part of the key/value pair referenced
 by atIndex.  Returns aValue."

super at: (anIndex + anIndex) put: aValue
%

category: 'Updating'
method: AbstractCollisionBucket
at: aKey put: aValue keyValDict_coll: aKeyValDict
 "Stores the aKey/aValue pair in the receiver.  
  Returns self size if this at:put: added a new key, 0 if this at:put: 
  replaced the value of an existing key."
 | emptySlotIdx startTableSize thisKey numElem |

  startTableSize := self tableSize .
  aKey ifNil:[ ^ self _error: #rtErrNilKey ] .
  (numElem := numElements) == 0 ifTrue:[
    emptySlotIdx := 1
  ] ifFalse:[ | idx |
    "search for aKey, or for the first empty slot "
    idx := 1 . 
    1 to: startTableSize do:[:n |
      thisKey := self _at: idx . "inline keyAt:"
      thisKey ifNotNil:[
        (self compareKey: aKey with: thisKey) ifTrue:[ "Key found.  Store given value"
          self _at: idx + 1 put: aValue .  "inline at:putValue:"
          aKeyValDict _markDirty .
          ^ 0
        ].
      ] ifNil:[
        emptySlotIdx ifNil:[ emptySlotIdx := idx ].
      ].
      idx := idx + 2 .
    ] .
    "Key not found so add key and value"
    emptySlotIdx ifNil:[ " bucket is full so grow it "
      emptySlotIdx := self _basicSize + 1 .
      self size: emptySlotIdx + 7  .  "accommodate 4 more key,value pairs"
    ] .
  ].
  numElem := numElem + 1.
  numElements := numElem .
  self _at: emptySlotIdx put: aKey. "inline at:putKey:"
  self _at: emptySlotIdx + 1 put: aValue . "inline at:putValue"
  ^ numElem 
%
category: 'Searching'
method: AbstractCollisionBucket
includesKey: aKey

"Returns true if the receiver contains a key that is equal to 'aKey'.
 Otherwise, returns false."

^(self searchForKey: aKey) ~~ nil
%

category: 'Searching'
method: AbstractCollisionBucket
searchForKey: argKey

"Returns the index of argKey, or if not found, nil."
   | idx |
   argKey ifNil:[ ^ self _error: #rtErrNilKey ] .
   idx := 1 .
   1 to: self tableSize do: [ :n | | aKey |
      aKey := self _at: idx . "inline keyAt:"
      aKey ifNotNil:[  
        (self compareKey: argKey with: aKey) ifTrue:[ ^ n ].
      ].
      idx := idx + 2
   ].
   ^ nil "Key not found"
%

category: 'Searching'
method: AbstractCollisionBucket
firstPair

"Returns an Association containing the receiver's first key/value pair.
 If the receiver is empty, returns an Association containing nils."

   | aKey |
   numElements == 0 ifFalse: [ 
      "Search for the first non-nil index"
      1 to: self tableSize do: [ :i |
         (aKey := self keyAt: i) ifNotNil: [  
            ^Association new key: aKey value: (self valueAt: i) 
         ]
      ]
   ].
   "No first pair was found"
   ^Association newWithKey: nil value: nil
%

category: 'Updating'
method: AbstractCollisionBucket
at: aKey put: aValue 

self shouldNotImplement: #at:put:  " use at:put:keyValDict_coll: "
%

category: 'Adding'
method: AbstractCollisionBucket
add: anAssociation

self shouldNotImplement: #add:  " use at:put:keyValDict_coll: "
%

category: 'Initializing'
method: AbstractCollisionBucket
initialize

"Initializes the instance variable of the receiver to be an empty
 CollisionBucket."

numElements := 0.
%

category: 'Private'
method: AbstractCollisionBucket
compareKey: key1 with: key2

"Returns true if key1 is equivalent to key2, and false otherwise."

^ key1 = key2
%

category: 'Formatting'
method: AbstractCollisionBucket
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

"Copy the implementation from Object so we don't inherit it from Collection."

aStream nextPutAll: self asString
%

category: 'Private'
method: AbstractCollisionBucket
_removeAll

"Dereferences the receiver from its parent and shrinks the receiver.
 Used while rebuilding a KeyValueDictionary."

numElements := 0.
"Gs64 v3.0, don't send size: 0 "
%

! deleted _canonicalizeSymbolAt: offset oldSymbol: oldSym newSymbol: newSym

! fix 46134
category: 'Comparing'
method: AbstractCollisionBucket
= anObject

  "Reimplemented to compare by identity."

  ^ self == anObject
%
category: 'Comparing'
method: AbstractCollisionBucket
hash
  ^ self identityHash
%
