!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   Boolean, Object.
!
!=========================================================================

removeallmethods Boolean
removeallclassmethods Boolean

category: 'For Documentation Installation only'
classmethod: Boolean
installDocumentation

self comment:
'The only two instances of Boolean represent the two logical truth values:
 true and false.

 You may not create new instances of Boolean.  You also may not create
 subclasses of Boolean.'
%

category: 'Instance Creation'
classmethod: Boolean
fromStream: aStream

"If the next characters in aStream are true or false (case insensitive, leading
 spaces permitted), this method returns the appropriate Boolean.  Otherwise,
 generates an error."

| char |

self _checkReadStream: aStream forClass: CharacterCollection.

char := aStream next.
[ char == $  ] whileTrue: [ char := aStream next ].
aStream skip: -1.

((char := aStream next) isEquivalent: $T)
ifTrue:
  [ ((aStream next isEquivalent: $R)
    and: [(aStream next isEquivalent: $U) 
    and: [aStream next isEquivalent: $E ]])
    ifTrue:
      [ ^ true ]
  ]
ifFalse:
  [ (char isEquivalent: $F)
    ifTrue:
      [ ((aStream next isEquivalent: $A)
        and: [(aStream next isEquivalent: $L)
        and: [(aStream next isEquivalent: $S)
        and: [aStream next isEquivalent: $E ]]])
        ifTrue:
          [ ^ false ]
      ]
  ].

self _errIncorrectFormat: aStream
%

category: 'Instance Creation'
classmethod: Boolean
fromString: aString

"If aString contains true or false, returns the appropriate Boolean.  Leading
 and trailing spaces are permitted in the String.  If aString contains any
 characters other than true or false, this method generates an error."

| s result |

s := ReadStream on: aString.
result := self fromStream: s.
[ s atEnd ]
whileFalse:
  [ (s next == $  )
    ifFalse:
      [ self _errIncorrectFormat: aString ]
  ].
^ result
%

category: 'Instance Creation'
classmethod: Boolean
new

"Disallowed.  You cannot create new instances of Boolean."

self shouldNotImplement: #new
%

category: 'Logical Operations'
method: Boolean
& aBoolean

"Evaluating conjunction (AND).  Returns true if both the receiver and the
 argument aBoolean are true."

<primitive: 42>
aBoolean _validateClass: Boolean.
self _primitiveFailed: #& args: { aBoolean }.
self _uncontinuableError
%

category: 'Logical Operations'
method: Boolean
| aBoolean

"Evaluating disjunction (OR).  Returns true if either the receiver or the
 argument aBoolean is true."

<primitive: 43>
aBoolean _validateClass: Boolean.
self _primitiveFailed: #| args: { aBoolean }.
self _uncontinuableError
%

category: 'Logical Operations'
method: Boolean
eqv: aBoolean

"Returns true if the receiver is identical to aBoolean."

^ self == aBoolean
%

category: 'Logical Operations'
method: Boolean
not

"Negation.  Returns true if the receiver is false.  Returns false if the
 receiver is true."

<primitive: 44>
self _primitiveFailed: #not .
self _uncontinuableError
%

category: 'Logical Operations'
method: Boolean
xor: aBoolean

"Exclusive OR.  Returns true if the receiver is not identical to aBoolean."

<primitive: 45>
aBoolean _validateClass: Boolean.
self _primitiveFailed: #xor: args: { aBoolean }.
self _uncontinuableError
%

category: 'Flow of Control'
method: Boolean
and: aBlock

"(Reserved selector.) Nonevaluating conjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is true.  Otherwise, returns false
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self and: [ aBlock value]
%

category: 'Flow of Control'
method: Boolean
_and: aBlock

"(Reserved selector.)  Nonevaluating conjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is true.  Otherwise, returns false
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self and: [ aBlock value]
%

category: 'Flow of Control'
method: Boolean
ifFalse: aBlock

"(Reserved selector.)  Returns the value of the zero-argument block aBlock if
 the receiver is false.  Otherwise, returns nil without evaluating the
 argument."

"The following is an optimized control structure, not a recursive send."

^ self ifFalse: [^ aBlock value ]
%

category: 'Flow of Control'
method: Boolean
ifFalse: falseBlock ifTrue: trueBlock

"(Reserved selector.)  Returns the value of the zero-argument block falseBlock
 if the receiver is false.  Otherwise, returns the value of the zero-argument
 block trueBlock without evaluating falseBlock."

"The following is an optimized control structure, not a recursive send."

self ifFalse: [^ falseBlock value]
      ifTrue: [^ trueBlock value]
%

category: 'Flow of Control'
method: Boolean
ifTrue: aBlock

"(Reserved selector.)  Returns the value of the zero-argument block aBlock if
 the receiver is true.  Otherwise, returns nil without evaluating the argument."

"The following is an optimized control structure, not a recursive send."

^ self ifTrue: [^ aBlock value]
%

category: 'Flow of Control'
method: Boolean
ifTrue: trueBlock ifFalse: falseBlock

"(Reserved selector.)  Returns the value of the zero-argument block falseBlock
 if the receiver is false.  Otherwise, returns the value of the zero-argument
 block trueBlock without evaluating falseBlock."

"The following is an optimized control structure, not a recursive send."

^ self ifTrue: [^ trueBlock value]
       ifFalse: [ ^ falseBlock value]
%

category: 'Flow of Control'
method: Boolean
or: aBlock

"(Reserved selector.) Nonevaluating disjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is false.  Otherwise, returns true
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self or:[ aBlock value ]
%
category: 'Flow of Control'
method: Boolean
_or: aBlock

"(Reserved selector.)  Nonevaluating disjunction.  Returns the value of the
 zero-argument block aBlock if the receiver is false.  Otherwise, returns true
 without evaluating the argument.

 The following is an optimized control structure, not a recursive send."

^ self or:[ aBlock value ]
%

category: 'Formatting'
method: Boolean
asString

"Returns a String containing true or false, depending on the receiver."

self
  ifTrue:
    [ ^ String withAll: 'true' ]
  ifFalse:
    [ ^ String withAll: 'false' ]
%

category: 'Clustering'
method: Boolean
clusterDepthFirst

"Returns true.  (Because Booleans are self-defining objects, this method has no
 effect.)"

^ true
%

category: 'Copying'
method: Boolean
copy

"Overrides the inherited method to return the receiver.  The pseudo-variables
 true and false are the only instances of Boolean, and must preserve identity."

^ self
%

! deleted _idxCompareLessThan: v2.0

! deleted _idxCompareLessThanOrEqual: v2.0

! deleted _idxCompareGreaterThan: v2.0

! deleted _idxCompareGreaterThanOrEqual: v2.0

category: 'Formatting'
method: Boolean
printString

"Returns a String whose contents are a displayable representation of the
 receiver."

"GemStone does not allow the creation of new kinds of Boolean, so there is no
 point in creating a stream and sending printOn:."

^self asString
%

category: 'Decompiling without Sources'
method: Boolean
_asSource

^ self asString
%

category: 'Storing and Loading'
classmethod: Boolean
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form and returns an equivalent instance of Boolean."

"This method supports filein of Booleans in the format required by GeODE
 versions before 2.0.1."

|inst|
inst := passiveObj next == $t.
passiveObj hasRead: inst.
^inst
%

category: 'Storing and Loading'
method: Boolean
writeTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."

passiveObj nextPut: (self ifTrue: [ $* ] ifFalse: [ $~ ])

%

category: 'Storing and Loading'
method: Boolean
containsIdentity

"Private."

^true
%
category: 'Testing'
method: Boolean
isSpecial

"Returns true if the receiver is a special object."

^ true
%
category: 'New Indexing Comparison'
method: Boolean
_classSortOrdinal

^ 20
%
category: 'New Indexing Comparison'
method: Boolean
_idxForCompareGreaterThan: arg

""

^arg _idxForCompareBooleanGreaterThanSelf: self
%
category: 'New Indexing Comparison'
method: Boolean
_idxForCompareGreaterThanOrEqualTo: arg

""

^arg _idxForCompareBooleanGreaterThanOrEqualToSelf: self
%
category: 'New Indexing Comparison'
method: Boolean
_idxForCompareLessThan: arg

""

^arg _idxForCompareBooleanLessThanSelf: self
%
category: 'New Indexing Comparison'
method: Boolean
_idxForCompareLessThanOrEqualTo: arg

""

^arg _idxForCompareBooleanLessThanOrEqualToSelf: self
%
category: 'New Indexing Comparison'
method: Boolean
_idxForSortGreaterThan: arg

""

^arg _idxForSortBooleanGreaterThanSelf: self
%
category: 'New Indexing Comparison'
method: Boolean
_idxForSortGreaterThanOrEqualTo: arg

""

^arg _idxForSortBooleanGreaterThanOrEqualToSelf: self
%
category: 'New Indexing Comparison'
method: Boolean
_idxForSortLessThan: arg

""

^arg _idxForSortBooleanLessThanSelf: self
%
category: 'New Indexing Comparison'
method: Boolean
_idxForSortLessThanOrEqualTo: arg

""

^arg _idxForSortBooleanLessThanOrEqualToSelf: self
%

category: 'New Indexing Comparison - for Sort'
method: Boolean
_idxForSortBooleanGreaterThanOrEqualToSelf: anObject

"second half of a double dispatch call from Boolean>>_idxForSortGreaterThanOrEqualTo:. Note that aBoolean should be the receiver in any >= comparison"

anObject = self
  ifTrue: [ ^ true ].
^anObject
%
category: 'New Indexing Comparison - for Sort'
method: Boolean
_idxForSortBooleanGreaterThanSelf: anObject

"second half of a double dispatch call from Boolean>>_idxForSortGreaterThan:. Note that aBoolean should be the receiver in any > comparison"

anObject = self
  ifTrue: [ ^ false ].
^anObject
%
category: 'New Indexing Comparison - for Sort'
method: Boolean
_idxForSortBooleanLessThanOrEqualToSelf: anObject

"second half of a double dispatch call from Boolean>>_idxForSortLessThanOrEqualTo:. Note that aBoolean should be the receiver in any <= comparison"

anObject = self
  ifTrue: [ ^ true ].
^anObject not
%
category: 'New Indexing Comparison - for Sort'
method: Boolean
_idxForSortBooleanLessThanSelf: anObject

"second half of a double dispatch call from Boolean>>_idxForSortLessThan:. Note that aBoolean 
should be the receiver in any < comparison"

anObject = self
  ifTrue: [ ^ false ].
^anObject not
%
category: 'New Indexing Comparison - for Compare'
method: Boolean
_idxForCompareBooleanGreaterThanOrEqualToSelf: aBoolean

"second half of a double dispatch call from Boolean>>_idxForCompareGreaterThanOrEqualTo:. Note that aBoolean should be the receiver in any >= comparison"

aBoolean = self
  ifTrue: [ ^true].
^aBoolean == true
%
category: 'New Indexing Comparison - for Compare'
method: Boolean
_idxForCompareBooleanGreaterThanSelf: aBoolean

"second half of a double dispatch call from Boolean>>_idxForCompareGreaterThan:. Note that aBoolean should be the receiver in any > comparison"

self = aBoolean
    ifTrue: [ ^ false ].
^aBoolean == true
%
category: 'New Indexing Comparison - for Compare'
method: Boolean
_idxForCompareBooleanLessThanOrEqualToSelf: aBoolean

"second half of a double dispatch call from Boolean>>_idxForCompareLessThanOrEqualTo:. Note that aBoolean should be the receiver in any <= comparison"

aBoolean = self
  ifTrue: [ ^true].
^aBoolean == false
%
category: 'New Indexing Comparison - for Compare'
method: Boolean
_idxForCompareBooleanLessThanSelf: aBoolean

"second half of a double dispatch call from Boolean>>_idxForCompareLessThan:. Note that aBoolean should be the receiver in any < comparison"

self = aBoolean
    ifTrue: [ ^ false ].
^aBoolean == false
%
category: 'Json'
method: Boolean
printJsonOn: aStream

	aStream nextPutAll: (self ifTrue: ['true'] ifFalse: ['false']).
%

