!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   Delay, Object.
!
!=========================================================================

expectvalue %String
run
^ Object _newKernelSubclass: #Delay
  instVarNames: #(#interval #signalTime #target)
  classVars: #()
  classInstVars: #()
  poolDictionaries: { }
  inDictionary: Globals
  options: #()
  reservedOop: 907
%
run
(Globals associationAt:#Delay) immediateInvariant . true
%

! Remove existing behavior from Delay
removeallmethods Delay
removeallclassmethods Delay

! ------------------- Class methods for Delay
category: 'For Documentation Installation only'
classmethod
installDocumentation

self comment:
'Delay is used in conjunction with GsProcess and ProcessorScheduler
 to suspend processes for specified amounts of time, and to resume 
 processes.   
 The in-memory state of a committed Delay is not changed by a transaction abort.

Constraints:
	interval: Integer     (units milliseconds)
	signalTime: Integer   (units milliseconds)
	target: Object'.
%

category: 'Instance Creation'
classmethod: Delay
new

"Disallowed."

self shouldNotImplement: #new
%

category: 'Instance Creation'
classmethod: Delay
forMilliseconds: millisecondCount
  "Answer a new instance that will suspend the active process
   for millisecondCount milliseconds when sent the message wait."
   
  ^ self _basicNew interval: millisecondCount
%
category: 'Instance Creation'
classmethod: Delay
forSeconds: secondCount
  "Answer a new instance that will suspend the active process
   for secondCount seconds when sent the message wait."
  
  ^ self forMilliseconds: secondCount * 1000
%
category: 'Instance Creation'
classmethod: Delay
untilMilliseconds: millisecondValue
  "Answer a new instance that will suspend the active process
   until the millisecond clock reaches the value millisecondValue."
  
  millisecondValue _isSmallInteger ifFalse:[
    millisecondValue _validateClass: SmallInteger.
  ].
  millisecondValue < 0 ifTrue:[
    ^ millisecondValue _error: #rtErrArgNotPositive 
  ].
  ^ self _basicNew _signalTime: millisecondValue
%
category: 'Private'
classmethod: Delay
_scheduler

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler
%
category: 'Private'
method: Delay
_scheduler

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler
%

! edited for 45575
category: 'Process Delay'
classmethod: Delay
waitForMilliseconds: millisecondCount
  "Suspends the active process for millisecondCount milliseconds.
   Note that this method does not create or use an instance of Delay.
   Returns the receiver."

  | sched |
  millisecondCount < 0 ifTrue:[
    ^ millisecondCount _error: #rtErrArgNotPositive 
  ].
  sched := self _scheduler _enterCritical .
  sched _waitForMilliseconds: millisecondCount .
  1 timesRepeat:[ self class ].  "check for a Break"
  ^self
%
category: 'GBS Access'
classmethod: Delay
_highPriorityWaitForMilliseconds: millisecondCount
  "Set the active process's priority high and then
   suspends the active process for millisecondCount milliseconds.
   Note that this method does not create or use an instance of Delay.
   Returns the receiver."

  | sched oldPrio proc |
  sched := self _scheduler _enterCritical .
  proc := sched activeProcess .
  oldPrio := proc _raisePriority .
  sched _waitForMilliseconds: millisecondCount .  "leaves critical"
  sched _enterCritical .
  sched activeProcess == proc ifTrue:[
    proc priority: oldPrio .
  ].
  sched _exitCritical .
  1 timesRepeat:[ self class ].  "check for a Break"
  ^ self
%
category: 'Process Delay'
classmethod: Delay
waitForSeconds: secondCount
  "Suspends the active process for secondCount seconds.
   Note that this method does not create or use an instance of Delay.
   Returns the receiver."
  
  | sched |
  secondCount < 0 ifTrue:[
    ^ secondCount _error: #rtErrArgNotPositive 
  ].
  sched := self _scheduler _enterCritical .
  sched _waitForMilliseconds: secondCount * 1000 .
  1 timesRepeat:[ self class ].  "check for a Break"
  ^self
%
category: 'Process Delay'
classmethod: Delay
waitUntilMilliseconds: millisecondValue
  "Suspends the active process until the millisecond clock
   reaches the value millisecondValue.
   Note that this method does not create or use an instance of Delay.
   Returns the receiver."

  | delta sched |
  millisecondValue _isSmallInteger ifFalse:[
    millisecondValue _validateClass: SmallInteger.
  ].
  millisecondValue < 0 ifTrue:[
    ^ millisecondValue _error: #rtErrArgNotPositive
  ].
  sched := self _scheduler _enterCritical .
  delta := millisecondValue -  sched _now .
  delta < 0 ifTrue:[ delta := 0 ].
  sched _waitForMilliseconds: delta.
  1 timesRepeat:[ self class ].  "check for a Break"
  ^self
%
category:  'General Inquiries'
classmethod: Delay
millisecondClockValue
  "Returns the current value of the millisecond clock.

   Gs64 v2.2, changed to no longer rollover to zero after 524287999 "

<primitive: 651>
^ self _primitiveFailed: #millisecondClockValue
%
! ------------------- Instance methods for Delay
category: 'Private'
method: Delay
_activate
  "Causes the receiver to signal any processes that have sent wait to it."

  target _signalAll .
  target := nil.
  ^self
%
category: 'Private'
method: Delay
_targetProcess
  "Returns the GsProcess that is waiting for the receiver."
  | targ | 
  ^ (targ := target) class == GsProcess ifTrue: [ targ ] ifFalse: [ nil ] 
%
category: 'Accessing'
method: Delay
priority
  | targ |
  ^ (targ := target) class == GsProcess ifTrue:[  targ priority ]
		ifFalse:[ 15 "inline userSchedulingPriority" ].
%
category: 'Process Delay'
method: Delay
signal
  "Causes the receiver to signal any processes that have sent wait to it."

  self _scheduler _delayUnschedule: self.
  ^self _activate.
%
category: 'Private'
method: Delay
_cancel: aGsProcess
  "Cancels a delay that is in progress.
   The receiver will not signal any processes that have sent wait to it.
   Only use if you know that the processes that are waiting on the receiver
   will be signalled in some other way."

  (aGsProcess == target) ifTrue: [
    self _scheduler _delayUnschedule: self.
    target := nil.
  ].
%
category: 'Private'
method: Delay
interval: millisecondCount
  "initialize the receiver to delay on an interval"
  
  millisecondCount _isSmallInteger ifFalse:[
    millisecondCount _validateClass: SmallInteger.
  ].
  millisecondCount < 0 ifTrue:[
    ^ millisecondCount _error: #rtErrArgNotPositive 
  ].
  interval := millisecondCount .
  signalTime := nil .
  target := nil .
%

category: 'Formatting'
method: Delay
printOn: aStream
  aStream
    nextPutAll: self class name;
    nextPut: $(.
  self asOop printOn: aStream.
  aStream nextPutAll: ',i='.
  interval printOn: aStream.
  aStream nextPutAll: ',t='.
  signalTime printOn: aStream.
  aStream nextPutAll: ',s='.
  target printOn: aStream.
  aStream nextPut: $).
%
category: 'Accessing'
method: Delay
resumptionTime
  "Answer the value of the millisecond clock at which
   the delayed process will be resumed.  This will be nil
   if the receiver was created with an interval and no
   wait has been issued."

  (interval == nil) ifTrue: [
    ^signalTime
  ] ifFalse: [
    (target ~~ nil"inline inProgress") ifTrue: [
      ^signalTime
    ] ifFalse: [
      ^nil
    ]. 
  ].
%
category: 'Private'
method: Delay
_signalTime
  ^signalTime
%

category: 'Testing'
method: Delay
inProgress
  "Returns true if a GsProcess is currently waiting on the receiver."

  ^(target ~~ nil)
%
category: 'Process Delay'
method: Delay
wait
  "Suspend the active process until the millisecond clock
   reaches the appropriate value."

self _wait: false
%

category: 'Process Delay'
method: Delay
highPriorityWait
  "Suspend the active process until the millisecond clock
   reaches the appropriate value.  The active process priority 
   is raised for the duration of the wait, so that expiration of the delay 
   will cause the active process to resume thus interrupting any infinite loop
   or infinite wait for socket activity in other processes."

 ^ self _wait: true 
%

category: 'Private'
method: Delay
_wait: highPriorityBoolean

  "Suspend the active process until the millisecond clock
   reaches the appropriate value."

  | interv targ proc oldPrio sched |
  sched := self _scheduler _enterCritical .
  (interv := interval) ifNotNil: [ 
    "If another process is already using this interval delay raise an error"
    target ifNotNil: [
      sched _exitCritical .
      ThreadError new _number: 2365 ; reason: #rtErrDelayInProgress ;
         object: self ; details:'Another GsProcess already waiting on this Delay';
         signal .
      ^ self
    ].
  ].
  highPriorityBoolean ifTrue:[
    proc := sched activeProcess .
    oldPrio := proc _raisePriority .
  ].
  interv ifNotNil: [ 
    signalTime := interv + (sched _now).
    targ := sched activeProcess .
    target := targ .
    targ _waitingOn: self.
    sched _delaySchedule: self.
  ] ifNil: [
    (targ := target) ifNil: [
      targ := Semaphore new.
      target := targ . 
      sched _delaySchedule: self.
    ].
  ].
  sched _exitCritical .
  [
    targ _wait .
  ] ensure: [
    sched _enterCritical .
    sched _delayUnschedule: self .
    sched _exitCritical .
  ].
  highPriorityBoolean ifTrue:[
    sched _enterCritical .
    sched activeProcess == proc ifTrue:[
      proc priority: oldPrio .
    ].
    sched _exitCritical .
  ].
  1 timesRepeat:[ self class ].  "check for a Break"
  ^self
%

category: 'Private'
method: Delay
_signalTime: millisecondClockValue
  "sets the receiver's signal time w/o regard to the current time
   or delay interval"
  
  signalTime := millisecondClockValue
%

category: 'Private'
method: Delay
_unscheduleProcess: aGsProcess
  "Used by ProcessorScheduler"
  self _cancel: aGsProcess.
%

category: 'Private'
method: Delay
_changePriority: aGsProcess from: oldPriority
  "Sent by code in GsProcess to change the priority of aGsProcess.waitingOn,
   when aGsProcess.waitingOn is a Delay.  Since a Delay does not contain
   a list of processes sorted by priority, there is nothing to do."
%

