!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   DepListTable, Collection, Object.
!
! class created in idxclasses.topaz
!=========================================================================

removeallmethods DepListTable
removeallclassmethods DepListTable

category: 'For Documentation Installation only'
classmethod: DepListTable
installDocumentation

self comment:
'The class DependencyListTable implements only GemStone internals.  That is, it 
provides only functionality required by GemStone itself.  It is not intended 
for customer use, by creating instances or by subclassing.

A DepListTable is used to hold a global collection of DependencyLists that
are shared by multiple objects.

Constraints:
	obsoletePathTerms: Object
	[elements]: DepListBucket' .
%

category: 'Instance Creation'
classmethod: DepListTable
new: size

"Returns a DepListTable with the given size."

| newOne |
newOne := (super new: size) initialize.
^ newOne
%

category: 'Repository Conversion'
method: DepListTable
reinitialize

"Reinitializes a given DepListTable.  This method is to be used only during
 conversion."

self initialize.
%

category: 'Instance Creation'
classmethod: DepListTable
new

"Returns a DepListTable with the default table size."

| newOne |
newOne := self new: 751.
^ newOne
%

! ------------------- Instance methods for DepListTable
category: 'Initialization'
method: DepListTable
initialize

"Initializes a new instance."

1 to: self _basicSize do: [ :i |
  self at: i put: DepListBucket new.	"Hard-code the class (#43087)"
].

"No need to use obsoletePathTerms set with strong dependency list references
obsoletePathTerms := IdentitySet new.
"
self objectSecurityPolicy: GsIndexingObjectSecurityPolicy.
%

category: 'Updating'
method: DepListTable
_add: depList
  "Replay the addition of the given dependency list to the receiver.
 If there is already an equivalent dependency list in the receiver,
 replace all references to the given dependency list with a reference
 to the dependency list in the table."

  | dl bkt |
  bkt := self depListBucketFor: depList.
  dl := bkt at: depList.
  dl == depList
    ifFalse: [ 
      " must replace references "
      depList _replaceReferencesWith: dl ].
  System _addEntireObjectToRcReadSet: bkt.
  System redoLog addLargeConflictObject: bkt for: self.
  ^ true
%

category: 'Updating'
method: DepListTable
_remove: depList
  "Replay the removal of the given dependency list from the receiver.
   If the dependencyList is obsolete (size 0) then there is no need to
   attempt the removal"

  depList size == 0
    ifTrue: [ ^ true ].
  ^ self remove: depList logging: false
%

category: 'Updating'
method: DepListTable
remove: depList logging: aBoolean
  "Remove the given dependency list from the receiver.
   Returns whether the removal occurred."

  | bkt result |
  bkt := self depListBucketFor: depList.
  result := bkt remove: depList.
  result
    ifTrue: [ 
      aBoolean
        ifTrue: [ self _logRemovalOf: depList inCollisionBucket: bkt ].
      System _addEntireObjectToRcReadSet: bkt.
      System redoLog addLargeConflictObject: bkt for: self ].
  ^ result
%

category: 'Private'
method: DepListTable
_logAdditionOf: depList inCollisionBucket: collisionBkt

"Logs the addition of the given dependency list in the system redo log."

| logEntry |

logEntry := LogEntry new.
logEntry receiver: self;
    selector: #_add:;
    argArray: { depList }.
System redoLog addLogEntry: logEntry
%

category: 'Private'
method: DepListTable
_logRemovalOf: depList inCollisionBucket: collisionBkt
  "Logs the removal of the given dependency list in the system redo log."

  | logEntry |
  logEntry := LogEntry new.
  logEntry
    receiver: self;
    selector: #'_remove:';
    argArray: {depList}.
  System redoLog addLogEntry: logEntry
%

category: 'Updating'
method: DepListTable
at: depList logging: aBoolean
  "Returns the dependency list in the receiver that is equivalent to the
 given dependency list.  If one does not exist, add the given dependency
 list to the receiver and returns the given dependency list.  If the
 collision bucket is updated and aBoolean is true, log the operation
 in the system redo log."

  | dl bkt |
  (depList == nil or: [ depList isEmpty ])
    ifTrue: [ ^ nil ].
  bkt := self depListBucketFor: depList.
  dl := bkt at: depList.
  dl == depList
    ifTrue: [ 
      " new entry was added "
      aBoolean
        ifTrue: [ self _logAdditionOf: depList inCollisionBucket: bkt ].
      depList objectSecurityPolicy: self objectSecurityPolicy ].
  System _addEntireObjectToRcReadSet: bkt.
  System redoLog addLargeConflictObject: bkt for: self.
  ^ dl
%

category: 'Updating'
method: DepListTable
removeEntriesContaining: arrayOfPathTerms
  "Remove all dependency lists from the receiver that have any of the
 path terms contained in the arrayOfPathTerms."

  | bkt depLists depList |
  depLists := {}.
  1 to: self _basicSize do: [ :i | 
    bkt := self _at: i.
    depLists size: 0.
    1 to: bkt size do: [ :j | 
      " build Array of dependency lists to remove "
      depList := bkt _at: j.
      (depList containsAnyOf: arrayOfPathTerms)
        ifTrue: [ depLists add: depList ] ].
    1 to: depLists size do: [ :j | 
      depList := depLists at: j.
      self remove: depList logging: true.
      depList size: 0	" resize dependency list to avoid object audit errors " ].
    bkt resizeIfNecessary.
    IndexManager current commitIndexMaintenance: nil at: 0	"fix 43647: call commitIndexMaintenance bug indexObj and progressCount not applicable" ].
  arrayOfPathTerms size: 0	" avoid object audit errors "
%

category: 'Accessing'
method: DepListTable
size

"Returns the number of entries in the receiver."
| num collisionBkt |
num := 0.
1 to: self _basicSize do: [ :i |
    collisionBkt := self _at: i.
    num := num + collisionBkt size
].
^ num
%

category: 'Accessing'
method: DepListTable
depListBucketFor: depList

"Returns the bucket where the given depList would be found."

<primitive: 524>
self _primitiveFailed: #depListBucketFor: args: { depList } .
self _uncontinuableError
%

category: 'Statistics'
method: DepListTable
statistics

"Returns a Dictionary containing statistics that can be useful in determining
 the performance of the dependency list table."

| bkt dict bktCnt maxBkt sizes cntDict sz total str phySize |
cntDict := KeyValueDictionary new.
dict := SymbolDictionary new.
maxBkt := self _at: 1.
bktCnt := 0.
sizes := 0.
phySize := self physicalSize.
1 to: self _basicSize do: [ :i |
  bkt := self _at: i.

  sz := bkt size.
  sz > 0
    ifTrue: [

      total := cntDict at: sz ifAbsent: [ cntDict at: sz put: 0 ].
      cntDict at: sz put: total + 1.

      bktCnt := bktCnt + 1.
      sizes := sizes + sz.
      " check for max bucket "
      sz > maxBkt size
        ifTrue: [ maxBkt := bkt ].
    ].
  phySize := phySize + bkt physicalSize.
].
dict at: #RootSize put: self _basicSize.
dict at: #TotalBuckets put: bktCnt.
dict at: #LargestBucketSize put: maxBkt size.
dict at: #PhysicalSize put: phySize.

bktCnt > 0
  ifTrue: [ dict at: #AvgEntriesPerBucket put: (sizes // bktCnt) ]
  ifFalse: [ dict at: #AvgEntriesPerBucket put: 0 ].

str := String new.
(cntDict keys sortAscending: #'') do: [ :i |
  str add: i asString; add: ' -> ';
    add: (cntDict at: i) asString; add: Character lf
].
dict at: #Histogram put: str.

^ dict
%

category: 'Adding'
method: DepListTable
add: newObject

""
self shouldNotImplement: #add:
%

category: 'Private'
method: DepListTable
_selectiveAbort

"Abort the collision buckets."

super _selectiveAbort.
1 to: self _basicSize do: [ :i | (self _at: i) _selectiveAbort ]
%

category: 'Enumerating'
method: DepListTable
do: aBlock

"For each dependency list in the receiver, evaluates the one-argument block
 aBlock with the dependency list as the argument."

aBlock _isExecBlock ifFalse:[ aBlock _validateClass: ExecBlock ].
1 to: self _basicSize do: [ :i |
  (self _at: i) do: aBlock
]
%

category: 'Accessing'
method: DepListTable
allEntries

"Returns an IdentitySet containing the receiver's entries."

| entries bkt |
entries := IdentitySet new.
1 to: self _basicSize do: [ :i |
  bkt := self _at: i.
  1 to: bkt size do: [ :j |
    entries add: (bkt _at: j).
  ]
].
^ entries
%

category: 'Private'
method: DepListTable
_resolveRcConflictsWith: conflictObjects

"A logical write-write conflict has occurred on the receiver.  The objects that
 had the actual physical write-write conflicts are in the conflictObjects
 Array.  Selectively abort the receiver and then attempt to replay the
 operations maintained in the System redo log.  Returns true if all the
 operations could be replayed; otherwise returns false."

| result |
result := self _abortAndReplay: conflictObjects.
^ result
%

category: 'Clustering'
method: DepListTable
clusterDepthFirst

"This method clusters the receiver.  Returns true if the receiver has
 already been clustered during the current transaction; returns false
 otherwise."

| result |
self cluster
  ifTrue: [ result := true ]
  ifFalse: [
    " obsoletePathTerms cluster. "
    1 to: self _basicSize do: [ :i | (self _at: i) cluster ].
    result := false
  ].
^ result
%

category: 'Hashing'
method: DepListTable
rebuildTable: newSize

"Rebuilds the table by saving the current state, initializing and changing
 the size of the table, and adding the entries saved back to the table.
 This method is intended to be used by the system administrator when the
 table has become too large.  If it is invoked directly by an application,
 concurrency conflicts may result."

| depLists origSize |
(newSize <= 0)
  ifTrue: [
    newSize _error: #rtErrArgNotPositive .
    ^ self
  ].

" get all entries "
depLists := self allEntries.

origSize := self _basicSize.
" reset each bucket to be empty "
1 to: origSize do: [ :i | (self _at: i) reset ].

super size: newSize.

newSize > origSize
  ifTrue: [ " growing the table "
    (origSize + 1) to: newSize do: [ :i |
      super at: i put: DepListBucket new
    ]
  ].

" now add the entries back to the table "
1 to: depLists size do: [ :i |
  self at: (depLists _at: i) logging: false
].

self objectSecurityPolicy: GsIndexingObjectSecurityPolicy.
%

category: 'Updating'
method: DepListTable
addObsoletePathTerm: aPathTerm

"Adds the given path term to the set of obsolete path terms."

" obsoletePathTerms add: aPathTerm "
%

category: 'Printing'
method: DepListTable
printOn: aStream

"Prints a string representation of the receiver on the given stream."

aStream nextPutAll: 'SharedDependencyLists'
%

category: 'Updating'
method: DepListTable
removeEntriesContainingTracker: aTracker

"Find each dependency list in the receiver that contains a tracker,
 and remove it from the receiver.  Return an Array of dependency lists
 that were removed."

| bkt depLists depList tmpList |

depLists := { } .
tmpList := { } .
1 to: self _basicSize do: [ :i |
  bkt := self _at: i.

  tmpList size: 0.
  " build Array of dependency lists to remove "
  1 to: bkt size do: [ :j |
    depList := bkt _at: j.
    (depList includesIdentical: aTracker)
      ifTrue: [ tmpList add: depList ]
  ].

  tmpList isEmpty
    ifFalse: [
      1 to: tmpList size do: [ :j |
        (self remove: (tmpList at: j) logging: true)
          ifFalse: [ self _error: #rtErrKeyNotFound args: { depList } ].
      ]
    ].

  depLists addAll: tmpList.
].

^ depLists
%

category: 'Updating'
method: DepListTable
_removeAllTrackingOf: aTracker

"Remove the receiver from all dependency lists.  The receiver will
 no longer be notified when objects are modified, and may become
 eligible for garbage collection."

| entries depList |
entries := self removeEntriesContainingTracker: aTracker.
1 to: entries size do: [ :i |
  depList := entries at: i.
  depList _removeCompletelyPathTerm: aTracker.
  self at: depList logging: false.
].
%

# delete _auditEntries .... functionality subsumed by _resortEntriesLogging:

category: 'Auditing'
method: DepListTable
_resortEntriesLogging: aString
  "Resort all entries in the receiver that need it.  Append a record of
what was done on the given string."

  | cnt depList bkt needsSorting empties zeros |
  cnt := 0.
  zeros := IdentitySet new.
  needsSorting := {}.
  empties := {}.
  1 to: self _basicSize do: [ :i | 
    bkt := self _at: i.
    needsSorting size: 0.
    empties size: 0.	" find entries that need sorting "
    1 to: bkt size do: [ :j | 
      (depList := bkt _at: j) size == 0
        ifTrue: [ empties add: depList ]
        ifFalse: [ 
          depList _needsSorting
            ifTrue: [ needsSorting add: depList ] ] ].
    zeros addAll: empties.
    1 to: empties size do: [ :j | 
      " remove empty dependency lists "
      bkt remove: (empties _at: j) ].
    cnt := cnt + needsSorting size.
    1 to: needsSorting size do: [ :j | 
      " remove, resort, then reinsert them "
      (needsSorting _at: j) _sortEntries: bkt ] ].
  zeros size > 0
    ifTrue: [ 
      aString
        add: zeros size asString;
        add: ' dependency list(s) were empty: ';
        add: Character lf;
        add: Character tab;
        add: 'dependency list oops: '.
      zeros do: [ :dl | aString add: dl asOop printString , ' ' ].
      aString lf ].
  cnt > 0
    ifTrue: [ 
      aString
        add: cnt asString;
        add: ' dependency list(s) needed their entries sorted.';
        add: Character lf ].
  ^ aString
%

category: 'Auditing'
method: DepListTable
findNscsWithIndexes

"Return any indexed NSCs we can find."

| result |
result := IdentitySet new.
self allEntries accompaniedBy: result do: [ :res :dl | 
  1 to: dl size by: 2 do: [ :i | | pt |
    pt := dl at: i.
    pt size > 0 ifTrue: [ res add: pt getRootNsc ].
  ].
].
^ result
%

category: 'Auditing'
method: DepListTable
_repairOutOfOrderBucketsLogging: aString

"Repair any collision buckets that have dependency lists out of order."

| bkt depList ooBkts cnt copy |
ooBkts := { } .
" first pass, find buckets that need rebuilding "
1 to: self _basicSize do: [ :i |
  (bkt := self _at: i) _hasEntryOutOfOrder
    ifTrue: [ ooBkts add: bkt ]
].
ooBkts size ~~ 0
  ifTrue: [
    aString add: 'Found '; add: ooBkts size asString;
      add: ' bucket(s) that contained entries out of order.'; add: Character lf.

    " second pass, make copies and reset buckets "
    1 to: ooBkts size do: [ :i |
      copy := (bkt := ooBkts at: i) copy.
      bkt reset.
      ooBkts at: i put: copy.
    ].
    " third pass, re-insert entries "
    cnt := 0.
    1 to: ooBkts size do: [ :i |
      bkt := ooBkts at: i.
      1 to: bkt size do: [ :j |
        depList := bkt _at: j.
        (self at: depList logging: false) == depList
          ifFalse: [ cnt := cnt + 1 ]
      ]
    ].
    cnt ~~ 0
      ifTrue: [
        aString add: 'Re-insertion found '; add: cnt asString;
          add: ' equivalent dependency list(s)'; add: Character lf.
      ]
  ].
^ aString
%

category: 'Auditing'
method: DepListTable
_auditAndRepair: aBool
  "Search for any dependency lists or collision buckets that have
entries out of order, and repair them.  Perform any repairs needed
and return a string describing the repairs."

  | aString |
  aString := String new.
  self _resortEntriesLogging: aString.
  self _repairOutOfOrderBucketsLogging: aString.
  aBool
    ifTrue: [ self _auditDepMapEntries: aString ].
  aString size == 0
    ifTrue: [ ^ 'SharedDependencyLists ok' ]
    ifFalse: [ 
      ^ 'Audit and repair of SharedDependencyLists:
' , aString ]
%

unprotectmethods

category: 'Auditing'
method: DepListTable
auditAndRepair

"Search for any dependency lists or collision buckets that have
entries out of order, and repair them.  Get all dependency
lists by directly enumerating through all dependency lists in the
SharedDependencyLists table.  This method only touches internal
indexing objects, not application objects.  Perform any repairs
needed and return a string describing the repairs."

^ self _auditAndRepair: false
%

category: 'Auditing'
method: DepListTable
fullAuditAndRepair

"Search for any dependency lists or collision buckets that have
entries out of order, and repair them.  Get all dependency
lists by finding all nsc's with indexes that we can, and iterate through
application objects, traversing down each index. path.  This
method performs a superset of the functionality of 'auditAndRepair'.
This can take much more time than 'auditAndRepair', due to the
number of objects fetched.  Perform any repairs needed and return
a string describing the repairs."

^ self _auditAndRepair: true
%

category: 'Updating'
method: DepListTable
objectSecurityPolicy: anObjectSecurityPolicy

"Place the receiver and its collision buckets in the given security policy."

super objectSecurityPolicy: anObjectSecurityPolicy.
1 to: self _basicSize do: [ :i |
  (self _at: i) objectSecurityPolicy: anObjectSecurityPolicy
].

%

category: 'Auditing'
set compile_env: 0
method: DepListTable
_auditDepMapEntries: str
  "verify that all of the dependencyLists tracked in DependencyMap exist in the 
   receiver and all of the dependencyLists in the receiver are tracked in 
   DependencyMap. Also look for duplicate dependencyLists"

  "repair:
     depLists in SharedDepencencyList but not in DepMap
     depList in DepMap not in SharedDependencyLists
   "

  | sharedBag sharedIdentityBag sharedIdentitySet depMapIdentitySet depMapDepListsNotInShared |
  sharedBag := Bag new.
  sharedIdentityBag := IdentityBag new.
  depMapDepListsNotInShared := 0.
  1 to: self _basicSize do: [ :i | 
    | bkt |
    bkt := self _at: i.
    1 to: bkt size do: [ :j | 
      | dl |
      dl := bkt _at: j.
      sharedBag add: dl.
      sharedIdentityBag add: dl ] ].
  sharedIdentityBag
    do: [ :dl | 
      | o |
      (o := sharedIdentityBag occurrencesOf: dl) > 1
        ifTrue: [ 
          "identical entries - should never happen"
          str
            add: 'DependencyList ';
            add: dl printString;
            add: ' [' , dl asOop asString , ']';
            add: ' has ';
            add: o asString;
            add: ' identical entries';
            add: Character lf ].
      (o := sharedBag occurrencesOf: dl) > 1
        ifTrue: [ 
          "not really sharing are we?"
          str
            add: 'DependencyList ';
            add: dl printString;
            add: ' [' , dl asOop asString , ']';
            add: ' has ';
            add: o asString;
            add: ' duplicate equivalent entries';
            add: Character lf ] ].
  depMapIdentitySet := IdentitySet new.
  sharedIdentitySet := sharedIdentityBag asIdentitySet.
  DependencyList depMapValuesToHiddenSet: 2.
  System
    hiddenSet: 2
    do: [ :depList | 
      depMapIdentitySet add: depList.
      (sharedIdentitySet includes: depList)
        ifFalse: [ 
          "depList in DepMap not in SharedDependencyLists"
          depMapDepListsNotInShared := depMapDepListsNotInShared + 1.
          SharedDependencyLists at: depList logging: false ] ].
  sharedIdentitySet removeAllPresent: depMapIdentitySet.
  sharedIdentitySet
    do: [ :depList | 
      "depLists in SharedDepencencyList but not in DepMap
       we can repair by removing depList from the SharedDependencyList"
      self remove: depList logging: false.
      depList size: 0	" resize dependency list to avoid object audit errors " ].
  depMapDepListsNotInShared > 0
    ifTrue: [ 
      str
        add: depMapDepListsNotInShared printString;
        add:
            ' dependency lists were in DepMap and not in SharedDepencyLists (repaired)';
        add: Character lf ]
%
category: 'Private'
classmethod: DepListTable
_hasDependencyBitFor: anObj

"If anObj is a SmallInteger, it is interpreted as an OopType,
 and a depMap lookup is done , without looking at the in-memory state
 (if any) of that object.

 If anObj a non-special object, returns the in-memory hasDependency bit
 for that object without regard to the state of the shadow or shared depMap.

 Returns a Boolean."
<primitive: 161>
self _primitiveFailed: #_hasDependencyBitFor: args: { anObj } .
self _uncontinuableError
%

category: 'Updating'
method: DepListTable
_buildWeakRefSet

"Put each dependency list in the weak references set."

| weakRef |
weakRef := (GsBitmap newForHiddenSet: #WeakReferences).
1 to: self _basicSize do: [ :i | | bkt |
  bkt := self _at: i.
  1 to: bkt size do: [ :j |
    weakRef add: (bkt _at: j)
  ]
].
%

