!=========================================================================
! Copyright (C) GemTalk Systems 2013-2020.  All Rights Reserved.
!
! $Id: idxmethods.topaz 30931 2013-05-29 22:46:30Z otisa $
!
!=========================================================================

! Remove existing behavior from CollectionBasedPathEvaluator
removeallmethods CollectionBasedPathEvaluator
removeallclassmethods CollectionBasedPathEvaluator

doit
CollectionBasedPathEvaluator comment:
'CollectionBasedPathEvaluator is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for CollectionBasedPathEvaluator
! ------------------- Instance methods for CollectionBasedPathEvaluator
category: 'Traversing'
method: CollectionBasedPathEvaluator
traverse: anObject startingAt: offset into: resultSet
  "Traverse the sub-objects of all the objects in the NSC, using the path names
 of the receiver starting at the path term at the given offset.  Add each
 object at the end of the traversal to the result set.  Returns the result
 set."

  self subclassResponsibility: #'traverse:startingAt:into:'
%
expectValue %Boolean
doit
CollectionBasedPathEvaluator category: 'Index-Evaluators'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from EnumeratedPathEvaluator
removeallmethods EnumeratedPathEvaluator
removeallclassmethods EnumeratedPathEvaluator

doit
EnumeratedPathEvaluator comment:
'EnumeratedPathEvaluator is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for EnumeratedPathEvaluator
! ------------------- Instance methods for EnumeratedPathEvaluator
category: 'Accessing'
method: EnumeratedPathEvaluator
enumeratedObjectsFor: theObj pathTerm: enumeratedPathName at: nscOffset
  | nextObj ivOffset enumeratedObjects |
  nextObj := theObj.
  nil == nextObj
    ifTrue: [ ^ #() ].
  enumeratedObjects := {}.
  (enumeratedPathName subStringsDelimitedBy: $|)
    do: [ :pathName | 
      ivOffset := nextObj class _ivOffsetOf: pathName asSymbol.
      ivOffset == nil
        ifTrue: [ enumeratedObjects add: nil ]
        ifFalse: [ enumeratedObjects add: (nextObj instVarAt: ivOffset) ] ].
  ^ enumeratedObjects
%
category: 'Testing'
method: EnumeratedPathEvaluator
hasEnumeratedTerm
  "Returns true if the path has a term that indicates an enumerated path term."

  ^ true
%
category: 'Testing'
method: EnumeratedPathEvaluator
isIndexOnNscElements
  "Returns whether the receiver is a path directly on the elements of an NSC
 (either the root NSC or an enumerated instance variable along the path)."

  ^ (self at: self size) includes: $|
%
category: 'Traversing'
method: EnumeratedPathEvaluator
traverse: anObject startingAt: offset into: resultSet
  "Traverse the sub-objects of all the objects in the NSC, using the path names
 of the receiver starting at the path term at the given offset.  Add each
 object at the end of the traversal to the result set.  Returns the result
 set."

  | nextObj ivOffset pathName nscOffset sz |
  (nil == anObject _and: [ self isIndexOnNscElements not ])
    ifTrue: [ ^ resultSet ].
  nextObj := anObject.
  sz := self size.
  offset to: sz do: [ :i | 
    pathName := self at: i.
    (pathName includes: $|)
      ifTrue: [ 
        nscOffset := i + 1.
        (self enumeratedObjectsFor: nextObj pathTerm: pathName at: nscOffset)
          do: [ :obj | self traverse: obj startingAt: nscOffset into: resultSet ].
        ^ resultSet ]
      ifFalse: [ 
        nil == nextObj
          ifTrue: [ ^ resultSet ].
        ivOffset := nextObj class _ivOffsetOf: pathName.
        ivOffset == nil
          ifTrue: [ 
            nextObj _errorInvalidOffset: pathName.
            nextObj := nil ]
          ifFalse: [ nextObj := nextObj instVarAt: ivOffset ] ].
    (nil == nextObj _and: [ i ~= sz ])
      ifTrue: [ ^ resultSet ] ].
  resultSet add: nextObj.
  ^ resultSet
%
expectValue %Boolean
doit
EnumeratedPathEvaluator category: 'Index-Evaluators'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from Gs32DisjunctiveClauseOptimizer
removeallmethods Gs32DisjunctiveClauseOptimizer
removeallclassmethods Gs32DisjunctiveClauseOptimizer

doit
Gs32DisjunctiveClauseOptimizer comment:
'Gs32DisjunctiveClauseOptimizer is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for Gs32DisjunctiveClauseOptimizer
! ------------------- Instance methods for Gs32DisjunctiveClauseOptimizer
category: 'optimizing'
method: Gs32DisjunctiveClauseOptimizer
consolidateEnumerablePredicates
  "Consolidate predicates that have a path terms that can be combined into a single 
	EnumeratedPath query, i.e., a clause of the form:

      (each.firstName = name) | (each.lastName = name)

    and an index with enumerated path term for 

      each.firstName|lastName
"

  | primaries |
  primaries := self predicates copy.
  1 to: primaries size do: [ :index | 
    (primaries at: index)
      ifNotNil: [ :primaryPredicate | self consolidateEnumeratedPredicates: primaries with: primaryPredicate ] ].
  predicates := primaries select: [ :each | each ~~ nil  ]
%
category: 'optimizing'
method: Gs32DisjunctiveClauseOptimizer
consolidateEnumeratedPredicates: primaries with: primaryPredicate
  | consolidatedPredicate |
  primaries
    do: [ :secondaryPredicate | 
      (secondaryPredicate ~~ nil  and: [ primaryPredicate ~~ secondaryPredicate ])
        ifTrue: [ 
          (primaryPredicate canConsolidateEnumeratedWith: secondaryPredicate)
            ifTrue: [ 
              primaries at: (primaries indexOf: secondaryPredicate) put: nil.
              consolidatedPredicate := primaryPredicate
                consolidateEnumeratedWith: secondaryPredicate.
              primaries
                at: (primaries indexOf: primaryPredicate)
                put: consolidatedPredicate.
              ^ self ] ] ]
%
category: 'optimizing'
method: Gs32DisjunctiveClauseOptimizer
runOptimizations
  self queryOptions removeRedundantPredicates
    ifTrue: [ self removeRedundantDisjunctivePredicates ].
  self queryOptions consolidateEnumerablePredicates
    ifTrue: [ self consolidateEnumerablePredicates ]
%
expectValue %Boolean
doit
Gs32DisjunctiveClauseOptimizer category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from Gs32Optimizer
removeallmethods Gs32Optimizer
removeallclassmethods Gs32Optimizer

doit
Gs32Optimizer comment:
'Gs32Optimizer is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for Gs32Optimizer
! ------------------- Instance methods for Gs32Optimizer
category: 'visiting'
method: Gs32Optimizer
acceptCompoundClause: aCompoundClause
  "replace conjunctive normal form clauses with optimized equivalent"

  | clause |
  clause := aCompoundClause clause1.
  clause isDisjunctiveNormalForm
    ifTrue: [ 
      clause isDisjunctiveClause
        ifTrue: [ clause := self optimizeDisjunctiveCompoundClause: clause ].
      clause := self optimizeDisjunctiveNormalClause: clause.
      aCompoundClause clause1: clause ]
    ifFalse: [ 
      clause isDisjunctiveClause
        ifTrue: [ 
          clause := self optimizeDisjunctiveCompoundClause: clause.
          aCompoundClause clause1: clause ] ].
  clause := aCompoundClause clause2.
  clause isDisjunctiveNormalForm
    ifTrue: [ 
      clause isDisjunctiveClause
        ifTrue: [ clause := self optimizeDisjunctiveCompoundClause: clause ].
      clause := self optimizeDisjunctiveNormalClause: clause.
      aCompoundClause clause2: clause ]
    ifFalse: [ 
      clause isDisjunctiveClause
        ifTrue: [ 
          clause := self optimizeDisjunctiveCompoundClause: clause.
          aCompoundClause clause2: clause ] ].
  super acceptCompoundClause: aCompoundClause
%
category: 'optimizing'
method: Gs32Optimizer
optimizeDisjunctiveCompoundClause: aClause
  | optimizer |
  self queryOptions consolidateUnaryConstantPredicates
    ifFalse: [ ^ aClause ].
  (optimizer := Gs32UnaryConstantPredicateOptimizer new)
    formula: aClause;
    nsc: self nsc;
    queryOptions: self queryOptions .
  ^ (optimizer optimizeFrom: self) copy
%
category: 'accessing'
method: Gs32Optimizer
transformed
  transformed ifNil: [ transformed := false ].
  ^transformed
%
category: 'accessing'
method: Gs32Optimizer
transformed: aBool
  transformed := aBool
%
expectValue %Boolean
doit
Gs32Optimizer category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsAbstractQueryOptimizer
removeallmethods GsAbstractQueryOptimizer
removeallclassmethods GsAbstractQueryOptimizer

doit
GsAbstractQueryOptimizer comment:
'GsAbstractQueryOptimizer is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsAbstractQueryOptimizer
category: 'instance creation'
classmethod: GsAbstractQueryOptimizer
optimize: aFormula on: anNsc options: aGsQueryOptions
  ^ self new
    formula: aFormula;
    nsc: anNsc;
    queryOptions: aGsQueryOptions;
    optimize
%
! ------------------- Instance methods for GsAbstractQueryOptimizer
category: 'accessing'
method: GsAbstractQueryOptimizer
formula
  ^ formula
%
category: 'accessing'
method: GsAbstractQueryOptimizer
formula: aFormula
  formula := aFormula copy
%
category: 'testing'
method: GsAbstractQueryOptimizer
hasNsc
  ^ nsc ~~ nil 
%
category: 'accessing'
method: GsAbstractQueryOptimizer
nsc
  ^ nsc
%
category: 'accessing'
method: GsAbstractQueryOptimizer
nsc: anNsc
  nsc := anNsc
%
category: 'optimizing'
method: GsAbstractQueryOptimizer
optimize
  self subclassResponsibility: #optimize
%
category: 'accessing'
method: GsAbstractQueryOptimizer
queryOptions
  ^ queryOptions
%
category: 'accessing'
method: GsAbstractQueryOptimizer
queryOptions: aGsQueryOptions
  queryOptions := aGsQueryOptions
%
expectValue %Boolean
doit
GsAbstractQueryOptimizer category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsAbstractQueryPredicate
removeallmethods GsAbstractQueryPredicate
removeallclassmethods GsAbstractQueryPredicate

doit
GsAbstractQueryPredicate comment: 
'The abstract class GsAbstractQueryPredicate is a predicate in a GsQueryFormula. A predicate is a boolean expression composed of two operands and an operator (GsQueryPredicate) or three operands and two operators (GsRangeQueryPredicate).

An operand may be a literal (instance of GsConstantReferenceAssociation), a variable (instance of GsVariableReferenceAssociation) or a path (instance of a GsQueryPathReferenceAssociation). 

A path operand is expected to be an instance of a GsQueryPathReferenceAssociation.
An operator may be one of the standard comparison operators: < > = == <= >= ~= ~~.'.
true
%

! ------------------- Class methods for GsAbstractQueryPredicate
! ------------------- Instance methods for GsAbstractQueryPredicate
category: 'visitors'
method: GsAbstractQueryPredicate
acceptVisitor: aFormulaVisitor
  super acceptVisitor: aFormulaVisitor.
  aFormulaVisitor acceptPredicate: self
%
category: 'optimizing'
method: GsAbstractQueryPredicate
applyDeMorgansTransform
  "do not transform, but propagate the transform"

  ^ self
%
category: 'querying-private'
method: GsAbstractQueryPredicate
elementValue: anObject
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  "with a set-valued path term, we may get multiple evaluations of the block ... a value along the path will result in no evaluation of the block"

  self elementValueEvaluator
    traverse: anObject
    do: [ :resolvedObject | 
      | res |
      res := self lastElementValue: resolvedObject.
      res
        ifTrue: [ ^ res ] ].
  ^ false
%
category: 'querying-private'
method: GsAbstractQueryPredicate
elementValueEvaluator
  "companion method to #elementValue:"

  "anObject is the value obtained by traversing path terms to the last value ... analgous to the last element class"

  "not all subclasses are required to implement this selector, if they do not implement this selector, then they must implement #elementValue:"

  self subclassResponsibility: #'elementValueEvaluator'
%
category: 'testing'
method: GsAbstractQueryPredicate
isPredicate
  ^ true
%
category: 'querying-private'
method: GsAbstractQueryPredicate
lastElementValue: anObject
  "companion method to #elementValue:"

  "anObject is the value obtained by traversing path terms to the last value ... analgous to the last element class"

  "not all subclasses are required to implement this selector, if they do not implement this selector, then they must implement #elementValue:"

  self subclassResponsibility: #'lastElementValue:'
%
expectValue %Boolean
doit
GsAbstractQueryPredicate category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsAbstractReferenceAssociation
removeallmethods GsAbstractReferenceAssociation
removeallclassmethods GsAbstractReferenceAssociation

doit
GsAbstractReferenceAssociation comment: 
'The abstract class GsAbstractReferenceAssociation represents the value of an operand in a GsAbstractQueryPredicate.'.
true
%

! ------------------- Class methods for GsAbstractReferenceAssociation
! ------------------- Instance methods for GsAbstractReferenceAssociation
category: 'converting'
method: GsAbstractReferenceAssociation
asFormula
  self subclassResponsibility: #asFormula
%
category: 'converting'
method: GsAbstractReferenceAssociation
asFormulaWithSelectorParts: selectorParts
  ^ GsUnaryClause
    clause: self asFormula
    operator: selectorParts first inputValue asSymbol
%
category: 'converting'
method: GsAbstractReferenceAssociation
asFormulaWithSelectorParts: selectorParts arguments: arguments
  ^ arguments first
    constructFormulaWithSelector: selectorParts first inputValue
    againstConstantReferenceAssoction: self
%
category: 'converting'
method: GsAbstractReferenceAssociation
asReferenceAssociation
  ^ self
%
category: 'accessing'
method: GsAbstractReferenceAssociation
bind: variableName to: anObject
  "noop"

%
category: 'private'
method: GsAbstractReferenceAssociation
constructFormulaWithSelector: selector againstConstantReferenceAssoction: aConstantReferenceAssociation
  ^ GsConstantConstantPredicate new
    operand1: aConstantReferenceAssociation;
    operator: selector asSymbol;
    operand2: self;
    immediateInvariant
%
category: 'private'
method: GsAbstractReferenceAssociation
constructFormulaWithSelector: selector againstPathReferenceAssoction: aPathReferenceAssociation
  ^ GsPathConstantPredicate new
    path1: aPathReferenceAssociation;
    operator: selector asSymbol;
    operand2: self;
    immediateInvariant
%
category: 'testing'
method: GsAbstractReferenceAssociation
hasSamePathAs: aReferenceAssoc
  ^ false
%
category: 'testing'
method: GsAbstractReferenceAssociation
isBound
  ^ true
%
category: 'printing'
method: GsAbstractReferenceAssociation
printOn: aStream
  aStream nextPutAll: self key asString
%
category: 'accessing'
method: GsAbstractReferenceAssociation
unbind
  "remove all bindings"

  "noop"

%
category: 'private'
method: GsAbstractReferenceAssociation
buildGsQueryParserNodeFor: aGsQueryParser
  aGsQueryParser build: self messages: nil
%
category: 'converting'
method: GsAbstractReferenceAssociation
  asString
    ^ self printString
%
expectValue %Boolean
doit
GsAbstractReferenceAssociation category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsClassicConjunctiveClauseOptimizer
removeallmethods GsClassicConjunctiveClauseOptimizer
removeallclassmethods GsClassicConjunctiveClauseOptimizer

doit
GsClassicConjunctiveClauseOptimizer comment:
'GsClassicConjunctiveClauseOptimizer is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsClassicConjunctiveClauseOptimizer
! ------------------- Instance methods for GsClassicConjunctiveClauseOptimizer
category: 'optimizing'
method: GsClassicConjunctiveClauseOptimizer
consolidatePredicates: primaries with: primaryPredicate
  "See if the primaryPredicates can be consolidated with any of the primiaries."

  "see QueryExecuter>>_consolidatePredicatesAt:and:"

  | consolidatedPredicate |
  primaries
    do: [ :secondaryPredicate | 
      (secondaryPredicate ~~ nil  and: [ primaryPredicate ~~ secondaryPredicate ])
        ifTrue: [ 
          (primaryPredicate canConsolidateWith: secondaryPredicate)
            ifTrue: [ 
              primaries at: (primaries indexOf: secondaryPredicate) put: nil.
              consolidatedPredicate := primaryPredicate
                consolidateWith: secondaryPredicate.
              consolidatedPredicate evaluator: primaryPredicate evaluator1.
              primaries
                at: (primaries indexOf: primaryPredicate)
                put: consolidatedPredicate.
              ^ self ] ] ]
%
category: 'optimizing'
method: GsClassicConjunctiveClauseOptimizer
consolidateRangePredicates
  "See if any predicates can be consolidated (i.e. can be satisfied 
	with a single lookup using a range predicate)."

  "see QueryExecuter>>_consolidatePredicatesAt:and:"

  | primaries |
  primaries := self predicates copy.
  1 to: primaries size do: [ :index | 
    (primaries at: index)
      ifNotNil: [ :primaryPredicate | self consolidatePredicates: primaries with: primaryPredicate ] ].
  predicates := primaries select: [ :each | each ~~ nil  ]
%
category: 'optimizing'
method: GsClassicConjunctiveClauseOptimizer
normalizePredicates
  predicates := self predicates
    collect: [ :each | 
      "normalize predicates to make simplify analysis"
      each normalize ]
%
category: 'optimizing'
method: GsClassicConjunctiveClauseOptimizer
removeRedundantPredicates
  "see QueryExecuter>>_removeRedundantOperatorsAt:and:"

  | primaries |
  [ 
  primaries := self predicates copy.
  1 to: primaries size do: [ :index | 
    (primaries at: index)
      ifNotNil: [ :primaryPredicate | self removeRedundantPredicates: primaries with: primaryPredicate ] ].
  predicates := primaries select: [ :each | each ~~ nil  ] ]
    on: GsUnsatisfiableQueryNotification
    do: [ :note | 
      "short circuit analysis and replace whole enchilada with a `false` predicate"
      predicates := OrderedCollection with: (GsQueryPredicate constant: false).
      ^ self ]
%
category: 'optimizing'
method: GsClassicConjunctiveClauseOptimizer
removeRedundantPredicates: primaries with: primaryPredicate
  "see QueryExecuter>>_removeRedundantOperatorsAt:and:"

  primaries
    do: [ :secondaryPredicate | 
      (secondaryPredicate ~~ nil  and: [ primaryPredicate ~~ secondaryPredicate ])
        ifTrue: [ 
          (primaryPredicate redundantPredicateBetween: secondaryPredicate)
            ifNotNil: [ :redundantPredicate | primaries at: (primaries indexOf: redundantPredicate) put: nil ] ] ]
%
category: 'optimizing'
method: GsClassicConjunctiveClauseOptimizer
reorderPredicates
  | constList identList equalList otherList indexList |
  self predicates size < 2
    ifTrue: [ ^ self ].
  constList := {}.
  identList := {}.
  equalList := {}.
  otherList := {}.
  indexList := {}.
  self predicates
    do: [ :each | 
      " if it is constant-constant "
      each isConstantConstant
        ifTrue: [ constList addLast: each ]
        ifFalse: [ 
          " if an index exists on the path and it is not path-path "
          (each isPathPath not and: [ each usesPathEvaluator not ])
            ifTrue: [ indexList addLast: each ]
            ifFalse: [ 
              " if it is an identity operation (== or ~~) "
              each usesIdentityOperation
                ifTrue: [ identList addLast: each ]
                ifFalse: [ 
                  " if operation is = or ~= "
                  each usesEqualityOperation
                    ifTrue: [ equalList addLast: each ]
                    ifFalse: [ otherList addLast: each ] ] ] ] ].
  predicates := OrderedCollection new.
  1 to: constList size do: [ :i | predicates addLast: (constList at: i) ].
  1 to: indexList size do: [ :i | predicates addLast: (indexList at: i) ].
  1 to: identList size do: [ :i | predicates addLast: (identList at: i) ].
  1 to: equalList size do: [ :i | predicates addLast: (equalList at: i) ].
  1 to: otherList size do: [ :i | predicates addLast: (otherList at: i) ]
%
category: 'optimizing'
method: GsClassicConjunctiveClauseOptimizer
runOptimizations
  self queryOptions transformCommonPaths
    ifTrue: [ self transformCommonPaths ].
  self queryOptions normalizePredicates
    ifTrue: [ self normalizePredicates ].
  self queryOptions removeRedundantPredicates
    ifTrue: [ self removeRedundantPredicates ].
  self queryOptions consolidateRangePredicates
    ifTrue: [ self consolidateRangePredicates ].
  self queryOptions consolidateUnaryConstantPredicates
    ifTrue: [ self consolidateUnaryConstantPredicates ].
  self queryOptions reorderPredicates
    ifTrue: [ self reorderPredicates ]
%
category: 'optimizing'
method: GsClassicConjunctiveClauseOptimizer
transformCommonPaths
  "see QueryExecuter>>_checkPathPathAt:"

  predicates := self predicates collect: [ :each | each transformCommonPaths ]
%
expectValue %Boolean
doit
GsClassicConjunctiveClauseOptimizer category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsClassicQueryBlockOptimizer
removeallmethods GsClassicQueryBlockOptimizer
removeallclassmethods GsClassicQueryBlockOptimizer

doit
GsClassicQueryBlockOptimizer comment:
'GsClassicQueryBlockOptimizer is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsClassicQueryBlockOptimizer
! ------------------- Instance methods for GsClassicQueryBlockOptimizer
category: 'visiting'
method: GsClassicQueryBlockOptimizer
acceptCompoundClause: aCompoundClause
  "replace conjunctive normal form clauses with optimized equivalent"

  aCompoundClause clause1 isConjunctiveNormalForm
    ifTrue: [ 
      aCompoundClause
        clause1: (self optimizeConjunctiveNormalClause: aCompoundClause clause1) ].
  aCompoundClause clause2 isConjunctiveNormalForm
    ifTrue: [ 
      aCompoundClause
        clause2: (self optimizeConjunctiveNormalClause: aCompoundClause clause2) ].
  super acceptCompoundClause: aCompoundClause
%
category: 'visiting'
method: GsClassicQueryBlockOptimizer
acceptUnaryClause: aUnaryClause
  | clause |
  self queryOptions applyDeMorgansLaws
    ifTrue: [ self applyDeMorgansLaws: aUnaryClause ].
  clause := aUnaryClause clause.
  clause isConjunctiveNormalForm
    ifTrue: [ clause := self optimizeConjunctiveNormalClause: clause ].
  aUnaryClause clause: clause.
  super acceptUnaryClause: aUnaryClause
%
category: 'optimizing'
method: GsClassicQueryBlockOptimizer
applyDeMorgansLaws: aClause
  ^ aClause applyDeMorgansTransform
%
category: 'optimizing'
method: GsClassicQueryBlockOptimizer
optimize
  | formulaCopy optimizedFormula |
  formulaCopy := self formula copy.
  optimizedFormula := formulaCopy isConjunctiveNormalForm
    ifTrue: [ self optimizeConjunctiveNormalClause: formulaCopy ]
    ifFalse: [ 
      self queryOptions applyDeMorgansLaws
        ifTrue: [ 
          formulaCopy := self applyDeMorgansLaws: formulaCopy.
          formulaCopy isConjunctiveNormalForm
            ifTrue: [ self optimizeConjunctiveNormalClause: formulaCopy ]
            ifFalse: [ self visitFormula: formulaCopy ] ]
        ifFalse: [ self visitFormula: formulaCopy ] ].
  ^ optimizedFormula immediateInvariant
%
category: 'optimizing'
method: GsClassicQueryBlockOptimizer
optimizeConjunctiveNormalClause: aClause
  ^ (GsClassicConjunctiveClauseOptimizer
    optimize: aClause
    on: self nsc
    options: self queryOptions) copy
%
expectValue %Boolean
doit
GsClassicQueryBlockOptimizer category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsClauseOptimizer
removeallmethods GsClauseOptimizer
removeallclassmethods GsClauseOptimizer

doit
GsClauseOptimizer comment:
'GsClauseOptimizer is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsClauseOptimizer
! ------------------- Instance methods for GsClauseOptimizer
category: 'visiting'
method: GsClauseOptimizer
acceptPredicate: aPredicate
  self predicates add: aPredicate
%
category: 'accessing'
method: GsClauseOptimizer
nsc: anNsc
  super nsc: anNsc.
  nsc
    ifNotNil: [ formula := formula bindEvaluatorsFor: anNsc collator: self collator ]
%
category: 'optimizing'
method: GsClauseOptimizer
optimize
  self visitFormula: self formula.
  self runOptimizations.
  ^ self predicatesAsFormula
%
category: 'accessing'
method: GsClauseOptimizer
predicates
  predicates ifNil: [ predicates := OrderedCollection new ].
  ^ predicates
%
category: 'accessing'
method: GsClauseOptimizer
predicatesAsFormula
  | newFormula |
  newFormula := (predicates at: 1) copy.
  2 to: predicates size do: [ :index | newFormula := newFormula & (predicates at: index) copy ].
  ^ (newFormula bindEvaluatorsFor: self nsc collator: self collator)
    immediateInvariant
%
category: 'optimizing'
method: GsClauseOptimizer
runOptimizations
  "noop"

%
expectValue %Boolean
doit
GsClauseOptimizer category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsCompoundClause
removeallmethods GsCompoundClause
removeallclassmethods GsCompoundClause

doit
GsCompoundClause comment: 
'The class GsCompoundClause is a predicate in a GsQueryFormula. This class applies an operator to two clauses or predicates. Each clause or predicate is expected to be a subclass of GsQueryFormula.

An operator may be one of the following: & |.'.
true
%

! ------------------- Class methods for GsCompoundClause
category: 'instance creation'
classmethod: GsCompoundClause
clause: clause1 operator: anOperator clause: clause2
  | res |
  (res := self new)
    clause1: clause1;
    operator: anOperator;
    clause2: clause2 .
  ^ res
%
! ------------------- Instance methods for GsCompoundClause
category: 'visiting'
method: GsCompoundClause
acceptVisitor: aFormulaVisitor
  super acceptVisitor: aFormulaVisitor.
  aFormulaVisitor acceptCompoundClause: self
%
category: 'optimizing'
method: GsCompoundClause
applyDeMorgansNegationTransform
  "actively apply De Morgan's laws ... operating on copy"

  self clause1: self clause1 applyDeMorgansNegationTransform.
  self operator: self operatorNegated.
  self clause2: self clause2 applyDeMorgansNegationTransform
%
category: 'optimizing'
method: GsCompoundClause
applyDeMorgansTransform
  "do not transform, but propagate the transform ... operating on copy"

  self clause1: self clause1 applyDeMorgansTransform.
  self clause2: self clause2 applyDeMorgansTransform.
  ^ self
%
category: 'converting'
method: GsCompoundClause
asFormulaWithSelectorParts: selectorParts arguments: arguments
  | selector |
  selector := selectorParts first inputValue asSymbol.
  ^ self perform: selector with: arguments first
%
category: 'transforming'
method: GsCompoundClause
bind: variableName to: value
  | bound |
  bound := self copy.
  bound clause1: (bound clause1 bind: variableName to: value).
  bound clause2: (bound clause2 bind: variableName to: value).
  ^ bound immediateInvariant
%
category: 'private'
method: GsCompoundClause
bindEvaluators
  clause1 := clause1 bindEvaluatorsFor: nsc collator: self collator.
  clause2 := clause2 bindEvaluatorsFor: nsc collator: self collator.
  super bindEvaluators
%
category: 'accessing'
method: GsCompoundClause
clause1
  ^ clause1
%
category: 'accessing'
method: GsCompoundClause
clause1: newValue
  clause1 := newValue
%
category: 'accessing'
method: GsCompoundClause
clause2
  ^ clause2
%
category: 'accessing'
method: GsCompoundClause
clause2: newValue
  clause2 := newValue
%
category: 'querying-private'
method: GsCompoundClause
elementValue: anObject
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  ^ (self clause1 elementValue: anObject)
    perform: self operator
    with: (self clause2 elementValue: anObject)
%
category: 'querying'
method: GsCompoundClause
executeAndDo: aBlock
  | preparedClause2 |
  preparedClause2 := self clause2 prepareForExecution.
  self operator == #'&'
    ifTrue: [ 
      self clause1
        executeAndDo: [ :obj | 
          (preparedClause2 elementValue: obj)
            ifTrue: [ aBlock value: obj ] ] ]
    ifFalse: [ 
      | preparedClause1 |
      preparedClause1 := self clause1 prepareForExecution.
      self _nsc
        do: [ :obj | 
          (preparedClause1 elementValue: obj)
            ifTrue: [ aBlock value: obj ]
            ifFalse: [ 
              (preparedClause2 elementValue: obj)
                ifTrue: [ aBlock value: obj ] ] ] ]
%
category: 'querying-private'
method: GsCompoundClause
executeClause
  ^ self executeClauseUsing: self operator
%
category: 'querying-private'
method: GsCompoundClause
executeClauseNegated
  | intermediateResult |
  intermediateResult := self clause1 executeClauseNegated.
  intermediateResult isEmpty
    ifTrue: [ 
      self operatorNegated == #'&'
        ifTrue: [ ^ intermediateResult ]
        ifFalse: [ ^ self clause2 executeClauseNegated ] ].
  ^ (self operatorNegated == #'&'
    and: [ 
      intermediateResult size < self bruteThreshold
        or: [ self clause2 usesPathEvaluatorForFirstClause ] ])
    ifTrue: [ 
      "Evaluate clause2 against intermediateResult when performing an #& and intermediateResult size
       is less than 1000. Otherwise it is more efficient to use an index for clause2."
      self clause2 executeClauseNegatedOn: intermediateResult ]
    ifFalse: [ 
      intermediateResult
        perform: (self resultOperatorFor: self operatorNegated)
        with: self clause2 executeClauseNegated ]
%
category: 'querying-private'
method: GsCompoundClause
executeClauseUsing: anOperator
  | intermediateResult |
  intermediateResult := self clause1 executeClause.
  intermediateResult isEmpty
    ifTrue: [ 
      anOperator == #'&'
        ifTrue: [ ^ intermediateResult ]
        ifFalse: [ ^ self clause2 executeClause ] ].
  ^ (anOperator == #'&'
    and: [ 
      intermediateResult size < self bruteThreshold
        or: [ self clause2 usesPathEvaluatorForFirstClause ] ])
    ifTrue: [ 
      "Evaluate clause2 against intermediateResult when performing an #& and intermediateResult size
       is less than 1000. Otherwise it is more efficient to use an index for clause2."
      self clause2 executeClauseOn: intermediateResult ]
    ifFalse: [ 
      intermediateResult
        perform: (self resultOperatorFor: anOperator)
        with: self clause2 executeClause ]
%
category: 'querying-private'
method: GsCompoundClause
executeNegatedAndDo: aBlock
  | preparedClause2 |
  preparedClause2 := self clause2 prepareForExecution.
  self operatorNegated == #'&'
    ifTrue: [ 
      self clause1
        executeNegatedAndDo: [ :obj | 
          (preparedClause2 elementValue: obj)
            ifFalse: [ aBlock value: obj ] ] ]
    ifFalse: [ 
      | preparedClause1 |
      preparedClause1 := self clause1 prepareForExecution.
      self _nsc
        do: [ :obj | 
          (preparedClause1 elementValue: obj)
            ifTrue: [ 
              (preparedClause2 elementValue: obj)
                ifFalse: [ aBlock value: obj ] ]
            ifFalse: [ aBlock value: obj ] ] ]
%
category: 'private'
method: GsCompoundClause
immediateInvariant
  super immediateInvariant.
  clause1 immediateInvariant.
  clause2 immediateInvariant
%
category: 'testing'
method: GsCompoundClause
isCompoundPredicate
  ^ true
%
category: 'testing'
method: GsCompoundClause
isConjunctiveNormalForm
  ^ self operator == #'&'
    and: [ 
      self clause1 isConjunctiveNormalForm
        and: [ self clause2 isConjunctiveNormalForm ] ]
%
category: 'operators'
method: GsCompoundClause
normalize
  | normalized |
  normalized := super normalize.
  normalized clause1: clause1 normalize.
  normalized clause2: clause2 normalize
%
category: 'accessing'
method: GsCompoundClause
operator
  ^operator
%
category: 'accessing'
method: GsCompoundClause
operator: aSymbol
  operator := aSymbol
%
category: 'accessing'
method: GsCompoundClause
operatorNegated
  ^ operator == #'&'
    ifTrue: [ #'|' ]
    ifFalse: [ #'&' ]
%
category: 'private'
method: GsCompoundClause
postCopy
  clause1 := clause1 copy.
  clause2 := clause2 copy.
  super postCopy
%
category: 'printing'
method: GsCompoundClause
printOn: aStream
  | parenthesizeClause |
  aStream nextPutAll: self clause1 printString.
  aStream nextPutAll: ' ' , self operator asString , ' '.
  parenthesizeClause := self clause2 isCompoundPredicate.
  parenthesizeClause
    ifTrue: [ aStream nextPut: $( ].
  aStream nextPutAll: self clause2 printString.
  parenthesizeClause
    ifTrue: [ aStream nextPut: $) ]
%
category: 'accessing'
method: GsCompoundClause
resultOperatorFor: anOperator
  ^ anOperator == #'&'
    ifTrue: [ 
      "intersection"
      #'*' ]
    ifFalse: [ 
      "minimal union"
      #'_union:' ]
%
category: 'transforming'
method: GsCompoundClause
unbind
  "remove all bindings"

  | unbound |
  unbound := super unbind.
  unbound clause1: unbound clause1 unbind.
  unbound clause2: unbound clause2 unbind.
  ^ unbound immediateInvariant
%
expectValue %Boolean
doit
GsCompoundClause category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsConstantConstantPredicate
removeallmethods GsConstantConstantPredicate
removeallclassmethods GsConstantConstantPredicate

doit
GsConstantConstantPredicate comment: 
'The class GsConstantConstantPredicate is a predicate in a GsQueryFormula. This class is a concrete subclass of GsQueryPredicate and applies an operator to two constant operands. 

A constant operand may be a literal (instance of GsConstantReferenceAssociation) or a variable (instance of GsVariableReferenceAssociation).

An operator may be one of the following: < > = == <= >= ~= ~~.'.
true
%

! ------------------- Class methods for GsConstantConstantPredicate
! ------------------- Instance methods for GsConstantConstantPredicate
category: 'transforming'
method: GsConstantConstantPredicate
bind: variableName to: value
  ^ self operator == #'unary'
    ifTrue: [ 
      | bound |
      bound := self copy.
      bound operand1: (bound operand1 bind: variableName to: value).
      ^ bound immediateInvariant ]
    ifFalse: [ super bind: variableName to: value ]
%
category: 'querying-private'
method: GsConstantConstantPredicate
elementValue: anObject
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  ^ self executePredicate
%
category: 'querying'
method: GsConstantConstantPredicate
executeAndDo: aBlock
  self executePredicate
    ifTrue: [ nsc do: aBlock ]
%
category: 'querying-private'
method: GsConstantConstantPredicate
executeClause
  ^ self executePredicate
    ifTrue: [ nsc ]
    ifFalse: [ nsc species new ]
%
category: 'querying-private'
method: GsConstantConstantPredicate
executeClauseNegated
  self executePredicateNegated
    ifTrue: [ ^ nsc ]
    ifFalse: [ ^ nsc species new ]
%
category: 'querying-private'
method: GsConstantConstantPredicate
executeNegatedAndDo: aBlock
  self executePredicateNegated
    ifTrue: [ nsc do: aBlock ]
%
category: 'querying-private'
method: GsConstantConstantPredicate
executePredicate
  ^ self operator == #'unary'
    ifTrue: [ self operand1 _idxValue ]
    ifFalse: [ 
      self operand1 _idxValue
        perform: (self comparisonSelectorFor: self operator)
        with: self operand2 _idxValue ]
%
category: 'querying-private'
method: GsConstantConstantPredicate
executePredicateNegated
  ^ self operator == #'unary'
    ifTrue: [ self operand1 _idxValue not ]
    ifFalse: [ self operand1 _idxValue perform: self operatorNegated with: self operand2 _idxValue ]
%
category: 'testing'
method: GsConstantConstantPredicate
isConstantConstant
  ^ true
%
category: 'printing'
method: GsConstantConstantPredicate
printOn: aStream
  self operator ~~ #'unary'
    ifTrue: [ ^ super printOn: aStream ].
  aStream nextPutAll: '( ' , self operand1 key asString , ' )'
%
category: 'initialization'
method: GsConstantConstantPredicate
_fromArray: varsArray terms: termsArray links: linksArray paths: pathsArray startingAt: i
  self constant1: (varsArray at: (termsArray at: i + 5)).
  operator := self operationSelectors at: (termsArray at: i) + 1.
  self constant2: (varsArray at: (termsArray at: i + 2))
%
expectValue %Boolean
doit
GsConstantConstantPredicate category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsConstantPathPredicate
removeallmethods GsConstantPathPredicate
removeallclassmethods GsConstantPathPredicate

doit
GsConstantPathPredicate comment: 
'The class GsConstantPathPredicate is a predicate in a GsQueryFormula. This class is a concrete subclass of GsQueryPredicate and applies an operator to a constant operand and a path operand. 

A constant operand may be a literal (instance of GsConstantReferenceAssociation) or a variable (instance of GsVariableReferenceAssociation). 

A path operand is expected to be an instance of a GsQueryPathReferenceAssociation.

An operator may be one of the following: < > = == <= >= ~= ~~.'.
true
%

! ------------------- Class methods for GsConstantPathPredicate
category: 'instance creation'
classmethod: GsConstantPathPredicate
constant: operand1 operator: operator path: operand2
  | res |
  (res := self new)
    constant1: operand1;
    operator: operator;
    path2: operand2 .
  ^ res
%
! ------------------- Instance methods for GsConstantPathPredicate
category: 'private'
method: GsConstantPathPredicate
bindEvaluators
  evaluator1 := self
    _bindEvaluator: nsc
    for: self path
    isRangeEqualityOperation: (self isRangeEqualityOperation: self operator).
  super bindEvaluators
%
category: 'transforming'
method: GsConstantPathPredicate
collapseToRangePredicateIfPossible: aPathXPredicate
  ^ aPathXPredicate collapseToRangePredicateForConstantPathPredicate: self
%
category: 'accessing'
method: GsConstantPathPredicate
constant
  ^ self operand1 _idxValue 
%
category: 'accessing'
method: GsConstantPathPredicate
evaluator
  ^ evaluator1
%
category: 'querying'
method: GsConstantPathPredicate
executeAndDo: aBlock
  ^ self executeAndDo: aBlock using: self operator
%
category: 'querying-private'
method: GsConstantPathPredicate
executeAndDo: aBlock using: anOperator
  | queryEvaluator |
  queryEvaluator := self evaluator1 asQueryEvaluator.
  queryEvaluator doBlock: aBlock.
  ^ self executeClauseUsing: anOperator queryEvaluator: queryEvaluator
%
category: 'querying-private'
method: GsConstantPathPredicate
executeClause
  ^ self executeClauseUsing: self operator
%
category: 'querying-private'
method: GsConstantPathPredicate
executeClauseNegated
  ^ self executeClauseUsing: self operatorNegated
%
category: 'querying-private'
method: GsConstantPathPredicate
executeClauseUsing: anOperator
  | queryEvaluator |
  queryEvaluator := self evaluator1 asQueryEvaluator.
  ^ self executeClauseUsing: anOperator queryEvaluator: queryEvaluator
%
category: 'querying-private'
method: GsConstantPathPredicate
executeNegatedAndDo: aBlock
  ^ self executeAndDo: aBlock using: self operatorNegated
%
category: 'testing'
method: GsConstantPathPredicate
isConstantBound
  ^ self operand1 isBound
%
category: 'querying-private'
method: GsConstantPathPredicate
lastElementValue: anObject
  "anObject is the value obtained by traversing path terms to the last value ... analgous to the last element class"

  ^ self constant
    perform: (self comparisonSelectorFor: self operator)
    with: anObject
%
category: 'operators'
method: GsConstantPathPredicate
normalize
  | res |
  (res := GsPathConstantPredicate new)
    operand1: self operand2 copy;
    evaluator1: self evaluator1;
    operator: (self inverseOperatorFor: self operator);
    operand2: self operand1 copy .
  ^ res
%
category: 'testing'
method: GsConstantPathPredicate
operand2IsPath
  ^ true
%
category: 'accessing'
method: GsConstantPathPredicate
rawPath
  ^ self operand2
%
category: 'initialization'
method: GsConstantPathPredicate
_fromArray: varsArray terms: termsArray links: linksArray paths: pathsArray startingAt: i
  self constant1: (varsArray at: (termsArray at: i + 5)).
  operator := self operationSelectors at: (termsArray at: i) + 1.
  self
    path2:
      (self _pathFrom: linksArray paths: pathsArray offset: (termsArray at: i + 3))
%
expectValue %Boolean
doit
GsConstantPathPredicate category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsConstantReferenceAssociation
removeallmethods GsConstantReferenceAssociation
removeallclassmethods GsConstantReferenceAssociation

doit
GsConstantReferenceAssociation comment: 
'The class GsConstantReferenceAssociation represents the value of a literal operand in a GsAbstractQueryPredicate.'.
true
%

! ------------------- Class methods for GsConstantReferenceAssociation
! ------------------- Instance methods for GsConstantReferenceAssociation
category: 'printing'
method: GsConstantReferenceAssociation
printOn: aStream
  aStream nextPutAll: self value printString
%
expectValue %Boolean
doit
GsConstantReferenceAssociation category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsIndexOptions
removeallmethods GsIndexOptions
removeallclassmethods GsIndexOptions

doit
GsIndexOptions comment:
'The class GsIndexOptions is part of the Indexing and Querying subsystem.  This class encapsulates the optional refinements that may be used when creating a particular index on a collection.  Each GsIndexSpec has an instance of GsIndexOptions, either the default one or an instance that is created and included in the GsIndexSpec specification method.

GsIndexOptions are stored in dynamic instance variables for ease of migration; all are false by default:
   reducedConflict
   optionalPathTerms
'.
true
%

! ------------------- Class methods for GsIndexOptions
category: 'Default'
classmethod: GsIndexOptions
default
  "default options "

  (SessionTemps current at: #GsDefaultIndexOptions ifAbsent: [])
    ifNotNil: [:sessionDefault | ^ sessionDefault ].
  (Globals at: #GsDefaultIndexOptions ifAbsent: [])
    ifNotNil: [ :default | ^ default ].
  "set state explicitly to avoid infinite loop getting default options when no default defined yet."
  ^ self basicNew
      btreePlusIndex: true;
      legacyIndex: false;
      reducedConflict: false;
      optionalPathTerms: false;
      optimizedComparison: false;
      yourself
%
category: 'Accessing'
classmethod: GsIndexOptions
reducedConflict
  "Create a reduced conflict index"

  "Use basicNew to bypass initialization, thus making it possible to use the default index options
   to supply the value for slots not explicitly set or cleared."

  ^ self basicNew
    setReducedConflict;
    yourself
%

! ------------------- Instance methods for GsIndexOptions
category: 'Initialize-Release'
method: GsIndexOptions
initialize
  | default |
  super initialize.
  default := self defaultOptions.
  self btreePlusIndex: default btreePlusIndex.
  self legacyIndex: default legacyIndex.
  self reducedConflict: false.
  self optionalPathTerms: false.
  self optimizedComparison: false
%
category: 'Operators'
method: GsIndexOptions
clearReducedConflict
  self reducedConflict: false
%
category: 'Operators'
method: GsIndexOptions
combineWith: aGsIndexOptions
  "all set/cleared option values in receiver set/cleared in <aGsIndexOptions>"

  self _reducedConflict == true
    ifTrue: [ aGsIndexOptions setReducedConflict ].
  self _reducedConflict == false
    ifTrue: [ aGsIndexOptions clearReducedConflict ].
  self _optionalPathTerms == true
    ifTrue: [ aGsIndexOptions setOptionalPathTerms ].
  self _optionalPathTerms == false
    ifTrue: [ aGsIndexOptions clearOptionalPathTerms ].
  self _legacyIndex == true
    ifTrue: [ aGsIndexOptions setLegacyIndex ].
  self _legacyIndex == false
    ifTrue: [ aGsIndexOptions clearLegacyIndex ].
  self _btreePlusIndex == true
    ifTrue: [ aGsIndexOptions setBtreePlusIndex ].
  self _btreePlusIndex == false
    ifTrue: [ aGsIndexOptions clearBtreePlusIndex ].
  self _optimizedComparison == true
    ifTrue: [ aGsIndexOptions setOptimizedComparison ].
  self _optimizedComparison == false
    ifTrue: [ aGsIndexOptions clearOptimizedComparison ].
  ^ aGsIndexOptions
%
category: 'Operators'
method: GsIndexOptions
negateWith: aGsIndexOptions
  "all set/cleared option values in receiver cleared/set in <aGsIndexOptions>"

  self _reducedConflict == true
    ifTrue: [ aGsIndexOptions clearReducedConflict ].
  self _reducedConflict == false
    ifTrue: [ aGsIndexOptions setReducedConflict ].
  self _optionalPathTerms == true
    ifTrue: [ aGsIndexOptions clearOptionalPathTerms ].
  self _optionalPathTerms == false
    ifTrue: [ aGsIndexOptions setOptionalPathTerms ].
  self _legacyIndex == true
    ifTrue: [ 
      aGsIndexOptions clearLegacyIndex.
      aGsIndexOptions setBtreePlusIndex ].
  self _legacyIndex == false
    ifTrue: [ 
      aGsIndexOptions setLegacyIndex.
      aGsIndexOptions clearBtreePlusIndex ].
  self _btreePlusIndex == true
    ifTrue: [ 
      aGsIndexOptions clearBtreePlusIndex.
      aGsIndexOptions setLegacyIndex].
  self _btreePlusIndex == false
    ifTrue: [ 
      aGsIndexOptions setBtreePlusIndex.
      aGsIndexOptions clearLegacyIndex].
  self _optimizedComparison == true
    ifTrue: [ aGsIndexOptions clearOptimizedComparison ].
  self _optimizedComparison == false
    ifTrue: [ aGsIndexOptions setOptimizedComparison ].
  ^ aGsIndexOptions
%
category: 'Accessing'
method: GsIndexOptions
reducedConflict
  ^ self _reducedConflict ifNil: [ self defaultOptions reducedConflict ]
%
category: 'Private'
method: GsIndexOptions
reducedConflict: aBool
  "Private method ... should never be called by method external to GsIndexOptions. 
   Use set* and clear* methods instead."

  self dynamicInstVarAt: #'reducedConflict' put: aBool
%
category: 'Operators'
method: GsIndexOptions
setReducedConflict
  self reducedConflict: true
%
expectValue %Boolean
doit
GsIndexOptions category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsIndexSpec
removeallmethods GsIndexSpec
removeallclassmethods GsIndexSpec

doit
GsIndexSpec comment:
'GsIndexSpec defines the specification used to create one or more indexes on a collection.  Instance protocol for GsIndexSpec includes methods to create particular kinds of indexes with specific features.  

Once the indexes are defined in the GsIndexSpec, the indexes can be created on a particular collection.  
Printing a GsIndexSpec outputs a string that can be used to recreate the same set of indexes and features.

Instance Variables
  specs -- Array of specifications for individual indexes
  defaultOptions -- instance of GsIndexOptions'.
true
%


! ------------------- Class methods for GsIndexSpec
! ------------------- Instance methods for GsIndexSpec
category: 'Index Creation'
method: GsIndexSpec
createIndexesOn: anNsc
  self specs do: [ :indexSpec | indexSpec createIndexOn: anNsc ]
%
category: 'Private'
method: GsIndexSpec
defaultOptions
  defaultOptions ifNil: [ defaultOptions := GsIndexOptions default ].
  ^ defaultOptions
%
category: 'Specification'
method: GsIndexSpec
equalityIndex: path lastElementClass: aClass
  "Specify that  an equality index on the given path and last element class, using default 
   index options be created."

  self equalityIndex: path lastElementClass: aClass options: self defaultOptions
%
category: 'Specification'
method: GsIndexSpec
equalityIndex: path lastElementClass: aClass options: aGsIndexOptions
  "Specify that an equality index with the given path, last element class and index options
   be created.

   If the last element class is a unicode class (one of the Unicode* classes or any 
     CharacterCollection when using unicode comparison mode), then create a unicode
     equality index using the default collator.

   If the reducedConflict option is specified, create a reduced conflict equality index.

   For btree plus indexes, optimized comparisons will not be used. If optimized comparisons is
     desired, use #stringOptimizedIndex:.
"

  | specClass options |
  (RangeEqualityIndex _isUnicodeLastElementClass: aClass)
    ifTrue: [ ^ self unicodeIndex: path collator: self defaultCollator options: aGsIndexOptions ].
  options := (aClass == CharacterCollection or: [aClass isSubclassOf: CharacterCollection])
    ifTrue: [ aGsIndexOptions - GsIndexOptions optimizedComparison ]
    ifFalse: [ aGsIndexOptions ].
  specClass := options reducedConflict
    ifTrue: [ RcEqualityIndexSpecification ]
    ifFalse: [ EqualityIndexSpecification ].
  self specs
    add: ((specClass path: (self preparedPath: path) lastElementClass: aClass)
        options: options;
        yourself)
%
category: 'Specification'
method: GsIndexSpec
identityIndex: path
  "Specify that  an identiy index on the given path, using default index options be created."

  self identityIndex: path options: self defaultOptions
%
category: 'Specification'
method: GsIndexSpec
identityIndex: path options: aGsIndexOptions
  "Specify that an identity index with the given pathand index options be created."

  | options |
  options := aGsIndexOptions.
  (options legacyIndex )
    ifTrue: [ 
      options reducedConflict 
        ifTrue: [ Warning signal: 'The reducedConflict option has no effect for an identity index' ] ]
    ifFalse: [
      "new indexes need to have reducedConflict option explicitly set"
      options := options + GsIndexOptions reducedConflict].
  self specs add: ((IdentityIndexSpecification path: (self preparedPath: path))
        options: options;
        yourself)
%
category: 'Private'
method: GsIndexSpec
preparedPath: aPath
  "strip each from <aPath>"

  | pathTerms stream |
  pathTerms := aPath asArrayOf32PathTerms.
  pathTerms first = #'each'
    ifFalse: [ ^ aPath ].
  stream := String new.
  2 to: pathTerms size do: [ :index | 
    stream addAll: (pathTerms at: index).
    index < pathTerms size
      ifTrue: [ stream add: $. ] ].
  ^ stream 
%
category: 'Printing'
method: GsIndexSpec
printOn: aStream
  aStream
    nextPutAll: self class name asString , ' new';
    lf.
  1 to: self specs size do: [ :index | 
    | spec |
    spec := self specs at: index.
    aStream tab.
    spec printSelectorOn: aStream.
    aStream
      nextPut: $;;
      lf ].
  aStream
    tab;
    nextPutAll: 'yourself.'
%
category: 'Index Creation'
method: GsIndexSpec
removeIndexesFrom: anNsc
  self specs do: [ :indexSpec | indexSpec removeIndexFrom: anNsc ]
%
category: 'Private'
method: GsIndexSpec
specs
  specs ifNil: [ specs := Array new ].
  ^ specs
%
category: 'Specification'
method: GsIndexSpec
unicodeIndex: path
  "Specify that a unicode equality index on the given path using the default collator and default 
    index options be created.

   When using string comparison mode, a unicode equality index is constrained to the unicode 
    string classes Unicode7, Unicode16, and Unicode32.

   When using unicode comparison mode, a unicode equality index is constrained to be a type
     of CharacterCollection.

   For btree plus indexes, optimized comparisons will not be used. If optimized comparisons is
     desired, use either #unicodeStringOptimizedIndex: or #symbolOptimizedIndex:collator:.
"

  self
    unicodeIndex: path
    collator: self defaultCollator
    options: self defaultOptions
%
category: 'Specification'
method: GsIndexSpec
unicodeIndex: path collator: collator
  "Specify that a unicode equality index on the given path using the given collator and default 
    index options be created.

   When using string comparison mode, a unicode equality index is constrained to the unicode 
    string classes Unicode7, Unicode16, and Unicode32.

   When using unicode comparison mode, a unicode equality index is constrained to be a type
     of CharacterCollection.

   For btree plus indexes, optimized comparisons will not be used. If optimized comparisons is
     desired, use either #unicodeStringOptimizedIndex: or #symbolOptimizedIndex:collator:.
"

  self unicodeIndex: path collator: collator options: self defaultOptions
%
category: 'Specification'
method: GsIndexSpec
unicodeIndex: path collator: collator options: aGsIndexOptions
  "Specify that a unicode equality index on the given path using the given collator and given 
    index options be created.

   When using string comparison mode, a unicode equality index is constrained to the unicode 
    string classes Unicode7, Unicode16, and Unicode32.

   When using unicode comparison mode, a unicode equality index is constrained to be a type
     of CharacterCollection.

   For btree plus indexes, optimized comparisons will not be used. If optimized comparisons is
     desired, use either #unicodeStringOptimizedIndex: or #symbolOptimizedIndex:collator:.
"

  | options specClass spec |
  options := aGsIndexOptions.
  specClass := UnicodeIndexSpecification.
  options := options - GsIndexOptions optimizedComparison.
  options legacyIndex
    ifTrue: [
      options reducedConflict
        ifTrue: [ specClass := RcUnicodeIndexSpecification ] ].
 spec := (specClass path: (self preparedPath: path) collator: collator)
    options: options;
    yourself.
  self specs add: spec
%
expectValue %Boolean
doit
GsIndexSpec category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsOptions

removeallmethods GsOptions
removeallclassmethods GsOptions

doit
GsOptions comment:
'The class GsOptions is part of the Indexing and Querying subsystem.  It is an abstract superclass for GsQueryOptions and GsIndexOptions, providing behavior for merging and subtracting between instances. '.
true
%

! ------------------- Class methods for GsOptions
category: 'instance creation'
classmethod: GsOptions
new
  ^ self basicNew initialize
%
! ------------------- Instance methods for GsOptions
category: 'initialization'
method: GsOptions
initialize

%
category: 'Operators'
method: GsOptions
+ aGsQueryOptions
  ^ aGsQueryOptions combineWith: self copy
%
category: 'Operators'
method: GsOptions
- aGsQueryOptions
  ^ aGsQueryOptions negateWith: self copy
%
category: 'Operators'
method: GsOptions
combineWith: aGsQueryOptions
  self subclassResponsibility: #'combineWith:'
%
category: 'Operators'
method: GsOptions
negateWith: aGsQueryOptions
  self subclassResponsibility: #'negateWith:'
%
expectValue %Boolean
doit
GsOptions category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsPathConstantPredicate
removeallmethods GsPathConstantPredicate
removeallclassmethods GsPathConstantPredicate

doit
GsPathConstantPredicate comment: 
'The class GsPathConstantPredicate is a predicate in a GsQueryFormula. This class is a concrete subclass of GsQueryPredicate and applies an operator to a path operand and a constant operand. 

A constant operand may be a literal (instance of GsConstantReferenceAssociation) or a variable (instance of GsVariableReferenceAssociation). 

A path operand is expected to be an instance of a GsQueryPathReferenceAssociation.

An operator may be one of the following: < > = == <= >= ~= ~~.'.
true
%

! ------------------- Class methods for GsPathConstantPredicate
! ------------------- Instance methods for GsPathConstantPredicate
category: 'transforming'
method: GsPathConstantPredicate
bind: variableName to: value
  ^ self operator == #'unary'
    ifTrue: [ 
      | bound |
      bound := self copy.
      bound operand1: (bound operand1 bind: variableName to: value).
      ^ bound immediateInvariant ]
    ifFalse: [ super bind: variableName to: value ]
%
category: 'private'
method: GsPathConstantPredicate
bindEvaluators
  evaluator1 := self
    _bindEvaluator: nsc
    for: self path
    isRangeEqualityOperation: (self isRangeEqualityOperation: self operator).
  super bindEvaluators
%
category: 'optimizing'
method: GsPathConstantPredicate
canConsolidateWith: secondaryPredicate
  ^ secondaryPredicate canConsolidateWithPathConstantPredicate: self
%
category: 'optimizing'
method: GsPathConstantPredicate
canConsolidateWithPathConstantPredicate: primaryPredicate
  | op1 op2 |
  (primaryPredicate operand1 hasSamePathAs: self operand1)
    ifFalse: [ ^ false ].
  (primaryPredicate isConstantBound and: [ self isConstantBound ])
    ifFalse: [ ^ false ].
  op1 := primaryPredicate operator.
  op2 := self operator.
  ((op1 == #'>' or: [ op1 == #'>=' ]) and: [ op2 == #'<' or: [ op2 == #'<=' ] ])
    ifTrue: [ 
      " if first operator is > or >= and last operator is < or <= "
      ^ primaryPredicate constant _idxForCompareLessThanOrEqualTo: self constant ].
  ((op1 == #'<' or: [ op1 == #'<=' ]) and: [ op2 == #'>' or: [ op2 == #'>=' ] ])
    ifTrue: [ 
      " if first operator is < or <= and last operator is > or >= "
      ^ primaryPredicate constant
        _idxForCompareGreaterThanOrEqualTo: self constant ].
  ^ false
%
category: 'transforming'
method: GsPathConstantPredicate
collapseToRangePredicateForConstantPathPredicate: aConstantPathPredicate
  (self compatibleRangeOperatorsFor: aConstantPathPredicate operator)
    ifFalse: [ ^ super collapseToRangePredicateForConstantPathPredicate: aConstantPathPredicate ].
  ^ GsRangeQueryPredicate
    operand: aConstantPathPredicate operand1 copy
    operator: aConstantPathPredicate operator
    path: self path
    operator: self operator
    operand: self operand2 copy
%
category: 'transforming'
method: GsPathConstantPredicate
collapseToRangePredicateForPathConstantPredicate: aPathConstantPredicate
  (self compatibleInverseRangeOperatorsFor: aPathConstantPredicate operator)
    ifFalse: [ ^ super collapseToRangePredicateForPathConstantPredicate: aPathConstantPredicate ].
  ^ GsRangeQueryPredicate
    operand: aPathConstantPredicate operand2 copy
    operator: (self inverseOperatorFor: aPathConstantPredicate operator)
    path: self path
    operator: self operator
    operand: self operand2 copy
%
category: 'transforming'
method: GsPathConstantPredicate
collapseToRangePredicateIfPossible: aPathXPredicate
  ^ aPathXPredicate collapseToRangePredicateForPathConstantPredicate: self
%
category: 'transforming'
method: GsPathConstantPredicate
consolidateWith: secondaryPredicate
  (secondaryPredicate canConsolidateWithPathConstantPredicate: self)
    ifFalse: [ 
      self
        error:
          'Cannot consolidate ' , self printString , ' with '
            , secondaryPredicate printString ].
  (self compatibleInverseRangeOperatorsFor: secondaryPredicate operator)
    ifFalse: [ self error: 'fail safe test failed' ].
  ^ (self operator == #'>' or: [ self operator == #'>=' ])
    ifTrue: [ 
      GsRangeQueryPredicate
        operand: self operand2 copy
        operator: (self inverseOperatorFor: self operator)
        path: self path
        operator: secondaryPredicate operator
        operand: secondaryPredicate operand2 copy ]
    ifFalse: [ 
      GsRangeQueryPredicate
        operand: secondaryPredicate operand2 copy
        operator:
          (secondaryPredicate inverseOperatorFor: secondaryPredicate operator)
        path: self path
        operator: self operator
        operand: self operand2 copy ]
%
category: 'accessing'
method: GsPathConstantPredicate
constant
  ^ self operand2 _idxValue
%
category: 'querying'
method: GsPathConstantPredicate
executeAndDo: aBlock
  ^ self executeAndDo: aBlock using: self operator
%
category: 'querying-private'
method: GsPathConstantPredicate
executeAndDo: aBlock using: anOperator
  | queryEvaluator |
  queryEvaluator := self evaluator1 asQueryEvaluator.
  queryEvaluator doBlock: aBlock.
  ^ self executeClauseUsing: anOperator queryEvaluator: queryEvaluator
%
category: 'querying-private'
method: GsPathConstantPredicate
executeClause
  ^ self executeClauseUsing: self operator
%
category: 'querying-private'
method: GsPathConstantPredicate
executeClauseNegated
  ^ self executeClauseUsing: self operatorNegated
%
category: 'querying-private'
method: GsPathConstantPredicate
executeClauseUsing: anOperator
  | queryEvaluator |
  queryEvaluator := self evaluator1 asQueryEvaluator.
  ^ self executeClauseUsing: anOperator queryEvaluator: queryEvaluator
%
category: 'querying-private'
method: GsPathConstantPredicate
executeNegatedAndDo: aBlock
  ^ self executeAndDo: aBlock using: self operatorNegated
%
category: 'testing'
method: GsPathConstantPredicate
isConstantBound
  (self operator == #'unary' or: [ self operator == #'unaryNot' ])
    ifTrue: [ ^ false ].
  ^ self operand2 isBound
%
category: 'querying-private'
method: GsPathConstantPredicate
lastElementValue: anObject
  "anObject is the value obtained by traversing path terms to the last value ... analgous to the last element class"

  | theOperator theConstant |
  theOperator := self operator.
  theConstant := self constant.
  theOperator == #'unary'
    ifTrue: [ 
      theOperator := #'=='.
      theConstant := true ].
  theOperator == #'unaryNot'
    ifTrue: [ 
      theOperator := #'~~'.
      theConstant := true ].
  ^ anObject
    perform: (self comparisonSelectorFor: theOperator)
    with: theConstant
%
category: 'testing'
method: GsPathConstantPredicate
operand1IsPath
  ^ true
%
category: 'printing'
method: GsPathConstantPredicate
printOn: aStream
  self operator ~~ #'unary'
    ifTrue: [ ^ super printOn: aStream ].
  aStream nextPutAll: '( ' , self operand1 key asString , ' )'
%
category: 'accessing'
method: GsPathConstantPredicate
rawPath
  ^ self operand1
%
category: 'optimizing'
method: GsPathConstantPredicate
redundantPredicateBetween: secondaryPredicate
  ^ secondaryPredicate redundantPredicateBetweenPathConstant: self
%
category: 'optimizing'
method: GsPathConstantPredicate
redundantPredicateBetweenPathConstant: primaryPredicate
  | op1 op2 |
  (primaryPredicate operand1 hasSamePathAs: self operand1)
    ifTrue: [ 
      primaryPredicate operand1 hasEnumeratedTerm
        ifTrue: [ ^ nil ] ]
    ifFalse: [ ^ nil ].
  (primaryPredicate isConstantBound and: [ self isConstantBound ])
    ifFalse: [ ^ nil ].
  (primaryPredicate constant == nil or: [ self constant == nil ])
    ifTrue: [ ^ nil ].
  op1 := primaryPredicate operator.
  op2 := self operator.
  op1 == op2
    ifTrue: [ 
      " operators are the same "
      (op1 == #'<' or: [ op1 == #'<=' ])
        ifTrue: [ 
          " operator: < or <= "
          " choose smallest value "
          ^ (primaryPredicate constant _idxForCompareLessThan: self constant)
            ifTrue: [ self ]
            ifFalse: [ primaryPredicate ] ].
      (op1 == #'>' or: [ op1 == #'>=' ])
        ifTrue: [ 
          " operator: > or >= "
          " choose largest value "
          ^ (primaryPredicate constant _idxForCompareGreaterThan: self constant)
            ifTrue: [ self ]
            ifFalse: [ primaryPredicate ] ].
      op1 == #'='
        ifTrue: [ 
          " operator: = "
          ^ (primaryPredicate constant _idxForCompareEqualTo: self constant)
            ifTrue: [ self ]
            ifFalse: [ self _unsatisfiableQuery ] ].
      op1 == #'~='
        ifTrue: [ 
          " operator: = "
          ^ (primaryPredicate constant _idxForCompareEqualTo: self constant)
            ifTrue: [ self ]
            ifFalse: [ nil ] ].
      op1 == #'=='
        ifTrue: [ 
          " operator: == "
          ^ primaryPredicate constant == self constant
            ifTrue: [ self ]
            ifFalse: [ self _unsatisfiableQuery ] ].
      op1 == #'~~'
        ifTrue: [ 
          " operator: == "
          ^ primaryPredicate constant == self constant
            ifTrue: [ self ]
            ifFalse: [ nil ] ] ]
    ifFalse: [ 
      " operators are different "
      ((op1 == #'<' and: [ op2 == #'<=' ])
        or: [ op1 == #'<=' and: [ op2 == #'<' ] ])
        ifTrue: [ 
          " op1: < and op2: <= "
          " op1: <= and op2: < "
          " choose smallest value "
          ^ (primaryPredicate constant _idxForCompareLessThan: self constant)
            ifTrue: [ self ]
            ifFalse: [ 
              (primaryPredicate constant _idxForCompareGreaterThan: self constant)
                ifTrue: [ primaryPredicate ]
                ifFalse: [ 
                  " operands are = "
                  op1 == #'<'
                    ifTrue: [ self ]
                    ifFalse: [ primaryPredicate ] ] ] ].
      ((op1 == #'>' and: [ op2 == #'>=' ])
        or: [ op1 == #'>=' and: [ op2 == #'>' ] ])
        ifTrue: [ 
          " op1: > and op2: >= "
          " op1: >= and op2: > "
          " choose largest value "
          ^ (primaryPredicate constant _idxForCompareGreaterThan: self constant)
            ifTrue: [ self ]
            ifFalse: [ 
              (primaryPredicate constant _idxForCompareLessThan: self constant)
                ifTrue: [ primaryPredicate ]
                ifFalse: [ 
                  " operands are = "
                  op1 == #'>'
                    ifTrue: [ self ]
                    ifFalse: [ primaryPredicate ] ] ] ].
      ((op1 == #'=' and: [ op2 == #'==' ])
        or: [ op1 == #'==' and: [ op2 == #'=' ] ])
        ifTrue: [ 
          " op1: = and op2: == "
          " op1: == and op2: = "
          ^ primaryPredicate constant == self constant
            ifTrue: [ 
              op2 == #'=='
                ifTrue: [ primaryPredicate ]
                ifFalse: [ self ] ]
            ifFalse: [ self _unsatisfiableQuery ] ].
      ((op1 == #'==' and: [ op2 == #'~~' ])
        or: [ op1 == #'~~' and: [ op2 == #'==' ] ])
        ifTrue: [ 
          " op1: == and op2: ~~ "
          " op1: ~~ and op2: == "
          ^ primaryPredicate constant == self constant
            ifTrue: [ self _unsatisfiableQuery ]
            ifFalse: [ 
              op1 == #'=='
                ifTrue: [ self ]
                ifFalse: [ primaryPredicate ] ] ].
      (op1 == #'=' or: [ op1 == #'==' ])
        ifTrue: [ 
          " op1: = or op1: =="
          op2 == #'<'
            ifTrue: [ 
              " op2: < "
              ^ (primaryPredicate constant _idxForCompareLessThan: self constant)
                ifTrue: [ self ]
                ifFalse: [ self _unsatisfiableQuery ] ].
          op2 == #'<='
            ifTrue: [ 
              " op2: <= "
              ^ (primaryPredicate constant
                _idxForCompareLessThanOrEqualTo: self constant)
                ifTrue: [ self ]
                ifFalse: [ self _unsatisfiableQuery ] ].
          op2 == #'>'
            ifTrue: [ 
              " op2: > "
              ^ (primaryPredicate constant
                _idxForCompareGreaterThan: self constant)
                ifTrue: [ self ]
                ifFalse: [ self _unsatisfiableQuery ] ].
          op2 == #'>='
            ifTrue: [ 
              " op2: >= "
              ^ (primaryPredicate constant
                _idxForCompareGreaterThanOrEqualTo: self constant)
                ifTrue: [ self ]
                ifFalse: [ self _unsatisfiableQuery ] ].
          op2 == #'~='
            ifTrue: [ 
              " op2: ~= "
              ^ (primaryPredicate constant _idxForCompareEqualTo: self constant)
                ifTrue: [ self _unsatisfiableQuery ]
                ifFalse: [ self ] ] ].
      (op2 == #'=' or: [ op2 == #'==' ])
        ifTrue: [ 
          " op2: = or op2: =="
          op1 == #'<'
            ifTrue: [ 
              " op1: < "
              ^ (self constant _idxForCompareLessThan: primaryPredicate constant)
                ifTrue: [ primaryPredicate ]
                ifFalse: [ self _unsatisfiableQuery ] ].
          op1 == #'<='
            ifTrue: [ 
              " op1: <= "
              ^ (self constant
                _idxForCompareLessThanOrEqualTo: primaryPredicate constant)
                ifTrue: [ primaryPredicate ]
                ifFalse: [ self _unsatisfiableQuery ] ].
          op1 == #'>'
            ifTrue: [ 
              " op1: > "
              ^ (self constant
                _idxForCompareGreaterThan: primaryPredicate constant)
                ifTrue: [ primaryPredicate ]
                ifFalse: [ self _unsatisfiableQuery ] ].
          op1 == #'>='
            ifTrue: [ 
              " op1: >= "
              ^ (self constant
                _idxForCompareGreaterThanOrEqualTo: primaryPredicate constant)
                ifTrue: [ primaryPredicate ]
                ifFalse: [ self _unsatisfiableQuery ] ].
          op1 == #'~='
            ifTrue: [ 
              " op1: ~= "
              ^ (self constant _idxForCompareEqualTo: primaryPredicate constant)
                ifTrue: [ self _unsatisfiableQuery ]
                ifFalse: [ primaryPredicate ] ] ] ].
  ^ nil
%
category: 'initialization'
method: GsPathConstantPredicate
_fromArray: varsArray terms: termsArray links: linksArray paths: pathsArray startingAt: i
  self
    path1:
      (self _pathFrom: linksArray paths: pathsArray offset: (termsArray at: i + 6)).
  operator := self operationSelectors at: (termsArray at: i) + 1.
  operator == #'unary'
    ifTrue: [ 
      "recast predicate to be '== true'"
      operator := #'=='.
      self constant2: true ]
    ifFalse: [ self constant2: (varsArray at: (termsArray at: i + 2)) ]
%
category: 'private'
method: GsPathConstantPredicate
_unsatisfiableQuery
  GsUnsatisfiableQueryNotification signal.
  ^ nil
%
expectValue %Boolean
doit
GsPathConstantPredicate category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsPathPathPredicate
removeallmethods GsPathPathPredicate
removeallclassmethods GsPathPathPredicate

doit
GsPathPathPredicate comment: 
'The class GsPathPathPredicate is a predicate in a GsQueryFormula. This class is a concrete subclass of GsQueryPredicate and applies an operator to two path operands. 

A path operand is expected to be an instance of a GsQueryPathReferenceAssociation.

An operator may be one of the following: < > = == <= >= ~= ~~.'.
true
%

! ------------------- Class methods for GsPathPathPredicate
! ------------------- Instance methods for GsPathPathPredicate
category: 'private'
method: GsPathPathPredicate
bindEvaluators
  "unconditionally use pathEvaluatorClass"

  | pathEvaluator |
  pathEvaluator := self pathEvaluatorFor: self path1.
  evaluator1 := pathEvaluator nsc: nsc.
  pathEvaluator := self pathEvaluatorFor: self path2.
  evaluator2 := pathEvaluator nsc: nsc.
  super bindEvaluators
%
category: 'querying-private'
method: GsPathPathPredicate
elementValue: anObject
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  | evaluatorA evaluatorB tmp1 tmp2 |
  evaluatorA := evaluator1 ifNil: [ self pathEvaluatorFor: self path1 ].
  evaluatorB := evaluator2 ifNil: [ self pathEvaluatorFor: self path2 ].
  tmp1 := Array new.
  tmp2 := Array new.
  evaluatorA traverse: anObject startingAt: 1 into: tmp1.
  evaluatorB traverse: anObject startingAt: 1 into: tmp2.
  tmp1
    do: [ :obj1 | 
      tmp2
        do: [ :obj2 | 
          (obj1 perform: (self comparisonSelectorFor: self operator) with: obj2)
            ifTrue: [ ^ true ] ] ].
  ^ false
%
category: 'accessing'
method: GsPathPathPredicate
evaluator
  ^ evaluator1
%
category: 'accessing'
method: GsPathPathPredicate
evaluator2
  ^ evaluator2
%
category: 'accessing'
method: GsPathPathPredicate
evaluator2: anEvaluator
  evaluator2 := anEvaluator
%
category: 'querying-private'
method: GsPathPathPredicate
executeAndDo: aBlock using: anOperator
  (self bagEnumeratorFor: anOperator)
    doBlock: aBlock;
    findAllValues
%
category: 'querying-private'
method: GsPathPathPredicate
executeClause
  ^ self executeClauseUsing: self operator
%
category: 'querying-private'
method: GsPathPathPredicate
executeClauseNegated
  ^ self executeClauseUsing: self operatorNegated
%
category: 'querying-private'
method: GsPathPathPredicate
executeClauseUsing: anOperator
  ^ (self bagEnumeratorFor: anOperator) findAllValues
%
category: 'querying-private'
method: GsPathPathPredicate
executeNegatedAndDo: aBlock
  ^ self executeAndDo: aBlock using: self operatorNegated
%
category: 'testing'
method: GsPathPathPredicate
isPathPath
  ^ true
%
category: 'testing'
method: GsPathPathPredicate
operand1IsPath
  ^ true
%
category: 'testing'
method: GsPathPathPredicate
operand2IsPath
  ^ true
%
category: 'accessing'
method: GsPathPathPredicate
path
  "Ambiguous - path1 and path2 both exist"

  ^ self shouldNotImplement: #'path'
%
category: 'accessing'
method: GsPathPathPredicate
path1
  ^ self operand1 key
%
category: 'accessing'
method: GsPathPathPredicate
path2
  ^ self operand2 key
%
category: 'transforming'
method: GsPathPathPredicate
transformCommonPaths
  "If the predicate is a path-path and path paths are the same:
	- if operator is =, ==, <=, >= predicate will always be true.
	  Transform  to unary true query
	- otherwise the predicate will always be false.
	  Transform  to unary false query"

  (self operand1 hasSamePathAs: self operand2)
    ifFalse: [ ^ self ].
  (#(#'=' #'==' #'<=' #'>=') includes: self operator)
    ifTrue: [ ^ GsQueryPredicate constant: true ].
  ^ GsQueryPredicate constant: false
%
category: 'private'
method: GsPathPathPredicate
unbindEvaluators
  "remove all bindings"

  super unbindEvaluators.
  evaluator2 := nil
%
category: 'testing'
method: GsPathPathPredicate
usesPathEvaluator
  ^ self evaluator1 isPathEvaluator and: [ self evaluator2 isPathEvaluator ]
%
category: 'initialization'
method: GsPathPathPredicate
_fromArray: varsArray terms: termsArray links: linksArray paths: pathsArray startingAt: i
  self
    path2:
      (self _pathFrom: linksArray paths: pathsArray offset: (termsArray at: i + 6)).
  operator := self
    inverseOperatorFor: (self operationSelectors at: (termsArray at: i) + 1).
  self
    path1:
      (self _pathFrom: linksArray paths: pathsArray offset: (termsArray at: i + 3))
%
expectValue %Boolean
doit
GsPathPathPredicate category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsPathReferenceAssociation
removeallmethods GsPathReferenceAssociation
removeallclassmethods GsPathReferenceAssociation

doit
GsPathReferenceAssociation comment: 
'GsPathReferenceAssociation  is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsPathReferenceAssociation
! ------------------- Instance methods for GsPathReferenceAssociation
expectValue %Boolean
doit
GsPathReferenceAssociation category: 'Index-Query'.
true
%

! Remove existing behavior from GsQuery
removeallmethods GsQuery
removeallclassmethods GsQuery

doit
GsQuery comment: 
'GsQuery is an abstraction of a query and conceptually of the query results.  

An instance of GsQuery is created based on:
  -- a string following query syntax,
  -- from a select block as used in the legacy indexing system, or
  --  from a formula output from another queery.  
This GsQuery must be bound to a collection, and any variables resolved, before it can be used as a query.  

Once it is fully bound, it can be treated as a collection of the results.  
However, note that operations will execute the query.'.
true
%

! ------------------- Class methods for GsQuery
category: 'defaults'
classmethod: GsQuery
defaultOptimizerClass
  DefaultOptimizerClass ifNil: [ DefaultOptimizerClass := Gs32Optimizer ].
  ^ DefaultOptimizerClass
%
category: 'defaults'
classmethod: GsQuery
defaultOptimizerClass: anOptimizerClass
  DefaultOptimizerClass := anOptimizerClass
%
category: 'defaults'
classmethod: GsQuery
defaultQueryOptions
  DefaultQueryOptions ifNil: [ DefaultQueryOptions := GsQueryOptions default ].
  ^ DefaultQueryOptions
%
category: 'defaults'
classmethod: GsQuery
defaultQueryOptions: aGsQueryOptions
  DefaultQueryOptions := aGsQueryOptions
%
category: 'initialization'
classmethod: GsQuery
initialize
  DefaultOptimizerClass := DefaultQueryOptions := nil.
  self
    defaultOptimizerClass;
    defaultQueryOptions
%
category: 'instance creation'
classmethod: GsQuery
fromFormula: aGsQueryFormula
  ^ self fromFormula: aGsQueryFormula on: nil options: self defaultQueryOptions
%
category: 'instance creation'
classmethod: GsQuery
fromFormula: aGsQueryFormula on: anNsc
  ^ self
    fromFormula: aGsQueryFormula
    on: anNsc
    options: self defaultQueryOptions
%
category: 'instance creation'
classmethod: GsQuery
fromFormula: aGsQueryFormula on: anNsc options: aGsQueryOptions
  ^ self new
    formula: aGsQueryFormula immediateInvariant;
    on: anNsc;
    queryOptions: aGsQueryOptions;
    yourself
%
category: 'instance creation'
classmethod: GsQuery
fromFormula: aGsQueryFormula options: aGsQueryOptions
  ^ self fromFormula: aGsQueryFormula on: nil options: aGsQueryOptions
%
category: 'instance creation'
classmethod: GsQuery
fromSelectBlock: aSelectBlock
  | formula |
  formula := GsQueryPredicate fromSelectBlock: aSelectBlock.
  ^ self fromFormula: formula immediateInvariant
%
category: 'instance creation'
classmethod: GsQuery
fromSelectBlock: aSelectBlock on: anNsc
  "formula returned by 
	GsQueryPredicate class>>fromSelectBlock:usingQueryExecuterWith: is already 
	bound to nsc, so we need to directly set nsc"

  | formula |
  formula := GsQueryPredicate
    fromSelectBlock: aSelectBlock
    usingQueryExecuterWith: anNsc.
  ^ self new
    formula: formula immediateInvariant;
    _nsc: anNsc;
    yourself
%
category: 'instance creation'
classmethod: GsQuery
fromString: aString
  ^ self fromFormula: (GsQueryPredicate fromQueryStatement: aString)
%
category: 'instance creation'
classmethod: GsQuery
fromString: aString on: anNsc
  ^ self fromFormula: (GsQueryPredicate fromQueryStatement: aString) on: anNsc
%
category: 'instance creation'
classmethod: GsQuery
fromString: aString on: anNsc options: aGsQueryOptions
  ^ self
    fromFormula: (GsQueryPredicate fromQueryStatement: aString)
    on: anNsc
    options: aGsQueryOptions
%
category: 'instance creation'
classmethod: GsQuery
fromString: aString options: aGsQueryOptions
  ^ self
    fromFormula: (GsQueryPredicate fromQueryStatement: aString)
    options: aGsQueryOptions
%
category: 'instance creation'
classmethod: GsQuery
initialize
  self 
    defaultQueryOptions;
    defaultOptimizerClass
%
! ------------------- Instance methods for GsQuery
category: 'collection'
method: GsQuery
any
  ^ self anyIfNone: [ ^ self error: 'this collection is empty' ]
%
category: 'collection'
method: GsQuery
anyIfNone: noneBlock
  ^ self detect: [ :each | ^ each ] ifNone: noneBlock
%
category: 'converting'
method: GsQuery
asIdentityBag
  ^ self queryResult asIdentityBag
%
category: 'querying'
method: GsQuery
bind: variableName to: value
  "bind the given <value> to the <variableName> reference in the query predicate"

  self _formula: (self _formula bind: variableName to: value)
%
category: 'private'
method: GsQuery
cacheQueryResult
  ^ self queryOptions cacheQueryResult
%
category: 'collection'
method: GsQuery
collect: collectBlock
  | resultSet cachedResult |
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self cacheQueryResult
    ifTrue: [ 
      queryResult ifNotNil: [ ^ queryResult collect: collectBlock ].
      cachedResult := nsc speciesForCollect new ].
  resultSet := nsc speciesForCollect new.
  self formula _checkForBug43764: self.
  self formula
    executeAndDo: [ :obj | 
      | res |
      res := collectBlock value: obj.
      resultSet add: res.
      self cacheQueryResult
        ifTrue: [ cachedResult add: obj ] ].
  queryResult := cachedResult.
  ^ resultSet
%
category: 'collection'
method: GsQuery
detect: detectBlock
  ^ self
    detect: detectBlock
    ifNone: [ ^ self _error: #'assocErrNoElementsDetected' args: {detectBlock} ]
%
category: 'collection'
method: GsQuery
detect: detectBlock ifNone: noneBlock
  | cachedResult |
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self cacheQueryResult
    ifTrue: [ 
      queryResult ifNotNil: [ ^ queryResult detect: detectBlock ].
      cachedResult := nsc species new ].
  self formula _checkForBug43764: self.
  self formula
    executeAndDo: [ :obj | 
      (detectBlock value: obj)
        ifTrue: [ ^ obj ].
      self cacheQueryResult
        ifTrue: [ cachedResult add: obj ] ].
  queryResult := cachedResult.
  ^ noneBlock value
%
category: 'collection'
method: GsQuery
do: doBlock
  | cachedResult |
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self cacheQueryResult
    ifTrue: [ 
      queryResult ifNotNil: [ ^ queryResult do: doBlock ].
      cachedResult := nsc species new ].
  self formula _checkForBug43764: self.
  self cacheQueryResult
    ifTrue: [ 
      self formula
        executeAndDo: [ :obj | 
          doBlock value: obj.
          cachedResult add: obj ] ]
    ifFalse: [ self formula executeAndDo: [ :obj | doBlock value: obj ] ].
  queryResult := cachedResult.
  ^ self
%
category: 'querying'
method: GsQuery
elementValue: each
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  self formula _checkForBug43764: self.
  ^ self formula elementValue: each
%
category: 'accessing'
method: GsQuery
formula
  ^ self queryOptions autoOptimize
    ifTrue: [ self optimizedFormula ]
    ifFalse: [ formula ]
%
category: 'private'
method: GsQuery
formula: aQueryFormula
  | theFormula |
  originalFormula := aQueryFormula unbind.
  theFormula := self _nsc
    ifNil: [ aQueryFormula ]
    ifNotNil: [ aQueryFormula bindEvaluatorsFor: self _nsc collator: self _collator ].
  self _formula: theFormula
%
category: 'private'
method: GsQuery
immediateInvariant
  super immediateInvariant.
  formula immediateInvariant
%
category: 'collection'
method: GsQuery
includes: anObject
  ^ self queryResult includes: anObject
%

category: 'collection'
method: GsQuery
isEmpty
  ^ self queryResult isEmpty
%
category: 'querying'
method: GsQuery
on: anNsc
  "Bind the query to the given <anNsc>"

  self on: anNsc collator: nil
%
category: 'querying'
method: GsQuery
optimize
  self _formula: self optimizedFormula
%
category: 'accessing'
method: GsQuery
optimizedFormula
  ^ self optimizeFormulaUsing: self class defaultOptimizerClass
%
category: 'private'
method: GsQuery
optimizeFormulaUsing: queryOptimizerClass
  formula ifNil: [ ^ nil ].
  ^ queryOptimizerClass
    optimize: formula
    on: self _nsc
    options: self queryOptions
%
category: 'accessing'
method: GsQuery
originalFormula
  ^ originalFormula
%
category: 'printing'
method: GsQuery
printOn: aStream
  self formula printOn: aStream
%
category: 'accessing'
method: GsQuery
queryOptions
  queryOptions ifNil: [ queryOptions := self class defaultQueryOptions ].
  ^ queryOptions
%
category: 'private'
method: GsQuery
queryOptions: aGsQueryOptions
  queryOptions := aGsQueryOptions.
  queryOptions optionalPathTerms
    ifTrue: [ formula := formula optionalPathTerms ]
%
category: 'accessing'
method: GsQuery
queryResult
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  ^ self cacheQueryResult
    ifTrue: [ 
      queryResult
        ifNil: [ 
          self formula _checkForBug43764: self.
          queryResult := self formula execute ].
      queryResult ]
    ifFalse: [ 
      self formula _checkForBug43764: self.
      self formula execute ]
%
category: 'collection'
method: GsQuery
readStream
  "Return a BtreeReadStream or RangeIndexReadStream on the query result"

  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self formula _checkForBug43764: self.
  self formula _checkForStreamableQuery: self.
  ^ self formula readStream
%
category: 'collection'
method: GsQuery
reject: rejectBlock
  | resultSet cachedResult |
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self cacheQueryResult
    ifTrue: [ 
      queryResult ifNotNil: [ ^ queryResult reject: rejectBlock ].
      cachedResult := nsc species new ].
  resultSet := nsc species new.
  self formula _checkForBug43764: self.
  self formula
    executeAndDo: [ :obj | 
      (rejectBlock value: obj)
        ifFalse: [ resultSet add: obj ].
      self cacheQueryResult
        ifTrue: [ cachedResult add: obj ] ].
  queryResult := cachedResult.
  ^ resultSet
%
category: 'querying'
method: GsQuery
resetCache
  queryResult := nil
%
category: 'collection'
method: GsQuery
reversedReadStream
  "Return a BtreeReadStream or RangeIndexReadStream on the query result"

  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self formula _checkForBug43764: self.
  self formula _checkForStreamableQuery: self.
  ^ self formula reversedReadStream
%
category: 'collection'
method: GsQuery
select: selectBlock
  | resultSet cachedResult |
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self cacheQueryResult
    ifTrue: [ 
      queryResult ifNotNil: [ ^ queryResult select: selectBlock ].
      cachedResult := nsc species new ].
  resultSet := nsc species new.
  self formula _checkForBug43764: self.
  self formula
    executeAndDo: [ :obj | 
      (selectBlock value: obj)
        ifTrue: [ resultSet add: obj ].
      self cacheQueryResult
        ifTrue: [ cachedResult add: obj ] ].
  queryResult := cachedResult.
  ^ resultSet
%
category: 'collection'
method: GsQuery
size
  ^ self queryResult size
%
category: 'converting'
method: GsQuery
withOptimizedFormula
  ^ self class new
    formula: self optimizedFormula immediateInvariant;
    on: self _nsc;
    queryOptions: self queryOptions copy;
    yourself
%
category: 'converting'
method: GsQuery
withOriginalFormula
  ^ self class new
    formula: self originalFormula immediateInvariant;
    on: self _nsc;
    queryOptions: self queryOptions copy;
    yourself
%
category: 'private'
method: GsQuery
_formula
  ^ formula
%
category: 'private'
method: GsQuery
_formula: aQueryFormula
  self resetCache.
  formula := aQueryFormula.
  self queryOptions optionalPathTerms
    ifTrue: [ formula := aQueryFormula optionalPathTerms ]
%
category: 'private'
method: GsQuery
_nsc
  ^ nsc
%
category: 'private'
method: GsQuery
_nsc: anNscOrSequenceableCollection
  self resetCache.
  nsc := anNscOrSequenceableCollection
%
category: 'private'
method: GsQuery
_queryResult
  ^ queryResult
%
expectValue %Boolean
doit
GsQuery category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryFormula
removeallmethods GsQueryFormula
removeallclassmethods GsQueryFormula

doit
GsQueryFormula comment: 
'The abstract class GsQueryFormula represents a formula in a GsQuery. A formula is composed of one or more predicates (instance of GsAbstractQueryPredicate).'.
true
%

! ------------------- Class methods for GsQueryFormula

category: 'initialization'
classmethod: GsQueryFormula
initialize
  self _addClassVar: #ComparisonSelectorMap
     value: (
      SymbolDictionary new
        at: #'<' put: #'_idxForCompareLessThan:';
        at: #'>' put: #'_idxForCompareGreaterThan:';
        at: #'=' put: #'_idxForCompareEqualTo:';
        at: #'==' put: #'==';
        at: #'<=' put: #'_idxForCompareLessThanOrEqualTo:';
        at: #'>=' put: #'_idxForCompareGreaterThanOrEqualTo:';
        at: #'~=' put: #'_idxForCompareNotEqualTo:';
        at: #'~~' put: #'~~';
        yourself ).
  self _addClassVar: #InverseOperatorMap 
      value: (SymbolDictionary new
        at: #'<' put: #'>';
        at: #'>' put: #'<';
        at: #'=' put: #'=';
        at: #'==' put: #'==';
        at: #'<=' put: #'>=';
        at: #'>=' put: #'<=';
        at: #'~=' put: #'~=';
        at: #'~~' put: #'~~';
        yourself ).
   self _addClassVar: #NegatedOperatorMap 
      value: ( SymbolDictionary new
        at: #'<' put: #'>=';
        at: #'>' put: #'<=';
        at: #'=' put: #'~=';
        at: #'==' put: #'~~';
        at: #'<=' put: #'>';
        at: #'>=' put: #'<';
        at: #'~=' put: #'=';
        at: #'~~' put: #'==';
        at: #'unary' put: #'unaryNot';
        at: #'unaryNot' put: #'unary';
        yourself ).
%

classmethod: GsQueryFormula
comparisonSelectorFor: operator
  "Returns the #_idxForCompare"

  ^ ComparisonSelectorMap at: operator
%

category: 'accessing'
classmethod: GsQueryFormula
inverseOperatorFor: operator
  "Returns the operator that is the inverse of the given operator.
 This is used to reverse the comparison order of operands in a predicate.  
 For example, the inverse of <= is >=; the inverse of = is = (no change)."

  ^ InverseOperatorMap at: operator
%
category: 'accessing'
classmethod: GsQueryFormula
negatedOperatorFor: operator
  "Returns the operator that is the negation of the given operator.
 This is used to implement executeClauseNegated.  
 For example, the negation of <= is >; the inverse of = is ~= ."

  ^ NegatedOperatorMap at: operator
%
! ------------------- Instance methods for GsQueryFormula
category: 'operators'
method: GsQueryFormula
& aClause
  ^ GsCompoundClause clause: self operator: #'&' clause: aClause
%
category: 'visitors'
method: GsQueryFormula
acceptVisitor: aFormulaVisitor
  aFormulaVisitor acceptFormula: self
%
category: 'optimizing'
method: GsQueryFormula
applyDeMorgansNegationTransform
  "actively apply De Morgan's laws"

  self subclassResponsibility: #'applyDeMorgansNegationTransform'
%
category: 'optimizing'
method: GsQueryFormula
applyDeMorgansTransform
  "do not transform, but propagate the transform"

  self subclassResponsibility: #'applyDeMorgansTransform'
%
category: 'converting'
method: GsQueryFormula
asFormula
  ^ self
%
category: 'converting'
method: GsQueryFormula
asFormulaWithSelectorParts: selectorParts
  ^ GsUnaryClause clause: self operator: selectorParts first inputValue asSymbol
%
category: 'converting'
method: GsQueryFormula
asFormulaWithSelectorParts: selectorParts arguments: arguments
  ^ self subclassResponsibility: #asFormulaWithSelectorParts:arguments:
%
category: 'converting'
method: GsQueryFormula
asQuery
  ^ GsQuery fromFormula: self
%
category: 'transforming'
method: GsQueryFormula
bind: variableName to: value
  self subclassResponsibility: #'bind:to:'
%
category: 'private'
method: GsQueryFormula
bindEvaluators
  self immediateInvariant
%
category: 'transforming'
method: GsQueryFormula
bindEvaluatorsFor: anNsc collator: anIcuCollatorOrNil
  | bound |
  bound := self copy.
  anNsc ifNil: [ ^ bound ].
  bound _nsc: anNsc collator: anIcuCollatorOrNil.
  bound bindEvaluators.
  ^ bound
%
category: 'optimizing'
method: GsQueryFormula
canConsolidateWith: secondaryPredicate
  ^ false
%
category: 'optimizing'
method: GsQueryFormula
canConsolidateWithPathConstantPredicate: primaryPredicate
  ^ false
%
category: 'transforming'
method: GsQueryFormula
collapseToRangePredicateForConstantPathPredicate: aPredicate
  "default is to and the two predicates together, last half of double dispatch so reverse order"

  ^ aPredicate & self
%
category: 'transforming'
method: GsQueryFormula
collapseToRangePredicateForPathConstantPredicate: aPredicate
  "default is to and the two predicates together, last half of double dispatch so reverse order"

  ^ aPredicate & self
%
category: 'transforming'
method: GsQueryFormula
collapseToRangePredicateIfPossible: aPathXPredicate
  "default is to and the two predicates together"

  ^ self & aPathXPredicate
%
category: 'accessing'
method: GsQueryFormula
comparisonSelectorFor: operator
  ^ self class comparisonSelectorFor: operator
%
category: 'accessing'
method: GsQueryFormula
constantReferenceFor: aName
  ^ GsConstantReferenceAssociation newWithKey: aName value: nil
%
category: 'querying-private'
method: GsQueryFormula
elementValue: anObject
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  self subclassResponsibility: #'elementValue:'
%
category: 'querying'
method: GsQueryFormula
execute
  ^ self executeClause
%
category: 'querying'
method: GsQueryFormula
executeAndDo: aBlock
  self subclassResponsibility: #'executeAndDo:'
%
category: 'querying-private'
method: GsQueryFormula
executeClause
  self subclassResponsibility: #'executeClause'
%
category: 'querying-private'
method: GsQueryFormula
executeClauseNegated
  "used when not encountered"

  self subclassResponsibility: #'executeClauseNegated'
%
category: 'querying-private'
method: GsQueryFormula
executeClauseNegatedOn: anNsc
  | bound |
  bound := self bindEvaluatorsFor: anNsc collator: self collator.
  ^ bound executeClauseNegated
%
category: 'querying-private'
method: GsQueryFormula
executeClauseOn: anNsc
  | bound |
  bound := self bindEvaluatorsFor: anNsc collator: self collator.
  ^ bound executeClause
%
category: 'querying-private'
method: GsQueryFormula
executeNegatedAndDo: aBlock
  self subclassResponsibility: #'executeNegatedAndDo:'
%
category: 'accessing'
method: GsQueryFormula
inverseOperatorFor: operator
  ^ self class inverseOperatorFor: operator
%
category: 'testing'
method: GsQueryFormula
isCompoundPredicate
  ^ false
%
category: 'testing'
method: GsQueryFormula
isConjunctiveNormalForm
  ^ true
%
category: 'testing'
method: GsQueryFormula
isConstantConstant
  ^ false
%
category: 'testing'
method: GsQueryFormula
isNegationClause
  ^ false
%
category: 'testing'
method: GsQueryFormula
isPathPath
  ^ false
%
category: 'testing'
method: GsQueryFormula
isPredicate
  ^ false
%
category: 'testing'
method: GsQueryFormula
isRangeEqualityOperation: queryOp
  "Returns true if the given search operation is one of < > = <= >= or ~=."

  ^ queryOp ~~ #'==' and: [ queryOp ~~ #'~~' ]
%
category: 'accessing'
method: GsQueryFormula
negatedOperatorFor: operator
  ^ self class negatedOperatorFor: operator
%
category: 'operators'
method: GsQueryFormula
normalize
  ^ self copy
%
category: 'operators'
method: GsQueryFormula
not
  ^ GsUnaryClause clause: self operator: #'not'
%
category: 'accessing'
method: GsQueryFormula
operationSelectors
  "Returns an Array of comparison operation selectors whose ordering matches the order used by the QueryExecutor."

  ^ #(#'<' #'>' #'=' #'==' #'<=' #'>=' #'~=' #'~~' #'unary' #'unaryNot')
%
category: 'accessing'
method: GsQueryFormula
pathEvaluatorClass
  pathTermsRequired ifNil: [ ^ PathEvaluator ].
  ^ pathTermsRequired
    ifTrue: [ PathEvaluator ]
    ifFalse: [ OptionalTermPathEvaluator ]
%
category: 'accessing'
method: GsQueryFormula
pathEvaluatorFor: path
  | pathEvaluator terms holder |
  pathEvaluator := self pathEvaluatorClass basicNew.
  terms := self _parsePath: path.
  terms do: [ :term | pathEvaluator add: term ].
  pathEvaluator collator: self collator.
  pathEvaluator initialize .
  holder := { pathEvaluator }.
  pathEvaluator := nil .
  ^ PathEvaluator asMostSpecificType: holder .
%
category: 'accessing'
method: GsQueryFormula
pathReferenceFor: aName
  ^ GsQueryPathReferenceAssociation newWithKey: aName value: nil
%
category: 'querying'
method: GsQueryFormula
query: anNsc
  ^ GsQuery new
    on: anNsc;
    formula: self;
    yourself
%
category: 'querying'
method: GsQueryFormula
readStream
  self subclassResponsibility: #'readStream'
%
category: 'optimizing'
method: GsQueryFormula
redundantPredicateBetween: secondaryPredicate
  ^ nil
%
category: 'optimizing'
method: GsQueryFormula
redundantPredicateBetweenPathConstant: primaryPredicate
  ^ nil
%
category: 'optimizing'
method: GsQueryFormula
transformCommonPaths
  "If the predicate is a path-path and path paths are the same:
	- if operator is =, ==, <=, >= predicate will always be true.
	  Transform  to unary true query"

  ^ self
%
category: 'transforming'
method: GsQueryFormula
unbind
  "remove all bindings"

  | unbound |
  unbound := self copy.
  unbound _nsc: nil collator: nil.
  ^ unbound
%
category: 'testing'
method: GsQueryFormula
usesEqualityOperation
  ^ false
%
category: 'testing'
method: GsQueryFormula
usesIdentityOperation
  ^ false
%
category: 'testing'
method: GsQueryFormula
usesPathEvaluator
  ^ false
%
category: 'accessing'
method: GsQueryFormula
variableReferenceFor: aName
  ^ GsVariableReferenceAssociation newWithKey: aName value: nil
%
category: 'private'
method: GsQueryFormula
_bindEvaluator: anNsc for: path isRangeEqualityOperation: isRangeOperation
  | pathEvaluator iList theIndexObj legacyIndexesOnNsc |
  pathEvaluator := self pathEvaluatorFor: path.
  iList := anNsc _indexedPaths.
  iList
    ifNil: [ 
      "no indexes defined on nsc ... do it the hard way"
      ^ pathEvaluator nsc: anNsc ].
  legacyIndexesOnNsc := false.
  1 to: iList size by: 2 do: [ :i | 
    | indexObj |
    indexObj := iList at: i.
    legacyIndexesOnNsc := indexObj isLegacyIndex. "all indexes on an nsc are either legacy or not"
    (iList at: i + 1) == 1
      ifTrue: [ 
        " see if the index matches the path "
        (indexObj hasIndexOnPath: pathEvaluator)
          ifTrue: [ 
            " if range index found for range operation or
          identity index found for identity operation "
            ((isRangeOperation and: [ indexObj isRangeEqualityIndex ])
              or: [ isRangeOperation not and: [ indexObj isIdentityIndex ] ])
              ifTrue: [ 
                (self _canIndexSatifyPathTermsRequired: indexObj)
                  ifTrue: [ ^ indexObj ] ]
              ifFalse: [ theIndexObj := indexObj ] ] ] ].
  isRangeOperation
    ifTrue: [ 
      "no range index found ... do it the hard way"
      ^ pathEvaluator nsc: anNsc ]
    ifFalse: [ 
      theIndexObj ~~ nil
        ifTrue: [ 
          (self _canIndexSatifyPathTermsRequired: theIndexObj)
            ifTrue: [ 
              "index found for non-range query, go ahead and use it"
              ^ theIndexObj ] ] ].
  theIndexObj := IdentityIndex new.
  theIndexObj
    indexDictionary: nsc _indexDictionary;
    nscRoot: anNsc.
  iList _putAllCommonPathTermsForPathArray: pathEvaluator into: theIndexObj.
  theIndexObj size == pathEvaluator size
    ifTrue: [ 
      (self _canIndexSatifyPathTermsRequired: theIndexObj)
        ifTrue: [ 
          legacyIndexesOnNsc
            ifTrue: [ ^ theIndexObj ] 
          ifFalse: [ 
            "eventually this should be conditional on a GsIndexOption ... for now it's naked"
            GsQueryExpectedImplicitIdentityIndexError signal: self ] ] ].
  ^ pathEvaluator nsc: anNsc	"no qualifying identity indexes found"
%
category: 'private'
method: GsQueryFormula
_parsePath: path
  "strip each off terms"

  | terms |
  terms := path asArrayOf32PathTerms.
  terms := terms size = 1
    ifTrue: [ #(#'') ]
    ifFalse: [ terms copyFrom: 2 to: terms size ].
  ^ terms
%
category: 'operators'
method: GsQueryFormula
| aClause
  ^ GsCompoundClause clause: self operator: #'|' clause: aClause
%
category: 'private'
method: GsQueryFormula
buildGsQueryParserNodeFor: aGsQueryParser
  aGsQueryParser build: self messages: nil
%
expectValue %Boolean
doit
GsQueryFormula category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryFormulaVisitor
removeallmethods GsQueryFormulaVisitor
removeallclassmethods GsQueryFormulaVisitor

doit
GsQueryFormulaVisitor comment:
'GsQueryFormulaVisitor is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsQueryFormulaVisitor
! ------------------- Instance methods for GsQueryFormulaVisitor
category: 'visiting'
method: GsQueryFormulaVisitor
acceptCompoundClause: aCompoundClause
  aCompoundClause clause1 acceptVisitor: self.
  aCompoundClause clause2 acceptVisitor: self
%
category: 'visiting'
method: GsQueryFormulaVisitor
acceptFormula: aFormula
%
category: 'visiting'
method: GsQueryFormulaVisitor
acceptPredicate: aPredicate
%
category: 'visiting'
method: GsQueryFormulaVisitor
acceptRangePredicate: aRangePredicate
%
category: 'visiting'
method: GsQueryFormulaVisitor
acceptUnaryClause: aUnaryClause
  aUnaryClause clause acceptVisitor: self
%
category: 'visiting'
method: GsQueryFormulaVisitor
visitFormula: aFormula
  aFormula acceptVisitor: self.
  ^ aFormula
%
expectValue %Boolean
doit
GsQueryFormulaVisitor category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryGrammar
removeallmethods GsQueryGrammar
removeallclassmethods GsQueryGrammar

doit
GsQueryGrammar comment:
'GsQueryGrammar is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsQueryGrammar
category: 'instance creation'
classmethod: GsQueryGrammar
parseFormula: aString
  ^ self new parseFormula: aString
%
category: 'instance creation'
classmethod: GsQueryGrammar
parseFormula: aString onError: aBlock
  ^ self new parseFormula: aString onError: aBlock
%
! ------------------- Instance methods for GsQueryGrammar
category: 'grammar-messages'
method: GsQueryGrammar
andBlock
  ^ [ :predicate1 :operator :predicate2 | 
  {predicate1.
  operator.
  predicate2} ]
%
category: 'grammar-literals'
method: GsQueryGrammar
arrayItem
  ^ literal / symbolLiteralArray / arrayLiteralArray / byteLiteralArray
%
category: 'grammar-literals'
method: GsQueryGrammar
arrayLiteral
  ^ '#(' asParser queryToken , arrayItem star , $) asParser queryToken
%
category: 'grammar-literals'
method: GsQueryGrammar
arrayLiteralArray
  ^ $( asParser queryToken , arrayItem star , $) asParser queryToken
%
category: 'primitives'
method: GsQueryGrammar
binary
	^ (PPPredicateObjectParser anyOf: '!%&*+,-/<=>?@\|~') plus
%
category: 'grammar-literals'
method: GsQueryGrammar
byteLiteral
  ^ '#[' asParser queryToken , numberLiteral star , $] asParser queryToken
%
category: 'grammar-literals'
method: GsQueryGrammar
byteLiteralArray
  ^ $[ asParser queryToken , numberLiteral star , $] asParser queryToken
%
category: 'primitives'
method: GsQueryGrammar
char
	^ $$ asParser , #any asParser
%
category: 'grammar-literals'
method: GsQueryGrammar
charLiteral
	^ charToken
%
category: 'token'
method: GsQueryGrammar
charToken
  ^ char queryToken
%
category: 'grammar-messages'
method: GsQueryGrammar
expression
  | exp |
  exp := PPExpressionParser new.
  exp term: parenExpression / parenPredicate.
  exp
    group: [ :g | 
          g
            postfix:
              (notToken
                handleGsQueryErrorMessage: '$|, $&, or not expected'
                nestMessages: false)
            do: self notBlock ];
    group: [ :g | 
          g left: $| asParser token trim do: self orBlock.
          g left: $& asParser token trim do: self andBlock ].
  ^ predicateExpression / exp
%
category: 'grammar-literals'
method: GsQueryGrammar
falseLiteral
	^ falseToken
%
category: 'token'
method: GsQueryGrammar
falseToken
  ^ ('false' asParser , #'word' asParser not) queryToken
%
category: 'primitives'
method: GsQueryGrammar
identifier
  ^ (PPPredicateObjectParser
    on: [ :each | each isLetter or: [ each == $_ ] ]
    message: 'letter expected')
    ,
      (PPPredicateObjectParser
        on: [ :each | each isAlphaNumeric or: [ each == $_ ] ]
        message: 'letter or digit expected') star
%
category: 'token'
method: GsQueryGrammar
identifierToken
  ^ primaryIdentifier queryToken
%
category: 'grammar'
method: GsQueryGrammar
literal
  ^ numberLiteral / stringLiteral / charLiteral / arrayLiteral / byteLiteral
    / symbolLiteral / nilLiteral / trueLiteral / falseLiteral
%
category: 'grammar-literals'
method: GsQueryGrammar
nilLiteral
	^ nilToken
%
category: 'token'
method: GsQueryGrammar
nilToken
  ^ ('nil' asParser , #'word' asParser not) queryToken
%
category: 'primitives'
method: GsQueryGrammar
not
  ^ 'not' asParser , #'word' asParser not
%
category: 'grammar-messages'
method: GsQueryGrammar
notBlock
  ^ [ :a :b | 
  {a.
  b} ]
%
category: 'grammar-messages'
method: GsQueryGrammar
notMessage
  ^ notToken ==> [ :node | { { node } . { } } ]
%
category: 'token'
method: GsQueryGrammar
notToken
  ^ not queryToken
%
category: 'primitives'
method: GsQueryGrammar
number
  ^ ($- asParser optional , #'digit' asParser) and
    ,
      [ :stream | 
      [ Number fromStream: stream ]
        on: Error
        do: [ :err | PPFailure message: err messageText at: stream position ] ]
        asParser
%
category: 'grammar-literals'
method: GsQueryGrammar
numberLiteral
	^ numberToken
%
category: 'token'
method: GsQueryGrammar
numberToken
  ^ number queryToken
%
category: 'grammar-messages'
method: GsQueryGrammar
orBlock
  ^ [ :predicate1 :operator :predicate2 | 
  {predicate1.
  {{{operator}.
  {predicate2}}}} ]
%
category: 'grammar-literals'
method: GsQueryGrammar
parenExpression
  ^ $( asParser queryToken , expression , $) asParser queryToken
%
category: 'grammar-literals'
method: GsQueryGrammar
parenPredicate
  ^ $( asParser queryToken , predicateExpression , $) asParser queryToken
%
category: 'parsing'
method: GsQueryGrammar
parseFormula: aString
  ^ self
    parseFormula: aString
    onError: [ :msg :pos | GsQueryParseError signal: msg , ' at position ' , pos printString position: pos ]
%
category: 'parsing'
method: GsQueryGrammar
parseFormula: aString onError: aBlock
  ^ self start parse: aString onError: aBlock
%
category: 'grammar-messages'
method: GsQueryGrammar
path
  ^ primaryIdentifier , ($. asParser , identifier) plus
%
category: 'grammar-messages'
method: GsQueryGrammar
pathToken
  ^ path queryToken
%
category: 'grammar-messages'
method: GsQueryGrammar
predicateExpression
  ^ rangePredicateExpression / simplePredicateExpression
%
category: 'grammar-messages'
method: GsQueryGrammar
predicateMessage
  ^ ((relopToken
    handleGsQueryErrorMessage: 'relational operator expected'
    nestMessages: false) , primary)
    ==> [ :nodes | 
      {{(nodes first)}.
      {(nodes at: 2)}} ]
%
category: 'grammar'
method: GsQueryGrammar
primary
  ^ queryPathToken / pathToken / literal / variable
    handleGsQueryErrorMessage: 'queryPath, path, literal or variable expected'
    nestMessages: false
%
category: 'grammar-messages'
method: GsQueryGrammar
queryPath
  ^ 'each' asParser
    ,
      ($. asParser
        ,
          (('#' asParser , identifier) / (identifier , '|' asParser , identifier)
            / identifier)
        ,
          ($. asParser
            ,
              (('#' asParser , identifier) / (identifier , '|' asParser , identifier)
                / '*' asParser / identifier))
            star
        min: 0)
%
category: 'grammar-messages'
method: GsQueryGrammar
queryPathToken
  ^ queryPath queryToken
%
category: 'grammar-messages'
method: GsQueryGrammar
rangePredicateExpression
  ^ rangePrimary
    ,
      (relopToken
        handleGsQueryErrorMessage: 'relational operator expected'
        nestMessages: false)
    , queryPathToken
    ,
      (relopToken
        handleGsQueryErrorMessage: 'relational operator expected'
        nestMessages: false)
    , rangePrimary
%
category: 'grammar'
method: GsQueryGrammar
rangePrimary
  ^ pathToken / literal / variable
    handleGsQueryErrorMessage: 'path, literal or variable expected'
    nestMessages: false
%
category: 'primitives'
method: GsQueryGrammar
relop
  ^ '<=' asParser / '<' asParser / '>=' asParser / '>' asParser / '==' asParser
    / '=' asParser / '~=' asParser / '~~' asParser
%
category: 'token'
method: GsQueryGrammar
relopToken
  ^ relop queryToken
%
category: 'primitives'
method: GsQueryGrammar
start
  "Default start production."

  ^ expression end handleGsQueryErrorMessage: 'invalid query expression'
%
category: 'primitives'
method: GsQueryGrammar
string
	^ $' asParser , ('''''' asParser / $' asParser negate) star , $' asParser
%
category: 'grammar-literals'
method: GsQueryGrammar
stringLiteral
	^ stringToken
%
category: 'token'
method: GsQueryGrammar
stringToken
  ^ string queryToken
%
category: 'primitives'
method: GsQueryGrammar
symbol
  ^ identifier / binary / string
%
category: 'grammar-literals'
method: GsQueryGrammar
symbolLiteral
  ^ ($# asParser queryToken plus , symbol queryToken)
    ==> [ :tokens | tokens first copyWith: tokens last ]
%
category: 'grammar-literals'
method: GsQueryGrammar
symbolLiteralArray
  ^ symbol queryToken
%
category: 'grammar-literals'
method: GsQueryGrammar
trueLiteral
	^ trueToken
%
category: 'token'
method: GsQueryGrammar
trueToken
  ^ ('true' asParser , #'word' asParser not) queryToken
%
category: 'grammar'
method: GsQueryGrammar
variable
	^ identifierToken
%
expectValue %Boolean
doit
GsQueryGrammar category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryLiteral
removeallmethods GsQueryLiteral
removeallclassmethods GsQueryLiteral

doit
GsQueryLiteral comment:
'GsQueryLiteral is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsQueryLiteral
! ------------------- Instance methods for GsQueryLiteral
category: 'converting'
method: GsQueryLiteral
asFormula
  (self _idxValue isKindOf: Boolean)
    ifFalse: [ self errorConstantPredicateMustBeBoolean ].
  ^ GsConstantConstantPredicate new
    operand1: self asReferenceAssociation;
    operator: #'unary';
    immediateInvariant
%
category: 'converting'
method: GsQueryLiteral
asFormulaWithSelectorParts: selectorParts
  ^ GsUnaryClause
    clause: self asFormula
    operator: selectorParts first inputValue asSymbol
%
category: 'converting'
method: GsQueryLiteral
asFormulaWithSelectorParts: selectorParts arguments: arguments
  ^ arguments first asReferenceAssociation
    constructFormulaWithSelector: selectorParts first inputValue
    againstConstantReferenceAssoction: self asReferenceAssociation
%
category: 'converting'
method: GsQueryLiteral
asReferenceAssociation
  ^ GsConstantReferenceAssociation newWithKey: self valuePrintString value: self _idxValue
%
category: 'private'
method: GsQueryLiteral
constructFormulaWithSelector: selector againstConstantReferenceAssoction: aConstantReferenceAssociation
  ^ GsConstantConstantPredicate new
    operand1: aConstantReferenceAssociation;
    operator: selector asSymbol;
    operand2: self asReferenceAssociation;
    immediateInvariant
%
category: 'private'
method: GsQueryLiteral
errorConstantPredicateMustBeBoolean
  self error: 'Constant predicate must be Boolean'
%
category: 'accessing'
method: GsQueryLiteral
_idxValue
  self subclassResponsibility: #'_idxValue'
%
category: 'accessing'
method: GsQueryLiteral
valuePrintOn: aStream
  self subclassResponsibility: #'valuePrintOn:'
%
category: 'accessing'
method: GsQueryLiteral
valuePrintString
  | aStream |
  aStream := AppendStream on: String new.
  self valuePrintOn: aStream.
  ^ aStream contents
%
category: 'private'
method: GsQueryLiteral
buildGsQueryParserNodeFor: aGsQueryParser
  aGsQueryParser build: self messages: nil
%
expectValue %Boolean
doit
GsQueryLiteral category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryLiteralArray
removeallmethods GsQueryLiteralArray
removeallclassmethods GsQueryLiteralArray

doit
GsQueryLiteralArray comment:
'GsQueryLiteralArray is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsQueryLiteralArray
! ------------------- Instance methods for GsQueryLiteralArray
category: 'converting'
method: GsQueryLiteralArray
asFormula
  self errorConstantPredicateMustBeBoolean
%
category: 'accessing'
method: GsQueryLiteralArray
contents
  ^ contents
%
category: 'accessing'
method: GsQueryLiteralArray
contents: anArray
  contents := anArray
%
category: 'accessing'
method: GsQueryLiteralArray
isByteArray
  ^ isByteArray
%
category: 'accessing'
method: GsQueryLiteralArray
isByteArray: aBool
  isByteArray := aBool
%
category: 'accessing'
method: GsQueryLiteralArray
_idxValue
  | array |
  array := (isByteArray
    ifTrue: [ ByteArray ]
    ifFalse: [ Array ]) new: contents size.
  1 to: contents size do: [ :index | array at: index put: (contents at: index) _idxValue ].
  ^ array
%
category: 'accessing'
method: GsQueryLiteralArray
valuePrintOn: aStream
  isByteArray
    ifTrue: [ aStream nextPutAll: '#[ ' ]
    ifFalse: [ aStream nextPutAll: '#( ' ].
  1 to: contents size do: [ :index | 
    (contents at: index) valuePrintOn: aStream.
    aStream space ].
  isByteArray
    ifTrue: [ aStream nextPutAll: ']' ]
    ifFalse: [ aStream nextPutAll: ')' ]
%

expectValue %Boolean
doit
GsQueryLiteralArray category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryLiteralValue
removeallmethods GsQueryLiteralValue
removeallclassmethods GsQueryLiteralValue

doit
GsQueryLiteralValue comment:
'GsQueryLiteralValue is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsQueryLiteralValue
! ------------------- Instance methods for GsQueryLiteralValue
category: 'accessing'
method: GsQueryLiteralValue
_idxValue

  ^self value
%
category: 'accessing'
method: GsQueryLiteralValue
value
  ^ value
%
category: 'accessing'
method: GsQueryLiteralValue
value: anObject
  value := anObject
%
category: 'accessing'
method: GsQueryLiteralValue
valuePrintOn: aStream
  ^ self value printOn: aStream
%
category: 'accessing'
method: GsQueryLiteralValue
valuePrintString
  ^ self value printString
%
expectValue %Boolean
doit
GsQueryLiteralValue category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryOptions
removeallmethods GsQueryOptions
removeallclassmethods GsQueryOptions

doit
GsQueryOptions comment:
'The class GsQueryOptions is part of the Indexing and Querying subsystem.  This class encapsulates the optional refinements that may be used when creating a particular query.  Each GsQuery has an associated instance of GsQueryOptions, either the default one or an instance that is created and included in the GsQuery specification method.

GsQueryOptions are stored in dynamic instance variables for ease of migration.  The following are false by default:
   cacheQueryResult
   optionalPathTerms
The following apply to optimization of queries and are true by default:
   applyDeMorgansLaws
   autoOptimize
   consolidateEnumerablePredicates
   consolidateRangePredicates
   consolidateUnaryConstantPredicates
   normalizePredicates
   removeRedundantPredicates
   reorderPredicates
   transformCommonPaths
'. 
true
%

! ------------------- Class methods for GsQueryOptions
category: 'options'
classmethod: GsQueryOptions
applyDeMorgansLaws
  "Apply De Morgan's Laws to eliminate `not` in queries:

    The negation of a conjunction is the disjunction of the negations.
    The negation of a disjunction is the conjunction of the negations.

    Where:

          (each.firstName = 'Dale') not

        becomes:

          (each.firstName ~= 'Dale')

    and:

          ((1 <= each.numberOfChildren) & (each.numberOfChildren <= 4)) not

        becomes:

          (each.numberOfChildren < 1) | (each.numberOfChildren > 4)
    and:

          ((1 > each.numberOfChildren) | (each.numberOfChildren > 4)) not

        becomes:

          (1 <= each.numberOfChildren) & (each.numberOfChildren <= 4)
"

  ^ self new
    setApplyDeMorgansLaws;
    yourself
%
category: 'options'
classmethod: GsQueryOptions
autoOptimize
  "Query formula is automatically optimized"

  ^ self new
    setAutoOptimize;
    yourself
%
category: 'options'
classmethod: GsQueryOptions
cacheQueryResult
  "Cache query result"

  ^ self new
    setCacheQueryResult;
    yourself
%
category: 'options'
classmethod: GsQueryOptions
consolidateEnumerablePredicates
  "Consolidate path-constant predicates to single enumerated predicate
      if an index exists that has an enumerated path term and predicates
      using the enumerated path  term can be combined into a single 
      predicate

          (each.firstName = 'Martin') | (each.lastName = 'Martin')

        becomes:

          (each.firstName|lastName = 'Martin')"

  ^ self new
    setConsolidateEnumerablePredicates;
    yourself
%
category: 'options'
classmethod: GsQueryOptions
consolidateRangePredicates
  "Convert 2 path-constant predicates into a single range predicate when possible

          (each.age > 4) & (each.age < 19)

        becomes:

          (4 < each.age < 19)"

  ^ self new
    setConsolidateRangePredicates;
    yourself
%
category: 'options'
classmethod: GsQueryOptions
default
  "all options set except cacheQueryResult"

  ^ self optimizerOptions + self autoOptimize
%
category: 'options'
classmethod: GsQueryOptions
normalizePredicates
  "Replace constant-path predicates with equivalent path-constant predicates.

          (19 > each.age) 

        becomes:

          (each.age < 19)"

  ^ self new
    setNormalizePredicates;
    yourself
%
category: 'options'
classmethod: GsQueryOptions
optimizerOptions
  "all options that control operation of optimizer"

  ^ self new
    setApplyDeMorgansLaws;
    setConsolidateEnumerablePredicates;
    setConsolidateRangePredicates;
    setConsolidateUnaryConstantPredicates;
    setNormalizePredicates;
    setRemoveRedundantPredicates;
    setReorderPredicates;
    setTransformCommonPaths
%
category: 'options'
classmethod: GsQueryOptions
removeRedundantPredicates
  "Eliminate redundant predicates. i.e., predicates that fall within range
	of other predicates

          (each.age < 19) & (each.age < 4)

        becomes:

          (each.age < 4)"

  ^ self new
    setRemoveRedundantPredicates;
    yourself
%
category: 'options'
classmethod: GsQueryOptions
reorderPredicates
  "Reorder predicates based on following ordering rules:
        constant predicates
        indexed predicates
        identity comparison predicates
        equality comparison predicates
        all others

          (each.age <= 21) & (each.gender == #male) & 
          (each.name = 'Dale') & (each.father = each.father)

        assuming each.name is indexed and each.age is not indexed, becomes:

          (true) & (each.name = 'Dale') & 
          (each.gender == #male) & (each.age <= 21)"

  ^ self new
    setReorderPredicates;
    yourself
%
category: 'options'
classmethod: GsQueryOptions
transformCommonPaths
  "Convert predicates with common path-path operands to an equivalent 
	constant predicate.

          (each.firstName = each.firstName) 

        becomes:

          (true)"

  ^ self new
    setTransformCommonPaths;
    yourself
%
! ------------------- Instance methods for GsQueryOptions
category: 'Initialize-Release'
method: GsQueryOptions
initialize
  super initialize.
  self applyDeMorgansLaws: false.
  self autoOptimize: false.
  self cacheQueryResult: false.
  self consolidateEnumerablePredicates: false.
  self consolidateRangePredicates: false.
  self normalizePredicates: false.
  self removeRedundantPredicates: false.
  self consolidateUnaryConstantPredicates: false.
  self reorderPredicates: false.
  self transformCommonPaths: false.
  self optionalPathTerms: false
%
category: 'Accessing'
method: GsQueryOptions
applyDeMorgansLaws
  | val |
  val := self dynamicInstVarAt: #'applyDeMorgansLaws'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
applyDeMorgansLaws: aBoolean
  self dynamicInstVarAt: #'applyDeMorgansLaws' put: aBoolean
%
category: 'Accessing'
method: GsQueryOptions
autoOptimize
  | val |
  val := self dynamicInstVarAt: #'autoOptimize'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
autoOptimize: aBoolean
  self dynamicInstVarAt: #'autoOptimize' put: aBoolean
%
category: 'Accessing'
method: GsQueryOptions
cacheQueryResult
  | val |
  val := self dynamicInstVarAt: #'cacheQueryResult'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
cacheQueryResult: aBoolean
  self dynamicInstVarAt: #'cacheQueryResult' put: aBoolean
%
category: 'Operators'
method: GsQueryOptions
clearApplyDeMorgansLaws
  self applyDeMorgansLaws: false
%
category: 'Operators'
method: GsQueryOptions
clearAutoOptimize
  self autoOptimize: false
%
category: 'Operators'
method: GsQueryOptions
clearCacheQueryResult
  self cacheQueryResult: false
%
category: 'Operators'
method: GsQueryOptions
clearConsolidateEnumerablePredicates
  self consolidateEnumerablePredicates: false
%
category: 'Operators'
method: GsQueryOptions
clearConsolidateRangePredicates
  self consolidateRangePredicates: false
%
category: 'Operators'
method: GsQueryOptions
clearNormalizePredicates
  self normalizePredicates: false
%
category: 'Operators'
method: GsQueryOptions
clearRemoveRedundantPredicates
  self removeRedundantPredicates: false
%
category: 'Operators'
method: GsQueryOptions
clearReorderPredicates
  self reorderPredicates: false
%
category: 'Operators'
method: GsQueryOptions
clearTransformCommonPaths
  self transformCommonPaths: false
%
category: 'Operators'
method: GsQueryOptions
combineWith: aGsQueryOptions
  self applyDeMorgansLaws
    ifTrue: [ aGsQueryOptions setApplyDeMorgansLaws ].
  self autoOptimize
    ifTrue: [ aGsQueryOptions setAutoOptimize ].
  self cacheQueryResult
    ifTrue: [ aGsQueryOptions setCacheQueryResult ].
  self consolidateEnumerablePredicates
    ifTrue: [ aGsQueryOptions setConsolidateEnumerablePredicates ].
  self consolidateRangePredicates
    ifTrue: [ aGsQueryOptions setConsolidateRangePredicates ].
  self normalizePredicates
    ifTrue: [ aGsQueryOptions setNormalizePredicates ].
  self removeRedundantPredicates
    ifTrue: [ aGsQueryOptions setRemoveRedundantPredicates ].
  self consolidateUnaryConstantPredicates
    ifTrue: [ aGsQueryOptions setConsolidateUnaryConstantPredicates ].
  self reorderPredicates
    ifTrue: [ aGsQueryOptions setReorderPredicates ].
  self transformCommonPaths
    ifTrue: [ aGsQueryOptions setTransformCommonPaths ].
  self optionalPathTerms
    ifTrue: [ aGsQueryOptions setOptionalPathTerms ].
  ^ aGsQueryOptions
%
category: 'Accessing'
method: GsQueryOptions
consolidateEnumerablePredicates
  | val |
  val := self dynamicInstVarAt: #'consolidateEnumerablePredicates'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
consolidateEnumerablePredicates: aBoolean
  self dynamicInstVarAt: #'consolidateEnumerablePredicates' put: aBoolean
%
category: 'Accessing'
method: GsQueryOptions
consolidateRangePredicates
  | val |
  val := self dynamicInstVarAt: #'consolidateRangePredicates'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
consolidateRangePredicates: aBoolean
  self dynamicInstVarAt: #'consolidateRangePredicates' put: aBoolean
%
category: 'Operators'
method: GsQueryOptions
negateWith: aGsQueryOptions
  self applyDeMorgansLaws
    ifTrue: [ aGsQueryOptions clearApplyDeMorgansLaws ].
  self autoOptimize
    ifTrue: [ aGsQueryOptions clearAutoOptimize ].
  self cacheQueryResult
    ifTrue: [ aGsQueryOptions clearCacheQueryResult ].
  self consolidateEnumerablePredicates
    ifTrue: [ aGsQueryOptions clearConsolidateEnumerablePredicates ].
  self consolidateRangePredicates
    ifTrue: [ aGsQueryOptions clearConsolidateRangePredicates ].
  self normalizePredicates
    ifTrue: [ aGsQueryOptions clearNormalizePredicates ].
  self removeRedundantPredicates
    ifTrue: [ aGsQueryOptions clearRemoveRedundantPredicates ].
  self consolidateUnaryConstantPredicates
    ifTrue: [ aGsQueryOptions clearConsolidateUnaryConstantPredicates ].
  self reorderPredicates
    ifTrue: [ aGsQueryOptions clearReorderPredicates ].
  self transformCommonPaths
    ifTrue: [ aGsQueryOptions clearTransformCommonPaths ].
  self optionalPathTerms
    ifTrue: [ aGsQueryOptions clearOptionalPathTerms ].
  ^ aGsQueryOptions
%
category: 'Accessing'
method: GsQueryOptions
normalizePredicates
  | val |
  val := self dynamicInstVarAt: #'normalizePredicates'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
normalizePredicates: aBoolean
  self dynamicInstVarAt: #'normalizePredicates' put: aBoolean
%
category: 'Accessing'
method: GsQueryOptions
removeRedundantPredicates
  | val |
  val := self dynamicInstVarAt: #'removeRedundantPredicates'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
removeRedundantPredicates: aBoolean
  self dynamicInstVarAt: #'removeRedundantPredicates' put: aBoolean
%
category: 'Accessing'
method: GsQueryOptions
reorderPredicates
  | val |
  val := self dynamicInstVarAt: #'reorderPredicates'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
reorderPredicates: aBoolean
  self dynamicInstVarAt: #'reorderPredicates' put: aBoolean
%
category: 'Operators'
method: GsQueryOptions
setApplyDeMorgansLaws
  self applyDeMorgansLaws: true
%
category: 'Operators'
method: GsQueryOptions
setAutoOptimize
  self autoOptimize: true
%
category: 'Operators'
method: GsQueryOptions
setCacheQueryResult
  self cacheQueryResult: true
%
category: 'Operators'
method: GsQueryOptions
setConsolidateEnumerablePredicates
  self consolidateEnumerablePredicates: true
%
category: 'Operators'
method: GsQueryOptions
setConsolidateRangePredicates
  self consolidateRangePredicates: true
%
category: 'Operators'
method: GsQueryOptions
setNormalizePredicates
  self normalizePredicates: true
%
category: 'Operators'
method: GsQueryOptions
setRemoveRedundantPredicates
  self removeRedundantPredicates: true
%
category: 'Operators'
method: GsQueryOptions
setReorderPredicates
  self reorderPredicates: true
%
category: 'Operators'
method: GsQueryOptions
setTransformCommonPaths
  self transformCommonPaths: true
%
category: 'Accessing'
method: GsQueryOptions
transformCommonPaths
  | val |
  val := self dynamicInstVarAt: #'transformCommonPaths'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
transformCommonPaths: aBoolean
  self dynamicInstVarAt: #'transformCommonPaths' put: aBoolean
%
expectValue %Boolean
doit
GsQueryOptions category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryParser
removeallmethods GsQueryParser
removeallclassmethods GsQueryParser

doit
GsQueryParser comment:
'GsQueryParser is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsQueryParser
category: 'instance creation'
classmethod: GsQueryParser
compileFormula: aString
  ^ (self parseFormula: aString) asFormula
%
category: 'instance creation'
classmethod: GsQueryParser
compileFormula: aString onError: aBlock
  ^ (self parseFormula: aString onError: aBlock) asFormula
%
! ------------------- Instance methods for GsQueryParser
category: 'grammar-literals'
method: GsQueryParser
andBlock
  ^ [ :predicate1 :operator :predicate2 | 
  predicate1 asFormula
    asFormulaWithSelectorParts: {operator}
    arguments: {(predicate2 asFormula)} ]
%
category: 'grammar-literals'
method: GsQueryParser
arrayLiteral
  ^ super arrayLiteral
    ==> [ :nodes | 
      GsQueryLiteralArray new
        contents: (nodes at: 2);
        isByteArray: false;
        yourself ]
%
category: 'grammar-literals'
method: GsQueryParser
arrayLiteralArray
  ^ super arrayLiteralArray
    ==> [ :nodes | 
      GsQueryLiteralArray new
        contents: (nodes at: 2);
        isByteArray: false;
        yourself ]
%
category: 'private'
method: GsQueryParser
build: aNode messages: anArray
  ^ (anArray == nil  or: [ anArray isEmpty ])
    ifTrue: [ aNode ]
    ifFalse: [ 
      anArray
        inject: aNode
        into: [ :receiver :messageArray | 
          messageArray == nil 
            ifTrue: [ receiver ]
            ifFalse: [ 
              (messageArray at: 2) isEmpty
                ifTrue: [ receiver asFormulaWithSelectorParts: messageArray first ]
                ifFalse: [ 
                  receiver
                    asFormulaWithSelectorParts: messageArray first
                    arguments: (messageArray at: 2) ] ] ] ]
%
category: 'private'
method: GsQueryParser
buildString: aString 
	(aString isEmpty or: [ aString first ~= $' or: [ aString last ~= $' ] ])
		ifTrue: [ ^ aString ].
	^ (aString 
		copyFrom: 2
		to: aString size - 1) 
		copyReplaceAll: ''''''
		with: ''''
%
category: 'grammar-literals'
method: GsQueryParser
byteLiteral
  ^ super byteLiteral
    ==> [ :nodes | 
      GsQueryLiteralArray new
        contents: (nodes at: 2);
        isByteArray: true;
        yourself ]
%
category: 'grammar-literals'
method: GsQueryParser
byteLiteralArray
  ^ super byteLiteralArray
    ==> [ :nodes | 
      GsQueryLiteralArray new
        contents: (nodes at: 2);
        isByteArray: true;
        yourself ]
%
category: 'grammar-literals'
method: GsQueryParser
charLiteral
  ^ super charLiteral
    ==> [ :token | 
      GsQueryLiteralValue new
        value: (token inputValue at: 2);
        yourself ]
%
category: 'grammar-messages'
method: GsQueryParser
expression
  ^ super expression
    map: [ :arrayOrNode | arrayOrNode buildGsQueryParserNodeFor: self ]
%
category: 'grammar-literals'
method: GsQueryParser
falseLiteral
  ^ super falseLiteral
    ==> [ :token | 
      GsQueryLiteralValue new
        value: false;
        yourself ]
%
category: 'grammar-literals'
method: GsQueryParser
nilLiteral
  ^ super nilLiteral
    ==> [ :token | 
      GsQueryLiteralValue new
        value: nil;
        yourself ]
%
category: 'grammar-literals'
method: GsQueryParser
notBlock
  ^ [ :predicate1 :operator | predicate1 asFormulaWithSelectorParts: {operator} ]
%
category: 'grammar-literals'
method: GsQueryParser
numberLiteral
  ^ super numberLiteral
    ==> [ :token | 
      GsQueryLiteralValue new
        value: (Number fromStream: token inputValue readStream);
        yourself ]
%
category: 'grammar-literals'
method: GsQueryParser
orBlock
  ^ [ :predicate1 :operator :predicate2 | 
  predicate1 asFormula
    asFormulaWithSelectorParts: {operator}
    arguments: {(predicate2 asFormula)} ]
%
category: 'grammar'
method: GsQueryParser
parenExpression
  ^ super parenExpression
    map: [ :openToken :expressionNode :closeToken | expressionNode ]
%
category: 'grammar'
method: GsQueryParser
parenPredicate
  ^ super parenPredicate
    map: [ :openToken :expressionNode :closeToken | expressionNode ]
%
category: 'grammar'
method: GsQueryParser
pathToken
  ^ super pathToken
    ==> [ :token | GsPathReferenceAssociation newWithKey: token inputValue asString value: nil ]
%
category: 'grammar'
method: GsQueryParser
queryPathToken
  ^ super queryPathToken
    ==> [ :token | GsQueryPathReferenceAssociation newWithKey: token inputValue asString value: nil ]
%
category: 'grammar-messages'
method: GsQueryParser
rangePredicateExpression
  ^ super rangePredicateExpression
    map: [ :primary1 :relop1 :queryPath1 :relop2 :primary2 | 
      GsRangeQueryPredicate
        operand: primary1 asReferenceAssociation
        operator: relop1 inputValue asSymbol
        path: queryPath1 asReferenceAssociation
        operator: relop2 inputValue asSymbol
        operand: primary2 asReferenceAssociation ]
%
category: 'grammar-literals'
method: GsQueryParser
stringLiteral
  ^ super stringLiteral
    ==> [ :token | 
      GsQueryLiteralValue new
        value: (self buildString: token inputValue);
        yourself ]
%
category: 'grammar-literals'
method: GsQueryParser
symbolLiteral
  ^ super symbolLiteral
    ==> [ :tokens | 
      GsQueryLiteralValue new
        value: (self buildString: tokens last inputValue) asSymbol;
        yourself ]
%
category: 'grammar-literals'
method: GsQueryParser
symbolLiteralArray
  ^ super symbolLiteralArray
    ==> [ :tokens | 
      GsQueryLiteralValue new
        value: (self buildString: tokens last inputValue) asSymbol;
        yourself ]
%
category: 'grammar-literals'
method: GsQueryParser
trueLiteral
  ^ super trueLiteral
    ==> [ :token | 
      GsQueryLiteralValue new
        value: true;
        yourself ]
%
category: 'grammar'
method: GsQueryParser
variable
  ^ super variable
    ==> [ :token | GsVariableReferenceAssociation newWithKey: token inputValue asString value: nil ]
%
expectValue %Boolean
doit
GsQueryParser category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryPathReferenceAssociation
removeallmethods GsQueryPathReferenceAssociation
removeallclassmethods GsQueryPathReferenceAssociation

doit
GsQueryPathReferenceAssociation comment: 
'The class GsQueryPathReferenceAssociation represents the value of a path operand in a GsAbstractQueryPredicate.'.
true
%

! ------------------- Class methods for GsQueryPathReferenceAssociation
! ------------------- Instance methods for GsQueryPathReferenceAssociation
category: 'converting'
method: GsQueryPathReferenceAssociation
asArrayOf32PathTerms
  ^ self key asArrayOf32PathTerms
%
category: 'converting'
method: GsQueryPathReferenceAssociation
asFormula
  ^ GsPathConstantPredicate new
    path1: self;
    operator: #'unary';
    immediateInvariant
%
category: 'converting'
method: GsQueryPathReferenceAssociation
asFormulaWithSelectorParts: selectorParts arguments: arguments
  ^ arguments first asReferenceAssociation
    constructFormulaWithSelector: selectorParts first inputValue
    againstPathReferenceAssoction: self
%
category: 'private'
method: GsQueryPathReferenceAssociation
constructFormulaWithSelector: selector againstConstantReferenceAssoction: aConstantReferenceAssociation
  ^ GsConstantPathPredicate new
    operand1: aConstantReferenceAssociation;
    operator: selector asSymbol;
    path2: self;
    immediateInvariant
%
category: 'private'
method: GsQueryPathReferenceAssociation
constructFormulaWithSelector: selector againstPathReferenceAssoction: aPathReferenceAssociation
  ^ GsPathPathPredicate new
    path1: aPathReferenceAssociation;
    operator: selector asSymbol;
    path2: self;
    immediateInvariant
%
category: 'testing'
method: GsQueryPathReferenceAssociation
hasSamePathAs: aReferenceAssoc
  (aReferenceAssoc isKindOf: self class)
    ifFalse: [ ^ false ].
  ^ self key = aReferenceAssoc key
%
expectValue %Boolean
doit
GsQueryPathReferenceAssociation category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryPredicate
removeallmethods GsQueryPredicate
removeallclassmethods GsQueryPredicate

doit
GsQueryPredicate comment: 
'The abstract class GsQueryPredicate is a predicate in a GsQueryFormula. This class applies an operator to two operands.

Operands may be a literal (instance of GsConstantReferenceAssociation), a variable (instance of GsVariableReferenceAssociation), or a path (instance of a GsQueryPathReferenceAssociation).

An operator may be one of the following: < > = == <= >= ~= ~~.'.
true
%

! ------------------- Class methods for GsQueryPredicate
category: 'instance creation'
classmethod: GsQueryPredicate
constant: operand1
  (operand1 _idxValue == true or: [ operand1 _idxValue == false ])
    ifFalse: [ self error: 'Unary constant must be Boolean' ].
  ^ GsConstantConstantPredicate new
    constant1: operand1;
    operator: #'unary';
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
constant: operand1 operator: operator constant: operand2
  ^ GsConstantConstantPredicate new
    constant1: operand1;
    operator: operator;
    constant2: operand2;
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
constant: operand1 operator: operator path: operand2
  ^ (GsConstantPathPredicate
    constant: operand1
    operator: operator
    path: operand2) immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
constant: constant1 operator: operator1 path: path operator: operator2 constant: constant2
  "e.g.: 4 < a.b.c < 6"

  ^ (GsRangeQueryPredicate
    constant: constant1
    operator: operator1
    path: path
    operator: operator2
    constant: constant2) immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
constant: constant1 operator: operator1 path: path operator: operator2 variable: constant2
  "e.g.: 4 < a.b.c < 6"

  ^ (GsRangeQueryPredicate
    constant: constant1
    operator: operator1
    path: path
    operator: operator2
    variable: constant2) immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
constant: operand1 operator: operator variable: operand2
  ^ GsConstantConstantPredicate new
    constant1: operand1;
    operator: operator;
    variable2: operand2;
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
fromQueryStatement: aString
  ^ (GsQueryParser compileFormula: aString) immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
fromSelectBlock: aSelectBlock
  "queryBlockArray is a four element Array of Arrays describing a SelectBlock
 in the manner needed to process a query on the receiver:

   1.  Array of the bound variables from the predicate
   2.  Array of the predicate's terms
   3.  Array of the predicate's paths
   4.  Array of Strings of the path names used in the predicate"

  | queryBlockArray varsArray termsArray linksArray pathsArray formula |
  queryBlockArray := aSelectBlock queryBlock value: nil.
  varsArray := queryBlockArray at: 1.
  termsArray := queryBlockArray at: 2.
  linksArray := queryBlockArray at: 3.
  pathsArray := queryBlockArray at: 4.
  formula := nil.
  1 to: termsArray size by: 7 do: [ :i | 
    | predicate |
    predicate := self _createPredicateFromTermsArray: termsArray startingAt: i.
    predicate
      _fromArray: varsArray
      terms: termsArray
      links: linksArray
      paths: pathsArray
      startingAt: i.
    formula == nil
      ifTrue: [ formula := predicate ]
      ifFalse: [ formula := formula & predicate ] ].
  ^ formula
%
category: 'instance creation'
classmethod: GsQueryPredicate
fromSelectBlock: aSelectBlock usingQueryExecuterWith: anNsc
  "Extract predicates from QueryExecuter using <aSelectBlock>
        and <anNsc>. Return an GsQuery. Note that the predicates are
        based on standard transforms applied by the QueryExecuter."

  | queryExecuter gueryBlockArray queryFormula |
  queryExecuter := QueryExecuter on: anNsc.
  gueryBlockArray := aSelectBlock queryBlock value: nil.
  queryExecuter
    _buildIndexOperationsList: gueryBlockArray;
    optimize.
  queryFormula := nil.
  queryExecuter _getOperationsOrder
    do: [ :offset |
      | predicate |
      predicate := self
        _createPredicateFromQueryExecuter: queryExecuter
        startingAt: offset.
      queryFormula == nil
        ifTrue: [ queryFormula := predicate ]
        ifFalse: [ queryFormula := queryFormula & predicate ] ].
  queryFormula
    ifNil: [
      "unsatisfiable query"
      queryFormula := (self constant: false)
        bindEvaluatorsFor: anNsc
        collator: nil ].
  ^ queryFormula
%
category: 'instance creation'
classmethod: GsQueryPredicate
operand: operand1 operator: operator operand: operand2
  ^ self new
    operand1: operand1;
    operator: operator;
    operand2: operand2;
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
operand: operand1 operator: operator1 path: path operator: operator2 operand: operand2
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (GsRangeQueryPredicate
    operand: operand1
    operator: operator1
    path: path
    operator: operator2
    operand: operand2) immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
path: operand1
  ^ GsPathConstantPredicate new
    path1: operand1;
    operator: #'unary';
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
path: operand1 operator: operator constant: operand2
  ^ GsPathConstantPredicate new
    path1: operand1;
    operator: operator;
    constant2: operand2;
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
path: operand1 operator: operator path: operand2
  ^ GsPathPathPredicate new
    path1: operand1;
    operator: operator;
    path2: operand2;
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
path: operand1 operator: operator variable: operand2
  ^ GsPathConstantPredicate new
    path1: operand1;
    operator: operator;
    variable2: operand2;
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
variable: operand1
  (operand1 _idxValue == true or: [ operand1 _idxValue == false ])
    ifFalse: [ self error: 'Unary constant must be Boolean' ].
  ^ GsConstantConstantPredicate new
    variable1: operand1;
    operator: #'unary';
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
variable: operand1 operator: operator constant: operand2
  ^ GsConstantConstantPredicate new
    variable1: operand1;
    operator: operator;
    constant2: operand2;
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
variable: operand1 operator: operator path: operand2
  ^ GsConstantPathPredicate new
    variable1: operand1;
    operator: operator;
    path2: operand2;
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
variable: constant1 operator: operator1 path: path operator: operator2 constant: constant2
  "e.g.: 4 < a.b.c < 6"

  ^ (GsRangeQueryPredicate
    variable: constant1
    operator: operator1
    path: path
    operator: operator2
    constant: constant2) immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
variable: constant1 operator: operator1 path: path operator: operator2 variable: constant2
  "e.g.: 4 < a.b.c < 6"

  ^ (GsRangeQueryPredicate
    variable: constant1
    operator: operator1
    path: path
    operator: operator2
    variable: constant2) immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
variable: operand1 operator: operator variable: operand2
  ^ GsConstantConstantPredicate new
    variable1: operand1;
    operator: operator;
    variable2: operand2;
    immediateInvariant
%
category: 'instance creation'
classmethod: GsQueryPredicate
_createPredicateFromQueryExecuter: aQueryExecuter startingAt: offset
  | firstOperand secondOperand searchOp predicateType predicate operator |
  firstOperand := aQueryExecuter firstOperandAt: offset.
  searchOp := aQueryExecuter operatorAt: offset.
  secondOperand := aQueryExecuter secondOperandAt: offset.
  predicateType := aQueryExecuter at: offset.
  searchOp == 8
    ifTrue: [ 
      predicate := (GsQueryPredicate constant: secondOperand) copy.
      predicate _nsc: aQueryExecuter nsc.
      ^ predicate ].
  searchOp == 9
    ifTrue: [ 
      | searchOp1 searchOp2 |
      predicateType ~~ 2
        ifTrue: [ self error: 'invalid predicateType for searchOp ''9''' ].
      searchOp1 := secondOperand at: 1.
      searchOp2 := secondOperand at: 3.
      predicate := (GsQueryPredicate
        constant: (secondOperand at: 2)
        operator:
          (self inverseOperatorFor: (aQueryExecuter operationSelectors at: searchOp1 + 1))
        path: 'each.' , firstOperand pathComponentsString
        operator: (aQueryExecuter operationSelectors at: searchOp2 + 1)
        constant: (secondOperand at: 4)) copy.
      predicate
        _nsc: aQueryExecuter nsc;
        evaluator: firstOperand.
      ^ predicate ].
  operator := aQueryExecuter operationSelectors at: searchOp + 1.
  predicateType == 2
    ifTrue: [ 
      predicate := (GsQueryPredicate
        path: 'each.' , firstOperand pathComponentsString
        operator: operator
        constant: secondOperand) copy.
      predicate
        _nsc: aQueryExecuter nsc;
        evaluator1: firstOperand.
      ^ predicate ].
  predicateType == 1
    ifTrue: [ 
      predicate := (GsQueryPredicate
        constant: firstOperand
        operator: operator
        constant: secondOperand) copy.
      predicate _nsc: aQueryExecuter nsc.
      ^ predicate ].
  predicateType == 4
    ifTrue: [ 
      predicate := (GsQueryPredicate
        path: 'each.' , firstOperand pathComponentsString
        operator: operator
        path: 'each.' , secondOperand pathComponentsString) copy.
      predicate
        _nsc: aQueryExecuter nsc;
        evaluator1: firstOperand;
        evaluator2: secondOperand.
      ^ predicate ].
  self error: 'Unknown predicate type: ' , predicateType printString
%
category: 'instance creation'
classmethod: GsQueryPredicate
_createPredicateFromTermsArray: termsArray startingAt: offset
  "Return the proper Query Predicate instance.
   Predicates are classified into four types:

 1.  constant-constant (such as 3 > 2)
 2.  path-constant     (such as 'spouse.age' > 30)
 3.  constant-path     (such as 30 <= 'spouse.age')
 4.  path-path         (such as 'spouse.age' < 'spouse.weight')"

  (termsArray at: offset + 4)
    ifTrue: [ 
      (termsArray at: offset + 1)
        ifTrue: [ 
          " its path-path "
          ^ GsPathPathPredicate new ]
        ifFalse: [ 
          " its path-constant "
          ^ GsPathConstantPredicate new ] ]
    ifFalse: [ 
      (termsArray at: offset + 1)
        ifTrue: [ 
          " its constant-path "
          ^ GsConstantPathPredicate new ]
        ifFalse: [ 
          " its constant-constant "
          ^ GsConstantConstantPredicate new ] ]
%
! ------------------- Instance methods for GsQueryPredicate
category: 'optimizing'
method: GsQueryPredicate
applyDeMorgansNegationTransform
  "actively apply De Morgan's laws ... operating on copy"

  self operator: self operatorNegated
%
category: 'converting'
method: GsQueryPredicate
asFormulaWithSelectorParts: selectorParts arguments: arguments
  | selector |
  selector := selectorParts first inputValue asSymbol.
  ^ (self operationSelectors includes: selector)
    ifTrue: [ 
      | clause2 |
      clause2 := self operand2
        asFormulaWithSelectorParts: selectorParts
        arguments: arguments.
      self collapseToRangePredicateIfPossible: clause2 ]
    ifFalse: [ self perform: selector with: arguments first ]
%
category: 'transforming'
method: GsQueryPredicate
bind: variableName to: value
  | bound |
  bound := self copy.
  bound operand1: (bound operand1 bind: variableName to: value).
  bound operand2: (bound operand2 bind: variableName to: value).
  ^ bound immediateInvariant
%
category: 'accessing'
method: GsQueryPredicate
compatibleInverseRangeOperatorsFor: anOperator
  | less greater |
  less := #(#'<' #'<=').
  greater := #(#'>' #'>=').
  ^ ((less includes: anOperator) and: [ greater includes: self operator ])
    or: [ (greater includes: anOperator) and: [ less includes: self operator ] ]
%
category: 'accessing'
method: GsQueryPredicate
compatibleRangeOperatorsFor: anOperator
  | less greater |
  less := #(#'<' #'<=').
  greater := #(#'>' #'>=').
  ^ ((less includes: anOperator) and: [ less includes: self operator ])
    or: [ (greater includes: anOperator) and: [ greater includes: self operator ] ]
%
category: 'accessing'
method: GsQueryPredicate
constant
  self subclassResponsibility: #'constant'
%
category: 'accessing'
method: GsQueryPredicate
constant1: anObject
  operand1 := GsConstantReferenceAssociation
    newWithKey: anObject asString
    value: anObject
%
category: 'accessing'
method: GsQueryPredicate
constant2: anObject
  operand2 := GsConstantReferenceAssociation
    newWithKey: anObject asString
    value: anObject
%
category: 'querying'
method: GsQueryPredicate
elementValueEvaluator
  "companion method to #elementValue:"

  "anObject is the value obtained by traversing path terms to the last value ... analgous to the last element class"

  "not all subclasses are required to implement this selector, if they do not implement this selector, then they must implement #elementValue:"

  ^ (evaluator1 == nil  or: [ evaluator1 isPathEvaluator not ])
    ifTrue: [ self pathEvaluatorFor: self path ]
    ifFalse: [ evaluator1 ]
%
category: 'accessing'
method: GsQueryPredicate
evaluator
  ^ self evaluator1
%
category: 'accessing'
method: GsQueryPredicate
evaluator1
  ^ evaluator1
%
category: 'accessing'
method: GsQueryPredicate
evaluator1: anEvaluator
  evaluator1 := anEvaluator
%
category: 'private'
method: GsQueryPredicate
immediateInvariant
  super immediateInvariant.
  operand1 immediateInvariant.
  operand2 immediateInvariant
%
category: 'testing'
method: GsQueryPredicate
isConstantBound
  self subclassResponsibility: #'isConstantBound'
%
category: 'accessing'
method: GsQueryPredicate
operand1
  ^ operand1
%
category: 'accessing'
method: GsQueryPredicate
operand1: anObject
  operand1 := anObject
%
category: 'testing'
method: GsQueryPredicate
operand1IsPath
  ^ false
%
category: 'accessing'
method: GsQueryPredicate
operand2
  ^ operand2
%
category: 'accessing'
method: GsQueryPredicate
operand2: anObject
  operand2 := anObject
%
category: 'testing'
method: GsQueryPredicate
operand2IsPath
  ^ false
%
category: 'accessing'
method: GsQueryPredicate
operator
  ^ operator
%
category: 'accessing'
method: GsQueryPredicate
operator: queryOp
  (self _isValidQueryOperator: queryOp)
    ifFalse: [ ^ self error: 'Invalid query operator: ' , queryOp printString ].
  operator := queryOp
%
category: 'accessing'
method: GsQueryPredicate
operatorNegated
  ^ self negatedOperatorFor: self operator
%
category: 'accessing'
method: GsQueryPredicate
path
  ^ self rawPath key
%
category: 'accessing'
method: GsQueryPredicate
path1: aPathString
  operand1 := self pathReferenceFor: aPathString
%
category: 'accessing'
method: GsQueryPredicate
path2: aPathString
  operand2 := self pathReferenceFor: aPathString
%
category: 'private'
method: GsQueryPredicate
postCopy
  operand1 := operand1 copy.
  operand2 := operand2 copy.
  super postCopy
%
category: 'printing'
method: GsQueryPredicate
printOn: aStream
  aStream
    nextPutAll:
      '(' , self operand1 printString , ' ' , self operator asString , ' '
        , self operand2 printString , ')'
%
category: 'accessing'
method: GsQueryPredicate
rawPath
  self subclassResponsibility: #'rawPath'
%
category: 'transforming'
method: GsQueryPredicate
unbind
  "remove all bindings"

  | unbound |
  unbound := super unbind.
  unbound operand1: unbound operand1 unbind.
  self operator == #'unary'
    ifFalse: [ unbound operand2: unbound operand2 unbind ].
  unbound unbindEvaluators.
  ^ unbound immediateInvariant
%
category: 'private'
method: GsQueryPredicate
unbindEvaluators
  "remove all bindings"

  evaluator1 := nil
%
category: 'testing'
method: GsQueryPredicate
usesEqualityOperation
  ^ operator == #'=' or: [ operator == #'~=' ]
%
category: 'testing'
method: GsQueryPredicate
usesIdentityOperation
  ^ operator == #'==' or: [ operator == #'~~' ]
%
category: 'testing'
method: GsQueryPredicate
usesPathEvaluator
  self evaluator1 ifNil: [ ^ true ].
  ^ self evaluator1 isPathEvaluator
%
category: 'accessing'
method: GsQueryPredicate
variable1: aVariableName
  operand1 := self variableReferenceFor: aVariableName
%
category: 'accessing'
method: GsQueryPredicate
variable2: aVariableName
  operand2 := self variableReferenceFor: aVariableName
%
category: 'private'
method: GsQueryPredicate
_check: indexObj compare: compareOp value: searchVal
  "Raise an error if the comparison operation is not valid for the index object
 and the search value."

  " check for range operation using an identity index "

  ((self isRangeEqualityOperation: compareOp)
    and: [ indexObj isRangeEqualityIndex not ])
    ifTrue: [ ^ indexObj _errorCannotInvokeRangeOperationOnIdentityIndex ]
%
category: 'private'
method: GsQueryPredicate
_fromArray: varsArray terms: termsArray links: linksArray paths: pathsArray startingAt: i
  self subclassResponsibility: #'_fromArray:terms:links:paths:startingAt:'
%
category: 'testing'
method: GsQueryPredicate
_isValidQueryOperator: queryOp
  "Returns true if the given search operation is one of == ~~ < > = <= >= or ~=."

  ^ self operationSelectors includes: queryOp
%
category: 'private'
method: GsQueryPredicate
_pathFrom: linksArray paths: pathsArray offset: offset
  | nextLink path i linkSize term |
  path := 'each'.
  offset == 0
    ifTrue: [ ^ path ].
  i := offset * 2 - 1.
  linkSize := linksArray size.
  [ i < linkSize ]
    whileTrue: [ 
      nextLink == 0
        ifTrue: [ ^ path ].
      nextLink := linksArray at: i.
      term := (pathsArray at: (linksArray at: i + 1)) asString.
      path := path , '.' , term.
      i := nextLink * 2 - 1 ].
  ^ path
%
expectValue %Boolean
doit
GsQueryPredicate category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsRangeQueryPredicate
removeallmethods GsRangeQueryPredicate
removeallclassmethods GsRangeQueryPredicate

doit
GsRangeQueryPredicate comment: 
'The class GsRangeQueryPredicate is a predicate in a GsQueryFormula. This class is a concrete subclass of GsAbstractQueryPredicate and applies two operators, two constant operands and a path operand.

The constant operands may be a literal (instance of GsConstantReferenceAssociation) or a variable (instance of GsVariableReferenceAssociation).

The path operand is expected to be an instance of a GsQueryPathReferenceAssociation.

Each operator may be one of the range comparison operators: < > <= >=. To qualify as a normalized GsRangeQueryPredicate, if one of the operators is < or <= then the other operator must also be  < or <=, alternately, if one of the operators is > or >= then the other operator must also be  > or >=. '.
true
%

! ------------------- Class methods for GsRangeQueryPredicate
category: 'instance creation'
classmethod: GsRangeQueryPredicate
constant: constant1 operator: operator1 path: path operator: operator2 constant: constant2
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (self new
    constant1: constant1;
    operator1: operator1;
    path: path;
    operator2: operator2;
    constant2: constant2;
    yourself) validate
%
category: 'instance creation'
classmethod: GsRangeQueryPredicate
constant: constant1 operator: operator1 path: path operator: operator2 variable: variable2
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (self new
    constant1: constant1;
    operator1: operator1;
    path: path;
    operator2: operator2;
    variable2: variable2;
    yourself) validate
%
category: 'instance creation'
classmethod: GsRangeQueryPredicate
operand: operand1 operator: operator1 path: path operator: operator2 operand: operand2
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (self new
    operand1: operand1;
    operator1: operator1;
    path: path;
    operator2: operator2;
    operand2: operand2;
    yourself) validate
%
category: 'instance creation'
classmethod: GsRangeQueryPredicate
variable: variable1 operator: operator1 path: path operator: operator2 constant: constant2
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (self new
    variable1: variable1;
    operator1: operator1;
    path: path;
    operator2: operator2;
    constant2: constant2;
    yourself) validate
%
category: 'instance creation'
classmethod: GsRangeQueryPredicate
variable: variable1 operator: operator1 path: path operator: operator2 variable: variable2
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (self new
    variable1: variable1;
    operator1: operator1;
    path: path;
    operator2: operator2;
    variable2: variable2;
    yourself) validate
%
! ------------------- Instance methods for GsRangeQueryPredicate
category: 'visiting'
method: GsRangeQueryPredicate
acceptVisitor: aFormulaVisitor
  super acceptVisitor: aFormulaVisitor.
  aFormulaVisitor acceptRangePredicate: self
%
category: 'optimizing'
method: GsRangeQueryPredicate
applyDeMorgansNegationTransform
  "actively apply De Morgan's laws ... convert to | of two negated predicates"

  | theConstant1 theConstant2 theOperator1 theOperator2 predicate1 predicate2 |
  self isNormal
    ifTrue: [ 
      theConstant1 := self constant1.
      theOperator1 := self operator1.
      theConstant2 := self constant2.
      theOperator2 := self operator2 ]
    ifFalse: [ 
      theConstant1 := self constant2.
      theOperator1 := self inverseOperatorFor: self operator2.
      theConstant2 := self constant1.
      theOperator2 := self inverseOperatorFor: self operator1 ].
  predicate1 := GsQueryPredicate
    constant: theConstant1
    operator: (self negatedOperatorFor: theOperator1)
    path: self path.
  predicate2 := GsQueryPredicate
    path: self path
    operator: (self negatedOperatorFor: theOperator2)
    constant: theConstant2.
  ^ predicate1 | predicate2
%
category: 'transforming'
method: GsRangeQueryPredicate
bind: variableName to: value
  | bound |
  bound := self copy.
  bound operand1: (bound operand1 bind: variableName to: value).
  bound operand2: (bound operand2 bind: variableName to: value).
  ^ bound immediateInvariant
%
category: 'private'
method: GsRangeQueryPredicate
bindEvaluators
  evaluator := self
    _bindEvaluator: nsc
    for: self path
    isRangeEqualityOperation: true.
  super bindEvaluators
%
category: 'accessing'
method: GsRangeQueryPredicate
constant1
  ^ operand1 _idxValue
%
category: 'accessing'
method: GsRangeQueryPredicate
constant1: anObject
  operand1 := GsConstantReferenceAssociation
    newWithKey: anObject asString
    value: anObject
%
category: 'accessing'
method: GsRangeQueryPredicate
constant2
  ^ operand2 _idxValue
%
category: 'accessing'
method: GsRangeQueryPredicate
constant2: anObject
  operand2 := GsConstantReferenceAssociation
    newWithKey: anObject asString
    value: anObject
%
category: 'querying'
method: GsRangeQueryPredicate
elementValueEvaluator
  "companion method to #elementValue:"

  "anObject is the value obtained by traversing path terms to the last value ... analgous to the last element class"

  "not all subclasses are required to implement this selector, if they do not implement this selector, then they must implement #elementValue:"

  ^ (evaluator == nil  or: [ evaluator isPathEvaluator not ])
    ifTrue: [ self pathEvaluatorFor: self path ]
    ifFalse: [ evaluator ]
%
category: 'accessing'
method: GsRangeQueryPredicate
evaluator
  ^ evaluator
%
category: 'accessing'
method: GsRangeQueryPredicate
evaluator: anEvaluator
  evaluator := anEvaluator
%
category: 'querying'
method: GsRangeQueryPredicate
executeAndDo: aBlock
  | queryEvaluator |
  queryEvaluator := self evaluator asQueryEvaluator.
  queryEvaluator doBlock: aBlock.
  ^ self executeClause: queryEvaluator
%
category: 'querying-private'
method: GsRangeQueryPredicate
executeClause
  | queryEvaluator |
  queryEvaluator := self evaluator asQueryEvaluator.
  ^ self executeClause: queryEvaluator
%
category: 'querying-private'
method: GsRangeQueryPredicate
executeClauseNegated
  "Deoptimize the range predicate and propogate executeClauseNegated"

  | theConstant1 theConstant2 theOperator1 theOperator2 predicate1 predicate2 |
  self isNormal
    ifTrue: [ 
      theConstant1 := self constant1.
      theOperator1 := self operator1.
      theConstant2 := self constant2.
      theOperator2 := self operator2 ]
    ifFalse: [ 
      theConstant1 := self constant2.
      theOperator1 := self inverseOperatorFor: self operator2.
      theConstant2 := self constant1.
      theOperator2 := self inverseOperatorFor: self operator1 ].
  predicate1 := (GsQueryPredicate
    constant: theConstant1
    operator: theOperator1
    path: self path) bindEvaluatorsFor: nsc collator: self collator.
  predicate2 := (GsQueryPredicate
    path: self path
    operator: theOperator2
    constant: theConstant2) bindEvaluatorsFor: nsc collator: self collator.
  ^ (predicate1 & predicate2) executeClauseNegated
%
category: 'querying-private'
method: GsRangeQueryPredicate
executeNegatedAndDo: aBlock
  "Deoptimize the range predicate and propogate executeClauseNegated"

  | theConstant1 theConstant2 theOperator1 theOperator2 predicate1 predicate2 |
  self isNormal
    ifTrue: [ 
      theConstant1 := self constant1.
      theOperator1 := self operator1.
      theConstant2 := self constant2.
      theOperator2 := self operator2 ]
    ifFalse: [ 
      theConstant1 := self constant2.
      theOperator1 := self inverseOperatorFor: self operator2.
      theConstant2 := self constant1.
      theOperator2 := self inverseOperatorFor: self operator1 ].
  predicate1 := (GsQueryPredicate
    constant: theConstant1
    operator: theOperator1
    path: self path) bindEvaluatorsFor: nsc collator: self collator.
  predicate2 := (GsQueryPredicate
    path: self path
    operator: theOperator2
    constant: theConstant2) bindEvaluatorsFor: nsc collator: self collator.
  ^ predicate1 & predicate2 executeNegatedAndDo: aBlock
%
category: 'private'
method: GsRangeQueryPredicate
immediateInvariant
  super immediateInvariant.
  operand1 immediateInvariant.
  operand2 immediateInvariant
%
category: 'testing'
method: GsRangeQueryPredicate
isNormal
  " operator1  and operator2 must be < or <= "

  ^ (self operator1 == #'<' or: [ self operator1 == #'<=' ])
    and: [ ^ self operator2 == #'<' or: [ self operator2 == #'<=' ] ]
%
category: 'querying-private'
method: GsRangeQueryPredicate
lastElementValue: anObject
  "anObject is the value obtained by traversing path terms to the last value ... analgous to the last element class"

  | selector1 selector2 |
  self isNormal
    ifTrue: [ 
      selector1 := self comparisonSelectorFor: self operator1.
      selector2 := self comparisonSelectorFor: self operator2 ]
    ifFalse: [ 
      selector1 := self
        comparisonSelectorFor: (self inverseOperatorFor: self operator1).
      selector2 := self
        comparisonSelectorFor: (self inverseOperatorFor: self operator2) ].
  ^ (self constant1 perform: selector1 with: anObject)
    & (anObject perform: selector2 with: self constant2)
%
category: 'accessing'
method: GsRangeQueryPredicate
operand1
  ^ operand1
%
category: 'accessing'
method: GsRangeQueryPredicate
operand1: anObject
  operand1 := anObject
%
category: 'accessing'
method: GsRangeQueryPredicate
operand2
  ^ operand2
%
category: 'accessing'
method: GsRangeQueryPredicate
operand2: anObject
  operand2 := anObject
%
category: 'accessing'
method: GsRangeQueryPredicate
operator1

   "Return the value of the instance variable 'operator1'."
   ^operator1
%
category: 'accessing'
method: GsRangeQueryPredicate
operator1: newValue

   "Modify the value of the instance variable 'operator1'."
   operator1 := newValue
%
category: 'accessing'
method: GsRangeQueryPredicate
operator2

   "Return the value of the instance variable 'operator2'."
   ^operator2
%
category: 'accessing'
method: GsRangeQueryPredicate
operator2: newValue

   "Modify the value of the instance variable 'operator2'."
   operator2 := newValue
%
category: 'accessing'
method: GsRangeQueryPredicate
path
  "don't like that we're obscuring instance variable reference this way ... rename instance variable"

  ^ path key
%
category: 'accessing'
method: GsRangeQueryPredicate
path: aPathString
  path := self pathReferenceFor: aPathString
%
category: 'private'
method: GsRangeQueryPredicate
postCopy
  operand1 := operand1 copy.
  path := path copy.
  operand2 := operand2 copy.
  super postCopy
%
category: 'printing'
method: GsRangeQueryPredicate
printOn: aStream
  aStream
    nextPutAll:
      '(' , self operand1 printString , ' ' , self operator1 asString , ' '
        , self rawPath printString , ' ' , self operator2 asString , ' '
        , self operand2 printString , ')'
%
category: 'accessing'
method: GsRangeQueryPredicate
rawPath
  "don't like that we're obscuring instance variable reference this way ... rename instance variable"

  ^ path
%
category: 'transforming'
method: GsRangeQueryPredicate
unbind
  "remove all bindings"

  | unbound |
  unbound := super unbind.
  unbound operand1: unbound operand1 unbind.
  unbound operand2: unbound operand2 unbind.
  unbound evaluator: nil.
  ^ unbound immediateInvariant
%
category: 'testing'
method: GsRangeQueryPredicate
usesPathEvaluator
  self evaluator ifNil: [ ^ true ].
  ^ self evaluator isPathEvaluator
%
category: 'private'
method: GsRangeQueryPredicate
validate
  " operator1 and operator2 must be < or <= 
      OR
	 operator1 and operator2 must be > or >= "

  self isNormal
    ifTrue: [ ^ self ].
  ((self operator1 == #'<' or: [ self operator1 == #'<=' ])
    and: [ self operator2 == #'<' or: [ self operator2 == #'<=' ] ])
    ifTrue: [ ^ self ].
  ((self operator1 == #'>' or: [ self operator1 == #'>=' ])
    and: [ self operator2 == #'>' or: [ self operator2 == #'>=' ] ])
    ifTrue: [ ^ self ].
  ^ (GsQueryPredicate
    operand: self operand1
    operator: self operator1
    operand: self rawPath)
    &
      (GsQueryPredicate
        operand: self rawPath
        operator: self operator2
        operand: self operand2)
%
category: 'accessing'
method: GsRangeQueryPredicate
variable1: aVariableName
  operand1 := self variableReferenceFor: aVariableName
%
category: 'accessing'
method: GsRangeQueryPredicate
variable2: aVariableName
  operand2 := self variableReferenceFor: aVariableName
%
expectValue %Boolean
doit
GsRangeQueryPredicate category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsUnaryClause
removeallmethods GsUnaryClause
removeallclassmethods GsUnaryClause

doit
GsUnaryClause comment: 
'The class GsUnaryClause is a predicate in a GsQueryFormula. This class applies the unary operator #not to a clause or a predicate. Each clause or predicate is expected to be a subclass of GsQueryFormula.'.
true
%

! ------------------- Class methods for GsUnaryClause
category: 'instance creation'
classmethod: GsUnaryClause
clause: clause operator: anOperator
  ^ self new
    clause: clause;
    operator: anOperator;
    yourself
%
! ------------------- Instance methods for GsUnaryClause
category: 'visiting'
method: GsUnaryClause
acceptVisitor: aFormulaVisitor
  super acceptVisitor: aFormulaVisitor.
  aFormulaVisitor acceptUnaryClause: self
%
category: 'optimizing'
method: GsUnaryClause
applyDeMorgansNegationTransform
  "actively apply De Morgan's laws ... operating on copy"

  | transformedClause |
  transformedClause := self operator == #'not'
    ifTrue: [ self clause applyDeMorgansTransform ]
    ifFalse: [ self clause applyDeMorgansNegationTransform ].
  self
    operator: #'null';
    clause: transformedClause.
  ^ self
%
category: 'optimizing'
method: GsUnaryClause
applyDeMorgansTransform
  "do not transform, but propagate the transform ... operating on copy"

  | transformedClause |
  transformedClause := self operator == #'not'
    ifTrue: [ self clause applyDeMorgansNegationTransform ]
    ifFalse: [ self clause applyDeMorgansTransform ].
  self
    operator: #'null';
    clause: transformedClause.
  ^ self
%
category: 'converting'
method: GsUnaryClause
asFormulaWithSelectorParts: selectorParts arguments: arguments
  | selector |
  selector := selectorParts first inputValue asSymbol.
  ^ self perform: selector with: arguments first
%
category: 'transforming'
method: GsUnaryClause
bind: variableName to: value
  | bound |
  bound := self copy.
  bound clause: (bound clause bind: variableName to: value).
  ^ bound immediateInvariant
%
category: 'private'
method: GsUnaryClause
bindEvaluators
  clause := clause bindEvaluatorsFor: nsc collator: self collator.
  super bindEvaluators
%
category: 'accessing'
method: GsUnaryClause
clause
  ^ clause
%
category: 'accessing'
method: GsUnaryClause
clause: newValue
  clause := newValue
%
category: 'querying-private'
method: GsUnaryClause
elementValue: anObject
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  | result |
  result := self clause elementValue: anObject.
  self operator == #'null'
    ifTrue: [ ^ result ].
  ^ result not
%
category: 'querying'
method: GsUnaryClause
executeAndDo: aBlock
  ^ self operator == #'not'
    ifTrue: [ self clause executeNegatedAndDo: aBlock ]
    ifFalse: [ self clause executeAndDo: aBlock ]
%
category: 'querying-private'
method: GsUnaryClause
executeClause
  ^ self operator == #'not'
    ifTrue: [ self clause executeClauseNegated ]
    ifFalse: [ self clause executeClause ]
%
category: 'querying-private'
method: GsUnaryClause
executeClauseNegated
  ^ self operator == #'not'
    ifTrue: [ self clause executeClause ]
    ifFalse: [ self clause executeClauseNegated ]
%
category: 'querying-private'
method: GsUnaryClause
executeNegatedAndDo: aBlock
  ^ self operator == #'not'
    ifTrue: [ self clause executeAndDo: aBlock ]
    ifFalse: [ self clause executeNegatedAndDo: aBlock ]
%
category: 'private'
method: GsUnaryClause
immediateInvariant
  super immediateInvariant.
  clause immediateInvariant
%
category: 'testing'
method: GsUnaryClause
isConjunctiveNormalForm
  ^ self operator == #'null' and: [ self clause isConjunctiveNormalForm ]
%
category: 'testing'
method: GsUnaryClause
isNegationClause
  ^ self operator == #'not'
%
category: 'operators'
method: GsUnaryClause
normalize
  | normalized |
  normalized := super normalize.
  normalized clause: clause normalize
%
category: 'accessing'
method: GsUnaryClause
operator
  ^operator
%
category: 'accessing'
method: GsUnaryClause
operator: aSymbol
  operator := aSymbol
%
category: 'private'
method: GsUnaryClause
postCopy
  clause := clause copy.
  super postCopy
%
category: 'printing'
method: GsUnaryClause
printOn: aStream
  | parenthesizeClause |
  parenthesizeClause := self clause isCompoundPredicate
    and: [ self operator ~~ #'null' ].
  parenthesizeClause
    ifTrue: [ aStream nextPut: $( ].
  aStream nextPutAll: self clause printString.
  parenthesizeClause
    ifTrue: [ aStream nextPut: $) ].
  self operator ~~ #'null'
    ifTrue: [ aStream nextPutAll: ' ' , self operator asString ]
%
category: 'transforming'
method: GsUnaryClause
unbind
  "remove all bindings"

  | unbound |
  unbound := super unbind.
  unbound clause: unbound clause unbind.
  ^ unbound immediateInvariant
%
expectValue %Boolean
doit
GsUnaryClause category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsMalformedQueryExpressionError
removeallmethods GsMalformedQueryExpressionError
removeallclassmethods GsMalformedQueryExpressionError

doit
GsMalformedQueryExpressionError comment: 
'The class GsMalformedQueryExpressionError is signaled when an attempt is made to execute a malformed GsQueryFormula.'.
true
%

! ------------------- Class methods for GsMalformedQueryExpressionError
! ------------------- Instance methods for GsMalformedQueryExpressionError
expectValue %Boolean
doit
GsMalformedQueryExpressionError category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryConjoinSetChecker
removeallmethods GsQueryConjoinSetChecker
removeallclassmethods GsQueryConjoinSetChecker

doit
GsQueryConjoinSetChecker comment:
'GsQueryConjoinSetChecker is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsQueryConjoinSetChecker
! ------------------- Instance methods for GsQueryConjoinSetChecker
expectValue %Boolean
doit
GsQueryConjoinSetChecker category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsStreamableQueryChecker
removeallmethods GsStreamableQueryChecker
removeallclassmethods GsStreamableQueryChecker

doit
GsStreamableQueryChecker comment:
'GsStreamableQueryChecker is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsStreamableQueryChecker
! ------------------- Instance methods for GsStreamableQueryChecker
expectValue %Boolean
doit
GsStreamableQueryChecker category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsStreamableConjunctiveClauseChecker
removeallmethods GsStreamableConjunctiveClauseChecker
removeallclassmethods GsStreamableConjunctiveClauseChecker

doit
GsStreamableConjunctiveClauseChecker comment:
'GsStreamableConjunctiveClauseChecker is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsStreamableConjunctiveClauseChecker
! ------------------- Instance methods for GsStreamableConjunctiveClauseChecker
expectValue %Boolean
doit
GsStreamableConjunctiveClauseChecker category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from GsUnsatisfiableQueryNotification
removeallmethods GsUnsatisfiableQueryNotification
removeallclassmethods GsUnsatisfiableQueryNotification

doit
GsUnsatisfiableQueryNotification comment: 
'The class GsUnsatisfiableQueryNotification is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsUnsatisfiableQueryNotification
! ------------------- Instance methods for GsUnsatisfiableQueryNotification
expectValue %Boolean
doit
GsUnsatisfiableQueryNotification category: 'Index-Query'.
true
%
category: 'Instance initialization'
method: GsUnsatisfiableQueryNotification
initialize
  gsNumber := ERR_GsUnsatisfiableQueryNotification .
  gsResumable := true .
  gsTrappable := true .
%


! --------------------------------------------------------------------
! Remove existing behavior from GsVariableReferenceAssociation
removeallmethods GsVariableReferenceAssociation
removeallclassmethods GsVariableReferenceAssociation

doit
GsVariableReferenceAssociation comment: 
'The class GsVariableReferenceAssociation  represents the value of a variable operand a GsAbstractQueryPredicate.'.
true
%

! ------------------- Class methods for GsVariableReferenceAssociation
! ------------------- Instance methods for GsVariableReferenceAssociation
category: 'converting'
method: GsVariableReferenceAssociation
asFormula
  ^ GsConstantConstantPredicate new
    constant1: self;
    operator: #'unary';
    immediateInvariant
%
category: 'accessing'
method: GsVariableReferenceAssociation
bind: variableName to: anObject
  variableName = self key
    ifFalse: [ ^ self ].
  bound := true.
  self value: anObject
%
category: 'accessing'
method: GsVariableReferenceAssociation
bound
  bound ifNil: [ ^ false ].
  ^ bound
%
category: 'testing'
method: GsVariableReferenceAssociation
isBound
  ^ self bound
%
category: 'accessing'
method: GsVariableReferenceAssociation
unbind
  "remove all bindings"

  bound := false.
  self value: nil
%
category: 'accessing'
method: GsVariableReferenceAssociation
value
  self bound
    ifFalse: [ self error: 'Variable: ' , self key printString , ' not bound' ].
  ^ super value
%
expectValue %Boolean
doit
GsVariableReferenceAssociation category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from OptionalTermPathEvaluator
removeallmethods OptionalTermPathEvaluator
removeallclassmethods OptionalTermPathEvaluator

doit
OptionalTermPathEvaluator comment:
'OptionalTermPathEvaluator is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for OptionalTermPathEvaluator
! ------------------- Instance methods for OptionalTermPathEvaluator
category: 'Traversing'
method: OptionalTermPathEvaluator
_nextObjectFor: anObject at: offset offsets: offsets classes: classes
  "Returns the next object along the path in anObject for the named instance
 variable at the given offset in the receiver."

  | ivOffset |
  (classes at: offset) ~~ anObject class
    ifTrue: [ 
      ivOffset := anObject class _ivOffsetOf: (self at: offset).
      ivOffset == nil
        ifTrue: [ ^ #'_incompletePathTraversal' ].
      classes at: offset put: anObject class.
      offsets at: offset put: ivOffset ]
    ifFalse: [ ivOffset := offsets at: offset ].
  ivOffset > anObject _primitiveSize
    ifTrue: [ ^ #'_incompletePathTraversal' ].
  ^ anObject instVarAt: ivOffset
%
expectValue %Boolean
doit
OptionalTermPathEvaluator category: 'Index-Evaluators'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from SelectorPathEvaluator
removeallmethods SelectorPathEvaluator
removeallclassmethods SelectorPathEvaluator

doit
SelectorPathEvaluator comment:
'SelectorPathEvaluator is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for SelectorPathEvaluator
! ------------------- Instance methods for SelectorPathEvaluator
category: 'Testing'
method: SelectorPathEvaluator
hasSelectorTerm
  "Returns true if the path has a term that indicates a selector."

  ^ false
%
category: 'Private'
method: SelectorPathEvaluator
termSelector
  "termSelector is always the last term"
  | term |
  term := self at: self size.
  ^ term copyFrom: 2 to: term size
%
category: 'Traversing'
method: SelectorPathEvaluator
_traverse: anObject cachedOffsets: offsets cachedClasses: classes incomplete: incompletePathTraversal
  | nextObj sz |
  nextObj := anObject.
  sz := self size.
  1 to: sz do: [ :i | 
    | pathName |
    pathName := self at: i.
    (pathName size > 0 and: [ (pathName at: 1) == $# ])
      ifTrue: [ nextObj := nextObj perform: self termSelector ]
      ifFalse: [ 
        offsets == nil
          ifTrue: [ nextObj := self _nextObjectFor: nextObj at: i ]
          ifFalse: [ 
            nextObj := self
              _nextObjectFor: nextObj
              at: i
              offsets: offsets
              classes: classes ] ].
    (nil == nextObj and: [ i ~~ sz ])
      ifTrue: [ ^ #'_incompletePathTraversal' ].
    nextObj == #'_incompletePathTraversal'
      ifTrue: [ ^ nextObj ] ].
  ^ nextObj
%
category: 'Traversing'
method: SelectorPathEvaluator
_traverseObject: anObject incompletesInto: incompleteArray incomplete: incomplete
  "Traverse the sub-objects of the given object using the path names of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, place the object
 into the appropriate Array in the incompleteArray of Arrays."

  "Intended for support of PathSort ... no collection-based terms"

  | nextObj sz |
  nextObj := anObject.
  sz := self size.
  1 to: sz do: [ :i | 
    | pathName |
    pathName := self at: i.
    (pathName size > 0 and: [ (pathName at: 1) == $# ])
      ifTrue: [ nextObj := nextObj perform: self termSelector ]
      ifFalse: [ nextObj := self _nextObjectFor: nextObj at: i ].
    ((nil == nextObj and: [ i ~~ sz ])
      or: [ nextObj == #'_incompletePathTraversal' ])
      ifTrue: [ 
        incompleteArray == nil
          ifTrue: [ ^ nil ].
        (incompleteArray at: i + 1) add: anObject.
        ^ incomplete ] ].
  ^ nextObj
%

! --------------------------------------------------------------------
! Remove existing behavior from RcUnicodeIndexSpecification
removeallmethods RcUnicodeIndexSpecification
removeallclassmethods RcUnicodeIndexSpecification

doit
RcUnicodeIndexSpecification comment: 
'The class RcUnicodeIndexSpecification provides details for specifying an index that is 
reduced-conflict and contains Unicode Strings. Since it contains Unicode Strings, it requires 
a persistent IcuCollator instance associated with the index.'. 
true
%

! ------------------- Class methods for RcUnicodeIndexSpecification

category: 'instance creation'
classmethod: RcUnicodeIndexSpecification
path: aString lastElementClass: aClass
  ^ self shouldNotImplement: #'path:lastElementClass:'
%
! ------------------- Instance methods for RcUnicodeIndexSpecification

expectValue %Boolean
doit
RcUnicodeIndexSpecification category: 'Index-Query'.
true
%

! --------------------------------------------------------------------
! Remove existing behavior from UnicodeIndexSpecification
removeallmethods UnicodeIndexSpecification
removeallclassmethods UnicodeIndexSpecification

doit
UnicodeIndexSpecification comment: 
'The class UnicodeIndexSpecification provides details for specifying an index that contains 
Unicode Strings. Since it contains Unicode Strings, it requires a persistent IcuCollator instance 
associated with the index.' .

true
%

! ------------------- Class methods for UnicodeIndexSpecification
category: 'instance creation'
classmethod: UnicodeIndexSpecification
path: aString lastElementClass: aClass
  ^ self shouldNotImplement: #'path:lastElementClass:'
%
! ------------------- Instance methods for UnicodeIndexSpecification

expectValue %Boolean
doit
UnicodeIndexSpecification category: 'Index-Query'.
true
%

category: 'accessing'
method: UnicodeIndexSpecification
collator
  "Returns IcuCollator to be used when comparing Unicode strings"

  collator ifNil: [ collator := IcuCollator default copy immediateInvariant ].
  ^ collator
%
category: 'accessing'
method: UnicodeIndexSpecification
collator: anIcuCollator
  "Set the receiver's collator. Use a copy of anIcuCollator to disallow 
   changes to strength, etc, that might affect the sort ordering. "

  collator := anIcuCollator copy immediateInvariant
%

! --------------------------------------------------------------------
! Remove existing behavior from Gs32UnaryConstantPredicateOptimizer
removeallmethods Gs32UnaryConstantPredicateOptimizer
removeallclassmethods Gs32UnaryConstantPredicateOptimizer

doit
Gs32UnaryConstantPredicateOptimizer comment:
'Gs32UnaryConstantPredicateOptimizer is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

expectValue %Boolean
doit
Gs32UnaryConstantPredicateOptimizer category: 'Index-Query'.
true
%

! ------------------- Class methods for Gs32UnaryConstantPredicateOptimizer
! ------------------- Instance methods for Gs32UnaryConstantPredicateOptimizer

category: 'optimizing'
method: Gs32UnaryConstantPredicateOptimizer
optimizeFrom: aGs32Optimizer
  "Simplify compound clause involving predicates that are unary-constant (true or false) or 
   constant-constant. The predicate is removed, or the expression is consolidated."

  | truePredicate falsePredicate clause1 clause2 |
  clause1 := self formula clause1.
  clause2 := self formula clause2.
  truePredicate := falsePredicate := nil.
  (clause1 isConstantConstant and: [ clause1 isConstantBound ])
    ifTrue: [ 
      clause1 executePredicate
        ifTrue: [ 
          aGs32Optimizer transformed: true.
          ^ clause1 ]
        ifFalse: [ 
          aGs32Optimizer transformed: true.
          ^ clause2 ] ]
    ifFalse: [ 
      (clause2 isConstantConstant and: [ clause2 isConstantBound ])
        ifTrue: [ 
          clause2 executePredicate
            ifTrue: [ 
              aGs32Optimizer transformed: true.
              ^ clause2 ]
            ifFalse: [ 
              aGs32Optimizer transformed: true.
              ^ clause1 ] ] ].
  ^ self formula
%

! --------------------------------------------------------------------
! Remove existing behavior from GsQueryParseError
removeallmethods GsQueryParseError
removeallclassmethods GsQueryParseError

doit
GsQueryParseError comment:
'GsQueryParseError is a subclass of Error, signalled when the parsing of a GsQuery finds incorrect syntax'.
true
%

! ------------------- Class methods for GsQueryParseError
category: 'instance creation'
classmethod: GsQueryParseError
signal: errorMessage position: errorPosition
  ^ self new
    position: errorPosition;
    signal: errorMessage
%
! ------------------- Instance methods for GsQueryParseError
category: 'accessing'
method: GsQueryParseError
position
  ^ position
%
category: 'accessing'
method: GsQueryParseError
position: errorPosition
  position := errorPosition
%

! --------------------------------------------------------------------
! Remove existing behavior from GsConjunctiveClauseConjoinSetChecker
removeallmethods GsConjunctiveClauseConjoinSetChecker
removeallclassmethods GsConjunctiveClauseConjoinSetChecker

doit
GsConjunctiveClauseConjoinSetChecker comment:
'GsConjunctiveClauseConjoinSetChecker is part of the Indexing and Querying subsystem. It implements 
GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsConjunctiveClauseConjoinSetChecker
category: 'instance creation'
classmethod: GsConjunctiveClauseConjoinSetChecker
check: aFormula on: anNsc
  ^ self new
    formula: aFormula;
    nsc: anNsc;
    queryOptions: GsQueryOptions default;
    check
%

! ------------------- Instance methods for GsConjunctiveClauseConjoinSetChecker
category: 'visiting'
method: GsConjunctiveClauseConjoinSetChecker
acceptPredicate: aPredicate
  "collect all predicates which use comparison operators, are not range predicates and have
   a collection based term"

  (aPredicate usesComparisonOperation not or: [ aPredicate usesRangeOperation ])
    ifTrue: [ 
      "not exposed to bug43764"
      ^ self ].
  aPredicate _evaluators
    detect: [ :evaluator | evaluator hasCollectionBasedTerm ]
    ifNone: [ 
      "not exposed to bug43764"
      ^ self ].
  super acceptPredicate: aPredicate
%
category: 'checking'
method: GsConjunctiveClauseConjoinSetChecker
check
  "collect all predicates which use comparison operators, are not range predicates and have
   a collection based term in common with another predicate"

  | collectionBasedEvaluators commonPathTerms |
  self visitFormula: self formula.
  collectionBasedEvaluators := {}.
  self predicates
    do: [ :predicate | 
      collectionBasedEvaluators
        addAll:
          (predicate _evaluators select: [ :each | each hasCollectionBasedTerm ]) ].
  collectionBasedEvaluators size <= 1
    ifTrue: [ 
      "not exposed"
      ^ self ].
  1 to: collectionBasedEvaluators size do: [ :outerIndex | 
    | outerEvaluator |
    outerEvaluator := collectionBasedEvaluators at: outerIndex.
    1 to: collectionBasedEvaluators size do: [ :innerIndex | 
      | innerEvaluator |
      innerEvaluator := collectionBasedEvaluators at: innerIndex.
      innerIndex == outerIndex
        ifFalse: [ 
          "see if there are common terms"
          commonPathTerms := innerEvaluator _commonPathTermsWith: outerEvaluator.
          (commonPathTerms includes: #'*')
            ifTrue: [ 
              "Cannot evaluate a formula where two conjoined predicates are collection based"
              ^ self
                _errorMalformedQuery: commonPathTerms
                senderLocationID: #'check' ].
          commonPathTerms
            do: [ :term | 
              (term includes: $|)
                ifTrue: [ 
                  "Cannot evaluate a formula where two conjoined predicates are collection based"
                  ^ self
                    _errorMalformedQuery: commonPathTerms
                    senderLocationID: #'check' ] ] ] ] ]
%
category: 'private'
method: GsConjunctiveClauseConjoinSetChecker
_errorMalformedQuery: commonPathTerms senderLocationID: senderLocationID
  | str |
  str := commonPathTerms at: 1.
  2 to: commonPathTerms size do: [ :index | str := str , '.' , (commonPathTerms at: index) ].
  ^ GsMalformedQueryExpressionError
    bug43764MalformedQuery:
      str printString , ' in clause ' , self formula printString
    senderLocationID: senderLocationID
%


! ------------------- Instance methods for GsQueryExpectedImplicitIdentityIndexError 

doit
GsQueryExpectedImplicitIdentityIndexError comment:
'GsQueryExpectedImplicitIdentityIndexError is part of the Indexing and Querying subsystem.  
It is used where an implicit identity index that was a side-effect of legacy indexing does
not exist, since btreeplus indexes do not have that side-effect'.
true
%

category: 'Accessing'
method: GsQueryExpectedImplicitIdentityIndexError
gsQuery

   ^ gsQuery
%
category: 'Accessing'
method: GsQueryExpectedImplicitIdentityIndexError
gsQuery: anObject

   gsQuery := anObject
%
category: 'Instance initialization'
method: GsQueryExpectedImplicitIdentityIndexError
initialize
  super initialize.
  gsResumable := true .
%
category: 'Instance creation'
classmethod: GsQueryExpectedImplicitIdentityIndexError
signal: aGsQuery
  ^ self new
    gsQuery: aGsQuery;
    signal
%




!--- end of ordered methods




category: 'accessing'
method: RcUnicodeIndexSpecification
collator
  "Returns IcuCollator to be used when comparing Unicode strings"

  collator ifNil: [ collator := IcuCollator default copy immediateInvariant ].
  ^ collator
%
category: 'accessing'
method: RcUnicodeIndexSpecification
collator: anIcuCollator
  "Set the receiver's collator. Use a copy of anIcuCollator to disallow 
   changes to strength, etc, that might affect the sort ordering. "

  collator := anIcuCollator copy immediateInvariant
%
category: 'querying'
method: GsQuery
on: anNscOrSequenceableCollection collator: anIcuCollator
  "Bind the query to the given <anNsc> and <anIcuCollator>"

  self _validateQueryableCollection: anNscOrSequenceableCollection.
  self _nsc: anNscOrSequenceableCollection.
  self _collator: anIcuCollator.
  self _formula
    ifNotNil: [ 
      self
        _formula:
          (self _formula bindEvaluatorsFor: self _nsc collator: self _collator) ]
%
category: 'private'
method: GsQuery
_collator
  "Returns IcuCollator to be used when comparing Unicode strings"

  ^ collator
%
category: 'private'
method: GsQuery
_collator: anIcuCollator
  "Set the receiver's collator. Use a copy of anIcuCollator to disallow 
   changes to strength, etc, that might affect the sort ordering. "

  collator := anIcuCollator copy immediateInvariant
%
category: 'instance creation'
classmethod: RcUnicodeIndexSpecification
path: aString collator: anIcuCollator
  | res |
  (res := self new)
    path: aString;
    collator: anIcuCollator.
  ^ res
%
category: 'accessing'
method: GsQueryFormula
collator
  "Returns IcuCollator to be used when comparing Unicode strings"

  ^ collator
%
category: 'accessing'
method: GsQueryFormula
_nsc: anNsc collator: anIcuCollatorOrNil
  nsc := anNsc.
  collator := anIcuCollatorOrNil
%
category: 'accessing'
method: GsAbstractQueryOptimizer
collator
  ^ self formula collator
%
category: 'instance creation'
classmethod: UnicodeIndexSpecification
path: aString collator: anIcuCollator
  | res |
  (res := self new)
    path: aString;
    collator: anIcuCollator .
  ^ res
%
category: 'Traversing'
method: SelectorPathEvaluator
traverse: anObject andCompare: selector with: aValue
  "Traverse the given object and perform the comparison operation
 on the object at the end of the path and the given value."

  "This method assumes that the index is not over a set-valued instance variable."

  "This method is intended for support of QueryExecutor only"

  self shouldNotImplement: #'traverse:andCompare:with:'
%
category: 'Traversing'
method: SelectorPathEvaluator
traverse: anObject andCompare: selector withTraverser: aTraverser
  "Traverse the given object and perform the comparison operation on the object
 at the end of the path and the value obtained by using the traverser."

  "This method assumes that the index is not over a set-valued instance variable."

  "This method is intended for support of QueryExecutor only"

  self shouldNotImplement: #'traverse:andCompare:withTraverser:'
%
category: 'Traversing'
method: SelectorPathEvaluator
traverse: anObject startingAt: offset do: aBlock
  "Traverse the sub-objects of anObject, using the path names of the receiver starting at the path name at the given offset.  For each object at the end of
 the traversal, evaluate the given block."

  | nextObj ivOffset pathName sz |
  (nil == anObject _and: [ self isIndexOnNscElements not ])
    ifTrue: [ ^ self ].
  nextObj := anObject.
  sz := self size.
  offset to: sz do: [ :i | 
    " get the next object along the path "
    nil == nextObj
      ifTrue: [ ^ self ].
    pathName := self at: i.
    (pathName size > 0 and: [ (pathName at: 1) == $# ])
      ifTrue: [ nextObj := nextObj perform: self termSelector ]
      ifFalse: [ 
        ivOffset := nextObj class _ivOffsetOf: pathName.
        ivOffset == nil
          ifTrue: [ 
            nextObj _errorInvalidOffset: pathName.
            nextObj := nil ]
          ifFalse: [ nextObj := nextObj instVarAt: ivOffset ] ].
    (nil == nextObj _and: [ i ~= sz ])
      ifTrue: [ ^ self ] ].
  ^ aBlock value: nextObj
%
category: 'Testing'
method: CollectionBasedPathEvaluator
isCollectionBased
  "Returns true."

  ^ true
%
category: 'Traversing'
method: CollectionBasedPathEvaluator
traverse: anObject andCompare: selector with: aValue
  "Traverse the given object and perform the comparison operation
 on the object at the end of the path and the given value."

  "This method assumes that the index is not over a set-valued instance variable."

  "This method is intended for support of QueryExecutor only"

  self shouldNotImplement: #'traverse:andCompare:with:'
%
category: 'Traversing'
method: CollectionBasedPathEvaluator
traverse: anObject andCompare: selector withTraverser: aTraverser
  "Traverse the given object and perform the comparison operation on the object
 at the end of the path and the value obtained by using the traverser."

  "This method assumes that the index is not over a set-valued instance variable and is in support Of QueryExecutor only."

  "This method is intended for support of QueryExecutor only"

  self shouldNotImplement: #'traverse:andCompare:withTraverser:'
%
category: 'Traversing'
method: CollectionBasedPathEvaluator
traverse: anObject startingAt: offset do: aBlock
  "Traverse the sub-objects of anObject, using the path names of the receiver
 starting at the path name at the given offset.  Evaluate <aBlock> for each endObject."

  self subclassResponsibility: #'traverse:startingAt:do:'
%
category: 'Traversing'
method: CollectionBasedPathEvaluator
traverseObject: anObject
  "This method is intended for support of QueryExecutor only"

  self shouldNotImplement: #'traverseObject:'
%
category: 'Traversing'
method: CollectionBasedPathEvaluator
traverseObject: anObject incompletesInto: incompleteArray incomplete: incomplete
  "This method is intended for support of QueryExecutor only"

  self shouldNotImplement: #'traverseObject:incompletesInto:incomplete:'
%
category: 'Traversing'
method: CollectionBasedPathEvaluator
_traverse: anObject cachedOffsets: offsets cachedClasses: classes incomplete: incompletePathTraversal
  "This method assumes that the index is not over a set-valued instance variable."

  self shouldNotImplement: #'_traverse:cachedOffsets:cachedClasses:incomplete:'
%
category: 'Traversing'
method: CollectionBasedPathEvaluator
_traverseObject: anObject
  "Traverse the sub-objects of the given object using the path names of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversalObject."

  "This method is intended for support of QueryExecutor only"

  self shouldNotImplement: #'_traverseObject:'
%
category: 'Traversing'
method: CollectionBasedPathEvaluator
_traverseObject: anObject cache: ivOffsetCache
  "Traverse the sub-objects of the given object using the path names of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversalObject."

  "This method is intended for support of QueryExecutor only"

  self shouldNotImplement: #'_traverseObject:cache:'
%
category: 'Traversing'
method: CollectionBasedPathEvaluator
_traverseObject: anObject incompletesInto: incompleteArray incomplete: incomplete
  "This method assumes that the index is not over a set-valued instance variable."

  "Intended for support of PathSort ... no collection-based terms"

  self shouldNotImplement: #'_traverse:incompletesInto:incomplete:'
%
category: 'Traversing'
method: EnumeratedPathEvaluator
traverse: anObject startingAt: offset do: aBlock
  "Traverse the sub-objects of anObject, using the path names of the receiver
 starting at the path name at the given offset.  For each object at the end of
 the traversal, evaluate the given block."

  | nextObj ivOffset pathName nscOffset sz |
  (nil == anObject _and: [ self isIndexOnNscElements not ])
    ifTrue: [ ^ self ].
  nextObj := anObject.
  sz := self size.
  offset to: sz do: [ :i | 
    pathName := self at: i.
    (pathName includes: $|)
      ifTrue: [ 
        nscOffset := i + 1.
        (self enumeratedObjectsFor: nextObj pathTerm: pathName at: nscOffset)
          do: [ :obj | self traverse: obj startingAt: nscOffset do: aBlock ].
        ^ self ]
      ifFalse: [ 
        nil == nextObj
          ifTrue: [ ^ self ].
        ivOffset := nextObj class _ivOffsetOf: pathName.
        ivOffset == nil
          ifTrue: [ nextObj := nil ]
          ifFalse: [ nextObj := nextObj instVarAt: ivOffset ] ].
    (nil == nextObj _and: [ i ~= sz ])
      ifTrue: [ ^ self ] ].
  ^ aBlock value: nextObj
%

category: 'accessing'
method: GsPathPathPredicate
bagEnumeratorFor: anOperator
  | op enumerator |
  op := self comparisonSelectorFor: anOperator.
  (self collator == nil or: [ op == #'==' or: [ op == #'~~' ] ])
    ifTrue: [ 
      enumerator := PathPathBagEnumerator new
        selector: op .
        ]
    ifFalse: [ 
      enumerator := PathPathBagUnicodeEnumerator new
        selector: op , #'collator:';
        collator: self collator .
        ].
  enumerator
    evaluator: self evaluator1;
    evaluator2: self evaluator2 .
  ^ enumerator
%
category: 'querying-private'
method: GsConstantPathPredicate
executeClauseUsing: anOperator queryEvaluator: queryEvaluator
  | invertedOperator |
  invertedOperator := self inverseOperatorFor: anOperator.
  self _check: self evaluator1 compare: invertedOperator value: self constant.
  (invertedOperator == #'<' or: [ invertedOperator == #'<=' ])
    ifTrue: [ 
      (self constant == nil and: [ invertedOperator == #'<=' ])
        ifTrue: [ ^ queryEvaluator findAllValuesEqualTo: nil ]
        ifFalse: [ 
          ^ queryEvaluator
            findAllValuesLessThanKey: self constant
            andEquals: invertedOperator == #'<=' ] ].
  (invertedOperator == #'>' or: [ invertedOperator == #'>=' ])
    ifTrue: [ 
      (self constant == nil and: [ invertedOperator == #'>=' ])
        ifTrue: [ ^ queryEvaluator findAllValuesEqualTo: nil ]
        ifFalse: [ 
          ^ queryEvaluator
            findAllValuesGreaterThanKey: self constant
            andEquals: invertedOperator == #'>=' ] ].
  invertedOperator == #'='
    ifTrue: [ ^ queryEvaluator findAllValuesEqualTo: self constant ].
  invertedOperator == #'~='
    ifTrue: [ ^ queryEvaluator findAllValuesNotEqualTo: self constant ].
  invertedOperator == #'=='
    ifTrue: [ ^ queryEvaluator findAllValuesIdenticalTo: self constant ].
  invertedOperator == #'~~'
    ifTrue: [ ^ queryEvaluator findAllValuesNotIdenticalTo: self constant ]
%
category: 'querying-private'
method: GsPathConstantPredicate
executeClauseUsing: anOperator queryEvaluator: queryEvaluator
  | theOperator theConstant |
  theOperator := anOperator.
  theConstant := self constant.
  theOperator == #'unary'
    ifTrue: [ 
      theOperator := #'=='.
      theConstant := true ].
  theOperator == #'unaryNot'
    ifTrue: [ 
      theOperator := #'~~'.
      theConstant := true ].
  self _check: self evaluator1 compare: theOperator value: theConstant.
  (theOperator == #'<' or: [ theOperator == #'<=' ])
    ifTrue: [ 
      (theConstant == nil and: [ theOperator == #'<=' ])
        ifTrue: [ ^ queryEvaluator findAllValuesEqualTo: nil ]
        ifFalse: [ 
          ^ queryEvaluator
            findAllValuesLessThanKey: theConstant
            andEquals: theOperator == #'<=' ] ].
  (theOperator == #'>' or: [ theOperator == #'>=' ])
    ifTrue: [ 
      (theConstant == nil and: [ theOperator == #'>=' ])
        ifTrue: [ ^ queryEvaluator findAllValuesEqualTo: nil ]
        ifFalse: [ 
          ^ queryEvaluator
            findAllValuesGreaterThanKey: theConstant
            andEquals: theOperator == #'>=' ] ].
  theOperator == #'='
    ifTrue: [ ^ queryEvaluator findAllValuesEqualTo: theConstant ].
  theOperator == #'~='
    ifTrue: [ ^ queryEvaluator findAllValuesNotEqualTo: theConstant ].
  theOperator == #'=='
    ifTrue: [ ^ queryEvaluator findAllValuesIdenticalTo: theConstant ].
  theOperator == #'~~'
    ifTrue: [ ^ queryEvaluator findAllValuesNotIdenticalTo: theConstant ]
%
category: 'Comparison'
method: GsQueryOptions
= aGsQueryOptions
  (aGsQueryOptions isKindOf: self class)
    ifFalse: [ ^ false ].
  ^ self applyDeMorgansLaws == aGsQueryOptions applyDeMorgansLaws
    and: [ 
      self autoOptimize == aGsQueryOptions autoOptimize
        and: [ 
          self cacheQueryResult == aGsQueryOptions cacheQueryResult
            and: [ 
              self consolidateEnumerablePredicates
                == aGsQueryOptions consolidateEnumerablePredicates
                and: [ 
                  self consolidateRangePredicates == aGsQueryOptions consolidateRangePredicates
                    and: [ 
                      self normalizePredicates == aGsQueryOptions normalizePredicates
                        and: [ 
                          self removeRedundantPredicates == aGsQueryOptions removeRedundantPredicates
                            and: [ 
                              self reorderPredicates == aGsQueryOptions reorderPredicates
                                and: [ 
                                  self transformCommonPaths == aGsQueryOptions transformCommonPaths
                                    and: [ 
                                      self consolidateUnaryConstantPredicates
                                        ==
                                          aGsQueryOptions consolidateUnaryConstantPredicates
                                        and: [ self optionalPathTerms == aGsQueryOptions optionalPathTerms ] ] ] ] ] ] ] ] ] ]
%
category: 'Comparison'
method: GsQueryOptions
hash
  ^ self applyDeMorgansLaws hash
    bitXor:
      ((self autoOptimize hash bitShift: 1)
        bitXor:
          ((self cacheQueryResult hash bitShift: 2)
            bitXor:
              ((self consolidateEnumerablePredicates hash bitShift: 3)
                bitXor:
                  ((self consolidateRangePredicates hash bitShift: 4)
                    bitXor:
                      ((self normalizePredicates hash bitShift: 5)
                        bitXor:
                          ((self removeRedundantPredicates hash bitShift: 6)
                            bitXor:
                              ((self reorderPredicates hash bitShift: 7)
                                bitXor:
                                  ((self transformCommonPaths hash bitShift: 8)
                                    bitXor:
                                      (self consolidateUnaryConstantPredicates hash bitShift: 9)))))))))
%
category: 'querying-private'
method: GsRangeQueryPredicate
executeClause: queryEvaluator
  | theConstant1 theConstant2 theOperator1 theOperator2 validNilTransformOperators |
  self isNormal
    ifTrue: [ 
      theConstant1 := self constant1.
      theOperator1 := self operator1.
      theConstant2 := self constant2.
      theOperator2 := self operator2 ]
    ifFalse: [ 
      theConstant1 := self constant2.
      theOperator1 := self inverseOperatorFor: self operator2.
      theConstant2 := self constant1.
      theOperator2 := self inverseOperatorFor: self operator1 ].
  validNilTransformOperators := #(#'<=' #'>=').
  ((theConstant1 == nil and: [ theConstant2 == nil ])
    and: [ 
      (validNilTransformOperators includes: self operator1)
        and: [ validNilTransformOperators includes: self operator2 ] ])
    ifTrue: [ ^ queryEvaluator findAllValuesEqualTo: nil ].
  ((theConstant1 == nil and: [ theConstant2 ~~ nil ])
    and: [ validNilTransformOperators includes: self operator1 ])
    ifTrue: [ theOperator1 := #'=' ].
  ((theConstant1 ~~ nil and: [ theConstant2 == nil ])
    and: [ validNilTransformOperators includes: self operator2 ])
    ifTrue: [ theOperator2 := #'=' ].
  ^ queryEvaluator
    findAllValuesGreaterThan: theConstant1
    andEquals: theOperator1 == #'<='
    andLessThan: theConstant2
    andEquals: theOperator2 == #'<='
%
category: 'instance creation'
classmethod: GsQueryPredicate
_createPredicateFromQueryExecuter: aQueryExecuter startingAt: offset
  | firstOperand secondOperand searchOp predicateType predicate operator |
  firstOperand := aQueryExecuter firstOperandAt: offset.
  searchOp := aQueryExecuter operatorAt: offset.
  secondOperand := aQueryExecuter secondOperandAt: offset.
  predicateType := aQueryExecuter at: offset.
  searchOp == 8
    ifTrue: [ 
      predicate := (GsQueryPredicate constant: secondOperand) copy.
      predicate _nsc: aQueryExecuter nsc collator: nil.
      ^ predicate ].
  searchOp == 9
    ifTrue: [ 
      | searchOp1 searchOp2 |
      predicateType ~~ 2
        ifTrue: [ self error: 'invalid predicateType for searchOp ''9''' ].
      searchOp1 := secondOperand at: 1.
      searchOp2 := secondOperand at: 3.
      predicate := (GsQueryPredicate
        constant: (secondOperand at: 2)
        operator:
          (self inverseOperatorFor: (aQueryExecuter operationSelectors at: searchOp1 + 1))
        path: 'each.' , firstOperand pathComponentsString
        operator: (aQueryExecuter operationSelectors at: searchOp2 + 1)
        constant: (secondOperand at: 4)) copy.
      predicate
        _nsc: aQueryExecuter nsc collator: nil;
        evaluator: firstOperand.
      ^ predicate ].
  operator := aQueryExecuter operationSelectors at: searchOp + 1.
  predicateType == 2
    ifTrue: [ 
      predicate := (GsQueryPredicate
        path: 'each.' , firstOperand pathComponentsString
        operator: operator
        constant: secondOperand) copy.
      predicate
        _nsc: aQueryExecuter nsc collator: nil;
        evaluator1: firstOperand.
      ^ predicate ].
  predicateType == 1
    ifTrue: [ 
      predicate := (GsQueryPredicate
        constant: firstOperand
        operator: operator
        constant: secondOperand) copy.
      predicate _nsc: aQueryExecuter nsc collator: nil.
      ^ predicate ].
  predicateType == 4
    ifTrue: [ 
      predicate := (GsQueryPredicate
        path: 'each.' , firstOperand pathComponentsString
        operator: operator
        path: 'each.' , secondOperand pathComponentsString) copy.
      predicate
        _nsc: aQueryExecuter nsc collator: nil;
        evaluator1: firstOperand;
        evaluator2: secondOperand.
      ^ predicate ].
  self error: 'Unknown predicate type: ' , predicateType printString
%

! delete UnicodeIndexSpecification createIndexOn:
! delete RcUnicodeIndexSpecification createIndexOn:

category: 'accessing'
method: GsPathPathPredicate
elementValueEvaluator
  "Ambiguous ... elementValue: does not use this method"

  ^ self shouldNotImplement: #'elementValueEvaluator
'
%
category: 'accessing'
method: RcUnicodeIndexSpecification
_createIndex
  | index |
  index := self equalityIndexClass newWithCollator: self collator.
  self legacyIndex
    ifFalse: [ index options: self options ].
  ^ index
%
category: 'private'
method: RcUnicodeIndexSpecification
_validateLastElementClassOn: anNsc
  "noop"

  
%
category: 'accessing'
method: UnicodeIndexSpecification
_createIndex
  | index |
  index := self equalityIndexClass newWithCollator: self collator.
  self legacyIndex
    ifFalse: [ 
      index 
        options: self options; 
        constraintType: self constraintType ].
  ^ index
%
category: 'private'
method: UnicodeIndexSpecification
_validateLastElementClassOn: anNsc
  "noop"

  
%

category: 'Private'
method: GsIndexSpec
defaultCollator
  ^ IcuCollator default
%
category: 'accessing'
method: RcUnicodeIndexSpecification
indexTypePrintString
  ^ 'unicodeIndex'
%
category: 'accessing'
method: UnicodeIndexSpecification
indexTypePrintString
  ^ self constraintType
      ifNil: [ 'unicodeIndex' ]
      ifNotNil: [:constraint |
        "constraint  == #unicodeString"
        'unicodeStringOptimizedIndex' ]
%
category: 'Comparison'
method: GsIndexOptions
= aGsIndexOptions
  (aGsIndexOptions isKindOf: self class)
    ifFalse: [ ^ false ].
  ^ self reducedConflict = aGsIndexOptions reducedConflict
    and: [ self optionalPathTerms = aGsIndexOptions optionalPathTerms
        and: [ self legacyIndex = aGsIndexOptions legacyIndex
            and: [ self optimizedComparison = aGsIndexOptions optimizedComparison ] ] ]
%
category: 'Comparison'
method: GsIndexOptions
hash
  "Returns an Integer hash code for the receiver."

  ^ (((((((self btreePlusIndex hash bitShift: -1) 
    bitXor: self reducedConflict hash) bitShift: -1)
    bitXor: self optionalPathTerms hash) bitShift: -1) 
    bitXor: self legacyIndex hash) bitShift: -1) 
    bitXor: self optimizedComparison hash
%
category: 'testing'
method: GsQueryFormula
_canIndexSatifyPathTermsRequired: indexObj
  pathTermsRequired
    ifNil: [ 
      "not set ... use index if at all possible"
      ^ true ].
  pathTermsRequired
    ifTrue: [ 
      indexObj termsRequired
        ifTrue: [ ^ true ] ]
    ifFalse: [ 
      indexObj termsRequired
        ifFalse: [ ^ true ] ].
  ^ false
%
category: 'accessing'
method: GsQueryFormula
_pathTermsRequired
  ^ pathTermsRequired
%
category: 'accessing'
method: GsQueryFormula
_pathTermsRequired: aBoolOrNil
  "set to true, if you want to quarantee that path terms are required during query 
   evaluation. If an index is available, but the index uses optional path terms,
   then the query will use brute force evaluation."

  "set to false if you want to guaranatee that path terms are optional during query 
   evaluation. If an index is available, but the index uses required path terms,
   then the query will use brute force evaluation."

  "set to nil (default) if you do not care. Indexes will be used if available 
   otherwise brute force evaluation will be used with optional path terms."

  pathTermsRequired := aBoolOrNil
%
category: 'private'
method: GsPathPathPredicate
_evaluators
  "return list of evaluators associated with predicate"

  ^ {(self evaluator1).
  (self evaluator2)}
%
category: 'private'
method: GsUnaryClause
_predicateAndEvaluatorsDo: aBlock
  "visit all predicates in receiver and provide access to predicate and list of evaluators 
   for the predicate"

  super _predicateAndEvaluatorsDo: aBlock.
  self clause _predicateAndEvaluatorsDo: aBlock
%
category: 'testing'
method: GsQueryFormula
usesComparisonOperation
  ^ false
%
category: 'testing'
method: GsQueryFormula
usesRangeOperation
  ^ false
%
category: 'private'
method: GsQueryFormula
_checkForBug43764: aGsQuery
  "Bug 43764 describes a query pattern involving collection-valued path terms that produce 
   incorrect query results. See the 3.2 Release Notes for more detailed information."

  | queryOptions |
  self _exposedToBug43764
    ifFalse: [ ^ self	"not exposed" ].
  queryOptions := aGsQuery queryOptions.
  [ GsQueryConjoinSetChecker check: self on: aGsQuery _nsc ]
    on: GsMalformedQueryExpressionError
    do: [ :ex | 
      "checker will throw an error if the query is exposed to bug 43764"
      (queryOptions autoOptimize
        and: [ queryOptions applyDeMorgansLaws and: [ queryOptions consolidateRangePredicates ] ])
        ifTrue: [ 
          "exposed to bug ... pass the exception"
          ex pass ]
        ifFalse: [ 
          [ 
          | newQuery newQueryOptions |
          "determine if query is evaluable with proper options set"
          newQueryOptions := queryOptions + GsQueryOptions applyDeMorgansLaws
            + GsQueryOptions consolidateRangePredicates
            + GsQueryOptions autoOptimize.
          newQuery := aGsQuery copy.
          newQuery queryOptions: newQueryOptions.
          GsQueryConjoinSetChecker check: newQuery formula on: aGsQuery _nsc ]
            on: GsMalformedQueryExpressionError
            do: [ :ex2 | 
              "exposed to bug ... pass the exception"
              ex2 pass ] ].
      GsMalformedQueryExpressionError
        bug43764QueryMustBeOptimized: #'_checkForBug43764:' ].
  ^ self	"not exposed"
%
category: 'private'
method: GsQueryFormula
_evaluators
  "return list of evaluators associated with predicate"

  ^ #()
%
category: 'private'
method: GsQueryFormula
_exposedToBug43764
  "Bug 43764 describes a query pattern involving collection-valued path terms that produce 
   incorrect query results. See the 3.2 Release Notes for more detailed information."

  | exposed |
  exposed := false.
  self
    _predicateAndEvaluatorsDo: [ :predicate :evaluators | 
      (predicate usesComparisonOperation and: [ predicate usesRangeOperation not ])
        ifTrue: [ 
          "potentially exposed if predicate uses comparison operators, and is not a 
           range predicate"
          evaluators
            do: [ :evaluator | 
              evaluator hasCollectionBasedTerm
                ifTrue: [ 
                  exposed
                    ifTrue: [ 
                      "exposed if receiver has two or more collection based path terms"
                      ^ true ].
                  exposed := true ] ] ] ].
  ^ false
%
category: 'private'
method: GsQueryFormula
_predicateAndEvaluatorsDo: aBlock
  "visit all predicates in receiver and provide access to predicate and list of evaluators 
   for the predicate"

  aBlock value: self value: self _evaluators
%


!-----------GsQueryConjoinSetChecker


category: 'visiting'
method: GsQueryConjoinSetChecker
acceptCompoundClause: aCompoundClause
  "disjunctive clauses are not exposed to Bug 43764. Only conjunctive normal form 
   clauses need to be checked"

  super acceptCompoundClause: aCompoundClause.
  aCompoundClause clause1 isConjunctiveNormalForm
    ifTrue: [ 
      self checkConjunctiveClause: aCompoundClause clause1.
      aCompoundClause clause2 isConjunctiveNormalForm
        ifTrue: [ self checkConjunctiveClause: aCompoundClause clause2 ].
      ^ self ].
  aCompoundClause clause2 isConjunctiveNormalForm
    ifTrue: [ self checkConjunctiveClause: aCompoundClause clause2 ]
%
category: 'checking'
method: GsQueryConjoinSetChecker
check
  self formula isConjunctiveNormalForm
    ifTrue: [ self checkConjunctiveClause: self formula ]
    ifFalse: [ 
      formula isNegationClause
        ifTrue: [ 
          "If clause is  negated, the applyDeMorgansLaws has not been applied to transform the formula. Cannot correctly analyze presence of Bug 43764 optimized if applyDeMorgansLaws and consolidateRangePredicates options not set."
          ^ GsMalformedQueryExpressionError
            bug43764QueryCannotBeAnalyzed: #'check' ].
      self visitFormula: self formula ]
%
category: 'instance creation'
classmethod: GsQueryConjoinSetChecker
check: aFormula on: anNsc
  ^ self new
    formula: aFormula;
    nsc: anNsc;
    queryOptions: GsQueryOptions default;
    check
%
category: 'checking'
method: GsQueryConjoinSetChecker
checkConjunctiveClause: aClause
  GsConjunctiveClauseConjoinSetChecker check: aClause on: self nsc
%





category: 'private'
method: GsCompoundClause
_predicateAndEvaluatorsDo: aBlock
  "visit all predicates in receiver and provide access to predicate and list of evaluators 
   for the predicate"

  super _predicateAndEvaluatorsDo: aBlock.
  self clause1 _predicateAndEvaluatorsDo: aBlock.
  self clause2 _predicateAndEvaluatorsDo: aBlock
%
category: 'private'
method: GsConstantConstantPredicate
_evaluators
  "return list of evaluators associated with predicate"

  ^ #()
%
category: 'testing'
method: GsQueryPredicate
usesComparisonOperation
  ^ #(#'<=' #'<' #'>' #'>=') includes: operator
%
category: 'private'
method: GsQueryPredicate
_evaluators
  "return list of evaluators associated with predicate"

  ^ {(self evaluator1)}
%
category: 'testing'
method: GsRangeQueryPredicate
usesComparisonOperation
  ^ true
%
category: 'testing'
method: GsRangeQueryPredicate
usesRangeOperation
  ^ true
%
category: 'private'
method: GsRangeQueryPredicate
_evaluators
  "return list of evaluators associated with predicate"

  ^ {(self evaluator)}
%



category: 'visiting'
method: GsQueryConjoinSetChecker
acceptUnaryClause: aUnaryClause
  aUnaryClause isNegationClause
    ifTrue: [ 
      "If clause is  negated, the applyDeMorgansLaws has not been applied to transform the formula. Cannot correctly analyze presence of Bug 43764 unless optimized with applyDeMorgansLaws and consolidateRangePredicates options set."
      ^ GsMalformedQueryExpressionError
        bug43764QueryCannotBeAnalyzed: #'acceptUnaryClause:' ].
  super acceptUnaryClause: aUnaryClause
%
category: 'Initialization'
method: GsMalformedQueryExpressionError
bug43764MalformedQuery: explanation
  "Invalid query formulation. Query cannot be properly evaluated ... 
   The query must be rewritten."

  self
    errorCondition: #'bug43764MalformedQuery';
    messageText:
        'Invalid query formulation with conjoined collection based path terms: '
            , explanation
            ,
              ' (#bug43764MalformedQuery). See bugnote: http://gemtalksystems.com/data/bugnotes/43764.html.'
%
category: 'Instance Creation'
classmethod: GsMalformedQueryExpressionError
bug43764MalformedQuery: explanation senderLocationID: senderLocationID
  "Invalid query formulation. Query cannot be properly evaluated ... 
   The query must be rewritten."

  "senderLocationID used to identify the method and/or class of the sender to help 
   validate test coverage"

  ^ (self new
    bug43764MalformedQuery: explanation;
    reason: senderLocationID) signal
%
category: 'Initialization'
method: GsMalformedQueryExpressionError
bug43764QueryCannotBeAnalyzed
  "query may be exposed to bug43764, but the query options applyDeMorgansLaws and consolidateRangePredicates must be set before proper analysis can be done"

  self
    errorCondition: #'bug43764QueryCannotBeAnalyzed';
    messageText:
        'Invalid query formulation involving collection based path terms. Query must be  optimized to properly evaluate exposure to Bug 43764. Set the query options applyDeMorgansLaws and consolidateRangePredicates and try again (#bug43764QueryCannotBeAnalyzed). See bugnote: http://gemtalksystems.com/data/bugnotes/43764.html.'
%
category: 'Instance Creation'
classmethod: GsMalformedQueryExpressionError
bug43764QueryCannotBeAnalyzed: senderLocationID
  "query may be exposed to bug43764, but the query options applyDeMorgansLaws and consolidateRangePredicates must be set before proper analysis can be done"

  "senderLocationID used to identify the method and/or class of the sender to help 
   validate test coverage"

  ^ (self new
    bug43764QueryCannotBeAnalyzed;
    reason: senderLocationID) signal
%
category: 'Initialization'
method: GsMalformedQueryExpressionError
bug43764QueryMustBeOptimized
  "Query is not exposed to Bug 43764, but the query must be optimized with the 
   appropriate query options to be successfully evaluated"

  self
    errorCondition: #'bug43764QueryMustBeOptimized';
    messageText:
        'Invalid query formulation involving collection based path terms. If you optimize the query and set the query options applyDeMorgansLaws and consolidateRangePredicates the query can be evaluated successfully (#bug43764QueryMustBeOptimized). See bugnote: http://gemtalksystems.com/data/bugnotes/43764.html.'
%
category: 'Instance Creation'
classmethod: GsMalformedQueryExpressionError
bug43764QueryMustBeOptimized: senderLocationID
  "Query is not exposed to Bug 43764, but the query must be optimized with the 
   appropriate query options to be successfully evaluated"

  "senderLocationID used to identify the method and/or class of the sender to help 
   validate test coverage"

  ^ (self new
    bug43764QueryMustBeOptimized;
    reason: senderLocationID) signal
%
category: 'Accessing'
method: GsMalformedQueryExpressionError
errorCondition
  ^ errorCondition
%
category: 'Accessing'
method: GsMalformedQueryExpressionError
errorCondition: aSymbol
  errorCondition := aSymbol
%
category: 'querying'
method: GsPathPathPredicate
readStream
  self shouldNotImplement: #'readStream'
%
category: 'visiting'
method: GsStreamableConjunctiveClauseChecker
acceptPredicate: aPredicate
  "streamable predicate:
  1. path-constant predicate (i.e., not path-path predicate)
  2. RangeEqualityIndex evaluator ... (not really valid for GsQuery, but could defer
     readstream on PathEvaluator if too much implementation involved)
  3. use range or equality operation for legacy indexes (#== or #~~ or #~= not allowed)
  4. use range, equality or identity operations for btree plus indexes (#~~ or #~= not allowed)
"

  aPredicate isPathPath
    ifTrue: [ 
      ^ GsMalformedQueryExpressionError
        queryIsNotStreamable:
          'Query with path-path predicate: ' , aPredicate printString printString
            , ' is not streamable.'
        senderLocationID: #'acceptPredicate:' ].
  aPredicate isConstantConstant
    ifTrue: [ 
      ^ GsMalformedQueryExpressionError
        queryIsNotStreamable:
          'Query with constant-constant predicate: ' , aPredicate printString printString
            , ' is not streamable.'
        senderLocationID: #'acceptPredicate:' ].
  aPredicate usesPathEvaluator
    ifTrue: [ 
      ^ GsMalformedQueryExpressionError
        queryIsNotStreamable:
          'Predicate: ' , aPredicate printString printString
            , ' must use an equality index or a  btreePlus identity index.'
        senderLocationID: #'acceptPredicate:' ].
  (aPredicate _evaluatorCanStreamQueries
    and: [ 
      aPredicate usesIdenticalToOperation
        or: [ aPredicate usesEqualOperation or: [ aPredicate usesComparisonOperation ] ] ])
    ifFalse: [ 
      ^ GsMalformedQueryExpressionError
        queryIsNotStreamable:
          'Predicate: ' , aPredicate printString printString
            ,
              ' must use an equality (#= #==) or comparison operator (#<= #< #> #>=).'
        senderLocationID: #'acceptPredicate:' ].
  super acceptPredicate: aPredicate
%
category: 'checking'
method: GsStreamableConjunctiveClauseChecker
check
  "streamable predicate:
  1. path-constant predicate (i.e., not path-path predicate)
  2. RangeEqualityIndex evaluator ... (not really valid for GsQuery, but could defer
     readstream on PathEvaluator if too much implementation involved)
  3. use range or equality operation for legacy indexes (#== or #~~ or #~= not allowed)
  4. use range, equality or identity operations for btree plus indexes (#~~ or #~= not allowed)
  5. 1 predicate (range predicate allowed)
"

  self visitFormula: self formula.
  self predicates size > 1
    ifTrue: [ 
      ^ GsMalformedQueryExpressionError
        queryIsNotStreamable:
          'Clause ' , self formula printString printString
            , ' must have a single predicate'
        senderLocationID: #'acceptPredicate:' ]
%
category: 'instance creation'
classmethod: GsStreamableConjunctiveClauseChecker
check: aFormula on: anNsc
  ^ self new
    formula: aFormula;
    nsc: anNsc;
    queryOptions: GsQueryOptions default;
    check
%
category: 'querying'
method: GsConstantConstantPredicate
readStream
  self shouldNotImplement: #'readStream'
%
category: 'querying'
method: GsRangeQueryPredicate
readStream
  | queryEvaluator theConstant1 theConstant2 theOperator1 theOperator2 validNilTransformOperators stream |
  queryEvaluator := self evaluator asQueryEvaluator.
  self isNormal
    ifTrue: [ 
      theConstant1 := self constant1.
      theOperator1 := self operator1.
      theConstant2 := self constant2.
      theOperator2 := self operator2 ]
    ifFalse: [ 
      theConstant1 := self constant2.
      theOperator1 := self inverseOperatorFor: self operator2.
      theConstant2 := self constant1.
      theOperator2 := self inverseOperatorFor: self operator1 ].
  validNilTransformOperators := #(#'<=' #'>=').
  stream := ((theConstant1 == nil and: [ theConstant2 == nil ])
    and: [ 
      (validNilTransformOperators includes: self operator1)
        and: [ validNilTransformOperators includes: self operator2 ] ])
    ifTrue: [ 
      queryEvaluator
        _findAllValuesGreaterThan: theConstant1
        andEquals: true
        andLessThan: theConstant2
        andEquals: true
        using: (BtreeComparisonForCompare newForComparison: nil) ]
    ifFalse: [ 
      ((theConstant1 == nil and: [ theConstant2 ~~ nil ])
        and: [ validNilTransformOperators includes: self operator1 ])
        ifTrue: [ theOperator1 := #'=' ].
      ((theConstant1 ~~ nil and: [ theConstant2 == nil ])
        and: [ validNilTransformOperators includes: self operator2 ])
        ifTrue: [ theOperator2 := #'=' ].
      queryEvaluator
        _findAllValuesGreaterThan: theConstant1
        andEquals: theOperator1 == #'<='
        andLessThan: theConstant2
        andEquals: theOperator2 == #'<='
        using: (BtreeComparisonForCompare newForComparison: nil) ].
  stream
    streamQuerySpec:
      (stream btreeRangeComparisonQuerySpec
         key: theConstant1
         selector:
           (theOperator1 == #<=
             ifTrue: [ #'>=' ]
             ifFalse: [ #'>' ]); "denormalize the selector"
         key2: theConstant2 selector2: theOperator2;
         yourself).
  ^ stream
%
category: 'querying'
method: GsQueryPredicate
readStream
  | queryEvaluator theOperator theConstant |
  queryEvaluator := self evaluator1 asQueryEvaluator.
  theOperator := self operator.
  theConstant := self constant.
  self _check: self evaluator1 compare: theOperator value: theConstant.
  (theOperator == #'<' or: [ theOperator == #'<=' ])
    ifTrue: [ 
      | stream querySpec |
      stream := queryEvaluator
        _findAllValuesGreaterThan: nil
        andEquals: true
        andLessThan: theConstant
        andEquals: theOperator == #'<='
        using: (BtreeComparisonForCompare newForComparison: self collator).
      querySpec := queryEvaluator btreeComparisonQuerySpec
        key: theConstant selector: theOperator;
        yourself.
      stream streamQuerySpec: querySpec.
      ^ stream ].
  (theOperator == #'>' or: [ theOperator == #'>=' ])
    ifTrue: [ 
      ^ queryEvaluator
        _findAllValuesGreaterThanKey: theConstant
        andEquals: theOperator == #'>='
        using: (BtreeComparisonForCompare newForComparison: self collator) ].
  theOperator == #'='
    ifTrue: [ 
      ^ queryEvaluator
        _findAllValuesEqualTo: theConstant
        using: (BtreeComparisonForCompare newForComparison: self collator) ].
  theOperator == #'=='
    ifTrue: [
      | querySpec stream |
      stream := queryEvaluator _findAllValuesIdenticalTo: theConstant.
      querySpec := queryEvaluator btreeComparisonQuerySpec
        key: theConstant selector: theOperator;
        yourself.
      stream streamQuerySpec: querySpec.
      ^  stream ]
%
category: 'testing'
method: GsQueryPredicate
usesEqualOperation
  ^ operator == #'='
%
category: 'Initialization'
method: GsMalformedQueryExpressionError
queryIsNotStreamable: explanation
  "Query is not streamable."

  self
    errorCondition: #'queryIsNotStreamable';
    messageText:
        'Query may not be streamed. '
            , explanation , ' (#queryIsNotStreamable).'
%
category: 'Instance Creation'
classmethod: GsMalformedQueryExpressionError
queryIsNotStreamable: explanation senderLocationID: senderLocationID
  "Query is not streamable."

  "senderLocationID used to identify the method and/or class of the sender to help 
   validate test coverage"

  ^ (self new
    queryIsNotStreamable: explanation;
    reason: senderLocationID) signal
%
category: 'Initialization'
method: GsMalformedQueryExpressionError
streamableQueryCannotBeAnalyzed
  "cannot determine if query can be streamed, the query options applyDeMorgansLaws and consolidateRangePredicates must be set before proper analysis can be done"

  self
    errorCondition: #'streamableQueryCannotBeAnalyzed';
    messageText:
        'Invalid query formulation involving collection based path terms. Query must be  optimized to properly evaluate whether or not the query is streamable. Set the query options applyDeMorgansLaws and consolidateRangePredicates and try again (#streamableQueryCannotBeAnalyzed).'
%
category: 'Instance Creation'
classmethod: GsMalformedQueryExpressionError
streamableQueryCannotBeAnalyzed: senderLocationID
  "cannot determine if query can be streamed, the query options applyDeMorgansLaws and consolidateRangePredicates must be set before proper analysis can be done"

  "senderLocationID used to identify the method and/or class of the sender to help 
   validate test coverage"

  ^ (self new
    streamableQueryCannotBeAnalyzed;
    reason: senderLocationID) signal
%
category: 'Initialization'
method: GsMalformedQueryExpressionError
streamableQueryMustBeOptimized
  "Query can be streamed, but the query must be optimized with the 
   appropriate query options to be successfully evaluated"

  self
    errorCondition: #'streamableQueryMustBeOptimized';
    messageText:
        'Query cannot be streamed as is. If you optimize the query and set the query options applyDeMorgansLaws and consolidateRangePredicates the query can be evaluated successfully (#streamableQueryMustBeOptimized).'
%
category: 'Instance Creation'
classmethod: GsMalformedQueryExpressionError
streamableQueryMustBeOptimized: senderLocationID
  "Query can be streamed, but the query must be optimized with the 
   appropriate query options to be successfully evaluated"

  "senderLocationID used to identify the method and/or class of the sender to help 
   validate test coverage"

  ^ (self new
    streamableQueryMustBeOptimized;
    reason: senderLocationID) signal
%
category: 'checking'
method: GsStreamableQueryChecker
check
  self formula isConjunctiveNormalForm
    ifTrue: [ self checkConjunctiveClause: self formula ]
    ifFalse: [ 
      "If clause is not in conjunctive normal form,  applyDeMorgansLaws has not been applied to transform the formula. Cannot properly determine whether or not a readStream can be used if applyDeMorgansLaws and consolidateRangePredicates options not set."
      (queryOptions applyDeMorgansLaws and: [ queryOptions consolidateRangePredicates ])
        ifTrue: [ 
          GsMalformedQueryExpressionError
            queryIsNotStreamable:
              'Clause ' , self formula printString printString,
              ' must have a single predicate and this clause cannot be reduced to a single predicate by the query optimizer.'
        senderLocationID: #'check' ]
        ifFalse: [ GsMalformedQueryExpressionError streamableQueryCannotBeAnalyzed: #'check' ] ]
%
category: 'instance creation'
classmethod: GsStreamableQueryChecker
check: aFormula on: anNsc
  ^ self new
    formula: aFormula;
    nsc: anNsc;
    queryOptions: GsQueryOptions default;
    check
%
category: 'checking'
method: GsStreamableQueryChecker
checkConjunctiveClause: aClause
  GsStreamableConjunctiveClauseChecker check: aClause on: self nsc
%
category: 'testing'
method: GsQueryFormula
usesEqualOperation
  ^ false
%
category: 'private'
method: GsQueryFormula
_checkForStreamableQuery: aGsQuery
  "Signal an error if the formula is not suited for streaming over result set"

  "a streamable query is not exposed to bug 43764, so no need to check"

  | queryOptions |
  queryOptions := aGsQuery queryOptions.
  [ GsStreamableQueryChecker check: self on: aGsQuery _nsc ]
    on: GsMalformedQueryExpressionError
    do: [ :ex | 
      "checker will throw an error if the query cannot be streamed"
      (queryOptions autoOptimize
        and: [ queryOptions applyDeMorgansLaws and: [ queryOptions consolidateRangePredicates ] ])
        ifTrue: [ 
          "cannot be streamed"
          ex pass ]
        ifFalse: [ 
          [ 
          | newQuery newQueryOptions |
          "determine if query is evaluable with proper options set"
          newQueryOptions := queryOptions + GsQueryOptions applyDeMorgansLaws
            + GsQueryOptions consolidateRangePredicates
            + GsQueryOptions autoOptimize.
          newQuery := aGsQuery copy.
          newQuery queryOptions: newQueryOptions.
          GsStreamableQueryChecker check: newQuery formula on: aGsQuery _nsc ]
            on: GsMalformedQueryExpressionError
            do: [ :ex2 | 
              "cannot stream ... pass the exception"
              ex2 pass ] ].
      GsMalformedQueryExpressionError
        streamableQueryMustBeOptimized: #'_checkForStreamableQuery:' ].
  ^ self	"streamable query"
%
category: 'optimizing'
method: GsClassicConjunctiveClauseOptimizer
consolidateUnaryConstantPredicates
  "Simplify formulas involving predicates that are unary-constant (true or false) or 
   constant-constant. The predicate is removed, or the expression is consolidated."

  | otherPredicates truePredicate falsePredicate |
  self predicates size <= 1
    ifTrue: [ 
      "a single predicate ... no more optimization called for"
      ^ self ].
  truePredicate := falsePredicate := nil.
  otherPredicates := OrderedCollection new.
  self predicates
    do: [ :predicate | 
      (predicate isConstantConstant and: [ predicate isConstantBound ])
        ifTrue: [ 
          predicate executePredicate
            ifTrue: [ truePredicate := predicate ]
            ifFalse: [ falsePredicate := predicate ] ]
        ifFalse: [ otherPredicates add: predicate ] ].
  falsePredicate
    ifNotNil: [ predicates := {falsePredicate} ]
    ifNil: [ truePredicate ifNotNil: [ predicates := otherPredicates ] ]
%
category: 'Operators'
method: GsQueryOptions
clearConsolidateUnaryConstantPredicates
  self consolidateUnaryConstantPredicates: false
%
category: 'options'
classmethod: GsQueryOptions
consolidateUnaryConstantPredicates
  "Simplify formulas involving predicates that are unary-constant (true or false) or 
   constant-constant. The predicate is removed, or the expression is consolidated.

  (true) & <other predicates> 
    becomes: 
  <other predicates> 

  (true) | <otherpredicates> 
    becomes: 
  (true)

  (false) & <other predicates> 
    becomes: 
  (false)

  (false) | <other predicates> 
    becomes: 
  <other predicates>

  (true) & (each.name = 'Dale') & (each.gender == #male)
    becomes:
  (each.name = 'Dale') & (each.gender == #male)"

  ^ self new
    setConsolidateUnaryConstantPredicates;
    yourself
%
category: 'Accessing'
method: GsQueryOptions
consolidateUnaryConstantPredicates
  | val |
  val := self dynamicInstVarAt: #'consolidateUnaryConstantPredicates'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
consolidateUnaryConstantPredicates: aBoolean
  self dynamicInstVarAt: #'consolidateUnaryConstantPredicates' put: aBoolean
%
category: 'Operators'
method: GsQueryOptions
setConsolidateUnaryConstantPredicates
  self consolidateUnaryConstantPredicates: true
%
category: 'testing'
method: GsConstantConstantPredicate
isConstantBound
  self operator == #'unary'
    ifTrue: [ ^ self operand1 isBound ].
  ^ self operand1 isBound and: [ self operand2 isBound ]
%
category: 'optimizing'
method: Gs32Optimizer
optimize
  | formulaCopy |
  formulaCopy := (GsClassicQueryBlockOptimizer
    optimize: self formula
    on: self nsc
    options: self queryOptions) copy.
  formulaCopy isDisjunctiveNormalForm
    ifTrue: [ 
      formulaCopy isDisjunctiveClause
        ifTrue: [ formulaCopy := self optimizeDisjunctiveCompoundClause: formulaCopy ].
      formulaCopy := self optimizeDisjunctiveNormalClause: formulaCopy ]
    ifFalse: [ 
      formulaCopy isDisjunctiveClause
        ifTrue: [ formulaCopy := self optimizeDisjunctiveCompoundClause: formulaCopy ] ].
  self visitFormula: formulaCopy.
  self transformed
    ifTrue: [ 
      "given the nature of the disjunctive clause transforms, we need to keep optimizing until nothing is changed"
      self formula: formulaCopy copy.
      self transformed: false.
      ^ self optimize ].
  ^ formulaCopy immediateInvariant
%
category: 'optimizing'
method: Gs32Optimizer
optimizeConjunctiveNormalClause: aClause
  ^ aClause
%
category: 'testing'
method: GsQueryFormula
isDisjunctiveClause
  ^ false
%
category: 'testing'
method: GsCompoundClause
isDisjunctiveClause
  ^ self operator == #'|'
%





category: 'testing'
method: GsCompoundClause
isDisjunctiveNormalForm
  ^ self operator == #'|'
    and: [ 
      self clause1 isDisjunctiveNormalForm
        and: [ self clause2 isDisjunctiveNormalForm ] ]
%
category: 'optimizing'
method: GsQueryFormula
canConsolidateEnumeratedWith: secondaryPredicate
  ^ false
%
category: 'optimizing'
method: GsQueryFormula
canConsolidateEnumeratedWithPathConstantPredicate: primaryPredicate
  ^ false
%
category: 'testing'
method: GsQueryFormula
isDisjunctiveNormalForm
  ^ true
%
category: 'optimizing'
method: GsQueryFormula
redundantDisjunctivePredicateBetween: secondaryPredicate
  ^ nil
%
category: 'optimizing'
method: GsQueryFormula
redundantDisjunctivePredicateBetweenPathConstant: primaryPredicate
  ^ nil
%
category: 'accessing'
method: GsQueryFormula
_nsc
  ^ nsc
%
category: 'accessing'
method: GsQueryFormula
_pathTerms
  ^ self _parsePath: self path
%
category: 'optimizing'
method: Gs32Optimizer
optimizeDisjunctiveNormalClause: aClause
  ^ (Gs32DisjunctiveClauseOptimizer
    optimize: aClause
    on: self nsc
    options: self queryOptions) copy
%
category: 'accessing'
method: GsPathPathPredicate
_pathTerms
  "ambiguous"

  ^ self shouldNotImplement: #'_pathTerms'
%
category: 'optimizing'
method: GsPathConstantPredicate
canConsolidateEnumeratedWith: secondaryPredicate
  ^ secondaryPredicate canConsolidateEnumeratedWithPathConstantPredicate: self
%
category: 'optimizing'
method: GsPathConstantPredicate
canConsolidateEnumeratedWithPathConstantPredicate: primaryPredicate
  | pathTerms primaryPathTerms |
  (primaryPredicate operand1 hasSamePathAs: self operand1)
    ifTrue: [ ^ false ].
  (primaryPredicate isConstantBound and: [ self isConstantBound ])
    ifFalse: [ ^ false ].
  primaryPredicate operator == self operator
    ifFalse: [ ^ false ].
  (primaryPredicate constant _idxForCompareEqualTo: self constant)
    ifFalse: [ ^ false ].
  pathTerms := self _pathTerms.
  primaryPathTerms := primaryPredicate _pathTerms.
  ^ pathTerms size = primaryPathTerms size
%
category: 'transforming'
method: GsPathConstantPredicate
consolidateEnumeratedWith: secondaryPredicate
  "caller must ensure that canConsolidateEnumeratedWith: has already been called"

  | pathTerms secondaryPathTerms candidateOperand1 candidatePathString |
  pathTerms := self _pathTerms.
  secondaryPathTerms := secondaryPredicate _pathTerms.
  candidatePathString := 'each'.
  1 to: pathTerms size do: [ :i | 
    | primaryTerm secondaryTerm |
    primaryTerm := pathTerms at: i.
    secondaryTerm := secondaryPathTerms at: i.
    primaryTerm = secondaryTerm
      ifTrue: [ candidatePathString := candidatePathString , '.' , primaryTerm ]
      ifFalse: [ 
        primaryTerm > secondaryTerm
          ifTrue: [ 
            | swap |
            swap := primaryTerm.
            primaryTerm := secondaryTerm.
            secondaryTerm := swap ].
        candidatePathString := candidatePathString , '.' , primaryTerm , '|'
          , secondaryTerm ] ].
  candidateOperand1 := GsQueryPathReferenceAssociation
    newWithKey: candidatePathString
    value: nil.
  ^ GsPathConstantPredicate
    operand: candidateOperand1
    operator: operator
    operand: operand2
%
category: 'optimizing'
method: GsPathConstantPredicate
redundantDisjunctivePredicateBetween: secondaryPredicate
  ^ secondaryPredicate redundantDisjunctivePredicateBetweenPathConstant: self
%
category: 'optimizing'
method: GsPathConstantPredicate
redundantDisjunctivePredicateBetweenPathConstant: primaryPredicate
  | op1 op2 |
  (primaryPredicate operand1 hasSamePathAs: self operand1)
    ifFalse: [ ^ nil ].
  (primaryPredicate isConstantBound and: [ self isConstantBound ])
    ifFalse: [ ^ nil ].
  (primaryPredicate constant == nil or: [ self constant == nil ])
    ifTrue: [ ^ nil ].
  op1 := primaryPredicate operator.
  op2 := self operator.
  op1 == op2
    ifTrue: [ 
      " operators are the same "
      (op1 == #'<' or: [ op1 == #'<=' ])
        ifTrue: [ 
          " operator: < or <= "
          " choose largest value "
          ^ (primaryPredicate constant _idxForCompareLessThan: self constant)
            ifTrue: [ primaryPredicate ]
            ifFalse: [ self ] ].
      (op1 == #'>' or: [ op1 == #'>=' ])
        ifTrue: [ 
          " operator: > or >= "
          " choose smallest value "
          ^ (primaryPredicate constant _idxForCompareGreaterThan: self constant)
            ifTrue: [ primaryPredicate ]
            ifFalse: [ self ] ].
      (op1 == #'=' or: [ op1 == #'~=' ])
        ifTrue: [ 
          " operator: = or ~="
          ^ (primaryPredicate constant _idxForCompareEqualTo: self constant)
            ifTrue: [ self ]
            ifFalse: [ nil ] ].
      (op1 == #'==' or: [ op1 == #'~~' ])
        ifTrue: [ 
          " operator: == or ~~"
          ^ primaryPredicate constant == self constant
            ifTrue: [ self ]
            ifFalse: [ nil ] ] ]
    ifFalse: [ 
      " operators are different "
      ((op1 == #'<' and: [ op2 == #'<=' ])
        or: [ op1 == #'<=' and: [ op2 == #'<' ] ])
        ifTrue: [ 
          " op1: < and op2: <= "
          " op1: <= and op2: < "
          " choose largest value "
          ^ (primaryPredicate constant _idxForCompareLessThan: self constant)
            ifTrue: [ primaryPredicate ]
            ifFalse: [ 
              (primaryPredicate constant _idxForCompareGreaterThan: self constant)
                ifTrue: [ self ]
                ifFalse: [ 
                  " operands are = "
                  op1 == #'<'
                    ifTrue: [ primaryPredicate ]
                    ifFalse: [ self ] ] ] ].
      ((op1 == #'>' and: [ op2 == #'>=' ])
        or: [ op1 == #'>=' and: [ op2 == #'>' ] ])
        ifTrue: [ 
          " op1: > and op2: >= "
          " op1: >= and op2: > "
          " choose largest value "
          ^ (primaryPredicate constant _idxForCompareGreaterThan: self constant)
            ifTrue: [ primaryPredicate ]
            ifFalse: [ 
              (primaryPredicate constant _idxForCompareLessThan: self constant)
                ifTrue: [ self ]
                ifFalse: [ 
                  " operands are = "
                  op1 == #'>'
                    ifTrue: [ primaryPredicate ]
                    ifFalse: [ self ] ] ] ].
      ((op1 == #'=' and: [ op2 == #'==' ])
        or: [ op1 == #'==' and: [ op2 == #'=' ] ])
        ifTrue: [ 
          " op1: = and op2: == "
          " op1: == and op2: = "
          ^ primaryPredicate constant == self constant
            ifTrue: [ 
              op2 == #'=='
                ifTrue: [ primaryPredicate ]
                ifFalse: [ self ] ]
            ifFalse: [ nil ] ].
      ((op1 == #'==' and: [ op2 == #'~~' ])
        or: [ op1 == #'~~' and: [ op2 == #'==' ] ])
        ifTrue: [ 
          " op1: == and op2: ~~ "
          " op1: ~~ and op2: == "
          ^ primaryPredicate constant == self constant
            ifTrue: [ self _unsatisfiableQuery ]
            ifFalse: [ self _unsatisfiableQuery ] ].
      (op1 == #'=' or: [ op1 == #'==' ])
        ifTrue: [ 
          " op1: = or op1: =="
          op2 == #'<'
            ifTrue: [ 
              " op2: < "
              ^ (primaryPredicate constant _idxForCompareLessThan: self constant)
                ifTrue: [ primaryPredicate ]
                ifFalse: [ nil ] ].
          op2 == #'<='
            ifTrue: [ 
              " op2: <= "
              ^ (primaryPredicate constant
                _idxForCompareLessThanOrEqualTo: self constant)
                ifTrue: [ primaryPredicate ]
                ifFalse: [ nil ] ].
          op2 == #'>'
            ifTrue: [ 
              " op2: > "
              ^ (primaryPredicate constant
                _idxForCompareGreaterThan: self constant)
                ifTrue: [ primaryPredicate ]
                ifFalse: [ nil ] ].
          op2 == #'>='
            ifTrue: [ 
              " op2: >= "
              ^ (primaryPredicate constant
                _idxForCompareGreaterThanOrEqualTo: self constant)
                ifTrue: [ primaryPredicate ]
                ifFalse: [ nil ] ].
          op2 == #'~='
            ifTrue: [ 
              " op2: ~= "
              ^ (primaryPredicate constant _idxForCompareEqualTo: self constant)
                ifTrue: [ self _unsatisfiableQuery ]
                ifFalse: [ self _unsatisfiableQuery ] ] ].
      (op2 == #'=' or: [ op2 == #'==' ])
        ifTrue: [ 
          " op2: = or op2: =="
          op1 == #'<'
            ifTrue: [ 
              " op1: < "
              ^ (self constant _idxForCompareLessThan: primaryPredicate constant)
                ifTrue: [ self ]
                ifFalse: [ nil ] ].
          op1 == #'<='
            ifTrue: [ 
              " op1: <= "
              ^ (self constant
                _idxForCompareLessThanOrEqualTo: primaryPredicate constant)
                ifTrue: [ self ]
                ifFalse: [ nil ] ].
          op1 == #'>'
            ifTrue: [ 
              " op1: > "
              ^ (self constant
                _idxForCompareGreaterThan: primaryPredicate constant)
                ifTrue: [ self ]
                ifFalse: [ nil ] ].
          op1 == #'>='
            ifTrue: [ 
              " op1: >= "
              ^ (self constant
                _idxForCompareGreaterThanOrEqualTo: primaryPredicate constant)
                ifTrue: [ self ]
                ifFalse: [ nil ] ].
          op1 == #'~='
            ifTrue: [ 
              " op1: ~= "
              ^ (self constant _idxForCompareEqualTo: primaryPredicate constant)
                ifTrue: [ self _unsatisfiableQuery ]
                ifFalse: [ self _unsatisfiableQuery ] ] ] ].
  ^ nil
%
category: 'testing'
method: GsUnaryClause
isDisjunctiveNormalForm
  ^ self operator == #'null' and: [ self clause isDisjunctiveNormalForm ]
%
category: 'accessing'
method: Gs32DisjunctiveClauseOptimizer
predicatesAsFormula
  | newFormula |
  newFormula := (predicates at: 1) copy.
  2 to: predicates size do: [ :index | newFormula := newFormula | (predicates at: index) copy ].
  ^ (newFormula bindEvaluatorsFor: self nsc collator: self collator)
    immediateInvariant
%
category: 'optimizing'
method: Gs32DisjunctiveClauseOptimizer
removeRedundantDisjunctivePredicates
  | primaries |
  [ 
  primaries := self predicates copy.
  1 to: primaries size do: [ :index | 
    (primaries at: index)
      ifNotNil: [ :primaryPredicate | self removeRedundantDisjunctivePredicates: primaries with: primaryPredicate ] ].
  predicates := primaries select: [ :each | each ~~ nil  ] ]
    on: GsUnsatisfiableQueryNotification
    do: [ :note | 
      "short circuit analysis and replace whole enchilada with a `true` predicate"
      predicates := OrderedCollection with: (GsQueryPredicate constant: true).
      ^ self ]
%
category: 'optimizing'
method: Gs32DisjunctiveClauseOptimizer
removeRedundantDisjunctivePredicates: primaries with: primaryPredicate
  primaries
    do: [ :secondaryPredicate | 
      (secondaryPredicate ~~ nil  and: [ primaryPredicate ~~ secondaryPredicate ])
        ifTrue: [ 
          (primaryPredicate redundantDisjunctivePredicateBetween: secondaryPredicate)
            ifNotNil: [ :redundantPredicate | primaries at: (primaries indexOf: redundantPredicate) put: nil ] ] ]
%
category: 'converting'
method: GsQuery
asArray
  ^ self queryResult asArray
%
category: 'Testing'
method: OptionalTermPathEvaluator
termsRequired
  "Answer true if receiver requires that instance variables of indexed objects are present "

  ^ false
%
category: 'Testing'
method: CollectionBasedPathEvaluator
termsRequired
  "Answer true if receiver requires that instance variables of indexed objects are present "

  ^ self _pathTermsRequired
%
category: 'Testing'
method: SelectorPathEvaluator
termsRequired
  "Answer true if receiver requires that instance variables of indexed objects are present "

  ^ false
%
category: 'Accessing'
method: CollectionBasedPathEvaluator
_pathTermsRequired
  pathTermsRequired ifNil: [ pathTermsRequired := true ].
  ^ pathTermsRequired
%
category: 'Accessing'
method: CollectionBasedPathEvaluator
_pathTermsRequired: aBoolOrNil
  pathTermsRequired := aBoolOrNil
%
category: 'accessing'
method: GsQueryFormula
optionalPathTerms
  | bound |
  bound := self copy _pathTermsRequired: false.
  nsc ifNil: [ ^ bound immediateInvariant ].
  ^ bound bindEvaluators
%
category: 'Operators'
method: GsIndexOptions
clearOptionalPathTerms
  self optionalPathTerms: false
%
category: 'Accessing'
classmethod: GsIndexOptions
optionalPathTerms
  "Do not signal an error if instance variable missing along index path."

  "Use basicNew to bypass initialization, thus making it possible to use the default index options
   to supply the value for slots not explicitly set or cleared."

  ^ self basicNew
    setOptionalPathTerms;
    yourself
%
category: 'Accessing'
method: GsIndexOptions
optionalPathTerms
  ^ self _optionalPathTerms ifNil: [ self defaultOptions optionalPathTerms ]
%
category: 'Private'
method: GsIndexOptions
optionalPathTerms: aBool
  "Private method ... should never be called by method external to GsIndexOptions. 
   Use set* and clear* methods instead."

  self dynamicInstVarAt: #'optionalPathTerms' put: aBool
%
category: 'Operators'
method: GsIndexOptions
setOptionalPathTerms
  self optionalPathTerms: true
%




category: 'querying'
method: GsQuery
optionalPathTerms
  self _formula ifNotNil: [ self _formula: self _formula optionalPathTerms ]
%
category: 'Specification'
method: GsIndexSpec
unicodeIndex: path options: aGsIndexOptions
  "Specify that a unicode equality index on the given path using the default collator and given 
    index options be created.

   When using string comparison mode, a unicode equality index is constrained to the unicode 
    string classes Unicode7, Unicode16, and Unicode32.

   When using unicode comparison mode, a unicode equality index is constrained to be a type
     of CharacterCollection.

   For btree plus indexes, optimized comparisons will not be used. If optimized comparisons is
     desired, use either #unicodeStringOptimizedIndex: or #symbolOptimizedIndex:collator:.
"

  self unicodeIndex: path collator: self defaultCollator options: aGsIndexOptions
%
category: 'converting'
method: GsRangeQueryPredicate
asFormulaWithSelectorParts: selectorParts arguments: arguments
  | selector |
  selector := selectorParts first inputValue asSymbol.
  ^ self perform: selector with: arguments first
%
category: 'private'
method: GsCompoundClause
bruteThreshold
  "number of elements in result set where brute force query is cheaper than using index."

  ^ 1000
%
category: 'testing'
method: GsCompoundClause
usesPathEvaluator
  ^ self clause1 usesPathEvaluator and: [ self clause2 usesPathEvaluator ]
%
category: 'testing'
method: GsCompoundClause
usesPathEvaluatorForFirstClause
  ^ self clause1 usesPathEvaluatorForFirstClause
%
category: 'testing'
method: GsQueryFormula
usesPathEvaluatorForFirstClause
  "only real distinction from usesPathEvaluator is for GsCompoundClause"

  ^ self usesPathEvaluator
%
! Remove existing behavior from GsQueryErrorHandlingParser
removeallmethods GsQueryErrorHandlingParser
removeallclassmethods GsQueryErrorHandlingParser

doit
GsQueryErrorHandlingParser comment:
'GsQueryErrorHandlingParser is part of the Indexing and Querying subsystem. It implements GemStone internals and is not intended for direct use by customers.'.
true
%

! ------------------- Class methods for GsQueryErrorHandlingParser
category: 'instance creation'
classmethod: GsQueryErrorHandlingParser
on: aParser errorMessage: aString
  ^ (self on: aParser) setErrorMessage: aString
%
! ------------------- Instance methods for GsQueryErrorHandlingParser
category: 'initialization'
method: GsQueryErrorHandlingParser
setErrorMessage: aString
  errorMessage := aString
%
category: 'parsing'
method: GsQueryErrorHandlingParser
parseOn: aPPContext
  | element |
  (element := parser parseOn: aPPContext) isPetitFailure
    ifTrue: [ 
      | msg failureElement |
      failureElement := element.
      aPPContext furthestFailure
        ifNotNil: [ :furthest | 
          furthest position >= element position
            ifTrue: [ failureElement := furthest ] ].
      msg := self nestMessages
        ifTrue: [ errorMessage , ' (' , failureElement message , ')' ]
        ifFalse: [ errorMessage ].
      element := PPFailure
        message: msg
        context: aPPContext
        at: failureElement position.
      aPPContext noteDominantFailure: element ].
  ^ element
%





category: 'grammar'
method: GsQueryGrammar
primaryIdentifier
  ^ 'each' asParser not , identifier
%
category: 'parsing'
method: GsQueryErrorHandlingParser
nestMessages
  nestMessages ifNil: [ nestMessages := true ].
  ^ nestMessages
%
category: 'parsing'
method: GsQueryErrorHandlingParser
nestMessages: aBool
  nestMessages := aBool
%
category: 'instance creation'
classmethod: GsQueryErrorHandlingParser
on: aParser errorMessage: aString nestMessages: aBool
  ^ (self on: aParser)
    setErrorMessage: aString;
    nestMessages: aBool;
    yourself
%
category: 'grammar-messages'
method: GsQueryParser
simplePredicateExpression
  ^ super simplePredicateExpression
    map: [ :receiverNode :messageNodes | self build: receiverNode messages: messageNodes ]
%
category: 'grammar-messages'
method: GsQueryGrammar
simplePredicateExpression
  ^ primary , (predicateMessage min: 0 max: 1)
%
category: 'querying'
method: GsQueryFormula
reversedReadStream
  ^ self readStream reversed
%
category: 'Operators'
method: GsQueryOptions
clearOptionalPathTerms
  self optionalPathTerms: false
%
category: 'Instance Creation'
classmethod: GsQueryOptions
optionalPathTerms
  "Do not signal an error if instance variable missing along index path."

  ^ self new
    setOptionalPathTerms;
    yourself
%
category: 'Accessing'
method: GsQueryOptions
optionalPathTerms
  | val |
  val := self dynamicInstVarAt: #'optionalPathTerms'.
  val ifNil: [ ^ false ].
  ^ val
%
category: 'Accessing'
method: GsQueryOptions
optionalPathTerms: aBool
  self dynamicInstVarAt: #'optionalPathTerms' put: aBool
%
category: 'Operators'
method: GsQueryOptions
setOptionalPathTerms
  self optionalPathTerms: true
%
! ------------------- Class initializers

expectvalue true
doit
GsQuery initialize.
GsQueryFormula initialize. 
true
%

category: 'accessing'
method: RcUnicodeIndexSpecification
equalityIndexClass
  ^ self legacyIndex
    ifTrue: [ RcUnicodeRangeEqualityIndex ]
    ifFalse: [ GsUnicodeRangeEqualityIndex ]
%
category: 'Operators'
method: GsIndexOptions
clearLegacyIndex
  self legacyIndex: false.
  self _btreePlusIndex == false
    ifTrue: [ self setBtreePlusIndex ]
%
category: 'Operators'
method: GsIndexOptions
clearOptimizedComparison
  self optimizedComparison: false
%
category: 'Accessing'
classmethod: GsIndexOptions
legacyIndex
  "Create a legacy index that uses BtreeNodes instead of the new BtreePlusNodes for indexing."

  "Use basicNew to bypass initialization, thus making it possible to use the default index options
   to supply the value for slots not explicitly set or cleared."

  ^ self basicNew
    clearBtreePlusIndex; "record that btreePlusIndex is false"
    setLegacyIndex;
    yourself
%
category: 'Accessing'
method: GsIndexOptions
legacyIndex
  ^ self _legacyIndex ifNil: [ self defaultOptions legacyIndex ]
%
category: 'Private'
method: GsIndexOptions
legacyIndex: aBool
  "Private method ... should never be called by method external to GsIndexOptions. 
   Use set* and clear* methods instead."

  self dynamicInstVarAt: #'legacyIndex' put: aBool.
%
category: 'Accessing'
classmethod: GsIndexOptions
optimizedComparison
  "Eliminate per element comparison testing for indexed queries and use native #= 
   instead of _idxForCompare* methods when comparisons are needed. In order to be
   able to do these optimizations, the following constraints are imposed on the 
   last element values:
    - values must be a kind of the last element class. 
    - nil is not allowed as a value.
    - For Float last element class, a NaN is not allowed as a value.
    - For String last element class, Symbols are not allowed as a value. 
    - For Symbol last element class, Strings are not allowed as a value. 
   "

  "Use basicNew to bypass initialization, thus making it possible to use the default index options
   to supply the value for slots not explicitly set or cleared."

  ^ self basicNew
    optimizedComparison: true; "set directly, to allow optimizedComparision option to be set, when lagacyIndex is default (bug 47145)"
    yourself
%
category: 'Accessing'
method: GsIndexOptions
optimizedComparison
  ^ self _optimizedComparison 
    ifNil: [ 
    self _legacyIndex == true
      ifTrue: [ false ]
      ifFalse: [ self defaultOptions optimizedComparison ] ]
%
category: 'Private'
method: GsIndexOptions
optimizedComparison: aBool
  "Private method ... should never be called by method external to GsIndexOptions. 
   Use set* and clear* methods instead."

  self dynamicInstVarAt: #'optimizedComparison' put: aBool
%
category: 'Operators'
method: GsIndexOptions
setLegacyIndex
  self legacyIndex: true.
  self btreePlusIndex
    ifTrue: [ self clearBtreePlusIndex ].
  self optimizedComparison
    ifTrue: [ self clearOptimizedComparison ]
%
category: 'Operators'
method: GsIndexOptions
setOptimizedComparison
  self legacyIndex
    ifTrue: [ 
      "do not set optimizedComparison if legacyIndex already set"
      ^ self ].
  self optimizedComparison: true
%
category: 'querying-private'
method: GsPathConstantPredicate
prepareForExecution
  "Return an object equivalent to the receiver that is prepared for execution within 
   the context of a block."

  | ans |
  ans := GsPreparedPathConstantPredicate new
    path1: operand1;
    operator: operator;
    constant2: self constant;
    _nsc: nsc collator: collator;
    evaluator1: evaluator1;
    prepare.
  ^ans
%
category: 'querying-private'
method: GsRangeQueryPredicate
prepareForExecution
  "Return an object equivalent to the receiver that is prepared for execution within 
   the context of a block."

  | ans |
  ans := GsPreparedRangeQueryPredicate new
    constant1: self constant1;
    operator1: self operator1;
    path: self path;
    operator2: self operator2;
    constant2: self constant2;
    _nsc: nsc collator: collator;
    evaluator: self evaluator;
    prepare.
  ^ans
%
category: 'querying-private'
method: GsQueryFormula
prepareForExecution
  "Return an object equivalent to the receiver that is prepared for execution within 
   the context of a block."

  ^ self
%
category: 'accessing'
method: UnicodeIndexSpecification
equalityIndexClass
  ^ self legacyIndex
    ifTrue: [ UnicodeRangeEqualityIndex ]
    ifFalse: [ GsUnicodeRangeEqualityIndex ]
%
category: 'accessing'
method: UnicodeIndexSpecification
options
  "if constraintType is not set, then ensure that optimizedComparison option is cleared"

  self constraintType ifNotNil: [ ^ super options ].
  ^ super options clearOptimizedComparison
%
category: 'testing'
method: GsUnaryClause
_queryUsesIndex
  "answer true if the one or more of the clause has an evaluator that is a range or identity index"

  ^ clause _queryUsesIndex
%
category: 'testing'
method: GsQueryPredicate
usesIdenticalToOperation
  ^ operator == #'=='
%
category: 'testing'
method: GsCompoundClause
_queryUsesIndex
  "answer true if the one or more of the clauses has an evaluator that is a range or identity index"

  ^ clause1 _queryUsesIndex or: [ clause2 _queryUsesIndex ]
%
category: 'testing'
method: GsQueryFormula
_evaluatorCanStreamQueries
  "Answer true if the evaluator(s) support streaming operations"

  self _evaluators
    do: [ :each | 
      each isStreamable
        ifFalse: [ ^ false ] ].
  ^ true
%
category: 'testing'
method: GsQueryFormula
_queryUsesIndex
  "answer true if the one or more of the predicates has an evaluator that is a range or identity index"

  self _evaluators
    do: [ :each | 
      (each isPathEvaluator not
        and: [ each isIdentityIndex or: [ each isRangeEqualityIndex ] ])
        ifTrue: [ ^ true ] ].
  ^ false
%
category: 'testing'
method: GsQueryFormula
usesIdenticalToOperation
  ^ false
%
category: 'testing'
method: GsQuery
_queryUsesIndex
  "answer true if the one or more of the predicates has an evaluator that is a range or identity index"

  self _formula ifNil: [ ^ false ].
  ^ self _formula _queryUsesIndex
%
category: 'testing'
method: GsQuery
_queryUsesIdentityIndex
  "answer true if the one or more of the predicates has an evaluator that is an identity index"

  self _formula ifNil: [ ^ false ].
  ^ self _formula _queryUsesIdentityIndex
%
category: 'testing'
method: GsQuery
_queryUsesRangeEqualityIndex
  "answer true if the one or more of the predicates has an evaluator that is a range equality index"

  self _formula ifNil: [ ^ false ].
  ^ self _formula _queryUsesIndex and: [ self _formula _queryUsesIdentityIndex not ]
%
category: 'testing'
method: GsQueryFormula
_queryUsesIdentityIndex
  "answer true if the one or more of the predicates has an evaluator that is an identity index"

  self _evaluators
    do: [ :each | 
      (each isPathEvaluator not
        and: [ each isIdentityIndex ])
        ifTrue: [ ^ true ] ].
  ^ false
%
category: 'testing'
method: GsCompoundClause
_queryUsesIdentityIndex
  "answer true if the one or more of the clauses has an evaluator that is an range or identity index"

  ^ clause1 _queryUsesIdentityIndex or: [ clause2 _queryUsesIdentityIndex ]
%
category: 'testing'
method: GsUnaryClause
_queryUsesIdentityIndex
  "answer true if the one or more of the clause has an evaluator that is an identity index"

  ^ clause _queryUsesIdentityIndex
%
category: 'private'
method: GsQuery
_validateQueryableCollection: anNscOrSequenceableCollection
  "throw an error if the receiver does not support queries on anNscOrSequenceableCollection"

  anNscOrSequenceableCollection ifNil: [ ^ self ].
  anNscOrSequenceableCollection class _canCreateQueryOnInstances
    ifFalse: [ 
      ^ GsMalformedQueryExpressionError
        collectionIsNotQueryable:
          anNscOrSequenceableCollection class name asString
        senderLocationID: #'_validateQueryableCollection:' ]
%
category: 'Initialization'
method: GsMalformedQueryExpressionError
collectionIsNotQueryable: explanation
  "Collection is not queryable."

  self
    errorCondition: #'collectionIsNotQueryable';
    messageText:
        'A GsQuery may not be created on an instance of class ' , explanation , '.'
            , ' (#collectionIsNotQueryable).'
%
category: 'Instance Creation'
classmethod: GsMalformedQueryExpressionError
collectionIsNotQueryable: explanation senderLocationID: senderLocationID
  "Collection is not queryable."

  ^ self new
    collectionIsNotQueryable: explanation;
    reason: senderLocationID;
    signal
%
category: 'testing'
method: GsAbstractReferenceAssociation
hasEnumeratedTerm
  ^ false
%
category: 'testing'
method: GsQueryPathReferenceAssociation
hasEnumeratedTerm
  self asArrayOf32PathTerms
    do: [ :term | 
      (term includes: $|)
        ifTrue: [ ^ true ] ].
  ^ false
%
category: 'Default'
classmethod: GsIndexOptions
default: aGsIndexOptions
  "set persistent default options"

  ^ Globals at: #GsDefaultIndexOptions put: aGsIndexOptions _reify immediateInvariant
%
category: 'Printing'
method: GsIndexOptions
printOn: aStream
  self legacyIndex
    ifTrue: [ 
      self btreePlusIndex
        ifTrue: [ 
          "should never get here"
          self _errorIncorrectIndexTypeOptionsCombo ].
      aStream nextPutAll: 'GsIndexOptions legacyIndex'.
      self optimizedComparison
        ifTrue: [ 
          "should never get here"
           self _errorOptimizedComparisonAndLegacyIndexSet ].
      self optionalPathTerms
        ifTrue: [ aStream nextPutAll: ' + GsIndexOptions optionalPathTerms' ].
      self reducedConflict
        ifTrue: [ aStream nextPutAll: ' + GsIndexOptions reducedConflict' ] ]
    ifFalse: [ 
      self btreePlusIndex
        ifFalse: [ 
          "should never get here"
          self _errorIncorrectIndexTypeOptionsCombo ].
      aStream nextPutAll: 'GsIndexOptions btreePlusIndex'.
      self optimizedComparison
        ifTrue: [ aStream nextPutAll: ' + GsIndexOptions optimizedComparison']. 
      self optionalPathTerms
        ifTrue: [ aStream nextPutAll: ' + GsIndexOptions optionalPathTerms' ].
      self reducedConflict
        ifTrue: [ aStream nextPutAll: ' + GsIndexOptions reducedConflict' ] ] 
%
category: 'Private'
method: GsIndexOptions
_btreePlusIndex
  ^ self dynamicInstVarAt: #'btreePlusIndex'
%
category: 'Private'
method: GsIndexOptions
_legacyIndex
  ^ self dynamicInstVarAt: #'legacyIndex'
%
category: 'Private'
method: GsIndexOptions
_optimizedComparison
  ^ self dynamicInstVarAt: #'optimizedComparison'
%
category: 'Private'
method: GsIndexOptions
_optionalPathTerms
  ^ self dynamicInstVarAt: #'optionalPathTerms'
%
category: 'Private'
method: GsIndexOptions
_reducedConflict
  ^ self dynamicInstVarAt: #'reducedConflict'
%
category: 'Accessing'
classmethod: GsIndexOptions
btreePlusIndex
  "Create a btree plus index that uses the new BtreePlusNodes for indexing."

  "Use basicNew to bypass initialization, thus making it possible to use the default index options
   to supply the value for slots not explicitly set or cleared."

  "Only one of legacyIndex or btreePlusIndex may be set in an instance"

  ^ self basicNew
    clearLegacyIndex; "record that legacyIndex is false"
    setBtreePlusIndex;
    yourself
%
category: 'Accessing'
method: GsIndexOptions
btreePlusIndex
  ^ self _btreePlusIndex ifNil: [ self defaultOptions btreePlusIndex ]
%
category: 'Private'
method: GsIndexOptions
btreePlusIndex: aBool
  "Private method ... should never be called by method external to GsIndexOptions. 
   Use set* and clear* methods instead."

  self dynamicInstVarAt: #'btreePlusIndex' put: aBool.
%
category: 'Operators'
method: GsIndexOptions
clearBtreePlusIndex
  self btreePlusIndex: false.
  self _legacyIndex == false
    ifTrue: [ self setLegacyIndex ]
%
category: 'Default'
classmethod: GsIndexOptions
sessionDefault: aGsIndexOptions
  "set default index options for this session only"

  aGsIndexOptions ifNotNil: [ aGsIndexOptions _reify immediateInvariant ].
  ^ SessionTemps current at: #GsDefaultIndexOptions put: aGsIndexOptions
%
category: 'Operators'
method: GsIndexOptions
setBtreePlusIndex
  self btreePlusIndex: true.
  self legacyIndex
    ifTrue: [ self clearLegacyIndex ]
%
category: 'Accessing'
method: GsIndexOptions
defaultOptions

  ^ self class default
%
category: 'Private'
method: GsIndexOptions
_reify
  "Ensure that all slots are set or cleared , no nil slots ... nil slots should be set 
   consistent with current default options"

  | default |
  default := self defaultOptions.
  self _reducedConflict ifNil: [ self reducedConflict: default reducedConflict ].
  self _optionalPathTerms ifNil: [ self optionalPathTerms: default optionalPathTerms ].
  self _legacyIndex ifNil: [ self legacyIndex: default legacyIndex ].
  self _btreePlusIndex ifNil: [ self btreePlusIndex: default btreePlusIndex ].
  self _optimizedComparison ifNil: [ 
    self legacyIndex
      ifTrue: [ self optimizedComparison: false ]
      ifFalse: [ self optimizedComparison: default optimizedComparison ] ].
 self legacyIndex
   ifTrue: [ 
     self optimizedComparison ifTrue: [ self _errorOptimizedComparisonAndLegacyIndexSet ].
     self btreePlusIndex ifTrue: [ self _errorIncorrectIndexTypeOptionsCombo ]].
 self btreePlusIndex
   ifTrue: [ self legacyIndex ifTrue: [ self _errorIncorrectIndexTypeOptionsCombo ]].

%
category: 'Operators'
method: GsOptions
not

  ^ self negateWith: self class basicNew 
%
category: 'Errors'
method: GsIndexOptions
_errorIncorrectIndexTypeOptionsCombo

  InternalError signal: 'Invalid GsIndexOptions instance: Only one of either legacyIndex or btreePlusIndex options should be set.' 
%
category: 'Errors'
method: GsIndexOptions
_errorOptimizedComparisonAndLegacyIndexSet

  InternalError signal: 'Invalid GsIndexOptions instance: optimizedComparison should never be set when legacyIndex is set'
%
category: 'Private'
method: GsIndexSpec
_optimizedCharacterCollectionIndex: path constraintType: constraintType collator: collatorOrNil options: aGsIndexOptions
  "Specify that an equality index on the given path using the given index options be created.
     The btreePlusIndex and optimizedComparison options will be unconditionally set in the 
     index options.

   If the collatorOrNil is nil and unicode comparison mode is in effect (see 
     CharacterCollection class>>isInUnicodeComparisonMode), the default collator will be used
     for comparison operations in the index.

   If the collatorOrNil is nil and unicode comparison mode is NOT in effect (see 
     CharacterCollection class>>isInUnicodeComparisonMode), then the native String 
     comparison operators (#<, #<=, #=, #~, #>, #>=, #==, #~~) will be used for 
     comparison operations in the index.

   If the collatorOrNil is not nil, the given collator will be used for comparison operations 
     in the index, whether or not unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode).
"

  "constraintType may be: #unicodeString, #symbol, or #string:

     If #unicodeString, then the elements of the collection will be constrained to any type of 
     String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString, Unicode32) and a
     collator will be used for comparison.

     If #symbol, then the elements of the collection will be constrained to be a type of 
     Symbol (Symbol, DoubleByteSymbol, or QuadByteSymbol).

     If #string, then the elements of the collection will be constrained based on whether
       or not unicode comparison mode is in effect or not (see CharacterCollection 
       class>>isInUnicodeComparisonMode).
   
       When unicode comparison mode is in effect the elements of the collection will be 
         constrained to be any type of String (String, Unicode7, DoubleByteString, Unicode16, 
         QuadByteString,  Unicode32).

       When unicode comparison mode is NOT in effect the elements of the collection will be 
         constrained to be a type of non-unicode String (String, DoubleByteString, 
         QuadByteString).
"

  | options spec |
  options := aGsIndexOptions + GsIndexOptions btreePlusIndex + GsIndexOptions optimizedComparison.
  spec := constraintType == #unicodeString
    ifTrue: [ | collator |
      collator := collatorOrNil ifNil: [ self defaultCollator ].
      (UnicodeIndexSpecification 
        path: (self preparedPath: path) 
        collator: collator)
          constraintType: constraintType; "#unicodeString"
          yourself]
    ifFalse: [ | theCollator |
      theCollator := collatorOrNil 
        ifNil: [ 
          CharacterCollection isInUnicodeComparisonMode 
            ifTrue: [ 
              "in unicode comparison mode, we always have a collator"
              self defaultCollator] 
            ifFalse: [ nil ] ]
        ifNotNil: [ collatorOrNil ].
      (ConstrainedCharacterCollectionIndexSpecification 
        path: (self preparedPath: path) 
        collator: theCollator)
          constraintType: constraintType;
          yourself ].
  spec options: options.
  self specs add: spec
%
category: 'Specification'
method: GsIndexSpec
stringOptimizedIndex: path
  "Specify that an equality index on the given path be created. The btreePlusIndex and 
     optimizedComparison options will be unconditionally set in the index options.

   When unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString, 
     Unicode32). The current default collator will be used for comparison operations in the
     index.

   When unicode comparison mode is NOT in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be a type of non-unicode String (String, DoubleByteString, QuadByteString). The 
     native String comparison operators (#<, #<=, #=, #~, #>, #>=, #==, #~~) will be used for 
     comparison operations in the index.
"

  self
    stringOptimizedIndex: path
    options: self defaultOptions
%
category: 'Specification'
method: GsIndexSpec
stringOptimizedIndex: path options: aGsIndexOptions
  "Specify that an equality index on the given path using the given index options be created. 
     The btreePlusIndex and optimizedComparison options will be unconditionally set in the
     index options.

   When unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString, 
     Unicode32). The current default collator will be used for comparison operations in the
     index.

   When unicode comparison mode is NOT in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be a type of non-unicode String (String, DoubleByteString, QuadByteString). The 
     native String comparison operators (#<, #<=, #=, #~, #>, #>=, #==, #~~) will be used 
     for comparison operations in the index.
"

  self 
    _optimizedCharacterCollectionIndex: path
    constraintType: #string
    collator: nil
    options: aGsIndexOptions
%
category: 'Specification'
method: GsIndexSpec
symbolOptimizedIndex: path
  "Specify that an equality index on the given path be created. The btreePlusIndex and 
     optimizedComparison options will be unconditionally set in the index options.

   When unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be any type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The current default 
     collator will be used for comparison operations in the index.

   When unicode comparison mode is NOT in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be a type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The native String 
     comparison operators (#<, #<=, #=, #~, #>, #>=, #==, #~~) will be used for 
     comparison operations in the index.
"

  self
    symbolOptimizedIndex: path
    options: self defaultOptions
%
category: 'Specification'
method: GsIndexSpec
symbolOptimizedIndex: path collator: icuCollator
  "Specify that an equality index on the given path be created. The btreePlusIndex and 
     optimizedComparison options will be unconditionally set in the index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be any type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The given 
     collator will be used for comparison operations in the index.
"

  self
    symbolOptimizedIndex: path
    collator: icuCollator
    options: self defaultOptions
%
category: 'Specification'
method: GsIndexSpec
symbolOptimizedIndex: path collator: icuCollator options: aGsIndexOptions
  "Specify that an equality index on the given path using the given index options be created. 
     The btreePlusIndex and optimizedComparison options will be unconditionally set in the 
     index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be any type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The given 
     collator will be used for comparison operations in the index.
"

  self 
    _optimizedCharacterCollectionIndex: path
    constraintType: #symbol
    collator: icuCollator
    options: aGsIndexOptions
%
category: 'Specification'
method: GsIndexSpec
symbolOptimizedIndex: path options: aGsIndexOptions
  "Specify that an equality index on the given path using the given index options be created. 
     The btreePlusIndex and optimizedComparison options will be unconditionally set in the
     index options.

   When unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be any type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The current default 
     collator will be used for comparison operations in the index.

   When unicode comparison mode is NOT in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be a type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The native String 
     comparison operators (#<, #<=, #=, #~, #>, #>=, #==, #~~) for comparison 
     operations in the index.
"

  self 
    _optimizedCharacterCollectionIndex: path
    constraintType: #symbol
    collator: nil
    options: aGsIndexOptions
%
category: 'Specification'
method: GsIndexSpec
unicodeStringOptimizedIndex: path
  "Specify that an equality index on the given path be created using the default collator and
     default index options. The btreePlusIndex and optimizedComparison options will be 
     unconditionally set in the index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString, 
     Unicode32). The current default collator will be used for comparison operations in the
     index.
"

  self
    unicodeStringOptimizedIndex: path
    collator: self defaultCollator
    options: self defaultOptions
%
category: 'Specification'
method: GsIndexSpec
unicodeStringOptimizedIndex: path collator: collator
  "Specify that an equality index on the given path be created using the given collator and
     the default index options. The btreePlusIndex and optimizedComparison options will be 
     unconditionally set in the index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString, 
     Unicode32). The current default collator will be used for comparison operations in the
     index.
"

  self
    unicodeStringOptimizedIndex: path
    collator: collator
    options: self defaultOptions
%
category: 'Specification'
method: GsIndexSpec
unicodeStringOptimizedIndex: path collator: collator options: aGsIndexOptions
  "Specify that an equality index on the given path be created using the given collator and
     the given index options. The btreePlusIndex and optimizedComparison options will be 
     unconditionally set in the index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString, 
     Unicode32). The current default collator will be used for comparison operations in the
     index.
"

  self
    _optimizedCharacterCollectionIndex: path
    constraintType: #'unicodeString'
    collator: collator
    options: aGsIndexOptions
%
category: 'Specification'
method: GsIndexSpec
unicodeStringOptimizedIndex: path options: aGsIndexOptions
  "Specify that an equality index on the given path be created using the default collator and
     the given index options. The btreePlusIndex and optimizedComparison options will be 
     unconditionally set in the index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection 
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to 
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString, 
     Unicode32). The current default collator will be used for comparison operations in the
     index.
"

  self
    unicodeStringOptimizedIndex: path
    collator: self defaultCollator
    options: aGsIndexOptions
%
category: 'accessing'
method: UnicodeIndexSpecification
constraintType
  ^ self dynamicInstVarAt: #'constraintType'
%
category: 'accessing'
method: UnicodeIndexSpecification
constraintType: aSymbolOrNil
  " #unicode ... if not nil, optimizedComparison option should be set as well. "

  self dynamicInstVarAt: #'constraintType' put: aSymbolOrNil
%
