category: 'Message Digests'
method:
_asMessageDigestKind: opCode

"
opCode   Digest Algorithm   Digest Bits
=======================================
  1          md5              128
  2          sha1             160
  3          sha2-256         256
  4          sha2-512         512
  5          sha3-224         224
  6          sha3-256         256
  7          sha3-384         384
  8          sha3-256         512
=======================================

"
<primitive: 666>
opCode _validateClass: SmallInteger .
((opCode < 1) or:[ opCode > 8])
  ifTrue:[ opCode _error: #rtErrArgOutOfRange args:{ 1 . 8 } ].
^ self _primitiveFailed: #_asMessageDigestKind:
%

category: 'Message Digests'
method:
md5sum

"Return the 128 bit MD5 checksum of the receiver as a LargeInteger. 

 Computation is per RFC 1321 , http://www.ietf.org/rfc/rfc1321.txt,
 using L. Peter Deutsch's C implementation from
 http://sourceforge.net/projects/libmd5-rfc/

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
" 

^ self _asMessageDigestKind: 1
%

category: 'Message Digests'
method:
asMd5String

"Compute the 128 bit MD5 message digest for the receiver and return it as
 a 32 character string of hexadecimal characters."

^ self md5sum asHexStringWithLength: 32
%

category: 'Message Digests'
method:
sha1Sum

"Return the 160 bit SHA1 checksum of the receiver as a LargeInteger. 

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
" 

^ self _asMessageDigestKind: 2
%

category: 'Message Digests'
method:
asSha1String

"Compute the 160 bit SHA1 message digest for the receiver and return it as
 a 40 character string of hexadecimal characters."

^ self sha1Sum asHexStringWithLength: 40
%

category: 'Message Digests'
method:
sha256Sum

"Return the 256 bit SHA256 checksum of the receiver as a LargeInteger. 

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
" 

^ self _asMessageDigestKind: 3
%

category: 'Message Digests'
method:
asSha256String

"Compute the 256 bit SHA-2 message digest for the receiver and return it as
 a 64 character string of hexadecimal characters."

^ self sha256Sum asHexStringWithLength: 64
%

category: 'Message Digests'
method:
sha512Sum

"Return the 512 bit SHA512 checksum of the receiver as a LargeInteger. 

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
" 

^ self _asMessageDigestKind: 4
%

category: 'Message Digests'
method:
asSha512String

"Compute the 512 bit SHA-2 message digest for the receiver and return it as
 a 128 character string of hexadecimal characters."

^ self sha512Sum asHexStringWithLength: 128
%

category: 'Message Authentication Codes'
method:
asHmacSumWithDigestKind: opCode key: keyString

"Computes the keyed-hash message authentication code (HMAC) of the receiver using
 the message digest algorithm indicated by opCode and secret key keyString.

 opCode must be one of the following:

 opCode   Digest Algorithm
 =========================
  1          md5         
  2          sha1        
  3          sha2-256      
  4          sha2-512      
  5          sha3-224
  6          sha3-256
  7          sha3-384
  8          sha3-256
 =========================

 secretKey must be an instance or subclass of a ByteArray or String and must
 have a character size of one, i.e.: its class must answer 1 when sent the 
 message #_bytesPerWord. secretKey must have a size between 1 and 64 bytes. 

 Answers the HMAC of the receiver as a LargeInteger."

<primitive: 1060>
opCode _validateClass: SmallInteger .
((opCode < 1) or:[ opCode > 8]) ifTrue:[ opCode _error: #rtErrArgOutOfRange args:{ 1 . 8 } ].
keyString _validateClasses: { String . ByteArray } .
((keyString _basicSize > 64) or:[keyString _basicSize < 1])
  ifTrue:[ keyString _error: #rtErrBadSize args: { 64 . keyString _basicSize } ] .
self _primitiveFailed: #asHmacSumWithDigestKind:key: args: { opCode . keyString } .
%

category: 'Message Authentication Codes'
method:
asMd5HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the MD5 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a LargeInteger."
 
^ self asHmacSumWithDigestKind: 1 key: keyString
%

category: 'Message Authentication Codes'
method:
asMd5HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the MD5 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asMd5HmacWithKey: keyString) asHexStringWithLength: 32
%

category: 'Message Authentication Codes'
method:
asSha1HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the sha1 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a LargeInteger."
 
^ self asHmacSumWithDigestKind: 2 key: keyString
%

category: 'Message Authentication Codes'
method:
asSha1HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the sha1 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha1HmacWithKey: keyString) asHexStringWithLength: 40
%

category: 'Message Authentication Codes'
method:
asSha256HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the sha256 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a LargeInteger."
 
^ self asHmacSumWithDigestKind: 3 key: keyString
%

category: 'Message Authentication Codes'
method:
asSha256HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the sha256 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha256HmacWithKey: keyString) asHexStringWithLength: 64
%

category: 'Message Authentication Codes'
method:
asSha512HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the sha512 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a LargeInteger."
 
^ self asHmacSumWithDigestKind: 4 key: keyString
%

category: 'Message Authentication Codes'
method:
asSha512HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the sha512 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha512HmacWithKey: keyString) asHexStringWithLength: 128
%

category: 'Converting'
method:
asBase64StringOnOneLine: aBoolean
"Return a String which represents the receiver represented in base64 format.
 If aBoolean is true, the resulting is one long line which does not contain
 newline characters.  If aBoolean is false, newline characters are inserted
 such that each line does not exceed 64 characters."
 
<primitive: 1062>
aBoolean _validateClass: Boolean .
self _primitiveFailed: #asBase64StringOnOneLine:
%

category: 'Converting'
method:
asBase64String
"Return a String which represents the receiver represented in base64 format."

^ self asBase64StringOnOneLine: true
%

! Begin SHA3 methods

category: 'Message Digests'
method:
sha3_224Sum

"Return the SHA3 224 bit checksum of the receiver as a LargeInteger. 

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
" 

^ self _asMessageDigestKind: 5
%
category: 'Message Digests'
method:
asSha3_224String

"Compute the SHA3 224 bit message digest for the receiver and return it as
 a 56 character string of hexadecimal characters."

^ self sha3_224Sum asHexStringWithLength: 56
%
category: 'Message Digests'
method:
sha3_256Sum

"Return the SHA3 256 bit checksum of the receiver as a LargeInteger. 

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
" 

^ self _asMessageDigestKind: 6
%
category: 'Message Digests'
method:
asSha3_256String

"Compute the SHA3 256 bit message digest for the receiver and return it as
 a 64 character string of hexadecimal characters."

^ self sha3_256Sum asHexStringWithLength: 64
%
category: 'Message Digests'
method:
sha3_384Sum

"Return the SHA3 384 bit checksum of the receiver as a LargeInteger. 

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
" 

^ self _asMessageDigestKind: 7
%
category: 'Message Digests'
method:
asSha3_384String

"Compute the SHA3 384 bit message digest for the receiver and return it as
 a 96 character string of hexadecimal characters."

^ self sha3_384Sum asHexStringWithLength: 96
%
category: 'Message Digests'
method:
sha3_512Sum

"Return the SHA3 512 bit checksum of the receiver as a LargeInteger. 

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
" 

^ self _asMessageDigestKind: 8
%

category: 'Message Digests'
method:
asSha3_512String

"Compute the SHA3 512 bit message digest for the receiver and return it as
 a 128 character string of hexadecimal characters."

^ self sha3_512Sum asHexStringWithLength: 128
%

!
! Add SHA3 HMAC Support
!
category: 'Message Authentication Codes'
method:
asSha3_224HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the SHA3 224 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a LargeInteger."
 
^ self asHmacSumWithDigestKind: 5 key: keyString
%
category: 'Message Authentication Codes'
method:
asSha3_224HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the SHA3 224 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_224HmacWithKey: keyString) asHexStringWithLength: 56
%
category: 'Message Authentication Codes'
method:
asSha3_256HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the SHA3 256 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a LargeInteger."
 
^ self asHmacSumWithDigestKind: 6 key: keyString
%
category: 'Message Authentication Codes'
method:
asSha3_256HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the SHA3 256 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_256HmacWithKey: keyString) asHexStringWithLength: 64
%
category: 'Message Authentication Codes'
method:
asSha3_384HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the SHA3 384 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a LargeInteger."
 
^ self asHmacSumWithDigestKind: 7 key: keyString
%
category: 'Message Authentication Codes'
method:
asSha3_384HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the SHA3 384 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_384HmacWithKey: keyString) asHexStringWithLength: 96
%
category: 'Message Authentication Codes'
method:
asSha3_512HmacWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the SHA3 512 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a LargeInteger."
 
^ self asHmacSumWithDigestKind: 8 key: keyString
%
category: 'Message Authentication Codes'
method:
asSha3_512HmacStringWithKey: keyString
"Computes the keyed-hash message authentication code (HMAC) of the receiver 
 using the SHA3 512 message digest algorithm and secret key keyString. Answers 
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_512HmacWithKey: keyString) asHexStringWithLength: 128
%
