!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   PathTerm, Array, SequenceableCollection, Collection, Object.
!
!=========================================================================

! class created in idxclasses.topaz

removeallmethods PathTerm
removeallclassmethods PathTerm

category: 'For Documentation Installation only'
classmethod: PathTerm
installDocumentation

self comment: '
PathTerm represents a single term/instance variable within an index or query
path. PathTerm provides a simple case; subclasses of PathTerm are used for 
specialized features, such as set-valued paths. 

PathTerm implements only GemStone internals, and is not intended for direct use
by customers. 

Constraints:
	name: CharacterCollection
	offset: SmallInteger
	securityPolicies: Object
	children: Object
	updateBtree: Object
	updateDict: Object
	needsDepList: Boolean'.
%

! ------------------- Class methods for PathTerm
category: 'Instance Creation'
classmethod: PathTerm
new

"Create an initialized instance of the receiver."

^ self basicNew initialize
%

! ------------------- Instance methods for PathTerm
category: 'private'
method: PathTerm
_findAllValuesForIdenticalRootObject: rootObject
	" if index directly on NSC elements or anObject is nil "

	| key tmpList |
	(self indicatesIndexOnNscElements or: [ nil == rootObject ])
		ifTrue: [ key := rootObject ]
		ifFalse: [ key := self _nextObjectFor: rootObject ].
	tmpList := IdentityBag new.
	self updateBtree btreeRoot _findAllValuesForIdenticalKey: key into: tmpList.
	^ (tmpList occurrencesOf: rootObject) <= 1
%
category: 'Updating Indexes'
method: PathTerm
addDirectMappingFor: anObject logging: aBoolean
  " Update the B-tree and the dependency list for anObject."

  updateBtree ~~ nil
    ifTrue: [ 
      updateBtree btreeAt: anObject put: anObject.
      needsDepList
        ifTrue: [ 
          " add an entry in the dependency list "
          anObject getDepListAndAddLastElementPathTerm: self logging: aBoolean ] ].
  updateDict ~~ nil
    ifTrue: [ 
      " insert an entry into the index dictionary "
      updateDict
        _at: anObject
        put: anObject
        term: self
        logging: aBoolean ]
%

category: 'Updating Indexes'
method: PathTerm
addMappingsUsing: mapInfo logging: aBoolean
  "Adds index dictionary and dependency list entries for the object described by
 the map info object.  If aBoolean is true, operations will be recorded in the
 system redo log."

  | pathTerm object nextObj nextMap |
  pathTerm := mapInfo pathTerm.
  pathTerm size == 0
    ifTrue: [ ^ self ].
  object := mapInfo object.
  nil == object
    ifTrue: [ ^ self ].
  object
    getDepListAndAddPathTerm: pathTerm
    withIVOffset: mapInfo ivOffset
    logging: aBoolean.	" add an entry to the objects dependency list "
  nextMap := mapInfo at: 1.	" nextObj is the same for all nextMaps, so choose the first one "
  nextObj := nextMap object.
  updateBtree ~~ nil
    ifTrue: [ 
      " insert an entry into the B-tree "
      updateBtree btreeAt: nextObj put: object.
      (needsDepList and: [ nil ~~ nextObj ])
        ifTrue: [ 
          " add an entry to the very last object's dependency list "
          nextObj
            getDepListAndAddLastElementPathTerm: pathTerm
            logging: aBoolean ] ].
  updateDict ~~ nil
    ifTrue: [ 
      " insert an entry into the index dictionary "
      updateDict
        _at: nextObj
        put: object
        term: pathTerm
        logging: aBoolean ].
  1 to: mapInfo size do: [ :i | 
    nextMap := mapInfo at: i.	" make recursive call to add mappings "
    nextMap pathTerm ~~ nil
      ifTrue: [ nextMap pathTerm addMappingsUsing: nextMap logging: aBoolean ] ]
%

category: 'Accessing'
method: PathTerm
_ivOffsetFor: anObject

"Returns the instance variable offset of anObject for the instance variable
 corresponding to the receiver path term."

^ (self _classOf: anObject) _ivOffsetOf: name
%

category: 'Accessing'
method: PathTerm
getRootNsc

"Returns the root NSC for the index in which the receiver is a term."

^ (self at: 1) nscRoot
%

category: 'Accessing'
method: PathTerm
getIndexDictionary

"Returns the index dictionary associated with the receiver."

| indexDict |
updateDict ~~ nil
    ifTrue: [ ^ updateDict ].

" get it from an index object if present "
1 to: self size do: [ :i |
    indexDict := (self at: i) indexDictionary.
    indexDict ~~ nil
        ifTrue: [ ^ indexDict ]
].
^ nil
%

! edited for 43084
category: 'Updating Indexes'
method: PathTerm
getMappingInfoFor: anObject ifObject: origObject atOffset: origOffset replaceWith: otherObject
  "Gathers all the objects, their dependency lists, and the instance variable
 offset for each path term in the tree of path terms.  Puts them in a
 MappingInfo object.  Builds a tree of MappingInfo objects for each child path
 term.

 It is possible that this method is invoked due to an instance variable
 modification.  In that case, we want to traverse the path terms as if the
 modification had occurred.  Thus, if anObject is the one that was modified
 (that is, if it is identical to origObject) and the instance variable offset of
 this path term is the same instance variable offset that was modified (that is,
 it is equal to origOffset), then rather than getting the next object along the
 path, use the otherObject."

  | ivOffset nextObj mapInfo aMapInfo |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self _mappingInfoClass new pathTerm: self ].
  ivOffset := self _ivOffsetFor: anObject.
  ivOffset ifNil: [ ^ self _mapInfoOrErrorForMissingIvOffset: anObject ].
  mapInfo := self _mappingInfoClass new
    object: anObject;
    ivOffset: ivOffset;
    pathTerm: self.
  (anObject == origObject and: [ ivOffset == origOffset ])
    ifTrue: [ nextObj := otherObject ]
    ifFalse: [ 
      " get the next object along the path "
      nextObj := self _nextObjectFor: anObject atInstVar: ivOffset ].
  updateBtree
    ifNotNil: [ 
      (self _checkBtreeComparisonWith: nextObj)
        ifFalse: [ 
          | exa |
          (exa := ImproperOperation new)
            _number:
                (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
            args: { nextObj class. self name. updateBtree lastElementClassDescription }.
          ^ {false.	" indicating no indexing objects were modified "
          exa} ] ].
  (nil == nextObj or: [ self children isEmpty ])
    ifTrue: [ 
      " adds nil to the end and returns "
      mapInfo addLast: (self _mappingInfoClass new object: nextObj).
      ^ mapInfo ].
  1 to: children size do: [ :i | 
    " make recursive call to get mapping info "
    aMapInfo := (children at: i)
      getMappingInfoFor: nextObj
      ifObject: origObject
      atOffset: origOffset
      replaceWith: otherObject.
    aMapInfo class == self _mappingInfoClass
      ifFalse: [ ^ aMapInfo	"an Array of error info" ].
    mapInfo addLast: aMapInfo ].
  ^ mapInfo
%

category: 'Accessing'
method: PathTerm
getParentTerm

"Returns the parent path term of the receiver.  This is determined by looking
 at the previous path term of the first index object that utilizes this path
 term.  If there is no parent, returns nil."

offset > 1
    ifTrue: [ ^ (self at: 1) at: offset - 1 ]
    ifFalse: [ ^ nil ]
%

category: 'Testing'
method: PathTerm
indicatesIndexOnNscElements

"Returns true if the path term indicates that the index is on elements of the
 NSC itself.  This is the case if the path is '' (two single quotes).  This
 method is overridden by SetValuedPathTerm to return true."

^ name ==  #'' 
%

category: 'Testing'
method: PathTerm
indicatesIndexOnRootNsc

"Returns true if the path term indicates that the index is on elements of the
 NSC itself.  This is the case if the path is '' (two single quotes).  This
 method is not overridden by SetValuedPathTerm."

^ name ==  #'' 
%

category: 'Testing'
method: PathTerm
isRangeEqualityIndexLastPathTerm

"Returns true if the receiver is the last path term of a RangeEqualityIndex."

1 to: self size do: [ :i |
    ( (self at: i) isRangeEqualityIndex and: [ (self at: i) size == offset ] )
        ifTrue: [ ^ true ]
].
^ false
%

category: 'Accessing'
method: PathTerm
name

"Returns the value of the instance variable 'name'."

^name
%

category: 'Updating'
method: PathTerm
name: newValue

"Modify the value of the instance variable 'name'."

name := newValue
%

category: 'Accessing'
method: PathTerm
offset

"Returns the value of the instance variable 'offset'."

^offset
%

category: 'Updating'
method: PathTerm
offset: newValue

"Modify the value of the instance variable 'offset'."

offset := newValue
%

category: 'Accessing'
method: PathTerm
updateBtree

"Returns the value of the instance variable 'updateBtree'."

^updateBtree
%

category: 'Updating'
method: PathTerm
updateBtree: newValue

"Modify the value of the instance variable 'updateBtree'."

updateBtree := newValue
%

category: 'Accessing'
method: PathTerm
updateDict

"Returns the value of the instance variable 'updateDict'."

^updateDict
%

category: 'Updating'
method: PathTerm
updateDict: newValue

"Modify the value of the instance variable 'updateDict'."

updateDict := newValue
%

category: 'Accessing'
method: PathTerm
needsDepList

"Returns the value of the instance variable 'needsDepList'."

^needsDepList
%

category: 'Updating'
method: PathTerm
needsDepList: newValue

"Modify the value of the instance variable 'needsDepList'."

needsDepList := newValue
%

category: 'Updating Indexes'
method: PathTerm
undoDirectMappingFor: anObject logging: doLogging

"There is a range index directly on the elements of the NSC."

updateDict ~~ nil
    ifTrue: [ " remove the index dictionary entry "
        updateDict
            removeKey: anObject
            value: anObject
            term: self
            logging: doLogging
    ]
%

category: 'Updating Indexes'
method: PathTerm
removeDirectMappingFor: anObject logging: doLogging
  "There is a range index directly on the elements of the NSC.  Update the
 B-tree."

  updateBtree ~~ nil
    ifTrue: [ 
      " remove it from the B-tree "
      updateBtree btreeRemoveKey: anObject value: anObject ].
  (self indicatesMultiValue _and: [ self indicatesIndexOnNscElements ])
    ifTrue: [ 
      " if it is a set-valued path term and the last term in an index... "
      ^ self ].
  updateDict ~~ nil
    ifTrue: [ 
      " remove the index dictionary entry "
      updateDict
        removeKey: anObject
        value: anObject
        term: self
        logging: doLogging ]
%

category: 'Index Maintenance'
method: PathTerm
removeIndex: indexObj

"The given index is being removed from this path term."

1 to: self size do: [ :i |
    indexObj == (self at: i)
        ifTrue: [ ^ self removeAtIndex: i ]
]

%

category: 'Testing'
method: PathTerm
isConstrained

"Returns false: the path term is not constrained."

^ false
%

category: 'Initialization'
method: PathTerm
initialize

"Initialize the receiver with default values."

securityPolicies := GsObjectSecurityPolicySet new.
children := { } .
needsDepList := false.
%

category: 'Accessing'
method: PathTerm
children

"Returns the value of the instance variable 'children'."

^children
%

category: 'Updating'
method: PathTerm
children: newValue

"Modify the value of the instance variable 'children'."

children := newValue
%

category: 'Updating'
method: PathTerm
_determineChildren
  "Determine the children of the receiver by getting the next path term for each
 index object that utilizes the receiver path term."

  | indexObj |
  children size: 0.
  1 to: self size do: [ :i | 
    indexObj := self at: i.
    (indexObj size > offset
      and: [ (children includesIdentical: (indexObj at: offset + 1)) not ])
      ifTrue: [ self _addChildTerm: (indexObj at: offset + 1) ] ]
%

category: 'Updating Indexes'
method: PathTerm
removeMappingsFor: anObject lastOne: aBoolean logging: doLogging
  "Remove entries in the index dictionary and dependency lists for anObject."

  | nextObj depList more isLast |
  self size == 0
    ifTrue: [ ^ self ].
  nextObj := self _nextObjectFor: anObject.
  updateBtree ~~ nil
    ifTrue: [ 
      " remove an entry from the B-tree "
      updateBtree btreeRemoveKey: nextObj value: anObject.
      DependencyList removePathTerm: self for: nextObj logging: doLogging	" remove dependency list entry for next object " ].
  more := false.
  updateDict ~~ nil
    ifTrue: [ 
      " remove the index dictionary entry "
      more := updateDict
        removeKey: nextObj
        value: anObject
        term: self
        logging: doLogging ].
  (nil ~~ nextObj and: [ children isEmpty not ])
    ifTrue: [ 
      " make recursive call to remove mappings "
      isLast := aBoolean and: [ more not ].
      1 to: children size do: [ :i | (children at: i) removeMappingsFor: nextObj lastOne: isLast logging: doLogging ].
      isLast
        ifTrue: [ 
          " remove dependency list entries for next object "
          (depList := DependencyList for: nextObj) ~~ nil
            ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ] ] ]
%


category: 'Updating Indexes'
method: PathTerm
removeMappingsSkipBtreeFor: anObject lastOne: aBoolean logging: doLogging

"Remove entries in the index dictionary and dependency lists for anObject.
 This version for index removal, it does not update the btree since the btree
 is going away (#40858)."

| nextObj depList more isLast |
self size == 0
  ifTrue: [ ^ self ].

nextObj := self _nextObjectFor: anObject.

updateBtree ~~ nil
  ifTrue: [
    " skip update of btree, index is going away"

    " remove dependency list entry for next object "
    DependencyList removePathTerm: self for: nextObj logging: doLogging.
  ].

more := false.
updateDict ~~ nil
  ifTrue: [
    " remove the index dictionary entry "
    more := updateDict removeKey: nextObj value: anObject term: self logging: doLogging.
  ].

( nil ~~ nextObj and: [ children isEmpty not ] )
  ifTrue: [ " make recursive call to remove mappings "
    isLast := aBoolean and: [ more not ].
    1 to: children size do: [ :i |
      (children at: i) removeMappingsFor: nextObj
        lastOne: isLast
        logging: doLogging
    ].
    isLast
      ifTrue: [
        " remove dependency list entries for next object "
        (depList := DependencyList for: nextObj) ~~ nil
          ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ]
      ]
  ]
%

category: 'Updating Indexes'
method: PathTerm
undoMappingsFor: anObject lastOne: aBoolean logging: doLogging

"Remove entries in the index dictionary and dependency lists for anObject."


| nextObj depList more isLast |
self size == 0
  ifTrue: [ ^ self ].

[
    nextObj := self _nextObjectFor: anObject.
] onException: Error do:[ :ex |  
    "Handle any error thrown during execution of _nextObjectFor:
     Cannot resolve the path for this guy --- can't have been added to index,
     in fact this object could have caused the failure in the first place"
    ^ self
].

updateBtree ~~ nil
  ifTrue: [
    " remove dependency list entry for next object "
    DependencyList removePathTerm: self for: nextObj logging: doLogging.
  ].

more := false.
updateDict ~~ nil
  ifTrue: [
    " remove the index dictionary entry "
    more := updateDict removeKey: nextObj value: anObject term: self logging: doLogging.
  ].

( nil ~~ nextObj and: [ children isEmpty not ] )
  ifTrue: [ " make recursive call to remove mappings "
    isLast := aBoolean and: [ more not ].
    1 to: children size do: [ :i |
      (children at: i) undoMappingsFor: nextObj
        lastOne: isLast
        logging: doLogging
    ].
    isLast
      ifTrue: [
        " remove dependency list entries for next object "
        (depList := DependencyList for: nextObj) ~~ nil
          ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ]
      ]
  ]
%

! edited for 43084
category: 'Updating Indexes'
method: PathTerm
update: anObject at: ivOffset to: newValue
  "The instance variable for anObject is being changed to newValue.
 Update the index dictionary and dependency lists for the old value and the
 newValue.  Returns true if the index objects were modified correctly;
 otherwise returns an Array containing error information."

  | maps depList more continue occurrences mapInfo doLogging oldValue exa |
  self size == 0
    ifTrue: [ ^ true ].
  maps := {}.
  self getIndexDictionary
    ifNotNil: [ " get the mapping info for all next path terms "
      1 to: children size do: [ :i | mapInfo := (children at: i)
          getMappingInfoFor: newValue
          ifObject: anObject
          atOffset: ivOffset
          replaceWith: newValue.
        mapInfo class == self _mappingInfoClass
          ifFalse: [ ^ mapInfo	"an Array of error info" ].
        maps addLast: mapInfo ] ].
  doLogging := UnorderedCollection _isRcIndexLoggingEnabled.
  oldValue := anObject instVarAt: ivOffset.
  occurrences := 0.
  updateBtree ifNotNil: [ (self _checkBtreeComparisonWith: newValue)
        ifFalse: [ (exa := ImproperOperation new)
            args: { newValue class. self name. updateBtree lastElementClassDescription};
            _number:
                (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree').
          ^ {false.	" indicating no indexing objects were modified "
          exa} ].	" remove B-tree entry for range index "
      [ updateBtree btreeRemoveKey: oldValue value: anObject ]
        whileTrue: [ occurrences := occurrences + 1 ].
      occurrences == 0
        ifTrue: [ (exa := ImproperOperation new)
            object: oldValue;
            arg: anObject;
            _number:
                (ErrorSymbols at: #'rtErrRangeEqualityIndexObjectNotInBtree').
          ^ {false.	" indicating no indexing objects were modified "
          exa} ].
      (needsDepList and: [ nil ~~ oldValue ])
        ifTrue: [ occurrences
            timesRepeat: [ DependencyList removePathTerm: self for: oldValue logging: doLogging ] ] ].
  updateDict ifNotNil: [ continue := true.
      [ continue ] whileTrue: [ " remove the index dictionary entry "
          more := updateDict
            removeKey: oldValue
            value: anObject
            term: self
            logging: doLogging
            ifAbsent: [ continue := false ].
          occurrences := occurrences + 1 ].
      occurrences := occurrences - 1.
      oldValue ifNotNil: [ " make recursive call to remove mappings "
          1 to: children size do: [ :i | occurrences timesRepeat: [ (children at: i)
                  removeMappingsFor: oldValue
                  lastOne: more not
                  logging: doLogging ] ].
          more
            ifFalse: [ depList := DependencyList for: oldValue.
              depList
                ifNotNil: [ depList removePathTerms: children for: oldValue logging: doLogging ] ] ] ].
  updateBtree ifNotNil: [ " insert an entry into the B-tree "
      occurrences timesRepeat: [ updateBtree btreeAt: newValue put: anObject ] ].
  updateDict ifNotNil: [ " insert an entry into the index dictionary "
      occurrences timesRepeat: [ updateDict
            _at: newValue
            put: anObject
            term: self
            logging: doLogging ] ].
  (needsDepList and: [ newValue ~~ nil ])
    ifTrue: [ occurrences
        timesRepeat: [ newValue getDepListAndAddLastElementPathTerm: self logging: true ] ].
  updateBtree ifNil: [ " make recursive call to add mappings "
      1 to: maps size do: [ :i | mapInfo := maps at: i.
        occurrences
          timesRepeat: [ mapInfo pathTerm addMappingsUsing: mapInfo logging: doLogging ] ] ].
  ^ true
%

category: 'Accessing'
method: PathTerm
_classOf: anObject

"Returns the class of the given object."

<primitive: 166>
self _primitiveFailed: #_classOf: args: { anObject } .
self _uncontinuableError
%

category: 'Accessing'
method: PathTerm
_nextObjectFor: anObject

"Returns the object at the instance variable that corresponds to the receiver
 path term."


^ self _nextObjectFor: anObject atInstVar: (self _ivOffsetFor: anObject)
%

category: 'Accessing'
method: PathTerm
_nextObjectFor: anObject atInstVar: ivOffset

"Returns the object at the instance variable that corresponds to the receiver
 path term."

<primitive: 167>
self _primitiveFailed: #_nextObjectFor:atInstVar: 
     args: { anObject . ivOffset } .
self _uncontinuableError
%

category: 'Updating Indexes'
method: PathTerm
addMappingsForObject: anObject logging: aBoolean
  "Add index dictionary and dependency list entries for anObject."

  | ivOffset nextObj |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (ivOffset := self _ivOffsetFor: anObject) == nil
    ifTrue: [ ^ self _invalidIvOffset: anObject ].
  anObject
    getDepListAndAddPathTerm: self
    withIVOffset: ivOffset
    logging: aBoolean.	" add an entry to the objects dependency list "
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.	" get the next object along the path "
  updateBtree ~~ nil
    ifTrue: [ 
      " insert an entry into the B-tree "
      updateBtree btreeAt: nextObj put: anObject.
      needsDepList
        ifTrue: [ 
          " add an entry to the very last object's dependency list "
          nextObj getDepListAndAddLastElementPathTerm: self logging: aBoolean ] ].
  updateDict ~~ nil
    ifTrue: [ 
      " insert an entry into the index dictionary "
      updateDict
        _at: nextObj
        put: anObject
        term: self
        logging: aBoolean ].
  nil == nextObj
    ifTrue: [ ^ self ].
  1 to: children size do: [ :i | 
    " make recursive call to add mappings "
    (children at: i) addMappingsForObject: nextObj logging: aBoolean ]
%

category: 'Reduced Conflict Support'
method: PathTerm
_selectiveAbort

 "Do nothing"
%

category: 'Audit'
method: PathTerm
auditNsc: nsc on: aString level: level

"Verifies that the equality index objects are consistent.
 Returns a string that describes any inconsistencies found." 

| indexObj btreeCounts count |
"aString add: 'entering obj, num is ', num printString; lf."

self size == 0
  ifTrue: [ ^ aString ].

" for each index that utilizes the path term "
btreeCounts := { } .
1 to: self size do: [ :i |
    indexObj := self at: i.
    ( indexObj isRangeEqualityIndex and: [ indexObj size == offset ] )
        ifTrue: [
            " verify B-tree has correct entries "
            count := indexObj btreeRoot auditNsc: nsc for: self offset: i on: aString.
            btreeCounts add: count.
        ]
].
1 to: children size do: [ :i |
      | resultArray |
    resultArray := (children at: i) auditNsc: nsc
                                            on: aString
                                            level: level + 1.
    btreeCounts add: resultArray
].
^btreeCounts
%

category: 'Deprecated'
set compile_env: 0
method: PathTerm
auditNscCounts: nsc for: obj on: aString count: btreeCounts
  "Private. "

  self deprecated: 'PathTerm>>auditNscCounts:for:on:count: deprecated v3.3. 
     Use auditNscCountsFor:on:count:'.  "Bug 43958"
  self auditNscCountsFor: obj on: aString count: btreeCounts
%

category: 'Deprecated'
set compile_env: 0
method: PathTerm
auditDirectNscCounts: nsc for: obj on: aString count: btreeCounts
  "Private. "

  self deprecated: 'PathTerm>>auditDirectNscCounts:for:on:count: deprecated v3.3. 
     Use auditDirectNscCountsFor:on:count:'.  "Bug 43958"
  self auditDirectNscCountsFor: obj on: aString count: btreeCounts
%

category: 'Audit'
method: PathTerm
auditBtree: btreeNode entryFor: obj and: nextObj occurrences: num on: aString builder: nscBuilder

"Private.  Unsupported method for GemStone Technical Support."

| bag |
self size == 0
  ifTrue: [ ^ self ].

btreeNode _findAllValuesForIdenticalKey: nextObj into: nscBuilder.
bag := nscBuilder completeBag.
(bag occurrencesOf: obj) == num 
    ifFalse: [
        self auditInfo: { nextObj . nil . nil } on: aString.
        aString add: 'Btree [';
            add: btreeNode asOop asString;
            add: '] has incorrect number of entries (';
            add: (bag occurrencesOf: obj) asString;
            add: ') for [';
            add: obj asOop asString;
            add: '] of class ';
            add: obj class asString;
            add: ' (should be '; add: num asString; add: ')';
            add: Character lf
    ].
^ bag size
%

category: 'Audit'
method: PathTerm
auditDepListFor: obj index: indexObj on: aString
  "Obsolete...use PathTerm>>auditDepListFor:index:on:optionalSentinel: instead"

  ^ self
    auditDepListFor: obj
    index: indexObj
    on: aString
    optionalSentinel: Object new
%

category: 'ObsoleteIDX - Audit'
set compile_env: 0
method: PathTerm
auditDepListForByteObject: obj occurrences: num index: indexObj on: aString
  "Obsolete ... Private.  Unsupported method for GemStone Technical Support."

  ^ self
    auditDepListForLastElementObject: obj
    occurrences: num
    index: indexObj
    on: aString
%

category: 'Audit'
method: PathTerm
auditDictionaryEntryFor: obj and: nextObj occurrences: num on: aString
  "Private.  Unsupported method for Technical Support."

  | val |
  self size == 0
    ifTrue: [ ^ 0 ].
  val := self getIndexDictionary
    at: nextObj
    term: self
    ifAbsent: [ 
      self isOptionalTerm
        ifFalse: [ 
          self
            auditInfo:
              {nextObj.
              nil.
              nil}
            on: aString.
          aString
            add: 'Index Dictionary has no entry for object [';
            add: obj asOop asString;
            add: '] of class ';
            add: obj class asString;
            add: Character lf ].
      ^ 0 ].
  (BucketValueBag _hasInstance: val)
    ifTrue: [ 
      | occ |
      (occ := val occurrencesOf: obj) == num
        ifFalse: [ 
          self
            auditInfo:
              {nextObj.
              nil.
              nil}
            on: aString.
          aString
            add: 'Index Dictionary has incorrect number of entries (';
            add: occ asString;
            add: ') for [';
            add: obj asOop asString;
            add: '] of class ';
            add: obj class asString;
            add: ' (should be ';
            add: num asString;
            add: ')';
            add: Character lf ].
      ^ val size ]
    ifFalse: [ 
      val == obj
        ifFalse: [ 
          self
            auditInfo:
              {nextObj.
              nil.
              nil}
            on: aString.
          aString
            add: 'Index Dictionary has incorrect entry for [';
            add: obj asOop asString;
            add: '] of class ';
            add: obj class asString;
            add: Character lf ].
      ^ 1 ]
%

category: 'Audit'
method: PathTerm
auditDirectObject: obj occurrences: num on: aString builder: nscBuilder
  "Private.  Unsupported method for GemStone Technical Support."

  "Verifies that the given object has the correct dependency list, index
 dictionary, and B-tree entries for the receiver."

  | indexObj dictChecked |
  self size == 0
    ifTrue: [ ^ self ].
  dictChecked := false.	" for each index that utilizes the path term "
  nscBuilder _resetForAudit.
  1 to: self size do: [ :i | 
    indexObj := self at: i.
    indexObj isRangeEqualityIndex
      ifTrue: [ 
        " verify B-tree has correct entries "
        self
          auditBtree: indexObj btreeRoot
          entryFor: obj
          and: obj
          occurrences: num
          on: aString
          builder: nscBuilder.
        nscBuilder _resetForAudit.
        (DependencyList for: obj) ~~ nil
          ifTrue: [ 
            " verify dependency list "
            self
              auditDepListForLastElementObject: obj
              occurrences: num
              index: indexObj
              on: aString ] ]
      ifFalse: [ 
        " verify index dictionary has correct entries "
        dictChecked
          ifFalse: [ 
            self
              auditDictionaryEntryFor: obj
              and: obj
              occurrences: num
              on: aString.
            dictChecked := true ] ] ]
%

category: 'Audit'
method: PathTerm
auditInfo: array on: aString
  "Private.  Unsupported method for GemStone Technical Support."

  | obj indexObj depList |
  self size == 0
    ifTrue: [ ^ self ].
  obj := array at: 1.
  indexObj := array at: 2.
  depList := array at: 3.
  aString add: 'object'.
  (obj isKindOf: CharacterCollection)
    ifTrue: [ 
      aString
        add: Character space;
        add: obj ].
  aString
    add: ' [';
    add: obj asOop asString;
    add: '] of class ';
    add: obj class asString.
  indexObj ~~ nil
    ifTrue: [ 
      aString
        add: ', ';
        add:
            (indexObj isRangeEqualityIndex
                ifTrue: [ 'equality' ]
                ifFalse: [ 'identity' ]);
        add: ' index <';
        add: indexObj pathComponentsString;
        add: '> [';
        add: indexObj asOop asString;
        add: ']' ].
  aString
    add: ', path term <';
    add: name.
  name == #'*'
    ifTrue: [ aString add: ' offset: ' , offset asString ].
  aString
    add: '> [';
    add: self asOop asString;
    add: ']'.
  depList ~~ nil
    ifTrue: [ 
      aString
        add: ', depList [';
        add: depList asOop asString;
        add: ']' ].
  aString
    add:
      '
    -- '
%

category: 'Deprecated'
set compile_env: 0
method: PathTerm
auditObject: obj occurrences: num on: aString builder: nscBuilder
  "Private.  Unsupported method for GemStone Technical Support."

  "Verifies that the given object has the correct dependency list, index
 dictionary, and B-tree entries for the receiver."

  | indexObj nextObj dictChecked newNum optionalSentinel |
  self size == 0
    ifTrue: [ ^ aString ].
  nscBuilder _resetForAudit.
  self indicatesIndexOnNscElements
    ifTrue: [ 
      ^ self
        auditDirectObject: obj
        occurrences: num
        on: aString
        builder: nscBuilder ].
  nil == obj
    ifTrue: [ ^ aString ].
  optionalSentinel := Object new.
  nextObj := self
    auditDepListFor: obj
    index: indexObj
    on: aString
    optionalSentinel: optionalSentinel.
  nextObj == optionalSentinel
    ifTrue: [ ^ aString ].
  dictChecked := false.
  1 to: self size do: [ :i | 
    " for each index that utilizes the path term "
    indexObj := self at: i.
    (indexObj isRangeEqualityIndex and: [ indexObj size == offset ])
      ifTrue: [ 
        " verify B-tree has correct entries "
        newNum := self
          auditBtree: indexObj btreeRoot
          entryFor: obj
          and: nextObj
          occurrences: num
          on: aString
          builder: nscBuilder.
        nscBuilder _resetForAudit.
        (DependencyList for: nextObj) ~~ nil
          ifTrue: [ 
            " if the path term is the last path term of a range index "
            self
              auditDepListForLastElementObject: nextObj
              occurrences: newNum
              index: indexObj
              on: aString ] ]
      ifFalse: [ 
        " verify index dictionary has correct entries "
        dictChecked
          ifFalse: [ 
            newNum := self
              auditDictionaryEntryFor: obj
              and: nextObj
              occurrences: num
              on: aString.
            dictChecked := true ] ] ].
  obj ~~ nil
    ifTrue: [ 
      1 to: children size do: [ :i | 
        (children at: i)
          auditObject: nextObj
          occurrences: newNum
          on: aString
          builder: nscBuilder.
        nscBuilder _resetForAudit ] ]
%

category: 'Audit'
method: PathTerm
auditIndexDictionaryForDirectObject: obj occurrences: num on: aString

"Private.  Unsupported method for GemStone Technical Support."

"Verifies that the given object has the correct index dictionary entries for 
the receiver."

| indexObj dictChecked |
self size == 0
  ifTrue: [ ^ self ].

dictChecked := false.
"since there's only one index dictionary, shared by all indexes, when you find
the first one audit that and stop"
1 to: self size do: 
    [ :i |
    indexObj := self at: i.
    indexObj isRangeEqualityIndex
        ifFalse: 
            [ " verify index dictionary has correct entries "
            dictChecked
                ifFalse: 
                     [self auditDictionaryEntryFor: obj
                          and: obj
                          occurrences: num
                          on: aString.
                    dictChecked := true]
            ]
    ]
%

category: 'Audit'
method: PathTerm
auditIndexDictionaryForObject: obj occurrences: num on: aString

"Private.  Unsupported method for GemStone Technical Support."

"Verifies that the given object has the correct index dictionary entries for the receiver."

| indexObj nextObj newNum dictChecked |
self size == 0
  ifTrue: [ ^ aString ].

self indicatesIndexOnNscElements
    ifTrue: [ ^ self auditIndexDictionaryForDirectObject: obj occurrences: num on: aString ].

nil == obj
    ifTrue: [ ^ aString ].

nextObj := self _nextObjectFor: obj.

dictChecked := false.
" for each index that utilizes the path term "
1 to: self size do: [ :i |
    indexObj := self at: i.
    ( indexObj isRangeEqualityIndex and: [ indexObj size == offset ] )
       ifFalse: [ " verify index dictionary has correct entries "
            dictChecked
                ifFalse: [
                    newNum := self auditDictionaryEntryFor: obj
                        and: nextObj
                        occurrences: num
                        on: aString.
                    dictChecked := true
                ]
        ]
].

obj ~~ nil
    ifTrue: [
        1 to: children size do: [ :i |
            (children at: i) auditIndexDictionaryForObject: nextObj
                occurrences: newNum
                on: aString
        ]
]
%

category: 'Audit'
method: PathTerm
auditBtreeForObject: obj occurrences: num on: aString builder: nscBuilder

"Private.  Unsupported method for GemStone Technical Support."

"Verifies that the given object has the B-tree entries for the receiver."

| indexObj nextObj newNum |
"aString add: 'entering obj, num is ', num printString; lf."

self size == 0
  ifTrue: [ ^ aString ].

nscBuilder _resetForAudit.

self indicatesIndexOnNscElements
  ifTrue: [ ^ self auditBtreeForDirectObject: obj occurrences: num on: aString builder: nscBuilder ].

nil == obj
    ifTrue: [ ^ aString ].

nextObj := self _nextObjectFor: obj.

" for each index that utilizes the path term "
1 to: self size do: [ :i |
    indexObj := self at: i.
    ( indexObj isRangeEqualityIndex and: [ indexObj size == offset ] )
        ifTrue: [
            " verify B-tree has correct entries "
            newNum := self auditBtree: indexObj btreeRoot
                entryFor: obj
                and: nextObj
                occurrences: num
                on: aString
                builder: nscBuilder.
            nscBuilder _resetForAudit.
        ]
].

obj ~~ nil
    ifTrue: [
        1 to: children size do: [ :i |
            (children at: i) auditBtreeForObject: nextObj
                occurrences: newNum
                on: aString
                builder: nscBuilder.
            nscBuilder _resetForAudit.
        ]
]

%

category: 'Audit'
method: PathTerm
auditBtreeForDirectObject: obj occurrences: num on: aString builder: nscBuilder

"Private.  Unsupported method for GemStone Technical Support."

"Verifies that the given object has the correct B-tree entries for the receiver."

| indexObj dictChecked |

self size == 0
  ifTrue: [ ^ self ].

dictChecked := false.
" for each index that utilizes the path term "
nscBuilder _resetForAudit.
1 to: self size do: [ :i |
    indexObj := self at: i.
    indexObj isRangeEqualityIndex
        ifTrue: [
            " verify B-tree has correct entries "
            self auditBtree: indexObj btreeRoot
                entryFor: obj
                and: obj
                occurrences: num
                on: aString
                builder: nscBuilder.
            nscBuilder _resetForAudit.
        ]
]

%

category: 'Accessing'
method: PathTerm
_thisAndAllChildTermsInto: array

"Returns an Array containing the receiver and all child path terms."


array add: self.

1 to: children size do: [ :i |
  (children at: i) _thisAndAllChildTermsInto: array
].

^ array
%

category: 'Accessing'
method: PathTerm
_thisAndAllChildTerms

"Returns an Array containing the receiver and all child path terms."


^ self _thisAndAllChildTermsInto: { } 
%

category: 'Updating Indexes'
method: PathTerm
cleanupDependencyListFor: anObject
  ""

  | ivOffset nextObj depList |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (depList := DependencyList for: anObject) == nil
    ifTrue: [ ^ self ].
  ivOffset := depList removeCompletelyPathTerm: self for: anObject.	" remove the receiver from the dependency list "
  (self indicatesIndexOnRootNsc
    or: [ 
      "last element dependency list"
      ivOffset < 0 ])
    ifTrue: [ ^ self ].
  ivOffset == 0
    ifTrue: [ nextObj := self _nextObjectFor: anObject ]
    ifFalse: [ 
      [ nextObj := self _nextObjectFor: anObject atInstVar: ivOffset ]
        onException: Error
        do: [ :ex | 
          "Handle any error thrown during execution of _nextObjectFor:
     Cannot resolve the path for this guy --- can't have been added to index,
     in fact this object could have caused the failure in the first place"
          ^ self ] ].
  nil == nextObj
    ifTrue: [ ^ self ].
  needsDepList
    ifTrue: [ 
      (depList := DependencyList for: nextObj) ~~ nil
        ifTrue: [ depList removeCompletelyPathTerm: self for: nextObj ].
      ^ self ].
  1 to: children size do: [ :i | (children at: i) cleanupDependencyListFor: nextObj ]
%

category: 'Testing'
method: PathTerm
_checkBtreeComparisonWith: anObject

"Returns whether the receiver's B-tree can hold an entry with anObject as the
 key."


^ updateBtree _canCompareWith: anObject
%

category: 'Updating Indexes'
method: PathTerm
_clear


"Assigns nil to important instance variables and sets the receiver's size
 to zero."

children := nil.
updateBtree := nil.
updateDict := nil.
self size: 0.
%

! deleted _doNotPerformPretraversalFor: v2.0

! deleted _totalNumSetValuedTerms v2.0

! deleted _pretraverse:ifObject:atOffset:replaceWith:

category: 'Clustering'
method: PathTerm
clusterDepthFirst

"Cluster the receiver."

name cluster.
securityPolicies cluster.
children cluster.
1 to: children size do: [ :i | (children at: i) cluster ]
%

! deleted _classOfNextObject in 3.2

category: 'Testing'
method: PathTerm
_isObsoletePathTerm

"Returns whether the receiver is obsolete."

^ self size == 0
%

category: 'Private'
method: PathTerm
_makeObsolete

"Makes the receiver and all its children obsolete."



1 to: children size do: [ :i | (children at: i) _makeObsolete ].
self _clear.
%

category: 'Testing'
method: PathTerm
isLastTermOfAnIndex

"Return if the receiver is the last term of one of its indexes."



1 to: self size do: [ :i |
  (self at: i) last == self
    ifTrue: [ ^ true ]
].
^ false
%

category: 'Deprecated'
set compile_env: 0
method: PathTerm
allDependencyListsFor: anObject into: all
  "Put any dependency lists that can be found from the receiver
into the given set."

  | ivOffset nextObj depList |
  self deprecated: 'PathTerm>>allDependencyListsFor:into: deprecated v3.3.  
     Use DependencyList class>>depMapValues to collect dependency lists.'.  "bug 43886"

  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (depList := DependencyList for: anObject) == nil
    ifTrue: [ ^ self ].
  all add: depList.
  self indicatesIndexOnRootNsc
    ifTrue: [ ^ self ].
  ivOffset := depList getInstVarOffsetWithPathTerm: self.
  ivOffset < 0
    ifTrue: [ 
      "reference count for last element"
      ^ self ].
  ivOffset == 0
    ifTrue: [ nextObj := self _nextObjectFor: anObject ]
    ifFalse: [ nextObj := self _nextObjectFor: anObject atInstVar: ivOffset ].
  nil == nextObj
    ifTrue: [ ^ self ].
  needsDepList
    ifTrue: [ 
      (depList := DependencyList for: nextObj) ~~ nil
        ifTrue: [ all add: depList ].
      ^ self ].
  1 to: children size do: [ :i | (children at: i) allDependencyListsFor: nextObj into: all ]
%

category: 'Testing'
method: PathTerm
isFast

""

^ true
%

category: 'Accessing'
method: PathTerm
objectSecurityPolicy: anObjectSecurityPolicy

"Assign the receiver and its children to the given security policy."

super objectSecurityPolicy: anObjectSecurityPolicy.
children objectSecurityPolicy: anObjectSecurityPolicy.
1 to: children size do: [ :i |
    (children at: i) objectSecurityPolicy: anObjectSecurityPolicy
].
securityPolicies objectSecurityPolicy: anObjectSecurityPolicy.
%

category: 'Testing'
set compile_env: 0
method: PathTerm
indicatesNsc
  "Returns false, the receiver does not indicate that the index is on an
 instance variable that is expected to be a collection (i.e., set valued pathterm."

  ^ false
%
category: 'Testing'
set compile_env: 0
method: PathTerm
_doNotPerformPretraversalFor: anObject
  "Returns whether to do a preparatory traversal (using MappingInfo objects)
 when anObject is being added to an indexed Bag or IdentityBag."

  ^ self _totalNumSetValuedTerms > 2
%
category: 'Testing'
set compile_env: 0
method: PathTerm
_totalNumSetValuedTerms
  ""

  | total |
  self indicatesNsc
    ifTrue: [ total := 1 ]
    ifFalse: [ total := 0 ].
  1 to: children size do: [ :i | total := total + (children at: i) _totalNumSetValuedTerms ].
  ^ total
%
category: 'Formatting'
set compile_env: 0
method: PathTerm
asString
  ^ super asString , ' on ' , self name asString printString
%
category: 'Private'
set compile_env: 0
method: PathTerm
_invalidIvOffset: anObject
  "If missing path slots are tolerated, return. Otherwise signal an error"

  self requirePathTerms
    ifFalse: [ ^ self ].
  anObject _errorInvalidOffset: name.
  self _uncontinuableError
%
category: 'Private'
set compile_env: 0
method: PathTerm
_mapInfoOrErrorForMissingIvOffset: anObject
  "If missing path slots are tolerated, return an empty MappingInfo. Otherwise 
   return an error array"

  | exa |
  (exa := LookupError new)
    _number:(ErrorSymbols at: #'rtErrObjectInvalidOffset');
    object: anObject;
    arg: name printString.
  ^ {false.	" indicating no indexing objects were modified "
  exa}
%
category: 'Audit'
set compile_env: 0
method: PathTerm
auditDepListFor: obj index: indexObj on: aString optionalSentinel: optionalSentinel
  "Private."

  | depList j nextObj ivOffset |
  (nil == obj or: [ self size == 0 ])
    ifTrue: [ ^ nil ].
  depList := DependencyList for: obj.
  depList == nil
    ifTrue: [ 
      (obj isInvariant not and: [ self isOptionalTerm not ])
        ifTrue: [ 
          self
            auditInfo:
              {obj.
              indexObj.
              depList}
            on: aString.
          aString
            add: ' Object has no dependency list';
            add: Character lf ].
      nextObj := self _nextObjectFor: obj ]
    ifFalse: [ 
      (DepListTable _hasDependencyBitFor: obj)
        ifFalse: [ 
          self
            auditInfo:
              {obj.
              indexObj.
              depList}
            on: aString.
          aString
            add: ' Object has DependencyList but no dependency bit is set';
            add: Character lf ].
      j := depList _findOffsetForPathTerm: self.
      j == nil
        ifTrue: [ 
          | ivOff |
          "no entry for self in depList"
          ivOff := self _ivOffsetFor: obj.
          (self isOptionalTerm and: [ ivOff isNil ])
            ifTrue: [ 
              "no dependencyList if there's no slot"
              nextObj := optionalSentinel ]
            ifFalse: [ 
              self
                auditInfo:
                  {obj.
                  indexObj.
                  depList}
                on: aString.
              aString
                add: 'DependencyList has no entry for path term';
                add: Character lf.
              nextObj := self _nextObjectFor: obj ] ]
        ifFalse: [ 
          "found entry for self in depList"
          ivOffset := depList at: j + 1.	"may be negative for last element reference count"
          ivOffset < 0
            ifTrue: [ 
              "reference count for last element, but we don't expect reference 
               count here ... see needsDepList clause below where depList 
               existence is sufficient"
              self
                auditInfo:
                  {obj.
                  indexObj.
                  depList}
                on: aString.
              aString
                add: 'DependencyList on last element in unexepected location ';
                add: ivOffset asString;
                add: Character lf.
              ^ obj ]
            ifFalse: [ 
              ivOffset ~~ (self _ivOffsetFor: obj)
                ifTrue: [ 
                  self
                    auditInfo:
                      {obj.
                      indexObj.
                      depList}
                    on: aString.
                  aString
                    add: 'DependencyList has incorrect iv offset ';
                    add: ivOffset asString.
                  ivOffset := self _ivOffsetFor: obj.
                  aString
                    add: ' instead of ';
                    add: ivOffset asString;
                    add: Character lf ].
              nextObj := obj instVarAt: ivOffset ] ] ].
  ((needsDepList and: [ nextObj ~~ nil ])
    and: [ DependencyList needsDepList: nextObj ])
    ifTrue: [ 
      depList := DependencyList for: nextObj.
      (depList == nil and: [ self isOptionalTerm not ])
        ifTrue: [ 
          self
            auditInfo:
              {nextObj.
              indexObj.
              depList}
            on: aString.
          aString
            add: ' Object has no dependency list';
            add: Character lf ] ].
  ^ nextObj
%
category: 'Testing'
set compile_env: 0
method: PathTerm
isOptionalTerm
  ^ false
%
category: 'Audit'
set compile_env: 0
method: PathTerm
auditDepListForLastElementObject: obj occurrences: num index: indexObj on: aString
  "Private.  Unsupported method for GemStone Technical Support."

  | depList j |
  (self size == 0 or: [ obj isInvariant ])
    ifTrue: [ ^ self ].
  depList := DependencyList for: obj.
  depList == nil
    ifTrue: [ 
      self
        auditInfo:
          {obj.
          indexObj.
          depList}
        on: aString.
      aString
        add: ' Object has no dependency list';
        add: Character lf ]
    ifFalse: [ 
      (DepListTable _hasDependencyBitFor: obj)
        ifFalse: [ 
          self
            auditInfo:
              {obj.
              indexObj.
              depList}
            on: aString.
          aString
            add: ' Object has DependencyList but no dependency bit is set';
            add: Character lf ].
      j := depList _findOffsetForPathTerm: self.
      j == nil
        ifTrue: [ 
          "hack to avoid false positives for index paths ending in *"
          self name = #'*'
            ifFalse: [ 
              self
                auditInfo:
                  {obj.
                  indexObj.
                  depList}
                on: aString.
              aString
                add: 'DependencyList has no entry for path term';
                add: Character lf ] ]
        ifFalse: [ 
          | referenceCount nonReferenceCountTerm |
          referenceCount := depList at: j + 1.
          nonReferenceCountTerm := false.
          referenceCount > 0
            ifTrue: [ 
              "upgraded indexes from pre-3.2 may have positive rerence count, but only for Byte indexable ojbects"
              obj class isBytes
                ifFalse: [ 
                  "ivOffset and not a reference count"
                  nonReferenceCountTerm := true ] ]
            ifFalse: [ referenceCount := referenceCount negated ].
          nonReferenceCountTerm
            ifTrue: [ 
              "not a reference count and will be audited by PathTerm>>auditDepListFor:index:on:optionalSentinel: "
              "noop"
               ]
            ifFalse: [ 
              referenceCount == num
                ifFalse: [ 
                  self
                    auditInfo:
                      {obj.
                      indexObj.
                      depList}
                    on: aString.
                  aString
                    add: ' DependencyList has incorrect ref count ';
                    add: referenceCount asString;
                    add: ' for occurrences ';
                    add: num asString;
                    add: Character lf ] ] ] ]
%
category: 'Testing'
set compile_env: 0
method: PathTerm
isCollectionBasedTerm
  "Returns true if the receiver is an enumerated path term or a term that indicates a 
   set-valued instance variable."

  ^ false
%
category: 'Testing'
set compile_env: 0
method: PathTerm
isEnumeratedTerm
  "Returns true if the receiver indicates an enumerated path term."

  ^ false
%
category: 'Testing'
set compile_env: 0
method: PathTerm
isSetValuedTerm
  "Returns true if the receiver indicates a set-valued instance variable."

  ^ false
%
category: 'Traversing'
method: PathTerm
_getNextObjectForTraversal: anObject
  " get the next object along the path "

  | ivOffset nextObj |
  ivOffset := anObject _findIVOffsetForPathTerm: self.
  ivOffset == nil
    ifTrue: [ 
      anObject _errorPathTermNotInDependencyList: self.
      nextObj := nil ]
    ifFalse: [ nextObj := anObject instVarAt: ivOffset ].
  ^ nextObj
%
category: 'Testing'
method: PathTerm
indicatesMultiValue
  "Returns false, the receiver does not indicate that the index is on an
 instance variable that may have multiple values (i.e., collection based path term)"

  ^ false
%
category: 'Testing'
method: PathTerm
hasMatchingTerm: aSymbol
  "Answer true if the receiver's term matches <aSymbol>"

  ^ self name == aSymbol
%
category: 'Accessing'
method: PathTerm
requirePathTerms
  "If missing path slots are tolerated, answer false"

  ^ true
%
category: 'Testing'
method: PathTerm
isSelectorTerm
  ^ false
%
category: 'Updating'
method: PathTerm
_addChildTerm: aPathTerm
  children addLast: aPathTerm
%
category: 'Testing'
method: PathTerm
termsRequired
  "Answer true if receiver requires that instance variables of indexed objects are present "

  ^ true
%
category: 'Audit'
method: PathTerm
auditDirectNscCountsFor: obj on: aString count: btreeCounts
  "Private. "

  | index |
  self size == 0
    ifTrue: [ ^ nil ].
  nil == obj
    ifFalse: [ 
      | depList |
      depList := DependencyList for: obj.
      (depList == nil and: [ obj isInvariant not ])
        ifTrue: [ 
          self
            auditInfo:
              {obj.
              nil.
              depList}
            on: aString.
          aString
            add: ' Object has no dependency list';
            add: Character lf ]
        ifFalse: [ 
          obj isInvariant not
            ifTrue: [ 
              (DepListTable _hasDependencyBitFor: obj)
                ifFalse: [ 
                  self
                    auditInfo:
                      {obj.
                      nil.
                      depList}
                    on: aString.
                  aString
                    add:
                        ' Object has DependencyList but no dependency bit is set';
                    add: Character lf ] ] ] ].
  index := 1.
  1 to: self size do: [ :i | 
    | count indexObj |
    indexObj := self at: i.
    (indexObj isRangeEqualityIndex and: [ indexObj size == offset ])
      ifTrue: [ 
        count := btreeCounts at: index.
        count == 0
          ifTrue: [ 
            aString
              add: Character lf;
              add: ' -- The number of entries in Btree [';
              add: (self at: i) btreeRoot asOop asString;
              add: '] for pathTerm [';
              add: self asOop asString;
              add: ' offset ';
              add: offset asString;
              add:
                  '] does not match the number of entries in the base collection (extra elements in either the base collection or btree).';
              add: Character lf ].
        btreeCounts at: index put: count - 1.
        index := index + 1 ] ]
%
category: 'Audit'
method: PathTerm
auditNscCountsFor: obj on: aString count: btreeCounts
  "Private. "

  | nextObj index optionalSentinel |
  self size == 0
    ifTrue: [ ^ self ].
  self indicatesIndexOnNscElements
    ifTrue: [ ^ self auditDirectNscCountsFor: obj on: aString count: btreeCounts ].
  nil == obj
    ifTrue: [ ^ self ].
  optionalSentinel := Object new.
  nextObj := self
    auditDepListFor: obj
    index: nil
    on: aString
    optionalSentinel: optionalSentinel.
  nextObj == optionalSentinel
    ifTrue: [ ^ self ].
  index := 1.
  1 to: self size do: [ :i | 
    | count indexObj |
    indexObj := self at: i.
    (indexObj isRangeEqualityIndex and: [ indexObj size == offset ])
      ifTrue: [ 
        count := btreeCounts at: index.
        (count == 0 and: [ self isOptionalTerm not ])
          ifTrue: [ 
            "an index with optional path terms may have 0 entries"
            aString
              add: Character lf;
              add: ' -- The number of entries in Btree [';
              add: (self at: i) btreeRoot asOop asString;
              add: '] for pathTerm [';
              add: self asOop asString;
              add: ' offset ';
              add: offset asString;
              add:
                  '] does not match the number of entries in the base collection (extra elements in either the base collection or btree).';
              add: Character lf ].
        (count == 0 and: [ self isOptionalTerm ])
          ifFalse: [ btreeCounts at: index put: count - 1 ].
        index := index + 1 ] ].
  1 to: children size do: [ :i | 
    (children at: i)
      auditNscCountsFor: nextObj
      on: aString
      count: (btreeCounts at: index).
    index := index + 1 ]
%
category: 'Private'
method: PathTerm
_mappingInfoClass
  ^ MappingInfo
%
category: 'Updating Indexes'
method: PathTerm
isLastOccurrenceInIndexObjects: anObject for: anNsc
  | pathTerm key tmpList val num val2 parentTerm |
  pathTerm := self.
  pathTerm isRangeEqualityIndexLastPathTerm
    ifTrue: [ pathTerm offset == 1
        ifTrue: [ " if index directly on NSC elements or anObject is nil "
          (pathTerm indicatesIndexOnNscElements or: [ nil == anObject ])
            ifTrue: [ key := anObject ]
            ifFalse: [ key := pathTerm _nextObjectFor: anObject ].
          tmpList := IdentityBag new.
          pathTerm updateBtree btreeRoot
            _findAllValuesForIdenticalKey: key
            into: tmpList.
          ^ (tmpList occurrencesOf: anObject) <= 1 ]
        ifFalse: [ pathTerm := pathTerm getParentTerm.
          val := pathTerm updateDict at: anObject term: pathTerm otherwise: nil.
          (BucketValueBag _hasInstance: val)
            ifTrue: [ " see if more than one mapping "
              pathTerm indicatesNsc
                ifTrue: [ num := val occurrencesOf: anNsc.
                  num < val size
                    ifTrue: [ " if anObject is contained in other NSCs "
                      ^ false ].
                  pathTerm := pathTerm getParentTerm.	" get path term before this one "
                  val2 := pathTerm updateDict
                    at: anNsc
                    term: pathTerm
                    otherwise: nil.
                  (BucketValueBag _hasInstance: val2)
                    ifTrue: [ ^ val2 size == num ]
                    ifFalse: [ ^ true ] ]
                ifFalse: [ ^ (val occurrencesOf: anNsc) <= 1 ] ]
            ifFalse: [ ^ true ] ] ]
    ifFalse: [ 
      " get key to look up in index dictionary "
      (pathTerm indicatesIndexOnNscElements or: [ nil == anObject ])
        ifTrue: [ key := anObject ]
        ifFalse: [ " see if a path with '*.*' in it "
          pathTerm indicatesNsc
            ifTrue: [ val := pathTerm updateDict
                at: anObject
                term: pathTerm getParentTerm
                otherwise: nil.
              ^ (BucketValueBag _hasInstance: val) not ]
            ifFalse: [ key := pathTerm _nextObjectFor: anObject ] ].
      val := pathTerm updateDict at: key term: pathTerm otherwise: nil.	" look up the mapping in the index dictionary "
      (BucketValueBag _hasInstance: val)
        ifTrue: [ 
          " see if more than one mapping "
          (num := val occurrencesOf: anObject) <= 1
            ifTrue: [ ^ true ]
            ifFalse: [ 
              " see if multiple occurrences are due to more than one object
                    referencing anNsc "
              " if there is a parent term, it is a SetValuedPathTerm "
              parentTerm := pathTerm getParentTerm.
              parentTerm == nil
                ifTrue: [ ^ false ].
              val2 := pathTerm updateDict
                at: anObject
                term: parentTerm
                otherwise: nil.
              (BucketValueBag _hasInstance: val2)
                ifTrue: [ ^ (val2 occurrencesOf: anNsc) == num ]
                ifFalse: [ ^ false ] ] ]
        ifFalse: [ ^ true ] ].
  self error: 'should not get here'
%
category: 'Updating Indexes'
method: PathTerm
addMappingsForObject: anObject traverseUpTo: aPathTerm for: anIndexObject logging: aBoolean
  | object |
  object := anIndexObject traverse: anObject upTo: aPathTerm.
  self addMappingsForObject: object logging: aBoolean
%
category: 'Testing'
method: PathTerm
coversIvOffset: anIvOffset for: anObject
  "Answer true if the receiver covers the given iv offset (see bug 46705)" 

  ^false
%
category: 'Testing'
method: PathTerm
hasIndexDictionary
  "Answer if receiver's index may have an indexDictionary."

  ^ true
%
category: 'Become Support'
method: PathTerm
_getPathTermIndexReferencesInto: refsToRcvr for: anObject
  "Place information about references to the receiver due to the receiver's 
 participation in an index into the given Array.  The Array consists of pairs:

 1) An object that references the receiver.  If the object is directly
    contained in an indexed NSC, then this object is the NSC.  If the object
    in the dependency list is a tracking object (for object modification
    tracking), then this object is the tracking object.
 2) The offset of the instance variable in that object that references the
    receiver.  
 3) If the object is the last path term, this number is < 0 and is a reference 
    count for the number of times .  
 3) If the object in the dependency list is a tracking
    object, this number is 0.

 This is used only in support of the 'become:' method."
 
 | vals indexObj obj |
 vals := nil.   
 self size = 0
   ifTrue: [ ^ self ].
 " check if it is the first path term in an index (which means anObject
  is directly contained in an indexed NSC "
 indexObj := self at: 1.
 self offset == 1
   ifTrue: [ 
     anObject
       _addToReferences: indexObj nscRoot
       offset: 0
       occurrences: (indexObj nscRoot occurrencesOf: anObject)
       into: refsToRcvr ]
   ifFalse: [
     | prevPathTerm | 
     "determine if mapping is contained in index dictionary or B-tree"
     self isRangeEqualityIndexLastPathTerm
       ifTrue: [ 
         " is either the next to the last or last object along the path "
         " if the last object does not need a dependency list, then
         anObject must be the next to the last object along the path
         (since only it could have a dependency list) "
         self needsDepList
           ifTrue: [ 
             " no choice but to check if its in the dictionary "
             vals := indexObj indexDictionary
               at: anObject
               term: (indexObj at: self offset - 1)
               otherwise: nil.
             vals
               ifNil: [ self error: 'Reference to anObject should have been found in indexDictionary.' ] ] ].	
     " get the previous path term in the index "
     prevPathTerm := indexObj at: self offset - 1.
     vals
       ifNil: [ 
         " get the objects that refer to anObject "
         vals := indexObj indexDictionary
           at: anObject
           term: prevPathTerm
           ifAbsent: [ indexObj indexDictionary _errorNoEntryForKey: anObject term: prevPathTerm value: nil ] ].
     (BucketValueBag _hasInstance: vals)
       ifTrue: [ 
         1 to: vals size do: [ :j | 
           obj := vals _at: j.
           prevPathTerm indicatesMultiValue
             ifTrue: [ 
               anObject
                 _addToReferences: obj
                 offset: 0
                 occurrences: (obj occurrencesOf: anObject)
                 into: refsToRcvr ]
             ifFalse: [ 
               anObject
                 _addToReferences: obj
                 offset: (prevPathTerm _ivOffsetFor: obj)
                 occurrences: 1
                 into: refsToRcvr ] ] ]
       ifFalse: [ 
         prevPathTerm indicatesMultiValue
           ifTrue: [ 
             anObject
               _addToReferences: vals
               offset: 0
               occurrences: (vals occurrencesOf: anObject)
               into: refsToRcvr ]
           ifFalse: [ 
             anObject
               _addToReferences: vals
               offset: (prevPathTerm _ivOffsetFor: vals)
               occurrences: 1
               into: refsToRcvr ] ] ] 
%
category: 'Index Maintenance'
method: PathTerm
_doRemoveIndex: indexObj nsc: nsc
      "Remove the index associated with the receiver from the nsc."
 
      | bag incomplete indexManager allTerms |
      indexManager := IndexManager current.
      bag := nsc _asIdentityBag.
      incomplete := #'_incompletePathTraversal'.
      indexObj _findSetValuedPathTermOffset ~~ 0
        ifTrue: [ 
          | updateIndexList objNsc |
          "See if there is a set-valued instance variable before the unshared path
     term."
          updateIndexList := false.
          (self == indexObj lastPathTerm
            _and: [ self size > 1 _and: [ (self at: 1) size ~= (self at: 2) size ] ])
            ifTrue: [ 
              " see if index is subsumed by an existing index "
              " if last path term is #*, will need to update the
              last objects' index list "
              self indicatesMultiValue
                ifTrue: [ 
                  updateIndexList := true.
                  objNsc := NscBuilder
                    for: IdentityBag new
                    max: indexObj sizeForNscBuilder ]
                ifFalse: [ objNsc := nil ] ]
            ifFalse: [ objNsc := NscBuilder for: IdentityBag new max: indexObj sizeForNscBuilder ].
          1 to: bag size do: [ :i | 
            " for each element in the NSC ... "
            " traverse up to the point of the unshared path term "
            indexObj
              traverse: (bag _at: i)
              upTo: self
              startingAt: 1
              into: objNsc
              addingIndex: false ].
          objNsc ~~ nil
            ifTrue: [ | endObj |
              updateIndexList
                ifTrue: [ self removeIndexListEntriesFor: objNsc completeBag index: indexObj ]
                ifFalse: [ 
                  objNsc := objNsc completeBag.
                  1 to: objNsc size do: [ :j | 
                    endObj := objNsc _at: j.
                    endObj ~~ incomplete
                      ifTrue: [ 
                        " only remove mappings once for a given object "
                        self
                          removeMappingsFor: endObj
                          lastOne: true
                          logging: false.
                        DependencyList removePathTerm: self for: endObj ] ] ] ] ]
        ifFalse: [ | obj |
          self indicatesIndexOnRootNsc
            ifTrue: [ 
              " see if the index is on elements of the NSC "
              1 to: bag size do: [ :j | 
                obj := bag _at: j.
                self removeDirectMappingFor: obj logging: false.
                DependencyList removePathTerm: self for: obj.
                indexManager commitIndexMaintenance: indexObj at: j ] ]
            ifFalse: [ 
              [ | endObj | 
              1 to: bag size do: [ :j | 
                endObj := indexObj traverse: (bag _at: j) upTo: self.
                (nil ~~ endObj and: [ incomplete ~~ endObj ])
                  ifTrue: [ 
                    self removeMappingsSkipBtreeFor: endObj lastOne: true logging: false.
                    DependencyList removePathTerm: self for: endObj.
                    indexManager commitIndexMaintenance: indexObj at: j ] ] ]
                onException: Error
                do: [ :ex | 
                  ex number == (ErrorSymbols at: #'rtErrObjectPathTermNotInDependencyList')
                    ifTrue: [ 
                      "continue execution"
                      ex resume ]
                    ifFalse: [ ex outer ] ] ].
          allTerms := self _thisAndAllChildTerms ].
      ^ allTerms
%
category: 'Index Maintenance'
method: PathTerm
_doRemoveSubsumedIndex: indexObj nsc: nsc
  "The given index is subsumed by an existing index and is being removed."

  | bag objNsc |
  " see if there is a set-valued term "
  indexObj _findSetValuedPathTermOffset == 0
    ifFalse: [ 
      " need to update index lists "
      bag := nsc _asIdentityBag.
      self indicatesMultiValue
        ifTrue: [ 
          " if last path term is *, need to update the last objects index list "
          objNsc := NscBuilder
            for: IdentityBag new
            max: indexObj sizeForNscBuilder ]
        ifFalse: [ objNsc := nil ].
      1 to: bag size do: [ :i | 
        " for each element in the NSC ... "
        indexObj
          traverse: (bag _at: i)
          upTo: self
          startingAt: 1
          into: objNsc
          addingIndex: false ].
      objNsc ~~ nil
        ifTrue: [ self removeIndexListEntriesFor: objNsc completeBag index: indexObj ] ]
%
category: 'Index Maintenance'
method: PathTerm
_doAddMappingsFor: anNSC on: indexObj indexManager: indexManager hasSet: hasSet
  "Adds the mappings in the index dictionary, dependency lists, and so on for all
 elements in the receiver."
 
  | objNsc bag |
  bag := anNSC _asIdentityBag.
  hasSet
    ifTrue: [ 
      | updateIndexList |
      " if there is a set-valued instance variable, collect objects in a
      Bag on which to update the index dictionary "
      updateIndexList := false.
      (self == indexObj lastPathTerm
        _and: [ self size > 1 _and: [ (self at: 1) size ~= (self at: 2) size ] ])
        ifTrue: [ 
          " see if index is subsumed by an existing index "
          " if last path term is #*, will need to update the
          last objects' index list "
          self indicatesMultiValue
            ifTrue: [ 
              updateIndexList := true.
              objNsc := NscBuilder
                for: IdentityBag new
                max: indexObj sizeForNscBuilder ]
            ifFalse: [ objNsc := nil ] ]
        ifFalse: [ objNsc := NscBuilder for: IdentityBag new max: indexObj sizeForNscBuilder ].
      1 to: bag size do: [ :i | 
        " traverse up to the point of the unshared path term "
        indexObj
          traverse: (bag _at: i)
          upTo: self
          startingAt: 1
          into: objNsc
          addingIndex: true ].
      objNsc ~~ nil
        ifTrue: [ 
          updateIndexList
            ifTrue: [ self addIndexListEntriesFor: objNsc completeBag index: indexObj ]
            ifFalse: [ 
              objNsc := objNsc completeBag.
              1 to: objNsc size do: [ :j | 
                self addMappingsForObject: (objNsc _at: j) logging: false.
                j \\ 100 == 0
                  ifTrue: [ indexManager commitIndexMaintenance: indexObj at: j ] ] ] ] ]
    ifFalse: [ 
      | object |
      "no set-valued terms"
      indexManager autoCommit
        ifTrue: [ anNSC _lockForIndexCreation ].
      indexObj firstPathTerm == self
        ifTrue: [ 
          " see if need to traverse to the first unshared path term "
          1 to: bag size do: [ :i | 
            object := bag _at: i.
            self addMappingsForObject: object logging: false.
            i \\ 100 == 0
              ifTrue: [ indexManager commitIndexMaintenance: indexObj at: i ] ] ]
        ifFalse: [ 
          1 to: bag size do: [ :i | 
            " traverse up to the point of the unshared path term "
            object := bag _at: i.
            self addMappingsForObject: object traverseUpTo: self for: indexObj logging: false.
            i \\ 100 == 0
              ifTrue: [ indexManager commitIndexMaintenance: indexObj at: i ] ] ] ].
  ^ anNSC
%
category: 'Index Maintenance'
method: PathTerm
_doAddSubsumedIndex: indexObj nsc: anNSC
  "Adds the mappings in the index dictionary, dependency lists, and so on for all
 elements in anNSC for a subsumed index on indexObj."
 
  | objNsc bag |
  " index is subsumed by an existing index "
  indexObj _findSetValuedPathTermOffset == 0
    ifTrue: [ ^ anNSC ].
  bag := anNSC _asIdentityBag.
  " need to update index lists "
  self indicatesMultiValue
    ifTrue: [ 
      " if last path term is #*, will need to update the
      last objects' index list "
      objNsc := NscBuilder
        for: IdentityBag new
        max: indexObj sizeForNscBuilder ]
    ifFalse: [ objNsc := nil ].
  1 to: bag size do: [ :i | 
    indexObj
      traverse: (bag _at: i)
      upTo: self
      startingAt: 1
      into: objNsc
      addingIndex: true ].
  objNsc ~~ nil
    ifTrue: [ self addIndexListEntriesFor: objNsc completeBag index: indexObj ].
  ^ anNSC 
%
