!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   ProcessorScheduler, Object.
!
!=========================================================================

!  ProcessorScheduler created in bom.c or modified in preconversion

set class ProcessorScheduler
removeallmethods   
removeallclassmethods

category: 'Documentation'
classmethod:
comment
^ '
 ProcessorScheduler implements a green-threads scheduler for instances of
 GsProcess.  The processes all share one native thread in the virtual
 machine.  There is exactly one instance of ProcessorScheduler in a session;
 this instance is initialized by the virtual machine and may not be 
 committed to disk . 

 InstVars:
   activeProcess - currently active process.
   readyQueue - a SortedCollection of processes ready to run.

   delayQueue - a SortedCollection of objects with signalTime not yet expired,
		  elements are instances of either Delay or GsProcess .
   lastGroup  -  a SmallInteger    
   suspendedSet - IdentitySet of suspended processes
   waitingSet  -  IdentitySet of processes waiting for objects to be ready

   socketsPolled - an Array maintained/used by primitives,
		   the list of GsSockets upon which processes are waiting . 
   pollResults - an Array used by the polling primitive.
   timeoutThreadActive - a Boolean
   criticalMethods  - used by in-vm debugging support

 A process can be in one of these states:
    active  		- the activeProcess
    ready to run 	  - in the readyQueue 
    waiting on a GsSocket - in the waitingSet , and a socket in socketsPolled
    waiting on a Semaphore - in the waitingSet
    waiting on a Delay    - in delayQueue or referenced from a Delay in delayQueue
    waiting on Ruby IoSelect -  referenced from a socket in the socketsPolled
    suspended		- in the suspendedSet

  Process state               onQueue             waitingOn
    0 active                    nil		     nil
    1 ready to run              readyQueue           nil
    2 waiting on a GsSocket     waitingSet 	    a GsSocket
    3 waiting on a Semaphore    waitingSet 	    a Semaphore
    4 waiting on a Delay    -   waitingSet 	    Delay or nil
    5 ref from a waiting Delay  ??waitingSet 	    a Delay
    6 suspended         -       suspendedSet         any

 If a process waiting on a socket is suspended 
 it will be woken up by activity on the socket.

 If a process waiting on a Semaphore is suspended, 
 it will be woken up by signaling the semaphore .

 If a process is waiting on a Delay, 
 it will be woken up when Delay expires.

 GsProcess>>suspend has no effect unless receiver is running or in the readyQueue .

 GsProcess>>terminate removes receiver from any thing it is waiting on.

 A Delay has one target, a GsProcess or a Semaphore .

Constraints:
	readyQueue: SortedCollection
	activeProcess: GsProcess
	pollResults: Array
	delayQueue: SortedCollection
	lastGroup: SmallInteger
	suspendedSet: Object
	waitingSet: Object
	socketsPolled: Array
	timeoutThreadActive: Boolean
	criticalMethods: IdentitySet
'
%


category: 'Instance Creation'
classmethod:
new
  self shouldNotImplement: #new
%
category: 'Private'
classmethod:
_isInitialized
  ^((System __sessionStateAt: 15) ~~ nil)
%
category: 'Accessing'
classmethod:
scheduler
  "This is equivalent to 'Processor' in the Blue Book.  It returns
   the instance of the receiver that is responsible for scheduling
   ready processes"

<primitive: 457>

^self _halt: 'Attempt to use process scheduler before it has been initialized' 
%

category: 'Logging'
method:
dbglog: aString
  "log a debugging string"
  GsFile gciLogServer: aString 
%
category: 'Logging'
method:
dbgfatallog: aString
  "log a fatal error string"
  GsFile gciLogServer: aString 
%
category: 'Priorities'
method:
highestPriority
  "return the highest allowable priority"
  ^ 40   "also in C code as PROCESSSCHEDULER_highestPriority"
         "also other in-line highestPriority logic"
%
category: 'Priorities'
method:
timingPriority
  "Answer the priority at which the system processes
   keeping track of real time should run. [Blue Book]"
  ^35
%
category: 'Priorities'
method:
highIOPriority
  "Answer the priority at which the most time
   critical input/output processes should run. [Blue Book]"
  ^30
%
category: 'Priorities'
method:
lowIOPriority
  "Answer the priority at which most input/output
   processes should run. [Blue Book]"
  ^25
%
category: 'Priorities'
method:
userInterruptPriority
  "Answer the priority at which processes created
   by the user and desiring immediate service
   should run. [Blue Book]"
  ^20
%
category: 'Priorities'
method:
userSchedulingPriority
  "Answer the priority at which the user inter-
   face processes should run. [Blue Book]"
  ^15  "this value inlined in some methods here and in GsProcess, and 
	in C as PROCESSSCHEDULER_userPriority"
%
category: 'Priorities'
method:
userBackgroundPriority
  "Answer the priority at which background
   processes created by the user should run. [Blue Book]"
  ^10
%
category: 'Priorities'
method:
systemBackgroundPriority
  "Answer the priority at which system background
   processes should run. [Blue Book]"
  ^5
%
category: 'Priorities'
method:
lowestPriority
  "return the lowest allowable priority"
  ^1   "also in C code as PROCESSSCHEDULER_lowestPriority"
       "also other inline lowestPriority logic"
%
category: 'Delay Access'
method:
_delaySchedule: aProcessOrDelay
  "Add aProcessOrDelay to the delayQueue ."

  self _addDelay: aProcessOrDelay to: delayQueue .
%

category: 'Delay Access'
method:
_now
  "Returns the time that should be used by a Delay to represent to
   current time in milliseconds."

<primitive: 651>
^ self _primitiveFailed: #_now
%

category: 'Delay Access'
method:
_timeoutNow: delayQsize

  "Returns current time in milliseconds consistent with most recently
   expired timeout.  If delayQsize == 0 , result may always be zero "
<primitive: 735>
^ self _primitiveFailed: #_timeoutNow:
%


category: 'Delay Access'
method:
_delayUnschedule: aDelay
  "Unschedule aDelay in the queue."

  delayQueue removeIdentical: aDelay otherwise: nil 
%
category: 'Initialization'
method:
_initialize: initSessionArg
  "This method is invoked once by the VM during session initialization.
   The single instances of ProcessorScheduler and GsCurrentSession
   are created by the VM before invoking this method.  

   If initSessionArg~~nil , invokes GsCurrentSession(C) initialize

   In the following discussion, the single instance of GsCurrentSession
   is 'currSess' . 

   Before invoking this method, the VM has completed these steps :
     1) default timeZone is fetched from (Globals at:#TimeZone) and
     installed in session state.   This wil be either the
     value of the class instance variable 'default' in TimeZone ,
     or the class variable #Default in the ObsoleteTimeZone class
     (if the 2007 upgrade to TimeZone has not been installed). 

   Initializes the receiver's process queues.  

   Invokes GsCurrentSession >> initialize  to execute any application
   defined initialization."

  readyQueue := SortedCollection sortBlock: [:p1:p2 | p1 priority >= p2 priority].
		"sort block must agree with prim 733"
  activeProcess := nil.
  delayQueue := SortedCollection sortBlock: [:d1:d2 | d1 _signalTime <= d2 _signalTime].
		"sort block must agree with prim 734"
  suspendedSet := IdentitySet new .
  waitingSet := IdentitySet new .
  lastGroup := 0.
  pollResults := Array new: 5 . pollResults size: 0 .
  socketsPolled := Array new: 5 . socketsPolled size: 0 .
  timeoutThreadActive := false .
  self _setInstVarsAlwaysValid .      "fix 44837"

  System __sessionStateAt: 15 put: self.
  (Globals at: #CharacterDataTables otherwise: nil) ifNotNil:[
    [ 
      (System gemEnvironmentVariable:'GS_DISABLE_CHARACTER_TABLE_LOAD') ifNil:[
        Character _loadCharTables.
      ]
    ] onException: AbstractException do:[:ex | "ignore" ].
  ].
  initSessionArg ifNotNil:[
    GsCurrentSession initialize
  ].
  "No user or application changes here. 
   Customization is to be done in GsCurrentSession >> initialize"
%

category 'Private'
method:
_setInstVarsAlwaysValid

"Returns receiver.  Prevent values of instVars from being changed by abort
if they do get committed."

<primitive: 995>
self _primitiveFailed: #_setInstVarsAlwaysValid
%


! deleted all uses of errorEventQueue

category 'Private'
method:
_newGroup
  lastGroup := lastGroup + 1.
  ^lastGroup
%

category 'Private'
method:
_switchFrom: oldProcess to: newProcess
  "Suspend the currentProcess and activate the receiver"

<primitive: 2010>
"prim clears OM.schedulerInReapEvents .
 No primitive failure code. Execution resumes upon resumption
 of oldProcess "
^ self  
%

! edits in _runNextProcess and other methods for bug 43245,
! replaced reads of activeProcess  with  'self activeProcess' ,
! deleted some assignments to activeProcess in favor of letting _switchFrom:to:
! primitive do all the assignments


category: 'Private'
method:
_runNextProcess
  "Used when a blocked thread exits to kick the scheduler so it will run
   another process."
  <primitive: 736> "prim always fails, sets OM.schedulerInReapEvents flag"

  | newProc ex oldProc |
  [ newProc := self _findReadyProcess: false .
    (newProc == nil or:[
      newProc == (oldProc := self activeProcess)]) ifTrue:[
      (ex := ThreadError new) details:
          'ProcessScheduler _runNextProcess failed, new ' , newProc asOop asString,
			' old ' , oldProc asOop asString .
      self _endYield .
      ex signalNotTrappable
    ].
    newProc _isTerminated 
  ] whileTrue .

  self _switchFrom: nil to: newProc .  "primitive updates activeProcess instVar"
  "Should never return"

  self dbgfatallog: 'leaving _runNextProcess!!!!'.
  self _uncontinuableError
%

category: 'Private'
method: 
_reschedule: honorOobInterrupts
  "Find a process that is ready to run. Once found suspend originalOldProcess
   and activate the new process."
  | newProc oldProc |
  oldProc := self activeProcess .
  honorOobInterrupts ifTrue:[
    [ 
      (newProc := self _findReadyProcess: honorOobInterrupts) ifNotNil:[ 
        self _switchFrom: oldProc to: newProc . "exits critical"
        "We only get here once the old process is once again active."
        true
      ] ifNil:[ 
        self _exitCritical .
        self _rescheduleHonorOob: oldProc .
        self _enterCritical .
        false
      ]
    ] whileFalse .
  ] ifFalse:[
    newProc := self _findReadyProcess: honorOobInterrupts .
    self _switchFrom: oldProc to: newProc . "primitive updates activeProcess instVar"
  ]
%

category: 'Private'
method: 
_rescheduleHonorOob: oldProc
  | noEx |
  [
    1 timesRepeat:[ self class ] . "honor interrupts"
    noEx := true .
  ] ensure: [  "if an Exception is not resumed, unschedule oldProc"
    noEx ifNil:[
      self _enterCritical .
      self _unschedule: oldProc .
      self  _exitCritical .
    ].
  ].
%

category: 'Private'
method:
_checkForExpiredDelays: delayQ
    "returns number of expired delays"
    | readyCount dlysSize now |
    readyCount := 0 .
    dlysSize := delayQ size .
    now := self _timeoutNow: dlysSize .
    dlysSize ~~ 0 ifTrue:[ | idx ofs dT |
      idx := 0 .
      ofs := 1 .
      [ ofs <= dlysSize ] whileTrue:[  | aDly |
        "search  delayQ for the first item not yet expired.
         For items that have expired, activate them so they will be scheduled.
        "  
        aDly := delayQ at: ofs . 
        ((dT := aDly _signalTime) ~~ nil and:[ dT > now])  ifTrue: [ "this one not expired"
	  idx := ofs .
          ofs := dlysSize + 1 . "exit loop"
        ] ifFalse: [			"this one expired"
	  "self dbglog: ('sched: activating a ', aDly class name , ' time: ',
		now asString, ', _signalTime: ', aDly _signalTime asString).  "
	  aDly _activate .
          readyCount := readyCount + 1 .
          ofs := ofs + 1 .
        ].
      ].
      (idx == 0) ifTrue: [
        "all delays have expired, tell timeout thread to wait for more work."
	delayQ size: 0 .
	dlysSize := 0 .
        timeoutThreadActive ifTrue:[
          self _updateTimerThread: -1 priority: 15"inline userSchedulingPriority" .
          timeoutThreadActive := false .
        ].
        "self dbglog: 'sched: all delays expired' ."
      ] ifFalse:[
        idx ~~ 1 ifTrue: [
          delayQ removeFrom: 1 to: (idx - 1). "delete expired items"
	  dlysSize := delayQ size .
        ].
	dlysSize ~~ 0 ifTrue:[ | nextDly |
	  nextDly := delayQ at: 1 .
	  self _updateTimerThread: nextDly _signalTime priority: nextDly priority .
          timeoutThreadActive := true .
        ].
      ].
    ] .
    ^ readyCount
%

category: 'Private'
method: 
_reapEvents: yieldInt honorOob: honorOobInterrupts

  "Reaps any pending events that will cause a thread to be ready to run.
   yieldInt should be 0 to wait for an event, or 1 for  yielding. 
   Returns nil if scheduler would deadlock, 
   returns 0 if an OOB byte from stone has interrupted the waiting,
   returns self otherwise.
  "
  "senders must enter critical region by using prim 736"
  
  | delayQ readyQ pollCount |
  delayQ := delayQueue .
  readyQ := readyQueue .
  pollCount := 0 .
  [
    | readyCount dlysSize |
    readyCount := readyQueue size + (self _checkForExpiredDelays: delayQ) .
    dlysSize := delayQ size .
    (dlysSize + socketsPolled size ) ~~ 0 ifTrue: [
      | resSize resArray |
         "yield use case:  poll for socket activity, sleepVal== 0
          otherwise:  waitforever for socket activity, or wakeup by timeout thread."
      resSize := self _doPoll: (resArray := pollResults) yield: readyCount + yieldInt .
      pollCount := pollCount + 1 .
      resSize ~~ 0 ifTrue:[
	1 to: resSize do:[:k | | aSock |
	  aSock := resArray at: k .
	  aSock _reapEvents .
        ].
        self _sizeNoShrink: resArray .
        readyCount := readyQ size.
      ].
      yieldInt ~~ 0 ifTrue:[ ^ self ].
      readyCount ~~ 0 ifTrue:[ ^ self ]
        ifFalse:[ 
          (self _checkForExpiredDelays: delayQ) ~~ 0 ifTrue:[ ^ self ]
      ].
      honorOobInterrupts ifTrue:[ 
        "self dbglog: 'sched: no events but interrupted' ."
        ^ 0 "no events but interrupted" 
      ].
    ] ifFalse:[
      readyCount := readyQ size .
      (readyCount + yieldInt) ~~ 0 ifTrue: [  
        ^ self 
      ] ifFalse:[
        "readyCount == 0 ifTrue:[  self pause ]." "uncomment to debug deadlock"
        ^ nil .  "would deadlock if caller is not yield"
      ].
    ] .
    pollCount > 100 ifTrue:[  | err |
      (err := ThreadError new) _number: 2366 ; reason: #rtErrSchedulerDeadlocked;
	  args: { self . nil } ; 
          details: 'The ProcessorScheduler is deadlocked, infinite loop in _reapEvents: ' .
       self _endYield .
       err signal  
    ]. 
    true 
  ] whileTrue "forever loop"
%

category: 'Private'
method:
_findReadyProcess: honorOobInterrupts

  "Find another process to run, waiting for events if no process is
   ready to run.  Signals a ThreadError if a deadlock would occur because
   no other processes are ready nor waiting. 
   Returns the next process to run or returns nil if interrupted by OOB."

  <primitive: 736> "prim always fails, sets OM.schedulerInReapEvents flag"
  | proc err oldProc rpt res |
  res := self _reapEvents: 0 honorOob: honorOobInterrupts .
  res ~~ self ifTrue:[
    res == 0 ifTrue:[ "no events but interrupted by OOB from stone"
      ^ nil
    ].
    "we have  res == nil"
    oldProc := self activeProcess .
    oldProc ifNotNil:[ | q |  "normally oldProc is the active process"
        (q := oldProc onQueue) ifNotNil:[
           q ~~ readyQueue ifTrue:[
             oldProc _onQueue: nil .
             self _remove: oldProc fromSet: q .  "waitingSet or suspendedSet"
           ].
        ]
    ].
    (rpt := '----- ProcessorScheduler deadlock ' copy) add: ' current process ' ;
                           add: oldProc asOop asString ; lf .
    rpt addAll: self _allStacksReport .
    GsFile gciLogServer: rpt .  
    (err := ThreadError new) _number: 2366 ; reason: #rtErrSchedulerDeadlocked;
	 args: { self . oldProc . rpt } ; 
         details: 'The ProcessorScheduler is deadlocked' .
    self _endYield .
    err signal .   
  ].
  proc := readyQueue removeAtIndex:1 . "removeFirst"
  proc _onQueue: nil .
  "caller must send _switchFrom:to: or _endYield"
  ^ proc
%

category: 'Process State Change'
method:
yield
  "schedule a different active process, if any are
   available, and put the current process in the
   appropriate ready queue."

  <primitive: 736> "prim always fails, sets OM.schedulerInReapEvents flag"
  | readyQ |
  self _reapEvents: 1 honorOob: false .
  (readyQ := readyQueue) size ~~ 0 ifTrue: [
    (readyQ at:1) priority >= self activePriority  ifTrue: [ | oldProc newProc |
      oldProc := self activeProcess .

      newProc := readyQueue removeAtIndex:1 .
      newProc _onQueue: nil .

      self _addReadyProcess: oldProc to: readyQ .  "active process never on ready que"
      oldProc _onQueue: readyQ .

      "tracing code [ 
        GsFile gciLogServer:'yield old clientData=' , oldProc _clientData asString ,
		' new clientData=', newProc _clientData asString .
      ]."

      self _switchFrom: oldProc to: newProc .   "prim does _endYield, updates activeProcess iv"

      ^ self
    ].
  ].
  self _endYield .
%
method:
_yieldForTimeout
  "schedule a different active process, if any are
   available at a higher priority, and put the current process in the
   appropriate ready queue."

  <primitive: 736> "enter critical region, prim always fails"
  | readyQ |
  self _reapEvents: 1 honorOob: false .
  (readyQ := readyQueue) size ~~ 0 ifTrue: [
    (readyQ at:1) priority > self activePriority  ifTrue: [ | oldProc newProc |
      oldProc := self activeProcess .

      newProc := readyQueue removeAtIndex:1 .
      newProc _onQueue: nil .

      self _addReadyProcess: oldProc to: readyQ .  "active process never on ready que"
      oldProc _onQueue: readyQ .

      "tracing code [ 
        GsFile gciLogServer:'yield old clientData=' , oldProc _clientData asString ,
		' new clientData=', newProc _clientData asString .
      ]."

      self _switchFrom: oldProc to: newProc .   "prim does _endYield, updates activeProcess iv"
      ^ self
    ].
  ].
  self _endYield .
%

category: 'Private'
method:
_endYield

"clears OM.schedulerInReapEvents, which is also cleared by prim 2010"

<primitive: 737>   
self _primitiveFailed: #_endYield
%

category: 'Private'
method:
_doPoll: resultArray yield: yieldInt

"Wait for a GsSocket to be ready or for a timeout to expire.
 yieldInt > 0 means yield , i.e. just poll for GsSocket(s) ready ,
 yieldInt == 0 means poll until GsSocket(s) ready or until
 timer C thread delivers EINTR to main thread when a timeout expires.

 resultArray must be an Array. It will have stored into it, each
 instance of GsSocket that had an event detected on it.
 Returns nil if there was a socket error, otherwise returns number of
 elements in resultArray, or zero if the primitive timed out."

<primitive: 192>
self _uncontinuableError
%

category: 'Private'
method:
_waitForMilliseconds: intervalMs
  "Causes the current thread to suspend for the specified time .
   Returns true if the timeout expired. Returns false if the thread
   woke up early for some reason."
  "Sender has done _enterCritical "
  intervalMs _isSmallInteger ifFalse:[
    intervalMs _validateClass: SmallInteger.
  ].
  intervalMs <= 0 ifTrue:[
    self yield .
    ^ false
  ] ifFalse:[ | nowMs endMs aProc proc res |
    nowMs := self _now .
    endMs := 16rfffffffffffffff "max SmallInteger" . 
    (endMs - intervalMs) > nowMs ifTrue:[ 
       endMs := nowMs + intervalMs
    ].
    (proc := self activeProcess) _signalTime: endMs  .
    self _addDelay: proc to: delayQueue .

    self _reschedule: true .
    self _enterCritical .

    "we have woken up."
    aProc := self activeProcess .
    aProc ~~ proc ifTrue:[
      self _exitCritical .
      Error signal:'Unexpected: activeProcess changed during _waitForMilliseconds:'.
    ].
    res := true .
    aProc _signalTime ifNotNil:[
      "Woke up without the delay having been activated, so clean it up."
      delayQueue removeIdentical: aProc otherwise: nil . "inline _delayUnschedule:" 
      aProc  _signalTime: nil.
       res := false
    ].
    self _exitCritical .
    ^ res .
  ].
%

category: 'Private'
method:
_waitForSocket: aGsSocket
  "Causes the current thread to suspend until aGsSocket is read/writable,
   or has an error.  Timeout is infinite.  Returns true if socket is ready."

  | proc ws |
  (proc := self activeProcess) _waitingOn: aGsSocket .  "clear done by _reapSignal:"
  "assertion code"   
  "true ifTrue:[  
    (socketsPolled includesIdentical: aGsSocket) ifFalse:[ self pause ].
  ]."
  
  self _add: proc toSet: (ws := waitingSet) .
  proc _onQueue: ws .
    
  self _reschedule: false  .
%

! fixed 44336
category: 'Private'
method:
_waitForSocket: aGsSocket timeout: msToWait forWrite: writeBool

  "Causes the current thread to suspend until aGsSocket is read/writable,
   has an error, or the specified timeout occurs.
   msToWait must be > 0 . Returns receiver.  
   Caller must have done _enableWrite or _enableRead for some GsSocket,
   and added current process to the socket's waiters list.
   writeBool should be nil or true .
   Caller has sent _enterCritical .
  "
  
  | proc aProc |
  (proc := self activeProcess) _waitingOn: aGsSocket .  "clear done by _reapSignal:"
  "assertion code"  
  "true ifTrue:[   
    (socketsPolled includesIdentical: aGsSocket) ifFalse:[ self pause ].
  ]."
  
  proc _signalTime: (self _now) + msToWait.
  self _addDelay: proc to: delayQueue .
  
  self _reschedule: false .
  self _enterCritical .
  aProc := self activeProcess.
  aProc ~~ proc ifTrue:[
    self _exitCritical .
    Error signal:'Unexpected: activeProcess changed during _waitForSocket:timeout:'.
  ].
  proc _signalTime ifNotNil: [
    delayQueue removeIdentical: proc otherwise: nil . "inline _delayUnschedule:" 
    proc _signalTime: nil.
    proc _waitingOn: nil .
    writeBool ifNotNil:[ aGsSocket _cancelWriteEventFor: proc ] 
		 ifNil:[ aGsSocket _cancelReadEventFor: proc ].
  ].
  self _exitCritical .
%

category: 'Private'
method:
_waitOnSema: aSemaphore

  | proc ws |
  (proc := self activeProcess) _waitingOn: aSemaphore .
  self _add: proc toSet: (ws := waitingSet) .
  proc _onQueue: ws .

  self _reschedule: false 
%

category: 'Private'
method: ProcessorScheduler
_isWhenReadableSet: aGsSocket signal: objToNotify
  "Returns true if read event is still enabled.
   Returns false if read event was never set or has happened and been
   automatically unset."
  
  ^ aGsSocket hasReadWaiter: objToNotify
%

category: 'Private'
method: ProcessorScheduler
_isWhenWritableSet: aGsSocket signal: objToNotify
  "Returns true if write envent is still enabled.
   Returns false if write event was never set or has happened and been
   automatically unset."
  ^ aGsSocket hasWriteWaiter: objToNotify
%


category: 'Event Scheduling'
method:
whenReadable: aGsSocket signal: objToNotify
  "Causes the receiver to signal objToNotify once when aGsSocket is 
   readable or has an exceptional condition.
   Once the receiver signals the object for this event the event is
   canceled so you need to call this method again to reenable the event.
   objToNotify is signalled by sending it the message '_reapSignal: source'
   with source being the socket that is readable.
   The kernel classes that respond to _reapSignal: are:
     Semaphore, SharedQueue, and GsProcess.
   Returns self."

  aGsSocket _whenReadableNotify: objToNotify .
  "true ifTrue:[  assertion code
    (socketsPolled includesIdentical: aGsSocket) ifFalse:[ self pause ].
  ]."
%

category: 'Event Scheduling'
method:
whenWritable: aGsSocket signal: objToNotify
  "Causes the receiver to signal objToNotify once when aGsSocket is 
   writable or has an exceptional condition.
   Once the receiver signals the object for this event the event is
   canceled so you need to call this method again to reenable the event.
   objToNotify is signalled by sending it the message '_reapSignal: source'
   with source being the socket that is writable.
   The kernel classes that respond to _reapSignal: are:
     Semaphore, SharedQueue, and GsProcess.
   Returns self."

  aGsSocket _whenWritableNotify: objToNotify .
  "true ifTrue:[ assertion code
    (socketsPolled includesIdentical: aGsSocket) ifFalse:[ self pause ].
  ]."
%

method:
cancelWhenReadable: aGsSocket signal: objToNotify
  "Cancels the event specified by aGsSocket and objToNotify.
   The event is created using the whenReadable:signal: method.
   Returns true if the event has already been cancelled, reaped, or does not exist.
   Returns false if the event is cancelled by this invocation."

  ^ aGsSocket _cancelReadEventFor: objToNotify .
%

category: 'Event Scheduling'
method:
cancelWhenWritable: aGsSocket signal: objToNotify
  "Cancels the event specified by aGsSocket and objToNotify.
   The event is created using the whenWritable:signal: method.
   Returns true if the event has already been cancelled, reaped, or does not exist.
   Returns false if the event is cancelled by this invocation."

  ^ aGsSocket _cancelWriteEventFor: objToNotify .
%

category: 'Private'
method:
_scheduleProcess: aProcess
  "add the given process to the correct ready-queue"

  | q readyQ |
  readyQ := readyQueue .
  q := aProcess onQueue . 
  q ~~ readyQ ifTrue:[
    q ifNotNil:[
      aProcess _onQueue: nil .
      self _remove: aProcess fromSet: q .  "waitingSet or suspendedSet"
    ].
    self _addReadyProcess: aProcess to: readyQ .
    aProcess _onQueue: readyQ  .
  ].
%

category: 'Accessing'
method:
activeProcess

"Returns the currently active process.
 Creates a new GsProcess if needed and stores it into the
  activeProcess instVar of the scheduler instance."
<primitive: 693>
self _primitiveFailed: #_currentProcess
%

category: 'Private'
method:
_unschedule: aGsProcess

  "remove the given process from any scheduler state it is in."

  | q |
  q := aGsProcess onQueue .
  q ifNotNil:[
    aGsProcess _onQueue: nil .
    q == readyQueue ifTrue:[
      q removeIdentical: aGsProcess otherwise: nil.
    ] ifFalse:[
      self _remove: aGsProcess fromSet: q . "suspendedSet or waitingSet"
    ].
  ].
%

method:
_findReadyProcessForExit: oldProc
  "Returns either nil or the next GsProcess to run.
   If result is nil, caller must signal an appropriate exception .
   If result is not nil, caller must send
    _switchFrom:to:  to the scheduler to switch to returned GsProcess."

  "caller has entered critical region"
  | proc |
  (self _reapEvents: 0 honorOob: false ) ifNil:[
     oldProc ifNotNil:[ | q |  "normally oldProc is the active process"
       (q := oldProc onQueue) ifNotNil:[
          q ~~ readyQueue ifTrue:[
            oldProc _onQueue: nil .
            self _remove: oldProc fromSet: q .  "waitingSet or suspendedSet"
          ].
       ]
     ].
     self _endYield .
     ^ nil   "no other processes available to run"
  ].
  proc := readyQueue removeAtIndex:1 . "removeFirst"
  proc _onQueue: nil .
  ^ proc
%

category: 'Process Access'
method:
_terminateScheduledProcess: aGsProcess
  "remove the given process from its queues it is in"

  <primitive: 736> "enter critical region, prim always fails"
  (aGsProcess == self activeProcess) ifTrue: [ | nxt |
    nxt := self _findReadyProcessForExit: aGsProcess .
    (nxt ~~ nil and:[ nxt ~~ aGsProcess]) ifTrue:[
      self _switchFrom: nil to: nxt .
      self _uncontinuableError . "should not reach here"
    ]. 
  ] ifFalse: [ 
    aGsProcess _unscheduleProcess .
    self _exitCritical .
  ].
%

category: 'Private'
method:
_suspendProcess: aProcess 
  "suspend the process and do not schedule it for further processing"

  | suspSet |
  (aProcess == self activeProcess) ifTrue: [ 
    self _add: aProcess toSet: (suspSet := suspendedSet) .
    aProcess _onQueue: suspSet  .
    self _reschedule: false .
  ] ifFalse: [ | q |
    aProcess onQueue == (q := readyQueue) ifTrue:[
      q removeIdentical: aProcess otherwise: nil .
      self _add: aProcess toSet: (suspSet := suspendedSet) .
      aProcess _onQueue: suspSet .
    ] " ifFalse:  do nothing"
  ].
%

category: 'Private'
method:
_suspendNewProcess: aProcess

 "put a newly created Process in the suspendedSet ."

  | suspSet |
  aProcess _setPriority: self activePriority .
  self _add: aProcess toSet: (suspSet := suspendedSet) .
  aProcess _onQueue: suspSet .
%

method:
_scheduleNewProcess: aProcess prio: aPriority
  "Put a newly created process on the ready queue"
  | readyQ p |
  p := aPriority ifNil:[ self activePriority ].
  aProcess _setPriority: p .
  self _addReadyProcess: aProcess to: (readyQ := readyQueue) .
  aProcess _onQueue: readyQ .
%

category: 'Private'
method:
_resumeProcess: aProcess
  "Put a suspended process back to the state it was in when it was
   suspended."

  | q suspSet readyQ |
  q := aProcess onQueue .
  q == (suspSet := suspendedSet) ifTrue:[
    suspSet removeIdentical: aProcess otherwise: nil .
    aProcess waitingOn ifNotNil:[ | ws |
      self _add: aProcess toSet: (ws := waitingSet) .
      aProcess _onQueue: ws .
    ] ifNil:[ 
      self _addReadyProcess: aProcess to: (readyQ := readyQueue) .
      aProcess _onQueue: readyQ .
    ].
  ] ifFalse:[ | proc |  "inline _cancelDelay"
    proc := delayQueue removeIdentical: aProcess otherwise: nil .
    proc == aProcess ifTrue:[
      proc _signalTime: 0 .
      self _addReadyProcess: proc to: (readyQ := readyQueue) .
      proc _onQueue: readyQ .
    ] ifFalse:[
      (aProcess waitingOn isKindOf: GsSocket) ifTrue:[
        proc := waitingSet removeIdentical: aProcess otherwise: nil .
        proc == aProcess ifTrue:[
          aProcess _waitingOn: nil .
          self _addReadyProcess: proc to: (readyQ := readyQueue) .
          proc _onQueue: readyQ .
        ].
      ]
    ]
  ].
%

category: 'Accessing'
classmethod:
topazStatusForProcess: aGsProcess

^ self scheduler _statusString: aGsProcess
%

category: 'Private'
method:
_statusString: aGsProcess
  "Returns a string that describes aGsProcess's status."

  | result q waitOn |

  aGsProcess == self activeProcess ifTrue: [
     ^ 'active'.
  ].
  q := aGsProcess onQueue .
  q == readyQueue ifTrue: [
    ^ 'ready'.
  ].
  q == suspendedSet ifTrue: [
    ^ 'suspended'.
  ].
  waitOn := aGsProcess waitingOn .
  (q == waitingSet or:[ waitOn ~~ nil]) ifTrue:[
    result := String new.
    waitOn _isSymbol ifTrue:[ 
      result add: 'waiting on ' ; add: waitOn . "usually for Ruby IO.select"
    ] ifFalse:[
      result add: 'waiting on a '; add: waitOn class name .
    ].
    aGsProcess _signalTime ifNotNil:[
        result addLast: ', with a timeout'
    ]. 
  ] ifFalse:[
    q ifNotNil:[
      self halt:'invalid process state'.
    ].
    (aGsProcess _isTerminated) ifTrue: [
      result := 'terminated'.
    ] ifFalse: [
      aGsProcess _terminationStarted ifTrue:[
       result := 'terminationStarted' .
      ] ifFalse:[
        (delayQueue indexOfIdentical: aGsProcess) ~~ 0 ifTrue:[
          result := 'on delayQueue' .
        ] ifFalse:[ 
          "GCI application must have it in the debug state"
          result := 'debug'.
        ].
      ].
    ].
  ].
  ^ result
%

category: 'Accessing'
method:
activePriority
  "return the priority of the active process"
  | ap |
  ^ (ap := activeProcess) ifNil:  [ 15 "inline userSchedulingPriority" ]
                          ifNotNil: [ ap priority ]
%

category: 'Accessing'
method:
allProcesses
  "Return an IdentitySet of all of the instances of GsProcess 
   that are not terminated."

  ^ self _allProcessesInto: IdentitySet new
%

category: 'Accessing'
classmethod:
topazAllProcesses
 "Returns an Array of all GsProcess's known to the 
  scheduler , excluding the current active process."
 | set |
 set :=  self scheduler allProcesses .
 set removeIfPresent: GsProcess _current .
 ^ set asArray .
%

category: 'Debugging'
method:
_allProcessesTerminate9
  "Send terminate 9 to all GsProcess's known to the
   scheduler , excluding the current active process."
  | set polled |
  set :=  self allProcesses .
  set removeIfPresent: GsProcess _current .
  [
    set do:[:aProcess | aProcess terminate9 ].
    polled := socketsPolled shallowCopy . 
    1 to: polled size do:[:j | 
      (polled at: j) _disableEvent: nil ."disable all events and polling" 
    ].
  ] onException: AbstractException do:[:ex |
    ex resume "ignore"
  ].
%
classmethod:
_allProcessesTerminate9
  self scheduler _allProcessesTerminate9
%

category: 'Accessing'
method:
_allProcessesInto: anIdentitySet

  "Add all of the instances of GsProcess that are not terminated
   to result and return aCollection."
  <primitive: 736> "enter critical region, prim always fails"
  | polled |
  anIdentitySet _addAll: readyQueue forReplay: false .
  self _waitingProcessesInto: anIdentitySet .
  anIdentitySet _addAll: suspendedSet forReplay: false .
  1 to: (polled := socketsPolled) size do:[:j |
    (polled at: j) _waitingProcessesInto: anIdentitySet . 
  ]. 
  anIdentitySet add: self activeProcess .
  self _exitCritical .
  ^ anIdentitySet
%

method:
_allProcessesInto: anIdentitySet inGroup: groupId

  "Add all of the instances of GsProcess that are not terminated
   and are in specified group to aSet and return anIdentitySet."

  <primitive: 736> "enter critical region, prim always fails"
  | set proc polled | 
  anIdentitySet _addAll: readyQueue forReplay: false .
  1 to: (set := readyQueue) size do:[:n|
    proc := set at: n . 
    proc _groupOrNil == groupId ifTrue:[ anIdentitySet add: proc ].
  ].
  self _waitingProcessesInto: anIdentitySet inGroup: groupId .
  1 to: (set := suspendedSet) size do:[:n|
    proc := set _at: n . 
    proc _groupOrNil == groupId ifTrue:[ anIdentitySet add: proc ].
  ] .
  1 to: (polled := socketsPolled) size do:[:j |
    (polled at: j) _waitingProcessesInto: anIdentitySet inGroup: groupId .
  ]. 
  self _exitCritical .
  ^ anIdentitySet
%

method:
_waitingProcessesInto: anIdentitySet

  "Add all of the instances of GsProcess that 
   are waiting for an event to anIdentitySet and return anIdentitySet"

  | que |
  anIdentitySet _addAll: waitingSet forReplay: false .
  (que := { } ) addAll: delayQueue .
  1 to: que size do: [ :idx | 
     (que atOrNil: idx) ifNotNil:[:p | 
        p _targetProcess ifNotNil:[:t| anIdentitySet add: t]
     ]
  ].
  ^ anIdentitySet
%

method:
_waitingProcessesInto: aCollection inGroup: groupId

  "Add all of the instances of GsProcess that 
   are waiting for an event and are in specified group
   to aCollection and return aCollection"

  | proc set |
  1 to: (set := waitingSet) size do:[:n |
    proc := set _at: n .
    proc _groupOrNil == groupId ifTrue:[ aCollection add: proc ].
  ].
  set := delayQueue .
  1 to: set size do: [ :n | 
    proc := (set at: n) _targetProcess .
    proc ifNotNil:[ 
      proc _groupOrNil == groupId ifTrue:[ aCollection add: proc ].
    ].
  ].
  ^ aCollection
%

category: 'Accessing'
method:
readyProcesses
  "Return all of the instances of GsProcess that are ready to run but
   are not active."
  <primitive: 736> "enter critical region, prim always fails"
  | s |
  (s := IdentitySet new ) _addAll: readyQueue  forReplay: false .
  self _exitCritical .
  ^ s
%
category: 'Accessing'
method:
waitingProcesses
  "Return an IdentitySet of all of the instances of GsProcess that 
   are waiting for an event."
  
  <primitive: 736> "enter critical region, prim always fails"
  | res |
  res := self _waitingProcessesInto: IdentitySet new .
  self _exitCritical .
  ^ res
%

category: 'Accessing'
method:
suspendedProcesses
  "Return all of the instances of GsProcess that were suspended using
   the GsProcess>>suspend method
   or created with the ExecBlock>>newProcess method and not yet
   sent resume."

  ^ suspendedSet shallowCopy
%
! fix 41229
method:
_isSuspended: aGsProcess
  "Returns a Boolean "

  ^ suspendedSet includesIdentical: aGsProcess
%

category: 'Private'
method:
_add: anObject toSet: anIdentitySet

"Returns receiver."

<primitive: 726>   
self _primitiveFailed: #_add:toSet: args: { anObject . anIdentitySet }
%

category: 'Private'
method:
_remove: anObject fromSet: anIdentitySet

"Returns receiver. If size of anIdentitySet goes to zero
and it is committed, regenerates the in-memory state as not committed."

<primitive: 727>   
self _primitiveFailed: #_remove:fromSet: args: { anObject . anIdentitySet }
%

category: 'Private'
method:
_changePriority: aProcess from: oldPriority

  | q |
  aProcess onQueue == (q := readyQueue) ifTrue:[
    q removeIdentical: aProcess otherwise: nil .
    self _addReadyProcess: aProcess to: q . 
  ].
%

category: 'Private
method:
_addReadyProcess: aProcess to: aReadyQ

"Receiver should be the ProcessorScheduler's readyQueue.
 Appends aProcess to the receiver. Returns receiver."

<primitive: 733>  "prim fails if aReadyQ is large"
aReadyQ add: aProcess  
%

category: 'Private'
method:
_addDelay: aDelayOrProcess to: aDelayQ

"Receiver should be the ProcessorScheduler's delayQueue.
 Appends aDelayOrProcess to the receiver. Returns receiver."

<primitive: 734>  "prim fails if aDelayQ is large"
aDelayQ add: aDelayOrProcess  
%

category: 'Private
method:
_updateTimerThread: timeMs priority: anInt

"Restart timer thread so it will wakeup at the specified signal time.
 aSignalTime == -1 means  timer thread should wait forever for more work."

<primitive: 738>
self _primitiveFailed: #_updateTimerThread: args: { timeMs . anInt }
%

category: 'Private'
method:
_sizeNoShrink: anArray

"Sets logical size of anArray to zero; if anArray is a small object,
 the operation does not shrink the physical size of anArray .
 Returns receiver. "

<primitive: 739>
self _primitiveFailed: #_sizeNoShrink: args: { anArray }
%
category: 'Private'
method: 
_exitCritical
  "exit critical region. see also prim 737 in scheduler"
<primitive: 955>

self _primitiveFailed: #_exitCritical
%

category: 'Private'
method:
_enterCritical

<primitive: 736> "enter critical region, prim always fails"
^ self
%
category: 'In-Process Debugger Support'
method:
_criticalMethods

"Returns an IdentitySet of GsNMethods."

^ criticalMethods ifNil:[ self _initCriticalMethods ]
%

category: 'Debugging'
method:
_allStacksReport
  "includes the stack of the currently executing process."
  | allP rpt |
  rpt := String new .
  allP := self allProcesses .
  allP do:[:aProc | rpt add: aProc _reportString ].
  ^ rpt
%

