!========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! File RcArray.gs
!
! definitions for RcArray
!
!========================================================================


!=========================================================================
! Superclass Hierarchy:
!   RcArray, Array, 
!
!=========================================================================

! ------------------- Class definition for RcArray
expectvalue %String
run
(Globals at: #RcArray otherwise: nil) ifNil:[
    Array subclass: 'RcArray'
      instVarNames: #( )
      classVars: #()
      classInstVars: #()
      poolDictionaries: #()
      inDictionary: Globals
      options: #().
   ^ 'Created class RcArray'
] 
ifNotNil:[ ^ 'RcArray class already exists' ].
%

removeallmethods RcArray
removeallclassmethods RcArray
set class RcArray

! ------------------- Class methods for RcArray
category: 'For Documentation Installation only'
classmethod: RcArray
installDocumentation

self comment:
'An RcArray is an implementation of an array that can automatically resolve
concurrency conflicts for certain operations on the RcArray.  All of the 
public methods implemented in this class support some form of reduced conflict
behavior.  If a conflict with other update operations on the RcArray occurs, 
the Rc operation is replayed.  In most cases these can succeed, but there are 
some, for example, at:put:, that can fail in the retry during the commmit. 
See the individual method descriptions for details.

One use case for the RcArray is in the implementation of an event logger where it 
is expected that there could be concurrent adds.  These can be safely resolved
by retrying the add on a new view of the object.

Because implementation relies on the replay of the operations when there are conflicts,
it should probably not be used in applications that require high levels of concurrency
since the replay can cause a convoy of sessions all trying to commit their changes
to the RcArray.  For applications with expected high rates of concurrency you should 
consider using an RcQueue to accumulate the additions and have a single gem process
remove them from the RcQueue and put them in an RcArray. 

The commit order determines the order and values of the elements in the RcArray.
'
%

! ------------------- Instance methods for RcArray
category: 'Private'
method:
__rcAt: anOffset
anOffset == 0 ifTrue:[
  System _addRootObjectToRcReadSet: self .
] ifFalse:[
  self _rcAt: anOffset. "add path to leaf node to rcReadSet"
].
%


category: 'Adding'
method: RcArray
add: anObject

"Adds anObject to the RcArray and returns anObject.  The replay always succeeds by 
 adding anObject to the end of the RcArray."

self __rcAt: self size . "add path to leaf node to rcReadSet"
self addRedoLogEntryFor: #_redoAdd:  withArgs: { anObject } .
^ super add: anObject
%

category: 'Adding'
method: RcArray
addAll: aCollection

"Adds all of the elements of aCollection to the receiver and returns
 aCollection.  The replay always succeeds by adding the elements of aCollection 
 to the end of the RcArray."

self __rcAt: self size . "add path to leaf node to rcReadSet"
self addRedoLogEntryFor: #_redoAddAll:  withArgs: { aCollection copy } .

^ super addAll: aCollection
%

category: 'Updating'
method: RcArray
at: anIndex put: aValue

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Returns aValue.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds.

 Can cause an Rc-write-write concurrency conflict if another concurrent transaction
 commits an update to the same index in the RcArray and the value stored by the
 other transaction is different than aValue.
"
| oldValue sz |
"add path to leaf node to rcReadSet and get previous value, if any "
(sz := self size) >= anIndex  ifTrue:[ oldValue := self _rcAt: anIndex ]
                             ifFalse:[  self __rcAt: sz   ] .

self addRedoLogEntryFor: #_redoAtPut:  withArgs: { { anIndex . oldValue . aValue } }.

^ super at: anIndex put: aValue.
%

category: 'Updating'
method: RcArray
size: anInteger

"Changes the size of the receiver to anInteger and returns the receiver.

 If anInteger is less than the current size of the receiver, the
 receiver is shrunk accordingly.  If anInteger is greater than the
 current size of the receiver, the receiver is extended and new
 elements are initialized to nil.

 Can cause an Rc-write-write concurrency conflict if another concurrent transaction
 commits an update that changes the size of the RcArray to a value that is 
 greater than it was at the start of this transaction and anInteger is less 
 than what the other transaction changed it to.

 Generates an error if anInteger is not a SmallInteger."
| sz |
anInteger >= (sz := self size) ifTrue:[
  self __rcAt: sz  "add path to last node to rcReadSet"
] ifFalse:[
  System _addToRcReadSet: self includingAllNodes: true.
].
self addRedoLogEntryFor: #_redoSize:  withArgs: { { sz . anInteger } }.
^ super size: anInteger
%

category: 'Private'
method: RcArray
_redoAdd: anObject

"Performs the replay of adding anObject to the RcArray and returns true."

self __rcAt: self size .
super add: anObject.
self _rcAt: self size .
^ true
%

category: 'Private'
method: RcArray
_redoAddAll: aCollection

"Performs the replay of adding aCollection to the RcArray and returns true."
| oldSize newSize |
self __rcAt: ( oldSize := self size ).
super addAll: aCollection.
newSize := self size .
(newSize - oldSize) < 2000 ifTrue:[
  self _rcAt: newSize .
] ifFalse:[
  System _addToRcReadSet: self includingAllNodes: true.
].
^ true
%

category: 'Private'
method: RcArray
_redoAtPut: args

"Performs the replay of the at:put to the RcArray and returns false if the
 value change conflicts with another concurrent transaction change."

| idx oldVal newVal |

idx := args at: 1.
oldVal := args at: 2.
newVal := args at: 3.

(self size >= idx) ifTrue: [ | currentVal |
  currentVal := self at: idx. "value after selective abort"
  (( currentVal ~= newVal ) and: [currentVal ~= oldVal ]) ifTrue: [
     ^ false "disallow this commit, another transaction changed this value"
  ]
].

super at: idx put: newVal.
self _rcAt: idx .
^ true
%

category: 'Private'
method: RcArray
_redoSize: args

"Performs the replay of the size: to the RcArray and returns false if the
 size change conflicts with another concurrent transaction change."

| oldSize newSize currentSize |

oldSize := args at: 1.
newSize := args at: 2.
currentSize := self size.

((oldSize < currentSize) and: [newSize < currentSize]) ifTrue: [
  ^ false. "disallow commit, size was made larger by a concurrent transaction"
].

super size: newSize.
System _addToRcReadSet: self includingAllNodes: true.
^ true
%

category: 'Private'
method: RcArray
addRedoLogEntryFor: aSelector withArgs: args

"Creates a redo log entry for the selector with the specified argument array,
 adds it to the redolog for this session .
 Sender responsible for adding objects to the rcReadSet."

| logEntry | 
(logEntry := LogEntry new)
    receiver: self;
         selector: aSelector;
         argArray: args.
System redoLog addLogEntry: logEntry ;
       addConflictObject: self for: self.
%


