!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: reposit2.gs 40520 2016-10-20 17:20:32Z bretlb $
! Superclass Hierarchy:
!   Repository, Collection, Object.
!
!=========================================================================

category: 'Backup and Restore'
method: Repository
lastFailoverTime

"If system is in restore from transaction logs, returns the
 last failOver timestamp detected while reading transaction logs,
 or nil if no failOver timestamp has been processed since restore started.
 If system is not in restore, returns the time from the most 
 recent suspendCommitsForFailover, or nil if no suspendCommitsForFailover
 has been executed.

 Result is a DateAndTime or nil."

 | info tm suspendTime |
 ((info := self restoreStatusInfo) at: 2) == 2 ifTrue:[
   (tm := info at: 13) == 0 ifTrue:[ ^ nil ]. 
   ^ DateAndTime posixSeconds: tm offset: Duration zero
 ].
 (suspendTime := info at: 14) == 0 ifTrue:[ ^ nil ].
 ^ DateAndTime posixSeconds: suspendTime offset: Duration zero
% 

category: 'Private'
method: Repository
_doScan: kind fast: isFast with: inputArgument

" This method calls does the appropriate tests, and calls the the _scan primitive 
  which does the bulk of the work.

  If the SessionTemps current at: #GsOverrideNumThreads is defined, then that value is
  used to determine the number of threads to use.

   scanKind  Function     
   =======================
     0     allInstances  
     1     allReferences
     2     allReferencesToInstancesOfClasses
     3     allReferencesByParentClass
     4     allObjectsInObjectSecurityPolicies
     5     allObjectsLargerThan
"
| maxThreads percentCpu argClass result inputArg |
inputArg := (inputArgument _isArray and: [inputArgument size > 1])
                ifTrue:[ Array withAll:(IdentitySet withAll: inputArgument)]
                ifFalse:[ inputArgument ].
System needsCommit ifTrue: [ System _error: #rtErrAbortWouldLoseData ] .

isFast ifTrue: [ maxThreads := self _aggressiveMaxThreadCount.
                 percentCpu := 95 ]
       ifFalse: [ maxThreads := self getDefaultNumThreads. percentCpu := 90 ].
result := self _scanWithMaxThreads: maxThreads waitForLock: 60 pageBufSize: 8
               percentCpuActiveLimit: percentCpu scanKind: kind with: inputArg .
argClass := inputArg class.
(kind ~= 3) ifTrue: [
  ((argClass == Array) or: [argClass == GsBitmap]) ifFalse: [
       result := (result at: 1) at: 2. ] .
  ].
^result
%
category: 'Repository Analysis'
method: Repository
loadGcCandidatesFromFile: aString intoHiddenSet: hiddenSetId

"Given a file which is binary list of dead objects (produced by one of the 
 #findDisconnectedObjects: methods), load the objects into the hidden set 
 with the given ID.  NOTE: in 3.4 the hiddenSetId changed from a SmallInteger to
 one of the symbols in GsBitmap>>hiddenSetSpecifiers.

 Only objects which exist and are not in the stone's dead object
 set are added to the hidden set.

 Used by GsObjectInventory to profile garbage in the repository.

 Returns an Array containing 3 SmallIntegers:
  1 - Total number of objects in the file.
  2 - Number of valid objects loaded into the hidden set.
  3 - Number of invalid objects not loaded.

 Requires the GarbageCollection privilege.

 Signals an error if hiddenSetId specifies a hidden set only writable
 by SystemUser and the session is not logged in as SystemUser.
"

| gsBitmap numInvalid result bmSize |

gsBitmap := GsBitmap newForHiddenSet: hiddenSetId.
numInvalid := gsBitmap readFromFile: aString.
bmSize := gsBitmap size.
result := Array with: (bmSize + numInvalid)
                with: bmSize 
                with: numInvalid.
^result
%

category: 'Deprecated (Single Ref Path)'
method: Repository
findReferencePathToObject: anObject

"Finds a reference path to an object.  Returns an instance of GsSingleRefPathResult, 
 for which each element represents a connection leading between the search 
 objects and the reference in the repository.  
 The first element will usually be one of the objects in the limit set, and the last 
 element will be the argument. 

 The searchObj must be a committed non-special object. This method is not to be used 
 for classes or metaclasses. Objects in the stone's list of dead objects are excluded from 
 the scan.  It is an error if the object is in the stone's dead objects set. 

 Only a single reference path to each object is returned. It is possible for any object 
 to be connected to the repository through multiple reference paths.

 It is possible for an object to be disonnected from the repository and therefore have no
 reference path. In this case, the result GsSingleRefPathResult isDead will be true. 

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData.

 A reference path scan for an object is terminated and considered completed if any class  
 object or GsObjectSecurityPolicy is found in the reference path.  Normally all classes are 
 connected to the repository and will be included in the limit set.  However, for garbage 
 collection purposes, disconnected classes are considered live objects if any other object 
 references them.  In other words, a class is a live object if any live instances of that 
 class exist in the repository.
"

self deprecated: 'Repository>>findReferencePathToObject: deprecated v3.4. ',
 'Use (GsSingleRefPathFinder newForSearchObjects: {anObject}) runScan buildResultObjects first'.
	
System needsCommit ifTrue: [ System _error: #rtErrAbortWouldLoseData ] .

^ (GsSingleRefPathFinder newForSearchObjects: { anObject }) runScan
      buildResultObjects first
%

category: 'Deprecated (Single Ref Path)'
method: Repository
findReferencePathToObjects: anArray findAllRefs: findAllRefs printToLog: printToLog

"Finds a reference path to an array of objects.  Returns an array of GsSingleRefPathResult
 showing how each object is connected to the repository by reference.  

 Note that arguments to this method other than <anArray> are not used in v3.4.

 See the comments in findReferencePathToObject: for more information.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

self deprecated: 'Repository>>findReferencePathToObjects:findAllRefs:printToLog: deprecated v3.4. ',
 'Use (GsSingleRefPathFinder newForSearchObjects: anArray) runScan buildResultObjects'.
		 
System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ (GsSingleRefPathFinder newForSearchObjects: anArray) runScan
      buildResultObjects 
%

category: 'Deprecated (Single Ref Path)'
method: Repository
findReferencePathToObjs: anArray limitObjArray: limitArray findAllRefs: findAllRefs printToLog: printToLog

"Finds a reference path to an array of objects.  Returns an array of GsSingleRefPathResult
 showing how each object is connected to the repository by reference.

 Note that arguments to this method other than <anArray> are not used in v3.4.

 See the comments in findReferencePathToObject: for more information.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

self deprecated: 'Repository>>findReferencePathToObjs:limitObjArray:findAllRefs:printToLog: deprecated v3.4. ',
 'Use (GsSingleRefPathFinder newForSearchObjects: anArray) runScan buildResultObjects'.

System needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^ (GsSingleRefPathFinder newForSearchObjects: anArray) runScan
      buildResultObjects 
%

category: 'Private'
method
_buildListInstResult: allInstResult

 | gsBitmap arrSize element resultArr |

 gsBitmap := GsBitmap newForHiddenSet: #ListInstances.
 gsBitmap removeAll.

 arrSize := allInstResult size.
 arrSize == 1 ifTrue: [
   element := allInstResult at: 1.
   gsBitmap addAll: (element at: 2).
   ^ (element at: 2) size
 ].

 resultArr := Array new.
 1 to: arrSize do: [ :i |
   element := allInstResult at: i.
   resultArr add: (element at: 1).
   resultArr add: ((element at: 2) size).
   gsBitmap addAll: (element at: 2).
 ].
 ^ resultArr.
%

category: 'Private'
method
_buildAllRefsResult: allRefsResult forInput: inputArr 

 | resultArr |
 resultArr := Array new: (inputArr size).
 1 to: (allRefsResult size) do: [ :i |  | element |
   element := allRefsResult at: i.
   resultArr at: (inputArr indexOfIdentical: (element at: 1)) put: ((element at: 2) asArray)
 ].
 ^ resultArr.
%

category: 'Private'
method
_buildAllRefsResult: allRefsResult forInput: inputArr withLimit: aSmallInt

 | resultArr |
 resultArr := Array new: (inputArr size).
 1 to: (allRefsResult size) do: [ :i |  | element |
   element := allRefsResult at: i.
   resultArr at: (inputArr indexOfIdentical: (element at: 1))
             put: ((element at: 2) enumerateWithLimit: aSmallInt startingAfter: 0)
 ].
 ^ resultArr.
%

category: 'Listing Instances'
method:
allInstances: inputArg threads: numThreads

"Similar to allInstances: except that it allows specifying the number of threads to use for the scan.
 The number of threads can be overridden for any other listing instances method by executing:
    setDefaultNumThreads: numThreads 
 before executing the method.

"
^self setDefaultNumThreads: numThreads during: [
    self _doScan: 0 "OP_ALL_INSTANCES-OP_ALL_INSTANCES" fast: false with: inputArg 
  ]
%

category: 'Listing References'
method:
allReferences: inputArg threads: numThreads

"Similar to allReferences: except that it allows specifying the number of threads to use for the scan.
 The number of threads can be overridden for any other listing references method by executing:
    setDefaultNumThreads: numThreads 
 before executing the method.
"

^self setDefaultNumThreads: numThreads during: [
   self _doScan: 1 "OP_ALL_REFERENCES-OP_ALL_INSTANCES" fast: false with: inputArg 
 ]
%

category: 'Listing By Security Policy'
method:
allObjectsInObjectSecurityPolicies: inputArg threads: numThreads

"Similar to allObjectsInObjectSecurityPolicies: except that it allows specifying the 
 number of threads to use for the scan.
 The number of threads can be overridden for any other listing references method by executing:
    setDefaultNumThreads: numThreads
 before executing the method.
"
self _checkObjectSecurityPolicyArg: inputArg.
^self setDefaultNumThreads: numThreads during: [
  self _doScan: 4"OP_ALL_OBJS_IN_SEC_POLICY-OP_ALL_INSTANCES" fast: false with: inputArg 
]
%

category: 'Listing By Size'
method:
allObjectsLargerThan: aSize threads: numThreads

"Similar to allObjectsLargerThan: except that it allows specifying the number of threads to use for the scan.
"

^self setDefaultNumThreads: numThreads during: [
   self _doScan: 5 "OP_ALL_OBJS_LARGER_THAN-OP_ALL_INSTANCES" 
    fast: false with: aSize 
  ]
%

category 'Private'
method: Repository
_getNumThreads
  ^ SessionTemps current at: #GsOverrideNumThreads 
      ifAbsent: [ (SystemRepository numberOfExtents) * 2 ]
%

category 'Default numThreads'
method: Repository
getDefaultNumThreads
"Returns the default number of threads to use for multi threaded operations"

  ^ SessionTemps current at: #GsOverrideNumThreads 
      ifAbsent: [ 2 ]
%

category 'Default numThreads'
method: Repository
setDefaultNumThreads: numThreads
"Sets default number of threads to use for multi threaded operations"

 SessionTemps current at: #GsOverrideNumThreads put: numThreads
%

category 'Default numThreads'
method: Repository
setDefaultNumThreads: numThreads during: aBlock
"Sets default number of threads to use for the execution of aBlock containing 
 a multi threaded operation and then resets it after the operation completes."

| prevDefault |
 prevDefault := SessionTemps current at: #GsOverrideNumThreads otherwise: nil .
 
 SessionTemps current at: #GsOverrideNumThreads put: numThreads.
 ^ aBlock ensure:
     [prevDefault 
        ifNil:[ SessionTemps current removeKey: #GsOverrideNumThreads ]
        ifNotNil:[  SessionTemps current at: #GsOverrideNumThreads put: prevDefault ]
     ].
%

category: 'Garbage Collection'
method:
markForCollectionWithMaxThreads: numThreads

"Similar to markForCollection: except that it allows specifying the 
 number of threads to use for the scan.
 The number of threads can be overridden for any other markForCollection method by executing:
    setDefaultNumThreads: numThreads 
 before executing the method.
"
^self setDefaultNumThreads: numThreads during: [  self markForCollection]
%

category: 'Audit and Repair'
method:
objectAuditWithMaxThreads: numThreads

"Similar to objectAudit: except that it allows specifying the number of threads to use for the scan.
 The number of threads can be overridden for any other listing references method by executing:
   SessionTemps current at: #GsOverrideNumThreads put: numThreads
 before executing the method.
"
^self setDefaultNumThreads: numThreads during: [ self objectAudit ]
%

category: 'Backup and Restore'
method:
fullBackupTo: fileNameOrArrayOfNames threads: numThreads
"The same as fullBackupTo: except that it specifies the number of
 threads to use in the backup. The number of threads can be overridden for any other
 backup method by executing:
   SessionTemps current at: #GsOverrideNumThreads put: numThreads
 before executing the method.
"
^ self setDefaultNumThreads: numThreads during: [
            self fullBackupTo: fileNameOrArrayOfNames 
  ]
%

category: 'Backup and Restore'
method:
fullBackupGzCompressedTo: fileNameOrArrayOfNames threads: numThreads
"The same as fullBackupGzCompressedTo: except that it specifies the number of
 threads to use in the backup. The number of threads can be overridden for any other
 backup method by executing:
   SessionTemps current at: #GsOverrideNumThreads put: numThreads
 before executing the method.
"
^ self setDefaultNumThreads: numThreads during: [ 
           self fullBackupGzCompressedTo: fileNameOrArrayOfNames 
  ]
%

category: 'Backup and Restore'
method:
fullBackupLz4CompressedTo: fileNameOrArrayOfNames threads: numThreads
"The same as fullBackupLz4CompressedTo: except that it specifies the number of
 threads to use in the backup. The number of threads can be overridden for any other
 backup method by executing:
   SessionTemps current at: #GsOverrideNumThreads put: numThreads
 before executing the method.
"
^ self setDefaultNumThreads: numThreads during: [ 
           self fullBackupLz4CompressedTo: fileNameOrArrayOfNames 
  ]
%

category: 'Backup and Restore'
method:
restoreFromBackups: fileNameOrArrayOfNames threads: numThreads
"The same as restoreFromBackups: except that it specifies the number of
 threads to use in the restore. The number of threads can be overridden for any other
 restore method by executing:
   SessionTemps current at: #GsOverrideNumThreads put: numThreads
 before executing the method.
"
^ self setDefaultNumThreads: numThreads during: [
      self restoreFromBackups: fileNameOrArrayOfNames 
  ]
%

category: 'Garbage Collection'
method:
reclaimAllWait: timeLimitSeconds

"Explicitly triggers the reclamation of all shadowed objects.
 If the Gc configuration parameter #reclaimDeadEnabled is true, then it waits
 until the dead objects are ready to reclaim (voteStateIdle) before it starts the reclaim.

 If the timeLimitSeconds value is -1, then it will wait indefinitely for the reclaim to complete.
 This should be done with caution, however, as under certain conditions
 the session could appear to wait forever.  To avoid this you need to:

     1.  Make sure that other sessions are committing/aborting to allow
         voting on possible dead to complete.

 This method signals an error if:
     1.  The voteState is not idle and AdminGem is not running.
     2.  ReclaimGem is not running.
     3.  The timeout expires before the reclaim completes.
          Explicit errors are returned to indicate whether it was waiting for voteStateIdle or the reclaim.
     4.  No progress is detected for 5 minutes.
     
 A warning is logged if a session takes more than 20 seconds to vote.
 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 This method requires the GarbageCollection and SystemControl privileges.
"

| voteState gcUserUg reclaimDead sleepCount sleepTime timeLimit
  pagesNeedReclaiming prevPagesNeedReclaiming 
  deadNotReclaimed prevDeadNotReclaimed noProgressCount |

  System needsCommit
    ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

  System myUserProfile _validatePrivilegeName:#SystemControl .
  System myUserProfile _validatePrivilegeName:#GarbageCollection .

  voteState := System voteState.

"Admin gem must be present if voteState is not 0"
  ((voteState ~~ 0) and:[System adminGemSessionId == 0])
    ifTrue:[self _error: #rtErrReclaimAllMissingGcGem].

"Reclaim gem must be present"
  (System reclaimGemSessionId == 0)
     ifTrue:[self _error: #rtErrReclaimAllMissingGcGem].

  timeLimit := timeLimitSeconds.
  timeLimit = -1 ifTrue: [ timeLimit := SmallInteger maximumValue ].

" Wait for voteState idle if needed"
  gcUserUg := ((AllUsers userWithId:'GcUser' ) resolveSymbol:#UserGlobals ) value.
  reclaimDead := gcUserUg at:#reclaimDeadEnabled .
  sleepCount := 0.
  (voteState == 0) ifFalse: [
    reclaimDead ifTrue: [ "wait for vote state idle"  | prevSessVoting currSessVoting voteCount |
      [(System voteState) > 0] whileTrue: [
        System voteState == 1 ifTrue: [  "voting" 
          currSessVoting := System stoneCacheStatisticWithName: 'WaitingForSessionToVote'.
          currSessVoting == prevSessVoting
	    ifTrue: [
	      sleepCount - voteCount > 20 ifTrue: [
	         GsFile gciLogServer:'WARNING -- session ', currSessVoting asString ,
		                     ' taking too long to vote'] ]
            ifFalse: [ prevSessVoting := currSessVoting. voteCount := sleepCount]
        ].
        System sleep: 1. System abortTransaction.
        sleepCount := sleepCount + 1.
        (sleepCount > timeLimit) ifTrue: [
          UserDefinedError new _number: 2318;
            reason: #fail ; details: 'Timeout expired while waiting for voteState idle' ; signal.
          ]
       ]
     ]
     ifFalse: [
       ((System deadNotReclaimedCount > 0) or: [System possibleDeadSize > 0]) ifTrue: [
           GsFile gciLogServer:'WARNING --reclaimAll: ', 
             'vote state is not idle and reclaimDeadEnabled is false, so ', 
             'dead objects will not be reclaimed at this time.'
         ]  
       ]
     ].

"do simple commits to ensure CRs are disposed and reclaim completes"
  sleepTime := 1.
  noProgressCount := 0.
  prevPagesNeedReclaiming := 0.
  prevDeadNotReclaimed := 0.
  System abortTransaction .
  [(System _simpleCommitForReclaim: reclaimDead)] whileFalse: [
    System sleep: sleepTime .
    sleepCount := sleepCount + 1 .
    sleepCount > timeLimit ifTrue: [
      UserDefinedError new _number: 2318;
         reason: #fail ; 
         details: 'Waited too long for reclaimAll with timeLimit ' , timeLimit asString;
         signal.
    ].
    deadNotReclaimed := System deadNotReclaimedCount.
    pagesNeedReclaiming := System scavengablePagesCount.

    (sleepCount \\ 10) == 0  ifTrue:[
      GsFile gciLogServer:
       '--reclaimAll: ', DateTime now asStringMs, '  simpleCommit loopCount ' , sleepCount asString , 
            ' crBacklog ' , ((System commitRecordBacklog) asString) ,
            ' deadNotReclamed ' , deadNotReclaimed asString, 
            ' pagesNeedReclaiming ' , pagesNeedReclaiming asString
    ].
    
    ((reclaimDead and: [deadNotReclaimed < prevDeadNotReclaimed]) 
         or: [pagesNeedReclaiming < prevPagesNeedReclaiming])
       ifTrue: [ noProgressCount := 0.
                 sleepTime := 1.
           ] 
       ifFalse: [ noProgressCount := noProgressCount + sleepTime.
                 sleepTime := (sleepTime + 1 ) min: 15.
           ].
    prevPagesNeedReclaiming := pagesNeedReclaiming.
    prevDeadNotReclaimed := deadNotReclaimed.
    (noProgressCount > 300) ifTrue:[ | debugged |
      (System gemVersionReport at: #gsBuildType) = 'SLOW' ifTrue:[
         (System gemConfigurationAt: #GEM_LISTEN_FOR_DEBUG) ifTrue:[ | tmps |
           debugged := true .
           tmps := SessionTemps current .
           tmps at: #GSTestCaseWaitForDebug put: true .
           GsFile gciLogServer:'Waiting for debugger to attach, use topaz -r, DEBUGGEM' .
           [ tmps at: #GSTestCaseWaitForDebug  ] whileTrue:[ 
              Delay waitForSeconds: 1 .
              System abortTransaction.
           ]
         ] ifFalse:[
           System _hostCallDebugger.
           debugged := true .
         ].
      ].
      debugged ifNotNil:[
        UserDefinedError new _number: 2318;
         reason: #fail ; details: 'Waited too long for reclaimAll' ; signal.
      ].
    ].
    System abortTransaction.
  ].
%

category: 'Garbage Collection'
method:
reclaimAll

"Explicitly triggers the reclamation of all shadowed and dead objects if
 the Gc configuration parameter #reclaimDeadEnabled.

 The need for running reclaimAll is reduced by the newer multi-threaded
 implementations of backup, restore and objectAudit.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost it signals an error, #rtErrAbortWouldLoseData.         

 This method requires the GarbageCollection and SystemControl privileges.

 Signals an error if the timeout expires before the reclaim completes.
 
 This method waits for voting to complete and then the reclaim of any shadowed pages or dead objects.

 It waits forever, but throws an error if it detects that there is no progress for 5 minutes.
 "

 self reclaimAllWait: -1 
%

