!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   StringKeyValueDictionary, KeyValueDictionary, AbstractDictionary, 
!   Collection, Object.
!
!=========================================================================

removeallmethods StringKeyValueDictionary
removeallclassmethods StringKeyValueDictionary
set class StringKeyValueDictionary

! at:, at:put: changed from primitive to Smalltalk implementation
!  so key #=  compares respect unicode string compare mode, to fix 44097 

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'A StringKeyValueDictionary is a KeyValueDictionary in which the keys are
 Strings or MultiByteStrings.  The hash function treats the strings as
 case-sensitive and is equivalent to String >> hash .
 Key comparisons always use code point comparison.

 Key arguments which are Symbols are converted to Strings prior
 to lookup in or addition to a StringKeyValueDictionary .

 The hashing algorithm is described in

 Pearson, Peter K.,
 "Fast Hashing of Variable-Length Text Strings,"
 Communications of the ACM, 33:6 (June 1990), 667-680.

 The implementation employs two modifications:

 * The hash function uses at most 2008 bytes of the string.  Strings that are
   identical within this range have the same hash value.

 * In his paper, Pearson describes a technique for producing a larger number of
   hash values by always computing the hash function to 24 bits.  The hash
   value is the result of this function modulo the table size.  Therefore,
   there is no value in specifying a table size greater than 2 to the 24th
   power.

Constraints:
	numElements: SmallInteger
	numCollisions: SmallInteger
	collisionLimit: SmallInteger
	tableSize: SmallInteger' .
%

category: 'Hashing'
method:
hashFunction: aKey

"No longer used by implementation of StringKeyValueDictionary"

self shouldNotImplement: #hashFunction:
%
method:
compareKey: key1 with: key2

"No longer used by implementation of StringKeyValueDictionary"

self shouldNotImplement: #compareKey:with:
%
category: 'Private'
method:
collisionBucketClass

"Returns the class of object to create when keys collide."

^ EqualityCollisionBucket
%
method:
_hashOfKey: aKey

"computes the hash of the given key, equivalent to String>>hash"
<primitive: 993>
^ self _primitiveFailed: #_hashOfKey: args: { aKey }
%


category: 'Accessing'
method:
at: aKey ifAbsent: aBlock

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the result of evaluating the zero-argument block aBlock."
| x keyOfs buckOfs |
x := self _lookup: aKey .
x _isSmallInteger
  ifTrue:[ keyOfs := x bitAnd: 16r3FFFFFFF . buckOfs := x bitShift: -30]
  ifFalse:[ x ifNil:[ aBlock _isExecBlock ifFalse:[ aBlock _validateClass: ExecBlock ].
                     ^ aBlock value ].
            keyOfs := x at: 1 . buckOfs := x at: 2 ].
buckOfs ~~ 0 ifTrue:[ | collisionBkt |
  collisionBkt := self _at: keyOfs + 1 .
  ^ collisionBkt _at: buckOfs + 1 .
].
^ self _at: keyOfs + 1.
%

category: 'Accessing'
method:
at: aKey otherwise: defaultValue

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the given defaultValue ."

| x keyOfs buckOfs |
x := self _lookup: aKey .
x _isSmallInteger
  ifTrue:[ keyOfs := x bitAnd: 16r3FFFFFFF . buckOfs := x bitShift: -30]
  ifFalse:[ x ifNil:[ ^ defaultValue ].
            keyOfs := x at: 1 . buckOfs := x at: 2 ].
buckOfs ~~ 0 ifTrue:[ | collisionBkt |
  collisionBkt := self _at: keyOfs + 1 .
  ^ collisionBkt _at: buckOfs + 1 .
].
^ self _at: keyOfs + 1.
%

category: 'Accessing'
method:
name
"Returns the key of a key/value pair whose value is the receiver.
 If the receiver contains no such Association, returns nil."

| hashKey hashValue collisionBkt |
1 to: tableSize do: [ :tableIndex |
  hashKey := self keyAt: tableIndex.
  hashValue := self valueAt: tableIndex.
  (hashKey == nil)
    ifTrue: [ collisionBkt := hashValue.
              (collisionBkt == nil)
                 ifFalse: [ 1 to: collisionBkt numElements do: [ :i |
                            ((collisionBkt valueAt: i) == self)
                              ifTrue: [ ^collisionBkt keyAt: i ] ] ] ]
    ifFalse: [ (hashValue == self)
                  ifTrue: [ ^hashKey ] ]
  ].
^nil
%

category: 'Updating'
method:
at: aKey put: aValue

"Stores the aKey/aValue pair in the receiver.  Rebuilds the hash table if the
 addition caused the number of collisions to exceed the limit allowed.
 Returns aValue.  Uses code point comparison.

 If aKey is being added for the first time, an invariant copy of it is stored
 as the key.  

 Gs64 v3.0 - behavior changed to match method comments.
 Previous versions made aKey invariant without making a copy of it, which
 could signal authorization errors or create unintended commit conflicts."

self _stringAt: aKey put: aValue .
(numCollisions > collisionLimit)
  ifTrue: [self rebuildTable: tableSize * 2].
^aValue
%

category: 'Updating'
method:
_stringAt: aKey put: aValue

"Stores the aKey/aValue pair in the hash dictionary, using code point comparison."

<primitive: 256>
self _primitiveFailed: #_stringAt:put: args: { aKey . aValue } .
self _uncontinuableError
%

category: 'Accessing'
method:
_lookup: aKey 

"Performs a search of the receiver for the key aKey, using code point comparison.
 Returns a SmallInteger,  least significant 30 bits are unsigned key offset 
                          higher 30 bits of are unsigned bucket offset,
 or an Array of two SmallIntegers { keyOffset . bucketOffset }.
 The offsets are 1 based, a zero offset means not found, or no bucket.
 A result of nil means the key was not found."
<primitive: 83>
aKey _validateKindOfClass: String .
self _primitiveFailed: #_lookup: args: { aKey }.
%

method:
keyAt: aKey otherwise: defaultValue

"Returns the key that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value. "

| x keyOfs buckOfs |
x := self _lookup: aKey .
x _isSmallInteger
  ifTrue:[ keyOfs := x bitAnd: 16r3FFFFFFF . buckOfs := x bitShift: -30 ]
  ifFalse:[ x ifNil:[ ^ defaultValue ].
            keyOfs := x at: 1 . buckOfs := x at: 2 ].
buckOfs ~~ 0 ifTrue:[ | collisionBkt |
  collisionBkt := self _at: keyOfs + 1 .
  ^ collisionBkt _at: buckOfs 
].
^ self _at: keyOfs .
%

category: 'Removing'
method:
removeKey: aKey ifAbsent: aBlock

"Removes the key/value pair with key aKey from the receiver and returns
 the value.  If no key/value pair is present with key aKey, evaluates
 the zero-argument block aBlock and returns the result of that evaluation."

| x keyOfs buckOfs oldValue |
x := self _lookup: aKey .
x _isSmallInteger 
  ifTrue:[ keyOfs := x bitAnd: 16r3FFFFFFF . buckOfs := x bitShift: -30]
  ifFalse:[ aKey ifNil:[ self _error: #rtErrNilKey].
            x ifNil:[ ^ aBlock value ].
            keyOfs := x at: 1 . buckOfs := x at: 2 ].

buckOfs ~~ 0 ifTrue:[ | collisionBkt |
  collisionBkt := self _at: keyOfs + 1 .
  oldValue := collisionBkt _at: buckOfs + 1 .
  collisionBkt _removePairAt: buckOfs  .
] ifFalse:[ | vOfs |
  oldValue := self _at: (vOfs := keyOfs + 1) . 
  self _at: keyOfs put: nil ; _at: vOfs put: nil
].
numElements := numElements - 1.
^ oldValue 
%


category: 'Removing'
method:
removeKey: aKey otherwise: defaultValue

  ^ self removeKey: aKey ifAbsent:[ defaultValue ]
%

category: 'Removing'
method:
_removeKey: aKey 

"Removes the key/value pair with key aKey from the receiver ,
 without accessing the value of the key/value pair.
 If no key/value pair is present with key aKey, generates an error.
 Does not remove empty collision buckets."
| x keyOfs buckOfs |
x := self _lookup: aKey .
x _isSmallInteger
  ifTrue:[ keyOfs := x bitAnd: 16r3FFFFFFF . buckOfs := x bitShift: -30]
  ifFalse:[ x ifNil:[ ^ self _errorKeyNotFound: aKey ].
            keyOfs := x at: 1 . buckOfs := x at: 2 ].
buckOfs ~~ 0 ifTrue:[ | collisionBkt |
  collisionBkt := self _at: keyOfs + 1 .
  collisionBkt _removePairAt: buckOfs  .
] ifFalse:[ 
  self _at: keyOfs put: nil ; _at: keyOfs + 1 put: nil
].
numElements := numElements - 1.
^ self
%

category: 'Private'
method:
_reportString
  | str |
  str := String new .
  self keysAndValuesDo:[:k :v |
   str add: k ; add: '  ' ; add: v asString ; lf .
  ].
  ^ str
%

category: 'Private'
method: StringKeyValueDictionary
_buildWeakRefSet

"Adds the leaf objects to the hiddenSet."

| hashKey hashValue collisionBkt weakRef |
weakRef := (GsBitmap newForHiddenSet: #WeakReferences).

1 to: tableSize do: [ :tableIndex |
  hashKey := self keyAt: tableIndex.
  hashValue := self valueAt: tableIndex.
  (hashKey == nil)
    ifTrue: [ collisionBkt := hashValue.
              (collisionBkt == nil)
                 ifFalse: [ weakRef add: collisionBkt]]
  ].
^nil
%

