!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!    SymbolDictionary, IdentityDictionary, IdentityKeyValueDictionary, 
!    KeyValueDictionary, AbstractDictionary, Collection, Object
!
!=========================================================================

removeallmethods SymbolDictionary
removeallclassmethods SymbolDictionary

category: 'For Documentation Installation only'
classmethod: SymbolDictionary
installDocumentation

self comment:
'A SymbolDictionary is an IdentityDictionary in which the keys are canonical
 symbols (Symbol, DoubleByteSymbol, or QuadByteSymbol) and the values are SymbolAssociations.
 The key of each SymbolAssociation is also the key used by the SymbolDictionary
 to access that SymbolAssociation.

 It is recommended that the symbol keys be valid identifiers in the ANSI sense, 
 but it is not disallowed to use other kinds of symbols.  SymbolDictionary names
 that are identifiers can be used directly in compiled code, while names that are
 not identifiers require lookup.

 Only SymbolDictionaries can be used in symbol lists.
'
%

classmethod: SymbolDictionary
_listClassesIn: aDict

"Returns a String describing the classes contained in the argument.  If aDict
 is not a kind of SymbolDictionary, returns false.
 Legacy topaz method."

| result aSet sortedNames lf |
(aDict isKindOf: self) ifFalse:[ ^ false ].
aSet := SymbolSet new .
aDict associationsDo:[:assoc |
  (assoc _value isKindOf:Class) ifTrue:[ aSet add: assoc key ].
  ].
lf := Character lf .
aSet size == 0 ifTrue:[
  ^ 'The dictionary contains no classes.
'
  ].
sortedNames := SortedCollection withAll: aSet .
result := String new .
1 to: sortedNames size do:[:j |
  result addAll: (sortedNames at: j); add: lf .
  ].
^ result
%

! fix 48050
category: 'Browser Methods'
classmethod: SymbolDictionary
_listClassesIn: dictArg matching: aString

"Returns a String containing names of classes, one per line; or false.
 If dictArg == nil, list the whole symbolList .
 If aString ~~ nil, limit result to those classes for which
    (aClass name includesString: aString)==true .
 If aDict is not a kind of SymbolDictionary, returns false.
 Used by Topaz."
| list result |
dictArg  ifNil:[ list := GsSession currentSession symbolList ]
      ifNotNil:[ (dictArg isKindOf: self ) ifFalse:[ ^ false ].
                 list := { dictArg } ].
result := String new .
list do:[:aDict | | aSet sortedNames |
  aSet := SymbolSet new .
  aDict associationsDo:[:assoc |
    (assoc _value isKindOf:Class) ifTrue:[ aSet add: assoc key ]. 
  ].
  aString ifNil:[
    result size > 0 ifTrue:[ result lf ].
    result addAll: 'dictionary ' ;
           addAll: (aDict name ifNil:[ '(unnamed)' ]) .
    aSet size == 0 ifTrue:[ result addAll:' is empty' ].
    result lf .
  ] ifNotNil:[ 
    aSet := aSet select:[:n | n includesString: aString]
  ].
  sortedNames := SortedCollection withAll: aSet .
  1 to: sortedNames size do:[:j |
    result addAll: (sortedNames at: j); lf .
  ].
].
^ result
%

! removed reimplementation of new . default table size of 503
!  caused bug 31421

category: 'Accessing'
method: SymbolDictionary
associationAt: aKey

"Returns the SymbolAssociation with key aKey.  Generates an error if
 no such SymbolAssociation exists."

| anAssoc |

anAssoc :=  self associationAt: aKey otherwise: nil.
anAssoc == nil ifTrue:[ anAssoc := self _errorKeyNotFound: aKey] .
^ anAssoc
%

category: 'Accessing'
method: SymbolDictionary
associationAt: aKey ifAbsent: aBlock

"Returns the SymbolAssociation with key aKey.  If no such SymbolAssociation
 exists, returns the result of evaluating the zero-argument block aBlock."

<primitive: 57>
| aSym |
aKey _isSymbol ifFalse:[
  (aSym := Symbol _existingWithAll: aKey ) ifNotNil:[
    ^ self associationAt: aSym ifAbsent: aBlock .
  ].
].
^ aBlock ifNotNil:[ aBlock value ].
%

! inherited:
! at: aKey 

! fix 45367
category: 'Accessing'
method: SymbolDictionary
at: aKey ifAbsent: aBlock

"Returns the value of the SymbolAssociation with key aKey.  If no such
 SymbolAssociation exists, returns the result of evaluating the
 zero-argument block aBlock."

<primitive: 240>
| aSym |
aKey _isSymbol ifFalse:[
  aKey ifNotNil:[
    aSym := Symbol _existingWithAll: aKey .
    aSym ifNotNil:[
      ^ self at: aSym ifAbsent: aBlock .
    ]
  ].
].
aBlock ifNil:[ ^ nil ] .
^ aBlock value
%

category: 'Searching'
method: SymbolDictionary
includesKey: aKey

"Reimplemented from KeyValueDictionary for efficiency."

^ (self associationAt: aKey otherwise: nil ) ~~ nil
%

category: 'Deprecated'
method: SymbolDictionary
detectValues: aBlock ifNone: exceptionBlock

self deprecated: 'SymbolDictionary>>detectValues:ifNone: deprecated long before v3.0. Use keysAndValuesDo: instead.'.

"Evaluates aBlock repeatedly, with values of the receiver as the argument.
 Returns the first key for which aBlock evaluates to true.  If none of the
 receiver's values evaluates to true, evaluates the argument exceptionBlock and
 returns its value.  The argument aBlock must be a one-argument block, and
 exceptionBlock must be a zero-argument block."

self keysAndValuesDo: [ :aKey :aValue |
  (aBlock value: aValue) ifTrue: [ ^ aKey ].
  ].
^ exceptionBlock value.
%

category: 'Accessing'
method: SymbolDictionary
at: aKey

"Returns the value at the given key.  Generates an error if aKey not found."

| assoc |
assoc := self associationAt: aKey otherwise: nil .
assoc == nil ifTrue:[ ^ self _errorKeyNotFound: aKey ].
^ assoc _value
%

category: 'Accessing'
method: SymbolDictionary
at: aKey otherwise: defaultValue

"Returns the value at the given key.  If aKey is not found, returns
 defaultValue."

| assoc |
assoc := self associationAt: aKey otherwise: nil .
assoc == nil ifTrue:[ ^ defaultValue ].
^ assoc _value
%

category: 'Accessing'
method: SymbolDictionary
associationAt: aKey otherwise: defaultValue

"Returns the SymbolAssociation with key aKey.  If no such SymbolAssociation
 exists, returns the given default value."

<primitive: 57>

| aSym |
aKey _isSymbol ifFalse:[
  (aSym := Symbol _existingWithAll: aKey ) ifNotNil:[
    ^ self associationAt: aSym otherwise: defaultValue
  ].
].
^ defaultValue
%

!inherited: at: aKey otherwise: aValue

category: 'Accessing'
method: SymbolDictionary
name

"Returns the key of a SymbolAssociation whose value is the receiver.  If the
 receiver contains no such SymbolAssociation, returns nil."

^ self keyAtValue: self ifAbsent:[ nil ]
%

category: 'Accessing'
method: SymbolDictionary
names

"Returns an Array that contains all the keys of entries in the receiver whose
 value is the receiver itself.  The order of the elements in the result is
 arbitrary.  If no such keys are found, returns an empty Array."

| result |
result := { }  .
self keysAndValuesDo:[:aKey :aValue | 
  aValue == self ifTrue:[ result add: aKey ].
  ].
^ result
%

category: 'Accessing'
method: SymbolDictionary
name: aSymbol

"Equivalent to self at: aSymbol put: self."

^ self at: aSymbol put: self
%

category: 'Accessing'
method: SymbolDictionary
keys

"Returns a SymbolSet containing the receiver's keys."

| result |
result := SymbolSet new .
self keysDo:[ :aKey | result add: aKey ].
^ result
%

! add inherited
! rebuildTable: newSize
! removeAssociation: aSymAssoc
! removeAssociation: aSymAssoc ifAbsent: aBlock
! removeKey: aKey
! removeKey: aKey ifAbsent: aBlock
! select: aBlock
! detect: aBlock
! reject: aBlock
! selectValues: aBlock
! detectValues: aBlock ifNone: exceptionBlock
! inherited
! rejectValues: aBlock

category: 'Updating'
method: SymbolDictionary
atHash: hashIndex putKey: aKey

"Updates the hash table by storing aKey at the specified hashIndex."

aKey _isSymbol ifFalse:[ 
  aKey ifNotNil:[ aKey _errorExpectedClass: Symbol ].
].
^super atHash: hashIndex putKey: aKey.
%

category: 'Updating'
method: SymbolDictionary
atHash: hashIndex putValue: aValue

"Updates the hash table by storing aValue at the specified hashIndex."

self _validateAtHash: hashIndex putValue: aValue .
^ super atHash: hashIndex putValue: aValue.
%
method: SymbolDictionary
_validateAtHash: hashIndex putValue: aValue

| valueCls |
(self _validatePrivilegeOld: (self valueAtHash: hashIndex) new: aValue) ifTrue:[
 valueCls := aValue class .
 (valueCls == SymbolAssociation) ifFalse:[
  (valueCls == CollisionBucket) ifFalse:[
    aValue ifNotNil:[
      (aValue isKindOf: SymbolAssociation) ifFalse:[
        (aValue isKindOf: CollisionBucket) ifFalse:[
          aValue _error: #rtErrInvalidArgClass args:
              { SymbolAssociation . CollisionBucket . UndefinedObject }
          ]
        ]
      ].
    ].
  ].
]
%

method: SymbolDictionary
atHash: hashIndex putKey: aKey value: aValue

aKey _isSymbol ifFalse:[
  aKey ifNotNil:[ aKey _errorExpectedClass: Symbol ].
].
self _validateAtHash: hashIndex putValue: aValue .

^ super atHash: hashIndex putKey: aKey value: aValue
%

category: 'Updating'
method: SymbolDictionary
addAssociation: aSymbolAssociation

"Add the argument to the receiver."

(aSymbolAssociation isKindOf: SymbolAssociation) ifFalse:[
  aSymbolAssociation _validateClass: SymbolAssociation
].
(self _validatePrivilegeOld: (self at: aSymbolAssociation key otherwise: nil)
                       new: aSymbolAssociation _value) ifTrue:[ 
  ^ self _at: aSymbolAssociation key put: aSymbolAssociation
].
^ nil
%

category: 'Updating'
method: SymbolDictionary
renameAssociationFrom: key1 to: key2 

"Look up the Association in the receiver that has key1, and change its
 key to key2.  Raises an error if key1 is not found, or if key2 already
 exists.  key1 and key2 must be Symbols."

| assoc |
key1 _validateClass: Symbol.
key2 _validateClass: Symbol.
assoc := self associationAt: key1 .
(self _validatePrivilegeOld: assoc _value new: assoc _value ) ifTrue:[
  (self associationAt: key2 otherwise: nil) ~~ nil ifTrue:[
    ^ self _error: #rtErrDuplicateKey args: { key2 }. 
  ].
  self removeKey: key1 .
  assoc key: key2 .
  self addAssociation: assoc .
]
%

category: 'Updating'
method: SymbolDictionary
swapKey: key1 with: key2

"In the receiver, look up the Associations for key1 and key2 and
 swap the keys of the two Associations.  Returns the receiver.
 If either key1 or key2 is not found in the receiver, raises an error."

| assoc1 assoc2 |

key1 _validateClass: Symbol.
key2 _validateClass: Symbol.
assoc1 := self associationAt: key1 .
assoc2 := self associationAt: key2 .
(self _validatePrivilegeOld: assoc1 _value new: assoc1 _value) ifTrue:[
  (self _validatePrivilegeOld: assoc2 _value new: assoc2 _value) ifTrue:[
    self removeKey: key1 .
    self removeKey: key2 .
    assoc1 key: key2 .
    assoc2 key: key1 .
    self addAssociation: assoc1 .
    self addAssociation: assoc2
  ]
]
%

category: 'Updating'
method: SymbolDictionary
at: aKey put: aValue

"If the receiver already contains a SymbolAssociation with the given key, this
 makes aValue the value of that SymbolAssociation.  Otherwise, this creates a
 new SymbolAssociation with the given key and value and adds it to the
 receiver.  aKey must be a Symbol.   Returns aValue."

| anAssoc |
(self _validatePrivilegeOld: (self at: aKey otherwise: nil) new: aValue) ifTrue:[
  anAssoc:= self associationAt: aKey otherwise: nil .
  anAssoc == nil ifTrue:[
       self _at: aKey put:
       (SymbolAssociation newWithKey: aKey value: aValue).
       ^aValue
  ].

  tableSize := tableSize.  "make sure SymbolDictionary is dirty, not just Association (#42383)"
  anAssoc value: aValue.
  ^aValue
].
%

! doAssociations removed; inherit from KeyValueDictionary
! collectAssociations: aBlock
! detectAssociations: aBlock
! detectAssociations: aBlock ifNone: exceptionBlock
! rejectAssociations: aBlock
! selectAssociations: aBlock
! Inherited: _lockableValues
! inherited:
! keyAtValue: anObject ifAbsent: aBlock

category: 'Accessing'
method: SymbolDictionary
_behaviorKeys

"Returns a SymbolSet containing keys in the receiver whose values are
 Behaviors."

"This is used as an optimization by the GemBuilder for Smalltalk browser."

| result |
result := SymbolSet new .
self keysAndValuesDo:[ :aKey :aValue |
  aValue isBehavior ifTrue: [ result add: aKey ]].
^ result
%

category: 'Browser Methods'
method: SymbolDictionary
_classAndVersionStrings

"For all Behaviors in the receiver, returns an OrderedCollection of Strings
 showing the class name and version.  This method is used as an optimization by
 the GemBuilder for Smalltalk browser."

| result |
result := OrderedCollection new .
self associationsDo: [ :anAssoc | | each |
  each := anAssoc _value.
  each isBehavior
  ifTrue: [ result add: 
              ( each classHistory size == 1
                 ifTrue: [ each name asString ]
                 ifFalse: [ each name , ' [ ' ,  
                      ( each classHistory indexOf: each ) printString , ' ]' ]
               )
    ] 
  ].
^result
%

! Fix for 42694: SymbolDictionary#printOn: fails to print key when one of the values is the collection itself
! #printOn: removed, since superclass implementation does all that is needed, and correctly at that.

! edited to fix 37218 , 45548
category: 'Evaluation'
method: SymbolDictionary
textForError: aNumber args: anArray

self deprecated: 'SymbolDictionary>>textForError:args: deprecated v3.3. Use ANSI API in
AbstractException'.

"This code no longer functions correctly and is not used in the server image."

^ [ 
    | descrsArray errDescriptor myLanguage msg arrSize bad |
    myLanguage := System myUserProfile nativeLanguage.
    descrsArray := self at: myLanguage otherwise: nil .
    descrsArray ifNil:[ ^ self _errorKeyNotFound: myLanguage].
    errDescriptor := descrsArray atOrNil: aNumber .
    errDescriptor ifNil:[  | exCls |
      exCls := AbstractException legacyNumberToClass: aNumber .
      (msg := String withAll:'a ') add: exCls name ; add: ' occurred'.
    ] ifNotNil:[
      msg := String new.
      arrSize := anArray size .
      errDescriptor do: [ :arg |
	(arg isKindOf: String) ifTrue: [
	    msg addAll: arg
	  ] ifFalse: [
	    (arg _isSmallInteger and:[ arg >= 1 and:[ arg <= arrSize]]) ifTrue: [
		msg addAll:  (anArray at: arg) asString
	    ] ifFalse: [
		msg addAll: '???'.  "peculiar error list"
		bad := true 
	    ]
	  ]
	].
      bad ifNotNil:[
	msg addAll: ' , for error number ' , aNumber asString
      ].
    ].
    msg
  ] onException: Error do:[:ex |
    ^ [
         ex return: 'Error building error string for ' , ex category asString , ':' ,
                     ex number asString
      ] onSynchronous: AbstractException do:[:exb |
         exb return: 'Error building error for uninitialized Exception'
      ]
  ].
%
! deleted conversionRebuild, #46923

! deleted convertClassesTo5, fix 41973

! deleted convRecompileAllClassesWith:

! deleted convertPoolDictionary, #46923

! delete rehashForConversion, #46923 

! fixed 32112
category: 'CodeModification Override'
method: SymbolDictionary
addAll:  aCollection

(aCollection isKindOf: AbstractDictionary)
ifTrue: [
    aCollection associationsDo: [:x |
        (self _validatePrivilegeOld: (self at: x key otherwise: nil)
                               new: x _value) ifFalse:[ ^ nil ] ] 
] ifFalse: [
    aCollection do: [:x | 
        (x _validateClass: Association) ifFalse:[ ^ nil ].
        (self _validatePrivilegeOld: (self at: x key otherwise: nil)
                               new: x _value) ifFalse:[ ^ nil ] ] 
].
^ super addAll: aCollection
%

category: 'CodeModification Override'
method: SymbolDictionary
removeAssociation: anAssociation

(self _validatePrivilegeOld: (self at: anAssociation key otherwise: nil)
                       new: anAssociation _value ) ifTrue:[
  ^ super removeAssociation: anAssociation
]
%

category: 'CodeModification Override'
method: SymbolDictionary
removeAssociation: anAssociation ifAbsent: aBlock

(self _validatePrivilegeOld: (self at: anAssociation key otherwise: nil)
                       new: anAssociation _value) ifTrue:[
  ^ super removeAssociation: anAssociation ifAbsent: aBlock
]
%

category: 'CodeModification Override'
method: SymbolDictionary
removeKey: aKey

(self _validatePrivilegeOld: ( self at: aKey otherwise: nil )
                       new: ( self at: aKey otherwise: nil )) ifTrue:[
  ^ super removeKey: aKey
]
%

category: 'CodeModification Override'
method: SymbolDictionary
removeKey: aKey ifAbsent: aBlock

(self _validatePrivilegeOld: ( self at: aKey otherwise: nil )
                       new: ( self at: aKey otherwise: nil )) ifTrue:[
  ^ super removeKey: aKey ifAbsent: aBlock
]
%

category: 'CodeModification Override'
method: SymbolDictionary
_at: aKey put: anObject

(self _validatePrivilegeGeneric: anObject) ifTrue:[
  ^ super _at: aKey put: anObject
]
%

category: 'CodeModification Override'
method: SymbolDictionary
_basicAt: anInteger put: anObject

(self _validatePrivilegeGeneric: anObject) ifTrue:[
  ^ super _basicAt: anInteger put: anObject
]
%

category: 'CodeModification Override'
method: SymbolDictionary
squeakBasicAt: anIndex put: aValue

^ self _basicAt: anIndex put: aValue
%

category: 'CodeModification Override'
method: SymbolDictionary
_primitiveAt: anInteger put: anObject

(self _validatePrivilegeGeneric: anObject) ifTrue:[
  ^ super _primitiveAt: anInteger put: anObject
]
%

category: 'CodeModification Override'
method: SymbolDictionary
_unsafeAt: anInteger put: anObject

(self _validatePrivilegeGeneric: anObject) ifTrue:[
  ^ super _unsafeAt: anInteger put: anObject
]
%

category: 'CodeModification Override'
method: SymbolDictionary
_validatePrivilegeOld: oldValue new: newValue

( oldValue isKindOf: Behavior ) ifTrue: [
  ^ System myUserProfile _validateCodeModificationPrivilege
  ].
( newValue isKindOf: Behavior ) ifTrue: [
  ^ System myUserProfile _validateCodeModificationPrivilege
  ].
^ true
%

category: 'CodeModification Override'
method: SymbolDictionary
_validatePrivilegeGeneric: newValue

" This version covers any weird arguments being passed 
  in primitive _at:put: methods and variations "

System _inProtectedMode 
    ifTrue: [ 
      "We go into protected mode when we rebuildTable:, do we don't need to worry about privileges"
      ^ true 
    ].

( newValue isKindOf: Behavior ) ifTrue: [
   ^ System myUserProfile _validateCodeModificationPrivilege
   ].

( newValue isKindOf: Association ) ifTrue: [ 
    ( newValue _value isKindOf: Behavior ) ifTrue: [
        ^ System myUserProfile _validateCodeModificationPrivilege
       ] 
    ].

( newValue isKindOf: AbstractCollisionBucket ) ifTrue: [
    1 to: newValue size do: [ :i |
      (( newValue valueAt: i ) isKindOf: Behavior ) ifTrue: [
	^ System myUserProfile _validateCodeModificationPrivilege
	].
      ].
    ].
^ true
%

category: 'CodeModification Override'
method: SymbolDictionary
rebuildTable: newSize

<primitive: 2001>  "enter protected mode, to disable CodeModification checks"
| prot |
prot := System _protectedMode .
[
  super rebuildTable: newSize.
] ensure:[
  prot _leaveProtectedMode
]
%

! try:on:do: deleted

! support for 42240
category: 'Enumerating'
method: SymbolDictionary
classesDo: aBlock

"Execute 3-argument block over all classes (plus those referenced by
 its class history) that are contained by this SymbolDictionary.

 Arguments to block are:
    [:aUserProfile :aSymbolDictionary :aClass | ... ]

"

  | coveredClasses |

  coveredClasses := IdentitySet new.
  self valuesDo: [ :namedClass |
    (namedClass isKindOf: Class) ifTrue: [
      namedClass classHistory do: [ :aClass |
        (coveredClasses includes: aClass) ifFalse: [
          coveredClasses add: aClass.
          aBlock value: (System myUserProfile)
                 value: self
                 value: aClass ]]]]
%

! fix 47888
category: 'Formatting'
method: SymbolDictionary
printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet
  
  self name ifNotNil:[:nam |
    aStream nextPutAll: self class name describeClassName .
    aStream nextPutAll: '( name: ';
            nextPutAll: nam printString ; 
            nextPutAll: ' )' .
 ] ifNil:[
   super printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet
 ]
%

