!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   SymbolList, Array, SequenceableCollection, Collection, Object.
!
!=========================================================================

removeallmethods SymbolList
removeallclassmethods SymbolList

! SymbolList disallows gci store, part of fix 31942 .
! This  class modification is done in  bomlastconv.gs , because we
! don't have methods filed in for UserProfile yet at this point of filein.
! run
! SymbolList _disallowGciCreateStore .   
! ^ SymbolList _structuralUpdatesDisallowed
! %

category: 'For Documentation Installation only'
classmethod: SymbolList
installDocumentation

self comment:
'A SymbolList is an Array whose elements are instances of SymbolDictionary.

 It is used in compilation of GemStone Smalltalk code, in order to resolve
 references to objects by name.  Given a Symbol as a name, the SymbolList
 searches its dictionary elements in order, and the first key that matches the
 given Symbol then resolves to the object that is the value at that key in that
 dictionary.

 Constraints on indexable fields are enforced by behavior in this class.

Constraints:
	[elements]: SymbolDictionary' .
%

category: 'Accessing'
method: SymbolList
names

"Returns an Array of Strings containing the names of the receiver's
 SymbolDictionaries.

 This method assumes that each SymbolDictionary in the receiver contains a
 SymbolAssociation whose value is that SymbolDictionary.  If any
 SymbolDictionary does not contain such a SymbolAssociation, it is
 represented in the result Array as '(unnamed Dictionary)'."

| result aDict aSymAssoc|
result:= { }  .
1 to: self size do:[:j|
  aDict := self at: j .
  aDict == nil
    ifTrue:[ result add: '(nil)' ]
    ifFalse:[
      aSymAssoc:= aDict associationsDetect:[:anAssoc | anAssoc value == aDict] 
			ifNone:[ nil ].
      aSymAssoc == nil
        ifTrue:[ result add: '(unnamed Dictionary)' ]
        ifFalse:[ result add: (aSymAssoc key)]
      ].
  ].
^ result
%
category: 'Formatting'
method: SymbolList
namesReport

"Returns a formatted String describing the position and name of each Dictionary
 in the receiver's symbol list.

 This method assumes that each Dictionary in the symbol list contains an
 Association whose value is that Dictionary.  If any Dictionary does not
 contain such an Association, it is represented in the result String as
 '(unnamed Dictionary)'."

"used by the topaz list dictionaries command"

| arrayOfNames result lf |

arrayOfNames := self names .
result := String new .
lf := Character lf .
1 to: arrayOfNames size do:[:j|
  result add: (j asString).
  result addAll: '  '.
  result addAll: (arrayOfNames at: j) .
  result add: lf.
  ].
^result
%

category: 'Searching'
method: SymbolList
resolveSymbol: aString

"Searches the receiver for an Association whose key is equal to aString, and
 returns that Association.  If no such Association is found, returns nil."

<primitive: 49>

(aString _isSymbol) ifFalse:[
  ^ (Symbol _existingWithAll: aString) ifNotNil:[ :sym | self resolveSymbol: sym]
].
aString _validateClasses: { String }.
self _primitiveFailed: #resolveSymbol: args: { aString } .
self _uncontinuableError
%

category: 'Searching'
method: SymbolList
objectNamed: aSymbol

"Returns the first object in the receiver that has the given name.  If no object
 is found with the given name, returns nil."

| assn |

assn := self resolveSymbol: aSymbol.
assn == nil ifTrue: [ ^nil ].
^assn value
%

category: 'Updating'
method: SymbolList
createDictionaryNamed: dictName at: anIndex 

"Creates a new SymbolDictionary in the GsObjectSecurityPolicy of the receiver.  Adds to the
 dictionary an Association whose key is dictName and whose value is the
 dictionary.  Inserts the dictionary in the receiver at anIndex.

 If anIndex is less than 1, the dictionary is inserted at the front of the list.
 If anIndex is greater than the size of the list, it is inserted at the end of
 the list.  If the receiver already contains a dictionary named dictName, raises
 an error."

| newDict nameSym offset |

self _validatePrivilege .
nameSym := dictName asSymbol .
self do:[ :aDict | 
(aDict names includesIdentical: nameSym) ifTrue:[ 
  aDict _error: #rtErrDuplicateKey args: { nameSym }.
  ].
].
(newDict := SymbolDictionary new) name: nameSym .
newDict objectSecurityPolicy: self objectSecurityPolicy .
offset := anIndex .
offset < 1 ifTrue:[ offset := 1 ].
offset > self size 
ifTrue:[ self addLast: newDict ]
ifFalse:[ self insertObject: newDict at: offset ].
%

category: 'Updating'
method: SymbolList
removeDictionaryNamed: aSymbol 

"Removes the first dictionary found in the receiver that contains an Association
 whose key is aSymbol and whose value is the dictionary.  Returns the removed
 dictionary.

 If no such dictionary is found, raises a KeyNotFound error."

self _validatePrivilege.
^ self removeDictionaryNamed: aSymbol ifAbsent:[ 
  self _error: #rtErrKeyNotFound args: { aSymbol }.  
  nil 
] 
%

category: 'Updating'
method: SymbolList
replaceElementsFrom: aSymbolList

"Removes all elements in the receiver and inserts all elements of aSymbolList in
 it, in the same order as in aSymbolList.
 
 If the argument is not a SymbolList, raises an error."

| newSize |
self _validatePrivilege: aSymbolList .
aSymbolList == self ifTrue:[ ^ self ].
(aSymbolList isKindOf: SymbolList) ifFalse:[ 
aSymbolList _validateClass: SymbolList .
^ self
].
self size: ( newSize := aSymbolList size) .
self replaceFrom: 1 to: newSize with: aSymbolList startingAt: 1 
%

category: 'Updating'
method: SymbolList
removeDictionaryNamed: aString ifAbsent: aBlock 

"Removes the first dictionary found in the receiver that contains an Association
 whose key is aSymbol and whose value is the dictionary.  Returns the removed
 dictionary.

 If no such dictionary is found, returns the result of evaluating the
 zero-argument Block aBlock." 

self _validatePrivilege .
(Symbol _existingWithAll: aString) ifNotNil:[ :nameSym |
	1 to: self size do:[ :j | | aDict |
		aDict := self at: j .
		(aDict names includesIdentical: nameSym) ifTrue:[ 
		self removeAtIndex: j .
		^ aDict .
		].
	].
].
^ aBlock value
%

category: 'Searching'
method: SymbolList
symbolResolutionOf: aString

"Searches the receiver for aString.  If aString is found, returns a formatted
 String that describes the position in the receiver of the Dictionary defining
 aString, the name of that Dictionary, and aString.

 Generates an error if aString is not defined in the receiver."

"Example: GsSession currentSession symbolList symbolResolutionOf: #Integer"

| lf result theAssoc aDict aSymbol |

aSymbol := Symbol _existingWithAll: aString.
aSymbol == nil ifTrue:[ self _error: #rtErrSymNotFound args: { aString } ].

result := String new.
lf := Character lf.
1 to: self size do:[ :j|
  aDict := self at: j .
  aDict == nil ifFalse:[
    (aDict at: aSymbol otherwise: nil) ifNotNil:[
      " found aSymbol in this dict "
      result add: lf; add: (j asString); addAll: '  '.
      theAssoc:= aDict associationsDetect: [:anAssoc | anAssoc value == aDict]
                               ifNone: [ nil ] .
      theAssoc == nil
        ifTrue:[ result add: '(Unnamed Dictionary)']
        ifFalse: [ result add: theAssoc key ].
      result add: lf; addAll: '      '; addAll: aSymbol;
                 addAll: '  '; add: ((aDict at: aSymbol) asString).
      ^ result 
      ].
    ].
  ].
^ self _error: #rtErrSymNotFound args: { aString }
%

! deleted convertContentsTo5 , fix 41973

! deleted convRecompileAllClasses

category: 'Searching'
method: SymbolList
dictionaryAndSymbolOf: anObject

"Returns an Array containing two elements for the first Association
 in the enumeration of the receiver for which value == anObject .
 
 * The Dictionary in the receiver that contains an Association whose
   value is anObject.
 * The Symbol which is that Association's key.

 The receiver is searched in the same order that the compiler searches it.
 (For more information about symbol resolution, see the GemStone Programming
 Guide.)  If anObject is not found in the receiver, returns nil."

| aDict |
self deprecated: 'SymbolList>>dictionaryAndSymbolOf: deprecated v3.1, use dictionariesAndSymbolsOf:' .
1 to: self size do:[ :j | 
  (aDict := self at: j) ifNotNil:[ 
    aDict associationsDo:[ :assoc |    
      assoc value == anObject ifTrue:[ ^ { aDict . assoc key } ].
    ].
  ].
].
^ nil
%

method: SymbolList
dictionariesAndSymbolsOf: anObject

"Returns an Array, possibly empty, of the form { { aDictionary . aSymbol } ... }.
 Searches the receiver for Associations whose value is identical to
 anObject and returns information for all matching Associations. "

| res aDict |
res := { } .
1 to: self size do:[ :j | 
  (aDict := self at: j) ifNotNil:[ 
    aDict associationsDo:[ :assoc |
      assoc _value == anObject ifTrue:[ res add: { aDict . assoc key }]
    ].
  ].
].
^ res
%

category: 'CodeModification Override'
method: SymbolList
, aSequenceableCollection

self _validatePrivilegeAdd: aSequenceableCollection .
^ super , aSequenceableCollection
%

!  _validateElement: added with fix 31942
category: 'Private'
method: SymbolList
_validateElement: anObj

"if anObj is a kind of SymbolDictionary, return true, otherwise return false"

(anObj isKindOf: SymbolDictionary) ifTrue: [ ^ true ].
ArgumentTypeError new constrainedIv: 'SymbolList.element'
expectedClass: SymbolDictionary actualArg: anObj ;
 signal .
self _uncontinuableError
%

category: 'CodeModification Override'
method: SymbolList
add: newObject

self _validatePrivilegeAdd: newObject .
self _validateElement: newObject .
^ super add: newObject
%

category: 'CodeModification Override'
method: SymbolList
add: newObject after: target

self _validatePrivilegeAdd: newObject .
self _validateElement: newObject .
^ super add: newObject after: target
%

category: 'CodeModification Override'
method: SymbolList
add: newObject before: target

self _validatePrivilegeAdd: newObject .
self _validateElement: newObject .
^ super add: newObject  before: target
%

category: 'CodeModification Override'
method: SymbolList
addAll: aCollection

self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj | 
	me _validateElement: anObj .
].
^ super addAll: aCollection
%

category: 'CodeModification Override'
method: SymbolList
addAll: aCollection afterIndex: index

self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj | 
	me _validateElement: anObj .
].
^ super addAll: aCollection afterIndex: index
%

category: 'CodeModification Override'
method: SymbolList
addAll: aCollection before: target

self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj | 
	me _validateElement: anObj .
].
^ super addAll: aCollection before: target .
%
category: 'CodeModification Override'
method: SymbolList
addAll: aCollection beforeIndex: index

self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj | 
	me _validateElement: anObj .
].
^ super addAll: aCollection beforeIndex: index
%

category: 'CodeModification Override'
method: SymbolList
addLast: newObject

self _validatePrivilegeAdd: newObject .
self _validateElement: newObject .
^ super addLast: newObject
%

category: 'CodeModification Override'
method: SymbolList
atAllPut: anObject

self _validatePrivilegeAdd: anObject .
self _validateElement: anObject .
^ super atAllPut: anObject
%

category: 'CodeModification Override'
method: SymbolList
first: anObject

self _validatePrivilegeAdd: anObject .
self _validateElement: anObject .
^ super first: anObject
%

category: 'CodeModification Override'
method: SymbolList
insertAll: aCollection at: anIndex

self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj | 
	me _validateElement: anObj .
].
^ super insertAll: aCollection at: anIndex
%

category: 'CodeModification Override'
method: SymbolList
insertObject: anObject at: anIndex

self _validatePrivilegeAdd: anObject .
self _validateElement: anObject .
^ super insertObject: anObject   at: anIndex
%

category: 'CodeModification Override'
method: SymbolList
last: anObject

self _validatePrivilegeAdd: anObject .
self _validateElement: anObject .
^ super last: anObject
%

category: 'CodeModification Override'
method: SymbolList
remove: oldObject

self _validatePrivilege: oldObject .
^ super remove: oldObject
%

category: 'CodeModification Override'
method: SymbolList
remove: oldObject ifAbsent: anExceptionBlock

self _validatePrivilege: oldObject .
^ super remove: oldObject ifAbsent: anExceptionBlock
%

category: 'CodeModification Override'
method: SymbolList
removeAll: aCollection

self _validatePrivilege: aCollection .
^ super removeAll: aCollection
%

category: 'CodeModification Override'
method: SymbolList
removeAllIdentical: aCollection

self _validatePrivilege: aCollection .
^ super removeAllIdentical: aCollection
%

category: 'CodeModification Override'
method: SymbolList
removeAllSuchThat: aBlock

self _validatePrivilege: self .
^ super removeAllSuchThat: aBlock
%

category: 'CodeModification Override'
method: SymbolList
removeAtIndex: anIndex

self _validatePrivilege: self .
^ super removeAtIndex: anIndex
%

category: 'CodeModification Override'
method: SymbolList
removeFirst

self _validatePrivilege: self .
^ super removeFirst
%

category: 'CodeModification Override'
method: SymbolList
removeFrom: startIndex to: stopIndex

self _validatePrivilege: self .
^ super removeFrom: startIndex to: stopIndex
%

! size: added with fix 31942
category: 'CodeModification Override'
method: SymbolList
size: anInteger

self _validatePrivilege: self .
^ super size: anInteger 
%

category: 'CodeModification Override'
method: SymbolList
protectedRemoveFrom: startIndex to: stopIndex

<protected>
^ super removeFrom: startIndex to: stopIndex
%


category: 'CodeModification Override'
method: SymbolList
removeIdentical: oldObject

self _validatePrivilege: oldObject .
^ super removeIdentical: oldObject
%

category: 'CodeModification Override'
method: SymbolList
removeIdentical: oldObject ifAbsent: anExceptionBlock

self _validatePrivilege: oldObject .
^ super removeIdentical: oldObject ifAbsent: anExceptionBlock
%

category: 'CodeModification Override'
method: SymbolList
removeLast

self _validatePrivilege: self .
^ super removeLast
%

category: 'CodeModification Override'
method: SymbolList
replaceFrom: startIndex to: stopIndex with: aCollection

self _validatePrivilege: self .
self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj | 
	me _validateElement: anObj.
].
^ super replaceFrom: startIndex to: stopIndex with: aCollection
%

category: 'CodeModification Override'
method: SymbolList
replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex

| count |
self _validatePrivilege: self .
self _validatePrivilegeAdd: aSeqCollection .
count := stopIndex - startIndex + 1 .
0 to: count -1 do:[:j |
	self _validateElement:( aSeqCollection at: repIndex + j) .
]. 
^ super replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex
%

category: 'CodeModification Override'
method: SymbolList
replaceFrom: startIndex to: stopIndex withObject: anObject

self _validatePrivilege: self .
self _validatePrivilegeAdd: anObject .
self _validateElement: anObject .
^ super replaceFrom: startIndex to: stopIndex withObject: anObject
%

category: 'CodeModification Override'
method: SymbolList
_addAllFromNsc: aBag

self _validatePrivilegeAdd: aBag .
aBag accompaniedBy: self do:[:me :anObj | 
	me _validateElement: anObj .
].
^ super _addAllFromNsc: aBag
%

category: 'CodeModification Override'
method: SymbolList
at: anIndex put: aValue

self _validatePrivilegeAdd: aValue .
self _validateElement: aValue .
^ super at: anIndex put: aValue
%

category: 'CodeModification Override'
method: SymbolList
_at: anIndex put: aValue

self _validatePrivilegeAdd: aValue .
self _validateElement: aValue .
^ super at: anIndex put: aValue
%

category: 'CodeModification Override'
method: SymbolList
_basicAt: anIndex put: aValue

self _validatePrivilegeAdd: aValue .
self _validateElement: aValue .
^ super _basicAt: anIndex put: aValue
%

category: 'CodeModification Override'
method: SymbolList
squeakBasicAt: anIndex put: aValue

^ self _basicAt: anIndex put: aValue
%

category: 'CodeModification Override'
method: SymbolList
_deleteNoShrinkFrom: startIndex to: stopIndex

self _validatePrivilege: self .
^ super _deleteNoShrinkFrom: startIndex to: stopIndex
%

category: 'CodeModification Override'
method: SymbolList
_insertAt: destOffset 
from: anArray 
fromStart: startOffset
fromEnd: endOffset 
numToMoveDown: numToMove 

self _validatePrivilegeAdd: anArray .
startOffset to: endOffset do:[ :j | 
	self _validateElement:(anArray at: j) .
].
^ super 
	_insertAt: destOffset 
	from: anArray 
	fromStart: startOffset
	fromEnd: endOffset 
	numToMoveDown: numToMove 
%

category: 'CodeModification Override'
method: SymbolList
_primitiveAt: anIndex put: aValue

self _validatePrivilegeAdd: aValue .
self _validateElement: aValue .
^ super _primitiveAt: anIndex put: aValue
%

category: 'CodeModification Override'
method: SymbolList
_unsafeAt: anIndex put: aValue

self _validatePrivilegeAdd: aValue .
self _validateElement: aValue .
^ super _unsafeAt: anIndex put: aValue
%

! fixed 44433, just check for CodeModification 
category: 'Updating'
method: SymbolList
_validatePrivilege

System myUserProfile _validateCodeModificationPrivilege
%

! added _validatePrivilegeAdd: for bug 31284
category: 'Updating'
method: SymbolList
_validatePrivilegeAdd: item

"Check that item satisfies constraints and then check privilege."

(item isKindOf: SymbolDictionary) ifTrue: [ 
  self _validatePrivilege: item .
  ^ self
].
(item isKindOf: Collection) ifTrue: [ 
     item do: [:x | x _validateClass: SymbolDictionary ]. 
] ifFalse:[ 
    item _validateClass: SymbolDictionary .
].
self _validatePrivilege: item 
%

category: 'Updating'
method: SymbolList
_validatePrivilege: item

"Only need to #_validatePrivilege if adding/removing a SymbolDictionary"

(item isKindOf: SymbolDictionary) ifTrue: [
   self _validatePrivilege .
   ^self
].
(item isKindOf: Collection) ifTrue: [
  item do: [:x |
    (x isKindOf: SymbolDictionary) ifTrue: [
      ^ self _validatePrivilege .
    ]
  ].
].
^ true
%

! _gciInitialize added with fix 31942
category: 'Instance Initialization'
method: SymbolList
_gciInitialize

self size: 0 .
^ true
%

! _deferredGciUpdateWith: added with fix 31942
category: 'Private'
method: SymbolList
_deferredGciUpdateWith: valueArray

self size == 0 ifTrue:[
  self addAll: valueArray .    
] ifFalse: [
  " semantics of the deferred update not defined"
  self _error: #errNoStructuralUpdate 
]
%

! fix 41791
category: 'Class Membership'
method:
species

"Returns a class, an instance of which should be used as the result of
 select: or other projections applied to the receiver."

^ Array
%

