!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
!
! Superclass Hierarchy:
!   PrivateObject 
!
!=========================================================================

set class PrivateObject
removeallmethods 
removeallclassmethods 

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'PrivateObject is the superclass of certain classes used in the implementation.
 Instances of subclasses of PrivateObject are normally not visible to Smaltalk,
 and if they are returned by a primitive, many of the methods defined in Object
 are not safe to send to these instances. '

%

category: 'Error Handling'
method:
pause

"Generates an error.  You can use this method to establish
 breakpoints in methods, aside from any debugger breakpoints that
 may be set.  If execution continued , returns receiver. "

 Halt signalNotTrappable 
%

input $upgradeDir/dnuErrorMethods.gs

category: 'Error Handling'
method:
doesNotUnderstand: aSymbol args: anArray envId: envId

"Generates an error reporting that the receiver cannot respond to a message.
 because no compiled method was found for aSymbol in method environment
 envId.   envId is a SmallInteger, 0 for Smalltalk , 1 for Ruby , 
 2..255 for future use by Smalltalk package managers .
"
| ex |
(ex := MessageNotUnderstood _basicNew) 
  receiver: self selector: aSymbol args: anArray envId: envId .
^ex signal .	"fix for #40871"

"Legacy behavior was that you could fall through from an exception handler
without an explicit resume/return. The fix for 40871 will break applications 
that rely on the old behavior."

"If we continue from the error, re-try the send of the message that was
 not understood."

"^ self perform: aSymbol env: envId withArguments: anArray "
%

category: 'Error Handling - Compatibility'
method:
doesNotUnderstand: aMessageDescriptor

"The method is for compatiblity with Gs64 v2.x, and assumes you are using 
   only method environment 0  for all of your Smalltalk code."

| ex sel args |
(ex := MessageNotUnderstood _basicNew)
  receiver: self selector: (sel := aMessageDescriptor at: 1) 
		args: (args := aMessageDescriptor at: 2) envId: 0 .
^ex signal .	"fix for #40871"

"Legacy behavior was that you could fall through from an exception handler
without an explicit resume/return. The fix for 40871 will break applications 
that rely on the old behavior."

"If we continue from the error, re-try the send of the message that was
 not understood."
"^ self perform: sel env: 0 withArguments: args "
%

! _errorSymbolToNumber:args: deleted
! _error... methods in object2.gs

category: 'Error Handling'
method:
_uncontinuableError

"An attempt was made to continue execution past an uncontinuable error."

 self pause 
%

category: 'Accessing'
method:
size

"Instance variables of PrivateObjects are not visible to Smalltalk."
^ 0
%


category: 'Accessing'
method:
_basicSize

"Instance variables of PrivateObjects are not visible to Smalltalk."
^ 0
%

category: 'Accessing'
method:
squeakBasicSize

^ 0
%

category: 'Accessing'
method:
_primitiveSize

"Instance variables of PrivateObjects are not visible to Smalltalk."
^ 0
%

! fix bug 11637 
category: 'Accessing'
method:
basicSize

"Instance variables of PrivateObjects are not visible to Smalltalk."
^ 0
%

category: 'Accessing'
method:
basicPhysicalSize

"Returns the number of bytes required to represent the receiver in memory.  If
 the receiver is in special format (which implies that its representation is the
 same as its OOP), returns zero.

 The basicPhysicalSize method returns the same result as the default
 implementation (in class Object) of the physicalSize method.  It makes that
 default implementation available even when the physicalSize method is
 reimplemented in a subclass.  The basicPhysicalSize method should not itself
 be reimplemented."

<primitive: 364>
self _primitiveFailed: #basicPhysicalSize .
^ 0
%

category: 'Accessing'
method:
physicalSize

"Returns the number of bytes required to represent the receiver in memory.  If
 the receiver is in special format (which implies that its representation is the
 same as its OOP), returns zero.

 This method should be reimplemented for subclasses whose instances are (or may
 be) a composite of component parts which are objects themselves (such as B-tree
 nodes).  Since the composite object cannot be represented independently of its
 components, its physical size should include that of its components.

 However, the component objects of collection (such as an NSC) should not be
 confused with its contents or elements.  Elements or contained objects are in a
 logical relationship with the collection, whereas its components are in a
 physical relationship.  Logically related objects can be represented and stored
 independently."

<primitive: 364>
self _primitiveFailed: #physicalSize .
^ 0
%

category: 'Accessing'
method:
physicalSizeOnDisk

"Returns the number of bytes required to represent the receiver on disk.  If
 the receiver is in special format (which implies that its representation is the
 same as its OOP), returns zero.

 See also Object >> physicalSize"

<primitive: 579>
self _primitiveFailed: #physicalSizeOnDisk .
^ 0
%

category: 'Formatting'
method:
describe

"Returns an instance of a subclass of CharacterCollection describing the
 receiver.  This method is required by Topaz and by GemStone internal error
 handling.  Any reimplementation must conform to the following rules:

 * This method must not return nil.

 * This method must return an instance of a subclass of CharacterCollection.

  Final version of this method is in bomlastconv.gs .
"
^ 'a ' , self class name  
%

method:
describe1K
  "Final version of this method is in bomlastconv.gs ."
  ^ self describe
%

category: 'Clustering'
method:
pageCreationTime

"Returns a DateTime that is the approximate beginning of the life of the page
 containing the receiver.  

 The result represents the time that the receiver was last modified, clustered,
 moved to a new page by the Garbage Collector Gem, regenerated from a
 transaction log or full backup file during recovery or restore, whichever
 happened last.  

 When an object is modified by a session, the resulting pageCreationTime is an
 approximate time of the object creation or modification, and may precede the
 time at which the modification was committed.

 If the receiver is not yet committed and has not yet been assigned to a page,
 returns the current time.

 If the receiver is special, returns the value of the class instance variable
 timeStamp of the receiver's class."

<primitive: 400>
self _primitiveFailed: #pageCreationTime .
^ nil
%

category: 'Tag Management'
method:
tagSize

^ 0
%

category: 'Disk Space Management'
method:
findReferences

"Searches GemStone for objects that reference the receiver, and returns an Array 
 of any such objects.  The search continues until all such objects have been
 found, or until the result contains 20 elements.  (The method
 findReferencesWithLimit: allows you to specify an arbitrarily large limit for
 the result Array.)

 If an object contains multiple references to the receiver, that object occurs
 only once in the result.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData.       

 The result contains both permanent and temporary objects.  The temporary
 objects found may vary from run to run. "
 
^self findReferencesWithLimit: 20
%

method:
findAllReferences

 "Returns an Array containing all references to the receiver.
 If an object contains multiple references to the receiver, that object occurs
 only once in the result.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData.

 The result contains both permanent and temporary objects.  The temporary
 objects found may vary from run to run. "

  ^ self findReferencesWithLimit: 0
%

category: 'Disk Space Management'
method:
findReferencesInMemory

"Returns an Array of objects in current sessions temporary object memory
 that reference the receiver.  If receiver is a committed object,
 the result includes both direct memory references from temporary or
 modified committed objects, and references by objectId from other
 in-memory committed objects.  See also findNonStubbedReferencesInMemory .

 The search continues until all such objects have been found, or until the
 size of the result reaches the specified maximum aSmallInt.

 The result may contain both permanent and temporary objects and may
 vary from run to run.  Does not abort the current transaction."

| argArray tmpArray resultArr |
argArray := { self } .
tmpArray := SystemRepository listReferencesInMemory: argArray .
resultArr := tmpArray at: 1.
resultArr removeIdentical: argArray ifAbsent:[ ].
^ resultArr
%

category: 'Disk Space Management'
method:
findReferencesWithLimit: aSmallInt

"Returns an Array of objects in GemStone that reference the receiver.  
 The search continues until all such objects have been found, or until the 
 size of the result reaches the specified maximum aSmallInt.

 If aSmallInt is <=  0, the result size is unlimited.

 The result contains both permanent and temporary objects.  The temporary
 objects found may vary from run to run.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData.       

 Note that this method may take a considerable length of time to
 execute, and the result may occupy a large amount of disk space.
 (Compare with findReferences, which limits the result to 20 elements.). "

| argArray tmpArray resultArr limit |

System needsCommit
  ifTrue: [ PrivateObject _error: #rtErrAbortWouldLoseData ] .

argArray := { self } .
limit :=  aSmallInt > 0 ifTrue:[ aSmallInt + 1] ifFalse:[ 0 ] .
tmpArray := SystemRepository listReferences: argArray withLimit: limit.

resultArr := tmpArray at: 1.
resultArr removeIdentical: argArray ifAbsent:[ ].
aSmallInt > 0 ifTrue:[
  resultArr size > aSmallInt ifTrue:[ resultArr size: aSmallInt ].
].
^ resultArr 
%

category: 'Class Membership'
method:
class

"Returns the object that is the receiver's class."

<primitive: 610 >
self _primitiveFailed: #class .
^ nil
%

category: 'Converting'
method:
asOop

"Returns the value of the receiver's object-oriented pointer (OOP) as a
 positive Integer .  This is the receiver's unique identifier that 
 distinguishes it from all other objects.  For non-special objects
 and for instances of Character, Boolean, UndefinedObject, and JISCharacter
 the result will always be a SmallInteger.  For instances of SmallInteger
 and SmallDouble the result may be a LargeInteger. 

 The result is different from the result obtained in Gemstone64 v1.x"

<primitive: 86>
self _primitiveFailed: #asOop .
^ nil
%

category: 'Encoded OOPs'
method:
asOopNumber

"If the receiver is a non-special object, return the receiver's 
 oopNumber as defined under OOP_TAG_POM_OOP in the file 
 $GEMSTONE/include/gcioop.ht  , otherwise return nil . "

| oop |
oop := self asOop .
oop _isSmallInteger ifFalse:[ ^ nil ].
(oop bitAnd:7) == 1 ifFalse:[ ^ nil ].
^ oop bitShift: -8
%

category: 'Formatting'
method:
printString

^ 'a ' , self class name  
%
method:
asString

^ 'a ' , self class name  
%
method:
_topazAsString
  ^ [ self asString
    ] onSynchronous: AbstractException do:[:ex |
      '<error during asString>'
    ]
%

category: 'Accessing'
method:
objectSecurityPolicy

"Returns the GsObjectSecurityPolicy where the receiver is located.
 If the result is nil, the object has World write permission (objectSecurityPolicyId 0)."

<primitive: 592>
self _primitiveFailed: #objectSecurityPolicy .
^ nil
%

category: 'Accessing'
method:
objectSecurityPolicyId

"Returns the objectSecurityPolicyId of the GsObjectSecurityPolicy where the receiver is located. 
 If the result is 0, the object has World write permission."
| seg | 
seg := self objectSecurityPolicy .
seg ifNotNil:[ ^ seg objectSecurityPolicyId ].
^ 0
%

category: 'Testing'
method: 
_status: checkClosureBool

"Returns a SmallInteger with status information about the receiver
 encoded in the bit whose masks are defined below:

 16r01 isSpecial
     Set if the receiver is an AbstractCharacter, Boolean,
     SmallInteger, or nil.
 16r02 isCommitted
     Set if the receiver existed in GemStone at the time the current
     transaction began.
     if checkClosureBool is true, and session's most recent commit
     failed and is not yet aborted, check the closureList to determine actual 
     committed state of temporary objects in the closure of the commit.
 16r04 isConnected  (obsolete bit, always 0)
 16r08 isWritten
     Set if the receiver has been written since the last
     commit, abort, or begin transaction command was executed.
 16r10 isWritable
     Set if the receiver canBeWritten by the current user.
 16r20 isTranlogged - true isCommitted and tranlogged, or
     not committed and beTranlogged has been sent .
 16r40 beTranlogged has been sent to the object while in-memory.

 16r700 levels  ( > 0 means a 'large object' implemented as a tree of
                smaller objects which are not visible to Smalltalk)"

<primitive: 39>
self _primitiveFailed: #_status args: { checkClosureBool } .
^ nil
%

category: 'Testing'
method: 
isCommitted

"Returns true if the receiver was a committed object at the time the current
 transaction began.  Returns false otherwise."

^((self _status: true) bitAnd: 2) ~~ 0
%
category: 'Testing'
method: 
isTranlogged

"Returns true if the receiver was a committed object at the time the current
 transaction began and is not a notTranlogged object, or was part of the
 closure of a failed commit and did not become reachabley by reference
 from a not tranlogged object."

^ ((self _status: true) bitAnd: 16r20) ~~ 0
%

category: 'Testing'
method: 
canBeWritten

"Returns true if the current user has write authorization for the
 receiver, false if not."

^ ((self _status: false)  bitAnd: 16r10) ~~ 0
%

category: 'Testing'
method: 
isWritten

"Returns true if the receiver was a committed object at the start of the
 current transaction and as been written by the current transaction.
 Returns false otherwise."

^((self _status: true) bitAnd: 16rA ) == 16rA
%

category: 'Testing'
method:
isSpecial
  ^ false
%

method:
isClass
 
  ^ false
%
method:
isBehavior
  ^ false
%
method:
isMeta
  ^ false
%
method:
isNil
  ^ false
%
method:
isInternalObject
  ^ true
%
method:
notNil
  ^ true
%

category: 'Comparing'
method: 
identityHash

"This method returns some Integer related to the identity of the receiver.  If
 two objects compare identically (==) to each other, the results of sending
identityHash to each of those objects will be equal."

<primitive: 609 >
self _primitiveFailed: #identityHash .
^ nil 
%
category: 'Error Handling'
method: 
_primitiveFailed: aSelector args: primArgs

"Methods which are implemented as primitives send _primitiveFailed:
 when a primitive fails and the failure is not attributable to any
 normal error such as bad argument kind, argument out of range, etc."
 
| args |
args := { self . aSelector } .
args add: primArgs.
^ ArgumentError new _number: 2258"#rtErrPrimFailed" ; args: args ; signal
%

category: 'Error Handling'
method: 
_primitiveFailed: aSelector

"Methods which are implemented as primitives send _primitiveFailed:
 when a primitive fails and the failure is not attributable to any
 normal error such as bad argument kind, argument out of range, etc."
 
^ self _primitiveFailed: aSelector args: { }
%

category: 'Instance Creation'
classmethod:
_basicNew
  "disallowed"
  self shouldNotImplement: #_basicNew
%
classmethod:
basicNew
  "disallowed"
  self shouldNotImplement: #basicNew
%
classmethod:
new
  "disallowed"
  self shouldNotImplement: #new
%

category: 'Reduced Conflict Support'
method:
_resolveRcConflictsWith: conflictObjects

"Return false, replay not possible"
System 
  rcValueCacheAt: #'Rc-Retry-Failure-Reason' 
    put: #'noReplayForPrivateObjects'
    for: System;
  rcValueCacheAt: #'Rc-Retry-Failure-Description' 
    put: 'Replay is not support for private object [', self asOop printString, '].' 
    for: System.
^ false
%
method:
_refreshAfterCommitFailure

 "Returns whether the receiver should be selectively aborted when there is a
 failed attempt to commit"

 ^ false
%
! fix  47796
method: 
_primitiveSelectiveAbort
"Performs an abort operation on the receiver. That is, if the object is
 committed, it removes any changes made by the current transaction and allows
 access to the committed state of the object.

 The error #rtErrSelectiveAbort is thrown if the receiver has depMap entries."

<primitive: 274>

self _primitiveFailed: #_primitiveSelectiveAbort .
self _uncontinuableError
%

method:  
_selectiveAbort

"Performs an abort operation on the receiver. That is, if the object is
 committed, it removes any changes made by the current transaction and allows
 access to the committed state of the object."

^ self _primitiveSelectiveAbort.
%
category: 'Private'
method: PrivateObject
_unsafeAt: anIndex
"Low level access to an object, node of a large object , or node of
 an IdentityBag
 Named instance variables are accessed by this method, so
 unnamed instance variables begin at (self class instSize + 1).

 Returns specifed instVar of the object.  If object is byte format,
 returns SmallInteger value of the specified byte.
 If anIndex is < 1 , the primtive fails.
 If anIndex is beyond the end of the object, returns OOP_REMOTE_NIL"

 <primitive: 596>
 (anIndex _isInteger)
   ifFalse: [ PrivateObject _errorNonIntegerIndex: anIndex .  self _uncontinuableError ].
 self _primitiveFailed: #_unsafeAt: args: { anIndex } .
 self _uncontinuableError
%

