!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Definition of TimeZone (from TimeZone2007.gs)
!
!=========================================================================

set class TimeZone

removeallmethods 
removeallclassmethods 

! ------------------- Class methods for TimeZone
category: 'For Documentation Installation only'
classmethod: TimeZone
installDocumentation

self comment:
'This provides GemStone/S overrides and extensions to the
ANSI TimeZone behavior.'.
%
category: 'cache'
classmethod: TimeZone
for: aPlace

"Returns a TimeZoneInfo object for the specified place if it has been defined
 and stored in the class. Returns nil if it is not defined. 
 E.g. TimeZoneInfo for: #'America/Los_Angeles'."

^self cacheAt: aPlace.
%
category: 'cache'
classmethod: TimeZone
for: aPlace put: aTimeZone

"Stores aTimeZone as the TimeZoneInfo object identified with a particular
 place. A single TimeZoneInfo can be associated with any number of places.
 E.g. TimeZoneInfo for: #'America/Los_Angeles' put: aTimeZone; 
 TimeZoneInfo for: #'Europe/Berlin' put: aTimeZone. 
 Returns aTimeZone."

self cacheAt: aPlace put: aTimeZone.
^aTimeZone.
%
category: 'instance creation'
classmethod: TimeZone
timeDifferenceHrs: hours dstHrs: dstHrs atTimeHrs: startTimeHrs 
fromDayNum: startDay toDayNum: endDay on: nameOfDay beginning: startYear
stdPrintString: stdString dstPrintString: dstString

| oldTZ |
oldTZ := ObsoleteTimeZone
	timeDifferenceHrs: hours 
	dstHrs: dstHrs  
	atTimeHrs: startTimeHrs 
	fromDayNum: startDay  
	toDayNum: endDay  
	on: nameOfDay  
	beginning: startYear
	stdPrintString: stdString  
	dstPrintString: dstString.
^self fromObsolete: oldTZ.
%
category: 'instance creation'
classmethod: TimeZone
timeDifferenceMin: minutes dstMin: dstMins atTimeMin: startTimeMins 
fromDayNum: startDay toDayNum: endDay on: nameOfDay beginning: startYear
stdPrintString: stdString dstPrintString: dstString

| oldTZ |
oldTZ := ObsoleteTimeZone
	timeDifferenceMin: minutes 
	dstMin: dstMins 
	atTimeMin: startTimeMins 
	fromDayNum: startDay 
	toDayNum: endDay 
	on: nameOfDay 
	beginning: startYear
	stdPrintString: stdString 
	dstPrintString: dstString.
^self fromObsolete: oldTZ.
%
category: 'other'
classmethod: TimeZone
migrateNew

"Override default migrateNew behavior with #_basicNew because
we disallow #new (which is called by Behavior>>migrateNew)."

^ self _basicNew
%
category: 'singleton'
classmethod: TimeZone
current

"Returns the current session's current TimeZoneInfo. E.g. TimeZoneInfo current."

^ System __sessionStateAt: 17.
%
category: 'singleton'
classmethod: TimeZone
default

	^default.
%
category: 'singleton'
classmethod: TimeZone
default: aTimeZone

"Makes the specified time zone the default time zone. Returns aTimeZone.
 Must be SystemUser to do so."

aTimeZone _validateClass: TimeZone.
System myUserProfile userId = 'SystemUser' ifFalse:[
  self error:'instance only modifiable by SystemUser'.
  self _uncontinuableError .
].
super default: aTimeZone.
ObsoleteTimeZone default: aTimeZone.  "work-around for #36178"
^aTimeZone.
%
category: 'instance creation'
classmethod: TimeZone
_fromPath: aString onClient: onClient
  | f |
  f := GsFile open: aString mode: 'rb' onClient: onClient .
  f ifNil:[
    Error signal: aString asString,' ' , 'does not specify a TimeZone file; ' , 
      GsFile lastErrorString
  ].
  ^ self fromGsFile: f .
%

classmethod: TimeZone
fromGciPath: pathString
  ^ self _fromPath: pathString onClient: true
%
category: 'instance creation'
classmethod: TimeZone
fromGemPath: pathString
  ^ self _fromPath: pathString onClient: false
%

category: 'instance creation'
classmethod: TimeZone
fromGsFile: aGsFile

	| instance |
	[
		instance := self fromStream: aGsFile.
	] ensure: [
		aGsFile close.
	].
	^instance.
%
category: 'instance creation'
classmethod: TimeZone
fromLinux

	^TimeZone fromGemPath: '/etc/localtime'.
%
category: 'instance creation'
classmethod: TimeZone
fromObsolete: anObsoleteTimeZone
"
TimeZone fromObsolete: ObsoleteTimeZone default.
"
  | res |
  (res := self basicNew) initializeFromObsolete: anObsoleteTimeZone.
  ^ res
%
category: 'instance creation'
classmethod: TimeZone
fromPath: aString

	| block |
	block := [:prefix | 
		| path |
		path := prefix , aString.
		(GsFile existsOnServer: path) == true ifTrue: [
			^self fromGemPath: path.
		].
		(GsFile exists: path) == true ifTrue: [
			^self fromGciPath: path.
		].
	].
	block 
		value: '';	"full path"
		value: '/usr/share/lib/zoneinfo/';	"Solaris"
		value: '/usr/share/zoneinfo/';		"Linux"
		value: '$GEMSTONE/pub/timezone/usr/share/zoneinfo/' .	"GemStone on linux/unix"
	^nil.
%
category: 'instance creation'
classmethod: TimeZone
fromOS

	| osName |
	osName := System gemVersionReport at: #osName.
	osName = 'Linux' ifTrue: [^self fromLinux  ].
	osName = 'SunOS' ifTrue: [^self fromSolaris].
        osName = 'Darwin' ifTrue: [^self fromLinux  ].
	^nil.
%
category: 'instance creation'
classmethod: TimeZone
fromSolaris

	^TimeZone fromGemPath: '/usr/share/lib/zoneinfo/' , (System gemEnvironmentVariable: 'TZ').
%

! fix 43322
category: 'private'
classmethod: TimeZone
_olsonPath
  | path list suffix sufSize |  
  path := (GsFile _expandEnvVariable:'GEMSTONE' isClient: false) , '/pub/timezone/usr/share'.
  " use contentsAndTypesOfDirectory on parent of target directory so as to expand symlinks"
  list := GsFile contentsAndTypesOfDirectory: path onClient: false .
  suffix := '/zoneinfo' . 
  sufSize := suffix size .
  1 to: list size by: 2 do:[:n | | kind |
    kind := list at: n + 1 .
    kind ifFalse:[ "a directory" | elem |
       elem := list at: n .
       (elem at:(elem size - sufSize + 1) equals: suffix) ifTrue:[ ^ elem].
    ]
  ].
  Error signal:'could not find zoneinfo subdirectory'
%

category: 'jrivate'
classmethod: TimeZone
_isTimeZoneFile: aPath

(GsFile openReadOnServer: aPath) ifNotNil:[:f | | str |
  str := String new .
  f read: 4 into: str .
  f close .
  str = 'TZif' ifTrue:[ ^ true ].
].
^ false
%
  

category 'querying'
classmethod: TimeZone
availableZones
 "Returns a sorted Array of Strings which are time zones supported by
  the Olson database shipped in $GEMSTONE/pub/timezone/usr/share/zoneinfo/.
  These Strings may be used as arguments to TimeZone(C)>>named: " 
  | res dirs ofs basePath basePathSize |
  res := { } .
  dirs := { (basePath := self _olsonPath , $/)  } .
  basePathSize := basePath size .
  ofs := 1 . 
  [ ofs <= dirs size ] whileTrue:[ | list aDir |
    aDir := dirs at: ofs .
    list := GsFile contentsAndTypesOfDirectory: aDir onClient: false .
    1 to: list size by: 2 do:[:n | | elem isFile elemSiz |
      elem := list at: n .  elemSiz := elem size .
      isFile := list at: n + 1 .
      ((elem at: elemSiz) == $. or:[ elem at: elemSiz - 3 equals: '.tab']) ifFalse:[ 
        isFile ifTrue:[
          (self _isTimeZoneFile: elem) ifTrue:[ | relPath |
            relPath := elem copyFrom: basePathSize + 1 to: elemSiz .
            res add:  relPath 
          ].
        ] ifFalse:[ dirs add: elem , $/ ].
      ].
    ].
    ofs := ofs + 1.
  ].
  ^ Array withAll: (SortedCollection withAll: res)
%

category: 'instance creation'
classmethod: TimeZone
named: aString
  "Return an instance of TimeZone using the specified zone
   from the Olson database shipped in $GEMSTONE/pub/timezone/usr/share/zoneinfo/.
  See TimeZone(C)>>availableZones for legal arguments"

  | f path |
  path := self _olsonPath , $/ , aString .
  f := GsFile openReadOnServer: path .
  f ifNil:[
    Error signal: aString asString,' ' , 'does not specify a TimeZone; ' , 
      GsFile lastErrorString
  ].
  ^ self fromGsFile: f .
%


! ------------------- Instance methods for TimeZone
category: 'Updating'
method: TimeZone
become: anObject

	super become: anObject.
	self initializeCache.
	anObject initializeCache.
%

category: 'internal'
method: TimeZone
initialize: aStream

	| transition |
	super initialize: aStream.
	transition := self detectLastTransition: [:each | each isDST].
	dstPrintString := transition == nil
		ifTrue:  ['']
		ifFalse: [transition abbreviation].
	transition := self detectLastTransition: [:each | each isDST not].
	standardPrintString := transition == nil
		ifTrue:  ['']
		ifFalse: [transition abbreviation].
	self initializeCache.
%
category: 'internal'
method: TimeZone
initializeCache

	dstStartTimeList := IntegerKeyValueDictionary new.
	dstEndTimeList := IntegerKeyValueDictionary new.
	self 
		populateCacheFor: (1950 to: 2050);
		"_yearStartDst;
		_secondsForDst;
		_secondsFromGmt;"
		yourself.
%
category: 'internal'
method: TimeZone
initializeFromObsolete: anObsoleteTimeZone

	| base startArray endArray old new |
	base := (DateTime newWithYear: 1970 dayOfYear: 1 seconds: 0) asSeconds.
	transitions := OrderedCollection new.
	standardPrintString := anObsoleteTimeZone standardPrintString.
	dstPrintString := anObsoleteTimeZone dstPrintString.
	startArray := { {
		anObsoleteTimeZone secondsFromGmt + anObsoleteTimeZone secondsForDst .
		true .
		dstPrintString } } .
	endArray := { {
		anObsoleteTimeZone secondsFromGmt .
		false .
		standardPrintString } } .
	anObsoleteTimeZone secondsForDst = 0 ifTrue: [
		| year trTim transition |
		year := (anObsoleteTimeZone yearStartDst max: 1900) printString.
		trTim := (DateTime fromStringGmt: '01/01/' ,  year , ' 00:00:00' ) asSecondsGmt - base.
		(transition := TimeZoneTransition new)
			localTimeTypeID: 1;
			transitionTime: trTim ;
			typeList: endArray.
		transitions add: transition.
		self initializeCache.
		^self.
	].
	anObsoleteTimeZone yearStartDst to: 2030 do: [:year | 
		| endDateTime startSec endSec start end |
		startSec := (anObsoleteTimeZone startOfDstFor: year) asSecondsGmt - base.
		(start := TimeZoneTransition new)
			localTimeTypeID: 1;
			transitionTime: startSec;
			typeList: startArray.
		(endDateTime := anObsoleteTimeZone endOfDstFor: year) ifNil: [
			transitions add: start.
		] ifNotNil: [
			endSec := endDateTime asSecondsGmt - base.
			(end := TimeZoneTransition new)
				localTimeTypeID: 1;
				transitionTime: endSec;
				typeList: endArray.
			start transitionTimeUTC < end transitionTimeUTC ifTrue: [
				transitions add: start; add: end.
			] ifFalse: [
				transitions add: end; add: start.
			].
		].
	].
	transitions := transitions asArray.
	self initializeCache.

	old := anObsoleteTimeZone.
	new := self.
	anObsoleteTimeZone yearStartDst to: 2030 do: [:year | 
		| oldStart oldEnd newStart newEnd |
		oldStart := old startOfDstFor: year.
		newStart := new startOfDstFor: year.
		oldEnd   := old endOfDstFor:   year.
		newEnd   := new endOfDstFor:   year.
		oldStart = newStart ifFalse: [self error: 'start date calculation error',
         oldStart asString , ', ' , newStart asString ].
		oldEnd   = newEnd   ifFalse: [self error: 'end date calculation error ',
		   oldEnd asString , ', ' , newEnd asString ].
	].
%
category: 'legacy protocol'
method: TimeZone
dateTimeClass

"Returns the class of DateTime objects that are to be created by the
 various methods in this class."

^ DateTime
%
category: 'legacy protocol'
method: TimeZone
dstPrintString

	^dstPrintString.
%
category: 'legacy protocol'
method: TimeZone
dstPrintString: aString

"Sets the dstPrintString instance variable. Returns the receiver."

dstPrintString := aString.
^ self
%
category: 'queries'
method: TimeZone
endOfDstFor: aYear

	^ (dstEndTimeList at: aYear otherwise: nil) 
		ifNil:[ self endOfDstForA: aYear].
%
! fixed 43322
category: 'internal'
method: TimeZone
endOfDstForA: aYear

	| dt transition next |
	dt := DateAndTime year: aYear + 1 day: 1 hour: 0 minute: 0 second: 0.
	[
		(next := self transitionAtUTC: dt) == nil ifTrue: [^nil].
		next = transition ifTrue: [^nil].
		transition := next.
		dt := transition asDateAndTimeUTC.
		dt year ~~ aYear ifTrue: [^nil].
		dt := dt - (Duration seconds: 1).
		transition isDST.
	] whileTrue: [].
	dt := DateTime
		newGmtWithYear: dt year 
		month: dt month
		day: dt dayOfMonth
		hours: dt hour
		minutes: dt minute
		seconds: dt second
		timeZone: self.
	^dt addSeconds: 1.
%
category: 'internal'
method: TimeZone
populateCacheFor: anInterval

	anInterval do: [:year | 
		dstStartTimeList 
			at: year 
			put: (self startOfDstForA: year).
		dstEndTimeList 
			at: year 
			put: (self endOfDstForA: year).
	].
%
category: 'legacy protocol'
method: TimeZone
shouldWriteInstVar: instVarName
 
"Returns whether the given instance variable should be written out."

"exclude the ditionaries"

instVarName == #dstStartTimeList ifTrue:[ ^ false ].
instVarName == #dstEndTimeList ifTrue:[ ^ false ].
^ true
%
category: 'legacy protocol'
method: TimeZone
standardPrintString

	standardPrintString == nil ifTrue: [
		standardPrintString := (self detectLastTransition: [:each | each isDST not]) abbreviation.
	].
	^standardPrintString.
%
category: 'legacy protocol'
method: TimeZone
standardPrintString: aString

"Sets the standardPrintString instance variable. Returns the receiver."

standardPrintString := aString.
^ self
%
category: 'queries'
method: TimeZone
startOfDstFor: aYear

	^ ( dstStartTimeList at: aYear otherwise: nil) 
		ifNil:[ self startOfDstForA: aYear ].
%
category: 'internal'
method: TimeZone
startOfDstForA: aYear

	| dt transition next |
	dt := DateAndTime year: aYear + 1 day: 1 hour: 0 minute: 0 second: 0.
	[
		(next := self transitionAtUTC: dt) == nil ifTrue: [^nil].
		next = transition ifTrue: [^nil].
		transition := next.
		dt := transition asDateAndTimeUTC.
		dt year ~~ aYear ifTrue: [^nil].
		dt := dt - (Duration seconds: 1).
		transition isDST not.
	] whileTrue: [
	].
	dt := DateTime
		newGmtWithYear: dt year 
		month: dt month 
		day: dt dayOfMonth 
		hours: dt hour
		minutes: dt minute 
		seconds: dt second
		timeZone: self.
	^dt addSeconds: 1.
%
category: 'obsolete accessors'
method: TimeZone
dayEndDst

	| transition dt |
	transition := self detectLastTransition: [:each | each isDST not].
	dt := transition asDateAndTimeUTC asLocal.
	^dt dayOfYear.
%
category: 'obsolete accessors'
method: TimeZone
dayStartDst

	| transition dt |
	transition := self detectLastTransition: [:each | each isDST].
	dt := transition asDateAndTimeUTC asLocal.
	dt := dt + (Duration seconds: self secondsForDst negated).
	^dt dayOfYear.
%
category: 'obsolete accessors'
method: TimeZone
dstEndTimeList

"Returns the dstEndTimeList instance variable." 

^ dstEndTimeList 
%
category: 'obsolete accessors'
method: TimeZone
dstStartTimeList

"Returns the dstStartTimeList instance variable." 

^ dstStartTimeList 
%
category: 'obsolete accessors'
method: TimeZone
secondsForDst

	| isDST isNotDST |
	isDST := self detectLastTransition: [:each | each isDST].
	isDST == nil ifTrue: [^0].
	isNotDST := self detectLastTransition: [:each | each isDST not].
	isNotDST == nil ifTrue: [^0].
	^isDST offsetFromUTC - isNotDST offsetFromUTC.
%
category: 'obsolete accessors'
method: TimeZone
secondsFromGmt

	| transition |
	transition := self detectLastTransition: [:each | each isDST not].
	^transition == nil
		ifTrue: [0]
		ifFalse: [transition offsetFromUTC].
%
category: 'obsolete accessors'
method: TimeZone
timeStartDst

	| transition dt |
	transition := self detectLastTransition: [:each | each isDST].
	dt := transition asDateAndTimeUTC - (Duration seconds: 1).
	dt := DateAndTime 
		secondsUTC: dt asSeconds
		offset: (Duration seconds: (self offsetAtUTC: dt)).
	^dt hour * 60 + dt minute * 60 + dt second + 1.
%
category: 'obsolete accessors'
method: TimeZone
weekDayStartDst

	| transition year dateTime date |
	transition := self detectLastTransition: [:each | each isDST].
	year := transition asDateAndTimeUTC year.
	dateTime := self startOfDstFor: year.
	dateTime := dateTime addSeconds: self secondsForDst negated.
	date := dateTime asDateIn: self.
	^date weekDayName asSymbol.
%
category: 'obsolete accessors'
method: TimeZone
yearStartDst
	"calculate and save in cache"
	
	| transition |
	transition := transitions 
		detect: [:each | each isDST]
		ifNone: [self error: 'DST is not observed in this TimeZone (see bugnote for 36401)!'].
	^transition asDateAndTimeUTC year.
%
category: 'singleton'
method: TimeZone
installAsCurrentTimeZone

"Sets the receiver as the current session's current Timezone. Returns the
 receiver."

System __sessionStateAt: 17 put: self.
^ self.
%
category: 'accessors'
method: TimeZone
transitions

	^transitions.
%
category: 'accessors'
method: TimeZone
leapSeconds

	^leapSeconds.
%
category: 'accessors'
method: TimeZone
= aTimeZone

	^(aTimeZone isKindOf: TimeZone)
		and: [self transitions = aTimeZone transitions
		and: [self leapSeconds = aTimeZone leapSeconds
		and: [self standardPrintString = aTimeZone standardPrintString 
		and: [self dstPrintString = aTimeZone dstPrintString]]]].
%
category: 'accessors'
method: TimeZone
hash

	^transitions first hash + standardPrintString hash.
%
category: 'Printing'
method: TimeZone
asString
  | str |
  str := self standardPrintString .
  (str size == 0 and:[ self secondsFromGmt == 0]) ifTrue:[ str := 'UTC'] .
  ^ str
%
category: 'Instance Migration'
method: TimeZone
migrateFrom: anotherObject instVarMap: otherivi

	super migrateFrom: anotherObject instVarMap: otherivi.
	self 
		_yearStartDst;
		_secondsForDst;
		_secondsFromGmt;
		yourself.
%
