!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   Class, Behavior, Object.
!
!=========================================================================

removeallmethods Class
removeallclassmethods Class 
set class Class

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'Each of the classes in the GemStone kernel inherits some of its behavior from
 Class.

 You may send the messages described here to any of the kernel classes
 (class-defining objects) defined in this manual.  However, you may not send
 these messages to instances of the kernel classes (that is, unless the
 receiver is an instance of Class).

 Consider the following example.  The description of class SmallInteger
 contains two kinds of protocol: instance methods and class methods.  Instance
 methods are understood by SmallIntegers (instances of the class SmallInteger,
 which inherit their protocol from Integer, Number, Magnitude, and Object).
 Class methods are understood by the class-defining object SmallInteger itself
 (which is the single instance of the metaclass "SmallInteger class", and
 inherits its protocol from Class, Behavior, and Object).  The messages
 described here (for Class) are understood by SmallInteger; that is, they are
 class methods for the class-defining object), but are not understood by
 instances of SmallInteger.

Constraints:
	superClass: Behavior
	format: SmallInteger
	instVarsInfo: SmallInteger
	instVarNames: Array
	constraints: Object
	classVars: SymbolDictionary
	methDicts: Object
	poolDictionaries: Array
	categorys: Object
	primaryCopy: Object
	name: Symbol
	classHistory: ClassHistory
	transientMethDicts: Object
	destClass: Module
	timeStamp: DateTime
	userId: CharacterCollection
	extraDict: SymbolDictionary
	classCategory: CharacterCollection
	subclasses: IdentitySet

--- instVar classCategory
A CharacterCollection that names the category of classes to which this class
 belongs.  Each subclass also belongs to this category, unless the subclass
 overrides it with its own category.  Class categorization can be used by
 browsers and schema design tools.

--- instVar classHistory
The ClassHistory to which the class belongs.  Every class belongs to exactly
 one class history, which tracks its ancestry and assists with changes to its
 structure (schema).  When a new class is created, it is considered to be either
 a new version of an existing class, or else it has no previous history.  A new
 class version becomes the most recent version in an existing ClassHistory.
 Otherwise, a new ClassHistory is created for the new class.

--- instVar destClass
A Class, generally considered to be the next later version of this class.  At
 an appropriate time, it may be desirable or necessary to migrate instances of
 this class to the newer version.  This variable remembers which class the
 instance should migrate to.

 You can mark a Class with a migration destination by sending it the message
 migrateTo:.  When so marked, instances of that Class can be migrated to the
 new Class while maintaining identity.  The destination Class should have the
 method migrateFrom: implemented to define the transformation.  A default
 implementation is provided in Object.

 Migration is triggered manually by sending the message migrate to an instance
 of the Class.  Other protocol for forcing migration is
 Class | migrateInstancesTo: and Repository | migrateInstancesOfClasses:.

--- instVar extraDict
Reserved for internal use by GemTalk Systems

--- instVar name
The class''s name for itself; a Symbol of up to 64 Characters.

--- instVar subclasses
An IdentitySet of the subclasses of this class.  This set is only present in
 modifiable classes, and is nil otherwise.

--- instVar timeStamp
A DateTime object that indicates when the class was created.

--- instVar userId
A CharacterCollection that gives the identity of the user that created the class.

--- Category:  Subclass Creation
Every new GemStone Smalltalk class must be a subclass of some other existing
 GemStone Smalltalk class.  To create the new class, you send a subclass
 creation message to its intended superclass.

 The following restrictions apply to creating classes:

 * The new class must be of the same implementation (storage format) as the
   receiver (its superclass), unless the receiver is a non-indexable pointer
   object.  In this case, there are no restrictions if the receiver has no
   instance variables.  If the receiver does have instance variables, the new
   class may not be of special or byte format.
 * The name of a class is a Symbol at most 1024 Characters long.
 * The name of an instance variable is a String at most 64 Characters long.
 * A class contains at most 2030 named instance variables.

 Implementation Format.

 Instance variables may be named or unnamed.  The class definition (often in
 the subclass creation method) explicitly declares the name and number of all
 named instance variables.  This definition must be fixed (class not
 modifiable) before instances of the class can be created.  The class
 definition also implicitly declares unnamed instance variables (if they
 exist), by the choice of implementation format.  Unnamed variables can vary in
 number independently for each instance.  Depending upon format, unnamed
 variables may be indexed (in which case they are accessed by index), or not
 (in which case they are unordered and are accessed associatively, by value).
 Classes in byte format have indexed instance variables that are stored by byte
 for efficiency of storage and access.

 You use different methods to create a byte class, an indexable class, or a
 class of another format.  For each of these possibilities there is a pair of
 standard methods.  Each of these methods provides a full (long) list of
 keywords that permit you to specify a new class fully.  One of them also
 allows you explicitly to specify the new class as a version of an existing
 class, while the other does not.  Additional methods provide selected shorter
 lists of keywords for convenience, and supply default values for some
 arguments.

 Pool dictionaries.

 If you want to add or remove pool dictionaries for the new class at some
 later time, the argument that supplies the Array of pool dictionaries must not
 be an Array literal.  The literal value produces an InvariantArray object,
 which cannot subsequently be modified.

 Dictionary.

 GemStone adds the new class to a dictionary.  The dictionary is typically
 already in the current user''s symbol list, but it can be added to the symbol
 list at a later time if it is not already there.  (The symbol list makes the
 class visible to the user.)  The specified dictionary is often UserGlobals,
 but may be Globals if the data curator has authorized the user to modify that
 dictionary.

 Constraints.

 Constraints are not supported as of GemStone/S 64bit v3.4. They may still be
 specified via the old method keywords, but the settings are ignored.

 Invariance.

 The invarBoolean argument of a subclass creation method deals with class-level
 invariance.  When that argument is true, GemStone thereafter forces all
 instances of the new class to become invariant as soon as they are committed
 to GemStone.  That is, invariance applies to all objects of that class.

 If instances of the new class''s superclass are invariant, then instances of
 the new class must also be invariant.  In this case, a subclass creation
 method generates an error if the invarBoolean argument is not true.

 Class Modification.

 The modifyBoolean argument of a subclass creation method deals with
 object-level invariance, the ability to modify the object that is the class
 itself.

 Classes are typically not modifiable.  As a result, this argument is generally
 given the value false.  The subclass creation method then makes the new class
 an invariant object, and instances of that class can be created at any time
 after.

 When the modifyBoolean argument is true, the new class is modifiable, not
 invariant.  Its instance variables can be modified.  However,
 no instances of it can yet be created.  Once all desired changes have been
 made, you must send the new class the message immediateInvariant.  That
 message then makes the new class an invariant object, and no further changes
 to it are possible.  However, instances of the class can then be created.

 For more information about invariance at all levels, see the GemStone
 Programming Guide.

 Classes and Schema.

 A class can be viewed as an implementation of a schema, or of part of a
 schema.  In order to define and develop a schema, you may create modifiable
 classes, which remain modifiable until the schema is stable.

 However, it is sometimes also necessary to change schema after classes are no
 longer modifiable, and after instances of them exist.  To accomplish this kind
 of change, you must create new classes to implement the new schema.  However,
 it may be desirable to consider a new class to be a new version of an existing
 class, so that a logical connection between them and their instances can be
 maintained.

 Speaking conceptually, a class history lists all the versions of a class.
 Speaking technically, the objects that are classes do not have versions.
 Versions are represented by the list of classes in a class history.  Every
 class (object) belongs to exactly one class history; therefore, all the
 classes that are listed in a class history share the same class history
 object.

 Subclass methods that have an oldClass argument return oldClass if the
 requested new class would be equivalent to oldClass .
 Otherwise the new class is created as a new version of oldClass, 
 and the two classes share the same class history.  
 If the oldClass argument is nil, then no equivalence checks are done and
 thew new class gets a new class history.

 When subclass methods that lack the oldClass argument create a new class with
 the same name an existing class in the specified dictionary, then 
 the existing class is treated as an "oldClass" in the implementation and
 equivalence checks are performed per the above paragraph.

 An oldClass is equivalent to the subclass that would be created using
 the other arguments to a subclass creation method if 
     instVar names match exactly ,
   and class instVar names match exactly ,
   and anArrayOfClassVars contains at least all of the classVars in oldClass
     (additional classVars will be added to oldClass if no subclass is created),
   and pool dictionaries match exactly.

  Many of the subclass creation methods have an options: keyword  which takes
  an Array of Symbols containing zero or more of  
    #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable , 
    #traverseByCallback #selfCanBeSpecial #logCreation
  and at most one of 
     #dbTransient, #instancesNonPersistent, #instancesInvariant .
  If present, #noInheritOptions must be the first element and it causes
  none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant 
  to be inherited from the superclass, nor copied from the
  current version of the class.
  #selfCanBeSpecial is never inherited and is needed only when modifying 
  superclass hierarchy above classes with special format.
  The option #logCreation, if present in the options array, causes logging
  with GsFile(C)>>gciLogSever:  of class creation / equivalence.

  With respect to options and format, oldClass is equivalent if
    The state of format bits dbTransient, and instancesInvariant match exactly ,
    and subclassesDisallowed cannot be set in the new subclass if it not set in oldClass ,
    and modifiable  cannot be set if it is not set in oldClass  ,
    and  (SELF_CAN_BE_SPECIAL, NSC_DUPLICATES, INDEXABLE, IMPLEMENTATION, NO_STRUCT_UPDATE bits)
        of the formats must match exactly.

  If all other equivalence tests pass, the following changes to oldClass may be 
  made to match the arguments and avoid creating a new subclass
    instancesNonPersistent may be set in the format of oldClass
    subclassesDisallowed bit may be cleared in format of oldClass 
    traverseByCallback bit may be set or cleared in format of oldClass
    oldClass may be changed from modifiable to not modifiable (by sending immediateInvariant)
    classVars may be added to oldClass. 
'.
%

category: 'Instance Creation'
method:
new

"Returns an instance of the receiver with no indexed variables."
<primitive: 51>
self _primitiveFailed: #new .
self _uncontinuableError
%

category: 'Instance Creation'
method:
new: anInteger

"Returns an instance of the receiver with the specified number of indexed
 variables.  Generates an error if the receiver is not indexable or if
 anInteger is not a positive SmallInteger.

 For new byte objects, all indexed variables are set to zero;
 for new pointer objects, all indexed variables are set to nil."

<primitive: 53>
(self isIndexable) ifFalse:[ self _errorNotIndexable .  ^ self new ].
(anInteger _isInteger)
  ifFalse:[ anInteger _validateClass: Integer . ^ self new ]
  ifTrue:[
    (anInteger < 0) ifTrue:[ anInteger _error: #rtErrArgNotPositive .
                            ^ self new].
    anInteger _error: #rtErrArgOutOfRange .
    ^ self new
  ].
self _primitiveFailed: #new: args: { anInteger }.
self _uncontinuableError
%

category: 'Accessing'
method:
classHistory

"Returns the classHistory instance variable for this class, which may be nil."

^ classHistory
%

method:
versionedName
| ofs str |
(classHistory atOrNil: classHistory size) == self ifTrue:[ ^ name ].
(ofs := classHistory indexOfIdentical: self) ~~ 0 ifTrue:[
  (str := String new) addAll: name ; add: $[ ; add: ofs asString; add:$] .
  ^ str
].
Error signal: 'oop ' , self asOop asString, ', not found in classHistory(oop ' ,
    classHistory asOop asString , $) .
%

category: 'Browser Methods'
method:
_versionedName
"used by topaz"

^ [ self versionedName ] onException: Error do:[:ex| ^ self describe]
%

category: 'Accessing'
method
versionNumber
 "Returns a SmallInteger"
  | ofs |
  (ofs := classHistory indexOfIdentical: self) ~~ 0 ifTrue:[
     ^ ofs
  ].
  ^ 1
%

method:
classHistoryAt: aSmallInteger

"Returns the specified version of this class."

^ classHistory at: aSmallInteger .
%

category: 'Class Membership'
method:
_classHistoryIncludesIdentical: aClass

 ^ (classHistory indexOfIdentical: aClass) ~~ 0
%

category: 'Private'
method: Class
_description
	"Returns the description of this class.
	Deprecated as of GS/64 3.1: subsumed by the new #comment field."

	self deprecated: 'Class>>_description deprecated v3.1. Replaced by comment'.
	^self _extraDictAt: #description
%

category: 'Accessing'
method:
description

"Returns the description of this class."

self deprecated: 'Class>>description deprecated v3.1. Replaced by comment.'.
^ self _description 
%

category: 'Accessing'
method:
migrationDestination

"Returns the migrationDestination instance variable of this class."

^ destClass
%

! name inherited from Module

category: 'Accessing'
method:
timeStamp

"Returns the timestamp instance variable of this class, a DateTime showing when
 the class was created."

^ timeStamp
%

category: 'Accessing'
method:
userId

"Returns the userId instance variable of this class, the ID of the user who
 created this class."

^ userId
%

! final implementation of _sortedClassVarNames is in class2.gs
!  this implementation is used during filein only
category: 'Private'
method:
_sortedClassVarNames

"Return an unsorted list because the image is still being bootstrapped."
  classVars ifNil:[ ^ { } ].
  ^ classVars keys
%

category: 'Browser Methods'
method: 
_definitionOfConstraints

" Returns a string of the form
     constraints: { <Array of instance-variable-symbol/class-name pairs> }
 or
     constraints: <class name> 

 As of GemStone 64bit v3.4, constraints are no longer implemented.
 This method is provided for examining classes in repositories upgraded from
 an older version."

| result  aConstraint firstElement |
result := String new.
result lf; add: '  constraints: '.
( constraints isKindOf: Array ) ifTrue: [
    result addAll: '{ '.
    firstElement := true.
    1 to: self instSize do: [ :x |
      aConstraint := constraints atOrNil: x .
      ((aConstraint ~~ nil _and: [aConstraint ~~ Object])
          _and:[ superClass == nil
            _or:[ (superClass _namedIvConstraintAt: x) ~~ aConstraint ]] )
      ifTrue: [
        " if not the first constraint, prefix with a period to separate
          from the last constraint "
        firstElement ifFalse: [
          result add: ' . '; lf; add: '                '
        ]
        ifTrue: [
          firstElement := false
        ].
        result add: '{ #'; add: (instVarNames at: x) ;
              add: ' . '; add: aConstraint name; addLast: $} .
      ]
    ].
    aConstraint:= self _varyingConstraint .
    ( (aConstraint ~~ Object) _and:
        [(superClass _varyingConstraint) ~~ aConstraint] )
    ifTrue:[
      firstElement ifFalse: [
          result add: ' . '; lf; add: '                '
      ]
      ifTrue: [
        firstElement := false
      ].
      result add: '   "the elements"  '; add: aConstraint name
    ].
    result add: ' }'.
  ]
  ifFalse: [
    constraints class class == Metaclass3 ifTrue: [
      result add: constraints name.
    ]
    ifFalse: [
      result add: ' nil'
    ].
  ].

^result
%

category: 'Browser Methods'
method: Class
_hasConstraints

"Determine if there are any constraints on any instance variable; return true if any
exist, false otherwise.  
As of GemStone 64bit v3.4, constraints are no longer implemented.
This method is provided for examining classes in repositories upgraded from
an older version."

| aConstraint constrs |
constrs := constraints .
 constrs _isArray  ifTrue: [
    1 to: self instSize do: [:x |
      aConstraint := constrs atOrNil: x .
      ((aConstraint ~~ nil _and: [aConstraint ~~ Object])
          _and: [ superClass == nil
          _or: [ (superClass _namedIvConstraintAt: x) ~~ aConstraint ]] )
                   ifTrue: [^true]
        ].
    aConstraint := self _varyingConstraint .
    ( (aConstraint ~~ Object) _and:
        [(superClass _varyingConstraint) ~~ aConstraint] )
    ifTrue: [^true].
  ] ifFalse: [
    constraints class class == Metaclass3 ifTrue: [^true]
  ].

^false
%

category: 'Browser Methods'
method:
_definitionInContext: aUserProfile 

^ self _definitionInContext: aUserProfile withConstraints: false
%

category: 'Browser Methods'
method:
_definitionInContext: aUserProfile withConstraints: withConstraintsBool

"Returns a description of the receiver using object names taken from the given
 UserProfile."

| result newByteSubclass anArray lfsp
  firstElement poolDicts civs supercls aSize iVs |
supercls := self superClass .
result := String new.
result addAll: (supercls == nil ifTrue: ['nil'] ifFalse: [supercls name]).
newByteSubclass := false.
(lfsp := Character lf asString) addAll: '  ' .

(self isBytes and: [supercls isBytes not]) ifTrue: [
  result addAll: ' byteSubclass: '''; addAll: name; addLast: $'.
  newByteSubclass := true.
] ifFalse: [
  (self isIndexable and:[ supercls == nil or:[ supercls isIndexable not]]) ifTrue: [
    result addAll: ' indexableSubclass: '''; addAll: name; addLast: $'.
  ] ifFalse: [
    result addAll: ' subclass: '''; addAll: name; addLast: $'.
  ]
].
" instVarNames: #( <list of strings> ) "
iVs := self instVarNames .
(newByteSubclass not or:[ iVs size > 0 ]) ifTrue: [
  result addAll: lfsp;
    addAll: 'instVarNames: #(';
    addAll: (self _instVarNamesWithSeparator: (lfsp , '                 '));
    add: $).
].
" classVars: #( <list of strings> ) "
result addAll: lfsp; addLast: 'classVars: #('.
self _sortedClassVarNames accompaniedBy: result do: [:res :aKey |
  res addLast: $  . 
  (aKey includesValue: $') 
    ifTrue:[ res addAll: aKey _asSource ]
    ifFalse:[ res addAll: aKey ].
  ].
result addLast: $).

" classInstVars: #( <list of strings> ) "
result addAll: lfsp; addLast: 'classInstVars: #('.
civs := self class allInstVarNames.
civs removeFrom: 1 to: (self class superClass instSize).
civs accompaniedBy: result do: [:res :civName |
  res addLast: $  .
  (civName includesValue: $') 
    ifTrue:[ res addAll: civName _asSource ]
    ifFalse:[ res addAll: civName ].
].
result addLast: $).

" poolDictionaries: { <list of dictionary names> } "
result addAll: lfsp; addAll: 'poolDictionaries: ' .
(poolDicts := self sharedPools) size > 0 ifTrue:[
  result addAll: '{ ' .
  firstElement := true.
  poolDicts do: [:each | 
    firstElement ifFalse: [ result add: ' . '].  
    anArray := aUserProfile dictionariesAndSymbolsOf: each.
    (aSize := anArray size) == 0 ifTrue:[ result add: '"(not named)"' ] 
      ifFalse:[ aSize = 1 ifTrue:[ result add:((anArray at:1) at: 2) ]
                      ifFalse:[ result add: '"(multiple names)"' ]].
    firstElement := false.
  ].
  result add: ' }' .
] ifFalse:[
  result add: '#()'
].

" inDictionary: <name of containing dictionary> "
result addAll: lfsp; addAll: 'inDictionary: ' ;
   addAll: (self _dictionaryNameForFileout: aUserProfile) .

withConstraintsBool ifTrue:[
 (newByteSubclass not and: [self _hasConstraints]) ifTrue: [
    result add: self _definitionOfConstraints; lf
  ].
].

"options:"
result add:  lfsp; add: self _optionsForDefinition .
result add: Character lf .
^result
%


! fixed 44352
category: 'Browser Methods'
method:
_dictionaryNameForFileout: aUserProfile
  | anArray |
  anArray := aUserProfile dictionariesAndSymbolsOf: self.
  (anArray size) == 0 ifTrue: [ | hist j |
    hist := self classHistory .
    anArray := nil .
    j := hist size .
    [ j >= 1 ] whileTrue:[
      anArray := aUserProfile dictionariesAndSymbolsOf: (hist at: j) .
      anArray size ~~ 0 ifTrue:[ j := 0"exit loop" ] .
      j := j - 1 .
    ] 
  ].
  (anArray size) ~~ 0 ifTrue:[
    anArray := aUserProfile dictionariesAndSymbolsOf: ((anArray at: 1) at: 1).
    anArray size == 0 ifTrue:[
      ^ '(dictionary not in your dictionaries)' 
    ] ifFalse: [ | dName |
      (dName := (anArray at: 1) at: 2) isValidIdentifier ifTrue: [
        ^ dName
      ] ifFalse: [  "this code moved from sessionmethods.topaz to here for v3.0"
        ^ ( '(GsCurrentSession currentSession symbolList objectNamed: ' , dName printString ) 
	       add: $) ; yourself .
      ]
    ]
  ].
  ^ 'UserGlobals' .
%

category: 'Browser Methods'
method:
_optionsForDefinition 
  | result arr |
  result :=  'options: #(' copy .
  arr := self _optionsArray .
  1 to: arr size do:[:j | | sym |
    result add: $  .
    sym := arr at: j .
    (sym at: 1) isDigit ifTrue:[ result add: $#; add: $' ; add: sym ; add: $' ]
                      ifFalse:[ result add: sym ].
  ].
  result add: $)  .
  ^ result
%

method:
_optionsArray
  | result optCount swiz | 
  result := { } .
  optCount := 0 .
  self instancesDbTransient ifTrue:[ result add: #dbTransient . optCount := optCount + 1 ].
  self instancesNonPersistent ifTrue:[ result add:  #instancesNonPersistent  . optCount := optCount + 1 ].
  self instancesInvariant ifTrue:[ result add:  #instancesInvariant  . optCount := optCount + 1 ].
  optCount > 1 ifTrue:[
    self _error: #classErrBadFormat
        with:'only one of #dbTransient #instancesNonPersistent  #instancesInvariant allowed' .
  ].
  self _structuralUpdatesDisallowed ifTrue:[ result add: #disallowGciStore  ].
  self isModifiable ifTrue:[ result add: #modifiable  ].
  self subclassesDisallowed ifTrue:[ result add: #subclassesDisallowed  ].
  self _traversalByCallback ifTrue:[ result add: #traverseByCallback  ].
  self selfCanBeSpecial ifTrue:[ result add: #selfCanBeSpecial  ].
  swiz := (format bitShift: -31"GC_BEHAV_byteSwizKind_shift") bitAnd: 3"GC_BEHAV_byteSwizKind_mask" . 
  swiz > 0 ifTrue:[ | signed |
    signed := (format bitShift: -33"GC_BEHAV_signedWords_shift") bitAnd: 1 .
    signed == 0 ifTrue:[ result add: ( #( #'2byteWords' #'4byteWords' #'8byteWords' ) at: swiz)]
             ifFalse:[ result add: ( #( #'signed2byteWords' #'signed4byteWords' #'signed8byteWords' ) at: swiz) ]
  ].
  ^ result 
%

method
_bytesPerWord
  "Returns a SmallInteger, the number of bytes per word in instances of the receiver.
   Returns 0 for special format, 8 for oop or nsc format, 
   and 1, 2, 4 or 8 for various byte format classes."

  self isBytes ifTrue:[ | swiz |
    swiz := (format bitShift: -31"GC_BEHAV_byteSwizKind_shift") bitAnd: 3"GC_BEHAV_byteSwizKind_mask".
    ^ 1 bitShift: swiz .
  ].
  self isSpecial ifTrue:[ ^ 0 ].
  ^ 8 
%

method
_signedWords

  "Return true if instances of the receiver are byte format with signed words,
   false otherwise."

  self isBytes ifTrue:[ 
     ^ ((format bitShift: -33"GC_BEHAV_signedWords_shift") bitAnd: 1) == 1
  ].
  ^ false .
%


category: 'Browser Methods'
method:
_nonInheritedOptions
  "Returns an Array like that from Class>>_optionsArray but without any 
   options that would be automatically inherited on a subclass creation."
  | set |
  set := (IdentitySet withAll: self _optionsArray) 
         - (IdentitySet withAll: self superClass _optionsArray) .
  self isModifiable ifTrue:[ set add: #modifiable ].
  ^ set asArray 
%

category: 'Browser Methods'
method:
_modifiableDefinitionInDictionary: dict named: dictName

"Returns a description of the receiver that is modifiable, 
 and that places the class in the given dictionary name.
 Byte classes will not use the modifiable form of class definition as
 they have no information that can be modified."

  | result newByteSubclass lfsp firstElement poolDicts cat nm resolver supercls |

  supercls := self superClass .
  nm := supercls == nil ifTrue: [ 'nil' ] ifFalse: [
      dict keyAtValue: supercls ifAbsent: [ supercls name ]
      ].
  (result := String new) add: $( ; addAll: nm .
  newByteSubclass := false.
  lfsp := (Character lf asString) addAll: '  '.

  (self isBytes and: [supercls isBytes not]) ifTrue:
    [
    result addAll: ' byteSubclass: '.
    newByteSubclass := true.
    ]
  ifFalse:
    [
    (self isIndexable and: [supercls isIndexable not]) ifTrue:
      [ result addAll: ' indexableSubclass: ' ]
    ifFalse:
      [ result addAll: ' subclass: ' ].
    ].

  nm := dict keyAtValue: self ifAbsent: [ self name ].
  result addAll: (String withAll: nm) quoted.

 " instVarNames: #( <list of strings> ) "
  newByteSubclass ifFalse: [
    result addAll: lfsp;
      addAll: 'instVarNames: #(';
      addAll: (self _instVarNamesWithSeparator: (lfsp , '                 '));
      add: $).
  ].

  " classVars: #( <list of strings> ) "

  result addAll: lfsp; addLast: 'classVars: #('.
  self _sortedClassVarNames accompaniedBy: result do: [:res :aKey |
    res addLast: $  . 
    (aKey includesValue: $') 
      ifTrue:[ res addAll: aKey _asSource ]
      ifFalse:[ res addAll: aKey ].
    ].

  result addLast: $).

  " classInstVars: #( <list of strings> ) "

  result addAll: lfsp; addLast: 'classInstVars: #('.
  self _classInstVars accompaniedBy: result do:[ :res :civName |
    res addLast: $  .
    (civName includesValue: $') 
      ifTrue:[ res addAll: civName _asSource ]
      ifFalse:[ res addAll: civName ].
    ].
  result addLast: $).

  " poolDictionaries: { <list of dictionary names> } "

  result addAll: lfsp; addAll: 'poolDictionaries: '.
  (poolDicts := self sharedPools ) size > 0 ifTrue:[
    firstElement := true.
    resolver := System myUserProfile .
    result add: ' {' .
    poolDicts do: [:each | | anArray aSize |
      firstElement ifFalse: [ result add: ' . '].  
      anArray := resolver dictionariesAndSymbolsOf: each.
      (aSize := anArray size) == 0 ifTrue:[ result add: '"(not named)"' ] 
        ifFalse:[ aSize = 1 ifTrue:[ result add:((anArray at:1) at: 2) ]
                      ifFalse:[ result add: '"(multiple names)"' ]].
      firstElement := false.
    ].
    result add: ' }' .
  ] ifFalse:[
    result add: '#()'
  ].

  " inDictionary: <name of containing dictionary> "
  result addAll: lfsp; addAll: 'inDictionary: '; addAll: dictName.

  "constraints omitted"

  "options:"
  result add:  lfsp; add: self _optionsForDefinition .

  result add: $) .
  (cat := classCategory ) ifNotNil:[
    result addAll: ' category: '; addAll: (String withAll: cat) quoted
  ].
  result addLast: Character lf .
  ^result
%

category: 'Category'
method:
_classCategory 

"Returns the classCategory of the receiver."

^ classCategory
%

category: 'Category'
method:
category: newCategory

"Sets the classCategory variable of the receiver.
 The argument should be a kind of CharacterCollection or nil."

newCategory ifNil:[
	classCategory := nil.
	^ self ].

(newCategory _validateClass: CharacterCollection ) ifFalse:[ ^ nil ].

classCategory := newCategory asString 
%

category: 'Modifying Classes'
method:
_subclasses

"Returns the class variable that is the list of subclasses, or
 nil if the receiver does not keep track of subclasses."

^ subclasses
%

category: 'Modifying Classes'
method:
_subclasses: anIdentitySet 

"Modifies the class variable that is the list of subclasses."

self _validatePrivilege ifTrue:[
  anIdentitySet ifNotNil:[ (anIdentitySet _validateClass: IdentitySet) ifFalse:[ ^ nil ]].
  subclasses := anIdentitySet
].
%

! changeNameTo: inherited from Module

! compileMissingAccessingMethods inherited from Behavior

category: 'Browser Methods'
method: Class
definition
	"Returns a String containing a GemStone Smalltalk definition for the receiver
 (that is, a subclass creation message).  This method uses the UserProfile
 of the owner of the current session as the correct context."
	"For use with the Topaz run command."

	^self _definitionInContext: System myUserProfile
%

category: 'Private'
method: Class
_definition
	"Returns a String. Sent by topaz "

	^self _definitionInContext: System myUserProfile
%

category: 'Private'
method: Class
__definition
	"Returns a String. 
	For insteractive use to show constraints in a class 
  in an upgraded repository."

	^self _definitionInContext: System myUserProfile withConstraints: true 
%

method:
_topazFileoutDefn
  "used by topaz fileout command"
  | str nm cat comm |
  nm := self name .
  (str := 'run' copy) lf ; 
       add: self _definition .
   str last ~~ Character lf ifTrue:[ str lf ].
   str  add: $% ; lf .
  cat := self category .
  comm := self commentForFileout .
  cat ifNotNil:[
    str add: 'run'; lf ;
       add: nm ; add: ' category: '; add: self category quoted ; lf ;
       add: $% ; lf 
  ].
  comm ifNotNil:[
    str add: 'run'; lf ;
       add: nm ; add: ' comment: ' ; add: self comment quoted ; lf ;
    add: $% ; lf .
  ].
  ^ str
%

category: 'Browser Methods'
method:
recompileWithDicts: symbolList

"Recompiles all the receiver's instance and class methods for envId 0.  
 Returns the CompiledMethods that fail to compile properly."
| failed |
failed := { }.
self _validatePrivilege ifTrue:[
  | cls envId |
  cls := self .
  envId := 0 .
  2 timesRepeat: [
    cls env: envId unifiedCategoriesDo:[ :categName :selectorList |
      selectorList copy do: [ :aSel| | oldMeth |
        [ oldMeth := cls compiledMethodAt: aSel environmentId: envId .
          cls compileMethod: oldMeth sourceString dictionaries: symbolList 
		          category: categName environmentId: envId
        ] onException: CompileError do:[:ex |
          failed add: oldMeth
        ].
      ].
    ].
    cls := self class .
  ].
].
^ failed.
%

category: 'Class Instance Variables'
method:
addClassInstanceVariable: civNameString

"Adds the given class instance variable to the receiver's metaclass.
 Generates an error if the receiver is either not modifiable or does
 not disallow subclasses."

self _validatePrivilege ifTrue:[
  self class addInstVarNames: { civNameString }
]
%

category: 'Class Instance Variables'
method:
atClassInstVar: varName

"Returns the value of the given class instance variable in the receiver.
 Generates an error if the argument does not name a class instance variable
 in the receiver.  In general, it is more efficient to implement a direct
 accessing method for a class instance variable."

| idx cls varSym |
varSym := Symbol _existingWithAll: varName .
varSym == nil ifTrue:[ 
  self _error: #classErrNotAVar args: { varName } .
  ^ nil
  ].
cls := self class.
idx := cls.instVarNames indexOf: varSym.
(idx < 0 or: [idx <= cls class instSize ]) ifTrue: [
  self _error: #classErrNotAVar args: { varName } .
  ^ nil
].
^ self instVarAt: idx
%

category: 'Class Instance Variables'
method: Class
atClassInstVar: varName put: newValue
	"Changes the value of the given class instance variable in the receiver,
 without regard to the variance or invariance of the receiver.  Generates an
 error if the argument does not name a class instance variable in the receiver.
 Returns the argument 'newValue'."

	self _validatePrivilege
		ifTrue: 
			[| idx cls varSym |
			varSym := Symbol _existingWithAll: varName.
			varSym == nil
				ifTrue: 
					[self _error: #classErrNotAVar args: {varName}.
					^nil].
			cls := self class.
			idx := cls.instVarNames indexOf: varSym.
			(idx < 0 or: [idx <= cls class instSize])
				ifTrue: 
					[self _error: #classErrNotAVar args: {varName}.
					^nil].
			self _unsafeAt: idx put: newValue.
			^newValue].
	^nil
%

! fix  47430
category: 'Class Instance Variables'
method: Class
instVarAt: anIndex put: aValue
  | sz |
  self _validatePrivilege ifTrue:[
    anIndex <= Class instSize ifTrue:[
      ImproperOperation new object: self ; 
         signal: 'instVarAt:put: not allowed to a class instance variable defined for Class'
    ].
    anIndex > (sz := self class instSize) ifTrue:[
       OutOfRange new name:#anIndex max: sz actual: anIndex ; signal .
    ].
    self _unsafeAt: anIndex put: aValue .
    ^ aValue
  ].
  ^ nil
%

category: 'Clustering'
method:
clusterBehavior

"Clusters elements of the receiver and its metaclass that are used
 for GemStone Smalltalk execution.

 It is recommended that when several classes are being clustered in a
 transaction, send clusterBehavior to all classes to be clustered, then send
 clusterDescription."

super clusterBehavior ifTrue:[ ^ true ].
self class clusterBehavior .
^ false
%

category: 'Clustering'
method:
clusterDescription

"Clusters elements of the receiver and its metaclass that are not required
 for GemStone Smalltalk execution.  

 It is recommended that when several classes are being clustered in a
 transaction, send clusterBehavior to all classes to be clustered, then send
 clusterDescription."

| result |
super clusterDescription ifTrue:[ ^ true ].
result := false .
extraDict ifNotNil:[ 
  result := extraDict cluster .
  result ifFalse:[ extraDict associationsDo:[:assoc| assoc cluster]].
].
result ifFalse:[ timeStamp ifNotNil:[ result := timeStamp cluster ]].
userId ifNotNil:[ userId cluster ].
result ifFalse:[ result := self class clusterDescription ].
^ result
%


category: 'Displaying'
method:
instanceString

"Returns a symbol that can be used to name an instance of the receiver."

^(('AEIOUYaeiouy' indexOf: (name at: 1) startingAt: 1) ~~ 0
   ifTrue: ['an']
   ifFalse: ['a'])
 , name
%

category: 'Displaying'
method:
instanceSymbol

"Returns a symbol that can be used to name an instance of the receiver."

^ ((('AEIOUYaeiouy' indexOf: (name at: 1) startingAt: 1) ~~ 0
  ifTrue: ['An']
  ifFalse: ['A'])
    , name) asSymbol
%

category: 'Filein Support'
method: Class
_newKernelByteSubclass: clsName 
classVars: classVarArg 
poolDictionaries: poolDicts 
inDictionary: aDict 
options: options 
reservedOop: reservedOopNum

	"Old (pre-GS64) behavior: 
   Created a new class always, to allow upgrade
   scripts to change the definition of a kernel class.  Preserved
   the identity of both the class and the metaClass if the class
   already existed.

 New (Gemstone64) behavior:  
   Does not allow redefinition of a
   class in the upgrade case.  Only create class if it does not exist
   by the given name in globals.  The _unsafeSet11Oop primitive does
   not currently allow changing the identity of an already existing
   object.

 Returns a String, either the definition of the new class or a
 message that the class already exists.

 If reservedOopNum is ~~ nil, reservedOopNum must be a positive Integer
 that is a legal gs64v1.1 objectId and the methods   Object >> _unsafeSet11Oop: 
 and   Object(C) >> _objectForOop:   must be installed."

	self _validatePrivilege
		ifTrue: 
			[| newClass className oldCls resolveRes |
			aDict == Globals
				ifFalse: 
					[aDict _error: #rtErrInvalidArgument.
					^false].
			className := Symbol withAll: clsName.
			resolveRes := Class _resolveReservedClass: reservedOopNum name: className.
			oldCls := resolveRes at: 1.
			oldCls
				ifNil: 
					[GsObjectSecurityPolicy setCurrent: Object objectSecurityPolicy
						while: 
							[newClass := self
										byteSubclass: className
										classVars: classVarArg
										classInstVars: #()
										poolDictionaries: poolDicts
										inDictionary: aDict
										newVersionOf: nil
										description: nil
										options: options.
							reservedOopNum ~~ nil
								ifTrue: [newClass := newClass _unsafeSet11Oop: reservedOopNum].
							(aDict associationAt: className) immediateInvariant]]
				ifNotNil: 
					[| fmt |
					fmt := (format bitAnd: 16r3 bitInvert) bitOr: 16r1 + 16r4.
					(self
						_equivalentSubclass: oldCls
						superCls: self
						name: className
						newOpts: ((options copy)
								add: #logCreation;
								yourself)
						newFormat: fmt
						newInstVars: #()
						newClassInstVars: #()
						newPools: poolDicts
						newClassVars: classVarArg
						inDict: aDict
						isKernel: true)
							ifFalse: [ArgumentError signal: 'existing class not equivalent to arguments']].
			^self _finishNewReservedClass: newClass old: oldCls resolv: resolveRes].
	^false
%

category: 'Filein Support'
method: Class
_newKernelIndexableSubclass: clsName 
instVarNames: ivArg 
classVars: classVarArg 
classInstVars: classInstVarArg 
poolDictionaries: poolDicts 
inDictionary: aDict 
options: options 
reservedOop: reservedOopNum

	"Used by filein.
Old (pre-GS64) behavior: 
   Created a new class always, to allow upgrade
   scripts to change the definition of a kernel class.  Perserved
   the identity of both the class and the metaClass if the class
   already existed.
 New (Gemstone64) behavior:  
   Does not allow redefinition of a
   class in the upgrade case.  Only create class if it does not exist
   by the given name in globals.  The _unsafeSet11Oop primitive does
   not currently allow changing the identity of an already existing
   object.
 Returns a String, either the definition of the new class or a
 message that the class already exists.

 If reservedOopNum is ~~ nil, reservedOopNum must be a positive Integer
 that is a legal gs64v1.1 objectId and the methods   Object >> _unsafeSet11Oop: 
 and   Object(C) >> _objectForOop:   must be installed."

	self _validatePrivilege
		ifTrue: 
			[| newClass className oldCls resolveRes fmt |
			aDict == Globals
				ifFalse: 
					[aDict _error: #rtErrInvalidArgument.
					^false].
			self isBytes
				ifTrue: 
					[^clsName _error: #classErrBadFormat
						with: 'cannot create indexable subclass of byte class'].
			self isNsc
				ifTrue: 
					[^clsName _error: #classErrBadFormat
						with: 'cannot create indexable subclass of Nsc class'].
			fmt := format bitOr: 16r4.	"add indexable bit"
			className := Symbol withAll: clsName.
			resolveRes := Class _resolveReservedClass: reservedOopNum name: className.
			oldCls := resolveRes at: 1.
			oldCls
				ifNil: 
					[aDict at: className put: nil.
					GsObjectSecurityPolicy setCurrent: Object objectSecurityPolicy
						while: 
							[newClass := self
										_subclass: className
										instVarNames: ivArg
										format: fmt
										classVars: classVarArg
										classInstVars: classInstVarArg
										poolDictionaries: poolDicts
										inDictionary: aDict
										inClassHistory: nil
										description: nil
										options: options.
							reservedOopNum ~~ nil
								ifTrue: 
									["change object identifier"
									newClass := newClass _unsafeSet11Oop: reservedOopNum].
							(aDict associationAt: className) immediateInvariant]]
				ifNotNil: 
					[(self
						_equivalentSubclass: oldCls
						superCls: self
						name: className
						newOpts: ((options copy)
								add: #logCreation;
								yourself)
						newFormat: fmt
						newInstVars: ivArg
						newClassInstVars: classInstVarArg
						newPools: poolDicts
						newClassVars: classVarArg
						inDict: aDict
						isKernel: true)
							ifFalse: [ArgumentError signal: 'existing class not equivalent to arguments']].
			^self _finishNewReservedClass: newClass old: oldCls resolv: resolveRes].
	^false
%

category: 'Filein Support'
classmethod:
_resolveReservedClass: reservedOopNum name: className

  " returns an Array of size 2.  
     result at:1 is already existing class or nil .
     result at: 2 is a String describing Globals modifications made"

  | oldCls result |
  result := Array new: 2 .
  result at: 2 put: String new .
  reservedOopNum ~~ nil ifTrue:[ | resOop20 |
    resOop20 := Object _oop11toOop20: reservedOopNum .
    oldCls := Object _objectForOop: resOop20 .
    oldCls ~~ nil ifTrue:[
      "ensure referenced from Globals"
      Globals at: className ifAbsent:[ 
         Globals at: className put: oldCls .
         (Globals associationAt: className) immediateInvariant .
         result at:2 put: (className asString , ' re-added to Globals').
      ].
    ]
  ]
  ifFalse:[
    oldCls := Globals at: className otherwise: nil .
  ].
  result at:1 put: oldCls .
  ^ result
%
category: 'Filein Support'
method:
_finishNewReservedClass: newClass old: oldCls resolv: resolveRes
  oldCls == nil ifTrue:[
    (((newClass class) superClass) == (self class)) ifFalse:[ 
       self _halt: 'Inconsistent class hierarchy' 
    ].
     ^ 'created class: '  , newClass definition 
  ] ifFalse:[ 
     ^ 'class already exists: ' , (resolveRes at: 2)
	    , ', ' , oldCls definition 
  ].
%

category: 'Filein Support'
method: Class
_newKernelSubclass: clsName
instVarNames: ivArg
inDictionary: aDict

^ self
  _newKernelSubclass: clsName
  subclassOf: self 
  instVarNames: ivArg
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: aDict 
  options: #() 
  reservedOop: nil
%

category: 'Filein Support'
method: Class
_newKernelSubclass: clsName 
subclassOf: actualSelf 
instVarNames: ivArg 
classVars: classVarArg 
classInstVars: anArrayOfClassInstVars 
poolDictionaries: poolDicts 
inDictionary: aDict 
options: options 
reservedOop: reservedOopNum

	"Old (pre-GS64) behavior: 
   Created a new class always, to allow upgrade
   scripts to change the definition of a kernel class.  Preserved
   the identity of both the class and the metaClass if the class
   already existed.

 New (Gemstone64) behavior:  
   Does not allow redefinition of a
   class in the upgrade case.  Only create class if it does not exist
   by the given name in globals.  The _unsafeSet11Oop primitive does
   not currently allow changing the identity of an already existing
   object.
 
 Returns a String, either the definition of the new class or a
 message that the class already exists.

 If reservedOopNum is ~~ nil, reservedOopNum must be a positive Integer
 that is a legal gs64v1.1 objectId and the methods   Object >> _unsafeSet11Oop: 
 and   Object(C) >> _objectForOop:   must be installed."

	self _validatePrivilege
		ifTrue: 
			[| newClass className oldCls resolveRes |
			aDict == Globals
				ifFalse: 
					[aDict _error: #rtErrInvalidArgument.
					^false].
			className := Symbol withAll: clsName.
			resolveRes := Class _resolveReservedClass: reservedOopNum name: className.
			oldCls := resolveRes at: 1.
			oldCls
				ifNil: 
					[aDict at: className put: nil.
					GsObjectSecurityPolicy setCurrent: Object objectSecurityPolicy
						while: 
							[newClass := self
										_subclass: className
										instVarNames: ivArg
										format: format
										classVars: classVarArg
										classInstVars: anArrayOfClassInstVars
										poolDictionaries: poolDicts
										inDictionary: aDict
										inClassHistory: nil
										description: nil
										options: options.
							reservedOopNum ifNotNil:
									["change object identifier"
									newClass := newClass _unsafeSet11Oop: reservedOopNum. 
							    (aDict associationAt: className) immediateInvariant] ]]
				ifNotNil: 
					[(self
						_equivalentSubclass: oldCls
						superCls: actualSelf
						name: className
						newOpts: ((options copy)
								add: #logCreation;
								yourself)
						newFormat: format
						newInstVars: ivArg
						newClassInstVars: anArrayOfClassInstVars
						newPools: poolDicts
						newClassVars: classVarArg
						inDict: aDict
						isKernel: true)
							ifFalse: [ArgumentError signal: 'existing class not equivalent to arguments']].
			^self _finishNewReservedClass: newClass old: oldCls resolv: resolveRes].
	^false
%

category: 'Filein Support'
method: Class
_newKernelSubclass: clsName
instVarNames: ivArg
classVars: classVarArg
classInstVars: anArrayOfClassInstVars
poolDictionaries: poolDicts
inDictionary: aDict
options: options 
reservedOop: reservedOopNum

^ self _newKernelSubclass: clsName
    subclassOf: self
    instVarNames: ivArg
    classVars: classVarArg
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: poolDicts
    inDictionary: aDict
    options: options
    reservedOop: reservedOopNum
%

! _redefineKernelSubclass:... not implemented in gemstone64

! fixed 31057
category: 'Instance Migration'
method:
allInstances

"Returns an Array that contains all instances of the receiver.
 Note: This method scans the entire GemStone repository, and may therefore
 take some time to execute.
 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

| list |

list := IdentitySet new.
list addAll: ((SystemRepository listInstancesInMemory: {self}) at: 1).
list addAll: ((SystemRepository listInstances: { self }) at: 1).
^list asArray
%

category: 'Queries'
method:
instancesInMemory
"Returns an Array that contains all instances of the receiver.
 that are in the temporary object memory of this session."
 
^ (SystemRepository listInstancesInMemory: { self }) at: 1
%

category: 'Instance Migration'
method:
cancelMigration

"Disables class migration by clearing the migrationDestination instance
 variable."

self migrationDestination: nil.
%

!  migrateInstances:to:  moved to Module in v3.1
!  migrateInstancesTo: moved to Module in v3.1


category: 'Instance Migration'
method:
migrateTo: aClass

"Enables class migration by setting the migrationDestination instance
 variable."

self _validatePrivilege ifTrue:[
  self migrationDestination: aClass.
].
%

category: 'Locking'
method:
lockableParts

"Returns an Array of the receiver's contents that are locked by browsers
 and folders."

| parts ed |
parts := super lockableParts.
(ed := extraDict) ifNotNil:[ 
	parts addLast: ed.
	ed do: [:each | parts addLast: each].
].
^ parts reject: [:each | each == nil ].
%

category: 'Private'
method:
__makeVariant

"Makes the receiver variant."

<primitive: 272>
self _primitiveFailed: #__makeVariant .
self _uncontinuableError
%

category: 'Private'
method:
_beVariantWhile: aBlock

"Executes the given block while the receiver is in a variant (modifiable) form.
 The block should not contain a return or the receiver will be left in a
 variant state.

 Great caution should be exercised in using this method as it allows
 modification of classes in uncontrolled ways that can lead to corrupted class
 objects."

self _validatePrivilege ifTrue:[
  | was |
  was := self isInvariant.
  was ifTrue: [ self __makeVariant; _refreshClassCache: false  ].
  [
    [  
      aBlock value 
    ] onException: Error do:[:ex |
      was ifTrue: [ 
	super immediateInvariant; _refreshClassCache: false.
	was := false 
      ].
      ex outer .
    ]
  ] ensure:[
    was ifTrue:[ super immediateInvariant; _refreshClassCache: false ].
  ]
].
%

! gemstone64: renamed _civSizeIncreasedAt: to _insertCivAt: and made primitive
category: 'Private'
method:
_insertCivAt: offset

"insert space for a new class instance variable at the specified offset.
 Each call will cause a markSweep GC of the VM local object memory."
 
<primitive: 486>

self _primitiveFailed: #_insertCivAt: args: { offset }.
self _uncontinuableError
%

category: 'Private'
method:
_gbsTraversalCallback

"Private.  When GemBuilder Smalltalk traverses a Class, this method
 is called to return a description of the class."

^self definition
%

! fixed 44684
category: 'Subclass Creation'
method:
_classDefiningClassVar: aSymbol

"Returns the receiver or the superclass that defines a class variable with
 name aSymbol, otherwise returns nil."

| aClass |
aClass := self .
[ 
  aClass _classVars ifNotNil:[ :cvs |
    ( cvs associationAt: aSymbol otherwise: nil ) ifNotNil:[
      ^ aClass 
    ].
  ].
  aClass := aClass superClass .
  aClass == nil
] untilTrue .
^ nil
%

category: 'Subclass Creation'
method:
_makeClassVarDict: anArrayOfClassVars

"Turns Array of class variable names into a SymbolDictionary.
 Returns nil if anArrayOfClassVars is empty or nil. 

 FINAL implementation in class2.gs "

| newDict |
self _validatePrivilege ifTrue:[
  anArrayOfClassVars ifNotNil:[
    anArrayOfClassVars _isArray ifFalse:[
      (anArrayOfClassVars _validateClass: Array) ifFalse:[ ^ nil ].
    ].
    anArrayOfClassVars size ~~ 0 ifTrue:[
      newDict := SymbolDictionary new.
      1 to: anArrayOfClassVars size do: [:index|
        | aVarName aSym definingClass |
        aVarName := anArrayOfClassVars at: index .
        (aVarName _isOneByteString or:[ aVarName _validateClass: CharacterCollection]) ifTrue:[
          aSym := aVarName asSymbol .
          aSym validateIsIdentifier "fix bug 9666" .
          definingClass := self _classDefiningClassVar: aSym .  "fix bug 10480"
          definingClass ifNotNil:[
            LookupError new object: definingClass; key: aSym ; reason: #classErrClassVarNameExists ;
                details: 'class variable already exists'; signal
          ] ifNil:[
            newDict at: aSym put: nil
          ]
        ].
      ].
    ].
  ].
].
^ newDict .
%



category: 'Private'
method:
_classNamed: aString inDictionary: aDictionary
  | v sym |
  aDictionary ifNil:[ ^ nil ].
  (sym := Symbol _existingWithAll: aString) ifNil:[ ^ nil ].
  v := aDictionary at: sym otherwise: nil .
  ^ (v isKindOf: Class) ifTrue:[ v ] ifFalse:[ nil ].
%

category: 'Private'
method: Class
_subclass: className 
instVarNames: anArrayOfInstvarNames 
format: theFormat 
classVars: anArrayOfClassVars 
classInstVars: anArrayOfClassInstVars 
poolDictionaries: anArrayOfPoolDicts 
inDictionary: aDictionary 
inClassHistory: aClassHistory 
description: aDescription 
options: optionsArray

	"The preferred private subclass creation method.
 optionsArray is an Array of Symbols containing zero or more of  
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable , 
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of 
   #dbTransient, #instancesNonPersistent, #instancesInvariant .
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant 
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying 
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.
"

	| cvDict result theName ivNames theHist poolDicts modifiableBool fmtArr fmt nCivs sza szb civNames |
	self _validatePrivilege ifFalse: [^nil].
	className _isOneByteString
		ifFalse: [(className _validateClass: CharacterCollection) ifFalse: [^nil]].
	self subclassesDisallowed
		ifTrue: [^self _error: #classErrSubclassDisallowed].
	anArrayOfClassInstVars
		ifNotNil: 
			[anArrayOfClassInstVars _isArray
				ifFalse: [(anArrayOfClassInstVars _validateClass: Array) ifFalse: [^nil]]].
	aDictionary
		ifNotNil: [(aDictionary _validateClass: SymbolDictionary) ifFalse: [^nil]].
	fmtArr := self _validateOptions: optionsArray withFormat: theFormat newClassName: className . 
	fmt := fmtArr at: 1.
	modifiableBool := fmtArr at: 2.
	(self instancesInvariant and: [(fmt bitAnd: 16r8) == 0])
		ifTrue: [^self _error: #classErrInvariantSuperClass].
	anArrayOfInstvarNames _isArray
		ifFalse: [(anArrayOfInstvarNames _validateClass: Array) ifFalse: [^nil]].
	ivNames := {}.
	1 to: anArrayOfInstvarNames size
		do: [:j | ivNames add: (anArrayOfInstvarNames at: j)].
	nCivs := anArrayOfClassInstVars size.
	civNames := anArrayOfClassInstVars.
	nCivs ~~ 0
		ifTrue: 
			[| aSet |
			civNames := Array new: nCivs.
			aSet := IdentitySet new.
			1 to: nCivs
				do: 
					[:k |
					| aName |
					aName := (anArrayOfClassInstVars at: k) asSymbol.
					self class _validateNewClassInstVar: aName.
					civNames at: k put: aName.
					aSet add: aName.
					aSet size < k
						ifTrue: 
							[ImproperOperation
								signal: 'array of new class instanceVariables contains a duplicate '
										, aName printString]]].

	"Gs64 v3.0 , cvDict and poolDicts maybe nil from caller,
    and will be converted to nil if caller passed an empty Array."
	cvDict := self _makeClassVarDict: anArrayOfClassVars.

	"undo the compiler's canonicalization of empty arrays (fix bug 14103) "
	poolDicts := anArrayOfPoolDicts.
	(poolDicts _isArray and: [poolDicts size == 0]) ifTrue: [poolDicts := nil].
	theName := className asSymbol.
	result := self
				_subclass: theName
				instVarNames: ivNames
				format: fmt
				classVars: cvDict
				poolDictionaries: poolDicts
				classInstanceVars: civNames.
	modifiableBool ifTrue: [result _subclasses: IdentitySet new].
	subclasses ifNotNil: [subclasses add: result].
	aDictionary ifNotNil: [aDictionary at: theName put: result].
	"leave extraDict as nil"
	result _commentOrDescription: aDescription.
	theHist := aClassHistory.
	theHist ifNil: [theHist := ClassHistory new name: className].
	theHist notEmpty ifTrue: [result category: theHist current _classCategory].
	theHist add: result.
	result classHistory: theHist.
	result timeStamp: DateTime now.
	result userId: System myUserProfile userId.
	sza := self class instSize + anArrayOfClassInstVars size.
	szb := result class instSize.
	sza == szb
		ifFalse: 
			[InternalError
				signal: 'prim 233: inconsistent class instance variables, superClass+args=>'
						, sza asString , '  newClass=>'
						, szb asString].
	modifiableBool ifFalse: [result immediateInvariant].
	result copyVariables.
	self _clearCachedOrganizer.
	(fmtArr at: 3) ifTrue: [GsFile gciLogServer: 'created class ' , className].
	^result
%


category: 'Private'
method:
_clearCachedOrganizer
  ^ self "bootstrap implementation , remplemented in class2.gs"
%

category: 'Subclass Creation'
method:
_validName: str

"Returns whether the given string contains Characters valid for a class name.
 The string need not be a Symbol."

| ch |

(str size == 0 or: [str size > 64]) ifTrue: [
  ^false
].

ch := str at: 1.
(ch == $_  or: [ch isLetter]) ifFalse: [
  ^false
].

2 to: str size do: [ :i |
  ch := str at: i.
  (ch == $_  or: [ch isAlphaNumeric]) ifFalse: [
    ^false
  ].
].

^true.
%

category: 'Deprecated'
method:
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
instancesInvariant: invarBoolean

 | opts |
 self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods 
  are in the Subclass Creation category (' , aString , ').'.

 opts := invarBoolean ifTrue:[ { #instancesInvariant } ]
           ifFalse:[ self instancesInvariant ifTrue:[ ^ self _error: #classErrInvariantSuperClass ].
                     #() ] .
^self
  byteSubclass: aString
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: nil 
  options: opts 
%

category: 'Deprecated'
method:
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
description: aDescription
isInvariant: invarBoolean

 | opts |
self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
 opts := invarBoolean ifTrue:[ { #instancesInvariant } ]
           ifFalse:[ self instancesInvariant ifTrue:[ ^ self _error: #classErrInvariantSuperClass ].
                     #() ] .
^self
  byteSubclass: aString
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: aDescription
  options: opts
%

category: 'Deprecated'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
newVersionOf: oldClass
instancesInvariant: invarBoolean

 | opts |
self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
 opts := invarBoolean ifTrue:[ { #instancesInvariant } ]
           ifFalse:[ self instancesInvariant ifTrue:[ ^ self _error: #classErrInvariantSuperClass ].
                     #() ] .
 ^ self byteSubclass: aString
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: oldClass
    description: (oldClass ifNotNil:[ [ oldClass commentForFileout ] on: Error do:[:ex | 'old comment not available']])
    options: opts
%

!1918
category: 'Deprecated'
method: Class
byteSubclass: aString 
	classVars: anArrayOfClassVars 
	classInstVars: anArrayOfClassInstVars 
	poolDictionaries: anArrayOfPoolDicts 
	inDictionary: aDictionary 
	newVersionOf: oldClass 
	description: aDescription 
	isInvariant: invarBoolean

	| hist theFormat opts descr |
	self
		deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category ('
				, aString , ').'.
	aDictionary _validateClass: SymbolDictionary.
	self instSize ~~ 0 ifTrue: [^self _error: #classErrByteObjInstVars].
	self isNsc
		ifTrue: 
			[^aString _error: #classErrBadFormat
				with: 'cannot create byte subclass of Nsc class'].
	theFormat := (format bitAnd: 16r3 bitInvert) bitOr: 16r1 + 16r4.
	opts := invarBoolean
				ifTrue: [{#instancesInvariant}]
				ifFalse: 
					[self instancesInvariant
						ifTrue: [^self _error: #classErrInvariantSuperClass].
					#()].
	descr := aDescription.
	oldClass
		ifNotNil: 
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: opts
				newFormat: theFormat
				newInstVars: #()
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false)
					ifTrue: 
						[oldClass _commentOrDescription: aDescription.
						^oldClass	"avoid creation of a new version"].
			hist := oldClass classHistory.
			descr
				ifNil: 
					[descr := [oldClass commentForFileout] on: Error
								do: [:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: #()
		format: theFormat
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: opts
%

category: 'Deprecated'
method:
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  indexableSubclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: nil
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method:
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  indexableSubclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method:
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
description: aDescription
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  indexableSubclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: aDescription
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method:
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
newVersionOf: oldClass
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self indexableSubclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    constraints: aConstraint
    instancesInvariant: invarBoolean
    newVersionOf: oldClass
    description: nil
    isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
instancesInvariant: invarBoolean
newVersionOf: oldClass
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self indexableSubclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    constraints: #()
    instancesInvariant: invarBoolean
    newVersionOf: oldClass
    description: (oldClass ifNotNil:[ [ oldClass commentForFileout ] on: Error do:[:ex | 'old comment not available']])
    isModifiable: modifyBoolean
%

! 2087
category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: ignored
instancesInvariant: invarBoolean
newVersionOf: oldClass
description: aDescription
isModifiable: modifyBoolean

	| opts fmt hist descr |
	self
		deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category ('
				, aString , ').'.
	"Any specified constraints are ignored."
	self isBytes
		ifTrue: 
			[^aString _error: #classErrBadFormat
				with: 'cannot create indexable subclass of byte class'].
	self isNsc
		ifTrue: 
			[^aString _error: #classErrBadFormat
				with: 'cannot create indexable subclass of Nsc class'].
	opts := {}.
	invarBoolean
		ifTrue: [opts add: #instancesInvariant]
		ifFalse: 
			[self instancesInvariant
				ifTrue: [^self _error: #classErrInvariantSuperClass]].
	modifyBoolean ifTrue: [opts add: #modifiable].
	fmt := format bitOr: 16r4.	"add indexable bit"
	descr := aDescription.
	oldClass
		ifNotNil: 
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: opts
				newFormat: fmt
				newInstVars: anArrayOfInstvarNames
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false)
					ifTrue: 
						["avoid creation of a new version"
						oldClass _commentOrDescription: aDescription.
						^oldClass].
			hist := oldClass classHistory.
			descr
				ifNil: 
					[descr := [oldClass commentForFileout] on: Error
								do: [:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: anArrayOfInstvarNames
		format: fmt
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: opts
%

! a
category: 'Deprecated'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: nil
  isModifiable: modifyBoolean
%

! b
category: 'Deprecated'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  isModifiable: modifyBoolean
%

! c
category: 'Deprecated'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
description: aDescription
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: anArrayOfClassInstVars
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: aDescription
  isModifiable: modifyBoolean
%

! d
category: 'Deprecated'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
newVersionOf: oldClass
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self subclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    constraints: aConstraint
    instancesInvariant: invarBoolean
    newVersionOf: oldClass
    description: nil
    isModifiable: modifyBoolean

%

! e 
category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
instancesInvariant: invarBoolean
newVersionOf: oldClass
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self subclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    constraints: #()
    instancesInvariant: invarBoolean
    newVersionOf: oldClass
    description: (oldClass ifNotNil:[ [ oldClass commentForFileout ] on: Error do:[:ex | 'old comment not available']])
    isModifiable: modifyBoolean
%

! f
! fixed 11833
category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: ignored
instancesInvariant: invarBoolean
newVersionOf: oldClass
description: aDescription
isModifiable: modifyBoolean

	| opts hist descr |
	self
		deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category ('
				, aString , ').'.
	"Any specified constraints are ignored."
	opts := {}.
	invarBoolean
		ifTrue: [opts add: #instancesInvariant]
		ifFalse: 
			[self instancesInvariant
				ifTrue: [^self _error: #classErrInvariantSuperClass]].
	modifyBoolean ifTrue: [opts add: #modifiable].
	descr := aDescription.
	oldClass
		ifNotNil: 
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: opts
				newFormat: self format
				newInstVars: anArrayOfInstvarNames
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false)
					ifTrue: 
						[oldClass _commentOrDescription: aDescription.
						^oldClass	"avoid creation of a new version"].
			hist := oldClass classHistory.
			descr
				ifNil: 
					[descr := [oldClass commentForFileout] on: Error
								do: [:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: anArrayOfInstvarNames
		format: format
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: opts
%

! g
! fixed 11833
category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: ignored
options: optionsArr

	| hist descr oldClass |
	self
		deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category ('
				, aString , ').'.
	"Any specified constraints are ignored."
	oldClass := self _classNamed: aString inDictionary: aDictionary.
	oldClass
		ifNotNil: 
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: optionsArr
				newFormat: self format
				newInstVars: anArrayOfInstvarNames
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false) ifTrue: [^oldClass	"avoid creation of a new version"].
			hist := oldClass classHistory.
			descr := [oldClass commentForFileout] on: Error
						do: [:ex | 'old comment not available']].
	^self
		_subclass: aString
		instVarNames: anArrayOfInstvarNames
		format: format
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: optionsArr
%

! fixed 32132
category: 'Updating'
method:
classHistory: aClassHistory

"Set the class history of the receiver. Returns the receiver."

self _validatePrivilege ifTrue:[
  aClassHistory ifNotNil:[ 
    (aClassHistory _validateInstanceOf: ClassHistory) ifFalse:[ ^ nil ].
  ].
  classHistory := aClassHistory .
  self _refreshClassCache: false .
].
%


category: 'Updating'
method:
addNewVersion: aClass

"Make aClass a new version of the receiver.  That is, add aClass to the
 receiver's history, and set aClass's history to be the same as the
 receiver's history.  The existing history of aClass will have aClass
 removed from it."

self _validatePrivilege ifTrue:[
  aClass classHistory removeVersion: aClass.
  aClass classHistory: classHistory.
  classHistory newVersion: aClass
].
%

! fix 42240: add some helper methods for handling extraDict

category: 'Private'
method:
_extraDictAt: key

  "Return value for key in extraDict.
   Return nil if extraDict or the key are not present. "

  ^ extraDict 
      ifNil: [ nil ] 
      ifNotNil: [ :ed | ed at: key otherwise: nil ]
%

category: 'Private'
method:
_extraDictAt: key put: value

  "Add value for key to extraDict.  Create extraDict if not present. "

  extraDict ifNil: [
    extraDict := SymbolDictionary new.
    extraDict objectSecurityPolicy: self objectSecurityPolicy ].

  ^ extraDict at: key put: value
%

category: 'Private'
method:
_extraDictRemoveKey: key

  " Remove key/value from extraDict.  
    Dont care if extraDict or the key itself are not present. "

  extraDict ifNotNil:[ :ed | ed removeKey: key ifAbsent: [] ].
%

! end 42240 extraDict helper methods


! 42240: new "official" #comment and #comment: methods

category: 'Documentation'
method: Class
comment

" As of GS/64 3.1, comments are now recorded in the class extraDict
  dictionary under the key #comment.  Comment information formerly
  recorded as a GsClassDocumentation under the key #description are
  converted to a string and placed under #comment during DB 
  conversion/upgrade. "
  
  | str |
  (self _extraDictAt: #comment) ifNotNil:[:cmt | ^ cmt ].
  str := 'No class-specific documentation for ' , self name .
  str add: ', hierarchy is: 
'; add: (self hierarchy: 0) .
  ^ str
%

category: 'Documentation'
! fixed 45394
method: Class
commentForFileout

"Returns a non-empty class comment or nil."

| str |
str := self _extraDictAt: #comment .
str size = 0 ifTrue:[ ^ nil ].
^ str
%

category: 'Documentation'
method: Class
comment: aString

  (aString isKindOf: CharacterCollection) ifFalse: [
    ArgumentTypeError signal: 'Comment must be a String' ].
  self _extraDictAt: #comment put: aString
%

! 42240: description methods
category: 'Private'
method: Class
_description: aDescription
	"Update the description of this Class.  Returns the argument.
 As of GS/64 3.1, this is deprecated in favor of #comment:"

	self deprecated: 'Deprecated as of GS/64 3.1'.
	self _extraDictAt: #description put: aDescription.
	^aDescription
%

category: 'Private'
method: Class
_commentOrDescription: newComment 
	"For backward compatibility, accept a GsClassDocumentation (although this is deprecated)
	or a string. Copy any GsClassDocumentation to avoid bug 41763."
newComment
	ifNotNil: 
		[(newComment isKindOf: GsClassDocumentation)
			ifTrue: 
				[| newDesc |
				newDesc := newComment copy.
				newDesc itsClass: self.
				self _description: newDesc]
			ifFalse: [self comment: newComment]].
%

category: 'Documentation'
method: Class
description: aDescription

"Update the description of this Class.  Returns the argument."
self deprecated: 'Deprecated as of GS/64 3.1'.
^ self _description: aDescription
%

category: 'Updating'
method:
migrationDestination: aClass

"Update the migrationDestination instance variable.  Returns the argument."

self _validatePrivilege ifTrue:[
  aClass ifNotNil:[ aClass _validateClass: Class ].
  destClass := aClass .
  self _refreshClassCache: false .
].
^ aClass
%

category: 'Updating'
method:
timeStamp: aDateTime

"Set the value of the timeStamp instance variable.
 For use only when creating a class, while the class is not yet invariant."

self _validatePrivilege ifTrue:[
  self isInvariant ifFalse:[
    aDateTime ifNotNil:[ (aDateTime _validateClass: DateTime) ifFalse:[ ^ self]].
    timeStamp := aDateTime
  ] ifTrue:[
    self validateIsVariant .
  ].
].
%

category: 'Updating'
method:
userId: aString

"Set the value of the userId instance variable.
 For use only when creating a class, while the class is not yet invariant."

(self _validatePrivilege) ifTrue:[
  self isInvariant ifFalse:[
    aString ifNotNil:[ (aString _validateClass: CharacterCollection ) ifFalse:[ ^ self]].
    userId := aString
  ] ifTrue:[
    self validateIsVariant .
  ].
].
%

category: 'Updating'
method:
extraDict: aSymbolDictionary

"Set the value of the extraDict instance variable."

self _validatePrivilege ifTrue:[
  aSymbolDictionary ifNotNil:[ 
    (aSymbolDictionary _validateClass: SymbolDictionary) ifFalse:[ ^ self].
  ].
  extraDict := aSymbolDictionary
].
%

category: 'Updating Variables'
method:
addClassVarName: aString

"Add aString to the class variable list for the receiver, if the
 class variable is not already defined."

| aSym definingClass |
self _validatePrivilege ifTrue:[
  aSym := aString asSymbol .
  aSym validateIsIdentifier .
  definingClass := self _classDefiningClassVar: aSym .  "fix bug 10480"
  definingClass ifNotNil:[
    definingClass == self ifTrue:[ 
      "if the receiver already defines the class variable, 
         do nothing and return silently    (fix bug 8094) "
      ^ self  
      ].
    LookupError new object: definingClass; key: aSym ; reason: #classErrClassVarNameExists ;
		details: 'class variable already exists'; signal .
    ^ self
  ].
  self _addClassVar: aSym value: nil .
]
%

category: 'Updating Variables'
method:
addSharedPool: aDictionary

"Add aDictionary to the end of the shared pool list for the receiver. "
| poolDicts |
self _validatePrivilege ifTrue:[
  (aDictionary _validateClass:  SymbolDictionary ) ifTrue:[
     poolDicts := poolDictionaries .
     (poolDicts ~~ nil and:[ poolDicts includesIdentical: aDictionary]) ifTrue:[
        ^ self _error: #classErrPoolDictExists args: { aDictionary }
     ].
     poolDicts ifNil:[ poolDicts := { } . poolDictionaries := poolDicts ]
       ifNotNil:[ poolDicts isInvariant ifTrue:[
                    poolDicts := Array withAll: poolDicts . poolDictionaries := poolDicts
                ]].
     poolDicts add: aDictionary
  ].
].
%

! fixed 41575
category: 'Updating Variables'
method:
removeClassVarName: aString

"Remove the class variable named aString from the receiver.   
 The value of the removed Association is set to nil. 
 Signals an error if there is no such class variable, or if the 
 Association for that class variable is invariant."

self _validatePrivilege ifTrue:[
  (Symbol _existingWithAll: aString) ifNotNil:[ :aSym |
    classVars ifNotNil:[ :cvs | | assoc |
      assoc := cvs associationAt: aSym otherwise: nil .
      assoc ifNotNil:[
         assoc value: nil .
         cvs removeKey: aSym  .
         ^ self
      ].
    ].
  ].
  ^ LookupError new reason: #classErrClassVarNotFound; key: aString ; object: self;
	signal
].
%

category: 'Updating Variables'
method:
removeSharedPool: aDictionary

"Remove aDictionary from the shared pool list for the receiver.  Generates an
 error if aDictionary is not a shared pool for the receiver.

 You may use this method only if, when the receiver was created, the argument
 to poolDictionaries: was an Array rather than a literal Array, which would
 create an InvariantArray.  (See Class >> subclass:.)"

self _validatePrivilege ifTrue:[
  poolDictionaries ifNotNil:[ | idx |
    (idx := poolDictionaries indexOfIdentical: aDictionary) ~~ 0 ifTrue:[
       poolDictionaries removeAtIndex: idx .
       ^ self
    ].
  ].
  ^ LookupError new reason: #classErrPoolDictNotFound;  key: aDictionary  ;
	object: self ; signal .
].
%

category: 'Versions'
method:
isVersionOf: anotherClass

"Returns whether the receiver and the given class share the same class
 history."

| hist |
^ ( hist := classHistory) ~~ nil and:[ anotherClass classHistory == hist ].
%

! currentVersion added to fix 42440
method:
currentVersion

  "return the most recent version of the receiver"
  classHistory ifNotNil:[ :hist | | sz |
    (sz := hist size) ~~ 0 ifTrue:[ ^ hist at: sz ].
  ].
  ^ self
%

category: 'Deprecated'
method:
byteSubclass: aString
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
isInvariant: invarBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self byteSubclass: aString
    classVars: anArrayOfClassVars
    classInstVars: #()
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    instancesInvariant: invarBoolean
%

category: 'Deprecated'
method:
byteSubclass: aString
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
inClassHistory: aClassHistory
description: aDescription
isInvariant: invarBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self byteSubclass: aString
  classVars: anArrayOfClassVars
  classInstVars: #()
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  inClassHistory: aClassHistory
  description: aDescription
  isInvariant: invarBoolean
%

category: 'Deprecated'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
inClassHistory: aClassHistory
description: aDescription
isInvariant: invarBoolean

	| theFormat opts |
	self
		deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category ('
				, aString , ').'.
	(aDictionary _validateClass: SymbolDictionary) ifFalse: [^nil].
	self instSize ~~ 0 ifTrue: [^self _error: #classErrByteObjInstVars].
	self isNsc
		ifTrue: 
			[^aString _error: #classErrBadFormat
				with: 'cannot create byte subclass of Nsc class'].
	theFormat := (format bitAnd: 16r3 bitInvert) bitOr: 16r1 + 16r4.
	opts := invarBoolean
				ifTrue: [{#instancesInvariant}]
				ifFalse: 
					[self instancesInvariant
						ifTrue: [^self _error: #classErrInvariantSuperClass].
					#()].
	^self
		_subclass: aString
		instVarNames: #()
		format: theFormat
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: aClassHistory
		description: aDescription
		options: opts
%

category: 'Deprecated'
method:
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
isInvariant: invarBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self indexableSubclass: aString
     instVarNames: anArrayOfInstvarNames
     classVars: anArrayOfClassVars
     classInstVars: #()
     poolDictionaries: anArrayOfPoolDicts
     inDictionary: aDictionary
     constraints: aConstraint
     instancesInvariant: invarBoolean
     isModifiable: false
%

category: 'Deprecated'
method:
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self indexableSubclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: #()
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    constraints: aConstraint
    instancesInvariant: invarBoolean
    isModifiable: modifyBoolean
%

category: 'Deprecated'
method:
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
inClassHistory: aClassHistory
description: aDescription
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self indexableSubclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: #()
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  inClassHistory: aClassHistory
  description: aDescription
  isModifiable: modifyBoolean
%

category: 'Deprecated'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: ignored
instancesInvariant: invarBoolean
inClassHistory: aClassHistory
description: aDescription
isModifiable: modifyBoolean

| opts fmt |
self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
"Any specified constraints are ignored."
(self isBytes) ifTrue: [ ^ aString _error: #classErrBadFormat with: 'cannot create indexable subclass of byte class'].
(self isNsc) ifTrue: [ ^ aString _error: #classErrBadFormat with: 'cannot create indexable subclass of Nsc class'].
opts := { } .
invarBoolean ifTrue:[ opts add: #instancesInvariant ] 
           ifFalse:[ self instancesInvariant ifTrue:[ ^ self _error: #classErrInvariantSuperClass ]].
modifyBoolean ifTrue:[ opts add: #modifiable ].
fmt := format bitOr: 16r4 "add indexable bit" .

^ self _subclass: aString
        instVarNames: anArrayOfInstvarNames
        format: fmt
        classVars: anArrayOfClassVars
        classInstVars: anArrayOfClassInstVars
        poolDictionaries: anArrayOfPoolDicts
        inDictionary: aDictionary
        inClassHistory: aClassHistory
        description: aDescription
        options: opts 
%

category: 'Subclass Creation'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
inDictionary: aDictionary

"Creates and returns a new subclass of the receiver.

 This method is a shortcut for convenience only.  It might not be retained in
 future GemStone releases.  Use it interactively or pedagogically, but avoid
 it in production code.

 The new class has no class variables, no class instance variables, 
 and no pool dictionaries.
 Instances of the new class are variant, but the new class itself is not
 modifiable.

 Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars:  #() 
  classInstVars:  #() 
  poolDictionaries: { } 
  inDictionary: aDictionary
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: nil 
  options: #()  
%

category: 'Subclass Creation'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
inDictionary: aDictionary
options: optionsArray

"Creates and returns a new subclass of the receiver.

 This method is a shortcut for convenience only.  It might not be retained in
 future GemStone releases.  Use it interactively or pedagogically, but avoid
 it in production code.

 The new class has no class variables, no class instance variables, 
 and no pool dictionaries.
 Instances of the new class are variant, but the new class itself is not
 modifiable.

 optionsArray is an Array of Symbols containing zero or more of  
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable , 
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of 
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant 
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying 
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars:  #() 
  classInstVars:  #() 
  poolDictionaries: { } 
  inDictionary: aDictionary
  newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
  description: nil 
  options: optionsArray
%

category: 'Deprecated'
method:
subclass: aString
inDictionary: aDictionary
constraints: constraintSpec

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self subclass: aString
    instVarNames: #()
    classVars:  #() 
    classInstVars:  #() 
    poolDictionaries: { }
    inDictionary: aDictionary
    constraints: constraintSpec
    instancesInvariant: false
    isModifiable: false
%

category: 'Deprecated'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
inDictionary: aDictionary
constraints: constraintSpec

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars:  #() 
  poolDictionaries: { }
  inDictionary: aDictionary
  constraints: constraintSpec
  instancesInvariant: false
  isModifiable: false
%

category: 'Deprecated'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
inDictionary: aDictionary
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars:  #() 
  classInstVars:  #() 
  poolDictionaries: { } 
  inDictionary: aDictionary
  constraints:  #() 
  instancesInvariant: false
  isModifiable: modifyBoolean
%

! fix bug 8684

category: 'Deprecated'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
classInstVars: anArrayOfClassInstVars
inDictionary: aDictionary
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^ self
    subclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars:  #() 
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: { } 
    inDictionary: aDictionary
    constraints:  #() 
    instancesInvariant: false
    isModifiable: modifyBoolean
%

category: 'Deprecated'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
isInvariant: invarBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
  subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: #()
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  isModifiable: false
%

category: 'Deprecated'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self
   subclass: aString
   instVarNames: anArrayOfInstvarNames
   classVars: anArrayOfClassVars
   classInstVars: #()
   poolDictionaries: anArrayOfPoolDicts
   inDictionary: aDictionary
   constraints: aConstraint
   instancesInvariant: invarBoolean
   isModifiable: modifyBoolean
%

! h
category: 'Deprecated'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: aConstraint
instancesInvariant: invarBoolean
inClassHistory: aClassHistory
description: aDescription
isModifiable: modifyBoolean

self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
^self subclass: aString
  instVarNames: anArrayOfInstvarNames
  classVars: anArrayOfClassVars
  classInstVars: #()
  poolDictionaries: anArrayOfPoolDicts
  inDictionary: aDictionary
  constraints: aConstraint
  instancesInvariant: invarBoolean
  inClassHistory: aClassHistory
  description: aDescription
  isModifiable: modifyBoolean
%

! i
category: 'Deprecated'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
constraints: ignored
instancesInvariant: invarBoolean
inClassHistory: aClassHistory
description: aDescription
isModifiable: modifyBoolean

| opts |
self deprecated: 'Obsolete in GemStone/S 64.  The preferred methods are in the Subclass Creation category (' , aString , ').'.
opts := { } .
invarBoolean ifTrue:[ opts add: #instancesInvariant ]
           ifFalse:[ self instancesInvariant ifTrue:[ ^ self _error: #classErrInvariantSuperClass ]].
modifyBoolean ifTrue:[ opts add: #modifiable ].
^ self _subclass: aString
          instVarNames: anArrayOfInstvarNames
          format: format
          classVars: anArrayOfClassVars
          classInstVars: anArrayOfClassInstVars
          poolDictionaries: anArrayOfPoolDicts
          inDictionary: aDictionary
          inClassHistory: aClassHistory
          description: aDescription
          options: opts 
%

! deleted _decompileMethods* , decompileMethods*
! _hasUncompiledMethods deleted

category: 'Accessing'
method:
extraDict

"Returns the SymbolDictionary held in extraDict that holds miscellaneous
 information about the receiver.  Result may be nil. "

^ extraDict
%
category: 'Accessing'
method:
extraDictForStore

"Returns the SymbolDictionary held in extraDict that holds miscellaneous
 information about the receiver.  Creates the dictionary if needed."

^ extraDict ifNil:[ extraDict := SymbolDictionary new ].
%

category: 'Accessing'
method:
_classInstVars

"Returns an Array of the receiver's class instance variables."

| civs |

civs := self class allInstVarNames.
civs removeFrom: 1 to: (self class superClass instSize).
^civs
%

! deleted :  Class >> convertTo5

category: 'Modifying Classes'
method:
makeInstancesNonPersistent

  "Takes effect immediately and will prevent committing new instances
   of the receiver in the current transaction.

   To change a non-modifiable class from persistent to non-persistent , 
   see  ClassOrganizer >> makeInstancesNonPersistent: . "

  <primitive: 2001>   "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [ self validateIsModifiable .
    self _makeInstancesNonPersistent .
  ] ensure:[
    prot _leaveProtectedMode
  ].
%

category: 'Private'
method:
_makeInstancesNonPersistent

<protected>
self _validatePrivilege ifTrue:[
  format := format bitOr: 16r800 .
  self _refreshClassCache: false .
].
%

category: 'Modifying Classes'
method:
makeInstancesDbTransient

  "Takes effect immediately and will cause any instances that get
   committed to be DbTransient. 

   For a DbTransient object, instVars on disk are always nil.   
   The first transaction which causes the object to be reachable
   from committed state will commit the object with all instVars on
   disk set to nil .  Thereafter any stores to instVars of the committed 
   object in any session do not cause the object to be written 
   by the transaction.   

   Clustering a committed DbTransient object will
   cause the object to be rewritten to disk with all instVars nil
   when the transaction commits.  So clustering is the only way
   that concurrency conflict could occur involving a DbTransient object.

   The format of the receiver must be    non-indexable, pointer 
   otherwise an error is generated.

   You cannot change a non-modifiable class to or from dbTransient ."

  <primitive: 2001>   "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [ 
    self validateIsModifiable .
    self _makeInstancesDbTransient: true .
  ] ensure: [
    prot _leaveProtectedMode
  ].
%

category: 'Modifying Classes'
method:
makeInstancesNotDbTransient

  "Takes effect immediately and cancels any previous makeInstancesDbTransient.

   The receiver must be a modifiable class."

  <primitive: 2001>   "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    self validateIsModifiable .
    self _makeInstancesDbTransient: false .
  ] ensure:[
    prot _leaveProtectedMode .
  ].
%

method:
_makeInstancesDbTransient: aBool

<protected>
self _validatePrivilege ifTrue:[
  aBool ifTrue:[
    (self isPointers and:[ self isIndexable not]) ifFalse:[
      ^ ImproperOperation new details:'Only non-indexable pointer objects may be DbTransient';
           object: self ; signal 
    ].
    format := format bitOr: 16r1000 .
  ] ifFalse:[
    format := format bitAnd: (16r1000 bitInvert) 
  ].
  self _refreshClassCache: false .
].
%

category: 'Modifying Classes'
method:
makeInstancesPersistent

  "Takes effect immediately.  
   To change a non-modifiable class from non-persistent to persistent, 
   see  ClassOrganizer >> makeInstancesPersistent: . "

  <primitive: 2001>   "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [ self isModifiable ifFalse:[ self validateIsModifiable . ^ self ].
    self _makeInstancesPersistent .
  ] ensure:[
    prot _leaveProtectedMode
  ].
%

category: 'Private'
method:
_makeInstancesPersistent

<protected>
self _validatePrivilege ifTrue:[
  self _validateInstancesPersistent ifTrue:[ 
    self superClass instancesNonPersistent ifTrue:[
      ^ ImproperOperation new reason: #rtErrSuperclassIsNP; 
	details: 'superclass is non-persistent';
	object: self ; signal .
    ]
  ]
].
format := (format bitOr: 16r800 ) bitXor: 16r800 .
self _refreshClassCache: false .
%

! final implementation of _validateInstancesPersistent is in class2.gs
!  this implementation is used during filein only
category: 'Private'
method:
_validateInstancesPersistent

"check that it is legal to make instances of the receiver 
 persistent."

^ true
%

category: 'Repository Conversion'
method: 
_makeClassObsolete: aSymbolDictionary
"remove association from <aSymbolDictionary>
 rename receiver
 if assoc is invariant:
   create new association for obsolete class
 if not:
   reuse the original association for obsolete class
"

 | newName assoc |
 assoc := aSymbolDictionary associationAt: self name.
 newName := ('Obsolete', self name asString) asSymbol.
 aSymbolDictionary removeAssociation: assoc.
 self _unsafeAt: 11 put: newName.
 assoc isInvariant
   ifTrue: [ aSymbolDictionary at: self name asSymbol put: self ]
   ifFalse: [
      assoc key: self name asSymbol.
      aSymbolDictionary addAssociation: assoc ].
 self removeAllMethods.
 self class removeAllMethods.
%

category: 'Repository Conversion'
method:
recompileAllMethods

"Recompile all methods for execution in a Gs64 v3.0 or later system."

super recompileAllMethods .
self class recompileAllMethods 
%
category: 'Queries'
method:
allSubclasses

	^ClassOrganizer new allSubclassesOf: self.
%
set compile_env: 0
category: 'Queries'
method: Class
allSuperclasses

	^ClassOrganizer new allSuperclassesOf: self.
%
category: 'Queries'
method:
subclasses

	^ClassOrganizer new subclassesOf: self.
%
category: 'Modifying Classes'
method:
_setClassVars: aDict old: previousDict

  classVars ~~ aDict ifTrue:[ 
    previousDict ~~ classVars ifTrue:[ self error:'invalid store to classVars'].
    classVars := aDict .
    self class _setClassVars: aDict old: previousDict
  ].
%
category: 'Accessing'
method:
thisClass
  ^ self
%
category: 'Method Timestamps'
method: 
methodStampDictName
  ^ #GSMethodStampDict
%
category: 'Pragmas'
method:
pragmaDictName

  ^ #GSMethodPragmaDict
%

! "fixed bug #40559 - copy class (instance) variables to new version"
! additional change for #40822
category: 'Private'
method:
copyVariables
| chSize chist priorVersion priorVars civNames priorCivNames toIgnoreCount |
(chSize := (chist := classHistory) size) = 1 ifTrue:[ 
  ^ self
].
priorVersion := chist at: chSize - 1.
(priorVars := priorVersion _classVars) notNil ifTrue: [ 
  | cvars |
  (cvars := classVars) notNil ifTrue: [
    priorVars associationsDo: [:anAssociation | |aKey |
      (cvars includesKey: (aKey := anAssociation key)) ifTrue: [
        cvars removeKey: aKey .
        cvars addAssociation: anAssociation.
      ].
    ].
  ].
].
priorCivNames := priorVersion class allInstVarNames.
toIgnoreCount := Class allInstVarNames size.
toIgnoreCount < priorCivNames size ifTrue: [
	priorCivNames := priorCivNames 
		copyFrom: toIgnoreCount + 1
		to: priorCivNames size.
	civNames := self class allInstVarNames.
	priorCivNames do: [:each | 
		(civNames includesIdentical: each) ifTrue: [
			self atClassInstVar: each put: (priorVersion atClassInstVar: each).
		].
	].
].
%

category: 'Subclass Creation'
method: Class
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
newVersionOf: oldClass
description: aDescription
options: optionsArray

	"Creates and returns a new byte subclass of the receiver.  You are not
 permitted to modify the new class after it is created.  If the receiver is not
 some kind of String class, then instances of the new class store and return
 SmallIntegers in the range 0 - 255.

 This method generates an error if instances of the receiver are of special
 storage format, if they are NSCs, or if they have instance variables.

 optionsArray is an Array of Symbols containing zero or more of  
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable , 
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of 
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 and at most one of the word size definitions for byte format (affects swizzling)
   #'2byteWords' #'4byteWords' #'8byteWords' 
   #'signed2byteWords' #'signed4byteWords' #'signed8byteWords' 
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant 
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying 
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

	| hist fmt descr |
	aDictionary
		ifNotNil: [ (aDictionary _validateClass: SymbolDictionary) ifFalse: [^nil] ].
	self isNsc
		ifTrue: 
			[^aString _error: #classErrBadFormat
				with: 'cannot create byte subclass of Nsc class'].
	fmt := (format bitAnd: 16r3 bitInvert) bitOr: 16r1 + 16r4.
	descr := aDescription.
	oldClass
		ifNotNil: 
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: optionsArray
				newFormat: fmt
				newInstVars: #() 
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false ) 
					ifTrue: 
						[oldClass _commentOrDescription: aDescription.
						^oldClass	"avoid creation of a new version" ].
			hist := oldClass classHistory.
			descr ifNil: [descr := [ oldClass commentForFileout ] on: Error do:[:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: #() 
		format: fmt
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: optionsArray
%

category: 'Subclass Creation'
method:
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
options: optionsArray


"Creates and returns a new byte subclass of the receiver.  You are not
 permitted to modify the new class after it is created.  If the receiver is not
 some kind of String class, then instances of the new class store and return
 SmallIntegers in the range 0 - 255.

 If aString is the name of a Class that is visible to the current user, this
 method creates the new class as a new version of the existing class, and they
 then share the same class history.  However, if no class named aString is
 visible to the user, then the new class is no relation to any existing class,
 and it has a new class history.

 This method generates an error if instances of the receiver are of special
 storage format, if they are NSCs, or if they have instance variables.

 optionsArray is an Array of Symbols containing zero or more of  
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable , 
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of 
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 and at most one of the word size definitions for byte format (affects swizzling)
   #'2byteWords' #'4byteWords' #'8byteWords' 
   #'signed2byteWords' #'signed4byteWords' #'signed8byteWords' 
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant 
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying 
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.
"

^ self byteSubclass: aString
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: optionsArray
%

category: 'Subclass Creation'
method:
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary

^ self byteSubclass: aString
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: #() 
%
! 3361
category: 'Subclass Creation'
method: Class
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
newVersionOf: oldClass
description: aDescription
options: optionsArray
	"Creates and returns a new indexable subclass of the receiver.  Instances of the
 new class are represented as pointer objects.

 This method generates an error if instances of the receiver are of special
 storage format or if they are NSCs.

 optionsArray is an Array of Symbols containing zero or more of  
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable , 
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of 
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant 
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying 
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

	| hist fmt descr |
	self isBytes
		ifTrue: 
			[^aString _error: #classErrBadFormat
				with: 'cannot create indexable subclass of byte class'].
	self isNsc
		ifTrue: 
			[^aString _error: #classErrBadFormat
				with: 'cannot create indexable subclass of Nsc class'].
	fmt := format bitOr: 16r4.	"add indexable bit"
	descr := aDescription.
	oldClass
		ifNotNil: 
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: optionsArray
				newFormat: fmt
				newInstVars: anArrayOfInstvarNames
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false )
					ifTrue: 
						[oldClass _commentOrDescription: aDescription.
						^oldClass	"avoid creation of a new version"].
			hist := oldClass classHistory.
			descr ifNil: [descr := [ oldClass commentForFileout ] on: Error do:[:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: anArrayOfInstvarNames
		format: fmt
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: optionsArray
%

category: 'Subclass Creation'
method:
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
options: optionsArray

"Creates and returns a new indexable subclass of the receiver.  Instances of the
 new class are represented as pointer objects.

 This method generates an error if instances of the receiver are of special
 storage format or if they are NSCs.

 optionsArray is an Array of Symbols containing zero or more of  
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable , 
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of 
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant 
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying 
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

^ self indexableSubclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: optionsArray
%

category: 'Subclass Creation'
method:
indexableSubclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary

"Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

^ self indexableSubclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: #() 
%

! 3472
! fixed 11833
category: 'Subclass Creation'
method: Class
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
newVersionOf: oldClass
description: aDescription
options: optionsArray

 "optionsArray is an Array of Symbols containing zero or more of  
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable , 
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of 
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant 
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying 
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

	| hist descr |
	descr := aDescription.
	oldClass
		ifNotNil: 
			[(self
				_equivalentSubclass: oldClass
				superCls: self
				name: aString
				newOpts: optionsArray
				newFormat: oldClass format
				newInstVars: anArrayOfInstvarNames
				newClassInstVars: anArrayOfClassInstVars
				newPools: anArrayOfPoolDicts
				newClassVars: anArrayOfClassVars
				inDict: aDictionary
				isKernel: false)
					ifTrue: 
						[oldClass _commentOrDescription: aDescription.
						^oldClass	"avoid creation of a new version"].
			hist := oldClass classHistory.
			descr
				ifNil: 
					[descr := [oldClass commentForFileout] on: Error
								do: [:ex | 'old comment not available']]].
	^self
		_subclass: aString
		instVarNames: anArrayOfInstvarNames
		format: format
		classVars: anArrayOfClassVars
		classInstVars: anArrayOfClassInstVars
		poolDictionaries: anArrayOfPoolDicts
		inDictionary: aDictionary
		inClassHistory: hist
		description: descr
		options: optionsArray
%

! j
category: 'Subclass Creation'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
options: optionsArray

"Creates and returns a new subclass of the receiver.

 optionsArray is an Array of Symbols containing zero or more of  
   #noInheritOptions,  #subclassesDisallowed, #disallowGciStore, #modifiable , 
   #traverseByCallback #selfCanBeSpecial #logCreation
 and at most one of 
   #dbTransient, #instancesNonPersistent, #instancesInvariant
 If present, #noInheritOptions must be the first element and it causes
 none of subclassesDisallowed, disallowGciStore, traverseByCallback,
         dbTransient, instancesNonPersistent, instancesInvariant 
 to be inherited from the superclass, nor copied from the
 current version of the class.
 #selfCanBeSpecial is never inherited and is needed only when modifying 
 superclass hierarchy above classes with special format.
 The option #logCreation, if present in the options array, causes logging
 with GsFile(C)>>gciLogSever:  of class creation / equivalence.

 Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "


^ self subclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: optionsArray
%

category: 'Subclass Creation'
method:
subclass: aString
instVarNames: anArrayOfInstvarNames
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary

"Executes
   oldClass := aDictionary at:(aString asSymbol) otherwise: nil
 and returns oldClass if it would be equivalent to the requested new class.
 (See the class comment for Class). "

^ self subclass: aString
    instVarNames: anArrayOfInstvarNames
    classVars: anArrayOfClassVars
    classInstVars: anArrayOfClassInstVars
    poolDictionaries: anArrayOfPoolDicts
    inDictionary: aDictionary
    newVersionOf: (self _classNamed: aString inDictionary: aDictionary)
    description: nil
    options: #() 
%

! -------------------------------------------------------
!  methods for _equivalentSubclass  funtionalilty
category: 'Private'
method:
_validateOptions: optionsArray  withFormat: theFormat newClassName: subClsName
  "theFormat is requested format and optionsArray is
   requested options for a new class .
   Returns an Array,
    { (theFormat with possible bits added) .  modifiableBBoolean . logBoolean } .
   signals an Error if the optionsArray contains an unregconized option."
| iOptCount fmt modifiableBool logBool swizSym nSwiz signed |
modifiableBool := false .
logBool := false .
nSwiz := 0 .
fmt := theFormat bitAnd: 16r2000 bitInvert "never inherit selfCanBeSpecial" .
iOptCount := 0 .
1 to: optionsArray size do:[:j | | oSym |
  oSym := optionsArray at: j .
  oSym == #noInheritOptions ifTrue:[
    j == 1 ifFalse:[ subClsName _error: #classErrBadFormat with: '#noInheritOptions must be first element of options'].
    "do not inherit dbTransient,instancesNonPersistent,instancesInvariant,subclassesDisallowed,
                    disallowGciStore,traverseByCallback "
    fmt := fmt bitAnd: (16r1E28  bitInvert) ] ifFalse:[
  oSym == #dbTransient ifTrue:[ iOptCount := iOptCount + 1 . fmt := fmt bitOr: 16r1000 ] ifFalse:[
  oSym == #instancesNonPersistent  ifTrue:[ iOptCount := iOptCount + 1 . fmt := fmt bitOr: 16r800 ] ifFalse:[
  oSym == #instancesInvariant ifTrue:[ iOptCount := iOptCount + 1 . fmt := fmt bitOr: 16r8 ] ifFalse:[
  oSym == #subclassesDisallowed ifTrue:[ fmt := fmt bitOr: 16r20 ] ifFalse:[
  oSym == #disallowGciStore ifTrue:[ fmt := fmt bitOr: 16r200 ] ifFalse:[
  oSym == #traverseByCallback ifTrue:[ fmt := fmt bitOr: 16r400 ] ifFalse:[
  oSym == #selfCanBeSpecial ifTrue:[ fmt := fmt bitOr: 16r2000 ] ifFalse:[
  oSym == #modifiable ifTrue:[ modifiableBool := true ] ifFalse:[
  oSym == #logCreation ifTrue:[ logBool := true ] ifFalse:[
  oSym == #'2byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=false ] ifFalse:[
  oSym == #'4byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=false ] ifFalse:[
  oSym == #'8byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=false ] ifFalse:[
  oSym == #'signed2byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=true ] ifFalse:[
  oSym == #'signed4byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=true ] ifFalse:[
  oSym == #'signed8byteWords' ifTrue:[ swizSym := oSym  . nSwiz := nSwiz + 1 . signed:=true ] ifFalse:[
    subClsName _error: #classErrBadFormat with: 'unrecognized option ' , oSym printString
  ]]]]]]]]]]]]]]]].
].
iOptCount > 1 ifTrue:[
  subClsName _error: #classErrBadFormat
        with:'only one of #dbTransient #instancesNonPersistent  #instancesInvariant allowed' .
].
nSwiz > 1 ifTrue:[
  subClsName _error: #classErrBadFormat
        with: 'only one of #''2byteWords'' #''4byteWords'' #''8byteWords'' #''signed2byteWords'' #''signed4byteWords'' #''signed8byteWords'' allowed'
].
swizSym ifNotNil:[
  (fmt bitAnd: 3) ~~ 1  ifTrue:[
    subClsName _error: #classErrBadFormat
        with:'A *byteWords options wa specified but new class would not be byte format'.
  ].
  fmt := self _addByteSwizzle: swizSym toFormat: fmt newClassName: subClsName signed: signed .
].
^ { fmt . modifiableBool . logBool }
%

category: 'Private'
method:
_addByteSwizzle: aSymbol toFormat: fmt newClassName: subClsName signed: isSigned
 | swiz newFmt oldSwiz signedMask |
  "uses GC_BEHAV_byteSwizKind_shift"
  aSymbol == #'2byteWords' ifTrue:[ swiz := 1 ].
  aSymbol == #'4byteWords' ifTrue:[ swiz := 2 ].
  aSymbol == #'8byteWords' ifTrue:[ swiz := 3 ].
  aSymbol == #'signed2byteWords' ifTrue:[ swiz := 1 ].
  aSymbol == #'signed4byteWords' ifTrue:[ swiz := 2 ].
  aSymbol == #'signed8byteWords' ifTrue:[ swiz := 3 ].
  swiz ifNil:[ Error signal:'invalid argument ' , aSymbol].

  oldSwiz := (format bitShift: -31"GC_BEHAV_byteSwizKind_shift") bitAnd: 3 .
  oldSwiz > 0 ifTrue:[
    oldSwiz = swiz ifFalse:[ 
       subClsName _error: #classErrBadFormat 
        with: ' option ', aSymbol printString,' incompatible with superclass'.
    ].
  ].
  newFmt := fmt bitOr: (swiz bitShift: 31"GC_BEHAV_byteSwizKind_shift" ) .
  signedMask := 1 bitShift: 33"GC_BEHAV_signedWords_shift" .
  isSigned ifTrue:[ newFmt := newFmt bitOr: signedMask ]
          ifFalse:[ newFmt := newFmt bitAnd: (signedMask bitInvert) ].
  ^ newFmt 
%
  

! fix 47903
method:
_optionsChangableTo: formatArr 

  "formatArr describes format and options that would be produced by
   a new class creation.
  Returns true or false, or signals an Error.
   Returns true if the format and options of the receiver could
   be changed to match formatArr without requiring creation of a new subclass
   of the receiver's superclass. " 
  | fmt newFmt |
  fmt := format .
  newFmt := formatArr at: 1 .
  ((fmt bitAnd: 16r800) ~~ 0 and:[ (newFmt bitAnd: 16r800) == 0 ]) ifTrue:[ 
     ^ false "cannot clear instancesNonPersistent if previously set" 
  ].
  (newFmt bitAnd: 16r1008 ) == (fmt bitAnd: 16r1008 ) ifFalse:[
    "difference in one of #dbTransient, #instancesInvariant "
    ^ false 
  ].
  ((fmt bitAnd: 16r800) == 0 and:[ (newFmt bitAnd: 16r800) ~~ 0 ]) ifTrue:[ 
     "setting instancesNonPersistent"
     (newFmt bitAnd: 16r1008) ~~ 0 ifTrue:[ 
       "instancesNonPersistent not allowed with dbTransient or instancesInvariant"
       ^ false .
     ].
  ]. 
  ((fmt bitAnd: 16r20) == 0 and:[ (newFmt bitAnd: 16r20) ~~ 0 ]) ifTrue:[ 
     ^ false "cannot set subclassesDisallowed if previously cleared" 
  ].
  self isModifiable ifFalse:[ 
    (formatArr at: 2 "modifiableBool") ifTrue:[ ^ false ] "cannot reenable modifiable"
  ].
  (fmt bitAnd: 16r2047) ~~ (newFmt bitAnd: 16r2047) ifTrue:[
     ^ false "difference in SELF_CAN_BE_SPECIAL,  NSC_DUPLICATES,
		or INDEXABLE, or IMPLEMENTATION  "
     "ignoring CONSTRAINTS_MASK"
  ]. 
  ((fmt bitOr: newFmt) bitAnd: 16r7FFFC000) ~~ 0 ifTrue:[
     ^ false "Ruby virtual , module or metaClass bit is present in either"
  ].
  (fmt bitShift: -31) = (newFmt bitShift: -31) ifFalse:[ "GC_BEHAV_byteSwizKind_shift"  
     ^ false "differing byte swizzle"
  ].
  "we allow changing traverseByCallback always"
  ^ true 
%


! fixed 42346
method:
_adjustOptions: opts formatArr: fmtArray
  | fmt newFmt modifiableBool fmtToStore |
  fmt := format .
  fmtToStore := fmt .
  newFmt := fmtArray at: 1 .
  modifiableBool := fmtArray at: 2 .
  ((fmt bitAnd: 16r20) ~~ 0 and:[ (newFmt bitAnd: 16r20) == 0 ]) ifTrue:[
     "clear subclassesDisallowed bit."
     fmtToStore := fmtToStore bitAnd: 16r20 bitInvert .
  ].
  ((fmt bitAnd: 16r800) == 0 and:[ (newFmt bitAnd: 16r800) ~~ 0 ]) ifTrue:[
    "set instancesNonPersistent bit"
    fmtToStore := fmtToStore bitOr: 16r800 .
  ].
  (fmt bitAnd: 16r400) ~~ (newFmt bitAnd: 16r400) ifTrue:[ 
     "change state of traverseByCallback"
     fmtToStore := (fmtToStore bitAnd:( 16r400 bitInvert)) bitOr:( newFmt bitAnd: 16r400) . 
  ].
  (fmt bitAnd: 16r200) ~~ (newFmt bitAnd: 16r200) ifTrue:[ 
     "change state of disallowGciStore"
     fmtToStore := (fmtToStore bitAnd:( 16r200 bitInvert)) bitOr:( newFmt bitAnd: 16r200) . 
  ].
  fmtToStore ~~ fmt ifTrue:[
     self _unsafeAt: 2 "offset of format" put: fmtToStore .
     (fmtArray at: 3) ifTrue:[ "logBool"
        GsFile gciLogServer: 'modified format of existing class ' , self name 
     ].
  ].
  "clear modifiable"
  (self isModifiable and:[ modifiableBool == false ]) ifTrue:[
     self immediateInvariant .
  ].
%

method:
_poolDictsEqual: anArray
  "pool dictionaries should be identical to ensure that compiled-in Associations can be shared"
  "fix bug 42279"

  | sharedPoolsArray |
  sharedPoolsArray := self sharedPools.
  sharedPoolsArray size ~= anArray size ifTrue: [ ^false ].
  1 to: sharedPoolsArray size do: [:index |
    (sharedPoolsArray at: index) == (anArray at: index) ifFalse: [ ^false ]].
  ^true
%

! fixed 41575
method:
_classVarsChangableTo: anArray 
  anArray ifNotNil:[ | argSet |
    anArray _isArray ifFalse:[
      (anArray _validateClass: Array) ifFalse:[ ^ nil ].
    ].
    argSet := IdentitySet new .
    1 to: anArray size do: [:index| | aVarName aSym definingClass |
      aVarName := anArray at: index .
      (aVarName _isOneByteString or:[ aVarName _validateClass: CharacterCollection]) ifTrue:[
	aSym := aVarName asSymbol .
	aSym validateIsIdentifier "fix bug 9666" .
	definingClass := self _classDefiningClassVar: aSym .  "fix bug 10480"
	definingClass ifNotNil:[ definingClass == self ifFalse:[  ^ false ]].
        argSet add: aSym .  "exists or ok to add"
        anArray at: index put: aSym .
      ].
    ].
    classVars ifNotNil:[ :cvs | | toRemove |
      toRemove := cvs keys - argSet  .
      toRemove do:[:aKey |
        (classVars associationAt: aKey) isInvariant ifTrue:[ ^ false ].
      ].
    ].
  ].
  ^ true
%

! fixed 41575 , 46715
method:
_adjustClassVars: anArray
  classVars ifNil:[
    1 to: anArray size do:[:j | self _addClassVar: (anArray at: j) value: nil ]
  ] ifNotNil:[
    | argSet existingSet toRemove toAdd assocs |
    existingSet := IdentitySet new . 
    assocs := { } .
    argSet := IdentitySet withAll: anArray  .
    "GsDevKit and Seaside need to be able to override classVarNames for
     implementation of SharedPool ,
     so must use   self classVarNames    here."
    self classVarNames do: [:classVarName | | a |
      a := classVars associationAt: classVarName.
      assocs add: a . existingSet add: a key ].
    toRemove := existingSet - argSet  .
    classHistory size > 1 ifTrue:[  | cvd | "create a new dictionary"
      cvd := SymbolDictionary new objectSecurityPolicy: classVars objectSecurityPolicy ; yourself.
      assocs do:[:a | (toRemove includes: a key) ifFalse:[ cvd addAssociation: a ]].
      self _setClassVars: cvd old: classVars .
    ] ifFalse:[
      toRemove do:[ :aKey | self removeClassVarName: aKey ].
    ].
    toAdd := argSet - existingSet .
    toAdd do:[ :aKey | self _addClassVar: aKey value: nil ].
  ].
%

category: 'Private'
method: Class
_equivalentSubclass: oldClass 
	superCls: actualSelf 
	name: aString 
	newOpts: optionsArray 
	newFormat: theFormat 
	newInstVars: anArrayOfInstvarNames 
	newClassInstVars: anArrayOfClassInstVars 
	newPools: anArrayOfPoolDicts 
	newClassVars: anArrayOfClassVars 
	inDict: aDictionary 
	isKernel: isKernelBool

 "oldClass is equivalent to the subclass that would be created using
  the other arguments if: 
     instVar names match exactly ,
     and class instVar names match exactly ,
     and the classVars in oldClass can be modified to add/remove Associations 
        to match anArrayOfClassVars ,
     and pool dictionaries match exactly

  With respect to options and format, oldClass is equivalent if
    The state of format bits dbTransient, instancesInvariant match exactly ,
    and subclassesDisallowed cannot be set in the new subclass if it not set in oldClass ,
    and modifiable  cannot be set if it is not set in oldClass  ,
    and (SELF_CAN_BE_SPECIAL, NSC_DUPLICATES, INDEXABLE, IMPLEMENTATION, NO_STRUCT_UPDATE bits)
        of the formats must match exactly.

  If all other equivalence tests pass, the following changes to oldClass may be 
  made to match the arguments and avoid creating a new subclass:
    instancesNonPersistent bit may be set (but not cleared) in the format of the oldClass
    subclassesDisallowed bit may be cleared (but not set) in format of oldClass 
    traverseByCallback bit may be set or cleared in format of oldClass
    oldClass may be changed from modifiable to not modifiable (by sending immediateInvariant)
    classVars may be added to oldClass 
    classVars having modifiable Associations may be removed from oldClass 
"

	| oldOk fmtArr nam supr opts ivs civs poolds cvars cvarsArray |
	fmtArr := self _validateOptions: optionsArray withFormat: theFormat newClassName: aString .
	(oldClass isKindOf: Class) ifFalse: [oldClass _validateClass: Class].
	nam := oldClass name asString = aString asString.
	supr := oldClass superClass == actualSelf.
	opts := oldClass _optionsChangableTo: fmtArr.
	ivs := oldClass _instVarsEqual: anArrayOfInstvarNames.
	civs := oldClass class _instVarsEqual: anArrayOfClassInstVars.
	poolds := oldClass _poolDictsEqual: anArrayOfPoolDicts.
	cvars := oldClass
				_classVarsChangableTo: (cvarsArray := anArrayOfClassVars copy).
	oldOk := nam
				and: [supr and: [opts and: [ivs and: [civs and: [poolds and: [cvars]]]]]].
	oldOk
		ifTrue: 
			[| oldVal newName |
			aDictionary ifNotNil: [
				newName := aString asSymbol.
				oldVal := aDictionary at: newName otherwise: nil.
				oldVal == oldClass
					ifFalse: 
						[ImproperOperation
							signal: 'no new subclass needed, but aDictionary at: oldClass name ~~ oldClass'.
						^false] ].
			oldClass _adjustOptions: optionsArray formatArr: fmtArr.
			oldClass _adjustClassVars: cvarsArray.
			(fmtArr at: 3)
				ifTrue: 
					[GsFile gciLogServer: 'class ' , aString , ' equivalent to requested class'].
			^true]
		ifFalse: 
			[(fmtArr at: 3)
				ifTrue: 
					[GsFile
						gciLogServer: '_equivalentSubclass false, nam:' , nam asString , 
                ' supr:' , supr asString , ' opts:' , opts asString , 
                ' ivs:' , ivs asString , ' civs:' , civs asString , 
                ' poolds:' , poolds asString , ' cvars:' , cvars asString]].
	^false
%

! fix 42718
category: 'Repository Conversion'
method:
needsRecompileFor30

 "Returns true if the receiver needs to have methods recompiled."

 ^ super needsRecompileFor30 or:[ self class needsRecompileFor30]
%

category: 'Repository Conversion'
method:
needsRecompileFor33

 "Returns true if the receiver needs to have methods recompiled."

 ^ super needsRecompileFor33 or:[ self class needsRecompileFor33]
%


