! Rewritten for 47763
category: 'Private'
method:
_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
into: destObjOrNil tag: tag extraData: eData

"Private method for encrypting or decrypting the receiver.
 The following encryption schemes are currently supported:

 ==================================================
                          Key     Salt    Tag
 opCode Cipher   Mode  bits/Bytes Size   Bytes
 ==================================================
   1     AES     CBC     128/16    16     N/A
   2     AES     CBC     192/24    16     N/A
   3     AES     CBC     256/32    16     N/A
   4     AES     OCB     128/16    12     16
   5     AES     OCB     192/24    12     16
   6     AES     OCB     256/32    12     16
   7     AES     GCM     128/16    12     16
   8     AES     GCM     192/24    12     16
   9     AES     GCM     256/32    12     16
  10   CHACHA20 Poly1305 256/32    12     16
 ==================================================

 AES encryption/decryption (Advanced Encryption Standard) is performed 
 using the OpenSSL open source package and the AES specification, 
 available at: 
   http://csrc.nist.gov/publications/fips/fips197/fips-197.pdf

 CBC is an acronym for cipher block chaining. See the referenced AES 
 specification document for further details.

 OCB is an acronym for Offset Cookbook Mode. See RFC 7253 for further
 details.

 GCM is an acronym for Galois Counter Mode. See RFC 5288 for further
 details.

 OCB, GCM and Poly1305 are AEAD modes. AEAD is an acronym for 
 Authenticated Encryption with Associated Data. AEAD provides data
 authenticity, confidentiality, and integrity. It also supports
 Additional Authenticated Data (AAD). AAD is not encrypted and therefore
 not kept confidential, however AAD authenticity and integrity are
 guaranteed. AAD is not included in the encrypted payload but must be 
 provided in order to decrypt the data. AAD is optional and the eData 
 argument may be nil if AAD is not required.

 AEAD encryption/decryption is performed using the OpenSSL open source
 package and is implemented to conform to ISO/IEC standard 19772:2009.
 See https://www.iso.org/standard/46345.html for further information.

 opCode must be an instance of SmallInteger and be one of the values from
 the first column of the above table. A positive value indicates the data
 is to be encrypted and a negative value indicates decryption.

 aKey and aSalt must be instances of ByteArray of the correct size
 per the above table, otherwise an error is raised.

 destObjOrNil must be nil or an instance of a non-invariant byte object.
 If destObjOrNil is nil, the result of the operation will be placed into a
 new instance of ByteArray (encryption) or String (decryption). Otherwise 
 the result will be placed into the given byte object starting at offset 1. 
 The size of destObjOrNil will be modified to correctly contain all 
 encrypted or decrypted data, and may differ from the size of the receiver 
 due to the automatic addition or removal of padding by the cipher 
 algorithm.

 When encrypting a receiver that has a character size greater than one, data
 is placed into big-endian byte order before encryption.

 When decrypting into a destObjOrNil object that a character size greater
 than one, data is converted to big-endian byte order after decryption.

 During AEAD encryption, a tag is generated which is used during decryption
 to ensure data integrity. The tag data will be stored into the tag 
 argument, which must an instance of a variant byte object. During AEAD 
 decryption, tag must be a byte object containing the tag bytes returned 
 during encryption. For non-AEAD modes, tag must be nil.

 During AEAD encryption, eData must be nil or a byte object with a character
 size of one containing additional data to be used in generating the tag
 value. eData is NOT added to the encrypted payload. During decryption, 
 eData must be a byte object with a character size of one containing the
 same bytes provided during encryption or nil if no byte object was
 provided. For non-AEAD modes, eData must be nil.

 Successful encryption or decryption returns encrypted or decrypted data, 
 which is stored into destObjOrNil if it was provided or a new byte object
 if it was not. An exception is raised if encryption or decryption fails.
"

<primitive: 953>
self _validateEncryptionOpCode: opCode ;
     _validateEncryptionKey: aKey forOpCode: opCode ;
     _validateEncryptionSalt: aSalt forOpCode: opCode ;
     _validateEncryptionExtraData: eData forOpCode: opCode .
destObjOrNil ifNotNil:[ destObjOrNil _validateIsBytes ; validateIsVariant ] .  
self _validateEncryptionTag: tag forOpCode: opCode .
^ self _primitiveFailed: #_primEncryptDecryptWithKey:salt:opCode:into:tag:extraData:
%

category: 'Private'
method:
_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode into: destObjOrNil
"Private method for invoking non-AEAD encrypt/decrypt modes. See the method:
    #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
    into: destObjOrNil tag: tag extraData: eData
 for more information."
 
^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: opCode
       into: destObjOrNil
       tag: nil
       extraData: nil
%

category: 'Encrypting'
method:
aesEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil
"Encrypts the receiver using 128 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 1  into: destObjOrNil
%

category: 'Encrypting'
method:
aesEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil
"Encrypts the receiver using 192 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 2 into: destObjOrNil
%

category: 'Encrypting'
method:
aesEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil
"Encrypts the receiver using 256 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 3 into: destObjOrNil
%

category: 'Encrypting'
method:
aesEncryptWith128BitKey: aKey salt: aSalt
"Encrypts the receiver using 128 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith128BitKey: aKey salt: aSalt into: nil
%

category: 'Encrypting'
method:
aesEncryptWith192BitKey: aKey salt: aSalt
"Encrypts the receiver using 192 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith192BitKey: aKey salt: aSalt into: nil
%

category: 'Encrypting'
method:
aesEncryptWith256BitKey: aKey salt: aSalt
"Encrypts the receiver using 256 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith256BitKey: aKey salt: aSalt into: nil
%

category: 'Decrypting'
method:
aesDecryptWith128BitKey: aKey salt: aSalt
"Decrypts the receiver using 128 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith128BitKey: aKey salt: aSalt into: nil
%

category: 'Decrypting'
method:
aesDecryptWith192BitKey: aKey salt: aSalt
"Decrypts the receiver using 192 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith192BitKey: aKey salt: aSalt into: nil
%

category: 'Decrypting'
method:
aesDecryptWith256BitKey: aKey salt: aSalt
"Decrypts the receiver using 256 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith256BitKey: aKey salt: aSalt into: nil
%

category: 'Decrypting'
method:
aesDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil
"Decrypts the receiver using 128 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -1 into: destObjOrNil
%

category: 'Decrypting'
method:
aesDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil
"Decrypts the receiver using 192 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -2 into: destObjOrNil
%

category: 'Decrypting'
method:
aesDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil
"Decrypts the receiver using 256 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -3 into: destObjOrNil
%


category: 'Encrypting'
method:
encryptionExample

| key salt enc dec hamlet |
key  := ByteArray withRandomBytes: 32 .
salt := ByteArray withRandomBytes: 16 .

hamlet := 
'Alas, poor Yorick! I knew him, Horatio: a fellow
of infinite jest, of most excellent fancy: he hath
borne me on his back a thousand times; and now, how
abhorred in my imagination it is! my gorge rims at
it. Here hung those lips that I have kissed I know
not how oft. Where be your gibes now? your
gambols? your songs? your flashes of merriment,
that were wont to set the table on a roar? Not one
now, to mock your own grinning? quite chap-fallen?
Now get you to my ladys chamber, and tell her, let
her paint an inch thick, to this favour she must
come; make her laugh at that. Prithee, Horatio, tell
me one thing.' .


enc := hamlet aesEncryptWith256BitKey: key salt: salt .
dec := enc aesDecryptWith256BitKey: key salt: salt into: String new.
^ dec = hamlet
%

category: 'Private'
method:
_saltSizeForOpCode: opCode

^ opCode abs >= 4
    ifTrue:[ 12 ]
   ifFalse:[ 16 ]
%

category: 'Private'
method:
_keySizeInBytesForOpCode: opCode

| code codes128Bit codes192Bit codes256Bit |
code := opCode abs .
codes128Bit := { 1 . 4 . 7 } .
codes192Bit := { 2 . 5 . 8 } .
codes256Bit := { 3 . 6 . 9 . 10 } .

(codes128Bit includesIdentical: code)
  ifTrue:[ ^ 16 ].
(codes192Bit includesIdentical: code)
  ifTrue:[ ^ 24 ].
(codes256Bit includesIdentical: code)
  ifTrue:[ ^ 32 ].
%

category: 'Private'
method:
_validateEncryptionOpCode: opCode
opCode _validateClass: SmallInteger .
(((opCode < -10) or:[opCode > 10]) or:[opCode == 0])
  ifTrue:[opCode _error: #rtErrArgOutOfRange args:{ -10 . 10 } ] .
^ true
%

category: 'Private'
method:
_validate: anObj isSize: expectedSize

anObj _basicSize == expectedSize
  ifFalse:[ anObj _error: #rtErrBadSize args: { expectedSize . anObj _basicSize } ] .
%

category: 'Private'
method:
_validateEncryptionKey: aKey forOpCode: opCode
aKey _validateClass: ByteArray .
self _validate: aKey isSize: (self _keySizeInBytesForOpCode: opCode)
%
category: 'Private'
method:
_validateEncryptionSalt: salt forOpCode: opCode
salt _validateClass: ByteArray .
self _validate: salt isSize: (self _saltSizeForOpCode: opCode)
%

category: 'Private'
method:
_validateEncryptionTag: tag forOpCode: opCode
| absCode |
absCode := opCode abs .
absCode > 3 "AEAD encrypt/decrypt"
  ifTrue:[
    tag _validateClass: ByteArray .
    opCode < 3 ifTrue:[ tag validateIsVariant ]			"AEAD Encrypting"
              ifFalse:[ self _validate: tag isSize: 16 ] . 	"AEAD Decrypting" 
  ] ifFalse:[ tag _validateClass: UndefinedObject ] "Not AEAD"	   
%

category: 'Private'
method:
_validateEncryptionExtraData: eData forOpCode: opCode
opCode abs > 3  "AEAD encrypt/decrypt"
  ifTrue:[ eData ifNotNil:[ eData _validateIsBytes ]]
 ifFalse:[ eData _validateClass: UndefinedObject ]
%

category: 'Authenticated Encrypting'
method:
aesOcbEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 128 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 4
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Decrypting'
method:
aesOcbDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 128 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -4
       into: destObjOrNil
       tag: tag
       extraData: eData
%


category: 'Authenticated Encrypting'
method:
aesOcbEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 192 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 5
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Decrypting'
method:
aesOcbDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 192 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -5
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method:
aesOcbEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 256 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 6
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Decrypting'
method:
aesOcbDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 256 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -6
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method:
aesGcmEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 128 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 7 into: destObjOrNil tag: tag extraData: eData
%

category: 'Authenticated Decrypting'
method:
aesGcmDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 128 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -7
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method:
aesGcmEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 192 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 8
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Decrypting'
method:
aesGcmDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 192 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -8
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method:
aesGcmEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using 256 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 9
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Decrypting'
method:
aesGcmDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using 256 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -9
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method:
chacha20Poly1305EncryptWithKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Encrypts the receiver using CHACHA20-Poly1305 encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 10
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Decrypting'
method:
chacha20Poly1305DecryptWithKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData
"Decrypts the receiver using CHACHA20-Poly1305 decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode 
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -10
       into: destObjOrNil
       tag: tag
       extraData: eData
%

category: 'Authenticated Encrypting'
method:
authenticatedEncryptionExample

| key salt enc dec hamlet extraData tag|
key  := ByteArray withRandomBytes: 32 .
salt := ByteArray withRandomBytes: 12 .

hamlet := 
'Alas, poor Yorick! I knew him, Horatio: a fellow
of infinite jest, of most excellent fancy: he hath
borne me on his back a thousand times; and now, how
abhorred in my imagination it is! my gorge rims at
it. Here hung those lips that I have kissed I know
not how oft. Where be your gibes now? your
gambols? your songs? your flashes of merriment,
that were wont to set the table on a roar? Not one
now, to mock your own grinning? quite chap-fallen?
Now get you to my ladys chamber, and tell her, let
her paint an inch thick, to this favour she must
come; make her laugh at that. Prithee, Horatio, tell
me one thing.' .

extraData := 'Hamlet: Act V, Scene i'.

tag := ByteArray new.
enc := hamlet aesOcbEncryptWith256BitKey: key salt: salt
       into: ByteArray new tag: tag extraData: extraData .
dec := enc aesOcbDecryptWith256BitKey: key salt: salt
       into: String new tag: tag extraData: extraData .
^ dec = hamlet
%
