!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   FixedPoint, Number, Magnitude, Object.
!
!=========================================================================

removeallmethods FixedPoint
removeallclassmethods FixedPoint
set class FixedPoint

category: 'For Documentation Installation only'
classmethod: 
installDocumentation

self comment:
'This class was named ScaledDecimal in Gemstone/64 v2.4 and prior versions.
 FixedPoint stores numerical values as a rational number, represented by a
 numerator and denominator that are Integers.  Since the numerator and
 denominator can be carried to arbitrary precision, FixedPoint can represent
 any rational number without loss of precision.  It also calculates based upon
 fractional arithmetic, and thus produces numerical results without loss of
 precision.

 FixedPoint also provides for automatic rounding to a fixed precision after
 the decimal point when converting to and from other types, such as String.

 The literal form for a FixedPoint uses $p; for example, 1.23p2, where the 
 final digit indicates the scale.

 One useful application of this kind of number is for financial instruments,
 which are always rounded off, but usually need more digits than a floating
 number can accurately express in order not to lose precision during
 computation.

--- instVar denominator
A positive Integer that represents the denominator of the rational value 
 of the instance.

--- instVar numerator
An Integer that represents the numerator of the rational value of the
 instance.

--- instVar scale
A non-negative SmallInteger that represents the number of decimal places of
 precision to the right of the decimal point.
' .
%

category: 'Instance Creation'
classmethod: 
numerator: numerator denominator: denominator scale: scale

"Returns an instance of FixedPoint with the given numerator and denominator.

 The arguments numerator and denominator must be Integers.
 The argument  scale  must be a SmallInteger . "

(denominator = 0) ifTrue: [ ^ numerator _errorDivideByZero ].
^ self basicNew _numerator: numerator denominator: denominator scale: scale
%

category: 'Private'
method:
_numerator: num denominator: den scale: sc

"Private.  Assigns the receiver's instance variables, reduces it, and makes it
 invariant."

num _isInteger ifTrue:[   "gemstone64, explicit constraint enforcement"
  den _isInteger ifTrue:[
    sc _isSmallInteger ifTrue:[
      numerator := num.
      denominator := den.
      sc < 0 ifTrue:[ scale := 0 ]
      	    ifFalse:[ scale := sc ].
      ^ self _reduce immediateInvariant
    ] ifFalse:[
      ArgumentTypeError new constrainedIv: 'FixedPoint.scale' 
	     expectedClass: SmallInteger actualArg: sc ;
	 signal
    ].
  ] ifFalse: [
    ArgumentTypeError new constrainedIv: 'FixedPoint.denominator' 
	expectedClass: Integer actualArg: den ;
      signal
  ].
] ifFalse:[ 
  ArgumentTypeError new constrainedIv: 'FixedPoint.numerator' 
	expectedClass: Integer actualArg: num ;
      signal
].
self _uncontinuableError .
%

category: 'Private'
method:
_reduce

"Private.  Reduces the receiver."

| gcd |
"now reduce it"
numerator = 0 ifTrue:[ 
  denominator := 1. 
  ^ self 
  ].
denominator < 0 ifTrue:[  "denominator is always positive "
  numerator := numerator negated .
  denominator := denominator negated
  ].
gcd := numerator gcd: denominator.
numerator := numerator // gcd.
denominator := denominator // gcd.
^ self 
%

category: 'Accessing'
method:
denominator

"Returns the denominator of the receiver."

^denominator
%

category: 'Testing'
method:
isZero

"Returns true if the receiver is zero."

^ numerator = 0 .
%

category: 'Accessing'
method:
instVarAt: anIndex put: aValue

"Disallowed.  You may not change the value of a FixedPoint."

self shouldNotImplement: #instVarAt:put:
%

category: 'Accessing'
method:
numerator

"Returns the numerator of the receiver."

^ numerator
%

category: 'Accessing'
method:
scale

"Returns the scale of the receiver."

^scale
%

category: 'Accessing'
method:
size: anInteger

"Disallowed.  You may not change the size of a FixedPoint."

self shouldNotImplement: #size:
%

category: 'Updating'
method:
reduced

"Returns a FixedPoint determined by finding the greatest common
 divisor of the numerator and denominator of the receiver."

"Reduce a fraction to its smallest terms."

| gcd numer denom |

(numerator = 0) ifTrue:[ 
  denominator == 1 ifFalse:[
    denominator := 1 .
  ].
  ^ self
].
gcd := numerator gcd: denominator.
numer := numerator // gcd.
denom := denominator // gcd.
(numer = numerator and:[ denom = denominator]) ifTrue:[ ^ self ].

^ FixedPoint numerator: numer denominator: denom scale: scale.
%

category: 'Private'
method:
_scale: aScale

"Private."

scale := aScale
%

category: 'Formatting'
method:
withScale: newScale

"Returns the receiver with the new scale."

scale == newScale ifTrue:[ ^ self ].
^ (self shallowCopy _scale: newScale) immediateInvariant
%

category: 'Formatting'
method:
_asString: dpChar

| x num numer denom aString wholePart fraction |

aString := String new.
scale ifNil:[ ^ '(uninitialized FixedPoint)' "fix bug 13190"].
x := 10 raisedToInteger: scale .
numer := numerator .
denom := denominator .
num := (( numer * x) + (denom quo: 2)) // denom .
(numer < 0) ifTrue:[
    aString add: $- .
    num := num negated .
].
wholePart := num // x .
fraction := num \\ x .

aString add: wholePart asString .
aString add: dpChar . 
scale timesRepeat: [
  fraction := fraction * 10 .
  aString add: (fraction // x) asString .
  fraction := fraction \\ x .
].
^ aString .
%

category: 'Formatting'
method:
asString

 "Returns a String of the form '123.56 for a number with scale = 2 ,
  where the decimal point character in the result is per the current Locale."

  ^ self _asString: Locale decimalPoint . "fix 36666"
%
! fixed 45339
category: 'Formatting'
method:
asStringLocaleC

 "Returns a String of the form '123.56 for a number with scale = 2.
  Does not use Locale , decimal point character is always $.  " 

  ^ self _asString: $. 
%


category: 'Private'
classmethod:
_fromString: aString decimalPoint: dp

"Private.  Given aString such as '34.23', returns a non-reduced FixedPoint.
 Using the specified decimal point character .
 If dp == nil , the session's locale state is used.
 Returns nil if there is a format error in aString"

<primitive: 465>
aString _validateClass: String.
dp ifNotNil:[ dp _validateClass: Character ].
self _errIncorrectFormat: aString.
self _primitiveFailed: #_fromString:decimalPoint: args: { aString . dp } .
%

category: 'Instance Creation'
classmethod:
fromStringLocaleC: aString

"Given aString such as '34.23', returns an instance of FixedPoint with
 appropriate numerator and denominator, and with scale equal to the number
 of digits to the right of the decimal point. 
 If a String includes the literal indicator $p, then an sequence of digits 
 immediately following $p are used for the scale. Characters other than this 
 after the $p are ignored.  
 The expected decimal point character is $.  "

^ ( self _fromString: aString decimalPoint: $. ) _reduce immediateInvariant
%

classmethod:
fromString: aString

"Given aString such as '34.23', returns an instance of FixedPoint with
 appropriate numerator and denominator, and with scale equal to the number
 of digits to the right of the decimal point. 
 If a String includes the literal indicator $p, then an sequence of digits
 immediately following $p are used for the scale. Characters other than this
 after the $p are ignored.
 The session's locale state determines the expected decimal point character.
"

^ ( self _fromString: aString decimalPoint: nil ) _reduce immediateInvariant
%

category: 'Converting'
method:
asFloat

"Returns an instance of SmallDouble or Float that has the value of the receiver."

^ numerator asFloat / denominator asFloat
%

category: 'Converting'
method:
asDecimalFloat

"Returns an instance of DecimalFloat that has the value of the receiver."

^ numerator asDecimalFloat / denominator asDecimalFloat
%

category: 'Converting'
method:
_coerce: aNumber

"Reimplemented from Number."

^ aNumber asFixedPoint: scale
%

category: 'Comparing'
method:
< aFixedPoint

"Returns true if the receiver is less than aFixedPoint; returns false
 otherwise."

(aFixedPoint _getKind > 4) ifTrue: [ "NaN" ^ false ].
(aFixedPoint class == FixedPoint) ifTrue: [
  (aFixedPoint numerator = 0)
      ifTrue:[ ^ numerator < 0]
      ifFalse:[ ^ self - aFixedPoint < 0 ]
  ].
  (aFixedPoint isKindOf: Fraction) ifTrue: [
    ((aFixedPoint numerator == nil) or: [aFixedPoint denominator == nil]) ifTrue:[ 
       ^false 
    ].
  ].
  ^ self _retry: #< coercing: aFixedPoint
%

category: 'Comparing'
method:
>= aNumber

^ aNumber <= self
%

category: 'Comparing'
method:
<= aFixedPoint

"Returns true if the receiver is less than or equal to aFixedPoint;
 returns false otherwise."

(aFixedPoint _getKind > 4) ifTrue: [ "NaN" ^ false ].
(aFixedPoint isKindOf: Fraction) ifTrue: [
  ((aFixedPoint numerator == nil) _or: [aFixedPoint denominator == nil]) ifTrue:[
     ^false 
  ].
].
^ (self > aFixedPoint) not
%

! fixed 41553
category: 'Comparing'
method:
= aFixedPoint

self == aFixedPoint ifTrue:[ ^ true ].
aFixedPoint _isNumber ifFalse:[ ^ false ].  
(aFixedPoint class == FixedPoint) ifTrue: [
  (aFixedPoint numerator = 0)
      ifTrue:[ ^ numerator = 0]
      ifFalse:[ ^ aFixedPoint numerator = numerator 
                  and:[ aFixedPoint denominator = denominator ]]. 
].
^ self _retry: #= coercing: aFixedPoint
%

method:
hash
 
^ self asFloat hash  
%

category: 'Truncation and Rounding'
method:
truncated

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located."

^ numerator quo: denominator
%

category: 'Arithmetic'
method:
* aFixedPoint 

"Returns the result of multiplying the receiver by aFixedPoint."

aFixedPoint class == FixedPoint ifTrue:[
  ^ FixedPoint numerator: numerator * aFixedPoint numerator
           denominator: denominator * aFixedPoint denominator
           scale: scale
].
^ self _retry: #* coercing: aFixedPoint
%

category: 'Arithmetic'
method: 
+ aFixedPoint

"Returns the sum of the receiver and aFixedPoint."

aFixedPoint class == FixedPoint ifTrue:[
   | denom argDenom commonDenominator newNumerator |
   (denom := denominator) = (argDenom := aFixedPoint denominator) ifTrue:[
     ^ FixedPoint numerator: numerator + aFixedPoint numerator
                  denominator: denom
                  scale: scale
   ].
   commonDenominator := denom lcm: argDenom .
   newNumerator := numerator
                     * (commonDenominator quo: denom)
                     + (aFixedPoint numerator * (commonDenominator quo: argDenom )).
   ^ FixedPoint numerator: newNumerator
         denominator: commonDenominator scale: scale
].
^ self _retry: #+ coercing: aFixedPoint
%

category: 'Arithmetic'
method:
- aFixedPoint

"Returns the difference between the receiver and aFixedPoint."

(aFixedPoint class == FixedPoint) ifTrue:[
  ^ self + aFixedPoint negated
].
^ self _retry: #- coercing: aFixedPoint
%

category: 'Arithmetic'
method: 
/ aFixedPoint
   
"Returns the result of dividing the receiver by aFixedPoint ."

aFixedPoint class == FixedPoint ifTrue:[
  ^ self * aFixedPoint  reciprocal
].
^ self _retry: #/ coercing: aFixedPoint 
%


method: 
negated

"Returns a Number that is the negation of the receiver."

^ FixedPoint numerator: numerator negated denominator: denominator
                scale: scale
%

category: 'Arithmetic'
method:
reciprocal

(numerator = 0) ifTrue: [ ^ self _errorDivideByZero].
^ FixedPoint numerator: denominator denominator: numerator scale: scale
%

category: 'Storing and Loading'
classmethod:
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst num den scale |

passiveObj readNamedIV.
num := passiveObj ivValue.
passiveObj readNamedIV.
den := passiveObj ivValue.
passiveObj readNamedIV.
scale := passiveObj ivValue.
  
passiveObj skipNamedInstVars.

inst := self numerator: num denominator: den scale: scale.
passiveObj hasRead: inst.
^inst.
%

category: 'Storing and Loading'
method:
writeTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."

"Reimplemented from Number since the receiver has a non-literal representation."

^super basicWriteTo: passiveObj
%

! deleted classmethod _generality

category: 'Private'
method:
_generality

"Returns an Integer representing the ordering of the receiver in
 the generality hierarchy."

^ 65
%

category: 'Testing'
method:
even

"Returns true if the receiver is an even integer, false otherwise."

 denominator = 1 ifFalse: [ ^ false ].
 ^ numerator even
%

category: 'Testing'
method:
odd

"Returns true if the receiver is an odd integer, false otherwise."

 denominator = 1 ifFalse: [ ^ false ].
 ^ numerator odd
%

category: 'Converting'
method:
asFraction

"Returns a Fraction that represents the receiver."

^ Fraction numerator: numerator denominator: denominator.
%

! fix 46790
category: 'Copying'
method:
postCopy
  ^ self immediateInvariant
%

