!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! File:   gsfile.gs
!
! Superclass Hierarchy:
!   GsFile, IO, Object.
!
!=========================================================================

! Gs64 v2.1, major edits to change semantics of isClient instVar to fix 35450.
!  35450 is caused by a open of a transient GsFile probing C state for isOpen
!  and objId reuse causing reuse of C state of a previous instance. This
! can happen only if an open transient GsFile was garbage collected without
!  having been explicitly closed prior to GC .
!  
! state 		     	isClient instVar values
! -----				----------------------
! committed pre-v2.1 instance   true or false
! temp , open 			true or false
! temp , closed                 0 or 1
! temp , closed, access error   2 or 3 
! committed v2.1              any of   true/false, 0/1, 2/3 
!
!  instance creation must produce "closed" state .
!  open of a transient instance must not probe C state if the
!    Smalltalk state is "closed" , rather it must delete previous
!    C state and create new C state.
!  close of any instance deletes C state .
!  close of a transient instance must change state to "closed"
!  attempt to access a closed transient instance must change state to
!   "closed, access error"

! Remove existing behavior from GsFile
set class GsFile
removeallmethods GsFile
removeallclassmethods GsFile

! edited for 45451
category: 'For Documentation Installation only'
classmethod:
installDocumentation
self comment:

'GsFile provides the means for creating and accessing files. These 
 files reside in the file system on either the machine that is running the
 current session''s Gem process (the server machine) or the machine that is
 running the client application (the client machine).  The files may be of any
 type, textual or binary, though separate protocol is provided for reading and
 writing these types of data.  File contents are in bytes and writing kinds of 
 String that require multiple bytes per code pint, the contents must be 
 explicitly encodedi before write, or written using nextPutAllUtf8:. 

 Beginning with Gs64 v3.0,  instances of GsFile automatically have
 their C state closed when the instance is garbage collected or 
 when a persistent instance drops out of memory.

                              Warning:
    Do not retain an instance of GsFile from one session to another.
    Instances of GsFile are intended to exist only within a given GemStone
    session.  GsFiles that are used across sessions always generate an error.

  Path arguments specifying a directory or file to open or create
  will be converted to Utf8 if they contain code points above 255 ,
  or are implemented as a MultiByteString . On Windows clients, for 
  compatibility with the OS operations, arguments are converted to Utf16.

Instance variables:
	
fileDescriptor -- SmallInteger
	
lineNumber -- SmallInteger

isClient -- a Boolean or SmallInteger,  with following stats and values
   committed pre-v2.1 instance   true or false
   temp , open                   true or false
   temp , closed                 0 or 1
   temp , closed, access error   2 or 3
   committed v2.1              any of   true/false, 0/1, 2/3

pathName -- A String that gives an absolute path name to the file.

mode -- A String that gives the file open mode for the file, as defined for the
 C-language open function.
' .
%

! ------------------- Class methods for GsFile
category: 'Drastic Measures'
classmethod:
closeAll

"Closes all open files on the client machine except stdin/stdout/stderr.
 Returns the receiver if successful, nil if not."

"make sure the sources file has been explicitly closed"

^ self classUserAction: #GsfCloseAll onClient: true
%

category: 'Drastic Measures'
classmethod:
closeAllOnServer

"Closes all open files on the server machine except stdin/stdout/stderr.
 Returns the receiver if successful, nil if not."

^ self classUserAction: #GsfCloseAll onClient: false
%

category: 'Garbage Collection'
classmethod:
finalizeAll

"Closes all open files on the client machine that were opened by a
 instance of GsFile but that are no longer being used by an instance
 of GsFile. The need for finalization is caused by instances being
 garbage collected before they are sent close.
 Returns nil if an error occurs.
 Returns the total number of GsFile instances that were not finalized
 because they were still in use."

System _generationScavenge.
^ self classUserAction: #GsfFinalizeAll onClient: true
%

category: 'Garbage Collection'
classmethod:
finalizeAllOnServer

"Closes all open files on the server machine that were opened by a
 instance of GsFile but that are no longer being used by an instance
 of GsFile. The need for finalization is caused by instances being
 garbage collected before they are sent close.
 Returns nil if an error occurs.
 Returns the total number of GsFile instances that were not finalized
 because they were still in use."

System _generationScavenge.
^ self classUserAction: #GsfFinalizeAll onClient: false
%

! fix 47325
category: 'Directory Operations'
classmethod:
contentsAndTypesOfDirectory: dirSpec onClient: onClient

"Returns an Array of objects describing the contents of the given directory.
 The argument bool indicates the location of the directory on the client or
 the server.  Successive pairs of elements of the Array are each the name of an
 entry, and a Boolean - true if the entry is a file, and false if not.
 The order of results is based on the file system.

 Sample:  { 'file.c'  true . 'subdir' . false . ... }

 Returns anArray if successful, nil if not."
| dirContents result |
onClient ifTrue:[
  dirContents := self contentsOfDirectory: dirSpec onClient: true .
  dirContents ifNil:[ ^ nil ].
  result := { }  .
  1 to: dirContents size do:[:n | | str |
    str := dirContents at: n .
    result add: str ;
           add: (self _fileKind: str onClient: true ) == 0  .
  ] .
] ifFalse:[  | unicodeEnabled |
  "use faster primitives available in VM for server file access"
  dirContents := self _contentsOfServerDirectory: dirSpec expandPath: true utf8Results: true .
  dirContents _isSmallInteger ifTrue:[ ^ nil ].
  result := { }  .
  unicodeEnabled := Unicode16 _unicodeCompareEnabled.
  1 to: dirContents size do:[:n | | elem aGsFileStat |
    result add: (elem := self _decodeUtfResult: (dirContents at: n) unicode: unicodeEnabled) .
    (aGsFileStat := self stat: elem isLstat: false) _isSmallInteger ifTrue:[ 
      result add: false "probably a symlink to a non-existant file"
    ] ifFalse:[
      result add: aGsFileStat isDirectory not
    ]
  ].
].
^ result
%

category: 'Private'
classmethod:
_fileKind: aPathName onClient: clientBool

"Returns a SmallInteger representing the kind of the file, or returns nil if an
 error occurs, in which case the class error buffer contains the error.

 The file kinds are enumerated as follows.  Symbolic links are reported as
 the type of the file pointed to.

    0. file 
    1. directory
    2. character device
    3. block device
    4. symbolic link (not applicable)
    5. other
    6. error
    7. FIFO
    8. stdin, stdout, or stderr
    9. stdin or stdout connected to a terminal (isatty(fileno(f) == 1)  "
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfFileKind onClient: clientBool with: path
%

! fix 47325
category: 'Directory Operations'
classmethod:
contentsOfDirectory: dirSpec onClient: clientBool

"Returns an Array describing the contents of the given directory.
 Elements of result are Unicode strings if the system is in Unicode Comparison
 Mode (String Configuration isInUnicodeComparsonMode = true), otherwise they are 
 traditional Strings.

 The argument bool indicates the location of the directory on the client or
 the server.  
 The ordering of the result is based on the file system implementation.
 Returns anArray if successful, nil if not."

| unicodeEnabled files uDir res |
unicodeEnabled := Unicode16 _unicodeCompareEnabled.
uDir := self _utfPath: dirSpec forClient: clientBool .
files :=  self classUserAction: #GsfDirectory onClient: clientBool
       with: uDir
       with: true.
files ifNil:[ ^ files ].
res := { } .
1 to: files size do:[:n |
  res add: (self _decodeUtfResult: (files at: n) unicode: unicodeEnabled) .
].
^ res
%

category: 'File Operations'
classmethod:
exists: aPathName

"Returns true if the given path points to a file on the client,
 false if not, and nil if an error occurs trying to find out."

^ self _exists: aPathName onClient: true
%

category: 'File Operations'
classmethod:
existsOnServer: aPathName

"Returns true if the given path points to a file, on the server,
 false if not, and nil if an error occurs trying to find out."

^ self _exists: aPathName onClient: false
%

category: 'Private'
classmethod:
_exists: aPathName onClient: clientBool

"Returns true if the given path points to a file, false if not, and nil if an
 error other than file-does-not-exist occurs trying to find out."

| result path |
path := self _utfPath: aPathName forClient: clientBool .
"GsfSize returns -1 for non-existant file , nil for other error ,
 >= 0 for a file that exists "
result := self classUserAction: #GsfSize onClient: clientBool with: path .
result == -1 ifTrue:[ ^ false "file does not exist" ].
result == nil ifTrue:[ ^ nil "some other error" ] .
^ true
%

category: 'Instance Creation'
classmethod:
open: aPathName mode: openMode 

"Creates an instance of the receiver and opens a file on the client machine.

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 Returns a GsFile if successful, nil if an error occurs."

^ self open: aPathName mode: openMode onClient: true
%

category: 'Instance Creation'
classmethod:
basicNew

"disallowed"

self shouldNotImplement: #basicNew
%
category: 'Private'
classmethod:
new

"disallowed"

self shouldNotImplement: #new
%

category: 'Private'
classmethod:
_basicNew

"creates an instance registered with VM for finalization of cData"

<primitive: 674>
self _primitiveFailed: #_basicNew
%
category: 'Private
method:
_newEmptyClientCData

"initialized cData for a client file, if cData not already
 initialized for a server file."

<primitive: 675>
self _primitiveFailed: #_newEmptyClientCData
%

! deleted GsFile(C)>>newWithFilePtr:pathname:mode:onClient:

category: 'Private'
method:
postCopy
	"Do cleanup on new copy. Ensure that it is closed."
	
super postCopy.
isClient := self isClient ifTrue:[ 1 ] ifFalse:[ 0 ]
%

category: 'Instance Creation'
classmethod:
open: aPathName mode: openMode  onClient: clientBool 

"Creates an instance of the receiver and opens a file on the client machine
 (if clientBool is true) or the server machine (if clientBool is false).

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 Returns a GsFile if successful, nil if an error occurs."

^self open: aPathName mode: openMode  onClient: clientBool withCompression: false
%

category: 'Instance Creation'
classmethod:
openOnServer: aPathName mode: openMode

"Creates an instance of the receiver and opens a file on the server machine.

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 Returns a GsFile if successful, nil if an error occurs."

^ self open: aPathName mode: openMode onClient: false
%

category: 'Instance Creation'
classmethod:
openAppend: aPathName

"Opens a text file on the client machine for writing.
 If the file does not exist it is created.
 The file position indicator is positioned at the end of the file
 before each write operation.
 Read operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'a' onClient: true
%

category: 'Instance Creation'
classmethod:
openRead: aPathName

"Opens an existing text file on the client machine for reading.
 Write operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'r' onClient: true
%

category: 'Instance Creation'
classmethod:
openUpdate: aPathName

"Opens an existing text file on the client machine for reading and writing.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'r+' onClient: true
%

category: 'Instance Creation'
classmethod:
openWrite: aPathName

"Opens a text file on the client machine for writing.
 If the file exists it is truncated to zero length.
 If the file does not exist it is created.
 Read operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'w' onClient: true
%

category: 'Instance Creation'
classmethod:
openAppendOnServer: aPathName

"Opens a text file on the server machine for writing.
 If the file does not exist it is created.
 The file position indicator is positioned at the end of the file
 before each write operation.
 Read operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'a' onClient: false
%

category: 'Instance Creation'
classmethod:
openReadOnServer: aPathName

"Opens an existing text file on the server machine for reading.
 Write operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'r' onClient: false
%

category: 'Instance Creation'
classmethod:
openUpdateOnServer: aPathName

"Opens an existing text file on the server machine for reading and writing.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'r+' onClient: false
%

category: 'Instance Creation'
classmethod:
openWriteOnServer: aPathName

"Opens a text file on the server machine for writing.
 If the file exists it is truncated to zero length.
 If the file does not exist it is created.
 Read operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'w' onClient: false
%

category: 'File Operations'
classmethod:
removeClientFile: aPathName

"Removes the named file from the client machine's file system.
 Returns the receiver if the file was deleted, nil if an error occurs."

^ self _removeFile: aPathName onClient: true .
%

category: 'File Operations'
classmethod:
removeServerFile: aPathName

"Removes the named file from the server machine.  Returns the receiver if the
 file was deleted, nil if an error occurs."

^ self _removeFile: aPathName onClient: false .
%

category: 'Private'
classmethod:
_removeFile: aPathName onClient: clientBool

"Returns receiver if removal succeeds.  Returns nil if an error occurs,
 in which case the class error buffer contains the error."
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfUnlink onClient: clientBool with: path
%

category: 'Directory Operations'
classmethod:
removeClientDirectory: aPathName

"Removes the named directory from the client machine's file system.
 Returns the receiver if the directory was deleted, nil if an error occurs.
 Note that this method will fail if the directory is not empty."

^ self _removeDirectory: aPathName onClient: true .
%

category: 'Directory Operations'
classmethod:
removeServerDirectory: aPathName

"Removes the named directory from the server machine.
 Returns the receiver if the directory was deleted, nil if an error occurs.
 Note that this method will fail if the directory is not empty."

^ self _removeDirectory: aPathName onClient: false .
%

category: 'Private'
classmethod:
_removeDirectory: aPathName onClient: clientBool

"Returns receiver if removal succeeds.  Returns nil if an error occurs,
 in which case the class error buffer contains the error."
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfRmdir onClient: clientBool with: path
%

category: 'Directory Operations'
classmethod:
isClientDirectory: aPathName
"Returns true if 'aPathName' names an existing directory on the client
 machine.
 Returns false if it is not a directory.
 Returns nil if an error occurs."

^ self _isDirectory: aPathName onClient: true .
%

category: 'Directory Operations'
classmethod:
isServerDirectory: aPathName
"Returns true if 'aPathName' names an existing directory on the server
 machine.
 Returns false if it is not a directory.
 Returns nil if an error occurs."

^ self _isDirectory: aPathName onClient: false .
%

category: 'Private'
classmethod:
_isDirectory: aPathName onClient: clientBool

"Returns true if 'aPathName' names an existing directory.
 Returns false if it is not a directory.
 Returns nil if an error occurs."
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfIsDir onClient: clientBool with: path
%

category: 'Directory Operations'
classmethod:
createClientDirectory: aPathName

"Creates the named directory on the client machine's file system.
 Returns the receiver if the directory was created, nil if an error occurs."

^ self _createDirectory: aPathName onClient: true mode: nil .
%

category: 'Directory Operations'
classmethod:
createServerDirectory: aPathName

"Creates the named directory on the server machine.
 Returns the receiver if the directory was created, nil if an error occurs."

^ self _createDirectory: aPathName onClient: false mode: nil .
%

! fixed 41298
classmethod:
createServerDirectory: aPathName mode: modeInt

"Creates the named directory on the server machine.
 Returns the receiver if the directory was created, nil if an error occurs.
 modeInt should be a SmallInteger, the value of which will be used
 as the second arg to mkdir() .  
 If modeInt == nil, a value of 8r770 will be used.
 Note that mkdir() takes the modeInt and removes bits set in the process' umask.
"

^ self _createDirectory: aPathName onClient: false mode: modeInt .
%

category: 'Private'
classmethod:
_contentsOfServerDirectory: aPathName expandPath: aBoolean utf8Results: u8Boolean

"Returns an Array, or a SmallInteger errno.  
 aPathName must be a String, Unicode string , Utf8 or Utf16  .
 The result is an Array of file names in the specified directory.
 If aBoolean is true, the result includes the expansion of aPathName
 as a prefix of each element of the result. 
 If u8Boolean is true, elements of result not representable with 7 bit code points
 will be Utf8 .
 "

^ self _primContentsOfServerDirectory: (self _utfPath: aPathName forClient: false)
         expandPath: aBoolean utf8Results: u8Boolean
%

category: 'Private'
classmethod:
_primContentsOfServerDirectory: aPathName expandPath: aBoolean utf8Results: u8Boolean

"Returns an Array, or a SmallInteger errno.  
 aPathName must be a String,  Utf8 or Utf16.
 The result is an Array of file names in the specified directory.
 If aBoolean is true, the result includes the expansion of aPathName
 as a prefix of each element of the result. 
 If u8Boolean is true, elements of result not representable with 7 bit code points
 will be Utf8 ."
<primitive: 431>
aPathName _validateClasses: { String . Utf8 }.
aBoolean _validateClass: Boolean .
u8Boolean _validateClass: Boolean .
self _primitiveFailed: #_primContentsOfServerDirectory: 
     args: { aPathName . aBoolean . u8Boolean}
%

category: 'Private'
classmethod:
_createDirectory: aPathName onClient: clientBool mode: modeInt

"Returns receiver if creation succeeds.  Returns nil if an error occurs,
 in which case the class error buffer contains the error."
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfMkdir onClient: clientBool with: path with: modeInt
%

category: 'File Operations'
classmethod:
lastModificationOfClientFile: aPathName

"Gets the date and time the named file on the client machine was last
 modified. Returns a DateTime if successful, nil if an error occurs."

^ self _fileModTime: aPathName onClient: true .
%

category: 'File Operations'
classmethod:
lastModificationOfServerFile: aPathName

"Gets the date and time the named file on the server machine was last 
 modified. Returns a DateTime if successful, nil if an error occurs."

^ self _fileModTime: aPathName onClient: false .
%

category: 'Accessing'
method:
lastModified
"Returns a DateTime that represents the last time the receiver's file
 was modified. Returns nil if an error occurs."

^ GsFile _fileModTime: (self pathName) onClient: isClient .
%

category: 'Private'
classmethod:
_fileModTime: aPathName onClient: clientBool

"Returns DateTime of files last modification time if successful.
 Returns nil if an error occurs,
 in which case the class error buffer contains the error."
| path |
path := self _utfPath: aPathName forClient: clientBool .
^ self classUserAction: #GsfModTime onClient: clientBool with: path
%

! documentation comments changed in v2.1 to match existing behavior
category: 'File Operations'
classmethod:
sizeOf: aPathName

"Returns the size in bytes of the given client file, false if
 file does not exist, or nil if any other error occurs."

^ self _sizeOf: aPathName onClient: true
%

! documentation comments changed in v2.1 to match existing behavior
category: 'File Operations'
classmethod:
sizeOfOnServer: aPathName

"Returns the size in bytes of the given server file, false if
 file does not exist, or nil if any other error occurs."

^ self _sizeOf: aPathName onClient: false
%

category: 'Private'
classmethod:
_sizeOf: aPathName onClient: clientBool

"Returns size of the file in bytes.  Returns nil if an error occurs,
 in which case the class error buffer contains the error."

| result path |
path := self _utfPath: aPathName forClient: clientBool .

"GsfSize returns -1 for non-existant file , nil for other error ,
 >= 0 for a file that exists "
result := self classUserAction: #GsfSize onClient: clientBool with: path .
result == nil ifTrue:[ ^ nil ] .
result == -1 ifTrue:[ ^ false "file does not exist" ].
^ result
%

category: 'Standard Files'
classmethod:
stdin

"Returns an instance of the receiver that is set up to read the standard input
 of the client process, or nil if an error occurs."

^self _getStdFile: 0 onClient: true
%
category: 'Standard Files'
classmethod:
stdinServer

"Returns an instance of the receiver that is set up to read the standard input
 of the server process, or nil if an error occurs."

^self _getStdFile: 0 onClient: false
%

! fixed 47283
method:
isTerminal

"Returns true if receiver is stdin or stdout and is connected to a terminal,
  i.e. in the client process , in C,  istty(fileno(f)) == 1 "

^ (pathName at: 1 equals:'std')
   and:[ (GsFile _fileKind: pathName onClient: self isClient) == 9 ]
%

category: 'Standard Files'
classmethod:
stdout

"Returns an instance of the receiver that is set up to write to the standard
 output of the client process, or nil if an error occurs."

^self _getStdFile: 1 onClient: true
%

! stdoutServer added for gemstone64 1.0
category: 'Standard Files'
classmethod:
stdoutServer

"Returns an instance of the receiver that is set up to write to the standard
 output of the server process, or nil if an error occurs."

^self _getStdFile: 1 onClient: false
%


category: 'Standard Files'
classmethod:
stderr

"Returns an instance of the receiver that is set up to write to the standard
 error output of the client process, or nil if an error occurs."

^self _getStdFile: 2 onClient: true
%

category: 'Private'
classmethod:
_getStdFile: stdId onClient: clientBool

"Returns an instance of the receiver that is set up to operate on a standard
 file of the client or nil if an error occurs.  stdId can be 0 (stdin),
 1 (stdout), or 2 (stderr)."

|result stdIdCache virtualId |

(stdIdCache := System __sessionStateAt: 18) ifNil:[
  stdIdCache := { nil . nil . nil . nil . nil . nil } .
  System __sessionStateAt: 18 put: stdIdCache.
].
clientBool ifTrue:[   virtualId := stdId + 1 ]
          ifFalse:[  virtualId := stdId + 4 ].
result := stdIdCache at: virtualId .
result == nil ifTrue: [ | status |
  "It has not yet been cached so create it"
  result := self _basicNew .
  result _newStdFile: stdId isClient: clientBool .
  "tell the client about it"
  status := result userAction: #GsfCreateStdFile onClient: clientBool with: stdId.
  status == nil ifFalse:[
    stdIdCache at: virtualId put: result.
    status == 1 ifTrue:[ result _newEmptyClientCData ].
  ] ifTrue: [ 
    result := nil. 
  ].
].

^ result
%

category: 'Error Reporting'
classmethod:
_setLastError: aString onClient: clientBool

self classUserAction: #GsfClassError onClient: clientBool with: aString
%

category: 'Error Reporting'
classmethod:
lastErrorString

"Returns the currently posted error string, for class operations on the
 client, or nil if no error has occurred.  Clears the error string."

^ self classUserAction: #GsfClassError onClient: true with: nil
%

category: 'Error Reporting'
classmethod:
serverErrorString

"Returns the currently posted error string, for class operations on the
 server, or nil if no error has occurred.  Clears the error string."

^ self classUserAction: #GsfClassError onClient: false with: nil
%

category: 'Private'
classmethod:
_clientVersionAt: verParamId

"Returns client process version information.  verInfoId should be one of the
 values in the global dictionary VersionParameterDict."

^ self classUserAction: #GsfVersionParam onClient: true with: verParamId
%

classmethod: 
_setEnvVariable: varName value: valueStr isClient: clientBool

"Both arguments must be either a String or a MultiByteString representable in 
 8000 bytes of Utf8 or Utf16 encoding.
 If valueStr is nil or of size 0 , implements unsetenv semantics.
"

| nam val res |
nam := self _utfPath: varName forClient: clientBool .
valueStr ifNotNil:[
  val := self _utfPath: valueStr forClient: clientBool .
  val size = 0 ifTrue:[ val := nil  "unsetenv"].
].
"check of NoGsFileOnServer or NoGsFileOnClient privilege is done by C code of prim 396"
res := self classUserAction: #GsfSetEnvVar onClient: clientBool with: nam with: val .
res ifNil:[ Error signal: 'GsfSetEnvVar failed' ].
^ self
%

category: 'Private'
classmethod:
_expandEnvVariable: varName isClient: clientBool

"Expands the environment variable named varName in either the GemBuilder for C
 or Gem process, returning a String. varName should be a kind of String.

 Returns nil if any of the following are true
    varName is not a byte format object.
    there is no environment variable defined with name  varName,
    the value of the environment variable is more than approximately 8000 bytes,
    clientBool is true and the size of varName exceeds approximately 8000 bytes."
| vn res |
vn := self _utfPath: varName forClient: clientBool .
res := self classUserAction: #GsfExpandEnvVar onClient: clientBool with: vn with: 0 .
^ self _decodeUtfResult: res unicode: nil .
%

classmethod
_decodeUtfResult: obj unicode: unicodeCompareEnabled
 | unic |
 unic := unicodeCompareEnabled ifNil:[ Unicode16 _unicodeCompareEnabled ].
 obj ifNil:[ ^ obj ].
 ((obj isKindOfClass: Utf8) or:[ obj isKindOfClass: Utf16]) ifTrue:[ 
   ^ unic ifTrue:[ obj decodeToUnicode] ifFalse:[ obj decodeToString].
 ].
 ^ unic ifTrue:[ obj asUnicodeString ] ifFalse:[ obj ].
%

classmethod:
_expandFilename: fName isClient: clientBool

"Expands fName using HostExpandEnvironmentVariable
 in either GCI or Gem process, returning a String. fName should be a kind of String.

 Returns nil if any of the following are true
    varName is not a byte format object.
    there is no environment variable defined with name  varName,
    the value of the environment variable is more than approximately 8000 bytes,
    clientBool is true and the size of varName exceeds approximately 8000 bytes."

| nam res |
nam := self _utfPath: fName forClient: clientBool .
res := self classUserAction: #GsfExpandEnvVar onClient: clientBool with: nam with: 1 .
^ self _decodeUtfResult: res unicode: nil
%

category: 'Private'
classmethod:
classUserAction: actionName onClient: onClient

"Executes a GsFile user action, passing it the following arguments:
    self
    any arguments to the primitive (see other uses of primitive 396)
 Maximum of 6 arguments (6 with: keywords) for this primitive.
 actionName must be a Symbol .
 Checks NoGsFileOnServer or NoGsFileOnClient privilege"

<primitive: 396>

^ self _primitiveFailed: #classUserAction:onClient: 
       args: { actionName . onClient }
%

category: 'Private'
classmethod:
classUserAction: actionName onClient: onClient with: arg1

"See GsFile | classUserAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #classUserAction:onClient:with:
       args: { actionName . onClient . arg1 }
%

category: 'Private'
classmethod:
classUserAction: actionName onClient: onClient with: arg1 with: arg2

"See GsFile | classUserAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #classUserAction:onClient:with:
       args: { actionName . onClient . arg1 . arg2 }
%

category: 'Private'
classmethod:
_log: string
  "Convenience method to print to stdout just like log: does."

  self stdout log: string
%

! ------------------- Instance methods for GsFile

category: 'Private'
method:
_open: aPath mode: aMode  onClient:  clientBool 

  "Initialize the receiver as specified by the arguments."

  "Does not actually open the underlying file."

  clientBool ifTrue:[ isClient := 3 ] ifFalse:[ isClient := 2 ].
  pathName := aPath .
  mode := aMode .
%

category: 'File Operations'
method:
fileSize

"Returns the size in bytes of the receiver, or nil if an error occurs.

 Note that the value returned is independent of the open mode used to
 create the receiver."

^ self class _sizeOf: pathName onClient: isClient
%

category: 'Writing'
method:
+ collection

"Writes the contents of the given collection to the receiver's file at
 the current position. The argument must be a Collection with byte format.  
 Returns a count of bytes added, or nil if an error occurs."

self deprecated: 'GsFile>>+ deprecated v3.2. Use the , method instead.'.
^  self nextPutAll: collection
%
! fixed 42534
category: 'Writing'
method:
, collection

"Writes the contents of the given collection to the receiver's file at
 the current position. The argument must be a Collection with byte format.  
 Returns a count of bytes added, or nil if an error occurs."

^  self nextPutAll: collection
%

category: 'Writing'
method:
add: char

"Writes the given Character to the receiver's file at the current position. 
 Returns true, or nil if an error occurs."

^ self nextPut: char
%

category: 'Writing'
method:
addAll: collection

"Writes the contents of the given collection to the receiver's file at
 the current position. The argument must be a Collection
 with byte format.  Returns a count of bytes added, or nil if an error occurs."

^ self nextPutAll: collection
%

category: 'Positioning'
method:
atEnd

"Returns true if the receiver is currently positioned at the end of its
 file, false if not, or nil if an error occurs."

<primitive: 724>  "primitive succeeds only if file is on server"
(isClient _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ ^ self _setNotOpenError ].
^ self userAction: #GsfAtEnd onClient: isClient
%

category: 'File Operations'
method:
close

"Closes the receiver's file.  Returns the receiver, or nil if an error occurs.
 If receiver is a standard file, has no effect and returns receiver."

 | res stdIdCache |
 self isOpen ifFalse: [^ self ].
 stdIdCache := System __sessionStateAt: 18.
 stdIdCache == nil ifFalse: [
   (stdIdCache includesIdentical: self) ifTrue:[
   fileDescriptor := -1 .
     ^ self
   ]
 ]. 
 fileDescriptor >= 0 ifTrue:[
   IO _forgetFileDescriptor: fileDescriptor
 ].
 fileDescriptor := -1 .
 res := self userAction: #GsfClose onClient: isClient . 
 res == nil ifFalse:[ self _setNotOpened ].
 ^ res
%

category: 'Reading'
!  edited to fix Trac788
method:
_contents: resultObj

"Read the contents of the receiver from the current
 position to the end of the file into resultObj.  
 Returns nil if an error occurs, otherwise returns resultObj."

| mySize myPosition |

mySize := self fileSize .
(mySize == nil or:[ mySize == 0]) ifTrue:[
  "a file not reporting size like stdin, or of size zero like /dev/zero"
  | buf |
  buf := resultObj class new .
  [ true ] whileTrue:[ | status |
    status := self _next:1024 into: buf .
    status ifNil:[ ^ resultObj ].
    resultObj addAll: buf
  ]
].
myPosition := self position .
(mySize + myPosition) == 0 ifTrue:[ ^ resultObj ]. "a zero length file"
myPosition == nil ifTrue:[ ^ nil ] .
(self next: ( mySize - myPosition ) into: resultObj ) ifNil:[ ^ nil ].
^ resultObj
%

category: 'Private'
method:
_isBinary

"Returns true if receiver is a binary file. Otherwise returns false."
| m |
(m := mode) ifNil: [ ^ false].
^ m includesValue: $b
%

method:
_isReadable
  
"Returns true if receiver is open for read, otherwise returns false."
| m |
(m := mode) ifNil:[ ^ false].
^ (m includesValue: $r) or:[ m includesValue: $+ ] 
% 


category: 'Writing'
method:
cr

"If the receiver is a text file then writes an end-of-line sequence to it.
 If the receiver is a binary file then writes a carriage return to it.
 Returns a count of bytes added, or nil if an error occurs."

(self _isBinary) ifTrue: [ | isCl |
  ((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ ^ self _setNotOpenError ].
  ^ self userAction: #GsfPutC onClient: isCl with: 13 "ASCII carriage return"
] ifFalse: [
  "ANSI-C text files will write eoln when they see a single lf character."
  ^ self lf
].
%

category: 'Error Reporting'
method:
lastErrorString

"Returns the currently posted error string, or nil if no error has occurred.
 Clears the error string and error code.  
 If the last error was a failure during reopen of a closed file,
 will retrieve and clear the error string from the class."

 isClient _isSmallInteger ifFalse:[
   "a pre v2.1 committed or open instance, get result from C"
   ^ self userAction: #GsfErrorString onClient: isClient
 ].
 self isCommitted ifTrue:[
   "v2.1 committed instance in any state, get result from C"
   ^ self userAction: #GsfErrorString onClient: isClient
 ].
 "at this point we have a v2.1 transient instance."
 isClient >= 2 ifTrue:[
   "change state from 'closed, access error' to 'closed' and return err string."
   isClient := isClient - 2 .
   ^ 'attempt to access a GsFile that was never opened or has been closed'.
 ].
 "get class' error string for last unsuccessful open/reopen."
 ^ self class lastErrorString .
%

category: 'Error Reporting'
method:
lastErrorCode

"Returns the currently posted error code, or zero if no error has occurred.
 A result of -1  means 
   'attempt to access a GsFile that was never opened or has been closed' .
 Does not clear the error code or error string.

 If the last error was a failure during reopen of a closed file, 
 this method returns zero;  you must use lastErrorString to get information 
 on such reopen  failures.   This situation applies to errors produce by
 sending  open   or   open:mode:    to an instance of GsFile. "

 isClient _isSmallInteger ifFalse:[
   "a pre v2.1 committed or open instance, get result from C"
   ^ self userAction: #GsfErrorCode onClient: isClient 
 ].
 self isCommitted ifTrue:[
   "v2.1 committed instance, get result from C"
   ^ self userAction: #GsfErrorCode onClient: isClient 
 ].
 "at this point we have a v2.1 transient instance."
 isClient >= 2 ifTrue:[ 
   ^ -1 "attempt to access a GsFile that was never opened or has been closed."
 ].
 ^ 0
%

category: 'Writing'
method:
ff

"Writes a form-feed (page break) to the receiver's file.
 Returns true, or nil if an error occurs."

^ self nextPut:  Character newPage
%

category: 'File Operations'
method:
flush

"Flushes all written bytes to the file.  Returns the receiver, or nil if an
 error occurs. On Unix, calls fflush() . "
| isCl |
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
   ^ self _setNotOpenError 
].
^ self userAction: #GsfFlush onClient: isCl
%

category: 'File Operations'
method:
sync

"Flushes all written bytes to the file's storage device.  Returns the receiver, or nil if an
 error occurs. On Unix, calls fflush() followed by fsync() . "
| isCl |
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
   ^ self _setNotOpenError 
].
^ self userAction: #GsfSync onClient: isCl
%

category: 'File Operations'
method: 
head: lineCount

"Returns a String containing the first lineCount lines from the
 receiver's file, or nil if an error occurs."

| result line |

self position: 0.
result := String new.
lineCount timesRepeat: [
  line := self nextLine.
  line ~~ nil ifTrue: [
    result addAll: line.
  ]
  ifFalse: [
    ^result
  ]
].
^result
%

category: 'Testing'
method:
isExternal

"Is the source for this stream is external to GemStone Smalltalk."

^true
%

category: 'Testing'
method:
isClient

"Returns true if the receiver's file is a client file, or nil if an error
 occurs."
| isCl |
(isCl := isClient) _isSmallInteger ifTrue:[ ^ (isCl bitAnd:1) == 1 ].
^ isCl  
%

category: 'Private'
method:
_setNotOpenError

  "set state of a transient instance to 'closed, access error' "

  self isCommitted ifFalse:[
    isClient _isSmallInteger ifFalse:[
      self error:'invalid state' .
      self _uncontinuableError 
    ].
    isClient < 2 ifTrue:[ isClient := isClient + 2 ].
  ].
  ^ nil
%

category: 'Private'
method:
_setIsOpen

  "for a transient instance, change state to 'open' "
  self isCommitted ifFalse:[
    isClient := (isClient bitAnd: 1) == 1  "convert 0/1, 2/3 to false/true"
  ] 
%

category: 'Private'
method:
_setNotOpened

  "for a transient instance,  change state to 'closed' "
  self isCommitted ifFalse:[
    isClient _isSmallInteger ifFalse:[
       isClient ifTrue:[ isClient:= 1 ] ifFalse:[ isClient:=0 ].
    ].
  ].
%

category: 'Testing'
method:
isOpen

"Returns true if the receiver's file is open, false otherwise"
| isCl |
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
  ^ false 
].
isClient ifNil:[ ^ false "an instance created with _basicNew" ].
^ self userAction: #GsfIsOpen onClient: isCl
%

category: 'Writing'
method:
lf

"Append an end-of-line to the receiver.
 Returns true if successful, nil if an error occurred."

^ self nextPut: 10"ASCII line-feed Character"
%
category: 'Writing'
method:
space

"Append an space character to the receiver.
 Returns true if successful, nil if an error occurred."

^ self nextPut: 32 " $   codePoint "
%
category: 'Writing'
method:
tab

"Append a tab to the receiver.
 Returns true if successful, nil if an error occurred."

^ self nextPut: 9  "ASCII tab"
%


category: 'Writing'
method:
log: string

"Writes the contents of the given collection to the receiver's file at the
 current position.  Appends a newline to the file if the string does not end
 with one.  The argument must be a kind of Collection with byte format.

 Returns the receiver if successful; returns nil otherwise."

| lf result |

result := self nextPutAll: string.
(result ~~ nil and:[ result > 0]) ifTrue:[
  lf := Character lf.
  string last == lf ifFalse: [ result := self nextPut: lf ]
  ].
result == nil ifTrue: [ ^nil ].
self flush.
^ self
%

! gemstone64, added gciLogServer: , gciLogClient:
category: 'Writing'
classmethod:
gciLogServer: aString

"Passes the contents of the given String to the GciGetLogger() function
 of the server process.  
 In a gem process, the default logger writes to stdout, i.e. to the gem
 log file.  In the linked login of a topaz -l process,
 the default logger writes to the current output file as controlled by
 topaz OUTPUT PUSH statements, else to stdout.
 If the collection has size > 0 and does not end in an Ascii LineFeed, 
 a LineFeed is appended to the bytes passed to the logging function."

<primitive: 1003>
"primitive accepts String , ByteArray, Utf8 args"
aString _validateClass: MultiByteString . 
[ self gciLogServer: aString encodeAsUTF8 ] onException: Error do:[:ex |
  self classUserAction: #GsfGciLog  onClient: false with: aString
].
^ self
%

category: 'Writing'
classmethod:
gciLogClient: aString

"Passes the contents of the given collection to the GciGetLogger() function
 of the client process.

 If the client is a topaz process, the default logger writes to the 
 current output file as controlled by topaz OUTPUT PUSH statements, 
 else to stdout.

 If the collection has size > 0 and does not end in an Ascii LineFeed,
 a LineFeed is appended to the bytes passed to the logging function.

 A check for client process disconnected is made at start of the GsfGciLog
 useraction invocation, and if the client has disconnected, this invocation
 will behave as if it were GsFile(C)>>gciLogServer: .
"

^ self classUserAction: #GsfGciLog  onClient: true with: aString
%

category: 'Writing'
classmethod:
gciLog: aString onClient: clientBool

"Passes the contents of the given collection to a GciGetLogger() function.
 The clientBool must be a Boolean , true means use GciGetLogger() on
 the client process, false means server process .
 If the collection has size > 0 and does not end in an Ascii LineFeed, 
 a LineFeed is appended to the bytes passed to the logging function.

 If clientBool==true,
 a check for client process disconnected is made at start of the GsfGciLog
 useraction invocation, and if the client has disconnected, this invocation
 will behave as if it were GsFile(C)>>gciLogServer: .
"

^ clientBool ifTrue:[ self gciLogClient: aString ] 
	    ifFalse:[ self gciLogServer: aString]
%


category: 'Accessing'
method:
mode

"Returns the access mode of the receiver's file."

^mode
%

category: 'Accessing'
method:
name

"Returns the receiver's file path name."

^ pathName
%

category: 'Reading'
method:
next

"Returns a Character representing the next byte from the receiver's file.
 Returns nil if an error occurs, or if receiver is at EOF."

<primitive: 1014>
"primitive fails if receiver not an open server file, or if EINTR occurs"
| isCl res |
((isCl :=isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
  ^ self _setNotOpenError 
].
[ 
  res := self userAction: #GsfGetC onClient: isCl .
  res _isSmallInteger   "repeat loop if EINTR , to allow ctl-C detection"
] untilFalse .
^ res
%

category: 'Reading'
method:
next: numberOfBytes

"Returns a String containing the next numberOfBytes Characters from the
 receiver's file, or nil if an error occurs."

| result count |
result := String new.
count := self _read: numberOfBytes into: result .
(count == 0 or:[ count == nil]) ifTrue:[ ^ nil ].
^ result
%

method:
_next: numberOfBytes into: resultObj

"Returns a String containing the next numberOfBytes Characters from the
 receiver's file, or nil if an error occurs."

| count |
count := self _read: numberOfBytes into: resultObj .
(count == 0 or:[ count == nil]) ifTrue:[ ^ nil ].
^ resultObj
%

category: 'Reading'
method:
read: numberOfBytes into: byteObj

"Reads up to the given number of bytes into the given byte object (for
 example, a String) starting at index 1.
 Returns the number of bytes read, or nil if an error occurs,
 or 0 if EOF on the receiver.  
 The destination object is grown as needed, but is not shrunk. "

^ self _read: numberOfBytes into: byteObj 
%

category: 'Reading'
method:
_read: numberOfBytes into: byteObj 

"Reads up to the given number of bytes into the given byte object
 starting at index 1.   byteObj must be a String or ByteArray .
 Returns the number of bytes read, or nil if an error occurs,
 or 0 if EOF on the receiver.
 The destination object is grown as needed, but is not shrunk"

<primitive: 874>  
"primitive succeeds only if file is open on server, and no EINTR occurred"
^ self _read: numberOfBytes ofSize: 1 into: byteObj .  "client file or EINTR"
%

category: 'Reading'
method:
next: amount byteStringsInto: byteObj

"Reads bytes written by printBytes: into the given byte object.
 Returns count of bytes read, or nil if an error occurs."
| isCl |
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
   ^ self _setNotOpenError 
].
^ self userAction: #GsfLoadByteStrings onClient: isCl with: byteObj with: amount
%

category: 'Reading'
method:
next: numberOfBytes into: aByteObject

"Reads the next numberOfBytes into the given collection object. The
 object's size is truncated to the amount of data actually read.
 Returns a count of bytes read, or nil if EOF or an error occurs."

| cnt |
cnt := self _read: numberOfBytes into: aByteObject .
cnt == 0 ifTrue:[ ^ nil ].
cnt ifNotNil:[
  cnt < aByteObject size ifTrue:[ aByteObject size: cnt ].
].
^cnt
%

category: 'Reading'
method:
_next: numberOfBytes basicInto: aByteObject

"GsFile doesn't distinguish Strings from EUCStrings, so no reimplementation
 needed."

^ self next: numberOfBytes into: aByteObject
%

category: 'Reading'
method:
next: numberOfItems ofSize: bytesPerItem into: byteObj

"Reads bytes for the next numberOfItems of the given bytesPerItem into 
 the given collection object. The object's size is truncated to the 
 amount of data actually read.  bytesPerItem must between 1 and 4096 inclusive.

 Returns a count of bytes read, or nil if an error occurs."
| count |
count := self _read: numberOfItems ofSize: bytesPerItem into: byteObj .
(count == 0 or:[ count == nil]) ifTrue:[ ^ nil ].
count < byteObj size ifTrue:[ byteObj size: count ].
^ count
%

category: 'Reading'
method:
_read: numberOfItems ofSize: bytesPerItem into: byteObj

"Reads bytes for the next numberOfItems of the given bytesPerItem into
 the given collection object. The object's size is truncated to the
 amount of data actually read.  bytesPerItem must between 1 and 4096 inclusive.

 Returns a count of bytes read, 0 if hit EOF, or nil if an error occurs."
| count isCl |
( (isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
   ^ self _setNotOpenError 
].
[ count := self userAction: #GsfRead onClient: isCl
       with: byteObj with: bytesPerItem with: numberOfItems .
  count == -2 
] whileTrue . "loop to handle EINTR and soft break"
^ count
%


category: 'Reading'
method:
nextByte

"Returns the next byte (integer) from the receiver's file.
 Returns nil if an error occurs, or if receiver is at EOF."

| c |
c := self next .
c ifNil:[ ^ nil ].
^ c codePoint 
%

category: 'Reading'
method:
nextLine

"Returns a String containing the next line from the receiver's file. The String
 will be terminated with a newline, unless the end of file is reached and there
 is no line terminator.  If receiver is positioned at the end of the file, or
 a read error occurs, returns nil. 

 There is no limit on line size."

^ self nextLineTo: Character lfValue prompt: nil 
%

category: 'Reading'
method:
nextLineToChar: eolCharacter

^ self nextLineTo: eolCharacter codePoint prompt: nil 
%

category: 'Reading'
method:
_nextLineTo: eolValue prompt: promptString

"Reads the next line from the receiver's file into a new String .

 eolValue is either a SmallInteger 0..255, or a String with
 size >= 1 and <= 128 .

 The String will be terminated with the codePoint per eolValue
 unless the end of file is reached and there is no line terminator.  
 Returns the new String, or nil if an error occurs.
 Result is nil if file is at EOF.

 If promptString is non-nil, and receiver is stdin, and 
   isatty(fileno(stdin)) == 1 , then the topaz line editor will be used
 to read the input
 Returns -2 if EINTR occurred."

<primitive: 723>  "primitive succeeds for server file"
| isCl |
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
  ^ self _setNotOpenError 
].
^ self userAction: #GsfGetLine onClient: isCl with: nil
                with: 1 with: eolValue with: promptString.
%

category: 'Reading'
method:
nextLineTo: eolValue prompt: promptString
| res |
[
  "if receiver is stdin, GsfGetLine will return one of
    nil (for EOF), a String, or a SmallInteger (if interrupted by a ctl-C) .
   otherwise GsfGetLine returns nil or a String .
   By rerunning this loop if a SmallInteger is obtained, we give a chance
   for the SIGINT handler to signal a soft-break error
  "
  res := self _nextLineTo: eolValue prompt: promptString .
  res _isSmallInteger 
] untilFalse .
^ res 
%

method:
nextLineTo: eolValue
  ^ self nextLineTo: eolValue prompt: nil
%

category: 'Reading'
method:
nextLineInto: str startingAt: pos

"Deprecated,  new code should use  nextLineTo: .
 Reads the next line from the receiver's file into the given collection object,
 starting at the given position in the collection. The collection will be
 terminated with a newline, unless the end of file is reached and there is no
 line terminator.  If the receiver is positioned at the end of the file, nothing
 is written.  Returns a count of bytes read, or nil if an error occurs."

"There is no limit on line size."

| res isCl |
self deprecated: 'GsFile>>nextLineInto:startingAt: deprecated in v3.2, use nextLineTo: instead'.
((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
   ^ self _setNotOpenError 
].
res := self userAction: #GsfGetLine onClient: isCl with: str with: pos 
		with: Character lfValue with: nil .
res ~~ nil ifTrue:[ ^ res size - pos + 1 ].
^ res
%

category: 'Writing'
method:
nextPut: aByte

"Writes the given byte to the receiver's file at the current position.
 aByte must be a Character or a SmallInteger in the range 0..255.

 If aByte is a SmallInteger, it will be interpreted logically as

 Character withValue: aByte 
   
 Returns true, or nil if an error occurs."

<primitive: 1011>
"handle file not open on server"
| isCl |
((isCl :=isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
  ^ self _setNotOpenError 
].
^ self userAction: #GsfPutC onClient: isCl with: aByte
%

category: 'Writing'
method:
nextPutAll: collection

"Writes the contents of the given collection to the receiver's file at the
 current position. The argument must be a kind of Collection with byte
 format.  Returns a count of bytes added, or nil if an error occurs."

^ self write: (collection _basicSize) from: collection
%

! fixed 44067
category: 'Writing'
method:
nextPutAllUtf8: aStringOrCharacter

"Writes the contents of the given String, MultiByteString or Utf8 to the 
 receiver's file using UTF8 encoding .
 Returns a count of bytes added, or nil if an error occurs."

| utf |
aStringOrCharacter class == Character ifTrue:[
  utf := Unicode16 with: aStringOrCharacter.
  utf := utf encodeAsUTF8 .
] ifFalse:[
  utf := aStringOrCharacter encodeAsUTF8 .
].
^ self write: utf size from: utf .
%

! fix 47974
method:
nextPutAsUtf8: aString
  "Will be deprecated.   Use nextPutAllUtf8: "
  ^ self nextPutAllUtf8: aString
%

category: 'Writing'
method:
nextPutAllBytes: collection

"Writes the byte contents of the given collection to the receiver's file at the
 current position.  The argument must be a kind of Collection with byte
 format.  Returns a count of bytes added, or nil if an error occurs."

^ self nextPutAll: collection
%

category: 'Private'
method:
_nextPutAllBytes: collection
  "used by PassiveObject"
| str |
(str := String new) addAllBytes: collection . "str is big-endian"
^ self nextPutAll: str .
%

category: 'File Operations'
method:
open

"If the receiver is not open, open it using the existing mode.
 Returns the receiver, or nil if an error occurs.
 If an error occurs,  there is no error information associated with
 the instance; you must use    GsFile lastErrorString   to retrieve the error. "

^self _openUsingGzip: false
%


category: 'File Operations'
method:
open: aPathName mode: openMode

"Opens the receiver's file with the given mode.  If the file is already open,
 it is closed and reopened.

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 Returns the receiver if successful, or nil if an error occurs.
 If an error occurs,  there is no error information associated with
 the instance; you must use   GsFile lastErrorString to retrieve the error. "

self isOpen ifTrue:[ 
  self close  .
  self isOpen ifTrue:[ 
    ^ self "a standard file that cannot be closed"
  ].
].
mode := openMode.
pathName := aPathName .
^self open
%

category: 'Accessing'
method:
pathName

"Returns the receiver's file path name."

^pathName
%

category: 'Private'
method: 
_peek: numBytes

"Returns nil for error on server file, false if EINTR occurred,
 true if receiver is a client file.
 Primitive fails for invalid numBytes argument."
<primitive: 1013>
(numBytes == 1 or:[ numBytes == 2]) ifFalse:[ 
   ArgumentError signal:'invalid number of bytes for peek'].
^ self _primitiveFailed: #_peek: args: { numBytes }
%

category: 'Reading'
method:
peek

"Returns the a Character representing the next byte in the receiver's file, 
 without advancing the current pointer.  
 Returns nil if an error occurs, or peek hits EOF."

| result |
[
  (result := self _peek: 1 ) == false 
] whileTrue . "loop to handle EINTR"
result == true ifTrue:[
  | pos |  "handle client file"
  pos := self position.
  result := self next.
  self position: pos.
].
^ result
%

category: 'Reading'
method:
peek2

"Returns the next byte plus one in the receiver's file, without advancing the
 current pointer.  Returns nil if an error occurs, or peek hits EOF."
| result |
[
  (result := self _peek: 2 ) == false
] whileTrue . "loop to handle EINTR"
result == true ifTrue:[
  | pos |  "handle client file"
  pos := self position.
  result := self next; next.
  self position: pos.
].
^result
%

category: 'Positioning'
method:
position
  
"Returns the current position of the receiver's file, 
 or nil if an error occurs.  Result is zero based."
 ^ (self _zeroArgPrim: 2) ifNil:[
   | isCl |
   ((isCl := isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
     ^ self _setNotOpenError 
   ].
   ^ self userAction: #GsfPosition onClient: isCl
 ].
%

category: 'Positioning'
method:
positionA
	"For compatibility with PositionableStream"

	^self position.
%

category: 'Positioning'
method:
positionL
	"For compatibility with PositionableStream"

	^self position.
%

category: 'Positioning'
method:
positionW
	"For compatibility with PositionableStream"

	^self position.
%

category: 'Positioning'
method:
position: offset

"Changes the receiver's position in its file to the given offset, which may be
 negative or zero.  Returns the new position, or nil if an error occurs."

^ self _seekTo: offset opcode: 0 .
%

category: 'Positioning'
method:
positionA: offset
	"For compatibility with PositionableStream"

	^self position: offset.
%

category: 'Positioning'
method:
positionL: offset
	"For compatibility with PositionableStream"

	^self position: offset.
%

category: 'Positioning'
method:
positionW: offset
	"For compatibility with PositionableStream"

	^self position: offset.
%

category: 'Writing'
method:
printBytes: byteObj

"Prints the bytes from the given byte object in decimal notation with
 line breaks to keep output lines from being too long.

 Returns the receiver, or nil if an error occurs.

 See also the method next:byteStringsInto:."

(isClient _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ ^ self _setNotOpenError ].
^ self userAction: #GsfPrintBytes onClient: isClient with: byteObj
%

category: 'Positioning'
method:
rewind

"Repositions the receiver's file to the beginning.  Returns 0, or nil if an
 error occurs."

self position: 0
%

category: 'Positioning'
method:
seekFromBeginning: offset

"Moves the receiver's position in its file to the given offset from the
 beginning of the file, which may be positive or zero, but not negative.
 Returns the new position, or nil if an error occurs."

^ self _seekTo: offset opcode: 0
%

category: 'Positioning'
method:
seekFromCurrent: offset

"Changes the receiver's position in its file by the given offset, which
 may be negative or zero.  Returns the new position, or nil if an error occurs."

^ self _seekTo: offset opcode: 1 
%

category: 'Positioning'
method:
seekFromEnd: offset

"Moves the receiver's position in its file to the given offset from the
 end of the file, which may be negative or zero, but not positive.
 Returns the new position, or nil if an error occurs.  Not supported
 for compressed files."

^ self _seekTo: offset opcode: 2
%

category: 'Reading'
method:
skip: count

"Changes the receiver's position in its file by the given offset, which may be
 zero, or negative.  Returns the new position, or nil if an error occurs."

| pos |
pos := self position.
^ self position: pos + count
%

category: 'Private'
method:
_newStdFile: stdId isClient: clientBool

  "Initialize the receiver to represent a standard file."

  clientBool ifTrue:[ 
    isClient := true .
    fileDescriptor := -1 .
  ] ifFalse:[ 
    isClient := false .
    fileDescriptor := stdId .
  ].
  stdId == 0 ifTrue: [mode := 'r'. pathName := 'stdin']
          ifFalse:[mode := 'w'.
                   stdId == 1 ifTrue: [pathName := 'stdout']
                             ifFalse:[pathName := 'stderr'].
                  ].

%

category: 'Writing'
method:
write: amount from: byteObj

"Write the given number of bytes from the given byte object.  Returns the
 number of bytes written, or nil if an error occurs."

<primitive: 875>  "primitive succeeds only if file is open on server"
| isCl |
((isCl:=isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
  ^ self _setNotOpenError 
].
^ self userAction: #GsfWrite onClient: isCl
       with: byteObj with: 1 with: amount
%

category: 'Writing'
method:
write: byteObj itemCount: itemCount ofSize: bytesPerItem

"Writes itemCount items of size bytesPerItem from the given byte object to
 the receiver's file.  bytesPerItem must be between 1 and 4096 inclusive.
 Returns a count of bytes written, or nil if an error occurs."

^ self write: itemCount * bytesPerItem from: byteObj
%

category: 'Private'
method:
_seekTo: offset opcode: seekKind

"seekKind  function
   0        seek to  start of file + offset
   1        seek  offset from current position
   2        seek to  end of file - offset

 Returns the new position, or nil if an error occurs."
<primitive: 1010>
"primitive failed, handle client file"
| isCl |
((isCl:=isClient) _isSmallInteger and:[ self isCommitted == false]) ifTrue:[ 
  ^ self _setNotOpenError 
].
^ self userAction: #GsfSeek onClient: isCl with: offset with: seekKind
%

category: 'Comparing'
method:
hash

"Returns a SmallInteger related to the value of the receiver.  If two instances
 of GsFile are equal (as compared by the = method), then they must have the same
 hash value.
 Gemstone 64 v3.0, hash algorithm has changed for GsFile
 "

^ pathName hash
%

category: 'Comparing'
method:
= aFile

"Returns true if the receiver and aFile represent the same file system file.
 Returns false otherwise.

 Result is always false if either receiver or argument are not open."

(self == aFile) ifTrue:[ ^ true].
aFile == nil ifTrue:[ ^ false].
(aFile isKindOf: self class) ifFalse:[ ^ false].
(self isOpen and:[ aFile isOpen]) ifTrue:[
   self isClient = aFile isClient  ifTrue:[
    pathName = aFile pathName ifFalse:[
      ^ fileDescriptor = aFile fileDescriptor
    ].
    ^ true
  ].
].
^ false
%

! GsFile>>id  deleted
!   result was ByteArray from GciPointerToByteArray, holding a C  FILE*

category: 'Comparing'
method:
~= aFile

"Returns false if the receiver and aFile represent the same file system file.
 Returns true otherwise."

^ (self = aFile) not
%

category: 'Private'
method:
userAction: actionName onClient: onClient

"Executes a GsFile user action, passing it the following arguments:
    self
    any arguments to the primitive (see other uses of primitive 396)
 Maximum of 6 arguments (6 with: keywords) for this primitive.
 actionName must be a Symbol "

<primitive: 396>

^ self _primitiveFailed: #userAction:onClient: args: { actionName . onClient }
%

category: 'Private'
method:
userAction: actionName onClient: onClient with: arg1

"See GsFile | userAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #userAction:onClient:with:
       args: { actionName . onClient . arg1 }
%

category: 'Private'
method:
userAction: actionName onClient: onClient with: arg1 with: arg2

"See GsFile | userAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #userAction:onClient:with:with:
       args: { actionName . onClient . arg1 . arg2 }
%

category: 'Private'
method:
userAction: actionName onClient: onClient with: arg1 with: arg2 with: arg3

"See GsFile | userAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #userAction:onClient:with:with:with:
       args: { actionName . onClient . arg1 . arg2 . arg3 }
%

category: 'Private'
method:
userAction: actionName onClient: onClient with: arg1 with: arg2 with: arg3 with: arg4

"See GsFile | userAction:onClient: for documentation."

<primitive: 396>

^ self _primitiveFailed: #userAction:onClient:with:with:with:with:
       args: { actionName . onClient . arg1 . arg2 . arg3 . arg4 }
%


category: 'Compressed File Operations'
method:
gzOpen: aPathName mode: openMode

"Opens the receiver's file with the given mode.  If the file is already open,
 it is closed and reopened.

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 Returns the receiver if successful, or nil if an error occurs.
 If an error occurs,  there is no error information associated with
 the instance; you must use   GsFile lastErrorString to retrieve the error. "

self isOpen ifTrue:[ self close ].
mode := openMode.
pathName := aPathName.
^self gzOpen
%

category: 'Compressed File Operations'
method:
gzOpen

"If the receiver is not open, open it using the existing mode.
 Returns the receiver, or nil if an error occurs.
 If an error occurs,  there is no error information associated with
 the instance; you must use    GsFile lastErrorString   to retrieve the error. "

^self _openUsingGzip: true
%

category: 'Compressed File Operations'
method:
_openUsingGzip: aBoolean

"If the receiver is not open, open it using the existing mode.
 If aBoolean is true, the gzip function gzopen() is used to open
 the file.  gzopen() can be used to open an existing file which was
 not compressed with gzip.  In that case, the file will be read
 without decompression.  

 Returns the receiver, or nil if an error occurs.
 If an error occurs,  there is no error information associated with
 the instance; you must use:
     GsFile lastErrorString
 to retrieve the error. "


| res path |
(pathName == nil or:[mode == nil]) ifTrue: [ ^nil ].
(self isOpen) ifTrue:[ ^self ].
mode stringCharSize == 1 ifFalse:[
   ArgumentError signal: 'mode must be a String or Unicode7'. ^nil.
].
path := GsFile _utfPath: pathName forClient: isClient .
res := self userAction: #GsfOpen onClient: isClient 
	         with: path with: mode with: aBoolean.
res ifNotNil:[
  self _setIsOpen .
  res == 1 ifTrue:[ self _newEmptyClientCData ].
  self isClient ifTrue:[
    fileDescriptor := -1 .
  ].
  fileDescriptor := self userAction: #GsfGetFileDesc onClient: false .
  " maglev only ifFalse:[
    IO _rememberFileDescriptor: fileDescriptor obj: self . 
  ]."
  ^ self .
].
^ res
%

classmethod:
_utfPath: aPath forClient: clientBool
  "Returns a String or Unicode7 with 7bit characters; or a Utf8 or Utf16"
  | isClient |
  (clientBool == false or:[ clientBool == 0 or:[ clientBool == 2]]) ifTrue:[
    isClient := false .
  ] ifFalse:[
    (clientBool == true or:[ clientBool == 1 or:[ clientBool == 3]]) ifTrue:[
      isClient := true .
    ] ifFalse:[
      ArgumentError signal:'invalid clientBool argument, a', clientBool class name
    ]
  ].
  (isClient and:[ self gciClientIsWindows]) ifTrue:[
    ^ aPath encodeAsUTF16  "file operation executing in Windows client"
  ].
  aPath _stringCharSize > 0 ifTrue:[
    aPath _asUnicode7 ifNotNil:[:p |
      "all chars are 7 bit"
      aPath isUnicodeString ifFalse:[ ^ String withAll: p ].
    ].
  ].
  ^ aPath encodeAsUTF8  "file operation on Unix client or in gem process"
%


category: 'Instance Creation (compression)'
classmethod:
open: aPathName mode: openMode  onClient: clientBool withCompression: compBool

"Creates an instance of the receiver and opens a file on the client machine
 (if clientBool is true) or the server machine (if clientBool is false).

 The openMode argument must be a String that that is equal to one of the
 following: 'r', 'w', 'a', 'r+', 'w+', 'a+', 'rb', 'wb', 'ab', 'r+b', 'w+b',
 'a+b', 'rb+', 'wb+', 'ab+'.  The mode has the same meaning as it does for the
 C library function, fopen().

 The classmethod GsFile>>validFileModes may be used to generate a list
 of all valid file modes.

 If compBool is true, the file is opened using gzip gzopen() call rather than
 fopen().  In this case, the file is assumed to be a compressed file, although
 gzopen() will also succeed when used to open uncompressed files.  When reading
 a compressed file, it is automatically decompressed as it is read.  Similarly,
 data written to a compressed file is automatically compressed before it is 
 written to the file.

 For compressed files, all open mode arguments listed above that do not contain
 a plus sign ('+') are valid.  In addition, any valid write mode argument 
 (those contaning 'a' or 'w') may have a '1' or '9' appended.   
 Appending a '1' means compress using the best possible speed while appending
  a '9' means compress to attain the best possible compression.

 The classmethod GsFile>>validFileModesForCompression may be used to 
 generate a list of all valid file modes for compressed files.

 Returns a GsFile if successful, nil if an error occurs."

| inst |
inst := self _basicNew.
inst _open: aPathName mode: openMode onClient: clientBool.
^ compBool ifTrue:[inst gzOpen] ifFalse:[inst open]
%

category: 'File Modes'
classmethod:
validFileModes

"Answers an array of strings, each of which is a valid mode for
 opening files without compression."

^self classUserAction: #GsfAllValidFileModes onClient: true with: false
%

category: 'File Modes'
classmethod:
validFileModesForCompression

"Answers an array of strings, each of which is a valid mode for
 opening files with compression."

^self classUserAction: #GsfAllValidFileModes onClient: true with: true
%

category: 'Instance Creation (compression)'
classmethod:
openCompressed: aPathName mode: openMode 

"Creates an instance of the receiver and opens a file on the client machine
 using the gzip compression routing gzopen().

 The openMode argument must be one of the strings return by the method
 GsFile>>validFileModesForCompression.

 Returns a GsFile if successful, nil if an error occurs."

^ self open: aPathName mode: openMode onClient: true withCompression: true
%

category: 'Instance Creation (compression)'
classmethod:
openCompressed: aPathName mode: openMode  onClient: clientBool 

"Creates an instance of the receiver and opens a file on the client machine
 (if clientBool is true) or the server machine (if clientBool is false).

 The openMode argument must be one of the strings return by the method
 GsFile>>validFileModesForCompression.

 Returns a GsFile if successful, nil if an error occurs."

^self open: aPathName mode: openMode  onClient: clientBool withCompression: true
%

category: 'Instance Creation (compression)'
classmethod:
openOnServerCompressed: aPathName mode: openMode

"Creates an instance of the receiver and opens a file on the server machine.

 The openMode argument must be one of the strings return by the method
 GsFile>>validFileModesForCompression.

 Returns a GsFile if successful, nil if an error occurs."

^ self open: aPathName mode: openMode onClient: false withCompression: true
%

category: 'Instance Creation (compression)'
classmethod:
openReadCompressed: aPathName

"Opens an existing text file on the client machine for reading.
 Write operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'rb' onClient: true withCompression: true
%

category: 'Instance Creation (compression)'
classmethod:
openWriteCompressed: aPathName

"Opens a text file on the client machine for writing.
 If the file exists it is truncated to zero length.
 If the file does not exist it is created.
 Read operations are not allowed.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'wb' onClient: true withCompression: true
%

category: 'Instance Creation (compression)'
classmethod:
openReadOnServerCompressed: aPathName

"Opens an existing text file on the server machine for reading.
 Write operations are not allowed.

 If the file was compressed with gzip, all data will be 
 uncompressed as it is read.

 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'rb' onClient: false withCompression: true
%

category: 'Instance Creation (compression)'
classmethod:
openWriteOnServerCompressed: aPathName

"Opens a text file on the server machine for writing.
 If the file exists it is truncated to zero length.
 If the file does not exist it is created.
 Read operations are not allowed.
 All data will be compressed in gzip format before being 
 written to the file.
 Returns aGsFile if successful, nil if an error occurs."

^self open: aPathName mode: 'wb' onClient: false withCompression: true
%

category: 'Testing'
method:
isCompressed

"Returns true if the file was opened in compressed mode or false
 if it was not or an uncompressed file was opened in compressed mode.

 Returns nil if the receiver is not in a valid open state."

^ (self _zeroArgPrim: 1) ifNil:[
    self userAction: #GsfIsCompressed onClient: isClient
  ]
%

category: 'Private'
method:
_zeroArgPrim: opcode

"returns nil for a client file or file error.

  opcode 1   isCompressed
  opcode 2   position
"
<primitive: 1012>
self _primitiveFailed: #_zeroArgPrim: args: { opcode }
%

category: 'Private'
method:
_contentsUncompressed: resultObj 

| buf |
buf := resultObj class new .
[ true ] whileTrue:[ | count |
  count := self _read: 4096 into: buf .
  count == 0 ifTrue:[ ^ resultObj ].
  count ifNil:[ ^ nil ].
  count < 4096 ifTrue:[
    buf size: count  "truncate to match amount read"
  ].
  resultObj addAll: buf 
].
%

category: 'Reading'
method:
contents

"Returns a String containing the contents of the receiver from the current
 position to the end of the file.  Returns nil if an error occurs."

^ self isCompressed
    ifTrue:[self _contentsUncompressed: String new ]
    ifFalse:[self _contents: String new ]
%

category: 'Reading'
method:
contentsAsUtf8

"Returns a Utf8 containing the contents of the receiver from the current
 position to the end of the file.  Returns nil if an error occurs."

^ self isCompressed
    ifTrue:[self _contentsUncompressed: Utf8 new ]
    ifFalse:[self _contents: Utf8 new ]
%

category: 'File Operations'
classmethod:
renameFileOnServer: oldName to: newName

"Rename the given file to the new name.  Returns 0 if successful, nil or a non-zero
  SmallInteger if not.  The resulting SmallInteger is the UNIX error number (errno)
  returned by the C function rename().  See the man page for rename on your system
  for errno value definitions."

| oldN newN |
oldN := self _utfPath: oldName forClient: false .
newN := self _utfPath: newName forClient: false .

^ self classUserAction: #GsfRename onClient: false with: oldN with: newN
%

category: 'File Operations'
classmethod:
renameFile: oldName to: newName

"Rename the given file to the new name.  Returns 0 if successful, nil or a non-zero
  SmallInteger if not.  The resulting SmallInteger is the UNIX error number (errno)
  returned by the C function rename().  See the man page for rename on your system
  for errno value definitions."

| oldN newN |
oldN := self _utfPath: oldName forClient: true .
newN := self _utfPath: newName forClient: true .
^ self classUserAction: #GsfRename onClient: true with: oldN with: newN
%

category: 'Private'
classmethod:
_stat: aName isLstat: aBoolean 

"uses the server file system access. 
 Returns a SmallInteger errno value if an error occurs or 
 or if aName is not a valid file or directory. Otherwise
 returns a new instance of GsFileStat. "
<primitive: 757>
aName _validateClasses: { String . Utf8 } .
aBoolean _validateClass: Boolean .
self _primitiveFailed: #_stat:isLstat: args: { aName . aBoolean }
%

classmethod:
_fstat: aFileDescriptor isLstat: aBoolean

"uses the server file system access.
 Returns a SmallInteger errno value if an error occurs or
 or if aFileDescriptor is not a valid file descriptor . Otherwise
 returns a new instance of GsFileStat. "
<primitive: 758>
aFileDescriptor _validateClass: SmallInteger .
aBoolean _validateClass: Boolean .
self _primitiveFailed: #_fstat:isLstat: args: { aFileDescriptor . aBoolean }
%


category: 'File Operations'
classmethod:
stat: aName isLstat: aBoolean 

"uses the server file system access. 
 Returns a SmallInteger errno value if an error occurs or 
 or if aName is not a valid file or directory. Otherwise
 returns a new instance of GsFileStat. "

^ self _stat: (self _utfPath: aName forClient: false) isLstat: aBoolean
%

category: 'ANSI Compatibility'
method:
do: aBlock
"To be compatible with ANSI gettableStream (5.9.2.2):
Each member of the receiver's future sequence values is, in turn, removed from the future
sequence values; appended to the past sequence values; and, passed as the argument to an
evaluation of operand. The argument, operation, is evaluated as if sent the message #value:.
The number of evaluations is equal to the initial size of the receiver's future sequence 
values. If there initially are no future sequence values, operation is not evaluated. 
The future sequence values are used as arguments in their sequence order. The result is 
undefined if any evaluation of operand changes the receiver's future sequence values"

	self positionA to: self fileSize - 1 do: [:i | 
		aBlock value: self next.
	].
%

category: 'ANSI Compatibility'
method:
isEmpty
"To be compatible with ANSI sequencedStream (5.9.1.3):
Returns true if both the set of past and future sequence values of the receiver are empty.
Otherwise returns false."

	^self fileSize == 0.
%

category: 'ANSI Compatibility'
method:
nextMatchFor: anObject
"To be compatible with ANSI gettableStream (5.9.2.2):
The first object is removed from the receiver's future sequence value and appended to 
the end of the receiver's past sequence values. The value that would result from sending 
#= to the object with anObject as the argument is returned.
The results are undefined if there are no future sequence values in the receiver."

	^self next = anObject.
%

category: 'ANSI Compatibility'
method:
peekFor: expectedObject
"To be compatible with ANSI gettableStream (5.9.2.8):
Returns the result of sending #= to the first object in the receiver's future sequence 
values with anObject as the argument. Returns false if the receiver has no future 
sequence values."

	| foundObject flag |
	foundObject := self peek.
	(flag := expectedObject = foundObject) ifTrue: [
		self next.
	].
	^flag.
%

category: 'ANSI Compatibility'
method:
reset
"To be compatible with ANSI sequencedStream (5.9.1.6):
Sets the receiver's future sequence values to be the current past sequence values appended 
with the current future sequence values. Make the receiver's past sequence values be empty."

	self positionA: 0.
%

category: 'ANSI Compatibility'
method:
setToEnd
"To be compatible with ANSI sequencedStream (5.9.1.7):
All of the receiver's future sequence values are appended, in sequence, to the receiver's 
past sequence values. The receiver then has no future sequence values."

	self positionA: self fileSize.
%

category: 'ANSI Compatibility'
method:
skipTo: anObject
"To be compatible with ANSI gettableStream (5.9.2.10):
Each object in the receiver's future sequence values up to and including the first 
occurrence of an object that is equivalent to anObject is removed from the future 
sequence values and appended to the receiver's past sequence values. If an object that 
is equivalent to anObject is not found in the receiver's future sequence values, all 
of the objects in future sequence values are removed from future sequence values and 
appended to past sequence values. If an object equivalent to anObject is not found 
false is returned. Otherwise return true."

	self positionA to: self fileSize - 1 do: [:i | 
		self next = anObject ifTrue: [^true].
	].
	^false.
%

category: 'ANSI Compatibility'
method:
upTo: anObject
"To be compatible with ANSI gettableStream (5.9.2.10):
Each object in the receiver's future sequence values up to and including the first 
occurrence of an object that is equivalent to anObject is removed from the future 
sequence values and appended to the receiver's past sequence values. A collection, 
containing, in order, all of the transferred objects except the object (if any) that 
is equivalent to anObject is returned. If the receiver's future sequence values is 
initially empty, an empty collection is returned."

	| result |
	result := String new.
	self positionA to: self fileSize - 1 do: [:i | 
		| char |
		(char := self next) = anObject ifTrue: [^result].
		result add: char.
	].
	^result.
%

category: 'Stream Compatibility'
method:
atBeginning
"Answer true if the stream is positioned at the beginning"

^self position == 0
%

category: 'Accessing'
method:
maxSize
  "For compatibility with Stream , return a large value"

  ^ SmallInteger maximumValue
%

category: 'Locking'
method:
_acquireLockKind: kind atOffset: offset forBytes: bytes waitTime: milliseconds

"Requests a lock on a portion the file referenced by the receiver.  kind
 is a symbol that indicates the lock kind and must be either #readLock or
 #writeLock.  A read lock is a shared lock such that a given byte in a file
 may have multiple read locks on it at the same time.  A write lock is an 
 exclusive lock which precludes all other read locks and write locks.

 offset indicates the position within the receiver in bytes.  
 offset is zero-based; the first by in the file is at offset 0.  bytes
 indicates the number of bytes to lock (0 means lock the file from offset to the
 end of the file).

 waitTime is the number of milliseconds to wait when requesting a lock and 
 the lock is unavailable.  0 means return immediately (do not wait).  -1 means 
 wait forever.  Specifying a waitTime greater than zero causes the session to 
 rapidly poll for the lock to be available.
 
 For processes running on UNIX based systems, file locks are advisory, which 
 means locks are not enforced by the operating system and applications may 
 freely ignore them.

 For processes running on Microsoft Windows-based systems, file locks are 
 mandatory, which means locks are enforced by the operating system and 
 applications cannot ignore them.

 *** Warning ***
 On UNIX-based systems only, closing any handle referencing a given physical
 file forces *all* locks on that file held by the process to be released.  For 
 this reason, locks should not be used on any file which the process opens more
 than once.

 This method is highly dependent upon the underlying operating system 
 implementation of advisory file locks.  On UNIX-based systems, the fcntl(F_SETLK)
 call is used.  On Microsoft Windows systems, the LockFileEx() call is used.  
 Consult the documentation for your host system to learn about the behavior of
 file locking for your system.

 Some systems do not support read locks if the file is opened in write-only 
 mode or write locks if the file is open in read-only mode.  Attempting to lock 
 files opened in these manners may raise an error indicating a bad file 
 descriptor.

 Returns true if the lock was acquired, false if the lock was denied.  
 Raises an IOError if an error occurs."

| result |
result := self userAction: #GsfLock onClient: self isClient with: kind with: offset with: bytes with: milliseconds .
^ result class == Boolean
     ifTrue:[ result ]
    ifFalse:[ IOError signal: result ]
%

category: 'Locking'
method:
unlockAtOffset: offset forBytes: bytes

"Releases a previously acquired lock on a portion the file referenced by the
 receiver.  offset indicates the position within the receiver in bytes and bytes 
 indicates the number of bytes to unlock (0 means unlock the file from offset to
 the end of the file).

 Please refer the comments in the following method for more details on file 
 locks:
   GSFile _acquireLockKind: atOffset: forBytes: waitTime:

 Returns true if the lock was released or if the region was not locked by the process.
 Raises an IOError if an error occurs."

| result |
result := self userAction: #GsfUnlock onClient: self isClient with: offset with: bytes .
^ result class == Boolean
     ifTrue:[ result ]
    ifFalse:[ IOError signal: result ]
%

category: 'Locking'
method:
readLockAtOffset: offset forBytes: bytes waitTime: milliseconds
"Requests a read lock on a portion of the receiver and blocks for up to 
 milliseconds to acquire the lock.  Returns true if successful, false if the lock 
 was denied. Raises an IOError if an error occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more 
 information about file locks."
 
 ^ self _acquireLockKind: #readLock atOffset: offset forBytes: bytes waitTime: milliseconds
%

category: 'Locking'
method:
writeLockAtOffset: offset forBytes: bytes waitTime: milliseconds
"Requests a write lock on a portion of the receiver and blocks for up to 
 milliseconds to acquire the lock.  Returns true if successful, false if the
 lock was denied. Raises an IOError if an error occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more 
 information about file locks."

 ^ self _acquireLockKind: #writeLock atOffset: offset forBytes: bytes waitTime: milliseconds
%

category: 'Locking'
method:
readLockAtOffset: offset forBytes: bytes
"Requests a read lock on a portion of the receiver without blocking.  Returns 
 true if successful, false if the lock was denied. Raises an IOError if an error
 occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more 
 information about file locks."

 ^ self readLockAtOffset: offset forBytes: bytes waitTime: 0
%

category: 'Locking'
method:
writeLockAtOffset: offset forBytes: bytes
"Requests a write lock on a portion of the receiver without blocking.
 Returns true if successful, false if the lock was denied. Raises an IOError if
 an error occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more 
 information about file locks."

 ^ self writeLockAtOffset: offset forBytes: bytes waitTime: 0
%

category: 'Locking'
method:
readLockContents
"Requests a read lock on the entire contents the receiver without blocking.
 Returns true if successful, false if the lock was denied. Raises an IOError if
 an error occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more 
 information about file locks."

 ^ self readLockAtOffset: 0 forBytes: 0
%

category: 'Locking'
method:
writeLockContents
"Requests a write lock on the entire contents the receiver without blocking.
 Returns true if successful, false if the lock was denied. Raises an IOError
 if an error occurred.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more
 information about file locks."

 ^ self writeLockAtOffset: 0 forBytes: 0
%

category: 'Locking'
method:
unlockContents
"Releases a previously acquired lock on contents of the receiver .

 Returns true if the lock was released successfully or if the contents were not
 locked by the process.  Raises an IOError if an error occurs.

 See the method #_acquireLockKind:atOffset:forBytes:waitTime: for more 
 information about file locks."

^ self unlockAtOffset: 0 forBytes: 0
%

category: 'Locking'
method:
getStatusForLockKind: kind atOffset: offset forBytes: bytes
"Tests if the receiver could be locked with the given arguments but without 
 actually acquiring the lock.  Returns information about a conflicting lock, 
 if any.

 The kind argument must be a symbol that indicates the lock kind requested, and
 must be either #readLock or #writeLock.  offset indicates the position within
 the receiver in bytes.  bytes indicates the number of bytes starting at offset
 (0 means to the end of the file).

 Returns an Array of 5 elements as described below.  If the first element is 
 false, then elements 2 through 5 contain information about a conflicting
 lock (there may be more than one) which prevented the lock from being available.
 If the first element is true, then element 2 will be nil and elements 3 
 through 5 will be -1.

 1 - a Boolean: true if the lock would have been granted, otherwise false.
 2 - a Symbol (#readLock or #writeLock): conflicting lock kind.
 3 - a SmallInteger: start position of the conflicting lock.
 4 - a SmallInteger: number of bytes held by the conflicting lock.
 5 - a SmallInteger: process ID of the owner of the conflicting lock.

 Raises an IOError if an error occurs.

 This method is not supported and will always raise an error if the receiver
 references a client file and the GCI client is running on the Microsoft 
 Windows platform."

| result |
result := self userAction: #GsfLockQuery onClient: self isClient 
               with: kind with: offset with: bytes .
^ result class == Array
     ifTrue:[ result ]
    ifFalse:[ IOError signal: result ]
%

category: 'Private'
classmethod:
_directoryPrim: opcode with: arg with: arg2

"opcode 6    gciClientIsWindows
 opcode 2    getcwd,
  other opcodes for Maglev only"
<primitive: 765>

^ self _primitiveFailed: #_directoryPrim:with:with: args: { opcode. arg . arg2 }
%

category: 'Directory Operations'
classmethod:
serverCurrentDirectory

"Returns a String, the current directory of this topaz -l  or gem process."
^ self _directoryPrim: 2 with: nil with: nil 
%
category: 'Directory Operations'
classmethod:
gciClientIsWindows

"Returns true if GCI client is on a Microsoft Windows operating system,
 false otherwise."

^ self _directoryPrim: 6 with: nil with: nil
%

category: 'Fileout'
method:
_fileOutAll: aString

  self nextPutAll: aString encodeAsUTF8
%

category: 'Reading'
classmethod:
getContentsOfServerFile: filePathAndName
"Return the contents of the named file in a single operation.
 Use caution when reading a large file as doing so may cause
 an out of memory error."
 
| gsf result |
gsf := GsFile openReadOnServer: filePathAndName.
gsf ifNil:[ IOError signal: GsFile serverErrorString ] .
result := gsf contents.
gsf close.
^ result
%
