!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   PositionableStream, Stream, Object.
! 
! Legacy implementation
!
! See installStreamHierarchy.topaz for filein information
!
!=========================================================================

! to filein this file interactively for debugging compile errors,
!   UserGlobals at: #PositionableStream put: PositionableStreamLegacy

set class PositionableStream
removeallmethods 
removeallclassmethods 


doit
PositionableStream
  comment: 'PositionableStream / PositionableStreamLegacy is an abstract superclass 
 that provides additional protocol appropriate to Streams whose objects 
 are externally named by indices.
 Concrete subclasses are ReadStream and WriteStream.

--- instVar itsCollection
A SequenceableCollection; the sequence of objects that the receiver may
 access.
--- instVar position
A SmallInteger; the current position reference for accessing the collection.
'.
true
%

category: 'Portable Methods'
classmethod: 
isLegacyStreamImplementation

^true
%

category: 'Portable Methods'
classmethod: 
isPortableStreamImplementation

^false
%

category: 'Instance Creation'
classmethod: 
on: aCollection

"Returns an instance of the receiver that can stream over the elements of
 aCollection."

| newStream |

newStream := self _basicNew.
newStream _initStreamWith: aCollection.
^ newStream
%

category: 'Accessing'
method: 
contents

"Returns the Collection associated with the receiver (that is,
 the sequence of objects that the receiver may access)."

^itsCollection
%

category: 'Accessing'
method: 
peek

"Returns the next element in the collection, but does not alter the current
 position reference.  If the receiver is at the end of the collection, returns
 nil."

| pos coll |
pos := position .
pos <= (coll := itsCollection) size ifFalse:[ ^ nil ]. "inline atEnd"
^ coll at: pos
%

category: 'Accessing'
method: 
skip: amount

"Sets the receiver's position to position+amount."

self position: self position + amount
%

category: 'Testing'
method: 
atEnd

"Returns true if the receiver cannot access any more objects, false if it can."

^ position > itsCollection size
%

category: 'Testing'
method: 
isEmpty

"Returns true if the collection that the receiver accesses contains
 no elements; otherwise returns false."

^ itsCollection size == 0
%

category: 'Positioning'
method: 
position

"This method replaced by positio1.gs."

^self positionL.
%

category: 'Positioning'
method: 
positionA

"Returns the receiver's current position reference for accessing the sequence
 of objects.  The position is actually the next element of the collection to be
 read or written; the position is incremented by each read or write.  In
 general, to start reading or writing at the nth element of a collection, the
 position must be set to n.
 
 This is the ANSI method. See Bug #39503."

^position - 1.
%

category: 'Positioning'
method: 
positionL

"Returns the receiver's current position reference for accessing the sequence
 of objects.  The position is actually the next element of the collection to be
 read or written; the position is incremented by each read or write.  In
 general, to start reading or writing at the nth element of a collection, the
 position must be set to n.
 
 This is the 'Legacy' (non-ANSI) method. See Bug #39503."

^position
%

category: 'Positioning'
method: 
positionW

"This is the 'Warning' behavior to be called when you think you have replaced
 all references to #'position' with #'positionL' or #'positionA'.
 Replace #'position' with this when you think there are no senders."

System addAllToStoneLog: (GsProcess stackReportToLevel: 5).
^self positionL.
%

category: 'Positioning'
method: 
position: anInteger

"This method replaced by positio1.gs."

^self positionL: anInteger.
%

category: 'Positioning'
method: 
positionA: anInteger

"Sets the receiver's current position reference for accessing the collection to
 be anInteger.  If anInteger is not within the bounds of the collection,
 generates an error.
 
 This is the ANSI method. See Bug #39503."

	self positionL: anInteger + 1.
%

category: 'Positioning'
method: 
positionL: anInteger

"Sets the receiver's current position reference for accessing the collection to
 be anInteger.  If anInteger is not within the bounds of the collection,
 generates an error.
 
 This is the 'Legacy' (non-ANSI) method. See Bug #39503."

(anInteger > 0) & (anInteger <= (itsCollection size + 1))
   ifTrue: [position := anInteger]
   ifFalse: [self _positionError: anInteger]
%

category: 'Positioning'
method: 
positionW: anInteger

"This is the 'Warning' behavior to be called when you think you have replaced
 all references to #'position:' with #'positionL:' or #'positionA:'.
 Replace #'position:' with this when you think there are no senders."

System addAllToStoneLog: (GsProcess stackReportToLevel: 5).
^self positionL: anInteger.
%

category: 'Positioning'
method: 
_positionError: anInteger

"Returns an error message that anInteger is out of bounds of the Collection."

^ self _error: #rtErrBadStreamPosition args: { anInteger }
%

category: 'Positioning'
method: 
reset

"Sets the receiver's position to the beginning of the sequence of objects."

position := 1
%

category: 'Positioning'
method: 
setToEnd

"Sets the receiver's position to the end of the sequence of objects."

self positionL: itsCollection size + 1
%

category: 'Positioning'
method: 
_initStreamWith: aCollection

"Initialize the receiver's 'itsCollection' instance variable to be
 aCollection."

itsCollection := aCollection.
position := 1.
%

category: 'Accessing'
method: 
backup

"Backs up the receiver one position."

self skip: -1
%

category: 'Accessing'
method: 
nextWord

"Assume that the receiver's collection is a kind of String.  Returns the
 next word in the string or nil if there is no next word."

| result |
result := itsCollection species new.
self skipSeparators.
[self atEnd not and: [self peek isSeparator not]] whileTrue: [
  result add: self next
].
result size == 0
    ifTrue: [^nil]
    ifFalse: [^result]
%

! fix bug 10257
category: 'Accessing'
method: 
peekWord

"Assume that the receiver's collection is a kind of String.  Returns the
 next word in the string without moving the receiver's position."

| result pos |
pos := self position.
result := self nextWord .
self position: pos.
^result
%

category: 'Accessing'
method: 
skipAny: chars

"Skip past all Characters in chars.  Returns the number of Characters skipped."

| skipped |
skipped := 0.
[self atEnd not and: [(chars includesIdentical: self peek)]] whileTrue: [
  self next.
  skipped := skipped + 1.
].
^skipped
%

category: 'Accessing'
method: 
skipSeparators

"Skip any objects immediately next in the stream that respond true to
 isSeparator."

[self atEnd not and: [self peek isSeparator]] whileTrue: [ self next ]
%


category: 'Accessing'
method: 
throughAll: matchCollection

"Returns a collection of objects from the receiver up to and including the
sequence of objects in the argument 'matchCollection', leaving the stream
positioned after the sequence.  If the sequence of objects is not found, this
returns the remaining contents of the receiver and leaves me positioned
at my end."

| numMatched numToMatch result  |

numMatched := 0.
result := itsCollection species new.
numToMatch := matchCollection size.
[self atEnd or: [numMatched = numToMatch]]
     whileFalse:
           [self next = (matchCollection at: numMatched + 1)
                ifTrue: [numMatched := numMatched + 1]
          ifFalse: [position := position - numMatched - 1.
                        result add: self next.
                        numMatched := 0]
].

"add matched or partially matched chars"
position := position - numMatched.
numMatched timesRepeat: [result add: self next].

^ result.
%

category: 'Accessing'
method: 
upTo: anObject

"Returns all objects up to the given value or the end of the stream."

| result obj |
result := itsCollection species new.
[ true ] whileTrue:[
  self atEnd ifTrue:[ ^ result ].
  obj := self next .
  anObject = obj ifTrue:[ ^ result ].
  result add: obj  
]
%

category: 'Accessing'
method: 
upTo: anObject do: aBlock

"Sends each object encountered to the given block until the end of stream
 or the given value is encountered.  Returns the receiver."

| obj |
[ true ] whileTrue:[
  self atEnd ifTrue:[ ^ self ].
  obj := self next .
  anObject = obj ifTrue:[ ^ self ].
  aBlock value: obj 
]
%

category: 'Accessing'
method: 
upToAll: matchCollection 

"Returns a collection of objects from the receiver up to, but not including,
 the sequence of objects in the argument 'matchCollection', leaving the stream
 positioned to read the sequence.  If the sequence of objects is not found,
 this returns the remaining contents of the receiver and leaves the stream 
 positioned at the end." 

| numMatched numToMatch result | 

numMatched := 0. 
result := itsCollection species new. 
numToMatch := matchCollection size. 
[self atEnd or: [numMatched == numToMatch]] whileFalse: [
  self next = (matchCollection at: numMatched + 1) 
      ifTrue: [numMatched := numMatched + 1] 
      ifFalse: [position := position - numMatched - 1. 
    		result add: self next. 
                numMatched := 0] 
]. 
"Position before any partial or complete match we might have found." 
position := position - numMatched. 

"If the match was not complete, must add any partially matched chars." 
numMatched ~~ numToMatch 
  ifTrue: [numMatched timesRepeat: [result add: self next]]. 

^ result. 
%

category: 'Accessing'
method: 
upToAny: objects

"Returns all objects up to one of the given collection of objects or the end
 of the stream."

| result |
result := itsCollection species new.
[self atEnd not and: [(objects includesIdentical: self peek) not]] whileTrue: [
  result add: self next
].
self atEnd not ifTrue: [
  self next
].
^result
%

category: 'Accessing'
method: 
upToAny: objects do: aBlock

"Send each Character encountered to aBlock until the end of stream
 or one of the given Characters is encountered."

[self atEnd not and: [(objects includesIdentical: self peek) not]] whileTrue: [
  aBlock value: self next
].
self atEnd not ifTrue: [ self next ].
%

! fix 34786 , upToEnd should advance position to one past end 
category: 'Accessing'
method: 
upToEnd

"Returns all Characters from the current position to the end of the stream."

| result end coll |
end := (coll := itsCollection) size .
result := coll copyFrom: position to: end .
position :=  end + 1 .
^ result
%

category: 'Accessing'
method: 
_collection

"Returns the collection of the receiver."

^itsCollection
%

category: 'Accessing'
method: 
next: count

"Returns the next count elements in the receiver's collection."

| result |
result := itsCollection species new.
count timesRepeat: [ result add: self next ].
^result
%

category: 'Accessing'
method: 
next: count into: anObject

"Stores the next count elements in the receiver's collection into the
 given object.  Returns the argument anObject."

| idx |
idx := 1.
count timesRepeat: [ anObject at: idx put: self next. idx := idx + 1 ].
^anObject
%

category: 'Accessing'
method: 
_next: count basicInto: anObject

"Stores the next count basic elements in the receiver's collection into the
 given object.  Returns the argument anObject."

| idx |
idx := 1.
count timesRepeat: [ 
   anObject _basicAt: idx put: self next codePoint. 
   idx := idx + 1 
   ].
^anObject
%

! fixed 41213
category: 'Accessing'
method: 
peek2

"Peeks at the second incoming object."

| pos coll |
pos := position.
pos < (coll := itsCollection) size ifFalse:[ ^ nil ]. "inline atEnd"
^ coll at: pos + 1
%

category: 'Accessing'
method: 
nextLine

| result cr lf char chrcls |
result := itsCollection species new.
cr := (chrcls:= Character) cr.
lf := chrcls  lf.
[self atEnd not and: [(char := self peek) ~~ cr and: [char ~~ lf]]] whileTrue: [
  result add: self next
].
self atEnd not ifTrue: [
  self next.
  (self atEnd not and: [char == cr and: [self peek == lf]]) ifTrue: [
    self next.
  ].
].
^result
%

category: 'Accessing'
method: 
peekFor: anObject

self peek = anObject ifTrue: [
  self next.
  ^true.
].
^false.
%

category: 'Testing'
method: 
atBeginning
"Answer true if the stream is positioned at the beginning"

^position == 1
%
