!=========================================================================
! Copyright (C) GemTalk Systems 2013-2020.  All Rights Reserved.
!=========================================================================

! Remove existing behavior from PPActionParser
removeallmethods PPActionParser
removeallclassmethods PPActionParser
! ------------------- Class methods for PPActionParser
category: 'instance creation'
classmethod: PPActionParser
on: aParser block: aBlock
	^ (self on: aParser) setBlock: aBlock
%
! ------------------- Instance methods for PPActionParser
category: 'accessing'
method: PPActionParser
block
	"Answer the action block of the receiver."

	^ block
%
category: 'parsing'
method: PPActionParser
parseOn: aPPContext
	| element |
	^ (element := parser parseOn: aPPContext) isPetitFailure
		ifFalse: [ block value: element ]
		ifTrue: [ element ]
%
category: 'initialization'
method: PPActionParser
setBlock: aBlockOrSymbol

	block := aBlockOrSymbol isSymbol
				ifFalse: [aBlockOrSymbol]
				ifTrue: [[:arg | arg perform: aBlockOrSymbol]]
%
expectValue %Boolean
doit
PPActionParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPAndParser
removeallmethods PPAndParser
removeallclassmethods PPAndParser
! ------------------- Class methods for PPAndParser
! ------------------- Instance methods for PPAndParser
category: 'operators'
method: PPAndParser
and
	^ self
%
category: 'parsing'
method: PPAndParser
parseOn: aPPContext
	| element memento |
	memento := aPPContext remember.
	element := parser parseOn: aPPContext.
	aPPContext restore: memento.
	^ element
%
expectValue %Boolean
doit
PPAndParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPCharSetPredicate
removeallmethods PPCharSetPredicate
removeallclassmethods PPCharSetPredicate
! ------------------- Class methods for PPCharSetPredicate
category: 'instance creation'
classmethod: PPCharSetPredicate
on: aBlock
	^ self basicNew initializeOn: aBlock
%
! ------------------- Instance methods for PPCharSetPredicate
category: 'initialization'
method: PPCharSetPredicate
initializeOn: aBlock
	block := aBlock.
	classification := Array new: 255.
	1 to: classification size do: [ :index |
		classification at: index put: (block
			value: (Character codePoint: index)) ]
%
category: 'evaluating'
method: PPCharSetPredicate
value: aCharacter
	| index |
	index := aCharacter asInteger.
	index == 0
		ifTrue: [ ^ block value: aCharacter ].
	index > 255
		ifTrue: [ ^ block value: aCharacter ].
	^ classification at: index
%
expectValue %Boolean
doit
PPCharSetPredicate category: 'PetitParser-Tools'.
true
%

! Remove existing behavior from PPChoiceParser
removeallmethods PPChoiceParser
removeallclassmethods PPChoiceParser
! ------------------- Class methods for PPChoiceParser
! ------------------- Instance methods for PPChoiceParser
category: 'operators'
method: PPChoiceParser
/ aRule 
	^ self copyWith: aRule
%
category: 'parsing'
method: PPChoiceParser
parseOn: aPPContext
	"This is optimized code that avoids unnecessary block activations, do not change. When all choices fail, the last failure is answered."

	| element |
	1 to: parsers size do: [ :index |
		element := (parsers at: index)
			parseOn: aPPContext.
		element isPetitFailure
			ifFalse: [ ^ element ] ].
	^ element
%
expectValue %Boolean
doit
PPChoiceParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPCompositeParser
removeallmethods PPCompositeParser
removeallclassmethods PPCompositeParser
! ------------------- Class methods for PPCompositeParser
category: 'accessing'
classmethod: PPCompositeParser
dependencies
	"Answer a collection of PPCompositeParser classes that this parser directly dependends on. Override this method in subclasses to declare dependent parsers. The default implementation does not depend on other PPCompositeParser."

	^ #()
%
category: 'accessing'
classmethod: PPCompositeParser
ignoredNames
	"Answer a collection of instance-variables that should not be automatically initialized with productions, but that are used internal to the composite parser."

	^ PPCompositeParser allInstVarNames
%
category: 'instance creation'
classmethod: PPCompositeParser
new
	"Answer a new parser starting at the default start symbol."

	^ self newStartingAt: self startSymbol
%
category: 'instance creation'
classmethod: PPCompositeParser
newStartingAt: aSymbol
	"Answer a new parser starting at aSymbol. The code makes sure to resolve all dependent parsers correctly."

	| parsers remaining |
	parsers := IdentityDictionary new.
	remaining := OrderedCollection with: self.
	[ remaining isEmpty ] whileFalse: [
		| dependency |
		dependency := remaining removeLast.
		(parsers includesKey: dependency) ifFalse: [
			parsers at: dependency put: dependency basicNew.
			remaining addAll: dependency dependencies ] ].
	parsers keysAndValuesDo: [ :class :parser |
		| dependencies |
		dependencies := IdentityDictionary new.
		class dependencies 
			do: [ :dependency | dependencies at: dependency put: (parsers at: dependency) ].
		parser 
			initializeStartingAt: (class == self
				ifTrue: [ aSymbol ]
				ifFalse: [ class startSymbol ]) 
			dependencies: dependencies ].
	parsers keysAndValuesDo: [ :class :parser |
		parser setParser: (parser perform: parser children first name).
		parser productionNames keysAndValuesDo: [ :key :value |
			(parser instVarAt: key) setParser: (parser perform: value) ] ].
	^ parsers at: self
%
category: 'parsing'
classmethod: PPCompositeParser
parse: anObject
	^ self parse: anObject startingAt: self startSymbol
%
category: 'parsing'
classmethod: PPCompositeParser
parse: anObject onError: aBlock
	^ self parse: anObject startingAt: self startSymbol onError: aBlock
%
category: 'parsing'
classmethod: PPCompositeParser
parse: anObject startingAt: aSymbol
	^ (self newStartingAt: aSymbol) parse: anObject
%
category: 'parsing'
classmethod: PPCompositeParser
parse: anObject startingAt: aSymbol onError: aBlock
	^ (self newStartingAt: aSymbol) parse: anObject onError: aBlock
%
category: 'accessing'
classmethod: PPCompositeParser
startSymbol
	"Answer the method that represents the default start symbol."

	^ #start
%
! ------------------- Instance methods for PPCompositeParser
category: 'querying'
method: PPCompositeParser
dependencyAt: aClass
	"Answer the dependent parser aClass. Throws an error if this parser class is not declared in the method #dependencies on the class-side of the receiver."
	
	^ dependencies at: aClass ifAbsent: [ self error: 'Undeclared dependency in ' , self class name , ' to ' , aClass name ]
%
category: 'initialization'
method: PPCompositeParser
initializeStartingAt: aSymbol dependencies: aDictionary
	self initialize.
	parser := PPDelegateParser named: aSymbol.
	self productionNames keysAndValuesDo: [ :key :value |
		self instVarAt: key put: (PPDelegateParser named: value) ].
	dependencies := aDictionary
%
category: 'querying'
method: PPCompositeParser
productionAt: aSymbol
	"Answer the production named aSymbol."
	
	^ self productionAt: aSymbol ifAbsent: [ nil ]
%
category: 'querying'
method: PPCompositeParser
productionAt: aSymbol ifAbsent: aBlock
	"Answer the production named aSymbol, if there is no such production answer the result of evaluating aBlock."
	
	(self class ignoredNames includes: aSymbol asString)
		ifTrue: [ ^ aBlock value ].
	(self class startSymbol = aSymbol)
		ifTrue: [ ^ parser ].
	^ self instVarAt: (self class allInstVarNames
		indexOf: aSymbol "asString"
		ifAbsent: [ ^ aBlock value ])
%
category: 'querying'
method: PPCompositeParser
productionNames
	"Answer a dictionary of slot indexes and production names."
	
	| productionNames ignoredNames |
	productionNames := Dictionary new.
	ignoredNames := self class ignoredNames
		collect: [ :each | each asSymbol ].
	self class allInstVarNames keysAndValuesDo: [ :key :value |
		(ignoredNames includes: value asSymbol)
			ifFalse: [ productionNames at: key put: value asSymbol ] ].
	^ productionNames
%
category: 'accessing'
method: PPCompositeParser
start
	"Answer the production to start this parser with."
	
	self subclassResponsibility
%
expectValue %Boolean
doit
PPCompositeParser category: 'PetitParser-Tools'.
true
%

! Remove existing behavior from PPConditionalParser
removeallmethods PPConditionalParser
removeallclassmethods PPConditionalParser
! ------------------- Class methods for PPConditionalParser
category: 'as yet unclassified'
classmethod: PPConditionalParser
on: aPPParser block: block
	^ (PPConditionalParser on: aPPParser)
		block: block;
		yourself
%
! ------------------- Instance methods for PPConditionalParser
category: 'accessing'
method: PPConditionalParser
block: aBlock
	block := aBlock
%
category: 'parsing'
method: PPConditionalParser
parseOn: aPPContext
	^ (block value: aPPContext) 
		ifTrue: [ parser parseOn: aPPContext ]
		ifFalse: [ PPFailure message: block asString, ' was not evaluated to true.' context: aPPContext ]
%
expectValue %Boolean
doit
PPConditionalParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPContext
removeallmethods PPContext
removeallclassmethods PPContext
! ------------------- Class methods for PPContext
category: 'as yet unclassified'
classmethod: PPContext
on: aPPParser stream: aStream
	^ self basicNew 
		initialize;
		root: aPPParser;
		stream: aStream asPetitStream;
		yourself
%
! ------------------- Instance methods for PPContext
category: 'stream mimicry'
method: PPContext
atEnd
	^ stream atEnd
%
category: 'stream mimicry'
method: PPContext
back
	^ stream back
%
category: 'stream mimicry'
method: PPContext
collection
	^ stream collection
%
category: 'stream mimicry'
method: PPContext
contents 
	^ stream contents
%
category: 'failures'
method: PPContext
furthestFailure
	" the furthest failure encountered while parsing the input stream "
	
	"^ self globalAt: #furthestFailure ifAbsent: [ nil ]"
	"performance optimization:"
	^ furthestFailure
%
category: 'accessing-globals'
method: PPContext
globalAt: aKey
	"Answer the global property value associated with aKey."
	
	^ self globalAt: aKey ifAbsent: [ self error: 'Property not found' ]
%
category: 'accessing-globals'
method: PPContext
globalAt: aKey ifAbsent: aBlock
	"Answer the global property value associated with aKey or, if aKey isn't found, answer the result of evaluating aBlock."
	
	^ globals == nil 
		ifTrue: [ aBlock value ]
		ifFalse: [ globals at: aKey ifAbsent: aBlock ]
%
category: 'accessing-globals'
method: PPContext
globalAt: aKey ifAbsentPut: aBlock
	"Answer the global property associated with aKey or, if aKey isn't found store the result of evaluating aBlock as new value."
	
	^ self globalAt: aKey ifAbsent: [ self globalAt: aKey put: aBlock value ]
%
category: 'accessing-globals'
method: PPContext
globalAt: aKey put: anObject
	"Set the global property at aKey to be anObject. If aKey is not found, create a new entry for aKey and set is value to anObject. Answer anObject."

	^ (globals ifNil: [ globals := Dictionary new: 1 ])
		at: aKey put: anObject
%
category: 'accessing-globals'
method: PPContext
globals
	^ globals
%
category: 'accessing-globals'
method: PPContext
hasGlobal: aKey
	"Test if the global property aKey is present."
	
	^ globals ~~ nil  and: [ globals includesKey: aKey ]
%
category: 'accessing-properties'
method: PPContext
hasProperty: aKey
	"Test if the property aKey is present."
	
	^ properties ~~ nil  and: [ properties includesKey: aKey ]
%
category: 'memoization'
method: PPContext
identifier
	"
		I provide an identifier that is used by memoizing parser to figure out if the
		cache should be flushed or not.
	"
	^ stream
%
category: 'initialization'
method: PPContext
initialize
	stream := nil.
%
category: 'initialization'
method: PPContext
initializeFor: parser
	root := parser.
%
category: 'stream mimicry'
method: PPContext
isEndOfLine
	^ stream isEndOfLine
%
category: 'stream mimicry'
method: PPContext
isStartOfLine
	^ stream isStartOfLine
%
category: 'stream mimicry'
method: PPContext
next
	^ stream next
%
category: 'stream mimicry'
method: PPContext
next: anInteger
	^ stream next: anInteger
%
category: 'failures'
method: PPContext
noteFailure: aPPFailure
  "record the furthest failure encountered while parsing the input stream "

  (furthestFailure == nil  or: [ aPPFailure position > furthestFailure position ])
    ifTrue: [ furthestFailure := aPPFailure ].
  " true
    ifTrue: [ 
      (self globalAt: #'failureStack' ifAbsentPut: [ OrderedCollection new ])
        add: aPPFailure ]"
%
category: 'stream mimicry'
method: PPContext
peek
	^ stream peek
%
category: 'stream mimicry'
method: PPContext
peekTwice
	^ stream peekTwice
%
category: 'stream mimicry'
method: PPContext
position
	^ stream position
%
category: 'stream mimicry'
method: PPContext
position: anInteger
	^ stream position: anInteger
%
category: 'initialization'
method: PPContext
postCopy
	super postCopy.
	globals := globals copy.
%
category: 'printing'
method: PPContext
printOn: aStream
	super printOn: aStream.
	aStream nextPut: $:.
	aStream nextPut: $ .
	stream printOn: aStream
%
category: 'accessing-properties'
method: PPContext
properties
	^ properties 
%
category: 'accessing-properties'
method: PPContext
propertyAt: aKey
	"Answer the property value associated with aKey."
	
	^ self propertyAt: aKey ifAbsent: [ self error: 'Property not found' ]
%
category: 'accessing-properties'
method: PPContext
propertyAt: aKey ifAbsent: aBlock
	"Answer the property value associated with aKey or, if aKey isn't found, answer the result of evaluating aBlock."
	
	^ properties == nil 
		ifTrue: [ aBlock value ]
		ifFalse: [ properties at: aKey ifAbsent: aBlock ]
%
category: 'accessing-properties'
method: PPContext
propertyAt: aKey ifAbsentPut: aBlock
	"Answer the property associated with aKey or, if aKey isn't found store the result of evaluating aBlock as new value."
	
	^ self propertyAt: aKey ifAbsent: [ self propertyAt: aKey put: aBlock value ]
%
category: 'accessing-properties'
method: PPContext
propertyAt: aKey put: anObject
	"Set the property at aKey to be anObject. If aKey is not found, create a new entry for aKey and set is value to anObject. Answer anObject."

	^ (properties ifNil: [ properties := Dictionary new: 1 ])
		at: aKey put: anObject
%
category: 'memoization'
method: PPContext
remember
	| memento |
	memento := PPContextMemento new
		stream: stream;
		position: stream position;
		yourself.
		
	self rememberProperties: memento.
	^ memento
%
category: 'memoization'
method: PPContext
rememberProperties: aPPContextMemento
	properties ifNil: [ ^ self ].
	
	properties keysAndValuesDo: [ :key :value |
		aPPContextMemento propertyAt: key put: value
	].
%
category: 'accessing-globals'
method: PPContext
removeGlobal: aKey
	"Remove the property with aKey. Answer the property or raise an error if aKey isn't found."
	
	^ self removeGlobal: aKey ifAbsent: [ self error: 'Property not found' ]
%
category: 'accessing-globals'
method: PPContext
removeGlobal: aKey ifAbsent: aBlock
	"Remove the global property with aKey. Answer the value or, if aKey isn't found, answer the result of evaluating aBlock."
	
	| answer |
	globals == nil  ifTrue: [ ^ aBlock value ].
	answer := globals removeKey: aKey ifAbsent: aBlock.
	globals isEmpty ifTrue: [ globals := nil ].
	^ answer
%
category: 'accessing-properties'
method: PPContext
removeProperty: aKey
	"Remove the property with aKey. Answer the property or raise an error if aKey isn't found."
	
	^ self removeProperty: aKey ifAbsent: [ self error: 'Property not found' ]
%
category: 'accessing-properties'
method: PPContext
removeProperty: aKey ifAbsent: aBlock
	"Remove the property with aKey. Answer the value or, if aKey isn't found, answer the result of evaluating aBlock."
	
	| answer |
	properties == nil  ifTrue: [ ^ aBlock value ].
	answer := properties removeKey: aKey ifAbsent: aBlock.
	properties isEmpty ifTrue: [ properties := nil ].
	^ answer
%
category: 'initialization'
method: PPContext
reset
	properties := nil.
	globals := nil.
%
category: 'memoization'
method: PPContext
restore: aPPContextMemento
	aPPContextMemento stream == stream ifFalse: [ self error: 'Oops!' ].

	stream position: aPPContextMemento position.
	self restoreProperties: aPPContextMemento.
%
category: 'memoization'
method: PPContext
restoreProperties: aPPContextMemento
	aPPContextMemento stream == stream ifFalse: [ self error: 'Oops!' ].
	
	properties ifNil: [ ^ self ].
	
	properties keysDo: [ :key |
		(aPPContextMemento hasProperty: key)
			ifTrue: [ properties at: key put: (aPPContextMemento propertyAt: key) ]
			ifFalse: [ properties removeKey: key  ]. 
	].

	aPPContextMemento keysAndValuesDo: [ :key :value |
		properties at: key put: value
	]
%
category: 'accessing'
method: PPContext
root
	^ root
%
category: 'memoization'
method: PPContext
size
	^ stream size
%
category: 'stream mimicry'
method: PPContext
skip: anInteger 
	^ stream skip: anInteger
%
category: 'stream mimicry'
method: PPContext
skipTo: anObject 
	^ stream skipTo: anObject
%
category: 'stream mimicry'
method: PPContext
skipToAll: aString
	"Set the access position of the receiver to be past the next occurrence of the subCollection. Answer whether subCollection is found.  No wildcards, and case does matter."
	| pattern startMatch |
	pattern := aString readStream.
	startMatch := nil.
	[ pattern atEnd ] whileFalse: 
		[ stream atEnd ifTrue: [ ^ false ].
		stream next = pattern next 
			ifTrue: [ pattern position = 1 ifTrue: [ startMatch := stream position ] ]
			ifFalse: 
				[ pattern position: 0.
				startMatch ifNotNil: 
					[ stream position: startMatch.
					startMatch := nil ] ] ].
	^ true
%
category: 'stream mimicry'
method: PPContext
skipToAnyOf: aCharacterSet 
	"Set the access position of the receiver to be past the next occurrence of
	a character in the character set. Answer whether a fitting character is found."

	[stream atEnd]
		whileFalse: [ (aCharacterSet includes: stream next) ifTrue: [^true]].
	^false
%
category: 'accessing'
method: PPContext
stream
	^ stream
%
category: 'accessing'
method: PPContext
stream: aStream
	stream := aStream.
%
category: 'stream mimicry'
method: PPContext
uncheckedPeek
	^ stream uncheckedPeek
%
category: 'stream mimicry'
method: PPContext
upTo: anObject
	^ stream upTo: anObject
%
category: 'stream mimicry'
method: PPContext
upToAll: whatever
	^ stream upToAll: whatever
%
category: 'stream mimicry'
method: PPContext
upToAnyOf: whatever
	^ stream upToAnyOf: whatever
%
expectValue %Boolean
doit
PPContext category: 'PetitParser-Core'.
true
%

! Remove existing behavior from PPContextMemento
removeallmethods PPContextMemento
removeallclassmethods PPContextMemento
! ------------------- Class methods for PPContextMemento
! ------------------- Instance methods for PPContextMemento
category: 'comparing'
method: PPContextMemento
= anObject
	
	(self == anObject) ifTrue: [ ^ true ].
	(anObject class = PPContextMemento) ifFalse: [ ^ false ].
	
	(anObject stream == stream) ifFalse: [ ^ false ].
	(anObject position == position) ifFalse: [ ^ false ].
	
	(self propertiesSize == anObject propertiesSize) ifFalse: [ ^ false ].

	self keysAndValuesDo: [ :key :value |
		(anObject hasProperty: key) ifFalse: [ ^ false ].
		((anObject propertyAt: key) = value) ifFalse: [ ^ false ]. 
 	].
	
	^ true.
%
category: 'comparing'
method: PPContextMemento
hash
	^ (position hash bitXor: stream hash) bitXor: properties hash.
%
category: 'accessing - properties'
method: PPContextMemento
hasProperty: aKey
	"Test if the property aKey is present."
	
	^ properties ~~ nil  and: [ properties includesKey: aKey ]
%
category: 'accessing - properties'
method: PPContextMemento
keysAndValuesDo: aBlock
	properties ifNil: [ ^ self ].
	properties keysAndValuesDo: [ :key :value | aBlock value: key value: value copy ]
%
category: 'accessing'
method: PPContextMemento
position
	^ position
%
category: 'accessing'
method: PPContextMemento
position: anInteger
	position := anInteger
%
category: 'accessing - properties'
method: PPContextMemento
propertiesSize
	properties ifNil: [ ^ 0 ].
	^ properties size.
%
category: 'accessing - properties'
method: PPContextMemento
propertyAt: aKey
	"Answer the property value associated with aKey."
	
	^ self propertyAt: aKey ifAbsent: [ self error: 'Property not found' ]
%
category: 'accessing - properties'
method: PPContextMemento
propertyAt: aKey ifAbsent: aBlock
	"Answer the property value associated with aKey or, if aKey isn't found, answer the result of evaluating aBlock."
	
	properties == nil 
		ifTrue: [ ^ aBlock value ]
		ifFalse: [ 
			(properties includesKey: aKey) ifTrue: [ 
				^ (properties at: aKey) copy
			].
			^ aBlock value
		]
%
category: 'accessing - properties'
method: PPContextMemento
propertyAt: aKey ifAbsentPut: aBlock
	"Answer the property associated with aKey or, if aKey isn't found store the result of evaluating aBlock as new value."
	
	^ self propertyAt: aKey ifAbsent: [ self propertyAt: aKey put: aBlock value ]
%
category: 'accessing - properties'
method: PPContextMemento
propertyAt: aKey put: anObject
	"Set the property at aKey to be anObject. If aKey is not found, create a new entry for aKey and set is value to anObject. Answer anObject."

	^ (properties ifNil: [ properties := Dictionary new: 1 ])
		at: aKey put: (anObject copy)
%
category: 'accessing - properties'
method: PPContextMemento
removeProperty: aKey
	"Remove the property with aKey. Answer the property or raise an error if aKey isn't found."
	
	^ self removeProperty: aKey ifAbsent: [ self error: 'Property not found' ]
%
category: 'accessing - properties'
method: PPContextMemento
removeProperty: aKey ifAbsent: aBlock
	"Remove the property with aKey. Answer the value or, if aKey isn't found, answer the result of evaluating aBlock."
	
	| answer |
	properties == nil  ifTrue: [ ^ aBlock value ].
	answer := properties removeKey: aKey ifAbsent: aBlock.
	properties isEmpty ifTrue: [ properties := nil ].
	^ answer
%
category: 'accessing'
method: PPContextMemento
stream
	^ stream
%
category: 'accessing'
method: PPContextMemento
stream: aStream
	stream := aStream
%
expectValue %Boolean
doit
PPContextMemento category: 'PetitParser-Core'.
true
%


! Remove existing behavior from PPDelegateParser
removeallmethods PPDelegateParser
removeallclassmethods PPDelegateParser
! ------------------- Class methods for PPDelegateParser
category: 'instance creation'
classmethod: PPDelegateParser
on: aParser
	^ self new setParser: aParser
%
! ------------------- Instance methods for PPDelegateParser
category: 'accessing'
method: PPDelegateParser
children
	^ Array with: parser
%
category: 'parsing'
method: PPDelegateParser
parseOn: aPPContext
	^ parser parseOn: aPPContext
%
category: 'initialization'
method: PPDelegateParser
setParser: aParser
	parser := aParser
%
expectValue %Boolean
doit
PPDelegateParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPEndOfFileParser
removeallmethods PPEndOfFileParser
removeallclassmethods PPEndOfFileParser
! ------------------- Class methods for PPEndOfFileParser
! ------------------- Instance methods for PPEndOfFileParser
category: 'parsing'
method: PPEndOfFileParser
parseOn: aPPContext
	(aPPContext atEnd) ifFalse:
	[
		^ PPFailure message: 'end of input expected' context: aPPContext.
	].
	^ #'end-of-input'
%
expectValue %Boolean
doit
PPEndOfFileParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPEndOfInputParser
removeallmethods PPEndOfInputParser
removeallclassmethods PPEndOfInputParser
! ------------------- Class methods for PPEndOfInputParser
! ------------------- Instance methods for PPEndOfInputParser
category: 'operators'
method: PPEndOfInputParser
end
	^ self
%
category: 'parsing'
method: PPEndOfInputParser
parseOn: aPPContext
	| memento result |
	memento := aPPContext remember.
	result := parser parseOn: aPPContext.
	(result isPetitFailure or: [ aPPContext stream atEnd ])
		ifTrue: [ ^ result ].
	result := PPFailure
		message: 'end of input expected'
		context: aPPContext.
	aPPContext restore: memento.
	^ result
%
expectValue %Boolean
doit
PPEndOfInputParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPEndOfLineParser
removeallmethods PPEndOfLineParser
removeallclassmethods PPEndOfLineParser
! ------------------- Class methods for PPEndOfLineParser
! ------------------- Instance methods for PPEndOfLineParser
category: 'parsing'
method: PPEndOfLineParser
parseOn: aPPContext
	(aPPContext isEndOfLine) ifTrue: [ 
		^ #endOfLine
	].
	^ PPFailure message: 'End of line expected' context: aPPContext at: aPPContext position
%
expectValue %Boolean
doit
PPEndOfLineParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPEpsilonParser
removeallmethods PPEpsilonParser
removeallclassmethods PPEpsilonParser
! ------------------- Class methods for PPEpsilonParser
! ------------------- Instance methods for PPEpsilonParser
category: 'parsing'
method: PPEpsilonParser
parseOn: aStream
	^ nil
%
expectValue %Boolean
doit
PPEpsilonParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPExpressionParser
removeallmethods PPExpressionParser
removeallclassmethods PPExpressionParser
! ------------------- Class methods for PPExpressionParser
! ------------------- Instance methods for PPExpressionParser
category: 'private'
method: PPExpressionParser
build: aParser left: aChoiceParser
	^ (aParser separatedBy: aChoiceParser) foldLeft: [ :a :op :b | op first value: a value: op second value: b ]
%
category: 'private'
method: PPExpressionParser
build: aParser postfix: aChoiceParser
	^ aParser , aChoiceParser star map: [ :term :ops | ops inject: term into: [ :result :operator | operator first value: result value: operator second ] ]
%
category: 'private'
method: PPExpressionParser
build: aParser prefix: aChoiceParser
	^ aChoiceParser star , aParser map: [ :ops :term | ops reverse inject: term into: [ :result :operator | operator first value: operator second value: result ] ]
%
category: 'private'
method: PPExpressionParser
build: aParser right: aChoiceParser
	^ (aParser separatedBy: aChoiceParser) foldRight: [ :a :op :b | op first value: a value: op second value: b ]
%
category: 'private'
method: PPExpressionParser
buildOn: aParser
	^ self buildSelectors inject: aParser into: [ :term :selector |
		| list |
		list := operators at: selector ifAbsent: [ #() ].
		list isEmpty
			ifTrue: [ term ]
			ifFalse: [
				self
					perform: selector with: term 
					with: (list size = 1
						ifTrue: [ list first first ==> [ :operator | Array with: list first second with: operator ] ]
						ifFalse: [ 
							list
								inject: PPChoiceParser new
								into: [ :choice :each | choice / (each first ==> [ :operator | Array with: each second with: operator ]) ] ]) ] ]
%
category: 'private'
method: PPExpressionParser
buildSelectors
	^ #(build:prefix: build:postfix: build:right: build:left:)
%
category: 'specifying'
method: PPExpressionParser
group: aOneArgumentBlock
	"Defines a priority group by evaluating aOneArgumentBlock."
	
	operators := Dictionary new.
	parser := [ 
		aOneArgumentBlock value: self.
	 	self buildOn: parser ]
			ensure: [ operators := nil ]
%
category: 'specifying'
method: PPExpressionParser
left: aParser do: aThreeArgumentBlock
	"Define an operator aParser that is left-associative. Evaluate aThreeArgumentBlock with the first term, the operator, and the second term."
	
	self operator: #build:left: parser: aParser do: aThreeArgumentBlock
%
category: 'private'
method: PPExpressionParser
operator: aSymbol parser: aParser do: aBlock
	parser == nil 
		ifTrue: [ ^ self error: 'You did not specify a term when creating the receiver.' ].
	operators == nil 
		ifTrue: [ ^ self error: 'Use #group: to define precedence groups in descending order.' ].
	(operators at: aSymbol ifAbsentPut: [ OrderedCollection new ])
		addLast: (Array with: aParser asParser with: aBlock)
%
category: 'specifying'
method: PPExpressionParser
postfix: aParser do: aTwoArgumentBlock
	"Define a postfix operator aParser. Evaluate aTwoArgumentBlock with the term and the operator."

	self operator: #build:postfix: parser: aParser do: aTwoArgumentBlock
%
category: 'specifying'
method: PPExpressionParser
prefix: aParser do: aTwoArgumentBlock
	"Define a prefix operator aParser. Evaluate aTwoArgumentBlock with the operator and the term."

	self operator: #build:prefix: parser: aParser do: aTwoArgumentBlock
%
category: 'specifying'
method: PPExpressionParser
right: aParser do: aThreeArgumentBlock
	"Define an operator aParser that is right-associative. Evaluate aThreeArgumentBlock with the first term, the operator, and the second term."
	
	self operator: #build:right: parser: aParser do: aThreeArgumentBlock
%
category: 'specifying'
method: PPExpressionParser
term: aParser
	"Defines the initial term aParser of the receiver."
	
	parser == nil 
		ifTrue: [ parser := aParser ]
		ifFalse: [ self error: 'Unable to redefine the term.' ]
%
expectValue %Boolean
doit
PPExpressionParser category: 'PetitParser-Tools'.
true
%

! Remove existing behavior from PPFailingParser
removeallmethods PPFailingParser
removeallclassmethods PPFailingParser
! ------------------- Class methods for PPFailingParser
category: 'instance creation'
classmethod: PPFailingParser
message: aString
	^ self new setMessage: aString
%
! ------------------- Instance methods for PPFailingParser
category: 'accessing'
method: PPFailingParser
message
	"Answer the error message of the receiving parser."

	^ message
%
category: 'parsing'
method: PPFailingParser
parseOn: aPPContext
	^ PPFailure message: message context: aPPContext
%
category: 'printing'
method: PPFailingParser
printNameOn: aStream
	super printNameOn: aStream.
	aStream nextPutAll: ', '; print: message
%
category: 'initialization'
method: PPFailingParser
setMessage: aString
	message := aString
%
expectValue %Boolean
doit
PPFailingParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPFailure
removeallmethods PPFailure
removeallclassmethods PPFailure
! ------------------- Class methods for PPFailure
category: 'instance creation'
classmethod: PPFailure
message: aString
	^ self basicNew initializeMessage: aString
%
category: 'instance creation'
classmethod: PPFailure
message: aString at: anInteger
	"One should not use this method if the furthest failure is supposed to be reported correctly"
	^ self basicNew initializeMessage: aString at: anInteger
%
category: 'instance creation'
classmethod: PPFailure
message: aString context: aPPContext
	^ self basicNew initializeMessage: aString context: aPPContext
%
category: 'instance creation'
classmethod: PPFailure
message: aString context: aPPContext at: position
	^ self basicNew initializeMessage: aString context: aPPContext position: position
%
! ------------------- Instance methods for PPFailure
category: 'initialization'
method: PPFailure
initializeMessage: aString	
	message := aString.
%
category: 'initialization'
method: PPFailure
initializeMessage: aString at: anInteger
	"One should not use this method if the furthest failure is supposed to be reported correctly"
	message := aString.
	position := anInteger.
%
category: 'initialization'
method: PPFailure
initializeMessage: aString context: aPPContext
	self initializeMessage: aString context:  aPPContext position: aPPContext position
%
category: 'initialization'
method: PPFailure
initializeMessage: aString context: aPPContext position: anInteger
	message := aString.
	context := aPPContext.
	position := anInteger.
	
	"record the furthest failure encountered while parsing the input stream "
	aPPContext noteFailure: self.	
%
category: 'testing'
method: PPFailure
isPetitFailure
	"I am the only class that should implement this method to return true."

	^ true
%
category: 'accessing'
method: PPFailure
message
	"Answer a human readable error message of this parse failure."
	
	^ message
%
category: 'accessing'
method: PPFailure
position
	"Answer the position in the source string that caused this parse failure."

	^ position
%
category: 'printing'
method: PPFailure
printOn: aStream
	aStream 
		nextPutAll: (self message ifNil: ['<message not specified>']); 
		nextPutAll: ' at '; print: self position
%
expectValue %Boolean
doit
PPFailure category: 'PetitParser-Core'.
true
%

! Remove existing behavior from PPFlattenParser
removeallmethods PPFlattenParser
removeallclassmethods PPFlattenParser
! ------------------- Class methods for PPFlattenParser
! ------------------- Instance methods for PPFlattenParser
category: 'private'
method: PPFlattenParser
on: aCollection start: aStartInteger stop: aStopInteger value: anObject
	^ aCollection copyFrom: aStartInteger to: aStopInteger
%
category: 'parsing'
method: PPFlattenParser
parseOn: aPPContext
	| start element |
	start := aPPContext position.
	element := parser parseOn: aPPContext.
	element isPetitFailure ifTrue: [ ^ element ].
	^ self on: aPPContext stream collection start: start + 1 stop: aPPContext position value: element
%
expectValue %Boolean
doit
PPFlattenParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPForwardingParser
removeallmethods PPForwardingParser
removeallclassmethods PPForwardingParser
! ------------------- Class methods for PPForwardingParser
category: 'instance creation'
classmethod: PPForwardingParser
to: forwardee

	^self new forwardee: forwardee
%
! ------------------- Instance methods for PPForwardingParser
category: 'forwarding'
method: PPForwardingParser
doesNotUnderstand: messageDescriptor
	"Forward them all"

	^forwardee
		perform: (messageDescriptor at: 1)
		env: 0
		withArguments: (messageDescriptor at: 2)
%
category: 'accessing'
method: PPForwardingParser
forwardee: aParser

	forwardee := aParser
%
category: 'testing'
method: PPForwardingParser
isPetitParser
	"Must override Object's implementation, or forward. Easier to override."

	^true
%
expectValue %Boolean
doit
PPForwardingParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPGreedyRepeatingParser
removeallmethods PPGreedyRepeatingParser
removeallclassmethods PPGreedyRepeatingParser
! ------------------- Class methods for PPGreedyRepeatingParser
! ------------------- Instance methods for PPGreedyRepeatingParser
category: 'parsing'
method: PPGreedyRepeatingParser
parseOn: aPPContext
	| memento element elements positions |
	memento := aPPContext remember.
	elements := OrderedCollection new.
	[ elements size < min ] whileTrue: [ 
		(element := parser parseOn: aPPContext) isPetitFailure ifTrue: [ 
			aPPContext restore: memento.
			^ element ].
		elements addLast: element ].
	positions := OrderedCollection with: aPPContext remember.
	[ elements size < max and: [ (element := parser parseOn: aPPContext) isPetitFailure not ] ] whileTrue: [
		elements addLast: element.
		positions addLast: aPPContext remember ].
	[ positions isEmpty ] whileFalse: [
		aPPContext restore: positions last.
		element := limit parseOn: aPPContext.
		element isPetitFailure ifFalse: [
			aPPContext restore: positions last.
			^ elements asArray ].
		elements isEmpty ifTrue: [
			aPPContext restore: memento.
			^ element ].
		elements removeLast.
		positions removeLast ].
	aPPContext restore: memento.
	^ PPFailure message: 'overflow' context: aPPContext at: memento position
%
expectValue %Boolean
doit
PPGreedyRepeatingParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPLazyRepeatingParser
removeallmethods PPLazyRepeatingParser
removeallclassmethods PPLazyRepeatingParser
! ------------------- Class methods for PPLazyRepeatingParser
! ------------------- Instance methods for PPLazyRepeatingParser
category: 'parsing'
method: PPLazyRepeatingParser
parseOn: aPPContext
	| memento element elements |
	memento := aPPContext remember.
	elements := OrderedCollection new.
	[ elements size < min ] whileTrue: [
		(element := parser parseOn: aPPContext) isPetitFailure ifTrue: [
			aPPContext restore: memento.
			^ element ].
		elements addLast: element ].
	[ self matchesLimitOn: aPPContext ] whileFalse: [
		elements size < max ifFalse: [
			aPPContext restore: memento.
			^ PPFailure message: 'overflow' context: aPPContext at: memento position ].
		element := parser parseOn: aPPContext.
		element isPetitFailure ifTrue: [
			aPPContext restore: memento.
			^ element ].
		elements addLast: element ].
	^ elements asArray
%
expectValue %Boolean
doit
PPLazyRepeatingParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPLimitedChoiceParser
removeallmethods PPLimitedChoiceParser
removeallclassmethods PPLimitedChoiceParser
! ------------------- Class methods for PPLimitedChoiceParser
! ------------------- Instance methods for PPLimitedChoiceParser
category: 'as yet unclassified'
method: PPLimitedChoiceParser
// aRule 
	^ self copyWith: aRule
%
category: 'as yet unclassified'
method: PPLimitedChoiceParser
initialize
	limit := nil asParser
%
category: 'accessing'
method: PPLimitedChoiceParser
limit
	
	^ limit
%
category: 'accessing'
method: PPLimitedChoiceParser
limit: anObject
	
	limit := anObject
%
category: 'as yet unclassified'
method: PPLimitedChoiceParser
parseOn: aPPContext
	"This is optimized code that avoids unnecessary block activations, do not change. When all choices fail, the last failure is answered."

	| element limitResult memento |
	"self halt."
	1 to: parsers size do: [ :index |
		memento := aPPContext remember.
		
		element := (parsers at: index)
			parseOn: aPPContext.
		
		(element isPetitFailure not) ifTrue: [ 
			"check limit"
			limitResult := limit parseOn: aPPContext.
			limitResult isPetitFailure ifTrue: [ 
				element := PPFailure message: 'limit failed' at: aPPContext position .
				aPPContext restore: memento.
			] ifFalse: [ ^ element ].
		].
	].	
	^ element
%
expectValue %Boolean
doit
PPLimitedChoiceParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPLimitedRepeatingParser
removeallmethods PPLimitedRepeatingParser
removeallclassmethods PPLimitedRepeatingParser
! ------------------- Class methods for PPLimitedRepeatingParser
category: 'instance creation'
classmethod: PPLimitedRepeatingParser
on: aParser limit: aLimitParser
	^ (self on: aParser) setLimit: aLimitParser
%
! ------------------- Instance methods for PPLimitedRepeatingParser
category: 'accessing'
method: PPLimitedRepeatingParser
children
	^ Array with: parser with: limit
%
category: 'accessing'
method: PPLimitedRepeatingParser
limit
	"Answer the parser that limits (or ends) this repetition."
	
	^ limit
%
category: 'private'
method: PPLimitedRepeatingParser
matchesLimitOn: aPPContext
	| element position |
	position := aPPContext remember.
	element := limit parseOn: aPPContext.
	aPPContext restore: position.
	^ element isPetitFailure not
%
category: 'initialization'
method: PPLimitedRepeatingParser
setLimit: aParser
	limit := aParser
%
expectValue %Boolean
doit
PPLimitedRepeatingParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPListParser
removeallmethods PPListParser
removeallclassmethods PPListParser
! ------------------- Class methods for PPListParser
category: 'instance creation'
classmethod: PPListParser
with: aParser
	^ self withAll: (Array with: aParser)
%
category: 'instance creation'
classmethod: PPListParser
with: aFirstParser with: aSecondParser
	^ self withAll: (Array with: aFirstParser with: aSecondParser)
%
category: 'instance creation'
classmethod: PPListParser
withAll: aCollection
	^ self basicNew initialize;
		setParsers: aCollection
%
! ------------------- Instance methods for PPListParser
category: 'accessing'
method: PPListParser
children
	^ parsers
%
category: 'copying'
method: PPListParser
copyWith: aParser
	^ self species withAll: (parsers copyWith: aParser)
%
category: 'initialization'
method: PPListParser
initialize
	super initialize.
	self setParsers: #()
%
category: 'copying'
method: PPListParser
postCopy
	super postCopy.
	parsers := parsers copy
%
category: 'initialization'
method: PPListParser
setParsers: aCollection
	parsers := aCollection asArray
%
expectValue %Boolean
doit
PPListParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPLiteralObjectParser
removeallmethods PPLiteralObjectParser
removeallclassmethods PPLiteralObjectParser
! ------------------- Class methods for PPLiteralObjectParser
! ------------------- Instance methods for PPLiteralObjectParser
category: 'operators'
method: PPLiteralObjectParser
caseInsensitive
	"Answer a parser that can parse the receiver case-insensitive."
	
	literal asUppercase = literal asLowercase ifTrue: [ ^ self ].
	^ PPPredicateObjectParser on: [ :value | literal "sameAs:" isEquivalent: value ] message: message
%
category: 'operators'
method: PPLiteralObjectParser
negate
	^ (PPPredicateObjectParser expect: literal message: message) negate
%
category: 'parsing'
method: PPLiteralObjectParser
parseOn: aPPContext
	^ (aPPContext stream atEnd not and: [ literal = aPPContext stream uncheckedPeek ])
		ifFalse: [ PPFailure message: message context: aPPContext ]
		ifTrue: [ aPPContext next ]
%
expectValue %Boolean
doit
PPLiteralObjectParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPLiteralParser
removeallmethods PPLiteralParser
removeallclassmethods PPLiteralParser
! ------------------- Class methods for PPLiteralParser
category: 'instance creation'
classmethod: PPLiteralParser
on: anObject
	^ self on: anObject message: anObject printString , ' expected'
%
category: 'instance creation'
classmethod: PPLiteralParser
on: anObject message: aString
	^ self new initializeOn: anObject message: aString
%
! ------------------- Instance methods for PPLiteralParser
category: 'operators'
method: PPLiteralParser
caseInsensitive
	"Answer a parser that can parse the receiver case-insensitive."
	
	self subclassResponsibility
%
category: 'initialization'
method: PPLiteralParser
initializeOn: anObject message: aString
	literal := anObject.
	message := aString
%
category: 'accessing'
method: PPLiteralParser
literal
	"Answer the parsed literal."

	^ literal
%
category: 'accessing'
method: PPLiteralParser
message
	"Answer the failure message."
	
	^ message
%
category: 'printing'
method: PPLiteralParser
printNameOn: aStream
	super printNameOn: aStream.
	aStream nextPutAll: ', '; print: literal
%
expectValue %Boolean
doit
PPLiteralParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPLiteralSequenceParser
removeallmethods PPLiteralSequenceParser
removeallclassmethods PPLiteralSequenceParser
! ------------------- Class methods for PPLiteralSequenceParser
! ------------------- Instance methods for PPLiteralSequenceParser
category: 'operators'
method: PPLiteralSequenceParser
caseInsensitive
	"Answer a parser that can parse the receiver case-insensitive."
	
	literal asUppercase = literal asLowercase ifTrue: [ ^ self ].
	^ PPPredicateSequenceParser on: [ :value | literal "sameAs:" isEquivalent: value ] message: message size: size
%
category: 'initialization'
method: PPLiteralSequenceParser
initializeOn: anObject message: aString
	super initializeOn: anObject message: aString.
	size := literal size
%
category: 'parsing'
method: PPLiteralSequenceParser
parseOn: aPPContext
	| memento result |
	memento := aPPContext remember.
	result := aPPContext next: size.
	literal = result ifTrue: [ ^ result ].
	aPPContext restore: memento.
	^ PPFailure message: message context: aPPContext
%
category: 'accessing'
method: PPLiteralSequenceParser
size
	"Answer the sequence size of the receiver."

	^ size
%
expectValue %Boolean
doit
PPLiteralSequenceParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPMemento
removeallmethods PPMemento
removeallclassmethods PPMemento
! ------------------- Class methods for PPMemento
category: 'instance creation'
classmethod: PPMemento
new
	^ self basicNew initialize
%
! ------------------- Instance methods for PPMemento
category: 'accessing-readonly'
method: PPMemento
count
	^ count
%
category: 'actions'
method: PPMemento
increment
	count := count + 1
%
category: 'initialization'
method: PPMemento
initialize
	count := 0
	
%
category: 'accessing'
method: PPMemento
contextMemento
	^ context
%
category: 'accessing'
method: PPMemento
contextMemento: aPPContextMemento
	context  := aPPContextMemento 
%
category: 'accessing'
method: PPMemento
result
	^ result
%
category: 'accessing'
method: PPMemento
result: anObject
	result := anObject
%
expectValue %Boolean
doit
PPMemento category: 'PetitParser-Core'.
true
%

! Remove existing behavior from PPMemoizedParser
removeallmethods PPMemoizedParser
removeallclassmethods PPMemoizedParser
! ------------------- Class methods for PPMemoizedParser
! ------------------- Instance methods for PPMemoizedParser
category: 'parsing'
method: PPMemoizedParser
check: aPPContext
	(identifier == aPPContext identifier)
		ifFalse: [ self reset: aPPContext ].
%
category: 'operators'
method: PPMemoizedParser
memoized
	"Ther is no point in memoizing more than once."

	^ self
%
category: 'operators'
method: PPMemoizedParser
nonMemoized
	^ parser
%
category: 'parsing'
method: PPMemoizedParser
parseOn: aPPContext
	| memento contextMemento  aStream |
	"TODO: JK memoizing needs review!"
	self check: aPPContext.
	contextMemento := aPPContext remember.
	memento := (buffer at: contextMemento ifAbsentPut: [ PPMemento new ]).
	
	memento contextMemento isNil
		ifTrue: [
			aStream := aPPContext stream.
			memento result: (aStream size - aStream position + 2 < memento count
				ifTrue: [ PPFailure message: 'overflow' context: aPPContext ]
				ifFalse: [ memento increment. parser parseOn: aPPContext ]).
			memento contextMemento: aPPContext remember ]
		ifFalse: [ aPPContext restore: memento contextMemento ].
	^ memento result.
%
category: 'private'
method: PPMemoizedParser
reset: aPPContext
	buffer := Dictionary new.
	identifier := aPPContext identifier.
%
expectValue %Boolean
doit
PPMemoizedParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPNotParser
removeallmethods PPNotParser
removeallclassmethods PPNotParser
! ------------------- Class methods for PPNotParser
! ------------------- Instance methods for PPNotParser
category: 'parsing'
method: PPNotParser
parseOn: aPPContext
	| element memento |
	memento := aPPContext remember.
	element := parser parseOn: aPPContext.
	aPPContext restore: memento.
	^ element isPetitFailure
		ifFalse: [ PPFailure message: '' context: aPPContext ]
%
expectValue %Boolean
doit
PPNotParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPOptionalParser
removeallmethods PPOptionalParser
removeallclassmethods PPOptionalParser
! ------------------- Class methods for PPOptionalParser
! ------------------- Instance methods for PPOptionalParser
category: 'parsing'
method: PPOptionalParser
parseOn: aPPContext
	| element |
	element := parser parseOn: aPPContext.
	^ element isPetitFailure ifFalse: [ element ]
%
expectValue %Boolean
doit
PPOptionalParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPParser
removeallmethods PPParser
removeallclassmethods PPParser
! ------------------- Class methods for PPParser
category: 'instance creation'
classmethod: PPParser
named: aString
	^ self new name: aString
%
category: 'instance creation'
classmethod: PPParser
new
	^ self basicNew initialize
%
! ------------------- Instance methods for PPParser
category: 'operators'
method: PPParser
, aParser 
	"Answer a new parser that parses the receiver followed by aParser."

	^ PPSequenceParser with: self with: aParser
%
category: 'operators'
method: PPParser
/ aParser 
	"Answer a new parser that parses the receiver, if the receiver fails try with aParser (ordered-choice)."
	
	^ PPChoiceParser with: self with: aParser
%
category: 'operators'
method: PPParser
// aParser 
	"
		Answer a new parser that parses the receiver, if the receiver fails try with aParser (ordered-choice).
		If the receiver passes, limit must pass as well.
	"
	
	^ PPLimitedChoiceParser with: self with: aParser
%
category: 'operators-mapping'
method: PPParser
==> aBlock
	"Answer a new parser that performs aBlock as action handler on success."

	^ PPActionParser on: self block: aBlock
%
category: 'operators-mapping'
method: PPParser
>=> aBlock
	"Answer a new parser that wraps the receiving parser with a two argument block. The first argument is the parsed stream, the second argument a continuation block on the delegate parser."

	^ PPWrappingParser on: self block: aBlock
%
category: 'enumerating'
method: PPParser
allParsers
	"Answer all the parse nodes of the receiver."

	| result |
	result := OrderedCollection new.
	self allParsersDo: [ :parser | result addLast: parser ].
	^ result
%
category: 'enumerating'
method: PPParser
allParsersDo: aBlock
	"Iterate over all the parse nodes of the receiver."

	self allParsersDo: aBlock seen: IdentitySet new
%
category: 'enumerating'
method: PPParser
allParsersDo: aBlock seen: aSet
	"Iterate over all the parse nodes of the receiver, do not visit and follow the ones contained in aSet."

	(aSet includes: self)
		ifTrue: [ ^ self ].
	aSet add: self.
	aBlock value: self.
	self children
		do: [ :each | each allParsersDo: aBlock seen: aSet ]
%
category: 'operators'
method: PPParser
and
	"Answer a new parser (logical and-predicate) that succeeds whenever the receiver does, but never consumes input."

	^ PPAndParser on: self
%
category: 'operators-mapping'
method: PPParser
answer: anObject
	"Answer a new parser that always returns anObject from a successful parse."

	^ self ==> [ :nodes | anObject ]
%
category: 'converting'
method: PPParser
asParser
	"Answer the receiving parser."
	
	^ self
%
category: 'accessing'
method: PPParser
child
	self assert: (self children size == 1).
	^ self children first
%
category: 'accessing'
method: PPParser
children
	"Answer a set of child parsers that could follow the receiver."

	^ #()
%
category: 'operators'
method: PPParser
def: aParser
	"Redefine the receiver as the argument aParser. This method is useful when defining recursive parsers: instantiate a PPUnresolvedParser and later redefine it with another one."

	"^ self becomeForward: (aParser name: self name)"
	aParser name: self name.
	self become: (PPForwardingParser to: aParser).
	^self
%
category: 'operators-convenience'
method: PPParser
delimitedBy: aParser
	"Answer a new parser that parses the receiver one or more times, separated and possibly ended by aParser."
	
	^ (self separatedBy: aParser) , (aParser optional) ==> [ :node |
		node second == nil 
			ifTrue: [ node first ]
			ifFalse: [ node first copyWith: node second ] ]
%
category: 'operators'
method: PPParser
end
	"Answer a new parser that succeeds at the end of the input and return the result of the receiver."

	^ PPEndOfInputParser on: self
%
category: 'operators-mapping'
method: PPParser
flatten
	"Answer a new parser that flattens the underlying collection."
	
	^ PPFlattenParser on: self
%
category: 'operators-mapping'
method: PPParser
foldLeft: aBlock
	"Answer a new parser that that folds the result of the receiver from left-to-right into aBlock. The argument aBlock must take two or more arguments."
	
	| size args |
	size := aBlock numArgs.
	args := Array new: size.
	^ self ==> [ :nodes |
		args at: 1 put: nodes first.
		2 to: nodes size by: size - 1 do: [ :index |
			args
				replaceFrom: 2 to: size with: nodes startingAt: index;
				at: 1 put: (aBlock valueWithArguments: args) ].
		args first ]
%
category: 'operators-mapping'
method: PPParser
foldRight: aBlock
	"Answer a new parser that that folds the result of the receiver from right-to-left into aBlock. The argument aBlock must take two or more arguments."

	| size args |
	size := aBlock numArgs.
	args := Array new: size.
	^ self ==> [ :nodes |
		args at: size put: nodes last.
		nodes size - size + 1 to: 1 by: 1 - size do: [ :index |
			args
				replaceFrom: 1 to: size - 1 with: nodes startingAt: index;
				at: size put: (aBlock valueWithArguments: args) ].
		args at: size ]
%
category: 'accessing-properties'
method: PPParser
hasProperty: aKey
	"Test if the property aKey is present."
	
	^ properties ~~ nil  and: [ properties includesKey: aKey ]
%
category: 'operators'
method: PPParser
if: aBlock
	^ PPConditionalParser on: self block: aBlock
%
category: 'initialization'
method: PPParser
initialize
%
category: 'testing'
method: PPParser
isPetitParser
	^ true
%
category: 'testing'
method: PPParser
isUnresolved
	^ false
%
category: 'operators-mapping'
method: PPParser
map: aBlock
	"Answer a new parser that works on the receiving sequence an passes in each element as a block argument."
	
	^ aBlock numArgs = 1
		ifTrue: [ self ==> aBlock ]
		ifFalse: [ self error: aBlock numArgs asString , ' arguments expected.' ]
%
category: 'parsing'
method: PPParser
matches: anObject
	"Answer if anObject can be parsed by the receiver."
	
	^ (self parse: anObject) isPetitFailure not
%
category: 'parsing'
method: PPParser
matchesIn: anObject
	"Search anObject repeatedly for the matches of the receiver. Answered an OrderedCollection of the matched parse-trees."

	| result |
	result := OrderedCollection new.
	self 
		matchesIn: anObject
		do: [ :each | result addLast: each ].
	^ result
%
category: 'parsing'
method: PPParser
matchesIn: anObject do: aBlock
	"Search anObject repeatedly for the matches of the receiver. Evaluate aBlock for each match with the matched parse-tree as the argument. Make sure to always consume exactly one character with each step, to not miss any match."

	((self and ==> aBlock , #any asParser) / #any asParser) star parse: anObject
%
category: 'parsing'
method: PPParser
matchesSkipIn: anObject
	"Search anObject repeatedly for the matches of the receiver. Answer an OrderedCollection of the matched parse-trees. Skip over matches."

	| result |
	result := OrderedCollection new.
	self 
		matchesSkipIn: anObject
		do: [ :each | result addLast: each ].
	^ result
%
category: 'parsing'
method: PPParser
matchesSkipIn: anObject do: aBlock
	"Search anObject repeatedly for the matches of the receiver. Evaluate aBlock for each match with the matched parse-tree as the argument. Skip over matches."

	(self ==> aBlock / #any asParser) star parse: anObject
%
category: 'parsing'
method: PPParser
matchingRangesIn: anObject
	"Search anObject repeatedly for the matches of the receiver. Answer an OrderedCollection of ranges of each match (index of first character to: index of last character)."
	
	| result |
	result := OrderedCollection new.
	self
		matchingRangesIn: anObject
		do: [ :value | result addLast: value ].
	^ result
%
category: 'parsing'
method: PPParser
matchingRangesIn: anObject do: aBlock
	"Search anObject repeatedly for the matches of the receiver. Evaluate aBlock with the range of each match (index of first character to: index of last character)."
	
	self token
		matchesIn: anObject
		do: [ :token | aBlock value: (token start to: token stop) ]
%
category: 'parsing'
method: PPParser
matchingSkipRangesIn: anObject
	"Search anObject repeatedly for the matches of the receiver. Skip over matches. Answer an OrderedCollection of ranges of each match (index of first character to: index of last character)."
	
	| result |
	result := OrderedCollection new.
	self
		matchingSkipRangesIn: anObject
		do: [ :value | result addLast: value ].
	^ result
%
category: 'parsing'
method: PPParser
matchingSkipRangesIn: anObject do: aBlock
	"Search anObject repeatedly for the matches of the receiver. Skip over matches. Evaluate aBlock with the range of each match (index of first character to: index of last character)."
	
	self token
		matchesSkipIn: anObject
		do: [ :token | aBlock value: (token start to: token stop) ]
%
category: 'operators-repeating'
method: PPParser
max: anInteger
	"Answer a new parser that parses the receiver at most anInteger times."
	
	^ self star setMax: anInteger
%
category: 'operators-repeating'
method: PPParser
max: anInteger greedy: aParser
	"Answer a new parser that parses the receiver at most anInteger times until it reaches aParser. This is a greedy non-blind implementation. aParser is not consumed."
	
	^ (self starGreedy: aParser) setMax: anInteger
%
category: 'operators-repeating'
method: PPParser
max: anInteger lazy: aParser
	"Answer a new parser that parses the receiver at most anInteger times until it reaches aParser. This is a lazy non-blind implementation. aParser is not consumed."
	
	^ (self starLazy: aParser) setMax: anInteger
%
category: 'operators'
method: PPParser
memoized
	"Answer a new memoized parser, for refraining redundant computations. This ensures polynomial time O(n^4) for left-recursive grammars and O(n^3) for non left-recursive grammars in the worst case. Not necessary for most grammars that are carefully written and in O(n) anyway."
	
	^ PPMemoizedParser on: self
%
category: 'operators-repeating'
method: PPParser
min: anInteger
	"Answer a new parser that parses the receiver at least anInteger times."
	
	^ self star setMin: anInteger
%
category: 'operators-repeating'
method: PPParser
min: anInteger greedy: aParser
	"Answer a new parser that parses the receiver at least anInteger times until it reaches aParser. This is a greedy non-blind implementation. aParser is not consumed."
	
	^ (self starGreedy: aParser) setMin: anInteger
%
category: 'operators-repeating'
method: PPParser
min: anInteger lazy: aParser
	"Answer a new parser that parses the receiver at least anInteger times until it reaches aParser. This is a lazy non-blind implementation. aParser is not consumed."
	
	^ (self starLazy: aParser) setMin: anInteger
%
category: 'operators-repeating'
method: PPParser
min: aMinInteger max: aMaxInteger
	"Answer a new parser that parses the receiver at least aMinInteger and at most aMaxInteger times."
	
	^ self star setMin: aMinInteger; setMax: aMaxInteger
%
category: 'operators-repeating'
method: PPParser
min: aMinInteger max: aMaxInteger greedy: aParser
	"Answer a new parser that parses the receiver at least aMinInteger and at most aMaxInteger times until it reaches aParser. This is a greedy non-blind implementation. aParser is not consumed."
	
	^ (self starGreedy: aParser) setMin: aMinInteger; setMax: aMaxInteger
%
category: 'operators-repeating'
method: PPParser
min: aMinInteger max: aMaxInteger lazy: aParser
	"Answer a new parser that parses the receiver at least aMinInteger and at most aMaxInteger times until it reaches aParser. This is a greedy non-blind implementation. aParser is not consumed."
	
	^ (self starLazy: aParser) setMin: aMinInteger; setMax: aMaxInteger
%
category: 'accessing'
method: PPParser
name
	"Answer the production name of the receiver."
	
	^ self propertyAt: #name ifAbsent: [ nil ]
%
category: 'accessing'
method: PPParser
name: aString
	self propertyAt: #name put: aString
%
category: 'operators'
method: PPParser
negate
	"Answer a new parser consumes any input token but the receiver."
	
	^ self not , #any asParser ==> #second
%
category: 'operators'
method: PPParser
nonMemoized
	^ self
%
category: 'operators'
method: PPParser
not
	"Answer a new parser (logical not-predicate) that succeeds whenever the receiver fails, but never consumes input."

	^ PPNotParser on: self
%
category: 'operators'
method: PPParser
optional
	"Answer a new parser that parses the receiver, if possible."

	^ PPOptionalParser on: self
%
category: 'parsing'
method: PPParser
parse: anObject
	"Parse anObject with the receiving parser and answer the parse-result or an instance of PPFailure."

	^ self parse: anObject withContext: PPContext new
%
category: 'parsing'
method: PPParser
parse: anObject onError: aBlock
	"Parse anObject with the receiving parser and answer the parse-result or answer the result of evaluating aBlock. Depending on the number of arguments of the block it is simply evaluated, evaluated with the failure object, or evaluated with the error message and position."
	
	| result |
	result := self parse: anObject.
	result isPetitFailure
		ifFalse: [ ^ result ].
	aBlock numArgs = 0
		ifTrue: [ ^ aBlock value ].
	aBlock numArgs = 1
		ifTrue: [ ^ aBlock value: result ].
	^ aBlock value: result message value: result position
%
category: 'context'
method: PPParser
parse: anObject withContext: aPPContext
	"Parse anObject with the receiving parser and answer the parse-result or an instance of PPFailure."

	aPPContext stream: anObject asPetitStream.
	^ self parseWithContext: aPPContext.
%
category: 'parsing'
method: PPParser
parseOn: aPPContext
	"Parse aStream with the receiving parser and answer the parse-result or an instance of PPFailure. Override this method in subclasses to specify custom parse behavior. Do not call this method from outside, instead use #parse:."
	
	self subclassResponsibility
%
category: 'context'
method: PPParser
parseWithContext: context
	| result |
	context initializeFor: self.
	result := self parseOn: context.
	
	"Return the furthest failure, it gives better results than the last failure"
	(result isPetitFailure and: [ context furthestFailure ~~ nil ]) 
		ifTrue: [ ^ context furthestFailure ].
	^ result
%
category: 'operators-repeating'
method: PPParser
plus
	"Answer a new parser that parses the receiver one or more times."

	^ self star setMin: 1
%
category: 'operators-repeating'
method: PPParser
plusGreedy: aParser
	"Answer a new parser that parses the receiver one or more times until it reaches aParser. This is a greedy non-blind implementation of the star operator. aParser is not consumed."
	
	^ (self starGreedy: aParser) setMin: 1
%
category: 'operators-repeating'
method: PPParser
plusLazy: aParser
	"Answer a new parser that parses the receiver one or more times until it reaches aParser. This is a lazy non-blind implementation of the star operator. aParser is not consumed."
	
	^ (self starLazy: aParser) setMin: 1
%
category: 'copying'
method: PPParser
postCopy
	super postCopy.
	properties := properties copy
%
category: 'printing'
method: PPParser
printNameOn: aStream
	self name == nil 
		ifTrue: [ aStream print: self hash ]
		ifFalse: [ aStream nextPutAll: self name ]
%
category: 'printing'
method: PPParser
printOn: aStream
	super printOn: aStream.
	aStream nextPut: $(.
	self printNameOn: aStream.
	aStream nextPut: $)
%
category: 'accessing-properties'
method: PPParser
properties
	^ properties
%
category: 'accessing-properties'
method: PPParser
propertyAt: aKey
	"Answer the property value associated with aKey."
	
	^ self propertyAt: aKey ifAbsent: [ self error: 'Property not found' ]
%
category: 'accessing-properties'
method: PPParser
propertyAt: aKey ifAbsent: aBlock
	"Answer the property value associated with aKey or, if aKey isn't found, answer the result of evaluating aBlock."
	
	^ properties == nil 
		ifTrue: [ aBlock value ]
		ifFalse: [ properties at: aKey ifAbsent: aBlock ]
%
category: 'accessing-properties'
method: PPParser
propertyAt: aKey ifAbsentPut: aBlock
	"Answer the property associated with aKey or, if aKey isn't found store the result of evaluating aBlock as new value."
	
	^ self propertyAt: aKey ifAbsent: [ self propertyAt: aKey put: aBlock value ]
%
category: 'accessing-properties'
method: PPParser
propertyAt: aKey put: anObject
	"Set the property at aKey to be anObject. If aKey is not found, create a new entry for aKey and set is value to anObject. Answer anObject."

	^ (properties ifNil: [ properties := Dictionary new: 1 ])
		at: aKey put: anObject
%
category: 'operators-mapping'
method: PPParser
queryToken
  ^ GsQueryTokenParser on: self
%
category: 'accessing-properties'
method: PPParser
removeProperty: aKey
	"Remove the property with aKey. Answer the property or raise an error if aKey isn't found."
	
	^ self removeProperty: aKey ifAbsent: [ self error: 'Property not found' ]
%
category: 'accessing-properties'
method: PPParser
removeProperty: aKey ifAbsent: aBlock
	"Remove the property with aKey. Answer the value or, if aKey isn't found, answer the result of evaluating aBlock."
	
	| answer |
	properties == nil  ifTrue: [ ^ aBlock value ].
	answer := properties removeKey: aKey ifAbsent: aBlock.
	properties isEmpty ifTrue: [ properties := nil ].
	^ answer
%
category: 'operators-convenience'
method: PPParser
separatedBy: aParser
	"Answer a new parser that parses the receiver one or more times, separated by aParser."
	
	^ (PPSequenceParser with: self with: (PPSequenceParser with: aParser with: self) star) ==> [ :nodes |
		| result |
		result := Array new: 2 * nodes second size + 1.
		result at: 1 put: nodes first.
		nodes second 
			keysAndValuesDo: [ :index :pair | result replaceFrom: 2 * index to: 2 * index + 1 with: pair startingAt: 1 ].
		result ]
%
category: 'operators-repeating'
method: PPParser
star
	"Answer a new parser that parses the receiver zero or more times. This is a greedy and blind implementation that tries to consume as much input as possible and it does not consider what comes afterwards."

	^ PPPossessiveRepeatingParser on: self
%
category: 'operators-repeating'
method: PPParser
starGreedy: aParser
	"Answer a new parser that parses the receiver zero or more times until it reaches aParser. This is a greedy non-blind implementation of the star operator. aParser is not consumed."
	
	^ PPGreedyRepeatingParser on: self limit: aParser
%
category: 'operators-repeating'
method: PPParser
starLazy: aParser
	"Answer a new parser that parses the receiver zero or more times until it reaches aParser. This is a lazy non-blind implementation of the star operator. aParser is not consumed."
	
	^ PPLazyRepeatingParser on: self limit: aParser
%
category: 'operators-repeating'
method: PPParser
times: anInteger
	"Answer a new parser that parses the receiver exactly anInteger times."
	
	^ self min: anInteger max: anInteger
%
category: 'operators-mapping'
method: PPParser
token
	"Answer a new parser that transforms the input to a token."
	
	^ PPTokenParser on: self
%
category: 'operators-mapping'
method: PPParser
token: aTokenClass
	"Answer a new parser that transforms the input to a token of class aTokenClass."
	
	^ self token tokenClass: aTokenClass
%
category: 'operators-mapping'
method: PPParser
trim
	"Answer a new parser that consumes spaces before and after the receiving parser."
	
	^ self trimSpaces
%
category: 'operators-mapping'
method: PPParser
trim: aParser
	"Answer a new parser that consumes and ignores aParser repeatedly before and after the receiving parser."
	
	^ PPTrimmingParser on: self trimmer: aParser
%
category: 'operators-mapping'
method: PPParser
trimBlanks
	"Answer a new parser that consumes blanks before and after the receiving parser."
	
	^ self trim: #blank asParser
%
category: 'operators-mapping'
method: PPParser
trimLeft
	"Answer a new parser that consumes spaces before the receiving parser."
	
	^ self trimSpacesLeft
%
category: 'operators-mapping'
method: PPParser
trimRight
	"Answer a new parser that consumes spaces after the receiving parser."
	
	^ self trimSpacesRight
%
category: 'operators-mapping'
method: PPParser
trimRight: trimmer
	"Answer a new parser that consumes spaces after the receiving parser."
	
	^ (self, trimmer star) ==> #first
%
category: 'operators-mapping'
method: PPParser
trimSpaces
	"Answer a new parser that consumes spaces before and after the receiving parser."
	
	^ self trim: #space asParser
%
category: 'operators-mapping'
method: PPParser
trimSpacesLeft
	"Answer a new parser that consumes spaces before the receiving parser."
	
	^ (#space asParser star, self) ==> #second
%
category: 'operators-mapping'
method: PPParser
trimSpacesRight
	"Answer a new parser that consumes spaces after the receiving parser."
	
	^ (self, #space asParser star) ==> #first
%
category: 'operators-convenience'
method: PPParser
withoutSeparators
	"Filters out the separators from a parse result produced by one of the productions #delimitedBy: or #separatedBy:."
	
	^ self ==> [ :items |
		| result |
		result := Array new: items size + 1 // 2.
		1 to: result size do: [ :index | result at: index put: (items at: 2 * index - 1) ].
		result ]
%
category: 'operators'
method: PPParser
wrapped
	"Answer a new parser that is simply wrapped."
	
	^ PPDelegateParser on: self
%
category: 'operators'
method: PPParser
| aParser
	"Answer a new parser that either parses the receiver or aParser. Fail if both pass or fail (exclusive choice, unordered choice)."

	^ (self not , aParser) / (aParser not , self) ==> #second
%
expectValue %Boolean
doit
PPParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPPluggableParser
removeallmethods PPPluggableParser
removeallclassmethods PPPluggableParser
! ------------------- Class methods for PPPluggableParser
category: 'instance creation'
classmethod: PPPluggableParser
on: aBlock
	^ self new initializeOn: aBlock
%
! ------------------- Instance methods for PPPluggableParser
category: 'accessing'
method: PPPluggableParser
block
	"Answer the pluggable block."

	^ block
%
category: 'initialization'
method: PPPluggableParser
initializeOn: aBlock
	block := aBlock
%
category: 'parsing'
method: PPPluggableParser
parseOn: aPPContext
	| memento result |
	memento := aPPContext remember.
	result := block value: aPPContext.
	result isPetitFailure
		ifTrue: [ aPPContext restore: memento ].
	^ result
%
expectValue %Boolean
doit
PPPluggableParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPPossessiveRepeatingParser
removeallmethods PPPossessiveRepeatingParser
removeallclassmethods PPPossessiveRepeatingParser
! ------------------- Class methods for PPPossessiveRepeatingParser
! ------------------- Instance methods for PPPossessiveRepeatingParser
category: 'parsing'
method: PPPossessiveRepeatingParser
parseOn: aPPContext
	| memento element elements |
	memento := aPPContext remember.
	elements := OrderedCollection new.
	[ elements size < min ] whileTrue: [
		(element := parser parseOn: aPPContext) isPetitFailure ifTrue: [
			aPPContext restore: memento.
			^ element ].
		elements addLast: element ].
	[ elements size < max ] whileTrue: [
	 	(element := parser parseOn: aPPContext) isPetitFailure
			ifTrue: [ ^ elements asArray ].
		elements addLast: element ].
	^ elements asArray
%
expectValue %Boolean
doit
PPPossessiveRepeatingParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPPredicateObjectParser
removeallmethods PPPredicateObjectParser
removeallclassmethods PPPredicateObjectParser
! ------------------- Class methods for PPPredicateObjectParser
category: 'factory-objects'
classmethod: PPPredicateObjectParser
any
	^self
		cacheAt: #'any'
		ifAbsentPut: [ self
			on: [ :each | true ] message: 'input expected'
			negated: [ :each | false ] message: 'no input expected' ]
%
category: 'factory-objects'
classmethod: PPPredicateObjectParser
anyExceptAnyOf: aCollection
	^ self
		on: [ :each | (aCollection includes: each) not ] message: 'any except ' , aCollection printString , ' expected'
		negated: [ :each | aCollection includes: each ] message: aCollection printString ,  ' not expected'
%
category: 'factory-objects'
classmethod: PPPredicateObjectParser
anyOf: aCollection
	^ self
		on: [ :each | aCollection includes: each ] message: 'any of ' , aCollection printString , ' expected'
		negated: [ :each | (aCollection includes: each) not ] message: 'none of ' , aCollection printString ,  'expected'
%
category: 'factory-objects'
classmethod: PPPredicateObjectParser
between: min and: max
	^ self
		on: [ :each | each >= min and: [ each <= max ] ] message: min printString , '..' , max printString , ' expected'
		negated: [ :each | each < min or: [ each > max ] ] message: min printString , '..' , max printString , ' not expected'
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
blank
	^self
		cacheAt: #'blank'
		ifAbsentPut: [ self
			chars: (String with: Character space with: Character tab) message: 'blank expected' ]
%
category: 'cache'
classmethod: PPPredicateObjectParser
cacheAt: aSymbol ifAbsentPut: aBlock
  | cache |
  cache := SessionTemps current
    at: #'PPPredicateObjectParser_cache'
    ifAbsentPut: [ SymbolKeyValueDictionary new ].
  ^ (cache at: aSymbol ifAbsentPut: aBlock) copy
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
char: aCharacter
	^ self expect: aCharacter message: (String with: $" with: aCharacter with: $") , ' expected'
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
char: aCharacter message: aString
	^ self expect: aCharacter message: aString
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
chars: aCollection message: aString
	^ self on: (PPCharSetPredicate on: [ :char | aCollection includes: char ]) message: aString
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
cr
	^self
		cacheAt: #'cr'
		ifAbsentPut: [ self char: (Character codePoint: 13) message: 'carriage return expected' ]
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
digit
	^self
		cacheAt: #'digit'
		ifAbsentPut: [ self on: (PPCharSetPredicate on: [ :char | char isDigit ]) message: 'digit expected' ]
%
category: 'factory-objects'
classmethod: PPPredicateObjectParser
endOfLine
	
	^ PPEndOfLineParser new.
%
category: 'instance creation'
classmethod: PPPredicateObjectParser
eof
	
	^ PPEndOfFileParser new
%
category: 'factory-objects'
classmethod: PPPredicateObjectParser
expect: anObject
	^ self expect: anObject message: anObject printString , ' expected'
%
category: 'factory-objects'
classmethod: PPPredicateObjectParser
expect: anObject message: aString
	^ self 
		on: [ :each | each = anObject ] message: aString
		negated: [ :each | each ~= anObject ] message: 'no ' , aString
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
hex
	^self
		cacheAt: #'hex'
		ifAbsentPut: [ self
		on: (PPCharSetPredicate on: [ :char | 
			(char between: $0 and: $9) 
				or: [ (char between: $a and: $f) 
				or: [ (char between: $A and: $F) ] ] ])
		message: 'hex digit expected' ]
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
letter
	^self
		cacheAt: #'letter'
		ifAbsentPut: [ self on: (PPCharSetPredicate on: [ :char | char isLetter ]) message: 'letter expected' ]
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
lf
	^self
		cacheAt: #'lf'
		ifAbsentPut: [ self char: (Character codePoint: 10) ]
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
lowercase
	^self
		cacheAt: #'lowercase'
		ifAbsentPut: [ self on: (PPCharSetPredicate on: [ :char | char isLowercase ]) message: 'lowercase letter expected' ]
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
newline
	^self
		cacheAt: #'newline'
		ifAbsentPut: [ self chars: (String with: (Character codePoint: 13) with: (Character codePoint: 10)) message: 'newline expected' ]
%
category: 'instance creation'
classmethod: PPPredicateObjectParser
on: aBlock message: aString
	^ self on: aBlock message: aString negated: [ :each | (aBlock value: each) not ] message: 'no ' , aString
%
category: 'instance creation'
classmethod: PPPredicateObjectParser
on: aBlock message: aString negated: aNegatedBlock message: aNegatedString
	^ self new initializeOn: aBlock message: aString negated: aNegatedBlock message: aNegatedString
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
punctuation
	^self
		cacheAt: #'punctuation'
		ifAbsentPut: [ self chars: '.,"''?!;:#$%&()*+-/<>=@[]\^_{}|~' message: 'punctuation expected' ]
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
space
	^self
		cacheAt: #'space'
		ifAbsentPut: [ self on: (PPCharSetPredicate on: [ :char | char isSeparator ]) message: 'separator expected' ]
%
category: 'instance creation'
classmethod: PPPredicateObjectParser
startOfLine
	
	^ PPStartOfLineParser new.
%
category: 'instance creation'
classmethod: PPPredicateObjectParser
startOfLogicalLine
	
	^ PPStartOfLogicalLineParser new.
%
category: 'instance creation'
classmethod: PPPredicateObjectParser
startOfWord
	
	^ PPStartOfWordParser new.
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
tab
	^self
		cacheAt: #'tab'
		ifAbsentPut: [ self char: Character tab message: 'tab expected' ]
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
uppercase
	^self
		cacheAt: #'uppercase'
		ifAbsentPut: [ self on: (PPCharSetPredicate on: [ :char | char isUppercase ]) message: 'uppercase letter expected' ]
%
category: 'factory-chars'
classmethod: PPPredicateObjectParser
word
	^self
		cacheAt: #'word'
		ifAbsentPut: [ self on: (PPCharSetPredicate on: [ :char | char isAlphaNumeric ]) message: 'letter or digit expected' ]
%
! ------------------- Instance methods for PPPredicateObjectParser
category: 'initialization'
method: PPPredicateObjectParser
initializeOn: aBlock message: aString negated: aNegatedBlock message: aNegatedString
	predicate := aBlock.
	predicateMessage := aString.
	negated := aNegatedBlock.
	negatedMessage := aNegatedString
%
category: 'operators'
method: PPPredicateObjectParser
negate
	"Answer a parser that is the negation of the receiving predicate parser."
	
	^ self class 
		on: negated message: negatedMessage 
		negated: predicate message: predicateMessage
%
category: 'parsing'
method: PPPredicateObjectParser
parseOn: aPPContext
	^ (aPPContext atEnd not and: [ predicate value: aPPContext uncheckedPeek ])
		ifFalse: [ PPFailure message: predicateMessage context: aPPContext ]
		ifTrue: [ aPPContext next ]
%
expectValue %Boolean
doit
PPPredicateObjectParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPPredicateParser
removeallmethods PPPredicateParser
removeallclassmethods PPPredicateParser
! ------------------- Class methods for PPPredicateParser
! ------------------- Instance methods for PPPredicateParser
category: 'accessing'
method: PPPredicateParser
block
	"Answer the predicate block of the receiver."
	
	^ predicate
%
category: 'accessing'
method: PPPredicateParser
message
	"Answer the failure message."
	
	^ predicateMessage
%
category: 'printing'
method: PPPredicateParser
printNameOn: aStream
	super printNameOn: aStream.
	aStream nextPutAll: ', '; print: predicateMessage
%
expectValue %Boolean
doit
PPPredicateParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPPredicateSequenceParser
removeallmethods PPPredicateSequenceParser
removeallclassmethods PPPredicateSequenceParser
! ------------------- Class methods for PPPredicateSequenceParser
category: 'instance creation'
classmethod: PPPredicateSequenceParser
on: aBlock message: aString negated: aNegatedBlock message: aNegatedString size: anInteger 
	^ self new initializeOn: aBlock message: aString negated: aNegatedBlock message: aNegatedString size: anInteger
%
category: 'instance creation'
classmethod: PPPredicateSequenceParser
on: aBlock message: aString size: anInteger
	^ self on: aBlock message: aString negated: [ :each | (aBlock value: each) not ] message: 'no ' , aString size: anInteger 
%
! ------------------- Instance methods for PPPredicateSequenceParser
category: 'initialization'
method: PPPredicateSequenceParser
initializeOn: aBlock message: aString negated: aNegatedBlock message: aNegatedString size: anInteger
	predicate := aBlock.
	predicateMessage := aString.
	negated := aNegatedBlock.
	negatedMessage := aNegatedString.
	size := anInteger 
%
category: 'operators'
method: PPPredicateSequenceParser
negate
	"Answer a parser that is the negation of the receiving predicate parser."
	
	^ self class 
		on: negated message: negatedMessage
		negated: predicate message: predicateMessage
		size: size
%
category: 'parsing'
method: PPPredicateSequenceParser
parseOn: aPPContext
	| memento result |
	memento := aPPContext remember.
	result := aPPContext stream next: size.
	(result size = size and: [ predicate value: result ])
		ifTrue: [ ^ result ].
	aPPContext restore: memento.
	^ PPFailure message: predicateMessage context: aPPContext
%
category: 'accessing'
method: PPPredicateSequenceParser
size
	"Answer the sequence size of the receiver."

	^ size
%
expectValue %Boolean
doit
PPPredicateSequenceParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPRepeatingParser
removeallmethods PPRepeatingParser
removeallclassmethods PPRepeatingParser
! ------------------- Class methods for PPRepeatingParser
! ------------------- Instance methods for PPRepeatingParser
category: 'initialization'
method: PPRepeatingParser
initialize
	super initialize.
	self setMin: 0; setMax: SmallInteger "maxVal" maximumValue
%
category: 'accessing'
method: PPRepeatingParser
max
	"Answer the maximum number of repetitions."

	^ max
%
category: 'accessing'
method: PPRepeatingParser
min
	"Answer the minimum number of repetitions."
	
	^ min
%
category: 'printing'
method: PPRepeatingParser
printOn: aStream
	super printOn: aStream.
	aStream nextPutAll: ' ['; print: min; nextPutAll: ', '; nextPutAll: (max = SmallInteger "maxVal" maximumValue
		ifTrue: [ '*' ] ifFalse: [ max printString ]); nextPut: $]
%
category: 'initialization'
method: PPRepeatingParser
setMax: anInteger
	max := anInteger
%
category: 'initialization'
method: PPRepeatingParser
setMin: anInteger
	min := anInteger
%
expectValue %Boolean
doit
PPRepeatingParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPSequenceParser
removeallmethods PPSequenceParser
removeallclassmethods PPSequenceParser
! ------------------- Class methods for PPSequenceParser
! ------------------- Instance methods for PPSequenceParser
category: 'operators'
method: PPSequenceParser
, aRule
	^ self copyWith: aRule
%
category: 'operators-mapping'
method: PPSequenceParser
map: aBlock
	^ aBlock numArgs = self children size
		ifTrue: [ self ==> [ :nodes | aBlock valueWithArguments: nodes ] ]
		ifFalse: [ self error: aBlock numArgs asString , ' arguments expected.' ]
%
category: 'parsing'
method: PPSequenceParser
parseOn: aPPContext
	"This is optimized code that avoids unnecessary block activations, do not change."
	
	| memento elements element |
	memento := aPPContext remember.
	elements := Array new: parsers size.
	1 to: parsers size do: [ :index |
		element := (parsers at: index) 
			parseOn: aPPContext.
		element isPetitFailure ifTrue: [
			aPPContext restore: memento.
			^ element ].
		elements at: index put: element ].
	^ elements
%
category: 'operators-mapping'
method: PPSequenceParser
permutation: anArrayOfIntegers
	"Answer a permutation of the receivers sequence."
	
	anArrayOfIntegers do: [ :index |
		(index _isInteger and: [ index between: 1 and: parsers size ])
			ifFalse: [ self error: 'Invalid permutation index: ' , index printString ] ].
	^ self ==> [ :nodes | anArrayOfIntegers collect: [ :index | nodes at: index ] ]
%
expectValue %Boolean
doit
PPSequenceParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPStartOfLineParser
removeallmethods PPStartOfLineParser
removeallclassmethods PPStartOfLineParser
! ------------------- Class methods for PPStartOfLineParser
! ------------------- Instance methods for PPStartOfLineParser
category: 'parasing'
method: PPStartOfLineParser
parseOn: aPPContext
	(aPPContext isStartOfLine) ifTrue: [ 
		^ #startOfLine
	].
	^ PPFailure message: 'Start of line expected' context: aPPContext at: aPPContext position
%
expectValue %Boolean
doit
PPStartOfLineParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPStartOfLogicalLineParser
removeallmethods PPStartOfLogicalLineParser
removeallclassmethods PPStartOfLogicalLineParser
! ------------------- Class methods for PPStartOfLogicalLineParser
! ------------------- Instance methods for PPStartOfLogicalLineParser
category: 'parasing'
method: PPStartOfLogicalLineParser
isBlank: character
	^ (character == Character space or: [character == Character tab])
%
category: 'parasing'
method: PPStartOfLogicalLineParser
parseOn: aPPContext
	aPPContext peek isAlphaNumeric ifFalse: [ 
		^ PPFailure message: 'Start of logical line expected' context: aPPContext 
	].

	aPPContext isStartOfLine ifTrue: [ ^ #startOfLogicalLine ].
	
	
	[ aPPContext position ~= 0 ] whileTrue: [  
		aPPContext back.
		(self isBlank: aPPContext peek) ifFalse: [ 
			^ PPFailure message: 'Start of logical line expected' context: aPPContext
		].
		aPPContext isStartOfLine ifTrue: [ ^ #startOfLogicalLine ].
	]
%
expectValue %Boolean
doit
PPStartOfLogicalLineParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPStartOfWordParser
removeallmethods PPStartOfWordParser
removeallclassmethods PPStartOfWordParser
! ------------------- Class methods for PPStartOfWordParser
! ------------------- Instance methods for PPStartOfWordParser
category: 'as yet unclassified'
method: PPStartOfWordParser
acceptsEpsilon
	^ false
%
category: 'as yet unclassified'
method: PPStartOfWordParser
parseOn: aPPContext
	aPPContext atEnd ifTrue: [  
		^ PPFailure message: 'Start of word expected' context: aPPContext at: aPPContext position 
	].

	(aPPContext position == 0) ifTrue: [ 
		(aPPContext peek isAlphaNumeric) ifTrue: [ 
			^ #startOfWord
		] ifFalse: [ 
			^ PPFailure message: 'Start of word expected' context: aPPContext at: aPPContext position 
	 	]
	].

	aPPContext back.
	aPPContext peek isAlphaNumeric ifTrue: [
		^ PPFailure message: 'Start of word expected' context: aPPContext at: aPPContext position 
	].
	aPPContext next.
	
	^ aPPContext peek isAlphaNumeric ifTrue: [ #startOfWord ] ifFalse: [ 
		PPFailure message: 'Start of word expected' context: aPPContext at: aPPContext position 
	]
%
expectValue %Boolean
doit
PPStartOfWordParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPToken
removeallmethods PPToken
removeallclassmethods PPToken
! ------------------- Class methods for PPToken
category: 'initialization'
classmethod: PPToken
initialize
	"Platform independent newline sequence. LF: Unix, CR+LF: Windows, and CR: Apple."
"
	| cr lf |
	cr := Character codePoint: 13.
	lf := Character codePoint: 10.
	NewLineParser := lf asParser / (cr asParser , lf asParser optional)
"
%
category: 'instance creation'
classmethod: PPToken
new
	self error: 'Token can only be created using a dedicated constructor.'
%
category: 'instance creation'
classmethod: PPToken
on: aSequenceableCollection
	^ self on: aSequenceableCollection start: 1 stop: aSequenceableCollection size value: nil
%
category: 'instance creation'
classmethod: PPToken
on: aSequenceableCollection start: aStartInteger stop: aStopInteger value: anObject
	^ self basicNew 
		initializeOn: aSequenceableCollection
		start: aStartInteger stop: aStopInteger
		value: anObject
%
! ------------------- Instance methods for PPToken
category: 'comparing'
method: PPToken
= anObject
	^ self class = anObject class and: [ self inputValue = anObject inputValue ]
%
category: 'accessing'
method: PPToken
collection
	"Answer the underlying collection of this token."

	^ collection
%
category: 'querying'
method: PPToken
column
	"Answer the column number of this token in the underlying collection."
	
	| position |
	position := 0.
	(newLineParser , [ :stream |
		start <= stream position
			ifTrue: [ ^ start - position ].
		position := stream position ] asParser
		/ #any asParser) star
			parse: collection.
	 ^ start - position
%
category: 'copying'
method: PPToken
copyFrom: aStartInteger to: aStopInteger
	^ self class on: collection start: start + aStartInteger - 1 stop: stop + aStopInteger - 3 value: value
%
category: 'comparing'
method: PPToken
hash
	^ self inputValue hash
%
category: 'initialization'
method: PPToken
initializeOn: aSequenceableCollection start: aStartInteger stop: aStopInteger value: anObject
	collection := aSequenceableCollection.
	start := aStartInteger.
	stop := aStopInteger.
	value := anObject.
	newLineParser := Character lf asParser / (Character cr asParser , Character lf asParser optional)
%
category: 'accessing-values'
method: PPToken
inputValue
	"Answer the consumed input of this token."

	^ collection copyFrom: start to: stop
%
category: 'querying'
method: PPToken
line
	"Answer the line number of this token in the underlying collection."
	
	| line |
	line := 1.
	(newLineParser , [ :stream |
		start <= stream position
			ifTrue: [ ^ line ].
		line := line + 1 ] asParser
		/ #any asParser) star
			parse: collection.
	^ line
%
category: 'accessing-values'
method: PPToken
parsedValue
	"Answer the parsed value of this token."

	^ value
%
category: 'printing'
method: PPToken
printOn: aStream
	super printOn: aStream.
	aStream nextPut: $[; print: self start; nextPut: $,; print: self stop; nextPut: $].
	aStream nextPut: $(; print: self inputValue; nextPut: $)
%
category: 'accessing'
method: PPToken
size
	"Answer the size of this token in the underlying collection."

	^ stop - start + 1
%
category: 'accessing'
method: PPToken
start
	"Answer the start position of this token in the underlying collection."

	^ start
%
category: 'accessing'
method: PPToken
stop
	"Answer the stop position of this token in the underlying collection."
	
	^ stop
%
category: 'accessing'
method: PPToken
value
	self notify: 'Token>>#value is no longer supported. Instead use Token>>#inputValue'.
	^ self inputValue
%
expectValue %Boolean
doit
PPToken category: 'PetitParser-Core'.
true
%

! Remove existing behavior from PPTokenParser
removeallmethods PPTokenParser
removeallclassmethods PPTokenParser
! ------------------- Class methods for PPTokenParser
! ------------------- Instance methods for PPTokenParser
category: 'private'
method: PPTokenParser
defaultTokenClass
	^ PPToken
%
category: 'initialization'
method: PPTokenParser
initialize
	tokenClass := self defaultTokenClass
	
%
category: 'private'
method: PPTokenParser
on: aCollection start: aStartInteger stop: aStopInteger value: anObject
	^ self tokenClass on: aCollection start: aStartInteger stop: aStopInteger value: anObject
%
category: 'accessing'
method: PPTokenParser
tokenClass
	^ tokenClass
%
category: 'accessing'
method: PPTokenParser
tokenClass: aTokenClass
	tokenClass := aTokenClass
%
expectValue %Boolean
doit
PPTokenParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPTrimmingParser
removeallmethods PPTrimmingParser
removeallclassmethods PPTrimmingParser
! ------------------- Class methods for PPTrimmingParser
category: 'instance creation'
classmethod: PPTrimmingParser
on: aParser trimmer: aTrimParser
	^ self new
		setParser: aParser;
		setTrimmer: aTrimParser;
		yourself
%
! ------------------- Instance methods for PPTrimmingParser
category: 'parsing'
method: PPTrimmingParser
parseOn: aPPContext
	| memento element trimContext |
	memento := aPPContext remember.
	trimContext := aPPContext copy.
	[ (trimmer parseOn: trimContext) isPetitFailure ]
		whileFalse.
	element := parser parseOn: aPPContext.
	element isPetitFailure ifTrue: [
		aPPContext restore: memento.
		^ element ].
	[ (trimmer parseOn: trimContext) isPetitFailure ]
		whileFalse.
	^ element
%
category: 'initialization'
method: PPTrimmingParser
setTrimmer: aParser
	trimmer := aParser
%
expectValue %Boolean
doit
PPTrimmingParser category: 'PetitParser-Parsers'.
true
%

! Remove existing behavior from PPUnresolvedParser
removeallmethods PPUnresolvedParser
removeallclassmethods PPUnresolvedParser
! ------------------- Class methods for PPUnresolvedParser
! ------------------- Instance methods for PPUnresolvedParser
category: 'testing'
method: PPUnresolvedParser
isUnresolved
	^ true
%
category: 'parsing'
method: PPUnresolvedParser
parseOn: aStream
	self error: self printString , ' need to be resolved before execution.'
%
expectValue %Boolean
doit
PPUnresolvedParser category: 'PetitParser-Tools'.
true
%

! Remove existing behavior from PPWrappingParser
removeallmethods PPWrappingParser
removeallclassmethods PPWrappingParser
! ------------------- Class methods for PPWrappingParser
! ------------------- Instance methods for PPWrappingParser
category: 'parsing'
method: PPWrappingParser
parseOn: aPPContext
	^ block value: aPPContext value: [ parser parseOn: aPPContext ]
%
expectValue %Boolean
doit
PPWrappingParser category: 'PetitParser-Parsers'.
true
%
category: 'operators-mapping'
method: PPParser
handleGsQueryErrorMessage: errorMessage
  ^ GsQueryErrorHandlingParser on: self errorMessage: errorMessage
%
category: 'copying'
method: PPContext
postCopy
  super postCopy.
  globals := globals copy
%
category: 'failures'
method: PPContext
noteDominantFailure: aDominantPPFailure
  "override an existing furthestFailure (at the same or greater position)"

  furthestFailure == nil 
    ifTrue: [ self error: 'expected an existing furthestFailure' ].
  aDominantPPFailure position >= furthestFailure position
    ifTrue: [ furthestFailure := aDominantPPFailure ].
  "true
    ifTrue: [ 
      (self globalAt: #'dominantFailureStack' ifAbsentPut: [ OrderedCollection new ])
        add: aDominantPPFailure ]"
%
category: 'operators-mapping'
method: PPParser
handleGsQueryErrorMessage: errorMessage nestMessages: aBool
  ^ GsQueryErrorHandlingParser
    on: self
    errorMessage: errorMessage
    nestMessages: aBool
%
category: 'cache'
classmethod: PPPredicateObjectParser
useCache: ignored
  "Cache is always enabled in GemStone"
%
