!=========================================================================
! Copyright (C) GemTalk Systems 2013-2020.  All Rights Reserved.
!
! $Id: setpathterm.gs 31614 2013-10-09 17:47:16Z dhenrich $
!
! Superclass Hierarchy:
!   SelectorPathTerm, PathTerm, Array, SequenceableCollection, 
!   Collection, Object.
!
!=========================================================================

! class created in idxclasses.topaz

removeallmethods SelectorPathTerm
removeallclassmethods SelectorPathTerm

category: 'For Documentation Installation only'
classmethod: SelectorPathTerm
installDocumentation

self comment: '
SelectorPathTerm represents a single term within an index or query path.  It represents a term that is a selector rather
than an instance variable, as specified by # before the selector name within the index or query path.

SelectorPathTerm implements only GemStone internals, and is not intended for direct use by customers.
'.
%

category: 'Updating Indexes'
set compile_env: 0
method: SelectorPathTerm
addDirectMappingFor: anObject logging: aBoolean
  " Update the B-tree. No dependencly list logic for receiver"

  updateBtree ~~ nil
    ifTrue: [ updateBtree btreeAt: anObject put: anObject ].
  updateDict ~~ nil
    ifTrue: [ 
      " insert an entry into the index dictionary "
      updateDict
        _at: anObject
        put: anObject
        term: self
        logging: aBoolean ]
%
category: 'Updating Indexes'
set compile_env: 0
method: SelectorPathTerm
addMappingsForObject: anObject logging: aBoolean
  "Add index dictionary. No dependency list logic for receiver."

  | nextObj |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  nextObj := self _nextObjectFor: anObject.	" get the next object along the path "
  updateBtree ~~ nil
    ifTrue: [ 
      " insert an entry into the B-tree "
      updateBtree btreeAt: nextObj put: anObject ].
  updateDict ~~ nil
    ifTrue: [ 
      " insert an entry into the index dictionary "
      updateDict
        _at: nextObj
        put: anObject
        term: self
        logging: aBoolean ].
  nil == nextObj
    ifTrue: [ ^ self ].
  1 to: children size do: [ :i | 
    " make recursive call to add mappings "
    (children at: i) addMappingsForObject: nextObj logging: aBoolean ]
%
category: 'Updating Indexes'
set compile_env: 0
method: SelectorPathTerm
addMappingsUsing: mapInfo logging: aBoolean
  "Adds index dictionary (no dependency list entries) for the object described by
 the map info object.  If aBoolean is true, operations will be recorded in the
 system redo log."

  | pathTerm object nextObj nextMap |
  pathTerm := mapInfo pathTerm.
  pathTerm size == 0
    ifTrue: [ ^ self ].
  object := mapInfo object.
  nil == object
    ifTrue: [ ^ self ].
  nextMap := mapInfo at: 1.	" nextObj is the same for all nextMaps, so choose the first one "
  nextObj := nextMap object.
  updateBtree ~~ nil
    ifTrue: [ 
      " insert an entry into the B-tree "
      updateBtree btreeAt: nextObj put: object ].
  updateDict ~~ nil
    ifTrue: [ 
      " insert an entry into the index dictionary "
      updateDict
        _at: nextObj
        put: object
        term: pathTerm
        logging: aBoolean ].
  1 to: mapInfo size do: [ :i | 
    nextMap := mapInfo at: i.	" make recursive call to add mappings "
    nextMap pathTerm ~~ nil
      ifTrue: [ nextMap pathTerm addMappingsUsing: nextMap logging: aBoolean ] ]
%
category: 'Audit'
set compile_env: 0
method: SelectorPathTerm
allDependencyListsFor: anObject into: all
  "No dependency list info"

  ^ self
%
category: 'Updating Indexes'
set compile_env: 0
method: SelectorPathTerm
cleanupDependencyListFor: anObject
  ""

  
%
category: 'Updating Indexes'
set compile_env: 0
method: SelectorPathTerm
getMappingInfoFor: anObject ifObject: origObject atOffset: origOffset replaceWith: otherObject
  "Gathers all the objects, their dependency lists, and the instance variable
 offset for each path term in the tree of path terms.  Puts them in a
 MappingInfo object.  Builds a tree of MappingInfo objects for each child path
 term.

 It is possible that this method is invoked due to an instance variable
 modification.  In that case, we want to traverse the path terms as if the
 modification had occurred.  Thus, if anObject is the one that was modified
 (that is, if it is identical to origObject) and the instance variable offset of
 this path term is the same instance variable offset that was modified (that is,
 it is equal to origOffset), then rather than getting the next object along the
 path, use the otherObject."

  | nextObj mapInfo aMapInfo |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self _mappingInfoClass new pathTerm: self ].
  mapInfo := self _mappingInfoClass new
    object: anObject;
    pathTerm: self.
  nextObj := self _nextObjectFor: anObject.	" get the next object along the path "
  updateBtree
    ifNotNil: [ 
      (self _checkBtreeComparisonWith: nextObj)
        ifFalse: [ 
          | exa |
          (exa := ImproperOperation new)
            _number:
                (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
            args: {nextObj class. self name. updateBtree lastElementClassDescription}.
          ^ {false.	" indicating no indexing objects were modified "
          exa} ] ].
  (nil == nextObj or: [ self children isEmpty ])
    ifTrue: [ 
      " adds nil to the end and returns "
      mapInfo addLast: (self _mappingInfoClass new object: nextObj).
      ^ mapInfo ].	" make recursive call to get mapping info "
  1 to: children size do: [ :i | 
    aMapInfo := (children at: i)
      getMappingInfoFor: nextObj
      ifObject: origObject
      atOffset: origOffset
      replaceWith: otherObject.
    aMapInfo class == self _mappingInfoClass
      ifFalse: [ ^ aMapInfo	"an Array of error info" ].
    mapInfo addLast: aMapInfo ].
  ^ mapInfo
%
category: 'Accessing'
set compile_env: 0
method: SelectorPathTerm
needsDepList
  "receiver does not do automatic maintenance updates"

  ^ false
%
category: 'Updating'
set compile_env: 0
method: SelectorPathTerm
needsDepList: ignored
  "receiver does not do automatic maintenance updates"

  
%
category: 'Updating Indexes'
set compile_env: 0
method: SelectorPathTerm
removeDirectMappingFor: anObject logging: doLogging
  "There is a range index directly on the elements of the NSC.  Update the
 B-tree."

  updateBtree ~~ nil
    ifTrue: [ 
      " remove it from the B-tree "
      updateBtree btreeRemoveKey: anObject value: anObject ].
  (self indicatesMultiValue _and: [ self indicatesIndexOnNscElements ])
    ifTrue: [ 
      " if it is a set-valued path term and the last term in an index... "
      ^ self ].
  updateDict ~~ nil
    ifTrue: [ 
      " remove the index dictionary entry "
      updateDict
        removeKey: anObject
        value: anObject
        term: self
        logging: doLogging ]
%
category: 'Updating Indexes'
set compile_env: 0
method: SelectorPathTerm
removeMappingsFor: anObject lastOne: aBoolean logging: doLogging
  "Remove entries in the index dictionary no dependency mappings"

  | nextObj more isLast |
  self size == 0
    ifTrue: [ ^ self ].
  nextObj := self _nextObjectFor: anObject.
  updateBtree ~~ nil
    ifTrue: [ 
      " remove an entry from the B-tree "
      updateBtree btreeRemoveKey: nextObj value: anObject ].
  more := false.
  updateDict ~~ nil
    ifTrue: [ 
      " remove the index dictionary entry "
      more := updateDict
        removeKey: nextObj
        value: anObject
        term: self
        logging: doLogging ].
  (nil ~~ nextObj and: [ children isEmpty not ])
    ifTrue: [ 
      " make recursive call to remove mappings "
      isLast := aBoolean and: [ more not ].
      1 to: children size do: [ :i | (children at: i) removeMappingsFor: nextObj lastOne: isLast logging: doLogging ] ]
%
category: 'Updating Indexes'
set compile_env: 0
method: SelectorPathTerm
removeMappingsSkipBtreeFor: anObject lastOne: aBoolean logging: doLogging
  "Remove entries in the index dictionary. Nodependency lists for anObject.
 This version for index removal, it does not update the btree since the btree
 is going away (#40858)."

  | nextObj more isLast |
  self size == 0
    ifTrue: [ ^ self ].
  nextObj := self _nextObjectFor: anObject.
  updateBtree ~~ nil
    ifTrue: [  ].
  more := false.
  updateDict ~~ nil
    ifTrue: [ 
      " remove the index dictionary entry "
      more := updateDict
        removeKey: nextObj
        value: anObject
        term: self
        logging: doLogging ].
  (nil ~~ nextObj and: [ children isEmpty not ])
    ifTrue: [ 
      " make recursive call to remove mappings "
      isLast := aBoolean and: [ more not ].
      1 to: children size do: [ :i | (children at: i) removeMappingsFor: nextObj lastOne: isLast logging: doLogging ] ]
%
category: 'Accessing'
set compile_env: 0
method: SelectorPathTerm
termSelector
  | term |
  term := self name.
  ^ term copyFrom: 2 to: term size
%
category: 'Updating Indexes'
set compile_env: 0
method: SelectorPathTerm
undoMappingsFor: anObject lastOne: aBoolean logging: doLogging
  "Remove entries in the index dictionary. No dependency lists for anObject."

  | nextObj more isLast |
  self size == 0
    ifTrue: [ ^ self ].
  [ nextObj := self _nextObjectFor: anObject ]
    onException: Error
    do: [ :ex | 
      "Handle any error thrown during execution of _nextObjectFor:
     Cannot resolve the path for this guy --- can't have been added to index,
     in fact this object could have caused the failure in the first place"
      ^ self ].
  updateBtree ~~ nil
    ifTrue: [  ].
  more := false.
  updateDict ~~ nil
    ifTrue: [ 
      " remove the index dictionary entry "
      more := updateDict
        removeKey: nextObj
        value: anObject
        term: self
        logging: doLogging ].
  (nil ~~ nextObj and: [ children isEmpty not ])
    ifTrue: [ 
      " make recursive call to remove mappings "
      isLast := aBoolean and: [ more not ].
      1 to: children size do: [ :i | (children at: i) undoMappingsFor: nextObj lastOne: isLast logging: doLogging ] ]
%
category: 'Updating Indexes'
set compile_env: 0
method: SelectorPathTerm
update: anObject at: ivOffset to: newValue
  "should never be sent to the receiver as automatic index maintenance is not supported"

  self shouldNotImplement: #'update:at:to:'
%
category: 'Accessing'
set compile_env: 0
method: SelectorPathTerm
_ivOffsetFor: anObject
  self shouldNotImplement: #'_ivOffsetFor:'
%
category: 'Accessing'
set compile_env: 0
method: SelectorPathTerm
_nextObjectFor: anObject
  "Returns the object at the instance variable that corresponds to the receiver
 path term."

  ^ anObject perform: self termSelector
%
category: 'Accessing'
set compile_env: 0
method: SelectorPathTerm
_nextObjectFor: anObject atInstVar: ivOffset
  self shouldNotImplement: #'_nextObjectFor:atInstVar:'
%
category: 'Audit'
method: SelectorPathTerm
auditDepListFor: obj index: indexObj on: aString optionalSentinel: optionalSentinel
  "No dependency lists"

  ^ obj
%
category: 'Testing'
method: SelectorPathTerm
isSelectorTerm
  ^ true
%
category: 'Updating'
method: SelectorPathTerm
_addChildTerm: aPathTerm
  self
    error:
      'Selector path term may not have child terms: ' , aPathTerm printString
%
category: 'Testing'
method: SelectorPathTerm
termsRequired
  "Answer true if receiver requires that instance variables of indexed objects are present "

  ^ false
%
category: 'Accessing'
method: SelectorPathTerm
requirePathTerms
  requirePathTerms ifNil: [ ^ true ].
  ^ requirePathTerms
%
category: 'Accessing'
method: SelectorPathTerm
requirePathTerms: aBool
  requirePathTerms := aBool
%
category: 'Testing'
method: SelectorPathTerm
termsRequired
  "Answer true if receiver requires that instance variables of indexed objects are present "

  ^ self requirePathTerms
%
