!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!  File: smalldouble.gs
!
! Superclass Hierarchy:
!   SmallDouble, Float, BinaryFloat, Number, Magnitude, Object.
!
! As of Gs64 v3.0, changed SmallDouble from being a subclass of BinaryFloat
!=========================================================================

! SmallDouble class is created in bom.c

! Most methods are inherited from Float ; the shared primitives
!  will handle receiver that are either a Float or a SmallDouble .

removeallmethods SmallDouble
removeallclassmethods SmallDouble
set class SmallDouble

! fixed 33609
category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'This class represents 8-byte binary floating point numbers, as defined in IEEE
 standard 754, but with a reduced exponent.  
 SmallDouble has 8 bits of exponent, compared to 11 bits of exponent in
 an IEEE-754 8-byte float.

 You may not create subclasses of SmallDouble.

 Each SmallDouble contains a 61 bit value.  The floats are stored on Disk and
 in object memory in big-endian IEEE format.  GemStone Smalltalk primitives and
 GemBuilder for C (GCI) float conversion functions automatically convert 
 the format of a float to or from the machine''s native format, as required.

 In big-endian layout,
  An IEEE 64 bit binary float (a C double) has this bit layout
   seeeeeee|eeeeffff|ffffffff|ffffffff|ffffffff|ffffffff|ffffffff|ffffffff
   If (e > 0) the double has value (-1 * s) * 2^^(e-0x3ff) * 1.f  .
   If (e == 0) and f != 0 ,
     the double has value (-1 * s) * 2^^(e-0x3fe) * 0.f  (i.e. subnormal) .
   zero is represented as s==0, e==0, f==0  . See Sparc V9 Arch Manual pg 28.

  An instance of SmallDouble has this OopType layout:
   eeeeeeee|ffffffff|ffffffff|ffffffff|ffffffff|ffffffff|ffffffff|ffffs110
  which is the same number of bits of mantissa as a C double, but
  only 8 bits of exponent as compared to 11 bits of exponent in C double.
  The bottom 3 bits are the tag bits of the OopType .

  If the C double has value zero , the SmallDouble has all e and f bits zero,
  otherwise the SmallDouble has non-zero exponent, and
  the value is (-1 * s) * 2^^(e - 0x7f ) * 1.f

  There are no subnormal, infinity, nor NaN SmallDoubles .
  Thus SmallDoubles can represent  C doubles that have value zero
  or that have exponent bits in range 0x381 to 0x3ff,
  which correspends to about 5.0e-39 to 6.0e+38 , which
  is also the exponent range of  C 4-byte float  .
 '.
%

category: 'Private'
method:
_asFloat    

"Returns an instance of Float equal to the receiver. 
 For use in activation of a PassiveObject and for testing the image.
 It is a performance and space degradation to use this in applications."

^ self _mathPrim: 10
%

category: 'Converting'
method:
_generality

"Returns the integer that represents the ordering of the receiver in the
 generality hierarchy."

"Reimplemented from Number"

^ 85
%

category: 'Storing and Loading'
classmethod:
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst |

inst := self fromStringLocaleC:  passiveObj upToSeparator .
"no hasRead: here, since SmallDouble is a special"
^inst
%

category: 'Storing and Loading'
method:
containsIdentity

"Private."

^true
%
category: 'Testing'
method:
isSpecial

"Returns true if the receiver is a special object."

^ true
%

category: 'Repository Conversion'
method: 
needsFixingAfterConversion
 ^ false
%

! Remove SmallDouble from classHistory of Float if present
! deletes Fix 35521
!
expectvalue %String
run
Float classHistory removeIdentical: SmallDouble ifAbsent:[ ^ 'not found'].
^ 'removed'
%
expectvalue 1
run
 Float classHistory size
%
category: 'Copying
method:
copy
  ^ self
%
