! ========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
! ========================================================================

set compile_env: 0
! ------------------- Class definition for CHeader
expectvalue /Class
doit
Object subclass: 'CHeader'
  instVarNames: #( stream types structs
                    unions functions enums enumTags
                    storage declarations preprocessor source)
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals
  options: #()

%
expectvalue /Class
doit
CHeader comment: 
'This class is part of the FFI implementation and represents a C header file. 
It is typically used to generate a Smalltalk class that wraps a function or structure.
Usage examples can be found in the following instance methods:
	#wrapperForLibraryAt:
	#wrapperNamed:forLibraryAt:select:
	#wrapperForTypeNamed:
	#wrapperNamed:forStruct:
'
%
expectvalue /Class
doit
CHeader category: 'FFI'
%
! ------------------- Remove existing behavior from CHeader
expectvalue /Metaclass3       
doit
CHeader removeAllMethods.
CHeader class removeAllMethods.
%
set compile_env: 0
! ------------------- Class methods for CHeader
category: 'Class Membership'
classmethod: CHeader
cPreprocessorSpecies
"Answer the class to be used for CPreprocessor objects. Subclasses may
 overload this method as needed."
^ CPreprocessor
%
category: 'other'
classmethod: CHeader
fetchGciLinkedLibraryName

"Answer a String containing a path to the linked GCI shared library 
 in a GemStone product tree, relative to the value of the GEMSTONE
 environment variable." 

| result |
result := String withAll: '$GEMSTONE/lib/' .
result addAll: System fetchLinkedGciLibraryName .
^ result
%
category: 'other'
classmethod: CHeader
fetchGciRpcLibraryName

"Answer a String containing a path to the 64-bit RPC GCI shared library 
 in a GemStone product tree, relative to the value of the GEMSTONE
 environment variable." 

| result |
result := String withAll: '$GEMSTONE/lib/' .
result addAll: System fetchRpcGciLibraryName .
^ result
%
category: 'other'
classmethod: CHeader
path: aString
  ^ self path: aString searchPaths: { } 
%
category: 'other'
classmethod: CHeader
path: aString searchPath: searchPath
	"Parse the given file path aString, searching the absolute path searchPath before the system include directories.
	Example:
	CHeader path: 'openssl/ssl.h' searchPath: '/home/me/where/i/put/private/builds/include/'.
	This will find /home/me/where/i/put/private/builds/include/openssl/ssl.h, and includes within
	that file of the form 
	#include <openssl/someFile.h> 
	will be found in /home/me/where/i/put/private/builds/include/openssl."

	^self path: aString searchPaths: {searchPath}
%
category: 'other'
classmethod: CHeader
path: aString searchPaths: anArray
	"Parse the given file path aString, searching the absolute paths in anArray before the system include directories.
	Example:
	CHeader path: 'openssl/ssl.h' searchPaths: #('/home/me/where/i/put/private/builds/include/').
	This will find /home/me/where/i/put/private/builds/include/openssl/ssl.h, and includes within
	that file of the form 
	#include <openssl/someFile.h> 
	will be found in /home/me/where/i/put/private/builds/include/openssl."

	| preprocessor fullPath |
	preprocessor := self cPreprocessorSpecies new.
	anArray reverseDo: [:path | preprocessor insertSearchPath: path].
	fullPath := preprocessor searchForInclude: aString excluding: nil.
	fullPath
		ifNil: [^UserDefinedError signal: 'Include file ' , aString , ' not found!'].
	preprocessor includePath: fullPath.
	^self preprocessor: preprocessor
%
category: 'other'
classmethod: CHeader
preprocessor: aCPreprocessor

	^self new
		initialize: aCPreprocessor;
		yourself.
%
category: 'other'
classmethod: CHeader
string: aString
	^ self preprocessor: (self cPreprocessorSpecies parseString: aString).
%
category: 'other'
classmethod: CHeader
string: aString ignoreWarnings: aBoolean
	^ self preprocessor: (self cPreprocessorSpecies parseString: aString ignoreWarnings: aBoolean).
%
category: 'Reporting'
classmethod: CHeader
_gciStructReport
^ self _gciStructsReport: '$GEMSTONE/include/gci.hf' for:
  {  "gcilegacy.ht" 'GciObjInfoSType' . 'GciFetchObjInfoArgsSType' . 'GciJmpBufSType' .
     "gci.ht" 'GciErrSType' .  'GciObjRepHdrSType' .  'GciObjRepSType' . 'GciX509LoginArg' .
     "gci.hf" 'GciCTimeStringType' .
     "gcicmn.ht" 'GciTravBufType' .  'GciTravBufHolder' .  'GciClampedTravArgsSType' .
         'GciStoreTravDoArgsSType' .  'GciDateTimeSType' }
%
category: 'Reporting'
classmethod: CHeader
_gciStructsReport: headerPath for: typeNames
  | h rpt | 
  h := CHeader path: headerPath .
  rpt := String new .
  typeNames do:[ :tName | (h _typeAt: tName ) _addToReport: rpt .  rpt lf ].
  ^ rpt 
%
category: 'Reporting'
classmethod: CHeader
_gciTsStructReport
 ^ self _gciStructsReport: '$GEMSTONE/include/gcits.hf' for:
   { "gci.ht" 'GciErrSType' .  'GciObjRepHdrSType' .  'GciObjRepSType' . "Add with TS x509 support is done: 'GciX509LoginArg' . "
     "gcicmn.ht" 'GciTravBufType' .  'GciTravBufHolder' .  'GciClampedTravArgsSType' .
         'GciStoreTravDoArgsSType' .  'GciDateTimeSType' .
     "gcits.hf"  'GciTsObjInfo' }
%
! ------------------- Instance methods for CHeader
category: 'CLibrary'
method: CHeader
add: aCDeclaration to: aClass

	aClass addClassVarName: aCDeclaration asClassVarName.
	aClass class
		compileMethod: aCDeclaration createFunctionInitializerCode
		dictionaries: GsCurrentSession currentSession symbolList
		category: 'Initializing - private' environmentId: 0 .
	aClass
		compileMethod: aCDeclaration invokeFunctionCode
		dictionaries: SymbolList new
		category: 'Functions' environmentId: 0.
%
category: 'Accessing'
method: CHeader
atEnd

	^preprocessor atEnd.
%
category: 'Class Membership'
method: CHeader
cByteArraySpecies
"Answer the class to be used for CByteArray objects. Subclasses may
 overload this method as needed."
^ CByteArray
%
category: 'Class Membership'
method: CHeader
cDeclarationSpecies
"Answer the class to be used for C Declarations. Subclasses may
 overload this method as needed."
^ CDeclaration
%
category: 'CLibrary'
method: CHeader
createDefinesInClass: aClass
	"preprocessor createDefinesInClass: aClass"
%
category: 'CLibrary'
method: CHeader
createFunctionsInClass: newClass libraryPathExpressionString: aString select: aBlock

	| localFunctions |
	localFunctions := aBlock 
		ifNil: [functions]
		ifNotNil: [functions select: aBlock].
	localFunctions do: [:each | 
		self 
			add: each 
			to: newClass.
	].
	self 
		createInitializerInClass: newClass
		forFunctions: localFunctions 
		libraryPathExpressionString: aString.
%
category: 'CLibrary'
method: CHeader
createInitializerInClass: aClass forFunctions: someFunctions libraryPathExpressionString: aString

	aClass class
		compileMethod: (self initializeFunctionsCodeForFunctions: someFunctions libraryPathExpressionString: aString)
		dictionaries: GsCurrentSession currentSession symbolList
		category: 'Initializing - private' environmentId: 0 .
%
category: 'Accessing'
method: CHeader
declarations

   ^declarations
%
category: 'Accessing'
method: CHeader
definitions

	^preprocessor definitions.
%
category: 'Accessing'
method: CHeader
enums

   ^enums
%
category: 'Accessing'
method: CHeader
enumTags

   ^enumTags
%
category: 'Accessing'
method: CHeader
fieldsForStruct: aString

   ^(structs at: aString) fields.
%
category: 'Accessing'
method: CHeader
fullPath
  "If not at completion of parsing, the filename may be a nested include file."
  ^ preprocessor path   
%
category: 'Accessing'
method: CHeader
functions

   ^functions
%
category: 'other'
method: CHeader
initialize: aCPreprocessor
 
	preprocessor := aCPreprocessor.
	types := KeyValueDictionary new.
	structs := KeyValueDictionary new.
	unions := KeyValueDictionary new.
	functions := KeyValueDictionary new.
	enums := KeyValueDictionary new.
	enumTags := KeyValueDictionary new.
	storage := KeyValueDictionary new.
	declarations := { } .

	source := String new.
	[
		self skipEmpty.
		self atEnd not.
	] whileTrue: [
		self readOne.
		source := String new.
	].
	preprocessor clearTokens.
%
category: 'CLibrary'
method: CHeader
initializeFunctionsCodeForFunctions: someFunctions libraryPathExpressionString: aString

	| myStream |
	myStream := AppendStream on: String new.
	myStream 
		nextPutAll: 'initializeFunctions

	| library |
	library := CLibrary named: ';
		nextPutAll: aString;
		nextPutAll: '.'; lf; tab;
		nextPutAll: 'self'; lf; tab; tab;
		yourself.
	(someFunctions asSortedCollection: [:a :b | a name <= b name]) do: [:aFunction |
		myStream 
			nextPutAll: aFunction initializerFunctionName;
			nextPutAll: ' library;'; lf; tab; tab;
			yourself.
	].
	^myStream 
		nextPutAll: 'yourself.'; lf;
		contents.
%
category: 'CLibrary'
method: CHeader
newClassNamed: aString

	^self
		newClassNamed: aString
		instVarNames: #().
%
category: 'CLibrary'
method: CHeader
newClassNamed: aString instVarNames: anArray

	^Object
		subclass: aString
		instVarNames: anArray
		classVars: #()
		classInstVars: #()
		poolDictionaries: #()
		inDictionary: nil 
		options: #().
%
category: 'Accessing'
method: CHeader
next
	| token src |
	preprocessor atEnd ifTrue:[ ^ nil].
	token := preprocessor next.
	source ifNotNil:[ | tkSrc |
     src := source .
     tkSrc := token source .
     (source size > 0 and:[ source last isAlphaNumeric ]) ifTrue:[
       tkSrc size > 0 ifTrue:[ | tC |
         tC := tkSrc at: 1.
         tC isAlphaNumeric ifTrue:[ source add:$  ]
       ]
     ].
     source add: token source 
  ].
	^ token.
%
category: 'Accessing'
method: CHeader
peek

	^preprocessor peek.
%
category: 'other'
method: CHeader
readClass
"                                    .----------------.
                                    V                   |
>>--+--class--identifier--{----member--;--+--}---><
"

	| token classDeclaration oldTypes oldStructs oldUnions oldFunctions oldEnums oldStorage newStorage |
	classDeclaration := declarations last.
	(token := self next) isSemicolonToken ifTrue: [
		classDeclaration source add: source.
		source := nil.
		^self.
	].
	token isOpenCurlyBracketToken ifFalse: [token error: 'syntax error, expected ''{'' '].
	types 
		at: classDeclaration name 
		put: classDeclaration.
	oldTypes := types copy.
	oldStructs := structs copy.
	oldUnions := unions copy.
	oldFunctions := functions copy.
	oldEnums := enums copy.
	oldStorage := storage copy.
	[
		self skipEmpty.
		(token := self peek) isCloseCurlyBracketToken not.
	] whileTrue: [
		(token type == #'identifier' and: [token value = 'public' or: [token value = 'private']]) ifTrue: [
			(token := self next; next) isColonToken ifFalse: [token error: 'Unexpected token'].
		].
		self readOne.
	].
	self next.
	types := oldTypes.
	structs := oldStructs.
	unions := oldUnions.
	functions := oldFunctions.
	enums := oldEnums.
	newStorage := storage.
	storage := oldStorage.
	storage keys do: [:each | newStorage removeKey: each].
	newStorage := newStorage asArray collect: [:each | 
		(declarations indexOf: each) -> each.
	].
	newStorage := newStorage asSortedCollection asArray collect: [:each | each value].
	classDeclaration setClassProperties: newStorage.
	self atEnd ifFalse: [
		(token := self next) isSemicolonToken ifFalse: [token error: 'expected '';'' '].
	].
	classDeclaration source: source.
%
category: 'other'
method: CHeader
readExternLinkageDeclarationsUnderStorageRegimeFrom: aCDeclaration
	"Process declarations of the form:
		extern <linkage_specification> { ... }
	Typically: 
		extern 'C' { }
	"

	| priorDeclaration priorSource token |
	(token := self next) isOpenCurlyBracketToken ifFalse: [ token error: 'expected ''{'' ']. 
	source := String new.
	[
		self skipEmpty.
		self atEnd not.
	] whileTrue: [
		self peek isCloseCurlyBracketToken ifTrue: [
			self next.
			^self
		].
		self readOneUnderStorageHandler: [:declaration |
			priorDeclaration := declaration.	"to help with debugging"
			(declaration isEmptyExternalLinkage and: [self peek isOpenCurlyBracketToken]) ifTrue: [
				self readExternLinkageDeclarationsUnderStorageRegimeFrom: declaration.
			] ifFalse: [
				declaration canStorageBeMadeExternal ifTrue: [
					declaration 
						storage: aCDeclaration storage
						linkageSpec: aCDeclaration linkageSpec.
				].
			].
		].
		priorSource := source.	"to help with debugging"
		source := String new.
	].
	token error: 'could not find end of ''extern <linkage_specification> {'' '
%
category: 'other'
method: CHeader
readOne

	self readOneUnderStorageHandler: 
			[:declaration |
			(declaration isEmptyExternalLinkage and: [self peek isOpenCurlyBracketToken])
				ifTrue: 
					[^self readExternLinkageDeclarationsUnderStorageRegimeFrom: declaration]]
%
category: 'other'
method: CHeader
readOneUnderStorageHandler: aMonadicBlock
	"Read a declaration and process its storage specification using the
	 specified monadic Block. The Block takes the new declaration as
	 an argument."

	| type declaration token |
	type := self cDeclarationSpecies readTypeFrom: self.
	type source: source.
	declaration := type copy readDeclaration.

	aMonadicBlock value: declaration.

	declaration isEmptyExternalLinkage
		ifFalse: 
			[declarations add: declaration.
			declaration includesCode ifTrue: [^self].
			(declaration storage == #typedef and: [declaration type == #class])
				ifTrue: [^self readClass].
			
			[self saveDeclaration: declaration.
			self atEnd ifTrue: [^self].
			(token := self next) isSemicolonToken ifTrue: [^self].
			token isCommaToken ifFalse: [token error: 'Expected comma or semicolon'].
			true]
					whileTrue: 
						[declaration := type copy readDeclaration.
						declarations add: declaration]]
%
category: 'other'
method: CHeader
saveDeclaration: aDeclaration
 
	| dictionary |
	aDeclaration type == #'enum' ifTrue: [
		aDeclaration enumTag ifNotNil: [
			aDeclaration enumList ifNil: [
				(enumTags includesKey: aDeclaration enumTag) ifFalse: [
					self error: 'Missing definition of enum ' , aDeclaration enumTag.
				].
				aDeclaration name ifNil: [
					self error: 'enum type should declare a variable or function'.
				].
			] ifNotNil: [
				aDeclaration enumList isEmpty ifTrue: [
					self error: 'enum should define some values'.
				].
				(enumTags includesKey: aDeclaration enumTag) ifTrue: [
					self error: 'Duplicate definition of enum ' , aDeclaration enumTag.
				].
				enumTags
					at: aDeclaration enumTag
					put: aDeclaration.
			].
		].
	].
	aDeclaration name ifNil: [^self].
	dictionary := 
		aDeclaration storage == #'typedef' ifTrue: [	types] ifFalse: [
		aDeclaration isFunction ifTrue: [					functions] ifFalse: [
																	storage]].
	dictionary
		at: aDeclaration name
		put: aDeclaration.
%
category: 'other'
method: CHeader
skipEmpty

	[
		self atEnd not and: [self peek isEmptyToken].
	] whileTrue: [
		| token |
		token := self next.
		"source ifNotNil: [source add: token source]."
	].
%
category: 'Accessing'
method: CHeader
storage

   ^storage
%
category: 'Accessing'
method: CHeader
structs

   ^structs
%
category: 'Accessing'
method: CHeader
types

   ^types
%
category: 'Accessing'
method: CHeader
unions
   ^unions
%
category: 'CLibrary'
method: CHeader
wrapperForLibraryAt: aString
"
| header class |
header := CHeader path: '$GEMSTONE/include/gci.hf'.
class := header wrapperForLibraryAt: CHeader fetchGciRpcLibraryName .
UserGlobals at: class name put: class.
"
	| name |
	name := (((aString subStrings: $/) last subStrings: $.) first subStrings: $-) first.
	(3 < name size and: [(name copyFrom: 1 to: 3) = 'lib']) ifTrue: [name := name copyFrom: 4 to: name size].
	name first isLowercase ifTrue: [name at: 1 put: name first asUppercase].
	^self 
		wrapperNamed: name 
		libraryPathExpressionString: aString printString 
		select: nil.
%
category: 'CByteArray'
method: CHeader
wrapperForType: aCDeclaration
  
	^self
		wrapperNamed: aCDeclaration name
		forType: aCDeclaration.
%
category: 'CByteArray'
method: CHeader
wrapperForTypeNamed: aString
"   
| header class |
UserGlobals removeKey: #'GciErrSType' ifAbsent: [].
header := CHeader path: '$GEMSTONE/include/gci.hf'.
class := header wrapperForTypeNamed: 'GciErrSType'.
UserGlobals at: class name put: class.
"
	^self
		wrapperNamed: aString
		forType: (types at: aString).
%
category: 'CLibrary'
method: CHeader
wrapperNamed: nameString forLibraryAt: pathString select: aBlock
"
| header class |
UserGlobals removeKey: #'GciLibrary' ifAbsent: [].
header := CHeader path: '$GEMSTONE/include/gci.hf'.
class := header
	wrapperNamed: 'GciLibrary'
	forLibraryAt: self class fetchGciRpcLibraryName
	select: [:each | each name first == $G ].
UserGlobals at: class name put: class.
"
	^self 
		wrapperNamed: nameString 
		libraryPathExpressionString: pathString printString 
		select: aBlock
%
category: 'CByteArray'
method: CHeader
wrapperNamed: nameString forStruct: aCDeclaration
"
| header class |
UserGlobals removeKey: #'GciErrSType' ifAbsent: [].
header := CHeader path: '$GEMSTONE/include/gci.hf'.
class := header
	wrapperNamed: 'GciErrSType'
	forType: 'GciErrSType'.
UserGlobals at: class name put: class.
"
	| class symbolList string writeStream |
	symbolList := GsCurrentSession currentSession symbolList.
	class := (self cByteArraySpecies)
		subclass: nameString
		instVarNames: #()
		classVars: #()
		classInstVars: #()
		poolDictionaries: #()
		inDictionary: nil 
		options: #().
	aCDeclaration addSourceTo: (writeStream := AppendStream on: String new).
	class comment: writeStream contents.
	string := 'on: aCByteArray

	^self withAll: aCByteArray.
'.
	class class
		compileMethod: string  "may signal CompileError or CompileWarning"
		dictionaries: symbolList
		category: 'Instance Creation' 
		environmentId: 0.
	string := 'new

	^self gcMalloc: ' , aCDeclaration byteSizeRounded printString , '.
'.
	class class
		compileMethod: string  "may signal CompileError or CompileWarning"
		dictionaries: symbolList
		category: 'Instance Creation' 
		environmentId: 0.
	string := 'initialize
'.
	class
		compileMethod: string  "may signal CompileError or CompileWarning"
		dictionaries: symbolList
		category: 'Initialization' 
		environmentId: 0.
	string := 'initialize: aCByteArray

	1 to: (self size min: aCByteArray size) do: [:i | 
		self uint8At: i put: (aCByteArray uint8At: i).
	].
'.
	class
		compileMethod: string  "may signal CompileError or CompileWarning"
		dictionaries: symbolList
		category: 'Initialization' 
		environmentId: 0.
	string := '_stringFromBytes: aByteArray

	| index |
	index := aByteArray indexOf: 0.
	^aByteArray
		at: 1 
		sizeBytes: 1 
		stringSize: (0 == index ifTrue: [aByteArray size] ifFalse: [index - 1]).
'.
	class
		compileMethod: string  "may signal CompileError or CompileWarning"
		dictionaries: symbolList
		category: 'Conversion' 
		environmentId: 0.
	aCDeclaration sourceStringsForAccessors do: [:each | 
		class
			compileMethod: each  "may signal CompileError or CompileWarning"
			dictionaries: symbolList
			category: 'Accessing' environmentId: 0 .
	].
	aCDeclaration sourceStringsForUpdators do: [:each | 
		class
			compileMethod: each  "may signal CompileError or CompileWarning"
			dictionaries: symbolList
			category: 'Updating' 
			environmentId: 0.
	].
	^class.
%
category: 'CByteArray'
method: CHeader
wrapperNamed: nameString forType: aCDeclaration
	| struct |
	aCDeclaration type == #'class' ifTrue: [
		^self
			wrapperNamed: nameString
			forStruct: aCDeclaration.
	].
	(((struct := aCDeclaration type) isKindOf: CDeclaration) and: [struct type == #'struct']) ifFalse:[
     Error signal: 'Should be a struct'
  ].
	^self
		wrapperNamed: nameString
		forStruct: struct.
%
category: 'CLibrary'
method: CHeader
wrapperNamed: nameString libraryPathExpressionString: aString select: aBlock

	| newClass |
	newClass := self newClassNamed: nameString.
	self 
		createFunctionsInClass: newClass libraryPathExpressionString: aString select: aBlock;
		createDefinesInClass: newClass.
	newClass initializeFunctions.
	^newClass.
%
category: 'Reporting'
method: CHeader
_typeAt: aName

^ (types at: aName otherwise: nil) ifNil:[
     (structs at: aName otherwise: nil) ifNil:[
       (unions at: aName otherwise: nil) ifNil:[ Error signal: aName , '  not found' ]]].
%
