!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Definition of DateAndTime (from TimeZone2007.gs)
!
!=========================================================================

set class DateAndTime

removeallmethods 
removeallclassmethods 

! ------------------- Class methods for DateAndTime
category: 'For Documentation Installation only'
classmethod: DateAndTime
installDocumentation

self comment:
'This is the GemStone/S override to access System data.

This protocol describes the behavior that is common to date 
time objects. Date time objects represent individual points 
in Coordinated Universal Time (UTC) as represented in an
implementation defined local time.

The exact properties of local times are unspecified. Local 
times may differ in their offset from UTC. A given local time 
may have different offsets from UTC at different points in time.

All dates and times in the UTC local time are in the Gregorian 
calendar. Date times prior to the adoption of the Gregorian 
calendar are given in the retrospective astronomical Gregorian 
calendar. The year 1 B.C. is astronomical Gregorian year 0. 
The year 2 B.C. is astronomical Gregorian year-1. The year 1 A.D. 
is astronomical Gregorian year 1. The offset of the UTC local 
time is zero.

By default, DateAndTime instances have float seconds.  The class 
variable DefaultScale can be used to allow DateAndTime instances to 
have ScaledDecimal seconds to a specific scale'.
%

category: 'Overrides'
classmethod: DateAndTime
clockPrecision
	"Answer a <Duration> such that after that period of time passes, 
	#now is guaranteed to give a different result.
	Ideally implementations should answer the least such duration."

	| is64bit seconds |
	is64bit := 536870911 < SmallInteger maximumValue.
	seconds := is64bit
		ifTrue:  [1e-6]
		ifFalse: [1e-3].
	^Duration seconds: seconds.
%
category: 'Overrides'
classmethod: DateAndTime
secondsSince2001

	"^DateTime now asSecondsGmt - 3155760000."
	^System _timeGmtFloat - 978307200.
%
category: 'initialize-release'
method: DateAndTime
currentTimeZone

	^TimeZone current.
%
category: 'New Indexing Comparison'
method: DateAndTime
_classSortOrdinal

^ 70
%
category: 'Default Scale'
classmethod: DateAndTime
setDefaultScale: anIntOrNil

   DefaultScale := anIntOrNil ifNotNil: [anIntOrNil asInteger].
%
	
category: 'DateAndTime factory'
classmethod: DateAndTime
now

   ^DefaultScale 
      ifNil: [super now]
      ifNotNil: [self nowWithScale: DefaultScale]
%

category: 'DateAndTime factory'
classmethod: DateAndTime
nowWithScale: subsecondResolution
  "Create a DateAndTime for the current time with the specified number of digits of subsecond resolution"

  | res |
  (res := self _basicNew) 
       _secondsUTC: (self secondsSince2001 asScaledDecimal: subsecondResolution)
       offset: nil.
   ^ res.
%

! ------------------- Instance methods for DateAndTime

! fix 47971
category: 'private'
method: DateAndTime
printLocalOn: aStream
  "print the receiver on aStream, omitting the timezone offset and using space instead of T"
  | anArray |
  anArray := self asFloatParts.
  self printDateFrom: anArray on: aStream.
  aStream nextPut: $ .
  self printTimeFrom: anArray on: aStream.
%

method: DateAndTime
printLocalMsOn: aStream
    "print the receiver on aStream, omitting the timezone offset and using space instead of T.
     printed time is rounded to 0.001 second." 
  | anArray |
  anArray := self asFloatParts.
  anArray at: 7 put:( ScaledDecimal for: (anArray at: 7) scale: 3 ) .
  self printDateFrom: anArray on: aStream.
  aStream nextPut: $ .
  self printTimeFrom: anArray on: aStream.
%

method: DateAndTime
asStringMs
| strm |
strm := AppendStream on: String new .
self printLocalMsOn: strm .
^ strm contents
%
