!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: object.gs 33686 2014-06-17 17:30:46Z otisa $
!
! Superclass Hierarchy:
!   UnauthorizedObjectStub
!
!=========================================================================

set class UnauthorizedObjectStub
removeallmethods 
removeallclassmethods 

category: 'Documentation'
classmethod: 
comment

^  'Instances of UnauthorizedObjectStub represent objects for which an object 
fault would signal a SecurityError for no read authorization. 
They are produced as the in-memory state of reading such an object when
GEM_READ_AUTH_ERR_STUBS=TRUE in the session''s gem configuration, or
in a session started with GciX509Login when the hostagent for that session
filters objects being transitted to remote cache used by the session.'
%

! fixed 45088
category: 'Accessing'
method:
_status: checkClosureBool

"See Object >> _status"
<primitive: 39>
self _primitiveFailed: #_status: args: { checkClosureBool } .
self _uncontinuableError
%

method:
canBeWritten

"Returns true if the current user has write authorization for the
 receiver, false if not."

^ ((self _status: false)  bitAnd: 16r10) ~~ 0 
%
method:
isCommitted

"Returns true if the receiver existed in GemStone at the time the current
 transaction began.  Returns false otherwise."

^((self _status: true) bitAnd: 2) ~~ 0
%
method:
isSpecial
  ^ false
%
method:
isWritten

"Returns true if the receiver was a committed object at the start of the
 current transaction and as been written by the current transaction.
 Returns false otherwise."

^((self _status: true) bitAnd: 16rA ) == 16rA
%

method:
yourself
  ^ self
%

category: 'Comparing'
method:
== anObject

"(Optimized selector.)  Returns true if the receiver and the argument are the
 same object.

 This selector is optimized by the compiler and may not be reimplemented
 in any subclass.  This implementation is so that perform will work."

^ self == anObject
%

method:
~~ anObject

"(Optimized selector.)  Returns true if the receiver and the argument are not
 the same object.

 This selector is optimized by the compiler and may not be reimplemented
 in any subclass. This implementation is so that perform will work."

^ self ~~ anObject
%

method:
basicIdentityHash

"This method returns some Integer related to the identity of the receiver.  If
 two objects compare identically (==) to each other, the results of sending
 basicIdentityHash to each of those objects is equal."

"Faster than identityHash.  Used in GsMethodDictionary ."


<primitive: 321>

self _primitiveFailed: #basicIdentityHash .
self _uncontinuableError
%

method:
identityHash

"This method returns some Integer related to the identity of the receiver.  If
 two objects compare identically (==) to each other, the results of sending
identityHash to each of those objects will be equal."

<primitive: 609 >

self _primitiveFailed: #identityHash .
self _uncontinuableError
%

category: 'Converting'
method:
asOop

"Returns the value of the receiver's object-oriented pointer (OOP) as a
 positive Integer .  This is the receiver's unique identifier that 
 distinguishes it from all other objects.  For non-special objects
 and for instances of Character, Boolean, UndefinedObject, and JISCharacter
 the result will always be a SmallInteger.  For instances of SmallInteger
 and SmallDouble the result may be a LargeInteger. 

 The result is different from the result obtained in Gemstone64 v1.x"

<primitive: 86>
self _primitiveFailed: #asOop
%

category: 'Encoded OOPs'
method:
asOopNumber

"If the receiver is a non-special object, return the receiver's 
 oopNumber as defined under OOP_TAG_POM_OOP in the file 
 $GEMSTONE/include/gcioop.ht  , otherwise return nil . "

| oop |
oop := self asOop .
oop _isSmallInteger ifFalse:[ ^ nil ].
(oop bitAnd:7) == 1 ifFalse:[ ^ nil ].
^ oop bitShift: -8
%

category: 'Private'
method:
__asString
  self isCommitted ifTrue:[
    | str |
    str := 'an UnauthorizedObjectStub(read authorization error on oop ' copy. 
    str add: self asOop asString; add: $) .
    ^ str
  ].
  ^ 'an UnauthorizedObjectStub(not committed)' copy .
%
method:
_topazAsString
 ^ [ self __asString
   ] onSynchronous: AbstractException do:[:ex |
     '<error during asString>'
   ]
%
method:
asString
^ self __asString
%
method:
printString
^ self __asString
%

! fixed 45760
category: 'Error Handling'
method:
doesNotUnderstand: aMessageDescriptor

"The method is for compatiblity with Gs64 v2.x, and assumes you are using 
   only method environment 0  for all of your Smalltalk code."

| sel str |
sel := aMessageDescriptor at: 1 .
(str := self __asString) 
   add: ' does not understand ';
   add: sel _asSource .
^ SecurityError new _number: 2115"AUTH_ERR_SEG_READ" ;  signal: str .
%

input $upgradeDir/dnuErrorMethods.gs

method: 
class
"Returns the object that is the receiver's class."
<primitive: 610 >
^ self _primitiveFailed: #class .
%

