!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   Number, Magnitude, Object.
!
!=========================================================================

removeallmethods Number
removeallclassmethods Number

category: 'For Documentation Installation only'
classmethod: Number
installDocumentation

self comment: 
'Number is an abstract superclass that establishes protocol for all GemStone
 Smalltalk numbers.  Concrete subclasses include Float, SmallInteger, and
 Fraction.

 You may not create new instances of Number.
'.
%

category: 'Instance Creation'
classmethod: Number
new

"Disallowed."

self shouldNotImplement: #new 
%

category: 'Instance Creation'
classmethod: Number
migrateNew

"Override default migrateNew behavior with #_basicNew."

^ self _basicNew
%

category: 'Instance Creation'
classmethod: Number
new: anInteger

"Disallowed."

self shouldNotImplement: #new:
%

category: 'Error Handling'
method: Number
_errorDivideByZero

"Generates a divide by 0 error."

^ ZeroDivide new _number: 2026 ; reason: 'numErrIntDivisionByZero'; 
    dividend: self ;
    signal
%

category: 'Accessing'
method: Number
denominator

"(Subclass responsibility.)  Returns the denominator of a Fraction representing
 the receiver."

Number subclassResponsibility: #denominator
%

category: 'Accessing'
method: Number
numerator

"(Subclass responsibility.)  Returns the numerator of a Fraction representing
 the receiver."

Number subclassResponsibility: #numerator
%

category: 'Accessing'
method: Number
sign

"Returns 1 if the receiver is greater than zero, -1 if the receiver is less
 than zero, and zero if the receiver is zero."

(self > 0) ifTrue:[ ^ 1 ].
(self < 0) ifTrue:[ ^ -1 ].
^0
%

category: 'Testing'
method: Number
negative

"Returns true if the receiver is less than zero, false if the receiver is zero
 or greater."

^self < 0
%

category: 'Testing'
method: Number
positive

"Returns true if the receiver is greater than or equal to zero, false if the
 receiver is less than zero."

^self >= 0
%

category: 'Testing'
method: Number
strictlyPositive

"Returns true if the receiver is greater than zero and false if it is less than
 or equal to zero."

^ self > 0
%

category: 'Converting'
method: Number
asDecimalFloat

"(Subclass responsibility.)  Returns a DecimalFloat representing the receiver."

Number subclassResponsibility: #asDecimalFloat

%

category: 'Converting'
method: Number
asFloat

"(Subclass responsibility.)  Returns a SmallDouble or Float representing the receiver."

Number subclassResponsibility: #asFloat
%

set compile_env: 0
category: 'Converting'
method: Number
asFloatD
	"We only have one precision of floating point."

	^self asFloat.
%

set compile_env: 0
category: 'Converting'
method: Number
asFloatE
	"We only have one precision of floating point."

	^self asFloat.
%

set compile_env: 0
category: 'Converting'
method: Number
asFloatQ
	"We only have one precision of floating point."

	^self asFloat.
%

category: 'Deprecated'
method: Number
asSmallFloat

"SmallFloat is deprecated.
 This method returns a SmallDouble or Float equal to the receiver."

self deprecated: 'asSmallFloat deprecated in v3.0 and later, use asFloat'.
^ self asFloat asSmallFloat 
%

category: 'Converting'
method: Number
asScaledDecimal: scale

"Returns a ScaledDecimal representing the receiver, with
 the specified scale.

 If the receiver is an Integer, or responds to asFraction with an
 Integer result, the result will have the specified scale.  This
 is to conform with other Smalltalk implementations. We do not
 conform to ANSI , which specifieds a ScaledDecimal result
 with scale 0 in this case."

| f |
f := self asFraction .
f _isInteger ifTrue:[ ^ ScaledDecimal for: f scale: scale  ].
^ f asScaledDecimal: scale.
%

category: 'Converting'
method: Number
asFixedPoint: scale 

| f |
f := self asFraction .
f _isInteger ifTrue:[ ^ FixedPoint numerator: f denominator: 1 scale: scale].
^ f asFixedPoint: scale
%

category: 'Converting'
method: Number
asFraction

"(Subclass responsibility.)  Returns a Fraction that represents the receiver."

Number subclassResponsibility: #asFraction
%

! reverted to remove fix 42560
category: 'Converting'
method: Number
asInteger

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located.  In particular, returns the
 receiver if the receiver is an integer.

 Truncates to be compatible with client Smalltalk implementations and 
 differs from ANSI which specifies rounding ."

^self truncated
%

category: 'Converting'
method: Number
_coerce: aNumber

"(Subclass responsibility.)  Returns aNumber as an instance of the class of the
 receiver, or an instance of any subclass of the receiver.  This method must be
 defined by all subclasses of Number."

Number subclassResponsibility: #_coerce
%

category: 'Converting'
method: Number
_generality

"(Subclass responsibility.)  Returns the integer that represents the ordering
 of the receiver in the generality hierarchy."

" Class           result of _generality
   SmallInteger     20
   Integer	    40
   ScaledDecimal    60
   FixedPoint       65
   Fraction         70
   SmallFloat       80
   SmallDouble      85
   Float            90
   DecimalFloat    100
"

Number subclassResponsibility: #_generality
%

category: 'Converting'
method: Number
_retry: aSelector coercing: aNumber

"A difference in representation between the receiver and aNumber has prevented
 aSelector from being executed.  Coerce either the receiver or aNumber to the
 higher of their generalities and attempt to execute the method represented by
 aSelector again. 

 This is the basis of the type coercion logic for all Numbers."
| gen argGen |
" numeric primitives #=, #~= check for   (anArg isKindOf: Number)
  and if that result is  false ,   the compare result is returned
  directly by those primitives
"
gen := self _generality .
argGen := aNumber _generality .
(gen < argGen ) ifTrue: [
  ^ (aNumber _coerce: self) perform: aSelector with: aNumber
].
(gen > argGen ) ifTrue: [
  ^self perform: aSelector with: (self _coerce: aNumber)
].
self _uncontinuableError.  "coercion failed"
%
method: Number
_retry: aSelector coercing: aNumber into: anArray

"A difference in representation between the receiver and aNumber has prevented
 aSelector from being executed.  Coerce either the receiver or aNumber to the
 higher of their generalities and attempt to execute the method represented by
 aSelector again.

 This is the basis of the type coercion logic for all Numbers."
| gen argGen |
gen := self _generality .
argGen := aNumber _generality .
(gen < argGen ) ifTrue: [
  ^ (aNumber _coerce: self) perform: aSelector with: aNumber with: anArray
].
(gen > argGen ) ifTrue: [
  ^self perform: aSelector with: (self _coerce: aNumber) with: anArray
].
self _uncontinuableError.  "coercion failed"
%

category: 'Repository Conversion'
method: Number
convert
"If receiver is a committed object and receiver's class on disk was one of
   ObsSmallFloat, ObsFloat, ObsLargePositiveInteger, ObsLargeNegativeInteger
 the receiver is added to the writeSet of current transaction,
 otherwise has no effect.
 Returns true if receiver was added to the writeSet, false otherwise."

<primitive: 744>
self _primitiveFailed:#convert
%

category: 'Truncation and Rounding'
method: Number
ceiling

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as positive infinity.  In particular, returns the
 receiver if the receiver is an integer."

"the actual implementation of this method is in bomlastconv.gs"

 self _uncontinuableError
%

category: 'Truncation and Rounding'
method: Number
floor

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as negative infinity.  In particular, returns the
 receiver if the receiver is an integer."

|result|

result := self truncated.
(self >= 0)
   ifTrue: [^result].
(self = result)
   ifTrue: [^result]
   ifFalse: [^ (result - 1) truncated]
%

category: 'Truncation and Rounding'
method: Number
rounded

"Returns the integer nearest in value to the receiver."

^(self + (self sign / 2)) truncated
%

category: 'Truncation and Rounding'
set compile_env: 0
method: Number
roundedHalfToEven

"Returns the integer nearest in value to the receiver. If the receiver is 
exactly halfway between two integers, return the even one."

^self asFraction roundedHalfToEven
%

category: 'Truncation and Rounding'
method: Number
roundTo: aNumber

"Returns the multiple of aNumber that is nearest in value to the receiver."

aNumber = 0
  ifTrue: [^0]
  ifFalse: [^(self / aNumber) rounded * aNumber]
%

method: Number
roundedNoFpe

^ self rounded
%

category: 'Truncation and Rounding'
method: Number
truncated

"(Subclass responsibility.)  Returns the integer nearest in value to
 the receiver, in the direction toward zero."

Number subclassResponsibility: #truncated
%

method: Number
truncatedNoFpe

^ self truncated
%

method: Number
integerPart
  "Returns the integer nearest in value to the receiver, 
   in the direction toward zero."
  ^ self truncated
%

category: 'Truncation and Rounding'
method: Number
truncateTo: aNumber

"Returns the multiple of aNumber that is closest to the receiver, on
 the same side of the receiver as zero is located.  In particular,
 returns the receiver if the receiver is a multiple of aNumber."

aNumber = 0
  ifTrue: [^0]
  ifFalse: [^(self quo: aNumber) * aNumber]
%

method: Number
fractionPart
 "Returns the fraction remaining after the receiver is truncated toward zero."
  ^ self - self truncated 
%

category: 'Arithmetic'
method: Number
* aNumber

"(Subclass responsibility.)  Returns the result of multiplying the receiver by
 aNumber."

Number subclassResponsibility: #*
%

category: 'Arithmetic'
method: Number
+ aNumber

"(Subclass responsibility.)  Returns the sum of the receiver and aNumber."

Number subclassResponsibility: #+
%

category: 'Arithmetic'
method: Number
- aNumber

"(Subclass responsibility.)  Returns the difference between the receiver and
 aNumber."

Number subclassResponsibility: #-
%

category: 'Arithmetic'
method: Number
/ aNumber

"(Subclass responsibility.)  Returns the result of dividing the receiver by
 aNumber."

Number subclassResponsibility: #/
%

category: 'Arithmetic'
method: Number
// aNumber

"Divides the receiver by aNumber.  Returns the integer quotient, with
 truncation toward negative infinity.  For example,

 9 // 4 = 2
 -9 // 4 = -3
 -0.9 // 0.4 = -3

 The selector \\ returns the remainder from this division."

^ (self / aNumber) floor
%

category: 'Arithmetic'
method: Number
abs

"Returns a Number that is the absolute value of the receiver."

(self < 0)
   ifTrue: [^self negated].
^self
%

category: 'Arithmetic'
method: Number
negated

"Returns a Number that is the negation of the receiver."

^0 - self
%

category: 'Arithmetic'
method: Number
quo: aNumber

"Divides the receiver by aNumber.  Returns the integer quotient, with
 truncation toward zero.  For example,

 -9 quo: 4 = -2
 -0.9 quo: 0.4 = -2

 The selector rem: returns the remainder from this division."

^ (self / aNumber) truncated
%

category: 'Arithmetic'
method: Number
reciprocal

"Returns 1 divided by the receiver.  Generates an error if the receiver is 0."

(self = 0)
    ifTrue:[^self _errorDivideByZero].  "division by zero error"
^ 1 / self
%

category: 'Arithmetic'
method: Number
rem: aNumber

"Returns the integer remainder defined in terms of quo: (division
 of the receiver by aNumber, with truncation toward zero)."

^ (self - ((self quo: aNumber) * aNumber)) 
%

category: 'Arithmetic'
method: Number
\\ aNumber

"Returns the modulo remainder defined in terms of //.  Returns a Number with
 the same sign as the argument aNumber.  For example,

 9 \\ 4 = 1
 -9 \\ 4 = 3
 9 \\ -4 = -3
 0.9 \\ 0.4 = 0.1"

^ self - (self // aNumber * aNumber)
%

category: 'Deprecated'
method: Number
_sqrt

self deprecated: '_sqrt obsolete,  Use #sqrt instead.'.
^ self sqrt
%

category: 'Arithmetic'
method: Number
squared

"Returns the multiplicative product of the receiver and itself."

^ self * self
%

category: 'Testing'
method: Number
even

"Returns true if the receiver is an even integer, false otherwise."

^self \\ 2 = 0
%

category: 'Testing'
method: Number
odd

"Returns true if the receiver is an odd integer, false otherwise."

^ self even == false
%

category: 'Arithmetic'
method: Number
sqrt

"Returns the square root of the receiver.  This method currently returns a
 Float.  This will eventually change, so that exact squares of Integers and
 Fractions return Integers and Fractions, for instance."

^ self asFloat sqrt
%

category: 'Formatting'
method: Number
printString

"Returns a String whose contents are a displayable representation of the
 receiver."

^self asString
%

category: 'Formatting'
method: Number
printOn: aStream

"Appends a printable representation of the receiver to aStream and returns 
 the receiver."

aStream nextPutAll: (self asString).
^ self.
%

! deleted _idxCompareLessThan: v2.0

! deleted _idxCompareLessThanOrEqual: v2.0

! deleted _idxCompareGreaterThan: v2.0

! deleted _idxCompareGreaterThanOrEqual: v2.0

category: 'Indexing Support'
method: Number
_isNaN

"Returns whether the receiver is quiet NaN or signaling NaN.
 This method is only to be used by the indexing subsystem."

^ false
%

category: 'Arithmetic'
method: Number
ln

"Returns the natural logarithm of the receiver."

^ self asFloat ln
%

category: 'Arithmetic'
method: Number
log: x

"Returns the base x logarithm of the receiver."

^ self ln / x ln
%

category: 'Arithmetic'
method: Number
floorLog: x

"Returns an Integer that is the floor of the receiver's log base x."

^ (self log: x) floor.
%

category: 'Arithmetic'
method: Number
exp

"Returns e raised to the power of the receiver."

^ self asFloat exp
%

category: 'Arithmetic'
method: Number
arcCos

"Returns the arc-cosine of the receiver in radians."

^ self asFloat arcCos
%

category: 'Arithmetic'
method: Number
arcSin

"Returns the arc-sine of the receiver in radians."

^ self asFloat arcSin
%

category: 'Arithmetic'
method: Number
arcTan

"Returns the arc-tangent of the receiver in radians."

^ self asFloat arcTan
%

category: 'Arithmetic'
method: Number
cos

"Returns the cosine of the receiver which is treated as an
 angle expressed in radians."

^ self asFloat cos
%

category: 'Arithmetic'
method: Number
sin

"Returns the sine of the receiver which is treated as an
 angle expressed in radians."

^ self asFloat sin
%

category: 'Arithmetic'
method: Number
tan

"Returns the tangent of the receiver which is treated as an
 angle expressed in radians."

^ self asFloat tan
%

category: 'Arithmetic'
method: Number
degreesToRadians

"Assuming the receiver represents an angle in degrees, returns
 the angle in radians."

"The actual implementation of this method is in bomlast.gs"

self _uncontinuableError
%

category: 'Arithmetic'
method: Number
radiansToDegrees

"Assuming the receiver represents an angle in radians, returns
 the angle in degrees."

"The actual implementation of this method is in bomlast.gs"

self _uncontinuableError
%

category: 'Arithmetic'
method: Number
raisedTo: aNumber

"Returns the receiver raised to the power of the argument."

(aNumber _isInteger) ifTrue: [
  ^ self raisedToInteger: aNumber
] ifFalse: [
  ^ self asFloat raisedTo: aNumber
]
%

category: 'Arithmetic'
method: Number
raisedToInteger: aNumber

"Returns the receiver raised to the power of the argument.
 aNumber must be a kind of Integer."

(aNumber _isInteger) ifFalse:[
  aNumber _validateClass: Integer
].
aNumber < 0 ifTrue:[ 
  ^ 1 / (self _raisedToPositiveInteger: (0 - aNumber)) 
].
^ self _raisedToPositiveInteger: aNumber
%

category: 'Private'
method: Number
_raisedToPositiveInteger: aNumber
  "Sender must ensure aNumber is an Integer >= 0"

aNumber <= 1 ifTrue:[  
  aNumber = 0 ifTrue:[ ^ 1 ]
              ifFalse:[ ^ self ].
].
aNumber _bottomBit == 1 ifTrue:[  "aNumber is odd"
  ^ self * (self _raisedToPositiveInteger: aNumber - 1 )
].
^ (self * self) _raisedToPositiveInteger: (aNumber bitShift: -1)
%

! radiansToDegrees,

category: 'Accessing'
method: Number
_getKind

"Returns the kind of the receiver based on the list in Number>>kind.  Only floats
 have a kind other than 1."

^ 1
%

category: 'Testing'
method: Number
isExceptionalFloat
"return true if the receiver is not a normal number"

| k |
k := self _getKind .
(k == 1 or:[ k == 4]) ifTrue:[ ^ false ].
^ true
%

category: 'Accessing'
method: Number
kind

"Returns a Symbol from the following list:

   #normal
   #subnormal
   #infinity
   #zero
   #quietNaN
   #signalingNaN

 The symbol tells what kind of floating-point number the receiver is.
 Refer to IEEE standards 754 and 854 for more information.  Numbers that
 are not instances of a floating point class are always normal."

^Number _kindList at: (self _getKind)
%


category: 'Private'
classmethod: Number
_kindList

""

"   1       2          3          4      5              6 "
^ #(#normal #subnormal #infinity #zero #quietNaN #signalingNaN)
%

category: 'Decompiling without Sources'
method: Number
_asSource

""

^ self asString
%

category: 'Comparing'
method: Number
hash

"Returns a SmallInteger related to the value of the receiver."

"Two Numbers for which = is true, must have the same hash value."

"This method should be reimplemented in Float and SmallInteger."

^ self asFloat hash
%

category: 'Storing and Loading'
method: Number
writeTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."
    
passiveObj nextPutAll: self describe.
passiveObj space
%

category: 'Flow of Control'
method: Number
downTo: aNumber by: stepValue do: aBlock

"Iteratively evaluates the one-argument block aBlock, using the block's single
 argument as the iteration control variable.  Initially, that control variable
 is set to the receiver.  
 The argument stepValue must be a Number greater than zero and
 aNumber must be a Number.

 The block is evaluated while the control variable is greater than or equal to
 aNumber.  After each evaluation, the control variable is decremented by
 stepValue.  If the receiver is less than aNumber, the block is not evaluated
 at all.  Returns the receiver."

| j |
j := self .
stepValue > 0 ifFalse:[ stepValue _error: #rtErrArgNotPositive ].
[ j >= aNumber ] whileTrue:[ 
   aBlock value: j . 
   j := j - stepValue 
   ] 
%

category: 'Flow of Control'
method: Number
downTo: aNumber do: aBlock

"Iteratively evaluates the one-argument block aBlock, using the block's single
 argument as the iteration control variable.  Initially, that control variable
 is set to the receiver.  The block is evaluated while the control variable is
 greater than or equal to aNumber (which must be a kind of Number).  After each
 evaluation, the control variable is decremented by 1.  If the receiver is less
 than aNumber, the block is not evaluated at all.  Returns the receiver."

self downTo: aNumber by: 1 do: aBlock
%

category: 'Flow of Control'
method: Number
timesRepeat: aBlock

"(Reserved selector.)  If the receiver is greater than zero, evaluates the
 zero-argument block aBlock the number of times represented by the receiver.
 (If the receiver is zero or negative, aBlock is not executed.).

 A method which sends timesRepeat: will have a step point generated for
 the send of the timesRepeat:  and a step point for the loop index increment
 and test instructions.

 The following is present for use by perform: , not a recursive send."

self timesRepeat: [aBlock value]
%

! to:by:do: moved to object.gs

category: 'Flow of Control'
method: Number
_downTo: aNumber by: delta do: aBlock

"(Reserved selector.) 
 Iteratively evaluates the one-argument block aBlock, using the block's single
 argument as the iteration control variable.  Initially, that control variable
 is set to the receiver.  The block is evaluated while the control variable is
 greater than or equal to aNumber (which must be a kind of Number).  After each
 evaluation, the control variable is decremented by delta.  If the receiver is less
 than aNumber for a positive delta, the block is not evaluated at all.  Returns the 
 receiver.

 The following is present for use by perform: , not a recursive send."

self _downTo: aNumber by: delta do: aBlock
%

category: 'Flow of Control'
method: Number
_downTo: aNumber do: aBlock

"(Reserved selector.) 
 Iteratively evaluates the one-argument block aBlock, using the block's single
 argument as the iteration control variable.  Initially, that control variable
 is set to the receiver.  The block is evaluated while the control variable is
 greater than or equal to aNumber (which must be a kind of Number).  After each
 evaluation, the control variable is decremented by 1.  If the receiver is less
 than aNumber, the block is not evaluated at all.  Returns the receiver.

 The following is present for use by perform: , not a recursive send."

self _downTo: aNumber do: aBlock
%

! fixed 43742
category: 'Intervals'
method: Number
to: aNumber by: stepValue

"Returns an Interval for the numbers between the receiver and aNumber, where 
 each number is the previous number plus stepValue."

stepValue = 0 ifTrue: [ArgumentError signal: 'stepValue may not be zero!'].
^ Interval from: self to: aNumber by: stepValue.
%

! to:do: moved to object.gs

category: 'Intervals'
method: Number
to: aNumber

"Returns an Interval for the numbers between the receiver and aNumber, where 
 each number is the previous number plus 1."

^ Interval from: self to: aNumber.
%

category: 'Generality'
method: Number
moreGeneralThan: anOperand

"Returns true if the receiver has higher generality than anOperand, 
 false otherwise."

^ self _generality > anOperand _generality
%

category: 'Generality'
method: Number
lessGeneralThan: anOperand

"Returns true if the receiver has lower generality than anOperand, 
 false otherwise."

^ self _generality < anOperand _generality
%

category: 'New Indexing Comparison'
method: Number
_classSortOrdinal

^ 90
%
category: 'New Indexing Comparison'
method: Number
_idxForCompareEqualTo: arg

""

^arg _idxForCompareEqualToNumber: self
%
category: 'New Indexing Comparison'
method: Number
_idxForSortEqualTo: arg

""

^arg _idxForSortEqualToNumber: self
%
category: 'New Indexing Comparison'
method: Number
_idxForSortGreaterThan: arg

""

^arg _idxForSortNumberGreaterThanSelf: self
%
category: 'New Indexing Comparison'
method: Number
_idxForSortGreaterThanOrEqualTo: arg

""

^arg _idxForSortNumberGreaterThanOrEqualToSelf: self
%
category: 'New Indexing Comparison'
method: Number
_idxForSortLessThan: arg

""

^arg _idxForSortNumberLessThanSelf: self
%
category: 'New Indexing Comparison'
method: Number
_idxForSortLessThanOrEqualTo: arg

""

^arg _idxForSortNumberLessThanOrEqualToSelf: self
%

category: 'New Indexing Comparison'
method: Number
_idxForSortNotEqualTo: arg

""

^arg _idxForSortNotEqualToNumber: self
%

category: 'New Indexing Comparison - for Sort'
method: Number
_idxForSortEqualToNumber: aNumber

"second half of a double dispatch call from Number>>_idxForSortEqualTo:."

( self _isNaN and: [ aNumber _isNaN ] ) ifTrue: [ ^ true ].
^ self = aNumber
%
category: 'New Indexing Comparison - for Sort'
method: Number
_idxForSortNumberGreaterThanOrEqualToSelf: aNumber

"second half of a double dispatch call from Number>>_idxForSortGreaterThanOrEqualTo:. Note that aNumber should be the receiver in any >= comparison"

( self _isNaN and: [ aNumber _isNaN ] ) ifTrue: [ ^ true ].
aNumber _isNaN ifTrue: [ ^ true ].
self _isNaN ifTrue: [ ^ false ].
^aNumber >= self
%
category: 'New Indexing Comparison - for Sort'
method: Number
_idxForSortNumberGreaterThanSelf: aNumber

"second half of a double dispatch call from Number>>_idxForSortGreaterThan:. Note that aNumber should be the receiver in any > comparison"

( self _isNaN and: [ aNumber _isNaN ] ) ifTrue: [ ^ false ].
aNumber _isNaN ifTrue: [ ^ true ].
self _isNaN ifTrue: [ ^ false ].
^aNumber > self
%
category: 'New Indexing Comparison - for Sort'
method: Number
_idxForSortNumberLessThanOrEqualToSelf: aNumber

"second half of a double dispatch call from Number>>_idxForSortLessThanOrEqualTo:. Note that aNumber should be the receiver in any <= comparison"

( self _isNaN and: [ aNumber _isNaN ] ) ifTrue: [ ^ true ].
aNumber _isNaN ifTrue: [ ^ false ].
self _isNaN ifTrue: [ ^ true ].
^aNumber <= self
%
category: 'New Indexing Comparison - for Sort'
method: Number
_idxForSortNumberLessThanSelf: aNumber

"second half of a double dispatch call from Number>>_idxForSortLessThan:. Note that aNumber should be the receiver in any < comparison"

( self _isNaN and: [ aNumber _isNaN ] ) ifTrue: [ ^ false ].
aNumber _isNaN ifTrue: [ ^ false ].
self _isNaN ifTrue: [ ^ true ].
^aNumber < self
%

category: 'New Indexing Comparison - for Sort'
method: Number
_idxForSortNotEqualToNumber: aNumber

"second half of a double dispatch call from Number>>_idxForSortNotEqualTo:."

( self _isNaN and: [ aNumber _isNaN ] ) ifTrue: [ ^ false ].
^ self ~= aNumber
%

category: 'New Indexing Comparison - for Compare'
method: Number
_idxForCompareEqualToNumber: aNumber

"second half of a double dispatch call from Number>>_idxForCompareEqualTo:."

^aNumber = self
%

category: 'Modifying Classes'
classmethod: Number
_addInvariantClassVar: aSymbol value: aVal
  | dict assoc |
  self _validatePrivilege ifFalse:[ ^ nil ].
  (dict := classVars) ifNil:[
    dict := self _createClassVarsDict .
  ].
  assoc := dict associationAt: aSymbol otherwise: nil .
  assoc ifNotNil:[ | oldVal |
    oldVal := assoc _value .
    "special comparison logic to handle NaN classVars in Float classes"
    (oldVal class == aVal class and:[ oldVal asString = aVal asString ]) ifTrue:[
      ^ self "no change"
    ].
  ].
  ^ super _addInvariantClassVar: aSymbol value: aVal
%
category: 'Json'
method: Number
printJsonOn: aStream

	aStream nextPutAll: self asFloat asStringLocaleC.
%

category: 'Private'
method: Number
_nonZeroGte: aNumber

"Sent by code emitted for to:by:do: , if the
 bytecode for _nonZeroGte fails .
 Signals an ArgumentError if self = 0 , 
 otherwise returns self >= aNumber "

0 = self ifTrue:[ 
   ArgumentError new 
    object: self ;
    signal:'argument to by: of to:by:do: must be nonzero'
].
^ self >= aNumber
%

category: 'Arithmetic'
method: Number
quoRem: aNumber into: anArray

"anArray should be an Array of size 2.
 Returns anArray "

 anArray at: 1 put: (self quo: aNumber) .
 anArray at: 2 put: (self rem: aNumber) .
 ^ anArray .
%
method: Number
divMod: aNumber into: anArray

"anArray should be an Array of size 2. 
 Returns anArray "

 anArray at: 1 put: (self // aNumber) .
 anArray at: 2 put: (self \\ aNumber) .
 ^ anArray
%


! REMINDER: you must filein bomlast.gs to get methods with float literals

category: 'Indexing Support'
set compile_env: 0
method: Number
_idxBasicCanCompareWithClass: aClass
  "Returns true if the receiver may be inserted into a basic BtreeNode whose 
   #lastElementClass is <aClass> (see RangeEqualityIndex class>>isBasicClass:)."

  (super _idxBasicCanCompareWithClass: aClass)
    ifTrue: [ ^ true ].
  ^ aClass isSubclassOf: Number
%

category: 'Custom Numeric Literals'
classmethod: Number
parseLiterals: aCharacter exponentRequired: aBoolean
  "Add a new numeric literal class to those recognized by the compiler.

   aCharacter specifies an new exponent letter to be recognized, and
   must be a Unicode alphabetic codePoint < 127 .

   If aBoolean==false the literal's exponent is parsed as for ScaledDecimal, 
      Exponent =  aCharacter [ ['-' ] Digits ]

   If aBoolean==true, the literal's exponent is parsed as for binary floats,
      Exponent = aCharacter ['-'] Digits 

   The compiler will send fromString: to the class which was the receiver of 
   parseLiterals:exponentRequired:  to construct an instance of a new 
   numeric literal class. The fromString: implementation must be able to 
   parse the string and return a new instance of the receiver. 

   Should be invoked from session initialization; once invoked
   the new literal will be recognized until the session logs out.

   To deinstall a literal class from the compiler.
   invoke with   aBoolean == nil  ; if the receiver and aCharacter
   do not match an installed literal class, an Error is signalled.  

   An Error is signaled if the receiver or aCharacter matches a
   predefined literal specified in the GemStone BNF.
   Thus aCharacter may not be one of   e E d D f F p q r s  .

   Returns an Array of triples listing additional literals recognized by 
   the compiler, of the form  { exponentCharacter, aClass, aBoolean  ... }

   If aCharacter==nil, has no effect and returns the Array of literals recognized."

  <primitive: 997>
  aBoolean ifNotNil:[ aBoolean _validateClass: Boolean ].
  aCharacter _validateClass: Character .
  ^ self _primitiveFailed: #parseLiterals:exponentRequired: args: { aCharacter . aBoolean }
%
category: 'Custom Numeric Literals'
classmethod: Number
_fromLiteralString: aString
  "Sent from C by the compiler to construct an instance of a literal which has
   been registered by invoking parseLiterals:exponentRequired: 
   Returns a kind of Number, or returns a String describing an error."
^ [
    self fromString: aString
  ] onSynchronous: Exception do:[:ex | | msg sfx |
    msg := ex asString .
    (msg isKindOfClass: String) ifFalse:[ msg := 'error during _fromLiteralString:'].
    sfx := ', for a String, ' . 
    (msg at:( msg size - sfx size + 1) equals: sfx) ifTrue:[
      msg size: msg size - sfx size . 
    ].
    ^ '; ', msg 
  ]
%

category: 'Instance Creation'
classmethod: Number
fromString: aString

"So topaz will work early in filein, 
  This implementation removed in number_2.gs"

^ Integer fromString: aString
%
