!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   RangeIndexReadStream, BtreeReadStream, Stream, Object.
!   ReversedRangeIndexReadStream, RangeIndexReadStream, BtreeReadStream, Stream, Object.
!
!=========================================================================

! RangeIndexReadStream class created in idxclasses.topaz

removeallmethods RangeIndexReadStream
removeallclassmethods RangeIndexReadStream

category: 'For Documentation Installation only'
classmethod: RangeIndexReadStream
installDocumentation

self comment:
'RangeIndexReadStream, like its superclass BtreeReadStream, supports the
 composition of query results by providing access to a B-tree structure.  Its
 ''next'' and ''atEnd'' methods are used the same way as those of
 BtreeReadStream in iterating through the B-tree.

 RangeIndexReadStream differs from BtreeReadStream in that it uses the reverse
 mappings to B-tree nodes that are found in a RangeEqualityIndex to obtain the
 next entry.  You can supply that index when you create the stream, and the
 index identifies the ordering used to return the entries.

Constraints:
	endNode: BtreeLeafNode
	endIndex: SmallInteger
	currentStack: Array
	rangeIndex: RangeEqualityIndex
	setIterationIndexes: Object

instvar rangeIndex - The RangeEqualityIndex for an instance of this class.

instvar setIterationIndexes -- An Array of Integers that indicates the 
  offset into BucketValueBags along the path.
'
%

! ------------------- Class methods for RangeIndexReadStream
category: 'Instance Creation'
classmethod: RangeIndexReadStream
new

"Create an initialized instance of the receiver."

| newOne |
newOne := super new.
newOne setIterationIndexes: { } .
^ newOne
%

category: 'Instance Creation'
classmethod: RangeIndexReadStream
on: aRangeIndex
  "Create a stream that can access the entire contents of the given RangeEqualityIndex."

  | newStream |
  newStream := super on: aRangeIndex btreeRoot.
  newStream rangeIndex: aRangeIndex.
  ^ newStream
%

! ------------------- Instance methods for RangeIndexReadStream
category: 'Private'
method: RangeIndexReadStream
_advance

"Advances the B-tree portion of the stream without advancing any of the set
 iteration indexes."

^ super next
%

category: 'Testing'
method: RangeIndexReadStream
_atEnd
  "Returns true if there are no more elements to return through the logical
 iteration of the stream."

  ^ self _atEndForStream: false
%

category: 'Testing'
method: RangeIndexReadStream
atEnd
  "Returns true if there are no more elements to return through the logical
 iteration of the stream."

  ^ self _atEndForStream: true
%

category: 'Private'
method: RangeIndexReadStream
getNext
  "Returns the next value on a stream of range index values."

  | val |
  " get the initial value at the end of the path (but do not advance the stream yet) "
  val := self _peekValue.
  rangeIndex isIndexOnRootNsc
    ifFalse: [ 
      | lastTerm |
      " if the index is not on the elements of the root NSC "
      lastTerm := rangeIndex at: rangeIndex size.
      lastTerm isSetValuedTerm
        ifTrue: [ 
          " if the last path term indicates that the index is on the elements of
            the NSC itself, then get the next value from the last term (the NSC)"
          val := self nextValueFor: val term: lastTerm ].
      rangeIndex lastPathComponentsDictionaryOffset
        _downTo: 1
        do: [ :i | 
          " now iterate through the path components to get the final value "
          val := self nextValueFor: val term: (rangeIndex at: i) ] ].
  super next.	" advance the B-tree current position "
  ^ val
%

category: 'Accessing'
method: RangeIndexReadStream
next
  "Returns the next value on a stream of range index values."

  self atEnd
    ifTrue: [ self _errorEndOfStream ].
  ^ self getNext
%

category: 'Private'
method: RangeIndexReadStream
nextValueFor: val term: pathTerm

"Returns the next value on a stream of range index values."

| nextVal |

nextVal := rangeIndex indexDictionary at: val term: pathTerm.
" see if the next value is a BucketValueBag "
(BucketValueBag _hasInstance: nextVal)
  ifTrue: [ nextVal := self nextValueInBag: nextVal level: pathTerm offset ].

^ nextVal
%

category: 'Accessing'
method: RangeIndexReadStream
rangeIndex

"Returns the value of the instance variable 'rangeIndex'."

^rangeIndex
%

category: 'Updating'
method: RangeIndexReadStream
rangeIndex: newValue

"Modify the value of the instance variable 'rangeIndex'."

rangeIndex := newValue.
self initializeSetIterationIndexes
%

category: 'Accessing'
method: RangeIndexReadStream
setIterationIndexes

"Returns the value of the instance variable 'setIterationIndexes'."

^setIterationIndexes
%

category: 'Updating'
method: RangeIndexReadStream
setIterationIndexes: newValue

"Modify the value of the instance variable 'setIterationIndexes'."

setIterationIndexes := newValue
%

category: 'Private'
method: RangeIndexReadStream
nextValueInBag: aBag level: offset

"Get the next value in the given BucketValueBag and update the set iteration
 indexes accordingly."

| setOffset |

setOffset := (setIterationIndexes at: offset) + 1.

setIterationIndexes at: offset put:
  (setOffset == aBag size ifTrue: [ 0 ] ifFalse: [ setOffset ]).

" return the object in the set "
^ aBag _at: setOffset.
%

category: 'Query Support'
method: RangeIndexReadStream
makeNsc

"Returns a new NSC that contains all the elements from the receiver."

| tmpHolder |
tmpHolder := NscBuilder for: rangeIndex nscRoot species new max: rangeIndex nscRoot size.

rangeIndex hasEnumeratedTerm 
  ifTrue: [ [ self atEnd ] whileFalse: [ tmpHolder add: self next ] ]
  ifFalse: [
    streamQuerySpec
      ifNil: [ 
        " If B-tree contains mapping to root object "
        rangeIndex size == 1
            ifTrue: [ self _valuesAddInto: tmpHolder]
            ifFalse: [ self _uniqueValuesAddInto: tmpHolder ] ]
      ifNotNil: [ 
        " If B-tree contains mapping to root object "
        rangeIndex size == 1
            ifTrue: [ self _valuesAddInto: tmpHolder spec: streamQuerySpec ]
            ifFalse: [ self _uniqueValuesAddInto: tmpHolder spec: streamQuerySpec ] ] ].

" make the stream be at the end "
currentStack at: 1 put: 0.

^ tmpHolder completeBag
%

category: 'Query Support'
method: RangeIndexReadStream
_uniqueValuesAddInto: collection

"Iterates over the unique values of the B-tree, traversing the reverse mappings
 to the root object.  Adds the root to the given NSC."

| querySpec |
querySpec := BtreeQuerySpec new.
^self _uniqueValuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: RangeIndexReadStream
_uniqueValuesAddInto: collection comparing: aSelector key: aKey
  "Iterates over the unique values of the B-tree, traversing the reverse mappings
 to the root object.  Adds the root to the given NSC."

  | querySpec |
  querySpec := self _btreeComparisonQuerySpecClass key: aKey selector: aSelector.
  endNode ifNotNil: [ querySpec collator: endNode collator ].
  ^ self _uniqueValuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: RangeIndexReadStream
_uniqueValuesAddInto: collection comparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2
  "Iterates over the unique values of the B-tree, traversing the reverse mappings
 to the root object.  Adds the root to the given NSC."

  | querySpec |
  querySpec := self _btreeRangeComparisonQuerySpecClass
    key: aKey1
    selector: aSelector1
    and: aKey2
    selector: aSelector2.
  endNode ifNotNil: [ querySpec collator: endNode collator ].
  ^ self _uniqueValuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: RangeIndexReadStream
_uniqueValuesAddInto: collection identicalTo: aKey
  "Iterate over the unique values of the B-tree, traversing the reverse mappings
 to the root object.  For each entry whose key is identical to the given key,
 add the root to the given NSC."

  | querySpec |
  querySpec := self _btreeComparisonQuerySpecClass key: aKey selector: #'=='.
  endNode ifNotNil: [ querySpec collator: endNode collator ].
  ^ self _uniqueValuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: RangeIndexReadStream
_uniqueValuesAddInto: collection notEqualTo: aKey
  "Iterate over the unique values of the B-tree, traversing the reverse mappings
 to the root object.  For each entry whose key is not identical to the given
 key, add the root to the given NSC."

  | querySpec |
  querySpec := self _btreeComparisonQuerySpecClass key: aKey selector: #'~='.
  endNode ifNotNil: [ querySpec collator: endNode collator ].
  ^ self _uniqueValuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: RangeIndexReadStream
_uniqueValuesAddInto: collection notIdenticalTo: aKey
  "Iterate over the unique values of the B-tree, traversing the reverse mappings
 to the root object.  For each entry whose key is not identical to the given
 key, add the root to the given NSC."

  | querySpec |
  querySpec := self _btreeComparisonQuerySpecClass key: aKey selector: #'~~'.
  endNode ifNotNil: [ querySpec collator: endNode collator ].
  ^ self _uniqueValuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: RangeIndexReadStream
_uniqueValuesAddInto: collection spec: querySpec

"Iterates over the unique values of the B-tree, traversing the reverse mappings
 to the root object.  Adds the root to the given NSC."

| leaf index eSize prevVal offset pathTerm |

" get the index into the leaf node and see if it has reached the end "
index := currentStack at: 1.
index == 0
    ifTrue: [ ^ self ].

rangeIndex isIndexOnNscElements
    ifTrue: [ offset := rangeIndex size ]
    ifFalse: [ offset := rangeIndex lastPathComponentsDictionaryOffset max: 1 ].
pathTerm := rangeIndex at: offset.

leaf := currentStack at: 2.
eSize := endNode entrySize.
prevVal := #_incompletePathTraversal.
querySpec rangeIndex: rangeIndex offset: offset pathTerm: pathTerm.
" while current node is not the end node "
[ leaf ~~ endNode ] whileTrue: [
    " iterate through the values of the current node "
  prevVal := querySpec traverseValuesFrom: leaf
                                  start: index
                                  end: leaf numElements * eSize 
                                  previous: prevVal
                                  into: collection.
    " look down the stack for the next leaf node "
    self _nextLeaf.

    leaf := currentStack at: 2.
    index := 1
].
" iterate through the values of the end node "
querySpec traverseValuesFrom: leaf
                 start: index
                 end: endIndex 
                 previous: prevVal
                 into: collection. 
%

category: 'Query Support'
method: RangeIndexReadStream
approxNumElements

"Returns the number of leaf node entries represented by the receiver.  This is
 approximately the number of elements because objects may share sub-objects
 which converge to the same leaf node entry."

| result |
self atEnd ifTrue: [ ^ 0 ].

result := { 0 } .
(currentStack at: currentStack size) numElementsInStack: currentStack 
    level:  currentStack size - 1
    endNode: endNode
    endIndex: endIndex 
    into: result.

" Add one because the last entry is not counted "
^ (result at: 1) + 1
%

category: 'Copying'
method: RangeIndexReadStream
postCopy

"Cleanup of new copy."

currentStack := currentStack copy.
%

! deleted RangeIndexReadStream>>makeNscFilterSymbols: v2.1

category: 'Query Support'
method: RangeIndexReadStream
makeNscComparing: aSelector key: aKey

"Returns a new NSC that contains all the elements from the receiver."

| tmpHolder |
tmpHolder := NscBuilder for: rangeIndex nscRoot species new max: rangeIndex nscRoot size.

rangeIndex hasEnumeratedTerm 
  ifTrue: [ [ self atEnd ] whileFalse: [ tmpHolder add: (self getNextValueComparing: aSelector key: aKey) ] ]
  ifFalse: [
    " If B-tree contains mapping to root object "
    rangeIndex size == 1
        ifTrue: [ self _valuesAddInto: tmpHolder comparing: aSelector key: aKey]
        ifFalse: [ self _uniqueValuesAddInto: tmpHolder comparing: aSelector key: aKey ] ].

" make the stream be at the end "
currentStack at: 1 put: 0.

^ tmpHolder completeBag
%

category: 'Query Support'
method: RangeIndexReadStream
makeNscComparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2

"Returns a new NSC that contains all the elements from the receiver."

| tmpHolder |
tmpHolder := NscBuilder for: rangeIndex nscRoot species new max: rangeIndex nscRoot size.

rangeIndex hasEnumeratedTerm 
  ifTrue: [ [ self atEnd ] whileFalse: [ tmpHolder add: (self getNextValueComparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2) ] ]
  ifFalse: [ 
    " If B-tree contains mapping to root object "
    rangeIndex size == 1
        ifTrue: [ self _valuesAddInto: tmpHolder comparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2]
        ifFalse: [ self _uniqueValuesAddInto: tmpHolder comparing: aSelector1 key: aKey1 and: aSelector2 key:     aKey2 ] ].

" make the stream be at the end "
currentStack at: 1 put: 0.

^ tmpHolder completeBag
%

category: 'Query Support'
method: RangeIndexReadStream
getNextValueComparing: aSelector key: aKey
  " iterate through the stream, return first value that satisfies query "

  [ self _atEnd not ]
    whileTrue: [ 
      (self _peekKey perform: aSelector with: aKey)
        ifTrue: [ ^ self getNext ].
      self _btreeNext ].
  ^ #'_incompletePathTraversal'
%

category: 'Query Support'
method: RangeIndexReadStream
getNextValueComparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2
  " iterate through the stream, return first value that satisfies query "

  [ self _atEnd not ]
    whileTrue: [ 
      | key |
      key := self _peekKey.
      ((key perform: aSelector1 with: aKey1)
        and: [ key perform: aSelector2 with: aKey2 ])
        ifTrue: [ ^ self getNext ].
      self _btreeNext ].
  ^ #'_incompletePathTraversal'
%

category: 'Private'
method: RangeIndexReadStream
initializeSetIterationIndexes
  "Initialize the setIterationIndexes for a new instance."

  | iterationIndexSize | 
  rangeIndex == nil
    ifTrue: [ ^ self ].
  iterationIndexSize := rangeIndex lastPathComponentsDictionaryOffset.
  (rangeIndex isIndexOnRootNsc not and: [(rangeIndex at: rangeIndex size) isSetValuedTerm])
    ifTrue: [iterationIndexSize := rangeIndex size].
  setIterationIndexes := Array new: iterationIndexSize.
  1 to: setIterationIndexes size do: [ :i | setIterationIndexes at: i put: 0 ]
%

category: 'Private'
method: RangeIndexReadStream
noMoreSetsToIterate

"Return whether the setIterationIndexes specifies any more sets over
which to iterate."

1 to: setIterationIndexes size do: [ :i |
  (setIterationIndexes at: i) > 0
    ifTrue: [ ^ false ]
].
^ true
%

category: 'Testing'
method: RangeIndexReadStream
_atEndForStream: aBool
  "Returns true if there are no more elements to return through the logical
 iteration of the stream."

  aBool
    ifTrue: [ 
      self streamQuerySpec
        ifNotNil: [ :querySpec | 
          "bug 44079: need to skip key/value pairs that do not satisfy the querySpec"
          [ 
          | aKey i |
          i := currentStack at: 1.
          i == 0
            ifTrue: [ ^ true ].
          aKey := (currentStack at: 2) _basicAt: i + 1.
          querySpec compareKey: aKey ]
            whileFalse: [ 
              super next.
              super _atEnd
                and: [ 
                  self noMoreSetsToIterate
                    ifTrue: [ ^ true ] ] ] ] ].
  ^ super _atEnd and: [ self noMoreSetsToIterate ]
%

category: 'Conversion'
method: RangeIndexReadStream
reversed
  | reversed |
  reversed := (self reversedRangeIndexClass on: self rangeIndex)
    streamQuerySpec: self streamQuerySpec;
    yourself.
  self atEnd
    ifTrue: [ reversed currentStack at: 1 put: 0 ]
    ifFalse: [
      "unwind currentStack until the endNode is `currentNode`"
      [ endNode == (reversed currentStack at: 2) ]
        whileFalse: [ reversed _previousLeaf ].
      "update, current index, endIndex and endNode"
      reversed currentStack at: 1 put: endIndex.
      reversed endIndex: (currentStack at: 1).
      reversed endNode: (currentStack at: 2) ].
  ^ reversed
%

! ReversedRangeIndexReadStream class created in idxclasses.topaz

removeallmethods  ReversedRangeIndexReadStream
removeallclassmethods  ReversedRangeIndexReadStream

doit
ReversedRangeIndexReadStream comment: 
'ReversedRangeIndexReadStream, like its superclasses BtreeReadStream and 
 RangeIndexStream, supports the composition of query results by providing 
 access to a B-tree structure.  Its ''next'' and ''atEnd'' methods are used 
 to iterate through the B-tree. The primary difference is that the traversal
 takes place from the end of the stream to the beginning, giving one access
 to the results of a query in reverse order.'.
true
%


! ------------------- Class methods for ReversedRangeIndexReadStream
category: 'Instance Creation'
classmethod: ReversedRangeIndexReadStream
on: aRangeIndex
  "Create a stream that can access the entire contents of the B-tree whose
 root node is aBtreeNode."

  | arr1 arr2 newStream btreeNode |
  arr1 := {}.
  arr2 := {}.
  btreeNode := aRangeIndex btreeRoot.
  btreeNode _putFirstIndexOfFirstChildInto: arr2.
  btreeNode _putLastIndexOfLastChildInto: arr1.
  newStream := self new.
  newStream endIndex: (arr2 at: 1).
  newStream endNode: (arr2 at: 2).
  newStream currentStack: arr1.
  newStream rangeIndex: aRangeIndex.
  ^ newStream
%

! ------------------- Instance methods for ReversedRangeIndexReadStream
category: 'Private'
method: RangeIndexReadStream
reversedRangeIndexClass
  ^ ReversedRangeIndexReadStream
%
category: 'Accessing'
method: ReversedRangeIndexReadStream
_btreeNext
  "Returns the previous value on a stream of B-tree values.  Updates the current
 stack for a subsequent 'next'."

  ^ self _btreePrevious
%
category: 'Accessing'
method: ReversedRangeIndexReadStream
_btreePrevious
  "Returns the previous value on a stream of B-tree values.  Updates the current
 stack for a subsequent 'next'."

  | val leaf index |
  index := currentStack at: 1.	" get the index into the leaf node and see if it has reached the end "
  index == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  leaf := currentStack at: 2.	" get the leaf and the value within the leaf "
  val := leaf _basicAt: index.
  (leaf == endNode and: [ endIndex == index ])
    ifTrue: [ 
      " if this is the end of the stream, sets the top index to zero and returns "
      currentStack at: 1 put: 0.
      ^ val ].
  index == 1
    ifTrue: [ 
      " must look down the stack for the previous leaf node "
      self _previousLeaf ]
    ifFalse: [ currentStack at: 1 put: index - leaf entrySize ].
  ^ val
%
category: 'Accessing'
method: ReversedRangeIndexReadStream
_previousLeaf
  "Looks down the stack for the previous leaf node and makes it the current node."

  | nodeIndex notPositioned currentNode currentIndex previousIndex node |
  nodeIndex := 4.
  notPositioned := true.
  [ notPositioned ]
    whileTrue: [ 
      currentIndex := currentStack at: nodeIndex - 1.
      nil == (currentNode := currentStack at: nodeIndex)
        ifTrue: [ self _error: #'rtErrInvalidBtreeReadStream' ].
      " if the current index is the fist in the node "
      currentIndex == 1
        ifTrue: [ 
          " must look down the stack another level "
          nodeIndex == currentStack size
            ifTrue: [ ^ self _errorEndOfStream ].
          nodeIndex := nodeIndex + 2 ]
        ifFalse: [ 
          " current node has a previous entry "
          previousIndex := currentIndex - currentNode entrySize.
          currentStack at: nodeIndex - 1 put: previousIndex.	" put the previous entry (index and node) on the stack "
          nil == (node := currentNode _basicAt: previousIndex)
            ifTrue: [ self _error: #'rtErrInvalidBtreeReadStream' ].
          currentStack at: nodeIndex - 2 put: node.
          nodeIndex - 3
            _downTo: 3
            by: 2
            do: [ :i | 
              | childNode |
              " put the last child on any remaining levels in the stack "
              childNode := (currentStack at: i + 1) _basicAt: (currentStack at: i + 1) _lastKeyIndex - 1.
              currentStack at: i put: childNode _lastKeyIndex - 1.
              currentStack at: i - 1 put: childNode ].
          currentStack at: 1 put: node _lastKeyIndex - 1.
          notPositioned := false ] ]
%
category: 'Private'
method: ReversedRangeIndexReadStream
reversedRangeIndexClass
  ^ RangeIndexReadStream
%
category: 'Query Support'
method: ReversedRangeIndexReadStream
_uniqueValuesAddInto: collection spec: querySpec

"Iterates over the unique values of the B-tree, traversing the reverse mappings
 to the root object.  Adds the root to the given NSC."

| leaf index eSize prevVal offset pathTerm |

" get the index into the leaf node and see if it has reached the end "
index := currentStack at: 1.
index == 0
    ifTrue: [ ^ self ].

rangeIndex isIndexOnNscElements
    ifTrue: [ offset := rangeIndex size ]
    ifFalse: [ offset := rangeIndex lastPathComponentsDictionaryOffset max: 1 ].
pathTerm := rangeIndex at: offset.

leaf := currentStack at: 2.
eSize := endNode entrySize.
prevVal := #_incompletePathTraversal.
querySpec rangeIndex: rangeIndex offset: offset pathTerm: pathTerm.
" while current node is not the end node "
[ leaf ~~ endNode ] whileTrue: [
    " iterate through the values of the current node "
  prevVal := querySpec traverseValuesFrom: leaf
                                  start: 1
                                  end: index
                                  previous: prevVal
                                  into: collection.
    " look down the stack for the next leaf node "
    self _previousLeaf.

    leaf := currentStack at: 2.
    index := currentStack at: 1
].
" iterate through the values of the end node "
querySpec traverseValuesFrom: leaf
                 start: endIndex
                 end: index 
                 previous: prevVal
                 into: collection. 
%
category: 'Query Support'
method: ReversedRangeIndexReadStream
_valuesAddInto: collection spec: querySpec

"Adds each entry value into the given NSC."

| leaf index eSize |

" get the index into the leaf node and see if it has reached the end "
index := currentStack at: 1.
index == 0
    ifTrue: [ ^ self ].

leaf := currentStack at: 2.
eSize := endNode entrySize.
" while current node is not the end node "
[ endNode ~~ leaf ] whileTrue: [
    " put the values for the current node in the set "
    querySpec addValuesFrom: leaf
                     start: 1
                     end: index 
                     into: collection.
    " look down the stack for the next leaf node "
    self _previousLeaf.

    leaf := currentStack at: 2.
    index := currentStack at: 1
].

" put values of end node in the NSC "
querySpec addValuesFrom: leaf
                 start: endIndex
                 end: index 
                 into: collection.
%
category: 'Updating'
method: RangeIndexReadStream
skip: aNum
"legacy indexes do not support skip: in read stream"

	self shouldNotImplement: #'skip:'
%
