!========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! definitions for RcPipe
!
!========================================================================

!=========================================================================
! Superclass Hierarchy:
!   RcPipe, GsPipe, Collection, Object.
!
!=========================================================================

expectvalue %String

! ------------------- Class definition for RcPipe
expectvalue %String
run
| cls |
cls := Globals at: #RcPipe otherwise: nil.
^ cls == nil ifTrue:[
    GsPipe subclass: 'RcPipe'
      instVarNames: #()
      classVars: #()
      classInstVars: #()
      poolDictionaries: #()
      inDictionary: Globals
      options: #().
          'Created class RcPipe'
   ] 
   ifFalse:[ 'RcPipe class already exists' ].
%

removeallmethods RcPipe
removeallclassmethods RcPipe

! ------------------- Class methods for RcPipe
category: 'For Documentation Installation only'
classmethod: RcPipe
installDocumentation

self comment:
'An RcPipe is similar to an GsPipe except that it has been extended to allow
concurrent additions to the pipe without generating a concurrency conflict.

The RcPipe is a very efficient for situations in which concurrent additions 
occur, but are not likely for every commit.  If it is known that almost every
addition is concurrent with some other additions, the RcQueue is likely 
a more efficient implementation.

Multiple concurrent producers (sessions adding to the RcPipe) will NOT experience
concurrency conflicts because as part of every add operation, they record the add
in a redo log in the session state.  If a transation containing a RcPipe object 
detects a conflict in its commit, it breaks serialization and then replays the 
addition before competing the commit.  The replays are serialized so that if 
there are multiple sessions performing additions, they will all complete 
sucessfully in the order in which they are serialized.  Because the RcPipe must 
perform the logging and replay operations, when there are lots of concurrent updaters, 
it may be more effient to use an RcQueue.  If the probability of concurrent updates
is relatively small, it can be much more efficient because it does not use as
much space and does not require cleanup like the RcQueue does.

Values in the RcPipe are always removed in the order that they were committed.
'
%

category: 'Instance Creation'
classmethod: RcPipe
new

"Returns a new RcPipe."

^super new
%

! ------------------- Instance methods for RcPipe

category: 'Adding'
method: RcPipe
add: aValue

"Adds aValue to the RcPipe and returns aValue."

| redo logEntry | 
redo := System redoLog.
logEntry := LogEntry new.
logEntry receiver: self;
         selector: #_redoAdd:;
         argArray: { aValue }.
redo addLogEntry: logEntry.
redo addConflictObject: tail for: self.
System _addToRcReadSet: self includingAllNodes: false.
System _addToRcReadSet: (self head) includingAllNodes: false.
System _addToRcReadSet: (self tail) includingAllNodes: false.
System _addToRcReadSet: (self tail next) includingAllNodes: false.
^ super add: aValue
%

category: 'Private'
method: RcPipe
_redoAdd: anObject

"Performs the replay of adding anObject to the RcPipe and returns true."
self add: anObject.
^ true
%

category: 'Reduced Conflict Support'
method: RcPipe
_abortAndReplay: conflictObjects
 
"Abort the receiver and replay operations on the receiver from the redo log."
 
| redoLog logEntries |
redoLog := System _redoLog.
 
" if no log entries to replay, then we're done "
redoLog == nil ifTrue: [ ^ false ].
logEntries := redoLog getLogEntriesFor: self .
logEntries == nil ifTrue:[ ^ false ].

" cannot perform selective abort if receiver has a dependency tag "
self _hasDependencyList ifTrue: [ ^ false ].
 
" refresh the state of the receiver and the tail.
  these should only be done once!"
self _selectiveAbort.
self tail _selectiveAbort.  


" tell the redo log to replay any operations on the receiver "
^ redoLog _redoOperationsForEntries: logEntries 
%
