! ========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
!
! $Id$
!
! Superclass Hierarchy
!   Set, UnorderedCollection, Collection, Object
!
! ========================================================================
expectvalue %String
run
^ UnorderedCollection _newKernelSubclass: 'Set'
        instVarNames: #( 'dict' )
        classVars: #()
        classInstVars: #()
        poolDictionaries: { }
        inDictionary: Globals
        options: #( disallowGciStore ) 
        reservedOop: 801
%

set class Set
removeallmethods 
removeallclassmethods 

! ------------------- Class methods for Set

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'A Set is an UnorderedCollection in which any distinct object can occur only
 once.  Adding the same (identical) object to a Set multiple times is redundant.
 The result is the same as adding it once.

 Since a Set is an equality-based collection, different (non-identical) but
 equivalent (equal) objects are not treated as distinct from each other.  In
 IdentitySets, they are distinct.  Adding multiple equivalent objects to a Set
 yields a Set with the object that was added last.  In short, two different
 elements of a Set are neither identical nor equivalent.

 You can create subclasses of Set to restrict the kind of elements it contains.
 When creating a subclass of Set, you must specify a class as the aConstraint
 argument.  This class is called the element kind of the new subclass.  For each
 instance of the new subclass, the class of each element must be of the element
 kind.

Constraints:
	_varyingSize: Object
	_numEntries: Object
	_indexedPaths: Object
	_levels: Object
	dict: KeyValueDictionary

--- instVar dict
A KeyValueDictionary that organizes the elements and element counts for the
 Set.
' .
%

category: 'Instance Creation'
classmethod:
new

"Returns an instance of the receiver whose contents are empty."

^ (self basicNew) initialize: 0
%

category: 'Instance Creation'
classmethod:
new: initialSize

"Returns an instance of the receiver whose contents are empty."

^ (self basicNew) initialize: initialSize.
%

! ------------------- Instance methods for Set
category: 'Adding'
method:
add: newObject

"Makes newObject one of the receiver's elements and returns newObject. 
 If an equivalent element is already present in the receiver, the 
 receiver is not modified.  A set can have only one occurrence of 
 equivalent objects."

| idxRes |

"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."
newObject == nil ifTrue: [ ^newObject ].

"v3.0.1 varyingConstraints no longer enforced."

(dict includesKey: newObject) ifFalse:[
  _levels := _levels .
  _indexedPaths ~~ nil ifTrue:[
    idxRes := self _updateIndexesForAdditionOf: newObject logging: true. 
    idxRes == true
      ifFalse: [ ^ self _raiseIndexingError: idxRes ]
    ].
  dict at: newObject put: newObject .
  ].
^ newObject.
%

category: 'Adding'
method:
add: anObject withOccurrences: anInteger

  anInteger == 1 ifTrue:[
    ^ self add: anObject
  ].
 "Disallowed.  Each element of a Set must be unique."
  self shouldNotImplement: #add:withOccurrences:
%

category: 'Accessing'
method:
at: anIndex

"Disallowed."

^ self shouldNotImplement: #at:
%

category: 'Updating'
method:
at: anIndex put: anObject

"Disallowed."

^ self shouldNotImplement: #at:put:
%

category: 'Enumerating'
method:
do: aBlock

"Evaluates the one-argument block aBlock using each element of the
 receiver.  Returns the receiver."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

dict keysDo: [ :aKey | aBlock value: aKey ].
^ self.
%

category: 'Searching'
method:
includesValue: anObject

"Returns true if anObject is equal to one of the elements of the receiver. 
 Returns false otherwise."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

^ dict includesKey: anObject 

%
category: 'Searching'
method:
includes: anObject

"Returns true if anObject is equal to one of the elements of the receiver. 
 Returns false otherwise."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

^ dict includesKey: anObject.
%

category: 'Searching'
method:
includesIdentical: anObject

"Returns true if anObject is identical to one of the elements of the receiver.
 Returns false otherwise."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

anObject == nil ifTrue:[ ^ false ].
^ anObject == (dict keyAt: anObject otherwise: nil) 
%

category: 'Private'
method:
initialize: initialSize

"Initializes the receiver immediately after creation.  Returns the receiver."

_indexedPaths ~~ nil ifTrue:[
   dict keysDo:[:anObj |  | idxRes |
     idxRes := self _updateIndexesForRemovalOf: anObj.
     idxRes == true ifFalse: 
       [ ^self _raiseIndexingError: idxRes]  ] 
   ].
dict := KeyValueDictionary new: initialSize.
%

category: 'Searching'
method:
occurrencesOf: anObject

"Returns the number of the receiver's elements that are equal to anObject."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

(dict includesKey: anObject)
  ifTrue: [ ^ 1 ].

^ 0.
%

category: 'Hashing'
method:
rehash

"Rebuilds the receiver to ensure its consistency.  Returns the receiver."

"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."
_levels := _levels .

dict _rebuild.
^self
%

category: 'Removing'
method:
remove: anObject 

"Removes from the receiver one object that is equivalent to anObject and 
 returns anObject.  Generates an error if anObject has no equivalent element
 in the receiver."

^ self remove: anObject ifAbsent:[ self _errorNotFound: anObject ]
%

category: 'Removing'
method:
remove: anObject ifAbsent: anExceptionBlock

"Removes from the receiver one object that is equivalent to anObject and 
 returns anObject. If anObject has no equivalent elements in the receiver,
 evaluates anExceptionBlock and returns the result."

| removedObj idxRes |

"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."

anObject ifNil:[ ^ anExceptionBlock value ].
_levels := _levels .
_indexedPaths ~~ nil ifTrue:[
  removedObj := dict at: anObject otherwise: nil .
  removedObj ~~ nil ifTrue:[
    _levels := _levels . 
    idxRes := self _updateIndexesForRemovalOf: removedObj.
    idxRes == true ifFalse: 
      [ ^self _raiseIndexingError: idxRes] .
    dict removeKey: removedObj ifAbsent:[ self _halt:'Set remove failed' ].
    ]
  ifFalse:[ ^ anExceptionBlock value ].
  ]
ifFalse:[
  dict removeKey: anObject ifAbsent:[ ^ anExceptionBlock value].
  _levels := _levels . 
  ].  
^ anObject.
%

category: 'Removing'
method:
removeAll: aCollection

"Removes each element of aCollection from the receiver and returns the receiver.
 Generates an error if any element of aCollection is not present in the
 receiver."

"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."
_levels := _levels .

aCollection == self ifTrue:[
   self initialize: 0 .
   ]
ifFalse:[
  aCollection accompaniedBy: self do: [:me :anObject | me remove: anObject ].
  ].
^aCollection
%

category: 'Accessing'
method:
size

"Returns the number of elements contained in the receiver."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

^ dict size
%

category: 'Converting'
method:
asSet

"Returns a Set with the contents of the receiver."

^ self
%

category: 'Converting'
method:
_asIdentityBag

"Private.  Returns an IdentitySet containing all of the elements of the
 receiver."

"Used by index creation."

| result  tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

result := IdentitySet new .
dict accompaniedBy: result keysAndValuesDo:[:res :aKey :aValue| res add: aKey ].
^ result
%

category: 'Private'
method:
_deferredGciUpdateWith: valueArray

"Private."

1 to: valueArray size do:[:j |
  self add: (valueArray at: j)
  ].
%

category: 'Instance Initialization'
method:
_gciInitialize

"Private."

self initialize: 0 .
^ true
%

category: 'Copying'
method:
postCopy

"Cleanup new copy."

 super postCopy .
 dict := dict copy .
%

category: 'Removing'
method:
removeIdentical: anObject ifAbsent: anExceptionBlock

"Removes from the receiver an object that is identical to anObject.
 Returns anObject.  If several elements of the receiver are identical to 
 anObject, only one instance is removed.  If anObject has no equivalent 
 elements in the receiver, evaluates anExceptionBlock and returns the result."

"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."

(self includesIdentical: anObject) ifFalse: [^anExceptionBlock value].
_levels := _levels .
^self remove: anObject
%

category: 'Updating'
method:
objectSecurityPolicy: anObjectSecurityPolicy

"Assigns the receiver and its private objects to the given security policy."

super objectSecurityPolicy: anObjectSecurityPolicy.
dict ifNotNil:[ dict objectSecurityPolicy: anObjectSecurityPolicy ]
%

category: 'Testing'
method: 
_isLarge

"Returns true if the object is implemented as a tree of private smaller objects"

^ true 
%

category: 'Private'
method:
_deepCopyWith: copiedObjDict

| copy |
copy := copiedObjDict at: self otherwise: nil.
copy ~~ nil ifTrue: [ ^ copy ].

copy := self class new: (self size).
copiedObjDict at: self put: copy.

self _deepCopyNamedIvsWith: copiedObjDict to: copy .

dict keysDo: [ :aKey |  
   copy add: (aKey _deepCopyWith: copiedObjDict)
  ].
  
^ copy.
%
category: 'Private'
method
_keysAndValuesDo: aBlock
  "aBlock should be a 2 argument block with arguments  element, count "

  dict keysAndValuesDo:[ :each :aVal | aBlock value: each value: 1 ]
%

category: 'Set Arithmetic'
method:
+ aBagOrSet

"Union.  The result contains exactly the elements that are
 present in either the receiver or the argument aBagOrSet.
 If aBagOrSet is a kind of IdentityBag, the result is an IdentityBag. 
 If aBagOrSet is a kind of IdentitySet, the result is an IdentitySet. 
 If aBagOrSet is a kind of Bag, the result is a Bag ,
 otherwise result is an instance of the class of the receiver."

| res argIsBag |
(aBagOrSet isKindOf: IdentityBag ) ifTrue:[
  res := (aBagOrSet isKindOf: IdentitySet) ifTrue:[ IdentitySet new] ifFalse:[ IdentityBag new].
  res addAll: aBagOrSet .
  dict keysAndValuesDo:[ :each :count | res add: each ]  .
  ^ res
].
((argIsBag := aBagOrSet isKindOf: Bag) or:[ aBagOrSet isKindOf: Set ]) ifFalse:[
  aBagOrSet _validateKindOfClasses: { Bag . Set } .
].
argIsBag
  ifTrue:[  res := Bag new .
            self _keysAndValuesDo:[ :each :count | res add: each withOccurrences: count ]]
  ifFalse:[ res := self copy ].
aBagOrSet _keysAndValuesDo:[ :each :count | res add: each withOccurrences: count ].
^ res
%

category: 'Set Arithmetic'
method:
- aBagOrSet

"Difference. The result containing exactly those elements of
 the receiver that have a greater number of occurrences in the receiver than in
 the argument.
 If argument is a kind of IdentityBag, result will be an IdentitySet ,
 otherwise result is an instance of the class of the receiver."

 | res |
 (aBagOrSet isKindOf: IdentityBag ) ifTrue:[ | s |
   s := IdentitySet new .
   dict keysAndValuesDo:[ :each :aVal | s add: each ] .
   ^ s - aBagOrSet 
 ].
 ((aBagOrSet isKindOf: Bag) or:[ aBagOrSet isKindOf: Set ]) ifFalse:[
   aBagOrSet _validateKindOfClasses: { Bag . Set } .
 ].
 res := self copy .
 aBagOrSet _keysAndValuesDo:[ :each :count | | oldVal |
   oldVal := dict at: each otherwise: nil .
   oldVal ifNotNil:[ res remove: each ifAbsent: nil ].
 ] .
 ^ res
%

category: 'Set Arithmetic'
method:
* aBagOrSet

"Intersection.  The result containing only the elements that
 are present in both the receiver and the argument aBagOrSet.

 If aBagOrSet is a kind of IdentityBag, the result is an IdentityBag,
 otherwise the result is a Bag ."

 | res argIsBag |
 (aBagOrSet isKindOf: IdentityBag ) ifTrue:[ | s |
   s := IdentitySet new .
   dict keysAndValuesDo:[ :each :aVal | s add: each ]  .
   ^ s * aBagOrSet
 ].
 ((argIsBag := aBagOrSet isKindOf: Bag) or:[ aBagOrSet isKindOf: Set ]) ifFalse:[
   aBagOrSet _validateKindOfClasses: { Bag . Set } .
 ].
 res := argIsBag ifTrue:[ Bag new ] ifFalse:[ self class new ].
 aBagOrSet _keysAndValuesDo:[ :each :count | | oldVal |
   oldVal := dict at: each otherwise: nil .
   oldVal ifNotNil:[ res add: each ].
 ].
 ^ res
%
category: 'Set Arithmetic'
method: Set
_union: aBagOrSet

^ self + aBagOrSet
%
