!=========================================================================
! Copyright (C) GemTalk Systems 2015-2024.  All Rights Reserved.
!
! SeletorPathTermExample.gs
!
! Description -
!   This file contains an example of the use of a SelectorPathTerm;
!   indexing and querying on a pathterm that is a method selector rather 
!   than an instance variable of the receiver.  Selector pathTerms 
!   should return values based on instance variables of elements in the 
!   collection. 
 
!   This example illustrates the use of a custom modification tracker for 
!   automatically updating a btree based on the results of a message send.
!
!   This example is provided to illustrate how to approach the problems of 
!   automatic updates for selector-based indexes.  GemTalk Systems has not 
!   fully tested automatic updates with selector pathTerms; production use 
!   is not supported. Use of this code is at your own risk.
!
!========================================================================


! ------------------- Class definition for Soldier
expectvalue /Class
doit
Object subclass: 'Soldier'
	instVarNames: #( name rank)
	classVars: #( Ranks)
	classInstVars: #()
	poolDictionaries: #()
	inDictionary: UserGlobals
	options: #()

%

! ------------------- Remove existing behavior from Soldier
doit
Soldier removeAllMethods.
Soldier class removeAllMethods.
%

! ------------------- Class definition for SoldierModificationTracker
expectvalue /Class
doit
Object subclass: 'SoldierModificationTracker'
	instVarNames: #( trackedOffset selectorPathTerm)
	classVars: #()
	classInstVars: #()
	poolDictionaries: #()
	inDictionary: UserGlobals
	options: #()

%

! ------------------- Remove existing behavior from SoldierModificationTracker
doit
SoldierModificationTracker removeAllMethods.
SoldierModificationTracker class removeAllMethods.
%

! ------------------- Class definition for SoldierNscModificationTracker
expectvalue /Class
doit
Object subclass: 'SoldierNscModificationTracker'
	instVarNames: #( selectorModificationTracker)
	classVars: #()
	classInstVars: #()
	poolDictionaries: #()
	inDictionary: UserGlobals
	options: #()

%

! ------------------- Remove existing behavior from SoldierNscModificationTracker
doit
SoldierNscModificationTracker removeAllMethods.
SoldierNscModificationTracker class removeAllMethods.
%


! ------------------- Class methods for Soldier
category: 'Initialization'
classmethod: Soldier
initialize
| index |
Ranks := SymbolKeyValueDictionary new.
index := 1.
#( #Lieutenant #Captain #Major #Colonel #General )
	do: [:e | Ranks at: e put: index.
		index := index + 1 ].
%
category: 'Ranking'
classmethod: Soldier
rankOrderForRank: aSymbol
^ (Ranks at: aSymbol otherwise: 0)
%
! ------------------- Instance methods for Soldier
category: 'Accessing'
method: Soldier
name
	^name
%
category: 'Updating'
method: Soldier
name: newValue
	name := newValue
%
category: 'Printing'
method: Soldier
printOn: aStream
aStream nextPutAll: self rank asString.
aStream nextPut: $ .
aStream nextPutAll: self name asString.
%
category: 'Promotion'
method: Soldier
promoteTo: newRank
rank := newRank
%
category: 'Accessing'
method: Soldier
rank
	^rank
%
category: 'Updating'
method: Soldier
rank: newValue
	rank := newValue
%
category: 'Rank'
method: Soldier
rankOrder
^self class rankOrderForRank: rank.
%

doit
Soldier initialize.
true
%

! ------------------- Class methods for SoldierModificationTracker
! ------------------- Instance methods for SoldierModificationTracker
category: 'Modification Tracking'
method: SoldierModificationTracker
aboutToModifyObject: anObject atOffset: anOffset to: newKey
  | pathTerm vals |
  anOffset == self trackedOffset
    ifFalse: [ ^ self ].
  pathTerm := self selectorPathTerm.
  vals := {}.
  1 to: pathTerm size do: [ :j | 
    | indexObj |
    " for each index utilizing this path term "
    indexObj := pathTerm at: j.
    vals
      add:
        {indexObj.
        (self updateIndex: indexObj forKeysWithValue: anObject toNewKey: newKey)} ].
  ^ vals
%
category: 'Modification Tracking'
method: SoldierModificationTracker
modifiedObject: anObject userData: ar
  | newKey indexObj vals |
  newKey := anObject rankOrder.
  indexObj := ar at: 1.
  vals := ar at: 2.
  vals do: [ :val | indexObj btreeAt: newKey put: anObject ]
%
category: 'Accessing'
method: SoldierModificationTracker
selectorPathTerm
	^selectorPathTerm
%
category: 'Updating'
method: SoldierModificationTracker
selectorPathTerm: newValue
	selectorPathTerm := newValue
%
category: 'Accessing'
method: SoldierModificationTracker
trackedOffset
	^trackedOffset
%
category: 'Updating'
method: SoldierModificationTracker
trackedOffset: newValue
	trackedOffset := newValue
%
category: 'Private'
method: SoldierModificationTracker
updateIndex: indexObj forKeysWithValue: anObject toNewKey: newValue
  | aKey stream vals spec |
  " first we need to find all values that have aKey as the key "
  aKey := anObject
    perform:
      (selectorPathTerm name copyFrom: 2 to: selectorPathTerm name size) asSymbol.
  stream := indexObj asQueryEvaluator
    _findAllValuesGreaterThanKey: aKey
    andEquals: true.
  vals := {}.
  [ stream _btreeAtEnd not and: [ stream _peekKey _idxForSortEqualTo: aKey ] ]
    whileTrue: [ 
      aKey == stream _peekKey 
        ifTrue: [ 
          | peeked |
          "pick out the values at the given key that are identical to anObject"
          peeked := stream _peekValue.
          peeked == anObject
            ifTrue: [ vals add: peeked ] ].
      stream _btreeNext ].
  spec := {}.
  1 to: vals size do: [ :i | 
    "remove the entry for each old value "
    spec add: (vals at: i).
    indexObj btreeRemoveKey: aKey value: (vals at: i) ].
  ^ spec
%

! ------------------- Class methods for SoldierNscModificationTracker
! ------------------- Instance methods for SoldierNscModificationTracker
category: 'Modification Tracking'
method: SoldierNscModificationTracker
adding: newObject to: trackedObject
  "add a dependency to the soldier modification tracker, so that changes to the 
   object that affect the selector-based index can be tracked"

  newObject _setModificationTrackingTo: self selectorModificationTracker
%
category: 'Modification Tracking'
method: SoldierNscModificationTracker
removing: anObject from: trackedObject
  anObject _clearModificationTrackingTo: self selectorModificationTracker
%
category: 'Accessing'
method: SoldierNscModificationTracker
selectorModificationTracker
	^selectorModificationTracker
%
category: 'Updating'
method: SoldierNscModificationTracker
selectorModificationTracker: newValue
	selectorModificationTracker := newValue
%

level 3
run
  | nsc selectorTracker nscTracker indexList selectorPathTerm terms indexedQueryResults selectQueryResults |

"Create the collection, and define and create the index."
  nsc := IdentityBag new.
  GsIndexSpec new
    equalityIndex: 'each.#rankOrder' lastElementClass: SmallInteger;
    createIndexesOn: nsc.

"Create the modification tracking infrastructure for the index."
  indexList := nsc _indexedPaths.
  terms := {}.
  indexList
    _putAllCommonPathTermsForPathArray: '#rankOrder' asArrayOfPathTerms
    into: terms.
  selectorPathTerm := terms first.
  selectorTracker := SoldierModificationTracker new
    selectorPathTerm: selectorPathTerm;
    trackedOffset: 2;
    yourself.
  nscTracker := SoldierNscModificationTracker new
    selectorModificationTracker: selectorTracker;
    yourself.
  nsc _setModificationTrackingTo: nscTracker.

"Create Soldiers and add them to the collection"
  nsc
    add:
        (Soldier new
            name: 'Patton';
            rank: #'General';
            yourself);
    add:
        (Soldier new
            name: 'Sanders';
            rank: #'Colonel';
            yourself);
    add:
        (Soldier new
            name: 'Houlihan';
            rank: #'Major';
            yourself);
    add:
        (Soldier new
            name: 'Vimes';
            rank: #'Captain';
            yourself);
    add:
        (Soldier new
            name: 'Dan';
            rank: #'Lieutenant';
            yourself);
    add:
        (Soldier new
            name: 'Pyle';
            rank: #'Private';
            yourself);
    yourself.

"Change #1: Modify a Soldier and trigger modification tracking"
  (nsc detect: [ :each | each name = 'Dan' ] ifNone: [ nil ])
    promoteTo: #'General'.

"Change #2: Add a new Soldier"
  nsc
    add:
      (Soldier new
        name: 'Problem';
        rank: #'Major';
        yourself).

"Change #3: Remove a Soldier"
  nsc remove: (nsc detect: [ :each | each name = 'Houlihan' ] ifNone: [ nil ]).

"Verify the changes: the indexed query should return the same results as non-indexed query"
  indexedQueryResults := ((GsQuery fromString: 'each.#rankOrder > rank')
    bind: 'rank' to: (Soldier rankOrderForRank: #'Captain');
    on: nsc;
    yourself) asArray.

  selectQueryResults := nsc
    select: [ :ea | ea rankOrder > (Soldier rankOrderForRank: #'Captain') ].

  ^{indexedQueryResults. selectQueryResults asArray}
%
