#!/bin/bash

# defaults
GSLOCKDIR_DEFAULT="/opt/gemstone"
ANSWERS="/var/tmp/gsanswers.txt"
ANSWERSBACKUP="/var/tmp/gsanswers.saved.txt"
OS=`uname`

if [ "$OS" = "Darwin" ]; then   
    ECHO=/bin/echo
else
    ECHO=echo
fi


checkFileExists(){
    if [ ! -f "$1" ]; then
        $ECHO "[Error]: File $1 was not found"
        return 0
    fi
    return 1
}

verifyFileExists(){
    if [ ! -f "$1" ]; then
        $ECHO "[Error]: File $1 was not found"
        exit 1
    fi
}

verifyFileNotExists(){
    if [ -f "$1" ]; then
        $ECHO "[Error]: File $1 already exists"
        exit 1
    fi
}

verifyDirExists(){
    if [ ! -d "$1" ]; then
        $ECHO "[Error]: Directory $1 was not found"
        exit 1
    fi
}

verifyDirNotExists(){
    if [ -d "$1" ]; then
        $ECHO "[Error]: Install directory $1 already exists"
        exit 1
    fi
}

verifyNotEmpty(){
    if [ -z "$2" -o "x$2" = "x" ]; then
        $ECHO "[Error]: Variable \"$1\" must be set"
        exit 1
    fi
}

checkPort(){
    expr "$2" + 0  > $DEBUG_LOG 2>&1
    if [ $? -ne 0 ]; then
        $ECHO "[Error]: $1 must be an integer between 1000 and 65535"
        return 0
    fi
    NETLDIPORT="$2"
    if [ "$NETLDIPORT" -lt "1000" -o "$NETLDIPORT" -gt "65535" ]; then
        $ECHO "[Error]: $1 must be between 1000 and 65535"
        return 0
    fi
    return 1
}

verifyIsRoot(){
    if [ `whoami` != "root" ]; then
        $ECHO "[Error]: This script must be run as root"
        exit 1
    fi
}

verifyIsNotRoot(){
    if [ `whoami` = "root" ]; then
        $ECHO "[Error]: This script must not be run as root"
        exit 1
    fi
}

validateKeyFile(){
    $GEMSTONE/sys/stoned -k $1 >/dev/null 2>&1
    if [ $? -ne 0 ]; then
	$ECHO "[Error]: key file $1 is invalid"
	return 0
    fi    
    $ECHO "[Info]: key file $1 is valid"
    return 1
}

readResponse() {
    ARG=""
    if [ "x$2" = "x" ]; then
	$ECHO -n "$1 "
    else	
	$ECHO -n "$1 ['$2'] "
    fi
    
    read ARG
    if [ "x$ARG" = "x" ]; then
#	$ECHO "Using default value $2"
	ARG=$2
    fi
}

readYesNoResponse() {
    done=0
    ARG=""
    $ECHO -n "$1 ['$2'] "
    while [ $done -eq 0 ]; do
	read ARG
	done=1
	if [ "x$ARG" = "x" ]; then
#	    $ECHO "[Info]: Using default value $2"
	    ARG=$2
	elif [ "$ARG" = "y" -o "$ARG" = "Y" ]; then
	    ARG="y"
	elif [ "$ARG" = "n" -o "$ARG" = "N" ]; then
	    ARG="n"
	else
	    $ECHO -n "[Info]: Please enter y or n"
	    done=0
	fi
    done
}

checkUserExistsNotRoot(){
    if [ "$1" = "root" ]; then
	$ECHO "[Error]: The user may not be root"
	return 0
    fi
    
    id "$1" > $DEBUG_LOG 2>&1
    if [ $? -ne 0 ]; then
        $ECHO "[Error]: User \"$1\" does not exist"
        return 0
    fi
    return 1
}

verifyUserExistsNotRoot(){
    checkUserExistsNotRoot $1
    if [ $? -ne 1 ]; then
	exit 1
    fi
}

checkGroupExists(){
    if [ "$OS" = "Linux" ]; then
	getent group "$1" > $DEBUG_LOG 2>&1
	if [ $? -ne 0 ]; then
            $ECHO "[Error]: Group \"$1\" does not exist"
            return 0
	fi	
    else	
	touch gsfoo >/dev/null 2>&1
	chgrp $1 gsfoo
	if [ $? -ne 0 ]; then
	    $ECHO "[Error]: Group \"$1\" does not exist"
	    return 0
	fi
    fi    
    return 1
}

verifyGroupExists(){
    checkGroupExists $1
    if [ $? -ne 1 ]; then
	exit 1
    fi    
}

doChmod(){
    FLAGS=$1
    PERMS=$2
    FILE=$3
    chmod $FLAGS $PERMS $FILE > $DEBUG_LOG 2>&1
    if [ $? -ne 0 ]; then
        $ECHO "[Error]: chmod \"$FLAGS\" \"$PERMS\" \"$FILE\" failed"
        exit 1
    fi
}

doCp(){
    FLAGS=$1
    SRC=$2
    DEST=$3
    cp $FLAGS $SRC $DEST > $DEBUG_LOG 2>&1
    if [ $? -ne 0 ]; then
        $ECHO "[Error]: cp \"$FLAGS\" \"$SRC\" \"$DEST\" failed"
        exit 1
    fi
}

doChown(){
    FLAGS=$1
    ID=$2
    FILE=$3
    chown $FLAGS $ID $FILE > $DEBUG_LOG 2>&1
    if [ $? -ne 0 ]; then
        $ECHO "[Error]: chown \"$FLAGS\" \"$ID\" \"$FILE\" failed"
        exit 1
    fi
}

doChgrp(){
    FLAGS=$1
    ID=$2
    FILE=$3
    chgrp $FLAGS $ID $FILE > $DEBUG_LOG 2>&1
    if [ $? -ne 0 ]; then
        $ECHO "[Error]: chgrp \"$FLAGS\" \"$ID\" \"$FILE\" failed"
        exit 1
    fi
}

doPushd(){
    DIR=$1
    pushd "$DIR" > $DEBUG_LOG 2>&1
    if [ $? -ne 0 ]; then
        $ECHO "[Error]: pushd \"$DIR\" failed"
        exit 1
    fi
}

doPopd(){
    popd > $DEBUG_LOG 2>&1
    if [ $? -ne 0 ]; then
        $ECHO "[Error]: popd failed"
        exit 1
    fi
}

doSetcap(){
    FLAGS=$1
    FILE=$2
    setcap "$FLAGS" $FILE > $DEBUG_LOG 2>&1
    if [ $? -ne 0 ]; then
        $ECHO "[Error]: setcap \"$FLAGS\" \"$FILE\" failed"
        exit 1
    fi
}

matchOnFile(){
    FILE=$1
    STR=$2
    grep "$STR" "$FILE" >/dev/null 2>&1
    if [ $? -eq 0 ]; then
	return 1
    else
	return 0
    fi
}

regexMatchOnFile(){
    FILE=$1
    STR=$2
    egrep "$STR" "$FILE" >/dev/null 2>&1
    if [ $? -eq 0 ]; then
	return 1
    else
	return 0
    fi
}

appendFile(){
    FILE=$1
    STR=$2
    $ECHO "$STR" >> $FILE
    if [ $? -ne 0 ]; then
	$ECHO "[Error]: Cannot append file $FILE"
	exit 1
    fi
}

makeDir(){
    if [ ! -d "$1" ]; then
        mkdir -p "$1" > $DEBUG_LOG 2>&1
        if [ $? -ne 0 ]; then
            $ECHO "[Error]: Unable to create directory $1"
            exit 1
        fi
    fi
}

verifyDirEmpty(){
    DIR=$1
    if [ "$(ls -A $DIR)" ]; then
	echo "[Error]: Directory $DIR is not empty"
	exit 1
    fi
}
