"
I'm a parser for tonel files. 
I parse a class with the following format: 

Tonel spec
====

    [comment]
    type { typeDefinition }
    (
        [{ methodMetadata }]
        method [
            methodBody ] 
    )*


comment
---
""
comment string
""
is optional (but it should be there, in good design ;)

type
---
Class|Trait|Extension

typeDefinition
---
a STON file with class/trait/extension metadata

methodMetadata
---
a STON file with method metadata
is optional (but also, recommended)

method
---
method declaration as this: 

Class[ class] >> selector

methodBody 
---
the method body (we do not parse contents, that's class builder task)
"
Class {
	#name : 'RwTonelParser',
	#superclass : 'Object',
	#instVars : [
		'packageReader',
		'stream',
		'lastSelectorParsed'
	],
	#classVars : [
		'Character_lf'
	],
	#category : 'Rowan-Tonel-Core'
}

{ #category : 'initialization' }
RwTonelParser class >> initialize [
	self _addInvariantClassVar: #Character_lf value: Character lf
]

{ #category : 'accessing' }
RwTonelParser class >> lineEnding [
  "Answer the os-specific line endings.  See also #lineEndingSize if changing "

  ^ String with: Character_lf
]

{ #category : 'instance creation' }
RwTonelParser class >> on: aStream forReader: aTonelReader [
	^ self new 
		stream: aStream;
		packageReader: aTonelReader;
		yourself

]

{ #category : 'instance creation' }
RwTonelParser class >> onString: aString forReader: aTonelReader [
  ^ self on: (self readStreamClass on: aString) forReader: aTonelReader

]

{ #category : 'parsing' }
RwTonelParser class >> parseStream: aStream forReader: aTonelReader [
	^ (self on: aStream forReader: aTonelReader)
		 start

]

{ #category : 'parsing' }
RwTonelParser class >> parseString: aString forReader: aTonelReader [
	^ self parseStream: (self readStreamClass on: aString) forReader: aTonelReader

]

{ #category : 'accessing' }
RwTonelParser class >> readStreamClass [

	^ ReadStreamPortable

]

{ #category : 'accessing' }
RwTonelParser class >> writeStreamClass [

	^ WriteStreamPortable

]

{ #category : 'private' }
RwTonelParser >> cleanSelector: aString [
	"BEWARE: I'm doing some heave assumptions here: I'm removing just ONE space (in case there 
	 is one) because I expect this to be a file generated by tonel, and tonel adds one space 
	 before start with the method body to make the format more readable. 
	 But of course this is not very good :("
  aString size == 0 ifTrue:[ RwTonelParseError signal:'empty selector string' ].
	^ (aString last = Character space
		ifTrue: [ aString allButLast ]
		ifFalse: [ aString ]) 
		trimLeft

]

{ #category : 'parsing' }
RwTonelParser >> comment [
	| result ch eatNext |
	
	result := String new writeStreamPortable.

	eatNext := false.
	stream next = $" ifFalse: [ RwTonelParseError signal: 'Can''t parse comment' ].	
	[ stream atEnd not 
		and: [ 
				(ch := stream next) ~= $" 
				or: [ eatNext := (stream peek = $") ] ] ]
	whileTrue: [ 
		result nextPut: ch.
		eatNext ifTrue: [ 
			stream skip: 1.
			eatNext := false ] ].
	
	^ self 
		removeFrom: '"',result contents,'"' 
		enclosingStart: $" 
		end: $"

]

{ #category : 'private factory' }
RwTonelParser >> definitionForType: aString [
  ^ self packageReader definitionForType: aString

]

{ #category : 'parsing' }
RwTonelParser >> document [
	^ { 
	self typeDef.
	self methodDefList.
	 } 
	select: [:each | each notNil ]

]

{ #category : 'error handling' }
RwTonelParser >> error: messageText [
	^ RwTonelParseError signal: messageText
]

{ #category : 'private' }
RwTonelParser >> extractSelector: aString [
	| separators selectorStream keywords |
	
	separators := { 
		Character space. 
		Character tab. 
		Character lf. 
		Character newPage. 
		Character cr. 
		$:}.

	keywords := Array new writeStreamPortable.
	selectorStream := aString readStream.
	[ selectorStream atEnd ]
	whileFalse: [ | word ch |
		word := String new writeStreamPortable.
		[ selectorStream atEnd not and: [ (separators includes: (ch := selectorStream next)) not ] ]
		whileTrue: [ word nextPut: ch ].
		ch = $: ifTrue: [ word nextPut: ch ]. 
		word contents trimBoth ifNotEmpty: [ :v | keywords nextPut: v ] ].
	keywords := keywords contents.

	^ (keywords size <= 2 
		ifTrue: [ keywords first]
		ifFalse: [ ('' join: (keywords pairsCollect: [ :keyword :argument | keyword ])) ])
		asSymbol

]

{ #category : 'testing' }
RwTonelParser >> isEnter: aCharacter [
	^ #(13 10) includes: aCharacter asciiValue

]

{ #category : 'testing' }
RwTonelParser >> isSeparator: aCharacter [ 
	^ aCharacter isSeparator

]

{ #category : 'parsing' }
RwTonelParser >> metadata [
	| result ch count |
	
	result := String new writeStreamPortable.

	count := 0.
	stream peek = ${ ifFalse: [ RwTonelParseError signal: 'Can''t parse metadata' ].	
	[ stream atEnd not ]
	whileTrue: [ 
		ch := stream next.
		result nextPut: ch.
		ch = ${ ifTrue: [ count := count +1 ].
		ch = $} ifTrue: [ count := count -1 ].
		count = 0 ifTrue: [ ^ STON fromString: result contents ]].

	RwTonelParseError signal: 'Can''t parse metadata'

]

{ #category : 'parsing' }
RwTonelParser >> method [
	| type selector |
	
	type := self untilIncluding: '>>'.
	selector := self cleanSelector: (self untilExcluding: '[').
	type := type trimBoth substrings: ' '.
	type size = 1 ifTrue: [ type := type copyWith: nil ].
  lastSelectorParsed := selector .
	^ { 
		type.
		selector.
	}

]

{ #category : 'parsing' }
RwTonelParser >> methodBody [
	"I read a methodbody (what is inside [ ... ])
	 Since a method body can contain enclosing brackets we need to be sure we will skip them and
	 correctly read the method. For that, I have to take into account: 
		- I can mention [] in comments
		- I can mention [] in strings
		- I can use $[, $] 
		- I can have inner blocks
		- I can mention a comment of the form ""$"" or a comment of the form '$'
	 all that needs to be skipped "
	| result char prevChar comment string count startPos |
	
	result := self class writeStreamClass on: String new.

	comment := false.
	string := false.
	prevChar := nil.
	count := 0.
        startPos := stream position .
        "startBody := stream peek: 300 ." "uncomment for debugging parse problems"
	stream peek = $[ ifFalse: [ RwTonelParseError signal: 'Can''t parse method body' ].
	[ stream atEnd not ]
	whileTrue: [ 
		char := stream next.
		result nextPut: char.
		(char = $" and: [ string not and: [ prevChar ~= $$ or: [ comment ] ] ]) 
			ifTrue: [ comment := comment not ]. 
		(char = $' and: [ comment not and: [ prevChar ~= $$ or: [ string ] ] ]) 
			ifTrue: [ string := string not ]. 
		(comment or: [ string ]) ifFalse: [ 
			(char = $[ and: [  prevChar ~= $$ ]) ifTrue: [ count := count +1 ].
			(char = $] and: [ prevChar ~= $$ ]) ifTrue: [ count := count -1 ] ].
		count = 0 ifTrue: [ 
			^ self 
				removeFrom: result contents 
				enclosingStart: $[ 
				end: $]
				clean: #right ].
		prevChar := char ].

	RwTonelParseError signal: 'Can''t parse method body'

]

{ #category : 'parsing' }
RwTonelParser >> methodDef [

	| methodDef |
	self methodDef: [:isMeta :mDef |
		methodDef :=  mDef.
		"skip possible spaces at the end"
		self separator ].
	^methodDef
]

{ #category : 'parsing' }
RwTonelParser >> methodDef: aBlock [
  | ar def offset |
  ar := {
    self separator.
    self try: [ self metadata ].
    self separator.
    [ offset := stream position . self method ] value .
    self methodBody
  }.
  (def := self newMethodDefinitionFrom: ar )
    offset: offset
    inFile: stream wrappedStreamName .

  aBlock
    value: ar fourth first second notNil
    value: def
]

{ #category : 'parsing' }
RwTonelParser >> methodDefList [
	| result classStream instanceStream |
	self separator. "to arrive to the end of the file in case there are no methods"
	result := { {}. {} }.
	classStream := (result at: 1) writeStreamPortable.
	instanceStream := (result at: 2) writeStreamPortable.
	[
		[ stream atEnd ]
			whileFalse: [ 
				self methodDef: [:isMeta :mDef |
					isMeta
						ifTrue: [ classStream nextPut: mDef ]
						ifFalse: [ instanceStream nextPut: mDef ].
					"skip possible spaces at the end"
					self separator ]
			] 
  ] on: (RwTonelParseError,STONReaderError,STONWriterError) do:[:ex | 
    lastSelectorParsed ifNotNil:[ | str |
      str := ex details ifNil:[ '' ].
      ex details: str, ', last method parsed: ', lastSelectorParsed printString
    ].
    ex pass 
  ].
  ^ result
]

{ #category : 'private factory' }
RwTonelParser >> newMethodDefinitionFrom: anArray [
	| metadata className meta selector source categ |
	metadata := anArray second ifNil: [ Dictionary new ].
	className := anArray fourth first first.	"avoid asSymbol sent to className before error checks."
	[ Metaclass3 _validateNewClassName: className asSymbol ]
		on: Error
		do: [ :ex | self error: 'Invalid class name ' , className printString ].
	meta := anArray fourth first second notNil.
	selector := self extractSelector: anArray fourth second trimBoth.
	source := String
		streamContents: [ :s | 
			s << anArray fourth second.
			anArray fifth ifNotEmpty: [ :src | s << src ] ].

	categ := metadata
		at: #'category'
		ifAbsent: [ 
			"to avoid error, resume with default category string"
			(RwTonelParseRequireMethodCategoryNotification
				className: className
				isMeta: meta
				selector: selector) signal ].

	^ self packageReader
		newMethodDefinitionForClassNamed: className
		classIsMeta: meta
		selector: selector
		category: categ
		source: source
]

{ #category : 'private factory' }
RwTonelParser >> newTypeDefinitionFrom: anArray [
	^ self packageReader newTypeDefinitionFrom: anArray

]

{ #category : 'accessing' }
RwTonelParser >> packageReader [
	^ packageReader

]

{ #category : 'accessing' }
RwTonelParser >> packageReader: aPackageReader [ 
	packageReader := aPackageReader

]

{ #category : 'private' }
RwTonelParser >> removeFrom: aString enclosingStart: startChar end: endChar [
	^ self 
		removeFrom: aString 
		enclosingStart: startChar 
		end: endChar
		clean: #both

]

{ #category : 'private' }
RwTonelParser >> removeFrom: aString enclosingStart: startChar end: endChar clean: cleanSymbol [
  "cleanSymbol can be #left, #rigth and #both"

  | result stop ch start end |
  result := self class readStreamClass on: aString trimBoth.
  result peek = startChar
    ifFalse: [ RwTonelParseError signal: 'I cannot remove enclosing start' ].
  result skip: 1.
  (#(#'both' #'left') includes: cleanSymbol)
    ifTrue: [ 
      stop := self class lineEnding size.
      [ stop > 0 and: [ self isSeparator: (ch := result peek) ] ]
        whileTrue: [ 
          (self isEnter: ch)
            ifTrue: [ stop := stop - 1 ].
          result skip: 1 ] ].
  start := result position.
  result setToEnd.
  result skip: -1.
  result peek = endChar
    ifFalse: [ RwTonelParseError signal: 'I cannot remove enclosing end' ].
  result skip: -1.
  (#(#'both' #'right') includes: cleanSymbol)
    ifTrue: [ 
      stop := self class lineEnding size.
      [ stop > 0 and: [ self isSeparator: (ch := result peek) ] ]
        whileTrue: [ 
          (self isEnter: ch)
            ifTrue: [ stop := stop - 1 ].
          result skip: -1 ] ].
  end := result position.
  ^ result contents copyFrom: start + 1 to: end + 1
]

{ #category : 'parsing' }
RwTonelParser >> separator [
	[ stream atEnd not and: [ self isSeparator: stream peek ] ]
	whileTrue: [ stream next ].
	^ nil

]

{ #category : 'parsing' }
RwTonelParser >> shebang [
	"look for a '#!' in first two character position and skip to next line if present"

	(stream peekFor: $#) ifFalse: [ ^ nil ].	
	(stream peekFor: $!) ifFalse: [ ^ nil ].
	^ stream  upTo: Character lf.
]

{ #category : 'accessing' }
RwTonelParser >> start [
	^ self document

]

{ #category : 'accessing' }
RwTonelParser >> stream: aStream [ 
	stream := aStream

]

{ #category : 'private parsing' }
RwTonelParser >> try: aBlock [
	^ self 
		try: aBlock 
		onSuccess: [ :parsedValue | parsedValue ] 
		onFailure: [ nil ]

]

{ #category : 'private parsing' }
RwTonelParser >> try: aBlock onSuccess: successBlock [
	^ self 
		try: aBlock 
		onSuccess: successBlock 
		onFailure: [ nil ]

]

{ #category : 'private parsing' }
RwTonelParser >> try: aBlock onSuccess: successBlock onFailure: failureBlock [
	| pos |
	
	pos := stream position.
	[ ^ successBlock value: aBlock value ]
	on: RwTonelParseError 
	do: [ :e | 
		stream position: pos.
		^ failureBlock value ]. 
	

]

{ #category : 'parsing' }
RwTonelParser >> type [
	self try: [ self word: 'Class' ] onSuccess: [ :word | ^ word  ].
	self try: [ self word: 'Trait' ] onSuccess: [ :word | ^ word  ].
	self try: [ self word: 'Extension' ] onSuccess: [ :word | ^ word  ].
	
	"at end"
	RwTonelParseError signal: 'Can''t parse type.'	

]

{ #category : 'parsing' }
RwTonelParser >> typeDef [
	| shebang |
	shebang := self shebang. "ignore shebang on first line of file if present"
	^ self newTypeDefinitionFrom: { 
		self separator.
		self try: [ self comment ]. 
		self separator. 
		self type. 
		self separator. 
		self try: [ 
			| typeMetadata normalizedMetadata |
			typeMetadata := self metadata.
			normalizedMetadata := Dictionary new.
			typeMetadata keysAndValuesDo: [:key :value |
				normalizedMetadata at: key asLowercase asSymbol put: value ].
			normalizedMetadata at: #shebang put: shebang.
			normalizedMetadata ] 
	}
]

{ #category : 'private parsing' }
RwTonelParser >> untilExcluding: aCollection [
	| result |
	result := stream upToAll: aCollection.
	stream position: stream position - aCollection size.
	^ result

]

{ #category : 'private parsing' }
RwTonelParser >> untilIncluding: aCollection [
	^ stream upToAll: aCollection

]

{ #category : 'private parsing' }
RwTonelParser >> word: aString [
	| result |
	result := stream next: aString size.
	result = aString
		ifFalse: [ RwTonelParseError signal: 'Can''t parse ', aString ].
	^ result

]
