Extension { #name : 'AbstractFraction' }

{ #category : 'Private' }
AbstractFraction class >> _coerce: aNumber [
	" Answer a fraction equal to aNumber (as long as aNumber is finite).
	Private, used for numeric comparisons."

	| f |
	f := aNumber asFraction.
	f _isInteger
		ifTrue: [ ^ AbstractFraction _noreduce_numerator: f denominator: 1 ].
	^ f
]

{ #category : 'Private' }
AbstractFraction class >> _newFraction: n denominator: d [

  "Returns an instance of Fraction , never reduced to a SmallFraction."

  ^ Fraction _basicNew _numerator: n denom: d

]

{ #category : 'Private' }
AbstractFraction class >> _newSmallFraction: n denom: d reduce: reduceBoolean [

"Returns nil or a reduced instance of SmallFraction ,
 or a Fraction containing reduced SmallInteger values for n and d.
 A SmallFraction is returned if, after moving overall sign to n
 and reducing n and d, n and d satisfy
                0 <  d <= 134217726
   and -536870912 <= n <= 536870911

 Signals an Error if d == 0 ."

<primitive: 996>
reduceBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_newSmallFraction:denom: args: { n . d }

]

{ #category : 'Private' }
AbstractFraction class >> _noreduce_numerator: n denominator: d [
  (self _newSmallFraction: n denom: d reduce: false) ifNotNil:[:res |
     ^ res
  ].
  ^ Fraction _basicNew _numerator: n denom: d

]

{ #category : 'Private' }
AbstractFraction class >> _reduce: numer denom: denom [
  | n d gcd |
  n := numer truncated .
  n == 0 ifTrue:[ ^ 0 ].
  d := denom truncated .
  (d == 0) ifTrue: [ ^ n _errorDivideByZero ].
  d < 0 ifTrue:[
    d := 0 - d .
    n := 0 - n .
  ].
  gcd := n gcd: d .
  n := n // gcd.
  d := d // gcd.
  (d == 1) ifTrue:[ ^ n ].
  (self _newSmallFraction: n denom: d reduce: false) ifNotNil:[:res |
     ^ res
  ].
  ^ Fraction _basicNew _numerator: n denom: d

]

{ #category : 'Instance Creation' }
AbstractFraction class >> fromStream: aStream [

"Returns a Fraction from the stream.  The stream must contain two Integers
 separated by a slash.  (There may be blanks around the slash.)  Generates an
 error if the stream contains anything else, or if an attempt is made to read
 beyond the end of the stream."

| n d |
self _checkReadStream: aStream forClass: CharacterCollection.
aStream peek unicodeIsWhitespace ifTrue:[
  [ aStream next unicodeIsWhitespace ] whileTrue.
  aStream skip: -1.
].
n := Integer _fromStream: aStream.
aStream peek unicodeIsWhitespace ifTrue:[
  [ aStream next unicodeIsWhitespace ] whileTrue.
  aStream skip: -1.
].
(aStream next == $/ ) ifFalse:[ ^ self _errIncorrectFormat: aStream ].
aStream peek unicodeIsWhitespace ifTrue:[
  [ aStream next unicodeIsWhitespace ] whileTrue.
  aStream skip: -1.
].
d := Integer _fromStream: aStream.
^ self numerator: n denominator: d

]

{ #category : 'Instance Creation' }
AbstractFraction class >> numerator: n denominator: d [
"Returns an instance of Fraction with numerator numInt and denominator
 denomInt.  If that Fraction can be reduced, this method returns the
 corresponding Integer instead.  The result is made invariant.

 If either argument (numerator or denominator) is not an Integer, that
 argument is truncated to the corresponding Integer."

 (self _newSmallFraction: n denom: d reduce: true) ifNotNil:[:res |
    ^ res
 ].
 ^ self _reduce: n denom: d

]

{ #category : 'Converting' }
AbstractFraction >> _coerce: aNumber [

"Reimplemented from Number."
| f |
f := aNumber asFraction .
f _isInteger ifTrue:[
  ^ AbstractFraction _noreduce_numerator: f denominator: 1
].
^ f

]

{ #category : 'Converting' }
AbstractFraction >> _generality [

"Reimplemented from Number."

^70

]

{ #category : 'Accessing' }
AbstractFraction >> _getKind [
	| num denom |
	(num := self numerator) ifNil: [ ^ 5	"nan" ].
	(denom := self denominator) ifNil: [ ^ 5 ].
	denom == 0
		ifTrue: [ 
			num == 0
				ifTrue: [ ^ 5	"NaN" ]
				ifFalse: [ ^ 3	"infinity" ] ].
	num == 0
		ifTrue: [ ^ 4	"zero" ].
	^ 1	"normal"
]

{ #category : 'Indexing Support' }
AbstractFraction >> _isNaN [

"Returns whether the receiver is quiet NaN or signaling NaN.
 This method is only to be used by the indexing subsystem."

^ (self numerator == nil) or: [self denominator == nil]

]

{ #category : 'Private' }
AbstractFraction >> _reciprocal [

"Returns 1 divided by the receiver.  Generates an error if the receiver is 0.
 Result is not reduced."
| numer denom |
((numer := self numerator) == 0 or:[ numer == nil]) ifTrue: [ ^ self _errorDivideByZero].
denom := self denominator .
numer < 0 ifTrue:[ denom := 0 - denom . numer := 0 - numer  ].

^  AbstractFraction _noreduce_numerator: denom denominator: numer

]

{ #category : 'Arithmetic' }
AbstractFraction >> - aFraction [

"Returns the difference between the receiver and aFraction."
(aFraction isKindOf: AbstractFraction) ifTrue: [
  ^ self + (Fraction _noreduce_numerator:  0 - aFraction numerator
		 denominator: aFraction denominator)
] ifFalse: [
  ^ self _retry: #- coercing: aFraction
]

]

{ #category : 'Arithmetic' }
AbstractFraction >> * aNumber [

"Returns the result of multiplying the receiver by aNumber."

(aNumber isKindOf: AbstractFraction) ifTrue: [
  ^ AbstractFraction numerator: (self numerator * aNumber numerator)
           denominator: (self denominator * aNumber denominator)
] ifFalse: [
  (aNumber _isInteger and:[ aNumber = self denominator ]) ifTrue:[
     ^ self numerator
  ].
  ^ self _retry: #* coercing: aNumber
]

]

{ #category : 'Arithmetic' }
AbstractFraction >> / aNumber [

"Returns the result of dividing the receiver by aFraction."

aNumber _isInteger ifTrue:[
  aNumber = 0 ifTrue: [ ^ self _errorDivideByZero ].
  ^ AbstractFraction numerator: self numerator denominator: (self denominator * aNumber)
].
(aNumber isKindOf: AbstractFraction) ifTrue:[
  ^ self * aNumber _reciprocal
].
^ self _retry: #/ coercing: aNumber

]

{ #category : 'Arithmetic' }
AbstractFraction >> + aFraction [

"Returns the sum of the receiver and aFraction."

(aFraction isKindOf: AbstractFraction) ifTrue:[
  | myDenom otherDenom commonDenominator newNumerator
    gcd myDemon_div_gcd |
  (myDenom := self denominator) = (otherDenom := aFraction denominator) ifTrue: [
    | myNumer otherNumer |
    (myNumer := self numerator) = (otherNumer := aFraction numerator) ifTrue:[
      myDenom _bottomBit == 0 ifTrue:[
        ^ AbstractFraction numerator: myNumer denominator: (myDenom bitShift: -1)
      ]
    ].
    ^ AbstractFraction numerator: (myNumer + otherNumer ) denominator: myDenom
  ].
  gcd := myDenom gcd: otherDenom .
  myDemon_div_gcd := myDenom // gcd .
  commonDenominator := myDemon_div_gcd * otherDenom .
  newNumerator := (self numerator * (otherDenom // gcd ))
                 + (aFraction numerator *  myDemon_div_gcd) .
  ^ AbstractFraction numerator: newNumerator denominator: commonDenominator
] ifFalse: [
  ^self _retry: #+ coercing: aFraction
]

]

{ #category : 'Comparing' }
AbstractFraction >> < aNumber [
	"Returns true if the receiver is less than aNumber; returns false otherwise."

	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].

	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'<' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].	"NaN"
	(aNumber isKindOf: AbstractFraction)
		ifTrue: [ 
			| otherNum num |
			num := self numerator.
			otherNum := aNumber numerator.
			otherNum == 0
				ifTrue: [ ^ num < 0 ].
			^ num * aNumber denominator < (self denominator * otherNum) ].
	ak == 3
		ifTrue: [ ^ aNumber sign == 1 ].	"Inf"
	^ self < (self _coerce: aNumber)
]

{ #category : 'Comparing' }
AbstractFraction >> <= aNumber [
	"Returns true if the receiver is less than or equal to aNumber; returns false
 otherwise."

	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].

	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'<=' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].	"NaN"
	(aNumber isKindOf: AbstractFraction)
		ifTrue: [ 
			| otherNum num |
			num := self numerator.
			otherNum := aNumber numerator.
			otherNum == 0
				ifTrue: [ ^ num <= 0 ].
			^ num * aNumber denominator <= (self denominator * otherNum) ].
	ak == 3
		ifTrue: [ ^ aNumber sign == 1 ].	"Inf"
	^ self <= (self _coerce: aNumber)
]

{ #category : 'Comparing' }
AbstractFraction >> = aNumber [
	"Returns true if the receiver is equal to aNumber; returns false otherwise."

	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ false ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'=' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].
	(aNumber isKindOf: AbstractFraction)
		ifTrue: [ 
			| otherNum num |
			otherNum := aNumber numerator.
			num := self numerator.
			otherNum == 0
				ifTrue: [ ^ num == 0 ].
			^ otherNum = num and: [ aNumber denominator = self denominator ] ].
	ak == 3
		ifTrue: [ ^ false ].	
	"If we get here, self and arg are both finite."
	^ self = (self _coerce: aNumber)
]

{ #category : 'Comparing' }
AbstractFraction >> > aNumber [
	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].

	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'>' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].	"NaN"
	(aNumber isKindOf: AbstractFraction)
		ifTrue: [ 
			| otherNum num |
			num := self numerator.
			otherNum := aNumber numerator.
			otherNum == 0
				ifTrue: [ ^ num > 0 ].
			^ num * aNumber denominator > (self denominator * otherNum) ].
	ak == 3
		ifTrue: [ ^ aNumber sign == -1 ].	"Inf"
	^ self > (self _coerce: aNumber)
]

{ #category : 'Comparing' }
AbstractFraction >> >= aNumber [
	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].

	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'>=' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].	"NaN"
	(aNumber isKindOf: AbstractFraction)
		ifTrue: [ 
			| otherNum num |
			num := self numerator.
			otherNum := aNumber numerator.
			otherNum == 0
				ifTrue: [ ^ num >= 0 ].
			^ num * aNumber denominator >= (self denominator * otherNum) ].
	ak == 3
		ifTrue: [ ^ aNumber sign == -1 ].	"Inf"
	^ self >= (self _coerce: aNumber)
]

{ #category : 'Converting' }
AbstractFraction >> asDecimalFloat [

"Returns an instance of DecimalFloat that has the value of the receiver."
| num denom |
(num := self numerator) ifNil:[ ^ PlusSignalingNaN ] .
(denom := self denominator) ifNil:[ ^ PlusSignalingNaN ] .
^ num asDecimalFloat / denom asDecimalFloat

]

{ #category : 'Converting' }
AbstractFraction >> asFixedPoint: scale [

^ FixedPoint numerator: self numerator denominator: self denominator scale: scale

]

{ #category : 'Converting' }
AbstractFraction >> asFloat [
	"Answer a Float that closely approximates the value of the receiver.
	This implementation will answer the closest floating point number to the receiver.
	In case of a tie, it will use the IEEE 754 round to nearest even mode.
	In case of overflow, it will answer +/- Float infinity.
	If numerator or denominator is nil, anwers NaN."

	| num den n d numeratorSize denominatorSize scalePower scaledValue scaledValueBits significantBits ulp halfULP significanceMask subSignificant scaledFloat |
	(num := self numerator) ifNil: [ ^ PlusSignalingNaN ].
	(den := self denominator) ifNil: [ ^ PlusSignalingNaN ].

	n := num abs.
	d := den.	"denominator is never negative"
	numeratorSize := n highBit.
	denominatorSize := d highBit.	

	"If both numerator and denominator can be represented exactly as floats,
	we can use float division and let it do the correct rounding."
	(numeratorSize <= 53 and: [ denominatorSize <=  53 ])
		ifTrue: [ ^ num asFloat / den asFloat ].	

	"Scale the fraction by a power of two scalePower so as to obtain an integer 
	value with 54 or 55 bits, depending on the values."
	scalePower := numeratorSize - denominatorSize - 54.
	scalePower >= 0
		ifTrue: [ d := d bitShift: scalePower ]
		ifFalse: [ n := n bitShift: scalePower negated ].
	scaledValue := n quo: d.
	scaledValueBits := scaledValue highBit.

	"How many high-order bits of scaledValue will end up in the significand?
	For tiny fractions that underflow to zero, this will be negative."
	significantBits := (scalePower + 1074 + scaledValueBits) min: 53.

	"The 54 or 55-bit integer scaledValue will be rounded to become the 
	significand of the float. Normal floats have a 53-bit significand.
	Subnormal floats have a significand with 0 to 52 significant bits.
	The 0-bit case includes very small fractions that round
	to 0.0 or to Float fmin. A negative number of significantBits will round to 0.0.

	Rounding:
	One or more low-order bits of the scaledValue will not be in the significand of the 
	float, and so are sub-significant. For normal floats, with 53-bit significands,
	there will be one or two sub-significant bits. Subnormal non-zero floats will
	have up to 55 sub-significant bits, and tiny fractions that round to 0.0 may 
	have more than that.
	The lowest-order bit of the significand is a ULP, a unit of least precision.
	If the sub-significant bits more than half a ULP, we round up, if less we round
	down. If equal, we consult the remainder to see whether there are any low-
	order one bits that were rounded away in computation of the scaledValue.
	If there are, we round up, but if there are none, we round either up or down,
	whichever makes the significand even (ULP bit zero).

	We increment the significant bits if rounding up, then clear any sub-significant
	bits to make the conversion to an integral float precise.
	We then scale the integral float to the correct magnitude by
	multiplying it by a power of two. For normal floats, the scale correction
	affects only the exponent of the float; the significand remains the same.
	If the exponent overflows the float becomes an infinity.
	If the exponent underflows the float becomes subnormal."

	ulp := 1 bitShift: scaledValueBits - significantBits.
	halfULP := ulp bitShift: -1.
	significanceMask := ulp - 1. 
	subSignificant := scaledValue bitAnd: significanceMask. "The low-order bits that will not end up in the significand."

	"Round up if needed"
	subSignificant >= halfULP 
		ifTrue: [(subSignificant > halfULP or: [n > (scaledValue * d)])
					ifTrue:  [ scaledValue := scaledValue + ulp ]
					ifFalse: ["Round half to even"
								scaledValue := scaledValue + (scaledValue bitAnd: ulp)]].

	"Clear any subSignificant bits so we only have bits that will go in the significand.
	Otherwise converting scaledValue to a float would round, in some cases incorrectly."
	scaledValue := scaledValue - subSignificant.

	scaledFloat := (self positive
		ifTrue: [ scaledValue asFloat ]
		ifFalse: [ scaledValue asFloat negated ]).
	
	"Scale correction"
	"For very negative exponents, must scale in two steps,
	since (2.0 raisedToInteger: -1075) rounds to 0.0, which would cause premature underflow to zero"
	^ scalePower < -1074 
			ifTrue: [scaledFloat * (2.0 raisedToInteger: scalePower + 1074) * (2.0 raisedToInteger: -1074)]
			ifFalse: [scaledFloat * (2.0 raisedToInteger: scalePower)]
]

{ #category : 'Converting' }
AbstractFraction >> asFraction [

"Returns the receiver."

^self

]

{ #category : 'Converting' }
AbstractFraction >> asScaledDecimal: scale [

"Returns a ScaledDecimal representation of the receiver."

^ ScaledDecimal numerator: self numerator denominator: self denominator scale: scale.

]

{ #category : 'Formatting' }
AbstractFraction >> asString [

"Returns a String of the form 'numerator/denominator'."

| result |
result := self numerator asString .
result add: $/ ; addAll: self denominator asString .
^ result

]

{ #category : 'Testing' }
AbstractFraction >> even [

"Returns true if the receiver is an even integer, false otherwise."

 self denominator = 1 ifFalse: [ ^ false ].
 ^ self numerator even


]

{ #category : 'Testing' }
AbstractFraction >> isZero [

"Returns true if the receiver is zero."

^ self numerator = 0 .

]

{ #category : 'Arithmetic' }
AbstractFraction >> negated [

"Returns a Number that is the negation of the receiver."

^ AbstractFraction _noreduce_numerator:  0 - self numerator
			denominator: self denominator

]

{ #category : 'Testing' }
AbstractFraction >> odd [

"Returns true if the receiver is an odd integer, false otherwise."

 self denominator = 1 ifFalse: [ ^ false ].
 ^ self numerator odd

]

{ #category : 'Arithmetic' }
AbstractFraction >> reciprocal [

"Returns 1 divided by the receiver.  Generates an error if the receiver is 0.
 Result is reduced."
| numer denom |
((numer := self numerator) == 0 or:[ numer == nil]) ifTrue: [ ^ self _errorDivideByZero].
denom := self denominator .
(numer == 1) ifTrue:[ ^ denom ].
numer < 0 ifTrue:[
  (numer == -1) ifTrue:[ ^ 0 - denom ].
  denom := 0 - denom . numer := 0 - numer .
].
^  AbstractFraction numerator: denom denominator: numer

]

{ #category : 'Truncation and Rounding' }
AbstractFraction >> roundAndCoerceTo: aNumber [
	"Returns the multiple of aNumber that is nearest in value to the receiver.
 If aNumber is a kind of Integer, ScaledDecimal, or AbstractFraction,
 the result will be an instance of the class of aNumber . "

	| r |
	aNumber = 0
		ifTrue: [ ^ 0 ].
	r := (self / aNumber) rounded * aNumber.
	aNumber _isFloat
		ifTrue: [ ^ r ].
	aNumber _isScaledDecimal
		ifTrue: [ ^ r ].
	^ r asFraction
]

{ #category : 'Truncation and Rounding' }
AbstractFraction >> roundedHalfToEven [

"Returns the integer nearest in value to the receiver. If the receiver is
exactly halfway between two integers, return the even one."

| remainder |
remainder := (self numerator abs \\ self denominator).
^ remainder * 2 = self denominator
	ifFalse: [self rounded]
	ifTrue: [| truncated | truncated := self truncated.
			truncated even
				ifTrue: [truncated]
				ifFalse: [truncated + self sign]]

]

{ #category : 'Truncation and Rounding' }
AbstractFraction >> roundTo: aNumber [
	"Returns the multiple of aNumber that is nearest in value to the receiver."

	| r |
	aNumber = 0
		ifTrue: [ ^ 0 ].
	r := (self / aNumber) rounded * aNumber.
	aNumber _isFloat
		ifTrue: [ ^ r ].
	^ r asFraction
]

{ #category : 'Accessing' }
AbstractFraction >> sign [
"Returns 1 if the receiver is greater than zero, -1 if the receiver is less
 than zero, and zero if the receiver is zero."

  ^ self numerator sign

]

{ #category : 'Truncation and Rounding' }
AbstractFraction >> truncateAndCoerceTo: aNumber [
	"Returns the multiple of aNumber that is closest to the receiver, on
 the same side of the receiver as zero is located.  In particular,
 returns the receiver if the receiver is a multiple of aNumber.

 If aNumber is a kind of Integer, ScaledDecimal, or AbstractFraction,
 the result will be an instance of the class of aNumber ."

	| t |
	aNumber = 0
		ifTrue: [ ^ 0 ].
	t := (self quo: aNumber) * aNumber.
	aNumber _isFloat
		ifTrue: [ ^ t ].
	aNumber _isScaledDecimal
		ifTrue: [ ^ t ].
	^ t asFraction
]

{ #category : 'Truncation and Rounding' }
AbstractFraction >> truncated [

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located."

^ self numerator quo: self denominator

]

{ #category : 'Truncation and Rounding' }
AbstractFraction >> truncateTo: aNumber [
	"Returns the multiple of aNumber that is closest to the receiver, on
 the same side of the receiver as zero is located.  In particular,
 returns the receiver if the receiver is a multiple of aNumber."

	| t |
	aNumber = 0
		ifTrue: [ ^ 0 ].
	t := (self quo: aNumber) * aNumber.
	aNumber _isFloat
		ifTrue: [ ^ t ].
	^ t asFraction
]
