Extension { #name : 'Behavior' }

{ #category : 'Instance Creation' }
Behavior class >> new [

"Disallowed.  To create a new Class or metaclass, use
 Class | subclass:instVarNames:... instead."

self shouldNotImplement: #new .
^ nil

]

{ #category : 'Instance Creation' }
Behavior class >> new: anInteger [

"Disallowed.  To create a new Class or metaclass, use
 Class | subclass:instVarNames:... instead."

self shouldNotImplement: #new: .
^ nil

]

{ #category : 'Accessing Categories' }
Behavior >> __categorys [
  ^ categorys

]

{ #category : 'Modifying Classes' }
Behavior >> _addInvariantClassVar: aSymbol value: aVal [

"Adds the class variable with name aSymbol to the receiver's
 class variables dictionary, and make the resulting SymbolAssocation
 invariant."
| assoc dict |
self _validatePrivilege ifTrue:[
  (dict := classVars) ifNil:[
    dict := self _createClassVarsDict .
  ].
  assoc := dict associationAt: aSymbol otherwise: nil .
  assoc ifNil:[
    assoc := SymbolAssociation new key: aSymbol value: aVal .
    assoc objectSecurityPolicy: self objectSecurityPolicy .
    dict addAssociation: assoc .
    assoc immediateInvariant
  ] ifNotNil:[
    assoc _value = aVal ifFalse:[
      assoc _value: aVal .
      assoc immediateInvariant
    ]
  ].
]

]

{ #category : 'Private' }
Behavior >> _allStrippedMethodSelectors [

"Returns an Array of two Arrays, signifying that all methods of the receiver
 are to have their source stripped."

  ^ { self selectors . self class selectors }.

]

{ #category : 'Jade browser support' }
Behavior >> _allSuperList [

 ^ self _allSuperList: 0

]

{ #category : 'Accessing the Class Hierarchy' }
Behavior >> _allSuperList: envId [

"Returns an Array of the superclasses of the receiver, beginning
 with the most remote superclass, and excluding the receiver itself."

  | result currClass |
  result:= { } .
  currClass := self .
  envId ~~ 0 ifTrue:[
    [ true] whileTrue:[
      currClass:= currClass superclassForEnv: envId .
      currClass == nil ifTrue: [^ result].
      result insertObject: currClass at: 1.
    ].
  ] ifFalse:[
    [ true] whileTrue:[
      currClass:= currClass superClass.
      currClass == nil ifTrue: [^ result].
      result insertObject: currClass at: 1.
    ].
  ]
]

{ #category : 'Private' }
Behavior >> _allSuperList: includeRubyVirtual env: envId [
  ^ self _allSuperList: envId 
]

{ #category : 'Enumerating' }
Behavior >> _baseCategorys: envId [

  "Returns nil or the category dictionary for specified environment"
  | cats |
  (cats := categorys) _isArray ifTrue:[
    cats := cats atOrNil:  envId + 1 .
  ] ifFalse:[
    envId ~~ 0 ifTrue:[ cats := nil ].
  ].
  ^ cats

]

{ #category : 'Accessing Categories' }
Behavior >> _baseCategorysForStore: envId [
  | dict carr ofs |
  (dict := self _baseCategorys: envId ) ifNil:[
     dict := GsMethodDictionary new  .
     dict objectSecurityPolicy: self objectSecurityPolicy .
     ofs := envId + 1 .
     (carr := categorys) _isArray ifTrue:[
       (carr size < ofs) ifTrue:[ carr size: ofs ].
        carr at: ofs put: dict .
     ] ifFalse:[
       envId > 0 ifTrue:[
         carr := Array new: ofs .
         carr at: 1 put: categorys .
         carr at: ofs put: dict .
         categorys := carr .
       ] ifFalse:[
         categorys := dict
       ].
     ].
  ].
  ^ dict

]

{ #category : 'Instance Creation' }
Behavior >> _basicNew [

"Returns an instance of the receiver, with no indexed
 variables.  Do not override this method; contrast with Behavior | new."

<primitive: 50>

(self _isInstanceDisallowed)
ifTrue:[ self _error: #objErrCantCreateInstance args:  #()  .
         self _uncontinuableError
       ].
self _primitiveFailed: #_basicNew .
self _uncontinuableError

]

{ #category : 'Instance Creation' }
Behavior >> _basicNew: anInteger [

"Returns with an instance of the receiver, with the given
 number of fields.  Generates an error if the Behavior is not indexable or if
 anInteger is bad.  Do not override this method; contrast with Behavior | new:."

<primitive: 52>
| isz |
(self _isInstanceDisallowed) ifTrue: [
   self _error: #objErrCantCreateInstance args:  #()  .
   self _primitiveFailed: #_basicNew: args: { anInteger }.
   self _uncontinuableError
].
(self isIndexable) ifFalse:[self _errorNotIndexable .  ^ self _basicNew ].
anInteger _isSmallInteger ifFalse:[
  anInteger _validateClass: SmallInteger . ^ self _basicNew
].
anInteger < 0 ifTrue:[
  anInteger _error: #rtErrArgNotPositive. ^ self _basicNew
].
(anInteger + (isz := self instSize)) _isSmallInteger ifFalse: [
  anInteger _error: #errArgTooLarge args:{ SmallInteger maximumValue - isz} .
  ^ self _basicNew
].
self _primitiveFailed: #_basicNew: args: { anInteger }.
self _uncontinuableError

]

{ #category : 'Browser Methods' }
Behavior >> _categoriesReport [

^ self _categoriesReportEnv: 0

]

{ #category : 'Browser Methods' }
Behavior >> _categoriesReportEnv: envId [

"Returns an Array containing key-value pairs from the receiver's categories
 for specified environment.
 The key in each key-value pair is the name of a category; the value in each
 key-value pair is a sorted Array of selectors.

 Used by the Topaz 'list categories' command."

| result k sz sortedCats |
sortedCats := SortedCollection new:[ :a :b | a key <= b key ].
self env: envId unifiedCategoriesDo:[ :categName :selectors |
  sortedCats add: ( Association newWithKey: categName value: selectors)
].
result := Array new: (sz := sortedCats size) * 2  .
k := 1 .
1 to: sz do:[:j | | anAssoc |
   anAssoc := sortedCats at: j .
   result at: k put: anAssoc key .
   result at: k + 1 put: (Array withAll:(SortedCollection withAll: anAssoc _value)).
   k := k + 2 .
].
^ result .

]

{ #category : 'Accessing Categories' }
Behavior >> _categoryNotFound: aString [
  self _error: #classErrMethCatNotFound args: { aString } .
  ^ nil

]

{ #category : 'Browsing' }
Behavior >> _categoryOfSelector: selector [

"Returns the category of the given selector, or 'unknown' if it isn't found."

 | result |
 (result := self categoryOfSelector: selector) == nil ifTrue:[
    ^ 'unknown'
    ].
 ^ result

]

{ #category : 'Private' }
Behavior >> _checkCompileResult: result source: sourceString suppressCompileWarning: suppressCompileWarning [
  "process the result Array from _primitiveCompileMethod:...
   Returns a GsNMethod or signals a CompileError or CompileWarning."

  result _isArray ifFalse:[ 
    ^ result "a GsNMethod or a selector "
  ].
  (result at: 2) ifNotNil:[ :errorArray|
    "Fill in the error message text for each error in the result."
    [ | errDict |
      errDict := GemStoneError at: System myUserProfile nativeLanguage .
      1 to: errorArray size do:[:j | | thisErr |
	thisErr := errorArray at: j .
	(thisErr atOrNil: 3) ifNil:[  | msg |
	  thisErr size < 3 ifTrue:[ thisErr size: 3 ].
	  msg := errDict atOrNil:(thisErr at: 1) .
	  thisErr at: 3 put: ( msg ifNil:[ '(unknown error number)' ]).
	].
      ].
    ] onException: Error do:[:ex | "ignore"].
    ^ CompileError new args: { errorArray . sourceString . self } ; signal
  ].
  (result at: 3) ifNotNil:[ :warnStr | | meth |
    (meth := result at: 1) class == GsNMethod ifTrue:[
       suppressCompileWarning ifFalse: [ CompileWarning signal: warnStr method: meth ].
       ^ meth
    ].
  ].
  InternalError signal:'unrecognized result from _primitiveCompileMethod'

]

{ #category : 'Private' }
Behavior >> _classCategory [

"Private."

"The classCategory instance variable is defined in Class, so return nil here."

^ nil

]

{ #category : 'Accessing Variables' }
Behavior >> _classVars [

"Returns the classVars instance variable."

^ classVars

]

{ #category : 'Private' }
Behavior >> _clearConstraints [
 "Used in repository upgrade only.
 As of GemStone 64bit v3.4, constraints are no longer implemented."

 constraints ifNotNil:[ constraints := nil ]

]

{ #category : 'Virtual Machine Control' }
Behavior >> _clearLookupCaches: envId [

"Invalidates method lookup caches for all classes.
 Invalidates all send-site caches for the specified environment.

 envId must be a SmallInteger >= 0 and <= 255 .
"
<primitive: 165>
envId _validateClass: SmallInteger .
self _primitiveFailed:#_clearLookupCaches: args: { envId }

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> _codeChangedForEnv: envId [
  "set bit in VM's cbCodeChanged word so it can propagate to
   other VMs upon commit.  Code which adds or removes entries
   in persistent method dictionaries must send this method. 
   envId >= 0   methods changed for specified environment .
   envId == -1  sessions methods changed .
  "
<primitive: 854>
envId _validateClass: SmallInteger .
self _primitiveFailed: #_codeChangedForEnv: args: {envId} "envId out of range"
]

{ #category : 'Private' }
Behavior >> _compileMethod: sourceString
symbolList: aSymbolList [
"Compile sourceString as an instance method in the receiver without installing
 in any dictionaries.
 Returns a GsNMethod or signals an Error ."

^ self _compileMethod: sourceString symbolList: aSymbolList environmentId: 0

]

{ #category : 'Private' }
Behavior >> _compileMethod: sourceString
symbolList: aSymbolList
environmentId: environmentId [
"Compile sourceString as an instance method in the receiver without installing
 in any dictionaries.
 Returns a GsNMethod or signals an Error ."

| res |
res := self _primCompileMethod: sourceString
   symbolList: aSymbolList
   category: nil
   oldLitVars: nil
   intoMethodDict: false
   intoCategories: nil
   environmentId: environmentId .
^ self _checkCompileResult: res source: sourceString suppressCompileWarning: false

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> _compileMethodTrappingErrors: sourceString
dictionaries: aSymbolList
category: aCategoryString
environmentId: envId [

"If there are no errors, add the resulting compiled method to the receiver's
 method dictionary and returns nil.

 If errors occur, returns an Array of pairs.  The first element of each pair is
 the GemStone error number, and the second element is the offset into the
 sourceString where the error occurred.

 This method differs from compileMethod:dictionaries:category:environmentId:
 in that it traps all errors (not just compiler errors).
 Non-compiler errors are reported with a source offset of 0."

  self _validatePrivilege ifFalse:[ ^ nil ].
  ^ [ self compileMethod: sourceString dictionaries: aSymbolList
           category: aCategoryString asSymbol  intoMethodDict: nil 
           intoCategories: nil environmentId: envId  .
      nil
    ] onException: { CompileError . Error } do:
      { [:ex | { ex gsArguments at: 1 }  "a CompileError" ] .
        [:ex | { { ex number . 0 } }  "any other Error" ]
      }
]

{ #category : 'Deprecated' }
Behavior >> _constraintAt: offset [
 "As of GemStone 64bit v3.4, constraints are no longer implemented.
 This method is provided for examining classes in repositories upgraded from
 an older version."

  constraints _isArray ifFalse:[ ^ Object].
  ^ (constraints atOrNil: offset) ifNil:[ Object ]
                 ifNotNil:[ :cls | cls currentVersion ]

]

{ #category : 'Deprecated' }
Behavior >> _constraints [
	"Returns nil or an Array.

 As of GemStone 64bit v3.4, constraints are no longer implemented.
 This method is provided for examining classes in repositories upgraded from
 an older version."

	self
		deprecated: 'Behavior>>_constraints  deprecated, Constraints are no longer supported'.
	^constraints

]

{ #category : 'Browsing' }
Behavior >> _copyMethodsAndVariablesFrom: sourceClass except: except dictionaries: dicts [

"Copies all instance and class methods, pool dictionaries, and references
 to class variables from the given class to ourselves.  Returns an Array of
 methods in the source class which failed to compile.  Some of them
 might be class methods.  The Array will be empty if none failed.

 Method environmentIds are copied from the source methods.

 The except: argument is a list of categories and/or methods that should
 not be copied.  Each exception is two elements in the Array: a Character
 followed by a String or Symbol.
    $C a category of class methods follows
    $c a category of instance methods follows
    $S the selector of a class method follows
    $s the selector of an instance method follows
    $V a list of class variable names follows
    $P a list of pool dictionaries follows"

| failed srccls targcls sel doit pds otherCvs excludeCvs envId |

self _validatePrivilege ifFalse:[ ^ nil ].
self == sourceClass ifTrue:[
  "because iteration is directly over the source's categories dictionaries"
  ^ self error: 'source of copy must not be self'
].
failed := { } .
pds :=  #() .
excludeCvs :=  #() .

except ifNotNil: [ | i |
  i := except indexOf: $P.
  i ~~ 0 ifTrue: [ pds := except at: i+1 ].
  i := except indexOf: $V.
  i ~~ 0 ifTrue: [ excludeCvs := except at: i+1 ].
  ].

"Copy pool dictionaries"
sourceClass _poolDictionaries do: [ :dict | | poolDicts |
  poolDicts := poolDictionaries .
  (poolDicts ~~ nil and:[ poolDicts includesIdentical: dict]) ifFalse: [
    (pds includesIdentical: dict) ifFalse: [
      poolDicts ifNil:[ poolDicts := { } . poolDictionaries := poolDicts ]
          ifNotNil:[ poolDicts isInvariant ifTrue:[
                       poolDicts := Array withAll: poolDicts . poolDictionaries := poolDicts
                     ]].
      poolDicts add: dict
    ].
  ].
].

"Copy class variables"
otherCvs := sourceClass _classVars .
otherCvs ifNotNil:[ | destCvs |
   destCvs := classVars .
   otherCvs associationsDo: [ :assn | | other |
    destCvs ifNotNil:[ other := destCvs associationAt: assn key otherwise: nil ].
    (other == nil or: [other value == nil and: [assn value ~~ nil]]) ifTrue: [
      (excludeCvs includesValue: assn key) ifFalse:[
        destCvs ifNil:[ destCvs := self _createClassVarsDict ].
        destCvs add: assn
      ].
    ].
  ].
].

"Copy class and instance methods"
envId := 0 .  "change for Maglev"
1 to: 2 do: [ :j |
  j == 1 ifTrue:[ srccls := sourceClass.  targcls := self ]
        ifFalse:[ srccls := sourceClass class.  targcls := self class ].
  srccls categorysDo:[ :cat :sels |
      1 to: sels size do: [ :s | | oldMeth |
	sel := sels at: s.
	doit := true.
	1 to: except size by: 2 do: [ :i | | ch exCat |
          ch := except at: i .
          exCat := except at: i + 1 .
	  (((( ch == $C and: [targcls isMeta and: [cat == exCat ]]) or:
	    [ ch == $S and: [targcls isMeta and: [sel == exCat ]]]) or:
	    [ ch == $c and: [targcls isMeta not and: [cat ==  exCat  ]]]) or:
	    [ ch == $s and: [targcls isMeta not and: [sel ==  exCat ]]]) ifTrue: [
	      doit := false
	  ].
	].
	doit ifTrue: [ | methEnvId |
          oldMeth := srccls compiledMethodAt: sel environmentId: envId .
	  methEnvId := oldMeth environmentId .
          methEnvId == envId ifFalse:[ self error:'environmentId mismatch'].
	  ( targcls
	     _compileMethodTrappingErrors: oldMeth sourceString
	     dictionaries: dicts category: cat environmentId: envId ) ifNotNil:[
	    failed add: oldMeth
          ].
        ].
      ].
  ].
].

^failed.

]

{ #category : 'Modifying Classes' }
Behavior >> _createClassVarsDict [
  | dict |
  (dict := classVars) ifNil:[
    (dict := SymbolDictionary new ) objectSecurityPolicy: self objectSecurityPolicy .
    self _setClassVars: dict old: nil .
  ].
  ^ dict

]

{ #category : 'Private' }
Behavior >> _deepCopyWith: copiedObjDict [

"Private. Used internally to implement deepCopy."

^ self.

]

{ #category : 'Private Methods for Class Modification' }
Behavior >> _disallowGciCreateStore [

"Private.

 Sets bit in the format instance variable to cause GemBuilder for C instance
 creation and updates to go through message sends.  Semantics are private to
 GemBuilder for Smalltalk."

self _validatePrivilege ifTrue:[
  format := format bitOr: 16r200 "no update through structural access" .
  self _refreshClassCache: false
]

]

{ #category : 'Private' }
Behavior >> _gciCompileMethod: sourceString dictionaries: aSymbolList category: aCategoryString environmentId: envId [
	"used by GCI implementation , when session methods are enabled.
   Returns nil for successful compilation, a warning String,
   or signals a CompileError"

  | warnStr |
  [
    self compileMethod: sourceString dictionaries: aSymbolList
          category: aCategoryString environmentId: envId .
  ] onException: CompileWarning do: [ :ex | "handle CompileWarning"
    warnStr := ex warningString .
    ex resume .
  ].
  ^ warnStr
]

{ #category : 'Stripping Sources' }
Behavior >> _hideSourceCode [

"For each environment 0 method defined for this class, hide the source code.  All that
 remains of the source is the method signature and the initial comment if one
 exists."

self _validatePrivilege ifFalse:[ ^ nil ].
self env: 0 methodsDo:[ :selector :method | method _removeAllSourceButFirstComment ]

]

{ #category : 'Private Methods for Class Modification' }
Behavior >> _incrementInstVars: delta [

"Increment instVars by 1 to account for adding a named instance variable."
| mask n info_old |
mask := Class_numIvs_mask .
info_old := instVarsInfo .
delta _isSmallInteger ifFalse:[ delta _validateClass: SmallInteger ].
n := (info_old bitAnd: mask) + delta .
n < 0 ifTrue:[ self error:'numInstVars would go negative'].
n > 2030"virtual machine constant GEN_MAX_INSTANCE_VARS" ifTrue:[
   ArgumentError signal:'number of instance variables  would exceed 2030' .
].
instVarsInfo := (info_old bitAnd:( mask bitInvert)) bitOr:( n bitAnd: mask)

]

{ #category : 'Private Methods for Class Modification' }
Behavior >> _insertNamedInstVar: aSymbol atOffset: offset [
	"Receiver and all subclasses must have be modifiable.  aSymbol must be unique
 with respect to existing instance variables."
	" add instance variable to self"

	| mySubclasses ivn |
	self _validatePrivilege
		ifTrue:
			[self _incrementInstVars: 1.
			(ivn := instVarNames) == #()
				ifTrue:
					[ivn := {}.
					instVarNames := ivn].
			ivn insertObject: aSymbol at: offset.
			self _refreshClassCache: false.
			mySubclasses := self _subclasses.
			mySubclasses ~~ nil
				ifTrue:
					[mySubclasses do: [:x | x _insertNamedInstVar: aSymbol atOffset: offset]]].
	^self

]

{ #category : 'Accessing the Class Format' }
Behavior >> _instancesNpDbtransient [

  ^ (format bitAnd: 16r1800)  "used by _become:fullChecks:  methods"

]

{ #category : 'Accessing Variables' }
Behavior >> _instVarNames [

"Returns the receiver's instance variables list.  Contrast with the public
 method #allInstVarNames."

^ instVarNames

]

{ #category : 'Private' }
Behavior >> _instVarsEqual: anArrayOfInstvarNames [

  "Return true if the argument matches the instVarNames
   defined by the receiver (excluding inherited instVars), false otherwise."

  ^ (Array withAll: self instVarNames) = (anArrayOfInstvarNames collect:[:n | n asSymbol ])

]

{ #category : 'Private' }
Behavior >> _instVarsInfo [
  ^ instVarsInfo

]

{ #category : 'Error Handling' }
Behavior >> _isInstanceDisallowed [

^ (InstancesDisallowed includesValue: self) or:[ self isMeta ]

]

{ #category : 'Private' }
Behavior >> _isKernel [

"Private.  Returns true if the given class is a GemStone kernel class."

<primitive: 480>
self _primitiveFailed: #_isKernel

]

{ #category : 'Accessing the Class Format' }
Behavior >> _isSpecial [

"Returns true if instances of the receiver are special objects.
 Otherwise, returns false."

^ (format bitAnd: 16r3) == 3

]

{ #category : 'Indexing Support' }
Behavior >> _ivOffsetOf: aSymbol [

"Searches the instVarNames instance variable of the receiver for an instance
 variable named aSymbol, and returns the offset for that instance variable.
 Returns nil if no instance variable exists with the name aSymbol."

| idx |
idx := instVarNames indexOfIdentical: aSymbol .
^ idx == 0 ifTrue:[ nil ] ifFalse:[ idx ].

]

{ #category : 'Indexing Support' }
Behavior >> _idxIvOffsetOf: aSymbol [

"Searches the instVarNames instance variable of the receiver for an instance
 variable named aSymbol, and returns the offset for that instance variable.
 Returns nil if no instance variable exists with the name aSymbol.
 Signals an error if the instance variable is dbTransient. "

  | idx |
  idx := instVarNames indexOfIdentical: aSymbol .
  idx == 0 ifTrue:[ ^ nil ] .
  dbTransientMask ifNotNil:[:dbTrMask |
    ((1 bitShift: idx - 1 ) bitAnd: dbTrMask) ~~ 0 ifTrue:[
       Error signal:'dbTransient instVar ', aSymbol printString, ' may not participate in an index'.
    ].
  ].
  ^ idx

]

{ #category : 'Private Methods for Class Modification' }
Behavior >> _makeProtected [

"Make the receiver a protected class by setting a bit in its format.  This
 protection disallows structural access through GemBuilder for C."

self _validatePrivilege ifTrue:[
  format := format bitOr: 128
]

]

{ #category : 'Private' }
Behavior >> _makeTraversalByCallback [

"Private.

 Make the receiver place its instances in a traversal buffer by
 invoking the clampSpecification's traversal callback method."

self _validatePrivilege ifTrue:[
  format := format bitOr: 16r400. "travByCallback"
  self _refreshClassCache: false .
]

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> _methodWithSource: aString [

"Returns the environment 0 method from the receivers' method dictionary
 whose source string is equal to aString , or nil if not found."

self env: 0 methodsDo:[ :aSelector :aMethod |
   aMethod sourceString = aString ifTrue:[ ^ aMethod ].
].
^ nil

]

{ #category : 'Private' }
Behavior >> _moveMethod: aSelector toCategory: categoryName [
  ^ self moveMethod: aSelector toCategory: categoryName environmentId:0
]

{ #category : 'Accessing the Class Format' }
Behavior >> _name [
  ^ 'aBehavior'

]

{ #category : 'Deprecated' }
Behavior >> _namedIvConstraintAt: anInteger [
  "Returns the constraint at the specified offset, or Object.

 As of GemStone 64bit v3.4, constraints are no longer implemented.
 This method is provided for examining classes in repositories upgraded from
 an older version."

  (anInteger > self instSize ) ifTrue:[ ^ Object].
  ^ self _constraintAt: anInteger

]

{ #category : 'Private' }
Behavior >> _nameForFileout [
  "used by topaz"

^ self theNonMetaClass name

]

{ #category : 'Private' }
Behavior >> _noStrippedMethodSelectors [

"Returns an Array of two empty Arrays, signifying that no methods are to have
 their source stripped."

  ^ #( #() #() ).

]

{ #category : 'Enumerating' }
Behavior >> _persistentMethodDicts [
  "Returns an Array of the persistent method dictionaries for all environments"

 methDicts
   ifNil:[ ^ #() ]
   ifNotNil:[ :mds |
     mds _isArray ifTrue:[ | arr |
       arr := { } .
       1 to: mds size by: 4 do:[:j |
         (mds at: j) ifNotNil:[:aDict | arr add: aDict ]
       ].
       ^ arr
     ] ifFalse:[
      ^ { mds }
     ]
   ].

]

{ #category : 'Browser Methods' }
Behavior >> _poolDictionaries [

"Returns the object containing this instance's pool dictionaries."

^  poolDictionaries ifNil:[ #() ]

]

{ #category : 'Private' }
Behavior >> _primCompileMethod: sourceString
symbolList: aSymbolList
category: categorySymbol
oldLitVars: litVarArray
intoMethodDict: aMethodDict
intoCategories: aCategDict
environmentId: environmentId [

"Compiles sourceString as a method for the receiver in category categorySymbol,
 using the symbol list aSymbolList.  If the compilation succeeds, the method
 dictionary of the receiver will have been updated.

 Returns the GsNMethod produced by the compilation if the compilation
 succeeded with no warnings or errors, or an Array of of the form
    { (GsNMethod or nil if the compilation has errors) .
       (nil or an Array of error descriptors as described for
        compileMethod:dictionaries:category: ) .
       (nil or a String describing warnings)
     } .

 If litVarArray is not nil, it must be an Array of Symbol,SymbolAssociation pairs
 and this Array will be searched prior to searching aSymbolList to
 resolve literal variables within the method.

 If the compilation succeeds,
   if aMethodDict == false, the method is added to no dictionaries,
      and aCategDict is ignored
   else if aMethodDict ~~ nil, the method is added to per-session
      method dictionaries,
   else aMethodDict == nil, the method is added to receiver's
      persistent method dictionary.

 If aMethodDict is neither nil nor false, and aCategDict is not nil and
 the compilation succeeds, the resulting method is added to aCategDict
 instead of the receiver's categories.

 If the compilation succeeds, the selector of the new method is
 removed from all method lookup caches for the receiver and all subclasses
 thereof,   independent of the value of aMethodDict argument.

 environmentId must be a SmallInteger >= 0 and <= 255 .
 0 denotes the base Smalltalk image.  1 was reserved for use by Ruby .

 You must have code modification privilege to execute this primitive.

 The IR graph produced by the parser (a GsComMethNode) is available as
   (System __sessionStateAt: 19)
 until being overwritten by the next invocation of this primitive.
 See GsNMethod(C)>>generateFromIR: for example code of printing this IR."

<primitive: 228>
sourceString _validateClasses: { String }.
aSymbolList ~~ nil  ifTrue:[ aSymbolList _validateClass: SymbolList ].
categorySymbol _validateClass: Symbol.
litVarArray ~~ nil ifTrue:[ litVarArray _validateClass: Array].
(aMethodDict ~~ nil and:[ aMethodDict ~~ false]) ifTrue:[ aMethodDict _validateClass: GsMethodDictionary ].
aCategDict ~~ nil ifTrue:[ aCategDict _validateClass: GsMethodDictionary ].
environmentId _validateClass: SmallInteger .

^ self _primitiveFailed:
  #_primCompileMethod:symbolList:category:oldLitVars:intoMethodDict:intoCategories:environmentId:
   args: { sourceString .  aSymbolList .  categorySymbol .  litVarArray .
	   aMethodDict .  aCategDict .  environmentId }

]

{ #category : 'Private' }
Behavior >> _primitiveCompileMethod: sourceString
symbolList: aSymbolList
category: categorySymbol
oldLitVars: litVarArray
intoMethodDict: aMethodDict
intoCategories: aCategDict
environmentId: environmentId [

  | src cvtAb |
  cvtAb := System _zeroArgPrim: 132 "fast System gemConfigurationAt:#GemConvertArrayBuilder" .
 cvtAb ifTrue:[ | cvtRes |
   sourceString _isOneByteString ifFalse:[
     sourceString _validateClasses: { String }.
   ].
   cvtRes := GsNMethod _convertArrayBuildersIfNeeded: sourceString .
   cvtRes _isArray ifTrue:[
     "error during ArrayBuilder conversion"
     ^ cvtRes "a compile error descriptor"
   ].
   src := cvtRes .
 ] ifFalse:[
   src := sourceString
 ].
 ^ self _primCompileMethod: src
     symbolList: aSymbolList
     category: categorySymbol
     oldLitVars: litVarArray
     intoMethodDict: aMethodDict
     intoCategories: aCategDict
     environmentId: environmentId

]

{ #category : 'Private' }
Behavior >> _recompileAllMethods [
  "unconditionally recompile all env 0 methods."
| meths symList |
meths := { } .
symList := GsSession currentSession symbolList .
self env: 0 methodsDo:[ :selector :aMethod | meths add: aMethod ].
1 to: meths size do:[:j |
  (meths at: j) recompileIntoMethodDict: nil intoCategories: nil symbolList: symList
].
]

{ #category : 'Private Methods for Class Modification' }
Behavior >> _recompileMethodsAfterNewIvOffset: newOffset [

| mySymList mySubclasses |
self _validatePrivilege ifTrue:[
  mySymList:= GsCurrentSession currentSession symbolList .
  (newOffset < self instSize) ifTrue:[
      self recompileAllMethodsInContext: mySymList ] .
  mySubclasses := self _subclasses .
  mySubclasses ~~ nil ifTrue:[
    mySubclasses do:[:x | x _recompileMethodsAfterNewIvOffset: newOffset ] .
    ].
].
^ self

]

{ #category : 'Private Methods for Class Modification' }
Behavior >> _refreshClassCache: clearLookupCachesBool [

"Refreshes in-memory state of the class .
 If clearLookupCachesBool==true, also clears all method lookup caches.
 Must also be sent immediately after any change to the
 format or named instance variables of the receiver."

<primitive: 227>
self _primitiveFailed:#_refreshClassCache: args: { clearLookupCachesBool } .
self _uncontinuableError

]

{ #category : 'Virtual Machine Control' }
Behavior >> _refreshLookupCache: aSelector oldMethod: aMethod env: envId [

"This message must be sent whenever a class's method dictionary
 changes, to keep lookup caches current.

 If not nil, aSelector specifieds a selector added or removed from receiver's
 method dictionary(s) for specified envId .

 If the argument aMethod is not nil, any breakpoints in the obsolete
 method aMethod are cleared.
 If aMethod is a SmallInteger, it is assumed to be a Ruby method-hidden
 value , and breakpoint clearing logic is skipped.

 envId must be a SmallInteger >= 0 and <= 255 .
"

<primitive: 374>
aSelector ifNotNil:[ aSelector _validateClass: Symbol ] .
aMethod ifNotNil:[ | gsMethod |
   gsMethod := ObsoleteClasses at: #GsMethod .
   (aMethod isKindOf: gsMethod) ifTrue: [
     ^ self "ignore unconverted method from Gs64 2.x"
   ].
   aMethod _validateClass: gsMethod
].
envId _validateClass: SmallInteger .
self _primitiveFailed: #_refreshLookupCache:oldMethod:env:
     args: { aSelector . aMethod . envId }.
self _uncontinuableError

]

{ #category : 'Updating Categories' }
Behavior >> _removeBaseCategory: categoryName environmentId: envId [

"Removes the specified category and all its methods from the receiver's
 method dictionary.  If categoryName is not in the receiver's categorys
 generates an error.  Any breakpoints in removed methods are cleared."
  | cats catName |
  self _validatePrivilege ifFalse:[ ^ nil ].
  catName := Symbol _existingWithAll: categoryName .
  catName ifNil:[ ^ self _error: #rtErrKeyNotFound args: { categoryName } ].
  cats := self _baseCategorys: envId .
  cats ifNotNil:[ | selectors |
    selectors := cats at: catName .
    selectors do:[ :aSelector |
       self _basicRemoveSelector: aSelector environmentId: envId
    ].
    cats removeKey: catName
  ]  ifNil:[
    self _error: #rtErrKeyNotFound args: { categoryName }
  ]

]

{ #category : 'Modifying Classes' }
Behavior >> _removeClassVar: aSymbol [

"Removes the class variable with name aSymbol from the receiver's
 class variables dictionary.  Generates an error if aSymbol is not
 the name of a class variable of the receiver."

  self _validatePrivilege ifTrue:[
    classVars ifNil:[ ^ self _errorKeyNotFound: aSymbol ].
    classVars removeKey: aSymbol
            ifAbsent: [classVars _errorKeyNotFound: aSymbol].
  ]

]

{ #category : 'Modifying Classes' }
Behavior >> _removeClassVar: aSymbol ifAbsent: exceptionBlock [

"Removes the class variable with name aSymbol from the receiver's
 class variables dictionary.  Executes exceptionBlock if aSymbol is not
 the name of a class variable of the receiver."
  | dict |
  self _validatePrivilege ifTrue:[
    (dict := classVars) ifNil:[ exceptionBlock value ]
       ifNotNil:[ dict removeKey: aSymbol ifAbsent: exceptionBlock ].
  ]

]

{ #category : 'Private Methods for Class Modification' }
Behavior >> _removeInstVarAtOffset: offset [
	"Remove named instance variable at specified offset from self and all
 subclasses assuming that all error checks have been done."

	self _validatePrivilege
		ifTrue:
			[self _incrementInstVars: -1.
			instVarNames removeFrom: offset to: offset.
			self _refreshClassCache: false.
			self _subclasses do: [:x | x _removeInstVarAtOffset: offset]]

]

{ #category : 'Reloading Decompiled Methods' }
Behavior >> _resolveClassOrPoolVar: aSymbol [

"Searches the receiver's class variables dictionary and pool dictionaries to
 attempt to resolve aSymbol.  Returns the SymbolAssociation for the variable
 with name aSymbol, or nil if aSymbol could not be found."

| assoc |
assoc := self _resolveClassVar: aSymbol .
assoc ~~ nil ifTrue:[ ^ assoc ].

poolDictionaries size ~~ 0 ifTrue:[
  1 to: poolDictionaries size do:[:j | | aDict |
    aDict := poolDictionaries at: j .
    assoc := aDict associationAt: aSymbol otherwise: nil .
    assoc ~~ nil ifTrue:[ ^ assoc ].
    ].
  ].
^ nil

]

{ #category : 'Reloading Decompiled Methods' }
Behavior >> _resolveClassVar: aSymbol [

"Searches the receiver's class variables dictionary, to attempt to resolve
 aSymbol.  Returns the SymbolAssociation for the variable with name aSymbol, or
 nil if aSymbol could not be found."

| assoc |

classVars ifNotNil:[ :cvs |
  assoc := cvs associationAt: aSymbol otherwise: nil.
  ].
^ assoc

]

{ #category : 'Reloading Decompiled Methods' }
Behavior >> _resolveLiteralVar: aSymbol [

"Attempts to resolve a literal variable with name aSymbol.  To attempt to
 resolve aSymbol, searches the receiver's class variable dictionary and its pool
 dictionaries, then the superclasses' class variables, then the current default
 SymbolList.

 Returns the SymbolAssociation for the variable or nil if aSymbol could not be
 found."

"Implementation here must agree with comgen.c:searchClassOrPool()"

| assoc aClass |

aClass := self .
"search receiver's class variables and pool variables"
assoc := aClass _resolveClassOrPoolVar: aSymbol .
assoc ~~ nil ifTrue:[ ^ assoc ].

"search for a class variable inherited from superclasses"
[ (aClass := aClass superClass) ~~ nil
 ]
whileTrue: [
  assoc := aClass _resolveClassVar: aSymbol .
  assoc ~~ nil ifTrue:[ ^ assoc ].
] .
"search the symbol list"
^ GsSession currentSession resolveSymbol: aSymbol

]

{ #category : 'Browser Methods' }
Behavior >> _selectorPrefixesReport: envId [
  "In base Smalltalk image, no ruby prefix logic"
  ^ self _selectorsReport: envId

]

{ #category : 'Accessing Categories' }
Behavior >> _selectorsInBaseCategory: aSymbol [

  ^ (self _baseCategorys: 0) at: aSymbol

]

{ #category : 'Browser Methods' }
Behavior >> _selectorsReport: envId [
  ^ self _selectorsReport: envId matching: nil

]

{ #category : 'Browser Methods' }
Behavior >> _selectorsReport: envId matching: aString [
 ^ self _selectorsReport: envId matching: aString primitivesOnly: false

]

{ #category : 'Browser Methods' }
Behavior >> _selectorsReport: envId matching: aString primitivesOnly: primsBoolean [

^ self _selectorsReport: envId matching: aString primitivesOnly: primsBoolean
       includeDeprecated: true

]

{ #category : 'Browser Methods' }
Behavior >> _selectorsReport: envId matching: aString primitivesOnly: primsBoolean includeDeprecated: inclDeprecBool [
 "Used by topaz (and GBS?).
  Result is a sorted SequenceableCollection  of Symbols, plus an optional string 'Omitted .. deprecated methods'.
  aString if not nil restricts the result to only include those selectors containing
  the case-insensitive substring aString .
  primsBoolean if true further restricts the result to only include only
  selectors of methods which are primitives."
 | list res deprecSet numDeprecated |
 numDeprecated := 0 .
 inclDeprecBool ifFalse:[
   deprecSet := Object _selectorsInBaseCategory:#'Deprecated Notification'.
 ].
 list := self selectorsForEnvironment: envId .
 res := SortedCollection new .
 list do:[:sym | | sel |
    (aString == nil or:[ (sym includesString: aString)]) ifTrue:[
       sel := sym .
       (primsBoolean or:[ deprecSet ~~ nil ]) ifTrue:[
         (self compiledMethodAt: sym environmentId: envId otherwise: nil) ifNotNil:[ :meth|
           primsBoolean ifTrue:[ meth _isPrimitive ifFalse:[ sel := nil ]].
           sel ifNotNil:[
             deprecSet ifNotNil:[ (meth _selectorPool * deprecSet) size ~~ 0 ifTrue:[
               sel := nil . numDeprecated := numDeprecated + 1 ]].
           ].
         ]
       ].
       sel ifNotNil:[ res add: sel ].
    ].
 ].
 numDeprecated > 0 ifTrue:[
   res := Array withAll: res .
   res add:'(Omitted ' , numDeprecated asString, ' deprecated methods)'.
 ].
 ^ res

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> _sessionMethodsChanged [
  "set bit in VM's cbCodeChanged word so it can propagate to
   other VMs upon commit. 
   System class >> _sessionMethodsChanged*  returns union of bit
   with respect to a view change."
  ^ self _codeChangedForEnv: -1 .
]

{ #category : 'Debugging Support' }
Behavior >> _setMethodBreak: aSelector breakpointLevel: brkLevel [

"Returns true to indicate success.  Otherwise returns a string describing the error.
 brkLevel  >=1  means signal to Smalltalk , 0 means to GCI"

^ self _setMethodBreak: aSelector stepPoint: 1 env: 0 breakpointLevel: brkLevel 

]

{ #category : 'Debugging Support' }
Behavior >> _setMethodBreak: aSelector stepPoint: anInt [
 "Set a breakpoint that will be signalled to the GCI.
  Returns true to indicate success.  Otherwise returns a string describing the error."

 ^ self _setMethodBreak: aSelector stepPoint: anInt env: 0 breakpointLevel: 0"To GCI"
]

{ #category : 'Debugging Support' }
Behavior >> _setMethodBreak: aSelector stepPoint: anInt env: envId [

 "Set a breakpoint that will be signalled to the GCI.
  Returns true to indicate success.  Otherwise returns a string describing the error."

^ self _setMethodBreak: aSelector stepPoint: anInt env: envId breakpointLevel: 0"to GCI"

]

{ #category : 'Debugging Support' }
Behavior >> _setMethodBreak: aSelector stepPoint: anInt env: envId breakpointLevel: brkLevel [

"Returns true to indicate success.  Otherwise returns a string describing the error.
 brkLevel  >=1  means signal to Smalltalk , 0 means to GCI"

  | selectorSym x |
  (aSelector isByteKindOf: CharacterCollection) ifFalse: [ ^ 'Illegal selector' ].
  selectorSym := Symbol _existingWithAll: aSelector .
  selectorSym ifNil:[ ^ 'selector is not an existing Symbol'].
  ((x := GsNMethod optimizedSelectors) includesIdentical: selectorSym) ifTrue:[
      ^ 'You may not set a method break on an optimized selector' ].

  (self == SmallInteger
      and: [ #(  #+  #-  #>=  #*  #=  ) includesIdentical: selectorSym])
        ifTrue:[ ^ 'You may not set a method break on an optimized selector' ].

  (anInt _isSmallInteger) ifFalse:[ ^ 'Step point must be a SmallInteger' ].
  (self includesSelector: selectorSym environmentId: envId)
      ifFalse: [ ^ 'Selector does not exist in class' ].

  ((self compiledMethodAt: selectorSym environmentId: envId) 
    setBreakAtStepPoint: anInt breakpointLevel: brkLevel ) 
  ifNil:[ ^ 'Step point does not exist in method' ].

  ^ true

]

{ #category : 'Private' }
Behavior >> _setSelfCanBeSpecial [

  "Set the SelfCanBeSpecial bit in the receiver.
   For use only after altering the superclass hierarchy in such a way
   as to insert new classes between a class for which isSpecial is true
   and Object."

format := format bitOr: 16r2000 .

]

{ #category : 'Debugging Support' }
Behavior >> _sourceCodeAndOffsets: aSelector [
  ^ self _sourceCodeAndOffsets: aSelector environmentId: 0

]

{ #category : 'Debugging Support' }
Behavior >> _sourceCodeAndOffsets: aSelector environmentId: envId [

"Returns an Array with two elements.  The first element is a String
 representing the source code for the argument, aSelector.  The second element
 is an InvariantArray (that holds SmallIntegers) is a list of offsets into
 sourceString, corresponding in order to the step points.  If aSelector (a
 String) is not a selector in the receiver's method dictionary, returns nil."

| method |
method := self compiledMethodAt: aSelector environmentId: envId otherwise: nil .
method ifNotNil:[ ^ { method _sourceString . method _sourceOffsets } ] .
^ nil

]

{ #category : 'Private' }
Behavior >> _stripAllMethodSources [

"Returns true if all method sources should be stripped for the receiver, and
 false otherwise."

"Returns true for the list of classes given in the code, false for others."

| classNameList |
" classNameList is an Array of Symbols because some of these classes
are not known at filein "
classNameList := #( ).

1 to: classNameList size do: [ :i |
  ((Globals at: (classNameList at: i)) == self)
    ifTrue: [ ^ true ]
].
^ false

]

{ #category : 'Private' }
Behavior >> _structuralUpdatesDisallowed [

"Private.

 Returns true if GemBuilder for C (GCI) direct structural update of instances
 is disallowed, false otherwise.  A result of true means that the deferred
 update mechanism is used (see GciProcessDeferredUpdates in gci.hf) by
 GemBuilder for C store operations on instances of the receiver."

^ (format bitAnd: 16r200 "no update through structural access") ~~ 0

]

{ #category : 'Modifying Classes' }
Behavior >> _subclasses [

"Returns the class variable that is the list of subclasses, or
 nil if the receiver does not keep track of subclasses."

^ nil

]

{ #category : 'Testing Inheritance' }
Behavior >> _subclassOf: aClass [
"Returns true if the receiver is identical to or is a subclass aClass.

 aClass must be a Class or a Metaclass3 or nil .
 ClassHistories of the receiver and argument are ignored.

 If argument is nil and the receiver is a subclass of Object, returns false.
 If argument is nil and the receiver is not a subclass of Object, returns true.
"

<primitive: 895>
self _primitiveFailed: #_subclassOf: args: { aClass } .
^ false

]

{ #category : 'Private' }
Behavior >> _superclass [
  ^ superClass

]

{ #category : 'Private' }
Behavior >> _traversalByCallback [

"Private.

 Returns true if GemBuilder for C (GCI) traversal results of instances
 are obtained by message send of aClampSpecification.traversalCallBackSelector,
 false otherwise."

^ (format bitAnd: 16r400"travByCallback") ~~ 0

]

{ #category : 'Private Methods for Class Modification' }
Behavior >> _validateNewNamedInstVar: aSymbol [

"Generate an error if the argument is the name of an already existing instance
 variable of the receiver or if the receiver is not modifiable and has not
 disallowed subclassing, otherwise return true. If execution is continued
 from the error, return false."

"reimplementation of self validateIsModifiable ."
self isModifiable ifFalse:[
  self isNsc ifTrue:[
    self _error: #rtErrClassNotModifiable .
    ^ false .
  ].
  self subclassesDisallowed ifFalse:[
    self _error: #rtErrClassNotModifiable .
    ^ false .
  ] .
  self isIndexable ifTrue:[
     self _error: #rtErrClassNotModifiable.
    ^ false
  ].
] .
(instVarNames includesIdentical: aSymbol) ifTrue:[
  self _error: #rtErrAddDupInstvar args:{ aSymbol } .
  ^ false
].
self _subclasses ifNotNil:[ :mySubclasses |
  mySubclasses do:[:aSubCls|
     (aSubCls _validateNewNamedInstVar: aSymbol) ifFalse:[ ^ false ].
  ].
].
^ true

]

{ #category : 'Private' }
Behavior >> _validatePrivilege [

^ System myUserProfile _validateCodeModificationPrivilege

]

{ #category : 'Private' }
Behavior >> _varyingConstraint [

"Returns the constraint on the unnamed part of the receiver (a kind of Class).
 If the receiver has no constraint on its unnamed part, or if it has no unnamed
 part, this method returns Object.

 As of GemStone 64bit v3.4, constraints are no longer implemented.
 This method is provided for examining classes in repositories upgraded from
 an older version."

 ^ self _constraintAt: self instSize + 1 .

]

{ #category : 'Modifying Classes' }
Behavior >> addInstVar: aString [

"Adds a new instance variable named aString to the receiver.  The argument
 aString must be a valid GemStone Smalltalk identifier and must be distinct from
 the names of all other instance variables previously defined for the receiver,
 its superclasses, and its subclasses.

 The new instance variable becomes the last named instance variable in the
 receiver, and is inserted at the appropriate position in each of the
 receiver's subclasses, to preserve the rules for inheritance of instance
 variables.  If, for any of the receiver's subclasses, the new instance
 variable is not the last named instance variable, then all instance methods
 for that subclass are recompiled using the symbol list of the current user.
 If an error occurs during recompilation of methods, the new instance
 variable will have been added to the receiver and all of its subclasses, but
 some methods in some subclasses will not have been recompiled.

 To successfully invoke this method, the receiver must meet one of these two
 conditions:

 * The receiver and all of its subclasses must be modifiable.
 * The receiver must disallow subclasses and must have no unnamed instance
   variables."

| newOffset theSymbol |
self _validatePrivilege ifTrue:[
  "3.2, no error check for NSCs"
  (self isBytes) ifTrue:[ ^ self _error: #rtErrInstvarAddToBytes ] .
  theSymbol:= aString asSymbol .
  theSymbol validateIsIdentifier .
  (self _validateNewNamedInstVar: theSymbol ) ifFalse:[ ^ nil ].

  newOffset := self instSize + 1 .  "the offset of the new instance variable"
  self _insertNamedInstVar: theSymbol atOffset: newOffset .
  self _recompileMethodsAfterNewIvOffset: newOffset .
]

]

{ #category : 'Accessing Variables' }
Behavior >> allClassVarNames [

"Returns an Array of Symbols, consisting of the names of the
 smalltalk class variables addressable by this class,
 including those inherited from superclasses.  Contrast with classVarNames."

| result currClass |

result:= { } .
currClass:= self.
[ true ] whileTrue:[
  currClass == nil ifTrue:[ ^ result ].
  result insertAll: (currClass classVarNames) at: 1.
  currClass:= currClass superClass .
]

]

{ #category : 'Accessing Variables' }
Behavior >> allInstVarNames [

"Returns an Array of Symbols, consisting of the names of all the receiver's
 instance variables, including those inherited from superclasses.  The ordering
 of the names in the Array follows the ordering of the superclass hierarchy;
 that is, instance variable names inherited from Object are listed first, and
 those peculiar to the receiver are last."

^ Array withAll: instVarNames

]

{ #category : 'Modifying Classes' }
Behavior >> allowSubclasses [

"Allows creation of subclasses of a class."

format := (format bitOr:32) bitXor: 32 .  "clear bit 32"

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> allSelectors [

^ self allSelectorsForEnvironment: 0

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> allSelectorsForEnvironment: envId [
"Returns an Array of Symbols, consisting of all of the message
 selectors that instances of the receiver can understand, including
 those inherited from superclasses.  For keyword messages, the
 Symbol includes each of the keywords, concatenated together."

 | result currClass |
 result:= IdentitySet new .
 currClass:= self.
 [ currClass == nil ] whileFalse: [
     result _addAll: (currClass selectorsForEnvironment: envId ) forReplay: false .
     currClass:= currClass superclassForEnv: envId .
 ].
 ^ Array withAll: result

]

{ #category : 'Accessing Variables' }
Behavior >> allSharedPools [

"Returns an Array of pool dictionaries used by this class and its superclasses.
 Contrast with sharedPools."

| result currClass |

result:= { } .
currClass:= self.
[ true ] whileTrue:[
  currClass == nil ifTrue:[ ^ result ].
  result insertAll: (currClass sharedPools) at: 1.
  currClass:= currClass superClass
].

^ result

]

{ #category : 'Accessing the Class Hierarchy' }
Behavior >> allSuperClasses [

"Returns an Array of the environment 0 superclasses of the receiver,
 beginning with the immediate superclass, and excluding the receiver."

| result cls |
result:= { } .
cls := self superClass .
[ true ] whileTrue:[
  cls == nil ifTrue:[ ^ result ].
  result add: cls .
  cls := cls superClass.
]

]

{ #category : 'Enumerating' }
Behavior >> allSuperClassesDo: aBlock [

"Evaluates aBlock with each of the receiver's superclasses as
 the argument, beginning with the immediate superclass."

| currClass |

currClass:= self superClass .
[ true ] whileTrue:[
  currClass ifNil:[ ^ self ].
  aBlock value: currClass .
  currClass:= currClass superClass
]

]

{ #category : 'Accessing the Class Hierarchy' }
Behavior >> allSuperClassesForEnv: envId [

"Returns an Array of the superclasses of the receiver, in the specified
 environment beginning with the immediate superclass, and excluding the receiver."

| result cls |
result:= { } .
cls := self superclassForEnv: envId .
[ true ] whileTrue:[
  cls == nil ifTrue:[ ^ result ].
  result add: cls .
  cls := cls superclassForEnv: envId .
]

]

{ #category : 'Queries' }
Behavior >> areInstancesSpecial [

"Returns whether instances of this class have their state encoded in their
 identities."  "fix 48559"

^ (format bitAnd: 16r3) == 3  "inline _isSpecial"

]

{ #category : 'Formatting' }
Behavior >> asString [

"Returns a String that indicates the class of the receiver."
| n |
n := self _name .  "avoid use of squeak Object>>name"
n ifNil:[ ^ '(unNamedClass)' copy ].
n _stringCharSize == 1 ifTrue:[
  n class == Symbol ifTrue:[
    ^ String withAll: n
  ].
  ^ n
].
^ DoubleByteString withAll: n

]

{ #category : 'Formatting' }
Behavior >> asUnicodeString [

"Returns a Unicode string that indicates the class of the receiver."
| n |
n := self _name .  "avoid use of squeak Object>>name"
n ifNil:[ ^ Unicode7 withAll: '(unNamedClass)' ].
n _stringCharSize == 1 ifTrue:[
  ^ Unicode7 withAll: n .
].
^ Unicode16 withAll: n  "Unicode representation of DoubleByteSymbol"

]

{ #category : 'Instance Creation' }
Behavior >> basicNew [

"Creates a new, uninitialized instance of the receiver."

^self _basicNew

]

{ #category : 'Instance Creation' }
Behavior >> basicNew: anInteger [

"Creates a new, uninitialized instance of the receiver with the given
 number of indexed instance variables."

^self _basicNew: anInteger

]

{ #category : 'Indexing Support' }
Behavior >> btreeLeafNodeClass [

"Returns the class of BtreeLeafNode to create for an equality index whose last
 object along the path is an instance of the receiver."

| result |
(RangeEqualityIndex isBasicClass: self)
    ifTrue: [ result := self indexManager btreeBasicLeafNodeClass]
    ifFalse: [ result := self indexManager btreeLeafNodeClass ].
^ result

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> canUnderstand: aSelector [

"Returns true if the receiver can respond to the message indicated by
 aSelector, returns false otherwise.  The selector (a String) can be in the
 method dictionary of the receiver or any of the receiver's superclasses."

| aSymbol |
aSymbol := Symbol _existingWithAll: aSelector .
aSymbol ifNil:[ ^ false ].

^ (self whichClassIncludesSelector: aSymbol) ~~ nil

]

{ #category : 'Accessing Categories' }
Behavior >> categoryNames [

"Returns an Array of Symbols.  The elements of the Array are the
 receiver's environment 0 category names (excluding names inherited from superclasses)."

 | set |
 set := IdentitySet new .
 self env: 0 categorysDo:[ :categName :selectors | set add: categName ].
 ^ Array withAll: set

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> categoryOfSelector: aSelector [
 ^ self categoryOfSelector: aSelector environmentId: 0

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> categoryOfSelector: aSelector environmentId: envId [

"Returns a Symbol which is the name of the category for the specified
 selector, or nil if the selector was not found in any category."

| aSymbol |
aSymbol := Symbol _existingWithAll: aSelector .
aSymbol ifNil:[ ^ nil ].
self env: envId categorysDo:[ :categName :selectors |
  (selectors includesIdentical: aSymbol) ifTrue:[ ^ categName ]
].
^ nil

]

{ #category : 'Accessing Categories' }
Behavior >> categorysDo: aBlock [
  "evaluates aBlock for each category in environment 0.

   aBlock should be a two argument block
   expecting the args  categoryNameSymbol ,   selectorsSet,
   If the package manager is active in the current session,
   aBlock may be invoked more than once for each category name."

  self env: 0 categorysDo: aBlock

]

{ #category : 'Accessing Variables' }
Behavior >> classVarAt: aString [

| key |
"Returns the value of the class variable named aString ."

classVars ifNil:[ ^ self _error: #rtErrKeyNotFound args: { aString } ].
key := Symbol _existingWithAll: aString .
key ifNil:[ ^ self _error: #rtErrKeyNotFound args: { aString } ].
^ classVars at: key .

]

{ #category : 'Accessing Variables' }
Behavior >> classVarAt: aString otherwise: defaultValue [

| key |
"Returns the value of the class variable named aString ."

classVars ifNil:[ ^ defaultValue ].
key := Symbol _existingWithAll: aString .
key ifNil:[ ^ defaultValue ].
^ classVars at: key otherwise: defaultValue

]

{ #category : 'Accessing Variables' }
Behavior >> classVarNames [

"Returns an Array of Symbols naming the class variables defined by this class.
 Inherited class variables are not included; contrast with allClassVarNames."

| cvars |
(cvars := classVars) ifNil:[ ^ { } ].
^ cvars keys asArray

]

{ #category : 'Clustering' }
Behavior >> clusterBehavior [
	"This method clusters, in depth-first order, the parts of the receiver required
 for executing GemStone Smalltalk code (the receiver and its method dictionary).
 Returns true if the receiver has already been clustered during the current
 transaction; returns false otherwise.

 Note that methods in the VM's temporary object memory that were
 loaded for execution in this session cannot be clustered by this session.

 It is recommended that when several classes are being clustered in a
 transaction, send clusterBehavior to all classes to be clustered, then send
 clusterDescription."

	self cluster ifTrue: [^true].
	self isMeta
		ifFalse:
			[| arr |
			(arr := instVarNames) ~~ #() ifTrue: [arr cluster]].
	methDicts _isArray ifTrue: [methDicts cluster].
	self persistentMethodDictsDo:
			[:aDict |
			aDict cluster.
			"note GsNmethod>>clusterDepthFirst clusters execution related parts
       to current bucket and other(debug) info info to bucket 5"
			aDict values do: [:meth | meth clusterDepthFirst]].
	self persistentNameSpacesDo: [:aNs | aNs cluster].
	classVars
		ifNotNil:
			[:cvs |
			cvs cluster ifFalse: [cvs associationsDo: [:assoc | assoc cluster]]].
	^false

]

{ #category : 'Clustering' }
Behavior >> clusterBehaviorExceptMethods: aCollectionOfMethodNames [

"This method allows you to cluster the receiver more efficiently by omitting
 infrequently-used methods from the clustering.  The methods that you designate
 as aCollectionOfMethodNames will not be clustered with the receiver.  Thus,
 the methods that are frequently used will be packed more densely.  Returns
 true if the receiver has already been clustered during the current
 transaction; returns false otherwise.

 This method works by first clustering all methods named into the max cluster
 bucket, preventing them from showing up in the default cluster bucket.  It
 then uses the standard method to cluster behavior."

| savedBucket otherBucket systm |
systm := System .
savedBucket := systm clusterBucket.
otherBucket := 6 .
systm clusterBucket: otherBucket.
self persistentMethodDictsDo:[ :aDict |
   aDict keysAndValuesDo:[ :aSelector :aMethod |
     (aCollectionOfMethodNames includesValue: aSelector )
       ifTrue: [ aMethod clusterDepthFirst ].
   ].
].
systm clusterBucket: savedBucket.
^ self clusterBehavior

]

{ #category : 'Clustering' }
Behavior >> clusterDescription [

"This method clusters, in depth-first order, those instance variables in the
 receiver that are not expected to be accessed heavily during method execution,
 specifically the  poolDictionaries, and categories dictionaries.

 (The receiver itself is not clustered.)  Returns true if the description
 already been clustered during the current transaction; returns
 false otherwise.

 It is recommended that when several classes are being clustered in a
 transaction, send clusterBehavior to all classes to be clustered, then send
 clusterDescription."

| result |
result := false .
self isMeta ifFalse:[
  poolDictionaries ifNotNil:[ poolDictionaries cluster] .
].
result ifFalse:[
  categorys _isArray ifTrue:[
    result := categorys cluster .
    categorys do:[ :aDict | aDict clusterDepthFirst ].
  ] ifFalse:[
    categorys ifNotNil:[ result := categorys clusterDepthFirst ].
  ]
].
^ result

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> compileAccessingMethodsFor: anArrayOfSymbols [

"This method create accessing methods for reading and modifying instance and
 other variables in instances of the receiver.  Each element of anArrayOfSymbols
 must be an instance variable, class variable, or pool dictionary variable in
 the receiver.  For each variable 'x' in the Array, two methods are created: 'x'
 reads the variable, in the category 'Accessing', and 'x:newValue' modifies the
 variable, in the category 'Updating'.

 To create class methods to access class variables, class instance variables,
 or pool variables, the message must be sent to the class of the class.

 The new methods have environmentId == 0 .

 Returns the receiver.  Generates an error if any element of anArrayOfSymbols
 is not an instance variable, class variable, or pool variable of the
 receiver."

| allVarNames nvName compileBlk |

self _validatePrivilege ifFalse:[ ^ nil ].
nvName := 'newValue'.
allVarNames := self allInstVarNames collect:[:s | s asString ].
[allVarNames includesValue: nvName ] whileTrue:[ nvName := 'z' , nvName ].
compileBlk := [ :src :cat :varName  |
    [ self compileMethod: src
       dictionaries:  #()
       category: cat
       environmentId: 0
    ] onException: CompileError do:[:ex |
      ^ self _error: #classErrNotAVar args: { varName }
    ]
  ].
anArrayOfSymbols do: [ :var | | getSrc putSrc lf tab |
  lf := Character lf.
  tab := Character tab.
  (getSrc := String new) add: var; add: lf; add: tab; add: '^' ; add: var; add: lf.
  (putSrc := String new) add: var; add: ': ' ; add: nvName; add: lf;
	add: tab; add: var; add: ' := '; add: nvName; add: lf.
  compileBlk value: getSrc value: #Accessing value: var.
  compileBlk value: putSrc value: #Updating  value: var.
]

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> compiledMethodAt: aSelector [
  ^ self compiledMethodAt: aSelector environmentId: 0

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> compiledMethodAt: aSelector environmentId: envId [

"Returns the compiled method associated with the argument aSelector (a String).
 The argument must be a selector in the receiver's method dictionary; if it is
 not, this method generates an error."

| aMeth |
aMeth := self compiledMethodAt: aSelector environmentId: envId otherwise: nil.
aMeth == nil ifTrue:[ self _error: #rtErrKeyNotFound args: { aSelector } ].
^ aMeth

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> compiledMethodAt: aSelector otherwise: notFoundVal [
  ^ self compiledMethodAt: aSelector environmentId: 0 otherwise: notFoundVal

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> compileMethod: sourceString [

"If there are no errors, this adds the resulting compiled method to the
 receiver's method dictionary and returns that method,
 otherwise signals a CompileError .
 A CompileWarning may be signaled, after adding the new method
 to a receiver's method dictionary."

^ self compileMethod: sourceString
   dictionaries: GsCurrentSession currentSession symbolList
   category: '(as yet unclassified)'
   environmentId: 0

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> compileMethod: sourceString
dictionaries: aSymbolList
category: aCategoryString [

"May be deprecated in the future.
 Returns nil, or an Array of error descriptors"

^ [ self compileMethod: sourceString
       dictionaries: aSymbolList
       category: aCategoryString
       environmentId: 0 .
    nil
  ] onException: CompileError do:[:ex | ex errorDetails ]

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> compileMethod: sourceString
dictionaries: aSymbolList
category: categorySymbol
intoMethodDict: aMethodDict
intoCategories: aCategDict [

  ^self
    compileMethod: sourceString
    dictionaries: aSymbolList
    category: categorySymbol
    intoMethodDict: aMethodDict
    intoCategories: aCategDict
    environmentId: 0


]

{ #category : 'Updating the Method Dictionary' }
Behavior >> compileMethod: sourceString
dictionaries: aSymbolList
category: categorySymbol
intoMethodDict: aMethodDict
intoCategories: aCategDict
environmentId: environmentId [

"Compiles sourceString as a method for the receiver into the method category
 categorySymbol, using the symbol list aSymbolList. If the compilation succeeds,
 returns the GsNMethod produced by the compilation.
 If the compilation fails, signals a CompileError .
 If the compilation has warnings, signals a CompileWarning, after the GsNMethod
 has been generated and installed in any method dictionaries.

 aMethodDict may be nil, in which case the resulting method is added to the
 receiver's persistent method dictionary and aCategDict is ignored. If
 aMethodDict is not nil it is added to aMethodDict instead of to the receiver's
 method dictionary.  If aMethodDict and aCategDict are both not nil, category
 and selector are added to aCategDict.
 This is used to add methods to per-session dictionaries.

 The caller is responsible for setting the current GsObjectSecurityPolicy to
 match the receiver's GsObjectSecurityPolicy if desired.

 environmentId must be a SmallInteger >= 0 and <= 255 .
 0 denotes the base Smalltalk image.  1 was reserved for use by Ruby .

 You must have code modification privilege to execute this method.
 "

^ self _checkCompileResult:( self _primitiveCompileMethod: sourceString
		symbolList: aSymbolList category: categorySymbol
             oldLitVars: nil intoMethodDict: aMethodDict intoCategories: aCategDict
	     environmentId: environmentId )
        source: sourceString suppressCompileWarning: false

]

{ #category : 'Private' }
Behavior >> _primExtractSelector: sourceString [
  "Return a Symbol or an Array containing error details"

  <primitive: 252>
  sourceString _validateClasses: { String }.
  ^ self _primitiveFailed:
  #_primExtractSelector: args: { sourceString }
   
]

{ #category : 'Updating the Method Dictionary' }
Behavior >> extractSelector: sourceString [
  "Extract selector from source of a method and return a Symbol, 
   or signal a CompileError."
  ^ self _checkCompileResult:( self _primExtractSelector: sourceString )
        source: sourceString suppressCompileWarning: true

]


{ #category : 'Updating the Method Dictionary' }
Behavior >> compileMethod: sourceString
dictionaries: aSymbolList
category: categorySymbol
intoMethodDict: aMethodDict
intoCategories: aCategDict
intoPragmas: aPragmasArray
environmentId: environmentId [

"pragmasArray is ignored.  use GsNMethod>>pragmas or GsNMethod>>_pragmasArray
 to fetch the pragmas for a method.

 Should eventually be Deprecated."

^ self
  compileMethod: sourceString
  dictionaries: aSymbolList
  category: categorySymbol
  intoMethodDict: aMethodDict
  intoCategories: aCategDict
  environmentId: environmentId

]

{ #category : 'Browser Methods' }
Behavior >> compileMissingAccessingMethods [

"Creates accessing and updating methods for all instance variables that do not
 already have such methods.  Sent to a class, creates accessor methods for class
 instance variables that do not have them."

| argName newLine tab allVarNames varNames |

self _validatePrivilege ifTrue:[
  argName := 'newValue' .
  allVarNames := self allInstVarNames collect:[:s | s asString ].
  [allVarNames includesValue: argName] whileTrue: [
    argName := 'z' , argName .
  ].
  newLine := Character lf asString.
  tab := Character tab asString.
  varNames := self instVarNames.
  varNames accompaniedBy: self do: [ :me :var |
    (me includesSelector: var ) ifFalse: [
      me compileMethod: (var , newLine , tab , '^' , var , newLine)
	    dictionaries:  #()
	    category: #Accessing environmentId: 0 .
    ].
    (me includesSelector: var , $: ) ifFalse: [
      me compileMethod: (var , ': ' , argName , newLine , tab , var , ' := ' , argName , newLine)
	    dictionaries:  #()
	    category: #Updating environmentId: 0 .
    ]
  ]
]

]

{ #category : 'Copying' }
Behavior >> copy [

"Returns the receiver. Copies of classes and metaclasses are not made."

^ self.

]

{ #category : 'Browsing' }
Behavior >> copyMethodsFrom: sourceClass dictionaries: dicts [

"Copies all instance and class methods from the sourceClass.  Returns an Array
 of methods in the source class which failed to compile in this class.  Some of
 them might be class methods.  The Array is empty if no methods failed to
 compile."

| failed srccls targcls envId |
self == sourceClass ifTrue:[
  "because iteration is directly over the source's categories dictionaries"
  ^ self error: 'source of copy must not be self'
].
self _validatePrivilege ifFalse:[ ^ nil ].
failed := { } .

"Copy class and instance methods"
envId := 0 .
1 to: 2 do: [ :i |
  i == 1 ifTrue:[ srccls := sourceClass.  targcls := self ]
        ifFalse:[ srccls := sourceClass class.  targcls := self class ].
  srccls categorysDo:[ :cat :sels |
     sels do: [ :sel | | oldMeth |
        [
          targcls compileMethod: (oldMeth := srccls sourceCodeAt: sel environmentId: envId)
                    dictionaries: dicts category: cat environmentId: envId .
        ] onException: CompileError do:[:ex |
          failed add: oldMeth
        ].
     ].
  ].
].

^failed.

]

{ #category : 'Versions' }
Behavior >> currentVersion [
  ^ self

]

{ #category : 'Modifying Classes' }
Behavior >> disallowSubclasses [

"Disallows creation of subclasses of a class.  If the receiver is not
 modifiable, this method generates an error.  If the receiver is modifiable and
 already has subclasses, this method generates an error."

self _validatePrivilege ifTrue:[
  self isModifiable ifFalse:[ ^ self validateIsModifiable ].
  self _subclasses size ~~ 0 ifTrue:[ ^ self _error: #rtErrAlreadyHasSubclasses ].
  format := format bitOr: 32
]

]

{ #category : 'Accessing Categories' }
Behavior >> env: environmentId baseCategorysDo: aBlock [
  "evaluates aBlock for each method category of receiver
   in specified environment. Returns the receiver.

   aBlock should be a two argument block
   expecting the args  categoryNameSymbol ,   selectorsSet.
   aBlock will be invoked once for each category in the receiver.
   The iteration is directly over the receiver's categories."

  | cats |
  cats := self _baseCategorys: environmentId .
  cats ifNotNil:[ cats keysAndValuesDo: aBlock ].

]

{ #category : 'Accessing Categories' }
Behavior >> env: envId methodsDo: aBlock [
  "evaluates aBlock for each method in receiver's method dictionaries
   for specified environment. Returns the receiver.

   aBlock should be a two argument block
   expecting the args  selectorSymbol ,   aGsNMethod ."

  (self persistentMethodDictForEnv: envId ) ifNotNil:[:dict | dict keysAndValuesDo: aBlock ]

]

{ #category : 'Accessing Categories' }
Behavior >> env: envId unifiedCategoriesDo: aBlock [
  "evaluates aBlock for each method category of receiver
   in specified environment. Returns the receiver.

   aBlock should be a two argument block
   expecting the args  categoryNameSymbol ,   selectorsSet.
   aBlock will be invoked once for each category in the receiver.
   The iteration is done over a copy of the receiver's categories,
   however the selectorSets are not copied ."

  (self _unifiedCategorys: envId ) keysAndValuesDo: aBlock

]

{ #category : 'Accessing the Class Format' }
Behavior >> firstPublicInstVar [

"Returns the index of the first publicly available instance variable storage
 location, whether or not a public instance variable has actually been
 defined."

"Currently, an instance variable is considered to be public if it is to be
 included when passivating the object through PassiveObject.  Being public has
 no relationship to whether or not accessing or updating methods are defined
 for the instance variable."

<primitive: 960>
self _primitiveFailed: #firstPublicInstVar

]

{ #category : 'Accessing the Class Format' }
Behavior >> format [

"Returns the value of the format instance variable."

^ format

]

{ #category : 'Accessing the Class Format' }
Behavior >> hasPublicInstVars [

"Returns true if the receiver has publicly-visible instance variables."

^ self instSize >= self firstPublicInstVar

]

{ #category : 'Modifying Classes' }
Behavior >> immediateInvariant [
	"If the receiver is not invariant, make it invariant and
 conditionally also make the receiver's instVarNames invariant.
 Also clears the subclasses instVar .

 After this method executes,  addInstVar: and removeInstVar: will
 no longer be allowed for the receiver."

	self isInvariant
		ifFalse:
			[self _validatePrivilege ifFalse: [^nil].
			" Gs64 v3.0 and later,
			We no longer recompile methods here, the assumption is that removeInstVar:
			has recompiled methods as needed or has failed and invoked
      		System _disallowCommitClassModFailure."
			self subclassesDisallowed
				ifFalse:
					["only make the instVarNames collection invariant if we
					are not permitting possible dynamic addition of named instance variables."
					instVarNames immediateInvariant].
			"clear the subclasses class instance variable to prevent concurrency problems"
			self _subclasses ~~ nil ifTrue: [self _subclasses: nil].
			super immediateInvariant.	"make self invariant"
			"_codeChangedForEnv: not needed, since no instances can exist yet"
			self _refreshClassCache: false]

]

{ #category : 'Accessing the Class Format' }
Behavior >> implementationFormat [

"Returns the three least-significant bits of the receiver's format instance
 variable.  The values of those bits mean the following:

 0   OOP       non-indexable
 1   Byte      non-indexable
 2   NSC       non-indexable
 3   Special   non-indexable
 4   OOP       indexable
 5   Byte      indexable"

^ format bitAnd: 16r7

]

{ #category : 'Accessing Categories' }
Behavior >> includesCategory: aString [

"Returns true if aString is equivalent to the name of a category in the
 receiver, false otherwise."

^ self includesCategory: aString environmentId: 0

]

{ #category : 'Accessing Categories' }
Behavior >> includesCategory: aString environmentId: envId [

"Returns true if aString is equivalent to the name of a category in the
 receiver, false otherwise."

| aSym |
aSym := Symbol _existingWithAll: aString.
aSym ifNil:[ ^ false ].
self env: envId categorysDo:[ :name :selectors |
   name == aSym ifTrue:[ ^ true ].
].
^ false

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> includesSelector: aString [
  ^ self includesSelector: aString environmentId: 0

]

{ #category : 'Accessing the Class Hierarchy' }
Behavior >> inheritsFrom: aClass [

"Returns true if the argument aClass is on the receiver's
 superclass chain; returns false if it isn't."


(aClass isKindOf: Behavior)
ifFalse:
   [self _error: #rtErrBadArgKind args: { Class }].

(self isSubclassOf: aClass)
   ifTrue: [
     self == aClass
       ifTrue: [^false]
       ifFalse: [^true]
   ]
   ifFalse: [^false] .
   self _uncontinuableError " should never get here"

]

{ #category : 'Accessing the Class Format' }
Behavior >> instancesDbTransient [

  "Return true if the class format has the DbTransient bit set.
   See also  makeInstancesDbTransient ."

^ (format bitAnd: 16r1000) ~~ 0

]

{ #category : 'Accessing the Class Format' }
Behavior >> instancesInvariant [

"Returns true if instances of the receiver may not change value after they have
 been committed to GemStone.  Otherwise, returns false."

^ (format bitAnd: 16r8) ~~ 0

]

{ #category : 'Accessing the Class Format' }
Behavior >> instancesNonPersistent [

 "Returns true if instances may not be committed.
  See also makeInstancesNonPersistent ."

^ (format bitAnd: 16r800) ~~ 0

]

{ #category : 'Accessing the Class Format' }
Behavior >> instSize [

"Returns the number of named instance variables in the receiver, including all
 inherited instance variables."

^ instVarsInfo bitAnd: Class_numIvs_mask

]

{ #category : 'Updating' }
Behavior >> instVarAt: anIndex put: aValue [

anIndex == 7 ifTrue:[
  "Gs64 v3.0 structural update of the methDicts instVar disallowed"
   OffsetError  new _number: 2350; reason: #errNoStructuralUpdate ;
	  details:'methDicts instVar'; object: self ; signal .
   self _uncontinuableError.
].
anIndex == 9 ifTrue:[
  "Gs64 v3.0 structural update of the categories instVar disallowed"
   OffsetError  new _number: 2350; reason: #errNoStructuralUpdate ;
	details:'categorys instVar'; object: self ; signal .
   self _uncontinuableError.
].
^ super instVarAt: anIndex put: aValue

]

{ #category : 'Accessing Variables' }
Behavior >> instVarNames [

"Returns an Array of Symbols naming the instance variables defined by the
 receiver, but not including those inherited from superclasses.  Contrast
 with allInstVarNames."

| inheritedInstVars "an Array of the inherited instance variables"
  size              "the size of inherited instance variables"
  myInstVars        "an Array of all of the instance variables"
  supercls     |

(myInstVars := instVarNames) ifNil:[ ^ { } ].
(supercls := self superClass) ifNil: [ ^ myInstVars ].

inheritedInstVars := supercls _instVarNames.
(size := inheritedInstVars size) == myInstVars size ifTrue:[ ^{ } ].

"Assume that each inherited instance variable is added to the end of
 the Array result of allInstVarNames."

^ myInstVars copyFrom: size + 1 to: myInstVars size .

]

{ #category : 'Testing' }
Behavior >> isBehavior [

"Returns true if the receiver is a kind of Behavior, and returns false
 otherwise."

  ^true

]

{ #category : 'Accessing the Class Format' }
Behavior >> isBytes [

"Returns true if instances of the receiver are byte objects.  Otherwise,
 returns false."

^ (format bitAnd: 16r3) == 1

]

{ #category : 'Accessing the Class Format' }
Behavior >> isBytesOrSpecial [

"Returns whether instances of the receiver are byte objects."

^ (format bitAnd: 1) == 1

]

{ #category : 'Accessing the Class Format' }
Behavior >> isIndexable [

"Returns true if instances of the receiver have indexed variables.
 Otherwise, returns false."

^ (format bitAnd: 16r4) ~~ 0 "that is, is indexable"

]

{ #category : 'Modifying Classes' }
Behavior >> isModifiable [
	"Returns true if the receiver may be modified (that is, if the receiver and its Array of
 instance variable names are both variant, and the receiver has a 'subclasses' class variable).
 Returns false otherwise."

	| ivn |
	self isInvariant ifTrue: [^false].
	((ivn := instVarNames) size ~~ 0 and: [ivn isInvariant]) ifTrue: [^false].
	^self _subclasses ~~ nil

]

{ #category : 'Accessing the Class Format' }
Behavior >> isNonByteVarying [

"Returns true if the instances of the receiver are not byte objects and have
 unnamed instance variables; returns false otherwise."

| bits |

bits := format bitAnd: 7.
^bits == 2 or: [bits == 4].

]

{ #category : 'Accessing the Class Format' }
Behavior >> isNsc [

"Returns true if instances of the receiver are non-sequenceable
 collections (UnorderedCollections).  Otherwise, returns false."

^ (format bitAnd: 16r3) == 2

]

{ #category : 'Accessing the Class Format' }
Behavior >> isPointers [

"Returns true if instances of the receiver are pointer objects.
 Otherwise, returns false."

^ (format bitAnd: 16r3) == 0

]

{ #category : 'Accessing the Class Format' }
Behavior >> isProtected [

"Returns true if instances of the receiver may not be accessed structurally
 through GemBuilder for C. "

^ (format bitAnd: 16r80) ~~ 0

]

{ #category : 'Testing Inheritance' }
Behavior >> isSubclassOf: aClassHistory [

"Returns true if the receiver is identical to or is a subclass of any class
 in aClassHistory; otherwise, returns false.

 nil is an allowed value for the argument.
 If argument is nil and the receiver is a subclass of Object, returns false.
 If argument is nil and the receiver is not a subclass of Object, returns true.

 If the aClassHistory argument is actually a class rather than a class history,
 then this method uses the class history of the argument, instead of the class
 itself."

<primitive: 70>
self _primitiveFailed: #isSubclassOf: args: { aClassHistory } .
^ false

]

{ #category : 'Accessing the Class Format' }
Behavior >> isVariable [

"Returns true if instances of the receiver have an unnamed part."

self isIndexable ifTrue: [^true].
self isNsc ifTrue: [^true].
^false

]

{ #category : 'Accessing Categories' }
Behavior >> methodsDo: aBlock [
  "evaluates aBlock for each method in receiver's method dictionaries
   for environment 0 . Returns the receiver.

   aBlock should be a two argument block
   expecting args  selectorSymbol ,   aGsNMethod ."

  self env: 0 methodsDo: aBlock

]

{ #category : 'Instance Creation' }
Behavior >> migrateNew [

"Create a new instance to use in migration.  By default, use #new.
Override in subclasses that can't use #new with #_basicNew. "

^ self new

]

{ #category : 'Private' }
Behavior >> needsRecompileFor30 [

 "Returns true if the receiver needs to have methods recompiled."

  methDicts _isArray ifTrue:[ ^ false ].
  ^ methDicts valueConstraint ~~ GsNMethod

]

{ #category : 'Private' }
Behavior >> needsRecompileFor33 [

 "Returns true if the receiver needs to have methods recompiled."

  self needsRecompileFor30 ifTrue:[ ^ true ].
  self env: 0 methodsDo:[ :selector :aMethod |
    aMethod needsRecompile ifTrue:[ ^ true ]
  ].
  ^ false .

]

{ #category : 'Instance Creation' }
Behavior >> new [

"Returns an instance of the receiver with no indexed variables."

<primitive: 51>

self _primitiveFailed: #new .
self _uncontinuableError

]

{ #category : 'Instance Creation' }
Behavior >> new: anInteger [

"Returns an instance of the receiver with the specified number of indexed
 variables.  Generates an error if the receiver is not indexable or if
 anInteger is not a positive SmallInteger.

 For new byte objects, all indexed variables are set to zero;
 for new pointer objects, all indexed variables are set to nil."

<primitive: 53>

(self isIndexable) ifFalse:[ self _errorNotIndexable. ^ self new ].
(anInteger _isSmallInteger) ifFalse:[
  anInteger _isInteger ifTrue:[
    anInteger _error: #errArgTooLarge args:{ SmallInteger maximumValue }.
    ^ self new
  ].
  anInteger _validateClass: SmallInteger.
  ^ self new
].
(anInteger < 0) ifTrue:[ anInteger _error: #rtErrArgNotPositive. ^ self new].
self _primitiveFailed: #new: args: { anInteger }.
self _uncontinuableError

]

{ #category : 'Accessing Variables' }
Behavior >> offsetOfInstVar: aSymbol [

"Returns the integer offset at which the instance variable named aString is
 stored in instances of the receiver.  Returns zero if the instance variable
 is not found."

^ instVarNames indexOfIdentical: aSymbol.

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> persistentMethodAt: aSelector [
  ^ (self persistentMethodAt: aSelector otherwise: nil)
     ifNil:[ self _error: #rtErrKeyNotFound args: { aSelector } ].

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> persistentMethodAt: aSelector otherwise: notFoundVal [
  | sym pmd |
  (sym := Symbol _existingWithAll: aSelector) ifNil:[ ^ notFoundVal ].
  (pmd := self persistentMethodDictForEnv: 0) ifNil:[ ^ notFoundVal ].
  ^ pmd at: sym otherwise: notFoundVal .

]

{ #category : 'Updating' }
Behavior >> persistentMethodDictForEnv: envId [
"result will may be nil if no methods exist for specified environmentId."
| mds |
(mds := methDicts) _isArray ifTrue:[
  ^ mds atOrNil: (envId*4 + 1)
].
envId == 0 ifTrue:[ ^ mds ].
^ nil

]

{ #category : 'Updating' }
Behavior >> persistentMethodDictForEnv: envId put: aValue [
  "aValue should be a GsMethodDictionary, or nil ,
   caller responsible for _refreshClassCache "

<protected>
| ofs mds |
aValue ifNotNil:[
  aValue class == GsMethodDictionary ifFalse:[ 
   ^ ArgumentTypeError new name: 'methDicts' expectedClass: GsMethodDictionary actualArg: aValue;
     signal.
  ].
].
(mds := methDicts) _isArray ifFalse:[
  envId == 0 ifTrue:[
    methDicts := aValue .
    ^ self
  ].
  mds := { mds }.
  methDicts := mds .
].
ofs := envId*4 + 1 .
mds size < ofs ifTrue:[ mds size: ofs ].
mds at: ofs put: aValue

]

{ #category : 'Enumerating' }
Behavior >> persistentMethodDictsDo: aBlock [
| mds |
(mds := methDicts) ifNotNil: [
  mds _isArray ifTrue:[
    1 to: mds size by: 4 do:[:j |  | aDict |
      (aDict := mds at: j) ifNotNil:[ aBlock value: aDict ].
    ].
  ] ifFalse:[
    aBlock value: mds
  ]
]

]

{ #category : 'Enumerating' }
Behavior >> persistentNameSpacesDo: aBlock [
| mds |
(mds := methDicts) ~~ nil ifTrue: [
  mds _isArray ifTrue:[
    2 to: mds size by: 4 do:[:j |  | aNs |
      (aNs := mds at: j) ifNotNil:[ aBlock value: aNs ].
    ].
  ]
]

]

{ #category : 'Indexing Support' }
Behavior >> rcBtreeLeafNodeClass [

"Returns the class of BtreeLeafNode to create for an equality index whose last
 object along the path is an instance of the receiver."

| result |
(RangeEqualityIndex isBasicClass: self)
    ifTrue: [ result := self indexManager rcBtreeBasicLeafNodeClass]
    ifFalse: [ result := self indexManager rcBtreeLeafNodeClass ].
^ result

]

{ #category : 'Repository Conversion' }
Behavior >> recompileAllMethods [

"Recompile all environment 0 methods for execution in a Gs64 v3.0 or later system."

| meths md |
meths := { } .
self env: 0 methodsDo:[ :selector :aMethod | meths add: aMethod ].
1 to: meths size do:[:j | (meths at: j) recompile ].
(md := methDicts) class == GsMethodDictionary ifTrue:[
  md valueConstraint == (ObsoleteClasses at: #GsMethod) ifTrue:[
     md size ~~ 0 ifTrue:[ Error signal:'inconsistent methodDict size'].
     md valueConstraint: GsNMethod
  ]
].

]

{ #category : 'Repository Conversion' }
Behavior >> recompileMethodAt: aSelector [
  (self compiledMethodAt: aSelector environmentId: 0) recompile
]

{ #category : 'Analysis' }
Behavior >> referencedStrings [

"Returns a Set containing all Strings and InvariantStrings referenced by
 the environment 0 methods in this Class and its metaclass."

| set blk |

set := IdentitySet new.
blk := [ :selector :method | method _referencedStringsInto: set ].
self env: 0 methodsDo: blk .
self class env: 0 methodsDo: blk .
^set.

]

{ #category : 'Updating Categories' }
Behavior >> removeCategory: categoryName [
  ^ self removeCategory: categoryName environmentId: 0

]

{ #category : 'Modifying Classes' }
Behavior >> removeInstVar: aString [
	"Removes the instance variable named aString from the receiver and from all of
 the receiver's subclasses.  The receiver and all of its subclasses must be
 modifiable.

 All instance methods for the receiver and its subclasses are recompiled using
 the symbol list of the current user.  If an error occurs during recompilation
 of methods, the instance variable will have been removed from the receiver and
 from all of its subclasses, but some methods in some subclasses will not have
 been recompiled, and subsquent commit will be disallowed.

 You may not use this method to remove an inherited instance variable."

	| offset aSymbol success |
	self _validatePrivilege ifFalse: [^nil].

	"validate that the instance variable exists"
	aSymbol := Symbol _existingWithAll: aString.
	aSymbol ifNotNil: [offset := self _ivOffsetOf: aSymbol].
	offset ifNil: [^self _error: #classErrNotAVar args: {aString}].
	(self superClass _ivOffsetOf: aSymbol) == nil
		ifFalse: [^self _error: #classErrRemoveInherIv args: {aSymbol}].

	"validate that self and all subclasses are modifiable"
	self validateSubclassesAreModifiable.
	success := false.

	[self _removeInstVarAtOffset: offset.	"remove from self and all subclasses"
	self recompileAllSubclassMethodsInContext: GsCurrentSession currentSession
				symbolList.
	success := true]
			ensure: [success ifFalse: [System _disallowCommitClassModFailure]]

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> removeSelector: aString [

  self removeSelector: aString environmentId: 0

]

{ #category : 'Browsing' }
Behavior >> removeSelector: aString environmentId: envId ifAbsent: aBlock [
"Removes the method whose selector is aString from the receiver's
 method dictionary.  If the selector is not in the method
 dictionary, returns the result of evaluating the
 zero-argument block aBlock.  Otherwise, returns the receiver."

| aKey meth |

self _validatePrivilege ifFalse:[ ^ nil ].
aKey := Symbol _existingWithAll: aString .
aKey ifNotNil:[
  meth := self compiledMethodAt: aKey environmentId: envId otherwise: nil .
].
meth ifNil:[ ^ aBlock value ].
self removeSelector: aKey environmentId: envId

]

{ #category : 'Browsing' }
Behavior >> removeSelector: aString ifAbsent: aBlock [
  ^ self removeSelector: aString environmentId: 0 ifAbsent: aBlock

]

{ #category : 'Accessing Variables' }
Behavior >> scopeHas: aVariableName
ifTrue: aBlock [

"If aVariableName (a String) is specified as a variable in the receiver or one
 of its superclasses, this evaluates the zero-argument block aBlock and returns
 the result of evaluating aBlock.  Otherwise, returns false."

| allSharedPools |

(aVariableName isKindOf: String)
ifFalse:
   [ ^ aVariableName _error: #rtErrBadArgKind args: { String }].

( ((self allInstVarNames ) includesValue: aVariableName) or:
 [((self allClassVarNames) includesValue: aVariableName)])
ifTrue:
   [^ aBlock value]
ifFalse: "now check sharedPools"
[
   allSharedPools:= self allSharedPools.
   allSharedPools do: [:poolDict |
                         (poolDict includesKey: aVariableName)
                         ifTrue:
                            [ ^ aBlock value]
                      ]
].
^ false

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> selectors [
^ self selectorsForEnvironment: 0

]

{ #category : 'Accessing Categories' }
Behavior >> selectorsIn: categoryName [
  ^ self selectorsIn: categoryName environmentId: 0

]

{ #category : 'Accessing Categories' }
Behavior >> selectorsIn: categoryName environmentId: envId [

"Returns an Array of all selectors in the specified category.  If categoryName
 is not in the receiver's method dictionary, generates an error."

| set count catSym |
catSym := Symbol _existingWithAll: categoryName .
catSym ifNil:[ ^ self _categoryNotFound:  categoryName ].
count := 0 .
set := IdentitySet new .
self env: envId categorysDo:[ :categName :selectors |
  categName == catSym ifTrue:[ set addAll: selectors . count := count + 1 ].
].
count == 0 ifTrue:[ ^ self _categoryNotFound:  categoryName ].
^ Array withAll: set

]

{ #category : 'Accessing the Class Format' }
Behavior >> selfCanBeSpecial [

"Return true if the class format has the SelfCanBeSpecial bit set.

Any class for which self can be a special object must be created
using the value  #selfCanBeSpecial  as an element of the argument
to the options: keyword of the subclass creation method .
If instances of any of
   Boolean Character SmallDate SmallDateAndTime SmallDouble
   SmallFraction SmallInteger SmallScaledDecimal SmallTime UndefinedObject
will be able to inherit methods from a class, that class
must be created with  #selfCanBeSpecial .

Example, the Pharo class ProtoObject needs #selfCanBeSpecial  .

#selfCanBeSpecial is not inherited from a superclass by subclass
creation 

#selfCanBeSpecial be present in a superclass to be able to specify it for a subclass."

^ (format bitAnd: 16r2000) ~~ 0

]

{ #category : 'Accessing Variables' }
Behavior >> sharedPools [

"Returns an Array of pool dictionaries used by this class.  Superclasses
 are not included; contrast with allSharedPools."

| pd |
^ (pd := poolDictionaries) ifNil:[ { } ]
		      ifNotNil:[ Array withAll: pd ]

]

{ #category : 'Accessing Categories' }
Behavior >> sortedCategoryNames [

"Returns an Array of Symbols.  The elements of the collection are the
 receiver's category names (excluding names inherited from superclasses)."

 | set |
 set := IdentitySet new .
 self env: 0 categorysDo:[ :categName :selectors | set add: categName ].
 ^ Array withAll: (SortedCollection withAll: set ) .

]

{ #category : 'Accessing Categories' }
Behavior >> sortedSelectorsIn: categoryName [
  ^ self sortedSelectorsIn: categoryName environmentId: 0

]

{ #category : 'Accessing Categories' }
Behavior >> sortedSelectorsIn: categoryName environmentId: envId [

"Returns an Array of all selectors in the specified category, sorted
 in ascending order."

 ^ Array withAll:( SortedCollection withAll:(
        self selectorsIn: categoryName environmentId: envId) )

]

{ #category : 'Indexing Support' }
Behavior >> sortNodeClass [

"Returns the class of SortNode to create for sorting on instances of the
 receiver."

| result |
(RangeEqualityIndex isBasicClass: self)
    ifTrue: [ result := self indexManager sortNodeClass ]
    ifFalse: [ result := SortNode ].
^ result

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> sourceCodeAt: aSelector [

^ self sourceCodeAt: aSelector environmentId: 0

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> sourceCodeAt: aSelector environmentId: envId [

"Returns a String representing the source code for the argument, aSelector.  If
 aSelector (a String) is not a selector in the receiver's method dictionary,
 this generates an error."

^ (self compiledMethodAt: aSelector environmentId: envId) sourceString

]

{ #category : 'Instance Creation' }
Behavior >> squeakBasicNew: anInteger [

"Returns with an instance of the receiver.
 If the result is oop format, anInteger specifies the number of varying instVars.
 For an bytes format result, anInteger specifies the number of words
 where (self _bytesPerWord) is the number of bytes per word.

 Signals an error if the Behavior is not indexable or if anInteger is bad."

<primitive: 1082>
| isz |
(self _isInstanceDisallowed) ifTrue: [
   self _error: #objErrCantCreateInstance args:  #()  .
   self _primitiveFailed: #_basicNew: args: { anInteger }.
   self _uncontinuableError
].
(self isIndexable) ifFalse:[self _errorNotIndexable .  ^ self _basicNew ].
anInteger _isSmallInteger ifFalse:[
  anInteger _validateClass: SmallInteger . ^ self _basicNew
].
anInteger < 0 ifTrue:[
  anInteger _error: #rtErrArgNotPositive. ^ self _basicNew
].
(anInteger + (isz := self instSize)) _isSmallInteger ifFalse: [
  anInteger _error: #errArgTooLarge args:{ SmallInteger maximumValue - isz} .
  ^ self _basicNew
].
self _primitiveFailed: #_basicNew: args: { anInteger }.
self _uncontinuableError

]

{ #category : 'Accessing the Class Format' }
Behavior >> subclassesDisallowed [

"Returns true if subclasses of the receiver have been disallowed by means of
 Behavior | disallowSubclasses.  Otherwise, returns false."

^ (format bitAnd: 16r20) ~~ 0

]

{ #category : 'Accessing the Class Hierarchy' }
Behavior >> superclass [
 "for squeak compatibility"
 ^ self superClass

]

{ #category : 'Accessing the Class Hierarchy' }
Behavior >> superClass [

"Returns the receiver's superclass."

 ^ superClass

]

{ #category : 'Accessing the Class Hierarchy' }
Behavior >> superclassForEnv: envId [

 "Return receiver's superclass in environment zero .
  Reimplemented in Module to handle envId ~~ 0 ."

  ^ superClass

]

{ #category : 'Updating the Class Hierarchy' }
Behavior >> superclassForEnv: envId put: aCls [
  "Update the receiver's persistent superclass for method lookup
   environment specified by envId.
   envId must be a SmallInteger > 0 .
   aClass must be a Behavior.

   It is necessary to execute
     Behavior _clearLookupCaches: envId ; _clearLookupCaches: 0 .
   after executing this method if you want the change in hierarchy
   to take effect immediately for all method lookups.
"
  <primitive: 2001>
  | prot |
  prot := System _protectedMode .
  [ | mds ofs |
    aCls isBehavior ifFalse:[ ^ ArgumentError signal:'expected a Class' ] .
    (self selfCanBeSpecial and:[ aCls selfCanBeSpecial not]) ifTrue:[ "fix 48261"
      ^ ArgumentError signal:'receiver has selfCanBeSpecial and argument does not'.
    ].
    envId > 0 ifFalse:[ ^ ArgumentError signal:'expected envId > 0' ].
    (mds := methDicts) _isArray ifFalse:[
      self persistentMethodDictForEnv: envId put: nil .
      mds := methDicts .
      ].
    ofs := envId*4 + 3 .
    mds size < ofs ifTrue:[ mds size: ofs ].
    mds at: ofs put: aCls
  ] ensure:[
    prot _leaveProtectedMode
  ].

]

{ #category : 'Modifying Classes' }
Behavior >> validateIsModifiable [

"Returns the receiver if the receiver and its
 Array of instance variables are modifiable.  Generates an error if the
 receiver cannot be modified (that is, if the receiver
 or its Array of instance variable names is not variant)."

self isModifiable ifFalse:[ ^ self _error: #rtErrClassNotModifiable ]

]

{ #category : 'Modifying Classes' }
Behavior >> validateSubclassesAreModifiable [

"Generates an error if the receiver or any of its subclasses cannot be
 modified."

self validateIsModifiable .
self _subclasses do:[:x| x validateSubclassesAreModifiable ].
^ self

]

{ #category : 'Testing Inheritance' }
Behavior >> validateSubclassOf: aClass [

"Returns true if receiver is identical to aClass or is a subclass
 of aClass; otherwise, generates an error."

( self isSubclassOf: aClass) ifFalse:[
     ^ self _error: #rtErrNotASubclassOf args:{ aClass } ].
^ true

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> whichClassIncludesSelector: aString [
  ^ self whichClassIncludesSelector: aString environmentId: 0

]

{ #category : 'Accessing the Method Dictionary' }
Behavior >> _topazMethodAt: aString env: envId [
  "Returns a GsNMethod, or signals an Error."
  ^ [ self compiledMethodAt: aString
           environmentId: envId
    ] on: LookupError do: [:ex |
      nil
    ]

]

{ #category : 'Listing Instances' }
Behavior >> allInstances [

"Returns an Array that contains all instances of the receiver.
 Note: This method scans the entire GemStone repository, and may therefore
 take some time to execute.

 Large result sets may cause out-of-memory issues. To avoid problems,
 use Repository >> allInstances:, which returns a GsBitmap that does not
 require keeping objects in temporary object space.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

^ (SystemRepository listInstances: { self }) at: 1.

]

{ #category : 'Listing Instances' }
Behavior >> fastAllInstances [

"Returns an Array that contains all instances of the receiver.
 Note: This method scans the entire GemStone repository, and may therefore
 take some time to execute.

 Large result sets may cause out-of-memory issues. To avoid problems,
 use Repository >> allInstances:, which returns a GsBitmap that does not
 require keeping objects in temporary object space.

 This method aborts the current transaction; if an abort would cause unsaved
 changes to be lost, it signals an error, #rtErrAbortWouldLoseData."

^ (SystemRepository fastListInstances: { self }) at: 1.

]

