Extension { #name : 'GsCurrentSession' }

{ #category : 'Instance Creation' }
GsCurrentSession class >> currentSession [

"Returns the sole instance of GsCurrentSession that represents this login
 session."

<primitive: 311>
self _primitiveFailed: #currentSession

]

{ #category : 'Initialization' }
GsCurrentSession class >> initialize [
 "If the GciLogin flags contained bit GCI_CLIENT_DOES_SESSION_INIT.
  the VM expects the GCI application to do
    GciPerform(OOP_CLASS_GSCURRENT_SESSION, 'initialize', NULL, 0);
  after a successful login, so that errors in initialize are debuggable.
  Otherwise the VM will do the message send itself during the login.

  topaz passes GCI_CLIENT_DOES_SESSION_INIT. to GciLogin and does the
  GciPerform itself.
 "

 | res |
 (res := self currentSession)
    "No application changes here, they belong in
     the instance method GsCurrentSession >> initialize"
    initialize .
 ^ res

]

{ #category : 'Instance Creation' }
GsCurrentSession class >> new [

"Disallowed.

 The only instance of GsCurrentSession that is permitted in a session is created
 automatically when a user logs in to GemStone.  Its default SymbolList is a
 copy of the user's SymbolList.  You can obtain that instance by sending the
 message GsSession currentSession."

self shouldNotImplement: #new.
self _uncontinuableError .

]

{ #category : 'Private' }
GsCurrentSession class >> _sessionStateSizeReport [
  | str | str := String new .
  1 to: 40 do:[:n |
    (System __sessionStateAt: n) ifNotNil:[ :obj |
      str add: n asString, ':  ', obj recursiveSizeInMemoryReport; lf .
    ]
  ].
  ^ str

]

{ #category : 'Reduced Conflict Support' }
GsCurrentSession >> _RcHasConflicts [

"Do the hasConflicts check for Rc objects but do not selectively abort."

| scanArray redoObject conflicts|

scanArray := System _getRedoAndConflictObjects.
" if no redo objects were found, cannot resolve conflicts "
scanArray ifNil: [ ^ true ].

1 to: scanArray size by: 2 do: [ :i |
  redoObject := scanArray at: i.
  conflicts := scanArray at: i + 1.

  (redoObject _validateRcConflictsWith: conflicts)
    ifFalse: [ ^ true ]
].
^false

]

{ #category : 'Reduced Conflict Support' }
GsCurrentSession >> _scanRedoLogForConflicts [

"Scan the redo log for entries on RC objects that could conflict
on replay.  If we find one, return true, indicating that we do not
know if we can replay successfully without selectively aborting."

| redoLog |
(redoLog := System _redoLog) == nil
  ifTrue: [ ^ false ].

redoLog redoObjects keysAndValuesDo: [ :redoObj :logEntries |
  (redoObj _validateRcConflictsWith: #())
    ifFalse: [ ^ true ]
].
^ false

]

{ #category : 'Accessing the Symbol List' }
GsCurrentSession >> _transientSymbolList [
	"Direct access to transentSymbolList .
  The transientSymbolList is nil unless one of
		GsCurrentSession >> transientSymbolList
    GsCurrentSession >> transientSymbolList:  
    System class >> refreshTransientSymbolList
 have been executed in the current session."

  ^ transientSymbolList
]

{ #category : 'Transaction Control' }
GsCurrentSession >> abortTransaction [

"Rolls back all modifications made to committed GemStone objects and provides
 the session with a new view of the most current committed state of the
 repository.

 These operations are performed whether or not the session was previously in a
 transaction.  If the transaction mode is set to #autoBegin, then a new
 transaction is started.  If the transaction mode is set to #manualBegin, then
 a new transaction is not started."

System abortTransaction

]

{ #category : 'Transaction Control' }
GsCurrentSession >> beginTransaction [

"Starts a new transaction for the session.  If the session is already
 in a transaction, aborts the current transaction and starts a new transaction.

 If the session changed any permanent objects without committing them, their
 state is aborted."

System beginTransaction

]

{ #category : 'Session Configuration Access' }
GsCurrentSession >> clientVersionAt: aSymbol [

"Returns the value of the GsSession's client version information parameter named
 aSymbol.  Returns nil if no version parameter named aSymbol exists."

^ System clientVersionAt: aSymbol

]

{ #category : 'Transaction Control' }
GsCurrentSession >> commitAndReleaseLocks [

"Attempts to commit the transaction for the session.

 This method is the same as 'commitTransaction' except for the handling of
 locks.  If the commit succeeds, this method releases all locks for the session
 and returns true.  Otherwise, it returns false and does not release locks.

 This method also clears the commit release locks and commit-or-abort release
 locks sets.  See the 'Releasing Locks' method category in class System for
 more information."

^ System commitAndReleaseLocks

]

{ #category : 'Transaction Control' }
GsCurrentSession >> commitTransaction [

"Attempts to update the persistent state of GemStone to include changes made
 by this session.

 If the commit operation succeeds, then this method returns true, and the
 current transaction's changes, if any, become a part of the persistent
 repository.  After the repository update, the session exits the current
 transaction.  If the transaction mode is autoBegin, then the session enters
 a new transaction.  If the transaction mode is #manualBegin, then the session
 remains outside of a transaction.

 If conflicts prevent the repository update, then this method returns false.
 Call the transactionConflicts method to determine the nature of the
 conflicts.  If the session is outside of a transaction, then this method
 raises the error #rtErrPrimOutsideTrans.

 This method also updates the session's view of GemStone.  If the commit
 operation succeeds, then all objects in the session's view are consistent with
 the current state of GemStone.  If the commit fails, then this method retains
 all the changes that were made to objects within the current transaction.
 However, commits made by other sessions are visible to the extent that changes
 in this transaction do not conflict with them."

^ System commitTransaction

]

{ #category : 'Session Configuration Access' }
GsCurrentSession >> configurationAt: aSymbol [

"Returns the value of the configuration parameter named aSymbol (for example,
 #GEM_HALT_ON_ERROR).  Raises an error if aSymbol is not a valid parameter
 name."

^ System configurationAt: aSymbol

]

{ #category : 'Session Configuration Access' }
GsCurrentSession >> configurationAt: aSymbol put: anObject [

"Sets the value of the configuration parameter named aSymbol to anObject.
 Raises an error if aSymbol is not a valid parameter name or anObject is an
 inappropriate value for the parameter."

^ System configurationAt: aSymbol put: anObject

]

{ #category : 'Session Configuration Access' }
GsCurrentSession >> configurationParameters [

"Returns a Set of Symbols containing the names of all valid configuration
 parameters for this GsSession."

^ ConfigurationParameterDict keys

]

{ #category : 'Transaction Control' }
GsCurrentSession >> continueTransaction [

"Updates the session's view to the most recently committed state of GemStone
 without rolling back modifications made to committed objects in the session.
 The read and write sets of the session are carried forward and continue to
 accumulate until the session either commits or aborts.  Changes made by this
 session to committed objects are not visible to other sessions until the
 session commits.

 Returns true if accumulated modifications to the committed state of GemStone
 would not cause concurrency conflict as of the new view; otherwise returns
 false.  If it returns false, you can call the transactionConflicts method to
 determine the nature of the conflicts.

 This method can be used whether or not the session is outside of a transaction.
 Of course, the session cannot commit the accumulated changes unless it is
 inside a transaction.

 If transaction mode is #manualBegin, then continueTransaction does not alter
 the inside/outside of transaction state of the session.

 Modifications made by other committed transactions are accumulated for
 retrieval by GciDirtyObjs() and GciDirtySavedObjs() just as they are
 accumulated for commitTransaction or abortTransaction.

 This method has no effect on object locks held by the session.  Locks in the
 release locks sets are not released."

^ System continueTransaction

]

{ #category : 'Method lookup control' }
GsCurrentSession >> enableSessionMethods: aBoolean env: envId [
  "sets omPtr->sessionMethodsEnabled := aBoolean  in VM
   and returns previous state ; this controls compilation logic
   but has no effect on method lookup.  Caller responsible for
   using transientMethodDictForEnv:put: to install session methods.
   Clears method lookup caches for all in-memory classes, and
   clears all send-site caches for all in-memory methods. "

<primitive: 650>
aBoolean _validateClass: Boolean .
envId _validateClass: SmallInteger .
self _primitiveFailed:#enableSessionMethods: args: { aBoolean . envId }

]

{ #category : 'Smalltalk Execution' }
GsCurrentSession >> execute: aString [

"Executes aString containing GemStone Smalltalk code in the session represented
 by the receiver.  Symbol resolution is from the default symbol list."

^ ( aString _compileInContext: nil symbolList: self symbolList )
     _executeInContext: nil

]

{ #category : 'Smalltalk Execution' }
GsCurrentSession >> execute: aString symbolList: aSymbolList [

"Executes aString containing GemStone Smalltalk code in the session represented
 by the receiver.  Symbol resolution is from the given symbol list."

^ ( aString _compileInContext: nil symbolList: aSymbolList )
     _executeInContext: nil

]

{ #category : 'Transaction Control' }
GsCurrentSession >> inTransaction [

"Returns true to indicate that the session is in a transaction, false
 otherwise."

^ (System _zeroArgPrim: 5) ~~ 0   "fix 48290"

]

{ #category : 'Session Control' }
GsCurrentSession >> logoutAndGoIdle [

"The session performs a relogin as the Nameless user and goes idle by
 setting the transactionMode to #transactionless."

| namelessUserProfile |
namelessUserProfile := AllUsers userWithId: 'Nameless' ifAbsent: [self error: 'There is no Nameless user profile.'].

System currentObjectSecurityPolicy: namelessUserProfile defaultObjectSecurityPolicy.
System abortTransaction.
System _generationScavenge.
self reloginAsUser: 'Nameless' withPassword: ''.
self transactionMode: #transactionless.


]

{ #category : 'Deprecated' }
GsCurrentSession >> nativeLanguage [

"Returns the String that designates the language that controls error message
 generation in the current session."

| language |
self deprecated: 'GsCurrentSession>>nativeLanguage deprecated v3.0; base system no longer uses nativeLanguage'.
language := self objectNamed: #'NativeLanguage'.
language ifNil: [^#'English'].
^language.

]

{ #category : 'Accessing the Symbol List' }
GsCurrentSession >> objectNamed: aSymbol [

"Returns the first object in the receiver's symbol list that has the given
 name.  If no object with the given name is found, returns nil."

| assn |

assn := (transientSymbolList ifNil:[ symbolList]) resolveSymbol: aSymbol.
assn ifNil: [ ^nil ].
^assn _value

]

{ #category : 'Session Control' }
GsCurrentSession >> reloginAsUser: aUserId withEncryptedPassword: aPassword [

"The session aborts its current transaction, if any, and attempts a relogin
 with the specified userid and encrypted password.  Errors in login leave
 the session in the idle state, as if logoutAndGoIdle had been successfully
 executed."

| result |
result := System _reloginAsUser: aUserId password: aPassword encrypted: true.
(System hasUserAction: #_topazReloginCallback)
  ifTrue: [ System userAction: #_topazReloginCallback ].

result ifFalse: [System _error: #rtErrWarningPasswordExpire]
       ifTrue: [System transactionMode: #autoBegin]

]

{ #category : 'Session Control' }
GsCurrentSession >> reloginAsUser: aUserId withPassword: aPassword [

"The session aborts its current transaction, if any, and attempts a relogin
 with the specified userid and password.  Errors in login leave the session
 in the idle state, as if logoutAndGoIdle had been successfully executed."

| result |
result := System _reloginAsUser: aUserId password: aPassword encrypted: false.

(System hasUserAction: #_topazReloginCallback)
  ifTrue: [ System userAction: #_topazReloginCallback ].

result ifFalse: [System _error: #rtErrWarningPasswordExpire]
       ifTrue: [System transactionMode: #autoBegin]

]

{ #category : 'Accessing the Symbol List' }
GsCurrentSession >> resolveSymbol: aSymbol [

"Searches the receiver's symbol list for an Association whose key is equal to
 aString, and returns that Association.  If no such Association is found in the
 symbol list, returns nil.

 Implemented to use the current session's transient copy of the symbol list.
 This method is the default mechanism for symbol-resolution during
 compilation of GemStone Smalltalk methods."

^ (transientSymbolList ifNil:[ symbolList]) resolveSymbol: aSymbol

]

{ #category : 'Session Configuration Access' }
GsCurrentSession >> serverVersionAt: aSymbol [

"Returns the value of the GsSession's Stone version information parameter named
 aSymbol."

^ System stoneVersionAt: aSymbol

]

{ #category : 'Session Configuration Access' }
GsCurrentSession >> sessionVersionAt: aSymbol [

"Returns the value of the GsSession's Gem version information parameter
 named aSymbol."

^ System gemVersionAt: aSymbol

]

{ #category : 'Deprecated' }
GsCurrentSession >> signalFromSession [

"Deprecated.
 Return a GsInterSessionSignal object containing information about a signal from
 another session, or nil if there is no signal waiting."

| sigArray result |
self deprecated:'GsCurrentSession>>signalFromSession replaced by InterSessionSignal(C)>>poll' .
sigArray := System _signalFromGemStoneSession .
sigArray ifNil:[ ^ nil ].
result := GsInterSessionSignal signal: (sigArray at: 2)
			      message: (sigArray at: 3) .
result session: (GsSession sessionWithSerialNumber: (sigArray at: 1)).
^ result

]

{ #category : 'Accessing the Symbol List' }
GsCurrentSession >> symbolList [

"If a transientSymbolList has been installed, return it.
 Otherwise return the symbolList installed at session login, which
 is  System myUserProfile symbolList .

If you have not executed any of
  System class >> refreshTransientSymbolList
  GsCurrentSession >> transientSymbolList:
  GsCurrentSession >> transientSymbolList
in the current session, then
  GsCurrentSession currentSession symbolList == System myUserProfile symbolList
"

^ transientSymbolList ifNil:[ symbolList]

]

{ #category : 'Transaction Control' }
GsCurrentSession >> transactionMode [

"Returns the current transaction mode for the current GemStone session, either
 #autoBegin, #manualBegin or #transactionless.  The default is #autoBegin."

^ System _zeroArgPrim: 4

]

{ #category : 'Transaction Control' }
GsCurrentSession >> transactionMode: newMode [

"Sets a new transaction mode for the current GemStone session and exits the
 previous mode by aborting the current transaction.  Valid arguments are
 #autoBegin, #manualBegin or #transactionless.
 The mode transactionless is intended primarily for idle sessions. Users
 may scan database objects, but are at risk of obtaining inconsistent views.
 When in transactionless mode, the session is exempt from voting on
 possibleDeadObjs and will not be terminated by STN_GEM_TIMEOUT expiring.
"

System transactionMode: newMode

]

{ #category : 'Accessing the Symbol List' }
GsCurrentSession >> transientSymbolList [
  "If transientSymbolList is nil , set it to be a copy of the
      current UserProfile's symbolList .
   Returns the receiver's transientSymbolList .

   transientSymbolList is nil unless one of
		GsCurrentSession >> transientSymbolList
    GsCurrentSession >> transientSymbolList:  
    System class >> refreshTransientSymbolList
  have been executed in the current session."

  ^ transientSymbolList 
		ifNil: [
			"System class refreshTransientSymbolList directly updates the transientSymbolList instance variable"
			System refreshTransientSymbolList.
			transientSymbolList ]
]

{ #category : 'Accessing the Symbol List' }
GsCurrentSession >> transientSymbolList: aSymbolListOrNil [
 "Installs aSymbolListOrNil as the transientSymbolList of the receiver.
  Requires CodeModification privilege .

  The transientSymbolList is nil unless one of
		GsCurrentSession >> transientSymbolList
    GsCurrentSession >> transientSymbolList:  
    System class >> refreshTransientSymbolList
 have been executed in the current session."
  (aSymbolListOrNil isNil or: [aSymbolListOrNil _validateClass: SymbolList]) ifTrue:[
    System myUserProfile _validateCodeModificationPrivilege .
    transientSymbolList := aSymbolListOrNil 
  ]
]

{ #category : 'Session Configuration Access' }
GsCurrentSession >> versionParameters [

"Returns a Set of Strings containing the names of all valid version parameters
 for this GsSession."

^ VersionParameterDict keys

]
