Extension { #name : 'IntegerKeyValueDictionary' }

{ #category : 'Private' }
IntegerKeyValueDictionary >> _at: aKey put: aValue [

"Stores the aKey/aValue pair in the hash dictionary.  Rebuilds the hash table
 if the addition caused the number of collisions to exceed the limit allowed."

| hofs hashKey hashValue collisionBkt cbStatus |
aKey ifNil: [ ^ self _error: #rtErrNilKey ].
hofs := (aKey hash \\ tableSize) . hofs := hofs + hofs .
hashKey := self _atZ: hofs  .
hashKey ifNil: [
  collisionBkt := self _atZ: hofs + 1 .
  collisionBkt ifNil: [
    self _atZ: hofs putKey: aKey value: aValue .
    numElements := numElements + 1
  ] ifNotNil: [
    cbStatus := collisionBkt at: aKey put: aValue keyValDict_coll: self.
    cbStatus ~~ 0 ifTrue:[
      numElements := numElements + 1.
      cbStatus ~~ 1 ifTrue:[
        numCollisions := numCollisions + 1
      ].
    ].
  ]
] ifNotNil: [
  aKey = hashKey ifTrue: [
    self _atZ: hofs putKey: aKey value: aValue .
  ] ifFalse: [
    hashValue := self _atZ: hofs + 1 .
    collisionBkt := self collisionBucketClass new.
    collisionBkt objectSecurityPolicy: self objectSecurityPolicy .
    collisionBkt keyValueDictionary: self.
    self _atZ: hofs putKey: nil value: collisionBkt .
    collisionBkt at: hashKey put: hashValue keyValDict_coll: self.
    numElements := numElements + 1.
    cbStatus := collisionBkt at: aKey put: aValue keyValDict_coll: self.
    cbStatus > 1 ifTrue:[
      numCollisions := numCollisions + 1
    ]
  ]
].
(numCollisions > collisionLimit) ifTrue: [
  self rebuildTable: self _nextTableSize .
].
^aValue

]

{ #category : 'Private' }
IntegerKeyValueDictionary >> _removeKey: aKey ifAbsent: aBlock [

"Removes the key/value pair with key aKey from the receiver and returns
 the value.  If no key/value pair is present with key aKey, evaluates
 the zero-argument block aBlock and returns the result of that evaluation."

| hofs thisKey collisionBkt aValue lastPair |

aKey ifNil:[ ^ self _error: #rtErrNilKey ].
hofs := (aKey hash \\ tableSize) . hofs := hofs + hofs .
(thisKey := (self _atZ: hofs )) ifNil:[ "It might be in a collision bucket"
      collisionBkt := self _atZ: hofs + 1.
      collisionBkt ifNil:[
        aBlock ifNil:[^ self _errorKeyNotFound: aKey ] .
        ^ aBlock value
      ].
      aValue := collisionBkt removeKey: aKey ifAbsent: [
        aBlock ifNil:[^ self _errorKeyNotFound: aKey ] .
        ^ aBlock value
      ].
      "If we got this far, a key/value pair was removed from collisionBkt"
      numCollisions := numCollisions - 1.
      (collisionBkt size <= 1) ifTrue: [
         "We don't need the collision bucket anymore.
          Put its last key/value pair in our table."
         lastPair := collisionBkt _firstPair.
         self _atZ: hofs putKey: (lastPair at:1) value: (lastPair at:2) .
      ]
] ifNotNil: [ "There is a key/value pair in this slot"
      (aKey = thisKey) ifFalse: [
        aBlock ifNil:[^ self _errorKeyNotFound: aKey ] .
        ^ aBlock value
      ].
      "The key matches - remove it"
      aValue := self _atZ: hofs + 1.
      self _atZ: hofs putKey: nil value: nil .
].
numElements := numElements - 1.
^aValue

]

{ #category : 'Accessing' }
IntegerKeyValueDictionary >> at: aKey ifAbsent: aBlock [

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value."

| hofs hashKey collisionBkt |

aKey ifNotNil:[
  hofs := (aKey hash \\ tableSize) . hofs := hofs + hofs .
  hashKey := self _atZ: hofs  .
  hashKey ifNil: [
      collisionBkt := self _atZ: hofs + 1 .
      collisionBkt ifNotNil: [ ^ collisionBkt at: aKey ifAbsent: aBlock ]
  ] ifNotNil: [
    (aKey = hashKey) ifTrue: [ ^ self _atZ: hofs + 1 ]
  ].
].
aBlock ifNil:[^ self _errorKeyNotFound: aKey ] .
^ aBlock value

]

{ #category : 'Accessing' }
IntegerKeyValueDictionary >> at: aKey otherwise: defaultValue [

"Returns the value that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value."

| hofs hashKey collisionBkt |

aKey ifNotNil:[
  hofs := (aKey hash \\ tableSize) . hofs := hofs + hofs .
  hashKey := self _atZ: hofs  .
  hashKey ifNil: [
    collisionBkt := self _atZ: hofs + 1 .
    collisionBkt ifNotNil: [ ^collisionBkt at: aKey otherwise: defaultValue ]
  ] ifNotNil: [
    (aKey = hashKey) ifTrue: [ ^ self _atZ: hofs + 1 ]
  ].
].
^ defaultValue

]

{ #category : 'Updating' }
IntegerKeyValueDictionary >> at: aKey put: aValue [

"Stores the aKey/aValue pair in the KeyValueDictionary.  Generates an error if
 aKey is not a kind of Integer."

aKey _isInteger ifFalse: [ ^ aKey _error: #rtErrBadArgKind args: { Integer }].

^ self _at: aKey put: aValue

]

{ #category : 'Private' }
IntegerKeyValueDictionary >> collisionBucketClass [

"Returns the class of object to create when keys collide."

^ EqualityCollisionBucket

]

{ #category : 'Hashing' }
IntegerKeyValueDictionary >> compareKey: key1 with: key2 [

"No longer used by implementation of IntegerKeyValueDictionary"

self shouldNotImplement: #compareKey:with:

]

{ #category : 'Hashing' }
IntegerKeyValueDictionary >> hashFunction: aKey [

"No longer used by implementation of IntegerKeyValueDictionary"

self shouldNotImplement: #hashFunction:

]

{ #category : 'Accessing' }
IntegerKeyValueDictionary >> keyAt: aKey otherwise: defaultValue [

"Returns the key that corresponds to aKey.  If no such key/value pair
 exists, returns the given alternate value. "

| hofs hashKey collisionBkt |

aKey ifNotNil:[
  hofs := (aKey hash \\ tableSize) . hofs := hofs + hofs .
  hashKey := self _atZ: hofs  .
  hashKey ifNil: [
    collisionBkt := self _atZ: hofs + 1 .
    collisionBkt ifNotNil:[ ^ collisionBkt keyAt: aKey otherwise: defaultValue ]
  ] ifNotNil: [
    (aKey = hashKey) ifTrue:[ ^ hashKey ].
  ].
].
^ defaultValue


]

{ #category : 'Hashing' }
IntegerKeyValueDictionary >> rehash [
	"Re-establish any hash invariants of the receiver.
	 Integer hashes cannot change."

]

{ #category : 'Removing' }
IntegerKeyValueDictionary >> removeKey: aKey ifAbsent: aBlock [
  ^ self _removeKey: aKey ifAbsent: aBlock

]

{ #category : 'Removing' }
IntegerKeyValueDictionary >> removeKey: aKey otherwise: defaultValue [

  ^ self _removeKey: aKey ifAbsent:[ ^ defaultValue]

]
