Extension { #name : 'OrderedCollection' }

{ #category : 'Instance Creation' }
OrderedCollection class >> new: anInteger [

"This method is supplied for compatibility with other implementations of
 Smalltalk.  Its behavior differs from other Smalltalks in that it ignores the
 extension and returns an instance with size 0.

 Other Smalltalks use an instance variable to keep track of the number of
 elements in what are fixed-size instances.  GemStone objects are dynamically
 extensible and the size of the object is used to keep track of the number of
 elements in the collection."

anInteger < 0 ifTrue:[ anInteger _error: #rtErrArgNotPositive ].
^ super new .

]

{ #category : 'Adding' }
OrderedCollection >> _addAllFromNsc: aBag [

""

<primitive: 202>
self _primitiveFailed: #_addAllFromNsc: args: { aBag } .
self _uncontinuableError

]

{ #category : 'Adding' }
OrderedCollection >> add: newObject [

"Makes newObject one of the receiver's elements and returns newObject.  The
 new element is actually added as the last element of the receiver.  This
 method is equivalent to #addLast:."

<primitive: 203>
self _primitiveFailed: #add: args: { newObject } .
self _uncontinuableError

]

{ #category : 'Adding' }
OrderedCollection >> add: newObject after: targetObject [

"Adds newObject to the receiver immediately following the first element that is
 equal to targetObject, and returns newObject.  If targetObject is not found,
 reports an error."

| index |

index := self indexOf: targetObject
             ifAbsent: [^ self _errorNotFound: targetObject].

^ self insertObject: newObject at: (index + 1).

]

{ #category : 'Adding' }
OrderedCollection >> add: newObject afterIndex: anIndex [

"Adds newObject to the receiver at the index immediately following anIndex."

^ self insertObject: newObject at: (anIndex + 1).

]

{ #category : 'Adding' }
OrderedCollection >> add: newObject before: targetObject [

"Adds newObject to the receiver immediately before the first element that is
 equal to targetObject, and returns newObject.  If targetObject is not found,
 reports an error."

| index |

index := self indexOf: targetObject
             ifAbsent: [^ self _errorNotFound: targetObject].

^ self insertObject: newObject at: index.

]

{ #category : 'Adding' }
OrderedCollection >> add: newObject beforeIndex: anIndex [

"Inserts newObject to the receiver at the index anIndex and returns newObject."

^ self insertObject: newObject at: anIndex.

]

{ #category : 'Adding' }
OrderedCollection >> addAll: aCollection [

"Adds all of the elements of aCollection to the receiver and returns
 aCollection."

<primitive: 213>
"primitive handles arg kinds Array, OrderedCollection and IdentityBag."

^ super addAll: aCollection

]

{ #category : 'Adding' }
OrderedCollection >> addAll: aCollection after: targetObject [

"Adds each element of aCollection to the receiver immediately following the
 first element of the receiver which is equal to targetObject.  Reports an error
 if targetObject is not found.  Returns aCollection."

| index |

index := self indexOf: targetObject
             ifAbsent: [^ self _errorNotFound: targetObject].

^ self insertAll: aCollection at: (index + 1).

]

{ #category : 'Adding' }
OrderedCollection >> addAll: aCollection afterIndex: anIndex [

"Adds each element of aCollection to the receiver immediately after the element
 of the receiver at index anIndex.  Reports an error if targetObject is not
 found.   Returns aCollection."

^ self insertAll: aCollection at: (anIndex + 1).

]

{ #category : 'Adding' }
OrderedCollection >> addAll: aCollection before: targetObject [

"Adds each element of aCollection to the receiver immediately before the
 first element of the receiver which is equal to targetObject.  Reports an error
 if targetObject is not found.  Returns aCollection."

| index |

index := self indexOf: targetObject
             ifAbsent: [^ self _errorNotFound: targetObject].

^ self insertAll: aCollection at: index.

]

{ #category : 'Adding' }
OrderedCollection >> addAll: aCollection beforeIndex: anIndex [

"Adds each element of aCollection to the receiver immediately before the
 element of the receiver at index anIndex.  Reports an error if targetObject is
 not found.  Returns aCollection."

^ self insertAll: aCollection at: anIndex.

]

{ #category : 'Adding' }
OrderedCollection >> addAllFirst: aCollection [

"Inserts the given objects at the beginning of the receiver.  Returns the
 argument, aCollection."

self insertAll: aCollection at: 1.
^aCollection

]

{ #category : 'Adding' }
OrderedCollection >> addAllLast: aCollection [

"Appends the objects in aCollection to the receiver.  Returns aCollection."

self addAll: aCollection.
^aCollection

]

{ #category : 'Adding' }
OrderedCollection >> addFirst: anObject [

"Inserts the given object at the beginning of the receiver.  Returns the
 inserted object."

self insertObject: anObject at: 1.
^anObject

]

{ #category : 'Adding' }
OrderedCollection >> addLast: newObject [

"Adds newObject as the last element of the receiver and returns newObject."

<primitive: 203>
self _primitiveFailed: #addLast: args: { newObject } .
self _uncontinuableError

]

{ #category : 'Copying' }
OrderedCollection >> copyFrom: startIndex to: stopIndex [

"Returns a new collection containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver.

 If startIndex > stopIndex then an empty collection is returned.
 Otherwise both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
"

<primitive: 817>
(startIndex < 1) ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

((stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

self _primitiveFailed: #copyFrom:to: args: { startIndex . stopIndex }

]

{ #category : 'Comparing' }
OrderedCollection >> hasIdenticalContents: anArray [

"Returns true if all of the following conditions are true:

 1.  The receiver and anArray are of the same class.
 2.  The two Arrays are the same size.
 3.  The corresponding elements of the receiver and anArray are equal.

 Returns false otherwise."

<primitive: 612 >

self _primitiveFailed: #hasIdenticalContents: args: { anArray }.
self _uncontinuableError

]

{ #category : 'Searching' }
OrderedCollection >> includesIdentical: anObject [

"Returns true if anObject is identical to one of the elements of the
 receiver.  Returns false otherwise."

<primitive: 59>
self _primitiveFailed: #includesIdentical: args: { anObject }

]

{ #category : 'Searching' }
OrderedCollection >> indexOfIdentical: anObject [

"Private.  Returns the index of the first element in the receiver that is
 identical to the argument.  If the receiver does not have any elements that are
 identical to the argument, returns zero."

<primitive: 494>
self _primitiveFailed: #indexOfIdentical: args: { anObject }

]

{ #category : 'Copying' }
OrderedCollection >> replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex [

"Replaces the elements of the receiver between the indexes startIndex and
 stopIndex inclusive with the elements of aSeqCollection starting at repIndex.
 If aSeqCollection is identical to the receiver, the source and
 destination blocks may overlap.

 Returns the receiver."

<primitive: 608>
startIndex _isSmallInteger ifFalse:[ startIndex _validateClass: SmallInteger ].
stopIndex _isSmallInteger ifFalse:[ stopIndex _validateClass: SmallInteger ].
repIndex _isSmallInteger ifFalse:[ repIndex _validateClass: SmallInteger ].

^ super replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex

]
