Extension { #name : 'SymbolList' }

{ #category : 'CodeModification Override' }
SymbolList >> _addAllFromNsc: aBag [

self _validatePrivilegeAdd: aBag .
aBag accompaniedBy: self do:[:me :anObj |
	me _validateElement: anObj .
].
^ super _addAllFromNsc: aBag

]

{ #category : 'CodeModification Override' }
SymbolList >> _at: anIndex put: aValue [

self _validatePrivilegeAdd: aValue .
self _validateElement: aValue .
^ super at: anIndex put: aValue

]

{ #category : 'CodeModification Override' }
SymbolList >> _basicAt: anIndex put: aValue [

self _validatePrivilegeAdd: aValue .
self _validateElement: aValue .
^ super _basicAt: anIndex put: aValue

]

{ #category : 'Private' }
SymbolList >> _deferredGciUpdateWith: valueArray [

self size == 0 ifTrue:[
  self addAll: valueArray .
] ifFalse: [
  " semantics of the deferred update not defined"
  self _error: #errNoStructuralUpdate
]

]

{ #category : 'CodeModification Override' }
SymbolList >> _deleteNoShrinkFrom: startIndex to: stopIndex [

self _validatePrivilege: self .
^ super _deleteNoShrinkFrom: startIndex to: stopIndex

]

{ #category : 'Instance Initialization' }
SymbolList >> _gciInitialize [

self size: 0 .
^ true

]

{ #category : 'CodeModification Override' }
SymbolList >> _insertAt: destOffset
from: anArray
fromStart: startOffset
fromEnd: endOffset
numToMoveDown: numToMove [

self _validatePrivilegeAdd: anArray .
startOffset to: endOffset do:[ :j |
	self _validateElement:(anArray at: j) .
].
^ super
	_insertAt: destOffset
	from: anArray
	fromStart: startOffset
	fromEnd: endOffset
	numToMoveDown: numToMove

]

{ #category : 'CodeModification Override' }
SymbolList >> _primitiveAt: anIndex put: aValue [

self _validatePrivilegeAdd: aValue .
self _validateElement: aValue .
^ super _primitiveAt: anIndex put: aValue

]

{ #category : 'CodeModification Override' }
SymbolList >> _unsafeAt: anIndex put: aValue [

self _validatePrivilegeAdd: aValue .
self _validateElement: aValue .
^ super _unsafeAt: anIndex put: aValue

]

{ #category : 'Private' }
SymbolList >> _validateElement: anObj [

"if anObj is a kind of SymbolDictionary, return true, otherwise return false"

(anObj isKindOf: SymbolDictionary) ifTrue: [ ^ true ].
ArgumentTypeError new constrainedIv: 'SymbolList.element'
expectedClass: SymbolDictionary actualArg: anObj ;
 signal .
self _uncontinuableError

]

{ #category : 'Updating' }
SymbolList >> _validatePrivilege [

System myUserProfile _validateCodeModificationPrivilege

]

{ #category : 'Updating' }
SymbolList >> _validatePrivilege: item [

"Only need to #_validatePrivilege if adding/removing a SymbolDictionary"

(item isKindOf: SymbolDictionary) ifTrue: [
   self _validatePrivilege .
   ^self
].
(item isKindOf: Collection) ifTrue: [
  item do: [:x |
    (x isKindOf: SymbolDictionary) ifTrue: [
      ^ self _validatePrivilege .
    ]
  ].
].
^ true

]

{ #category : 'Updating' }
SymbolList >> _validatePrivilegeAdd: item [

"Check that item satisfies constraints and then check privilege."

(item isKindOf: SymbolDictionary) ifTrue: [
  self _validatePrivilege: item .
  ^ self
].
(item isKindOf: Collection) ifTrue: [
     item do: [:x | x _validateClass: SymbolDictionary ].
] ifFalse:[
    item _validateClass: SymbolDictionary .
].
self _validatePrivilege: item

]

{ #category : 'CodeModification Override' }
SymbolList >> , aSequenceableCollection [

self _validatePrivilegeAdd: aSequenceableCollection .
^ super , aSequenceableCollection

]

{ #category : 'CodeModification Override' }
SymbolList >> add: newObject [

self _validatePrivilegeAdd: newObject .
self _validateElement: newObject .
^ super add: newObject

]

{ #category : 'CodeModification Override' }
SymbolList >> add: newObject after: target [

self _validatePrivilegeAdd: newObject .
self _validateElement: newObject .
^ super add: newObject after: target

]

{ #category : 'CodeModification Override' }
SymbolList >> add: newObject before: target [

self _validatePrivilegeAdd: newObject .
self _validateElement: newObject .
^ super add: newObject  before: target

]

{ #category : 'CodeModification Override' }
SymbolList >> addAll: aCollection [

self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj |
	me _validateElement: anObj .
].
^ super addAll: aCollection

]

{ #category : 'CodeModification Override' }
SymbolList >> addAll: aCollection afterIndex: index [

self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj |
	me _validateElement: anObj .
].
^ super addAll: aCollection afterIndex: index

]

{ #category : 'CodeModification Override' }
SymbolList >> addAll: aCollection before: target [

self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj |
	me _validateElement: anObj .
].
^ super addAll: aCollection before: target .

]

{ #category : 'CodeModification Override' }
SymbolList >> addAll: aCollection beforeIndex: index [

self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj |
	me _validateElement: anObj .
].
^ super addAll: aCollection beforeIndex: index

]

{ #category : 'CodeModification Override' }
SymbolList >> addLast: newObject [

self _validatePrivilegeAdd: newObject .
self _validateElement: newObject .
^ super addLast: newObject

]

{ #category : 'CodeModification Override' }
SymbolList >> at: anIndex put: aValue [

self _validatePrivilegeAdd: aValue .
self _validateElement: aValue .
^ super at: anIndex put: aValue

]

{ #category : 'CodeModification Override' }
SymbolList >> atAllPut: anObject [

self _validatePrivilegeAdd: anObject .
self _validateElement: anObject .
^ super atAllPut: anObject

]

{ #category : 'Updating' }
SymbolList >> createDictionaryNamed: dictName at: anIndex [

"Creates a new SymbolDictionary in the GsObjectSecurityPolicy of the receiver.  Adds to the
 dictionary an Association whose key is dictName and whose value is the
 dictionary.  Inserts the dictionary in the receiver at anIndex.

 If anIndex is less than 1, the dictionary is inserted at the front of the list.
 If anIndex is greater than the size of the list, it is inserted at the end of
 the list.  If the receiver already contains a dictionary named dictName, raises
 an error."

| newDict nameSym offset |

self _validatePrivilege .
nameSym := dictName asSymbol .
self do:[ :aDict |
(aDict names includesIdentical: nameSym) ifTrue:[
  aDict _error: #rtErrDuplicateKey args: { nameSym }.
  ].
].
(newDict := SymbolDictionary new) name: nameSym .
newDict objectSecurityPolicy: self objectSecurityPolicy .
offset := anIndex .
offset < 1 ifTrue:[ offset := 1 ].
offset > self size
ifTrue:[ self addLast: newDict ]
ifFalse:[ self insertObject: newDict at: offset ].

]

{ #category : 'Searching' }
SymbolList >> dictionariesAndSymbolsOf: anObject [

"Returns an Array, possibly empty, of the form { { aDictionary . aSymbol } ... }.
 Searches the receiver for Associations whose value is identical to
 anObject and returns information for all matching Associations. "

| res aDict |
res := { } .
1 to: self size do:[ :j |
  (aDict := self at: j) ifNotNil:[
    aDict associationsDo:[ :assoc |
      assoc _value == anObject ifTrue:[ res add: { aDict . assoc key }]
    ].
  ].
].
^ res

]

{ #category : 'Searching' }
SymbolList >> dictionariesAndAssociationsOf: aSymbol [
  "Returns nil or an Array of the form { { dictionary . association }  ... }
  For all of the Associations in the enumeration of the receiver for which key == aSymbol ."

 | res |
 1 to: self size do:[ :j | 
   (self at: j) ifNotNil:[ :aDict |
     (aDict associationAt: aSymbol otherwise: nil) ifNotNil:[:assoc|
       res ifNil:[ res := { } ].
       res add: { aDict . assoc }
     ].
   ].
 ].
 ^ res .
]

{ #category : 'Searching' }
SymbolList >> dictionaryAndSymbolOf: anObject [

"Returns an Array containing two elements for the first Association
 in the enumeration of the receiver for which value == anObject .

 * The Dictionary in the receiver that contains an Association whose
   value is anObject.
 * The Symbol which is that Association's key.

 The receiver is searched in the same order that the compiler searches it.
 (For more information about symbol resolution, see the GemStone Programming
 Guide.)  If anObject is not found in the receiver, returns nil."

| aDict |
self deprecated: 'SymbolList>>dictionaryAndSymbolOf: deprecated v3.1, use dictionariesAndSymbolsOf:' .
1 to: self size do:[ :j |
  (aDict := self at: j) ifNotNil:[
    aDict associationsDo:[ :assoc |
      assoc value == anObject ifTrue:[ ^ { aDict . assoc key } ].
    ].
  ].
].
^ nil

]

{ #category : 'CodeModification Override' }
SymbolList >> first: anObject [

self _validatePrivilegeAdd: anObject .
self _validateElement: anObject .
^ super first: anObject

]

{ #category : 'CodeModification Override' }
SymbolList >> insertAll: aCollection at: anIndex [

self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj |
	me _validateElement: anObj .
].
^ super insertAll: aCollection at: anIndex

]

{ #category : 'CodeModification Override' }
SymbolList >> insertObject: anObject at: anIndex [

self _validatePrivilegeAdd: anObject .
self _validateElement: anObject .
^ super insertObject: anObject   at: anIndex

]

{ #category : 'CodeModification Override' }
SymbolList >> last: anObject [

self _validatePrivilegeAdd: anObject .
self _validateElement: anObject .
^ super last: anObject

]

{ #category : 'Accessing' }
SymbolList >> names [

"Returns an Array of Strings containing the names of the receiver's
 SymbolDictionaries.

 This method assumes that each SymbolDictionary in the receiver contains a
 SymbolAssociation whose value is that SymbolDictionary.  If any
 SymbolDictionary does not contain such a SymbolAssociation, it is
 represented in the result Array as '(unnamed Dictionary)'."

| result aDict aSymAssoc|
result:= { }  .
1 to: self size do:[:j|
  aDict := self at: j .
  aDict == nil
    ifTrue:[ result add: '(nil)' ]
    ifFalse:[
      aSymAssoc:= aDict associationsDetect:[:anAssoc | anAssoc value == aDict]
			ifNone:[ nil ].
      aSymAssoc == nil
        ifTrue:[ result add: '(unnamed Dictionary)' ]
        ifFalse:[ result add: (aSymAssoc key)]
      ].
  ].
^ result

]

{ #category : 'Formatting' }
SymbolList >> namesReport [

"Returns a formatted String describing the position and name of each Dictionary
 in the receiver's symbol list.

 This method assumes that each Dictionary in the symbol list contains an
 Association whose value is that Dictionary.  If any Dictionary does not
 contain such an Association, it is represented in the result String as
 '(unnamed Dictionary)'."

"used by the topaz list dictionaries command"

| arrayOfNames result lf |

arrayOfNames := self names .
result := String new .
lf := Character lf .
1 to: arrayOfNames size do:[:j|
  result add: (j asString).
  result addAll: '  '.
  result addAll: (arrayOfNames at: j) .
  result add: lf.
  ].
^result

]

{ #category : 'Searching' }
SymbolList >> objectNamed: aSymbol [

"Returns the first object in the receiver that has the given name.  If no object
 is found with the given name, returns nil."

| assn |

assn := self resolveSymbol: aSymbol.
assn == nil ifTrue: [ ^nil ].
^assn value

]

{ #category : 'CodeModification Override' }
SymbolList >> protectedRemoveFrom: startIndex to: stopIndex [

<protected>
^ super removeFrom: startIndex to: stopIndex

]

{ #category : 'CodeModification Override' }
SymbolList >> remove: oldObject [

self _validatePrivilege: oldObject .
^ super remove: oldObject

]

{ #category : 'CodeModification Override' }
SymbolList >> remove: oldObject ifAbsent: anExceptionBlock [

self _validatePrivilege: oldObject .
^ super remove: oldObject ifAbsent: anExceptionBlock

]

{ #category : 'CodeModification Override' }
SymbolList >> removeAll: aCollection [

self _validatePrivilege: aCollection .
^ super removeAll: aCollection

]

{ #category : 'CodeModification Override' }
SymbolList >> removeAllIdentical: aCollection [

self _validatePrivilege: aCollection .
^ super removeAllIdentical: aCollection

]

{ #category : 'CodeModification Override' }
SymbolList >> removeAllSuchThat: aBlock [

self _validatePrivilege: self .
^ super removeAllSuchThat: aBlock

]

{ #category : 'CodeModification Override' }
SymbolList >> removeAtIndex: anIndex [

self _validatePrivilege: self .
^ super removeAtIndex: anIndex

]

{ #category : 'Updating' }
SymbolList >> removeDictionaryNamed: aSymbol [

"Removes the first dictionary found in the receiver that contains an Association
 whose key is aSymbol and whose value is the dictionary.  Returns the removed
 dictionary.

 If no such dictionary is found, raises a KeyNotFound error."

self _validatePrivilege.
^ self removeDictionaryNamed: aSymbol ifAbsent:[
  self _error: #rtErrKeyNotFound args: { aSymbol }.
  nil
]

]

{ #category : 'Updating' }
SymbolList >> removeDictionaryNamed: aString ifAbsent: aBlock [

"Removes the first dictionary found in the receiver that contains an Association
 whose key is aSymbol and whose value is the dictionary.  Returns the removed
 dictionary.

 If no such dictionary is found, returns the result of evaluating the
 zero-argument Block aBlock."

self _validatePrivilege .
(Symbol _existingWithAll: aString) ifNotNil:[ :nameSym |
	1 to: self size do:[ :j | | aDict |
		aDict := self at: j .
		(aDict names includesIdentical: nameSym) ifTrue:[
		self removeAtIndex: j .
		^ aDict .
		].
	].
].
^ aBlock value

]

{ #category : 'CodeModification Override' }
SymbolList >> removeFirst [

self _validatePrivilege: self .
^ super removeFirst

]

{ #category : 'CodeModification Override' }
SymbolList >> removeFrom: startIndex to: stopIndex [

self _validatePrivilege: self .
^ super removeFrom: startIndex to: stopIndex

]

{ #category : 'CodeModification Override' }
SymbolList >> removeIdentical: oldObject [

self _validatePrivilege: oldObject .
^ super removeIdentical: oldObject

]

{ #category : 'CodeModification Override' }
SymbolList >> removeIdentical: oldObject ifAbsent: anExceptionBlock [

self _validatePrivilege: oldObject .
^ super removeIdentical: oldObject ifAbsent: anExceptionBlock

]

{ #category : 'CodeModification Override' }
SymbolList >> removeLast [

self _validatePrivilege: self .
^ super removeLast

]

{ #category : 'Updating' }
SymbolList >> replaceElementsFrom: aSymbolList [

"Removes all elements in the receiver and inserts all elements of aSymbolList in
 it, in the same order as in aSymbolList.

 If the argument is not a SymbolList, raises an error."

| newSize |
self _validatePrivilege: aSymbolList .
aSymbolList == self ifTrue:[ ^ self ].
(aSymbolList isKindOf: SymbolList) ifFalse:[
aSymbolList _validateClass: SymbolList .
^ self
].
self size: ( newSize := aSymbolList size) .
self replaceFrom: 1 to: newSize with: aSymbolList startingAt: 1

]

{ #category : 'CodeModification Override' }
SymbolList >> replaceFrom: startIndex to: stopIndex with: aCollection [

self _validatePrivilege: self .
self _validatePrivilegeAdd: aCollection .
aCollection accompaniedBy: self do:[:me :anObj |
	me _validateElement: anObj.
].
^ super replaceFrom: startIndex to: stopIndex with: aCollection

]

{ #category : 'CodeModification Override' }
SymbolList >> replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex [

| count |
self _validatePrivilege: self .
self _validatePrivilegeAdd: aSeqCollection .
count := stopIndex - startIndex + 1 .
0 to: count -1 do:[:j |
	self _validateElement:( aSeqCollection at: repIndex + j) .
].
^ super replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex

]

{ #category : 'CodeModification Override' }
SymbolList >> replaceFrom: startIndex to: stopIndex withObject: anObject [

self _validatePrivilege: self .
self _validatePrivilegeAdd: anObject .
self _validateElement: anObject .
^ super replaceFrom: startIndex to: stopIndex withObject: anObject

]

{ #category : 'Searching' }
SymbolList >> resolveSymbol: aString [

"Searches the receiver for an Association whose key is equal to aString, and
 returns that Association.  If no such Association is found, returns nil."

<primitive: 49>

(aString _isSymbol) ifFalse:[
  ^ (Symbol _existingWithAll: aString) ifNotNil:[ :sym | self resolveSymbol: sym]
].
aString _validateClasses: { String }.
self _primitiveFailed: #resolveSymbol: args: { aString } .
self _uncontinuableError

]

{ #category : 'CodeModification Override' }
SymbolList >> size: anInteger [

self _validatePrivilege: self .
^ super size: anInteger

]

{ #category : 'Class Membership' }
SymbolList >> species [

"Returns a class, an instance of which should be used as the result of
 select: or other projections applied to the receiver."

^ Array

]

{ #category : 'CodeModification Override' }
SymbolList >> squeakBasicAt: anIndex put: aValue [

^ self _basicAt: anIndex put: aValue

]

{ #category : 'Searching' }
SymbolList >> symbolResolutionOf: aString [

"Searches the receiver for aString.  If aString is found, returns a formatted
 String that describes the position in the receiver of the Dictionary defining
 aString, the name of that Dictionary, and aString.

 Generates an error if aString is not defined in the receiver."

"Example: GsSession currentSession symbolList symbolResolutionOf: #Integer"

| lf result theAssoc aDict aSymbol |

aSymbol := Symbol _existingWithAll: aString.
aSymbol == nil ifTrue:[ self _error: #rtErrSymNotFound args: { aString } ].

result := String new.
lf := Character lf.
1 to: self size do:[ :j|
  aDict := self at: j .
  aDict == nil ifFalse:[
    (aDict at: aSymbol otherwise: nil) ifNotNil:[
      " found aSymbol in this dict "
      result add: lf; add: (j asString); addAll: '  '.
      theAssoc:= aDict associationsDetect: [:anAssoc | anAssoc value == aDict]
                               ifNone: [ nil ] .
      theAssoc == nil
        ifTrue:[ result add: '(Unnamed Dictionary)']
        ifFalse: [ result add: theAssoc key ].
      result add: lf; addAll: '      '; addAll: aSymbol;
                 addAll: '  '; add: ((aDict at: aSymbol) asString).
      ^ result
      ].
    ].
  ].
^ self _error: #rtErrSymNotFound args: { aString }

]
