Extension { #name : 'System' }

{ #category : 'Transient Session State' }
System class >> __sessionStateAt: anIndex [
"
 Returns an indexed instance variable of the Transient Session State object.
 This object is a temporary instance of Array created after each GciLogin.

 If the specified instance variable of the Session State has not been stored
 into during this session, the result is nil.  This is different from
 Array>>at: semantics, where accessing beyond the end of an Array
 generates an error.

 Users of this method must observe the following instance variable allocations
 for this Array.

    anIndex      assignedUse
    -------      -----------
    1            Reserved for SymbolDictionary of Temporaries
    2            Reserved for ephemerons finalization queue .
    3            Topaz session state
    4            Reduced Conflict classes logging info
    5            Cache for Reduced Conflict values
    6            The one instance of GsCurrentSession
    7            Pending commit/abort errorNumber or userAction
    8            Reserved for GemStone GBJ support
    9            Other known sessions (a dictionary keyed on sessionSerialNum)
    10           Reserved for GemStone relational support
    11           Reserved for GemStone IndexManager support
    12           GemBuilder for C (GCI) deferred update Dictionary.
    13           GemBuilder for C (GCI) deferred update Array.
    14           Commit coordinator.
    15           Reserved for GemStone ProcessorScheduler support and prim 457
    16           Reserved for GemStone Locale support.
    17           Reserved for timezone support.
    18           Reserved for GemStone GsFile support, caching of std files.
    19           Last GsComMethNode produced by primitive 228.
    20           IcuCollator for the current default IcuLocale.
    21           a Boolean, IcuCollator.caseInsensitive
    22           Boolean controlling Notification >> defaultAction .
    23           nil or a Boolean,  usingUnicodeCompares .
    24           nil or an Array, detailed transaction conflicts of last failed commit
    25           Reserved for GemConnect for Postgres
    26           Used by System class >> waitForDebug 
    27           Progress print interval for long operations
    28..39       reserved for GemStone internal use.
    40..         Available for customer use . Customer range changed in v3.1"

<primitive: 111>

(anIndex _isSmallInteger)
  ifTrue: [self _errorIndexOutOfRange: anIndex .  ^ nil ]
  ifFalse: [self _errorNonIntegerIndex: anIndex .  ^ nil ].
self _primitiveFailed: #__sessionStateAt: args: { anIndex } .
self _uncontinuableError

]

{ #category : 'Transient Session State' }
System class >> __sessionStateAt: anIndex put: aValue [

"Modifies an indexed instance variable of the Transient Session State Array.
 The Session State Array will be grown as required to accommodate a store
 into the specified instance variable.

 anIndex must be < approx 2034 (Session State Array may not be a large object).

 See additional important documentation in the method #__sessionStateAt: .
 Returns receiver."

<primitive: 372>
(anIndex _isSmallInteger)
  ifTrue: [self _errorIndexOutOfRange: anIndex]
  ifFalse: [self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #__sessionStateAt:put: args: { anIndex . aValue } .
self _uncontinuableError

]

{ #category : 'Session Control' }
System class >> __sleepMs: milliSecondsToSleep [

"Sleep for the number of milliseconds specified by milliSecondsToSleep.
 The argument milliSecondsToSleep must be a positive SmallInteger.
 If milliSecondsToSleep is zero, this method has no effect.
 Returns time left to sleep , which will be > 0 if interrupted.
 The ProcessorScheduler is not used, and other waiting GsProcesses
 will not execute during the sleep. Use a Delay or Semaphore to wait
 while allowing other GsProcesses to run."

<primitive: 344>
milliSecondsToSleep _validateClass: SmallInteger.
milliSecondsToSleep < 0
   ifTrue: [ milliSecondsToSleep _error: #errArgTooSmall args:{ 0 } ] .
^ self _primitiveFailed: #_sleepMs: args: { milliSecondsToSleep }

]

{ #category : 'Private' }
System class >> _abortNestedTransaction [
  (self __sessionStateAt: 4) ifNotNil:[:rlArray |  | lev |
     (rlArray atOrNil: (lev := self transactionLevel)) ifNotNil:[
       "deref redo log entries for level being aborted"
       rlArray at: lev put: nil .
     ].
  ].
  self clearRcValueCache .
  (self _zeroArgPrim: 148 "_abortNested") ifFalse:[
    InternalError signal:'inconsistent transaction nesting'
  ].

]

{ #category : 'System Control' }
System class >> _addAllToStoneLog: aString [

"Appends text to the Stone's informational log file.  First, this method writes
 a banner that identifies the session from which aString came.  It then appends
 aString itself.  The argument must be a Utf8.

 Generates an error if aString is larger than approximately 16000 bytes."

<primitive: 463>
aString _validateClasses: { Utf8 }.
self _primitiveFailed: #addAllToStoneLog: args: { aString }

]

{ #category : 'Reduced Conflict Support' }
System class >> _addEntireObjectToRcReadSet: anObject [

"Adds anObject to the reduced-conflict read set (RcReadSet).
 If the object is a large object, all nodes of the object are added.

 See also documentation for _addToRcReadSet:includingAllNodes: "

self _addToRcReadSet: anObject includingAllNodes: true.

]

{ #category : 'Reduced Conflict Support' }
System class >> _addRootObjectToRcReadSet: anObject [

"Adds anObject to the reduced-conflict read set (RcReadSet).
 Only the root object is added (even if the object is a large object).

 See also documentation for _addToRcReadSet:includingAllNodes: "

self _addToRcReadSet: anObject includingAllNodes: false.

]

{ #category : 'Reduced Conflict Support' }
System class >> _addToRcReadSet: anObject includingAllNodes: aBoolean [

"Adds anObject to the reduced-conflict read set (RcReadSet).
 All objects that are part of composite used to represent a reduced
 conflict object should be added to this set to avoid unnecessary
 conflict handling.
 If aBoolean is true, the entire object (including all interior nodes
 of a large object) are added.  If aBoolean is false, only the root object
 is added.

 The RcReadSet is used as follows during a commit attempt .
   readWriteConflicts =   (strongReadSet * writeSetUnion) - RcReadSet .
   writeWriteConflicts = (writeSet * writeSetUnion) .
 If there are readWriteConflicts, the commit always fails.
 If writeWriteConflicts is not empty,
 but (writeWriteConflicts - RcReadSet ) is empty ,
 then RC replay is executed and the commit is reattempted once .
 "

<primitive: 116>
aBoolean _validateClass: Boolean.
self _primitiveFailed: #_addToRcReadSet:includingAllNodes:
     args: { anObject . aBoolean } .
self _uncontinuableError

]

{ #category : 'Performance Monitoring' }
System class >> _approxOopHighWaterMark [

"Returns an Integer that gives the approximate highest oop allocated in GemStone."

^self _zeroArgPrim: 23

]

{ #category : 'Performance Monitoring' }
System class >> _approxOopNumberHighWaterMark [

"Returns an Integer that gives the approximate highest oop number allocated in GemStone."

^self _zeroArgPrim: 36

]

{ #category : 'Runtime Configuration Access' }
System class >> _atConfigId: rtConfigId put: aValue [

"Change the specified runtime configuration parameter.  The current user
 must be SystemUser to set any configuration parameter except the parameters

 Changing a runtime parameter does not alter the configuration file.
 The new value will show up in the result of configuration report methods,
 but will not be used when a Gem or Stone next starts up."

<primitive: 355>
rtConfigId _validateClass: SmallInteger .
aValue _validateKindOfClasses: { SmallInteger . Array } .
(self _configParameterName: rtConfigId kind: $C ) == nil
   ifTrue:[ rtConfigId _error: #rtErrArgOutOfRange ].
rtConfigId >= 0 ifTrue:[
  (self _configParameterName: rtConfigId kind: $C ) _error: #rtErrConfigReadOnly
  ].

self _primitiveFailed: #_atConfigId:put: args: { rtConfigId . aValue } .
self _uncontinuableError

]

{ #category : 'Private' }
System class >> _bypassReadAuth [

"For use by indexing and query subsystem only.

 Subsequent object read-faults will bypass
 read-authorization checks to allow indexed queries and index maintenance
 to execute.  This allows indexing code to execute comparisons on keys
 in btree nodes without regard to read-authorization of the keys.
 Query results will never contain objects for which the current user
 is not read-authorized.

 Has no effect if current user is SystemUser.

 Indexing code should use this method in combination with an ensure: block
 which contains   _exitBypassReadAuth

 The bypass is shutoff automatically (equivalent to _exitBypassReadAuth)
 upon return from a userAction, from a recurFromOm, , upon return to the GCI,
 or upon generation of an error which would unwind the VM C stack.
"

<protected primitive: 595>
self _primitiveFailed: #_bypassReadAuth

]

{ #category : 'Shared Cache Management' }
System class >> _cacheAddresses: midLevelBool includeMyCache: includeMyCacheBool [
| idx result batch |
idx := 0 .
result := { }  .
[ idx >= 0 ] whileTrue:[ | batSize |
  batch := self _remoteCachesListStartingAt: idx midLevelOnly: midLevelBool
					    includeMyCache: includeMyCacheBool .
  batSize := batch size .
  batSize > 0 ifTrue:[
    1 to: batSize - 1 do:[:n | result add: ((batch at:n) at: 8) ].
    idx := batch at: batSize .
  ] ifFalse:[
    idx := -1 . "done"
  ].
].
^ result

]

{ #category : 'Private' }
System class >> _cacheName: aString [

"Sets the name the current session is known by in the shared cache
 to 'aString'. Does nothing if the current session is not attached
 to a shared cache. Raises an error if 'aString' is too long.
 Returns self."

<primitive: 509>
self _primitiveFailed: #_cacheName: args: { aString }

]

{ #category : 'Private' }
System class >> _cacheStatWithName: aString opCode: anInt [

"Return the value of the cache stat with the given name, which must match an
 element of the Array returned by the #cacheStatisticsDescription method.

 opCode 1 means the statistic is for the primary shared page cache monitor.
 opCode 2 means the statistic is for the stone.
 opCode 8 means the statistic is for the current session.
 opCode 128 means the statistic is for the page manager thread
 opCode 512 means the statistic is for a thread in a gem.

 This method may be used to access primary shared page cache monitor, stone and page manager
 statistics from hosts other than that on which the stone is running.

 Returns nil if a statistic matching aString was not found for the process kind
 specified by the opCode."

<primitive: 430>
aString _validateClass: String .
anInt _validateClass: SmallInteger .
^ self _primitiveFailed: #_cacheStatWithName:opCode:
       args: { aString . anInt }

]

{ #category : 'Private' }
System class >> _checkCacheAndClear: aBool [

"Scans the cache and if aBool is false, returns an array of statistics
 for the current contents of the cache:
    total number of pages in the cache
    number of free pages
    number of pinned pages
    number of locked pages
    number of dirty pages
    number of clean pages
    number of pages written (only non zero if arg is true)
    number of pages preemted (only non zero if arg is true)

If the aBool argument is true, as it scans it also tries to write and
preempt pages from the cache and fills in the last two array elements with
the number of operations performed.

Requires the SystemControl privilege if aBool is true.

No guarantees are made that the values will be consistent from run to run
since the contents of the cache may change depending on other commits,
aborts and garbage collection (reclaims) that might occur. "

<primitive: 1001>
aBool _validateClass: Boolean .
^ self _primitiveFailed: #_checkCacheAndClear:
       args: { aBool }

]

{ #category : 'Private' }
System class >> _cleanupZombieSession: sessionId [

"Forces the stone to finish logout processing for the given session ID.
 Takes no action if the given session has not begun the logout process or
 if the session no longer exists or if the session is the current session.
 When logout processing is completed, references to a commit record by
 that session are released and resources consumed by the session are recycled.

 To execute this method, you must have SessionAccess and SystemControl
 privileges.

 WARNING: This method bypasses certain internal mechanisms which
          prevent the immediate termination of active sessions,
          and therefore should only be used if the session
          could not be stopped by any other mechanisms.

 Returns true if the action was successful and the stone acted on the
 specified session.  A message will be written to the stone log in this
 case.  Returns false if no action was taken.  No message is added to the
 stone log when false is returned."

<primitive: 774>

sessionId _validateClass: SmallInteger .
((sessionId < 1) or:[sessionId > 10000])
  ifTrue:[ sessionId _error: #rtErrArgOutOfRange args:{ 1 . 10000 } ].
self _primitiveFailed: #_cleanupZombieSession: args: { sessionId }

]

{ #category : 'Private' }
System class >> _clearDeadObjects [
  "If admin, reclaim, or symbol sessions are running, does nothing and returns false.
   Signals an error if the vote state is 'in wsUnion sweep'.
   Otherwise resets Vote state to idle and
   clears the deadObjs, possibleDeadObjs, possibleDeadWsUnion, and possibleDeadWithSymbols
   bitmaps in stone, and returns true ."

^ self _zeroArgPrim: 39

]

{ #category : 'Private' }
System class >> _closureObjects [

"Returns an Array which is the result of enumerating the closure list
 of outer level transaction.
 For use after a failed commit and before subsequent abort, result empty otherwise."

^ self _dirtyListOp: 3 id: 0

]

{ #category : 'Clustering' }
System class >> _clusterImage [

"For all classes in Globals, clusters behavior and descriptions "

| classes nonClasses behaviorBucket descriptionBucket otherBucket |
behaviorBucket := AllClusterBuckets at: 4 .
descriptionBucket := AllClusterBuckets at: 5 .
otherBucket := AllClusterBuckets at: 6  .

classes := Globals select:[:i| i isKindOf: Behavior ] .
nonClasses := Globals reject:[:i| i isKindOf: Behavior ] .
nonClasses removeAssociation: (Globals associationAt: #Globals ).

self clusterBucket: behaviorBucket.
classes associationsDo: [ :each |
    each value _alias . "for debugging"
    each value clusterBehavior.
    ].

self clusterBucket: descriptionBucket.
classes associationsDo: [ :each |
    each value clusterDescription.
    ] .
^ true

]

{ #category : 'Clustering' }
System class >> _clusterUserClassesVisibleFromUserProfile: aUserProfile [

"Cluster all classes contained in symbol list dictionaries for the
 given user profile.  Kernel classes in Globals are not clustered;
 the _clusterImage method should be used to cluster kernel classes."

|behaviorBucket descriptionBucket symbolList|

symbolList := aUserProfile symbolList copy.
symbolList remove: Globals.
symbolList removeDictionaryNamed: #Published ifAbsent: [ ].

"Identify cluster buckets for class and method objects"
behaviorBucket := AllClusterBuckets at: 4.
descriptionBucket := AllClusterBuckets at: 5.
self clusterBucket: behaviorBucket.
symbolList do:[:eachSymbolDictionary|
  eachSymbolDictionary rebuildIfNeeded.
  eachSymbolDictionary cluster.
  eachSymbolDictionary associationsDo:[ :eachAssociation |
    eachAssociation cluster.
    (eachAssociation value isKindOf: Behavior)
      ifTrue:[eachAssociation value clusterBehavior.].
  ].
].

self clusterBucket: descriptionBucket.
symbolList do:[:eachSymbolDictionary|
  eachSymbolDictionary associationsDo:[ :eachAssociation | |val|
  val := eachAssociation value.
  (val isKindOf: Behavior)
    ifTrue:[val clusterDescription.].
  ].
].

]

{ #category : 'Private' }
System class >> _comClearOpcodesEmitted [

^ self _zeroArgPrim: 92

]

{ #category : 'Reduced Conflict Support' }
System class >> _commit: commitMode [

"commitMode is a SmallInteger:

 0 for normal,
 1 for release locks,
 2 for checkpoint and release locks.

 Returns true if commit was read-only or succeeded ,
 false if there was a failure.  See also _localCommit: .  "

| coordinator |

^ (coordinator := self _commitCoordinator) == nil
  ifTrue: [ self _localCommit: commitMode ]
  ifFalse: [ coordinator commit: commitMode ].

]

{ #category : 'Private' }
System class >> _commitCoordinator [

"Returns the commit coordinator stored in temporary session state."

^ self __sessionStateAt: 14

]

{ #category : 'Private' }
System class >> _commitCoordinator: aCoordinator [

"Sets the commit coordinator stored in temporary session state."

^ self __sessionStateAt: 14 put: aCoordinator

]

{ #category : 'Private' }
System class >> _commitNestedTransaction [
  | lev log outerLog rlArray |
  lev := self transactionLevel.
  (self _zeroArgPrim: 147 "commitNestedTransaction" ) ifFalse:[
     InternalError signal:'inconsistent transaction nesting'
  ].
  (rlArray := self __sessionStateAt: 4) ifNil:[
    ^ true "no entries at any level"
  ].
  (log := rlArray atOrNil: lev) ifNil:[
     ^ true "no Rc operations in level we are committing"
  ].
  (outerLog := rlArray atOrNil: lev - 1) ifNil:[
    "no entries yet in parent level, just move nested entries to parent level."
    rlArray at: lev - 1 put: log .
    rlArray size: lev - 1.
    ^ true.
  ].
  outerLog _commitNested: log .  "merge redo log into parent level redo log"
  rlArray at: lev put: nil .

]

{ #category : 'Private' }
System class >> _commitPrintingDiagnostics [

"Commits the current transaction. Answers true if successful,
 otherwise false. If false, it also prints the OOPs for
 each kind of conflict, ordered by kind. This is used in the
 upgrade scripts."

self commitTransaction
  ifTrue: [^true]
  ifFalse: [
    | conflicts |
    conflicts := self transactionConflicts.
    (conflicts at: #commitResult) == #success ifFalse: [
      GsFile gciLogServer: (self conflictReportString: conflicts) .
    ].
    ^false
  ]

]

{ #category : 'Private' }
System class >> _commitResult [

"Private.  Returns a SmallInteger which represents the result of the last
 commitTransaction or continueTransaction operation.  If abortTransaction was
 issued since the last commit or continue then it returns SmallInteger 0.

 See   _primitiveCommit:  for documentation of the return values."

^ self _zeroArgPrim: 30

]

{ #category : 'Private' }
System class >> _committedDataPages [

"Returns an Array of the committed data pages referenced from the session's
 view of GemStone.  The contents of this Array are invalid after the next
 commit or abort."

^ self _zeroArgCateg2Prim: 10

]

{ #category : 'Version Management' }
System class >> _compareGlobalsWith: passiveSourceFileName
writeReportTo: reportFileName
generateFileinTo: fileinFileName [
"returns a string, a report on Smalltalk method differences between this
 version and a previous version.

 If reportFileName is nil, no report file is written. If reportFile is not
 nil, the report string will be written to a file.

 If fileinFileName is nil, no filein file will be generated. If fileinFileName
 is not nil, a topaz filein script will be generated.

 See also System(C)>>_generatePassiveSourceFileTo:
"
| prevClasses prevClassesFile report reportFile lf addedMeths changedMeths
  deletedMeths addedClasses deletedClasses fileinFile
  theCls prevDict clsMethStr |

lf := Character lf.
report := String new.
addedMeths := { } .
changedMeths := { } .
deletedMeths := { } .
addedClasses := { } .
deletedClasses := { } .

fileinFileName ifNotNil: [
  fileinFile := GsFile openWriteOnServer: fileinFileName.
  (fileinFile == nil) ifTrue: [
    ^'Can''t open ', fileinFileName, ' for writing.'
  ].
].

prevClassesFile := GsFile openReadOnServer: passiveSourceFileName.
prevClassesFile ifNil: [
  ^'Can''t open ', passiveSourceFileName, ' for reading.'
].

"read in the prevClasses from passivated object file"
prevClasses := (PassiveObject newOnStream: prevClassesFile) activate.
prevClasses ifNil: [
  ^'Failed to activate source dictionaries from ', passiveSourceFileName, '.'
].

"iterate classes in Globals and compare method source strings"
Globals do: [ :ea | | cls |
  (ea isBehavior) ifTrue: [ | className |
    cls := ea thisClass .
    className := cls name .
    (prevClasses includesKey: className) ifFalse: [
      addedClasses add: className.
      fileinFile ifNotNil: [
        cls fileOutClassOn: fileinFile.
      ].
    ] ifTrue: [ | methsBlk |
      methsBlk := [ :mDict |
        mDict keysAndValuesDo: [ :selector :aMeth | | envId aName |
          envId := aMeth environmentId .
          aName := envId == 0 ifTrue:[ selector ] ifFalse:[
             ':env_', envId asString , ':', selector asString
          ].
          " check to see if same method exists in prevClasses"
          (prevDict includesKey: aName) ifFalse: [
            addedMeths add: className, '>>', clsMethStr, aName .
            fileinFile ifNotNil: [
              fileinFile nextPutAll: (theCls _fileOutMethod: selector environmentId: envId ).
            ].
          ] ifTrue: [
            " method exists in both versions, do a diff."
            (aMeth sourceString = (prevDict at: aName)) ifFalse: [
              changedMeths add: className, '>>', clsMethStr, aName.
              fileinFile ifNotNil: [
                fileinFile nextPutAll: (theCls fileOutMethod: selector environmentId: envId).
              ].
            ].
          ].
        ].
      ].

      clsMethStr := '' .
      prevDict := (prevClasses at: className) at: 1 .
      theCls := ea .
      theCls persistentMethodDictsDo: methsBlk .

      clsMethStr := '(C)' .
      prevDict := (prevClasses at: className) at: 2 .
      theCls := ea class .
      theCls persistentMethodDictsDo: methsBlk .
    ]
  ]
]. "Globals do:"

"Now check to see if there are any deleted methods or classes"
prevClasses keysAndValuesDo: [ :clsNam :v |
  ((Globals includesKey: clsNam) and: [(Globals at: clsNam) isBehavior]) ifFalse: [
    deletedClasses add: clsNam.
  ] ifTrue: [ | meths envPrefix prefixSize |
    meths := v at: 1 .
    clsMethStr := '' .
    theCls := Globals at: clsNam .
    envPrefix := ':env_' .
    prefixSize := envPrefix size .
    2 timesRepeat:[
      meths keysDo:[ :aName | | envId selectr |
        envId := 0 .
        selectr := aName .
        (aName at: 1 equals: envPrefix) ifTrue:[ | endIdx |
          endIdx := aName indexOf: $: startingAt: 2 .
          envId := Integer fromString:( aName copyFrom: prefixSize + 1 to: endIdx - 1 ).
          selectr := aName copyFrom: endIdx + 1 to: aName size .
        ].
        (theCls includesSelector: selectr environmentId: envId) ifFalse:[
          deletedMeths add: clsNam, '>>', clsMethStr, aName.
          fileinFile ifNotNil: [
            fileinFile nextPutAll: 'doit', lf,
	      clsNam , ' removeSelector: #', selectr, ' environmentId:', envId asString ,
		lf, $%, lf.
          ].
        ].
      ].
      meths := v at: 2 .
      clsMethStr := '(C)' .
      theCls := theCls class .
    ].
  ].
].
fileinFile ifNotNil:[ fileinFile close ].

"put together the report"
report := '====', lf, 'Image change report', lf, '====', lf.
(addedClasses size ~~ 0) ifTrue: [
  report := report, '----', lf, 'Added Classes:', lf, '----', lf.
  (addedClasses sortAscending) do: [ :ea | report := report, ea, lf ].
].
(deletedClasses size ~~ 0) ifTrue: [
  report := report, '----', lf, 'Deleted Classes:', lf, '----', lf.
  (deletedClasses sortAscending) do: [ :ea | report := report, ea, lf ].
].
(addedMeths size ~~ 0) ifTrue: [
  report := report, '----', lf, 'Added Methods:', lf, '----', lf.
  (addedMeths sortAscending) do: [ :ea | report := report, ea, lf ].
].
(deletedMeths size ~~ 0) ifTrue: [
  report := report, '----', lf, 'Deleted Methods', lf, '----', lf.
  (deletedMeths sortAscending) do: [ :ea | report := report, ea, lf ].
].
(changedMeths size ~~ 0) ifTrue: [
  report := report, '----', lf, 'Changed Methods', lf, '----', lf.
  (changedMeths sortAscending) do: [ :ea | report := report, ea, lf ].
].

reportFileName ifNotNil:[
  reportFile := GsFile openWriteOnServer: reportFileName.
  reportFile ifNotNil:[
    reportFile nextPutAll: report; close.
  ].
].
^report

]

{ #category : 'Private' }
System class >> _comPrintOpcodesEmitted [

^ self _zeroArgPrim: 87

]

{ #category : 'Private' }
System class >> _comPrintOpcodesNotEmitted [

^ self _zeroArgPrim: 88

]

{ #category : 'Configuration File Access' }
System class >> _configConstantsDict [

"Returns a Dictionary of internal configuration constants that
 are fixed at compilation of the gem and stone executables. The dictionary
 keys are Strings.  Its values are SmallInteger configuration constants."


| anArray result |
anArray := self _zeroArgCateg2Prim: 8 .
result := StringKeyValueDictionary new .
1 to: anArray size do:[ :j | | anAssoc |
  anAssoc := anArray at: j .
  result add: anAssoc .
  ].
^ result

]

{ #category : 'Configuration File Access' }
System class >> _configFileParameterDict [

"Returns a Dictionary of names for configuration file parameters.  The
 dictionary keys are Symbols.  Its values are SmallInteger configuration IDs."

| result cfgId cfgName |
result := SymbolKeyValueDictionary new .
cfgId := 0 .
[
  cfgName := self _configParameterName: cfgId kind: $C .
  cfgName == nil ifFalse:[
    result at: cfgName put: cfgId .
    cfgId := cfgId + 1 .
    ] .
  cfgName == nil
  ] untilTrue .

cfgId := -1 .
[
  cfgName := self _configParameterName: cfgId kind: $C .
  cfgName == nil ifFalse:[
    result at: cfgName put: cfgId .
    cfgId := cfgId - 1 .
    ] .
  cfgName == nil
  ] untilTrue .

^ result

]

{ #category : 'Configuration File Access' }
System class >> _configParameterName: anInt kind: aChar [

"Returns a String, which is the name of the configuration parameter whose
 internal identifier is anInt.  Returns nil if anInt is out of range.

 aChar must be one of
   $C - anInt is identifier of a configuration parameter
   $V - anInt is identifier of a version information item
"

<primitive: 342>
anInt _validateClass: SmallInteger .
^ self _primitiveFailed: #_configParameterName:kind: args: { anInt . aChar }

]

{ #category : 'Configuration File Access' }
System class >> _configurationAt: cfgId isStone: aBool kind: aKind [

"Returns the configuration value specified. If aBool is true, get the
 configuration value from the Stone process, otherwise from the current
 session.

 cfgId must be a SmallInteger; if >= 0, returns the value of the specified
 configuration parameter;  if < 0, returns the value of the specified
 runtime control parameter.

 aKind must be either $C (for configuration info) or $V (for version info)"

<primitive: 338>
aBool _validateClass: Boolean .
cfgId _validateClass: SmallInteger .
aKind == $C ifTrue:[
  ConfigurationParameterDict keyAtValue: cfgId ifAbsent:[
     cfgId _error: #rtErrArgOutOfRange
     ].
  ]
  ifFalse:[
  VersionParameterDict keyAtValue: cfgId ifAbsent:[
     cfgId _error: #rtErrArgOutOfRange
     ].
  ].
self _primitiveFailed: #_configurationAt:isStone:kind:
     args: { cfgId . aBool . aKind } .
self _uncontinuableError

]

{ #category : 'Configuration File Access' }
System class >> _configurationReport: isStone [

"Private."

| result |
result := SymbolDictionary new .
ConfigurationParameterDict keysAndValuesDo:[ :aName :anId |
  (self _configurationAt: anId isStone: isStone kind: $C) ifNotNil:[:val |
    result at: aName put: val
    ].
  ].
^ result

]

{ #category : 'Private' }
System class >> _cpuClockTicksPerSecond [

"Calls the UNIX function sysconf(_SC_CLK_TCK) to determine the number
 of clock ticks in a second.  Returns zero if the call is not supported."

^self _zeroArgPrim: 57

]

{ #category : 'Private' }
System class >> _currentSessionProfiles [

"Returns an Array containing pairs of sessionId, UserProfile
 for all sessions currently logged in.
 Some sessionIds may have a userProfile of nil, including those which
 are part way through login.
 logsender processes have userProfile==1 ,
 logreceiver processes have userProfile==2.

 This method requires SessionAccess privilege if there is more
 than one session logged in."

| sessId result batch |
GsSession currentSession isSolo ifTrue:[
  ^ { 1 . System myUserProfile }
].
sessId := 1 .
result := { }  .
[ sessId > 0 ] whileTrue:[
  batch := self _sessionProfilesStartingAt: sessId .
  batch size > 0 ifTrue:[
    result addAll: batch .
    sessId := (batch at: (batch size - 1)) + 1 .
  ] ifFalse:[
    sessId := -1 . "done"
  ].
].
^ result

]

{ #category : 'Session Control' }
System class >> _currentSessionsIncludingLogout [

"Returns an Array of SmallIntegers corresponding to all of the sessions
 currently running on the GemStone system.  Includes sessions in before-login
 or after-logout states.
 Gets the result from the stone process."

^self _zeroArgPrim: 83

]

{ #category : 'Session Control' }
System class >> _currentUserSessionCountIncludingLogout [
"Return a SmallInteger which is the number of user sessions present in the system.
 Garbage collection sessions, the Symbol Gem, and the Page Manager Gem are not
 included in the count.
 Includes sessions in before-login or after logout state.
 Gets the result from the stone process."

^self _zeroArgPrim: 70

]

{ #category : 'Private' }
System class >> _deadNotReclaimed [

"Private, deprecated.  Returns a GsBitmap that contains the current contents of the
 global set of objects which have been identified as dead but have not been reclaimed."

  self deprecated: 'System class>>_deadNotReclaimed: deprecated v3.6. Use GsBitmap class>>_deadNotReclaimed'.

^ GsBitmap _deadNotReclaimed

]

{ #category : 'Private' }
System class >> _debugInfoForPage: anInt [

"If anInt is the page ID of a data page, return an Array of 12 elements:

  1 - Time page was written in seconds since January 1, 1970 (SmallInteger)
  2 - Session ID that wrote the page. (SmallInteger)
  3 - Number of valid objects in the page. (SmallInteger)
  4 - Number of invalid objects in the page. (SmallInteger)
  5 - Array of valid object IDs (Array of SmallIntegers)
  6 - Array of classes of valid objects.
  7 - Array of invalid object IDs (Array of SmallIntegers)
  8 - Array of classes of invalid objects.
  9 - Array of correct page IDs for invalid objects.
 10 - Array of correct classes for invalid objects.
 11 - Array of session IDs that wrote the correct pages in element 9.
 12 - Array of timestamps that the correct pages in element 9 were written.

 This method is intended to be used only by GemStone staff to aid in the debugging of
 repository problems.

 This method may only be executed by SystemUser.

 Returns nil if the page is not a data page."

<primitive: 870>

anInt _validateClass: SmallInteger .
self _primitiveFailed: #_debugInfoForPage: args: { anInt }

]

{ #category : 'Private' }
System class >> _debugToggle [
"returns self. calls  extern C void OmDebugToggle
  which can be registered with callgrind."

^ self _zeroArgPrim: 197
]

{ #category : 'Deprecated' }
System class >> _deleteServerFile: aFileSpec [

<primitive: 346>
aFileSpec _validateClass: String.
self _primitiveFailed: #_deleteServerFile: args: { aFileSpec }
]

{ #category : 'Session Control' }
System class >> _descriptionOfSessionSerialNum: serialNum sessionId: aSessionId [

"Returns an Array describing the session identified by the arguments.
 The session is looked up by serial number if serialNum is > 0.  Otherwise
 it is looked up by sessionId.

 See System (C) | descriptionOfSession: for documentation on the contents of the
 result Array.
 Requires SessionAccess privilege if the session being looked up is not
 the current session."

<primitive: 334>
aSessionId _validateClass: SmallInteger.
^ self _primitiveFailed: #_descriptionOfSessionSerialNum:sessionId:
       args: { serialNum . aSessionId }

]

{ #category : 'Private' }
System class >> _detailedConflicts [
  "Returns nil or an Array information about which commits by other sessions
   caused commit conflicts.  They are ordered by youngest commit first.

    { attemptedCommitTimeMs . previousViewCrPage .
      { sessionIdInteger . commitTimeMsInteger . aUserProfile . wwArray . depArray . crPage }
      ...
    }
    wwArray is nil or an Array of objects with write-write conflicts.
    depArray is nil or an Array of objects with dependency conflicts.

    wwArray and depArray are limited to the first 100 objects

    The time values are Integers, milliseconds since 1 Jan 1970 UTC .
  "

^  self __sessionStateAt: 24

]

{ #category : 'Private' }
System class >> _dirtyListOp: opcode id: listId [

"execute an operation on one of the object manager dirty lists
 opcode 1  not supported (was rollBackDirtyList in v3.0 )
 opcode 2  enumerateDirtyList.
 opcode 3  enumerate closureList, for use after failed commit and before abort , listId=0

   listId may be -1 (current),
    or a specific dirty list where listId is  (System transactionLevel - 1)

 Not-tranlogged objects are no longer segregated in a separate list
"

<primitive: 276>
self _primitiveFailed: #_dirtyListOp:id: args: { opcode . listId } .
self _uncontinuableError

]

{ #category : 'Private' }
System class >> _disableAddToDirtySets: aBoolean [

"Disable/reenable additions to the GCI export lists .
 Returns previous state of the flag"

<primitive: 569>
aBoolean _validateClass: Boolean .
self _primitiveFailed: #_disableAddToDirtySets: args: { aBoolean }

]

{ #category : 'Private' }
System class >> _disableCommitsWithReason: aString untilAbort: aBool [

"Private - this method should not be called directly.  Call the #disableCommitsWithReason:
 method instead of this one."

<primitive: 701>
aBool _validateClass: Boolean .
aString == nil
  ifFalse:[ aString _validateClass: String].
self _primitiveFailed: #_disableCommitsWithReason:untilAbort:
     args: { aString . aBool }

]

{ #category : 'Private' }
System class >> _disallowCommitClassModFailure [

"Disallows subsequent attempts due to failure during class modification."
^ self _zeroArgPrim: 37

]

{ #category : 'Reduced Conflict Support' }
System class >> _disallowSubsequentCommit: traceInt [
  "Forces transaction status to be WORK_COMMIT_RETRY_FAILURE .
   Abort will clear this state and allow a subsequent Begin, Commit ."
  "GsFile gciLogServer:'_disallowSubsequentCommit: ' , traceInt asString .""uncomment to debug"
  ^ self _zeroArgPrim: 22

]

{ #category : 'Private' }
System class >> _dumpFrameData [

"Private.  Prints out shared page cache frame data to the session log file."

^ self _zeroArgPrim: 58

]

{ #category : 'Private' }
System class >> _exitBypassReadAuth [

"For use by indexing and query subsystem only.

 Any in-memory objects which were faulted in while bypass was enabled by
 a preceding _bypassReadAuth, and for which a read-authorization error would
 have otherwise occurred, will be marked not-valid, so that subsequent
 access will get a read-authorization error.

 Has no effect if current user is SystemUser."

<primitive: 692>
self _primitiveFailed: #_exitBypassReadAuth

]

{ #category : 'Reduced Conflict Support' }
System class >> _failedCommitWithResult: commitResult [

"Increments the failedCommitCount stat for the session and sets
 the lastFailedCommitReasonCode stat to commitResult.
 Returns true."

  ^ self _primFailedCommitWithResult: commitResult

]

{ #category : 'Private' }
System class >> _findAllPublishedOn: machine [
  "Description:
     Find all published servers on the given machine.
   Input:
     machine <String or nil>: name of machine to find on or nil if local.
   Result <Array or nil>:
     nil if no servers were found.
     instance of Array that contains one element for each server found.
     Each element will itself be an Array with the contents described
     for the _findPublished:on: method.
   Exceptions:
     Illegal input raises various errors.
     Error 2367 raised if low level C code fails with system error.
  "

<primitive: 499>
self _primitiveFailed: #_findAllPublishedOn: args: { machine }

]

{ #category : 'Private' }
System class >> _findPublished: server on: machine [
  "Description:
     Find a published server on the given machine.
   Input:
     server <String>: name of server to findunpublish.
     machine <String or nil>: name of machine to find on or nil if local.
   Result <Array or nil>:
     nil if server was not found.
     instance of Array if server was found. The array contents are:
       1. <String>: the name of the server
       2. <SmallInteger>: the port number of ther servers listening socket.
       3. <Integer>: the process id of the server's main process.
       4. <Boolean>: true if process exists, false if not.
       5. <String>: type of server: 'Stone', 'NetLDI', 'Cache', 'Broker',
                                    'Agent', 'RCPD', 'Service-Stone',
                                    'Service-NetLDI', or '?'.
       6. <String>: name of user that created the server.
       7. <String>: time server was created.
       8. <String>: name of machine server is running on.
       9. <String or nil>: version of GemStone used by server.
       10.<String or nil>: GemStone product directory.
       11.<String or nil>: log file name of server.
       12.<String or nil>: startup options of server.
       13.<String or nil>: GemStone system configuration file used by server.
       14.<String or nil>: GemStone executable configuration file used by server.
       15.<String or nil>: linux executable path, value of /proc/<processId>/exe
   Exceptions:
     Illegal input raises various errors.
     Error 2367 raised if low level C code fails with system error.
  "

<primitive: 498>
self _primitiveFailed: #_findPublished:on: args: { server . machine }

]

{ #category : 'Shared Cache Management' }
System class >> _formatCacheReport: anArray [
"given an Array produced by remoteCachesList or midLevelCachesList
 return a String containing one line per element of the list."

| rpt body formatBlk header anyX509 stnLine |
formatBlk := [:aCache :aRpt | |kind kindStr hostAgentSessionId haStr aSize |
  aRpt addAll: ((aCache at: 1) width: 30) . "hostname"
  hostAgentSessionId := aCache atOrNil: 7 .
  kind := aCache at: 2 .
  kind == nil ifTrue:[ kindStr := 'stone' ] ifFalse:[
  kind == true ifTrue:[
    kindStr := hostAgentSessionId > 0 ifTrue:[ 'x509mid'] ifFalse:[ 'mid'].  ] ifFalse:[
  kind == false
     ifTrue:[ kindStr := hostAgentSessionId > 0 ifTrue:[ 'x509remote'] ifFalse:['remote']]
    ifFalse:[ kindStr := 'kind' . haStr := hostAgentSessionId.  "header line"].
  ]].
  kindStr := String withAll: kindStr .
  aRpt addAll: (kindStr width: 10) .
  aRpt addAll: ((aCache at: 3) asString width: -11 ). "sessions"
  aRpt addAll: ((aCache at: 6) asString width: -13 ). "sessions using as midCache"
  aRpt addAll: ((aCache at: 4) asString width: -10 ). "maxsessions"
  aSize := aCache at: 5 .
  aSize _isSmallInteger ifTrue:[ aSize := aSize // 1024 ].
  aRpt addAll: (aSize asString width: -10 ). "sizeMB"
  haStr ifNil:[
    hostAgentSessionId > 0 ifTrue:[
      haStr := '    ', hostAgentSessionId asString . anyX509 := true .
      (aCache atOrNil: 9 ) ifNotNil:[ :midHaSess |
         midHaSess > 0 ifTrue:[
           haStr addAll: '  midHa=' ; addAll: midHaSess asString .
         ].
      ].
    ] ifFalse:[ haStr := String new ]
  ].
  aRpt addAll: haStr ; lf .
].
body := String new .
1 to: anArray size do:[:k || aCache |
  aCache := anArray at: k .
  formatBlk value: aCache value: body .
].
header := { 'hostname' . 'kind' . 'sessions' . 'maxSess' . 'sizeMB' . 'mid-sess''s' }.
anyX509 ifNil:[ header add:''] ifNotNil:[ header add: '  hostAgentSession' ].
header := header collect:[ :s | s copy ]. "make strings variant"
rpt := String new .
formatBlk value: header value: rpt . "header line"
rpt addAll: body .
header := { System stoneVersionReport at: #nodeName .
            nil . System currentSessionCount .
            System stoneConfigurationAt: 'STN_MAX_SESSIONS' .
            (System stoneConfigurationAt: 'SHR_PAGE_CACHE_SIZE_KB'). 0 . 0 } .
stnLine := String new .
formatBlk value: header value: stnLine . "stone cache line"
rpt addAll: stnLine .
^ rpt

]

{ #category : 'Private' }
System class >> _gcGemConfig: isReclaim symb: configSymbol toValue: aVal [

"Primitive implementation to fetch or set gcGem configs at runtime.
 If the gcUserGlobals argument is passed, it also updates the corresponding persistent
 entry in the GcUser's UserGlobals and commits.

 Use nil for the aVal argument to fetch the current value."

<primitive: 178>
isReclaim _validateClass: Boolean.
configSymbol _validateClass: Symbol.
aVal _validateClass: SmallInteger.
^ self _primitiveFailed: #_gcGemConfig:symb:toValue:
       args: { isReclaim . configSymbol . aVal }

]

{ #category : 'Transaction Control' }
System class >> _gciCommit [
  "used by GciCommit"

  (self _zeroArgPrim: 5) <= 1 ifTrue:[ "level 0 or 1"
    (self _commit: 3"from GCI") ifFalse:[
      ^ false "commit conflicts"
    ].
  ] ifFalse:[
    self _commitNestedTransaction.
  ] .
  ^ true

]

{ #category : 'Private' }
System class >> _gcLockKindToStr: anInt [
  | val |
  val := #( 'free'
       'mfc'
       'fdc'
       'epoch'
       'wsUnionSweep'
       'restore'
       'backup'
       'reposScan') atOrNil: (anInt"zero based" + 1).
  ^ val ifNil:[ 'unknown' ].
]

{ #category : 'Version Management' }
System class >> _gemBuildShaString [

 "Private. result is first 64 bits of the git SHA of the build as a Hex string"
 | str count |
 str :=  (self _zeroArgCateg2Prim: 1) asHexString .
 count := 16 - str size .
 count > 0 ifTrue:[ | pad |
   pad := String new .
   count timesRepeat:[ pad add: $0 ].
   ^ pad , str
 ].
 ^ str

]

{ #category : 'Private' }
System class >> _gemCommitConflictDetails [
  "Optimized implementation of  (self gemConfigurationAt: #GemCommitConflictDetails) ,
   returns a SmallInteger, values
     >= 1 enables saving data for detailedConflictReportString 
     2 enables printing conflicts and Rc replay to gem log or topaz -l output 
  "
^ self _zeroArgPrim: 72

]

{ #category : 'Version Management' }
System class >> _gemVersion [

"Private."

^ self _zeroArgCateg2Prim: 3

]

{ #category : 'Version Management' }
System class >> _gemVersionNum [

 "Private. result is of form 30501 for 3.5.1"

 ^ self _zeroArgCateg2Prim: 2

]

{ #category : 'Version Management' }
System class >> _generatePassiveSourceFileTo: fileName [
"generate a passivated data structure containing all source code for classes
 in Globals. The file generated by this method can be used by
 System(C)>>_compareGlobalsWith:writeReportTo:generateFileinTo: to generate
 a report and/or filein script to reconcile kernel class differences between
 two versions of the image. If fileName is nil, file will be given a useful
 name like '1.0.0-sourceStrings.obj'.
"
|ver classes outfile outfilename processedClasses |
fileName ifNil: [ ver := self gemVersionReport at: #gsRelease.
                  ver := (ver copyReplaceAll: ' ' with: '').
                  outfilename := ver, '-sourceStrings.obj' ]
       ifNotNil: [outfilename := fileName].
classes := Dictionary new.
processedClasses := IdentitySet new .
Globals do: [ :ea | | cls methSrcDict classMethSrcDict |
  (ea isKindOf: Behavior) ifTrue: [  | methsBlk |
    cls := ea thisClass .
    (processedClasses includes: cls) ifFalse:[
      processedClasses add: cls .
      methSrcDict := Dictionary new.
      classMethSrcDict := Dictionary new.
      classes at: ea name put: { methSrcDict . classMethSrcDict } .
      methsBlk := [ :mDict :targetDict |
        mDict keysAndValuesDo: [ :k :aMeth | | envId aName |
          envId := aMeth environmentId .
          aName := envId == 0 ifTrue:[ k ] ifFalse:[ k asString ,':env_', envId asString].
          targetDict at: aName put: aMeth sourceString.
        ].
      ].
      ea _persistentMethodDicts do:[:aDict | methsBlk value: aDict value: methSrcDict].
      ea class _persistentMethodDicts do:[:aDict | methsBlk value: aDict value: classMethSrcDict].
    ].
  ].
].
outfile := GsFile openWriteOnServer: outfilename.
outfile ifNil: [^'error opening outfile', outfilename].
(PassiveObject passivate: classes toStream: outfile)
     ifNil:[ 'error writing outfile', outfilename].
outfile close.
^true

]

{ #category : 'Private GC' }
System class >> _generationScavenge [

"Returns true if the scavenge succeeded, false if it got promoted
 to a markSweep

 Explicitly triggers in-memory scavenge collection of temporary objects.
 Should only be used when debugging internal problems. Frequent use
 can degrade performance."

^ self _zeroArgPrim: 14

]

{ #category : 'Private GC' }
System class >> _generationScavenge_vmMarkSweep [

"Execute an in-memory scavenge and an in-memory markSweep.
 Should only be used when debugging internal problems. Frequent use
 can degrade performance."

self _generationScavenge ifTrue:[
  self _vmMarkSweep
  ].

]

{ #category : 'Private' }
System class >> _getMidCachePort: aHostName [

"Get the mid cache IP and port from stone
 for an already running mid cache on the
 specified host, without actually connecting to the mid cache.

 returns an Array  { sourceMidCacheIp . sourceMidCachePort }
 or a String with error details.
"

<primitive: 1104>
aHostName _validateInstanceOf: String .
self _primitiveFailed: #_getMidCachePort: args: { aHostName }

]

{ #category : 'Private' }
System class >> _hostAgentVersionString [
  "Returns a version and build string used by the hostagent."
  ^ self _zeroArgCateg2Prim: 4

]

{ #category : 'Private' }
System class >> _hostCallDebugger [

"Private, for debugging.
 calls HostCallDebugger in the VM which causes a core dump in customer
 executables and sleeps in non-customer executables.  Depending on the
 value of STN_HALT_ON_FATAL_ERR in stone's config file, this method may also
 cause the stone process to halt."

^ self _zeroArgPrim: 25

]

{ #category : 'Host System Access' }
System class >> _hostProcessExists: aProcessId [
  "Returns true if aProcessId exists as a process on this sessions' machine,
   false otherwise."

<primitive: 1023>
aProcessId _validateClass: SmallInteger .
^ self _primitiveFailed: #_hostProcessExists:

]

{ #category : 'Host System Access' }
System class >> _hostTimes [
" return a 4 element Array holding the elements of the struct tms result
  from
      clock_t times(struct tms *buffer)

  The Array elements are tms_utime, tms_stime, tms_cutime, tms_cstime ,
  and each element ia s Float in units of seconds.

  tms_utime is the CPU time  used  while  executing
  instructions in the user space of the calling process.

  tms_stime is the CPU time used by the  system  on
  behalf of the calling process.

  tms_cutime is the sum of the  tms_utime  and  the
  tms_cutime of the child processes.

  tms_cstime is the sum of the  tms_stime  and  the
  tms_cstime of the child processes.
"
^ self _zeroArgPrim: 119

]

{ #category : 'Private' }
System class >> _hostWaitForDebugger [

"Private, for debugging,
 calls HostExplicitWaitForDebugger, in the VM which puts this session's
 VM into an infinite wait for gdb or dbx to be attached.  Does not
 notify stone, nor trigger core dump."

^ self _zeroArgPrim: 16
]

{ #category : 'Private' }
System class >> _ignoreModTracking [

"Private. Disables Modification Tracking."

<primitive: 516>
self _primitiveFailed: #_ignoreModTracking

]

{ #category : 'Version Management' }
System class >> _imageVersion [

"Returns a String, which contains the history of the image."

^Globals at: #DbfHistory.

]

{ #category : 'Private' }
System class >> _initMap: aMap [

"Initializes the underlying data structures used to perform the mapping
 of passivated objects.
 aMap must be a SmallInteger  1 or 2 .
 Returns a SmallInteger serial number to be used for calls
 to _inMap:at:putIfAbsent:serialNum:
"

<primitive: 319>
aMap _validateClass: SmallInteger .
self _primitiveFailed: #_initMap: args: { aMap }

]

{ #category : 'Private' }
System class >> _inMap: aMap at: anOop putIfAbsent: newValue serialNum: serialNum [

"If there is no current value at anOop in the passive object map aMap,
 (that is, it is absent), then this method stores newValue and returns newValue.
 If the current value at anOop is valid however, it returns the value and does
 not store the newValue in the map.

 The argument aMap must be in the range 1 - 4
 and newValue must be a positive SmallInteger.

 anOop must not be a special object.
 serialNum must be a SmallInteger
"

<primitive: 320>
aMap _validateClass: SmallInteger .
(aMap < 1 or:[ aMap > 2]) ifTrue:[
  OutOfRange new name: #aMap min: 1 max: 2 actual: aMap ; signal
].
anOop isSpecial ifTrue:[ ArgumentError signal:'special object not allowed' ].
newValue _validateClass: SmallInteger .
serialNum _validateClass: SmallInteger .
newValue < 1 ifTrue:[ newValue _error: #rtErrArgNotPositive ].
self _primitiveFailed: #_inMap:at:putIfAbsent:serialNum:
     args: { aMap . anOop . newValue . serialNum }

]

{ #category : 'Private' }
System class >> _inProtectedMode [

 "Return true if in protected mode"

  ^self _zeroArgPrim: 84

]

{ #category : 'Private' }
System class >> _internalMessageForErrorNum: aNumber [

"Private.  Returns a String describing the error message generated from the
 errmsg.c module for GemStone error numbered aNumber.  Returns nil if no such
 error number or message exists in errmsg.c."

<primitive: 384>
aNumber _validateClass: SmallInteger .
self _primitiveFailed: #_internalMessageForErrorNum: args: { aNumber }

]

{ #category : 'Error Handling' }
System class >> _lastGsErrorNumber [

"Return the value of the last error number.  Also clears the error number back to zero."
^self _zeroArgPrim: 47

]

{ #category : 'Private' }
System class >> _lastReservedOopNumber [
  "Used in image upgrade, result is a constant from the VM."

^ self _zeroArgPrim: 115

]

{ #category : 'Transaction Control' }
System class >> _localAbort [

"Rolls back all modifications made to committed GemStone objects and provides
 the session with a new view of the most recently committed GemStone state.

 These operations are performed whether or not the session was previously in a
 transaction.  If the transaction mode is set to #autoBegin, then a new
 transaction is started.  If the transaction mode is set to #manualBegin, then
 a new transaction is not started.

 Signals an error if  GsSession isSolo==true .  "
 | res |
 self _pendingCommitAbort .  "Signal error to application if appropriate"
 res := self _primitiveAbort: false . "Abort the current transaction"
 (self _zeroArgPrim: 82 "_sessionMethodsChangedForAbort") ifTrue:[ | policy|
   (policy := GsPackagePolicy current) enabled ifTrue:[
     policy refreshSessionMethodDictionary
  ].
 ].
 ^ res

]

{ #category : 'Transaction Control' }
System class >> _localBeginTransaction [

"Starts a new transaction for the session."

self _pendingCommitAbort .  "Signal error to application if appropriate"
self _primitiveBegin

]

{ #category : 'Reduced Conflict Support' }
System class >> _localCommit: commitMode [

  "commitMode is a SmallInteger:

   0 for normal,
   1 for release locks,
   2 for checkpoint and release locks,
   3 for normal commit called from GciCommit .

   Returns true if commit was read-only or succeeded ,
   false if there was a failure."

  | commitResult actualMode |
  self _processDeferredGciUpdates .
  commitMode == 3 ifTrue:[
    "called from GciCommit, no pending commit action."
    actualMode := 0
  ] ifFalse:[
    self _pendingCommitAbort .
    actualMode := commitMode
  ] .
  commitResult := [ self _primitiveCommit: actualMode
		  ] onException: Error do:[:ex |
		    "ensure primitive commit is not on stack, and won't be retried
		     on continuation of execution."
		    ex pass .
		    5
		  ].
  commitResult <= 0 ifTrue: [
    (self _zeroArgPrim: 81 "_sessionMethodsChangedForCommit") ifTrue:[ | policy |
      (policy := GsPackagePolicy current) enabled ifTrue:[
        policy refreshSessionMethodDictionary
      ].
    ].
    ^ true
  ] ifFalse: [
    commitResult == 1 ifTrue: [ | resolveCommitResult  gemCommitConflictDetails |
      gemCommitConflictDetails := self _gemCommitConflictDetails > 1 .
      gemCommitConflictDetails ifTrue: [ GsFile 
				gciLogServer:'---- Conflicts before replay';
				gciLogServer: self detailedConflictReportString;
				gciLogServer: '---- BEGIN replay' ].
      SessionTemps current at: #LogEntry_redoCount put: 0 .
      resolveCommitResult := self _resolveRcConflictsForCommit: actualMode.
      gemCommitConflictDetails ifTrue: [ 
			  GsFile gciLogServer:'---- result after replay (', resolveCommitResult printString , ' ) ' 
      ]. 
      ^ resolveCommitResult
    ].
    self _failedCommitWithResult: commitResult . "48387"
  ].
  self _disallowSubsequentCommit: 1 .
  ^ false
]

{ #category : 'Private' }
System class >> _locale [
"Returns a Locale object setup according to the session's Locale configuration."

^ self _zeroArgPrim: 90.

]

{ #category : 'Transaction Control' }
System class >> _localTransactionMode: newMode [

"Sets a new transaction mode for the current GemStone session and exits the
 previous mode by aborting the current transaction.  Valid arguments are
 #autoBegin, #manualBegin and #transactionless."

self _pendingCommitAbort .
^ self _primTransactionMode: newMode

]

{ #category : 'Setting Locks' }
System class >> _lock: anObject kind: lockKind  autoRelease: autoReleaseBoolean [

"Attempt to lock a single object.
 lockKind argument must be a SmallInteger , one of
    2  read
    3  write
    4  (not used)
    5  wait for Application write lock 1
    ...
   14  wait for Application write lock 10

 lockKinds 5..14  may be used with only one unique persistent object
 during the life of a stone process .  The first use of that lockKind
 registers anObject with that transient queue in the stone process, and
 subsequent uses of that lockKind with an object not identical to anObject
 will generate an error.  An error will be generated if anObject is
 not committed.  The queues are reinitialized each time stone process restarts.

 Any attempt to lock AllSymbols generates an error.

 This method returns a SmallInteger.
 Attempt to lock a special object always returns  2075  denied .
 If lockKind is < 2 or > 14 ,   returns 2071 .

 read and write requests return one of
   1  granted
   2074  dirty (object written by other session since start of this transaction)
   2075  denied (locked by another session)

 'wait for' requests return one of
   2074  dirty (object written by other session since start of this transaction)
   2075  denied (locked by another session)
   2418  deadlock
   2419  timeout  (per STN_OBJ_LOCK_TIMEOUT config parameter)
   2420  a different object is already registered with the specified
         Application write lock queue

 If autoReleaseBoolean == true and result is either 1 or 2074 ,
 then the object is added to  both the CommitReleaseLocksSet
 and CommitOrAbortReleaseLocksSet hidden sets.
"

<primitive: 97>
self _primitiveFailed: #_lock:kind:autoRelease:
     args: { anObject . lockKind . autoReleaseBoolean } .
self _uncontinuableError

]

{ #category : 'Setting Locks' }
System class >> _lockAll: aCollection kind: lockKind [

"Attempts to lock all the objects in aCollection.

 The lockKind argument specified the kind of lock and
 must be a SmallInteger, one of
    2  read
    3  write

 An error is generated and no locks are obtained
 if aCollection is a byte format object.

 An error is generated and no locks are obtained if any element
 of aCollection is identical to AllSymbols .

 Returns the receiver if all locks were granted and not dirty .
 Otherwise returns an Array of size 3 with elements :
   1.  An Array of objects that could not be locked.
   2.  An Array of objects that were locked but whose locks are dirty.
   3.  An empty Array, retained for backward compatibility with old GemStone versions.
"

<primitive: 100>
"primitive fails if aCollection is not a kind of SequenceableCollection
 or IdentityBag "
| anArray |
anArray := aCollection asArray.
anArray == aCollection ifFalse:[
  ^ self _lockAll: anArray kind: lockKind
  ] .
aCollection _validateClass: Collection.
lockKind _validateClass: SmallInteger .
self _primitiveFailed: #_lockAll:kind: args: { aCollection . lockKind } .
self _uncontinuableError

]

{ #category : 'Setting Locks' }
System class >> _lockError: errNum obj: anObj details: details [

"Translate from SmallInteger to Symbol, for error reporting."
"Ugh."

| ex sym |
ex := LockError new _number: errNum ; object: anObj ; details: details .

(errNum == 2075) ifTrue: [  sym := #lockErrDenied ] ifFalse:[
(errNum == 2074) ifTrue: [  sym := #lockErrObjHasChanged] ifFalse:[
(errNum == 2071) ifTrue: [  sym := #lockErrUndefinedLock ] ifFalse:[
(errNum == 2073) ifTrue: [  sym := #lockErrIncomplete ] ifFalse:[
(errNum == 2418) ifTrue: [  sym := #lockErrDeadlock ] ifFalse:[
(errNum == 2419) ifTrue: [  sym := #lockErrTimeout ] ifFalse:[
]]]]]].

^ ex reason: sym ; signal

]

{ #category : 'Setting Locks' }
System class >> _lockEvaluateErr: errNum obj: anObject denied: denyBlock changed: changeBlock [

"Private."

(errNum == 2074) ifTrue:[ ^ changeBlock value ]. "object  written by other session"
(errNum == 2075) ifTrue:[ ^ denyBlock value ]. " lock denied "

^ self _lockError: errNum obj: anObject details: 'unexpected lock error'

]

{ #category : 'Private' }
System class >> _markTranlogWith: aString [

"Writes a debug record to the tranlog with the specified string.
 String should be <= 16 characters, the rest are truncated."

<primitive: 1015>
self _primitiveFailed: #_markTranlogWith: args: { aString }

]

{ #category : 'Private' }
System class >> _maxClusterId [

" return the maximum clusterId supported vy the VM."
^ self _zeroArgPrim: 91

]

{ #category : 'Private' }
System class >> _maxPrimitiveNumber [

"Return a SmallInteger, the maximum primitive number for this VM."

^ self _zeroArgPrim: 184

]

{ #category : 'Private' }
System class >> _messageForErrorNum: aNumber [

"Private.  Returns a String describing the error message produced for GemStone
 error numbered aNumber in native language English, or nil if no such error
 number or language exists."

^ self _messageForErrorNum: aNumber inLanguage: #English

]

{ #category : 'Private' }
System class >> _messageForErrorNum: aNumber inLanguage: aSymbol [

"Private.  Returns a String describing the error message produced for GemStone
 error numbered aNumber in native language aSymbol, or nil if no such error
 number or language exists."

 | errArray msgParts result |

 errArray := GemStoneError at: aSymbol otherwise: nil .
 errArray == nil ifTrue:[ ^ nil ].
 (aNumber < 1 or:[ aNumber > errArray size]) ifTrue:[ ^ nil ].
 msgParts := errArray at: aNumber .
 msgParts == nil ifTrue:[ ^ nil ].
 result := String new .
 1 to: msgParts size do:[:j | | element |
   element := msgParts at: j .
   element _isSmallInteger
     ifTrue:[ result addAll: '<arg'; addAll: element asString; add: $> ]
    ifFalse:[ result addAll: element ].
   ].
 ^ result

]

{ #category : 'Private' }
System class >> _midCachePgsvrListeningPort [

 "Returns nil or the listening port of the multithreaded pgsvr
  on the mid cache for this gem."

^ self _zeroArgPrim: 69

]

{ #category : 'Private' }
System class >> _midLevelCacheConnect: hostName options: anArg [

"If (hostName == nil) and (anArg class == Array) returns a String printout of the
 results of parsing the options array.

 If (anArg == 1) then semantics are disconnect any existing mid level cache
 and connect to an already running mid level cache on host  hostName .

 If (anArg == 2) then hostNames is ignored;
 return a String, the IP address of an existing mid cache
 connection, or nil if no connection to a mid cache.

 Otherwise behaves per  System(C)>>midLevelCacheConnect:options:  "

<primitive: 664 >
hostName ifNotNil:[ hostName _validateInstanceOf: String ] .
anArg ifNotNil:[
  (anArg ~~ 1 and:[ anArg ~~ 2]) ifTrue:[ anArg _validateInstanceOf: Array ] .
].
hostName size > 2000 ifTrue:[ hostName _error: #errArgTooLarge args:{ 2000 } ].
^ self _primitiveFailed: #_midLevelCacheConnect:options:
       args: { hostName . anArg }

]

{ #category : 'Private' }
System class >> _notVotedSessions: resultBlock into: res [
  | list userPro num desc |
  list := self _currentSessionProfiles .
  1 to: list size by: 2 do:[:j |
    (userPro := list at: j + 1) ifNotNil:[
      userPro _isInteger ifFalse:[  "ignore logsender, logreceiver"
        desc := self _descriptionOfSessionSerialNum: 0 sessionId: (num := list at: j).
        (desc at: 20) < 2 ifTrue:[  resultBlock value: res value: num value: userPro ].
      ].
    ].
  ].
  ^ res

]

{ #category : 'Private' }
System class >> _numPagesInSharedOt [

"Returns the number of pages in the shared object table.
 This method aborts the current transaction."

^self _zeroArgPrim: 55

]

{ #category : 'Transaction Control' }
System class >> _numPersistentObjsModified [

"Returns number of persistent objects modified during current transaction.
 Includes number of objects added/removed to/from depMap."

^ self _zeroArgPrim: 54

]

{ #category : 'Shared Counters' }
System class >> _numSharedCounters [

"Returns the number of shared counters supported by this session's shared
 page cache. This is set by the value for SHR_PAGE_CACHE_NUM_SHARED_COUNTERS
 that is specified in the configuration file parameter used by this shared cache;
 this parameter has a default of 1900."

  ^self _zeroArgPrim: 100

]

{ #category : 'Configuration File Access' }
System class >> _objectReadLogInfo [
 "Returns nil if object read logging is not enabled.
  Otherwise returns an Array containing
    a String , path to object read log file that stone is writing to
    an Integer, the zero based offset of element of STN_OBJECT_READ_LOG_DIRECTORIES in use,
    an Integer,  size in bytes of the current object read log file"
  ^ self _zeroArgPrim: 193
]

{ #category : 'Private' }
System class >> _objectsInNscEnumCache [

"Returns an array with the objects in the NscEnumerationCache"

^self _zeroArgPrim: 56

]

{ #category : 'Private' }
System class >> _objectTableCleanup [

"Enters a transaction critical region and writes all deltas in the
 object table to the leaves using a multithreaded algorithm.
 Requires SystemControl and GarbageCollection privilege."

^self _zeroArgPrim: 15

]

{ #category : 'Performance Monitoring' }
System class >> _objectTableSizeInMegabytes [

"Returns an Integer that gives the size of the object table in megabytes"

^ self _zeroArgPrim: 101

]

{ #category : 'Performance Monitoring' }
System class >> _oopHighWaterMark [

"Returns an Integer that gives the highest oop allocated in GemStone."

^self _zeroArgPrim: 93

]

{ #category : 'Performance Monitoring' }
System class >> _oopNumberHighWaterMark [

"Returns an Integer that gives the highest oop number allocated in GemStone."

^self _zeroArgPrim: 94

]

{ #category : 'Shared Cache Management' }
System class >> _otherCacheAddresses [

"Return an Array of the ipAddresses of the currently active leaf caches,
 excluding the shared cache on which this session is running."

^ self _cacheAddresses: false includeMyCache: false

]

{ #category : 'Shared Cache Management' }
System class >> _otherMidCacheAddresses [

"Return an Array of the ipAddresses of the currently active midLevel caches,
 excluding the shared cache on which this session is running."

^ self _cacheAddresses: true includeMyCache: false

]

{ #category : 'Private' }
System class >> _parseClientNrs: aString [

"aString should be the result of reading a request sent to a netldi
 on a socket. A session reading nrs requests from clients can tell
 that a full request has come in by reading until it gets a zero byte.
 This method raises the error #genericKernelError if something is
 wrong with the input.
 Otherwise it returns a 8 element Array with the following contents:
 1  Symbol describing type of client. It will be one of the following:
      #RpcApplication, #Gem, #PageServer, #Stone, #Unknown
 2  Boolean: true if client is trusted.
 3  String: name of user.
 4  Boolean: (obsolete)
 5  Symbol describing type of request. It will be one of the following:
      #invalid, #file, #monitor, #server, #task, #spawn, #cmd, #dbf #fork
 6  String: working directory
 7  String: logfile name
 8  String: body of request
"

<primitive: 515>
| syms |
" make sure we don't have to create new symbols at runtime"
syms := { #RpcApplication . #Gem . #PageServer . #Stone . #Unknown .
	#invalid . #file . #monitor . #server . #task . #spawn . #cmd .
        #dbf . #fork } .
self _primitiveFailed: #_parseClientNrs: args: { aString }

]

{ #category : 'Transaction Control' }
System class >> _pendingCommitAbort [

""

| commitAction |
commitAction := self __sessionStateAt: 7 .
commitAction ifNotNil:[
  commitAction _isSmallInteger ifTrue:[
      "Signal commit-abort pending error"
      TransactionError new reason:#rtErrCommitAbortPending; _number: 6007 ; signal
      "expect the application to continue with GciContinue"
    ]
    ifFalse: [ self userAction: commitAction ]
].

]

{ #category : 'Private' }
System class >> _performOnServer: aString [

"For compatibility, used by seaside"

^ self _performOnServer: aString withShell: nil

]

{ #category : 'Private' }
System class >> _performOnServer: aString withShell: aShellOrNil [

"Attempts to execute aString using the shell aShellOrNil.
 Uses the default shell /bin/sh if aShellOrNil is nil.
 Otherwise uses aShellOrNil, which must accept the -c argument.

 Returns an Array of 5 elements:
  raw status Integer,
  child process status Integer (after WEXITSTATUS macro applied),
  result String ,
  error string from script file write, fork, or result file read ,
  errno value, a SmallInteger from file write, fork, or file read
"
<primitive: 347>
aString _validateClasses: { String . Utf8 }.
aShellOrNil ifNotNil:[ aShellOrNil _validateClass: String ] .
^ aString _error: #hostErrPerform args:  { aShellOrNil }

]

{ #category : 'Private' }
System class >> _prepareToCommit [

"This is a placeholder for the implementation of voting in a two-phase commit
 protocol.  The future implementation should use #hasConflicts to return
 whether the current transaction could commit at this exact point in time.

 The current design of #hasConflicts however will issue false positives,
 indicating that there are conflicts even when there are not (bug 40094).
 Because of this, certain types of operations would always cause
 #_prepareToCommit to fail.  Until this is resolved, we'll default to
 always returning true -- later steps in the psuedo two-phase commit will
 still catch any problems.

 Note: See the comment for GsCurrentSession | hasConflicts to get a
 description of the side-effects of hasConflicts."

 " ^ GsSession currentSession hasConflicts not "
 ^ true

]

{ #category : 'Private' }
System class >> _primCacheStatsForSlotOrSession: arg opCode: code [

"This primitive performs various functions relating to cache statistics per
 the following table:

arg  code    meaning
=======================================================================
 +     0     arg is slot.  Return all stats for all kinds*
 +     1     arg is slot.  Return only stats for the process.
 -     0     arg is a gem -session ID.  Return all stats for all kinds*
 -     1     arg is a gem -session ID.  Return only stats for the gem.
 +     2     arg is a PID.  Return only stats for the process.

 str   3     arg is a cache name string.  Return only stats for the
             process matching the name.
 +     4     arg is a slot.  Return  the Pid for slot.
 +     5     arg is a PID.  Return the slot for the PID.
 +     6     arg is a slot.  Return host cpu stats for slot.
 +     7     arg is a PID.  Host cpu stats for pid.
 +     8     arg is a pgsvr session ID.  Return only stats for the
             pgsvr.
 +     9     arg is a slot.  Return the process kind for the slot
             (1 = monitor, 2 = stone, 3 = pgsvr, 4 = gem) or -1 if
             the slot is not in use.
 nil  10     arg is nil.  Return an array of strings which are the
             system stats available for each process.
  +   10     arg is a process ID.  Return an array of Integers which
             are the system statistics for the given process. arg == -1
             indicates the current gem process.
 nil  11     Return a Solo session's own stats .
  +   12     Linux only: arg is a process ID.  Return an array of 
             Integers which are the 'easy' (inexpensive to obtain) 
             system statistics for the given process. arg == -1
             indicates the current gem process.
=======================================================================

Legend:

+     a postive SmallInteger
-     a negative SmallInteger
str   aString
*     deprecated functionality which may be removed in a future release.
"

<primitive: 220>
code _validateClass: SmallInteger .
code == 3
  ifTrue:[ arg _validateClass: String ]
  ifFalse:[ arg _validateClass: SmallInteger] .
(code < 0 or:[ code > 12])
  ifTrue:[ code _error: #rtErrArgOutOfRange args:{ 0 . 12 } ] .
arg _error: #rtErrArgOutOfRange .
^ self _primitiveFailed: #_primCacheStatsForSlotOrSession:opCode:
       args: { arg . code }

]

{ #category : 'Reduced Conflict Support' }
System class >> _primFailedCommitWithResult: commitResult [

"Increments the failedCommitCount stat for the session and sets
 the lastFailedCommitReasonCode stat to commitResult.
 Returns true."

  <primitive: 1108>
  self _primitiveFailed: #_primFailedCommitWithResult: args: { commitResult } .

]

{ #category : 'Private' }
System class >> _primitiveAbort: soloAbort [

"Do not call this method directly; it is private to GemStone.  You must use
 abortTransaction.  Otherwise the transparent GemStone Smalltalk interface will
 break.

 This method rolls back all modifications made to committed GemStone objects
 (connected to the root) and provides the session with a new view of the most
 current committed state.  These operations are performed whether or not the
 session was previously in a transaction.  If the transactionMode is set to
 #autoBegin, then a new transaction is started.  If the transactionMode is set
 to #manualBegin or #transactionless, then a new transaction is not started,
 but the session's view of the database is updated.
 This method always returns the receiver (System).

 Signals an error if  GsSession isSolo==true , and soloAbort ~~ true, and there
 are modified committed objects . (See System class >> soloAbort )"

<primitive: 335>
soloAbort _validateClass: Boolean.
self _primitiveFailed: #_primitiveAbort .
self _uncontinuableError

]

{ #category : 'Private' }
System class >> _primitiveBegin [

"Starts a new transaction for the session.  If the session is already
 in a transaction, aborts the transaction and starts a new transaction.

 If any permanent objects had been written by the session, their
 state is aborted.  This method returns the receiver (System).

 Signals an error if  GsSession isSolo==true , and there
 are modified committed objects (See System class >> soloAbort) ."

<primitive: 362>
self _primitiveFailed: #primitiveBegin.
self _uncontinuableError

]

{ #category : 'Private' }
System class >> _primitiveCommit: commitMode [

"Do not call this method directly; it is private to GemStone.  You must use
 commitTransaction or commitAndReleaseLocks.  Otherwise Gem Builder breaks.

 The argument commitMode is a SmallInteger:
    0 for normal,
    1 for release locks,
    2 for checkpoint and release locks.
    3  value not used , primitive will fail
    4 checkpoint and start dpNsUnion (used by full backup)

 The result of this primitive is a SmallInteger:
      -1 read-only  (there were no modified objects to commit),
       0 success,
       1 rcFailure  (replay of changes to instances of Rc classes failed),
       2 dependencyMap failure  (concurrency conflict on dependencyMap),
       3 validationFailure  (concurrency conflict),
       4 retryFailure,   (previous commit attempt failed with an rcFailure)
       5 commitDisallowed (disallowed due to other error),
       6 retry limit exceeded
       7 symbol failure (symbol VM missing or restarted)
       8 lock failure "

<primitive: 336>
commitMode _validateClass: SmallInteger .

(commitMode < 0 or:[ commitMode > 2 ])
  ifTrue:[ commitMode _error: #rtErrArgOutOfRange args:{ 0 . 2 } .  ^ 1 "failure" ] .

self _primitiveFailed: #_primitiveCommit: args: { commitMode } .
self _uncontinuableError

]

{ #category : 'Transaction Control' }
System class >> _primTransactionMode: newMode [

""

<primitive: 368>
self _primitiveFailed: #_primTransactionMode: args: { newMode } .
self _uncontinuableError

]

{ #category : 'Private' }
System class >> _printCHeap [

"in a slow or fastdebug VM , calls UtlHeapPrint()"
^ self _zeroArgPrim: 89

]

{ #category : 'Private' }
System class >> _printSocketTrace [
  ^ self _zeroArgPrim: 605 "has effect only in debug build of VM"

]

{ #category : 'Private' }
System class >> _processDeferredGciUpdates [

"Process deferred GemBuilder for C updates and return the number of objects that
 had deferred updates."

"The deferred updates are stored in __sessionStateAt: 12.  These
 are updates to objects that do not allow direct structural update such
 as AbstractDictionary, Bag, and Set ."

| res |
self _disableAddToDirtySets: true .
[ | arr count |
  arr := self __sessionStateAt: 13 "virtual machine constant".
  (count := arr size) ~~ 0 ifTrue:[
    1 to: count by: 2 do:[ :j |
      (arr at: j) _deferredGciUpdateWith: (arr at: j + 1).
      ].
    arr size: 0 .
    "reinitialize the identity dictionary"
    (self __sessionStateAt: 12 "virtual machine constant")
	initialize: 751"a prime number for a large small object"
    ].
  res := count // 2
] ensure:[
  self _disableAddToDirtySets: false .
].
^ res

]

{ #category : 'Private' }
System class >> _protectedMode [

"Returns a SmallInteger, the value of protected mode .

 The usage pattern for writing a protected method is

     <primitive: 2001>
     | prot |
     prot := System _protectedMode .
     [
       ""method body""
     ] ensure:[
       prot _leaveProtectedMode
     ]

    primitive 2001 enters protected mode.
    The _protectedMode method returns the current nesting level of
    protected methods (a SmallInteger).
    _leaveProtectedMode is a special selector which compiles to a
     bytecode ."

<protected primitive: 691>
self _primitiveFailed: #_protectedMode .
self _uncontinuableError

]

{ #category : 'Private' }
System class >> _publish: server port: port log: logname options: optstr [
  "Description:
     Publishes information about a server so that the gslist utility
     can display it. The information will be published on the local machine.
   Input:
     server <String>: name of server to publish.
     port <SmallInteger>: port number of server listening socket.
     logname <String or nil>: name of log file or nil if none.
     optstr <String or nil>: startup options or nil if none.
   Result <Boolean>:
     true if publish was successful.
     false if server was already published.
   Exceptions:
     Illegal input raises various errors.
     Error 2367 raised if low level C code fails with system error.
  "

<primitive: 496>
self _primitiveFailed: #_publish:port:log:options:
     args: { server . port . logname . optstr }

]

{ #category : 'Reduced Conflict Support' }
System class >> _rcValueCache [

"Returns the cache dictionary that is stored in temporary session state
 used to hold calculated values for reduced conflict classes. "

^ self __sessionStateAt: 5

]

{ #category : 'Performance Monitoring' }
System class >> _readClock [

"Returns an Integer indicating the amount of CPU time used by the current
 process, in units of milliseconds.  The resolution of the result is operating
 system dependent; typical resolution is 10 to 50 milliseconds."

^ self _zeroArgPrim: 10

]

{ #category : 'Performance Monitoring' }
System class >> _gemCpuTimeMs [

"Returns an Integer indicating the amount of CPU time used by the current
 process, in units of milliseconds.  The resolution of the result is operating
 system dependent; typical resolution is 10 to 50 milliseconds."

^ self _zeroArgPrim: 10

]

{ #category : 'Performance Monitoring' }
System class >> _stoneCpuTimeMs [

"Returns an Integer indicating the amount of CPU time used by the stone
 process, in units of milliseconds.  The resolution of the result is operating
 system dependent; typical resolution is 10 to 50 milliseconds."

^ self _zeroArgPrim: 199

]

{ #category : 'Private' }
System class >> _redoLog [

"Returns the redo log for transaction level 1.
 For use by Rc replay during commit."

| lev |
(lev := self _zeroArgPrim: 5"transactionLevel") == 1 ifFalse:[
  Error signal:'Replay not allowed at transactionLevel ' , lev asString .
].
(self __sessionStateAt: 4) ifNotNil:[:a | ^ a atOrNil: 1 ].
^ nil

]

{ #category : 'Private' }
System class >> _reinitializeSharedCache [

"This method shuts down the GcGem process and verifies that this session
 is the only one logged in.  If not the only user, then the primitive fails.
 If it is the only user, all attached pages for this session are detached
 and the cache is swept.  All dirty pages are written.  All remaining
 unattached pages are removed from the cache and their frames are put back
 in the free list.

 Returns self."

^self _zeroArgPrim: 36

]

{ #category : 'Private' }
System class >> _reloginAsUser: aUserId password: aPassword encrypted: aBoolean [

"Relogin this session with the specified user identification."

<primitive: 495>
aUserId _validateClass: String.
aPassword _validateClass: String.
aBoolean _validateClass: Boolean.
self _primitiveFailed: #_reloginAsUser:password:encrypted:
     args: { aUserId . aPassword . aBoolean }

]

{ #category : 'Shared Cache Management' }
System class >> _remoteCachesList: forMidLevel [

"Return an Array describing shared caches, per remoteCachesList."

| idx result batch |
idx := 0 .
result := { }  .
[ idx >= 0 ] whileTrue:[ | nextIdx batSize |
  batch := self _remoteCachesListStartingAt: idx midLevelOnly: forMidLevel
                       includeMyCache: true .
  batSize := batch size .
  batSize > 0 ifTrue:[
    nextIdx := batch at: batSize .
    batch size: batSize - 1.
    result addAll: batch .
    idx := nextIdx .
  ] ifFalse:[
    idx := -1 . "done"
  ].
].
^ result

]

{ #category : 'Shared Cache Management' }
System class >> _remoteCachesListStartingAt: anOffset midLevelOnly: midBoolean
includeMyCache: includeMyCacheBoolean [

"Return a portion of the result for remoteCachesList,
 starting at anOffset in stone's list.    anOffset is zero-based.

 Result size is zero if anOffset is beyond end of stone's list,
 otherwise last element of result is a SmallInteger, the nextEnumIdx.
 If nextEnumIdx is < 0 there are no more elements in stone's list,
 otherwise pass nextEnumIdx as anOffset argument in next call to this
 primitive to continue the enumeration.

 Result size is limited by size of gem to stone communication buffers."

<primitive: 665>
anOffset _validateClass: SmallInteger .
anOffset < 0 ifTrue:[ anOffset _error: #errArgTooSmall args:{ 0 } ].
midBoolean _validateClass: Boolean .
includeMyCacheBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_remoteCachesListStartingAt:midLevelOnly:includeMyCache:
       args: { anOffset . midBoolean . includeMyCacheBoolean  }

]

{ #category : 'Private' }
System class >> _breakSerialization [

"Release commit token if it is held by this session."

^ self _zeroArgPrim: 71

]

{ #category : 'Private' }
System class >> _removeLockWriteSubset [

"Removes the locks acquired on any objects that were locked using
 _writeLockWriteSubset and empties the hidden set."

^self _zeroArgPrim: 35


]

{ #category : 'Private' }
System class >> _removePageFromCache: aPageId [

"aPageId must be an Integer >= 0 and <= 549755813887  .
 If the specified page is in gem's shared cache,
 prints the page header and attempts to remove the page
 from the shared cache.   If the gem is running on stone's machine
 will also print the page header from the state of the page on disk.

 Returns true if successfully removed, false otherwise.
 The result will be false if the page was dirty, locked or pinned."

<primitive: 590>
aPageId  _validateClass: Integer.
(aPageId < 0 or:[aPageId > 549755813887 ]) ifTrue:[
  aPageId _error: #rtErrArgOutOfRange args:{ 0 . 549755813887 }
] .
^self _primitiveFailed: #_removePageFromCache: args: { aPageId }

]

{ #category : 'Reduced Conflict Support' }
System class >> _resolveRcConflicts [

  "Checks for selected conflicts.  If some are found, attempt to resolve those
   conflicts.  If any conflicts could not be resolved, returns false."
 [
  | scanArray redoObject replayFailed conflicts idx limit gemCommitConflictDetails |
  scanArray := self _getRedoAndConflictObjects .
  scanArray ifNil: [
    "no redo objects were found, cannot resolve conflicts"
    self _breakSerialization; _disallowSubsequentCommit: 3 .
    ^ false
  ].
  gemCommitConflictDetails := self _gemCommitConflictDetails.
  replayFailed := false.
  limit := scanArray size .
  idx := 1 .
  [ idx <= limit ] whileTrue:[
		redoObject := scanArray at: idx .
		conflicts := scanArray at: idx + 1 .
		idx := idx + 2 .
		(redoObject _resolveRcConflictsWith: conflicts) ifFalse: [
	    self rcValueCacheAt: #'Rc-Retry-Failure' put: redoObject for: self.
	    replayFailed := true.
      limit := 0 . "exit loop"
    ]
  ].
  "If write write conflicts not empty, we have failed to selective abort enough objs"
  (replayFailed 
    or:[ self _writeWriteConflictsIsEmpty ~~ true ]) ifTrue: [
    " force subsequent attempts to commit to fail with retry limit error"
    self _breakSerialization; _disallowSubsequentCommit: 4 .
    gemCommitConflictDetails ~~ 0
      ifTrue: [GsFile gciLogServer: '--- replay FAILED'].
   ^ false
  ].
  gemCommitConflictDetails > 1 ifTrue:[
     GsFile gciLogServer: '--- replay SUCCEEDED'.
     self printRcReadSet .
  ].
  ^ true
 ] onException: Error do:[:ex |
   self _breakSerialization; _disallowSubsequentCommit: 5 .
   ex pass .
   ^ false
 ]
]

{ #category : 'Reduced Conflict Support' }
System class >> _resolveRcConflictsForCommit: commitMode [

| result |
self _processDeferredGciUpdates .
self _resolveRcConflicts ifFalse: [
  ^ false
].

result := self _primitiveCommit: commitMode.
(result == 0 or: [result == -1]) ifTrue: [
  ^ true.
].
self _failedCommitWithResult: result ; "48387"
     _disallowSubsequentCommit: 2 .
^ false
]

{ #category : 'Private' }
System class >> _restoreModTracking [

"Private. Enables Modification Tracking.

 Has no effect in this release"

<primitive: 517>
self _primitiveFailed: #_RestoreModTracking

]

{ #category : 'Signals' }
System class >> _sendSignal: aSignal toSess: sessionIdOrSerial withMessage: aString [

"Sends a signal (a SmallInteger) to the specified session with aString as a
 message.  The aString argument is currently limited to 1023 bytes.

 The sessionIdOrSerial argument is a SmallInteger specifying a session.  A
 positive value is interpreted as a serialNumber.  A negative value is
 interpreted as a negated sessionId."

^ self _sendSignal: aSignal toSess: sessionIdOrSerial withMessage: aString oob: nil

]

{ #category : 'Private' }
System class >> _sendSignal: aSignal toSess: sessionIdOrSerial withMessage: aString oob: oobInt [
"If oobInt nil or zero,  Send aSignal and aString to specified session .
 aString is limited to 1023 bytes.
 The sessionIdOrSerial argument is a SmallInteger specifying a session.  A
 positive value is interpreted as a serialNumber.  A negative value is
 interpreted as a negated sessionId.

 If oobInt > 0, send only specified GscOobMsgEType value to the session
 for debugging purposes only.  Requires SystemControl privilege.
 aSignal and aString are ignored, but must be a SmallInteger and a String.

 Signals an error if the specified session cannot be found, or if the session
 is the pagemanager thread, a logreceiver or a logsender .

 Warning, some values sent to some sessions might cause a session logout."

<primitive: 330>
aSignal _validateClass: SmallInteger.
sessionIdOrSerial _validateClass: SmallInteger.
aString _validateClass: String.
oobInt ifNotNil:[ oobInt _validateClass: SmallInteger ].
self _primitiveFailed: #_sendSignal:toSerialNum:withMessage:oob:
     args: { aSignal . sessionIdOrSerial . aString . oobInt } .
self _uncontinuableError

]

{ #category : 'Private' }
System class >> _sendTransactionSignal: sessionIdOrSerial kind: aKind [

"send a sigAbort or a finishTransaction signal to the specified
 session.  The signal will be ignored in the session
 if the session has not enabled that specific signal.
 Also supports sendLostOt to a session.

 Returns a Boolean , true if stone sent the OOB byte to the
 specified session, false of the OOB socket was blocked.

 Generates an Error if no session exists for sessionIdOrSerial .

 You must have SystemControl privilege .

 aKind == 0 : sigAbort
 aKind == 1 : finishTransaction
 aKind == 2 : lostOt "

<primitive: 327>
sessionIdOrSerial _validateClasses:{ SmallInteger }.
aKind _validateClasses:{ SmallInteger }.
self _primitiveFailed: #sendSigAbortToSession:
     args: { sessionIdOrSerial . aKind }

]

{ #category : 'Version Management' }
System class >> _serverVersionReport: isStone [

"Private, returns a StringKeyValueDictionary."

| result |
result := StringKeyValueDictionary new .
VersionParameterDict keysAndValuesDo:[ :aName :anId | | aVal |
  aVal := self _configurationAt: anId isStone: isStone kind: $V .
  aVal ~~ nil ifTrue:[
    result at: aName put: aVal
    ].
  ].
^ result

]

{ #category : 'Private' }
System class >> _sessionCacheStatAt: anIndex [

"Returns the value of the session statistic at the specified index, which
 must be in the range -2 to 47.  -1 means return the value of the
 progressCount statistic.  -2 means return the value of the indexProgressCount
 statistic."

^ self _sessionCacheStatAt: anIndex incrementBy: 0

]

{ #category : 'Private' }
System class >> _sessionCacheStatAt: anIndex decrementBy: aSmallInt [

"Decrements the value of the session statistic at the specified index
 by aSmallInt and returns the new value.  anIndex must be in the range
 of -2 to 47.  -1 means decrement the progressCount statistic.  -2 means
 decrement the indexProgressCount statistic."

^ self _sessionCacheStatAt: anIndex incrementBy: aSmallInt negated

]

{ #category : 'Private' }
System class >> _sessionCacheStatAt: anIndex incrementBy: aSmallInt [

"Increments the value of the session statistic at the specified index
 by aSmallInt and returns the new value.  anIndex must be in the range
 of -2 to 47.  -1 means increment the progressCount statistic.  -2 means
 increment the indexProgressCount statistic."

<primitive: 477>
anIndex _validateClass: SmallInteger.
aSmallInt _validateClass: SmallInteger.
((anIndex < -2) or: [ anIndex > 47]) ifTrue: [
   anIndex _error: #rtErrArgOutOfRange args: { -2 . 47 } ].
self _primitiveFailed: #_sessionCacheStatAt:incrementBy:
     args: { anIndex . aSmallInt } .

]

{ #category : 'Private' }
System class >> _sessionCacheStatAt: anIndex put: i [

"This method sets the session statistic at the specified index (which should be
 in the range -2 to 47) to the specified value i, which must be a SmallInteger.
 -1 means set the progessCount statistic to the value i.  -2 means set the
 indexProgessCount statistic to the value i. -3 means set the GemKind statistic
 to value i."

<primitive: 476>
anIndex _validateClass: SmallInteger.
i _validateClass: SmallInteger.
((anIndex < -3) or: [ anIndex > 47]) ifTrue: [
   anIndex _error: #rtErrArgOutOfRange args: { -3 . 47 } ].
self _primitiveFailed: #_sessionCacheStatAt:put: args: { anIndex . i } .

]

{ #category : 'Private' }
System class >> _sessionCacheStatsForProcessSlot: aProcessSlot [

"Return an array containing the 48 SmallIntegers representing the
 session statistics for the given process slot.

 Returns nil if the given process slot is not found or if the slot
 is not in use by a gem process.

 See additional information in the method #_sessionCacheStatAt: put:"

^self _sessionCacheStatsForSlotOrSessionId: aProcessSlot

]

{ #category : 'Private' }
System class >> _sessionCacheStatsForSessionId: aSessionId [

"Return an array containing the 48 session statistics for the given
 session ID.

 Returns nil if the given session is not found or if the slot is not
 in use by a gem process.

 See additional information in the method #_sessionCacheStatAt: put:"

^self _sessionCacheStatsForSlotOrSessionId: aSessionId negated.

]

{ #category : 'Private' }
System class >> _sessionCacheStatsForSlotOrSessionId: aProcessSlotOrSessionId [

"Return an array containing the 48 session statistics for the given
 session ID or process slot number.  If the argument is positive
 it is assumed to be a process slot.  If it is negative it is assumed
 to be a negated session ID.

 Returns nil if the given session or cache slot is not found or
 if the slot is not in use by a gem process.

 See additional information in the method #_sessionCacheStatAt: put:"

<primitive: 566>
aProcessSlotOrSessionId _validateClass: SmallInteger.
aProcessSlotOrSessionId _error: #rtErrArgOutOfRange .
^self _primitiveFailed: #_sessionCacheStatsForSlotOrSessionId:
      args: { aProcessSlotOrSessionId } .

]

{ #category : 'Updating the Method Dictionary' }
System class >> _sessionMethodsChangedForAbort [
  "Returns a Boolean, the union of other sessions commits of
   Behavior >> _sessionMethodsChanged , unioned with our own
    Behavior >> _sessionMethodsChanged bit "
  ^ self _zeroArgPrim: 82

]

{ #category : 'Updating the Method Dictionary' }
System class >> _sessionMethodsChangedForCommit [
  "Returns a Boolean, the union of other sessions commits of
   Behavior >> _sessionMethodsChanged ."
  ^ self _zeroArgPrim: 81

]

{ #category : 'Session Control' }
System class >> _sessionPriorityPrim: anInt forSessionId: aSessionId [

"Fetches or sets the priority for the given session ID.
 If anInt is -1, then the current priority for the given
 session ID is returned.  Otherwise anInt must be a valid
 priority between 0 and 4.  See the method:
   System(C)>>setSessionPriority:forSessionId:
 for more information on session priority."

<primitive: 898>
anInt _validateClass: SmallInteger .
aSessionId _validateClass: SmallInteger .
((anInt < -1) or:[ anInt > 4])
  ifTrue:[anInt _error: #rtErrArgOutOfRange args:{ -1 . 4 } .].
aSessionId < 1
  ifTrue:[aSessionId _error: #rtErrArgOutOfRange ].
^self _primitiveFailed: #_setSessionPriorityPrim:forSessionId:
      args: { anInt . aSessionId }

]

{ #category : 'Private' }
System class >> _sessionProfilesStartingAt: startSessionId [

"Returns an Array containing pairs of sessionId, UserProfile ,
 for some sessions currently logged in, beginning at the specified sessionId.
 Some sessionIds may have a userProfile of nil
 if that UserProfile is not visible in the transactional view of
 the current session.
 The size of the Array is limited by number of sessions logged in
 and/or gem to stone communication buffer sizes.  If there are
 no more valid sessionIds beginning with startSessionId, an
 Array of size 0 is returned.

 This method requires SessionAccess privilege if there is more
 than one session logged in."

<primitive: 329>
startSessionId _validateClass: SmallInteger .
startSessionId _error: #rtErrArgOutOfRange .
^ self _primitiveFailed: #_sessionProfilesStartingAt: args: { startSessionId }

]

{ #category : 'Private' }
System class >> _reportDetailsFor: sessionId desc: desc [
  "desc is result of descriptionOfSession;"
  | str uId |
  str := String new .
  (desc at: 1) ifNotNil:[:up|  uId := up userId ]
               ifNil:[ (desc at: 17) ifNotNil:[:x | uId := '    ' ]].
   uId ifNotNil:[
     str add: sessionId asString; add:' '; add: uId; add: ' '.
     (desc at: 17) ifNotNil:[ :sysgem | 
       str add: sysgem ; add: ' ' ; add:(desc at:2) asString 
     ] ifNil:[  |  gemPid |
       (desc at: 21) == -1 ifTrue:[ str add:'topaz -l ' ] ifFalse:[ str add:'gem ' ].
       gemPid := desc at: 2 . 
       gemPid > 0 ifTrue:[ str add: gemPid asString; add:' on ' ; add: (desc at: 3) ;
            add:' serial '; add: (desc at: 9) asString.
       ].
       sessionId == System session ifTrue:[ str add:', this session'].
    ].
  ].
  ^ str
]

{ #category : 'Private' }
System class >> _sessionsReport: arrayOfSessionIds [
  | str committingSessId inLogin |
  str := String new .
  committingSessId := System stoneCacheStatisticWithName: 'CommitTokenSession'.
  inLogin := 0 .
  arrayOfSessionIds do:[:sessId | | desc |
    sessId ifNotNil:[ | sVal | 
      desc := self descriptionOfSession: sessId .
      sVal := self _reportDetailsFor: sessId desc: desc .
      sVal size > 0 ifTrue:[
        str addAll: sVal .
        sessId == committingSessId ifTrue:[ str add: ', has commit token' ].
        str lf.
      ] ifFalse:[ inLogin := inLogin + 1 "userProfile is nil"].
    ] ifNil:[ inLogin := inLogin + 1 ].
  ].
  inLogin ~~ 0 ifTrue:[ str add: '  ', inLogin asString , ' sessions in login'; lf  ].
  ^ str
]

{ #category : 'Deprecated' }
System class >> _sessionStateAt: anIndex [

"Deprecated, use sessionStateAt:, and note changes in allowed range"

self deprecated: 'System class>>_sessionStateAt: deprecated v3.2.
Use sessionStateAt:, with available range starting at 1.'.

^ self __sessionStateAt: anIndex + 20

]

{ #category : 'Deprecated' }
System class >> _sessionStateAt: anIndex put: aValue [

"Deprecated, use sessionStateAt:put:, and note changes in allowed range"

self deprecated: 'System class>>_sessionStateAt:put: deprecated v3.2.
Use sessionStateAt:put:, with available range starting at 1.'.
^ self __sessionStateAt: anIndex + 20 put: aValue

]

{ #category : 'Transient Session State' }
System class >> _sessionStateSize [

"Return the current size of the Session State Array."

<primitive: 214>
self _primitiveFailed: #_sessionStateSize .
self _uncontinuableError

]

{ #category : 'System Control' }
System class >> _sessionTrackReads [
"Returns true if (System objectReadLogEnabled == true)
 and ((System myUserProfile _hasPrivilegeName: #DisableObjectReadLogging) == false)"

^ self _zeroArgPrim: 194
]

{ #category : 'Private' }
System class >> _sessionUproString: rawUp [
  ^  rawUp _isInteger ifTrue:[
       rawUp == 1 ifTrue:[ ^ '    logsender'].
       rawUp == 2 ifTrue:[^ '    logreceiver' ].
       rawUp == 3 ifTrue:[ ^ '    stoneTranlogReplay'].
       Error signal:'invalid userProfile ', rawUp asString .
     ] ifFalse:[
       '    UserId: ' , rawUp userId .
     ]

]

{ #category : 'Private' }
System class >> _setCategory: categorySym locale: localeStr [

" Set Locale information.  See man page for setlocale( ) for details.

Valid category types include:

   LC_CTYPE        /* locale's ctype handline */
   LC_NUMERIC      /* locale's decimal handling */
   LC_TIME         /* locale's time handling */
   LC_COLLATE      /* locale's collation data */
   LC_MONETARY     /* locale's monetary handling */
   LC_MESSAGES     /* locale's messages handling */
   LC_ALL          /* name of locale's category name */

Applications should not call this directly -- instead use:

  Locale setCategory: <sym> locale: <str>'

"

<primitive: 661>
categorySym _validateClass: Symbol.
localeStr _validateClass: String.
self _primitiveFailed: #_setCategory:locale: args: { categorySym . localeStr }

]

{ #category : 'Private' }
System class >> _setPersistentConfig: isReclaim symbol: configSymbol toValue: aVal [

| gcUg result|
self needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

gcUg := ((AllUsers userWithId:'GcUser') resolveSymbol: 'UserGlobals') value .
[
  self beginTransaction.
  gcUg at: configSymbol put: aVal.
  result := self _gcGemConfig: isReclaim symb: configSymbol toValue: aVal.
  self commitTransaction.
] on: Error do:[:ex |
  System abortTransaction .
  ex pass
].
^result

]

{ #category : 'Debugging Support' }
System class >> _setPrintStackAtError [
  "Set a flag in the VM so that a gem process will print error information
   and a Smalltalk stack to the gem log file when any error is generated."

  self _zeroArgPrim:102

]

{ #category : 'Configuration File Access' }
System class >> _sharedCacheAttributes [
"Returns an Array of attributes for the shared page cache to which the current
 session is attached as follows:

 1.  (String)       Name of the cache
 2.  (SmallInteger) Total number of 16Kb frames in the cache.
 3.  (Integer)      Physical size of the shared memory segment in bytes.
 4.  (SmallInteger) Number of shared counters in the cache
                    (see config option SHR_PAGE_CACHE_NUM_SHARED_COUNTERS).
 5.  (SmallInteger) Maximum number of processes that can attach the cache
                    (see config option SHR_PAGE_CACHE_NUM_PROCS).
 6.  (SmallInteger) Spin lock count (see config option SHR_SPIN_LOCK_COUNT).
 7.  (SmallInteger) Large memory page policy
                    (see config option SHR_PAGE_CACHE_LARGE_MEMORY_PAGE_POLICY).
 8.  (SmallInteger) Large memory page size in megabytes
                    (see config option SHR_PAGE_CACHE_LARGE_MEMORY_PAGE_SIZE_MB).
 9.  (SmallInteger) Number of large memory pages used to create the cache.
10.  (SmallInteger) Target free frame count
                    (see config option SHR_TARGET_FREE_FRAME_COUNT).
11.  (Boolean)      Cache locked in memory
                    (see config option SHR_PAGE_CACHE_LOCKED).
12.  (Boolean)      True if the shared cache is remote from the stone, false otherwise.
13.  (Boolean)      True if the shared cache is a mid-level shared cache, false otherwise.
Returns nil if the current session is a solo session.
"
^ self _zeroArgPrim: 187

]

{ #category : 'Shared Counters' }
System class >> _sharedCounter: n decrementBy: i withFloor: f [

"The shared counter specified by n is decremented by i.
The new value of the counter will not be less than f .
Returns the new value of the counter.

The argument n must a SmallInteger in the range 0 to
(System _numSharedCounters - 1) .

The argument i must be a SmallInteger or LargeInteger in the range
0 to 2**63 - 1 (9223372036854775807).

f specifies the minimum final value of the shared counter.  Specifying a floor
of nil means the minimum value is INT64_MIN (-9223372036854775808, or -2**63),
which is the lowest possible value for any shared counter.  Attempting to
decrement any counter to a final value less than f will set the counter
to floor value f."

<primitive: 551>
| sz |
n _validateClass: SmallInteger.
i _validateClass: Integer .
(f == nil)
  ifFalse:[
     f _validateClass: Integer .
     (f < -9223372036854775808 or:[ f > 9223372036854775807 ])
   ifTrue:[f _error: #rtErrArgOutOfRange args: { -9223372036854775808 . 9223372036854775807 } ] .
  ].
sz := self _numSharedCounters .
(n < 0 or:[ n >= sz ])
  ifTrue:[ n _error: #rtErrArgOutOfRange args: { 0 . sz - 1 } ] .
(i < 0 or:[ i > 9223372036854775807 ])
  ifTrue:[i _error: #rtErrArgOutOfRange args: { 0 . 9223372036854775807 } ] .
self _primitiveFailed: #_sharedCounter:decrementBy:withFloor:
     args: { n . i . f }

]

{ #category : 'Shared Counters' }
System class >> _sharedCounterFetchValuesFrom: firstCounter to: lastCounter [

 "Returns an array containing the values from all shared counters starting
 with the counter at index firstCounter, up to and including the value
 from the counter lastCounter.

 Both arguments must be in the range 0...(System numSharedCounters - 1), and
 lastCounter must be greater than or equal to firstCounter."

<primitive: 589>
| sz |
firstCounter _validateClass: SmallInteger.
lastCounter _validateClass: SmallInteger.
(firstCounter < 0 or:[ firstCounter > lastCounter])
  ifTrue:[firstCounter _error: #rtErrArgOutOfRange args:{ 0 . lastCounter } ] .
sz := self _numSharedCounters .
(lastCounter < 0 or:[ lastCounter >= sz ])
  ifTrue:[lastCounter _error: #rtErrArgOutOfRange args:{ 0 . sz - 1 }].
self _primitiveFailed: #_sharedCounterFetchValuesFrom:to:
     args: { firstCounter . lastCounter }

]

{ #category : 'Private' }
System class >> _signalErrorStatus: aSignal [

"Returns status of the specified signal .
 Signals are specified by SmallIntegers, the following are defined:

 1   SignaledObjects
 2   SignaledAbort (if in transaction, reflects status after transation finishes)
 3   SignaledGemStoneSession
 4   SignaledFinishTransaction
 5   AlmostOutOfMemory (a Notification)
 6   SignalTranlogsFull
 9   AlmostOutOfMemoryError
 10  AlmostOutOfStackError
"
<primitive: 366>
self _primitiveFailed: #_signalErrorStatus: args: { aSignal } .
self _uncontinuableError

]

{ #category : 'Private' }
System class >> _signalFromGemStoneSession [

"Returns a 4 element Array containing information about a signal from
 another GemStone session:

 1.  The sessionSerialNumber of the session that sent the signal.
 2.  The signal value (a SmallInteger).
 3.  A signal message (a String).
 4.  A SmallInteger, number of additional signals pending in stone

 If there is no signal in the queue, returns nil."

<primitive: 369>
self _primitiveFailed: #signalFromGemStoneSession .
self _uncontinuableError

]

{ #category : 'Private' }
System class >> _simpleCommitForReclaim: includeDeadBool [

  "Calls stone to request a simple commit be performed.
   if includeDeadBool==true, returns true if both the
   number of dead objects and number of shadow pages needing reclaiming
   are zero .
   if includeDeadBool==false, returns true if
   the number of shadow pages needing reclaiming is zero .

   Has no effect on this session's transactional view."

<primitive: 290>

self _primitiveFailed: #_simpleCommitForReclaim: args: { includeDeadBool } .
self _uncontinuableError

]

{ #category : 'Session Control' }
System class >> _sleep: aTime [

"Sleep for aTime seconds.  aTime must be a positive SmallInteger.
 If aTime is zero, this method has no effect.

 Returns time left to sleep , which will be > 0 if interrupted.
 The ProcessorScheduler is not used, and other waiting GsProcesses
 will not execute during the sleep. Use a Delay or Semaphore to wait
 while allowing other GsProcesses to run."

<primitive: 370>
aTime _validateClass: SmallInteger .
aTime < 0 ifTrue: [ aTime _error: #errArgTooSmall args:{ 0 } ] .
^ self _primitiveFailed: #_sleep: args: { aTime }

]

{ #category : 'Session Control' }
System class >> _sleepMs: milliSecondsToSleep [

"Sleep for the number of milliseconds specified by milliSecondsToSleep.
 The argument milliSecondsToSleep must be a positive SmallInteger.
 If milliSecondsToSleep is zero, this method has no effect.
 Returns the receiver."

| timeLeft |
milliSecondsToSleep _isSmallInteger ifFalse:[
  milliSecondsToSleep _validateClass: SmallInteger
].
milliSecondsToSleep < 0 ifTrue:[
  milliSecondsToSleep _error: #errArgTooSmall args:{ 0 } ].
timeLeft := milliSecondsToSleep .
[ true ] whileTrue:[
  timeLeft := self __sleepMs: timeLeft .
  timeLeft <= 0 ifTrue:[ ^ self ].
]

]

{ #category : 'Private' }
System class >> _stoneClearCommitQueues [

"Cause all sessions waiting in stone's TransQueue or stone's RcTransQueue
 to have the session's call  System(C>>_primitiveCommit return with an error.
 Returns a SmallInteger, the number of sessions that were sent the error.

 For use in debugging commit deadlocks.

 Only SystemUser may execute this method, otherwise an error is generated.
"

^ self _zeroArgPrim: 43

]

{ #category : 'Private' }
System class >> _stoneCommandsReport [

"Returns a String describing heavily used commands to stone
 since the last report.  Result is non-empty only from slow stoned executable."

^ self _zeroArgPrim: 183

]

{ #category : 'Private' }
System class >> _stonePgsvrListeningPort [

 "Returns nil or the listening port of the multithreaded pgsvr
  on stone's host for this gem's pgsvr."

^ self _zeroArgPrim: 68

]

{ #category : 'Session Control' }
System class >> _stopHostAgents: aKind timeOut: waitSeconds [
| list midList endMs waitMs |
endMs := DateTime now millisecondsGmt + (waitMs := waitSeconds * 1000) .
list := self hostAgentSessions .
midList := { } .
1 to: list size do:[:j | | sessionId descr |
  "first stop the midcache hostagents on mid cache hosts"
  sessionId := list at: j .
  descr := self descriptionOfSession: sessionId.
  (descr at: 17) ifNotNil:[:str |
     (str at: 1 equals: 'midcache') ifTrue:[
       list at: j put: nil .
       midList add: sessionId ; add: (descr at: 9) .
       self _stopHostAgentSession: sessionId kind: aKind .
     ].
  ].
].
aKind == 16r4"Break" ifTrue:[ | midEndMs waitingFor  |
  "Wait for a clean logout from the mid hostagents"
  midEndMs := DateTime now millisecondsGmt + (10000 max:( waitMs // 2)) .
  waitingFor := midList size // 2 .
  [ waitingFor > 0 ] whileTrue:[
    Delay waitForMilliseconds: 100 .
    1 to: midList size by: 2 do:[:n |
      (midList at: n) ifNotNil:[ :sessId || serialNum descr |
        serialNum := midList at: n + 1 .
        descr := self descriptionOfSession: sessId.
        ((descr at: 1"userPro") == nil or:[ (descr at: 9) ~~ serialNum]) ifTrue:[
           midList at: n put: nil .
           waitingFor := waitingFor - 1.
         ].
      ]
    ].
    DateTime now millisecondsGmt > midEndMs ifTrue:[ waitingFor := 0 ].
  ].
].
list do:[:sessionId |
   sessionId ifNotNil:[ self _stopHostAgentSession: sessionId kind: aKind ]
].
[ list size > 0 ] whileTrue:[
  (DateTime now millisecondsGmt > endMs )
    ifTrue:[ Error signal: list size asString , ' hostagents could not be stopped'].
  Delay waitForMilliseconds: 100 .
  list := self hostAgentSessions
]

]

{ #category : 'Session Control' }
System class >> _stopHostAgentSession: sessionId  kind: aKind [

| serialNum |
sessionId _validateClass: SmallInteger.
serialNum := GsSession serialOfSession: sessionId .
self _stopSession: serialNum kind: aKind timeout: -1

]

{ #category : 'Session Control' }
System class >> _stopSession: aSerialNumber  kind: aKind timeout: aSeconds [

"Aborts the transaction of the specified session (a SmallInteger), then
 terminates that session.  Returns the receiver.  If the indicated session is
 not active, no operation is performed.

 aKind has the following bits
   16r1  include GcGem sessions
   16r2  include SymbolCreation session
   16r4  include Hostagent sessions

   16r10  timeout aSeconds per terminateSession:timeout: ,
	  and do not inhibit   restart of gc gems or symbol gem

 To execute this method, you must have explicit privilege from your system
 data curator."

<primitive: 332>
aSerialNumber _validateClass: SmallInteger.
aSeconds _validateClass: SmallInteger.
aKind _validateClass: SmallInteger.
aKind _error: #rtErrArgOutOfRange .
^ self _primitiveFailed: #_stopSession:kind:timeout:
       args: { aSerialNumber . aKind . aSeconds }

]

{ #category : 'Private' }
System class >> _systemLocksPrim [

"For use in implementation of System(C) | systemLocks.
 This is an expensive method, it makes a call to stone for each element
 of the result of
    System currentSessions

 Returns an Array of Associations.
 Each Association has key sessionId,
  and value an Array of size 3.
  The elements of the Array are each Arrays ,
    first element  -  objects that are read locked
    second element -  objects that are write locked
    third  element -  objects for which unlock is deferred while a commit is in progress
 "
<primitive: 353>

self _primitiveFailed: #_systemLocksPrim .
self _uncontinuableError

]

{ #category : 'Performance Monitoring' }
System class >> _tempObjSpaceMax [

"Returns the approximate maximum number of bytes of temporary object memory which is
 usable for storing objects."

^self _zeroArgPrim: 78

]

{ #category : 'Performance Monitoring' }
System class >> _tempObjSpacePercentUsed [

"Returns the approximate percentage of temporary object memory which is
 in use to store temporary objects.  This is equivalent to the
 expression:

 (self _tempObjSpaceUsed * 100) // self _tempObjSpaceMax.

 Note that it is possible for the result to be slightly greater than 100%.
 This result indicates temporary memory is almost completely full."

^self _zeroArgPrim: 79

]

{ #category : 'Performance Monitoring' }
System class >> _tempObjSpacePercentUsedLastMark [

"Returns the approximate percentage of temporary object memory which is
 in use to store temporary objects, as of the last MarkSweep."

^self _zeroArgPrim: 86

]

{ #category : 'Performance Monitoring' }
System class >> _tempObjSpaceUsed [

"Returns the approximate number of bytes of temporary object memory being used
 to store objects."

^self _zeroArgPrim: 77

]

{ #category : 'Time' }
System class >> _timeGmt: aBoolean [

"Returns a SmallInteger, the time since January 1, 1970, in seconds.
 aBoolean is true, call Stone and update our offset from Stone's clock,
 otherwise compute the current time locally using the offset from Stone's time
 that was cached in the session at login.

 The result is a posixSeconds value as used in class DateAndTimeANSI ."

<primitive: 371>
^ self _primitiveFailed: #_timeGmt: args: { aBoolean }

]

{ #category : 'Time' }
System class >> _timeGmt2005: aBoolean [

"Returns a SmallInteger, the time since January 1, 2005, in seconds.
 aBoolean is true, call Stone and update our offset from Stone's clock,
 otherwise compute the current time locally using the offset from Stone's time
 that was cached in the session at login."

<primitive: 598>
^ self _primitiveFailed: #_timeGmt2005: args: { aBoolean }

]

{ #category : 'Time' }
System class >> _timeGmt95: aBoolean [

"Returns a SmallInteger, the time since January 1, 1995, in seconds.
 aBoolean is true, call Stone and update our offset from Stone's clock,
 otherwise compute the current time locally using the offset from Stone's time
 that was cached in the session at login."

self deprecated: 'System class>>_timeGmt95 is deprecated v3.0. Use timeGmt2005'.
^self _timeGmt95Prim: aBoolean.

]

{ #category : 'Time' }
System class >> _timeGmt95Prim: aBoolean [

"Returns a SmallInteger, the time since January 1, 1995, in seconds.
 If aBoolean is true, call Stone and update our offset from Stone's clock,
 otherwise compute the current time locally using the offset from Stone's time
 that was cached in the session at login.

 Deprecated, use   System (C) >> _timeGmt2005:   .
"
<primitive: 381>
^ self _primitiveFailed: #_timeGmt95: args: { aBoolean }

]

{ #category : 'Time' }
System class >> _timeGmtFloat [

"Returns a Float representing the time since January 1, 1970 in units
 of seconds, to microsecond resolution, or such resolution as provided
 by the operating system's gettimeofday() call.  This time is computed locally
 in the session process, using the offset from Stone's time that was
 cached in the session at login.

 The result is a posixSeconds value as used in class DateAndTimeANSI ."

^ self _zeroArgPrim: 13

]

{ #category : 'Time' }
System class >> _timeGmtMicroSeconds [
  "Returns a SmallInteger representing time since January 1, 1970 in microseconds,
  or such resolution as provided
  by the operating system's gettimeofday() call.  This time is computed locally
  in the session process, using the offset from Stone's time that was
  cached in the session at login."

^ self _zeroArgPrim: 188

]

{ #category : 'Time' }
System class >> _timeMs [

"Returns a SmallInteger representing the time since January 1, 1970 in milliseconds.
 The result is a SmallInteger equivalent to
    (System _timeGmtFloat * 1000) asInteger

 The result is computed locally in the session process, using the offset
 from Stone's time that was cached in the session at login.

 Gs64 v2.2, changed to no longer rollover to zero after 524287999 "

<primitive: 651>
^ self _primitiveFailed: #_timeMs

]

{ #category : 'Time' }
System class >> _timeMsLegacy [

"Returns a SmallInteger representing the current relative time in milliseconds.
 The result is a SmallInteger equivalent to
    (System _timeGmtFloat * 1000) asInteger \\ 524288000

 The result is computed locally in the session process, using the offset
 from Stone's time that was cached in the session at login."

^ self _zeroArgPrim: 12

]

{ #category : 'Time' }
System class >> _timeUs [

"Returns a SmallInteger between 0 and 999,999 representing the current number
 of microseconds that have passed within the current second."

^ self _zeroArgPrim: 103 .

]

{ #category : 'Private' }
System class >> _trapMarkDirty: anObject [

 "Returns the previous value of the mark dirty trap, or nil.
  anObject may be one of
    any committed object that does not have special format,
    nil  to disable the printing,
    the SmallInteger 1 to print any invariant object that is not a class.
  When the specified object is added to the dirtyList of the
  outer level transaction, C and Smalltalk stacks will be printed to
  topaz -l stdout, or to the gem log file."

<primitive: 1087>

]

{ #category : 'Private' }
System class >> _uncommittedDataPages [

"Returns an Array of the data pages referenced from the session's shadowed
 view of GemStone.  After the next commit or abort, the contents of this
 Array are invalid and may be incomplete.  That is, very shortly after the
 commit or abort operation, they may not show all data pages, because more pages
 may be added by the currently running methods.  However, it might be useful to
 check on modified objects (in particular, reclustering) before the transaction
 commits."

^ self _zeroArgCateg2Prim: 11

]

{ #category : 'Private' }
System class >> _unpublish: server on: machine [
  "Description:
     Unpublishes information about a server on the given machine.
   Input:
     server <String>: name of server to unpublish.
     machine <String or nil>: name of machine to unplubish on or nil if local.
   Result <Boolean>:
     true if unpublish was successful.
     false if server was not published.
   Exceptions:
     Illegal input raises various errors.
     Error 2367 raised if low level C code fails with system error.
  "

<primitive: 497>
self _primitiveFailed: #_unpublish:on: args: { server . machine }

]

{ #category : 'Cache Statistics - Global Session' }
System class >> _updateGlobalSessionStat: index by: i overwrite: aBool [

"Global session statistics are similar to session statistics except that global
 statistics are accessible by every logged in session on every host.  Global
 sessions statistics are not transactional.  Therefore every session will see the
 same value in a given statistic regardless of its transactional view.  Reads and
 updates to these statistics are guaranteed to be atomic.

 This method updates the global session statistic at the specified index (which must be
 in the range 0 to 47) by the specified value i, which must be a SmallInteger.
 If overwrite is true, then the current value of the  statistic is replaced by i.
 If overwrite is false, then the current value of the  statistic is incremented by i.

 Returns the new value of the statistic."

<primitive: 518>
index _validateClass: SmallInteger.
i _validateClass: SmallInteger.
aBool _validateClass: Boolean.
self _primitiveFailed: #_updateGlobalSessionStat:by:overwrite:
     args: { index . i . aBool } .

]

{ #category : 'Persistent Counters' }
System class >> _updateSharedCounterAt: index by: amount withOpCode: code [

"Persistent shared counters provide a means for multiple sessions
 to share common integer values.  There are 1536 persistent shared
 counters, numbered from 1 to 1536 (the index of the first counter is 1).

 Each update to a persistent shared counter causes a roundtrip
 to the stone process.  However reading the value of a counter is
 handled by the gem (and its page server, if any) and does not
 cause a roundtrip to the stone.

 Persistent shared counters are globally visible to all sessions on all
 shared page caches.

 Persistent shared counters hold 64 bit values and may be set to any
 signed 64 bit integer value.  No limit checks are done when
 incrementing or decrementing a counter.  Attempts to increment/decrement
 the counter above/below the minimum/maximum value of a signed
 64-bit integer will cause the counter to 'roll over'.

 Persistent shared counters are independent of database transactions.
 Updates to counters are visible immediately and aborts have no
 effect on them.

 The values of all persistent shared counters are written to the
 primary database extent at checkpoint time.  Updates between
 checkpoints are written to the transaction log by the stone.
 Therefore the state of the persistent shared counters is
 recoverable after a crash, restore from backup, and restore from
 transaction logs.

 Persistent shared counter performance is affected by the stone
 configuration option STN_COMMITS_ASYNC.  Setting this option
 to TRUE will result in faster update performance because
 the stone will respond to update requests after the tranlog
 write has been queued but before it completes.  Operating
 in this mode leaves a small chance of losing data should the
 stone or host machine crash after the tranlog write was queued
 but before it completes.  If this value is set to FALSE, the
 stone will only respond to update requests after the tranlog
 write has completed."

<primitive: 669>
| sz |
index _validateClass: SmallInteger .
amount _validateClass: Integer .
code _validateClass: SmallInteger .
sz := self numberOfPersistentSharedCounters .
(index < 1 or:[ index > sz ])
  ifTrue:[ ^index _error: #rtErrArgOutOfRange args:{ 1 . sz }] .
(code < 0 or:[ code > 1])
  ifTrue:[ ^code _error: #rtErrArgOutOfRange args: { 0 . 1 }] .
(amount < -9223372036854775808 or:[amount > 9223372036854775807])
  ifTrue:[ ^amount _error: #rtErrArgOutOfRange args:{ -9223372036854775808 . 9223372036854775807 }] .
^self _primitiveFailed: #_updateSharedCounterAt:by:withOpCode:
      args: { index . amount . code }

]

{ #category : 'Private' }
System class >> _updateSignalErrorStatus: aSignal toState: newState [

"This method changes the generation of the error for the specified signal
 generation is disabled.  Signals are specified by SmallIntegers, and the
 following are defined:

 1.  SignaledObjects
 2.  SignaledAbort       (delayed effect if currently in transaction)
 3.  SignaledGemStoneSession
 4.  SignaledFinishTransaction
 5.  AlmostOutOfMemory (a Notification) (disable/enable/enable and set threshold)
 6   SignalTranlogsFull
 7.  AlmostOutOfMemory threshold (read/set independent of enabling)
 8   both SignaledAbort and SignaledFinishTransaction
 9   AlmostOutOfMemoryError  ((disable/enable/enable and set threshold)
 10  AlmostOutOfStackError
"
<primitive: 367>
self _primitiveFailed: #_updateSignalErrorStatus:toState:
     args: { aSignal . newState } .
self _uncontinuableError

]

{ #category : 'Host System Access' }
System class >> _validatePasswordForUser: uid password: pw [

<primitive: 864>
uid _validateClass: String .
pw _validateClass: String .
^ self _primitiveFailed: #validatePasswordForUser:password: args: { uid . pw }

]

{ #category : 'Private' }
System class >> _validateTransaction [

" Determine whether the current transaction could commit.

  See   _primitiveCommit:  for documentation of the return values.

  If the result is > 0 , the current transaction cannot commit and
  the transaction conflict hidden sets have
  been updated to reflect the conflicts preventing commit .  "


^ self _zeroArgPrim: 32

]

{ #category : 'Version Management' }
System class >> _version [

"Returns a String describing the versions of the pieces of the running GemStone
 system."

| result |
result := String new .
result addAll: 'GEMSTONE: ' ; addAll: self _gemVersion ;
	addAll: '; IMAGE: ' ; addAll: self _imageVersion .
^ result

]

{ #category : 'Private' }
System class >> _versionParameterDict [

"Returns a Dictionary of names for version information parameters.  The
 dictionary keys are Strings.  Its values are SmallInteger parameter IDs."

"gemstone64 changed  result keys from Symbols to Strings, to reduce creation
  of new Symbols at runtime. "

| result cfgId cfgName |
result := StringKeyValueDictionary new .
cfgId := 0 .
[
  cfgName := self _configParameterName: cfgId kind: $V .
  cfgName == nil ifFalse:[
    result at: cfgName put: cfgId .
    cfgId := cfgId + 1 .
    ] .
  cfgName == nil
  ] untilTrue .

^ result

]

{ #category : 'Private GC' }
System class >> _vmInstanceCounts: anInt [
"
 Report on object memory usage by instances of each in-memory class.
 (anInt bitAnd: 16r10)  if 1 returns a String,
                        if 0 prints to gem stdout or topaz -l  output file
 (anInt bitAnd: 16r20)  if 1 report is in CSV format, 0 is legacy format.

 (anInt bitAnd:7) = 0   all of object memory
           1   new generation (young temporary objects)
           2   pom generation (copies of committed objects)
           3   old generation (old temporary objects)
           4   perm generation (loaded classes)
           5   code generation (loaded methods)
"
<primitive: 153>
| arg |
anInt _validateClass: SmallInteger.
arg := anInt bitAnd: 16rF .
(arg < 0 or:[ arg > 5 ])
  ifTrue:[ anInt _error: #rtErrArgOutOfRange args:{ 0 . 5 } ].
self _primitiveFailed: #_vmInstanceCounts: args: { anInt }

]

{ #category : 'Private GC' }
System class >> _vmInstanceCountsArray: anInt [

"Returns an Array of Associations. Each Association
 is of the form  Class->{ instanceCount . byteSize } .
 Returns nil if argument is invalid.
 The argument specifies a portion of object memory to
 include in the results
   anInt = 0   all of object memory
           1   new generation (young temporary objects)
           2   pom generation (copies of committed objects)
           3   old generation (old temporary objects)
           4   perm generation (loaded classes)
           5   code generation (loaded methods)"
<primitive: 652>
anInt _validateClass: SmallInteger.
(anInt < 0 or:[ anInt > 5 ])
  ifTrue:[ anInt _error: #rtErrArgOutOfRange args:{ 1 . 4 } ].
self _primitiveFailed: #_vmInstanceCountsArray: args: { anInt }

]

{ #category : 'Private GC' }
System class >> _vmInstanceCountsReport: anInt [

"Returns a String which is
 a report about how many instances of each in-memory class
 are present in the specified portion of temporary object memory.

   (anInt bitAnd: 16rF)
           0   all of object memory
           1   new generation (young temporary objects)
           2   pom generation (copies of committed objects)
           3   old generation (old temporary objects)
           4   perm generation (loaded classes)
           5   code generation (loaded methods)

  (anInt bitAnd: 16r20)  if 1 report is in CSV format, 0 is legacy format.

   If covering all of object memory , also reports those classes
   which have zero instances in memory , otherwise classes with
   zero instances are not printed.  The report contains one
   line per class, with className, number of instances, and total
   bytes occupied by the instances.
"
^ self _vmInstanceCounts: (anInt bitOr: 16r10)

]

{ #category : 'Private GC' }
System class >> _vmMarkSweep [

"Explicitly triggers in-memory markSweep collection of temporary objects.
 Should only be used when debugging internal problems. Frequent use
 can degrade performance."

^ self _zeroArgPrim: 29

]

{ #category : 'Private GC' }
System class >> _vmMarkSweepCode [

"Explicitly triggers in-memory markSweep collection of temporary objects,
 and attempts to also collect code and perm space.
 Should only be used when debugging internal problems. Frequent use
 can degrade performance."

  self _vmRequestCodePermGc .
  2 timesRepeat:[  self class ].  "service code gen gc interrupt here"

]

{ #category : 'Private GC' }
System class >> _vmRequestCodePermGc [
  "Request code_gen and perm_gen gc. 
   Caller should contain a backwards branch for interrupt service"

  ^ self _zeroArgPrim: 200

]

{ #category : 'Private GC' }
System class >> _vmPrunePomGen [
  "Prunes 100% of the pom generation of temporary object memory.
   Equivalent to the pruning specified by
    (System gemConfigurationAt:#GemPomGenPruneOnVote) == 100"

  ^ self _zeroArgPrim: 201
]

{ #category : 'Private GC' }
System class >> _vmPrintInstanceCounts: anInt [

"Print to VM's stdout or  topaz -l  output file
 a report about how many instances of each in-memory class
 are present in the specified portion of temporary object memory.

 The argument specifies the portion of temporary object memory
 to include in the report.
   (anInt bitAnd 16rF)
           0   all of object memory
           1   new generation (young temporary objects)
           2   pom generation (copies of committed objects)
           3   old generation (old temporary objects)
           4   perm generation (loaded classes)
           5   code generation (loaded methods)

   (anInt bitAnd: 16r20)  if 1 report is in CSV format, 0 is legacy format.

   If printing all of object memory , also prints those classes
   which have zero instances in memory , otherwise classes with
   zero instances are not printed.  The printout contains one
   line per class, with className, number of instances, and total
   bytes occupied by the instances.
"

^ self _vmInstanceCounts: (anInt bitAnd: 16r2F)

]

{ #category : 'Private' }
System class >> _vmStEntryCount [
"returns a SmallInteger,  omPtr->currStack->stEntryCount_() "

^ self _zeroArgPrim: 196
]

{ #category : 'Private' }
System class >> _writeLockWriteSubset [

"Start with the object manager dirty list ,
 and perform filtering of objects already locked, putting the results
 in a new hidden set (writeLockWriteSubset).  Next, attempt to acquire a
 write lock on the objects in the hidden set.  If all locks are acquired
 cleanly, return true.  If all locks acquired but some are dirty, return
 an array of dirty locked objects.  If not all locks could be acquired,
 return false (locks that were acquired, including dirty locks are available
 in the writeLockWriteSubset hidden set.  Any locks that are acquired are
 also put in the CommitOrAbortReleaseLocksSet."

^self _zeroArgPrim: 34

]

{ #category : 'Private' }
System class >> _writtenObjects [

"Returns an Array which is the result of enumerating
 the dirty list of the outer level transaction."

^ self _dirtyListOp: 2 id: 0

]

{ #category : 'Private' }
System class >> _zeroArgCateg2Prim: opcode [

"Private."

"opcode 0 = activeRepositories
        1 = _gemBuildSha
        2 = _gemVersionNum
        3 = _gemVersion
        4 = _hostAgentVersionString
        5 = (not used)
        6 = clusterAllSymbols
        7 = userActionReport - returns an Array of SymbolAssociations
        8 = configConstantsArray - returns an Array of Assocations
        9 = systemUserActionReport
       10 = _committedDataPages
       11 = _uncommittedDataPages"

<primitive: 325>

self _primitiveFailed: #_zeroArgCateg2Prim: args: { opcode } .
self _uncontinuableError

]

{ #category : 'Private' }
System class >> _zeroArgPrim: opcode [

"Private."

"opcode 0 = removeLocksForSession
        1 = stackLimit
        2 = pageReads
        3 = pageWrites
        4 = transactionMode   (also sent by GsCurrentSession)
        5 = transactionLevel   (also sent by GsCurrentSession)
        6 = maxSessionId
        7 = myCacheProcessSlot
        8 = shutDown
        9 = continueTransaction
       10 = _readClock   _gemCpuTimeMs
       11 = stoneName
       12 = _timeMsLegacy
       13 = _timeGmtFloat
       14 = _generationScavenge
       15 = _objectTableCleanup
       16 = _hostWaitForDebugger  HostExplicitWaitForDebugger
       17 = voteState  (gets GC voteState from stone)
       18 = startAllGcSessions
       19 = voteStateString
       20 = clientIsRemote
       21 = sessionsReferencingOldestCr
       22 = _disallowSubsequentCommit  (Rc failure)
       23 = _approxOopHighWaterMark
       24 = _sharedAttached (result always zero)
       25 = _hostCallDebugger   HostCallDebugger
       26 = stackDepth
       27 = stackDepthHighwater
       28 = writtenObjects
       29 = _vmMarkSweep
       30 = _commitResult
       31 = _deadNotReclaimed
       32 = _validateTransaction
       33 = session
       34 = _writeLockWriteSubset
       35 = _removeLockWriteSubset
       36 = _approxOopNumberHighWaterMark
       37 = _disallowCommitClassModFailure
       38 = clearEpochGcState
       39 = _clearDeadObjects
       40 = stopAllGcGems
       41 = forceEpochGc
       42 = stopAdminGcGem
       43 = _stoneClearCommitQueue
       44 = stopSymbolCreationSession,
       45 = startSymbolCreationSession,
       46 = symbolCreationSessionId,
       47 = _lastGsErrorNumber
       48 = currentUserSessionCount
       49 = currentSessionCount
       50 = Repository pageSize
       51 = _enteringIncompleteIndexRemoval (controls assertions in C, returns previous state)
       52 = _leavingIncompleteIndexRemoval (controls assertions in C, returns previous state)
       53 = getHighAllocatedOopNum
       54 = _numPersistentObjsModified
       55 = _numPagesInSharedOt
       56 = _objectsInNscEnumCache
       57 = _cpuClockTicksPerSecond
       58 = _dumpFrameData
       59 = myPageServerProcessId
       60 = resumeCheckpoints
       61 = checkpointStatus
       62 = startCheckpointSync
       63 = startCheckpointAsync
       64 = adminGemSessionId
       65 = reclaimGemSessionCount
       66 = reclaimGemSessionId
       67 = startAdminGem
       68 = _stonePgsvrListeningPort
       69 = _midCachePgsvrListeningPort
       70 = _currentUserSessionCountIncludingLogout
       71 = _breakSerialization
       72 = _gemCommitConflictDetails
       73 = cacheStatusCount from stone
       74 = _remoteSharedCacheCount
       75 = _enableTraceNewPomObjs
       76 = _disableTraceNewPomObjs
       77 = _tempObjSpaceUsed
       78 = _tempObjSpaceMax
       79 = _tempObjSpacePercentUsed
       80 = failOverStatus
       81 = _getSessionMethodsChangedForCommit
       82 = _getSessionMethodsChangedForAbort
       83 = _currentSessionsIncludingLogout
       84 = _inProtectedMode
       85 = currentSessions
       86 = _tempObjSpacePercentUsedLastMark
       87 = _comPrintOpcodesEmitted
       88 = _comPrintOpcodesNotEmitted
       89 = _printCHeap
       90 = _locale
       91 = _maxClusterId
       92 = _comClearOpcodesEmitted
       93 = _oopHighWaterMark
       94 = _oopNumberHighWaterMark
       95 = timeNs
       96 = _updateObjectsRead
       97 = _disableTraceObjectsRead
       98 = _resumeTraceObjectsRead
       99 = _enableTraceObjectsRead
      100 = _numSharedCounters
      101 = _objectTableSizeInMegabytes
      102 = _setPrintStackAtError
      103 = _timeUs
      104 = logout
      105 = inContinueTransaction
      106 = gemIsBigEndian
      107 = stoneIsBigEndian
      108 = latestCommitRecordPageId
      109 = sessionsReferencingOldestCrInTransaction
      110 = sessionsReferencingOldestCrNotInTransaction
      111 = myReadPageServerProcessId
      112 = commitsDisabledUntilLogout
      113 = commitsDisabledUntilAbort
      114 = continuationsEnabled   (always true)
      115 = _lastReservedOopNumber
      116 = listenForDebugConnection
      117 = fetchSystemStatNames
      118 = fetchSystemStats
      119 = _hostTimes
      120 = disableStoneGemTimeout
      121 = enableStoneGemTimeout
      122 = millisecondsSinceTransactionBoundary
      123 = millisecondsSinceLogin
      124 = secondsSinceStoneStarted
      125 = _enableTracePagesAccessed
      126 = _resumeTracePagesAccessed
      127 = _updatePagesAccessed
      128 = _disableTracePagesAccessed
      129 = refreshTransientSymbolList
      130 = hostCpuUsage
      131 = hostId
      132 = fast   System gemConfigurationAt:#GemConvertArrayBuilder
      133 = stoneMessageKinds
      134 = commitsSinceCurrentView
      135 = hostCpuCount
      136 = gcLocksCount
      137 = sessionIsOnStoneHost
      138 = cacheStatisticsForAllShort
      139 = cacheStatisticsForAll
      140 = failOverToSlave
      141 = commitRestore
      142 = restoreStatusInfo
      143 = _restoreDestroyOt
      144 = commitRestoreWithLostData
      145 = reclaimedSymbols
      146 = begin nested transaction
      147 = commit nested transaction
      148 = abort nested transaction
      149 = stoneStartupId
      150 = gemLogFileName
      151 =  get logsender or logreceiver sessionId compatible with pre v3.7.2 image
      152 = linkedGciLibraryName64
      153 = rpcGciLibraryName64
      154 = numberOfPersistentSharedCounters
      155 = suspendCommits
      156 = resumeCommits
      157 = profMonSample
      158 = gemProcessId
      159 = gemLogPath
      160 = commandLineArguments
      161 = terminateAllSessionsReferencingOldestCr
      162 = stopAllSessionsReferencingOldestCr
      163 = hostname
      164 = oldestCommitRecordAgeMs
      165 = allZombieSessions
      166 = allZombieSessionsRefOldestCr
      167 = realUserId
      168 = realUserIdName
      169 = effectiveUserId
      170 = effectiveUserIdName
      171 = loginUserId
      172 = loginUserIdName
      173 = startNewLoginLog
      174 = loginLogFileName
      175 = sessionIsRpc
      176 = pageServerConnectionIsEncrypted
      177 = gciClientConnectionIsEncrypted
      178 = readClockNano
      179 = refPathCleanup
      180 = myKerberosPrincipal
      181 = getKeyfileAttributes
      182 = fullyQualifiedDomainName
      183 = _stoneCommandsReport
      184 = _maxPrimitiveNumber
      185 = repositoryPublicKeys
      186 = repositoryPrivateKey
      187 = _sharedCacheAttributes
      188 = _timeGmtMicroSeconds
      189 =  used by pageManagerProcessSlot 
      190 = _hostWaitForDebuggerIfSlow
      191 = startNewObjectReadLog
      192 = flushObjectReadBuffer
      193 = _objectReadLogInfo
      194 = _sessionTrackReads
      195 = objectReadTrackingEnabled
      196 = _vmStEntryCount
      197 = call OmDebugToggle 
      198 = _suspendCommitsForDebug
      199 =  _stoneCpuTimeMs    stone cpu time
      200 = _vmRequestCodePermGc
      201 = _vmPrunePomGen
      202 = commitsDisabled
      203 = killLogSender
      204 = killLogReceiver
      205 = _debugIndexingCode
      206 = _keyFilePermissions
      207 = _writeWriteConflictsIsEmpty
      208 = _readWriteConflicts_or_writeDepConflicts
      209 = killLogSenderSplitLogs
      210 = get logsender, logreceiver sessionIds 

     1036 is  freezeCache, must be SystemUser
     1037 is  RDbfTestPgsvrSegv , must be SystemUser
     1038 = stoneCommandLineArgs
     1039 = extentEncrpytionScheme
     4001 = cacheStatisticsDescriptionForMonitor
     4002 = cacheStatisticsDescriptionForStone
     4004 = cacheStatisticsDescriptionForPageServer
     4008 = cacheStatisticsDescriptionForGem
     4128 = cacheStatisticsDescriptionForPageManager
     4512 = cacheStatisticsDescriptionForGemThreads
"

<primitive: 98>

self _primitiveFailed: #_zeroArgPrim: args: { opcode } .
self _uncontinuableError

]

{ #category : 'Transaction Control' }
System class >> abort [

^ System abortTransaction

]

{ #category : 'Transaction Control' }
System class >> abortAll [
"Abort all levels of nested transactions and then
 do an outer level abortTransaction."

 [ (self _zeroArgPrim: 5) > 1 ] whileTrue:[
   self _abortNestedTransaction.
 ].
 ^ self abortTransaction

]

{ #category : 'Transaction Control' }
System class >> abortTransaction [

"If in a nested transaction (System transactionLevel > 1) ,
 aborts the current nested transaction.
 and leaves the session in the parent level of transaction.
 The in-memory state of persistent objects modified during the nested transaction
 is rolled back to the in-memory state as of the start of the nested transaction.
 Temporary objects which were part of the closure of the outer level transaction
 at the start of this nested transaction, and were modified during this
 nested transaction, will be rolled back to their state as of the start of
 this nested transaction.

 If in an outer level transaction, (System transactionLevel <= 1) ,
 rolls back all modifications made to committed GemStone objects and provides
 the session with a new view of the most recently committed GemStone state.
 The roll back and new view happen whether or not the session was previously in a
 transaction.  If the transaction mode is set to #autoBegin, then a new
 transaction is started.  If the transaction mode is set to #manualBegin, then
 a new transaction is not started."

 (self _zeroArgPrim: 5) > 1 ifTrue:[
   self _abortNestedTransaction.
   ^ self
 ] ifFalse:[ | coordinator |
  ^ (coordinator := self _commitCoordinator) == nil
    ifTrue: [ self _localAbort ]
    ifFalse: [ coordinator abort ].
 ].

]

{ #category : 'System Control' }
System class >> activeRepositories [

"Returns an Array containing references to the repositories that are
 attached at the time the message is sent."

^ self _zeroArgCateg2Prim: 0

]

{ #category : 'System Control' }
System class >> addAllToStoneLog: srcString [

"Appends text to the Stone's informational log file.  First, this method writes
 a banner that identifies the session from which aString came.  It then appends
 aString itself.  The argument must be convertable to a Utf8, and the total
 number of bytes of the converted string be 16270 or less."

| aString  |


aString := srcString encodeAsUTF8.
aString basicSize < 16271  "truncate message if too long"
	ifFalse: [(aString := aString size: 16200) addAll: '..truncated..' encodeAsUTF8].

^ self _addAllToStoneLog: aString


]

{ #category : 'Deprecated' }
System class >> adminGcGemSessionId [

"Return the session ID of the Admin GC session.  Returns
 zero if the Admin GC session is not running."

self deprecated: 'System class >> adminGcGemSessionId deprecated v3.2.  Use adminGemSessionId instead'.
^self adminGemSessionId

]

{ #category : 'Garbage Collection Management' }
System class >> adminGemConfigs [
"Returns a String that documents the admin gem configuration options"

^'#adminVerboseLogging
   Controls the amount of logging information written to the log file.
   If set to false, normal logging information will be written.  If set to
   true, extra logging information is written to the log file.
     (default: false)

 #epochGcMaxThreads
   MaxThreads used for next epochGc
     (default:1, min:1, max: 32)

 #epochGcPageBufferSize
   Size in pages of buffer used for epoch GC (must be power of 2)
     (default: 64, min:8, max 1024)

 #epochGcPercentCpuActiveLimit
   "Limit active epoch threads when system percentCpuActive is above this limit
     (default: 90, min: 0, max: 100)

 #epochGcTimeLimit
   Controls the maximum frequency of epochs, in seconds. It is recommended that
   this value not be less than 1800 (i.e. 30 minutes), since the aging of objects
   faulted into gem memory uses 5 minute aging for each of 10 subspaces of the pom
   generation.  Ideally epochGcTimeLimit should be several hours.
     (default: 3600, min: 5, max: 2147483647)

 #epochGcTransLimit
   Minimum number of commits that must have occurred in order to start an
   epoch garbage collection.
     (default: 5000, min: 0, max: 2147483647)

 #saveWriteSetUnionToFile
   If true, causes the Admin GC gem to write out the write set union bitmap to
   a binary file before sweeping the union.  The file will be placed in the
   same directory as the Admin GC gem log file.
     (default: false)

 #sweepWsUnionMaxThreads
   MaxThreads used for next wsUnion sweep
     (default: 1 min: 1, max: 32)

 #sweepWsUnionPageBufferSize
   Size in pages of buffer used for wsUnion sweep.
     (default: 64, min: 8, max: 1024)

 #sweepWsUnionPercentCpuActiveLimit
   Limit active wsUnion threads when system percentCpuActive is above this limit.
     (default: 90, min: 0, max: 100)
'

]

{ #category : 'Garbage Collection Management' }
System class >> adminGemSessionId [

"Return the session ID of the Admin GC session.  Returns
 zero if the Admin GC session is not running."

^self _zeroArgPrim: 64

]

{ #category : 'Session Control' }
System class >> allZombieSessions [

"Returns a list of session IDs that are partially logged out and are in the process of being
 disposed by the stone.  These zombie sessions may or may not reference a commit record."

^ self _zeroArgPrim: 165

]

{ #category : 'Session Control' }
System class >> allZombieSessionsRefOldestCr [

"Returns a list of session IDs are partially logged out and are in the process of being
 disposed by the stone which also reference the oldest commit record."

^ self _zeroArgPrim: 166

]

{ #category : 'Signals' }
System class >> almostOutOfMemoryErrorThreshold [

"Returns a positive percentage of memory full at which the AlmostOutOfMemory
 error will be raised, or -1  if the error is not enabled. "

^self _signalErrorStatus: 5

]

{ #category : 'System Control' }
System class >> applyKeyfileNamed: arg [

"Causes the stone to read the keyfile with file name arg.  If arg is nil, the
 stone will use the same file name used when the stone was started.

 Requires the SystemControl privilege.  The stone must have permission to
 open the given keyfile for reading.

 The new keyfile permissions apply only as long as the stone is running.
 To permanently use the new keyfile, modify the KEYFILE option in the
 stone configuration file.

 Returns true upon success.  In this case, the keyfile was read and one or more
 changes were made to the system.  The system is now running as if it was
 restarted with the new keyfile.

 Returns false if the keyfile was successfully read but contained no
 meaningful changes from the current configuration.  For example,
 if the only change from the previous keyfile is a new license number.

 Raises an exception if the keyfile could not be read, has expired,
 or if applying it would modify parameters which may not be changed
 at runtime."

<primitive: 1031>
arg ifNotNil:[
  arg _validateClass: String
].
^ self _primitiveFailed: #applyKeyfileNamed:

]

{ #category : 'Transaction Control' }
System class >> beginNestedTransaction [
"Enter a new nested transaction.
 If session is outside of a transaction, equivalent to beginTransaction.
 Signals a ImproperOperation exception if the begin would exceed
 16 levels of nested transactions."

 (self _zeroArgPrim: 5) == 0 ifTrue:[
    ^ self beginTransaction
 ] ifFalse:[
   (self _zeroArgPrim: 146 "_beginNested") ifFalse:[
      ImproperOperation signal:'attempt to exceed 16 levels of nested transactions'
   ].
   ^ self
 ]

]

{ #category : 'Transaction Control' }
System class >> beginTransaction [

"Starts a new transaction for the session.  An abort is done before the new
 transaction is started - giving the session a new snapshot of the repository.

 If any permanent objects had been written by the session, their
 state is aborted.  This method returns the receiver (System).

 If in a nested transaction (System transactionLevel > 1) ,
 an error is signaled. and no change is made to the transaction state."

| coordinator |
^ (coordinator := self _commitCoordinator) == nil
  ifTrue: [ self _localBeginTransaction ]
  ifFalse: [ coordinator beginTransaction ].

]

{ #category : 'Cache Statistics' }
System class >> cacheName: aString [

"Sets the name the current session is known by in the shared cache
 to 'aString'. Does nothing if the current session is not attached
 to a shared cache. Raises an error if 'aString' is too long.
 Returns self."

^ self _cacheName: aString

]

{ #category : 'Cache Statistics' }
System class >> cacheSlotForProcessId: anInt [

"Return the cache slot at being used by the given process ID.
 Return -1 if the slot was not found."

^ self _primCacheStatsForSlotOrSession: anInt opCode: 5

]

{ #category : 'Cache Statistics' }
System class >> cacheSlotForSessionId: aSessionId [

"Return the cache process slot number (a SmallInteger) for the given session
 ID.  The session must be connected to the same shared page cache as the
 session invoking this method.

 A return of nil indicates the session could not be located."

<primitive: 567>
aSessionId _validateClass: SmallInteger.
aSessionId _error: #rtErrArgOutOfRange .
^ self _primitiveFailed: #cacheSlotForSessionId: args: { aSessionId }

]

{ #category : 'Deprecated' }
System class >> cacheStatistics: aProcessSlot [

"Returns an Array whose contents are described by the result of the
 cacheStatisticsDescription method.  The Array contains statistics for the
 specified slot in the GemStone shared memory cache to which this session is
 attached.

 The argument aProcessSlot should be a SmallInteger between 0 and the number of
 process slots in the shared cache minus 1, inclusive.  If aProcessSlot is
 outside the range of valid process slots, or the session executing this
 method is not using a shared cache, generate an error.  If the slot specified
 by aProcessSlot is an inactive slot, returns nil.  The method
 cacheSlotForSessionId: may be used to determine the process slot of a session
 on the same shared cache.

 The process slots that are predefined are:

    slot 0: The shared page cache monitor.

    slot 1: The Stone if the cache is on the same machine as the Stone.
            Otherwise, a page server that is used to monitor the cache for
            the Stone.

 No other slots are guaranteed.  However, slot 2 is the often the page server
 and slot 3 is often the GcGem.  These depend to some extent on the relative
 speed of the processes during startup.  In addition, the GcGem can be
 shut down, and when it is restarted, it is unlikely to end up at the same
 position.

 This method is deprecated and may be removed in a future release.
 It is replaced by the following classmethod in class System:

  cacheStatisticsAt:
"

self deprecated: 'System class>>cacheStatistics: deprecated v3.0.
Use #cacheStatisticsAt: instead'.
 ^self _primCacheStatsForSlotOrSession: aProcessSlot opCode: 0

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsAt: aProcessSlot [

"Returns an array of statistics appropriate to the specific executable (common + unique).
 Per-process host statistics are not included, see hostStatisticsForProcess: <pid>.
 The argument aProcessSlot is the same as described in #'cacheStatistics:' method.
 The contents are described by the result of sending element 4 of the result to the
 #cacheStatisticsDescriptionFor: method."

^ self _primCacheStatsForSlotOrSession: aProcessSlot opCode: 1

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsDescriptionAt: aSlot [

"Returns an array of strings which describe the cache statistics for the
 kind using the cache slot with index anInt.  Returns an empty array if
 the slot is not in use."

| kind |
kind := self processKindForSlot: aSlot .
^ kind == -1
    ifTrue:[ Array new ]
    ifFalse:[ self cacheStatisticsDescriptionForType: kind ]

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsDescriptionForGem [

"Returns an Array of Strings describing cache statistics applicable to a
 gem process."

^ self _zeroArgPrim: 4008

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsDescriptionForMonitor [

"Returns an Array of Strings describing cache statistics applicable to the shared
 page cache monitor."

^ self _zeroArgPrim: 4001

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsDescriptionForPageServer [

"Returns an Array of Strings describing cache statistics applicable to a
 page server process."

^ self _zeroArgPrim: 4004

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsDescriptionForStone [

"Returns an Array of Strings describing cache statistics applicable to the
 stone."

^ self _zeroArgPrim: 4002

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsDescriptionForType: aProcessKind [

"Returns an Array of Strings describing the result of the method #'cacheStatisticsAt:'. The
 argument <aProcessKind> is the StatType (as returned as element #4 by #'cacheStatisticsAt:')
 and must be one of the following:

   1 = Shared Page Cache Monitor
   2 = Stone
   4 = Remote gem page server
   8 = Gem main thread (including Topaz, GBS, and other GCI applications).
 128 = Page Manager thread
 256 = Stone restore thread
 512 = A gem thread
1024 = Stone Aio thread
2048 = Stone free frame thread
4096 = Remote cache page server thread
8192 = Remote gem page server thread

 The result includes only those appropriate to the specific executable (common and unique).
 Note that Statmonitor will capture additional statistics (e.g., on itself and the system)
 that are not actually stored in shared memory."

^ self _zeroArgPrim: aProcessKind + 4000

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsForAllSlots [

"Return an Array with one entry for each process attached to the shared
 cache.  Each element in the result contains the cache statistis for that
 slot.  The last element in each array is the slot ID of the slot.
 The subarrays will be different sizes depending on what type of process
 is using the slot."

^ self _zeroArgPrim: 139

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsForAllSlotsShort [

"Return an Array with one entry for each process attached to the shared
 cache.  Each element in the result is an array containing 5 elements.
 The first 4 elements are the same first 4 elements returned when fetching
 cache statistics for a session, slot, or process.  The fifth element is the
 process slot for the process.  Each sub-array of 5 elements contain the
 following elements:

  1 - ProcessName (String)
  2 - ProcessId   (SmallInteger)
  3 - SessionId   (SmallInteger)
  4 - StatType    (SmallInteger)
  5 - Cache Slot  (SmallInteger)
"

^ self _zeroArgPrim: 138

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsForProcessWithCacheName: aString [

"Search the cache for the first process with cache name matching aString.
 If found, return the statistics appropriate for that process. Returns nil if 
 the process with the given cache name was not found. 
 Per-process host statistics are not included, see hostStatisticsForProcess: <pid>."

^ self _primCacheStatsForSlotOrSession: aString opCode: 3

]

{ #category : 'Deprecated' }
System class >> cacheStatisticsForSessionId: aSessionId [

"Same as the cacheStatistics: method except the argument is the session ID
 of a session currently connected to the shared memory cache.  In systems
 that use multiple shared memory caches, the session must exist on the same
 cache as the session invoking this method.

 nil is returned if the session cannot be located.

 This method is deprecated and may be removed in a future release.
 It is replaced by the following classmethod in class System:

  gemCacheStatisticsForSessionId:
"

self deprecated: 'System class>>cacheStatisticsForSessionId: deprecated v3.0.
Use #gemCacheStatisticsForSessionId: instead'.
 ^self _primCacheStatsForSlotOrSession: aSessionId negated opCode: 0

]

{ #category : 'Cache Statistics' }
System class >> cacheStatisticsProcessId: aPid [

"Search the cache for a process id matching aPid.  If found, return the
 statistics appropriate for that process.  Returns nil if the process
 was not found. Per-process host statistics are not included, see 
 hostStatisticsForProcess: <pid>."

^ self _primCacheStatsForSlotOrSession: aPid opCode: 2

]

{ #category : 'Cache Statistics' }
System class >> cacheStatsForGemWithName: aStringOrSymbol [

"This method does a case-sensitive search for the first gem with the given
 cache name and returns an array of cache statistics for the gem if found.
 Returns nil if the gem could not be located. Per-process host statistics 
 are not included, see hostStatisticsForProcess: <pid>."

<primitive: 565>
aStringOrSymbol _validateClass: String.
^ self _primitiveFailed: #cacheStatsForGemWithName: args: { aStringOrSymbol }

]

{ #category : 'Cache Statistics' }
System class >> cacheStatsForPageServerWithSessionId: anInt [

"Return the cache statistics for the page server with the given session ID.  
 Page servers assume the same session ID as their client gem.
 Return -1 if the page server with the given session ID was not found.
 Per-process host statistics are not included, see hostStatisticsForProcess: <pid>."

^ self _primCacheStatsForSlotOrSession: anInt opCode: 8

]

{ #category : 'Shared Cache Management' }
System class >> cacheStatusCount [
  "Returns a positive SmallInteger value.
   The value maintained in the stone process increments
   whenever a remote cache starts up , shuts down, or becomes a midCache"

^ self _zeroArgPrim: 73

]

{ #category : 'Session Control' }
System class >> cacheWarmerSessions [
  "Returns an Array of sessionIds of cache warmer sessions.
   Requires SessionAccess privilege."
  ^ self currentSessions select:[:sessId |
      ((self descriptionOfSession: sessId) at: 17) = 'cachewarmer'
    ].
]

{ #category : 'Session Control' }
System class >> cacheWarmerSessionsReport [
  "Returns a String describing cache warmer sessions.
   Requires SessionAccess privilege."
  ^ self _sessionsReport: self cacheWarmerSessions .
]

{ #category : 'Authorization' }
System class >> canRead: anObject [

"In Gemstone64 , an object must be faulted into memory before
 it can be passed as an argument to a method.  Thus by definition
 if you enter this method you can read the argument. "

^ true

]

{ #category : 'Authorization' }
System class >> canWrite: anObject [

"This method tests whether the user has authorization to write anObject without
 adding it to write set and returns a Boolean result."

<primitive: 103>
self _primitiveFailed: #canWrite: args: { anObject } .
self _uncontinuableError

]

{ #category : 'Private' }
System class >> changeCacheSlotFreeFrameLimit: aSlot to: aValue [

"Changes the freeFrameLimit for the process associated with a cache slot to
 aValue, which must be a positive SmallInteger.  The argument aSlot
 should be a SmallInteger between 0 and the number of process slots in the
 shared cache minus 1, inclusive.

 To execute this method, you must have the SystemControl privilege."

<primitive: 349>

aSlot _validateClasses:{ SmallInteger }.
aValue _validateClasses:{ SmallInteger }.
self _primitiveFailed: #changeCacheSlotFreeFrameLimit:to:
     args: { aSlot . aValue }

]

{ #category : 'Garbage Collection Management' }
System class >> changeNumberOfReclaimGemSessions: num [

  ^ self changeNumberOfReclaimThreads: num
]

{ #category : 'Garbage Collection Management' }
System class >> changeNumberOfReclaimThreads: num [

"This method attempts to reconfigure the system to have num reclaim
 threads running and returns the actual number that it is able to
 accommodate. If num is smaller than the current number of reclaim
 threads, then threads are stopped and num is returned.  If the value
 of num is 0 and the reclaim gem is running then it is stopped.
 If num is larger than the current number of reclaim threads and
 if the number is less than STN_MAX_GC_RECLAIM_SESSIONS, then threads
 are started until that number are running and the value of num
 is returned.  If num is larger than STN_MAX_GC_RECLAIM_SESSIONS then the
 system may be able to accommodate the request by using some of the
 cache slots allocated for user sessions (from STN_MAX_SESSIONS).
 If this occurs then this method does not limit to STN_MAX_GC_RECLAIM_SESSIONS,
 and the value returned may be less than or equal to num depending on
 whether cache slots are currently considered available.

 A new value > 0 will be reflected in result of any subsequent
    System stoneConfigurationAt: #StnNumGcReclaimSessions .

 If the result of this method is more than STN_MAX_GC_RECLAIM_SESSIONS, then
 a warning is posted in the stone log indicating that user cache slots
 were allocated for reclaim threads.

 It is the callers responsibility to reset the number of reclaim threads to
 a value less than or equal to STN_MAX_GC_RECLAIM_SESSIONS to prevent users from
 getting login errors after the reclaim that needed more resources has
 completed.
"

<primitive: 287>
num _validateClass: SmallInteger.

]

{ #category : 'Online Backup Support' }
System class >> checkpointStatus [
"Returns an array of 2 elements.  The first element is a boolean indicating
 if checkpoints are currently suspended.  The second element is an
 Integer indicating the number of seconds before checkpoints will be
 resumed by the stone."


 ^self _zeroArgPrim: 61

]

{ #category : 'Garbage Collection Management' }
System class >> clearEpochGcState [

"Resets the epoch GC state which keeps track of objects eligible to be
 marked as possible dead during the next epoch GC operation.  Has no effect
 on objects already marked possible dead by previously completed epochs."

^self _zeroArgPrim: 38

]

{ #category : 'Reduced Conflict Support' }
System class >> clearRcValueCache [

"Clears the cache of calculated values for reduced conflict classes by setting
 the temporary session state slot to nil."

self __sessionStateAt: 5 put: nil

]

{ #category : 'Reduced Conflict Support' }
System class >> clearRedoLog [

"Clear the redo log by setting the temporary session state slot to nil.

                            Warning:
 Clearing the redo log will probably prevent Reduced Conflict
 classes from resolving conflicts.  Sending this message negates this
 capability for the current transaction.

 This is a protected method."

< protected >

self __sessionStateAt: 4 put: nil

]

{ #category : 'Environment Access' }
System class >> clientEnvironmentVariable: varName [

"Expands the environment variable named varName in the GemBuilder for C client
 process, returning a String.  The varName argument should be a kind of String.

 Returns nil if any of the following are true:

 * varName is not a byte format object.
 * There is no environment variable defined with name varName.
 * The value of the environment variable is more than approximately 8000 bytes.
 * The size of varName exceeds approximately 8000 bytes.

 Signals an error if NoGsFileOnServer privilege is set in the UserProfile.
 "

^ GsFile _expandEnvVariable: varName isClient: true

]

{ #category : 'Environment Access' }
System class >> clientEnvironmentVariable: varName put: valueStr [

"Sets the environment variable named varName in the GCI client process
 to value valueStr .
 The first argument must be either a String or a Unicode7 not larger than 8000 bytes.
 The second argument must be either a String or a Unicode7 not larger than 8000 bytes, 
 or nil (nil specifies  clearenv  semantics).
 The NoGsFileOnClient privilege must not be set in the session's UserProfile.

 Returns self or signals an Error if the operation failed.
"
GsFile _setEnvVariable: varName value: valueStr isClient: true.
^ self

]

{ #category : 'Session Control' }
System class >> clientIsRemote [

"Returns true if the GemBuilder for C client for this session is in a different
 process than the Gem, otherwise returns false."

^ self _zeroArgPrim: 20

]

{ #category : 'Version Management' }
System class >> clientVersionAt: aString [

"Returns version information about the client GemBuilder for C process.  If the
 client is a session using the linkable GemBuilder for C login, this method is
 equivalent to gemVersionAt:.

 See System(C) | gemVersionAt: for further details."

| verId |
verId := VersionParameterDict at: aString otherwise: nil .
verId ifNil:[ ^ nil ].
^ GsFile _clientVersionAt: verId

]

{ #category : 'Version Management' }
System class >> clientVersionReport [

"Return a StringKeyValueDictionary whose keys are the names of operating system,
 hardware, or GemStone version attributes, and whose values are the
 current values of those attributes in the client GemBuilder for C process."

| result |
result := StringKeyValueDictionary new .
VersionParameterDict keysAndValuesDo:[ :aName :anId | | aVal |
  aVal := GsFile _clientVersionAt: anId .
  aVal ~~ nil ifTrue:[
    result at: aName put: aVal
    ].
  ].
^ result

]

{ #category : 'Clustering' }
System class >> clusterAllSymbols [

"This method clusters the AllSymbols hash dictionary and all of the symbols to
 which it refers.
 The clustering is performed by the Symbol Creation Gem, and other
 sessions may see a long latency on symbol creation requests while
 the clustering is in progress."

| result |
GsFile gciLogServer:'-- sending clusterAllSymbols to Symbol Creation gem'.
result := self _zeroArgCateg2Prim: 6 .
GsFile gciLogServer:'-- waiting 10 seconds for Symbol Creation gem commit'.
20 timesRepeat:[
  self _sleepMs: 500 .
  System inTransaction ifFalse:[ System abortTransaction ].
].
^ result

]

{ #category : 'Clustering' }
System class >> clusterBucket: aClusterBucketOrId [

"This method sets the current default ClusterBucket to the ClusterBucket with
 the specified clusterId.  The argument may be an instance of ClusterBucket, or
 a positive SmallInteger which specifies an instance of ClusterBucket."

<primitive: 348>
self _primitiveFailed: #clusterBucket: args: { aClusterBucketOrId } .
self _uncontinuableError

]

{ #category : 'Host System Access' }
System class >> commandLineArguments [
  "Returns an Array of Strings, the command line arguments to
   the gem or topaz -l  process for this session.
   Each invocation of this method returns a new instance of the result."

  ^ self _zeroArgPrim: 160

]

{ #category : 'Transaction Control' }
System class >> commit [

"
 If session is not in transaction, signals an ImproperOperation exception.

 If in a nested transaction (System transactionLevel > 1) ,
 commits the current level of transaction
 and leaves the session in the parent level of transaction.

 The following applies if the current level is 1 (i.e. outer level):
     Attempts to update the persistent state of the Repository to include changes
     made by this transaction.  If the outer level commit succeeds,
     then this method returns true, and the current transaction's changes, if any,
     become a part of the persistent Repository.
     After the repository update, the session exits the current
     transaction.  If the transaction mode is #autoBegin, then the session enters
     a new transaction.  If the transaction mode is #manualBegin, then the session
     remains outside of a transaction.

     If conflicts prevent the repository update, then a TransactionError is signalled.
     Call the transactionConflicts method to determine the nature of the
     conflicts.  If the session is outside of a transaction, then this method
     signals the Error #rtErrPrimOutsideTrans.

     This method also updates the session's view of GemStone.  If the commit
     operation succeeds, then all objects in the session's view are consistent with
     the current state of GemStone.  If the commit fails, then this method retains
     all the changes that were made to objects within the current transaction.
     However, commits made by other sessions are visible to the extent that changes
     in this transaction do not conflict with them.

 Signals an error if   GsCurrentSession isSolo == true .

 Returns true if commit was read-only or succeeded .
 Signals an Error if there was a failure.  "

  (self _zeroArgPrim: 5) <= 1 ifTrue:[ "level 0 or 1"
    (self _commit: 0 "don't release locks") ifFalse:[
      TransactionError new reason: 'commitConflicts' ; signal:'commit conflicts'
    ].
  ] ifFalse:[
    self _commitNestedTransaction.
  ] .
  ^ true

]

{ #category : 'Transaction Control' }
System class >> commitAll [
"Commit all levels of nested transactions, if any, and then attempt
 to commit the outer level transaction.

 If conflicts prevent the outer level commit, a TransactionError is signalled.
 If session is outside of a transaction, an Error is signalled.
 Signals an error if   GsCurrentSession isSolo == true . "

 [ (self _zeroArgPrim: 5) > 1 ] whileTrue:[
   self _commitNestedTransaction.
 ].
 ^ self commit

]

{ #category : 'Transaction Control' }
System class >> commitAndReleaseLocks [

"Attempt to commit the transaction for the current session.

 If in a nested transaction (System transactionLevel > 1) ,
 an error is signalled and no change is made to the transaction state.

 If in an outer level transaction, this method is the same
 as 'commitTransaction' except for the handling of locks.
 If the commit succeeds, this method releases all locks for the session
 and returns true.  Otherwise, it returns false and does not release locks.

 This method also clears the commit release locks and commit-or-abort release
 locks sets.  See the 'Releasing Locks' method category for more
 information.

 Signals an error if   GsCurrentSession isSolo == true .

 Returns true if commit was read-only or succeeded ,
 false if there was a failure.  "

 (self _zeroArgPrim: 5) <= 1 ifTrue:[ "level 0 or 1"
   ^ self _commit: 1 "release locks" .
 ].
 ImproperOperation signal:'operation illegal while in a nested transaction'

]

{ #category : 'System Control' }
System class >> commitRecordBacklog [

"Return the number of commit records in the system."
^self stoneCacheStatisticWithName: 'CommitRecordCount'

]

{ #category : 'Session Control' }
System class >> commitRecordPageForSessionId: aSessionId [

"Return the page ID of the commit record referenced by the given session ID.
 The result will usually be a SmallInteger but could also be a
 LargeInteger.

 Returns -1 if the session does not exist or if it does not currently reference a
 commit record (such as after a Lost OT root error).

 To execute this method for any session other than your current session, you
 must have the SessionAccess privilege.
 Gs64 v3.5+  returns view.viewCommitRecord ; previously returned view.endingCr "

<primitive: 663>
aSessionId _validateClass: SmallInteger .
^ self _primitiveFailed: #commitRecordPageForSessionId: args: { aSessionId } .

]

{ #category : 'Transaction Control' }
System class >> commitsDisabledUntilAbort [

"Answers true if further commits have been disabled until the session aborts.
 Otherwise answers false.

 It is possible the session may still not be able to commit when this
 method returns true.  Possbile reasons are the session is not in transaction,
 commits are disabled at the UserProfile level, or commits have been disabled
 until the session logs out.

 Use the #sessionCanCommit method to definitively determine if the current session
 can commit."

^self _zeroArgPrim: 113.

]

{ #category : 'Transaction Control' }
System class >> commitsDisabledUntilLogout [

"Answers true if further commits have been disabled for the session.
 Otherwise answers false.

 It is possible the session may still not be able to commit when this
 method returns true.  Possbile reasons are the session is not in transaction,
 commits are disabled at the UserProfile level, or commits have been disabled
 only until the next abort.

 Use the #sessionCanCommit method to definitively determine if the current session
 can commit."

^self _zeroArgPrim: 112.

]

{ #category : 'Transaction Control' }
System class >> commitsDisabled [

"Answers true if further commits have been disabled for the session.
  either until abort or until logout .
 Otherwise answers false.

 It is possible the session may still not be able to commit when this
 method returns true.  Possbile reasons are the session is not in transaction,
 commits are disabled at the UserProfile level.

 Use the #sessionCanCommit method to definitively determine if the current session
 can commit."

^self _zeroArgPrim: 202.

]

{ #category : 'Transaction Control' }
System class >> commitsSinceCurrentView [

"Returns a SmallInteger which is the number of commits of
 transaction level 1  which have occurred since
 the session obtained its current view.  If the session has no current view due to
 a lost OT root error, then -1 is returned.  A result of 0 indicates the session
 has the most current view of the repository."

^ self _zeroArgPrim: 134

]

{ #category : 'Transaction Control' }
System class >> commitTransaction [

"Same as System(C)>>commit, except that if transaction conflicts are
  detected, returns false instead of signalling a TransactionError.
 Signals an error if   GsCurrentSession isSolo == true .
"
  (self _zeroArgPrim: 5) <= 1 ifTrue:[ "level 0 or 1"
    (self _commit: 0 "don't release locks") ifFalse:[
      ^ false "commit conflicts"
    ].
  ] ifFalse:[
    self _commitNestedTransaction.
  ] .
  ^ true

]

{ #category : 'Deprecated' }
System class >> concurrencyMode [

"This method is obsolete, concurrency mode is always #NO_RW_CHECKS"

self deprecated: 'System class>>concurrencyMode is obsolete in GemStone/S 64 Bit.'.

^ #NO_RW_CHECKS

]

{ #category : 'Configuration File Access' }
System class >> configurationAt: aName [

"Returns the value of the specified configuration file parameter, giving
 preference to the Gem process if the parameter applies to the Gem."

| result |
result := self gemConfigurationAt: aName .
result == nil ifTrue:[
  result := self stoneConfigurationAt: aName
  ].
^ result

]

{ #category : 'Runtime Configuration Access' }
System class >> configurationAt: aName put: aValue [

"Change the value of the specified configuration parameter.

 The changeable parameters all require aValue to be a SmallInteger.

 Configuration parameters should not be changed unless there is a clear
 reason for doing so, since incorrect settings of parameters can
 have serious adverse effects on GemStone performance.

 Configuration parameters for Stone that are transferred to Gem processes
 are only read by the Gem at login, so changes using this method to
 Stone parameters may have no effect on existing sessions.

 Parameters in the Gem with the following names may be changed by any user at
 any time:
   * #GemAbortMaxCrs
   * #GemCommitConflictDetails   (1 allows use of detailedConflictReportString, 2 enables trace of Rc replay)
   * #GemCommitStubsForNpObjects
   * #GemConvertArrayBuilder (takes effect at start of next method compile)
   * #GemDropCommittedExportedObjs
   * #GemExceptionSignalCapturesStack
   * #GemFreeFrameLimit
   * #GemFreePageIdsCache
   * #GemHaltOnError
   * #GemKeepMinSoftRefs
   * #GemKeyRingDirs
   * #GemNativeCodeEnabled (only can disable at runtime)
   * #GemPgsvrCompressPageTransfers
   * #GemPgsvrUpdateCacheOnRead
   * #GemPomGenPruneOnVote
   * #GemReadAuthErrStubs
   * #GemRepositoryInMemory
   * #GemSoftRefCleanupPercentMem
   * #GemTempObjOomstatsCsv
   * #GemTempObjConsecutiveMarksweepLimit
   * #GemTempObjPomgenScavengeInterval

 Parameters with the following names may be changed only by users who have
 the correct privilege:

   * #StnAdminGcSessionEnabled - Requires GarbageCollection privilege.
   * #StnDisableLoginFailureLimit - Requires OtherPassword privilege.
   * #StnDisableLoginFailureTimeLimit - Requires OtherPassword privilege.
   * #StnEpochGcEnabled - Requires GarbageCollection privilege.
   * #StnLoginsSuspended - Requires SystemControl privilege.
   * #StnLogLoginFailureLimit - Requires OtherPassword privilege.
   * #StnLogLoginFailureTimeLimit - Requires OtherPassword privilege.
   * #StnMaxLoginLockSpinCount - Requires SystemControl privilege.
   * #StnNumGcReclaimSessions - Requires GarbageCollection privilege.
   * #StnObjLockTimeout - Requires SystemControl privilege.
   * #StnRemoteCachePgsvrTimeout - Requires SystemControl privilege.
   * #StnSignalAbortCrBacklog - Requires GarbageCollection privilege.
   * #StnSignalAbortAggressive - Requires GarbageCollection privilege.
   * #StnSmcSpinLockCount - Requires SystemControl privilege.
   * #StnSymbolGcEnabled - Requires GarbageCollection privilege.
   * #StnTranLogDebugLevel - Requires SystemControl privilege.

 All other parameters that are changeable at run time may be changed
 only by SystemUser, and should not normally need to be changed in the
 course of GemStone operation."

| cfgId |

cfgId := ConfigurationParameterDict at: aName .
self _atConfigId: cfgId put: aValue .
^ self configurationAt: aName

]

{ #category : 'Transaction Control' }
System class >> conflictReportString [
"Return a String describing the result of  System(C)>>transactionConflicts"

^ self conflictReportString: self transactionConflicts

]

{ #category : 'Transaction Control' }
System class >> conflictReportString: conflicts [
"Return a String describing the objects that are causing commit conflicts. conflicts is the
 result of  System(C)>>transactionConflicts"

^ self conflictReportString: conflicts maxOops: 100

]

{ #category : 'Transaction Control' }
System class >> conflictReportString: conflicts maxOops: maxOops [
 "Report on objects that are associated with commit conflicts.
  The conflicts argument is Array returned by System(C)>>transactionConflicts .
  The report contains oops and class names for each object in conflict,
  limited to the first maxOops oops in each category of conflict.
  "
  | result |
  result := String new .
  #( #'Read-Write' #'Write-Write' #'Write-Dependency' #'Write-ReadLock'
	 #'Write-WriteLock' 'Rc-Retry-Failure' #'Synchronized-Commit' #RcReadSet
   ) do:[:aSym | | v sz |
      v := conflicts at: aSym otherwise: nil .
      sz := v size .
      sz == 0 
       ifTrue:[ aSym == #RcReadSet ifTrue:[ result add: 'RcReadSet empty'; lf ]]
       ifFalse:[  | limit |
        result add: sz asString; add: $  ; add: aSym .
        aSym == #RcReadSet 
          ifTrue:[  result add: ' Conflicts(' ; lf ]
          ifFalse:[ result add: ' Oops('; lf ].
        limit := sz min: maxOops .
        1 to: limit do:[ :n | | oopStr clsName  |
          clsName := ' ' .
          oopStr := '<authErr>' .
          [ | each | 
            each := v at: n .
            oopStr := each asOop asString .
            clsName := ' (a ' , each class name , $) .
          ] onException: SecurityError do:[:ex | 
            (ex gsArguments atOrNil: 1) ifNotNil:[:oop | 
               oopStr := oop asString .
               clsName := ' (authErr on class)'.
            ].
          ].  
          result add:' ' ; add: oopStr ; add: clsName .
	  (0 == (n \\ 8)) ifTrue:[ result lf ].
        ] .
        sz > maxOops ifTrue:[ result lf ; addAll:' ('; addAll: (sz - maxOops)  asString;
                            addAll:' more oops)' ].
        result add: $) ;  lf .
      ]
  ].
  ^ result

]

{ #category : 'Transaction Control' }
System class >> conflictsReport [

"Returns an Array describing conflicts or success of the last attempted commit.
 First element of result is a symbol describing the status.
 If status was not #readOnly or #success,
 the following elements are pairs of aKey,anArray  with each aKey
 describing the kind of conflict.  Empty sets are omitted from the result.

     Key                Conflicts
 Read-Write          StrongReadSet and WriteSetUnion conflicts.
 Write-Write         WriteSet and WriteSetUnion conflicts.
 Write-Dependency    WriteSet and DependencyChangeSetUnion conflicts.
 Write-WriteLock     WriteSet and WriteLockSet conflicts.
 Write-ReadLock      WriteSet and ReadLockSet conflicts.
 Rc-Retry-Failure    objects for which Rc replay failed.
 WriteWrite_minusRcReadSet  ((WriteSet and WriteSetUnion conflicts) - RcReadSet)
 RcReadSet           rcRead set for information

 see also System>>transactionConflicts for more documentation.
"

^ self _conflictsReport: false withDetails: false

]

{ #category : 'Deprecated' }
System class >> contentsOfServerDirectory: aSpecString [

"Obsolete: Use the GsFile>>contentsOfDirectory: onClient:  method instead."

| result |
self deprecated: 'System class>>contentsOfServerDirectory: deprecated v3.0.
Replace with GsFile class >> contentsOfDirectory:onClient:'.
result := GsFile contentsOfDirectory: aSpecString onClient: false.
result ifNil: [ ^ { }  ].
^ result

]

{ #category : 'Session Control' }
System class >> continuationsEnabled [

"Returns true if Smalltalk (i.e. Seaside style) continuations are enabled
 for this session."

^ true

]

{ #category : 'Transaction Control' }
System class >> continueTransaction [

"Updates the session's view to the most recently committed GemStone state.

 If in a nested transaction (System transactionLevel > 1) ,
 an error is signalled and no change is made to the transaction state.

 If the session is in an outer level transaction (System transactionLevel == 1)
 this method preserves modifications made to committed GemStone objects.
 The read and write sets of the session are carried forward and continue
 to accumulate until the session either commits or aborts.

 If the session is not in a transaction, this method performs an abort,
 discarding any accumulated changes to committed objects.

 If in an outer level transaction and a
 previous attempt to commit the transaction
 failed due to conflicts, then continueTransaction will generate
 error 2409.  After a failed attempt to commit, you must abort before
 continueTransaction can be used again.

 Returns true if accumulated modifications to the committed objects would not
 cause concurrency conflicts; otherwise returns false.  If the method
 returns false, you can call the transactionConflicts method to determine
 the nature of the conflicts.

 This method can be used whether or not the session is outside of a transaction.
 Of course, the session cannot commit accumulated changes unless it is
 inside a transaction.

 If transaction mode is #manualBegin, then continueTransaction does not alter
 the inside/outside of transaction state of the session.

 Modifications made by other committed transactions are accumulated for
 retrieval by GciDirtyObjs() and GciDirtySavedObjs() just as they are
 accumulated for commitTransaction or abortTransaction.

 This method has no effect on object locks.  Locks in the release locks sets
 are not released."

| continueResult |

(self _zeroArgPrim: 5) > 1 ifTrue:[
  ImproperOperation signal:'operation illegal while in a nested transaction'
].
continueResult := self _zeroArgPrim: 9.

continueResult <= 0 ifTrue: [
  continueResult >= -1 ifTrue:[
    ^ true  "read only or success"
  ].
] ifFalse: [
  continueResult == 1 ifTrue: [
    ^ self _resolveRcConflicts .
  ] ifFalse: [
    ^ false "validation failure"
    "3 means validation failure,
    4 means maximum number of commit retries was reached,
    5 means commit disallowed (usually because of indexes)"
  ]
].
self _uncontinuableError "logic error if we get here"

]

{ #category : 'Clustering' }
System class >> currentClusterBucket [

"Returns the instance of ClusterBucket that is the current default."

^ AllClusterBuckets at: self currentClusterId

]

{ #category : 'Clustering' }
System class >> currentClusterId [

"This method returns a SmallInteger that is the ID of the ClusterBucket that
 is the current default bucket."

<primitive: 96>
self _primitiveFailed: #currentClusterId .
self _uncontinuableError

]

{ #category : 'Deprecated' }
System class >> currentGcReclaimSessionsByExtent [

"Deprecated - all extents are covered by a single reclaim gem.
Return an array where the size is the number of extents, and in which
each element of the array is the session ID for the reclaim gem for that
extents. A 0 means no reclaim gcgems are running; 1 is used in v3.2 to
indicate that the extent has reclaim coverage."

self deprecated: 'System class>>currentGcReclaimSessionsByExtent deprecated in v3.2.'.
^(Array new: SystemRepository numberOfExtents)
     atAllPut: (self reclaimGemSessionCount max: 1);
     yourself.

]

{ #category : 'Runtime Configuration Access' }
System class >> currentObjectReadLogFile [
 "Answer a string containing the current object read log file which the stone
 is writing to, or nil if the object read logging feature is disabled."
   self _objectReadLogInfo ifNotNil:[:arr | ^ arr at: 1 ].
   ^ nil
]

{ #category : 'Session Control' }
System class >> currentObjectSecurityPolicy [

"Returns the GsObjectSecurityPolicy in which objects created in the current session
 are stored.  The result can be nil, in which case objects are
 being created with World write  permission.

 At login, the current security policy is the default security policy of the
 UserProfile for the session of the sender."

<primitive: 341>
self _primitiveFailed: #currentObjectSecurityPolicy .
^ nil

]

{ #category : 'Session Control' }
System class >> currentObjectSecurityPolicy: anObjectSecurityPolicy [

"Redefines the GsObjectSecurityPolicy in which subsequent objects created in the
 current session will be stored.  Returns the receiver.

 If the argument is nil , subsequent objects are created
 with World write permission.

 The argument must be a committed GsObjectSecurityPolicy and you must have
 writeAuthorization to that objectSecurityPolicy, otherwise an error is generated.
"

<primitive: 339>
anObjectSecurityPolicy _validateClass: GsObjectSecurityPolicy.
self _primitiveFailed: #currentObjectSecurityPolicy:
     args: { anObjectSecurityPolicy } .
self _uncontinuableError

]

{ #category : 'Deprecated' }
System class >> currentSegment [

self deprecated: 'System class>>currentSegment deprecated v3.0.
Use #currentObjectSecurityPolicy instead'.
^self currentObjectSecurityPolicy.

]

{ #category : 'Deprecated' }
System class >> currentSegment: anObjectSecurityPolicy [

self deprecated: 'System class>>currentSegment: deprecated v3.0.
Use #currentObjectSecurityPolicy: instead'.
^self currentObjectSecurityPolicy: anObjectSecurityPolicy

]

{ #category : 'Session Control' }
System class >> currentSessionCount [

"Return a SmallInteger which is the number of sessions present in the system,
 including the Symbol Gem, garbage collection sessions, but not the page manager
 session.  Gets the result from the stone process."

^self _zeroArgPrim: 49

]

{ #category : 'Session Control' }
System class >> currentSessionNames [

"Returns a formatted String containing, for each current GemStone session, the
 session number and userId.

 This method requires SessionAccess privilege if there is more
 than one session logged in."

| result profiles |
result := String new.
profiles := self _currentSessionProfiles.  "sessId, userProfile pairs"
1 to: profiles size by: 2 do:[:j| | sessId aUserProfile |
   sessId := profiles at: j .
   aUserProfile := profiles at: j + 1 .
   j > 1 ifTrue:[ result lf ].
   result
     addAll: 'session number: ';
     addAll: (sessId asString) .
   aUserProfile ifNotNil:[ 
     result addAll: (self _sessionUproString: aUserProfile) 
   ] ifNil:[ | desc |
     desc := System descriptionOfSession: sessId .
     result add: '  '; add: ((desc at: 17) ifNil:[ 'in login']) .
   ].
 ].
^ result

]

{ #category : 'Session Control' }
System class >> currentSessions [

"Returns an Array of SmallIntegers corresponding to all of the sessions
 currently running on the GemStone system.  Excludes sessions in before-login
 or after-logout states."

^self _zeroArgPrim: 85

]

{ #category : 'Session Control' }
System class >> currentSessionsReport [
  "Returns a String describing all sessions.
   Requires SessionAccess privilege."
  ^ self _sessionsReport: self currentSessions .

]

{ #category : 'Transaction Control' }
System class >> currentTransactionHasWDConflicts [

"Returns a Boolean indicating if the current transaction has one or
 more write-dependency conflicts.  A result of true indicates a commit
 will most likely fail.  A result of false indicates no write-dependency
 conflicts exist.

 It is inefficient to invoke this method and then invoke
 currentTransactionWDConflicts. If you want conflict details, invoke
 currentTransactionWDConflicts directly.
 "

 ^ self currentTransactionWDConflicts size ~~ 0

]

{ #category : 'Transaction Control' }
System class >> currentTransactionHasWWConflicts [

"Returns a Boolean indicating if the current transaction has one or
 more write-write conflicts.  A result of true indicates a commit
 will most likely fail.  However some RC objects have conflict
 resolution mechanisms which could allow a commit to suceed.
 A result of false indicates no write-write conflicts exist.

 It is inefficient to invoke this method and then invoke
 currentTransactionWWConflicts. If you want conflict details, invoke
 currentTransactionWWConflicts directly.
 "

 ^ self currentTransactionWWConflicts size > 0

]

{ #category : 'Transaction Control' }
System class >> currentTransactionWDConflicts [

"Returns an Array of objects which have write-dependency conflicts.
 The array is created by building the union of all dependency
 change sets from all commit records created since the session's
 current transaction and intersecting it with the session's write
 set."

 | status conflictDict wdConflicts |
 status := self _validateTransaction .
 status <= 0 ifTrue:[
   ^ #()  "read only or empty conflicts"
 ].
 conflictDict := self transactionConflicts .
 wdConflicts := conflictDict at: #'Write-Dependency' otherwise: #() .
 ^ wdConflicts .

]

{ #category : 'Transaction Control' }
System class >> currentTransactionWWConflicts [

"Returns an Array of objects which have write-write conflicts.
 The array is created by building the write set union of all
 commit records created since the session's current transaction
 and intersecting it with the session's write set."

 | status conflictDict wwConflicts |
 status := self _validateTransaction .
 status <= 0 ifTrue:[
   ^ #()  "read only or empty conflicts"
 ].
 conflictDict := self transactionConflicts .
 wwConflicts := conflictDict at: #'Write-Write' otherwise: #() .
 ^ wwConflicts .

]

{ #category : 'Session Control' }
System class >> currentUserSessionCount [

"Return a SmallInteger which is the number of user sessions present in the system.
 Garbage collection sessions, the Symbol Gem, and the Page Manager Gem are not
 included in the count.  Gets the result from the stone process."

^self _zeroArgPrim: 48

]

{ #category : 'System Control' }
System class >> deadNotReclaimedCount [

"Return the number of dead not reclaimed objects in the system."
^self stoneCacheStatisticWithName: 'DeadNotReclaimedObjs'

]

{ #category : 'Transient Session State' }
System class >> decrementIndexProgressCountBy: aSmallInt [

"Decrements the value of the indexProgressCount statistic by aSmallInt and
 returns the new value."

^ self _sessionCacheStatAt: -2 incrementBy: aSmallInt negated

]

{ #category : 'Transient Session State' }
System class >> decrementProgressCountBy: aSmallInt [

"Decrements the value of the progressCount statistic by aSmallInt and
 returns the new value."

^ self _sessionCacheStatAt: -1 incrementBy: aSmallInt negated

]

{ #category : 'Deprecated' }
System class >> deleteServerFile: aFileSpec [

self deprecated: 'System class>>deleteServerFile: deprecated v3.0. Use GsFile class>>removeServerFile: instead.'.
^self _deleteServerFile: aFileSpec.

]

{ #category : 'Session Control' }
System class >> descriptionOfSession: aSessionId [

"Returns an Array describing the session as follows:
 Most elements are zero or a String of size zero if no session exists for aSessionId.

 1.  The UserProfile of the session; nil if the UserProfile is recently
     created and not visible from this session's transactional view
     or the session is in login or processing, or has logged out.
     nil for a logsender,  logreceiver, or stone's internal session executing tranlog replay.
 2.  A SmallInteger, the process ID of the Gem or topaz -l process, zero if no
     process registered with stone.
 3.  If the gem is running on the same host as stone the value is 'localHost'.  If the
     gem is on a remote host then the value is the host's ipAddress.
 4.  Primitive number in which the Gem is executing, or 0 if it is not executing
     in a long primitive.
 5.  Time of the session's most recent beginTransaction, commitTransaction, or
     abortTransaction (from System timeGmt).
 6.  The session state (a SmallInteger).
 7.  A SmallInteger whose value is -1 if the session is in transactionless mode,
     0 if it is not in a transaction and 1 if it is in a transaction.
 8.  A Boolean whose value is true if the session is currently referencing the
     oldest commit record, and false if it is not.
 9.  The session's serial number (a SmallInteger).
 10. The session's sessionId (a SmallInteger), zero if no session exists.
 11. A String containing the ip address of host running the GCI process.
     If the GCI application is remote,  the peer address as seen by
     the gem of the GCI app to gem network connection.
     For a hostagent, this is the ip address of the remote host being serviced,
     otherwise if the GCI application is linked (using libgcilnk*.so or gcilnk*.dll)
     this is the peer's ip address as seen by stone, for the gem to
     stone network connection used for login.
 12. The priority of the session (a SmallInteger).
 13. Unique host ID of the host where the session is running (an Integer)
 14. Time of the session's most recent request to stone (from System timeGmt)
 15. Time the session logged in (from System timeGmt)
 16. Number of commits which have occurred since the session obtained its view.
 17. Nil or a String describing a system or gc gem , logsender, logreceiver, stone tranlog replay.
 18. Number of temporary (uncommitted) object IDs allocated to the session.
 19. Number of temporary (non-persistent) page IDs allocated to the session.
 20. A SmallInteger, 0 session has not voted, 1 session voting in progress,
     2 session has voted, or voting not active.
 21. A SmallInteger, processId of the remote GCI client process,
     or -1 if the session has no remote GCI client .
 22. The KerberosPrincipal object used for passwordless login to the session,
     or nil if passwordless login was not used.
 23. The sessionId of the hostagent session through which this session is
     communicating to stone, or -1 if session is not using a hostagent .
 24. SmallInteger listening port if this session is a hostagent, or -1 .
 25. gcLockKind , 0 or the type of gcLock or repository scan lock held by the session.
 26. UserProfile which created the onetime password for the session, or nil if
     no onetime password was used.
 27. Process ID of the gem that created the onetime password for the session, or -1 if
     no onetime password was used.
 28. a Boolean , true if the session is waiting for reclaimGem to acknowledge setting gcHighWater
     
 Because a session can update its commit record without committing a
 transaction, it is possible that no session actually references the oldest
 commit record.  Therefore, the eighth element may be false for all current
 sessions.

 To execute this method for any session other than your current session, you
 must have the SessionAccess privilege."

^ self _descriptionOfSessionSerialNum: 0 sessionId: aSessionId
]

{ #category : 'Session Control' }
System class >> descriptionOfSessionSerialNum: aSerialNumber [

"Returns an Array describing the session identified by aSerialNumber.

 See System (C) | descriptionOfSession: for documentation on the contents of the
 result Array.

 Requires SessionAccess privilege if aSerialNumber is not the current session."

^ self _descriptionOfSessionSerialNum: aSerialNumber sessionId: -1

]

{ #category : 'Transaction Control' }
System class >> detailedConflictReportString [
  "Return a String describing which commits by other sessions caused this
   sessions most recent commit attempt to fail."
| arr result printObjsBlk details maxO |
(details := self _gemCommitConflictDetails) == 0 ifTrue:[
  ^ 'No details.  You must execute
      System gemConfigurationAt: #GemCommitConflictDetails put: 1
   prior to attempting commit in order to get per-session conflict details.'
].
result := String new .
arr := self _conflictsReport: false withDetails: true .
(arr at: 1) == #success ifTrue:[
  ^ 'Commit succeeded' copy lf .
].
(arr at: 1) == #readOnly ifTrue:[
  ^ 'Commit was readOnly' copy lf .
].
maxO := details > 2 ifTrue:[ SmallInteger maximumValue ] ifFalse:[ 20 ].
printObjsBlk := [:objs :isRcDetail | | numObjs limit str cnt |
  str := '  ('  copy .
  limit := (numObjs := objs size) min: maxO .
  cnt := 0 .
  1 to: limit do:[ :n | | each clsName oopStr didObj |
    clsName := ' ' .
    oopStr := '<authErr>' .
    [ each := objs at: n .
      oopStr := each asOop asString .
      clsName := 'a ' , each class name  .
    ] onException: SecurityError do:[:ex | 
      (ex gsArguments atOrNil: 1) ifNotNil:[:oop | 
         oopStr := oop asString .  clsName := 'authErr on class'.
      ].
    ].  
    str add:' ' ; add: oopStr ; add: $( .
    (isRcDetail and:[ each ~~ nil ]) ifTrue:[
      each isSymbol ifTrue:[ str add: each printString . didObj := true ] ifFalse:[
      each _isOneByteString ifTrue:[ str add: (each copyFrom: 1 to: (each size min:100)).
                                    didObj := true ]].
    ].
    didObj ifNil:[ str add: clsName ].
    str add: $) .
    cnt := cnt + 1 .
    (cnt >= 5 and:[ cnt < limit]) ifTrue:[ str lf ].
  ].
  numObjs > limit ifTrue:[
     str addAll:' ('; addAll: (numObjs - limit)  asString; addAll:' more oops)' 
  ].
  str add: $) ;  lf .
  str
  ].

result add: 'Commit failed , ' ; add: (arr at: 1) asString ; lf .
result add: ((arr at: 2) ifNil:[ DateAndTime now asStringMs ]);
       lf .
3 to: arr size by: 3 do:[:j | | aSym objArr sessArr nObj |
  aSym :=   arr at: j .
  objArr := arr at: j + 1 .
  sessArr := arr at: j + 2 .
  nObj := objArr size .
  (nObj > 0 or:[ aSym == #RcReadSet ]) ifTrue:[ | nOtherSess |
     result add: nObj asString; add: $  ; add: aSym .
     aSym == #RcReadSet ifTrue:[ result add: ' Objects' ] ifFalse:[ result add:' Conflicts'].
     result lf .
     result add: ( printObjsBlk value: objArr
                            value: (aSym at: 1 equals: 'Rc-Retry-Failure') ).
     nOtherSess := sessArr size // 2 .
     nOtherSess > 0 ifTrue:[
       result add:'  '; add: nOtherSess asString ; add: ' commits by other sessions'; lf .
       1 to: sessArr size by: 2 do:[:k |
         result add:'  '; add: (sessArr at: k) ; lf .
         result add:'  '; add: ( printObjsBlk value: (sessArr at: k + 1) value: false ).
       ].
     ].
  ].
].
^ result
]

{ #category : 'Transaction Control' }
System class >> dirtyListId [

"Return a SmallInteger identifying the currently active dirtyList.
 Result is always zero."

^ self _zeroArgPrim:28

]

{ #category : 'Signals' }
System class >> disableAlmostOutOfMemoryError [

"Disables raising an error error when session's temporary object memory
 is almost full.

 See signalAlmostOutOfMemoryThreshold:  for more details. "

self _updateSignalErrorStatus: 5 toState: -1

]

{ #category : 'Session Control' }
System class >> disableCommits [

"Disables further commits by the session.  See the method #disableCommitsWithReason:
 for more information."

^ self disableCommitsWithReason: nil

]

{ #category : 'Session Control' }
System class >> disableCommitsUntilAbortWithReason: aString [

"Prevents this session from performing any further commits until the session
 performs an abort.  Attempting to commit a transaction after this method is
 successfully executed before aborting will raise an #rtErrCommitDisallowed
 error and will include 'aString' as the reason.

 aString must be an instance or subclass of String, or nil.  If aString is nil,
 a default reason of 'Commits disabled by user' will be used.

 Care should be taken when using this method.  If the session has any uncommitted
 changes, those changes will be lost after this method is executed.  Use the
 #needsCommit method to determine if uncommitted changes exist.

 This method returns true on success.  A return of false indicates commits were
 previously disabled by the user or by the system."

^ self _disableCommitsWithReason: aString untilAbort: true

]

{ #category : 'Session Control' }
System class >> disableCommitsWithReason: aString [

"Prevents this session from performing any further commits.  Attempting to
 commit a transaction after this method is successfully executed will raise
 an #rtErrCommitDisallowed error and will include 'aString' as the reason.

 This method disables commits only for the current session, not permanently.
 Commits may be performed if the session logs out and logs in again.  Use the
 UserProfile>>disableCommits method to permanently disable commits for a
 UserProfile.

 aString must be an instance or subclass of String, or nil.  If aString is nil,
 a default reason of 'Commits disabled by user' will be used.

 Care should be taken when using this method.  If the session has any uncommitted
 changes, those changes will be lost after this method is executed.  Use the
 #needsCommit method to determine if uncommitted changes exist.

 This method returns true on success.  A return of false indicates commits were
 previously disabled by the user or by the system."

^ self _disableCommitsWithReason: aString untilAbort: false

]

{ #category : 'Garbage Collection Management' }
System class >> disableEpochGc [
"Disables epoch garbage collection from running and resets
 the epoch GC state.  Appends the new configuration state
 to the stones configuration file.

 No further epoch GC operations will be run after this method is
 successfully executed.  An epoch GC operation already in progress
 when this method is executed will not be interrupted.  Has no
 effect if epoch GC is already disabled.

 Requires the GarbageCollection privilege."

^self stoneConfigurationAt: #StnEpochGcEnabled put: 0.

]

{ #category : 'Transaction Control' }
System class >> disableSignaledAbortError [

"Disables the generation of an error when Stone signals the Gem session that it
 should abort when running outside of a transaction."

self _updateSignalErrorStatus: 2 toState: false

]

{ #category : 'Transaction Control' }
System class >> disableSignaledFinishTransactionError [

"Disables the generation of the TransactionBacklog notification ."

self _updateSignalErrorStatus: 4 toState: false

]

{ #category : 'Notification' }
System class >> disableSignaledGemStoneSessionError [

"Set the current GemStone session so that it cannot receive signals from
 other GemStone sessions."

self _updateSignalErrorStatus: 3 toState: false

]

{ #category : 'Notification' }
System class >> disableSignalTranlogsFull [

"Disables generation of error 2339 to this session when stone detects
 a tranlogs full condition."

self _updateSignalErrorStatus: 6 toState: false

]

{ #category : 'Session Control' }
System class >> disableStoneGemTimeout [

"Prevents the current session from being killed due to inactivity
 (lack of communication with the stone).

 Use the #enableStoneGemTimeout method to re-enable the stone gem timeout.

 Returns true if the action was successful or false if the stone gem timeout
 was previously disabled for this session.

 Refer to the STN_GEM_TIMEOUT configuration parameter for more information.

 This sets the PrimitiveNumber statistic for this session to 9999 .
"

^ self _zeroArgPrim: 120

]

{ #category : 'Session Control' }
System class >> effectiveUserId [

"Returns a SmallInteger indicating the effective Unix user ID of the gem process."
^ self _zeroArgPrim: 169

]

{ #category : 'Session Control' }
System class >> effectiveUserIdName [

"Returns a String indicating the effective Unix user ID of the gem process."
^ self _zeroArgPrim: 170

]

{ #category : 'Signals' }
System class >> enableAlmostOutOfMemoryError [

"Enables or reenables error when session's temporary object memory
 is almost full , with previous threshold .

 See also AlmostOutOfMemory(C)>>enable: .
 You must install a handler for AlmostOutOfMemory to take action
 or execute
   Notification enableSignalling
 to have AlmostOutOfMemory and other Notifications signalled to the
 application to avoid having AlmostOutOfMemory silently ignored.

 See signalAlmostOutOfMemoryThreshold:  for more details.

 This method or  signalAlmostOutOfMemoryThreshold:  must be
 invoked after each delivery of the
 AlmostOutOfMemory error to reenable generation of the error.
 "

self _updateSignalErrorStatus: 5 toState: 0

]

{ #category : 'Garbage Collection Management' }
System class >> enableEpochGc [
"Enables epoch garbage collection to run and resets
 the epoch GC state.  Appends the new configuration state
 to the stones configuration file.

 Has no effect if epoch GC is already enabled.

 Requires the GarbageCollection privilege."

^self stoneConfigurationAt: #StnEpochGcEnabled put: 1.

]

{ #category : 'Transaction Control' }
System class >> enableSignaledAbortError [

"Enables the generation of an error when the Stone has signaled that the Gem
 process should abort to connect to a more current GemStone root.

 This method must be invoked after each delivery of the signal-abort error, to
 reenable generation of the error.

 If invoked when in a transaction, the new state will take effect
 after the next commit or abort which exits the transaction.
 "

self _updateSignalErrorStatus: 2 toState: true

]

{ #category : 'Transaction Control' }
System class >> enableSignaledFinishTransactionError [

"Enables the generation of an error when the Stone has signaled that a Gem
 process which is in-transaction should abort, commit, or continueTransaction
 to move to a newer transactional view.

 This method must be invoked after each delivery of the TransactionBacklog
 notification to reenable generation of the error."

self _updateSignalErrorStatus: 4 toState: true

]

{ #category : 'Notification' }
System class >> enableSignaledGemStoneSessionError [

"Enable the current GemStone session to receive signals from other GemStone
 sessions.  One GemStone session receives a signal from another session when a
 InterSessionSignal exception is signalled.

 The receiving session processes the signal with an exception handler.  When
 GemStone raises one signal exception, it also disables further signal
 exceptions, to allow the exception handler to run without receiving another
 interrupt.  The exception handler should therefore re-enable signal exceptions
 when it is done with its other processing.

 A signal is not exactly an interrupt, and it does not automatically awaken an
 idle session.  Both the GemStone Smalltalk virtual machine and GemBuilder for C
 can raise the signal exception.  But the process of the session must activate
 the virtual machine or interface before the signal can be received."

self _updateSignalErrorStatus: 3 toState: true

]

{ #category : 'Notification' }
System class >> enableSignalTranlogsFull [

"Enables generation of error 2339 to this session when stone detects
 a tranlogs full condition."

self _updateSignalErrorStatus: 6 toState: true

]

{ #category : 'Session Control' }
System class >> enableStoneGemTimeout [

"Re-enables the stone gem timeout for the current session after it was previously
 disabled by the #disableStoneGemTimeout method.

 Returns true if the action was successful.  Returns false if the stone gem timeout
 was already re-enabled or if it was not disabled for this session.

 Refer to the STN_GEM_TIMEOUT configuration parameter for more information."

^ self _zeroArgPrim: 121

]

{ #category : 'Host System Access' }
System class >> encryptPassword: pw withSalt: salt [

"Calls the C function crypt() to encrypt pw, which is assumed to be the
 clear text password to be encrypted.  Returns a new String which contains
 the encrypted password.

 Both the pw and salt arguments must be instances of String."

<primitive: 865>
pw _validateClass: String .
salt _validateClass: String .
^self _primitiveFailed: #encryptPassword:withSalt: args: { pw . salt }

]

{ #category : 'Garbage Collection Management' }
System class >> ensureGcRunning [
 "Makes sure that the reclaimGem and adminGem are running."

(self reclaimGemSessionCount == 0) ifTrue:[
  self startReclaimGem.
  ].
((SystemRepository restoreStatusInfo at: 2) == 0 and: [self adminGemSessionId == 0])
 ifTrue:[
    System startAdminGem .
  ].
^System waitForAllGcGemsToStartForUpToSeconds: 60 .

]

{ #category : 'Transaction Control' }
System class >> enumerateDirtyList: listId [

"Returns an Array which is the result of enumerating
 the dirtyList(s) specified by listId.
 listId = -1  specifies the currently active dirtyList .

 For listId >= 0,   listId + 1 is the  transaction level being enumerated,
 listId == 0 is the outer level transaction, listId == 1 is first level
 of nested transactions.
 When entering a nested transaction, the dirtyList of that transaction
 is initialized to be the closure of the dirtyList of the parent level.

 Example: to enumerate the closure of System _writtenObjects,
   System beginNestedTransaction .
   closureList := System enumerateDirtyList: -1 .
   System commit .
"

^ self _dirtyListOp: 2 id: listId

]

{ #category : 'Host System Access' }
System class >> fetchLinkedGciLibraryName [

"Returns a string containing the name of the linked 64-bit GCI shared library
 file for this version of the product.
 The string does not include a path to the file, only the file name."

^ self _zeroArgPrim: 152

]

{ #category : 'Host System Access' }
System class >> fetchRpcGciLibraryName [

"Returns a string containing the name of the RPC 64-bit GCI shared library
 file for this version of the product.
 The string does not include a path to the file, only the file name."

^ self _zeroArgPrim: 153

]

{ #category : 'Host System Access' }
System class >> fetchSystemStatNames [

"On systems that support it, returns an array of Strings which describe
 the statistics returned by the #fetchSystemStats method.  The length
 of the result array is host technology dependent.

 Returns nil if the host system does not support system statistics."

^ self _zeroArgPrim: 117

]

{ #category : 'Host System Access' }
System class >> fetchSystemStats [

"On systems that support it, returns an array of Numbers which describe
 the statistics returned by the #fetchSystemStats method.  The length
 of the result array is host technology dependent.  A description of
 each element in the array can be obtained using the #fetchSystemStatNames
 method.

 While most elements in the result array will be SmallIntegers, the
 result may also contain other types of Numbers such as SmallDoubles,
 Floats, LargeInteger, etc.

 Returns nil if the host system does not support system statistics."

^ self _zeroArgPrim: 118

]

{ #category : 'Private' }
System class >> flushAllExtents [

"Deprecated, has no effect, extents are only flushed to file system
 by the stone process at checkpoints."

self deprecated: 'System class>>flushAllExtents deprecated in v3.2. flush is done at checkpoints'.
^ self

]

{ #category : 'System Control' }
System class >> flushObjectReadBuffer [
  "Forces the gem to immediately send any buffered object read records to the stone.
   Has no effect if object read logging is not enabled.
   Returns the receiver."

  ^ self _zeroArgPrim: 192
]

{ #category : 'Garbage Collection Management' }
System class >> forceEpochGc [
"Force an Epoch GC to run as soon as possible, regardless of the setting
of the GcUser configuration parameters epochGc[Time|Trans]Limit.
Note that the stone configuration parameter STN_EPOCH_GC_ENABLED must be
TRUE for this method to have any effect.

This method will fail and return false under the following conditions:
  *Checkpoints are suspended
  *Another garbage collection operation is in progress
  *Unfinalized possible dead objects exist (i.e., System>>voteState
    returns any value except 0).
  *The system is in restore mode.
  *The Admin GC session is not running.
  *Epoch GC is not enabled (STN_EPOCH_GC_ENABLED is set to FALSE)
  *The system is performing a reclaimAll.
  *A previous forceEpochGc operation was performed and the epoch has
   not yet started or completed.

If successful, this method sets the stone cache statistic EpochForceGc
to 1.  Once the Admin GcGem has started the epoch GC, EpochForceGc
will return back to 0.

This method returns true if the epoch GC was started, false if not.

You must have GarbageCollection privilege to be able to run this method."

^self _zeroArgPrim: 41

]

{ #category : 'Host System Access' }
System class >> fullyQualifiedDomainName [

"Returns a string representing the fully qualified domain name (FQDN) of the
 host system for this session.  Raises an exception if the FQDN could not
 be determined."

^ self _zeroArgPrim: 182

]

{ #category : 'Garbage Collection Management' }
System class >> gcConfigurationReport [

"Returns a string describing the current garbage collection settings.
 Sender must have permission to read GcUsers UserGlobals."

| gcug keys result lf |

result := String new.
lf := Character lf.
result addAll: 'Reclaim and Admin GC gem settings:' ; add: lf.
gcug := ((AllUsers userWithId: 'GcUser') resolveSymbol: #UserGlobals) value.
(keys := gcug keys)
	remove: #NativeLanguage ifAbsent: [] ;
	remove: #UserGlobals ;
	remove: #GcUser ;
	remove: #defaultAdminConfigValues ;
	remove: #defaultReclaimConfigValues.

keys do:[:eachSymbol| | val |
  val := gcug at: eachSymbol.
  result add: $#; addAll: eachSymbol asString; addAll: ' -> '; addAll: val asString; add: lf.
].
result add: lf; addAll: 'Stone GC settings:' ; add: lf.
result add: $# ; add: 'StnEpochGcEnabled'; addAll: ' -> ' .
(System stoneConfigurationAt: #StnEpochGcEnabled) == 1
  ifTrue:[result addAll: 'true']
  ifFalse:[result addAll: 'false'].
result add: lf.

^result

]

{ #category : 'Runtime Configuration Access' }
System class >> gciClientConnectionIsEncrypted [

"Answer a Boolean indicating whether the network connection to the gem's GCI client
 encrypted (using TLS).  Always answers false for gems linked gems and for gems running
 on the same host as the GCI client."

^ self _zeroArgPrim: 177

]

{ #category : 'Gci Set Support' }
System class >> gciDirtyObjsInit [

"Enable GCI tracking of dirty objects, equivalent
 to GciDirtyObjsInit()."

^ self _gciDirtyInit: 22

]

{ #category : 'Session Control' }
System class >> gcLocksCount [

"Returns a SmallInteger which is the number of sessions holding a garbage collection 
 or repository scan lock.  

 Requires the SessionAccess privilege."

^ self _zeroArgPrim: 136

]

{ #category : 'Session Control' }
System class >> gcLocksReport [
  | sess str |
  sess := self currentSessions .
  str := String new .
  sess do:[:sessId | | desc gcLock waitingForGcHwAck |
    sessId ifNotNil:[
      desc := self descriptionOfSession: sessId .
      gcLock := desc at: 25 .
      waitingForGcHwAck := (desc at: 28) == true .
      (gcLock ~~ 0 or:[ waitingForGcHwAck]) ifTrue:[
        str addAll: (self _reportDetailsFor: sessId desc: desc) .
        str add:'  lock: '; add: (self _gcLockKindToStr: gcLock) .
        waitingForGcHwAck ifTrue:[ str add: ', waitingForGcHwAck '].
        str lf.
      ].
    ]
  ].
  ^ str
]

{ #category : 'Cache Statistics' }
System class >> gemCacheStatisticsForSessionId: aSessionId [

"Same as the cacheStatisticsAt: method except the argument is the session ID
 of a session currently connected to the shared memory cache.  In systems
 that use multiple shared memory caches, the session must exist on the same
 cache as the session invoking this method.

 Only cache statistics applicable to a Gem process are returned.
 The description of these statistics can be determined by evaluating:
   'System cacheStatisticsDescriptionForGem'
 Per-process host statistics are not included, see hostStatisticsForProcess: <pid>.

 nil is returned if the session cannot be located."

(aSessionId == 1 and:[ GsSession currentSession isSolo ]) ifTrue:[
  ^ self _primCacheStatsForSlotOrSession: nil opCode: 11 .
].
^ self _primCacheStatsForSlotOrSession: aSessionId negated opCode: 1

]

{ #category : 'Configuration File Access' }
System class >> gemConfigurationAt: aName [

"Returns the value of the specified configuration file parameter from the
 current session.  Returns nil if that parameter is not applicable to a Gem."

| cfgId |
cfgId := ConfigurationParameterDict at: aName otherwise: nil .
cfgId == nil ifTrue:[ ^ nil ].
^ self _configurationAt: cfgId isStone: false kind: $C

]

{ #category : 'Runtime Configuration Access' }
System class >> gemConfigurationAt: aName put: aValue [

"Changes the value of the specified Gem configuration parameter.

 See comments in the method configurationAt:put: for complete documentation."

^ self configurationAt: aName put: aValue

]

{ #category : 'Configuration File Access' }
System class >> gemConfigurationFileNames [
"Answer a SymbolDictionary containing two keys-value pairs:
  #GEMSTONE_SYS_CONF -> (String) Filename of the gem's system configuration file.
  #GEMSTONE_EXE_CONF -> (String) Filename of the gem's executable configuration file.

 If either file does not exist, the value will be an empty String."

|array result|
array := self gemConfigurationAt: #GemConfigFileNames .
result := SymbolDictionary new.
result at: #GEMSTONE_SYS_CONF put: (array at: 1) ;
       at: #GEMSTONE_EXE_CONF put: (array at: 2) .
^ result

]

{ #category : 'Configuration File Access' }
System class >> gemConfigurationReport [

"Returns a SymbolDictionary whose keys are the names of configuration file
 parameters, and whose values are the current settings of those parameters in
 the current session's Gem process.  Parameters that are not applicable to Gem
 and those that are undefined are not included in the result."

^ self _configurationReport: false

]

{ #category : 'Environment Access' }
System class >> gemEnvironmentVariable: varName [

"Expands the environment variable named varName in the Gem
 process, returning a String. varName should be a kind of String.

 Returns nil if any of the following are true:

 * varName is not a byte format object.
 * There is no environment variable defined with name varName.
 * The value of the environment variable is more than approximately 8000 bytes.
 * The size of varName exceeds approximately 8000 bytes.

 Signals an error if NoGsFileOnServer privilege is set in the UserProfile.
"

^ GsFile _expandEnvVariable: varName isClient: false

]

{ #category : 'Environment Access' }
System class >> gemEnvironmentVariable: varName put: valueStr [

"Sets the environment variable named varName in the Gem
 to value valueStr .
 The first argument must be either a String or a Unicode7 not larger than 8000 bytes.
 The second argument must be either a String or a Unicode7 not larger than 8000 bytes, 
 or nil (nil specifies  clearenv  semantics).
 The NoGsFileOnServer privilege must not be set in the session's UserProfile.

 Returns self or signals an Error if the operation failed.
"
GsFile _setEnvVariable: varName value: valueStr isClient: false.
^ self

]

{ #category : 'Runtime Configuration Access' }
System class >> gemIsBigEndian [

" Returns true if the gem process is running
  on a machine using big endian byte ordering "

^ self _zeroArgPrim: 106


]

{ #category : 'Host System Access' }
System class >> gemLogFileName [
  "Returns a String, the full path of the file to which the session's
   gem log file is being written.
   Returns an empty String in a topaz -l process.
   Each invocation of this method returns a new instance of String."

  ^ self _zeroArgPrim: 150

]

{ #category : 'Host System Access' }
System class >> gemLogPath [
  "Returns a String, the path of the directory in which the session's
   gem log file is being written.
   Returns an empty String in a topaz -l process.
   Each invocation of this method returns a new instance of String."

  ^ self _zeroArgPrim: 159

]

{ #category : 'Host System Access' }
System class >> gemProcessId [

  "Returns a SmallInteger, the processId from Unix getpid() of this session's
  gem or topaz -l  process ."

  ^ self _zeroArgPrim: 158

]

{ #category : 'Version Management' }
System class >> gemVersionAt: aString [

"Returns information about the Gem process of the current session.
 aString must be equal to a key in VersionParameterDict, otherwise nil
 is returned.  The semantics of these keys are:

 aString              meaning
 -------              -------
 cpuKind             detailed CPU type obtained at runtime: 'sun4u', 'x86_64'.
 cpuArchitecture     target CPU for which GemStone was compiled:
                       'SPARC', 'x86-64'.
 gsBuildArchitecture operating system name and CPU for which GemStone
                       was compiled: 'SPARC (Solaris)', 'x86-64 (Linux)'.
 gsBuildDate         time at which the Gem executable was compiled (a String).
 gsBuildType         usually FAST, but could be SLOW, PROFILE or NOOP
 gsRelease           major and minor version of GemStone, such as '3.0.0'.
 gsVersion           major version of GemStone, such as '3.0'.
 imageKind           a Symbol: #server.
 nodeName            network node name: 'speedy'.
 osName              operating system name: 'SunOS', 'Linux'.
 osRelease           release number of the operating system:
                       '5.10', '2.6.16.46-0.12-smp'.
 osVersion           vendor defined major version of the OS:
                       'Generic_141444-09', '#1 SMP Thu May 17 14:00:09 UTC 2007'.
 processId           operating system process identifier (an Integer): 13529.
 processorCount      number of processors on the machine running the process."

| verId |
aString = 'imageKind' ifTrue:[ ^ self imageVersionAt: aString ] .
verId := VersionParameterDict at: aString otherwise: nil .
verId ifNil:[ ^ nil ].

^ self _configurationAt: verId isStone: false kind: $V

]

{ #category : 'Version Management' }
System class >> gemVersionReport [

"Return a SymbolDictionary whose keys are the names of operating system,
 hardware, or GemStone version attributes, and whose values are the
 current values of those attributes in the Gem process."

^ self _serverVersionReport: false

]

{ #category : 'Version Management' }
System class >> gemVersionReportString [
  "Returns a String with keys and values of gemVersionReport one pair per line."
  ^ self gemVersionReport _reportString

]

{ #category : 'Error Handling' }
System class >> genericSignal: errIdentifier text: aString [

"Will be deprecated.
 Raise a user-defined signal with no arguments.

 The argument errIdentifier is a user-defined object, to distinguish user
 errors, and may be nil.  The argument aString appears in GemStone's error
 message for this error, and may be nil."

^ UserDefinedError new _number: 2318;
         reason: errIdentifier ; details: aString ; signal

]

{ #category : 'Error Handling' }
System class >> genericSignal: errIdentifier text: aString arg: anArg [

"Will be deprecated.
 Raise a user-defined signal with one argument.

 The argument errIdentifier is a user-defined object, to distinguish user
 errors, and may be nil.  The argument aString appears in GemStone's error
 message for this error, and may be nil. "

^ UserDefinedError new _number: 2318;
         reason: errIdentifier ; details: aString;
	args: { anArg } ; signal .

]

{ #category : 'Error Handling' }
System class >> genericSignal: errIdentifier text: aString args: anArray [

"Will be deprecated.
 Raise a user-defined signal.

 The argument errIdentifier is a user-defined object, to distinguish user
 errors, and may be nil.  The argument aString appears in GemStone's error
 message for this error, and may be nil.  "

^ UserDefinedError new _number: 2318;
     reason: errIdentifier ; details: aString ;
     args: (anArray  _isArray ifTrue:[ anArray ] ifFalse:[ { anArray } ]) ;
     signal .

]

{ #category : 'Garbage Collection Management' }
System class >> getAdminConfig: configSymbol [

"Returns the current runtime
 Valid configSymbols are:
    #adminVerboseLogging
    #epochGcMaxThreads
    #epochGcPageBufferSize
    #epochGcPercentCpuActiveLimit
    #epochGcTimeLimit
    #epochGcTransLimit
    #saveWriteSetUnionToFile
    #sweepWsUnionMaxThreads
    #sweepWsUnionPageBufferSize
    #sweepWsUnionPercentCpuActiveLimit
"

^self _gcGemConfig: false symb: configSymbol toValue: nil

]

{ #category : 'Configuration File Access' }
System class >> getKeyfileAttributes [

"Returns a SymbolDictionary which contains attributes of the current keyfile as follows:

   #allowsGbj -> (Boolean) true if GemBuilder for Java is allowed.
   #allowsGciTraversal -> (Boolean) true if GBS applications are allowed.
   #allowsGemConnect -> (Boolean) true if GemConnect is allowed.
   #allowsRemoteSpc -> (Boolean) true if remote shared page caches are allowed.
   #creationDate -> (DateTime) representing when the key was created.
   #customerName -> (String) name of the customer for which the keyfile was created.
   #fileName -> (String) full path to the current keyfile on the stone's host.
   #license ->  (String) license number of the keyfile.
   #machineKind -> (String) machine kind for the keyfile.
   #maxCpus -> (SmallInteger) maximum number of CPUs, or 0 for no limit.
   #maxDbfSizeMb -> (SmallInteger) maximum database size in megabytes, or 0 if unlimited.
   #maxOops -> (SmallInteger) maximum number of objects.
   #maxSpcSizeMb -> (SmallInteger) maximum shared page cache size in megabytes.
   #sessionLimit -> (SmallInteger) maximum number of sessions allowed.
   #sunsetDate -> (DateTime) when the keyfile expires, or nil if no expiry date is set."

^ self _zeroArgPrim: 181

]

{ #category : 'Garbage Collection Management' }
System class >> getReclaimConfig: configSymbol [

"Returns the current runtime value of the specified reclaim gem configuration option.
 Valid configSymbols are:
   #deferReclaimCacheDirtyThreshold
   #deadObjsReclaimedCommitThreshold
   #maxTransactionDurationUs
   #objsMovedPerCommitThreshold
   #reclaimDeadEnabled
   #reclaimMinPages
   #reclaimVerboseLogging
   #sleepTimeBetweenReclaimUs
   #sleepTimeWithCrBacklogUs
"
^self _gcGemConfig: true symb: configSymbol toValue: nil

]

{ #category : 'Configuration File Access' }
System class >> getSharedCacheAttributes [
"Returns a SymbolDictionary which contains attributes for the shared page cache to
 which the current session is attached as follows:

   #cacheName ->              (String) Name of the cache
   #numFrames ->              (SmallInteger) Total number of 16Kb frames in the cache.
   #physicalSizeInBytes ->    (Integer) Physical size of the shared memory
                                segment in bytes.
   #numSharedCounters ->      (SmallInteger) Number of shared counters in the cache
                                (see config option SHR_PAGE_CACHE_NUM_SHARED_COUNTERS).
   #numProcesses ->           (SmallInteger) Maximum number of processes that can attach
                                the cache (see config option SHR_PAGE_CACHE_NUM_PROCS).
   #spinLockCount ->          (SmallInteger) Spin lock count
                                (see config option SHR_SPIN_LOCK_COUNT).
   #largeMemoryPagePolicy ->  (SmallInteger) Large memory page policy
                                (see config option SHR_PAGE_CACHE_LARGE_MEMORY_PAGE_POLICY).
   #largeMemoryPageSizeMb ->  (SmallInteger) Large memory page size in megabytes
                                (see config option SHR_PAGE_CACHE_LARGE_MEMORY_PAGE_SIZE_MB).
   #largeMemoryPagesUsed ->   (SmallInteger) Number of large memory pages used
                                to create the cache.
   #targetFreeFrameCount ->   (SmallInteger) Target free frame count
                                (see config option SHR_TARGET_FREE_FRAME_COUNT).
   #cacheLocked ->            (Boolean) Cache locked in memory
                                (see config option SHR_PAGE_CACHE_LOCKED).
   #isRemoteCache ->          (Boolean) true if the cache is remote from the stone.
   #isRemoteMidCache ->       (Boolean) true if the cache is a mid-level remote cache.

Returns nil if the current session is a solo session.
"

| list result |
list := self _sharedCacheAttributes .
list ifNil:[ ^ nil ].
result := SymbolDictionary new: 11 .
^ result at: #cacheName put: (list at: 1) ;
         at: #numFrames put: (list at: 2) ;
         at: #physicalSizeInBytes put: (list at: 3) ;
         at: #numSharedCounters put: (list at: 4) ;
         at: #numProcesses put: (list at: 5) ;
         at: #spinLockCount put: (list at: 6) ;
         at: #largeMemoryPagePolicy put: (list at: 7) ;
         at: #largeMemoryPageSizeMb put: (list at: 8) ;
         at: #largeMemoryPagesUsed put: (list at: 9) ;
         at: #targetFreeFrameCount put: (list at: 10) ;
         at: #cacheLocked put: (list at: 11) ;
	 at: #isRemoteCache put: (list at: 12) ;
	 at: #isRemoteMidCache put: (list at: 13) ;
         yourself

]

{ #category : 'Cache Statistics - Global Session' }
System class >> globalSessionStatAt: index [
"Returns the current value of a global session statistic.
 See the method System>>_updateGlobalSessionStat: index by: i overwrite: aBool
 for more information on global session statistics."

^self _updateGlobalSessionStat: index by: 0 overwrite: false

]

{ #category : 'Cache Statistics - Global Session' }
System class >> globalSessionStatAt: index put: newVal [
"Overwrites the value of the given global session statistic with newVal
 and returns the new value.  See the method
 System>>_updateGlobalSessionStat: index by: i overwrite: aBool
 for more information on global session statistics."

^self _updateGlobalSessionStat: index by: newVal overwrite: true

]

{ #category : 'Garbage Collection Management' }
System class >> hasMissingGcGems [
"Return a Boolean indicating if any garbage collection systems are
 missing from the system.  Answers true if either admin gem or
 reclaim gem is not running."

((SystemRepository restoreStatusInfo at: 2) == 0 and: [self adminGemSessionId == 0])
  ifTrue:[^true]. "missing the Admin gem"

(self reclaimGemSessionCount == 0)
  ifTrue:[^true]. "missing reclaim gem"

^false "all GC gems accounted for"

]

{ #category : 'User-Defined Actions' }
System class >> hasUserAction: aSymbol [

"Returns true if the user action named aSymbol is installed in this GemStone
 session.  Returns false otherwise."

<primitive: 105>
aSymbol _validateClass: String.
^ self _primitiveFailed: #hasUserAction: args: { aSymbol }

]

{ #category : 'Session Control' }
System class >> hostAgentSessions [

"Returns an Array of SmallIntegers which are stone sessionIds
 corresponding to all of the hostagent sessions currently running."

| haUp pairs res |
pairs := self _currentSessionProfiles.
haUp := AllUsers userWithId:'HostAgentUser'.
res := { } .
1 to: pairs size by: 2 do:[:j |
  (pairs at: j+1 ) == haUp ifTrue:[ | desc sesId |
    sesId := pairs at: j .
    desc := System descriptionOfSession: sesId .
    (desc at: 17) ifNotNil:[ "a hostagent process"
      res add: sesId
    ].
  ].
].
^ res

]

{ #category : 'Session Control' }
System class >> hostAgentSessionsReport [
  "Returns a String describing all hostgent sessions."
| haUp pairs sess rpt |
pairs := self _currentSessionProfiles.
haUp := AllUsers userWithId:'HostAgentUser'.
sess := { } .
1 to: pairs size by: 2 do:[:j |
  (pairs at: j+1 ) == haUp ifTrue:[ sess add: (pairs at: j)].
].
rpt := String new .
sess do:[:sesId | | desc port leafIp |
  rpt add: 'session '; add: (sesId asString width: -3) ; add:'  '.
  desc := System descriptionOfSession: sesId .
  rpt add: ' gemPid: ', (desc at:2) asString ; add: '  '  .
  (desc at: 17) ifNotNil:[:elem |
    rpt add: elem .
    rpt add: (elem = 'midcache_hostagent' ifTrue:[ ' on: '] ifFalse:[ ' servicing: ' ]).
    rpt add: (leafIp := desc at: 11) .   "host"
    rpt add: $( ; add: (GsSocket getHostNameByAddress:leafIp); add: $) .
  ] ifNil:[ rpt add: '(normal login by HostAgentUser)' ].
  port := desc at: 24.
  port > 0 ifTrue:[ rpt add: ' listeningPort: '; add: port asString ].
  rpt lf .
].
^ rpt

]

{ #category : 'Host System Access' }
System class >> hostCpuCount [

"Returns a SmallInteger which is the number of CPUs on the host system.  In most cases,
 this number represents the total number of CPU cores on the host system."

^ self _zeroArgPrim: 135

]

{ #category : 'Cache Statistics - Host Process' }
System class >> hostCpuStatsForProcessId: anInt [

"Return an Array of 2 integers as follows:
  array[1] - user mode CPU milliseconds
  array[2] - system mode CPU milliseconds

 Both array elements will be -1 if the process slot is out of range
 or not in use or if this method is not supported for the host
 architecture.

 Is not required to be attached to the shared page cache or even be
 a GemStone process.  The method will success for any process for which
 the gem session has permission to view its CPU usage statistics."

^ self _primCacheStatsForSlotOrSession: anInt opCode: 7

]

{ #category : 'Cache Statistics - Host Process' }
System class >> hostCpuStatsForProcessSlot: anInt [

"For the process using the cache process slot anInt, return an Array of
 2 integers as follows:
   array[1] - user mode CPU milliseconds used
   array[2] - system mode CPU milliseconds used

 Both array elements are set to -1 if the process slot is out of range
 or not in use, or if this method is not supported for the host
 architecture."

^ self _primCacheStatsForSlotOrSession: anInt opCode: 6

]

{ #category : 'Host System Access' }
System class >> hostCpuUsage [

"Returns an Array of 5 SmallIntegers with values between 0 and 100 which
 have the following meanings:

 1 - Percent CPU active (user + system)
 2 - Percent CPU idle
 3 - Percent CPU user
 4 - Percent CPU system (kernel)
 5 - Percent CPU I/O wait

 On hosts with multiple CPUs, these figure represent the average across all
 processors.

 The values on first invocation or with invocations very close together may show
 100% idle; these results are unreliable.

 Returns nil if the host system does not support collecting CPU statistics or an
 error occurs."

^ self _zeroArgPrim: 130

]

{ #category : 'Host System Access' }
System class >> hostId [

"Return an Integer which represents a unique 64 bit identifier for the host
 where the session is running.  This identifier is created and managed by
 GemStone and is not related in anyway to the hostname or IP address of the
 host."

^ self _zeroArgPrim: 131

]

{ #category : 'Host System Access' }
System class >> hostname [

"Return a String which is the name of the host on which the session is running."
^ self _zeroArgPrim: 163

]

{ #category : 'Cache Statistics - Host Process' }
System class >> hostProcessStatisticsNames [

"Returns an array of Strings which are the names of the per-process statistics
 provided by this host."

^ self _primCacheStatsForSlotOrSession: nil opCode: 10

]

{ #category : 'Cache Statistics - Host Process' }
System class >> hostStatisticsForMyProcess [

"Returns an array of SmallIntegers which represent the host statistics for this process.
 The names of each statistic are returned by the #hostProcessStatisticsNames
 method."

^ self _primCacheStatsForSlotOrSession: -1 opCode: 10

]

{ #category : 'Cache Statistics - Host Process' }
System class >> hostEasyStatisticsForMyProcess [

"Same as #hostStatisticsForMyProcess on all platforms except Linux.
 On Linux, same as #hostStatisticsForMyProcess except certain statistics which
 are expensive to collect are not computed and have zeros stored in their place."

^ self _primCacheStatsForSlotOrSession: -1 opCode: 12

]

{ #category : 'Cache Statistics - Host Process' }
System class >> hostStatisticsForProcess: anInteger [

"Returns an array of SmallIntegers which represent the host statistics for the given
 process ID.  The names of each statistic are returned by the #hostProcessStatisticsNames
 method.

 Returns nil if the process could not be found or if this process does not have
 permission to fetch the statistics."

^ self _primCacheStatsForSlotOrSession: anInteger opCode: 10

]

{ #category : 'Cache Statistics - Host Process' }
System class >> hostEasyStatisticsForProcess: anInteger [

"Same as #hostStatisticsForProcess: on all platforms except Linux.
 On Linux, same as #hostStatisticsForProcess: except certain statistics which
 are expensive to collect are not computed and have zeros stored in their place."

^ self _primCacheStatsForSlotOrSession: anInteger opCode: 12

]

{ #category : 'Session Control' }
System class >> iAmSystemUser [

"Returns the true if the current session is logged in as SystemUser, otherwise false"

^ self myUserProfile isSystemUserProfile

]

{ #category : 'Version Management' }
System class >> imageVersionAt: aSymbol [

"Returns information about the GemStone kernel class image where aSymbol is one
 of the following:

 #gsBuildDate  DateTime of last kernel class filein or upgrade.
 #gsRelease    Version String of last kernel class filein or upgrade: '3.0.0'.
 #gsVersion    Major version of image, such as '3.0' .
 #imageKind    A Symbol: #server.

 If aSymbol is not one of the above, returns nil."

^ ImageVersion at: aSymbol otherwise: nil

]

{ #category : 'Transaction Control' }
System class >> inContinueTransaction [

"Returns a boolean indicating if the current transaction has been
 continued.  Answers true if System>>continueTransaction has been
 executed and no abort or successful commit has been performed.
 Also answer true if a commit was attempted and failed and no
 abort has been executed.  Otherwise returns false."

^ self _zeroArgPrim: 105

]

{ #category : 'Cache Statistics - Global Session' }
System class >> incrementGlobalSessionStatAt: index by: value [
"Increments the value of the given global session statistic by newVal
 and returns the new value.  See the method
 System>>_updateGlobalSessionStat: index by: i overwrite: aBool
 for more information on global session statistics."

^self _updateGlobalSessionStat: index by: value overwrite: false

]

{ #category : 'Transient Session State' }
System class >> incrementIndexProgressCountBy: aSmallInt [

"Increments the value of the indexProgressCount statistic by aSmallInt and
 returns the new value."

^ self _sessionCacheStatAt: -2 incrementBy: aSmallInt

]

{ #category : 'Transient Session State' }
System class >> incrementProgressCountBy: aSmallInt [

"Increments the value of the progressCount statistic by aSmallInt and
 returns the new value."

^ self _sessionCacheStatAt: -1 incrementBy: aSmallInt

]

{ #category : 'Transaction Control' }
System class >> inTransaction [

"Returns true to indicate that the session is in a transaction, false
 otherwise."

^ (self _zeroArgPrim: 5) ~~ 0

]

{ #category : 'Session Control' }
System class >> killHostAgents [

"Stops all HostAgent processes that are logged in to this stone.
 Each hostagent receives a fatal stop session Error.
 Signals an Error if unable to stop all hostagents."

^ self _stopHostAgents: 16r8"Fatal stop session" timeOut: 10

]

{ #category : 'Session Control' }
System class >> killHostAgentSession: sessionId [

 "Shutdown of the HostAgent with the specified sessionId with fatal Error."

^ self _stopHostAgentSession: sessionId kind: 16r8 "Fatal stop session"

]

{ #category : 'Session Control' }
System class >> killLogReceiver [

 "Causes stone to send SIGTERM to the log receiver process.
  Requires the SystemControl privilege.
  Returns true if the SIGTERM signal was sent, false if no signal was
  sent because no log receiver process was running or a SIGTERM signal
  had already been sent."

^self _zeroArgPrim: 204
]

{ #category : 'Session Control' }
System class >> killLogSender [

 "Causes stone to send SIGTERM to the log sender process.
  Requires the SystemControl privilege.
  Returns true if the SIGTERM signal was sent, false if no signal was
  sent because no log sender process was running or a SIGTERM signal
  had already been sent."

^self _zeroArgPrim: 203
]

{ #category : 'Session Control' }
System class >> killLogSenderSplitLogs [

 "Causes stone to send SIGTERM to the log sender process that was started with -F -W .
  Requires the SystemControl privilege.
  Returns true if the SIGTERM signal was sent, false if no signal was
  sent because no log sender process was running or a SIGTERM signal
  had already been sent."

^self _zeroArgPrim: 209
]

{ #category : 'Session Control' }
System class >> lastTransactionTimeForSessionWithId: aSessionId [

"Returns a SmallInteger representing the time of the session's most recent
 beginTransaction, commitTransaction, or abortTransaction (from System timeGmt).

 Equivalent to executing:
   (System descriptionOfSession: aSessionId) at: 5

 Returns -1 if the session with aSessionId could not be found."

<primitive: 834>
aSessionId _validateClass: SmallInteger .
^self _primitiveFailed: #lastTransactionTimeForSessionWithId:
      args: { aSessionId }

]

{ #category : 'Session Control' }
System class >> latestCommitRecordPageId [

"Return the page ID of the most recent (newest) commit record on the system.
 Returns -1 if an error occurs."

^ self _zeroArgPrim: 108

]

{ #category : 'Debugging Support' }
System class >> listenForDebugConnection [

"Requires CodeModification privilege.
 Returns an Integer (the debug token) or signals an Error .
 If this session was configured with GEM_LISTEN_FOR_DEBUG=false ,
 starts a thread in the vm which listens on localhost for a process
 to connect with GciDebugConnectToGem, GciDebugStartDebugService,
 or for a topaz -r to connect with DEBUGGEM command.  Prints the
 required DEBUGGEM command including token to gem log file or stdout of a topaz -l
 process.  The token is a random 64 bit integer .

 If the debug thread is already running or GEM_LISTEN_FOR_DEBUG=true,
 returns the Integer debug token."
 | token |
 token := self _zeroArgPrim: 116.
 token == 0 ifTrue:[
    Error signal:'listenForDebugConnection failed, check gem log or topaz -l sterr'.
 ].
 ^ token

]

{ #category : 'User-Defined Actions' }
System class >> loadUserActionLibrary: aString [

"Loads the session user action library specified by aString.
 This method always returns the receiver (System).

 Searches for the shared library in current directory,
 $GEMSTONE/ualib ,
 directory containing the executable that loaded the libgci library,
   $GEMSTONE/bin for a topaz or $GEMSTONE/sys for a gem .
 and on Linux the directories specified by LD_LIBRARY_PATH environment variable."

<primitive: 426>
^ self _primitiveFailed: #loadUserActionLibrary: args: { aString }

]

{ #category : 'Lock Status' }
System class >> lockKind: anObject [

"Returns a Symbol (#none, #deferredUnlock, #read, or #write) representing the kind
 of lock held on anObject by any session in the system."

| anArray |

anArray := self lockStatus: anObject.
^anArray at: 1.

]

{ #category : 'Lock Status' }
System class >> lockOwners: anObject [

"Returns an Array of session numbers (SmallIntegers) representing the sessions
 that hold a lock on anObject.  If the object is not locked by any session, the
 result Array is empty.  Note that a write lock can have only one owner."

| anArray |

anArray := self lockStatus: anObject.
^anArray at: 2.

]

{ #category : 'Lock Status' }
System class >> lockStatus: anObject [

"Returns a two-element Array, where the first element is a Symbol representing
 the kind of lock held on anObject (#none, #read or #write) and
 the second element is an Array of session numbers (SmallIntegers) representing
 the sessions that hold the lock.

 If there are no locks on anObject, the first element is the Symbol #none and
 the second element is an empty Array.

 Only locks on permanent objects are reported."

<primitive: 351>
self _primitiveFailed: #lockStatus: args: { anObject } .
self _uncontinuableError

]

{ #category : 'System Control' }
System class >> loginLogFileName [

"Returns a String containing the name of the current login log file,
 or nil if the login log file is not enabled."

^ self _zeroArgPrim: 174

]

{ #category : 'Session Control' }
System class >> loginUserId [

"Returns a SmallInteger which is the numeric value of the Unix userId of
 the gem or topaz -l process of the current session.

 Returns -1 if the Unix user ID could not be determined or if the host
 system does not support this feature."

^ self _zeroArgPrim: 171

]

{ #category : 'Session Control' }
System class >> loginUserIdName [

"Returns a String containing the Unix login user ID of the gem or topaz -l process
 for the current session.

 Returns 'unknown user' if the Unix user ID could not be determined or if the host
 system does not support this feature."

^ self _zeroArgPrim: 172

]

{ #category : 'Session Control' }
System class >> logout [

"Immediately logs out the current session and generates a fatal error
 (GS_ERR_GEM_NORMAL_SHUTDOWN).  This method does not return."

self _zeroArgPrim: 104

]

{ #category : 'Private' }
System class >> _standbyvmSessionId: offset [
  | ary id |
  ary := self _zeroArgPrim: 210 .
  id := ary at: offset .
  id > 0 ifTrue:[ ^ id ].
  ^ 0
]

{ #category : 'Session Control' }
System class >> logreceiverSessionId [

"Return the session ID of the logreceiver process.
 Returns zero if no logreceiver is connected to the stone process.
 Note a logreceiver can exist without a connection to the stone."

 ^ self _standbyvmSessionId: 3 . 
]

{ #category : 'Session Control' }
System class >> logsenderSessionId [

"Return the session ID of the logsender process.
 Returns zero if no logsender is connected to the stone process.
 Note a logsender can exist without a connection to the stone."

 ^ self _standbyvmSessionId: 1 . 
]

{ #category : 'Session Control' }
System class >> logsenderSessionIdSplitLogs [

"Return the session ID of the logsender process started with -W -F .
 Returns zero if no logsender is connected to the stone process.
 Note a logsender can exist without a connection to the stone."

 ^ self _standbyvmSessionId: 2 . 
]

{ #category : 'Clustering' }
System class >> maxClusterBucket [

"Returns the maximum legal clusterId as a SmallInteger."

^ AllClusterBuckets size

]

{ #category : 'Session Control' }
System class >> maxSessionId [

"Returns a SmallInteger representing the maximum number of sessions
 allowed on the system based upon the Stone configuration parameter."

^ self _zeroArgPrim: 6

]

{ #category : 'Performance Monitoring' }
System class >> microsecondsToRun: aBlock [

"Returns the number of CPU microseconds used while evaluating aBlock.
 The argument aBlock must be a zero-argument block."

| startTime endTime |

startTime := self readClockNano.
aBlock value.
endTime := self readClockNano.
^ (endTime - startTime) // 1000

]

{ #category : 'Shared Cache Management' }
System class >> midLevelCacheAddress [

"If connected to a mid level cache, return a String , the IP
 address of the mid cache host.  Otherwise return nil."

^ self _midLevelCacheConnect: nil options: 2

]

{ #category : 'Shared Cache Management' }
System class >> midLevelCacheConnect: hostName [

"Attempt to connect to mid-level cache on the
 specified host, if cache already exists, using
 the same netldi name as was used to login to the stone.

 Returns true if connection succeeded, or a String
 describing the failure.

 The session's gem process must be on a machine different from
 the machine running the stone process in order for this to succeed."

^ self midLevelCacheConnect: hostName options: nil .

]

{ #category : 'Shared Cache Management' }
System class >> midLevelCacheConnect: hostName cacheSizeKB: aSize maxSessions: nSess [

"If a cache already exists on host specified by hostName,
 connects to that cache.
 If current session was created by GciX509Login, and the cache does
 not exist, returns a String describing the connection failure.
 Arguments aSize and nSess are ignored in X509 sessions.

 Otherwise if a mid-level cache does not exist on a specified host,
 and aSize > 0, attempt to start one.
 Then attempt to connect to the mid-level cache.
 Connection may fail if cache could not be started or
 if the cache could not accept another session .
 Uses the same netldi name as was used to login to the stone.

 The session's gem process must be on a machine different from
 the machine running the stone process in order for this to succeed.

 If session A's gem process is running on a machine where the
 shared cache is in use as a mid-level cache by gem processes
 on other machines, then session A is cannot use this method;
 i.e. cross-connected mid-level caches are not allowed .

 hostName must be a String, else an error is generated.

 If aSize <= 0, nSess is ignored ,
 otherwise nSess must be >= 11 and aSize must be >= 16000 .
 The nSess value may be reduced by stone to the
 number of sessions stone is configured and licensed for.

 When creating a cache
   aSize is cache size in K bytes (minumum 16000) ,
   nSess is maximum number of sessions (minimum 11) .

 The spinLockCount, targetFreeFrameCount, numFreeFrameServers,
 and numSharedCounters for creating a cache come from gem's config file
 just as for a cache the gem would have created as a side effect
 of login .

 Returns true if connection succeeded, or a String
 describing the connection failure."

aSize <= 0 ifTrue:[
  "attempt to connect to an existing cache"
  ^ self midLevelCacheConnect: hostName options: nil
].
nSess < 1 ifTrue:[ Error signal:'nSess arg must be >= 1' ].
^ self midLevelCacheConnect: hostName
   options: { 'SHR_PAGE_CACHE_SIZE_KB' . aSize  .
	      'SHR_PAGE_CACHE_NUM_PROCS' . nSess + 4 "pcmon, cachePgsvr, 2 FF slots"}

]

{ #category : 'Shared Cache Management' }
System class >> midLevelCacheConnect: hostName options: anArray [

"Attempts to connects to a mid-level cache on the host specified by hostName.
 If anArray is nil, or if current session is from GciX509Login,
 the cache must exist.  Otherwise if cache
 does not exist an attempt is made to start the cache.
 Argument anArray is ignored in X509 sessions.

 Connection may fail if cache could not be started or
 if the cache could not accept another session .
 Uses the same netldi name as was used to login to the stone.

 The session's gem process must be on a machine different from
 the machine running the stone process in order for this to succeed.

 If session A's gem process is running on a machine where the
 shared cache is in use as a mid-level cache by gem processes
 on other machines, then session A is cannot use this method;
 i.e. cross-connected mid-level caches are not allowed .

 hostName must be a String, else an error is generated.

 If not nil, anArray must be an Array of pairs specifing config item and value
 for a mid-level cache.
 The first element of each pair must be one of the
 following case-insensitive strings , with second element of a pair
 being a value as follows

  'SHR_PAGE_CACHE_SIZE_KB'  anInteger representing KB,
                            or a String with a units suffix such as '3GB' or '3000MB'
  'SHR_PAGE_CACHE_NUM_PROCS'  anInteger
  'SHR_TARGET_FREE_FRAME_COUNT' anInteger
  'SHR_NUM_FREE_FRAME_SERVERS' anInteger
  'SHR_PAGE_CACHE_NUM_SHARED_COUNTERS' anInteger
  'SHR_PAGE_CACHE_LOCKED'   0, 1 or a Boolean
  'GEM_PGSVR_COMPRESS_PAGE_TRANSFERS'  0, 1 or a Boolean
  'SHR_PAGE_CACHE_LARGE_MEMORY_PAGE_POLICY'   0, 1 or 2
  'SHR_PAGE_CACHE_LARGE_MEMORY_PAGE_SIZE_MB' anInteger: Linux: 0, 2, 1024; Others: 0
  'SHR_PAGE_CACHE_PERMISSIONS'   anInteger
  'SHR_WELL_KNOWN_PORT_NUMBER' anInteger
  'MidHostRoute'  a String , a hostname or ipAddress .

 NOTE: These are expected to match the names and behavior of configuration
   parameters, however 'MidHostRoute' is not a existing configuration parameter
   and is specific to this method.

 MidHostRoute if specified is used to access an already running cache;
 it specifies the host used by the gem to contact the mid cache.
 hostName is always used to ask stone for the existance of the cache regardless
 of whether MidHostRoute is defined.  hostName must be the route to the
 mid cache as seen by stone ; MidHostRoute allows for a leaf node to
 have a different route to a multi homed mid cache host.  All other
 config items are used only to start a cache .

 Any other item which is not specified will have a value
 taken from this session's gem config file.
 The SHR_TARGET_FREE_FRAME_COUNT is normally computed automatically
 based on the cache size .
 The default for GEM_PGSVR_COMPRESS_PAGE_TRANSFERS is taken from
 the current runtime value in the gem process.

 See the comments in the file $GEMSTONE/data/system.conf for
 more details on each item.

 Signals an error if a config item is invalid or out of range.
 Returns true if connection succeeded, or a String
 describing a connection failure."

hostName  _validateClass: String .
anArray ifNotNil:[ anArray _validateClass: Array ].
^ self _midLevelCacheConnect: hostName options: anArray

]

{ #category : 'Shared Cache Management' }
System class >> midLevelCacheReconnect: hostName [

"If already connected to a mid level cache, disconnect from that cache.
 Then attempt to connect to mid-level cache on the
 specified host, if cache already exists, using
 the same netldi name as was used to login to the stone.

 Returns true if connection succeeded, or a String
 describing the failure.

 The session's gem process must be on a machine different from
 the machine running the stone process in order for this to succeed."

 hostName  _validateClass: String .
 ^ self _midLevelCacheConnect: hostName options: 1 .

]

{ #category : 'Shared Cache Management' }
System class >> midLevelCachesList [

"Return an Array as described for remoteCachesList,
 filtered to include only the mid-level caches, plus the stone cache. "

^ self _remoteCachesList: true

]

{ #category : 'Shared Cache Management' }
System class >> midLevelCachesReport [

"Return the result of midLevelCachesList formatted as a string,
 one line per cache."

^ self _formatCacheReport: (self _remoteCachesList: true)

]

{ #category : 'Transaction Control' }
System class >> millisecondsSinceLogin [

"Answer a SmallInteger representing the number of milliseconds since this
 session logged in to the repository."

^ self _zeroArgPrim: 123

]

{ #category : 'Transaction Control' }
System class >> millisecondsSinceTransactionBoundary [

"Answer a SmallInteger representing the number of milliseconds since this
 session updated its view of the repository by executing #abortTransaction,
 #beginTransaction, #continueTransaction, or #commitTransaction.  Note that
 an attempt to commit which fails (due to transaction conflicts) is considered
 a transaction boundary since the session is shifted to the most recent view
 by the failed commit attempt."

^ self _zeroArgPrim: 122

]

{ #category : 'Performance Monitoring' }
System class >> millisecondsToRun: aBlock [

"Returns the number of CPU milliseconds used while evaluating aBlock.
 The argument aBlock must be a zero-argument block.

 The resolution of the result is operating system dependent."

| startTime endTime |
aBlock _isExecBlock ifFalse:[ ArgumentError signal:'expected a block'].
startTime := self readClockNano .
aBlock value.
endTime := self readClockNano .
^ (endTime - startTime) // 1000000

]

{ #category : 'Garbage Collection Management' }
System class >> missingGcGem [
"Return a string indicating which gc gem is missing"
((SystemRepository restoreStatusInfo at: 2) == 0 and: [self adminGemSessionId == 0])
  ifTrue:[^ 'missing the Admin gem' ].

(self reclaimGemSessionCount == 0)
  ifTrue:[^ 'missing reclaim gem' ].

^ 'all GC gems accounted for'

]

{ #category : 'Cache Statistics' }
System class >> myCacheProcessSlot [

"Returns the process slot in the SharedPageCache that corresponds
 to my process.  If the SharedPageCache is not in use, returns -1."

^ self _zeroArgPrim: 7

]

{ #category : 'Cache Statistics' }
System class >> myCacheStatistics [

"Returns the cache statistics for the current session. Per-process host 
 statistics are not included, see hostStatisticsForMyProcess."

GsSession currentSession isSolo ifTrue:[
  ^ self _primCacheStatsForSlotOrSession: nil opCode: 11 .
].
^ self cacheStatisticsAt: self myCacheProcessSlot

]

{ #category : 'Cache Statistics' }
System class >> myCacheStatisticWithName: aString [

"Return the value of the cache stat with the given name for the current session.
 The name must match an element of the Array returned by the
 #cacheStatisticsDescription method applicable to a gem process.  The UserTime
 and SysTime statistics cannot be accessed using this method.

 Returns nil if a statistic matching aString was not found for a gem.

 This method may be used on hosts remote from the stone process."

^ self _cacheStatWithName: aString opCode: 8

]

{ #category : 'Session Control' }
System class >> myKerberosPrincipal [

"Returns the KerberosPrincipal of the current session, or nil
 if Kerberos was not used by the session during login or the
 session does not have read access to the KerberosPrincipal
 object."

^ self _zeroArgPrim: 180

]

{ #category : 'Lock Status' }
System class >> myLockKind: anObject [

"Returns a Symbol that indicates what kind of lock the current session has on
 anObject: one of #none, #read or #write."

| status |
status := self lockStatus: anObject.
( (status at: 1) ~~ #none and:
[ (status at: 2) includesIdentical: self session] )
  ifTrue: [ ^status at: 1 ].

^#none

]

{ #category : 'Private' }
System class >> myPageServerProcessId [
"Answer an Integer which is the process ID of the page server on stone's
 machine for this session.
 Returns 0 if the session does not have a page server."

^ self _zeroArgPrim: 59

]

{ #category : 'Private' }
System class >> myReadPageServerProcessId [
"Answer an Integer which is the process ID of the page server
being used for disk reads.
Returns 0 if the session does not have a page server.
Result may be different from the result of myPageServerProcessId if
the session is using a mid-level cache."

^ self _zeroArgPrim: 108

]

{ #category : 'Deprecated' }
System class >> myUserGlobals [

"Obsolete in GemStone/64."

| assn |

self deprecated: 'System class>>myUserGlobals is deprecated in GemStone/S 64 Bit.
Replace with (System myUserProfile resolveSymbol: #UserGlobals) value'.

(assn := self myUserProfile resolveSymbol: #UserGlobals) == nil ifTrue: [
  ^self _error: #rtErrKeyNotFound args: #(#UserGlobals)
].
^assn value.

]

{ #category : 'Session Control' }
System class >> myUserProfile [

"Returns the UserProfile of the current session."

"Implemented as a primitive to handle image bootstrap problems."

<primitive: 318>
^ self _primitiveFailed: #myUserProfile

]

{ #category : 'Transaction Control' }
System class >> needsCommit [

"Returns true if a non-solo session has made changes to the repository.  
 Returns false otherwise."

GsSession currentSession isSolo ifTrue:[ ^ false ].
^ self _numPersistentObjsModified > 0

]

{ #category : 'Instance Creation' }
System class >> new [

"Disallowed.  You may not create new instances of System."

self shouldNotImplement: #new

]

{ #category : 'Session Control' }
System class >> notVotedSessionNames [
  "Return a string formatted as for System(C)>>currentSessionNames
   for each session that has not voted during the last epochGc or
   markForCollection .
   Requires SessionAccess privilege if there is more
   than one session logged in."
  | resBlock |
  resBlock := [ :res :num :userPro |
              res lf ; add: 'session number: '; add: num asString ;
                  add: '    UserId: ' ; add: userPro userId ].
  ^ self _notVotedSessions: resBlock into: String new
]

{ #category : 'Session Control' }
System class >> notVotedSessions [
  "Return an array of session numbers as for System(C)>>currentSessions
   for each session that has not voted since the last epochGc or markForCollection .
   Requires SessionAccess privilege if there is more
   than one session logged in."
  | resBlock |
  resBlock := [ :res :num :userPro | res add: num ].
  ^ self _notVotedSessions: resBlock into: { } .

]

{ #category : 'Session Control' }
System class >> notVotedSessionsReport [
  "Returns a String describing sessions that have not voted 
   since the last epochGc or markForCollection. 
   Requires SessionAccess privilege."
  ^ self _sessionsReport: self notVotedSessions 
]

{ #category : 'Deprecated' }
System class >> numberOfExtentRangesWithoutGC [
"Deprecated - all extents are covered by a single reclaim gem.
 Return a SmallInteger indicating the minimum number of additional
 reclaim gems that would need to be started to have all extents
 covered by a reclaim gem."

self deprecated: 'System class>>numberOfExtentRangesWithoutGC deprecated in v3.2.'.
^self reclaimGemSessionCount > 0
  ifTrue: [ 0 ]
  ifFalse: [ 1 ]

]

{ #category : 'Deprecated' }
System class >> numberOfExtentsWithoutGC [

"Deprecated - all extents are covered by a single reclaim gem.
 Return a SmallInteger indicating the number of extents not subject to
 garbage collection because they have no reclaim gem assigned."

self deprecated: 'System class>>numberOfExtentsWithoutGC deprecated in v3.2. '.
^self reclaimGemSessionCount > 0
  ifTrue: [ 0 ]
  ifFalse: [ SystemRepository numberOfExtents ]

]

{ #category : 'Persistent Counters' }
System class >> numberOfPersistentSharedCounters [

"Returns a SmallInteger which is the number of persistent shared counters
 the system supports."

^ self _zeroArgPrim: 154

]

{ #category : 'Shared Counters' }
System class >> numSharedCounters [

"Returns the number of shared counters supported by this session's shared
 page cache. This is set by the configuration file parameter
 SHR_PAGE_CACHE_NUM_SHARED_COUNTERS, which has a default of 1900."

  ^self _numSharedCounters

]

{ #category : 'Configuration File Access' }
System class >> objectReadLogEnabled [
 "Answer a Boolean indicating if the object read logging feature is currently enabled."
 ^ System stoneConfigurationAt: #STN_OBJECT_READ_LOG_ENABLED .
]

{ #category : 'System Control' }
System class >> objectReadTrackingEnabled [
"Returns true if read tracking is currently enabled for this session.
 The result will be true if (System objectReadLogEnabled == true) and
 ((System myUserProfile _hasPrivilegeName: #DisableObjectReadLogging) == false
 and (System setObjectReadTracking: false) has not been successfully executed as the most recent
 invocation of System class >> setObjectReadTracking: ."

^ self _zeroArgPrim: 195
]

{ #category : 'Session Control' }
System class >> oldestCommitRecordAge [
"Return a SmallInteger representing the age of the oldest commit record in seconds.
 This value represents the elapsed time since the commit that created the oldest
 commit record."

  ^ (self _zeroArgPrim: 164) // 1000

]

{ #category : 'Session Control' }
System class >> oldestCommitRecordAgeMilliseconds [

"Return a SmallInteger representing the age of the oldest commit record in milliseconds.
 This value represents the elapsed time since the commit that created the oldest
 commit record."

^ self _zeroArgPrim: 164

]

{ #category : 'Session Control' }
System class >> otherSessionNames [

"Returns a formatted String containing, for each current GemStone session
  other than the session executing this method, the
 session number and userId.

 This method requires SessionAccess privilege."

| lf result mySessNum profiles |

result := String new.  " string to return "
lf := Character lf.
mySessNum := self session .

profiles := self _currentSessionProfiles.  "sessId, userProfile pairs"
1 to: profiles size by: 2 do:[:j| | aSessionNumber aUserProfile |
   aSessionNumber := profiles at: j .
   aSessionNumber == mySessNum ifFalse:[
     aUserProfile := profiles at: j+ 1 .
     aUserProfile ifNotNil: [
       result add: lf;
	  addAll: 'session number: ';
	  addAll: (aSessionNumber asString);
	  addAll: (self _sessionUproString: aUserProfile) .
     ].
   ].
].
^result

]

{ #category : 'Session Control' }
System class >> otherSessionsReport [
  "Returns a String describing sessions other than the one executing this method.
   Requires SessionAccess privilege."
  | mySession |
  mySession := System session .
  ^ self _sessionsReport: (self currentSessions reject:[:n | n == mySession ])

]

{ #category : 'Performance Monitoring' }
System class >> pageReads [

"Returns the number of Repository page read operations performed since the start
 of the Gem process.  If the Gem is remote, this corresponds to reads performed
 in the current session.  If the Gem is linked, it corresponds to reads
 performed since the application was invoked."

^ self _zeroArgPrim: 2

]

{ #category : 'Cache Statistics' }
System class >> pageServerCacheName: aString [

"Sets the cache name for the page server thread serving the current session.
 Also sets the cache name for the page server on the mid-level cache, if it
 exists.

 Raises an error if aString is too long (exceeds 31 characters) or is empty.

 Has no effect and returns false if the current session exists on the same host
 as the stone process.  Otherwise returns true."

<primitive: 1019>
self _primitiveFailed: #pageServerCacheName: args: { aString }

]

{ #category : 'Runtime Configuration Access' }
System class >> pageServerConnectionIsEncrypted [

"Answer a Boolean indicating whether the network connection to the gem's page server is
 encrypted (using TLS).  Always answers false for gems without a page server, i.e., gems
 that run on the same host as the stone process."

^ self _zeroArgPrim: 176

]

{ #category : 'Performance Monitoring' }
System class >> pageWrites [

"Returns the number of Repository page write operations performed since the
 start of the Gem process.  If the Gem is remote, this corresponds to writes
 performed in the current session.  If the Gem is linked, it corresponds to
 writes performed since the application was invoked."

^ self _zeroArgPrim: 3

]

{ #category : 'Private Cache Warming' }
System class >> parseWarmerConfig: aString [
"Returns a SymbolKeyValueDictionary with zero or more keys
   #midHost #d #n #w
"
 | res |
 res := SymbolKeyValueDictionary new .
 aString ifNotNil:[ | arr n |
   arr := aString subStringsDelimitedByAny: { Character space . Character tab }.
   n := 1 .
   [ n <= arr size ] whileTrue:[ | s val |
     s := arr at: n . n := n + 1 .
     s = '-d' ifTrue:[ res at: #d put: true ] ifFalse:[
     s = '-n' ifTrue:[
        val := arr atOrNil: n. n := n + 1.
        val ifNil:[ Error signal:'missing value for -n'].
        val := Integer fromString: val .
        (val < 1 or:[ val > 20]) ifTrue:[
          Error signal:'''-n ', val asString, ''' , number of threads not within 1..20'].
        res at: #n put: val .
     ] ifFalse:[
     s = '-M' ifTrue:[
       val := arr atOrNil: n. n := n + 1.
       val ifNil:[ Error signal:'missing value for -M'].
       res at: #midHost put: val .
     ] ifFalse:[
     s = '-w' ifTrue:[
       val := arr atOrNil: n. n := n + 1.
       val ifNil:[ Error signal:'missing value for -w'].
       val := Integer fromString: val .
       res at: #workingSetInterval put: val .
     ]]]].
   ]
 ].
 ^ res
]

{ #category : 'Host System Access' }
System class >> performOnServer: aString [
"This method causes the operating system commands in aString to be executed as
 a spawned subprocess using the default shell, /bin/sh.  Generates an error
 if aString cannot be executed by the operating system.

 Under Unix, commands in aString can have exactly the same form as a shell
 script.  For example, newlines or semicolons can separate commands, and a
 backslash can be used as an escape Character.

 Returns the stdout result of the spawned process as a String .
 If the result is actually UTF8 data, the application must determine
 when to send decodeFromUTF8 to the result.

 Signals an error if the privilege NoPerformOnServer is true and aString is
 not present in the allowlist of allowed commands for the session's UserProfile.
 See the methods in UserProfile under category PerformOnServer for more
 information."

^ self performOnServer: aString withShell: nil

]

{ #category : 'Host System Access' }
System class >> performOnServer: aString withShell: aShellOrNil [

"This method causes the operating system commands in aString to be executed as
 a spawned subprocess.  Generates an error if aString cannot be executed by the
 operating system.

 Under Unix, commands in aString can have exactly the same form as a shell
 script.  For example, newlines or semicolons can separate commands, and a
 backslash can be used as an escape Character.

 aShellOrNil may be used to specify the shell to use to process the command.
 If aShellOrNil is nil, the default shell (/bin/sh) is used. Otherwise the
 specified shell is used. The specified shell must accept the -c command line
 argument. Note that C SHell (/bin/csh) and its variants (/bin/tcsh) are known
 to not work with this method. However Korn SHell (/bin/ksh), Z SHell
 (/bin/zsh) and Bourne-Again SHell (/bin/bash) are known to work.

 Returns the stdout result of the spawned process as a String .
 If the result is actually UTF8 data, the application must determine
 when to send decodeFromUTF8 to the result.

 Signals an error if the privilege NoPerformOnServer is true and aString is
 not present in the allowlist of allowed commands for the session's UserProfile.
 See the methods in UserProfile under category PerformOnServer for more
 information."

| resultStr arr |
arr := self _performOnServer: aString withShell: aShellOrNil.
resultStr := arr at: 3 .
resultStr ifNil:[  | rawStatus childStatus errno errMsg |
  rawStatus := arr at: 1 .
  childStatus := arr at: 2 .
  errno := arr at: 5 .
  errMsg := arr at: 4 .
  aString _error: #hostErrPerform args:{ errMsg . errno . rawStatus . childStatus }.
  ^ nil
].
resultStr _stringCharSize == 0 ifTrue:[
  ^ resultStr decodeToString  "decode a Utf8"
].
^ resultStr

]

{ #category : 'Persistent Counters' }
System class >> persistentCounterAt: index [

"Returns the value of the persistent shared counter at index, which
 may be a SmallInteger or LargeInteger.

 See the method:
   System>>_updateSharedCounterAt: index by: amount withOpCode: code
 for more information on persistent shared counters."

^self _updateSharedCounterAt: index by: 0 withOpCode: 0

]

{ #category : 'Persistent Counters' }
System class >> persistentCounterAt: index decrementBy: amount [

"Decrements the persistent shared counter at index by the specified
 amount. Amount must be a SmallInteger or LargeInteger.
 For a LargeInteger,  amount must be representable as a signed
 64-bit integer (between -9223372036854775808 and 9223372036854775807).

 Returns the new value of the counter after the decrement.

 See the method:
   System>>_updateSharedCounterAt: index by: amount withOpCode: code
 for more information on persistent shared counters."

^self _updateSharedCounterAt: index by: amount negated withOpCode: 0

]

{ #category : 'Persistent Counters' }
System class >> persistentCounterAt: index incrementBy: amount [

"Increments the persistent shared counter at index by the specified
 amount.  Amount must be a SmallInteger or LargeInteger.
 For a LargeInteger,  amount must be representable as a signed
 64-bit integer (between -9223372036854775808 and 9223372036854775807).

 Returns the new value of the counter after the increment.

 See the method:
   System>>_updateSharedCounterAt: index by: amount withOpCode: code
 for more information on persistent shared counters."

^self _updateSharedCounterAt: index by: amount withOpCode: 0

]

{ #category : 'Persistent Counters' }
System class >> persistentCounterAt: index put: value [

"Sets the persistent shared counter at index to the specified
 value.  value must be a SmallInteger or LargeInteger.
 For a LargeInteger,  amount must be representable as a signed
 64-bit integer (between -9223372036854775808 and 9223372036854775807).

 Returns value, the new value of the counter.

 See the method:
   System>>_updateSharedCounterAt: index by: amount withOpCode: code
 for more information on persistent shared counters."

^self _updateSharedCounterAt: index by: value withOpCode: 1

]

{ #category : 'System Control' }
System class >> possibleDeadSize [

"Return the number of possible dead objects in the system."
^self stoneCacheStatisticWithName: 'PossibleDeadObjs'

]

{ #category : 'System Control' }
System class >> possibleDeadSymbols [

"Return the number of possible dead Symbols in the system."
^self stoneCacheStatisticWithName: 'PossibleDeadSymbols'

]

{ #category : 'Cache Statistics' }
System class >> primaryCacheMonitorCacheStatisticWithName: aString [

"Return the value of the cache stat with the given name for the cache monitor
 process running on the stone's host.  The name must match an element of the
 Array returned by the #cacheStatisticsDescription method  applicable to the
 cache monitor process.  The UserTime and SysTime statistics cannot be accessed
 using this method.

 Returns nil if a statistic matching aString was not found for the cache monitor.

 This method may be used on hosts remote from the stone process."

^ self _cacheStatWithName: aString opCode: 1

]

{ #category : 'Session Control' }
System class >> priorityForSessionId: aSessionId [

"Returns an integer between 0 and 4 indicating the current priority
 of the session with the given session ID.  Returns -1 if aSession ID
 does not refer to a valid session.

 Requires the SessionAccess privilege to query the priority of another
 session.  No privilege is required to query the priority of the current
 session."

^self _sessionPriorityPrim: -1 forSessionId: aSessionId

]

{ #category : 'Cache Statistics' }
System class >> processIdForSlot: anInt [

"Return the process ID of the process using the ncache slot at
 index anInt.  Return -1 if the slot is not in use or out of
 range."

^ self _primCacheStatsForSlotOrSession: anInt opCode: 4

]

{ #category : 'Cache Statistics' }
System class >> processKindForSlot: anInt [

"Return the process kind occupying the given cache slot as follows:

   1 = Shared Page Cache Monitor
   2 = Stone
   4 = Remote gem page server
   8 = Gem main thread (including Topaz, GBS, and other GCI applications).
 128 = Page Manager thread
 256 = Stone restore thread
 512 = A gem thread
1024 = Stone Aio thread
2048 = Stone free frame thread
4096 = Remote cache page server thread
8192 = Remote gem page server thread

 A return value of -1 means the slot index was out of range or the
 slot is not in use."

^ self _primCacheStatsForSlotOrSession: anInt opCode: 9

]

{ #category : 'Performance Monitoring' }
System class >> profMonSample [

"Take a snapshot of the current execution stack for a ProfMonitor report.
 Only applies if a ProfMonitor instance is actively monitoring and its
 timing interval has been set to zero.  See documention for ProfMonitor
 for more details."

^ self _zeroArgPrim: 157

]

{ #category : 'Reduced Conflict Support' }
System class >> rcValueCache [

"Returns the cache dictionary that is stored in temporary session state
 used to hold calculated values for reduced conflict classes.  If it does
 not exist, create it."

| dict |
dict := self __sessionStateAt: 5.
dict ifNil:[
    dict := IdentityKeyValueDictionary new: 47.
    self __sessionStateAt: 5 put: dict
].
^ dict

]

{ #category : 'Reduced Conflict Support' }
System class >> rcValueCacheFor: anObject [
  (self __sessionStateAt: 5"_rcValueCache") ifNotNil:[:dict | 
     ^ dict at: anObject otherwise: nil 
  ].
  ^ nil
]

{ #category : 'Reduced Conflict Support' }
System class >>  _rcv: valueArray at: aSymbol  [
  "For a  valueArray of the rcValueCache, return value for specfied key, or return nil."
  1 to: valueArray size by: 2 do: [ :i |
    aSymbol == (valueArray at: i) ifTrue:[ ^ valueArray at: (i + 1) ]
  ].
  ^ nil .
]

{ #category : 'Reduced Conflict Support' }
System class >> _rcv: valueArray increment: aSymbol by: delta [
  "For a valueArray of the rcValueCache, return offset of value for specfied key, or return nil."
  1 to: valueArray size by: 2 do: [ :i |
    aSymbol == (valueArray at: i) ifTrue:[ | j |
      j := i + 1 .
      valueArray at: j put: (valueArray at:j) + delta .
    ].
  ].
]

{ #category : 'Reduced Conflict Support' }
System class >> rcValueCacheAt: aKey for: anObject ifAbsent: aBlock [

"Returns the associated value at the given key for anObject.
 If the key is not present, execute the zero-argument block."

| valueArray cache |
cache := (self __sessionStateAt: 5"_rcValueCache") .
" if the cache does not exist, then the key is not present "
cache ifNil: [ ^ aBlock value ].

valueArray := cache at: anObject otherwise: nil .
valueArray ifNil:[ ^ aBlock value ].

1 to: valueArray size by: 2 do: [ :i |
    aKey == (valueArray at: i)
        ifTrue: [ ^ valueArray at: (i + 1) ]
].

^ aBlock value

]

{ #category : 'Reduced Conflict Support' }
System class >> rcValueCacheAt: aKey for: anObject otherwise: aValue [

 "Returns the associated value at the given key for anObject.
  If the key is not present, returns aValue."
  | valueArray cache |
  cache := (self __sessionStateAt: 5"_rcValueCache").
  " if the cache does not exist, then the key is not present "
  cache ifNil:[ ^ aValue ].
  (valueArray := cache at: anObject otherwise: nil) ifNil:[
    ^ aValue 
  ].
  1 to: valueArray size by: 2 do: [ :i |
    aKey == (valueArray at: i) ifTrue:[ 
      ^ valueArray at: (i + 1) 
    ]
  ].
  ^ aValue
]

{ #category : 'Reduced Conflict Support' }
System class >> rcValueCacheAt: aKey otherwise: aValue [

"Returns the associated value at the given key for anObject.
 If the key is not present, returns aValue."

  | cache |
  (cache := (self __sessionStateAt: 5"_rcValueCache")) ifNil:[
     ^ aValue .
  ].
  ^ cache at: aKey otherwise: aValue.

]

{ #category : 'Reduced Conflict Support' }
System class >> rcValueCacheIncrement: aKey for: anObject by: delta [

 (self __sessionStateAt: 5"_rcValueCache") ifNotNil:[:cache|
   (cache at: anObject otherwise: nil) ifNotNil:[:valueArray |
      1 to: valueArray size by: 2 do: [ :i |
        aKey == (valueArray at: i) ifTrue:[ | j |
          j := i + 1 .
          valueArray at: j put: (valueArray at: j) + delta
        ].
      ]
    ]
  ]
]

{ #category : 'Reduced Conflict Support' }
System class >> rcValueCacheIfPresent: aKey put: aValue for: anObject [

 (self __sessionStateAt: 5"_rcValueCache") ifNotNil:[:cache|
   (cache at: anObject otherwise: nil) ifNotNil:[:valueArray |
      1 to: valueArray size by: 2 do: [ :i |
        aKey == (valueArray at: i) ifTrue:[ | j |
          j := i + 1 .
          valueArray at: j put: aValue .
        ].
      ]
    ]
  ]
]

{ #category : 'Reduced Conflict Support' }
System class >> rcValueCacheAt: aKey put: aValue for: anObject [

"Adds the given key/value pair for anObject.  Returns the receiver."
  | valueArray cache |
  cache := self rcValueCache .
  (valueArray := cache at: anObject otherwise: nil) ifNil:[
    valueArray := { } .
    cache at: anObject put: valueArray.
  ].
  1 to: valueArray size by: 2 do: [ :i |
    aKey == (valueArray at: i) ifTrue: [
            valueArray at: (i + 1) put: aValue.
            ^ self
    ]
  ].
  valueArray addLast: aKey; addLast: aValue
]

{ #category : 'Performance Monitoring' }
System class >> readClockNano [

"Returns an Integer indicating the amount of CPU time used by the current
 process, in units of nanoseconds.  The resolution of the result is operating
 system dependent."

^ self _zeroArgPrim: 178

]

{ #category : 'Setting Locks' }
System class >> readLock: anObject [

"Requests a read lock on anObject. This method denies a read lock
 on anObject under any one of the following circumstances:

 * Another session has a write lock on the object.

 * The object is special.

 Note that once a readLock is obtained, any write of the object will
 cause a commit failure, even by the session holding the read lock."

| result |
result := self _lock: anObject kind: 2 autoRelease: false .
(result == 1) ifTrue:[ ^ self ].
self _lockError: result obj: anObject details: 'read lock'

]

{ #category : 'Setting Locks' }
System class >> readLock: anObject ifDenied: denyBlock ifChanged: changeBlock [

"Requests a read lock on anObject.  This method denies a read lock
 on anObject under any one of the following circumstances:

 * Another session has a write lock on the object.

 * The object is special.

 Returns the receiver if the requested lock was granted and was not dirty.

 Note that once a readLock is obtained, any write of the object will
 cause a commit failure, even by the session holding the read lock.

 If the requested lock is otherwise denied, it returns the value of
 the zero-argument block denyBlock.  If it grants a dirty lock, then it
 returns the value of the zero-argument block changeBlock.  In that case the
 lock remains, even after the transaction is aborted."

| result |

result := self _lock: anObject kind: 2 autoRelease: false .
(result == 1) ifTrue:[ ^ self ].
^ self _lockEvaluateErr: result obj: anObject denied: denyBlock changed: changeBlock

]

{ #category : 'Setting Locks' }
System class >> readLockAll: aCollection [

"Requests a read lock on each object in aCollection.  This method denies
 a read lock on an object under any one of the following circumstances:

 * Another session already holds a write lock on the object.

 * The object is special.

 This method grants a read lock on an object whenever it finds no reason
 to deny it.  However, a lock that it grants may be dirty.  One session's
 lock is dirty if another session has committed a change to the locked object
 since the beginning of the first session's current transaction.  A session
 that holds a dirty lock cannot commit its transaction.  To clean its locks,
 it must abort the transaction and obtain updated values for each object whose
 lock is dirty.

 Note that once a readLock is obtained, any write of the object will
 cause a commit failure, even by the session holding the read lock.

 If a lock was acquired for every element of aCollection, and no locks are
 dirty, returns the receiver.

 This method generates an error if it is unable to acquire a lock for every
 element of aCollection, or if any lock that it acquires is dirty.  However,
 all the locks that it acquires remain in place, even after the current
 transaction is aborted."

 | result str nErr nDirty err |
 result := self _lockAll: aCollection kind: 2 .
 (result == self ) ifTrue: [^self] .
 
 str := (nErr := (result at: 1) size) asString , ' locks denied, ', 
      (nDirty := (result at: 2) size) asString , ' dirty locks ' .
 err := LockError new .
 err _number: ( nErr == 0 ifTrue:[ 2074 ] ifFalse:[ 2073 ]).
 ^ err args: result ; signal: str .
]

{ #category : 'Setting Locks' }
System class >> readLockAll: aCollection ifIncomplete: incompleteBlock [

"Requests a read lock on each object in aCollection.  This method denies
 a read lock on an object in the collection under any one of the following
 circumstances:

 * Another session already holds a write lock an object.

 * The object is special.

 If all requested locks were granted and none of the locks are 'dirty',
 returns the receiver.

 Note that once a readLock is obtained, any write of the object will
 cause a commit failure, even by the session holding the read lock.

 This method grants a read lock on an object whenever it finds no reason
 to deny it.  However, a lock that it grants may be dirty.  One session's
 lock is dirty if another session has committed a change to the locked object
 since the beginning of the first session's current transaction.  A session
 that holds a dirty lock cannot commit its transaction.  To clean its locks,
 it must abort the transaction and obtain updated values for each object whose
 lock is dirty.

 If this method is unable to acquire a lock for every element of aCollection,
 or if any lock that it acquires is dirty, then it returns the value of the
 three-argument block incompleteBlock.  The arguments to the block are:

 1.  An Array of objects that could not be locked.

 2.  An Array of objects that were locked but whose locks are dirty.

 3.  An empty Array, retained for backward compatibility with old GemStone versions.

 All the locks that it acquires remain in place, even after the current
 transaction is aborted."

| result |
result := self _lockAll: aCollection kind: 2 .
(result == System)
  ifTrue: [^self]
  ifFalse: " else we execute the incomplete block with 3 arguments "
          [^incompleteBlock value: (result at: 1)
                            value: (result at:2)
                            value: (result at:3)].

]

{ #category : 'Session Control' }
System class >> realUserId [

"Returns a SmallInteger indicating the numeric value of the real Unix user ID of the gem process."
^ self _zeroArgPrim: 167

]

{ #category : 'Session Control' }
System class >> realUserIdName [

"Returns a String indicating the real Unix user ID of the gem process."
^ self _zeroArgPrim: 168

]

{ #category : 'System Control' }
System class >> reclaimedSymbols [

"Return the number of Symbols reclaimed since stone was started."
^self stoneCacheStatisticWithName: 'ReclaimedSymbols'

]

{ #category : 'Deprecated' }
System class >> reclaimGcSessionCount [

"Return the number of reclaim GC sessions currently running."

self deprecated: 'System class >> reclaimGcSessionCount deprecated v3.2.  Use reclaimGemSessionCount instead'.
^self reclaimGemSessionCount

]

{ #category : 'Garbage Collection Management' }
System class >> reclaimGemConfigs [
"Returns a String that documents the reclaim gem configuration options"

^'
#deadObjsReclaimedCommitThreshold
   The maximum number of dead objects to reclaim in a single transaction.
     (default:20000, min:32, max: 2147483647)

#deferReclaimCacheDirtyThreshold
   Specifies a threshold percentage of dirty pages in the stones shared
   page cache.  If more than this percentage of pages in the stone
   shared cache are dirty pages, page reclaims will be deferred until
   the dirty percentage drops below the threshold minus 5%.
   Setting the value to 100 disables this feature.
     (default:75, min:10, max: 100)

#maxTransactionDurationUs
   Controls the approximate maximum length of a reclaim transaction in
   microseconds.  If the Reclaim gem has been in transaction longer than
   this duration, it will commit even if the #objsMovedPerCommitThreshold
   condition has not been satisfied.
     (default:100000, min:100, max: 20000000)

#objsMovedPerCommitThreshold
   Controls the approximate number of object table updates to be performed
   per transaction.  Reclaim gems will commit when at least this many
   live objects have been moved to new data pages.
     (default:20000, min:100, max: 2147483647)

#reclaimDeadEnabled
   Controls whether the reclaim will try to clean up dead objects.
     (default:true, min:false, max:true)

#reclaimMinPages
   The minimum number of pages to process in a single reclaim commit.  If there are not this
   many pages to reclaim, the reclaim gem sleeps until there are at least this number available.
     (default:40, min:1, max: 2147483647)

(reclaimThreads - For reclaim threads, see
  System class >> changeNumberOfReclaimThreads:
  and STN_NUM_GC_RECLAIM_SESSIONS in the stone config file) .

#reclaimVerboseLogging
   Controls the amount of logging information written to the log file.
   For compatibility with earlier versions that only accepted booleans, false = 0, true = 1
     (default: 0, min:0, max: 9)
   Warning: levels higher than 3 can add a significant number of entries to the reclaimGem log,
            so their use should be limited to diagnosing problems.
   These are the definitions for the levels:
      0 - only summary information every 5 minutes in slow, 15 minutes in a fast
      1 - log sigAbort, gcGemAlert, gcHwPage, etc
      2 - numDead from stone, numDead processed per thread
      3 - summary: commitCount, stayInTrans, crPage, numLive, numDead, numReclPages
      4 - abort, start phases
      5 - conflict set
      6 - num live and dead found per page
      7 - summary of info added to ot slot
      8 - deltas in rootPage
      9 - push live or dead

#sleepTimeBetweenReclaimUs
   Amount of time (in microseconds) to sleep after a reclaim that did work.
     (default:0, min:0, max:300000000)

#sleepTimeWithCrBacklogUs
   Amount of time (in microseconds) to sleep after a commit when the
   commit record backlog is larger than 1.25 * the current setting for
   STN_CR_BACKLOG_THRESHOLD.
     (default:0, min:0, max:300000000)
'
]

{ #category : 'Garbage Collection Management' }
System class >> reclaimGemSessionCount [

"Return the number of reclaim GC sessions (threads) currently running."

^ self _zeroArgPrim: 65

]

{ #category : 'Garbage Collection Management' }
System class >> reclaimGemSessionId [

"Returns the reclaim sessionId, zero if the reclaimGem is not running."

^ self _zeroArgPrim: 66

]

{ #category : 'Garbage Collection Management' }
System class >> reclaimGemSessionIds [

"Return an array with the reclaim sessionIds."
self deprecated: 'System class>>reclaimGemSessionIds deprecated v3.6. Use reclaimGemSessionId '.

^ { self reclaimGemSessionId }

]

{ #category : 'Reduced Conflict Support' }
System class >> redoLog [

  "Returns the redo log for the current transaction level
   Creates it if it does not exist.
   The redo log is used by Reduced Conflict classes."

  | redoLog arr lev |
  (lev := self _zeroArgPrim: 5"transactionLevel") == 0 ifTrue:[ lev := 1 ].
  arr := self __sessionStateAt: 4.
  arr ifNotNil:[
    (redoLog := arr atOrNil: lev) ifNotNil:[ ^ redoLog]
  ] ifNil:[
    arr := Array new: lev .
    self __sessionStateAt: 4 put: arr .
  ].
  arr size < lev ifTrue:[
    arr size: lev .
  ].
  (redoLog := arr at: lev) ifNil:[
    redoLog := RedoLog new.
    arr at: lev put: redoLog .
  ].
  ^ redoLog

]

{ #category : 'Transaction Control' }
System class >> refreshTransientSymbolList [

"Installs a transient symbol list in the current session by
 copying the symbolList of the current UserProfile.
 Does not require CodeModification privilege .

 GsCurrentSession currentSession transientSymbolList: (System myUserProfile symbolList copy)

 Returns the receiver."

^ self _zeroArgPrim: 129

]

{ #category : 'Shared Cache Management' }
System class >> remoteCachesList [

"Return an Array describing all shared caches that the
 stone process is managing, including the cache on the stone machine.

 Result contains one element per cache. Each
 element of result is a 8 element Array containing
   hostName, a String
   a Boolean - true if cache was created as a mid-level cache
   a SmallInteger - total number of sessions connected to the cache
   a SmallInteger - max number of sessions for the cache
   a SmallInteger - size of cache in KB
   a SmallInteger - number of sessions using cache as a mid-level cache
   a SmallInteger - zero or sessionId of hostagent on stone host servicing the cache
   a String       - ipAddress of the host of the cache
   a SmallInteger - zero or sessionId hostagent running on a mid-level cache host"

^ self _remoteCachesList: false

]

{ #category : 'Shared Cache Management' }
System class >> remoteCachesReport [

"Return the result of remoteCachesList formatted as a string,
 one line per cache."

^ self _formatCacheReport: (self _remoteCachesList: false)


]

{ #category : 'System Control' }
System class >> remoteSharedCacheCount [

"Return the number of remote shared page caches on the system."
^self stoneCacheStatisticWithName: 'RemoteSharedPageCacheCount'

]

{ #category : 'Runtime Configuration Access' }
System class >> removeGemLogOnExit: aBoolean [

"Set state in a gem process that overrides the state of the
 GEMSTONE_KEEP_LOG environment variable. See $GEMSTONE/sys/gemnetdebug
 for documentation on GEMSTONE_KEEP_LOG .

 If aBoolean is true, the gem log file will be deleted if the
 gem process exits normally. If aBoolean is false, the
 the gem log file will not be deleted .

 Has no effect in a linked session"

<primitive: 554>
aBoolean _validateClass: Boolean .
^self _primitiveFailed: #removeGemLogOnExit: args: { aBoolean }

]

{ #category : 'Removing Locks' }
System class >> removeLock: anObject [

"Removes the lock held by the current session on anObject.
 Returns the receiver."

self removeLockAll: { anObject }.
^self

]

{ #category : 'Removing Locks' }
System class >> removeLockAll: aCollection [

"Removes all locks held by the current session on the objects in aCollection.
 If an object in aCollection is not locked by the current session, that
 object is ignored. Returns the receiver."

<primitive: 99>
| anArray |
anArray := aCollection asArray.
anArray == aCollection ifFalse:[
  ^ self removeLockAll: anArray .
  ] .
aCollection _validateClass: Collection.
self _primitiveFailed: #removeLockAll: args: { aCollection } .
self _uncontinuableError

]

{ #category : 'Removing Locks' }
System class >> removeLocksForSession [

"Removes all locks held by this session.  Returns the receiver.
 This method succeeds even if the session no longer has
 read authorization for one or more of its locked objects."

^ self _zeroArgPrim: 0

]

{ #category : 'Runtime Configuration Access' }
System class >> repositoryEncryptionScheme [

"Answers a symbol describing the algorithm used to encrypt the repository (extents and tranlogs) or
 #none if the repository is not encrypted."

^ self _zeroArgPrim: 1039

]

{ #category : 'Runtime Configuration Access' }
System class >> repositoryIsEncrypted [

"Answers a boolean indicating if the repository is encrypted."
^self repositoryEncryptionScheme ~~ #none

]

{ #category : 'Runtime Configuration Access' }
System class >> repositoryPrivateKey [

"If the repository is encrypted and the current user is SystemUser, returns an
 instance of GsTlsPrivateKey which represents the private key used to decrypt
 the repository. Returns nil if the repository is not encrypted. Raises an
 exception if this method is invoked by any user except SystemUser."

^self _zeroArgPrim: 186

]

{ #category : 'Runtime Configuration Access' }
System class >> repositoryPublicKeys [

"If the repository is encrypted, returns an Array of instances of GsTlsPublicKey or
 GsX509Certificate which represents the public keys used to encrypt the repository.
 Returns nil if the repository is not encrypted."

^self _zeroArgPrim: 185

]

{ #category : 'Runtime Configuration Access' }
System class >> repositoryPublicKey [

"Deprecated. Use System repositoryPublicKeys"

  self deprecated: 'System class>>repositoryPublicKey deprecated v3.7.1. Use System repositoryPublicKeys'.

^ self repositoryPublicKeys ifNil:[ nil ] ifNotNil:[:a| a first ]
]

{ #category : 'Online Backup Support' }
System class >> resumeCheckpoints [
"Resumes regular checkpoints if they were previously suspended by
 the System suspendCheckpointsForMinutes: method.

 Requires the SystemControl privilege.

 Returns the previous checkpoint state.  Returns true if checkpoints
 were suspended or false if checkpoints were not suspended."

 ^self _zeroArgPrim: 60

]

{ #category : 'System Control' }
System class >> resumeLogins [

"Allows new sessions to be initiated.  (Enables users to login.)  Logins are
 enabled when the GemStone system is started.  This message reverses the effect
 of System | suspendLogins.  Requires the SystemControl privilege."

self configurationAt: #StnLoginsSuspended put: 0

]

{ #category : 'System Control' }
System class >> scavengablePagesCount [

"Legacy method.
 Return the number of scavengable pages in the system."
^self stoneCacheStatisticWithName: 'PagesNeedReclaimSize'

]

{ #category : 'System Control' }
System class >> pagesNeedReclaimCount [
  "Return number of pages not yet reclaimed by the reclaimGem."
  ^ self stoneCacheStatisticWithName: 'PagesNeedReclaimSize'
]

{ #category : 'Environment Access' }
System class >> secondsSinceStoneStarted [

"Answer a SmallInteger which is the number of seconds since the stone started."

^ self _zeroArgPrim: 124

]

{ #category : 'System Control' }
System class >> sendLostOtToSession: sessionIdOrSerial [

"Sends a lostOt signal to the specified session.
 To execute this method, you must have the SystemControl privilege.

 The sessionIdOrSerial argument is a SmallInteger specifying a session.  A
 positive value is interpreted as a serialNumber.  A negative value is
 interpreted as a negated sessionId.

 Returns a Boolean , true if stone sent the OOB byte to the
 specified session, false of the OOB socket was blocked."

^ self _sendTransactionSignal: sessionIdOrSerial kind: 2

]

{ #category : 'System Control' }
System class >> sendSigAbortToSession: sessionIdOrSerial [

"Sends a sigAbort signal to the specified session.
 To execute this method, you must have the SystemControl privilege.

 The sessionIdOrSerial argument is a SmallInteger specifying a session.  A
 positive value is interpreted as a serialNumber.  A negative value is
 interpreted as a negated sessionId.

 SigAborts sent to any session more frequently than
 every 2 seconds are ignored.
 Returns a Boolean , true if stone sent the OOB byte to the
 specified session, false of the OOB socket was blocked."

^ self _sendTransactionSignal: sessionIdOrSerial kind: 0

]

{ #category : 'Signals' }
System class >> sendSignal: aSignal to: aSessionId withMessage: aString [

"Sends a signal (a SmallInteger) to the specified session (a SmallInteger) with
 aString as a message.  The aString argument is currently limited to 1023
 bytes."

aSessionId _validateClass: SmallInteger.
aSessionId < 1 ifTrue:[ aSessionId _error: #rtErrArgNotPositive  ].
self _sendSignal: aSignal toSess: aSessionId negated withMessage: aString oob: nil

]

{ #category : 'System Control' }
System class >> sendSignalFinishTransactionToSession: sessionIdOrSerial [

"Sends a FinishTransaction signal to the specified session.
 To execute this method, you must have the SystemControl privilege.

 The sessionIdOrSerial argument is a SmallInteger specifying a session.
 A positive value is interpreted as a serialNumber.  A negative value is
 interpreted as a negated sessionId.

 Returns a Boolean , true if stone sent the OOB byte to the
 specified session, false of the OOB socket was blocked."

^ self _sendTransactionSignal: sessionIdOrSerial kind: 1

]

{ #category : 'Session Control' }
System class >> session [

"Returns a SmallInteger representing the stone sessionId of the sender."

^self _zeroArgPrim: 33

]

{ #category : 'Cache Statistics - Session' }
System class >> sessionCacheStatAt: anIndex [

"Returns the value of the session statistic at the specified index, which
 must be in the range -2 to 47.  0 to 47 are for application use.  -1 means
 return the value of the progressCount statistic.  -2 means return the value
 of the indexProgressCount statistic."

^ self _sessionCacheStatAt: anIndex incrementBy: 0

]

{ #category : 'Cache Statistics - Session' }
System class >> sessionCacheStatAt: anIndex decrementBy: aSmallInt [

"Decrements the value of the session statistic at the specified index
 by aSmallInt and returns the new value.  anIndex must be in the range
 of -2 to 47. 0 to 47 are for application use.  -1 means decrement the
 progressCount statistic.  -2 means decrement the indexProgressCount
 statistic."

^ self _sessionCacheStatAt: anIndex incrementBy: aSmallInt negated

]

{ #category : 'Cache Statistics - Session' }
System class >> sessionCacheStatAt: anIndex incrementBy: aSmallInt [

"Increments the value of the session statistic at the specified index
 by aSmallInt and returns the new value.  anIndex must be in the range
 of -2 to 47.  0 to 47 are for application use.
 -1 means increment the progressCount statistic.  -2 means increment the
 indexProgressCount statistic."

 ^self _sessionCacheStatAt: anIndex incrementBy: aSmallInt

]

{ #category : 'Cache Statistics - Session' }
System class >> sessionCacheStatAt: anIndex put: i [

"This method sets the session statistic at the specified index (which should be
 in the range -2 to 47) to the specified value i, which must be a SmallInteger.
 0 to 47 are for application use.  -1 means set the progessCount statistic to
 the value i.  -2 means set the indexProgessCount statistic to the value i."

 ^ self _sessionCacheStatAt: anIndex put: i

]

{ #category : 'Cache Statistics - Session' }
System class >> sessionCacheStatsForProcessSlot: aProcessSlot [

"Return an array containing the 48 SmallIntegers representing the
 session statistics for the given process slot; the stats
 corresponding to indexes 0 to 47, not including the stats for -2
 and -1.

 Returns nil if the given process slot is not found or if the slot
 is not in use by a gem process.

 See additional information in the method #_sessionCacheStatAt: put:"

^self _sessionCacheStatsForSlotOrSessionId: aProcessSlot

]

{ #category : 'Cache Statistics - Session' }
System class >> sessionCacheStatsForSessionId: aSessionId [

"Return an array containing the 48 session statistics for the given
 session ID; the stats corresponding to indexes 0 to 47, not including
 the stats for -2 and -1.

 Returns nil if the given session is not found or if the slot is not
 in use by a gem process.

 See additional information in the method #_sessionCacheStatAt: put:"

^self _sessionCacheStatsForSlotOrSessionId: aSessionId negated.

]

{ #category : 'Transaction Control' }
System class >> sessionCanCommit [

"Answers true if the session may commit the current transaction (if any).
 Otherwise answers false."

"Must be in a transaction to commit."
self inTransaction
  ifFalse:[ ^ false].
self commitsDisabled 
  ifTrue:[ ^ false].
self myUserProfile isReadOnly
  ifTrue:[^ false]. "This user can never commit"
(SystemRepository restoreActive)
  ifTrue:[^ false]. "restore from log or backup is active"
GsSession currentSession isSolo 
  ifTrue:[ ^ false ].
^true

]

{ #category : 'System Control' }
System class >> sessionIsLinked [

"Answer true if the session and its client run in the same operating system
 process.  Otherwise answer false."

^ self sessionIsRpc not

]

{ #category : 'Session Control' }
System class >> sessionIsOnStoneHost [

"Answer true if this session is running on the same host computer as the stone
 process.  Otherwise answer false."

^ self _zeroArgPrim: 137

]

{ #category : 'System Control' }
System class >> sessionIsRpc [

"Answer true if the session and its client run in different operating system
 processes.  Otherwise answer false."

^ self _zeroArgPrim: 175

]

{ #category : 'Session Control' }
System class >> sessionIsUsingHostagent [
"Answer true if this session is connected to stone via a hostagent process,
 i.e. true if this session used GciX509Login .
 Otherwise answer false."

^ ((System descriptionOfSession: System session) at: 23) > 0

]

{ #category : 'Lock Status' }
System class >> sessionLocks [

"Returns a three-element Array describing locks held by current session.
 The elements are
   an Array of read-locked objects
   an Array of write-locked objects
   an Array of objects with deferred unlocks
 If the current session holds no locks of a particular kind (read, write),
 then the corresponding Array is empty.
 Deferred unlocks are objects for which the unlock request was received by
 stone while another session was holding the commit token; they will
 be unlocked as soon as the commit token is released.
 "

<primitive: 352>
self _primitiveFailed: #sessionLocks .
self _uncontinuableError

]

{ #category : 'Environment Access' }
System class >> sessionPerformingBackup [

"Returns the session ID of the session that is performing a backup.
 If there is no such session, returns -1."

^ self configurationAt: #SessionInBackup

]

{ #category : 'Deprecated' }
System class >> sessionIdHoldingGcLock [
 "Deprecated. Returns a SmallInteger which is the session ID of the session currently holding
  the garbage collection lock.  Returns 0 if the GC lock is free."

  self deprecated: 'System class >> sessionIdHoldingGcLock deprecated v3.7.  Use sessionsHoldingGcLock'.
  ^ (self sessionsHoldingGcLock atOrNil: 1) ifNil:[ 0 ].
]

{ #category : 'Session Control' }
System class >> sessionsHoldingGcLock [
  "Returns an Array of sessionIds of sessions holding a garbage collection or
   repository scan lock."
  | sess |
  self gcLocksCount == 0 ifTrue:[ ^ #() ].
  sess := self currentSessions .
  ^ sess select:[:sessId | ((self descriptionOfSession: sessId) at: 25) > 0 ] 
]

{ #category : 'Session Control' }
System class >> sessionsReferencingOldestCr [

"Returns an Array containing the sessionIds of the sessions that are
 currently referencing the oldest commit record.  Sessions both inside and outside
 of a transaction will be returned.  Because a session can update its commit
 record without committing a transaction, it is possible that no session
 actually references the oldest commit record.  Therefore, this method
 may return an empty Array."

^self _zeroArgPrim: 21

]

{ #category : 'Session Control' }
System class >> sessionsReferencingOldestCrInTransaction [

"Returns an Array containing the sessionIds of the sessions that are in
 transaction and currently reference the oldest commit record."
^self _zeroArgPrim: 109

]

{ #category : 'Session Control' }
System class >> sessionsReferencingOldestCrNotInTransaction [

"Returns an Array containing the sessionIds of the sessions that are not in
 transaction and currently reference the oldest commit record."
^self _zeroArgPrim: 110

]

{ #category : 'Transient Session State' }
System class >> sessionStateAt: anIndex [

"Access to customer-available session states."

anIndex < 1 ifTrue: [  ArgumentError new
     object: anIndex ;
     signal: 'argument out of range in sessionStateAt:'].
^ self __sessionStateAt: anIndex + 39

]

{ #category : 'Transient Session State' }
System class >> sessionStateAt: anIndex put: aValue [

"Access to customer-available session states."
(anIndex < 1 or: [anIndex > 1994]) ifTrue: [  ArgumentError new
     object: anIndex ;
     signal: 'argument out of range in sessionStateAt:put:'].
^ self __sessionStateAt: anIndex + 39 put: aValue

]

{ #category : 'Transient Session State' }
System class >> sessionStateSize [

"Return the current size of the public part of the Session State Array."

^self _sessionStateSize - 39.

]

{ #category : 'Garbage Collection Management' }
System class >> setAdminConfig: configSymbol toValue: aVal [

"Sets the specified Admin gem runtime configuration option to aVal.  This change
 remains in effect until it is changed with this method or setPersistentAdminConfig:.

 Boolean values can be set using either true/false or 1/0.
 See getAdminConfig documentation for the list of valid config symbols.
 Returns the new value.
 Requires garbage collection privilege.
 The new setting should take effect within a few seconds unless the admin process is in a long
 operation such as epochGc or a wsUnionSweep.
 The adminGem will print 'Parameter changes noticed:' message to its log file when it discovers the change.
"
^self _gcGemConfig: false symb: configSymbol toValue: aVal

]

{ #category : 'Garbage Collection Management' }
System class >> setDefaultAdminConfigs [
  "Sets all of the admin configuration parameters to their default values.

   ***** Note: changes to these values need to be made in the comments in the
         adminGemConfigs method and gcgemutl.c

   This method aborts the current transaction; if an abort would cause unsaved
   changes to be lost it signals an error, #rtErrAbortWouldLoseData.

   You must have the GarbageCollection privilege to execute this method.
  "

  self needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

  self setPersistentAdminConfig: #adminVerboseLogging toValue: false .
  self setPersistentAdminConfig: #epochGcMaxThreads toValue: 1 .
  self setPersistentAdminConfig: #epochGcPageBufferSize toValue: 64 .
  self setPersistentAdminConfig: #epochGcPercentCpuActiveLimit toValue: 90.
  self setPersistentAdminConfig: #epochGcTimeLimit toValue: 3600.
  self setPersistentAdminConfig: #epochGcTransLimit toValue: 5000.
  self setPersistentAdminConfig: #saveWriteSetUnionToFile toValue: false .
  self setPersistentAdminConfig: #sweepWsUnionMaxThreads toValue: 1 .
  self setPersistentAdminConfig: #sweepWsUnionPageBufferSize toValue: 64 .
  self setPersistentAdminConfig: #sweepWsUnionPercentCpuActiveLimit toValue: 90 .

]

{ #category : 'Garbage Collection Management' }
System class >> setDefaultReclaimConfigs [
  "Sets all of the reclaim configuration options to their default values.

   ***** Note: changes to these values need to be made in the comments in the
         reclaimGemConfigs method and gcgemutl.c

   This method aborts the current transaction; if an abort would cause unsaved
   changes to be lost it signals an error, #rtErrAbortWouldLoseData.

  "

  self needsCommit ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

  self setPersistentReclaimConfig: #deadObjsReclaimedCommitThreshold toValue: 20000 .
  self setPersistentReclaimConfig: #deferReclaimCacheDirtyThreshold toValue: 75 .
  self setPersistentReclaimConfig: #maxTransactionDurationUs toValue: 10 * 1000000.
  self setPersistentReclaimConfig: #objsMovedPerCommitThreshold toValue: 20000 .
  self setPersistentReclaimConfig: #reclaimDeadEnabled toValue: true.
  self setPersistentReclaimConfig: #reclaimMinPages toValue: 40 .
  self setPersistentReclaimConfig: #reclaimVerboseLogging toValue: false .
  self setPersistentReclaimConfig: #sleepTimeBetweenReclaimUs toValue: 0.
  self setPersistentReclaimConfig: #sleepTimeWithCrBacklogUs toValue: 0.
]

{ #category : 'Cache Statistics - Session' }
System class >> setGemKind: aSmallInt [

"Set the gemKind cache stat to be aSmallInt.
 The default gemKind is zero. Customers are free to use any positive value for
 this statistic. Negative values are reserved for internal use."

^ self _sessionCacheStatAt: -3 put: aSmallInt
]

{ #category : 'Transient Session State' }
System class >> setIndexProgressCountTo: aSmallInt [

"Sets the value of the indexProgressCount statistic to aSmallInt.
 Returns aSmallInt."

^ self _sessionCacheStatAt: -2 put: aSmallInt

]

{ #category : 'System Control' }
System class >> setObjectReadTracking: aBoolean [
  "If (System objectReadLogEnabled == false)
   or ((System myUserProfile _hasPrivilegeName: #DisableObjectReadLogging) == true)
   has no effect and always returns false .

   If (System objectReadLogEnabled == true)
   and ((System myUserProfile _hasPrivilegeName: #DisableObjectReadLogging) == false)
   and ((System myUserProfile _hasPrivilegeName: #DynamicDisableObjectReadLogging) == true)
   returns the previous state of readLoggingEnabled,
   and sets readLoggingEnabled for this session to aBoolean .
   aBoolean == true enables read tracking.
   aBoolean == false will  disable any read tracking implied by
   GsObjectSecurityPolicy >> trackReads .

   Signals an Exception if aBoolean == false and
   ((System myUserProfile _hasPrivilegeName: #DynamicDisableObjectReadLogging) == false) .
  "
<primitive: 824>
aBoolean _validateClass: Boolean .
self _primitiveFailed: #setObjectReadTracking: args: { aBoolean }
]

{ #category : 'Garbage Collection Management' }
System class >> setPersistentAdminConfig: configSymbol toValue: aVal [

"Updates the runtime admin configuration option, see setAdminConfig:toValue:.
 If that is successful, then it commits the change to the persistent configuration.
 If an abort would cause unsaved changes to be lost, it does not execute and
 signals an error, #rtErrAbortWouldLoseData.

 User must have 'Group System write' permissions for the GcUser's UserGlobals
 and garbage collection privilege.
"
^self _setPersistentConfig: false symbol: configSymbol toValue: aVal

]

{ #category : 'Garbage Collection Management' }
System class >> setPersistentReclaimConfig: configSymbol toValue: aVal [

"Updates the runtime reclaim configuration option, see setReclaimConfig:toValue:.
 If that is successful, then it commits the change to the persistent configuration.
 If an abort would cause unsaved changes to be lost, it does not execute and
 signals an error, #rtErrAbortWouldLoseData.

 User must have 'Group System write' permissions for the GcUser's UserGlobals
 and garbage collection privilege.
"
^self _setPersistentConfig: true symbol: configSymbol toValue: aVal

]

{ #category : 'Transient Session State' }
System class >> setProgressCountTo: aSmallInt [

"Sets the value of the progressCount statistic to aSmallInt.
 Returns aSmallInt."

^ self _sessionCacheStatAt: -1 put: aSmallInt

]

{ #category : 'Garbage Collection Management' }
System class >> setReclaimConfig: configSymbol toValue: aVal [

"Sets the specified runtime reclaim gem configuration option to aVal.  This change
 remains in effect until it is changed with this method or setPersistentReclaimConfig:.

 Boolean values can be set using either true/false or 1/0.
 See    System class >> getReclaimConfig:    for the list of valid config symbols.
 Returns the new value.
 Requires garbage collection privilege.
 The new setting should take effect within a few seconds unless the reclaim process
 is in a long commit.
 The reclaimGem will print 'Parameter changes noticed:' message to its log file when it discovers the change.
"

^self _gcGemConfig: true symb: configSymbol toValue: aVal

]

{ #category : 'Session Control' }
System class >> setSessionPriority: anInt forSessionId: aSessionId [

"Sets the priority of the given session to a new value.

 Session priority is used by the stone to order requests for
 service by sessions.  anInt must be one of the following valid
 priorities:

 0 - lowest priority
 1 - low priority
 2 - normal priority (default)
 3 - high priority
 4 - highest priority

 By default, user sessions have normal (medium) priority and reclaim and
 Admin GC gems have lowest priority.

 A session which holds or is about the receive the commit token is always
 given highest priority by the stone regardless of what priority is assigned
 to the session.

 Session priority is only pertinent when more than one session requests
 service from the stone during the same stone wakeup interval.  A single
 session requesting service is always processed immediately.

 Sessions with the same priority are serviced in ascending session ID order.

 Requires the SessionAccess and SessionPriority privileges.

 The priority of the PageManager session may not be changed.

 If successful, this method answers a SmallInteger which indicates the previous
 priority of the session.  -1 is returned if aSessionId does not refer to
 a valid session.  An exception is raised if anInt is not a valid priority or
 the session executing this method does not have the required privileges."

^self _sessionPriorityPrim: anInt forSessionId: aSessionId

]

{ #category : 'System Control' }
System class >> shadowPagesCount [

"Return the number of shadow pages in the system, same as scavengablePagesCount."
^self stoneCacheStatisticWithName: 'PagesNeedReclaimSize'

]

{ #category : 'Configuration File Access' }
System class >> sharedCacheAttributesReport [
"Returns a string describing attributes for the shared page cache to which the current
 session is attached. Returns nil if the current session is a solo session."

 | dict cacheName keys ws |
 dict := self getSharedCacheAttributes .
 dict ifNil:[ ^ nil ].
 cacheName := dict at: #cacheName .
 keys := SortedCollection withAll: dict keys .
 ws := WriteStream on: String new.
 ws nextPutAll: 'Attributes Report For Shared Cache Named ''';
    nextPutAll: cacheName;
    nextPut: $' ;
    nextPut: $: .

 keys do:[:key|
   ws lf ; tab ;
      nextPutAll: key asString;
      nextPutAll: ' -> ';
      nextPutAll: (dict at: key) asString .
].
^ ws contents

]

{ #category : 'Shared Counters' }
System class >> sharedCounter: index [

"Return the value at the shared counter at the specified index.
 index must be in the range 0...(System numSharedCounters - 1) .
 See sharedCounter:setValue: for more information on shared counters."

  ^(self _sharedCounterFetchValuesFrom: index to: index) first

]

{ #category : 'Shared Counters' }
System class >> sharedCounter: index decrementBy: amount [

"Decrement the value of the shared counter at index by the amount amount.
 Return the new value of that shared counter.
 See sharedCounter:setValue: for more information on shared counters.

 index argument must a SmallInteger in the range 0 to (System _numSharedCounters - 1).

 amount argument must be in the range 0 to 2**63 - 1 (9223372036854775807).
 If decrementing by <amount> would decrease the counter value to be less than the
 minimum (-2**63 or -9223372036854775808), the value is set to the minmum instead."

  ^self _sharedCounter: index decrementBy: amount withFloor: nil

]

{ #category : 'Shared Counters' }
System class >> sharedCounter: index decrementBy: amount withFloor: floorValue [

"Decrement the value of the shared counter at <index> by the amount <amount>.
 Return the new value of that shared counter.
 See sharedCounter:setValue: for more information on shared counters.

 index argument must a SmallInteger in the range 0 to (System _numSharedCounters - 1).

 amount argument must be in the range 0 to 2**63 - 1 (9223372036854775807).

 floorValue specifies the lowest counter value. If decrementing by <amount> would
 decrease the counter value to be less than the floorValue, the value is set to
 floorValue instead."

  ^self _sharedCounter: index decrementBy: amount withFloor: floorValue

]

{ #category : 'Shared Counters' }
System class >> sharedCounter: index incrementBy: amount [

"Increment the value of the shared counter at the specified index by the specified
 amount. Return the new value of that shared counter.
 See sharedCounter:setValue: for more information on shared counters.

 index argument must a SmallInteger in the range 0 to (System _numSharedCounters - 1).

 amount argument must be in the range 0 to 2**63 - 1 (9223372036854775807).
 If incrementing by <amount> would increase the counter value to be greater than the
 maximum (2**63 - 1 or 9223372036854775807), the value is set to the maximum instead."

<primitive: 354>
| sz |
index _validateClass: SmallInteger.
sz := self _numSharedCounters .
(index < 0 or:[ index >= sz ])
  ifTrue:[ index _error: #rtErrArgOutOfRange args: { 0 . sz - 1 } ] .
amount _validateClass: Integer .
(amount < 0 or:[ amount > 9223372036854775807 ])
  ifTrue:[ amount _error: #rtErrArgOutOfRange args: { 0 . } ] .
self _primitiveFailed: #sharedCounter:incrementBy: args: { index . amount }

]

{ #category : 'Shared Counters' }
System class >> sharedCounter: n setValue: i [

"Set the shared counter at the specified index to ths specified value.
 Return the receiver.

 The argument n must a SmallInteger in the range 0 to
(System _numSharedCounters - 1) .

 The argument i must be a SmallInteger or LargeInteger in the range
 -2**63 (-9223372036854775808) to 2**63 - 1 (9223372036854775807).

 Shared counters allow multiple sessions on the same host to read and update
 a common counter value. Each counter is protected by a unique spinlock.
 Shared counters are transient, i.e. do not persist across cache restart.
 See persistentCounterAt:put:, etc. for accessing persistent shared counters,
 for which values do persist over shutdown and are recoverable from tranlogs.
 Shared counter values are recorded by statmonitor when using the -n option
 and recorded as AppStats.

 Shared counters are indexed from 0 to (System numSharedCounters - 1), which is
 set by the config parameter SHR_PAGE_CACHE_NUM_SHARED_COUNTERS, and defaults
 to 1900."

<primitive: 350>
| sz |
n _validateClass: SmallInteger.
i _validateClass: Integer .
sz := self _numSharedCounters .

(n < 0 or:[ n >= sz ])
  ifTrue:[n _error: #rtErrArgOutOfRange args:{ 0 . sz - 1 } ] .
(i < -9223372036854775808 or:[ i > 9223372036854775807 ])
   ifTrue:[i _error: #rtErrArgOutOfRange args:{ -9223372036854775808 . 9223372036854775807 } ] .
self _primitiveFailed: #sharedCounter:setValue: args: { n . i }

]

{ #category : 'Shared Counters' }
System class >> sharedCounterFetchValuesFrom: firstCounter to: lastCounter [

"Return an array containing the values from all shared counters starting
 with the counter at index <firstCounter>, up to and including the value
 from the counter <lastCounter>.  Note that shared counters are indexed
 starting at 0 and the result array indexes starting at 1.

 Both arguments must be in the range 0...(System numSharedCounters - 1), and
 lastCounter must be greater than or equal to firstCounter."

  ^self _sharedCounterFetchValuesFrom: firstCounter to: lastCounter

]

{ #category : 'Cache Statistics' }
System class >> sharedPageCacheMonitorCacheStatistics [

"Return the cache statistics for the SPC monitor process.

 Only cache statistics applicable to the SPC monitor process are returned.
 The description of these statistics can be determined by evaluating:
   'System cacheStatisticsDescriptionForMonitor'
 Per-process host statistics are not included, see hostStatisticsForProcess: <pid>."

^ self cacheStatisticsAt: 0  "monitor is always slot 0"

]

{ #category : 'System Control' }
System class >> shutDown [

"Aborts all current sessions, then terminates them.  Finally, the GemStone
 system is shut down.  The session issuing this message terminates with a
 broken connection.  Requires the SystemControl privilege.
 On success, blocks until the repository shutdown sequence has completed."

^ self _zeroArgPrim: 8

]

{ #category : 'Error Handling' }
System class >> signal: anInteger args: anArray signalDictionary: anErrorDict [

"Will be deprecated.
 This method generates the specified Exception, along with its
 associated arguments, and signals it.

 Gs64 v3.0  anErrorDict is ignored. "

^ (AbstractException _new: anInteger args: anArray ) signal

]

{ #category : 'Signals' }
System class >> signalAlmostOutOfMemoryThreshold: anInteger [

"Controls the generation of an error when session's temporary object memory
 is almost full .
 anInteger = -1, disable the generation of the error, resets threshold to 85% ,
		and suppress smalltalk stack printing before OOM.
 anInteger = 0,  enable generation of the error with previous threshold,
		default threshold after session login is 85% .
 0 < anInteger < 125 , enable generation of the error with specified threshold.

 The error is generated at the end of an in-memory markSweep,
 when the amount of memory used exceeds the specified threshold.
 If the session is executing a user action, or in index maintenance,
 the error is deferred and generated when execution returns to the
 bottom-level Smalltalk or GCI portion of the stack.

 This method or enableAlmostOutOfMemoryError must be invoked after
 each delivery of the AlmostOutOfMemory error to reenable generation
 of the error."

self _updateSignalErrorStatus: 5 toState: anInteger

]

{ #category : 'Transaction Control' }
System class >> signaledAbortErrorStatus [

"Returns true to indicate that the system generates an error when it receives
 the abort signal from Stone.  (In other words, verify that
 enableSignaledAbortError has been called to activate detection of the
 RT_ERR_SIGNAL_ABORT signal.)  Returns false otherwise.

 If in transaction, the result reflects what will happen to abort signals
 after the next commit or abort which exits the transaction.
 "

^self _signalErrorStatus: 2

]

{ #category : 'Transaction Control' }
System class >> signaledFinishTransactionErrorStatus [

"Returns true to indicate that the system generates an error when it receives
 the finishTranaction signal from Stone.  (In other words, verify that
 enableSignaledFinishTransactionError has been called ).  Returns false otherwise."

^self _signalErrorStatus: 4

]

{ #category : 'Notification' }
System class >> signaledGemStoneSessionErrorStatus [

"Returns true to indicate that the current GemStone session can receive signals
 from other GemStone sessions.  Returns false otherwise."

^self _signalErrorStatus: 3

]

{ #category : 'Deprecated' }
System class >> signalFromGemStoneSession [

"Returns a 4 element Array containing information about a signal from
 another GemStone session:

 1.  An instance of GsSession representing the session that sent the signal.
 2.  The signal value (a SmallInteger).
 3.  A signal message (a String).
 4.  A SmallInteger, number of additional signals pending in stone

 If there is no signal in the queue, returns an empty Array."

| result |
self deprecated: 'System class >> signalFromGemStoneSession: deprecated in v3.0; use InterSessionSignal'.
result := self _signalFromGemStoneSession .
result == nil ifTrue:[ ^ { } ].
result at: 1 put: (GsSession sessionWithSerialNumber: (result at: 1)).
^ result

]

{ #category : 'Notification' }
System class >> signalTranlogsFullStatus [

"Returns true to indicate that the session will get an error when
 stone detects a tranlogs full condition.  Returns false otherwise."

^self _signalErrorStatus: 6

]

{ #category : 'Session Control' }
System class >> sleep: aTime [

"Sleep for aTime seconds.  aTime must be a positive SmallInteger.
 If aTime is zero, this method has no effect.
 Returns the receiver.
 The ProcessorScheduler is not used, and other waiting GsProcesses
 will not execute during the sleep. Use a Delay or Semaphore to wait
 while allowing other GsProcesses to run."

| timeLeft |
aTime _isSmallInteger ifFalse:[ aTime _validateClass: SmallInteger ].
aTime < 0 ifTrue: [ aTime _error: #errArgTooSmall args:{ 0 } ] .
timeLeft := aTime .
[ true ] whileTrue:[
  timeLeft := self _sleep: timeLeft .
  timeLeft <= 0 ifTrue:[ ^ self ].
]

]

{ #category : 'Transaction Control' }
System class >> soloAbort [
  "For use in a Solo session that desires to abort changes objects that exist
   in the solo extent, such as abort changes to UserGlobals, etc.

   Signals an error if   GsCurrentSession isSolo == false .

   Has no effect if there no changes to objects in the solo extent.
   Returns the receiver.  
  "
  GsSession isSolo ifFalse:[ Error signal:'current session is not a solo session'].
  ^ self _primitiveAbort: true
]

{ #category : 'Debugging Support' }
System class >> stackDepth [

"Returns current depth of the GemStone Smalltalk stack."

^ self _zeroArgPrim: 26

]

{ #category : 'Debugging Support' }
System class >> stackDepthHighwater [

"Returns largest depth of the GemStone Smalltalk stack since session login.
 Result not accurate if running native code .
"

^ self _zeroArgPrim: 27

]

{ #category : 'Debugging Support' }
System class >> stackLimit [

"Returns the approximate limit on the depth of the GemStone Smalltalk stack.
 The stack size is determined by the configuration file parameter
 GEM_MAX_SMALLTALK_STACK_DEPTH."

^ self _zeroArgPrim: 1

]

{ #category : 'Debugging Support' }
System class >> stackLimit: anInteger [

"Has no effect in this release.  Provided for compatibility."

anInteger _validateClass: SmallInteger .
(anInteger < 0) ifTrue: [ anInteger _error: #rtErrArgNotPositive ].
^ self

]

{ #category : 'Deprecated' }
System class >> startAdminGcSession [

"Start the Admin GC session if it is not running.
 Returns true if the gem was started and false if
 it could not be started because it is already running
 or logins are suspended.
 Requires the GarbageCollection privilege."

self deprecated: 'System class >> startAdminGcSession deprecated v3.2.  Use startAdminGem instead'.
^self startAdminGem

]

{ #category : 'Garbage Collection Management' }
System class >> startAdminGem [

"Start the Admin GC session if it is not running.
 Returns true if the gem was started and false if
 it could not be started because it is already running.
 Requires the GarbageCollection privilege."

^self _zeroArgPrim: 67

]

{ #category : 'Garbage Collection Management' }
System class >> startAllGcGems [

"Start all enabled garbage collector gems that are not running.
 Does not start the Symbol Creation Gem.
 Requires the GarbageCollection privilege."

^ self _zeroArgPrim: 18

]

{ #category : 'Deprecated' }
System class >> startAllGcSessions [

"Start all enabled garbage collector gems that are not running.
 Does not start the Symbol Creation Gem.
 Requires the GarbageCollection privilege."

self deprecated: 'System class >> startAllGcSessions deprecated v3.2.  Used startAllGcGems instead'.
^ self startAllGcGems

]

{ #category : 'Deprecated' }
System class >> startAllReclaimGcSessions [

"Start all recalim GC sessions that are configured to run.
 Returns the number of reclaim sessions started.

 Requires the GarbageCollection privilege."

self deprecated: 'System class >> startAllReclaimGcSessions deprecated v3.2.  Use startReclaimGem instead'.
^self startReclaimGem

]

{ #category : 'Transaction Control' }
System class >> startCheckpointAsync [
"Starts an asynchronous checkpoint.  If a checkpoint is already
 in progress, this method will not start another.  Instead, it
 will return 'true', indicating a checkpoint is in progress.   If a
 checkpoint is not in progress, a new checkpoint is started and
 this method returns immediately.   To block until the checkpoint
 completes, use the startCheckpointSync method instead.

 Unlike the commitTransactionWithCheckpoint method, this method
 does not commit (or otherwise modify) the current transaction.

 Requires the SystemControl privilege.

 A result of true means a checkpoint was successfully started.
 A result of false means a checkpoint could not be started.  This
 usually happens because checkpoints are suspended or no tranlog
 space is available, or the repository is in restore-from-log state.
 It can also happen if more than one session
 calls this method at the same time."

 | status |
 status := self _zeroArgPrim: 63 .
 ^ status == 1"in progress" or:[ status == 2"started"].

]

{ #category : 'Transaction Control' }
System class >> startCheckpointSync [
"Starts a synchronous checkpoint.  If a checkpoint is already in progress,
 this method waits for it to complete, and then starts another checkpoint.
 The wait for the already running checkpoint to complete.
 can be interrupted by GciSoftBreak  (ctl-C in topaz).  Once this
 method starts its own checkpoint, the session is suspended in stone
 (waiting in a primitive method)  until stone completes the checkpoint.

 If successful, this method returns after starting and completing a checkpoint.
 Otherwise it signals an Error.
 To start a new checkpoint and return before it completes, use the
 startCheckpointAsync method instead of this one.

 This method does not commit (or otherwise modify) the current transaction.

 Requires the SystemControl privilege.

 A result of true means a checkpoint was successfully finished,
 otherwise an Error is signaled.  Failure to start a checkpoint can occur
 because checkpoints are suspended or tranlog space is full, or
 the repository is in restore-from-log mode.
"
 | printed |
 [ true ] whileTrue:[ | status |
   "Session waits in prim for stone to reply, if checkpoint was not in progress
    and we started one."
   status := self _zeroArgPrim: 62 .
   status == 0 ifTrue:[
     (self checkpointStatus at: 1) == true ifTrue:[
        Error signal:'startCheckpointSync failed, checkpoints are suspended'.
     ].
     Error signal:'startCheckpointSync failed'.
   ].
   status == 2 ifTrue:[ ^ true "we started and completed a checkpoint"].

   "status == 1 , a checkpoint already running but not one that we started"
   printed ifNil:[
     GsFile gciLogServer:'Waiting for preceding checkpoint to finish'.
     printed := true .
   ].
   Delay waitForSeconds: 5
 ].

]

{ #category : 'Garbage Collection Management' }
System class >> startMaxReclaimGemSessions [

"Starts the configured maximum number of reclaim threads."

| maxReclaim |

maxReclaim := System stoneConfigurationAt: #STN_MAX_GC_RECLAIM_SESSIONS.
maxReclaim == 0 ifTrue: [ maxReclaim := SystemRepository numberOfExtents ].

^self changeNumberOfReclaimGemSessions: maxReclaim.

]

{ #category : 'Host System Access' }
System class >> startNewGemLog: aFileName [
 "aFileName must be a file name (no '/' allowed) which will be appended to
  the result of System class >> gemLogPath to open a new log file.
  The existing log, the path for which is System class >> gemLogFileName prior to sending
  startNewGemLog:,  will be closed,  then stdout and stderr of this sessions' gem process
  will be redirected to the new log file. If aFileName cannot be created or written to, 
  the existing log file will continue to be used.

  Has no effect and signals a Warning in a topaz -l or other linked GCI application.
  Returns receiver if successful. "

 <primitive: 79>
 self clientIsRemote ifFalse:[
   ^ Warning signal:'System class>>startNewLog:  has no effect in a linked GCI application'.
 ].
 aFileName _validateClass: String.
 self _primitiveFailed: #startNewGemLog args: { aFileName }
]

{ #category : 'System Control' }
System class >> startNewLoginLog [

"Tells the stone to close the current login log file and open a new one.

 Requires the FileControl privilege.

 Returns true if the action was successful or false if the action could
 not be completed or if the login log file is not enabled."

^ self _zeroArgPrim: 173

]

{ #category : 'System Control' }
System class >> startNewObjectReadLog [
 "Requests stone close the current object read log file and start a new one.
 Blocks until the operation has completed. Returns true on success or false if
 the object read logging feature is disabled. Raises an exception on error.
 Requires the FileControl privilege."

  ^ self _zeroArgPrim: 191
]

{ #category : 'Garbage Collection Management' }
System class >> startReclaimGem [

"If a reclaimGem is running and the number of threads is greater than the value of
 the runtime config #StnNumGcReclaimSessions, then simply return the current number of threads.
 If the number of threads running is less than the config value then set it to the config.

 If a reclaimGem is not running it starts a reclaimGem using the value of the
 #StnNumGcReclaimSessions runtime configuration to determine the number of sessions to start.

 Returns the number of reclaim threads that are running.
"
| numReclaim configReclaimSessions|

numReclaim := self reclaimGemSessionCount.
configReclaimSessions := System stoneConfigurationAt: #StnNumGcReclaimSessions.
(numReclaim > 0) ifTrue: [
     (numReclaim > configReclaimSessions) ifTrue: [ ^ numReclaim ].
   ].
^ self changeNumberOfReclaimGemSessions: configReclaimSessions.

]

{ #category : 'Deprecated' }
System class >> startReclaimGemForExtentRange: startExtent to: endExtent [

"Deprecated - all extents are covered by a single reclaim gem"

self deprecated: 'System class>>startReclaimGemForExtentRange:to: deprecated in v3.2.'.
^self startReclaimGem

]

{ #category : 'Deprecated' }
System class >> startReclaimGemForExtentRange: startExtent to: endExtent onHost: aHostNameOrNil [

"Deprecated - all extents are covered by a single reclaim gem"

self deprecated: 'System class>>startReclaimGemForExtentRange:to:onHost: deprecated in v3.2.'.
^self startReclaimGem

]

{ #category : 'Deprecated' }
System class >> startReclaimGemForExtentRange: startExtent to: endExtent onHost: aHostNameOrNil stoneHost: stoneHostOrNil [

"Deprecated - all extents are covered by a single reclaim gem"

self deprecated: 'System class>>startReclaimGemForExtentRange:to:onHost:stoneHost: deprecated in v3.2.'.
^self startReclaimGem

]

{ #category : 'Deprecated' }
System class >> startSymbolCreationSession [

"Start the Symbol Creation session if it is not running.
 Returns true if the gem was started and false if
 it could not be started because it is already running
 or logins are suspended.

 Requires the GarbageCollection privilege."

self deprecated: 'System class >> startSymbolCreationSession deprecated v3.2.  Use startSymbolGem instead'.
^self startSymbolGem

]

{ #category : 'Garbage Collection Management' }
System class >> startSymbolGem [

"Start the Symbol Creation session if it is not running.
 Returns true if the gem was started and false if
 it could not be started because it is already running.

 Requires the GarbageCollection privilege."

^self _zeroArgPrim: 45

]

{ #category : 'Cache Statistics' }
System class >> stoneCacheStatistics [

"Return the cache statistics for the stone if this session is located on
 the same host as the stone process.

 Only cache statistics applicable to the Stone process are returned.
 The description of these statistics can be determined by evaluating:
   'System cacheStatisticsDescriptionForStone'
 Per-process host statistics are not included, see hostStatisticsForProcess: <pid>.

 nil is returned if the stone is not running the session's host."

^ self sessionIsOnStoneHost
    ifTrue:[ self cacheStatisticsAt: 1 ] "stone is always slot 1, if it's here at all"
    ifFalse:[ nil ]

]

{ #category : 'Cache Statistics' }
System class >> stoneCacheStatisticWithName: aString [

"Return the value of the cache stat with the given name, which must match an
 element of the Array returned by the #cacheStatisticsDescription method
 applicable to the stone process.  The UserTime and SysTime statistics cannot
 be accessed using this method.

 Returns nil if a statistic matching aString was not found for the stone.

 This method may be used on hosts remote from the stone process."

^ self _cacheStatWithName: aString opCode: 2

]

{ #category : 'Runtime Configuration Access' }
System class >> stoneCommandLineArgs [

"Returns a String representing the command line arguments used to start the stone,
 or nil if an error occurs."

^ self _zeroArgPrim: 1038

]

{ #category : 'Configuration File Access' }
System class >> stoneConfigurationAt: aName [

"Returns the value of the specified configuration file parameter from the
 repository monitor process (Stone).  Returns nil if that parameter is not
 applicable to the Stone."

| cfgId |
cfgId := ConfigurationParameterDict at: aName otherwise: nil .
cfgId == nil ifTrue:[ ^ nil ].
^ self _configurationAt: cfgId isStone: true kind: $C

]

{ #category : 'Configuration File Access' }
System class >> stoneConfigurationAt: aName put: aValue [

"Changes the value of the specified Stone configuration parameter.

 See comments in the method configurationAt:put: for complete documentation."

^ self configurationAt: aName put: aValue

]

{ #category : 'Configuration File Access' }
System class >> stoneConfigurationFileNames [
"Answer a SymbolDictionary containing two keys-value pairs:
  #GEMSTONE_SYS_CONF -> (String) Filename of the stone's system configuration file.
  #GEMSTONE_EXE_CONF -> (String) Filename of the stone's executable configuration file.

 If either file does not exist, the value an empty String."

|array result|
array := self stoneConfigurationAt: #StnConfigFileNames .
result := SymbolDictionary new.
result at: #GEMSTONE_SYS_CONF put: (array at: 1) ;
       at: #GEMSTONE_EXE_CONF put: (array at: 2) .
^ result

]

{ #category : 'Configuration File Access' }
System class >> stoneConfigurationReport [

"Returns a SymbolDictionary whose keys are the names of configuration file
 parameters, and whose values are the current settings of those parameters in
 the repository monitor process (Stone).  Parameters that are not applicable to
 Stone and those that are undefined are not include in the result."

^ self _configurationReport: true

]

{ #category : 'Runtime Configuration Access' }
System class >> stoneIsBigEndian [

" Returns true if the stone process is running
  on a machine using big endian byte ordering "

^ self _zeroArgPrim: 107


]

{ #category : 'Cache Statistics' }
System class >> stoneMessageKinds [

"Returns an array of Strings that lists the request kinds which can be sent
 from a session to the stone.  The numeric index of a message may be obtained
 from the 'MessagesToStone' (gem and page servers) or 'GsMsgKind' (stone)
 cache statistics.

 The result of this method may change with every new release of GemStone/64."

^ self _zeroArgPrim: 133

]

{ #category : 'Environment Access' }
System class >> stoneName [

"Returns a String whose value is the full network name of the
 Stone to which this session is logged in."

^ self _zeroArgPrim: 11

]

{ #category : 'Session Control' }
System class >> stoneStartupId [
"Returns an Integer, a 64bit random number created in stone startup."

^ self _zeroArgPrim: 149

]

{ #category : 'Version Management' }
System class >> stoneVersionAt: aString [

"Returns version information about the Stone (repository monitor) process.

 See System (C) | gemVersionAt: for further details.
 aString = 'imageKind' returns nil for the Stone."

| verId |

aString = 'imageKind' ifTrue:[ ^ nil "image not applicable to Stone"].

verId := VersionParameterDict at: aString otherwise: nil .
verId ifNil:[ ^ nil ].

^ self _configurationAt: verId isStone: true kind: $V

]

{ #category : 'Version Management' }
System class >> stoneVersionReport [

"Return a StringKeyValueDictionary whose keys are the names of operating system,
 hardware, or GemStone version attributes, and whose values are the
 current values of those attributes in the Gem process."

^ self _serverVersionReport: true

]

{ #category : 'Version Management' }
System class >> stoneVersionReportString [
  "Returns a String with keys and values of stoneVersionReport one pair per line."
^ self stoneVersionReport _reportString

]

{ #category : 'Deprecated' }
System class >> stopAdminGcSession [

"Stop the Admin GC session if it is running.
 Requires the GarbageCollection privilege."

self deprecated: 'System class >> stopAdminGcSession deprecated v3.2.  Use stopAdminGem instead'.
^self stopAdminGem

]

{ #category : 'Garbage Collection Management' }
System class >> stopAdminGem [

"Stop the Admin GC session if it is running.
 Requires the GarbageCollection privilege."

^self _zeroArgPrim: 42

]

{ #category : 'Garbage Collection Management' }
System class >> stopAllGcGems [

"Stop any and all Garbage Collection sessions.
 Does not stop the Symbol Creation Gem .
 Requires the GarbageCollection privilege."

^self _zeroArgPrim: 40

]

{ #category : 'Deprecated' }
System class >> stopAllGcSessions [

"Stop any and all Garbage Collection sessions.
 Does not stop the Symbol Creation Gem .
 Requires the GarbageCollection privilege."

self deprecated: 'System class >> stopAllGcSessions deprecated v3.2.  Used stopAllGcGems instead'.
^self stopAllGcGems

]

{ #category : 'Deprecated' }
System class >> stopAllReclaimGcSessions [

"Stop any and all reclaim GC sessions.
 Requires the GarbageCollection privilege."

self deprecated: 'System class >> stopAllReclaimGcSession deprecated v3.2.  Use stopReclaimGem instead'.
^self stopReclaimGem

]

{ #category : 'Session Control' }
System class >> stopAllSessionsReferencingOldestCr [

"Causes the stone to take the following actions:
 1 - send a stopSession message to all sessions referencing the oldest
     commit record (see System(C)>>stopSession:)
 2 - forces all sessions referencing the oldest commit record to immediately
     log out.
 3 - disposes all commit records eligible for disposal.

 Does not stop the current session, even if it references the
 oldest commit record.  Has no effect if there is only one commit
 record present.

 Does not force the immediate disposal of the oldest commit record.

 Requires the SystemControl privilege.

 Returns a SmallInteger indicating the number of commit records
 disposed."

^ self _zeroArgPrim: 162

]

{ #category : 'Deprecated' }
System class >> stopGcSession: aSessionId [

"Aborts the transaction of the specified session (a SmallInteger), then
 terminates that session.  Returns the receiver.  If the indicated
 session is not active, no operation is performed.

 Will stop Garbage Collector sessions but will not stop the
 Symbol Creation Gem session .

 To execute this method, you must have SystemControl and SessionAccess
 privileges."

| serialNum |

self deprecated: 'System class>>stopGcSession: is deprecated v3.3. Use stopAdminGem or stopReclaimGem.'.

aSessionId _validateClass: SmallInteger.
serialNum := GsSession serialOfSession: aSessionId .
self _stopSession: serialNum kind: 16r1 timeout: -1

]

{ #category : 'Session Control' }
System class >> stopHostAgentForHost: hostNameOrIp [
  "hostNameOrIp is a String.
   Returns a string describing action taken."
  | targetIp pairs haUp sess |
  targetIp := GsSignalingSocket getHostAddressByName: hostNameOrIp .
  pairs := self _currentSessionProfiles.
  haUp := AllUsers userWithId:'HostAgentUser'.
  sess := { } .
  1 to: pairs size by: 2 do:[:j |
    (pairs at: j+1 ) == haUp ifTrue:[ sess add: (pairs at: j)].
  ].
  sess do:[:sesId | | desc leafIp |
    desc := System descriptionOfSession: sesId .
    leafIp := desc at: 11 .
    leafIp = targetIp ifTrue:[
      self stopHostAgentSession: sesId .
      ^ 'stopped session ' , sesId asString .
    ]
  ].
  ^ 'hostagent not found'

]

{ #category : 'Session Control' }
System class >> stopHostAgents [

"Stops all HostAgent processes that are logged in to this stone.
 Each hostagent executes graceful shutdown.
 Signals an Error if unable to stop all hostagents."

^ self _stopHostAgents: 16r4"Break" timeOut: 20

]

{ #category : 'Session Control' }
System class >> stopHostAgentSession: sessionId [
  "Graceful shutdown of the HostAgent with the specified sessionId"

^ self _stopHostAgentSession: sessionId kind: 16r4"Break"

]

{ #category : 'Session Control' }
System class >> stopOtherSessions [
  "Stops user sessions other than the current session and reenables logins.
   Does not stop reclaim, admin nor symbol creation Gems."

  self stopUserSessions ; resumeLogins

]

{ #category : 'Garbage Collection Management' }
System class >> stopReclaimGem [

"Stop all reclaim GC sessions.
 Requires the GarbageCollection privilege."

^ self changeNumberOfReclaimGemSessions: 0

]

{ #category : 'Session Control' }
System class >> stopSession: aSessionId [

"Aborts the transaction of the specified session (a SmallInteger), then
 terminates that session.  Returns the receiver.  If the indicated
 session is not active, no operation is performed.

 Implemented by sending a stopSession out-of-band byte to the session's
 gem process.   If the session is not responsive to this request,
 it will eventually be killed per STN_GEM_TIMEOUT in stone's config file.

 Does not stop Garbage Collector or Symbol Creation Gem sessions.

 To execute this method, you must have SystemControl and SessionAccess
 privileges."

| serialNum |

aSessionId _validateClass: SmallInteger.
serialNum := GsSession serialOfSession: aSessionId .
self _stopSession: serialNum kind: 16r0 timeout: -1

]

{ #category : 'Deprecated' }
System class >> stopSymbolCreationSession [

"Stop the Symbol Creation session if it is running.

 This is an extreme action that should never be done in normal
 operation of a system.  Stopping the Symbol Creation session
 can prevent other sessions from being able to commit until
 after the Symbol Creation session restarts and after
 the other sessions logout and login .

 Only SystemUser may execute this method, otherwise an error is generated."

self deprecated: 'System class >> stopSymbolCreationSession deprecated v3.2.  Use stopSymbolGem instead'.
^self stopSymbolGem

]

{ #category : 'Session Control' }
System class >> stopSymbolCreationSession: aSessionId [

"Aborts the transaction of the specified session (a SmallInteger), then
 terminates that session.  Returns the receiver.  If the indicated
 session is not active, no operation is performed.

 Will stop any session including garbage Collector and
 Symbol Creation Gem sessions.

 Only SystemUser may execute this method, otherwise an error is generated."

| serialNum |

aSessionId _validateClass: SmallInteger.
serialNum := GsSession serialOfSession: aSessionId .
self _stopSession: serialNum kind: 16r3  timeout: -1

]

{ #category : 'Garbage Collection Management' }
System class >> stopSymbolGem [

"Stop the Symbol Creation session if it is running.

 This is an extreme action that should never be done in normal
 operation of a system.  Stopping the Symbol Creation session
 can prevent other sessions from being able to commit until
 after the Symbol Creation session restarts and after
 the other sessions logout and login .

 Only SystemUser may execute this method, otherwise an error is generated."

^self _zeroArgPrim: 44

]

{ #category : 'Session Control' }
System class >> stopUserSessions [

"Prevents any new sessions from being initiated; then, for each
 active user session other than the session of the user executing this
 method, aborts the transaction and terminates that session.

 This method does not stop Gc Gem sessions nor does it stop
 the Symbol Creation Gem .

 To reenable logins, send the message System resumeLogins.  Otherwise,
 logins are automatically reenabled when this session logs out.

 To execute this method, you must have explicit privilege for
 SessionAccess and SystemControl in your UserProfile."


"Performs a System | suspendLogins, then
 performs a System | stopSession."

| anArrayOfSessions mySession myPrivs |

myPrivs := self myUserProfile privileges.
((myPrivs includesValue: #SystemControl) and:
   [myPrivs includesValue: #SessionAccess])
   ifFalse: [ self _error: #rtErrNoPriv .  ^ self "do nothing" ].
self suspendLogins.  " Prevent anyone from logging in "

anArrayOfSessions := self currentSessions.  " All Session running "
mySession := self session.  " My Session number "

anArrayOfSessions do: " Loop all session and log then off if not me "
   [:aSessionNumber | (aSessionNumber == mySession )
                         ifFalse:[self stopSession: aSessionNumber].
   ].

]

{ #category : 'Session Control' }
System class >> stopZombieSession: aSessionId [

"Forces the stone to finish logout processing for the given session ID.
 Takes no action if the given session has not begun the logout process or
 if the session no longer exists or if the session is the current session.
 When logout processing is completed, references to a commit record by
 that session are released and resources consumed by the session are recycled.

 To execute this method, you must have SessionAccess and SystemControl
 privileges.

 WARNING: This method bypasses certain internal mechanisms which
          prevent the immediate termination of active sessions,
          and therefore should only be used if the session
          could not be stopped by any other mechanisms.

 Returns true if the action was successful and the stone acted on the
 specified session.  A message will be written to the stone log in this
 case.  Returns false if no action was taken.  No message is added to the
 stone log when false is returned."

^ self _cleanupZombieSession: aSessionId

]

{ #category : 'Online Backup Support' }
System class >> suspendCheckpointsForMinutes: anInt [
"Suspend all checkpoints for the given number of minutes or until the
 resumeCheckpoints method is executed, which ever occurs first.  anInt
 must be a positive SmallInteger.

 Requires the SystemControl privilege.

 It is safe to copy the repository extents for backup purposes while
 checkpoints are suspended.

 Checkpoint suspension is not supported in partial tranlog mode.  This
 method will always return false STN_TRAN_FULL_LOGGING is set to FALSE.

 Certain operations which require checkpoints are not permitted while
 checkpoints are suspended, such as full backups.

 Calling this method while checkpoints are already suspended has
 the effect of changing the duration of the suspension.

 If a checkpoint is in progress when this method is called, the call
 will block until the current checkpoint completes, at which time
 checkpoints will be suspended.  If any session has made this
 call and is waiting for the current checkpoint to complete, calls
 to this method by other sessions will fail.

 Returns true if checkpoints were successfully suspended.  Returns
 false if checkpoints could not be suspended because the repository
 is in partial log mode or is in restore from backup or log mode."

<primitive: 285>
anInt  _validateClass: SmallInteger.
(anInt < 0)
  ifTrue:[ anInt _error: #errArgTooSmall args:{ 0 } ] .
^self _primitiveFailed: #suspendCheckpointsForMinutes: args: { anInt }

]

{ #category : 'System Control' }
System class >> suspendLogins [

"Prevents any new sessions from being initiated.  That is, no new user is
 allowed to login.  However, users already active will be allowed to continue
 processing.

 To reenable logins, send the message System | resumeLogins.  (If you fail to do
 so, GemStone automatically reenables logins when the last user logs out.)

 Requires the SystemControl privilege."

self configurationAt: #StnLoginsSuspended put: 1

]

{ #category : 'Deprecated' }
System class >> symbolCreationSessionId [

"Return the session ID of the SymbolCreation session.  Returns
 zero if the SymbolCreation session is not running."

self deprecated: 'System class >> symbolCreationSessionId deprecated v3.2.  Use symbolGemSessionId instead'.
^self symbolGemSessionId

]

{ #category : 'Garbage Collection Management' }
System class >> symbolGemSessionId [

"Return the session ID of the SymbolCreation session.  Returns
 zero if the SymbolCreation session is not running."

^self _zeroArgPrim: 46

]

{ #category : 'Lock Status' }
System class >> systemLocks [
"
 This is an expensive method, it makes a call to stone for each element
 of the result of
    System currentSessions

 Returns a Dictionary describing all objects that are currently locked.  For
 each Association in the result Dictionary, the key is a SmallInteger (the
 session number of a GemStone session that holds locks) and the value is the
 three-element Array described in the sessionLocks method.  If no sessions hold
 any locks, the result Dictionary is empty.

 The Arrays in the result Dictionary contain by identity those objects that are
 visible to the current session.  For locks on objects that the
 current session cannot see , the Arrays contain uncommitted Strings
 using one of the forms
    uncommitted object with oop NNN
    object with oop NNN (no read auth)  ."

| anArray result |
anArray := self _systemLocksPrim  .
result := Dictionary new .
1 to: anArray size do:[ :j |
   result add: (anArray at:j)
].
^ result

]

{ #category : 'Lock Status' }
System class >> systemLocksDetailedReport [

"Returns a String describing all of the locked objects.
 This is an expensive method, it makes a call to stone for each element
 of the result of
    System currentSessions

 For each object the oop and the class are printed, example:
session 5(
  (1 writeLocks: 11074305(a SymbolDictionary)))
session 6(  (1 readLocks: 240641(a Repository))
  (2 writeLocks: 979969(a UserProfile) 11074561(a SymbolDictionary)))
"
| str kinds assocs |
kinds := #( 'readLocks' 'writeLocks' 'deferredUnlocks' ).
assocs := self _systemLocksPrim .
assocs size == 0 ifTrue:[ ^ 'no locks' ].
str := String new .
1 to: assocs size do:[:j | | assoc val |
  assoc := assocs at: j .
  str add: 'session '; add: assoc key asString ; add: $( .
  val := assoc value .
  1 to: val size do:[:n | | arr |
    arr := val at: n .
    arr size > 0 ifTrue:[
      n > 1 ifTrue:[ str lf ].
      str add: '  ('; add: arr size asString; add:' ';
          add: (kinds at: n); add: ':' .
      arr do:[:obj |
        (obj isCommitted not and:[ obj class == String
          and:[ obj includesString:'object with oop']]) ifTrue:[ 
          str add: ' ('; add: obj; add: $) . "descriptive string generated by primitive."
        ] ifFalse:[
          str add: ' '; add: obj asOop asString ; add: '(a '; add: obj class name .  str add: $)  .
        ]
      ].
      str add: $).
    ].
  ].
  str add: $) .
  j < assocs size ifTrue:[ str lf ].
].
^ str

]

{ #category : 'Lock Status' }
System class >> systemLocksQuick [
 "Returns an Array of 3 Arrays. The elements of the result are
    readLocks
    writeLocks
    deferredUnlocks  (will be unlocked after commit token is released)
 Details about which sessions hold the locks are not included.
 readLocks and writeLocks include locks on not-committed objects from
 the current session.
 "
<primitive: 1083>
^ self _primitiveFailed: #systemLocksQuick .

]

{ #category : 'Lock Status' }
System class >> systemLocksReport [

"Returns a String that summarizes all of the locked objects.
 details about which session holds the locks are not included.
 Locks on not-committed objects held by the current session are included.

 Example report:
  (1 readLocks: 207361(a SymbolDictionary))
  (1 writeLocks: 11079425(a SymbolDictionary))
  (2 deferredUnlocks: 207361(a SymbolDictionary) 11079425(a SymbolDictionary))
"

 | val kinds str |
 val := self systemLocksQuick .
 str := String new .
 kinds := #( 'readLocks' 'writeLocks' 'deferredUnlocks' ).
 1 to: val size do:[:n | | arr |
    arr := val at: n .
    arr size > 0 ifTrue:[
      n > 1 ifTrue:[ str lf ].
      str add: '  ('; add: arr size asString; add:' ';
          add: (kinds at: n); add: ':' .
      arr do:[:obj |
        (obj isCommitted not and:[ obj class == String
          and:[ obj includesString:'object with oop']]) ifTrue:[ 
          str add: ' ('; add: obj; add: $) . "descriptive string generated by primitive."
        ] ifFalse:[
          str add: ' '; add: obj asOop asString ; add: '(a '; add: obj class name .  str add: $)  .
        ]
      ].
      str add: $).
    ].
 ].
 ^ str

]

{ #category : 'User-Defined Actions' }
System class >> systemUserActionReport [

"Returns a SymbolDictionary that provides information about GemStone system user
 actions.  These are user actions that are automatically installed in every
 GemStone session to support classes such as GsFile and GsSocket.

 In the resulting SymbolDictionary, the keys are the symbolic names of the user
 actions, and the values are Booleans.  A values is true if the user action is
 linked with your application, and false if the user action is linked with the
 current GemStone session."

| anArray result |

anArray :=  self _zeroArgCateg2Prim: 9 . "get an Array of SymbolAssociations"
result := SymbolDictionary new .
1 to: anArray size do:[:j | result addAssociation: (anArray at: j ) ].
^ result

]

{ #category : 'Extreme Measures' }
System class >> terminateAllSessionsReferencingOldestCr [

"Causes the stone to take the following actions:
 1 - send a stopSession message to all sessions referencing the oldest
     commit record (see System(C)>>stopSession:)
 2 - forces all sessions referencing the oldest commit record to immediately
     log out.
 3 - forces all references to the oldest commit record to be released.
 4 - disposes all commit records eligible for disposal.

 Does not terminate the current session, even if it references the
 oldest commit record.  Has no effect if there is only one commit
 record present.

 WARNING: This method bypasses certain internal mechanims which
          prevent the immediate termination of active sessions,
          and therefore should only be used if the oldest CR session(s)
          could not be stopped by any other mechanisms.  The
          System(C)>>stopAllSessionsReferencingOldestCr method should be
          attempted before this one.  Usage of this method when one or more
          session referencing the oldest CR is active may cause cache coherency
          errors.

 Requires the SystemControl privilege.

 Returns a SmallInteger indicating the number of commit records disposed."

^ self _zeroArgPrim: 161

]

{ #category : 'Session Control' }
System class >> terminateSession: aSessionId timeout: aSeconds [

"Supports faster termination of the specified session than is
 possible with   stopSession:   .

 Sends a stopSession out-of-band byte to the specified session's gem
 process.
 The argument aSeconds is a SmallInteger. aSeconds >= 0 specifies number of
 seconds to wait after sending the stopSession out-of-band byte
 before killing specified session's gem process by sending
 it a SIGTERM.  aSeconds < 0 means move the session to after_logout state
 immediately, but don't kill .

 If the stone config file specifies non-default STN_GEM_TIMEOUT ,
 the actual timeout will be limited by    aSeconds min: STN_GEM_TIMEOUT*60  ,
 otherwise timeout is limited in stone by     aSeconds min: 300 .

 If the current session is not logged in as SystemUser,
 this method requires SystemControl and SessionAccess privileges .

 Will stop normal sessions and Garbage Collector Gem sessions.
 Does not prevent auto-restart of Garbage Collector Gem sessions
 that are stopped by this method.

 Will not stop the Symbol Creation Gem."

| serialNum |
aSessionId _validateClass: SmallInteger.
serialNum := GsSession serialOfSession: aSessionId .
self _stopSession: serialNum kind: 16r11 timeout: aSeconds.

]

{ #category : 'Session Control' }
System class >> terminateSymbolCreationSession: aSessionId timeout: aSeconds [

"Will stop normal sessions, Garbage Collector Gem sessions,
 and  Symbol Creation Gem .

 Only SystemUser may use this method.
 Does not prevent auto-restart of Symbol Creation Gem sessions
 stopped by this method.

 Otherwise same as terminateSession:timeout:"

| serialNum |
aSessionId _validateClass: SmallInteger.
serialNum := GsSession serialOfSession: aSessionId .
self _stopSession: serialNum kind: 16r13 timeout: aSeconds.

]

{ #category : 'Time' }
System class >> timeGmt [

"Returns a SmallInteger, the time since January 1, 1970, in seconds.
 The time is computed from the clock of the machine on which the session is
 running, using the offset from the clock on the Stone's (GemStone repository
 monitor process) machine which is cached in the session at login."

^ self _timeGmt: false

]

{ #category : 'Time' }
System class >> timeGmt2005 [

"Returns a SmallInteger, the time since January 1, 2005, in seconds.
 The time is computed from the clock of the machine on which the session is
 running, using the offset from the clock on the Stone's (GemStone repository
 monitor process) machine which is cached in the session at login."

^ self _timeGmt2005: false

]

{ #category : 'Deprecated' }
System class >> timeGmt95 [

"Obsolete - use timeGmt2005 instead of this method.

 Returns a SmallInteger, the time since January 1, 1995, in seconds.
 The time is computed from the clock of the machine on which the session is
 running, using the offset from the clock on the Stone's (GemStone repository
 monitor process) machine which is cached in the session at login."

self deprecated: 'System class>>timeGmt95 deprecated v3.0. Use #timeGmt2005 instead.'.

^ self _timeGmt95: false

]

{ #category : 'Time' }
System class >> timeNs [

"Returns a SmallInteger representing the current high-resolution
 real time in nanoseconds since some arbitrary time in the past.  The
 result is not correlated in any way with the time of day."

^ self _zeroArgPrim: 95

]

{ #category : 'Session Control' }
System class >> totalSessionsCount [

"Return the approximate number of sessions logged into the system.
 Avoids a call to stone."
^self stoneCacheStatisticWithName: 'TotalSessionsCount'

]

{ #category : 'Transaction Control' }
System class >> transactionConflicts [

"Returns a SymbolDictionary that contains an Association with key #commitResult
 and  value is one of the following Symbols:
   #readOnly #success #rcFailure #failure #dependencyFailure #retryFailure
   #commitDisallowed #retryLimitExceeded #symbolFailure #lockFailure  .

 The remaining Associations in the dictionary are used to report the conflicts
 found.  Each Association's key indicates the kind of conflict detected; its
 associated value is an Array of objects that are conflicting.
 If there are no conflicts for the transaction, the returned SymbolDictionary
 has no additional Associations.

 The conflict sets are cleared at the beginning of a commit or abort and
 therefore may be examined until the next commit, continue or abort.
 Conflict sets are empty if the value of commitResult was #readOnly or #success.

 The keys for the conflicts are as follows:

     Key                Conflicts
 Read-Write          StrongReadSet and WriteSetUnion conflicts.
 Write-Write         WriteSet and WriteSetUnion conflicts.
 Write-Dependency    WriteSet and DependencyChangeSetUnion conflicts.
 Write-WriteLock     WriteSet and WriteLockSet conflicts.
 Write-ReadLock      WriteSet and ReadLockSet conflicts.
 Rc-Retry-Failure    objects for which Rc replay failed.
 WriteWrite_minusRcReadSet  (WriteSet and WriteSetUnion conflicts) - RcReadSet)
 Synchronized-Commit Details on specific failures in synchronized commits
 RcReadSet           Objects in the RcReadSet
 Rc-Retry-Failure-Reason      Specific RC retry failures
 Rc-Retry-Failure-Description Details on specific RC retry failures.

 The Read-Write conflict set has already had RcReadSet subtracted from it.
 The Write-Write conflict set does not have RcReadSet subtracted .

 The WriteSet does not includes objects newly committed by this transaction.
 So a conflict between a lock and a newly committed object will not be a conflict.

 The Write-Dependency conflict set contains objects modified (including DependencyMap
 operations) in the current transaction that were either added to, removed from,
 or changed in the DependencyMap by another transaction. Objects in the
 Write-Dependency conflict set may be in the Write-Write conflict set.

 Note: If you create a reference to a conflict set, be sure to disconnect it before
 committing, to avoid creating a persistent reference to the objects."

 | arr res |

 arr := self _conflictsReport: true withDetails: false .

 res := SymbolDictionary new.
 res at: #commitResult put: (arr at: 1) .

 3 to: arr size by: 3 do:[:j |
   res at: (arr at: j) put: ( arr at: j + 1 "conflict set") .
 ].
 ^ res

]

{ #category : 'Transaction Control' }
System class >> transactionLevel [
  "Returns 0 if not in a transaction, or a SmallInteger > 0
   indicating the transaction level. > 1 means a nested transaction."
  ^ self _zeroArgPrim: 5

]

{ #category : 'Transaction Control' }
System class >> transactionMode [

"Returns the current transaction mode for the current GemStone session, either
 #autoBegin, #manualBegin or #transactionless.
 A solo session is always in autoBegin mode.

 The default is controlled by the stone configuration item
 STN_GEM_INITIAL_TRANSACTION_MODE ."

^ self _zeroArgPrim: 4

]

{ #category : 'Transaction Control' }
System class >> transactionMode: newMode [

"Sets a new transaction mode for the current GemStone session and exits the
 previous mode by aborting the current transaction.  Valid arguments are
 #autoBegin, #manualBegin and #transactionless.
 The mode transactionless is intended primarily for idle sessions. Users
 may scan database objects, but are at risk of obtaining inconsistent views.
 In a solo session, signals an error if newMode ~~ #autoBegin .
 In a solo session, has no effect if newMode == #autoBegin .
"

| coordinator |
^ (coordinator := self _commitCoordinator) == nil
  ifTrue: [ self _localTransactionMode: newMode ]
  ifFalse: [ coordinator transactionMode: newMode ].

]

{ #category : 'Transaction Control' }
System class >> trapAddToClosureList: anObject [

"If anObject is a non-committed object, signal an error
 when a subsequent attempt to commit adds anObject to the closure list
 for the commit.  The error will be not-trappable within Gemstone Smalltalk.
 The error information will include the parent dirty or not-committed
 object which caused anObject to be added to the closure list.
 After the error is returned to topaz or GBS, you can then save the
 stack and use Object>>findReferencesInMemory:
 to determine what paths might be causing anObject to become reachable
 from persistent state.

 If anObject == nil, then cancel a previous trap object.
 Returns receiver."

<primitive: 850>
((anObject _status: false) bitAnd: 2) ~~ 0 ifTrue:[
  ArgumentError signal:
   'arg to trapAddToClosureList: must be nil or an object that is neither committed nor in closure of a failed commit'
].
^self _primitiveFailed: #trapAddToClosureList: args: { anObject }

]

{ #category : 'Host System Access' }
System class >> unixUserIdExists: aString [

"Returns true or false indicating whether or not the UNIX user
 ID representing by aString exists on the host system."

<primitive: 820>
aString _validateClass: String .
self _primitiveFailed: #unixUserIdExists: args: { aString }

]

{ #category : 'User-Defined Actions' }
System class >> userAction: aSymbol [

"Invokes the user-defined action represented by aSymbol.  Generates an error if
 the user action is not installed in this session, or if it expects any
 arguments, or if the user has the NoUserAction privilege.

 A maximum of 47 user actions may be active at any one time on the current
 GemStone Smalltalk stack."

<primitive: 200>
^ self _primitiveFailed: #userAction: args: { aSymbol }

]

{ #category : 'User-Defined Actions' }
System class >> userAction: aSymbol with: anArg [

"Invokes the user-defined action represented by aSymbol, passing it the argument
 anArg.  Generates an error if the user action is not installed in this session,
 or if the number of arguments expected by the user action is not 1, or if the
 user has the NoUserAction privilege.

 A maximum of 47 user actions may be active at any one time on the current
 GemStone Smalltalk stack."

<primitive: 200>
^ self _primitiveFailed: #userAction:with: args: { aSymbol . anArg }

]

{ #category : 'User-Defined Actions' }
System class >> userAction: aSymbol with: firstArg with: secondArg [

"Invokes the user-defined action represented by aSymbol, passing it the
 arguments firstArg and secondArg.  Generates an error if the user action is not
 installed in this session, or if the number of arguments expected by the user
 action is not 2, or if the user has the NoUserAction privilege.

 A maximum of 47 user actions may be active at any one time on the current
 GemStone Smalltalk stack."

<primitive: 200>
^ self _primitiveFailed: #userAction:with:with:
       args: { aSymbol . firstArg . secondArg }

]

{ #category : 'User-Defined Actions' }
System class >> userAction: aSymbol with: firstArg with: secondArg with: thirdArg [

"Invokes the user-defined action represented by aSymbol, passing it the
 arguments firstArg, secondArg, and thirdArg.  Generates an error if the user
 action is not installed in this session, or if the number of arguments expected
 by the user action is not 3, or if the user has the NoUserAction privilege.

 A maximum of 47 user actions may be active at any one time on the current
 GemStone Smalltalk stack."

<primitive: 200>
^ self _primitiveFailed: #userAction:with:with:with:
       args: { aSymbol . firstArg . secondArg . thirdArg }

]

{ #category : 'User-Defined Actions' }
System class >> userAction: aSymbol with: firstArg with: secondArg with: thirdArg
  with: fourthArg [

"Invokes the user-defined action represented by aSymbol, passing it the
 arguments firstArg, secondArg, thirdArg, and fourthArg.  Generates an error
 if the user action is not installed in this session, or if the number of
 arguments expected by the user action is not 4, or if the user has the
 NoUserAction privilege."

<primitive: 200>
^ self _primitiveFailed: #userAction:with:with:with:with:
       args: { aSymbol . firstArg . secondArg . thirdArg . fourthArg }

]

{ #category : 'User-Defined Actions' }
System class >> userAction: aSymbol with: firstArg with: secondArg with: thirdArg
  with: fourthArg with: fifthArg [

"Invokes the user-defined action represented by aSymbol, passing it the
 arguments firstArg, secondArg, thirdArg, fourthArg, and fifthArg.  Generates
 an error if the user action is not installed in this session, or if the number
 of arguments expected by the user action is not 5, or if the user has the
 NoUserAction privilege."

<primitive: 200>
^ self _primitiveFailed: #userAction:with:with:with:with:with:
       args: { aSymbol . firstArg . secondArg .
               thirdArg . fourthArg . fifthArg}

]

{ #category : 'User-Defined Actions' }
System class >> userAction: aSymbol with: firstArg with: secondArg with: thirdArg
  with: fourthArg with: fifthArg with: sixthArg [

"Invokes the user-defined action represented by aSymbol, passing it the
 arguments firstArg, secondArg, thirdArg, fourthArg, fifthArg, and sixthArg.
 Generates an error if the user action is not installed in this session, or if
 the number of arguments expected by the user action is not 6, or if the user
 has the NoUserAction privilege."

<primitive: 200>
^ self _primitiveFailed: #userAction:with:with:with:with:with:with:
       args: { aSymbol . firstArg . secondArg . thirdArg .
               fourthArg . fifthArg . sixthArg}

]

{ #category : 'User-Defined Actions' }
System class >> userAction: aSymbol with: firstArg with: secondArg with: thirdArg
  with: fourthArg with: fifthArg with: sixthArg with: seventhArg [

"Invokes the user-defined action represented by aSymbol, passing it the
 arguments firstArg, secondArg, thirdArg, fourthArg, fifthArg, sixthArg, and
 seventhArg.  Generates an error if the user action is not installed in this
 session, or if the number of arguments expected by the user action is not 7,
 or if the user has the NoUserAction privilege."

<primitive: 200>
^ self _primitiveFailed: #userAction:with:with:with:with:with:with:with:
       args: { aSymbol . firstArg . secondArg . thirdArg .
               fourthArg . fifthArg . sixthArg . seventhArg }

]

{ #category : 'User-Defined Actions' }
System class >> userAction: aSymbol with: firstArg with: secondArg with: thirdArg
  with: fourthArg with: fifthArg with: sixthArg with: seventhArg
  with: eighthArg [

"Invokes the user-defined action represented by aSymbol, passing it the
 arguments firstArg, secondArg, thirdArg, fourthArg, fifthArg, sixthArg,
 seventhArg, and eighthArg.  Generates an error if the user action is not
 installed in this session, or if the number of arguments expected by the
 user action is not 8, or if the user has the NoUserAction privilege."

<primitive: 200>
^ self _primitiveFailed: #userAction:with:with:with:with:with:with:with:with:
       args: { aSymbol . firstArg . secondArg . thirdArg .
               fourthArg . fifthArg . sixthArg . seventhArg . eighthArg }

]

{ #category : 'User-Defined Actions' }
System class >> userAction: aSymbol withArgs: anArray [

"Invokes the user-defined action represented by aSymbol, passing it the elements
 of anArray as arguments.  Generates an error if the user action is not
 installed in this session, or if the number of arguments expected by the user
 action is not the same as the number of elements in anArray, or if the user has
 the NoUserAction privilege.

 A maximum of 47 user actions may be active at any one time on the current
 GemStone Smalltalk stack."

<primitive: 201>
^ self _primitiveFailed: #userAction:withArgs: args: { aSymbol . anArray }

]

{ #category : 'User-Defined Actions' }
System class >> userActionReport [

"Returns a SymbolDictionary that provides information about all user actions
 installed in this GemStone session.  In that SymbolDictionary, the keys are
 the symbolic names of the user actions, and the values are Booleans (true if
 the user action is linked with your application, false if the user action is
 linked with the current GemStone session)."

 | anArray result |
 anArray :=  self _zeroArgCateg2Prim: 7 . "get an Array of SymbolAssociations"
 result := SymbolDictionary new .
 1 to: anArray size do:[:j | result addAssociation: (anArray at: j ) ].
 ^ result


]

{ #category : 'Session Control' }
System class >> userProfileForSession: aSessionId [

"Returns the UserProfile attached to the specified session (a SmallInteger).
 If the indicated session is not active, returns nil.
 Returns nil if the session is a logsender or logreceiver connection to stone.

 Requires SessionAccess privilege if aSessionId is not the current session."

 ^ (self descriptionOfSession: aSessionId) at: 1 .
]

{ #category : 'Session Control' }
System class >> users [

"Returns a Set of UserProfiles for all users known to the system."

^ AllUsers

]

{ #category : 'Session Control' }
System class >> userSessionsCount [

"Return the approximate number of sessions logged into the system.
 Avoids a call to stone."
^ self stoneCacheStatisticWithName: 'UserSessionsCount'

]

{ #category : 'Host System Access' }
System class >> validatePasswordForUser: uid password: pw [

"Validate the password pw for the user id uid using the Pluggable Authentication
 Modules (PAM) interface.

 Both the uid and pw arguments must be instances of String.

 Returns true if the password was successfully authenticated or false if
 the password is incorrect. Raises an ArgumentTypeError error if the arguments
 are not correct.

 Requires the UserPassword privilege."

^ [ System _validatePasswordForUser: uid password: pw ]
  on: InternalError do: [:ex |
	(Delay forSeconds: 1) wait.
	ex return: false.
    ]

]

{ #category : 'Deprecated' }
System class >> validatePasswordForUser: uid password: pw isEncrypted: encrypted [

"Obsolete.  Use System class>>validatePasswordForUser:password: instead of this
 method."

self deprecated: 'System class>>validatePasswordForUser:password:isEncrypted
deprecated in v3.2.  Replace with:
System class>>validatePasswordForUser:password:'.

^ [
    System _validatePasswordForUser: uid password: pw
] on: InternalError do: [:ex |
	(Delay forSeconds: 1) wait.
	ex return: false.
].

]

{ #category : 'LDAP Support' }
System class >> validatePasswordUsingLdapServers: aUriArray baseDn: aBaseDn filterDn: aFilterDn userId: aUserId password: pw [

"Authenticates a password using one or more LDAP servers.
 See the comments in the method
   #validatePasswordUsingLdapServers:baseDn:filterDn:userId:password:bindDn:bindPassword:
 for more information."

^ self validatePasswordUsingLdapServers: aUriArray
       baseDn: aBaseDn
       filterDn: aFilterDn
       userId: aUserId
       password: pw
       bindDn: nil "anonymous bind for searches"
       bindPassword: nil "anonymous bind for searches"

]

{ #category : 'LDAP Support' }
System class >> validatePasswordUsingLdapServers: aUriArray baseDn: aBaseDn filterDn: aFilterDn userId: aUserId password: pw bindDn: bindDn bindPassword: bindPw [

"Use the LDAP server URI(s) specified in aUriArray to validate the password pw
 is valid for aUserId.

 aUriArray must be an array of strings containing LDAP URI addresses
 (for example: 'ldaps://foo.bar.com').

 In order to validate the password, the complete distinguish name (DN) for
 userId must be determined.  The DN can either be constructed from the baseDn
 pattern (explicit mode) or the DN may be resolved by searching the LDAP directory
 (search mode).  In explicit mode, baseDn must be a string that contains the string
 wildcard sequence '%s'.  GemStone will substitute '%s' with userId before
 doing the password validation.  aFilterDn must be nil in explicit mode.

 In search mode, baseDn is the search pattern used to resolve the DN and must
 NOT contain the string wildcard sequence '%s'.  In search mode, aFilterDn must
 be a string that contains the string wildcard sequence '%s'.  See the examples
 below.

 aUserId must be a string which represents the userId to be validated.
 pw must be a string which is the password for userId.

 bindDn and bindPw must both be either strings or nils.  If both are strings,
 they are used as credentials to bind to the LDAP server to perform the search in
 search mode.  If bindDn and bindPw are both nil, the search is attempted using an
 anonymous bind.  Not all LDAP servers are configured to support anonymous
 binds.  bindDn and bindPw are not used in explicit mode and should be set to nil.

 Returns true if pw is the correct password for userId. Otherwise returns
 false if the password is incorrect or an error occurred while communicating
 with the LDAP directory.

 Setting the variable GS_DEBUG_LDAP=7 in the gem's environment will cause LDAP debugging
 information to be printed to stdout. Setting the variable GS_DEBUG_LDAP_DIR in the gem's
 environment will cause LDAP debugging information to be written to a new file in that
 directory.

 Example 1: Explicit mode

 System validatePasswordUsingLdapServers: (Array with: 'ldaps://myldap.mydomain.com')
        baseDn: 'uid=%s,ou=Users,dc=mycompany,dc=com' filterDn: nil
        userId: 'MyUserId' password: 'swordfish' bindDn: nil bindPassword: nil


 Example 2: Search mode with anonymous bind

 System validatePasswordUsingLdapServers: (Array with: 'ldaps://myldap.mydomain.com')
        baseDn: 'ou=Users,dc=mycompany,dc=com' filterDn: '(uid=%s)'
        userId: 'MyUserId' password: 'swordfish' bindDn: nil bindPassword: nil


 Example 3: Search mode with authenticated bind

 System validatePasswordUsingLdapServers: (Array with: 'ldaps://myldap.mydomain.com')
        baseDn: 'ou=Users,dc=mycompany,dc=com' filterDn: '(uid=%s)'
        userId: 'MyUserId' password: 'swordfish' bindDn: 'LdapBindUser'
        bindPassword: 'LdapBindPassword'
"

<primitive: 859>
aUriArray == nil
  ifFalse:[ aUriArray _validateClass: Array ] .
aUriArray do:[:e| e _validateClass: String ].
aBaseDn _validateClass: String .
aFilterDn == nil
  ifFalse:[ aFilterDn _validateClass: String ] .
aUserId _validateClass: String .
pw _validateClass: String .

bindDn == nil
   ifTrue:[ bindPw _validateClass: UndefinedObject ]
  ifFalse:[ bindDn _validateClass: String ] .
bindPw == nil
   ifTrue:[ bindDn _validateClass: UndefinedObject ]
  ifFalse:[ bindPw _validateClass: String ] .

^ self _primitiveFailed:
       #validatePasswordUsingLdapServers:baseDn:filterDn:userId:password:bindDn:bindPassword
       args: { aUriArray . aBaseDn . aFilterDn . aUserId . pw . bindDn . bindPw }

]

{ #category : 'Garbage Collection Management' }
System class >> voteState [

"Returns an integer, the voteState of the Stone garbage
 collection voting state machine.
 The states are  0 IDLE, 1 HIDING_SYMBOLS, 2 VOTING,
	 3 DONE_VOTING, 4 IN_PDWSU_SWEEP, 5 PDWSU_DONE"

 ^ self _zeroArgPrim: 17

]

{ #category : 'Garbage Collection Management' }
System class >> voteStateString [

"Returns a String which is the name of the current value of
 the voteState of the Stone garbage collection voting state
 machine. "

 ^ self _zeroArgPrim: 19

]

{ #category : 'Garbage Collection Management' }
System class >> waitForAllGcGemsToStartForUpToSeconds: anInt [
"Wait for all GC gems on the system to start for up to anInt
 seconds.  Returns true if all GC gems started, false if they
 did not within the specified amount of time."

|count|

count := 0.
[self hasMissingGcGems] whileTrue:[
  self sleep: 1.
  count := count + 1.
  (count > anInt)
   ifTrue:[^false]. "timeout waiting for GC gems to start"
].
^true "success!"

]

{ #category : 'Setting Locks' }
System class >> waitForApplicationWriteLock: lockObject queue: lockIdx autoRelease: aBoolean [

"Waits for a write lock using the specified object and
 the Application lock queue specified by lockIdx .

 lockIdx must be a SmallInteger,  >= 1  and  <= 10

 Throws error LOCK_ERR_ALL_SYMBOLS (2331) if you attempt to lock AllSymbols.
 Throws error RT_ERR_OBJ_MUST_BE_COMMITTED (2405) if you attempt to lock a temporary object.

 returns a SmallInteger, one of
   1  granted
   2071  undefined lock (lockIdx out of range)
   2074  dirty (object written by other session since start of this transaction)
   2075  lockDenied (lock object is a special object)
   2418  deadlock
   2419  timeout  (per STN_OBJ_LOCK_TIMEOUT config parameter)

 For a given value of lockIdx, all calls must use a persistent lockObject
 which is identical during the life of a stone process .

 If result is 1 or 2074 and aBoolean == true,  the lockObject is added to both
 the CommitReleaseLocksSet and CommitOrAbortReleaseLocksSet
 hidden sets of this session.

 The result 2418 (deadlock)   is returned if
   the requesting session already holds a readLock on lockObject
 or if
  the session would have to wait for the lock on lockObject, but doing
  so would create a cycle in the graph of locks being waited for, i.e:

    the session already holds a read or write lock on some object X
        waitForApplicationWriteLock: X ...
    and this session's attempt at
       waitForApplicationWriteLock: lockObject
    would have to wait to get the lock on lockObject
"

lockIdx < 1 ifTrue:[ ^ 2071 "lockIdx out of range"].
^ self _lock: lockObject kind: lockIdx + 4 autoRelease: aBoolean

]

{ #category : 'Garbage Collection Management' }
System class >> waitForVoteStateIdle [

"Returns when the vote state is idle."

self waitForVoteStateIdleSecs: -1. "wait forever for vote state idle"
^ self

]

{ #category : 'Garbage Collection Management' }
System class >> waitForVoteStateIdleSecs: waitTime [
| timeLimit sleepCount voteState voteCount currSessVoting prevSessVoting |
  timeLimit := waitTime.
  timeLimit = -1 ifTrue: [ timeLimit := SmallInteger maximumValue ].

  self needsCommit
    ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

  self abortTransaction .

  voteState := System voteState.

  "Admin gem must be present if voteState is not 0"
  ((voteState ~~ 0) and:[System adminGemSessionId == 0])
    ifTrue:[self _error: #rtErrReclaimAllMissingGcGem args:
             {'missing admin or reclaim gem needed to get vote state idle'}].
  sleepCount := 0.
  [ voteState := System voteState .
    voteState ~~ 0
  ] whileTrue:[
    voteState == 2 ifTrue: [  "voting"
       currSessVoting := System stoneCacheStatisticWithName: 'WaitingForSessionToVote'.
       currSessVoting == prevSessVoting
       ifTrue: [
         self abortTransaction .
         sleepCount - voteCount > 20
	 ifTrue: [ GsFile gciLogServer:'WARNING -- session ', currSessVoting asString ,
		                     ' taking too long to vote']]
       ifFalse: [ prevSessVoting := currSessVoting. voteCount := sleepCount].
     ].
     System sleep: 1. System abortTransaction.
     sleepCount := sleepCount + 1.
     (sleepCount > timeLimit) ifTrue: [
       UserDefinedError new _number: 2318;
         reason: #fail ; details: 'Timeout of ' , timeLimit asString , 
                                  ' seconds expired while waiting for voteState idle' ; signal.
     ]
  ].
^ timeLimit - sleepCount

]

{ #category : 'Private Cache Warming' }
System class >> warmRemoteCache: aString [

"Used to warm an X509 remote cache.
Invoked by a topaz -r forked from a netldi started with -e and -S ,
see $GEMSTONE/sys/cachewarmertopaz.ini .
aString is the value of either GEM_CACHE_WARMER_ARGS or GEM_CACHE_WARMER_MID_CACHE_ARGS
from the config file specified by -e arg to startnetldi.

Reads pages into the local cache
"
<primitive: 1103>
aString _validateClass: String .
self _primitiveFailed: #warmRemoteCache: args: { aString }

]

{ #category : 'Private Cache Warming' }
System class >> warmX509LeafCache [
  "Executes in a topaz -r forked by a netldi -S -e with NETLDI_START_MIDCACHE=false."
  | cfg dict midHost cfgStr val |
  cfg := self clientEnvironmentVariable:'GS_NETLDI_WARMER_ARGS' .  "from putenv in netldi C code"
  cfg ifNotNil:[
    GsFile gciLogClient:'Using GS_NETLDI_WARMER_ARGS=''', cfg asString, '''' .
    dict := self parseWarmerConfig: cfg
  ] ifNil:[ Error signal:'missing GS_NETLDI_WARMER_ARGS environment variable' ].
  midHost := dict at: #midHost otherwise: nil .
  midHost ifNotNil:[ | status |
    (GsSocket hostIsLocalhost: midHost) ifTrue:[
      GsFile gciLogClient: '''-M ', midHost, ''' specifies localhost, warming direct from stone cache'.
    ] ifFalse:[
      status := self midLevelCacheConnect: midHost .
      status ~~ true ifTrue:[
        ^ 'Failed to connect to mid level cache on ', midHost, '; ',  status asString.
      ].
      GsFile gciLogClient: 'Successful connection to mid cache on ' , midHost .
    ]
  ].
  cfgStr := String new .
  (dict at: #d otherwise: nil) ifNotNil:[ cfgStr add: '-d ' ].
  (val := dict at: #n otherwise: nil) ifNotNil:[
    val < 1 ifTrue:[ val := 1 ].
    cfgStr add: '-n ' , val asString, ' '.
  ] ifNil:[ 
    cfgStr add: '-n 2 ' . "default 2 warmer threads"
  ].
  (val := dict at: #workingSetInterval otherwise: nil) ifNotNil:[
    val < 1 ifTrue:[ val := 1 ].
    cfgStr add: '-w ', val asString ,' ' .
  ].
  GsFile gciLogClient:'Using ', cfgStr .
  GsFile gciLogClient:'Detailed warming status is in ' , System gemLogFileName .
  ^ self warmRemoteCache: cfgStr .

]

{ #category : 'Setting Locks' }
System class >> writeLock: anObject [

"Analogous to System | readLock:.  However, this method requests and
 grants write locks."

| result |

result := self _lock: anObject kind: 3 autoRelease: false .
(result == 1) ifTrue:[ ^ self ].
self _lockError: result obj: anObject details: 'write lock'

]

{ #category : 'Setting Locks' }
System class >> writeLock: anObject ifDenied: denyBlock ifChanged: changeBlock [

"Analogous to System | readLock:ifDenied:ifChanged:.  However,
 this method requests and grants write locks."

| result |
result := self _lock: anObject kind: 3 autoRelease: false .
(result == 1) ifTrue:[ ^ self ].
^ self _lockEvaluateErr: result obj: anObject denied: denyBlock changed: changeBlock

]

{ #category : 'Setting Locks' }
System class >> writeLockAll: aCollection [

"Analogous to System | readLockAll:.  However, this method requests and
 grants write locks."

 | result str nErr nDirty err |
 result := self _lockAll: aCollection kind: 3 .
 (result == self ) ifTrue: [^self] .
 
 str := (nErr := (result at: 1) size) asString , ' locks denied, ', 
      (nDirty := (result at: 2) size) asString , ' dirty locks ' .
 err := LockError new .
 err _number: ( nErr == 0 ifTrue:[ 2074 ] ifFalse:[ 2073 ]).
 ^ err args: result ; signal: str .
]

{ #category : 'Setting Locks' }
System class >> writeLockAll: aCollection ifIncomplete: incompleteBlock [

"Analogous to System | readLockAll:ifIncomplete:.  However, this method
 requests and grants write locks."

| result |
result := self _lockAll: aCollection kind: 3 .
(result == System)
  ifTrue: [^self]
  ifFalse: " else we execute the incomplete block with 3 arguments "
          [^incompleteBlock value: (result at: 1)
                            value: (result at:2)
                            value: (result at:3)].

]

{ #category : 'Private' }
System class >> _debugIndexingCode [
  "Returns a Boolean the state of FLG_DEBUG_INDEXING_CODE from C compile time of the VM."
  ^self _zeroArgPrim: 205
]

{ #category : 'Private' }
System class >> _keyFilePermissions [
  "Returns a SmallInteger, the value of rwks->keyfilePermissions received from stone at login"
  ^ self _zeroArgPrim: 206
]

{ #category : 'Transient Session State' }
System class >> progressPrintInterval: seconds [
"Sets the progress print interval for long operations such as backup, restore,
 markForCollection, etc. A value of 0 disables printing. Negative values and
 objects other than SmallIntegers are silently ignored."

^ self __sessionStateAt: 27 put: seconds

]

{ #category : 'Transient Session State' }
System class >> progressPrintInterval [
"Answers the progress print interval for long operations such as backup, restore,
 markForCollection, etc. A value of 0 means progress will not be printed."

^ (self __sessionStateAt: 27) ifNil:[ 180 ] ifNotNil:[:secs| secs ]

]

{ #category : 'Private' }
System class >> _configurationReportString: aDictionary [
  | sc rpt |
  sc := SortedCollection withAll: aDictionary keys .
  rpt := String new .
  sc do:[:aKey |
    rpt add: aKey; add:'  '; add: (aDictionary at: aKey) printString ; lf
  ].
  ^ rpt
]

{ #category : 'Configuration File Access' }
System class >> gemConfigurationReportString [
 "Returns a String representation of  gemConfigurationReport"
  ^ self _configurationReportString: self gemConfigurationReport .
]

{ #category : 'Configuration File Access' }
System class >> stoneConfigurationReportString [
 "Returns a String representation of  stoneConfigurationReport"
  ^ self _configurationReportString: self stoneConfigurationReport .
]

{ #category : 'Private' }
System class >> _writeWriteConflictsIsEmpty [
  "Returns a Boolean reflecting state of omPtr->workspace()->writeWriteConflicts"
  ^ self _zeroArgPrim: 207
]

{ #category : 'Private' }
System class >> printRcReadSet [

 | printObjsBlk ary |
 printObjsBlk := [:objs :isRcDetail | | limit str cnt sz |
  limit := (sz := objs size) min: 50 .
  str := ' ' , sz asString,' oops (' .
  cnt := 0 .
  1 to: limit do:[ :n | | each clsName oopStr didObj |
    clsName := ' ' .
    oopStr := '<authErr>' .
    [ each := objs at: n .
      oopStr := each asOop asString .
      clsName := 'a ' , each class name  .
    ] onException: SecurityError do:[:ex |
      (ex gsArguments atOrNil: 1) ifNotNil:[:oop |
         oopStr := oop asString .  clsName := 'authErr on class'.
      ].
    ].
    str add:' ' ; add: oopStr ; add: $( .
    (isRcDetail and:[ each ~~ nil ]) ifTrue:[
      each _isSymbol ifTrue:[ str add: each printString . didObj := true ] ifFalse:[
      each _isOneByteString ifTrue:[ str add: (each copyFrom: 1 to: (each size min:100)).
                                    didObj := true ]].
    ].
    didObj ifNil:[ str add: clsName ].
    str add: $) .
    cnt := cnt + 1 .
    (cnt >= 5 and:[ n < limit]) ifTrue:[ str lf . cnt := 0 ].
  ].
  limit < sz ifTrue:[ str add: ' ... '].
  str add: $) ;  lf .
  ].
 ary := (GsBitmap newForHiddenSet: #RcReadSet) asArray.
 GsFile gciLogServer: 'RcReadSet ' ; gciLogServer: (printObjsBlk value: ary value: true ).
]

