Extension { #name : 'UserProfile' }

{ #category : 'External Authentication - Private' }
UserProfile class >> _initPrivilegeNames [
  classVars
    at: #PrivilegeNames
        put:
           #(#SystemControl #ObsoleteStatistics #SessionAccess #UserPassword
             #DefaultObjectSecurityPolicy #PrivUnused5 #OtherPassword
             #ObjectSecurityPolicyCreation #ObjectSecurityPolicyProtection #FileControl
             #GarbageCollection #CodeModification
             #NoPerformOnServer #NoUserAction
             #NoGsFileOnServer #NoGsFileOnClient
             #SessionPriority #CompilePrimitives #ChangeUserId
             #MigrateObjects #DisableObjectReadLogging  
             #DynamicDisableObjectReadLogging #ReadOtherUserProfile
	     #CreateOnetimePassword ) ;
    at: #InversePrivileges put:
        #( NoPerformOnServer NoUserAction NoGsFileOnServer NoGsFileOnClient ).

]

{ #category : 'Private' }
UserProfile class >> _newWithUserId: userIdString
password: passwordArg
defaultObjectSecurityPolicy: anObjectSecurityPolicy
privileges: anArrayOfStrings
inGroups: groupStringsOrObjs [

"Creates a new UserProfile with the associated characteristics.  In so doing,
 creates a symbol list with three dictionaries: UserGlobals, Globals, and
 Published.  The first Dictionary (UserGlobals) is created for the user's
 private symbols.  Returns the new UserProfile.

 Adds the new UserProfile to AllUsers.  Creates the new UserProfile as a member
 of the group Subscribers and of the groups in groupStringsOrObjs.

 groupStringsOrObjs must be a collection of either Strings that represent group names,
 or UserProfileGroup objects.

 UserIdString must be a single-byte string with only alphanumeric characters or  
 underscore, period, dash or space.  Generates an error if passwordString is 
 equivalent to userIdString ignoring case (equalsNoCase:).

 anObjectSecurityPolicy must be nil or a committed GsObjectSecurityPolicy.
"

| result newDict packagePolicy newSymList myAssociations mySymbols
  userGroups newPrivSet publ |

self _validatePrivilege ifFalse:[ ^ nil ].
(self _validateUserId: userIdString ) ifFalse:[ ^ nil ].
passwordArg ~~ 509 ifTrue:[
  self _validate__password: passwordArg forUserId: userIdString .
].
anObjectSecurityPolicy == nil ifFalse:[
  (anObjectSecurityPolicy _validateClass: GsObjectSecurityPolicy ) ifFalse:[ ^ nil ].
  anObjectSecurityPolicy isCommitted ifFalse:[
    anObjectSecurityPolicy _error: #rtErrObjMustBeCommitted .
    ^ nil
  ].
].
newPrivSet := UserProfile _privilegesSetFromArray: anArrayOfStrings .

(groupStringsOrObjs _validateClass: Collection) ifFalse:[ ^ nil ].
userGroups := { }  .
groupStringsOrObjs do: [ :aStrOrGroup | | groupObj |
  (aStrOrGroup isKindOf: CharacterCollection)
     ifTrue:[ groupObj := UserProfileGroup _validateGroupString: aStrOrGroup ]
    ifFalse:[ aStrOrGroup _validateClass: UserProfileGroup.
              groupObj := aStrOrGroup ] .
  userGroups add: groupObj .
  ].
"by default all users are a member of Subscribers."
userGroups add: (UserProfileGroup groupWithName: 'Subscribers').

    "Create the new UserProfile and populate it using the given arguments."

GsObjectSecurityPolicy setCurrent: (Globals at: #AllUsers) objectSecurityPolicy  while:[
  | theUserId |
  result := super new _initialize .
  theUserId := userIdString asString copy immediateInvariant .
  result _userId: theUserId .
  result privileges: anArrayOfStrings.
].
result defaultObjectSecurityPolicy: anObjectSecurityPolicy .

   "Create UserGlobals Dictionary "
newDict := SymbolDictionary new.
newDict objectSecurityPolicy: anObjectSecurityPolicy .

   "Create GsPackagePolicy current (fix 41433)"
packagePolicy := GsPackagePolicy new.
packagePolicy objectSecurityPolicy: anObjectSecurityPolicy.

   "Create all SymbolAssociations and Symbols and UserGlobals."

myAssociations := { } .
mySymbols := { } .
myAssociations add: (SymbolAssociation newWithKey: #UserGlobals
                                       value: newDict).
myAssociations add: (SymbolAssociation newWithKey: GsPackagePolicy globalName
                                       value: packagePolicy).

myAssociations do:[:assoc|
  assoc objectSecurityPolicy: anObjectSecurityPolicy .
  newDict add: assoc
].

"Create the new user's symbol list."

newSymList := SymbolList new .
newSymList add: newDict; add: Globals .
"fix 40939"
publ := (AllUsers userWithId: 'DataCurator') symbolList objectNamed: #Published.
publ ifNotNil: [newSymList add: publ].
result symbolList: newSymList.
newSymList objectSecurityPolicy: anObjectSecurityPolicy.

GsObjectSecurityPolicy setCurrent: (Globals at: #AllUsers) objectSecurityPolicy  while:[
  AllUsers add: result .  "add to AllUsers after all of above succeeded"
    " password must be assigned after adding to AllUsers"
  [
    passwordArg _isOneByteString ifTrue:[ result password: passwordArg ]
      ifFalse:[ passwordArg == 509 ifTrue:[ result enableX509Authentication]
                     ifFalse:[ ArgumentError signal:'invalid password argument']].

    userGroups do: [:aGrp | result addToUserProfileGroup: aGrp ]
  ] on: Error do:[:ex |
    AllUsers removeAndCleanup: result .
    ex pass .
    ^ nil
  ]
].
^ result

]

{ #category : 'Private' }
UserProfile class >> _privilegeMask: aPrivilegeString [

| index privSet privSymbol |
privSet := UserProfile _privilegesSetFromArray: { aPrivilegeString }.
privSymbol := privSet _at: 1 .
(index := PrivilegeNames indexOf: privSymbol ) == 0 ifTrue:[
  aPrivilegeString _error: #rtErrBadPriv .
  ^ self
].
^ 1 bitShift: index - 1

]

{ #category : 'Private' }
UserProfile class >> _privilegesSetFromArray: anArrayOfStrings [

"Convert an Array of Strings to an IdentitySet of legal Privilege names.
 Raises an error if anArrayOfStrings is not an Array, or if any String
 within anArrayOfStrings cannot be translated to a canonical Symbol
 of a legal privilege name, otherwise returns an IdentitySet of Symbols."

| result |
(anArrayOfStrings _validateClass: Array) ifFalse:[ ^ nil ].
result := IdentitySet new .
anArrayOfStrings do:[ :aString | | aSymbol |
  aSymbol :=  Symbol _existingWithAll: aString .
  aSymbol ifNil:[ aString _error: #rtErrBadPriv ].
  "translate Segment for compatibility."
  aSymbol == #DefaultSegment ifTrue:[ aSymbol := #DefaultObjectSecurityPolicy ] .
  aSymbol == #SegmentCreation ifTrue:[ aSymbol := #ObjectSecurityPolicyCreation ] .
  aSymbol == #SegmentProtection ifTrue:[ aSymbol := #ObjectSecurityPolicyProtection ] .
  (PrivilegeNames includesIdentical: aSymbol) ifFalse:[
    aString _error: #rtErrBadPriv .
    ^ self
  ].
  result add: aSymbol
].
^ result

]

{ #category : 'Private' }
UserProfile class >> _validate__password: aString forUserId: aUserIdString [

  (aString _validateClass: String) ifFalse:[ ^ false ] .

  "disallow aUserIdString as a password"
  (aString equalsNoCase: aUserIdString) ifTrue:[
    aString _error: #rtDisallowedPassword . ^ false
  ].
  aString size > 1024 ifTrue:[
   aString _error: #rtMaxPasswordSize args: { 1024 } . ^ false
  ].
  "disallow empty string as a password"
  (aString equalsNoCase: '') ifTrue:[
    aString _error: #rtMinPasswordSize args: { 1 } . ^ false
  ].

  ^ true

]

{ #category : 'External Authentication - Private' }
UserProfile class >> _validateAuthenticationScheme: aSymbol [

^ (self authenticationSchemeSymbolToId: aSymbol ) ~~ nil

]

{ #category : 'Private' }
UserProfile class >> _validatePrivilege [

" You must have the #OtherPassword to modify UserProfiles "

  ^ System myUserProfile _validatePrivilegeName: #OtherPassword

]

{ #category : 'Private' }
UserProfile class >> _validateUserId: aString [
  | sz specialChars | 
  aString _validateKindOfClass: String .
  aString _isOneByteString ifFalse:[ 
     ArgumentError signal:'userId must be a kind of String with codePoints <= 255'.
     ^ false .
  ].
  (sz := aString size) == 0 ifTrue:[ 
    ArgumentError signal:'userId may not be an empty String' . ^ false 
  ].
  sz > 1024 ifTrue:[ ArgumentError signal:'userId exceeds 1024 characters'. ^ false ].
  specialChars := #( $_ $. $- $  ) .
  1 to: sz do:[:n | | ch | ch := aString at: n .
      (ch isAlphaNumeric or:[ specialChars includesIdentical: ch] ) ifFalse:[
      ArgumentError signal:'userId contains invalid Character(s)'.
      ^ false .
    ].   
  ].
  ^ true 
]

{ #category : 'External Authentication - Private' }
UserProfile class >> authenticationSchemeIdToSymbol: aSmallInt [

aSmallInt == 0 ifTrue:[ ^ #GemStone ] .
aSmallInt == 1 ifTrue:[ ^ #UNIX ] .
aSmallInt == 2 ifTrue:[ ^ #LDAP ] .
aSmallInt == 3 ifTrue:[ ^ #SingleSignOn ] .
aSmallInt == 4 ifTrue:[ ^ #X509  ] .
aSmallInt _validateClass: SmallInteger .
aSmallInt _error: #rtErrInvalidAuthSchemeId .
^ nil

]

{ #category : 'External Authentication - Private' }
UserProfile class >> authenticationSchemeSymbolToId: aSymbol [

aSymbol == #GemStone  ifTrue:[ ^ 0 ] .
aSymbol == #UNIX      ifTrue:[ ^ 1 ] .
aSymbol == #LDAP      ifTrue:[ ^ 2 ] .
aSymbol == #SingleSignOn ifTrue:[ ^ 3 ] .
aSymbol == #X509      ifTrue:[ ^ 4 ] .
aSymbol _validateClass: Symbol .
aSymbol _error: #rtErrInvalidAuthSchemeSymbol .
^ nil

]

{ #category : 'Private' }
UserProfile class >> isSpecialUserId: aUserId [

 aUserId = 'SystemUser'  ifTrue:[ ^ true ].
 aUserId = 'SymbolUser'  ifTrue:[ ^ true ].
 aUserId = 'GcUser'      ifTrue:[ ^ true ].
 aUserId = 'DataCurator' ifTrue:[ ^ true ].
 aUserId = 'Nameless'    ifTrue:[ ^ true ].
 aUserId = 'HostAgentUser'    ifTrue:[ ^ true ].
 "For 3.6,  CodeLibrarianUser no longer special."
 ^ false

]

{ #category : 'Instance Creation' }
UserProfile class >> new [

"Disallowed.  To create a new UserProfile, use newWithUserId:... instead."

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
UserProfile class >> newWithUserId: userIdString password: passwordString [

"Creates a new UserProfile with the given password and no privileges.
 In so doing, creates a symbol list with three dictionaries: UserGlobals,
 Globals, and Published.  The first Dictionary (UserGlobals) is created
 for the user's private symbols.  Returns the new UserProfile.

 Adds the new UserProfile to AllUsers. Creates the new UserProfile as a member
 of the group Subscribers.

 UserIdString must be a single-byte string with only alphanumeric characters or  
 underscore, period, dash or space. Generates an error if passwordString is 
 equivalent to userIdString ignoring case (equalsNoCase:).

 The new UserProfile will have the default object security policy, which is nil.
"

^ self newWithUserId: userIdString password: passwordString
  defaultObjectSecurityPolicy: nil privileges: #() inGroups: #()

]

{ #category : 'Instance Creation' }
UserProfile class >> newWithUserId: userIdString password: passwordString defaultObjectSecurityPolicy: aPolicy [

"Creates a new UserProfile with the given password and no privileges.
 In so doing, creates a symbol list with three dictionaries: UserGlobals,
 Globals, and Published.  The first Dictionary (UserGlobals) is created
 for the user's private symbols.  Returns the new UserProfile.

 Adds the new UserProfile to AllUsers.  Creates the new UserProfile as a member
 of the group Subscribers and of the groups in aCollectionOfGroupStrings.

 UserIdString must be a single-byte string with only alphanumeric characters or  
 underscore, period, dash or space.  Generates an error if passwordString is 
 equivalent to userIdString ignoring case (equalsNoCase:).

 aPolicy must be nil or a committed instance of GsObjectSecurityPolicy.
"

^ self newWithUserId: userIdString password: passwordString
  defaultObjectSecurityPolicy: aPolicy privileges: #() inGroups: #()

]

{ #category : 'Instance Creation' }
UserProfile class >> newWithUserId: userIdString
password: passwordString
defaultObjectSecurityPolicy: anObjectSecurityPolicy
privileges: anArrayOfStrings
inGroups: groupStringsOrObjs [

"Creates a new UserProfile with the associated characteristics.  In so doing,
 creates a symbol list with three dictionaries: UserGlobals, Globals, and
 Published.  The first Dictionary (UserGlobals) is created for the user's
 private symbols.  Returns the new UserProfile.

 Adds the new UserProfile to AllUsers.  Creates the new UserProfile as a member
 of the group Subscribers and of the groups in groupStringsOrObjs.

 groupStringsOrObjs must be a collection of either Strings that represent group names,
 or UserProfileGroup objects.

 UserIdString must be a single-byte string with only alphanumeric characters or  
 underscore, period, dash or space.  Generates an error if passwordString is 
 equivalent to userIdString ignoring case (equalsNoCase:).

 anObjectSecurityPolicy must be nil or a committed GsObjectSecurityPolicy.
"

self _validatePrivilege ifFalse:[ ^ nil ].
self _validate__password: passwordString forUserId: userIdString .
^ self _newWithUserId: userIdString
  password: passwordString
  defaultObjectSecurityPolicy: anObjectSecurityPolicy
  privileges: anArrayOfStrings
  inGroups: groupStringsOrObjs

]

{ #category : 'Deprecated' }
UserProfile class >> newWithUserId: userIdString
password: passwordString
defaultSegment: anObjectSecurityPolicy
privileges: anArrayOfStrings
inGroups: aCollectionOfGroupStrings [

"Deprecated - use methods in the Instance Creation category instead"

self deprecated: 'UserProfile class>>newWithUserId:password:defaultSegment:privileges:inGroups: deprecated v3.0.
Use #newWithUserId:password:defaultObjectSecurityPolicy:privileges:inGroups: instead.'.

^self
	newWithUserId: userIdString
	password: passwordString
	defaultObjectSecurityPolicy: anObjectSecurityPolicy
	privileges: anArrayOfStrings
	inGroups: aCollectionOfGroupStrings

]

{ #category : 'Instance Creation' }
UserProfile class >> newWithUserId: userIdString password: passwordString privileges: anArrayOfStrings inGroups: aCollectionOfGroupStrings [

"Creates a new UserProfile with the given password and the given privileges.
 In so doing, creates a symbol list with three dictionaries: UserGlobals,
 Globals, and Published.  The first Dictionary (UserGlobals) is created
 for the user's private symbols.  Returns the new UserProfile.

 Adds the new UserProfile to AllUsers.  Creates the new UserProfile as a member
 of the group Subscribers and of the groups in aCollectionOfGroupStrings.

 UserIdString must be a single-byte string with only alphanumeric characters or  
 underscore, period, dash or space.  Generates an error if passwordString is 
 equivalent to userIdString ignoring case (equalsNoCase:).

 The new UserProfile will have the default object security policy, which is nil.
"

^ self newWithUserId: userIdString password: passwordString
  defaultObjectSecurityPolicy: nil privileges: anArrayOfStrings inGroups: aCollectionOfGroupStrings

]

{ #category : 'Instance Creation' }
UserProfile class >> newX509WithUserId: userIdString [
"Creates a new UserProfile for which authenticationSchemeIsX509 will return true.
 The new UserProfile has no privileges, and has a symbol list with three dictionaries: 
 UserGlobals, Globals, and Published.  The first Dictionary (UserGlobals) is created
 for the user's private symbols.  Returns the new UserProfile.

 Adds the new UserProfile to AllUsers.  Creates the new UserProfile as a member
 of the group Subscribers.

 The new UserProfile will have the default object security policy, which is nil.
"

^ self _newWithUserId: userIdString password: 509
  defaultObjectSecurityPolicy: nil privileges: #() inGroups: #()

]

{ #category : 'Accessing' }
UserProfile class >> privilegeNames [
  ^ PrivilegeNames

]

{ #category : 'Updating Privileges' }
UserProfile >> _addPrivilege: aPrivilegeString [

< protected >
| index privSet privSymbol |
privSet := UserProfile _privilegesSetFromArray: { aPrivilegeString }.
privSymbol := privSet _at: 1 .
(index := PrivilegeNames indexOf: privSymbol ) == 0 ifTrue:[
  aPrivilegeString _error: #rtErrBadPriv .
  ^ self
].
(self class isSpecialUserId: userId) ifTrue:[
  System iAmSystemUser
    ifFalse:[ ^ self _error: #rtErrMustBeSystemUser ] .
].
privileges := privileges bitOr:( 1 bitShift: index - 1) .

]

{ #category : 'Private' }
UserProfile >> _changePasswordFrom: firstString to: secondString [

"Private.  Change the given password of the receiver, which is the UserProfile
 of the current session (firstString), to the new given password
 (secondString)."

"Tests AllUsers.disallowUsedPasswords, and if ok, saves firstString
  in encrypted form in receiver's UserSecurityData.oldPasswords. "

<protected primitive: 308>
firstString _validateByteClass: CharacterCollection.
secondString _validateByteClass: CharacterCollection.
self _primitiveFailed: #_changePasswordFrom:to:
     args: { firstString . secondString } .
self _uncontinuableError

]

{ #category : 'Updating' }
UserProfile >> _defaultObjectSecurityPolicy: anObjectSecurityPolicy [

"Change the default login GsObjectSecurityPolicy of the receiver, which is the UserProfile of
 the current session.

 This method requires the #DefaultObjectSecurityPolicy privilege,
 and requires write authorization for the receiver.

 This method does not change the GsObjectSecurityPolicy into which newly created objects
 are placed, that will happen after the current session commits
 at the next login using this UserProfile .
 See also System (C) >> currentObjectSecurityPolicy:  .

 Gs64 v2.2 , the requirement for write authorization for the receiver is new;
 it did not exist in Gemstone/S 6.x "

<primitive: 307>
anObjectSecurityPolicy _validateClass: GsObjectSecurityPolicy.
self _primitiveFailed: #_defaultObjectSecurityPolicy:
     args: { anObjectSecurityPolicy } .
^ nil

]

{ #category : 'Updating Privileges' }
UserProfile >> _deletePrivilege: aPrivilegeString [

< protected >
| index privSet privSymbol |
privSet := UserProfile _privilegesSetFromArray: { aPrivilegeString }.
privSymbol := privSet _at: 1 .
(index := PrivilegeNames indexOf: privSymbol ) == 0 ifTrue:[
  aPrivilegeString _error: #rtErrBadPriv .
  ^ self
].
(self class isSpecialUserId: userId) ifTrue:[
  System iAmSystemUser
    ifFalse:[ ^ self _error: #rtErrMustBeSystemUser ] .
].
privileges := privileges bitAnd:( 1 bitShift: index - 1) bitInvert

]

{ #category : 'Private' }
UserProfile >> _enableNonGemStoneAuthentication [
"Private. Customers should not call this method directly."

self _securityDataInstVarAt: OC_USER_SECUR_DATA_VERIFIER put: 0 .
self _securityDataInstVarAt: OC_USER_SECUR_DATA_REASON_DISABLED put: nil .
^ self

]

{ #category : 'Private' }
UserProfile >> _fetchPasswordNeverExpires [

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_PASSWD_NEVER_EXPIRES .
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Accessing Privileges' }
UserProfile >> _hasPrivilegeName: aSymbol [

 "Returns true if receiver has the privilege specified by aSymbol,
  false otherwise."

 | bitNumber |
 bitNumber := (PrivilegeNames indexOfIdentical: aSymbol) .
 bitNumber < 1 ifTrue:[ ^ false ].
 ^ (privileges bitAt: bitNumber ) == 1

]

{ #category : 'Accessing Privileges' }
UserProfile >> _hasPrivilegeBit: anInteger [

 "Returns true if receiver has the privilege specified by anInteger,
  false otherwise."

 anInteger < 1 ifTrue:[ ^ false ].
 ^ (privileges bitAt: anInteger ) == 1

]

{ #category : 'Private' }
UserProfile >> _initialize [

"Private.  Initialize private instance variables."

self _validatePrivilege.

]

{ #category : 'External Authentication - Private' }
UserProfile >> _invalidDnError: aString [

self _error: #rtErrIllegalDn args: { aString }  .
self _uncontinuableError

]

{ #category : 'Repository Conversion' }
UserProfile >> _migrateGroups: migrateBlock [

"Converts the groups collections from IdentitySets of Strings to
 IdentitySets of UserProfileGroups."

groups ~~ nil  ifTrue:[ migrateBlock value: groups ] .
groups do:[:eachGroup| eachGroup add: self ].

]

{ #category : 'Private' }
UserProfile >> _newSecurityData [

"Creates a new instance of UserSecurityData for the receiver."

" only called from UserProfileSet>>_add: "
<protected primitive: 462>

self _primitiveFailed: #_newSecurityData .
self _uncontinuableError

]

{ #category : 'Private' }
UserProfile >> _oldPasswordsIncludes: aString [

"Returns true if the receiver's old passwords include the argument."

<protected primitive: 459>

self _primitiveFailed: #_oldPasswordsIncludes: args: { aString } .
self _uncontinuableError

]

{ #category : 'Private' }
UserProfile >> _password: aString [

"Modifies the receiver's password to be aString, and returns the receiver.  If
 the argument is not a String, generates an error.  The new password (aString)
 may not be longer than 1024 Characters.

 Raises an #rtErrAuthIllegalPasswordChange error if the authentication mode of
 the receiver is not set to #GemStone.

 This method allows you to change the password of another GemStone user.
 To change your own password, use oldPassword:newPassword: instead.

 This method requires the #OtherPassword privilege."

<protected primitive: 309>
aString _validateKindOfClass:  String  . "MultiByteString not allowed"
aString class isBytes ifFalse:[ aString _error: #objErrNotByteKind ] .
self _primitiveFailed: #_password: args: { aString } .
self _uncontinuableError

]

{ #category : 'Private' }
UserProfile >> _pendingReasonForDisabledAccount [

" Returns either 'StaleAccount' or 'PasswordAgeLimit' if this
  UserProfile will be disabled for that reason the next time it logs in.
  Otherwise returns nil."

| staleLimit passwordLimit |
self passwordNeverExpires ifTrue: [ ^ nil ].

" Check and see if we're using staleAccountLimits
  and if this account has exceeded the limit "
staleLimit := AllUsers staleAccountAgeLimit.
(staleLimit ~~ nil and: [ staleLimit > 0 ]) ifTrue: [
   | lastLogin |
   (lastLogin := self lastLoginTime) ifNotNil:[
         ((DateTime now) > (lastLogin addHours: staleLimit))
         ifTrue: [ ^ 'StaleAccount' ]]].

" Check and see if we're using passwordAging
  and if this account has exceeded the limit"
passwordLimit := AllUsers passwordAgeLimit.
(passwordLimit ~~ nil and: [ passwordLimit > 0 ]) ifTrue: [
      | lastPasswordChange |
      (lastPasswordChange := self lastPasswordChange) ifNotNil:[
            ((DateTime now) > (lastPasswordChange addHours: passwordLimit))
            ifTrue: [ ^ 'PasswordAgeLimit' ]]].
^ nil

]

{ #category : 'Private' }
UserProfile >> _postLoginInitialize [

  "Invoked from GsCurrentSession >> initialize."

  loginHook ifNotNil:[ :v |
    v _isExecBlock ifTrue:[ ^ v value ] .
    v _isSymbol ifTrue:[ ^ self perform: v ]
  ].

]

{ #category : 'Accessing Privileges' }
UserProfile >> _privileges [

"Returns a SmallInteger specifying the user's level of access to certain
 privileged system functions ordinarily performed by the GemStone data
 curator."

"SystemControl (bit 0); ObsoleteStatistics (1); SessionAccess (2);  UserPassword (3);
 DefaultObjectSecurityPolicy (4); PrivUnused5 (5);  OtherPassword (6);
 ObjectSecurityPolicyCreation (7); ObjectSecurityPolicyProtection (8); FileControl (9);
 GarbageCollection (10); CodeModification (11) ;
 NoPerformOnServer (12); NoUserAction (13); NoGsFileOnServer (14);
 NoGsFileOnClient (15); SessionPriority (16);  CompilePrimitives (17);
 ChangeUserId (18) MigrateObjects(19) DisableObjectReadLogging (20) 
 DynamicDisableObjectReadLogging (21) ReadOtherUserProfile(22)
 CreateOnetimePassword(23)"

^ privileges

]

{ #category : 'Updating Privileges' }
UserProfile >> _privileges: aSmallInteger [

"Modifies the user's privileges (level of access to privileged system
 functions) to the level represented by the argument aSmallInteger."

self _validatePrivilege ifTrue:[
  (aSmallInteger _validateInstanceOf: SmallInteger ) ifTrue:[
    privileges := aSmallInteger
  ].
].

]

{ #category : 'Private' }
UserProfile >> _removeSymbol: aSymbol [

"Remove aSymbol from the Symbol List by searching the symbol list for an
 Association whose key is equal to aSymbol.  Returns true if successful.
 Otherwise no such Association was found, and it returns false."

| temp dict |
self _validatePrivilege ifTrue:[
  1 to: symbolList size do:[:j |
    dict := symbolList at: j .
    temp := dict at: aSymbol otherwise: nil .
    temp ifNotNil:[
      dict removeKey: aSymbol.
      ^ true
    ].
  ].
].
^ false

]

{ #category : 'Private' }
UserProfile >> _securityDataInstVarAt: offset [

"Returns the value of the specified instance variable in the UserSecurityData
 of the receiver.  If the value is not a special object, returns a copy of the
 value of that instance variable.

 If the receiver is not your UserProfile,
 offset=OC_USER_SECUR_X509_STATUS requires SessionAccess privilege,
 Most other values offset require either OtherPassword or ReadOtherUserProfile
 privilege, otherwise signals an error.

 offset is zero based.  0 implements 'isDisabled' and returns true or false.
 The primitive fails if you attempt to access the password or the
 oldPasswords instVars."

<protected primitive: 460>
self _primitiveFailed: #_securityDataInstVarAt: args: { offset } .
self _uncontinuableError

]

{ #category : 'Private' }
UserProfile >> _securityDataInstVarAt: offset put: aValue [

"Private.  Modifies the specified instance variable in the UserSecurityData of
 the receiver.  The sender is responsible for making a copy of aValue and
 making the copy invariant, if appropriate.  If aValue is not a special
 object, the primitive will assign aValue to the SecurityDataObjectSecurityPolicy.

 offset is zero based.

 Generates an error if the receiver is not your UserProfile and you do not
 have the #OtherPassword privilege.

 The primitive fails if you attempt to modify the password instance variable
 of the receiver's security data."

<protected primitive: 461>
self _primitiveFailed: #_securityDataInstVarAt:put: args: { offset . aValue } .
self _uncontinuableError

]

{ #category : 'Private' }
UserProfile >> _setPassword: aString [
"Private. Customers should not call this method directly."

"no checks for format of new password, omit _validateNewPassword: "
(self _validate__password: aString)
  ifFalse:[ ^ userId _error: #rtErrBadPassword ].
self _password: aString . "change the password"
AllUsers addMsgToSecurityLog: '      new password set for ' , self userId , ' .' .
^ self

]

{ #category : 'Private' }
UserProfile >> _setPasswordNeverExpires: aBoolean [

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validatePrivilege ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_PASSWD_NEVER_EXPIRES put: aBoolean .
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Private' }
UserProfile >> _symbolListNamesAndDictionaries [

"Returns an Array of Associations which name the dictionaries in the
 receiver's SymbolList."

| result |

result := { }  .
symbolList do: [ :aDict |
  result add:
    ( aDict associationsDetect: [ :anAssoc | anAssoc value == aDict ]
             ifNone: [ SymbolAssociation newWithKey: #unnamed value: aDict ] ).
  ].
^ result.

]

{ #category : 'Private' }
UserProfile >> _userId: newUserId [

"Modifies the userId associated with the receiver to be aString."

self _validatePrivilege ifTrue:[
  userId ifNotNil:[ Error signal:'illegal reassignment of userId'].
  userId := newUserId .
].

]

{ #category : 'Private' }
UserProfile >> _validate__password: aString [

^ self class _validate__password: aString forUserId: userId

]

{ #category : 'External Authentication - Private' }
UserProfile >> _validateAuthenticationSchemeIsExternal [

"Raises an error if the authentication scheme for the recevier is
 #GemStone, otherwise returns true. If execution continued from the
 error, returns false."

self authenticationSchemeIsGemStone ifTrue:[
   self _error: #rtErrAuthIllegalOperation
                 args: { self authenticationScheme } .
   ^ false
].
^ true

]

{ #category : 'External Authentication - Private' }
UserProfile >> _validateAuthenticationSchemeIsGemStone [

"Raises an error if the authentication scheme for the recevier is not
 #GemStone, otherwise returns true. If execution continued from the
 error, returns false."

self authenticationSchemeIsGemStone ifFalse:[
   self _error: #rtErrAuthIllegalOperation
                 args: { self authenticationScheme } .
   ^ false
].
^ true

]

{ #category : 'External Authentication - Private' }
UserProfile >> _validateBaseDn: baseDn searchDn: filterDn [

"The baseDn must be a String.  If the filterDn is nil,
 then the baseDn must contain the '%s' character sequence.
 If the filterDn is not nil, then it must be a string that
 contains the '%s' sequence, and baseDn must not contain
 the sequence.  If all of these requirements are met
 returns true, otherwise signals an error.  If execution is
 continued from the error, returns false.
"

(baseDn _validateClass: String ) ifFalse:[ ^ false ].
filterDn ifNil:[
  baseDn hasLdapWildcardPattern ifFalse:[
     self _invalidDnError: baseDn . ^ false
  ].
] ifNotNil:[
   (filterDn _validateClass: String ) ifFalse:[  ^ false ].
   filterDn hasLdapWildcardPattern
     ifFalse:[ self _invalidDnError: filterDn . ^ false ].
   baseDn hasLdapWildcardPattern
     ifTrue:[ self _invalidDnError: baseDn . ^ false ].
  ].
^ true

]

{ #category : 'Accessing Privileges' }
UserProfile >> _validateCodeModificationPrivilege [

"Generates an error if the receiver does not have
 either the CodeModification or the OtherPassword privilege.
 Uses hardcoded bit number to handle filein and upgrade cases."

self isSystemUserProfile ifTrue: [ ^ true ].
(System _inProtectedMode or: [(privileges bitAnd: 16r840) ~~ 0]) ifTrue: [ ^ true ].
self _error: #rtErrNoPriv .
self _uncontinuableError

]

{ #category : 'Private' }
UserProfile >> _validateIsMyUserProfileOrHasPrivilege: aPrivSymbol [

"Validate that the receiver is my own UserProfile or that I have the
 privilege aPrivSymbol."

| myUserProfile |

myUserProfile := System myUserProfile .
self == myUserProfile ifFalse:[
  ^ myUserProfile _validatePrivilegeName: aPrivSymbol
].
^ true

]

{ #category : 'Private' }
UserProfile >> _validateNewPassword: aString [

"Generates an error if aString is not a valid new password for the receiver.
 Otherwise, returns true .  Note that the userId of the receiver is
 always a disallowed password."

<protected>

(self _validate__password: aString)
  ifFalse:[ ^ false ] .

(AllUsers _validateNewPassword: aString ) ifFalse:[ ^ false ].

AllUsers disallowUsedPasswords ifTrue:[
  (self _oldPasswordsIncludes: aString) ifTrue:[
      aString _error: #rtDisallowedPassword .
      ^ false
  ].
].
^ true

]

{ #category : 'Updating Privileges' }
UserProfile >> _validatePrivilege [

" You must have the #OtherPassword to modify UserProfiles "

^  System myUserProfile _validatePrivilegeName: #OtherPassword

]

{ #category : 'Accessing Privileges' }
UserProfile >> _validatePrivilege: anInt [

"Generates an error if the receiver does not have the privilege specified
 by anInt.  anInt is one based"

(self isSystemUserProfile or:[(privileges bitAt: anInt ) == 1]) ifTrue: [ ^ true ].
self _error: #rtErrNoPriv .
self _uncontinuableError

]

{ #category : 'Accessing Privileges' }
UserProfile >> _validatePrivilegeName: aSymbol [

"Generates an error if the receiver does not have the privilege specified
 by aSymbol, or if aSymbol is not the name of a privilege."

| bitNumber |
self isSystemUserProfile ifTrue:[ ^ true ] .
bitNumber := PrivilegeNames indexOfIdentical: aSymbol .
bitNumber > 0 ifTrue: [ ^ self _validatePrivilege: bitNumber ].
self _error: #rtErrNoPriv .
self _uncontinuableError

]

{ #category : 'Private' }
UserProfile >> _validateOneOfPrivileges: anArray  [
 anArray do:[:sym | | bitNum |
   bitNum := PrivilegeNames indexOfIdentical: sym .
   bitNum <= 0 ifTrue:[ Error signal:'invalid privilege ', sym asString ].
   (privileges bitAt: bitNum ) == 1 ifTrue:[ ^ true ].
 ].
 self _error: #rtErrNoPriv .
 self _uncontinuableError
]
  

{ #category : 'External Authentication - Private' }
UserProfile >> _validateUnixUserIdOrNil: aString [

"Confirm that aString is nil or is a valid UNIX user ID on this host"

aString ifNil:[ ^ true ].
(System unixUserIdExists: aString) ifFalse:[
  aString _error: #rtErrInvalidUnixUserId .
  ^ false
].
^ true

]

{ #category : 'Private' }
UserProfile >> _validateUserProfileWrite [

"Validate that the receiver is my own UserProfile or that I have the
 #OtherPassword privilege."

^ self _validateIsMyUserProfileOrHasPrivilege: #OtherPassword

]

{ #category : 'Private' }
UserProfile >> _validateUserProfileRead [

"Validate that the receiver is my own UserProfile or that I have the
 at least one of the #OtherPassword or #ReadOtherUserProfile privilege."

 | myPro |
 myPro := System myUserProfile .
 self == myPro  ifTrue:[ ^ true].
 myPro isSystemUserProfile ifTrue:[ ^ true ] .
 ^ myPro _validateOneOfPrivileges: #( OtherPassword ReadOtherUserProfile )
]

{ #category : 'Accessing' }
UserProfile >> activeUserIdLimit [

"Returns the maximum number of concurrent logins allowed for the receiver.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_ACTIVE_USR_LIM  .
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Updating' }
UserProfile >> activeUserIdLimit: aPositiveInteger [

"Sets the maximum number of concurrent logins for the receiver to be
 aPositiveInteger.  This change will take effect after this session commits.

 Generates an error if you do not have the #OtherPassword privilege."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  self _validatePrivilege ifTrue:[
    aPositiveInteger ifNotNil:[
      aPositiveInteger _isSmallInteger ifFalse:[
        ^ aPositiveInteger _validateClass: SmallInteger .
      ].
      aPositiveInteger < 0 ifTrue:[
        ^ aPositiveInteger _error: #rtErrArgNotPositive
      ].
    ].
    self _securityDataInstVarAt: OC_USER_SECUR_DATA_ACTIVE_USR_LIM put: aPositiveInteger .
  ].
] ensure:[
  prot _leaveProtectedMode
]

]

{ #category : 'Group Membership' }
UserProfile >> addGroup: aGroupString [

"Adds the user to the group with the name aGroupString, and returns the
 receiver.  If the user already belongs to the group aGroupString, no action
 occurs.

 If the group with name aGroupString is not defined in UserProfileGroup,
 generates an error.

 Legacy method; use addToUserProfileGroup: or UserProfileGroup >> addUser: instead."

| theGroup |
(theGroup := UserProfileGroup _validateGroupString: aGroupString) ifNil:[ ^ nil ] .
^self addToUserProfileGroup: theGroup

]

{ #category : 'Updating Privileges' }
UserProfile >> addPrivilege: aPrivilegeString [

"Adds aPrivilegeString to the receiver's privileges.  Generates an error if
 aPrivilegeString is not a valid privilege name (as defined in the class
 variable PrivilegeNames).

 Only SystemUser may add privileges to special UserProfiles. Attempts which 
 violate this restriction will raise an exception.

 Example: 'System myUserProfile addPrivilege: #GarbageCollection' "

<primitive: 2001> "enter protected mode"
| prot |
prot := System _protectedMode .
[ self _validatePrivilege ifTrue:[
    self _addPrivilege: aPrivilegeString.
  ].
] ensure:[
  prot _leaveProtectedMode
]

]

{ #category : 'Group Membership' }
UserProfile >> addToUserProfileGroup: aUserProfileGroup [

"Adds the user to the UserProfileGroup. If the user already belongs to
 the group, no action occurs."

self _validatePrivilege ifTrue:[
    | userGroups |
    userGroups := groups .
    userGroups == nil ifTrue:[
      userGroups := IdentitySet new .
      userGroups objectSecurityPolicy: self objectSecurityPolicy .
    ].
  (userGroups includesIdentical: aUserProfileGroup) ifFalse:[
    groups ifNil:[ groups := userGroups ].
    groups add: aUserProfileGroup.
    aUserProfileGroup add: self .
    ] .
].

]

{ #category : 'Group Membership' }
UserProfile >> auditGroups [
"Audits the receiver's groups collection to ensure:
  -every UserProfileGroup in the collection contains the receiver.
  -every UserProfileGroup in the collection is present in the AllGroups global.

Raises an error if the audit fails and returns true if successful."

groups do:[:aGroup| |name obj|
  name := aGroup groupName .
  obj := AllGroups at: name ifAbsent:[ aGroup _error: 'Group is missing from AllGroups'. ^ false ].
  obj == aGroup
    ifFalse:[ aGroup error: 'Non-canonical group found' . ^ false ] .
  (aGroup includes: self)
    ifFalse: [ aGroup error: 'Inconsistent group state' . ^ false ] .
].
^ true

]

{ #category : 'External Authentication' }
UserProfile >> authenticationScheme [

"Returns a Symbol representing the authentication scheme for the receiver,
 result is one of #GemStone, #UNIX, #LDAP, #SingleSignOn , #X509 .

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[ self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_AUTH_SCHEME.
  ]
] ensure:[
  prot _leaveProtectedMode  "exit protected mode"
].
^ self class authenticationSchemeIdToSymbol: result

]

{ #category : 'External Authentication - Private' }
UserProfile >> authenticationScheme: aSymbol [

"Private - customers should not call this method directly.

 Sets the athentication scheme for the receiver to match aSymbol, which
 must be one of  #GemStone, #UNIX, #LDAP, #SingleSignOn, or #X509.
 The system user profiles, SystemUser, DataCurator, GcUser, SymbolUser and Nameless,
 must always use 'GemStone' authentication.  Attempts to change the authentication
 scheme for these UserProfiles will raise an error.

 Requires the OtherPassword privilege."

self _securityDataInstVarAt: OC_USER_SECUR_DATA_AUTH_SCHEME
        put: (self class authenticationSchemeSymbolToId: aSymbol).
^ self

]

{ #category : 'External Authentication' }
UserProfile >> authenticationSchemeIsGemStone [

"Answer true if the receiver uses GemStone for password authentication.
 Otherwise answer false.

 Requires the #OtherPassword privilege for any UserProfile other than
 your own."

^ self authenticationScheme == #GemStone

]

{ #category : 'External Authentication' }
UserProfile >> authenticationSchemeIsLDAP [

"Answer true if the receiver uses LDAP for password authentication.
 Otherwise answer false.

 Requires the #OtherPassword privilege for any UserProfile other than
 your own."

^ self authenticationScheme == #LDAP

]

{ #category : 'External Authentication' }
UserProfile >> authenticationSchemeIsSingleSignOn [

"Answer true if the receiver uses Single Sign On (Kerberos) for password authentication.
 Otherwise answer false.

 Requires the #OtherPassword privilege for any UserProfile other than
 your own."

^ self authenticationScheme == #SingleSignOn

]

{ #category : 'External Authentication' }
UserProfile >> authenticationSchemeIsUNIX [

"Answer true if the receiver uses UNIX for password authentication.
 Otherwise answer false.

 Requires the #OtherPassword privilege for any UserProfile other than
 your own."

^ self authenticationScheme == #UNIX

]

{ #category : 'External Authentication' }
UserProfile >> authenticationSchemeIsX509 [

"Answer true if the receiver uses X509 certificates for authentication.
 Otherwise answer false.

 Requires the #OtherPassword privilege for any UserProfile other than
 your own."

^ self authenticationScheme == #X509

]

{ #category : 'Enumerating' }
UserProfile >> classesDo: aBlock [

"Execute 3-argument block over all classes (plus those referenced by
 its class history) that are contained by all SymbolDictionaries in
 this user's symbol list.  Checks are made to avoid  redundant processing
 of classes/symbol dictionaries already scanned.

 Arguments to block are:
    [:aUserProfile :aSymbolDictionary :aClass | ... ]

"

  | coveredSymDics coveredClasses |

  coveredSymDics := IdentitySet new.
  coveredClasses := IdentitySet new.

  self symbolList do: [ :aSymbolDictionary |
    (coveredSymDics includes: aSymbolDictionary) ifFalse: [
      coveredSymDics add: aSymbolDictionary.
      aSymbolDictionary valuesDo: [ :namedClass |
        (namedClass isKindOf: Class) ifTrue: [
          namedClass classHistory do: [ :aClass |
            (coveredClasses includes: aClass) ifFalse: [
              coveredClasses add: aClass.
              aBlock value: self
                     value: aSymbolDictionary
                     value: aClass ]]]]]]

]

{ #category : 'Updating' }
UserProfile >> clearOldPasswords [

"Clears the set of old passwords for the receiver, thus permitting reuse of
 some passwords that had been previously disallowed.

 Generates an error if you do not have the #OtherPassword privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validatePrivilege ifTrue:[
    self _securityDataInstVarAt: OC_USER_SECUR_DATA_USED_VERIFIERS put: { } . "old salts"
    self _securityDataInstVarAt: OC_USER_SECUR_DATA_USED_SALTS put: { } . "old verifiers"
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_OLD_PASSWDS_ARRAY put: { } .
  ]
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Clustering' }
UserProfile >> clusterDepthFirst [

"This method clusters the receiver's user ID, password, and symbol list.
 Because the password and user ID are assumed to be byte objects, and because
 the symbol list may contain shared elements, no further traversal of the
 objects is done.  Returns true if the receiver has already been clustered
 during the current transaction; returns false otherwise."

self cluster
  ifTrue: [ ^ true ]
  ifFalse: [
    encryptedPassword cluster.
    userId cluster.
    symbolList cluster.
    ^ false
  ]

]

{ #category : 'Updating' }
UserProfile >> createDictionary: aSymbol [

"Creates a new SymbolDictionary. The new Dictionary is created with a single
 SymbolAssociation, whose key is aSymbol and whose value is the new
 SymbolDictionary itself. It is recommended that the symbol be a valid
 identifer, though it is not disallowed to use any kind of symbol.

 Also creates a SymbolAssociation in the receiver's UserGlobals dictionary, with
 aSymbol as the key and the new dictionary as its value.

 Returns the new SymbolDictionary. Generates an error if aSymbol is already
 defined in the receiver's symbol list, or if aSymbol is not a Symbol."

| newDictionary anAssoc userGlobalsOfMe |

self _validatePrivilege ifFalse:[ ^ nil ].
(aSymbol _validateClass: Symbol) ifFalse:[ ^ nil ].

anAssoc := self resolveSymbol: aSymbol.
 (anAssoc == nil)
    ifFalse:[ ^ self _error: #rtErrSymAlreadyDefined args: { aSymbol }].

newDictionary := (SymbolDictionary new).  "Create new dictionary"
"Name it by making a reference to itself"
newDictionary at: aSymbol put: newDictionary.

userGlobalsOfMe := (self resolveSymbol: #UserGlobals) value.
"Get my UserGlobals"
userGlobalsOfMe at: aSymbol put: newDictionary.

^newDictionary

]

{ #category : 'Accessing' }
UserProfile >> defaultObjectSecurityPolicy [

"Returns the default login GsObjectSecurityPolicy associated with the receiver."

^ defaultObjectSecurityPolicy

]

{ #category : 'Updating' }
UserProfile >> defaultObjectSecurityPolicy: anObjectSecurityPolicy [

"Redefines the default login GsObjectSecurityPolicy associated with the receiver,
 and returns the receiver.  The argument may be nil (which implies World write for
 all newly created objects).  The new value will take effect for subsequent
 logins using the receiver.

 If the receiver is the UserProfile under which this session is logged in,
 this method requires the DefaultObjectSecurityPolicy privilege.

 If the receiver is not the UserProfile under which this session is logged
 in, you must have write authorization for the GsObjectSecurityPolicy where the receiver
 resides and the #OtherPassword privilege.

 Exercise extreme caution when executing this method.  If, at the time you commit your
 transaction, the receiver no longer had write authorization for anObjectSecurityPolicy,
 that user's GemStone session will be terminated and the user will be unable to log back
 in to GemStone."

anObjectSecurityPolicy ifNotNil:[
  (anObjectSecurityPolicy _validateClass: GsObjectSecurityPolicy) ifFalse:[ ^ nil].
] .
(self == System myUserProfile) ifTrue: [
  ^ self _defaultObjectSecurityPolicy: anObjectSecurityPolicy
] ifFalse:[
  self _validatePrivilege ifFalse:[ ^ nil ].
].
defaultObjectSecurityPolicy := anObjectSecurityPolicy .

]

{ #category : 'Deprecated' }
UserProfile >> defaultSegment [

self deprecated: 'UserProfile>>defaultSegment deprecated v3.0.
Use #defaultObjectSecurityPolicy instead.'.
^self defaultObjectSecurityPolicy.

]

{ #category : 'Deprecated' }
UserProfile >> defaultSegment: anObjectSecurityPolicy [

 self deprecated: 'UserProfile>>defaultSegment: deprecated v3.0.
Use #defaultObjectSecurityPolicy: instead.'.
 ^ self defaultObjectSecurityPolicy: anObjectSecurityPolicy

]

{ #category : 'Updating Privileges' }
UserProfile >> deletePrivilege: aPrivilegeString [

"Removes aPrivilegeString from the receiver's privileges.  Generates an error
 if aPrivilegeString is not a valid privilege name (as defined in the class
 variable PrivilegeNames).

 Only SystemUser may delete privileges from special UserProfiles.
 Attempts which violate this restriction will raise an exception."

"Example: 'System myUserProfile deletePrivilege: #GarbageCollection'"

<primitive: 2001> "enter protected mode"
| prot |
prot := System _protectedMode .
[ self _validatePrivilege ifTrue:[
    self _deletePrivilege: aPrivilegeString.
  ].
] ensure:[
  prot _leaveProtectedMode
]

]

{ #category : 'Accessing the Symbol List' }
UserProfile >> dictionariesAndSymbolsOf: anObject [

"Returns a Array, possibly empty, of the form { { aDictionary . aSymbol } ... }.
 Searches the receiver for Associations whose value is identical to
 anObject and returns information on all matching Associations ."

  ^ symbolList dictionariesAndSymbolsOf: anObject

]

{ #category : 'Accessing the Symbol List' }
UserProfile >> dictionaryAndSymbolOf: anObject [

"Returns the symbol list Dictionary that names anObject, and also returns the
 name which that Dictionary associates with anObject.  More precisely, this
 returns an Array containing two elements:

 * The Dictionary in the user's symbol list that contains an Association whose
   value is anObject.
 * The Symbol which is that Association's key.

 The symbol list is searched in the same order that the compiler searches it.
 (For more information about symbol resolution, see the GemStone Programming
 Guide.)  If anObject is not found in the symbol list, returns nil."

self deprecated: 'UserProfile class>>dictionaryAndSymbolOf: deprecated v3.1, use dictionariesAndSymbolsOf:' .

^ symbolList dictionaryAndSymbolOf: anObject

]

{ #category : 'Accessing the Symbol List' }
UserProfile >> dictionaryNames [

"Returns a formatted String describing the position and name of each Dictionary
 in the receiver's symbol list.

 This method assumes that each Dictionary in the symbol list contains an
 Association whose value is that Dictionary.  If any Dictionary does not
 contain such an Association, it is represented in the result String as
 '(unnamed Dictionary)'."

^ symbolList namesReport

]

{ #category : 'Updating' }
UserProfile >> disable [

"Disables the receiver when the transaction is committed.
 Generates an error if you do not have OtherPassword privilege.
 Returns the receiver."

^ self disableWithReason: 'Disabled by administrator'

]

{ #category : 'Updating Privileges' }
UserProfile >> disableCommits [

"Disables commits for the receiver if they were previously disabled.
 The current transaction must be committed to apply this change.

 Requires the #OtherPassword privilege.

 Has no effect on existing sessions.

 Commits will be disabled for the receiver at the next login attempt
 which occurs AFTER the current transaction is committed.

 Disabling commits for the reserved system accounts
 (SystemUser, DataCurator, GcUser, SymbolUser and Nameless) is not allowed
 and will always fail.

 Returns true if the operation was successful or if commits were already
 disabled for the receiver.  Returns false if the operation failed."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[ result := true.
  self _validatePrivilege ifTrue:[
    (self class isSpecialUserId: userId)
      ifTrue:[ result := false] "illegal"
      ifFalse:[
        self isReadOnly
          ifFalse:[ result := (self _securityDataInstVarAt: OC_USER_SECUR_DATA_READ_ONLY put: true) == true]].
  ].
] ensure:[
    prot _leaveProtectedMode "exit protected mode"
].
^ result

]

{ #category : 'Updating' }
UserProfile >> disableLoginLogging [

"Disables the printing of login/logout events for the receiver to the
 stone's login log file. Has no effect (i.e. logins still not logged)
 if the stone does not have login logging enabled using
 STN_LOGIN_LOG_ENABLED.

 This method requires the #OtherPassword privilege.

 Returns the receiver."

^ self exemptFromLoginLogging: true

]

{ #category : 'Updating' }
UserProfile >> disableWithReason: aString [

"Disables the receiver when the transaction is committed and specifies aString
 as the reason.  Generates an error if you do not have OtherPassword privilege.
 Returns the receiver."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  self _validateUserProfileWrite ifTrue:[
    (aString _validateClasses: { String } ) ifFalse:[ ^ nil ].
    self _securityDataInstVarAt: OC_USER_SECUR_DATA_VERIFIER put: nil.
    self _securityDataInstVarAt: OC_USER_SECUR_DATA_REASON_DISABLED put: aString copy immediateInvariant .
  ]
] ensure:[
  prot _leaveProtectedMode
].
^ self

]

{ #category : 'Updating Privileges' }
UserProfile >> enableCommits [

"Enables commits for the receiver if they were previously disabled.
 The current transaction must be committed to apply this change.

 Requires the #OtherPassword privilege.

 Has no effect on existing sessions.

 Commits will be reenabled for the receiver at the next login attempt
 which occurs AFTER the current transaction is committed.

 Disabling commits for the reserved system accounts
 (SystemUser, DataCurator, GcUser, SymbolUser and Nameless) is not allowed.
 Therefore enabling commits for these user profiles will always succeed.

 Returns true if the operation was successful or if commits were already
 enabled for the receiver."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[ result := true.
  self _validatePrivilege ifTrue:[
     (self class isSpecialUserId: userId)
     ifFalse:[
      self isReadOnly
        ifTrue:[  result := (self _securityDataInstVarAt: OC_USER_SECUR_DATA_READ_ONLY put: false) == false]].
  ]
] ensure:[
  prot _leaveProtectedMode "exit protected mode"
].
^ result

]

{ #category : 'External Authentication' }
UserProfile >> enableGemStoneAuthenticationWithPassword: aPassword [

"Modify the receiver so that GemStone authentication is used to validate
 logins to GemStone by the receiver.  After this method has
 been run and successfully committed, all future logins of the receiver
 must specify a valid GemStone password.

 Requires the OtherPassword privilege."

<primitive: 2001>
| prot |
prot := System _protectedMode .

[ self _validatePrivilege ifTrue:[
    self _validate__password: aPassword . "fix 47327"
    self authenticationScheme: #GemStone  .
    self userIdAlias: nil.
    self kerberosPrincipal: nil.
    self _setPassword: aPassword .
  ]
] ensure:[
prot _leaveProtectedMode  "exit protected mode"].

^ self

]

{ #category : 'External Authentication' }
UserProfile >> enableLDAPAuthenticationWithAlias: aString baseDn: baseDn filterDn: filterDn [

"Modify the receiver so that LDAP authentication is used to validate
 all logins to GemStone by the receiver.  After this method has
 been run and successfully committed, all future logins of the receiver
 must specify the password for the given LDAP user ID.  GemStone will then
 validate the password using an LDAP bind.

 If aString is nil, then the UNIX user ID and the UserProfile
 ID of the receiver are identical.  At the next login, an LDAP bind
 will be performed using the user ID of the receiver and the password
 specified during the login.

 Most LDAP binds require a fully qualified DN (distinguished name) to be
 specified in order to get a successful bind.  There are 2 ways to get
 thee fully qualified DN: specify it as baseDn, or have GemStone perform
 a query to resovle the DN before attempting the bind.

 If filterDn is aString, this filter will be used to search for the fully
 qualified DN, and the string must include the character sequence '%s' in
 place of the user ID.  The baseDn argument must not contain the '%s'
 sequence in this case.

 If no DN search is to be performed, then nil should be passed in the filterDn
 argument and the baseDn string must contain the character sequencey '%s'
 where the userId is to be inserted in the string.

 Refer to the LDAP section of the System Administrators Guide for more
 information on configuring GemStone for LDAP authentication.

 Special UserProfiles may not use LDAP authentication.  Attempting
 to change the authentication scheme to LDAP for a special UserProfile will
 raise an error.  See the method #isSpecialUserId: in this class to see which
 UserProfiles are considered special.

 Requires the OtherPassword privilege."

<primitive: 2001>
| prot |
prot := System _protectedMode .

[ self _validatePrivilege ifFalse:[ ^ nil ].
  (self _validateBaseDn: baseDn searchDn: filterDn ) ifTrue:[
    aString ifNotNil:[
      (aString _validateClass: String) ifFalse:[ ^ nil ].
               self userIdAlias: aString
    ].
    self kerberosPrincipal: nil .
    self ldapBaseDn: baseDn .
    filterDn ifNotNil:[ self ldapFilterDn: filterDn ].
    self authenticationScheme: #LDAP .
    self _enableNonGemStoneAuthentication .
  ]
] ensure:[ prot _leaveProtectedMode  "exit protected mode"].

^ self

]

{ #category : 'Updating' }
UserProfile >> enableLoginLogging [

"Enables the printing of login/logout events for the receiver to the
 stone's login log file. Has no effect (i.e. logins still not logged)
 if the stone does not have login logging enabled using
 STN_LOGIN_LOG_ENABLED.

 This method requires the #OtherPassword privilege.

 Returns the receiver."

^ self exemptFromLoginLogging: false

]

{ #category : 'External Authentication' }
UserProfile >> enableSingleSignOnAuthenticationWithPrincipal: aKerberosPrincipal [

"Modify the receiver so that single sign on (SSO) authentication is
 required to validate all logins to GemStone by the receiver.  After this method
 has been run and successfully committed, all future logins of the receiver
 must be authenticated by the Kerberos single sign on mechanism. No password may
 be specified during the GemStone login.

 Refer to the SSO section of the System Administrators Guide for more
 information on configuring GemStone for SSO authentication.

 aKerberosPrincipal must be an instance of KerberosPrincipal or nil.

 SystemUser must always use GemStone authentication.  Attempting
 to change the authentication scheme for a SystemUser will raise
 an error.

 Requires the OtherPassword privilege."

<primitive: 2001>
| prot |
prot := System _protectedMode .

[ self _validatePrivilege ifFalse:[ ^ nil ].
  aKerberosPrincipal ifNotNil:[
     aKerberosPrincipal _validateClass: KerberosPrincipal.
     self kerberosPrincipal: aKerberosPrincipal .
  ].
  self userIdAlias: nil .
  self ldapBaseDn: nil .
  self ldapFilterDn: nil .
  self authenticationScheme: #SingleSignOn .
  self _enableNonGemStoneAuthentication .
] ensure:[ prot _leaveProtectedMode  "exit protected mode"].

^ self

]

{ #category : 'External Authentication' }
UserProfile >> enableUnixAuthenticationWithAlias: aUnixUserIdString [

"Modify the receiver so that UNIX authentication is used to validate
 all logins to GemStone by the receiver.  After this method has
 been run and successfully committed, all future logins of the receiver
 must specify the password for the given UNIX user ID.  GemStone will then
 validate the password using UNIX password calls.

 If aUnixUserIdString is nil, then the UNIX user ID and the UserProfile
 ID of the receiver are assumed to be the same.  At the next login,
 UNIX password authentication will be performed using the user ID of the
 receiver and the password specified during the login.

 Special UserProfiles may not use UNIX authentication.  Attempting
 to change the authentication scheme to UNIX for a special UserProfile will raise
 an error.  See the method #isSpecialUserId: in this class to see which
 UserProfiles are considered special.

 Requires the OtherPassword privilege."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[ self _validatePrivilege ifTrue:[
    (self _validateUnixUserIdOrNil: aUnixUserIdString ) ifTrue:[
      self authenticationScheme: #UNIX .
      self userIdAlias: nil .
      self kerberosPrincipal: nil .
      self ldapBaseDn: nil .
      self ldapFilterDn: nil .
      self _enableNonGemStoneAuthentication .
      aUnixUserIdString ifNotNil:[ self userIdAlias: aUnixUserIdString ].
    ].
  ]
] ensure:[
  prot _leaveProtectedMode  "exit protected mode"
].
^ self

]

{ #category : 'External Authentication' }
UserProfile >> enableX509Authentication [

"Modify the receiver so that GciX509Login with certificates must be
 used to validate all logins to GemStone by the receiver.
 After this method has been run and successfully committed,
 all future logins of the receiver must use GciX509Login.

 Special UserProfiles may not use X509 authentication.  Attempting
 to change the authentication scheme to X509 for a special UserProfile will
 signal an error.  See the method #isSpecialUserId: in this class to see which
 UserProfiles are considered special.

 GciX509Login is permitted regardless of the authentication scheme of a UserProfile.
 This method restricts a UserProfile to allow only GciX509Login .

 Requires the OtherPassword privilege."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[ self _validatePrivilege ifTrue:[
    self userIdAlias: nil .
    self ldapBaseDn: nil .
    self ldapFilterDn: nil .
    self kerberosPrincipal: nil .
    self authenticationScheme: #X509  .
    self _enableNonGemStoneAuthentication .
  ]
] ensure:[
  prot _leaveProtectedMode  "exit protected mode"
].
^ self

]

{ #category : 'Updating' }
UserProfile >> exemptFromLoginLogging: aBoolean [

"Enables or disables the printing of login/logout events for the receiver to the
 stone's login log file. Has no effect (i.e. logins still not logged)
 if the stone does not have login logging enabled using
 STN_LOGIN_LOG_ENABLED.

 This method requires the #OtherPassword privilege.

 Returns the receiver."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  self _validatePrivilege ifTrue:[
    | newBits oldBits |
    ((aBoolean ~~ true) and:[ aBoolean ~~ false ])
      ifTrue:[ ^ aBoolean _validateClass: Boolean ] .
    oldBits := self _securityDataInstVarAt: OC_USER_SECUR_DATA_BIT_FLAGS .
    oldBits == nil ifTrue:[ oldBits := 0 ].
    aBoolean
      ifTrue:[ newBits := oldBits bitOr: GSC_SET_USER_PROFILE_no_login_logging ]
     ifFalse:[ newBits := oldBits bitAnd: GSC_SET_USER_PROFILE_no_login_logging bitInvert ] .
    oldBits ~~ newBits
      ifTrue:[  self _securityDataInstVarAt: OC_USER_SECUR_DATA_BIT_FLAGS put: newBits ] .
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ self

]

{ #category : 'Group Membership' }
UserProfile >> groupNames [
"Answers an IdentitySet of symbols which represent the names of all UserProfileGroups of
 which the receiver is a member."

^ groups collect:[:eachGroup| eachGroup groupName ]

]

{ #category : 'Group Membership' }
UserProfile >> groups [

"Returns an IdentitySet, the set of groups to which the user belongs."

groups == nil ifTrue:[ ^ IdentitySet new ].
^ groups copy


]

{ #category : 'Testing' }
UserProfile >> hasLoginLogging [

"Answer true if the receiver is configured to write login and logout
 events to the stone's login log file, false otherwise.

 If the stone is not configured to log logins (STN_LOGIN_LOG_ENABLED is
 FALSE), then logins and logouts are not logged, regardless of this
 setting.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_BIT_FLAGS .
    result == nil ifTrue:[ result := 0 ] .
    result := (result bitAnd: GSC_SET_USER_PROFILE_no_login_logging) == 0 .
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Updating the Symbol List' }
UserProfile >> insertDictionary: aSymbolDictionary at: anIndex [

"Inserts aSymbolDictionary into the receiver's symbol list.  If the
 receiver is identical to 'GsSession currentSession userProfile', inserts
 aSymbolDictionary into the transient session symbol list as well.  The
 insertion into the receiver's symbol list occurs first.

 If anIndex is less than or equal to the size of the receiver's symbol
 list, inserts aSymbolDictionary into the symbol list at anIndex.

 If anIndex is 1 greater than the size of the receiver's symbol list,
 appends aSymbolDictionary to the receiver's symbol list.

 If anIndex is  more than 1 greater than the size of the receiver's
 symbol list, or if anIndex is less than 1, generates an error.

 If an error occurs as the result of the persistent insertion, no
 further action is taken and an error is generated.  If the insertion
 completes correctly, aSymbolDictionary is inserted into the transient
 session symbol list.

 If anIndex is 1, prepend aSymbolDictionary to the beginning of the
 transient symbol list.

 Otherwise, finds the dictionary at (anIndex - 1) in the persistent symbol
 list, and searches for it by identity in the transient symbol list.  If
 found, insert aSymbolDictionary just after it in the transient symbol
 list.  If not found, append aSymbolDictionary to the end of the transient
 symbol list.

 If an error occurs as a result of the insertion in the transient symbol
 list, the persistent symbol list is left in its new state, the
 transient symbol list is left in its old state and the error is
 silently ignored.

 Note if the transient and persistent lists have different contents when
 an abort transaction occurs they will not be automatically synchronized
 after the abort.  The persistent list will be rolled back to the
 committed state, but the transient list will not be rolled back."

"Example: System myUserProfile
                        insertDictionary: (SymbolDictionary new) at: 3'"
(self == System myUserProfile
  ifTrue: [ self _validatePrivilegeName: #CodeModification  ]
  ifFalse: [ self _validatePrivilege ]) ifTrue:[

  (aSymbolDictionary _validateClass: SymbolDictionary) ifTrue:[ | ses |
    self symbolList insertObject:  aSymbolDictionary at: anIndex.
    (self == (ses := GsSession currentSession) userProfile) ifTrue: [
      ses _transientSymbolList ifNotNil:[ :transient |
        anIndex == 1 ifTrue: [
	  transient insertObject: aSymbolDictionary at: 1.
	] ifFalse: [ | dict index |
	  dict := self symbolList at: (anIndex - 1).
	  index := transient indexOfIdentical: dict.
	  index ~~ 0 ifTrue: [
	    transient insertObject: aSymbolDictionary at: index + 1.
	  ] ifFalse: [
	    transient add: aSymbolDictionary.
	  ].
	].
      ].
    ].
  ]
].
^self.

]

{ #category : 'Accessing' }
UserProfile >> isDisabled [

"Returns true if logins for the receiver are disabled or if the next
 login attempt will fail and trigger disabling of the account, false otherwise.
 Generates an error if the receiver is not your UserProfile and you do not
 have the #OtherPassword privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_VERIFIER . "isDisabled"
  result ifFalse: [ result := self _pendingReasonForDisabledAccount ~~ nil ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Accessing' }
UserProfile >> _isDisabled [

"Returns true if logins for the receiver are disabled, false otherwise.
 Returns false if the disable is pending .
 Generates an error if the receiver is not your UserProfile and you do not
 have the #OtherPassword privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_VERIFIER . "isDisabled"
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Accessing Privileges' }
UserProfile >> isReadOnly [

"Returns true if commits have been disabled for this user profile by the
 #disableCommits method.  Otherwise returns false.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .

[ self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_READ_ONLY.
  ]
] ensure:[
  prot _leaveProtectedMode  "exit protected mode"
].
^ result == true

]

{ #category : 'Private' }
UserProfile >> isSystemProfile [
 "Return true if the receiver is one used for 'system sessions'"
 self isSystemUserProfile  ifTrue:[ ^ true ].
 userId = 'SymbolUser'  ifTrue:[ ^ true ].
 userId = 'GcUser'      ifTrue:[ ^ true ].
 userId = 'HostAgentUser'    ifTrue:[ ^ true ].
 ^ false

]

{ #category : 'Accessing' }
UserProfile >> kerberosPrincipal [

^ kerberosPrincipal

]

{ #category : 'Updating' }
UserProfile >> kerberosPrincipal: aKerberosPrincipal [

kerberosPrincipal := aKerberosPrincipal

]

{ #category : 'Accessing' }
UserProfile >> lastLoginTime [

"Returns a DateTime of the last login time of the receiver, or nil if
 no login time has been recorded.  If no stale account limits are set
 on the receiver, then no login times are recorded.  (See updating methods
 in UserProfileSet.)

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_LAST_LOGIN_TIM .
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Updating' }
UserProfile >> lastLoginTime: aVal [

"Updates the lastLoginTime for the receiver to be aVal, which must be an
 instance of DateTime or nil.

 Raises an error if the authentication scheme of the receiver is not #GemStone.

 Requires the OtherPassword privilege.

 Returns the receiver."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  aVal ifNotNil:[ aVal _validateClass: DateTime ].
  self _validateUserProfileWrite ifTrue:[
    self _validateAuthenticationSchemeIsGemStone ifTrue:[
      self _securityDataInstVarAt: OC_USER_SECUR_DATA_LAST_LOGIN_TIM put: aVal .
    ].
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ self

]

{ #category : 'Accessing' }
UserProfile >> lastPasswordChange [

"Returns a DateTime that specifies when the password was last changed.

 Raises an error if the authentication scheme of the receiver is not #GemStone.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    self _validateAuthenticationSchemeIsGemStone ifTrue:[
      result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_LAST_PWD_CHNG .
    ].
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Accessing' }
UserProfile >> ldapBaseDn [

"If the receiver is using LDAP authentication, returns a copy of the base
 distinguished name (DN) string.  Otherwise returns nil.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_LDAP_BASE_DN .
  ]
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'External Authentication - Private' }
UserProfile >> ldapBaseDn: aString [

"Private - customers should not call this method directly."

self _securityDataInstVarAt: OC_USER_SECUR_DATA_LDAP_BASE_DN put: aString .
^ self

]

{ #category : 'External Authentication - LDAP' }
UserProfile >> ldapBaseDn: aString searchFilterDn: stringOrNil [

"Specify the base distinguished name (DN) and optionally the search filter
 DN for LDAP logins done by the receiver.  Has no meaningful effect unless
 LDAP authenticaion is enabled for the receiver.

 Requires the OtherPassword privilege."


<primitive: 2001>
| prot |
prot := System _protectedMode .

[ self _validatePrivilege ifTrue:[
    (self _validateBaseDn: aString searchDn: stringOrNil ) ifTrue:[
       self ldapBaseDn: aString ; ldapFilterDn: stringOrNil .
    ]
  ]
] ensure:[
prot _leaveProtectedMode  "exit protected mode"].

^ self

]

{ #category : 'External Authentication - Private' }
UserProfile >> ldapFilterDn: aString [

"Private - customers should not call this method directly."

self _securityDataInstVarAt: OC_USER_SECUR_DATA_LDAP_FILTER put: aString .
^ self

]

{ #category : 'Accessing' }
UserProfile >> ldapSearchFilterDn [

"If the receiver is using LDAP authentication, returns a copy of the search
 filter string used to resolve the receiver complete distinguished name
 during LDAP authentication.  Otherwise returns nil.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |

prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_LDAP_FILTER .
  ]
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Session Initialization' }
UserProfile >> loginHook [
  "Returns the method or block installed by loginHook:  or nil ."
  ^ loginHook
]

{ #category : 'Session Initialization' }
UserProfile >> loginHook: blockOrSymbol [
  "block must be one of:
     nil - removes any previously specified block or method.
     a zero argument ExecBlock
     the selector of a zero argument instance method in UserProfile .

   If not nil, then after authentication of the receiver,
   at the start of each session using the receiver,
   the specified Block or method is invoked.
   Invocation is from GsCurrentSession >> initialize,
   after the ProcessorScheduler is initalized, and
   after the GsPackagePolicy is initialized (in the absense
   of customer modifications to GsCurrentSession >> initialize )."

  | meth nArgs |
  blockOrSymbol
    ifNotNil: [
      blockOrSymbol _isSymbol
        ifTrue: [
          meth := self class
            compiledMethodAt: blockOrSymbol
            environmentId: 0
            otherwise: nil
            usePackages: false.
          meth
            ifNil: [ ArgumentError signal: 'In loginHook:, method ' , blockOrSymbol , ' not found' ]
            ifNotNil: [
              (nArgs := meth numArgs) == 0
                ifFalse: [
                  ArgumentError
                    signal:
                      'Symbol argument to loginHook: must specify a zero argument method' ] ] ]
        ifFalse: [
          blockOrSymbol _isExecBlock
            ifTrue: [
              (nArgs := blockOrSymbol numArgs) == 0
                ifFalse: [
                  ArgumentError
                    signal:
                      'block argument to loginHook: must be a zero argument ExecBlock' ] ]
            ifFalse: [
              blockOrSymbol
                _validateClasses:
                  {Symbol.
                  ExecBlock} ] ] ].
  loginHook := blockOrSymbol

]

{ #category : 'Accessing' }
UserProfile >> loginsAllowedBeforeExpiration [

"Returns the number of logins allowed before the receiver's password will
 expire.  Zero or nil means unlimited logins.  A positive number is the number
 remaining before the password will expire.  A negative number means the account
 has expired.  The internal value is reset to nil when the password is
 changed.

 This method will always return nil if the authentication scheme of the
 receiver is not #GemStone.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    self authenticationSchemeIsGemStone ifTrue:[
      result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_LOGINS_BEFORE_EXPIR .
    ].
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Updating' }
UserProfile >> loginsAllowedBeforeExpiration: aPositiveInteger [

"Sets the number of logins allowed using the receiver before the receiver's
 password will expire.  Zero means unlimited logins.  A positive number is
 the number of logins to be allowed before the current password will expire.
 The internal value is reset to nil when the password is next changed.

 Has no effect if the receiver is one of the special UserProfiles
 (SystemUser, SymbolUser, DataCurator, GcUser, and Nameless), which are always
 immune to login expiration.

 Has no effect on UserProfiles which do not use GemStone authentication.

 Raises an error if the authentication scheme of the receiver is not #GemStone.

 Generates an error if you do not have the #OtherPassword privilege."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  self _validateAuthenticationSchemeIsGemStone ifTrue:[
    self _validatePrivilege ifTrue:[
      aPositiveInteger ifNotNil:[
        aPositiveInteger _isSmallInteger ifFalse:[
          ^ aPositiveInteger _validateClass: SmallInteger .
        ].
        aPositiveInteger < 0 ifTrue:[
          ^ aPositiveInteger _error: #rtErrArgNotPositive
        ].
      ].
      self _securityDataInstVarAt: OC_USER_SECUR_DATA_LOGINS_BEFORE_EXPIR put: aPositiveInteger .
    ].
  ].
] ensure:[
  prot _leaveProtectedMode
]

]

{ #category : 'Repository Conversion' }
UserProfile >> migrateSecurityData [

"Migrate the UserSecurityData object to the correct size for this release
 of GemStone/64.  Returns true if the migration was performed, false if
 no migration is required.

 This method may only be executed by SystemUser."

<primitive: 2001>
| prot |
prot := System _protectedMode .
^ [
  self _securityDataInstVarAt: -1 put: nil .
] ensure:[
  prot _leaveProtectedMode
].

]

{ #category : 'Accessing' }
UserProfile >> nativeLanguage [

"Returns a Symbol specifying the user's native language.  This value
 is no longer used by the base system, but is left for use by applications."

^ (self resolveSymbol: #NativeLanguage) ifNotNil:[ :assoc | assoc _value ]
					 ifNil:[ #English ]

]

{ #category : 'Updating' }
UserProfile >> nativeLanguage: aLanguageSymbol [

"Redefines the user's native language to be aLanguageSymbol.
 aLanguageSymbol must be a Symbol or DoubleByteSymbol .

 If you use this method, you must commit, and the change will
 take effect in sessions which login using the receiver after the commit."

self _validatePrivilege ifTrue:[
  (aLanguageSymbol _validateClass: Symbol) ifTrue:[
    ((self resolveSymbol: #UserGlobals) value)
       at: #NativeLanguage
       put: aLanguageSymbol
  ].
].

]

{ #category : 'Accessing the Symbol List' }
UserProfile >> objectNamed: aSymbol [

"Returns the first object in the receiver's symbol list that has the given
 name.  If no object is found with the given name, this returns nil."

| assn |
assn := symbolList resolveSymbol: aSymbol.
assn == nil ifTrue: [ ^nil ].
^assn value

]

{ #category : 'Updating' }
UserProfile >> oldPassword: firstString
newPassword: secondString [

"Modifies the receiver's password to be secondString.  Generates an error if
 either argument is not a String, if firstString is not the receiver's
 password, or if the receiver is not the UserProfile of the current session.
 Generates an error if secondString is equivalent to the userId of the
 receiver.  The new password (secondString) may not be longer than
 1024 Characters and may not be an empty string.

 Raises an #rtErrAuthIllegalPasswordChange error if the authentication mode of
 the receiver is not set to #GemStone or the receiver is disabled.

 This method allows you to change your own password.  To change the password of
 a disabled account or any other user, use password: instead.

 This method requires the UserPassword privilege."


<primitive: 2001> "enter protected mode"
| prot |
prot := System _protectedMode .
[
  self == System myUserProfile ifFalse:[
    ^ self _error: #rtErrCantChangePassword
  ].
  (self _validatePrivilegeName: #UserPassword ) ifTrue:[
    "check old password"
    firstString _validateClass: String .
    secondString _validateClass: String .
    (self validatePassword: firstString) ifFalse:[
        ^ userId _error: #rtErrBadPassword.
    ].
    (self _validateNewPassword:  secondString) ifTrue:[
      (AllUsers disallowUsedPasswords and:[ firstString = secondString]) ifTrue:[
        SecurityError signal:'A password may not have the value ', secondString .
      ].
      self _changePasswordFrom: firstString to: secondString .
    ].
  ].
] ensure:[
  prot _leaveProtectedMode
]

]

{ #category : 'Updating' }
UserProfile >> password: aString [

"Modifies the receiver's password to be aString, and returns the receiver.  If
 the argument is not a String, generates an error.
 Generates an error if aString is equivalent to the userId of the
 receiver.  aString may not be longer than 1024 Characters and may not be an
 empty string. In order to login, the characters in aString should all be
 within the 8-bit range, but this is not enforced.

 This method allows you to change the password of another GemStone user.
 To change your own password, use oldPassword:newPassword: instead.

 Raises an #rtErrAuthIllegalPasswordChange error if the authentication mode of
 the receiver is not set to #GemStone.

 This method requires the #OtherPassword privilege.

 If the repository was restored using
    Repository>>restoreFromBackup:newSystemUserPassword: (or equivalent)
 with a non-nil SystemUser password, then Repository>>commitRestore 
 must occur and then a login by SystemUser must occur before this method 
 can be executed for SystemUser's UserProfile by a user other than SystemUser."

<primitive: 2001> "enter protected mode"
| prot |
prot := System _protectedMode .
[
  self _validatePrivilege
    ifTrue:[ self _setPassword: aString ].
] ensure:[
  prot _leaveProtectedMode
].

]

{ #category : 'Accessing' }
UserProfile >> passwordAgeLimit [

"Answers the maximum number of hours for which the receiver can retain a password.
 When a password is set, it expires this number of hours later.  The user must change
 the password before it expires, or else GemStone disables the account.  Once
 a password has expired, an administrator must reset the password from another
 account before the user can login again.

 Returns nil if the repository-wide passwordAgeLimit, if any, will be used to
 enforce the password age limit.

 Returns zero if password aging is disabled for the receiver.

 Sending this message to any of the special UserProfiles (SystemUser, SymbolUser,
 DataCurator, GcUser, and Nameless) always returns zero or an error.

 Raises an error if the authentication scheme of the receiver is not #GemStone.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    self _validateAuthenticationSchemeIsGemStone ifTrue:[
      result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_PASSWD_AGE_LIMIT .
    ]
  ]
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Updating' }
UserProfile >> passwordAgeLimit: numberOfHours [

"Sets the maximum number of hours for which the receiver can retain a password.
 When a password is set, it expires this number of hours later.  The user must change
 the password before it expires, or else GemStone disables the account.  Once
 a password has expired, an administrator must reset the password from another
 account before the user can login again.

 Setting this value to nil causes the receiver to use the repository-wide
 passwordAgeLimit setting in AllUsers, if enabled.

 Setting this value to zero disables password aging for the receiver.

 Requires the OtherPassword privilege.

 Has no effect if the receiver is one of the special UserProfiles
 (SystemUser, SymbolUser, DataCurator, GcUser, and Nameless), which are always
 immune to the password age limit.

 Raises an error if the authentication scheme of the receiver is not #GemStone.

 Returns the receiver."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  self _validatePrivilege ifTrue:[
  self _validateAuthenticationSchemeIsGemStone ifTrue:[
    (self class isSpecialUserId: userId) ifFalse:[
      numberOfHours == nil ifFalse: [
        (numberOfHours _validateClasses: { SmallInteger }) ifFalse: [^ nil ].
        numberOfHours < 0 ifTrue: [^numberOfHours _error: #rtErrArgNotPositive].
      ].
    self _securityDataInstVarAt: OC_USER_SECUR_DATA_PASSWD_AGE_LIMIT put: numberOfHours ] .
    ]
  ]
] ensure:[ prot _leaveProtectedMode "exit protected mode"].
^ self

]

{ #category : 'Accessing' }
UserProfile >> passwordAgeWarning [

"The maximum number of hours prior to a password expiration time for which
 a login as the receiver can succeed without a warning.  If the user logs in
 to GemStone within this number of hours before the password is due to expire,
 GemStone issues a warning about the impending expiration.  This feature grants
 a user the opportunity to change the password conveniently and to prevent the
 account from being disabled.

 Returns nil if the repository-wide passwordAgeWarning, if any, will be used to
 generate password age warnings.

 Returns zero if password age warning is disabled for the receiver.

 Sending this message to any of the special UserProfiles (SystemUser, SymbolUser,
 DataCurator, GcUser, and Nameless) always returns zero or an error.

 Raises an error if the authentication scheme of the receiver is not #GemStone.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    self _validateAuthenticationSchemeIsGemStone ifTrue:[
      result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_PASSWD_AGE_WARNING .
    ]
  ]
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Updating' }
UserProfile >> passwordAgeWarning: numberOfHours [

"Sets the maximum number of hours prior to a password expiration time for which
 a login as the receiver can succeed without a warning.  If the user logs in
 to GemStone within this number of hours before the password is due to expire,
 GemStone issues a warning about the impending expiration.  This feature grants
 a user the opportunity to change the password conveniently and to prevent the
 account from being disabled.

 Has no effect if the a password age limit is not set for the receiver.

 Setting this value to nil causes the receiver to use the repository-wide
 passwordAgeWarning setting, if is enabled.

 Setting this value to zero disables the password age warning for the receiver.

 Requires the OtherPassword privilege.

 Has no effect if the receiver is one of the special UserProfiles
 (SystemUser, SymbolUser, DataCurator, GcUser, and Nameless), which are always
 immune to the password age warning.

 Raises an error if the authentication scheme of the receiver is not #GemStone.

 Returns the receiver."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  self _validatePrivilege ifTrue:[
    self _validateAuthenticationSchemeIsGemStone ifTrue:[
      (self class isSpecialUserId: userId) ifFalse:[
        numberOfHours == nil ifFalse: [
          (numberOfHours _validateClasses: { SmallInteger }) ifFalse: [^ nil ].
          numberOfHours < 0 ifTrue: [^numberOfHours _error: #rtErrArgNotPositive].
        ].
        self _securityDataInstVarAt: OC_USER_SECUR_DATA_PASSWD_AGE_WARNING put: numberOfHours
      ] .
    ]
  ]
] ensure:[ prot _leaveProtectedMode "exit protected mode"].
^ self

]

{ #category : 'Accessing' }
UserProfile >> passwordNeverExpires [

"Returns true if the receiver is an account that is immune from password
 expiration, or if the receiver uses an authentication scheme other than
 #GemStone.

 The special UserProfiles (SystemUser, SymbolUser, DataCurator, GcUser,
 and Nameless) are immune to password expiration and always answer true.

 Generates an error if the receiver is not your UserProfile and you 
 have neither #OtherPassword nor #ReadOtherUserProfile  privilege."

self authenticationSchemeIsGemStone
  ifFalse:[ ^ true] .

(self class isSpecialUserId: userId)
  ifTrue:[ ^ true ].

^self _fetchPasswordNeverExpires

]

{ #category : 'Accessing Privileges' }
UserProfile >> privileges [

"Returns an Array of Symbols describing the receiver's privileges.  Those
 privilege Symbols specify the user's level of access to certain privileged
 system functions ordinarily performed by the GemStone data curator.

 See individual methods for specific required privileges.  For more
 information about privileges, see the GemStone Programming Guide.

 The following privileges are granted to UserProfiles, to allow them to
 perform privileged methods:

 DefaultObjectSecurityPolicy
   DefaultSegment - translated to DefaultObjectSecurityPolicy
 FileControl
 GarbageCollection
 OtherPassword
 ObjectSecurityPolicyCreation
   SegmentCreation - translated to ObjectSecurityPolicyCreation
 ObjectSecurityPolicyProtection
   SegmentProtection - translated to ObjectSecurityPolicyProtection
 SessionAccess
 SystemControl
 UserPassword
 CodeModification
 SessionPriority
 CompilePrimitives
 ChangeUserId
 MigrateObjects
 DisableObjectReadLogging 
 DynamicDisableObjectReadLogging
 ReadOtherUserProfile
 CreateOnetimePassword

 The following four privileges are inverse privilege; the operations can be
 performed unless the privilege is given to the user.  UserProfiles with
 these privileges cannot perform the function.

 NoPerformOnServer
    prevents perform of System class >> performOnServer: and
    GsHostProcess >> fork  unless the command argument appears on a
    allowlist of allowed server commands. Also affects methods in
    class GsSysLog.

 NoUserAction
    prevents loading of any user action library and prevents use of the FFI.

 NoGsFileOnServer
    prevents any GsFile operation which accesses a file on the server,
    i.e. access from the gem process to the file system.

 NoGsFileOnClient
    prevents any GsFile operation which accesses a file on the client,
    i.e. access from the remote GCI process to the file system.

 The state of NoUserAction, NoGsFileOnServer, and NoGsFileOnClient are
 cached in the Virtual Machine at session login, and changes to a
 UserProfile for these 3 privileges will only be seen in sessions that
 login after those changes have been committed.

 "

| result |

result := { } .

1 to: (PrivilegeNames size) do: [:counter |
    (privileges bitAt: counter) = 1 ifTrue: [
       result add: (PrivilegeNames at: counter).
    ].
].
^result

]

{ #category : 'Updating Privileges' }
UserProfile >> privileges: anArrayOfStrings [

"Redefines the receiver's privileges to be those specified by anArrayOfStrings.
 Any privileges not contained in anArrayOfStrings will be turned off.
 Generates an error if any element of anArrayOfStrings is not a valid privilege
 name (as defined in the class variable PrivilegeNames).

 Only SystemUser may send this message to special UserProfiles. Attempts which
 violate this restriction will raise an exception. SystemUser automatically has
 all privileges, regardless of any defined privileges.

Example: System myUserProfile privileges: #('SessionAccess' 'DefaultObjectSecurityPolicy')"


| privSet myUserProfile |
self _validatePrivilege ifFalse:[ ^ nil ].
(self class isSpecialUserId: userId) ifTrue:[
  System iAmSystemUser
    ifFalse:[ ^ self _error: #rtErrMustBeSystemUser ] .
] .

myUserProfile := ( self == System myUserProfile ).
privSet := UserProfile _privilegesSetFromArray: anArrayOfStrings .

" clear all privileges -- unless this is me,
  in which case leave #OtherPassword on "

myUserProfile
    ifTrue: [ privileges :=
              ( 1 bitShift: ( PrivilegeNames indexOf: #OtherPassword ) - 1 ) ]
    ifFalse: [ privileges := 0 ].

privSet do: [ :aPrivSymbol |
   self addPrivilege: aPrivSymbol .
   ].

" if me, clean up the #OtherPassword flag "
myUserProfile ifTrue: [
    ( privSet includes: #OtherPassword ) ifFalse: [
        self deletePrivilege: #OtherPassword ] ].

]

{ #category : 'Accessing' }
UserProfile >> reasonForDisabledAccount [

"Returns the String that contains the reason why the receiver's password
 has been automatically disabled or expired. Some possible strings are:
  'LoginLimitReached'
     The account's password in no longer valid because its login limit
     was reached.  See UserProfile>>loginsAllowedBeforeExpiration.
  'LoginsWithInvalidPassword'
     The account is disabled due to an attempt to login to it too many
     times with an incorrect password.  See the stone configuration
     option STN_DISABLE_LOGIN_FAILURE_LIMIT.
  'PasswordAgeLimit'
     The account's password is no longer valid due to it being too
     old as determined by UserProfileSet>>passwordAgeLimit.
  'StaleAccount'
     The account is disabled due to it being too old
     as determined by UserProfileSet>>staleAccountAgeLimit.
  'Disabled by administrator'
     The account was disabled by an Administrator using the
     UserProfile>>disable method.

 Returns nil if the account has not been disabled.

 Raises an error if the receiver is not your UserProfile and you do not
 have the #OtherPassword privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_REASON_DISABLED .
  result ifNil:[ result := self _pendingReasonForDisabledAccount ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Updating' }
UserProfile >> reasonForDisabledAccount: aString [

"Updates the String that contains the reason why the receiver's password
 has been automatically disabled or expired.

 Raises an error if you do not have the #OtherPassword privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validatePrivilege ifTrue:[
   (aString isKindOf: String) ifFalse:[
     ^ aString _validateClass: String
    ].
    "Note: primitive now stores a copy of aString"
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_REASON_DISABLED put: aString .
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'External Authentication' }
UserProfile >> reenable [
"Resumes previously disabled logins for the receiver which must be using
 external authentication.

 Raises an error if the account is using GemStone authentication. Use the
 #reenableWithPassword: method to reenable a UserProfile which uses GemStone
 authentication.

 Returns the receiver on success or raises an exception and returns nil
 on error.
 Requires the #OtherPassword privilege."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[ self _validatePrivilege ifFalse:[ ^ nil ].
  self _validateAuthenticationSchemeIsExternal ifFalse:[ ^ nil ] .
  self isDisabled ifFalse:[ ^ self ] . "No action needed"
  self _enableNonGemStoneAuthentication .
] ensure:[ prot _leaveProtectedMode  "exit protected mode"].

^ self

]

{ #category : 'Updating' }
UserProfile >> reenableWithPassword: aString [
"Resumes previously disabled logins for the receiver, which must be using
 GemStone authentication.

 Raises an error if the account is using external authentication. Use the
 #reenable method to re-enable a UserProfile which uses external
 authentication.

 Returns the receiver on success or raises an exception and returns nil
 on error.
 Requires the #OtherPassword privilege."

^ self _validateAuthenticationSchemeIsGemStone
    ifTrue:[ self password: aString. self ]
   ifFalse:[ nil ].

]

{ #category : 'Updating the Symbol List' }
UserProfile >> removeDictionaryAt: anIndex [

"Removes the SymbolDictionary at position anIndex from the receiver's symbol
 list, thus decreasing the size of the receiver's symbol list by one,
 and, if the receiver is identical to 'GsSession currentSession
 userProfile', removes the SymbolDictionary at position anIndex from the
 transient symbol list's symbol list (subject to the constraints below).
 Returns the receiver's dictionaryNames String.

 Generates an error if anIndex is not both a positive integer and less than the
 size of the receiver's symbol list.  If an error occurs in removing
 from the receiver's symbol list, the transient symbol list is left
 alone, and the error is handled immediately.

 If the removal from the receiver's symbol list succeeds, search in the
 transient symbol list for a SymbolDictionary identical to the one
 removed from the transient symbol list, and if found, remove it from
 the list.  Removes only the first such element.  If no such dictionary
 is found, silently return.

 Note if the transient and persistent lists have different contents when
 an abort transaction occurs they will not be automatically synchronized
 after the abort.  The persistent list will be rolled back to the
 committed state, but the transient list will not be rolled back."

"Example: System myUserProfile insertDictionary: (SymbolDictionary new) at: 3.
                  System myUserProfile dictionaryNames.
                  System myUserProfile removeDictionaryAt: 3.
                  System myUserProfile dictionaryNames."

| dict |
(self == System myUserProfile
  ifTrue: [ self _validatePrivilegeName: #CodeModification  ]
  ifFalse: [ self _validatePrivilege]) ifTrue:[

   (anIndex _validateClass: Integer) ifTrue:[ | ses |
     dict := self symbolList removeAtIndex: anIndex.
     ses := GsSession currentSession .
     (self == ses userProfile) ifTrue: [
        ses _transientSymbolList ifNotNil:[ :list |
             list removeIdentical: dict ifAbsent: [ ]
        ]
     ].
  ].
].
^symbolList namesReport.

]

{ #category : 'Group Membership' }
UserProfile >> removeFromAllGroups [

"Removes the receiver from all groups of which it is a member.
 Answers the receiver."

self groups do: [:aGroup | self removeFromUserProfileGroup: aGroup].
^ self

]

{ #category : 'Group Membership' }
UserProfile >> removeFromUserProfileGroup: aUserProfileGroup [

"Removes the user from the group UserProfileGroup.
 If the user does not belong to the group, no action occurs."

self _validatePrivilege ifTrue:[
  groups size > 0 ifTrue:[
    groups remove: aUserProfileGroup ifAbsent:[] .
    aUserProfileGroup remove: self ifAbsent: [] .
  ].
].

]

{ #category : 'Group Membership' }
UserProfile >> removeGroup: aGroupString [

"Removes the user from the group represented by aGroupString.  If aGroupString
 is not a group defined in UserProfileGroup, generates an error.
 If the user does not belong to the group, no action occurs.

 Legacy  method; use removeFromUserProfilGroup: or
    UserProfileGroup >> removeUser: instead."

| theGroup |
(theGroup := UserProfileGroup _validateGroupString: aGroupString) ifNotNil:[
  self removeFromUserProfileGroup: theGroup
].

]

{ #category : 'Accessing the Symbol List' }
UserProfile >> resolveSymbol: aString [

"Searches the receiver's symbol list for an Association whose key is equal to
 aString, and returns that Association.  If no such Association is found in the
 symbol list, this returns nil."

^ symbolList resolveSymbol: aString

]

{ #category : 'Updating' }
UserProfile >> setPasswordNeverExpires: aBoolean [

"Makes the receiver immune from all forms of password expiration.  Once
 set, the receiver is never required to change the password and will
 never be denied access because too much time has passed since the last
 successful login.

 This method does not modify any password validity restrictions for
 the receiver such as maximum repeating characters, disallowed passwords,
 etc.

 Password expiration is always disabled for the special UserProfiles
 (SystemUser, SymbolUser, DataCurator, GcUser, and Nameless).  Attempting
 to enabled it for any of these UserProfiles will raise an error.  An
 attempt to disable password expiration for a special user profile is
 ignored.

 Running this method raises an error if the receiver is not using the
 #GemStone authentication scheme.

 Requires the #OtherPassword privilege."

self authenticationSchemeIsGemStone
  ifFalse:[ ^ self _error: #rtErrAuthIllegalOperation ].

(self class isSpecialUserId: userId)
  ifTrue:[ aBoolean  ifTrue:[ ^ self ] "Do nothing"
                    ifFalse:[ ^ self _error: #rtErrIllegalPasswordExpiration ].].

^ self _setPasswordNeverExpires: aBoolean

]

{ #category : 'Accessing' }
UserProfile >> staleAccountAgeLimit [

"The maximum number of hours for which the receiver will remain enabled without
 a login.  Once the user logs in for the first time, he or she has up to this
 number of hours to login again, or else GemStone disables the account.  Once
 the account has been disabled, an administrator must reset the password from
 another account before the user can login again.

 Returns nil if the repository-wide staleAccountAgeLimit, if any, will be used to
 suspend stale accounts.

 Returns zero if the stale account age limit is disbled for the receiver.

 Sending this message to any of the special UserProfiles (SystemUser, SymbolUser,
 DataCurator, GcUser, and Nameless) always returns zero or an error.

 Raises an error if the authentication scheme of the receiver is not #GemStone.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    self _validateAuthenticationSchemeIsGemStone ifTrue:[
      result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_STALE_ACCOUNT_AGE_LIMIT .
    ]
  ]
] ensure:[ prot _leaveProtectedMode ].
^ result

]

{ #category : 'Updating' }
UserProfile >> staleAccountAgeLimit: numberOfHours [

"Sets maximum number of hours for which the receiver will remain enabled without
 a login.  Once the user logs in, he or she has up to this number of hours to
 login again, or else GemStone disables the account.  Once the account has been
 disabled, an administrator must reset the password from another account before
 the user can login again.

 Setting this value to nil causes the receiver to use repository-wide
 staleAccountAgeLimit setting in AllUsers, if enabled.

 Setting this value to zero disables the stale account age limit for the
 receiver.

 Requires the OtherPassword privilege.

 Has no effect if the receiver is one of the special UserProfiles
 (SystemUser, SymbolUser, DataCurator, GcUser, and Nameless), which are always
 immune to the stale account limit.

 Raises an error if the authentication scheme of the receiver is not #GemStone.

 Returns the receiver."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  self _validatePrivilege ifTrue:[
    self _validateAuthenticationSchemeIsGemStone ifTrue:[
      (self class isSpecialUserId: userId) ifFalse:[
         numberOfHours == nil ifFalse: [
           (numberOfHours _validateClasses: { SmallInteger }) ifFalse: [^ nil ].
           numberOfHours < 0 ifTrue: [^numberOfHours _error: #rtErrArgNotPositive].
         ] .
         self _securityDataInstVarAt: OC_USER_SECUR_DATA_STALE_ACCOUNT_AGE_LIMIT put: numberOfHours
      ] .
    ]
  ]
] ensure:[ prot _leaveProtectedMode "exit protected mode"].
^ self

]

{ #category : 'Accessing the Symbol List' }
UserProfile >> symbolList [

"Whenever the compiler is invoked, tokens in the method's source are bound to
 either Symbols found in the source, or to SymbolAssociations found in one of a
 number of SymbolDictionary passed to the compiler.  This method returns
 the Array of such SymbolDictionary that are associated with the
 receiver."

^ symbolList

]

{ #category : 'Updating the Symbol List' }
UserProfile >> symbolList: aSymbolList [

"Modifies the list of SymbolDictionaries associated with the receiver.
 If the receiver is identical to 'GsSession currentSession userProfile',
 replaces the contents of the session transient symbol
 list with the contents of aSymbolList.  If an error occurs as a result
 of the modification to the persistent symbol list, the transient list
 is left unmodified and the error is handled immediately.  If an error
 occurs as a result of the modification of the transient symbol list,
 the persistent symbol list is left in its new state, the transient
 symbol list is left in its old state, and the error is silently ignored.

 Note that if the transient and persistent lists have different contents when
 an abort transaction occurs they will not be automatically synchronized
 after the abort.  The persistent list will be rolled back to the
 committed state, but the transient list will not be rolled back."

"relax privilege to not require #OtherPassword if doing this to ourself"
(self == System myUserProfile
  ifTrue: [ self _validatePrivilegeName: #CodeModification ]
  ifFalse: [ self _validatePrivilege ]) ifTrue:[

  (aSymbolList _validateClass: SymbolList ) ifTrue:[ | ses |
      symbolList ifNil:[
         symbolList := aSymbolList
      ] ifNotNil:[ "must preserve old identity to avoid authorization errors"
         symbolList ~~ aSymbolList ifTrue:[		"fix 41658"
           symbolList size: 0.
           symbolList addAll: aSymbolList .
         ]
      ] .
      ses := GsSession currentSession .
      (self == ses userProfile) ifTrue: [
         [
           ses _transientSymbolList ifNotNil:[ :list |
             list replaceElementsFrom: symbolList
           ]
         ] onException: Error do:[:ex | ^ nil ].
      ].
  ].
].
^ self

]

{ #category : 'Accessing the Symbol List' }
UserProfile >> symbolResolutionOf: aString [

"Searches the receiver's symbol list for aString.  If aString is found, returns
 a formatted String describing the position in the symbol list of the
 Dictionary defining aString, the name of that Dictionary, and aString.

 Generates an error if aString is not defined for the receiver."

"Example: System myUserProfile symbolResolutionOf: #Integer"

^ self symbolList symbolResolutionOf: aString

]

{ #category : 'Accessing' }
UserProfile >> userId [

"Answers a String which is the user name of the receiver."

^ userId

]

{ #category : 'Updating' }
UserProfile >> userId: newUserIdString password: passwordString [

"Modifies the userId associated with the receiver to be newUserIdString,
 and installs the password passwordString .
 newUserIdString must not be the user ID of some other UserProfile in
 AllUsers, or an error will be raised.  Has no effect if newUserIdString
 is equal to the current userId of the receiver.

 newUserIdString must be a kind of String , with size <= 1024 characters .

 WARNING: Be careful when using this method with UserProfiles which use an
          external authentication scheme.  Executing this method will
          NOT change the userId used with external authentication if a user ID
          alias has been set.  See the method #userIdAlias: to set a
          alias for a UserProfile.

 Requires OtherPassword and ChangeUserId privileges."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[ | oldUserPro newUserId oldUserId duplicateUser|

  self _validatePrivilege ifFalse:[ ^ nil ].
  System myUserProfile _validatePrivilegeName: #ChangeUserId .
  (UserProfile _validateUserId: newUserIdString ) ifFalse:[ ^ nil ].

  (self class _validate__password: passwordString forUserId: newUserIdString)
    ifFalse:[ ^ nil ] .

  (self class isSpecialUserId: userId)
    ifTrue:[ ArgumentError signal: 'illegal attempt to modify a special UserProfile' ].

  (self class isSpecialUserId: newUserIdString)
    ifTrue:[ ArgumentError signal: 'illegal attempt to assume the name of a special UserProfile' ].

  oldUserId := userId .
  newUserId := newUserIdString asString .
  oldUserId = newUserId ifTrue:[
    ^ self
    ].
  oldUserPro := AllUsers userWithId: oldUserId ifAbsent:[
    ArgumentError signal:'userId ' , oldUserId , ' not found in AllUsers'
  ].

  duplicateUser := AllUsers userWithId: newUserId ifAbsent:[ nil ].
  duplicateUser ifNotNil:[
    ^ self _error: #rtErrUserIdAlreadyExists args: { newUserIdString }
    ].

  passwordString ifNotNil:[
    (oldUserPro _validate__password: passwordString) ifFalse:[ ^ nil ].
  ].
  AllUsers _oldUserId: oldUserId newUserId: newUserId for: self .
  userId := newUserId .
  oldUserPro password: passwordString .
  AllUsers addMsgToSecurityLog: '      userId ', oldUserId , ' changed to ',
	newUserId , ' and new password set'.
] ensure:[
  prot _leaveProtectedMode
]

]

{ #category : 'External Authentication' }
UserProfile >> userIdAlias [

"Returns a String which is the alias used by the receiver
 for external authentication, or nil if an alias is not used.

 Generates an error if the receiver is not your UserProfile and 
 you have neither the #OtherPassword nor #ReadOtherUserProfile privilege."

<primitive: 2001>
| prot result |
prot := System _protectedMode .
[ self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_ALIAS_USERID.
  ]
] ensure:[
  prot _leaveProtectedMode  "exit protected mode"
].
^ result

]

{ #category : 'External Authentication - Private' }
UserProfile >> userIdAlias: aString [

"Private - customers should not call this method directly.

 Sets the alias user ID for the receiver to be aString.  When external
 authentication is performed, the alias user ID will be used instead of
 the userId instance variable.  If no alias is specified, then the userId
 instance variable is used.

 If UNIX authentication is enabled for the receiver, then aString must
 be a valid UNIX user ID, otherwise an error will be raised.

 Requires the OtherPassword privilege."

self _securityDataInstVarAt: OC_USER_SECUR_DATA_ALIAS_USERID put: aString .
^ self

]

{ #category : 'Accessing' }
UserProfile >> usesOldPasswordEncryption [

"Answer a Boolean indicating if the receiver uses the old
 password encryption scheme.  The old encryption scheme was
 used in versions prior to 3.1.0.

 For upgraded repositories, passwords are automatically
 encrypted using the new scheme during the first successful
 login after upgrading to version 3.1.0 or later.

 Generates an error if the receiver is not your UserProfile and 
 you do not have #OtherPassword privilege."

<primitive: 2001>
| result prot |
result := false .
prot := System _protectedMode .
[
  self _validateUserProfileWrite ifTrue:[
    self authenticationSchemeIsGemStone ifTrue:[
      result := (self _securityDataInstVarAt: OC_USER_SECUR_DATA_SALT) == nil .
    ].
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'User Authorization' }
UserProfile >> validatePassword: aString [

"This method allows an application to validate an operation that requires
 authentication by multiple individuals.  Returns true if aString is the
 password of the receiver, returns false if the receiver is disabled or
 the password is incorrect.

 Generates an error if the argument is not a String.

 Generates an error if the receiver is not your UserProfile and you do not
 have the #OtherPassword privilege."

<primitive: 306>
aString _validateByteClass: CharacterCollection.
self _primitiveFailed: #validatePassword: args: { aString } .
self _uncontinuableError

]

{ #category : 'Storing and Loading' }
UserProfile >> writeTo: aPassiveObject [

"Instances of UserProfile cannot be converted to passive form.  This method
 writes nil to aPassiveObject and stops GemStone Smalltalk execution with a
 notifier."

  aPassiveObject writeObject: nil.
  self _error: #rtErrAttemptToPassivateInvalidObject.

]

{ #category : 'Accessing' }
UserProfile >> x509loginStatus [

  "Returns a SmallInteger with bits 16r1 isDisabled , 16r2 isReadOnly .
   The isDisabled bit does not include pending disable per
   UserProfile >> _pendingReasonForDisabledAccount.
   Requires SessionAccess privilege, does not require OtherPassword privilege."
<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  result := self _securityDataInstVarAt: OC_USER_SECUR_X509_STATUS.
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category: 'Private' }
UserProfile >> _primUpdatePerformOnServerCommands: anArray opCode: anInt [

"opcode 1  add elements of anArray to list of allowed commands
        2  remove elements of anArray from list of allowed commands
"
<primitive: 1124>
anArray _validateClass: Array .
anInt _validateClass: SmallInteger .
self _primitiveFailed: #_primUpdatePerformOnServerCommands:opCode:
     args: { anArray . anInt } .    
self _uncontinuableError
]

{ #category : 'Private' }
UserProfile >> _fetchPerformOnServerCommandList [

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_POS_WHITELIST
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result
]

{ #category: 'PerformOnServer' }
UserProfile >> performOnServerCommands [

"Answer an Array of Strings which represents a list of commands the receiver
 is permitted to use when calling either the System(C)>>performOnServer: or
 GsHostProcess>>fork: methods.

 Raises an exception if the receiver is not the caller's UserProfile and the
 caller has neither the #OtherPassword nor #ReadOtherUserProfile privilege."

^self _fetchPerformOnServerCommandList
]

{ #category: 'PerformOnServer' }
UserProfile >> addPerformOnServerCommands: anArray [

"Adds the commands in anArray to the list of commands the receiver is permitted
 to execute when calling either the System(C)>>performOnServer: or
 GsHostProcess>>fork methods when the receiver has the NoPerformOnServer
 privilege. Each element of anArray must be a non-empty instance of String,
 which represents the full path  (the first character of the string must
 be a forward slash [/]) to a file with execute permission. Command paths may
 not contain whitespace characters.

 Returns the receiver on success. Raises an exception if the caller does not have
 the OtherPassword privilege, if any element of anArray is empty, contains
 whitespace characters, or is not an instance of String.
 Raises an exception if any command string in anArray is already in the list."
 
^ self _primUpdatePerformOnServerCommands: anArray opCode: 1
]

{ #category: 'PerformOnServer' }
UserProfile >> addPerformOnServerCommand: aString [

"Adds the command in aString to the list of commands the receiver is permitted
 to execute when calling either the System(C)>>performOnServer: or
 GsHostProcess>>fork methods when the receiver has the NoPerformOnServer
 privilege. aString must be an instance of String which represents the full
 path  (the first character of the string must be a forward slash [/]) to a
 file with execute permission. Command paths may not contain whitespace
 characters.

 Returns the receiver on success. Raises an exception if the caller does not have
 the OtherPassword privilege, if aString contains whitespace characters, 
 or is not an instance of String. Raises an exception if the command string is
 already present in the list."

^ self addPerformOnServerCommands: { aString }
]

{ #category: 'PerformOnServer' }
UserProfile >> removePerformOnServerCommands: anArray [

"Removes the commands in anArray from the list of commands the receiver is
 permitted to execute when calling either the System(C)>>performOnServer: or
 GsHostProcess>>fork methods when the receiver has the NoPerformOnServer
 privilege. Each element of anArray must be a non-empty instance of String.

 Returns the receiver on success. Raises an exception if the caller does not
 have the OtherPassword privilege or any command in anArray is not present in
 the receiver's list of allowed commands."

^ self _primUpdatePerformOnServerCommands: anArray opCode: 2
]

{ #category: 'PerformOnServer' }
UserProfile >> removePerformOnServerCommand: aString [

"Removes the command aString from the list of commands the receiver is
 permitted to execute when calling either the System(C)>>performOnServer: or
 GsHostProcess>>fork methods when the receiver has the NoPerformOnServer
 privilege.

 Returns the receiver on success. Raises an exception if the caller does not
 have the OtherPassword privilege or aString is not present in the receiver's
 list of allowed commands."

^ self removePerformOnServerCommands: { aString }
]

{ #category: 'PerformOnServer' }
UserProfile >> removeAllPerformOnServerCommands [

"Removes all commands from the list of commands the receiver is permitted to
 use when calling either the System(C)>>performOnServer: or GsHostProcess>>fork
 methods. This method effectively disables these methods when the receiver has the
 NoPerformOnServer privilege.
 
 Returns the receiver on success. Raises an exception if the caller does not have
 the OtherPassword privilege."

^ self removePerformOnServerCommands: self performOnServerCommands
]

{ #category: 'PerformOnServer' }
UserProfile >> enableSystemLogAccess [

"Allows the receiver to write messages to the system log by executing privileged methods
 in class GsSysLog. Has no effect unless the receiver also has the NoPerformOnServer
 privilege.

 Returns the receiver on success. Raises an exception if the caller does not have
 the OtherPassword privilege or if the receiver already has GsSysLog access."

^ self addPerformOnServerCommand: self class gsSysLogFileName
]

{ #category: 'PerformOnServer' }
UserProfile >> disableSystemLogAccess [

"Disables the receiver from writing messages to the system log via executing 
 privileged methods in class GsSysLog. Has no effect unless the receiver also has
 the NoPerformOnServer privilege.

 Returns the receiver on success. Raises an exception if the caller does not have
 the OtherPassword privilege or if the receiver does not have GsSysLog access."

^ self removePerformOnServerCommand: self class gsSysLogFileName
]

{ #category: 'PerformOnServer' }
UserProfile >> hasSystemLogAccess [

"Returns a boolean indicating if the receiver is allowed to write to the system log
 via the methods in class GsSysLog."

 self isSystemUserProfile ifTrue:[ ^ true ] .
 (self privileges includesIdentical: #NoPerformOnServer) ifFalse:[ ^ true ] .
 ^ self performOnServerCommands includes: self class gsSysLogFileName
]

{ #category: 'PerformOnServer' }
UserProfile class >> gsSysLogFileName [
 ^ '/GsSysLog/writeLogMessage'
]

{ #category : 'Private' }
UserProfile >> isSystemUserProfile [
 "Return true if the receiver is the profile for SystemUser"
 ^ userId = 'SystemUser'
]

{ #category : 'Onetime Passwords' }
UserProfile >> onetimePasswordUserProfiles [

"Answer an IdentitySet of UserProfiles which is a copy of the allowlist
 of UserProfiles for which the receiver may generate onetime passwords.

 SystemUser may generate a onetime password for any UserProfile except 
 SystemUser.

 Raises an exception if the receiver is not the caller's UserProfile and the
 caller has neither the #OtherPassword nor #ReadOtherUserProfile privilege."

^ self isSystemUserProfile
	ifTrue:[ (IdentitySet withAll: AllUsers) removeIdentical: self ; yourself ]
	ifFalse:[ self _fetchOnetimePasswordUserProfiles ]
	
]

{ #category : 'Onetime Passwords' }
UserProfile >> addOnetimePasswordUserProfiles: anArray [

"Adds the contents of anArray to the allowlist of UserProfiles for which
 the receiver may create onetime passwords. anArray must be an instance 
 of Array which contains only one or more instances of UserProfile.
 UserProfiles appearing more than once in anArray are ignored as are
 entries identical to the receiver.

 An error is raised if anArray contains objects which are not instances of
 UserProfile.

 Requires the #OtherPassword privilege.

 Returns the receiver on success."

^ self _primUpdateOnetimePasswordList: anArray opCode: 1

]

{ #category : 'Onetime Passwords' }
UserProfile >> addOnetimePasswordUserProfile: aUserProfile [

"Adds a aUserProfile to the allowlist of UserProfiles for which the
 receiver may create onetime passwords. No action is taken if aUserProfile
 is already in the receiver's allowlist or if aUserProfile is identical
 to the receiver.

 An error is raised if aUserProfile is not an instance of UserProfile.

 Requires the #OtherPassword privilege.

 Returns the receiver on success."

^ self addOnetimePasswordUserProfiles: { aUserProfile }

]

{ #category : 'Onetime Passwords' }
UserProfile >> removeOnetimePasswordUserProfiles: anArray [

"Removes the contents of anArray from the allowlist of UserProfiles for
 which the receiver may create onetime passwords. anArray must be an
 instance of Array which contains one or more instances of UserProfile only.
 UserProfiles appearing more than once in anArray are ignored as are
 elements which are not present in the receiver's allowlist.

 An error is raised if anArray contains objects which are not instances of
 UserProfile.

 Requires the #OtherPassword privilege.

 Returns the receiver on success."

^ self _primUpdateOnetimePasswordList: anArray opCode: 2

]

{ #category : 'Onetime Passwords' }
UserProfile >> removeOnetimePasswordUserProfile: aUserProfile [

"Removes aUserProfile from the allowlist of UserProfiles for which the
 receiver may create onetime passwords. aUserProfile must be an instance
 of UserProfile.

 No action is taken if aUserProfile is not in the receiver's allowlist.

 An error is raised if aUserProfile is not an instance of UserProfile.

 Requires the #OtherPassword privilege.

 Returns the receiver on success."

^ self removeOnetimePasswordUserProfiles: { aUserProfile }

]

{ #category : 'Private' }
UserProfile >> _primUpdateOnetimePasswordList: anArray opCode: aSmallInt [

<primitive: 245>
anArray _validateClass: Array .
aSmallInt _validateClass: SmallInteger .
^ self _primitiveFailed: #_primUpdateOnetimePasswordList:opCode: 
     args: { anArray . aSmallInt }

]

{ #category : 'Private' }
UserProfile >> _fetchOnetimePasswordUserProfiles [

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validateUserProfileRead ifTrue:[
    result := self _securityDataInstVarAt: OC_USER_SECUR_DATA_ONETIME_PW_WL
  ].
] ensure:[
  prot _leaveProtectedMode
].
^ result
]

{ #category : 'Onetime Passwords' }
UserProfile >> removeOnetimePasswordUserId: aUserId [

"Removes the UserProfile with aUserId from the allowlist of 
 UserProfiles for which the receiver may create onetime passwords. 
 aUserId must be an instance of String.

 No action is taken if the UserProfile is not in the receiver's 
 allowlist.

 An error is raised if no UserProfile matching aUserId exists.

 Requires the #OtherPassword privilege.

 Returns the receiver on success."

^ self removeOnetimePasswordUserProfile: (AllUsers userWithId: aUserId)

]

{ #category : 'Onetime Passwords' }
UserProfile >> addOnetimePasswordUserId: aUserId [

"Adds the aUserProfile with aUserId to the allowlist of UserProfiles
 for which the receiver may create onetime passwords. No action is taken
 if the UserProfile is already in the receiver's allowlist or if the 
 UserProfile is identical to the receiver. aUserId must be an instance 
 of String.

 An error is raised if no UserProfile matching aUserId exists.

 Requires the #OtherPassword privilege.

 Returns the receiver on success."

^ self addOnetimePasswordUserProfile: (AllUsers userWithId: aUserId)

]

{ #category : 'Onetime Passwords' }
UserProfile >> removeAllOnetimePasswordUserProfiles [

"Removes all entries from the allowlist of UserProfiles for which the
 receiver may create onetime passwords. Has no effect if the receiver
 is SystemUser or the allowlist is empty.

 Requires the #OtherPassword privilege.

 Returns the receiver on success."

^ self removeOnetimePasswordUserProfiles:
                   self onetimePasswordUserProfiles asArray

]
