"
This is the abstract superclass that contains a dialect-independent
implementation of the TimeZone behavior. It reads a binary file
generated by the Olsen ZoneInfo database.
"
Class {
	#name : 'TimeZoneInfo',
	#superclass : 'Object',
	#instVars : [
		'transitions',
		'leapSeconds',
		'stream',
		'types',
		'charcnt'
	],
	#classInstVars : [
		'default',
		'cache'
	],
	#category : nil
}

{ #category : 'samples' }
TimeZoneInfo class >> buildSample [
"
	See #sampleLosAngelesDataB for an example of what this gives.
	(TimeZoneInfo buildSample).
"

	| file data stream i |
	file := GsFile
		openOnServer: '/usr/share/lib/zoneinfo/America/Los_Angeles'
		mode: 'rb'.
	[
		data := file contents.
	] ensure: [
		file close.
	].
	stream := AppendStream on: String new.
	stream nextPutAll: '#('.
	i := 1.
	data do: [:each |
		stream space.
		each codePoint printOn: stream.
		(i := i + 1) \\ 20 == 0 ifTrue: [stream cr].
	].
	stream nextPutAll: ' ).'.
	^stream contents.

]

{ #category : 'cache' }
TimeZoneInfo class >> cache [

	^cache.

]

{ #category : 'cache' }
TimeZoneInfo class >> cacheAt: aString [
	| c |
	c := cache ifNil: [cache := SymbolDictionary new].
	^ c at: aString otherwise: nil .

]

{ #category : 'cache' }
TimeZoneInfo class >> cacheAt: aString put: aTimeZoneInfo [
	| c |
	c := cache ifNil: [cache := SymbolDictionary new].
	^c at: aString put: aTimeZoneInfo.

]

{ #category : 'singleton' }
TimeZoneInfo class >> default [

	default == nil ifTrue: [
		default := self sampleLosAngelesB.
	].
	^default.

]

{ #category : 'singleton' }
TimeZoneInfo class >> default: aTimeZoneInfo [

	default := aTimeZoneInfo.

]

{ #category : 'instance creation' }
TimeZoneInfo class >> fromStream: aStream [
  | res |
  (res := super new) initialize: aStream.
  ^ res

]

{ #category : 'instance creation' }
TimeZoneInfo class >> new [

"Disallowed.  To create a new TimeZone, use another instance creation method."

self shouldNotImplement: #new

]

{ #category : 'instance creation' }
TimeZoneInfo class >> new: anInteger [

"Disallowed.  To create a new TimeZone, use another instance creation method."

self shouldNotImplement: #new:

]

{ #category : 'other' }
TimeZoneInfo class >> sampleLosAngelesA [

	^self fromStream: (ReadStreamPortable on: self sampleLosAngelesDataA).

]

{ #category : 'other' }
TimeZoneInfo class >> sampleLosAngelesB [

	^self fromStream: (ReadStreamPortable on: self sampleLosAngelesDataB).

]

{ #category : 'other' }
TimeZoneInfo class >> sampleLosAngelesDataA [
	"without leap seconds and without 2007 fix"

^#( 84 90 105 102 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0
 0 0 0 0 4 0 0 0 4 0 0 0 0 0 0 0 185 0 0 0
 4 0 0 0 16 158 166 72 160 159 187 21 144 160 134 42 160 161 154 247
 144 203 137 26 160 210 35 244 112 210 97 38 16 214 254 116 32 216 128 173
 144 218 254 209 160 219 192 144 16 220 222 179 160 221 169 172 144 222 190 149
 160 223 137 142 144 224 158 119 160 225 105 112 144 226 126 89 160 227 73 82
 144 228 94 59 160 229 41 52 144 230 71 88 32 231 18 81 16 232 39 58
 32 232 242 51 16 234 7 28 32 234 210 21 16 235 230 254 32 236 177 247
 16 237 198 224 32 238 145 217 16 239 175 252 160 240 113 187 16 241 143 222
 160 242 127 193 144 243 111 192 160 244 95 163 144 245 79 162 160 246 63 133
 144 247 47 132 160 248 40 162 16 249 15 102 160 250 8 132 16 250 248 131
 32 251 232 102 16 252 216 101 32 253 200 72 16 254 184 71 32 255 168 42
 16 0 152 41 32 1 136 12 16 2 120 11 32 3 113 40 144 4 97 39
 160 5 81 10 144 6 65 9 160 7 48 236 144 7 141 67 160 9 16 206
 144 9 173 191 32 10 240 176 144 11 224 175 160 12 217 205 16 13 192 145
 160 14 185 175 16 15 169 174 32 16 153 145 16 17 137 144 32 18 121 115
 16 19 105 114 32 20 89 85 16 21 73 84 32 22 57 55 16 23 41 54
 32 24 34 83 144 25 9 24 32 26 2 53 144 26 242 52 160 27 226 23
 144 28 210 22 160 29 193 249 144 30 177 248 160 31 161 219 144 32 118 43
 32 33 129 189 144 34 86 13 32 35 106 218 16 36 53 239 32 37 74 188
 16 38 21 209 32 39 42 158 16 39 254 237 160 41 10 128 16 41 222 207
 160 42 234 98 16 43 190 177 160 44 211 126 144 45 158 147 160 46 179 96
 144 47 126 117 160 48 147 66 144 49 103 146 32 50 115 36 144 51 71 116
 32 52 83 6 144 53 39 86 32 54 50 232 144 55 7 56 32 56 28 5
 16 56 231 26 32 57 251 231 16 58 198 252 32 59 219 201 16 60 176 24
 160 61 187 171 16 62 143 250 160 63 155 141 16 64 111 220 160 65 132 169
 144 66 79 190 160 67 100 139 144 68 47 160 160 69 68 109 144 70 15 130
 160 71 36 79 144 71 248 159 32 73 4 49 144 73 216 129 32 74 228 19
 144 75 184 99 32 76 205 48 16 77 152 69 32 78 173 18 16 79 120 39
 32 80 140 244 16 81 97 67 160 82 108 214 16 83 65 37 160 84 76 184
 16 85 33 7 160 86 44 154 16 87 0 233 160 88 21 182 144 88 224 203
 160 89 245 152 144 90 192 173 160 91 213 122 144 92 169 202 32 93 181 92
 144 94 137 172 32 95 149 62 144 96 105 142 32 97 126 91 16 98 73 112
 32 99 94 61 16 100 41 82 32 101 62 31 16 102 18 110 160 103 30 1
 16 103 242 80 160 104 253 227 16 105 210 50 160 106 221 197 16 107 178 20
 160 108 198 225 144 109 145 246 160 110 166 195 144 111 113 216 160 112 134 165
 144 113 90 245 32 114 102 135 144 115 58 215 32 116 70 105 144 117 26 185
 32 118 47 134 16 118 250 155 32 120 15 104 16 120 218 125 32 121 239 74
 16 122 186 95 32 123 207 44 16 124 163 123 160 125 175 14 16 126 131 93
 160 127 142 240 16 0 1 0 1 2 3 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 255 255 157 144 1 0 255 255 143 128
 0 4 255 255 157 144 1 8 255 255 157 144 1 12 80 68 84 0 80 83
 84 0 80 87 84 0 80 80 84 0 0 0 0 1 0 0 0 1 ).

]

{ #category : 'other' }
TimeZoneInfo class >> sampleLosAngelesDataB [
	"without leap seconds and with 2007 fix"

^#( 84 90 105 102 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0
 0 0 0 0 4 0 0 0 4 0 0 0 0 0 0 0 185 0 0 0
 4 0 0 0 16 158 166 72 160 159 187 21 144 160 134 42 160 161 154 247
 144 203 137 26 160 210 35 244 112 210 97 38 16 214 254 116 32 216 128 173
 144 218 254 209 160 219 192 144 16 220 222 179 160 221 169 172 144 222 190 149
 160 223 137 142 144 224 158 119 160 225 105 112 144 226 126 89 160 227 73 82
 144 228 94 59 160 229 41 52 144 230 71 88 32 231 18 81 16 232 39 58
 32 232 242 51 16 234 7 28 32 234 210 21 16 235 230 254 32 236 177 247
 16 237 198 224 32 238 145 217 16 239 175 252 160 240 113 187 16 241 143 222
 160 242 127 193 144 243 111 192 160 244 95 163 144 245 79 162 160 246 63 133
 144 247 47 132 160 248 40 162 16 249 15 102 160 250 8 132 16 250 248 131
 32 251 232 102 16 252 216 101 32 253 200 72 16 254 184 71 32 255 168 42
 16 0 152 41 32 1 136 12 16 2 120 11 32 3 113 40 144 4 97 39
 160 5 81 10 144 6 65 9 160 7 48 236 144 7 141 67 160 9 16 206
 144 9 173 191 32 10 240 176 144 11 224 175 160 12 217 205 16 13 192 145
 160 14 185 175 16 15 169 174 32 16 153 145 16 17 137 144 32 18 121 115
 16 19 105 114 32 20 89 85 16 21 73 84 32 22 57 55 16 23 41 54
 32 24 34 83 144 25 9 24 32 26 2 53 144 26 242 52 160 27 226 23
 144 28 210 22 160 29 193 249 144 30 177 248 160 31 161 219 144 32 118 43
 32 33 129 189 144 34 86 13 32 35 106 218 16 36 53 239 32 37 74 188
 16 38 21 209 32 39 42 158 16 39 254 237 160 41 10 128 16 41 222 207
 160 42 234 98 16 43 190 177 160 44 211 126 144 45 158 147 160 46 179 96
 144 47 126 117 160 48 147 66 144 49 103 146 32 50 115 36 144 51 71 116
 32 52 83 6 144 53 39 86 32 54 50 232 144 55 7 56 32 56 28 5
 16 56 231 26 32 57 251 231 16 58 198 252 32 59 219 201 16 60 176 24
 160 61 187 171 16 62 143 250 160 63 155 141 16 64 111 220 160 65 132 169
 144 66 79 190 160 67 100 139 144 68 47 160 160 69 68 109 144 69 243 211
 32 71 45 138 16 71 211 181 32 73 13 108 16 73 179 151 32 74 237 78
 16 75 156 179 160 76 214 106 144 77 124 149 160 78 182 76 144 79 92 119
 160 80 150 46 144 81 60 89 160 82 118 16 144 83 28 59 160 84 85 242
 144 84 252 29 160 86 53 212 144 86 229 58 32 88 30 241 16 88 197 28
 32 89 254 211 16 90 164 254 32 91 222 181 16 92 132 224 32 93 190 151
 16 94 100 194 32 95 158 121 16 96 77 222 160 97 135 149 144 98 45 192
 160 99 103 119 144 100 13 162 160 101 71 89 144 101 237 132 160 103 39 59
 144 103 205 102 160 105 7 29 144 105 173 72 160 106 230 255 144 107 150 101
 32 108 208 28 16 109 118 71 32 110 175 254 16 111 86 41 32 112 143 224
 16 113 54 11 32 114 111 194 16 115 21 237 32 116 79 164 16 116 255 9
 160 118 56 192 144 118 222 235 160 120 24 162 144 120 190 205 160 121 248 132
 144 122 158 175 160 123 216 102 144 124 126 145 160 125 184 72 144 126 94 115
 160 127 152 42 144 0 1 0 1 2 3 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1
 0 1 0 1 0 1 0 1 0 1 255 255 157 144 1 0 255 255 143 128
 0 4 255 255 157 144 1 8 255 255 157 144 1 12 80 68 84 0 80 83
 84 0 80 87 84 0 80 80 84 0 0 0 0 1 0 0 0 1 ).

]

{ #category : 'other' }
TimeZoneInfo class >> sampleTokyo [

	^self fromStream: (ReadStreamPortable on: self sampleTokyoData).

]

{ #category : 'other' }
TimeZoneInfo class >> sampleTokyoData [
	"without leap seconds and without 2007 fix"

^#( 84 90 105 102 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0
 0 0 0 0 2 0 0 0 2 0 0 0 0 0 0 0 1 0 0 0
 2 0 0 0 8 195 206 133 112 1 0 0 126 144 0 0 0 0 126 144
 0 4 67 74 84 0 74 83 84 0 0 0 0 0 )

]

{ #category : 'public' }
TimeZoneInfo >> detectLastTransition: aBlock [
		"#'to:by:do: is slightly faster than #'reverseDo:' (#40674)"
	transitions size to: 1 by: -1 do: [:i |
		| each |
		each := transitions at: i.
		(aBlock value: each) ifTrue: [^each].
	].
	^nil.

]

{ #category : 'instance creation' }
TimeZoneInfo >> initialize: aStream [

	stream := aStream.
	self
		readHeader;
		readTransitions;
		readTypes;
		readAbbreviations;
		updateTransitions;
		readLeapSeconds .
	stream  := nil.
	types   := nil.
	charcnt := nil.

]

{ #category : 'instance creation' }
TimeZoneInfo >> nextByte [

	| object |
	object := stream next.
	(object _isInteger) ifTrue: [^object].
	(object isKindOf: Character) ifTrue: [^object codePoint].
	self error: 'invalid file format'.

]

{ #category : 'instance creation' }
TimeZoneInfo >> nextSignedInt [

	| b0 b1 b2 w h |
	b0 := self nextByte.
	b1 := self nextByte.
	b2 := self nextByte.
	w :=  self nextByte.

	"Minimize LargeInteger arithmetic"
	h := ((b0 bitAnd: 16r7F) - (b0 bitAnd: 16r80) bitShift: 8) + b1.
	b2 == 0 ifFalse: [w := (b2 bitShift: 8) + w].
	h == 0 ifFalse: [w := (h bitShift: 16) + w].
	^w

]

{ #category : 'instance creation' }
TimeZoneInfo >> nextUnsignedInt [

	^(self nextByte * 16r1000000) +
		(self nextByte * 16r10000) +
		(self nextByte *   16r100) +
		self nextByte.

]

{ #category : 'public' }
TimeZoneInfo >> offsetAtLocal: aDateAndTime [

	| transition |
	transition := self transitionAtLocal: aDateAndTime.
	transition == nil ifTrue: [
		transition := transitions
			detect: [:each | each isDST not]
			ifNone: [^0].
	].
	^transition offsetFromUTC.

]

{ #category : 'public' }
TimeZoneInfo >> offsetAtUTC:   aDateAndTime [
	| transition |
	transition := self transitionAtUTC: aDateAndTime.
	transition == nil ifTrue: [
		transition := transitions
			detect: [:each | each isDST not]
			ifNone: [^0].
	].
	^transition offsetFromUTC.
]
{ #category : 'public' }
TimeZoneInfo >> _offsetAtPosix: posixSeconds [
	| transition |
	transition := self _transitionAtPosix: posixSeconds.
	transition == nil ifTrue: [
		transition := transitions
			detect: [:each | each isDST not]
			ifNone: [^0].
	].
	^transition offsetFromUTC.
]

{ #category : 'instance creation' }
TimeZoneInfo >> readAbbreviations [

	| list index dict |
  list := String new .
	(stream next: charcnt) do: [:each |
		list add: ((each _isInteger) ifTrue:[ Character codePoint: each]
			                           ifFalse: [each]).
	].
	list := list subStrings: (Character codePoint: 0).
	dict := Dictionary new.
	index := 1.
	list do: [:each |
		dict
			at: index
			put: each.
		index := index + each size + 1.
	].
	1 to: types size do: [:i |
		| array offset |
		array := types at: i.
		offset := array at: 3.
		array at: 3 put: (dict at: offset otherwise: '' ).
	].

]

{ #category : 'instance creation' }
TimeZoneInfo >> readHeader [

	| version unused ttisgmtcnt ttisstdcnt |
	'TZif' do: [:each |
		self nextByte == each codePoint ifFalse: [self error: 'Invalid file format'].
	].
	version := stream next: 1.			"either an ASCII NUL ('\0') or a '2'"
	unused := stream next: 15.		"fifteen bytes containing zeros reserved for future use"
	ttisgmtcnt := self nextUnsignedInt.	"the number of UTC/local indicators stored in the file"
	ttisstdcnt := self nextUnsignedInt. 	"the number of standard/wall indicators stored in the file"
	leapSeconds := Array new: self nextUnsignedInt.	"size is the number of leap seconds for which data is stored in the file"
	transitions := (Array new: self nextUnsignedInt) 	"size is the number of 'transition times' for which data is stored in the file"
		collect: [:ignore | TimeZoneTransition new].
	types      := Array new: self nextUnsignedInt.		"size is the number of 'local time types' for which data is stored in the file (must not be zero)"
	charcnt    := self nextUnsignedInt.						"The number of characters of 'timezone abbreviation strings' stored in the file"

]

{ #category : 'instance creation' }
TimeZoneInfo >> readLeapSeconds [

	leapSeconds := leapSeconds collect: [:each |
                { self nextSignedInt . self nextSignedInt }
	].

]

{ #category : 'instance creation' }
TimeZoneInfo >> readTransitions [

        | tsize |
	1 to: (tsize := transitions size) do: [:j |
		(transitions at: j) transitionTime: self nextSignedInt.
	].
	1 to: tsize do: [:j |
		(transitions at: j) localTimeTypeID: self nextByte + 1.
	].
	tsize == 0 ifTrue: [ | tt |
                (tt := TimeZoneTransition new )
			transitionTime: 0;
			localTimeTypeID: 1.
		transitions add: tt .
	].

]

{ #category : 'instance creation' }
TimeZoneInfo >> readTypes [

	types := types collect: [:ignore |
	         { self nextSignedInt	"UTC offset" .
		    self nextByte == 1		"is DST" .
		    self nextByte + 1	"abbreviation" }
	].

]

{ #category : 'public' }
TimeZoneInfo >> transitionAtLocal: aDateAndTime [

	| seconds |
	seconds := aDateAndTime asPosixSeconds.
	^self detectLastTransition: [:each | each transitionTimeLocal <= seconds].

]

{ #category : 'public' }
TimeZoneInfo >> transitionAtUTC:   aDateAndTime [
	| seconds |
	seconds := aDateAndTime asPosixSeconds asFloat"optimization, fix 49425".
	^self detectLastTransition: [:each | each transitionTimeUTC   <= seconds].
]

{ #category : 'public' }
TimeZoneInfo >> _transitionAtPosix: posixSeconds [
  transitions size to: 1 by: -1 do: [:i |
    | each |
    each := transitions at: i .
    each transitionTimeUTC <= posixSeconds ifTrue:[ ^ each ].
  ].
  ^ nil.
]

{ #category : 'instance creation' }
TimeZoneInfo >> tzfile_h [
	"from tzfile.h in ftp://elsie.nci.nih.gov/pub/tzcode2006n.tar.gz on 2006-10-30"

^'
#define	TZ_MAGIC	"TZif"

struct tzhead {
	char	tzh_magic[4];			/* TZ_MAGIC */
	char	tzh_version[1];		/* ''\0'' or ''2'' as of 2005 */
	char	tzh_reserved[15];		/* reserved--must be zero */
	char	tzh_ttisgmtcnt[4];	/* coded number of trans. time flags */
	char	tzh_ttisstdcnt[4];	/* coded number of trans. time flags */
	char	tzh_leapcnt[4];		/* coded number of leap seconds */
	char	tzh_timecnt[4];		/* coded number of transition times */
	char	tzh_typecnt[4];		/* coded number of local time types */
	char	tzh_charcnt[4];		/* coded number of abbr. chars */
};

/*
** . . .followed by. . .
**
**	tzh_timecnt (char [4])s		coded transition times a la time(2)
**	tzh_timecnt (unsigned char)s	types of local time starting at above
**	tzh_typecnt repetitions of
**		one (char [4])		coded UTC offset in seconds
**		one (unsigned char)	used to set tm_isdst
**		one (unsigned char)	that''s an abbreviation list index
**	tzh_charcnt (char)s		''\0''-terminated zone abbreviations
**	tzh_leapcnt repetitions of
**		one (char [4])		coded leap second transition times
**		one (char [4])		total correction after above
**	tzh_ttisstdcnt (char)s		indexed by type; if TRUE, transition
**					time is standard time, if FALSE,
**					transition time is wall clock time
**					if absent, transition times are
**					assumed to be wall clock time
**	tzh_ttisgmtcnt (char)s		indexed by type; if TRUE, transition
**					time is UTC, if FALSE,
**					transition time is local time
**					if absent, transition times are
**					assumed to be local time
*/'

]

{ #category : 'instance creation' }
TimeZoneInfo >> updateTransitions [

   1 to: transitions size do: [:j |
      (transitions at:j) typeList: types
   ].

]
